/* GTK DBF Editor -- a simple dbf editor built with GTK+
 * Copyright (C) 2001-2012 Steffen Macke
 *
 * dbf.c: dbf editing functionality
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>

#include "interface.h"
#include "support.h"
#include "shapefil.h"
#include "table.h"
#include "dbf.h"

/* global variables */

/* DBFHandle used by shapelib */
DBFHandle dbfh_file = NULL;

/* the record that is currently selected */
int n_current_record = 0;

/* whether the file has changed (and needs saving)*/
gboolean dbf_has_changed = FALSE;

/* the dbf file that has been opened */
gchar *dbf_filename = NULL;

/* whether the application shall exit */
gboolean application_is_closing = FALSE;

/* the main window */
GtkWidget *dbf_editor_window = NULL;

gchar *dbf_encoding = "ISO-8859-15";
/* remove widgets for all the fields in the DBF file */
void
dbf_remove_widgets (GtkBox * vbox_fields)
{
  GList *children;

  children = gtk_container_children (GTK_CONTAINER (vbox_fields));
  while (children != NULL)
    {
      gtk_container_remove (GTK_CONTAINER (vbox_fields),
			    GTK_WIDGET (children->data));
      children = g_list_next (children);
    }
}

/* create widgets for all the fields in the DBF file */
void
dbf_create_widgets (gchar * filename, GtkBox * vbox_fields)
{

  GtkWidget *label1;
  GtkWidget *hbox1 = NULL;
  GtkWidget *entry1;
  GtkAdjustment *spinner_adj;
  int n_fields;
  int i;
  char field_name[12];
  int n_field_width;
  int n_field_decimals;
  DBFFieldType dbf_field_type;

  dbf_remove_widgets (vbox_fields);

  if (dbf_open_file (filename) == FALSE)
    return;

  dbf_filename = g_strdup (filename);

  n_fields = DBFGetFieldCount (dbfh_file);

  dbf_editor_window = glade_xml_get_widget (GLADE_XML (glade_xml),
					    "dbf_editor_window");

  for (i = 0; i < n_fields; i++)
    {
      dbf_field_type = DBFGetFieldInfo (dbfh_file, i, field_name,
					&n_field_width, &n_field_decimals);
      hbox1 = gtk_hbox_new (FALSE, 3);
      gtk_widget_ref (hbox1);
      gtk_object_set_data_full (GTK_OBJECT (dbf_editor_window), "hbox1",
				hbox1, (GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show (hbox1);
      gtk_box_pack_start (GTK_BOX (vbox_fields), hbox1, TRUE, TRUE, 0);

      label1 = gtk_label_new (field_name);
      gtk_widget_ref (label1);
      gtk_object_set_data_full (GTK_OBJECT (dbf_editor_window), "label1",
				label1, (GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show (label1);
      gtk_box_pack_start (GTK_BOX (hbox1), label1, FALSE, FALSE, 0);

      switch (dbf_field_type)
	{
	case FTString:
	  entry1 = gtk_entry_new_with_max_length ((guint16) n_field_width);
	  break;
	case FTDouble:
	  spinner_adj = GTK_ADJUSTMENT (gtk_adjustment_new (1.0, pow (10,
								      -n_field_decimals)
							    - pow (10,
								   n_field_width
								   -
								   n_field_decimals
								   - 1),
							    pow (10,
								 n_field_width
								 -
								 n_field_decimals
								 - 1) -
							    pow (10,
								 -n_field_decimals),
							    pow (10,
								 -n_field_decimals
								 - 1),
							    pow (10,
								 -n_field_decimals),
							    0.0));
	  entry1 = GTK_WIDGET (gtk_spin_button_new (spinner_adj, 1.0, 0));
	  gtk_spin_button_set_digits (GTK_SPIN_BUTTON (entry1),
				      n_field_decimals);
	  break;
	case FTInteger:
	  spinner_adj = GTK_ADJUSTMENT (gtk_adjustment_new (1.0, -pow (10,
								       n_field_width
								       - 1),
							    pow (10,
								 n_field_width
								 - 1), 1.0,
							    10.0, 0.0));
	  entry1 = GTK_WIDGET (gtk_spin_button_new (spinner_adj, 1.0, 0));
	  gtk_spin_button_set_digits (GTK_SPIN_BUTTON (entry1), 0);
	  break;
	}

      gtk_widget_ref (entry1);
      gtk_object_set_data_full (GTK_OBJECT (dbf_editor_window), field_name,
				entry1, (GtkDestroyNotify) gtk_widget_unref);
      gtk_signal_connect (GTK_OBJECT (entry1), "changed",
			  GTK_SIGNAL_FUNC (on_entry_changed), NULL);

      gtk_signal_connect (GTK_OBJECT (entry1), "focus_out_event",
			  GTK_SIGNAL_FUNC (on_entry_focus_out_event), NULL);

      gtk_widget_show (entry1);
      gtk_widget_set_name (entry1, field_name);
      gtk_box_pack_start (GTK_BOX (hbox1), entry1, TRUE, TRUE, 3);

    }
  dbf_display_record (dbf_editor_window, 0);
  gtk_container_foreach (GTK_CONTAINER
			 (glade_xml_get_widget (glade_xml, "hbox2")),
			 (GtkCallback *) gtk_widget_set_sensitive,
			 (gpointer) TRUE);
}

gboolean
dbf_load_file (gchar * filename)
{
  GtkWidget *vbox_fields;

  vbox_fields = GTK_WIDGET (glade_xml_get_widget (glade_xml, "vbox_fields"));
  n_current_record = 0;
  gtk_widget_set_sensitive (glade_xml_get_widget
			    (glade_xml, "toolbutton_save"), FALSE);

  dbf_create_widgets (filename, GTK_BOX (vbox_fields));
  if (DBFGetRecordCount (dbfh_file) == 0)
    {
      gtk_widget_set_sensitive (glade_xml_get_widget (glade_xml,
						      "go_first_record"),
				FALSE);
      gtk_widget_set_sensitive (glade_xml_get_widget
				(glade_xml, "go_next_record"), FALSE);
      gtk_widget_set_sensitive (glade_xml_get_widget
				(glade_xml, "go_previous_record"), FALSE);
      gtk_widget_set_sensitive (glade_xml_get_widget
				(glade_xml, "go_last_record"), FALSE);
    }
  else
    {
      dbf_fill_table ();
      dbf_table_select_row (n_current_record);
    }
  return TRUE;
}

/* add a record to the dbf file */
void
dbf_add_record (GtkWidget * dbf_editor_window)
{
  int n_record_count;

  dbf_table_add_row ();
  n_record_count = DBFGetRecordCount (dbfh_file);

  if (n_record_count == 0)
    {
      gtk_widget_set_sensitive (glade_xml_get_widget (glade_xml,
						      "go_first_record"),
				TRUE);
      gtk_widget_set_sensitive (glade_xml_get_widget
				(glade_xml, "go_next_record"), TRUE);
      gtk_widget_set_sensitive (glade_xml_get_widget
				(glade_xml, "go_previous_record"), TRUE);
      gtk_widget_set_sensitive (glade_xml_get_widget
				(glade_xml, "go_last_record"), TRUE);
    }

  DBFWriteNULLAttribute (dbfh_file, n_record_count, 0);
  dbf_display_record (dbf_editor_window, n_record_count);
  dbf_has_changed = TRUE;
}

/* display the last record */
void
dbf_display_last_record (GtkWidget * dbf_editor_window)
{
  int n_record_count;

  n_record_count = DBFGetRecordCount (dbfh_file);

  if (n_record_count > 0)
    dbf_display_record (dbf_editor_window, n_record_count - 1);
}

/* move to the next record */
void
dbf_display_next_record (GtkWidget * dbf_editor_window)
{
  int n_record_count;

  n_record_count = DBFGetRecordCount (dbfh_file);

  if (n_record_count == 0)
    return;
  if (n_current_record < n_record_count - 1)
    {
      dbf_display_record (dbf_editor_window, n_current_record + 1);
    }
  else
    {
      dbf_display_record (dbf_editor_window, 0);
    }
}

/* move the previous record */
void
dbf_display_previous_record (GtkWidget * dbf_editor_window)
{
  int n_record_count;

  n_record_count = DBFGetRecordCount (dbfh_file);

  if (n_record_count == 0)
    return;
  if (n_current_record > 0)
    {
      dbf_display_record (dbf_editor_window, n_current_record - 1);
    }
  else
    {
      dbf_display_last_record (dbf_editor_window);
    }
}

/* display message in the statusbar */
void
window_display_status_message (GtkWidget * dbf_editor_window, gchar * message)
{
  GtkStatusbar *statusbar1;
  static guint message_id = 0;
  static guint context_id = 0;

  statusbar1 = GTK_STATUSBAR (glade_xml_get_widget (glade_xml, "statusbar1"));
  if (message_id != 0)
    gtk_statusbar_pop (statusbar1, message_id);
  if (context_id == 0)
    context_id =
      gtk_statusbar_get_context_id (statusbar1, _("Display Record"));
  message_id = gtk_statusbar_push (statusbar1, context_id, message);
}

/* display one record */
void
dbf_display_record (GtkWidget * dbf_editor_window, gint n_record)
{
  GtkWidget *text_entry;
  GtkStatusbar *statusbar1;
  static guint message_id = 0;
  static guint context_id = 0;
  gchar status_message[32];
  DBFFieldType dbf_field_type = FTInvalid;

  int i;
  int n_fields;
  int n_record_count;
  char field_name[12];
  gchar *field_attribute;

  n_fields = DBFGetFieldCount (dbfh_file);
  n_record_count = DBFGetRecordCount (dbfh_file);
  if (n_record >= n_record_count)
    return;

  n_current_record = n_record;

  for (i = 0; i < n_fields; i++)
    {
      dbf_field_type = DBFGetFieldInfo (dbfh_file, i, field_name, NULL, NULL);
      text_entry =
	GTK_WIDGET (gtk_object_get_data
		    (GTK_OBJECT (dbf_editor_window), field_name));
      switch (dbf_field_type)
	{
	case FTString:
	  field_attribute =
	    g_convert ((gchar *)
		       DBFReadStringAttribute (dbfh_file, (int) n_record, i),
		       -1, "UTF-8", dbf_encoding, NULL, NULL, NULL);
	  gtk_entry_set_text (GTK_ENTRY (text_entry), field_attribute);
	  g_free (field_attribute);
	  break;
	case FTInteger:
	  gtk_spin_button_set_value (GTK_SPIN_BUTTON (text_entry),
				     (gdouble)
				     DBFReadIntegerAttribute (dbfh_file,
							      n_record, i));
	  break;
	case FTDouble:
	  gtk_spin_button_set_value (GTK_SPIN_BUTTON (text_entry),
				     (gdouble)
				     DBFReadDoubleAttribute (dbfh_file,
							     n_record, i));
	  break;
	}

    }

  statusbar1 = GTK_STATUSBAR (glade_xml_get_widget (glade_xml, "statusbar1"));
  if (message_id != 0)
    gtk_statusbar_pop (statusbar1, message_id);
  if (context_id == 0)
    context_id =
      gtk_statusbar_get_context_id (statusbar1, _("Display Record"));
  g_snprintf (status_message, 32, _("Record %d of %d"), n_record + 1,
	      n_record_count);
  message_id = gtk_statusbar_push (statusbar1, context_id, status_message);
  dbf_table_select_row (n_record);
}

/* open the dbf file */
gboolean
dbf_open_file (gchar * filename)
{
  if (dbf_can_close () == FALSE)
    {
      return FALSE;
    }
  dbfh_file = DBFOpen ((const char *) filename, "rb+");
  if (dbfh_file == NULL)
    {
      fprintf (stderr, _("ERROR: Unable to open dbf file '%s'.\n"), filename);
      return FALSE;
    }
  return TRUE;
  dbf_has_changed = FALSE;
}

/* save the dbf file */
void
dbf_save_file ()
{
  if (dbfh_file != NULL)
    {
      dbf_close_file ();
      dbf_open_file (dbf_filename);
    }
}

/* close the dbf file */
void
dbf_close_file ()
{
  if (dbfh_file != NULL)
    {
      DBFClose (dbfh_file);
      dbfh_file = NULL;
      dbf_has_changed = FALSE;
    }
}

/* return wether or not the file can be closed */
/* saves the file if necessary */
gboolean
dbf_can_close ()
{
  GtkWidget *dialog_dbf_has_changed;
  GtkLabel *label_unsaved_file;

  if (dbfh_file != NULL)
    {
      if (dbf_has_changed == TRUE)
	{
	  dialog_dbf_has_changed =
	    GTK_WIDGET (glade_xml_get_widget
			(glade_xml, "dialog_dbf_has_changed"));

	  label_unsaved_file = GTK_LABEL (glade_xml_get_widget (glade_xml,
								"label_unsaved_file"));
	  gtk_label_set_text (label_unsaved_file, dbf_filename);
	  gtk_widget_show (dialog_dbf_has_changed);
	  gtk_widget_grab_focus (dialog_dbf_has_changed);
	  gtk_grab_add (dialog_dbf_has_changed);

	  return FALSE;
	}
      else
	{
	  return TRUE;
	}
    }
  return TRUE;
}

G_MODULE_EXPORT void
on_entry_changed (GtkEditable * editable, gpointer user_data)
{
  gchar *widget_name;
  int i;
  int n_fields;
  char dbf_field_name[12];
  int dbf_field_width;
  DBFFieldType dbf_field_type = FTInvalid;
  int dbf_field_decimals;
  int dbf_field_number;
  char dbf_field_value[256];
  gchar *widget_text;
  gchar **tokens;

  widget_name = gtk_widget_get_name (GTK_WIDGET (editable));

  dbf_field_number = dbf_get_field_number (widget_name);

  widget_text = gtk_entry_get_text (GTK_ENTRY (editable));

  dbf_table_set_cell_value (widget_text, dbf_field_number, n_current_record);
}

/* write record to shapefile if it has changed */
G_MODULE_EXPORT gboolean
on_entry_focus_out_event (GtkWidget * widget, GdkEventFocus * event,
			  gpointer user_data)
{
  gchar *widget_text;
  gchar *widget_name;
  int dbf_field_number;
  DBFFieldType dbf_field_type;
  gchar *field_attribute;
  gchar *converted_text;

  widget_text = gtk_entry_get_text (GTK_ENTRY (widget));

  widget_name = gtk_widget_get_name (widget);
  dbf_field_number = dbf_get_field_number (widget_name);

  dbf_field_type = DBFGetFieldInfo (dbfh_file, dbf_field_number, NULL, NULL,
				    NULL);
  field_attribute = (gchar *) DBFReadStringAttribute (dbfh_file,
						      n_current_record,
						      dbf_field_number);

  switch (dbf_field_type)
    {
    case FTString:
      /**
       * \TODO Use g_ascii_strcasecmp
       */
      if (strcmp (field_attribute, widget_text) != 0)
	{
	  converted_text =
	    g_convert (widget_text, -1, dbf_encoding, "UTF-8", NULL, NULL,
		       NULL);
	  if (DBFWriteStringAttribute
	      (dbfh_file, n_current_record, dbf_field_number,
	       converted_text) != TRUE)
	    {
	      g_printerr (_("ERROR: Could not write record %d to '%s'\n"),
			  n_current_record, dbf_filename);
	    }
	  else
	    {
	      set_dbf_has_changed ();
	    }
	  g_free (converted_text);
	  break;
    case FTDouble:
	  if (DBFReadDoubleAttribute (dbfh_file, n_current_record,
				      dbf_field_number)
	      != gtk_spin_button_get_value (GTK_SPIN_BUTTON (widget)))
	    {
	      if (DBFWriteDoubleAttribute (dbfh_file, n_current_record,
					   dbf_field_number, (double)
					   gtk_spin_button_get_value
					   (GTK_SPIN_BUTTON (widget))) !=
		  TRUE)
		{
		  g_printerr (_("ERROR: Could not write record %d to '%s'\n"),
			      n_current_record, dbf_filename);
		}
	      else
		{
		  set_dbf_has_changed ();
		}
	      break;
    case FTInteger:
	      if (DBFReadIntegerAttribute (dbfh_file, n_current_record,
					   dbf_field_number)
		  !=
		  gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget)))
		{
		  if (DBFWriteIntegerAttribute (dbfh_file, n_current_record,
						dbf_field_number,
						gtk_spin_button_get_value_as_int
						(GTK_SPIN_BUTTON (widget))) !=
		      TRUE)
		    {
		      g_printerr (_
				  ("ERROR: Could not write record %d to '%s'\n"),
				  n_current_record, dbf_filename);
		    }
		  else
		    {
		      set_dbf_has_changed ();
		    }
		}
	      break;
	    }
	}
    }
  return FALSE;
}

int
dbf_get_field_number (char *field_name)
{
  int i;
  int n_fields;
  char dbf_field_name[12];

  n_fields = DBFGetFieldCount (dbfh_file);
  for (i = 0; i < n_fields; i++)
    {
      DBFGetFieldInfo (dbfh_file, i, dbf_field_name, NULL, NULL);
      if (strcmp (dbf_field_name, field_name) == 0)
	{
	  return i;
	  break;
	}
    }
  fprintf (stderr, _("ERROR: Can't find field '%s'.\n"), field_name);
  return -1;
}

/**
 * Enable Save button and menu once the file has changed.
 */
void
set_dbf_has_changed ()
{
  GtkWidget *widget;

  dbf_has_changed = TRUE;
  widget = GTK_WIDGET (glade_xml_get_widget (glade_xml, "toolbutton_save"));
  gtk_widget_set_sensitive (widget, TRUE);
}
