/* Code to read Garmin GPS 25/35LP binary output
 * 
 * For license terms, see the file COPYING in this directory
 *
 * Takes two Garmin binary files and performs post processed DPGS. More
 * doc to follow later....
 * 
 * Last update: 19 Dec 1999 by Joe Desbonnet, joe@wombat.ie
 *
 * Joe Desbonnet, joe@wombat.ie
 * 
 * 10 July 1999
 * Minor mods
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <signal.h>

#include "libgps.h"
#include "garmin.h"


/* Define functions */
void usage ();

void read_rcv_record (unsigned char *, FILE *fp);
void read_pvt_record (unsigned char *, FILE *fp);
void read_eph_record (unsigned char *, FILE *fp);
int verify_checksum (void *memaddr, int len, int expected_checksum);
int retrieve_record (unsigned char *buf, int length, FILE *fp);
boolean calc_posn (vec3 Xr, garmin_rcv_type rcv_rec, double tow);
void output_to_screen ();
void skymap (FILE *);
void handle_packet (FILE *, garmin_rcv_type *, garmin_pvt_type *, double *);
void do_diff_stuff (void);


/* Define signal handlers */
void sig_hup_handler (int);
void sig_pipe_handler (int);
void sig_term_handler (int);

/* Globals */

/* Array of latest ephemeris records */
eph_type current_eph[32];
boolean  got_eph[32];

/* Array of latest corrections (meters) */
double prc[32];

/* Array of validity times of latest corrections */
double prt[32];


/* Current SV range/phase records */
garmin_rcv_type ref_rcv;
garmin_rcv_type rov_rcv;

/* Current PVT record */
garmin_pvt_type ref_pvt;
garmin_pvt_type rov_pvt;

/* current position estimate */
vec3     current_Xr;

/* current calculated clock error (in range meters;  *NOT* seconds) */
double   cclke_m;

/* current position as reported by the Garmin unit */
vec3     current_reported_lla;

/* Reference station position. -- experimental stuff -- ignore */
vec3     ref_Xr;


double   ref_tow = -1;
double   rov_tow = -1;

#define MASK_ANGLE 20

/* Main Program */
int main (int argc, char **argv) 
{
    
    int i, nrefpacket=0, nrovpacket=0;
    vec3 lla;
    double prev_ref_tow, prev_rov_tow;
    FILE *ref_fp=NULL, *rov_fp=NULL;
    char *ref_filename=NULL, *rov_filename=NULL;


    /* Setup signal handlers */
    signal (SIGHUP, sig_hup_handler);
    signal (SIGPIPE, sig_pipe_handler);
    signal (SIGTERM, sig_term_handler);
    signal (SIGINT, sig_term_handler);

    /* Initialize global variables */
    bzero (current_eph, sizeof (eph_type) * 32);
    bzero (got_eph, sizeof (boolean) * 32);
    bzero (&ref_pvt, sizeof (garmin_pvt_type));
    bzero (&rov_pvt, sizeof (garmin_pvt_type));
    bzero (&ref_rcv, sizeof (garmin_rcv_type));
    bzero (&rov_rcv, sizeof (garmin_rcv_type));
    bzero (prc, sizeof (double) * 32);
    bzero (prt, sizeof (double) * 32);
    
    cclke_m=0;
    lla[0] = 53 * DEG_TO_RAD;
    lla[1] = -6 * DEG_TO_RAD;
    lla[2] = 70;
    
    /* This is where my units sit when I do zero basline tests. +/- 20m or so */
    ref_Xr[0]= 3795025;
    ref_Xr[1]= -416238;
    ref_Xr[2]= 5092264;

    gps_lla2xyz(lla,current_Xr);
  

    /* Parse switches. Will move to getopt some day. */

    for (i=1; i<argc; i++) 
    {
	if (*argv[i] == '-') 
	{
	    switch (*(argv[i]+1)) {
	     case 'r':
		i++;
		if (i == argc)
		{
		    break;
		}
		ref_filename = argv[i];
		break;
		
	     case 'm':
		i++;
		if (i == argc)
		{
		    break;
		}
		rov_filename = argv[i];
		break;
		
	    }
	}
    }

    if (ref_filename==NULL || rov_filename==NULL) 
    {
	usage();
	exit(1);
    }
    
    ref_fp = fopen (ref_filename,"r");
    if (ref_fp==NULL)
    {
	fprintf (stderr, "Error opening ref_filename %s", ref_filename);
    }
    
    rov_fp = fopen (rov_filename,"r");
    if (rov_fp==NULL)
    {
	fprintf (stderr, "Error opening ref_filename %s", rov_filename);
    }
    
    
    /* Skip to first valid record */
    garmin_skip_to_next_record(ref_fp);
    garmin_skip_to_next_record(rov_fp);

    /* Main loop */
    while (!feof(ref_fp) && !feof(rov_fp)) 
    {
	/*
	fprintf (stderr,"nref=%5d nrov=%5d  ref_tow=%15.8f  rov_tow=%15.8f\n",
		 nrefpacket,
		 nrovpacket,
		 ref_tow,
		 rov_tow
		 );
	*/
	
	/* Rov file ahead by atleast one second -- read a Ref packet */
	if ( (rov_tow - ref_tow) > 0.0001) 
	{
	    handle_packet (ref_fp, &ref_rcv, &ref_pvt, &ref_tow);
	    nrefpacket++;
	    continue;
	}
	
	/* Ref file ahead by atleast one second -- read a Rov packet */
	if ( (ref_tow - rov_tow) > 0.0001)
	{
	    handle_packet (rov_fp, &rov_rcv, &rov_pvt, &rov_tow);
	    nrovpacket++;
	    continue;
	}

	/* At this point tow should coincide */
	if (ref_tow > 0  &&  rov_tow > 0) 
	{
	    do_diff_stuff();
	}
	
	/* Read one Ref packet and one Rov packet */
	prev_ref_tow = ref_tow;
	prev_rov_tow = rov_tow;
	while (ref_tow == prev_ref_tow && rov_tow == prev_rov_tow)
	{
	    handle_packet (ref_fp, &ref_rcv, &ref_pvt, &ref_tow);
	    nrefpacket++;
	    handle_packet (rov_fp, &rov_rcv, &rov_pvt, &rov_tow);
	    nrovpacket++;
	}
	
	
    }
    return 0;
}


void handle_packet (FILE *fp, 
		    garmin_rcv_type *rcv,
		    garmin_pvt_type *pvt,
		    double *tow)
{
    int rec_type, rec_len, prn;
    char buf[256];
    
    eph_type eph;
    
    /* Read record type and length */
    rec_type = garmin_get_data_byte(fp);
    rec_len = garmin_get_data_byte(fp);
    
    if (DEBUG) 
    {
	fprintf (stderr,"rec_type = 0x%02x, rec_len = %d\n",rec_type,rec_len);
    }
    
    
    retrieve_record(buf,rec_len,fp);
    
    switch (rec_type) {
     case RCV_REC:
	if (DEBUG) fprintf (stderr,"Receiver record found.\n");
	garmin_rcv_record(buf, rcv);
	break;
	
     case POS_REC:
	if (DEBUG) fprintf (stderr,"Position record found.\n");
	garmin_pvt_record(buf, pvt);
	*tow = pvt->tow;
	break;
	
     case EPH_REC:
	if (DEBUG) fprintf (stderr,"Ephemeris record found. len=%d\n",rec_len);
	garmin_eph_record(buf, &eph);
	prn = (int)eph.prn;
	if ( (got_eph[prn]==false) ||  (current_eph[prn].iod != eph.iod)  ) 
	{
	    memcpy (&current_eph[prn],&eph, sizeof(eph_type));
	    got_eph[prn]=true;
	}
	
	break;
	
     default:
	garmin_skip_to_next_record(fp);
	break;
    }
    
}

int retrieve_record (unsigned char *buf, int length, FILE *fp) 
{
    int i, checksum;
    
    for (i=0; i<length; i++) 
    {
      buf[i] = garmin_get_data_byte (fp);
    }
    
    checksum = garmin_get_data_byte (fp);
   
    return 0; /* FIXME */
}


void do_diff_stuff ()
{
    vec3 Xs,errorvec, Xrg, current_lla, dXrXs;
    double Trel,az,el,errormag,real_range, range_error;
    double gclke_m, gclke_s, cclke_t,gre,cre;
    boolean status;
    int i,prn,nvalid;
    boolean have_valid=false;

    /* Calculate pseudorange corrections */
    
    for (i=0; i<12; i++)
    {
	
	/* Only continue with this channel record if it contains valid 
	 * data AND we have a valid ephemeris record for this SV */
	if (ref_rcv.sv[i].valid == false)
	{
	    continue;
	}
	have_valid=true;
	
	prn = ref_rcv.sv[i].prn;
	
	if (got_eph[prn] == false)
	{
	    continue;
	}
	
	nvalid++;
	
	/* Where is the satellite (SV) in space and where in the sky? */
	gps_satpos (current_eph[prn], ref_tow, &Trel, Xs);
	gps_calcAzEl (Xs, ref_Xr, &az, &el, &status);

	/* Displacement between ref position and SV. This sould be very
	 * accurate as the only error is that in ephemeris and our
	 * surveyed position (and the latter is a constant error) */
	vec3sub (dXrXs, Xs, ref_Xr);
	real_range = vec3mag (dXrXs);
	
	/* Clock error (according to Garmin data) */
	gclke_s = ref_rcv.tow - ref_tow; /* Clock error in seconds */
	gclke_m = gclke_s * const_c;         /* Clock error in range meters */
	gre = ref_rcv.sv[i].pr - gclke_m;    /* Range error */
	
	prc[prn] = ref_rcv.sv[i].pr - gclke_m -  real_range; /* Calculated Range Error */
	prt[prn] = ref_tow;

	printf ("%15.6f %f   666777666\n", ref_tow, gclke_s);
	
	if (DEBUG)
	{
	    fprintf (stderr,"tow=%f  prc[%02d] = %f\n",ref_tow,prn,prc[prn]);
	}
	
    }
    
    if (nvalid == 0) 
    {
	fprintf (stderr,"No valid channel records at %f!!\n",ref_tow);
    }
    
    /* How many rov SV records? */
    nvalid=0;
    for (i=0; i<12; i++) 
    {
	if (rov_rcv.sv[i].valid) 
	{
	    nvalid++;
	}
    }
    
    if (nvalid < 4) 
    {
	return;
    }
    
    for (i=0; i<12; i++) 
    {
	prn = rov_rcv.sv[i].prn;
	if (!rov_rcv.sv[i].valid)
	{
	    continue;
	}
	if (!got_eph[prn]) 
	{
	    continue;
	}
	/* Make sure corrections are still valid. FIXME does not handle EOW */
	if (rov_tow - prt[prn] > 10) 
	{
	    fprintf (stderr,"Not correcting prn %d at %d -- too old by %f\n",
		     (int)prn,
		     (int)(rov_tow+0.5),
		     rov_tow - prt[prn]
		     );
	    continue;
	}
	
	rov_rcv.sv[i].pr -= prc[prn];
    }
    
    status = calc_posn (current_Xr, rov_rcv, rov_tow);
    
    if (status)
    {
	gps_xyz2lla (current_Xr, current_lla);
	/* The results */
	printf ("%15.8f %10.6f %10.6f %10.6f  %10.6f %10.6f %10.6f 666444666\n",
		rov_tow,
		current_lla[0] * RAD_TO_DEG,
		current_lla[1] * RAD_TO_DEG,
		current_lla[2],
		rov_pvt.lat * RAD_TO_DEG,
		rov_pvt.lon * RAD_TO_DEG,
		rov_pvt.alt);
	
	printf ("%15.8f " , ref_tow);
	for (i=0; i<32; i++) 
	{
	    printf ("%8.2f ",prc[i]);
	}
	printf (" 666999666\n");
    }
    
}

		
boolean calc_posn (vec3 Xr, garmin_rcv_type  rcv_rec, double tow)
{
    boolean status;
    int i,prn;
    double pseudorange[32];
    boolean pseudorange_valid[32];
    
    /* Zero arrays */
    for (i=0; i<32; i++)
    {
	pseudorange[i]=0.0;
	pseudorange_valid[i]=0;
    }
    
    for (i=0; i<12; i++)
    {
	if (rcv_rec.sv[i].valid == false) 
	{
	    continue;
	}
	
	prn = rcv_rec.sv[i].prn;
	
	if (got_eph[prn] == false)
	{
	    continue;
	}
	
	pseudorange[prn] = rcv_rec.sv[i].pr;
	pseudorange_valid[prn] = true;
    }
    
    return gps_sps_calc (Xr, 
		  tow, 
		  pseudorange_valid,
		  current_eph, 
		  NULL,
		  pseudorange,
		  &cclke_m);

}

/* Handle the HUP signal */
void sig_hup_handler (int i)
{
    if (DEBUG)
    {
	fprintf (stderr,"Caught SIGHUP\n");
	fflush (stderr);
    }

    /* reestablish signal handler */
    signal (SIGHUP, sig_hup_handler);
}
void sig_pipe_handler (int i)
{
    if (DEBUG)
    {
	fprintf (stderr,"Caught SIGPIPE.\n");
	fflush (stderr);
    }
    
    exit(0);
}

void sig_term_handler (int i)
{
    fprintf (stderr,"Caught SIGTERM\n");
    fflush (stderr);
    exit(0);
}


void usage ()
{
   fprintf (stderr,"\
Usage: garminbin -r <file> -m <file> \n\n\
-r  Garmin binary data from reference receiver  \n\
-m  Garmin binary data from moving receiver \n\
\n\
Text output is on stdout\n\
Currently does not work!!!\n\n");
}
