// Copyright 2024 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package accesscontextmanager provides access to the Access Context Manager API.
//
// For product documentation, see: https://cloud.google.com/access-context-manager/docs/reference/rest/
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/accesscontextmanager/v1"
//	...
//	ctx := context.Background()
//	accesscontextmanagerService, err := accesscontextmanager.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	accesscontextmanagerService, err := accesscontextmanager.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	accesscontextmanagerService, err := accesscontextmanager.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package accesscontextmanager // import "google.golang.org/api/accesscontextmanager/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version

const apiId = "accesscontextmanager:v1"
const apiName = "accesscontextmanager"
const apiVersion = "v1"
const basePath = "https://accesscontextmanager.googleapis.com/"
const basePathTemplate = "https://accesscontextmanager.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://accesscontextmanager.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, configure, and delete your Google Cloud data and see the email
	// address for your Google Account.
	CloudPlatformScope = "https://www.googleapis.com/auth/cloud-platform"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/cloud-platform",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s, err := New(client)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	s := &Service{client: client, BasePath: basePath}
	s.AccessPolicies = NewAccessPoliciesService(s)
	s.Operations = NewOperationsService(s)
	s.Organizations = NewOrganizationsService(s)
	s.Services = NewServicesService(s)
	return s, nil
}

type Service struct {
	client    *http.Client
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	AccessPolicies *AccessPoliciesService

	Operations *OperationsService

	Organizations *OrganizationsService

	Services *ServicesService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAccessPoliciesService(s *Service) *AccessPoliciesService {
	rs := &AccessPoliciesService{s: s}
	rs.AccessLevels = NewAccessPoliciesAccessLevelsService(s)
	rs.AuthorizedOrgsDescs = NewAccessPoliciesAuthorizedOrgsDescsService(s)
	rs.ServicePerimeters = NewAccessPoliciesServicePerimetersService(s)
	return rs
}

type AccessPoliciesService struct {
	s *Service

	AccessLevels *AccessPoliciesAccessLevelsService

	AuthorizedOrgsDescs *AccessPoliciesAuthorizedOrgsDescsService

	ServicePerimeters *AccessPoliciesServicePerimetersService
}

func NewAccessPoliciesAccessLevelsService(s *Service) *AccessPoliciesAccessLevelsService {
	rs := &AccessPoliciesAccessLevelsService{s: s}
	return rs
}

type AccessPoliciesAccessLevelsService struct {
	s *Service
}

func NewAccessPoliciesAuthorizedOrgsDescsService(s *Service) *AccessPoliciesAuthorizedOrgsDescsService {
	rs := &AccessPoliciesAuthorizedOrgsDescsService{s: s}
	return rs
}

type AccessPoliciesAuthorizedOrgsDescsService struct {
	s *Service
}

func NewAccessPoliciesServicePerimetersService(s *Service) *AccessPoliciesServicePerimetersService {
	rs := &AccessPoliciesServicePerimetersService{s: s}
	return rs
}

type AccessPoliciesServicePerimetersService struct {
	s *Service
}

func NewOperationsService(s *Service) *OperationsService {
	rs := &OperationsService{s: s}
	return rs
}

type OperationsService struct {
	s *Service
}

func NewOrganizationsService(s *Service) *OrganizationsService {
	rs := &OrganizationsService{s: s}
	rs.GcpUserAccessBindings = NewOrganizationsGcpUserAccessBindingsService(s)
	return rs
}

type OrganizationsService struct {
	s *Service

	GcpUserAccessBindings *OrganizationsGcpUserAccessBindingsService
}

func NewOrganizationsGcpUserAccessBindingsService(s *Service) *OrganizationsGcpUserAccessBindingsService {
	rs := &OrganizationsGcpUserAccessBindingsService{s: s}
	return rs
}

type OrganizationsGcpUserAccessBindingsService struct {
	s *Service
}

func NewServicesService(s *Service) *ServicesService {
	rs := &ServicesService{s: s}
	return rs
}

type ServicesService struct {
	s *Service
}

// AccessContextManagerOperationMetadata: Metadata of Access Context Manager's
// Long Running Operations.
type AccessContextManagerOperationMetadata struct {
}

// AccessLevel: An `AccessLevel` is a label that can be applied to requests to
// Google Cloud services, along with a list of requirements necessary for the
// label to be applied.
type AccessLevel struct {
	// Basic: A `BasicLevel` composed of `Conditions`.
	Basic *BasicLevel `json:"basic,omitempty"`
	// Custom: A `CustomLevel` written in the Common Expression Language.
	Custom *CustomLevel `json:"custom,omitempty"`
	// Description: Description of the `AccessLevel` and its use. Does not affect
	// behavior.
	Description string `json:"description,omitempty"`
	// Name: Identifier. Resource name for the `AccessLevel`. Format:
	// `accessPolicies/{access_policy}/accessLevels/{access_level}`. The
	// `access_level` component must begin with a letter, followed by alphanumeric
	// characters or `_`. Its maximum length is 50 characters. After you create an
	// `AccessLevel`, you cannot change its `name`.
	Name string `json:"name,omitempty"`
	// Title: Human readable title. Must be unique within the Policy.
	Title string `json:"title,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Basic") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Basic") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessLevel) MarshalJSON() ([]byte, error) {
	type NoMethod AccessLevel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AccessPolicy: `AccessPolicy` is a container for `AccessLevels` (which define
// the necessary attributes to use Google Cloud services) and
// `ServicePerimeters` (which define regions of services able to freely pass
// data within a perimeter). An access policy is globally visible within an
// organization, and the restrictions it specifies apply to all projects within
// an organization.
type AccessPolicy struct {
	// Etag: Output only. An opaque identifier for the current version of the
	// `AccessPolicy`. This will always be a strongly validated etag, meaning that
	// two Access Policies will be identical if and only if their etags are
	// identical. Clients should not expect this to be in any specific format.
	Etag string `json:"etag,omitempty"`
	// Name: Output only. Identifier. Resource name of the `AccessPolicy`. Format:
	// `accessPolicies/{access_policy}`
	Name string `json:"name,omitempty"`
	// Parent: Required. The parent of this `AccessPolicy` in the Cloud Resource
	// Hierarchy. Currently immutable once created. Format:
	// `organizations/{organization_id}`
	Parent string `json:"parent,omitempty"`
	// Scopes: The scopes of the AccessPolicy. Scopes define which resources a
	// policy can restrict and where its resources can be referenced. For example,
	// policy A with `scopes=["folders/123"]` has the following behavior: -
	// ServicePerimeter can only restrict projects within `folders/123`. -
	// ServicePerimeter within policy A can only reference access levels defined
	// within policy A. - Only one policy can include a given scope; thus,
	// attempting to create a second policy which includes `folders/123` will
	// result in an error. If no scopes are provided, then any resource within the
	// organization can be restricted. Scopes cannot be modified after a policy is
	// created. Policies can only have a single scope. Format: list of
	// `folders/{folder_number}` or `projects/{project_number}`
	Scopes []string `json:"scopes,omitempty"`
	// Title: Required. Human readable title. Does not affect behavior.
	Title string `json:"title,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessPolicy) MarshalJSON() ([]byte, error) {
	type NoMethod AccessPolicy
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ApiOperation: Identification for an API Operation.
type ApiOperation struct {
	// MethodSelectors: API methods or permissions to allow. Method or permission
	// must belong to the service specified by `service_name` field. A single
	// MethodSelector entry with `*` specified for the `method` field will allow
	// all methods AND permissions for the service specified in `service_name`.
	MethodSelectors []*MethodSelector `json:"methodSelectors,omitempty"`
	// ServiceName: The name of the API whose methods or permissions the
	// IngressPolicy or EgressPolicy want to allow. A single ApiOperation with
	// `service_name` field set to `*` will allow all methods AND permissions for
	// all services.
	ServiceName string `json:"serviceName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MethodSelectors") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MethodSelectors") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ApiOperation) MarshalJSON() ([]byte, error) {
	type NoMethod ApiOperation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Application: An application that accesses Google Cloud APIs.
type Application struct {
	// ClientId: The OAuth client ID of the application.
	ClientId string `json:"clientId,omitempty"`
	// Name: The name of the application. Example: "Cloud Console"
	Name string `json:"name,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ClientId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ClientId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Application) MarshalJSON() ([]byte, error) {
	type NoMethod Application
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AuditConfig: Specifies the audit configuration for a service. The
// configuration determines which permission types are logged, and what
// identities, if any, are exempted from logging. An AuditConfig must have one
// or more AuditLogConfigs. If there are AuditConfigs for both `allServices`
// and a specific service, the union of the two AuditConfigs is used for that
// service: the log_types specified in each AuditConfig are enabled, and the
// exempted_members in each AuditLogConfig are exempted. Example Policy with
// multiple AuditConfigs: { "audit_configs": [ { "service": "allServices",
// "audit_log_configs": [ { "log_type": "DATA_READ", "exempted_members": [
// "user:jose@example.com" ] }, { "log_type": "DATA_WRITE" }, { "log_type":
// "ADMIN_READ" } ] }, { "service": "sampleservice.googleapis.com",
// "audit_log_configs": [ { "log_type": "DATA_READ" }, { "log_type":
// "DATA_WRITE", "exempted_members": [ "user:aliya@example.com" ] } ] } ] } For
// sampleservice, this policy enables DATA_READ, DATA_WRITE and ADMIN_READ
// logging. It also exempts `jose@example.com` from DATA_READ logging, and
// `aliya@example.com` from DATA_WRITE logging.
type AuditConfig struct {
	// AuditLogConfigs: The configuration for logging of each type of permission.
	AuditLogConfigs []*AuditLogConfig `json:"auditLogConfigs,omitempty"`
	// Service: Specifies a service that will be enabled for audit logging. For
	// example, `storage.googleapis.com`, `cloudsql.googleapis.com`. `allServices`
	// is a special value that covers all services.
	Service string `json:"service,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AuditLogConfigs") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AuditLogConfigs") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AuditConfig) MarshalJSON() ([]byte, error) {
	type NoMethod AuditConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AuditLogConfig: Provides the configuration for logging a type of
// permissions. Example: { "audit_log_configs": [ { "log_type": "DATA_READ",
// "exempted_members": [ "user:jose@example.com" ] }, { "log_type":
// "DATA_WRITE" } ] } This enables 'DATA_READ' and 'DATA_WRITE' logging, while
// exempting jose@example.com from DATA_READ logging.
type AuditLogConfig struct {
	// ExemptedMembers: Specifies the identities that do not cause logging for this
	// type of permission. Follows the same format of Binding.members.
	ExemptedMembers []string `json:"exemptedMembers,omitempty"`
	// LogType: The log type that this config enables.
	//
	// Possible values:
	//   "LOG_TYPE_UNSPECIFIED" - Default case. Should never be this.
	//   "ADMIN_READ" - Admin reads. Example: CloudIAM getIamPolicy
	//   "DATA_WRITE" - Data writes. Example: CloudSQL Users create
	//   "DATA_READ" - Data reads. Example: CloudSQL Users list
	LogType string `json:"logType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ExemptedMembers") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ExemptedMembers") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AuditLogConfig) MarshalJSON() ([]byte, error) {
	type NoMethod AuditLogConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AuthorizedOrgsDesc: `AuthorizedOrgsDesc` contains data for an organization's
// authorization policy.
type AuthorizedOrgsDesc struct {
	// AssetType: The asset type of this authorized orgs desc. Valid values are
	// `ASSET_TYPE_DEVICE`, and `ASSET_TYPE_CREDENTIAL_STRENGTH`.
	//
	// Possible values:
	//   "ASSET_TYPE_UNSPECIFIED" - No asset type specified.
	//   "ASSET_TYPE_DEVICE" - Device asset type.
	//   "ASSET_TYPE_CREDENTIAL_STRENGTH" - Credential strength asset type.
	AssetType string `json:"assetType,omitempty"`
	// AuthorizationDirection: The direction of the authorization relationship
	// between this organization and the organizations listed in the `orgs` field.
	// The valid values for this field include the following:
	// `AUTHORIZATION_DIRECTION_FROM`: Allows this organization to evaluate traffic
	// in the organizations listed in the `orgs` field.
	// `AUTHORIZATION_DIRECTION_TO`: Allows the organizations listed in the `orgs`
	// field to evaluate the traffic in this organization. For the authorization
	// relationship to take effect, all of the organizations must authorize and
	// specify the appropriate relationship direction. For example, if organization
	// A authorized organization B and C to evaluate its traffic, by specifying
	// `AUTHORIZATION_DIRECTION_TO` as the authorization direction, organizations B
	// and C must specify `AUTHORIZATION_DIRECTION_FROM` as the authorization
	// direction in their `AuthorizedOrgsDesc` resource.
	//
	// Possible values:
	//   "AUTHORIZATION_DIRECTION_UNSPECIFIED" - No direction specified.
	//   "AUTHORIZATION_DIRECTION_TO" - The specified organizations are authorized
	// to evaluate traffic in this organization.
	//   "AUTHORIZATION_DIRECTION_FROM" - The traffic of the specified
	// organizations can be evaluated by this organization.
	AuthorizationDirection string `json:"authorizationDirection,omitempty"`
	// AuthorizationType: A granular control type for authorization levels. Valid
	// value is `AUTHORIZATION_TYPE_TRUST`.
	//
	// Possible values:
	//   "AUTHORIZATION_TYPE_UNSPECIFIED" - No authorization type specified.
	//   "AUTHORIZATION_TYPE_TRUST" - This authorization relationship is "trust".
	AuthorizationType string `json:"authorizationType,omitempty"`
	// Name: Identifier. Resource name for the `AuthorizedOrgsDesc`. Format:
	// `accessPolicies/{access_policy}/authorizedOrgsDescs/{authorized_orgs_desc}`.
	// The `authorized_orgs_desc` component must begin with a letter, followed by
	// alphanumeric characters or `_`. After you create an `AuthorizedOrgsDesc`,
	// you cannot change its `name`.
	Name string `json:"name,omitempty"`
	// Orgs: The list of organization ids in this AuthorizedOrgsDesc. Format:
	// `organizations/` Example: `organizations/123456`
	Orgs []string `json:"orgs,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AssetType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AssetType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AuthorizedOrgsDesc) MarshalJSON() ([]byte, error) {
	type NoMethod AuthorizedOrgsDesc
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// BasicLevel: `BasicLevel` is an `AccessLevel` using a set of recommended
// features.
type BasicLevel struct {
	// CombiningFunction: How the `conditions` list should be combined to determine
	// if a request is granted this `AccessLevel`. If AND is used, each `Condition`
	// in `conditions` must be satisfied for the `AccessLevel` to be applied. If OR
	// is used, at least one `Condition` in `conditions` must be satisfied for the
	// `AccessLevel` to be applied. Default behavior is AND.
	//
	// Possible values:
	//   "AND" - All `Conditions` must be true for the `BasicLevel` to be true.
	//   "OR" - If at least one `Condition` is true, then the `BasicLevel` is true.
	CombiningFunction string `json:"combiningFunction,omitempty"`
	// Conditions: Required. A list of requirements for the `AccessLevel` to be
	// granted.
	Conditions []*Condition `json:"conditions,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CombiningFunction") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CombiningFunction") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s BasicLevel) MarshalJSON() ([]byte, error) {
	type NoMethod BasicLevel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Binding: Associates `members`, or principals, with a `role`.
type Binding struct {
	// Condition: The condition that is associated with this binding. If the
	// condition evaluates to `true`, then this binding applies to the current
	// request. If the condition evaluates to `false`, then this binding does not
	// apply to the current request. However, a different role binding might grant
	// the same role to one or more of the principals in this binding. To learn
	// which resources support conditions in their IAM policies, see the IAM
	// documentation
	// (https://cloud.google.com/iam/help/conditions/resource-policies).
	Condition *Expr `json:"condition,omitempty"`
	// Members: Specifies the principals requesting access for a Google Cloud
	// resource. `members` can have the following values: * `allUsers`: A special
	// identifier that represents anyone who is on the internet; with or without a
	// Google account. * `allAuthenticatedUsers`: A special identifier that
	// represents anyone who is authenticated with a Google account or a service
	// account. Does not include identities that come from external identity
	// providers (IdPs) through identity federation. * `user:{emailid}`: An email
	// address that represents a specific Google account. For example,
	// `alice@example.com` . * `serviceAccount:{emailid}`: An email address that
	// represents a Google service account. For example,
	// `my-other-app@appspot.gserviceaccount.com`. *
	// `serviceAccount:{projectid}.svc.id.goog[{namespace}/{kubernetes-sa}]`: An
	// identifier for a Kubernetes service account
	// (https://cloud.google.com/kubernetes-engine/docs/how-to/kubernetes-service-accounts).
	// For example, `my-project.svc.id.goog[my-namespace/my-kubernetes-sa]`. *
	// `group:{emailid}`: An email address that represents a Google group. For
	// example, `admins@example.com`. * `domain:{domain}`: The G Suite domain
	// (primary) that represents all the users of that domain. For example,
	// `google.com` or `example.com`. *
	// `principal://iam.googleapis.com/locations/global/workforcePools/{pool_id}/sub
	// ject/{subject_attribute_value}`: A single identity in a workforce identity
	// pool. *
	// `principalSet://iam.googleapis.com/locations/global/workforcePools/{pool_id}/
	// group/{group_id}`: All workforce identities in a group. *
	// `principalSet://iam.googleapis.com/locations/global/workforcePools/{pool_id}/
	// attribute.{attribute_name}/{attribute_value}`: All workforce identities with
	// a specific attribute value. *
	// `principalSet://iam.googleapis.com/locations/global/workforcePools/{pool_id}/
	// *`: All identities in a workforce identity pool. *
	// `principal://iam.googleapis.com/projects/{project_number}/locations/global/wo
	// rkloadIdentityPools/{pool_id}/subject/{subject_attribute_value}`: A single
	// identity in a workload identity pool. *
	// `principalSet://iam.googleapis.com/projects/{project_number}/locations/global
	// /workloadIdentityPools/{pool_id}/group/{group_id}`: A workload identity pool
	// group. *
	// `principalSet://iam.googleapis.com/projects/{project_number}/locations/global
	// /workloadIdentityPools/{pool_id}/attribute.{attribute_name}/{attribute_value}
	// `: All identities in a workload identity pool with a certain attribute. *
	// `principalSet://iam.googleapis.com/projects/{project_number}/locations/global
	// /workloadIdentityPools/{pool_id}/*`: All identities in a workload identity
	// pool. * `deleted:user:{emailid}?uid={uniqueid}`: An email address (plus
	// unique identifier) representing a user that has been recently deleted. For
	// example, `alice@example.com?uid=123456789012345678901`. If the user is
	// recovered, this value reverts to `user:{emailid}` and the recovered user
	// retains the role in the binding. *
	// `deleted:serviceAccount:{emailid}?uid={uniqueid}`: An email address (plus
	// unique identifier) representing a service account that has been recently
	// deleted. For example,
	// `my-other-app@appspot.gserviceaccount.com?uid=123456789012345678901`. If the
	// service account is undeleted, this value reverts to
	// `serviceAccount:{emailid}` and the undeleted service account retains the
	// role in the binding. * `deleted:group:{emailid}?uid={uniqueid}`: An email
	// address (plus unique identifier) representing a Google group that has been
	// recently deleted. For example,
	// `admins@example.com?uid=123456789012345678901`. If the group is recovered,
	// this value reverts to `group:{emailid}` and the recovered group retains the
	// role in the binding. *
	// `deleted:principal://iam.googleapis.com/locations/global/workforcePools/{pool
	// _id}/subject/{subject_attribute_value}`: Deleted single identity in a
	// workforce identity pool. For example,
	// `deleted:principal://iam.googleapis.com/locations/global/workforcePools/my-po
	// ol-id/subject/my-subject-attribute-value`.
	Members []string `json:"members,omitempty"`
	// Role: Role that is assigned to the list of `members`, or principals. For
	// example, `roles/viewer`, `roles/editor`, or `roles/owner`. For an overview
	// of the IAM roles and permissions, see the IAM documentation
	// (https://cloud.google.com/iam/docs/roles-overview). For a list of the
	// available pre-defined roles, see here
	// (https://cloud.google.com/iam/docs/understanding-roles).
	Role string `json:"role,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Condition") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Condition") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Binding) MarshalJSON() ([]byte, error) {
	type NoMethod Binding
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CancelOperationRequest: The request message for Operations.CancelOperation.
type CancelOperationRequest struct {
}

// CommitServicePerimetersRequest: A request to commit dry-run specs in all
// Service Perimeters belonging to an Access Policy.
type CommitServicePerimetersRequest struct {
	// Etag: Optional. The etag for the version of the Access Policy that this
	// commit operation is to be performed on. If, at the time of commit, the etag
	// for the Access Policy stored in Access Context Manager is different from the
	// specified etag, then the commit operation will not be performed and the call
	// will fail. This field is not required. If etag is not provided, the
	// operation will be performed as if a valid etag is provided.
	Etag string `json:"etag,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CommitServicePerimetersRequest) MarshalJSON() ([]byte, error) {
	type NoMethod CommitServicePerimetersRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CommitServicePerimetersResponse: A response to
// CommitServicePerimetersRequest. This will be put inside of
// Operation.response field.
type CommitServicePerimetersResponse struct {
	// ServicePerimeters: List of all the Service Perimeter instances in the Access
	// Policy.
	ServicePerimeters []*ServicePerimeter `json:"servicePerimeters,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ServicePerimeters") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ServicePerimeters") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CommitServicePerimetersResponse) MarshalJSON() ([]byte, error) {
	type NoMethod CommitServicePerimetersResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Condition: A condition necessary for an `AccessLevel` to be granted. The
// Condition is an AND over its fields. So a Condition is true if: 1) the
// request IP is from one of the listed subnetworks AND 2) the originating
// device complies with the listed device policy AND 3) all listed access
// levels are granted AND 4) the request was sent at a time allowed by the
// DateTimeRestriction.
type Condition struct {
	// DevicePolicy: Device specific restrictions, all restrictions must hold for
	// the Condition to be true. If not specified, all devices are allowed.
	DevicePolicy *DevicePolicy `json:"devicePolicy,omitempty"`
	// IpSubnetworks: CIDR block IP subnetwork specification. May be IPv4 or IPv6.
	// Note that for a CIDR IP address block, the specified IP address portion must
	// be properly truncated (i.e. all the host bits must be zero) or the input is
	// considered malformed. For example, "192.0.2.0/24" is accepted but
	// "192.0.2.1/24" is not. Similarly, for IPv6, "2001:db8::/32" is accepted
	// whereas "2001:db8::1/32" is not. The originating IP of a request must be in
	// one of the listed subnets in order for this Condition to be true. If empty,
	// all IP addresses are allowed.
	IpSubnetworks []string `json:"ipSubnetworks,omitempty"`
	// Members: The request must be made by one of the provided user or service
	// accounts. Groups are not supported. Syntax: `user:{emailid}`
	// `serviceAccount:{emailid}` If not specified, a request may come from any
	// user.
	Members []string `json:"members,omitempty"`
	// Negate: Whether to negate the Condition. If true, the Condition becomes a
	// NAND over its non-empty fields. Any non-empty field criteria evaluating to
	// false will result in the Condition to be satisfied. Defaults to false.
	Negate bool `json:"negate,omitempty"`
	// Regions: The request must originate from one of the provided
	// countries/regions. Must be valid ISO 3166-1 alpha-2 codes.
	Regions []string `json:"regions,omitempty"`
	// RequiredAccessLevels: A list of other access levels defined in the same
	// `Policy`, referenced by resource name. Referencing an `AccessLevel` which
	// does not exist is an error. All access levels listed must be granted for the
	// Condition to be true. Example:
	// "accessPolicies/MY_POLICY/accessLevels/LEVEL_NAME"
	RequiredAccessLevels []string `json:"requiredAccessLevels,omitempty"`
	// VpcNetworkSources: The request must originate from one of the provided VPC
	// networks in Google Cloud. Cannot specify this field together with
	// `ip_subnetworks`.
	VpcNetworkSources []*VpcNetworkSource `json:"vpcNetworkSources,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DevicePolicy") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DevicePolicy") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Condition) MarshalJSON() ([]byte, error) {
	type NoMethod Condition
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CustomLevel: `CustomLevel` is an `AccessLevel` using the Cloud Common
// Expression Language to represent the necessary conditions for the level to
// apply to a request. See CEL spec at: https://github.com/google/cel-spec
type CustomLevel struct {
	// Expr: Required. A Cloud CEL expression evaluating to a boolean.
	Expr *Expr `json:"expr,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Expr") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Expr") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomLevel) MarshalJSON() ([]byte, error) {
	type NoMethod CustomLevel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DevicePolicy: `DevicePolicy` specifies device specific restrictions
// necessary to acquire a given access level. A `DevicePolicy` specifies
// requirements for requests from devices to be granted access levels, it does
// not do any enforcement on the device. `DevicePolicy` acts as an AND over all
// specified fields, and each repeated field is an OR over its elements. Any
// unset fields are ignored. For example, if the proto is { os_type :
// DESKTOP_WINDOWS, os_type : DESKTOP_LINUX, encryption_status: ENCRYPTED},
// then the DevicePolicy will be true for requests originating from encrypted
// Linux desktops and encrypted Windows desktops.
type DevicePolicy struct {
	// AllowedDeviceManagementLevels: Allowed device management levels, an empty
	// list allows all management levels.
	//
	// Possible values:
	//   "MANAGEMENT_UNSPECIFIED" - The device's management level is not specified
	// or not known.
	//   "NONE" - The device is not managed.
	//   "BASIC" - Basic management is enabled, which is generally limited to
	// monitoring and wiping the corporate account.
	//   "COMPLETE" - Complete device management. This includes more thorough
	// monitoring and the ability to directly manage the device (such as remote
	// wiping). This can be enabled through the Android Enterprise Platform.
	AllowedDeviceManagementLevels []string `json:"allowedDeviceManagementLevels,omitempty"`
	// AllowedEncryptionStatuses: Allowed encryptions statuses, an empty list
	// allows all statuses.
	//
	// Possible values:
	//   "ENCRYPTION_UNSPECIFIED" - The encryption status of the device is not
	// specified or not known.
	//   "ENCRYPTION_UNSUPPORTED" - The device does not support encryption.
	//   "UNENCRYPTED" - The device supports encryption, but is currently
	// unencrypted.
	//   "ENCRYPTED" - The device is encrypted.
	AllowedEncryptionStatuses []string `json:"allowedEncryptionStatuses,omitempty"`
	// OsConstraints: Allowed OS versions, an empty list allows all types and all
	// versions.
	OsConstraints []*OsConstraint `json:"osConstraints,omitempty"`
	// RequireAdminApproval: Whether the device needs to be approved by the
	// customer admin.
	RequireAdminApproval bool `json:"requireAdminApproval,omitempty"`
	// RequireCorpOwned: Whether the device needs to be corp owned.
	RequireCorpOwned bool `json:"requireCorpOwned,omitempty"`
	// RequireScreenlock: Whether or not screenlock is required for the
	// DevicePolicy to be true. Defaults to `false`.
	RequireScreenlock bool `json:"requireScreenlock,omitempty"`
	// ForceSendFields is a list of field names (e.g.
	// "AllowedDeviceManagementLevels") to unconditionally include in API requests.
	// By default, fields with empty or default values are omitted from API
	// requests. See https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields
	// for more details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AllowedDeviceManagementLevels")
	// to include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DevicePolicy) MarshalJSON() ([]byte, error) {
	type NoMethod DevicePolicy
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// EgressFrom: Defines the conditions under which an EgressPolicy matches a
// request. Conditions based on information about the source of the request.
// Note that if the destination of the request is also protected by a
// ServicePerimeter, then that ServicePerimeter must have an IngressPolicy
// which allows access in order for this request to succeed.
type EgressFrom struct {
	// Identities: A list of identities that are allowed access through
	// [EgressPolicy]. Identities can be an individual user, service account,
	// Google group, or third-party identity. The `v1` identities that have the
	// prefix `user`, `group`, `serviceAccount`, `principal`, and `principalSet` in
	// https://cloud.google.com/iam/docs/principal-identifiers#v1 are supported.
	Identities []string `json:"identities,omitempty"`
	// IdentityType: Specifies the type of identities that are allowed access to
	// outside the perimeter. If left unspecified, then members of `identities`
	// field will be allowed access.
	//
	// Possible values:
	//   "IDENTITY_TYPE_UNSPECIFIED" - No blanket identity group specified.
	//   "ANY_IDENTITY" - Authorize access from all identities outside the
	// perimeter.
	//   "ANY_USER_ACCOUNT" - Authorize access from all human users outside the
	// perimeter.
	//   "ANY_SERVICE_ACCOUNT" - Authorize access from all service accounts outside
	// the perimeter.
	IdentityType string `json:"identityType,omitempty"`
	// SourceRestriction: Whether to enforce traffic restrictions based on
	// `sources` field. If the `sources` fields is non-empty, then this field must
	// be set to `SOURCE_RESTRICTION_ENABLED`.
	//
	// Possible values:
	//   "SOURCE_RESTRICTION_UNSPECIFIED" - Enforcement preference unspecified,
	// will not enforce traffic restrictions based on `sources` in EgressFrom.
	//   "SOURCE_RESTRICTION_ENABLED" - Enforcement preference enabled, traffic
	// restrictions will be enforced based on `sources` in EgressFrom.
	//   "SOURCE_RESTRICTION_DISABLED" - Enforcement preference disabled, will not
	// enforce traffic restrictions based on `sources` in EgressFrom.
	SourceRestriction string `json:"sourceRestriction,omitempty"`
	// Sources: Sources that this EgressPolicy authorizes access from. If this
	// field is not empty, then `source_restriction` must be set to
	// `SOURCE_RESTRICTION_ENABLED`.
	Sources []*EgressSource `json:"sources,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Identities") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Identities") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s EgressFrom) MarshalJSON() ([]byte, error) {
	type NoMethod EgressFrom
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// EgressPolicy: Policy for egress from perimeter. EgressPolicies match
// requests based on `egress_from` and `egress_to` stanzas. For an EgressPolicy
// to match, both `egress_from` and `egress_to` stanzas must be matched. If an
// EgressPolicy matches a request, the request is allowed to span the
// ServicePerimeter boundary. For example, an EgressPolicy can be used to allow
// VMs on networks within the ServicePerimeter to access a defined set of
// projects outside the perimeter in certain contexts (e.g. to read data from a
// Cloud Storage bucket or query against a BigQuery dataset). EgressPolicies
// are concerned with the *resources* that a request relates as well as the API
// services and API actions being used. They do not related to the direction of
// data movement. More detailed documentation for this concept can be found in
// the descriptions of EgressFrom and EgressTo.
type EgressPolicy struct {
	// EgressFrom: Defines conditions on the source of a request causing this
	// EgressPolicy to apply.
	EgressFrom *EgressFrom `json:"egressFrom,omitempty"`
	// EgressTo: Defines the conditions on the ApiOperation and destination
	// resources that cause this EgressPolicy to apply.
	EgressTo *EgressTo `json:"egressTo,omitempty"`
	// ForceSendFields is a list of field names (e.g. "EgressFrom") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EgressFrom") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s EgressPolicy) MarshalJSON() ([]byte, error) {
	type NoMethod EgressPolicy
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// EgressSource: The source that EgressPolicy authorizes access from inside the
// ServicePerimeter to somewhere outside the ServicePerimeter boundaries.
type EgressSource struct {
	// AccessLevel: An AccessLevel resource name that allows protected resources
	// inside the ServicePerimeters to access outside the ServicePerimeter
	// boundaries. AccessLevels listed must be in the same policy as this
	// ServicePerimeter. Referencing a nonexistent AccessLevel will cause an error.
	// If an AccessLevel name is not specified, only resources within the perimeter
	// can be accessed through Google Cloud calls with request origins within the
	// perimeter. Example: `accessPolicies/MY_POLICY/accessLevels/MY_LEVEL`. If a
	// single `*` is specified for `access_level`, then all EgressSources will be
	// allowed.
	AccessLevel string `json:"accessLevel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccessLevel") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessLevel") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s EgressSource) MarshalJSON() ([]byte, error) {
	type NoMethod EgressSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// EgressTo: Defines the conditions under which an EgressPolicy matches a
// request. Conditions are based on information about the ApiOperation intended
// to be performed on the `resources` specified. Note that if the destination
// of the request is also protected by a ServicePerimeter, then that
// ServicePerimeter must have an IngressPolicy which allows access in order for
// this request to succeed. The request must match `operations` AND `resources`
// fields in order to be allowed egress out of the perimeter.
type EgressTo struct {
	// ExternalResources: A list of external resources that are allowed to be
	// accessed. Only AWS and Azure resources are supported. For Amazon S3, the
	// supported formats are s3://BUCKET_NAME, s3a://BUCKET_NAME, and
	// s3n://BUCKET_NAME. For Azure Storage, the supported format is
	// azure://myaccount.blob.core.windows.net/CONTAINER_NAME. A request matches if
	// it contains an external resource in this list (Example: s3://bucket/path).
	// Currently '*' is not allowed.
	ExternalResources []string `json:"externalResources,omitempty"`
	// Operations: A list of ApiOperations allowed to be performed by the sources
	// specified in the corresponding EgressFrom. A request matches if it uses an
	// operation/service in this list.
	Operations []*ApiOperation `json:"operations,omitempty"`
	// Resources: A list of resources, currently only projects in the form
	// `projects/`, that are allowed to be accessed by sources defined in the
	// corresponding EgressFrom. A request matches if it contains a resource in
	// this list. If `*` is specified for `resources`, then this EgressTo rule will
	// authorize access to all resources outside the perimeter.
	Resources []string `json:"resources,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ExternalResources") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ExternalResources") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s EgressTo) MarshalJSON() ([]byte, error) {
	type NoMethod EgressTo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// Expr: Represents a textual expression in the Common Expression Language
// (CEL) syntax. CEL is a C-like expression language. The syntax and semantics
// of CEL are documented at https://github.com/google/cel-spec. Example
// (Comparison): title: "Summary size limit" description: "Determines if a
// summary is less than 100 chars" expression: "document.summary.size() < 100"
// Example (Equality): title: "Requestor is owner" description: "Determines if
// requestor is the document owner" expression: "document.owner ==
// request.auth.claims.email" Example (Logic): title: "Public documents"
// description: "Determine whether the document should be publicly visible"
// expression: "document.type != 'private' && document.type != 'internal'"
// Example (Data Manipulation): title: "Notification string" description:
// "Create a notification string with a timestamp." expression: "'New message
// received at ' + string(document.create_time)" The exact variables and
// functions that may be referenced within an expression are determined by the
// service that evaluates it. See the service documentation for additional
// information.
type Expr struct {
	// Description: Optional. Description of the expression. This is a longer text
	// which describes the expression, e.g. when hovered over it in a UI.
	Description string `json:"description,omitempty"`
	// Expression: Textual representation of an expression in Common Expression
	// Language syntax.
	Expression string `json:"expression,omitempty"`
	// Location: Optional. String indicating the location of the expression for
	// error reporting, e.g. a file name and a position in the file.
	Location string `json:"location,omitempty"`
	// Title: Optional. Title for the expression, i.e. a short string describing
	// its purpose. This can be used e.g. in UIs which allow to enter the
	// expression.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Description") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Description") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Expr) MarshalJSON() ([]byte, error) {
	type NoMethod Expr
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GcpUserAccessBinding: Restricts access to Cloud Console and Google Cloud
// APIs for a set of users using Context-Aware Access.
type GcpUserAccessBinding struct {
	// AccessLevels: Optional. Access level that a user must have to be granted
	// access. Only one access level is supported, not multiple. This repeated
	// field must have exactly one element. Example:
	// "accessPolicies/9522/accessLevels/device_trusted"
	AccessLevels []string `json:"accessLevels,omitempty"`
	// DryRunAccessLevels: Optional. Dry run access level that will be evaluated
	// but will not be enforced. The access denial based on dry run policy will be
	// logged. Only one access level is supported, not multiple. This list must
	// have exactly one element. Example:
	// "accessPolicies/9522/accessLevels/device_trusted"
	DryRunAccessLevels []string `json:"dryRunAccessLevels,omitempty"`
	// GroupKey: Required. Immutable. Google Group id whose members are subject to
	// this binding's restrictions. See "id" in the [G Suite Directory API's Groups
	// resource]
	// (https://developers.google.com/admin-sdk/directory/v1/reference/groups#resource).
	// If a group's email address/alias is changed, this resource will continue to
	// point at the changed group. This field does not accept group email addresses
	// or aliases. Example: "01d520gv4vjcrht"
	GroupKey string `json:"groupKey,omitempty"`
	// Name: Immutable. Assigned by the server during creation. The last segment
	// has an arbitrary length and has only URI unreserved characters (as defined
	// by RFC 3986 Section 2.3 (https://tools.ietf.org/html/rfc3986#section-2.3)).
	// Should not be specified by the client during creation. Example:
	// "organizations/256/gcpUserAccessBindings/b3-BhcX_Ud5N"
	Name string `json:"name,omitempty"`
	// RestrictedClientApplications: Optional. A list of applications that are
	// subject to this binding's restrictions. If the list is empty, the binding
	// restrictions will universally apply to all applications.
	RestrictedClientApplications []*Application `json:"restrictedClientApplications,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccessLevels") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessLevels") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GcpUserAccessBinding) MarshalJSON() ([]byte, error) {
	type NoMethod GcpUserAccessBinding
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GcpUserAccessBindingOperationMetadata: Metadata of GCP Access Binding Long
// Running Operations.
type GcpUserAccessBindingOperationMetadata struct {
}

// GetIamPolicyRequest: Request message for `GetIamPolicy` method.
type GetIamPolicyRequest struct {
	// Options: OPTIONAL: A `GetPolicyOptions` object for specifying options to
	// `GetIamPolicy`.
	Options *GetPolicyOptions `json:"options,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Options") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Options") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GetIamPolicyRequest) MarshalJSON() ([]byte, error) {
	type NoMethod GetIamPolicyRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GetPolicyOptions: Encapsulates settings provided to GetIamPolicy.
type GetPolicyOptions struct {
	// RequestedPolicyVersion: Optional. The maximum policy version that will be
	// used to format the policy. Valid values are 0, 1, and 3. Requests specifying
	// an invalid value will be rejected. Requests for policies with any
	// conditional role bindings must specify version 3. Policies with no
	// conditional role bindings may specify any valid value or leave the field
	// unset. The policy in the response might use the policy version that you
	// specified, or it might use a lower policy version. For example, if you
	// specify version 3, but the policy has no conditional role bindings, the
	// response uses version 1. To learn which resources support conditions in
	// their IAM policies, see the IAM documentation
	// (https://cloud.google.com/iam/help/conditions/resource-policies).
	RequestedPolicyVersion int64 `json:"requestedPolicyVersion,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RequestedPolicyVersion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RequestedPolicyVersion") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GetPolicyOptions) MarshalJSON() ([]byte, error) {
	type NoMethod GetPolicyOptions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngressFrom: Defines the conditions under which an IngressPolicy matches a
// request. Conditions are based on information about the source of the
// request. The request must satisfy what is defined in `sources` AND identity
// related fields in order to match.
type IngressFrom struct {
	// Identities: A list of identities that are allowed access through
	// [IngressPolicy]. Identities can be an individual user, service account,
	// Google group, or third-party identity. The `v1` identities that have the
	// prefix `user`, `group`, `serviceAccount`, `principal`, and `principalSet` in
	// https://cloud.google.com/iam/docs/principal-identifiers#v1 are supported.
	Identities []string `json:"identities,omitempty"`
	// IdentityType: Specifies the type of identities that are allowed access from
	// outside the perimeter. If left unspecified, then members of `identities`
	// field will be allowed access.
	//
	// Possible values:
	//   "IDENTITY_TYPE_UNSPECIFIED" - No blanket identity group specified.
	//   "ANY_IDENTITY" - Authorize access from all identities outside the
	// perimeter.
	//   "ANY_USER_ACCOUNT" - Authorize access from all human users outside the
	// perimeter.
	//   "ANY_SERVICE_ACCOUNT" - Authorize access from all service accounts outside
	// the perimeter.
	IdentityType string `json:"identityType,omitempty"`
	// Sources: Sources that this IngressPolicy authorizes access from.
	Sources []*IngressSource `json:"sources,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Identities") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Identities") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngressFrom) MarshalJSON() ([]byte, error) {
	type NoMethod IngressFrom
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngressPolicy: Policy for ingress into ServicePerimeter. IngressPolicies
// match requests based on `ingress_from` and `ingress_to` stanzas. For an
// ingress policy to match, both the `ingress_from` and `ingress_to` stanzas
// must be matched. If an IngressPolicy matches a request, the request is
// allowed through the perimeter boundary from outside the perimeter. For
// example, access from the internet can be allowed either based on an
// AccessLevel or, for traffic hosted on Google Cloud, the project of the
// source network. For access from private networks, using the project of the
// hosting network is required. Individual ingress policies can be limited by
// restricting which services and/or actions they match using the `ingress_to`
// field.
type IngressPolicy struct {
	// IngressFrom: Defines the conditions on the source of a request causing this
	// IngressPolicy to apply.
	IngressFrom *IngressFrom `json:"ingressFrom,omitempty"`
	// IngressTo: Defines the conditions on the ApiOperation and request
	// destination that cause this IngressPolicy to apply.
	IngressTo *IngressTo `json:"ingressTo,omitempty"`
	// ForceSendFields is a list of field names (e.g. "IngressFrom") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IngressFrom") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngressPolicy) MarshalJSON() ([]byte, error) {
	type NoMethod IngressPolicy
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngressSource: The source that IngressPolicy authorizes access from.
type IngressSource struct {
	// AccessLevel: An AccessLevel resource name that allow resources within the
	// ServicePerimeters to be accessed from the internet. AccessLevels listed must
	// be in the same policy as this ServicePerimeter. Referencing a nonexistent
	// AccessLevel will cause an error. If no AccessLevel names are listed,
	// resources within the perimeter can only be accessed via Google Cloud calls
	// with request origins within the perimeter. Example:
	// `accessPolicies/MY_POLICY/accessLevels/MY_LEVEL`. If a single `*` is
	// specified for `access_level`, then all IngressSources will be allowed.
	AccessLevel string `json:"accessLevel,omitempty"`
	// Resource: A Google Cloud resource that is allowed to ingress the perimeter.
	// Requests from these resources will be allowed to access perimeter data.
	// Currently only projects and VPCs are allowed. Project format:
	// `projects/{project_number}` VPC network format:
	// `//compute.googleapis.com/projects/{PROJECT_ID}/global/networks/{NAME}`. The
	// project may be in any Google Cloud organization, not just the organization
	// that the perimeter is defined in. `*` is not allowed, the case of allowing
	// all Google Cloud resources only is not supported.
	Resource string `json:"resource,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccessLevel") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessLevel") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngressSource) MarshalJSON() ([]byte, error) {
	type NoMethod IngressSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngressTo: Defines the conditions under which an IngressPolicy matches a
// request. Conditions are based on information about the ApiOperation intended
// to be performed on the target resource of the request. The request must
// satisfy what is defined in `operations` AND `resources` in order to match.
type IngressTo struct {
	// Operations: A list of ApiOperations allowed to be performed by the sources
	// specified in corresponding IngressFrom in this ServicePerimeter.
	Operations []*ApiOperation `json:"operations,omitempty"`
	// Resources: A list of resources, currently only projects in the form
	// `projects/`, protected by this ServicePerimeter that are allowed to be
	// accessed by sources defined in the corresponding IngressFrom. If a single
	// `*` is specified, then access to all resources inside the perimeter are
	// allowed.
	Resources []string `json:"resources,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Operations") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Operations") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngressTo) MarshalJSON() ([]byte, error) {
	type NoMethod IngressTo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAccessLevelsResponse: A response to `ListAccessLevelsRequest`.
type ListAccessLevelsResponse struct {
	// AccessLevels: List of the Access Level instances.
	AccessLevels []*AccessLevel `json:"accessLevels,omitempty"`
	// NextPageToken: The pagination token to retrieve the next page of results. If
	// the value is empty, no further results remain.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccessLevels") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessLevels") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAccessLevelsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAccessLevelsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAccessPoliciesResponse: A response to `ListAccessPoliciesRequest`.
type ListAccessPoliciesResponse struct {
	// AccessPolicies: List of the AccessPolicy instances.
	AccessPolicies []*AccessPolicy `json:"accessPolicies,omitempty"`
	// NextPageToken: The pagination token to retrieve the next page of results. If
	// the value is empty, no further results remain.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccessPolicies") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessPolicies") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAccessPoliciesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAccessPoliciesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAuthorizedOrgsDescsResponse: A response to
// `ListAuthorizedOrgsDescsRequest`.
type ListAuthorizedOrgsDescsResponse struct {
	// AuthorizedOrgsDescs: List of all the Authorized Orgs Desc instances.
	AuthorizedOrgsDescs []*AuthorizedOrgsDesc `json:"authorizedOrgsDescs,omitempty"`
	// NextPageToken: The pagination token to retrieve the next page of results. If
	// the value is empty, no further results remain.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AuthorizedOrgsDescs") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AuthorizedOrgsDescs") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAuthorizedOrgsDescsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAuthorizedOrgsDescsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListGcpUserAccessBindingsResponse: Response of ListGcpUserAccessBindings.
type ListGcpUserAccessBindingsResponse struct {
	// GcpUserAccessBindings: GcpUserAccessBinding
	GcpUserAccessBindings []*GcpUserAccessBinding `json:"gcpUserAccessBindings,omitempty"`
	// NextPageToken: Token to get the next page of items. If blank, there are no
	// more items.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "GcpUserAccessBindings") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "GcpUserAccessBindings") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListGcpUserAccessBindingsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListGcpUserAccessBindingsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListOperationsResponse: The response message for Operations.ListOperations.
type ListOperationsResponse struct {
	// NextPageToken: The standard List next-page token.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Operations: A list of operations that matches the specified filter in the
	// request.
	Operations []*Operation `json:"operations,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListOperationsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListOperationsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListServicePerimetersResponse: A response to `ListServicePerimetersRequest`.
type ListServicePerimetersResponse struct {
	// NextPageToken: The pagination token to retrieve the next page of results. If
	// the value is empty, no further results remain.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// ServicePerimeters: List of the Service Perimeter instances.
	ServicePerimeters []*ServicePerimeter `json:"servicePerimeters,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListServicePerimetersResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListServicePerimetersResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListSupportedServicesResponse: A response to `ListSupportedServicesRequest`.
type ListSupportedServicesResponse struct {
	// NextPageToken: The pagination token to retrieve the next page of results. If
	// the value is empty, no further results remain.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// SupportedServices: List of services supported by VPC Service Controls
	// instances.
	SupportedServices []*SupportedService `json:"supportedServices,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListSupportedServicesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListSupportedServicesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MethodSelector: An allowed method or permission of a service specified in
// ApiOperation.
type MethodSelector struct {
	// Method: A valid method name for the corresponding `service_name` in
	// ApiOperation. If `*` is used as the value for the `method`, then ALL methods
	// and permissions are allowed.
	Method string `json:"method,omitempty"`
	// Permission: A valid Cloud IAM permission for the corresponding
	// `service_name` in ApiOperation.
	Permission string `json:"permission,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Method") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Method") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MethodSelector) MarshalJSON() ([]byte, error) {
	type NoMethod MethodSelector
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Operation: This resource represents a long-running operation that is the
// result of a network API call.
type Operation struct {
	// Done: If the value is `false`, it means the operation is still in progress.
	// If `true`, the operation is completed, and either `error` or `response` is
	// available.
	Done bool `json:"done,omitempty"`
	// Error: The error result of the operation in case of failure or cancellation.
	Error *Status `json:"error,omitempty"`
	// Metadata: Service-specific metadata associated with the operation. It
	// typically contains progress information and common metadata such as create
	// time. Some services might not provide such metadata. Any method that returns
	// a long-running operation should document the metadata type, if any.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// Name: The server-assigned name, which is only unique within the same service
	// that originally returns it. If you use the default HTTP mapping, the `name`
	// should be a resource name ending with `operations/{unique_id}`.
	Name string `json:"name,omitempty"`
	// Response: The normal, successful response of the operation. If the original
	// method returns no data on success, such as `Delete`, the response is
	// `google.protobuf.Empty`. If the original method is standard
	// `Get`/`Create`/`Update`, the response should be the resource. For other
	// methods, the response should have the type `XxxResponse`, where `Xxx` is the
	// original method name. For example, if the original method name is
	// `TakeSnapshot()`, the inferred response type is `TakeSnapshotResponse`.
	Response googleapi.RawMessage `json:"response,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Done") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Done") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Operation) MarshalJSON() ([]byte, error) {
	type NoMethod Operation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// OsConstraint: A restriction on the OS type and version of devices making
// requests.
type OsConstraint struct {
	// MinimumVersion: The minimum allowed OS version. If not set, any version of
	// this OS satisfies the constraint. Format: "major.minor.patch". Examples:
	// "10.5.301", "9.2.1".
	MinimumVersion string `json:"minimumVersion,omitempty"`
	// OsType: Required. The allowed OS type.
	//
	// Possible values:
	//   "OS_UNSPECIFIED" - The operating system of the device is not specified or
	// not known.
	//   "DESKTOP_MAC" - A desktop Mac operating system.
	//   "DESKTOP_WINDOWS" - A desktop Windows operating system.
	//   "DESKTOP_LINUX" - A desktop Linux operating system.
	//   "DESKTOP_CHROME_OS" - A desktop ChromeOS operating system.
	//   "ANDROID" - An Android operating system.
	//   "IOS" - An iOS operating system.
	OsType string `json:"osType,omitempty"`
	// RequireVerifiedChromeOs: Only allows requests from devices with a verified
	// Chrome OS. Verifications includes requirements that the device is
	// enterprise-managed, conformant to domain policies, and the caller has
	// permission to call the API targeted by the request.
	RequireVerifiedChromeOs bool `json:"requireVerifiedChromeOs,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MinimumVersion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MinimumVersion") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s OsConstraint) MarshalJSON() ([]byte, error) {
	type NoMethod OsConstraint
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Policy: An Identity and Access Management (IAM) policy, which specifies
// access controls for Google Cloud resources. A `Policy` is a collection of
// `bindings`. A `binding` binds one or more `members`, or principals, to a
// single `role`. Principals can be user accounts, service accounts, Google
// groups, and domains (such as G Suite). A `role` is a named list of
// permissions; each `role` can be an IAM predefined role or a user-created
// custom role. For some types of Google Cloud resources, a `binding` can also
// specify a `condition`, which is a logical expression that allows access to a
// resource only if the expression evaluates to `true`. A condition can add
// constraints based on attributes of the request, the resource, or both. To
// learn which resources support conditions in their IAM policies, see the IAM
// documentation
// (https://cloud.google.com/iam/help/conditions/resource-policies). **JSON
// example:** ``` { "bindings": [ { "role":
// "roles/resourcemanager.organizationAdmin", "members": [
// "user:mike@example.com", "group:admins@example.com", "domain:google.com",
// "serviceAccount:my-project-id@appspot.gserviceaccount.com" ] }, { "role":
// "roles/resourcemanager.organizationViewer", "members": [
// "user:eve@example.com" ], "condition": { "title": "expirable access",
// "description": "Does not grant access after Sep 2020", "expression":
// "request.time < timestamp('2020-10-01T00:00:00.000Z')", } } ], "etag":
// "BwWWja0YfJA=", "version": 3 } ``` **YAML example:** ``` bindings: -
// members: - user:mike@example.com - group:admins@example.com -
// domain:google.com - serviceAccount:my-project-id@appspot.gserviceaccount.com
// role: roles/resourcemanager.organizationAdmin - members: -
// user:eve@example.com role: roles/resourcemanager.organizationViewer
// condition: title: expirable access description: Does not grant access after
// Sep 2020 expression: request.time < timestamp('2020-10-01T00:00:00.000Z')
// etag: BwWWja0YfJA= version: 3 ``` For a description of IAM and its features,
// see the IAM documentation (https://cloud.google.com/iam/docs/).
type Policy struct {
	// AuditConfigs: Specifies cloud audit logging configuration for this policy.
	AuditConfigs []*AuditConfig `json:"auditConfigs,omitempty"`
	// Bindings: Associates a list of `members`, or principals, with a `role`.
	// Optionally, may specify a `condition` that determines how and when the
	// `bindings` are applied. Each of the `bindings` must contain at least one
	// principal. The `bindings` in a `Policy` can refer to up to 1,500 principals;
	// up to 250 of these principals can be Google groups. Each occurrence of a
	// principal counts towards these limits. For example, if the `bindings` grant
	// 50 different roles to `user:alice@example.com`, and not to any other
	// principal, then you can add another 1,450 principals to the `bindings` in
	// the `Policy`.
	Bindings []*Binding `json:"bindings,omitempty"`
	// Etag: `etag` is used for optimistic concurrency control as a way to help
	// prevent simultaneous updates of a policy from overwriting each other. It is
	// strongly suggested that systems make use of the `etag` in the
	// read-modify-write cycle to perform policy updates in order to avoid race
	// conditions: An `etag` is returned in the response to `getIamPolicy`, and
	// systems are expected to put that etag in the request to `setIamPolicy` to
	// ensure that their change will be applied to the same version of the policy.
	// **Important:** If you use IAM Conditions, you must include the `etag` field
	// whenever you call `setIamPolicy`. If you omit this field, then IAM allows
	// you to overwrite a version `3` policy with a version `1` policy, and all of
	// the conditions in the version `3` policy are lost.
	Etag string `json:"etag,omitempty"`
	// Version: Specifies the format of the policy. Valid values are `0`, `1`, and
	// `3`. Requests that specify an invalid value are rejected. Any operation that
	// affects conditional role bindings must specify version `3`. This requirement
	// applies to the following operations: * Getting a policy that includes a
	// conditional role binding * Adding a conditional role binding to a policy *
	// Changing a conditional role binding in a policy * Removing any role binding,
	// with or without a condition, from a policy that includes conditions
	// **Important:** If you use IAM Conditions, you must include the `etag` field
	// whenever you call `setIamPolicy`. If you omit this field, then IAM allows
	// you to overwrite a version `3` policy with a version `1` policy, and all of
	// the conditions in the version `3` policy are lost. If a policy does not
	// include any conditions, operations on that policy may specify any valid
	// version or leave the field unset. To learn which resources support
	// conditions in their IAM policies, see the IAM documentation
	// (https://cloud.google.com/iam/help/conditions/resource-policies).
	Version int64 `json:"version,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AuditConfigs") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AuditConfigs") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Policy) MarshalJSON() ([]byte, error) {
	type NoMethod Policy
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReplaceAccessLevelsRequest: A request to replace all existing Access Levels
// in an Access Policy with the Access Levels provided. This is done
// atomically.
type ReplaceAccessLevelsRequest struct {
	// AccessLevels: Required. The desired Access Levels that should replace all
	// existing Access Levels in the Access Policy.
	AccessLevels []*AccessLevel `json:"accessLevels,omitempty"`
	// Etag: Optional. The etag for the version of the Access Policy that this
	// replace operation is to be performed on. If, at the time of replace, the
	// etag for the Access Policy stored in Access Context Manager is different
	// from the specified etag, then the replace operation will not be performed
	// and the call will fail. This field is not required. If etag is not provided,
	// the operation will be performed as if a valid etag is provided.
	Etag string `json:"etag,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccessLevels") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessLevels") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReplaceAccessLevelsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod ReplaceAccessLevelsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReplaceAccessLevelsResponse: A response to ReplaceAccessLevelsRequest. This
// will be put inside of Operation.response field.
type ReplaceAccessLevelsResponse struct {
	// AccessLevels: List of the Access Level instances.
	AccessLevels []*AccessLevel `json:"accessLevels,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccessLevels") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessLevels") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReplaceAccessLevelsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ReplaceAccessLevelsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReplaceServicePerimetersRequest: A request to replace all existing Service
// Perimeters in an Access Policy with the Service Perimeters provided. This is
// done atomically.
type ReplaceServicePerimetersRequest struct {
	// Etag: Optional. The etag for the version of the Access Policy that this
	// replace operation is to be performed on. If, at the time of replace, the
	// etag for the Access Policy stored in Access Context Manager is different
	// from the specified etag, then the replace operation will not be performed
	// and the call will fail. This field is not required. If etag is not provided,
	// the operation will be performed as if a valid etag is provided.
	Etag string `json:"etag,omitempty"`
	// ServicePerimeters: Required. The desired Service Perimeters that should
	// replace all existing Service Perimeters in the Access Policy.
	ServicePerimeters []*ServicePerimeter `json:"servicePerimeters,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReplaceServicePerimetersRequest) MarshalJSON() ([]byte, error) {
	type NoMethod ReplaceServicePerimetersRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReplaceServicePerimetersResponse: A response to
// ReplaceServicePerimetersRequest. This will be put inside of
// Operation.response field.
type ReplaceServicePerimetersResponse struct {
	// ServicePerimeters: List of the Service Perimeter instances.
	ServicePerimeters []*ServicePerimeter `json:"servicePerimeters,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ServicePerimeters") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ServicePerimeters") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReplaceServicePerimetersResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ReplaceServicePerimetersResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ServicePerimeter: `ServicePerimeter` describes a set of Google Cloud
// resources which can freely import and export data amongst themselves, but
// not export outside of the `ServicePerimeter`. If a request with a source
// within this `ServicePerimeter` has a target outside of the
// `ServicePerimeter`, the request will be blocked. Otherwise the request is
// allowed. There are two types of Service Perimeter - Regular and Bridge.
// Regular Service Perimeters cannot overlap, a single Google Cloud project or
// VPC network can only belong to a single regular Service Perimeter. Service
// Perimeter Bridges can contain only Google Cloud projects as members, a
// single Google Cloud project may belong to multiple Service Perimeter
// Bridges.
type ServicePerimeter struct {
	// Description: Description of the `ServicePerimeter` and its use. Does not
	// affect behavior.
	Description string `json:"description,omitempty"`
	// Name: Identifier. Resource name for the `ServicePerimeter`. Format:
	// `accessPolicies/{access_policy}/servicePerimeters/{service_perimeter}`. The
	// `service_perimeter` component must begin with a letter, followed by
	// alphanumeric characters or `_`. After you create a `ServicePerimeter`, you
	// cannot change its `name`.
	Name string `json:"name,omitempty"`
	// PerimeterType: Perimeter type indicator. A single project or VPC network is
	// allowed to be a member of single regular perimeter, but multiple service
	// perimeter bridges. A project cannot be a included in a perimeter bridge
	// without being included in regular perimeter. For perimeter bridges, the
	// restricted service list as well as access level lists must be empty.
	//
	// Possible values:
	//   "PERIMETER_TYPE_REGULAR" - Regular Perimeter. When no value is specified,
	// the perimeter uses this type.
	//   "PERIMETER_TYPE_BRIDGE" - Perimeter Bridge.
	PerimeterType string `json:"perimeterType,omitempty"`
	// Spec: Proposed (or dry run) ServicePerimeter configuration. This
	// configuration allows to specify and test ServicePerimeter configuration
	// without enforcing actual access restrictions. Only allowed to be set when
	// the "use_explicit_dry_run_spec" flag is set.
	Spec *ServicePerimeterConfig `json:"spec,omitempty"`
	// Status: Current ServicePerimeter configuration. Specifies sets of resources,
	// restricted services and access levels that determine perimeter content and
	// boundaries.
	Status *ServicePerimeterConfig `json:"status,omitempty"`
	// Title: Human readable title. Must be unique within the Policy.
	Title string `json:"title,omitempty"`
	// UseExplicitDryRunSpec: Use explicit dry run spec flag. Ordinarily, a dry-run
	// spec implicitly exists for all Service Perimeters, and that spec is
	// identical to the status for those Service Perimeters. When this flag is set,
	// it inhibits the generation of the implicit spec, thereby allowing the user
	// to explicitly provide a configuration ("spec") to use in a dry-run version
	// of the Service Perimeter. This allows the user to test changes to the
	// enforced config ("status") without actually enforcing them. This testing is
	// done through analyzing the differences between currently enforced and
	// suggested restrictions. use_explicit_dry_run_spec must bet set to True if
	// any of the fields in the spec are set to non-default values.
	UseExplicitDryRunSpec bool `json:"useExplicitDryRunSpec,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Description") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Description") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ServicePerimeter) MarshalJSON() ([]byte, error) {
	type NoMethod ServicePerimeter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ServicePerimeterConfig: `ServicePerimeterConfig` specifies a set of Google
// Cloud resources that describe specific Service Perimeter configuration.
type ServicePerimeterConfig struct {
	// AccessLevels: A list of `AccessLevel` resource names that allow resources
	// within the `ServicePerimeter` to be accessed from the internet.
	// `AccessLevels` listed must be in the same policy as this `ServicePerimeter`.
	// Referencing a nonexistent `AccessLevel` is a syntax error. If no
	// `AccessLevel` names are listed, resources within the perimeter can only be
	// accessed via Google Cloud calls with request origins within the perimeter.
	// Example: "accessPolicies/MY_POLICY/accessLevels/MY_LEVEL". For Service
	// Perimeter Bridge, must be empty.
	AccessLevels []string `json:"accessLevels,omitempty"`
	// EgressPolicies: List of EgressPolicies to apply to the perimeter. A
	// perimeter may have multiple EgressPolicies, each of which is evaluated
	// separately. Access is granted if any EgressPolicy grants it. Must be empty
	// for a perimeter bridge.
	EgressPolicies []*EgressPolicy `json:"egressPolicies,omitempty"`
	// IngressPolicies: List of IngressPolicies to apply to the perimeter. A
	// perimeter may have multiple IngressPolicies, each of which is evaluated
	// separately. Access is granted if any Ingress Policy grants it. Must be empty
	// for a perimeter bridge.
	IngressPolicies []*IngressPolicy `json:"ingressPolicies,omitempty"`
	// Resources: A list of Google Cloud resources that are inside of the service
	// perimeter. Currently only projects and VPCs are allowed. Project format:
	// `projects/{project_number}` VPC network format:
	// `//compute.googleapis.com/projects/{PROJECT_ID}/global/networks/{NAME}`.
	Resources []string `json:"resources,omitempty"`
	// RestrictedServices: Google Cloud services that are subject to the Service
	// Perimeter restrictions. For example, if `storage.googleapis.com` is
	// specified, access to the storage buckets inside the perimeter must meet the
	// perimeter's access restrictions.
	RestrictedServices []string `json:"restrictedServices,omitempty"`
	// VpcAccessibleServices: Configuration for APIs allowed within Perimeter.
	VpcAccessibleServices *VpcAccessibleServices `json:"vpcAccessibleServices,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccessLevels") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessLevels") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ServicePerimeterConfig) MarshalJSON() ([]byte, error) {
	type NoMethod ServicePerimeterConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SetIamPolicyRequest: Request message for `SetIamPolicy` method.
type SetIamPolicyRequest struct {
	// Policy: REQUIRED: The complete policy to be applied to the `resource`. The
	// size of the policy is limited to a few 10s of KB. An empty policy is a valid
	// policy but certain Google Cloud services (such as Projects) might reject
	// them.
	Policy *Policy `json:"policy,omitempty"`
	// UpdateMask: OPTIONAL: A FieldMask specifying which fields of the policy to
	// modify. Only the fields in the mask will be modified. If no mask is
	// provided, the following default mask is used: `paths: "bindings, etag"
	UpdateMask string `json:"updateMask,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Policy") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Policy") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SetIamPolicyRequest) MarshalJSON() ([]byte, error) {
	type NoMethod SetIamPolicyRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Status: The `Status` type defines a logical error model that is suitable for
// different programming environments, including REST APIs and RPC APIs. It is
// used by gRPC (https://github.com/grpc). Each `Status` message contains three
// pieces of data: error code, error message, and error details. You can find
// out more about this error model and how to work with it in the API Design
// Guide (https://cloud.google.com/apis/design/errors).
type Status struct {
	// Code: The status code, which should be an enum value of google.rpc.Code.
	Code int64 `json:"code,omitempty"`
	// Details: A list of messages that carry the error details. There is a common
	// set of message types for APIs to use.
	Details []googleapi.RawMessage `json:"details,omitempty"`
	// Message: A developer-facing error message, which should be in English. Any
	// user-facing error message should be localized and sent in the
	// google.rpc.Status.details field, or localized by the client.
	Message string `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Code") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Code") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Status) MarshalJSON() ([]byte, error) {
	type NoMethod Status
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SupportedService: `SupportedService` specifies the VPC Service Controls and
// its properties.
type SupportedService struct {
	// AvailableOnRestrictedVip: True if the service is available on the restricted
	// VIP. Services on the restricted VIP typically either support VPC Service
	// Controls or are core infrastructure services required for the functioning of
	// Google Cloud.
	AvailableOnRestrictedVip bool `json:"availableOnRestrictedVip,omitempty"`
	// KnownLimitations: True if the service is supported with some limitations.
	// Check documentation
	// (https://cloud.google.com/vpc-service-controls/docs/supported-products) for
	// details.
	KnownLimitations bool `json:"knownLimitations,omitempty"`
	// Name: The service name or address of the supported service, such as
	// `service.googleapis.com`.
	Name string `json:"name,omitempty"`
	// ServiceSupportStage: The support stage of the service.
	//
	// Possible values:
	//   "SERVICE_SUPPORT_STAGE_UNSPECIFIED" - Do not use this default value.
	//   "GA" - GA features are open to all developers and are considered stable
	// and fully qualified for production use.
	//   "PREVIEW" - PREVIEW indicates a pre-release stage where the product is
	// functionally complete but undergoing real-world testing.
	//   "DEPRECATED" - Deprecated features are scheduled to be shut down and
	// removed.
	ServiceSupportStage string `json:"serviceSupportStage,omitempty"`
	// SupportStage: The support stage of the service.
	//
	// Possible values:
	//   "LAUNCH_STAGE_UNSPECIFIED" - Do not use this default value.
	//   "UNIMPLEMENTED" - The feature is not yet implemented. Users can not use
	// it.
	//   "PRELAUNCH" - Prelaunch features are hidden from users and are only
	// visible internally.
	//   "EARLY_ACCESS" - Early Access features are limited to a closed group of
	// testers. To use these features, you must sign up in advance and sign a
	// Trusted Tester agreement (which includes confidentiality provisions). These
	// features may be unstable, changed in backward-incompatible ways, and are not
	// guaranteed to be released.
	//   "ALPHA" - Alpha is a limited availability test for releases before they
	// are cleared for widespread use. By Alpha, all significant design issues are
	// resolved and we are in the process of verifying functionality. Alpha
	// customers need to apply for access, agree to applicable terms, and have
	// their projects allowlisted. Alpha releases don't have to be feature
	// complete, no SLAs are provided, and there are no technical support
	// obligations, but they will be far enough along that customers can actually
	// use them in test environments or for limited-use tests -- just like they
	// would in normal production cases.
	//   "BETA" - Beta is the point at which we are ready to open a release for any
	// customer to use. There are no SLA or technical support obligations in a Beta
	// release. Products will be complete from a feature perspective, but may have
	// some open outstanding issues. Beta releases are suitable for limited
	// production use cases.
	//   "GA" - GA features are open to all developers and are considered stable
	// and fully qualified for production use.
	//   "DEPRECATED" - Deprecated features are scheduled to be shut down and
	// removed. For more information, see the "Deprecation Policy" section of our
	// [Terms of Service](https://cloud.google.com/terms/) and the [Google Cloud
	// Platform Subject to the Deprecation
	// Policy](https://cloud.google.com/terms/deprecation) documentation.
	SupportStage string `json:"supportStage,omitempty"`
	// SupportedMethods: The list of the supported methods. This field exists only
	// in response to GetSupportedService
	SupportedMethods []*MethodSelector `json:"supportedMethods,omitempty"`
	// Title: The name of the supported product, such as 'Cloud Product API'.
	Title string `json:"title,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AvailableOnRestrictedVip")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AvailableOnRestrictedVip") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SupportedService) MarshalJSON() ([]byte, error) {
	type NoMethod SupportedService
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TestIamPermissionsRequest: Request message for `TestIamPermissions` method.
type TestIamPermissionsRequest struct {
	// Permissions: The set of permissions to check for the `resource`. Permissions
	// with wildcards (such as `*` or `storage.*`) are not allowed. For more
	// information see IAM Overview
	// (https://cloud.google.com/iam/docs/overview#permissions).
	Permissions []string `json:"permissions,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Permissions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Permissions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TestIamPermissionsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod TestIamPermissionsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TestIamPermissionsResponse: Response message for `TestIamPermissions`
// method.
type TestIamPermissionsResponse struct {
	// Permissions: A subset of `TestPermissionsRequest.permissions` that the
	// caller is allowed.
	Permissions []string `json:"permissions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Permissions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Permissions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TestIamPermissionsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod TestIamPermissionsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// VpcAccessibleServices: Specifies how APIs are allowed to communicate within
// the Service Perimeter.
type VpcAccessibleServices struct {
	// AllowedServices: The list of APIs usable within the Service Perimeter. Must
	// be empty unless 'enable_restriction' is True. You can specify a list of
	// individual services, as well as include the 'RESTRICTED-SERVICES' value,
	// which automatically includes all of the services protected by the perimeter.
	AllowedServices []string `json:"allowedServices,omitempty"`
	// EnableRestriction: Whether to restrict API calls within the Service
	// Perimeter to the list of APIs specified in 'allowed_services'.
	EnableRestriction bool `json:"enableRestriction,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AllowedServices") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AllowedServices") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s VpcAccessibleServices) MarshalJSON() ([]byte, error) {
	type NoMethod VpcAccessibleServices
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// VpcNetworkSource: The originating network source in Google Cloud.
type VpcNetworkSource struct {
	// VpcSubnetwork: Sub-segment ranges of a VPC network.
	VpcSubnetwork *VpcSubNetwork `json:"vpcSubnetwork,omitempty"`
	// ForceSendFields is a list of field names (e.g. "VpcSubnetwork") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "VpcSubnetwork") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s VpcNetworkSource) MarshalJSON() ([]byte, error) {
	type NoMethod VpcNetworkSource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// VpcSubNetwork: Sub-segment ranges inside of a VPC Network.
type VpcSubNetwork struct {
	// Network: Required. Network name. If the network is not part of the
	// organization, the `compute.network.get` permission must be granted to the
	// caller. Format:
	// `//compute.googleapis.com/projects/{PROJECT_ID}/global/networks/{NETWORK_NAME
	// }` Example:
	// `//compute.googleapis.com/projects/my-project/global/networks/network-1`
	Network string `json:"network,omitempty"`
	// VpcIpSubnetworks: CIDR block IP subnetwork specification. The IP address
	// must be an IPv4 address and can be a public or private IP address. Note that
	// for a CIDR IP address block, the specified IP address portion must be
	// properly truncated (i.e. all the host bits must be zero) or the input is
	// considered malformed. For example, "192.0.2.0/24" is accepted but
	// "192.0.2.1/24" is not. If empty, all IP addresses are allowed.
	VpcIpSubnetworks []string `json:"vpcIpSubnetworks,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Network") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Network") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s VpcSubNetwork) MarshalJSON() ([]byte, error) {
	type NoMethod VpcSubNetwork
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AccessPoliciesCreateCall struct {
	s            *Service
	accesspolicy *AccessPolicy
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Create: Creates an access policy. This method fails if the organization
// already has an access policy. The long-running operation has a successful
// status after the access policy propagates to long-lasting storage. Syntactic
// and basic semantic errors are returned in `metadata` as a BadRequest proto.
func (r *AccessPoliciesService) Create(accesspolicy *AccessPolicy) *AccessPoliciesCreateCall {
	c := &AccessPoliciesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.accesspolicy = accesspolicy
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesCreateCall) Fields(s ...googleapi.Field) *AccessPoliciesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesCreateCall) Context(ctx context.Context) *AccessPoliciesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.accesspolicy)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accessPolicies")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesCreateCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes an access policy based on the resource name. The
// long-running operation has a successful status after the access policy is
// removed from long-lasting storage.
//
//   - name: Resource name for the access policy to delete. Format
//     `accessPolicies/{policy_id}`.
func (r *AccessPoliciesService) Delete(name string) *AccessPoliciesDeleteCall {
	c := &AccessPoliciesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesDeleteCall) Fields(s ...googleapi.Field) *AccessPoliciesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesDeleteCall) Context(ctx context.Context) *AccessPoliciesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesDeleteCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns an access policy based on the name.
//
//   - name: Resource name for the access policy to get. Format
//     `accessPolicies/{policy_id}`.
func (r *AccessPoliciesService) Get(name string) *AccessPoliciesGetCall {
	c := &AccessPoliciesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesGetCall) Fields(s ...googleapi.Field) *AccessPoliciesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccessPoliciesGetCall) IfNoneMatch(entityTag string) *AccessPoliciesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesGetCall) Context(ctx context.Context) *AccessPoliciesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessPolicy.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesGetCall) Do(opts ...googleapi.CallOption) (*AccessPolicy, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessPolicy{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesGetIamPolicyCall struct {
	s                   *Service
	resource            string
	getiampolicyrequest *GetIamPolicyRequest
	urlParams_          gensupport.URLParams
	ctx_                context.Context
	header_             http.Header
}

// GetIamPolicy: Gets the IAM policy for the specified Access Context Manager
// access policy.
//
//   - resource: REQUIRED: The resource for which the policy is being requested.
//     See Resource names (https://cloud.google.com/apis/design/resource_names)
//     for the appropriate value for this field.
func (r *AccessPoliciesService) GetIamPolicy(resource string, getiampolicyrequest *GetIamPolicyRequest) *AccessPoliciesGetIamPolicyCall {
	c := &AccessPoliciesGetIamPolicyCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resource = resource
	c.getiampolicyrequest = getiampolicyrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesGetIamPolicyCall) Fields(s ...googleapi.Field) *AccessPoliciesGetIamPolicyCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesGetIamPolicyCall) Context(ctx context.Context) *AccessPoliciesGetIamPolicyCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesGetIamPolicyCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesGetIamPolicyCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.getiampolicyrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+resource}:getIamPolicy")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resource": c.resource,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.getIamPolicy" call.
// Any non-2xx status code is an error. Response headers are in either
// *Policy.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesGetIamPolicyCall) Do(opts ...googleapi.CallOption) (*Policy, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Policy{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all access policies in an organization.
func (r *AccessPoliciesService) List() *AccessPoliciesListCall {
	c := &AccessPoliciesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// PageSize sets the optional parameter "pageSize": Number of AccessPolicy
// instances to include in the list. Default 100.
func (c *AccessPoliciesListCall) PageSize(pageSize int64) *AccessPoliciesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Next page token for the
// next batch of AccessPolicy instances. Defaults to the first page of results.
func (c *AccessPoliciesListCall) PageToken(pageToken string) *AccessPoliciesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Parent sets the optional parameter "parent": Required. Resource name for the
// container to list AccessPolicy instances from. Format:
// `organizations/{org_id}`
func (c *AccessPoliciesListCall) Parent(parent string) *AccessPoliciesListCall {
	c.urlParams_.Set("parent", parent)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesListCall) Fields(s ...googleapi.Field) *AccessPoliciesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccessPoliciesListCall) IfNoneMatch(entityTag string) *AccessPoliciesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesListCall) Context(ctx context.Context) *AccessPoliciesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/accessPolicies")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAccessPoliciesResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccessPoliciesListCall) Do(opts ...googleapi.CallOption) (*ListAccessPoliciesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAccessPoliciesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccessPoliciesListCall) Pages(ctx context.Context, f func(*ListAccessPoliciesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccessPoliciesPatchCall struct {
	s            *Service
	name         string
	accesspolicy *AccessPolicy
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Patch: Updates an access policy. The long-running operation from this RPC
// has a successful status after the changes to the access policy propagate to
// long-lasting storage.
//
//   - name: Output only. Identifier. Resource name of the `AccessPolicy`.
//     Format: `accessPolicies/{access_policy}`.
func (r *AccessPoliciesService) Patch(name string, accesspolicy *AccessPolicy) *AccessPoliciesPatchCall {
	c := &AccessPoliciesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.accesspolicy = accesspolicy
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. Mask to
// control which fields get updated. Must be non-empty.
func (c *AccessPoliciesPatchCall) UpdateMask(updateMask string) *AccessPoliciesPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesPatchCall) Fields(s ...googleapi.Field) *AccessPoliciesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesPatchCall) Context(ctx context.Context) *AccessPoliciesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.accesspolicy)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesPatchCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesSetIamPolicyCall struct {
	s                   *Service
	resource            string
	setiampolicyrequest *SetIamPolicyRequest
	urlParams_          gensupport.URLParams
	ctx_                context.Context
	header_             http.Header
}

// SetIamPolicy: Sets the IAM policy for the specified Access Context Manager
// access policy. This method replaces the existing IAM policy on the access
// policy. The IAM policy controls the set of users who can perform specific
// operations on the Access Context Manager access policy.
//
//   - resource: REQUIRED: The resource for which the policy is being specified.
//     See Resource names (https://cloud.google.com/apis/design/resource_names)
//     for the appropriate value for this field.
func (r *AccessPoliciesService) SetIamPolicy(resource string, setiampolicyrequest *SetIamPolicyRequest) *AccessPoliciesSetIamPolicyCall {
	c := &AccessPoliciesSetIamPolicyCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resource = resource
	c.setiampolicyrequest = setiampolicyrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesSetIamPolicyCall) Fields(s ...googleapi.Field) *AccessPoliciesSetIamPolicyCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesSetIamPolicyCall) Context(ctx context.Context) *AccessPoliciesSetIamPolicyCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesSetIamPolicyCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesSetIamPolicyCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.setiampolicyrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+resource}:setIamPolicy")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resource": c.resource,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.setIamPolicy" call.
// Any non-2xx status code is an error. Response headers are in either
// *Policy.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesSetIamPolicyCall) Do(opts ...googleapi.CallOption) (*Policy, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Policy{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesTestIamPermissionsCall struct {
	s                         *Service
	resource                  string
	testiampermissionsrequest *TestIamPermissionsRequest
	urlParams_                gensupport.URLParams
	ctx_                      context.Context
	header_                   http.Header
}

// TestIamPermissions: Returns the IAM permissions that the caller has on the
// specified Access Context Manager resource. The resource can be an
// AccessPolicy, AccessLevel, or ServicePerimeter. This method does not support
// other resources.
//
//   - resource: REQUIRED: The resource for which the policy detail is being
//     requested. See Resource names
//     (https://cloud.google.com/apis/design/resource_names) for the appropriate
//     value for this field.
func (r *AccessPoliciesService) TestIamPermissions(resource string, testiampermissionsrequest *TestIamPermissionsRequest) *AccessPoliciesTestIamPermissionsCall {
	c := &AccessPoliciesTestIamPermissionsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resource = resource
	c.testiampermissionsrequest = testiampermissionsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesTestIamPermissionsCall) Fields(s ...googleapi.Field) *AccessPoliciesTestIamPermissionsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesTestIamPermissionsCall) Context(ctx context.Context) *AccessPoliciesTestIamPermissionsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesTestIamPermissionsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesTestIamPermissionsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.testiampermissionsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+resource}:testIamPermissions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resource": c.resource,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.testIamPermissions" call.
// Any non-2xx status code is an error. Response headers are in either
// *TestIamPermissionsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccessPoliciesTestIamPermissionsCall) Do(opts ...googleapi.CallOption) (*TestIamPermissionsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TestIamPermissionsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesAccessLevelsCreateCall struct {
	s           *Service
	parent      string
	accesslevel *AccessLevel
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// Create: Creates an access level. The long-running operation from this RPC
// has a successful status after the access level propagates to long-lasting
// storage. If access levels contain errors, an error response is returned for
// the first error encountered.
//
//   - parent: Resource name for the access policy which owns this Access Level.
//     Format: `accessPolicies/{policy_id}`.
func (r *AccessPoliciesAccessLevelsService) Create(parent string, accesslevel *AccessLevel) *AccessPoliciesAccessLevelsCreateCall {
	c := &AccessPoliciesAccessLevelsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.accesslevel = accesslevel
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAccessLevelsCreateCall) Fields(s ...googleapi.Field) *AccessPoliciesAccessLevelsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAccessLevelsCreateCall) Context(ctx context.Context) *AccessPoliciesAccessLevelsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAccessLevelsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAccessLevelsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.accesslevel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/accessLevels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.accessLevels.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesAccessLevelsCreateCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesAccessLevelsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes an access level based on the resource name. The long-running
// operation from this RPC has a successful status after the access level has
// been removed from long-lasting storage.
//
//   - name: Resource name for the Access Level. Format:
//     `accessPolicies/{policy_id}/accessLevels/{access_level_id}`.
func (r *AccessPoliciesAccessLevelsService) Delete(name string) *AccessPoliciesAccessLevelsDeleteCall {
	c := &AccessPoliciesAccessLevelsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAccessLevelsDeleteCall) Fields(s ...googleapi.Field) *AccessPoliciesAccessLevelsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAccessLevelsDeleteCall) Context(ctx context.Context) *AccessPoliciesAccessLevelsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAccessLevelsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAccessLevelsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.accessLevels.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesAccessLevelsDeleteCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesAccessLevelsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets an access level based on the resource name.
//
//   - name: Resource name for the Access Level. Format:
//     `accessPolicies/{policy_id}/accessLevels/{access_level_id}`.
func (r *AccessPoliciesAccessLevelsService) Get(name string) *AccessPoliciesAccessLevelsGetCall {
	c := &AccessPoliciesAccessLevelsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// AccessLevelFormat sets the optional parameter "accessLevelFormat": Whether
// to return `BasicLevels` in the Cloud Common Expression Language rather than
// as `BasicLevels`. Defaults to AS_DEFINED, where Access Levels are returned
// as `BasicLevels` or `CustomLevels` based on how they were created. If set to
// CEL, all Access Levels are returned as `CustomLevels`. In the CEL case,
// `BasicLevels` are translated to equivalent `CustomLevels`.
//
// Possible values:
//
//	"LEVEL_FORMAT_UNSPECIFIED" - The format was not specified.
//	"AS_DEFINED" - Uses the format the resource was defined in. BasicLevels
//
// are returned as BasicLevels, CustomLevels are returned as CustomLevels.
//
//	"CEL" - Use Cloud Common Expression Language when returning the resource.
//
// Both BasicLevels and CustomLevels are returned as CustomLevels.
func (c *AccessPoliciesAccessLevelsGetCall) AccessLevelFormat(accessLevelFormat string) *AccessPoliciesAccessLevelsGetCall {
	c.urlParams_.Set("accessLevelFormat", accessLevelFormat)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAccessLevelsGetCall) Fields(s ...googleapi.Field) *AccessPoliciesAccessLevelsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccessPoliciesAccessLevelsGetCall) IfNoneMatch(entityTag string) *AccessPoliciesAccessLevelsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAccessLevelsGetCall) Context(ctx context.Context) *AccessPoliciesAccessLevelsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAccessLevelsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAccessLevelsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.accessLevels.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessLevel.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesAccessLevelsGetCall) Do(opts ...googleapi.CallOption) (*AccessLevel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessLevel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesAccessLevelsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all access levels for an access policy.
//
//   - parent: Resource name for the access policy to list Access Levels from.
//     Format: `accessPolicies/{policy_id}`.
func (r *AccessPoliciesAccessLevelsService) List(parent string) *AccessPoliciesAccessLevelsListCall {
	c := &AccessPoliciesAccessLevelsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// AccessLevelFormat sets the optional parameter "accessLevelFormat": Whether
// to return `BasicLevels` in the Cloud Common Expression language, as
// `CustomLevels`, rather than as `BasicLevels`. Defaults to returning
// `AccessLevels` in the format they were defined.
//
// Possible values:
//
//	"LEVEL_FORMAT_UNSPECIFIED" - The format was not specified.
//	"AS_DEFINED" - Uses the format the resource was defined in. BasicLevels
//
// are returned as BasicLevels, CustomLevels are returned as CustomLevels.
//
//	"CEL" - Use Cloud Common Expression Language when returning the resource.
//
// Both BasicLevels and CustomLevels are returned as CustomLevels.
func (c *AccessPoliciesAccessLevelsListCall) AccessLevelFormat(accessLevelFormat string) *AccessPoliciesAccessLevelsListCall {
	c.urlParams_.Set("accessLevelFormat", accessLevelFormat)
	return c
}

// PageSize sets the optional parameter "pageSize": Number of Access Levels to
// include in the list. Default 100.
func (c *AccessPoliciesAccessLevelsListCall) PageSize(pageSize int64) *AccessPoliciesAccessLevelsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Next page token for the
// next batch of Access Level instances. Defaults to the first page of results.
func (c *AccessPoliciesAccessLevelsListCall) PageToken(pageToken string) *AccessPoliciesAccessLevelsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAccessLevelsListCall) Fields(s ...googleapi.Field) *AccessPoliciesAccessLevelsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccessPoliciesAccessLevelsListCall) IfNoneMatch(entityTag string) *AccessPoliciesAccessLevelsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAccessLevelsListCall) Context(ctx context.Context) *AccessPoliciesAccessLevelsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAccessLevelsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAccessLevelsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/accessLevels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.accessLevels.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAccessLevelsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccessPoliciesAccessLevelsListCall) Do(opts ...googleapi.CallOption) (*ListAccessLevelsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAccessLevelsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccessPoliciesAccessLevelsListCall) Pages(ctx context.Context, f func(*ListAccessLevelsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccessPoliciesAccessLevelsPatchCall struct {
	s           *Service
	name        string
	accesslevel *AccessLevel
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// Patch: Updates an access level. The long-running operation from this RPC has
// a successful status after the changes to the access level propagate to
// long-lasting storage. If access levels contain errors, an error response is
// returned for the first error encountered.
//
//   - name: Identifier. Resource name for the `AccessLevel`. Format:
//     `accessPolicies/{access_policy}/accessLevels/{access_level}`. The
//     `access_level` component must begin with a letter, followed by
//     alphanumeric characters or `_`. Its maximum length is 50 characters. After
//     you create an `AccessLevel`, you cannot change its `name`.
func (r *AccessPoliciesAccessLevelsService) Patch(name string, accesslevel *AccessLevel) *AccessPoliciesAccessLevelsPatchCall {
	c := &AccessPoliciesAccessLevelsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.accesslevel = accesslevel
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. Mask to
// control which fields get updated. Must be non-empty.
func (c *AccessPoliciesAccessLevelsPatchCall) UpdateMask(updateMask string) *AccessPoliciesAccessLevelsPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAccessLevelsPatchCall) Fields(s ...googleapi.Field) *AccessPoliciesAccessLevelsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAccessLevelsPatchCall) Context(ctx context.Context) *AccessPoliciesAccessLevelsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAccessLevelsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAccessLevelsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.accesslevel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.accessLevels.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesAccessLevelsPatchCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesAccessLevelsReplaceAllCall struct {
	s                          *Service
	parent                     string
	replaceaccesslevelsrequest *ReplaceAccessLevelsRequest
	urlParams_                 gensupport.URLParams
	ctx_                       context.Context
	header_                    http.Header
}

// ReplaceAll: Replaces all existing access levels in an access policy with the
// access levels provided. This is done atomically. The long-running operation
// from this RPC has a successful status after all replacements propagate to
// long-lasting storage. If the replacement contains errors, an error response
// is returned for the first error encountered. Upon error, the replacement is
// cancelled, and existing access levels are not affected. The
// Operation.response field contains ReplaceAccessLevelsResponse. Removing
// access levels contained in existing service perimeters result in an error.
//
//   - parent: Resource name for the access policy which owns these Access
//     Levels. Format: `accessPolicies/{policy_id}`.
func (r *AccessPoliciesAccessLevelsService) ReplaceAll(parent string, replaceaccesslevelsrequest *ReplaceAccessLevelsRequest) *AccessPoliciesAccessLevelsReplaceAllCall {
	c := &AccessPoliciesAccessLevelsReplaceAllCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.replaceaccesslevelsrequest = replaceaccesslevelsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAccessLevelsReplaceAllCall) Fields(s ...googleapi.Field) *AccessPoliciesAccessLevelsReplaceAllCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAccessLevelsReplaceAllCall) Context(ctx context.Context) *AccessPoliciesAccessLevelsReplaceAllCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAccessLevelsReplaceAllCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAccessLevelsReplaceAllCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.replaceaccesslevelsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/accessLevels:replaceAll")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.accessLevels.replaceAll" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesAccessLevelsReplaceAllCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesAccessLevelsTestIamPermissionsCall struct {
	s                         *Service
	resource                  string
	testiampermissionsrequest *TestIamPermissionsRequest
	urlParams_                gensupport.URLParams
	ctx_                      context.Context
	header_                   http.Header
}

// TestIamPermissions: Returns the IAM permissions that the caller has on the
// specified Access Context Manager resource. The resource can be an
// AccessPolicy, AccessLevel, or ServicePerimeter. This method does not support
// other resources.
//
//   - resource: REQUIRED: The resource for which the policy detail is being
//     requested. See Resource names
//     (https://cloud.google.com/apis/design/resource_names) for the appropriate
//     value for this field.
func (r *AccessPoliciesAccessLevelsService) TestIamPermissions(resource string, testiampermissionsrequest *TestIamPermissionsRequest) *AccessPoliciesAccessLevelsTestIamPermissionsCall {
	c := &AccessPoliciesAccessLevelsTestIamPermissionsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resource = resource
	c.testiampermissionsrequest = testiampermissionsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAccessLevelsTestIamPermissionsCall) Fields(s ...googleapi.Field) *AccessPoliciesAccessLevelsTestIamPermissionsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAccessLevelsTestIamPermissionsCall) Context(ctx context.Context) *AccessPoliciesAccessLevelsTestIamPermissionsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAccessLevelsTestIamPermissionsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAccessLevelsTestIamPermissionsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.testiampermissionsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+resource}:testIamPermissions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resource": c.resource,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.accessLevels.testIamPermissions" call.
// Any non-2xx status code is an error. Response headers are in either
// *TestIamPermissionsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccessPoliciesAccessLevelsTestIamPermissionsCall) Do(opts ...googleapi.CallOption) (*TestIamPermissionsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TestIamPermissionsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesAuthorizedOrgsDescsCreateCall struct {
	s                  *Service
	parent             string
	authorizedorgsdesc *AuthorizedOrgsDesc
	urlParams_         gensupport.URLParams
	ctx_               context.Context
	header_            http.Header
}

// Create: Creates an authorized orgs desc. The long-running operation from
// this RPC has a successful status after the authorized orgs desc propagates
// to long-lasting storage. If a authorized orgs desc contains errors, an error
// response is returned for the first error encountered. The name of this
// `AuthorizedOrgsDesc` will be assigned during creation.
//
//   - parent: Resource name for the access policy which owns this Authorized
//     Orgs Desc. Format: `accessPolicies/{policy_id}`.
func (r *AccessPoliciesAuthorizedOrgsDescsService) Create(parent string, authorizedorgsdesc *AuthorizedOrgsDesc) *AccessPoliciesAuthorizedOrgsDescsCreateCall {
	c := &AccessPoliciesAuthorizedOrgsDescsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.authorizedorgsdesc = authorizedorgsdesc
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAuthorizedOrgsDescsCreateCall) Fields(s ...googleapi.Field) *AccessPoliciesAuthorizedOrgsDescsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAuthorizedOrgsDescsCreateCall) Context(ctx context.Context) *AccessPoliciesAuthorizedOrgsDescsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAuthorizedOrgsDescsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAuthorizedOrgsDescsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.authorizedorgsdesc)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/authorizedOrgsDescs")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.authorizedOrgsDescs.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesAuthorizedOrgsDescsCreateCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesAuthorizedOrgsDescsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes an authorized orgs desc based on the resource name. The
// long-running operation from this RPC has a successful status after the
// authorized orgs desc is removed from long-lasting storage.
//
//   - name: Resource name for the Authorized Orgs Desc. Format:
//     `accessPolicies/{policy_id}/authorizedOrgsDesc/{authorized_orgs_desc_id}`.
func (r *AccessPoliciesAuthorizedOrgsDescsService) Delete(name string) *AccessPoliciesAuthorizedOrgsDescsDeleteCall {
	c := &AccessPoliciesAuthorizedOrgsDescsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAuthorizedOrgsDescsDeleteCall) Fields(s ...googleapi.Field) *AccessPoliciesAuthorizedOrgsDescsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAuthorizedOrgsDescsDeleteCall) Context(ctx context.Context) *AccessPoliciesAuthorizedOrgsDescsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAuthorizedOrgsDescsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAuthorizedOrgsDescsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.authorizedOrgsDescs.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesAuthorizedOrgsDescsDeleteCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesAuthorizedOrgsDescsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets an authorized orgs desc based on the resource name.
//
//   - name: Resource name for the Authorized Orgs Desc. Format:
//     `accessPolicies/{policy_id}/authorizedOrgsDescs/{authorized_orgs_descs_id}`.
func (r *AccessPoliciesAuthorizedOrgsDescsService) Get(name string) *AccessPoliciesAuthorizedOrgsDescsGetCall {
	c := &AccessPoliciesAuthorizedOrgsDescsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAuthorizedOrgsDescsGetCall) Fields(s ...googleapi.Field) *AccessPoliciesAuthorizedOrgsDescsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccessPoliciesAuthorizedOrgsDescsGetCall) IfNoneMatch(entityTag string) *AccessPoliciesAuthorizedOrgsDescsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAuthorizedOrgsDescsGetCall) Context(ctx context.Context) *AccessPoliciesAuthorizedOrgsDescsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAuthorizedOrgsDescsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAuthorizedOrgsDescsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.authorizedOrgsDescs.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *AuthorizedOrgsDesc.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccessPoliciesAuthorizedOrgsDescsGetCall) Do(opts ...googleapi.CallOption) (*AuthorizedOrgsDesc, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AuthorizedOrgsDesc{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesAuthorizedOrgsDescsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all authorized orgs descs for an access policy.
//
//   - parent: Resource name for the access policy to list Authorized Orgs Desc
//     from. Format: `accessPolicies/{policy_id}`.
func (r *AccessPoliciesAuthorizedOrgsDescsService) List(parent string) *AccessPoliciesAuthorizedOrgsDescsListCall {
	c := &AccessPoliciesAuthorizedOrgsDescsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": Number of Authorized Orgs
// Descs to include in the list. Default 100.
func (c *AccessPoliciesAuthorizedOrgsDescsListCall) PageSize(pageSize int64) *AccessPoliciesAuthorizedOrgsDescsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Next page token for the
// next batch of Authorized Orgs Desc instances. Defaults to the first page of
// results.
func (c *AccessPoliciesAuthorizedOrgsDescsListCall) PageToken(pageToken string) *AccessPoliciesAuthorizedOrgsDescsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAuthorizedOrgsDescsListCall) Fields(s ...googleapi.Field) *AccessPoliciesAuthorizedOrgsDescsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccessPoliciesAuthorizedOrgsDescsListCall) IfNoneMatch(entityTag string) *AccessPoliciesAuthorizedOrgsDescsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAuthorizedOrgsDescsListCall) Context(ctx context.Context) *AccessPoliciesAuthorizedOrgsDescsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAuthorizedOrgsDescsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAuthorizedOrgsDescsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/authorizedOrgsDescs")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.authorizedOrgsDescs.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAuthorizedOrgsDescsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccessPoliciesAuthorizedOrgsDescsListCall) Do(opts ...googleapi.CallOption) (*ListAuthorizedOrgsDescsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAuthorizedOrgsDescsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccessPoliciesAuthorizedOrgsDescsListCall) Pages(ctx context.Context, f func(*ListAuthorizedOrgsDescsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccessPoliciesAuthorizedOrgsDescsPatchCall struct {
	s                  *Service
	name               string
	authorizedorgsdesc *AuthorizedOrgsDesc
	urlParams_         gensupport.URLParams
	ctx_               context.Context
	header_            http.Header
}

// Patch: Updates an authorized orgs desc. The long-running operation from this
// RPC has a successful status after the authorized orgs desc propagates to
// long-lasting storage. If a authorized orgs desc contains errors, an error
// response is returned for the first error encountered. Only the organization
// list in `AuthorizedOrgsDesc` can be updated. The name, authorization_type,
// asset_type and authorization_direction cannot be updated.
//
//   - name: Identifier. Resource name for the `AuthorizedOrgsDesc`. Format:
//     `accessPolicies/{access_policy}/authorizedOrgsDescs/{authorized_orgs_desc}`
//     . The `authorized_orgs_desc` component must begin with a letter, followed
//     by alphanumeric characters or `_`. After you create an
//     `AuthorizedOrgsDesc`, you cannot change its `name`.
func (r *AccessPoliciesAuthorizedOrgsDescsService) Patch(name string, authorizedorgsdesc *AuthorizedOrgsDesc) *AccessPoliciesAuthorizedOrgsDescsPatchCall {
	c := &AccessPoliciesAuthorizedOrgsDescsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.authorizedorgsdesc = authorizedorgsdesc
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. Mask to
// control which fields get updated. Must be non-empty.
func (c *AccessPoliciesAuthorizedOrgsDescsPatchCall) UpdateMask(updateMask string) *AccessPoliciesAuthorizedOrgsDescsPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesAuthorizedOrgsDescsPatchCall) Fields(s ...googleapi.Field) *AccessPoliciesAuthorizedOrgsDescsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesAuthorizedOrgsDescsPatchCall) Context(ctx context.Context) *AccessPoliciesAuthorizedOrgsDescsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesAuthorizedOrgsDescsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesAuthorizedOrgsDescsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.authorizedorgsdesc)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.authorizedOrgsDescs.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesAuthorizedOrgsDescsPatchCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesServicePerimetersCommitCall struct {
	s                              *Service
	parent                         string
	commitserviceperimetersrequest *CommitServicePerimetersRequest
	urlParams_                     gensupport.URLParams
	ctx_                           context.Context
	header_                        http.Header
}

// Commit: Commits the dry-run specification for all the service perimeters in
// an access policy. A commit operation on a service perimeter involves copying
// its `spec` field to the `status` field of the service perimeter. Only
// service perimeters with `use_explicit_dry_run_spec` field set to true are
// affected by a commit operation. The long-running operation from this RPC has
// a successful status after the dry-run specifications for all the service
// perimeters have been committed. If a commit fails, it causes the
// long-running operation to return an error response and the entire commit
// operation is cancelled. When successful, the Operation.response field
// contains CommitServicePerimetersResponse. The `dry_run` and the `spec`
// fields are cleared after a successful commit operation.
//
//   - parent: Resource name for the parent Access Policy which owns all Service
//     Perimeters in scope for the commit operation. Format:
//     `accessPolicies/{policy_id}`.
func (r *AccessPoliciesServicePerimetersService) Commit(parent string, commitserviceperimetersrequest *CommitServicePerimetersRequest) *AccessPoliciesServicePerimetersCommitCall {
	c := &AccessPoliciesServicePerimetersCommitCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.commitserviceperimetersrequest = commitserviceperimetersrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesServicePerimetersCommitCall) Fields(s ...googleapi.Field) *AccessPoliciesServicePerimetersCommitCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesServicePerimetersCommitCall) Context(ctx context.Context) *AccessPoliciesServicePerimetersCommitCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesServicePerimetersCommitCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesServicePerimetersCommitCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.commitserviceperimetersrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/servicePerimeters:commit")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.servicePerimeters.commit" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesServicePerimetersCommitCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesServicePerimetersCreateCall struct {
	s                *Service
	parent           string
	serviceperimeter *ServicePerimeter
	urlParams_       gensupport.URLParams
	ctx_             context.Context
	header_          http.Header
}

// Create: Creates a service perimeter. The long-running operation from this
// RPC has a successful status after the service perimeter propagates to
// long-lasting storage. If a service perimeter contains errors, an error
// response is returned for the first error encountered.
//
//   - parent: Resource name for the access policy which owns this Service
//     Perimeter. Format: `accessPolicies/{policy_id}`.
func (r *AccessPoliciesServicePerimetersService) Create(parent string, serviceperimeter *ServicePerimeter) *AccessPoliciesServicePerimetersCreateCall {
	c := &AccessPoliciesServicePerimetersCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.serviceperimeter = serviceperimeter
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesServicePerimetersCreateCall) Fields(s ...googleapi.Field) *AccessPoliciesServicePerimetersCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesServicePerimetersCreateCall) Context(ctx context.Context) *AccessPoliciesServicePerimetersCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesServicePerimetersCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesServicePerimetersCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.serviceperimeter)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/servicePerimeters")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.servicePerimeters.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesServicePerimetersCreateCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesServicePerimetersDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a service perimeter based on the resource name. The
// long-running operation from this RPC has a successful status after the
// service perimeter is removed from long-lasting storage.
//
//   - name: Resource name for the Service Perimeter. Format:
//     `accessPolicies/{policy_id}/servicePerimeters/{service_perimeter_id}`.
func (r *AccessPoliciesServicePerimetersService) Delete(name string) *AccessPoliciesServicePerimetersDeleteCall {
	c := &AccessPoliciesServicePerimetersDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesServicePerimetersDeleteCall) Fields(s ...googleapi.Field) *AccessPoliciesServicePerimetersDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesServicePerimetersDeleteCall) Context(ctx context.Context) *AccessPoliciesServicePerimetersDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesServicePerimetersDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesServicePerimetersDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.servicePerimeters.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesServicePerimetersDeleteCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesServicePerimetersGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a service perimeter based on the resource name.
//
//   - name: Resource name for the Service Perimeter. Format:
//     `accessPolicies/{policy_id}/servicePerimeters/{service_perimeters_id}`.
func (r *AccessPoliciesServicePerimetersService) Get(name string) *AccessPoliciesServicePerimetersGetCall {
	c := &AccessPoliciesServicePerimetersGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesServicePerimetersGetCall) Fields(s ...googleapi.Field) *AccessPoliciesServicePerimetersGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccessPoliciesServicePerimetersGetCall) IfNoneMatch(entityTag string) *AccessPoliciesServicePerimetersGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesServicePerimetersGetCall) Context(ctx context.Context) *AccessPoliciesServicePerimetersGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesServicePerimetersGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesServicePerimetersGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.servicePerimeters.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *ServicePerimeter.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccessPoliciesServicePerimetersGetCall) Do(opts ...googleapi.CallOption) (*ServicePerimeter, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ServicePerimeter{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesServicePerimetersListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all service perimeters for an access policy.
//
//   - parent: Resource name for the access policy to list Service Perimeters
//     from. Format: `accessPolicies/{policy_id}`.
func (r *AccessPoliciesServicePerimetersService) List(parent string) *AccessPoliciesServicePerimetersListCall {
	c := &AccessPoliciesServicePerimetersListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": Number of Service
// Perimeters to include in the list. Default 100.
func (c *AccessPoliciesServicePerimetersListCall) PageSize(pageSize int64) *AccessPoliciesServicePerimetersListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Next page token for the
// next batch of Service Perimeter instances. Defaults to the first page of
// results.
func (c *AccessPoliciesServicePerimetersListCall) PageToken(pageToken string) *AccessPoliciesServicePerimetersListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesServicePerimetersListCall) Fields(s ...googleapi.Field) *AccessPoliciesServicePerimetersListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccessPoliciesServicePerimetersListCall) IfNoneMatch(entityTag string) *AccessPoliciesServicePerimetersListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesServicePerimetersListCall) Context(ctx context.Context) *AccessPoliciesServicePerimetersListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesServicePerimetersListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesServicePerimetersListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/servicePerimeters")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.servicePerimeters.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListServicePerimetersResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccessPoliciesServicePerimetersListCall) Do(opts ...googleapi.CallOption) (*ListServicePerimetersResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListServicePerimetersResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccessPoliciesServicePerimetersListCall) Pages(ctx context.Context, f func(*ListServicePerimetersResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccessPoliciesServicePerimetersPatchCall struct {
	s                *Service
	name             string
	serviceperimeter *ServicePerimeter
	urlParams_       gensupport.URLParams
	ctx_             context.Context
	header_          http.Header
}

// Patch: Updates a service perimeter. The long-running operation from this RPC
// has a successful status after the service perimeter propagates to
// long-lasting storage. If a service perimeter contains errors, an error
// response is returned for the first error encountered.
//
//   - name: Identifier. Resource name for the `ServicePerimeter`. Format:
//     `accessPolicies/{access_policy}/servicePerimeters/{service_perimeter}`.
//     The `service_perimeter` component must begin with a letter, followed by
//     alphanumeric characters or `_`. After you create a `ServicePerimeter`, you
//     cannot change its `name`.
func (r *AccessPoliciesServicePerimetersService) Patch(name string, serviceperimeter *ServicePerimeter) *AccessPoliciesServicePerimetersPatchCall {
	c := &AccessPoliciesServicePerimetersPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.serviceperimeter = serviceperimeter
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. Mask to
// control which fields get updated. Must be non-empty.
func (c *AccessPoliciesServicePerimetersPatchCall) UpdateMask(updateMask string) *AccessPoliciesServicePerimetersPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesServicePerimetersPatchCall) Fields(s ...googleapi.Field) *AccessPoliciesServicePerimetersPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesServicePerimetersPatchCall) Context(ctx context.Context) *AccessPoliciesServicePerimetersPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesServicePerimetersPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesServicePerimetersPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.serviceperimeter)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.servicePerimeters.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesServicePerimetersPatchCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesServicePerimetersReplaceAllCall struct {
	s                               *Service
	parent                          string
	replaceserviceperimetersrequest *ReplaceServicePerimetersRequest
	urlParams_                      gensupport.URLParams
	ctx_                            context.Context
	header_                         http.Header
}

// ReplaceAll: Replace all existing service perimeters in an access policy with
// the service perimeters provided. This is done atomically. The long-running
// operation from this RPC has a successful status after all replacements
// propagate to long-lasting storage. Replacements containing errors result in
// an error response for the first error encountered. Upon an error,
// replacement are cancelled and existing service perimeters are not affected.
// The Operation.response field contains ReplaceServicePerimetersResponse.
//
//   - parent: Resource name for the access policy which owns these Service
//     Perimeters. Format: `accessPolicies/{policy_id}`.
func (r *AccessPoliciesServicePerimetersService) ReplaceAll(parent string, replaceserviceperimetersrequest *ReplaceServicePerimetersRequest) *AccessPoliciesServicePerimetersReplaceAllCall {
	c := &AccessPoliciesServicePerimetersReplaceAllCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.replaceserviceperimetersrequest = replaceserviceperimetersrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesServicePerimetersReplaceAllCall) Fields(s ...googleapi.Field) *AccessPoliciesServicePerimetersReplaceAllCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesServicePerimetersReplaceAllCall) Context(ctx context.Context) *AccessPoliciesServicePerimetersReplaceAllCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesServicePerimetersReplaceAllCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesServicePerimetersReplaceAllCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.replaceserviceperimetersrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/servicePerimeters:replaceAll")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.servicePerimeters.replaceAll" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccessPoliciesServicePerimetersReplaceAllCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccessPoliciesServicePerimetersTestIamPermissionsCall struct {
	s                         *Service
	resource                  string
	testiampermissionsrequest *TestIamPermissionsRequest
	urlParams_                gensupport.URLParams
	ctx_                      context.Context
	header_                   http.Header
}

// TestIamPermissions: Returns the IAM permissions that the caller has on the
// specified Access Context Manager resource. The resource can be an
// AccessPolicy, AccessLevel, or ServicePerimeter. This method does not support
// other resources.
//
//   - resource: REQUIRED: The resource for which the policy detail is being
//     requested. See Resource names
//     (https://cloud.google.com/apis/design/resource_names) for the appropriate
//     value for this field.
func (r *AccessPoliciesServicePerimetersService) TestIamPermissions(resource string, testiampermissionsrequest *TestIamPermissionsRequest) *AccessPoliciesServicePerimetersTestIamPermissionsCall {
	c := &AccessPoliciesServicePerimetersTestIamPermissionsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resource = resource
	c.testiampermissionsrequest = testiampermissionsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccessPoliciesServicePerimetersTestIamPermissionsCall) Fields(s ...googleapi.Field) *AccessPoliciesServicePerimetersTestIamPermissionsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccessPoliciesServicePerimetersTestIamPermissionsCall) Context(ctx context.Context) *AccessPoliciesServicePerimetersTestIamPermissionsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccessPoliciesServicePerimetersTestIamPermissionsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccessPoliciesServicePerimetersTestIamPermissionsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.testiampermissionsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+resource}:testIamPermissions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resource": c.resource,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.accessPolicies.servicePerimeters.testIamPermissions" call.
// Any non-2xx status code is an error. Response headers are in either
// *TestIamPermissionsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccessPoliciesServicePerimetersTestIamPermissionsCall) Do(opts ...googleapi.CallOption) (*TestIamPermissionsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TestIamPermissionsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type OperationsCancelCall struct {
	s                      *Service
	name                   string
	canceloperationrequest *CancelOperationRequest
	urlParams_             gensupport.URLParams
	ctx_                   context.Context
	header_                http.Header
}

// Cancel: Starts asynchronous cancellation on a long-running operation. The
// server makes a best effort to cancel the operation, but success is not
// guaranteed. If the server doesn't support this method, it returns
// `google.rpc.Code.UNIMPLEMENTED`. Clients can use Operations.GetOperation or
// other methods to check whether the cancellation succeeded or whether the
// operation completed despite cancellation. On successful cancellation, the
// operation is not deleted; instead, it becomes an operation with an
// Operation.error value with a google.rpc.Status.code of 1, corresponding to
// `Code.CANCELLED`.
//
// - name: The name of the operation resource to be cancelled.
func (r *OperationsService) Cancel(name string, canceloperationrequest *CancelOperationRequest) *OperationsCancelCall {
	c := &OperationsCancelCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.canceloperationrequest = canceloperationrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OperationsCancelCall) Fields(s ...googleapi.Field) *OperationsCancelCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OperationsCancelCall) Context(ctx context.Context) *OperationsCancelCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OperationsCancelCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OperationsCancelCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.canceloperationrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:cancel")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.operations.cancel" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *OperationsCancelCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type OperationsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a long-running operation. This method indicates that the
// client is no longer interested in the operation result. It does not cancel
// the operation. If the server doesn't support this method, it returns
// `google.rpc.Code.UNIMPLEMENTED`.
//
// - name: The name of the operation resource to be deleted.
func (r *OperationsService) Delete(name string) *OperationsDeleteCall {
	c := &OperationsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OperationsDeleteCall) Fields(s ...googleapi.Field) *OperationsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OperationsDeleteCall) Context(ctx context.Context) *OperationsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OperationsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OperationsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.operations.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *OperationsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type OperationsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the latest state of a long-running operation. Clients can use this
// method to poll the operation result at intervals as recommended by the API
// service.
//
// - name: The name of the operation resource.
func (r *OperationsService) Get(name string) *OperationsGetCall {
	c := &OperationsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OperationsGetCall) Fields(s ...googleapi.Field) *OperationsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OperationsGetCall) IfNoneMatch(entityTag string) *OperationsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OperationsGetCall) Context(ctx context.Context) *OperationsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OperationsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OperationsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.operations.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *OperationsGetCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type OperationsListCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists operations that match the specified filter in the request. If
// the server doesn't support this method, it returns `UNIMPLEMENTED`.
//
// - name: The name of the operation's parent resource.
func (r *OperationsService) List(name string) *OperationsListCall {
	c := &OperationsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Filter sets the optional parameter "filter": The standard list filter.
func (c *OperationsListCall) Filter(filter string) *OperationsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The standard list page
// size.
func (c *OperationsListCall) PageSize(pageSize int64) *OperationsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": The standard list page
// token.
func (c *OperationsListCall) PageToken(pageToken string) *OperationsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OperationsListCall) Fields(s ...googleapi.Field) *OperationsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OperationsListCall) IfNoneMatch(entityTag string) *OperationsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OperationsListCall) Context(ctx context.Context) *OperationsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OperationsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OperationsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.operations.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListOperationsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *OperationsListCall) Do(opts ...googleapi.CallOption) (*ListOperationsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListOperationsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *OperationsListCall) Pages(ctx context.Context, f func(*ListOperationsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type OrganizationsGcpUserAccessBindingsCreateCall struct {
	s                    *Service
	parent               string
	gcpuseraccessbinding *GcpUserAccessBinding
	urlParams_           gensupport.URLParams
	ctx_                 context.Context
	header_              http.Header
}

// Create: Creates a GcpUserAccessBinding. If the client specifies a name, the
// server ignores it. Fails if a resource already exists with the same
// group_key. Completion of this long-running operation does not necessarily
// signify that the new binding is deployed onto all affected users, which may
// take more time.
//
// - parent: Example: "organizations/256".
func (r *OrganizationsGcpUserAccessBindingsService) Create(parent string, gcpuseraccessbinding *GcpUserAccessBinding) *OrganizationsGcpUserAccessBindingsCreateCall {
	c := &OrganizationsGcpUserAccessBindingsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.gcpuseraccessbinding = gcpuseraccessbinding
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsGcpUserAccessBindingsCreateCall) Fields(s ...googleapi.Field) *OrganizationsGcpUserAccessBindingsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsGcpUserAccessBindingsCreateCall) Context(ctx context.Context) *OrganizationsGcpUserAccessBindingsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsGcpUserAccessBindingsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsGcpUserAccessBindingsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.gcpuseraccessbinding)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/gcpUserAccessBindings")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.organizations.gcpUserAccessBindings.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *OrganizationsGcpUserAccessBindingsCreateCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type OrganizationsGcpUserAccessBindingsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a GcpUserAccessBinding. Completion of this long-running
// operation does not necessarily signify that the binding deletion is deployed
// onto all affected users, which may take more time.
//
// - name: Example: "organizations/256/gcpUserAccessBindings/b3-BhcX_Ud5N".
func (r *OrganizationsGcpUserAccessBindingsService) Delete(name string) *OrganizationsGcpUserAccessBindingsDeleteCall {
	c := &OrganizationsGcpUserAccessBindingsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsGcpUserAccessBindingsDeleteCall) Fields(s ...googleapi.Field) *OrganizationsGcpUserAccessBindingsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsGcpUserAccessBindingsDeleteCall) Context(ctx context.Context) *OrganizationsGcpUserAccessBindingsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsGcpUserAccessBindingsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsGcpUserAccessBindingsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.organizations.gcpUserAccessBindings.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *OrganizationsGcpUserAccessBindingsDeleteCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type OrganizationsGcpUserAccessBindingsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the GcpUserAccessBinding with the given name.
//
// - name: Example: "organizations/256/gcpUserAccessBindings/b3-BhcX_Ud5N".
func (r *OrganizationsGcpUserAccessBindingsService) Get(name string) *OrganizationsGcpUserAccessBindingsGetCall {
	c := &OrganizationsGcpUserAccessBindingsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsGcpUserAccessBindingsGetCall) Fields(s ...googleapi.Field) *OrganizationsGcpUserAccessBindingsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OrganizationsGcpUserAccessBindingsGetCall) IfNoneMatch(entityTag string) *OrganizationsGcpUserAccessBindingsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsGcpUserAccessBindingsGetCall) Context(ctx context.Context) *OrganizationsGcpUserAccessBindingsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsGcpUserAccessBindingsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsGcpUserAccessBindingsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.organizations.gcpUserAccessBindings.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *GcpUserAccessBinding.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *OrganizationsGcpUserAccessBindingsGetCall) Do(opts ...googleapi.CallOption) (*GcpUserAccessBinding, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GcpUserAccessBinding{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type OrganizationsGcpUserAccessBindingsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all GcpUserAccessBindings for a Google Cloud organization.
//
// - parent: Example: "organizations/256".
func (r *OrganizationsGcpUserAccessBindingsService) List(parent string) *OrganizationsGcpUserAccessBindingsListCall {
	c := &OrganizationsGcpUserAccessBindingsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of items to
// return. The server may return fewer items. If left blank, the server may
// return any number of items.
func (c *OrganizationsGcpUserAccessBindingsListCall) PageSize(pageSize int64) *OrganizationsGcpUserAccessBindingsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": If left blank, returns
// the first page. To enumerate all items, use the next_page_token from your
// previous list operation.
func (c *OrganizationsGcpUserAccessBindingsListCall) PageToken(pageToken string) *OrganizationsGcpUserAccessBindingsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsGcpUserAccessBindingsListCall) Fields(s ...googleapi.Field) *OrganizationsGcpUserAccessBindingsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OrganizationsGcpUserAccessBindingsListCall) IfNoneMatch(entityTag string) *OrganizationsGcpUserAccessBindingsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsGcpUserAccessBindingsListCall) Context(ctx context.Context) *OrganizationsGcpUserAccessBindingsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsGcpUserAccessBindingsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsGcpUserAccessBindingsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/gcpUserAccessBindings")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.organizations.gcpUserAccessBindings.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListGcpUserAccessBindingsResponse.ServerResponse.Header or (if a response
// was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *OrganizationsGcpUserAccessBindingsListCall) Do(opts ...googleapi.CallOption) (*ListGcpUserAccessBindingsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListGcpUserAccessBindingsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *OrganizationsGcpUserAccessBindingsListCall) Pages(ctx context.Context, f func(*ListGcpUserAccessBindingsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type OrganizationsGcpUserAccessBindingsPatchCall struct {
	s                    *Service
	name                 string
	gcpuseraccessbinding *GcpUserAccessBinding
	urlParams_           gensupport.URLParams
	ctx_                 context.Context
	header_              http.Header
}

// Patch: Updates a GcpUserAccessBinding. Completion of this long-running
// operation does not necessarily signify that the changed binding is deployed
// onto all affected users, which may take more time.
//
//   - name: Immutable. Assigned by the server during creation. The last segment
//     has an arbitrary length and has only URI unreserved characters (as defined
//     by RFC 3986 Section 2.3
//     (https://tools.ietf.org/html/rfc3986#section-2.3)). Should not be
//     specified by the client during creation. Example:
//     "organizations/256/gcpUserAccessBindings/b3-BhcX_Ud5N".
func (r *OrganizationsGcpUserAccessBindingsService) Patch(name string, gcpuseraccessbinding *GcpUserAccessBinding) *OrganizationsGcpUserAccessBindingsPatchCall {
	c := &OrganizationsGcpUserAccessBindingsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.gcpuseraccessbinding = gcpuseraccessbinding
	return c
}

// UpdateMask sets the optional parameter "updateMask": Required. Only the
// fields specified in this mask are updated. Because name and group_key cannot
// be changed, update_mask is required and may only contain the following
// fields: `access_levels`, `dry_run_access_levels`. update_mask { paths:
// "access_levels" }
func (c *OrganizationsGcpUserAccessBindingsPatchCall) UpdateMask(updateMask string) *OrganizationsGcpUserAccessBindingsPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsGcpUserAccessBindingsPatchCall) Fields(s ...googleapi.Field) *OrganizationsGcpUserAccessBindingsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsGcpUserAccessBindingsPatchCall) Context(ctx context.Context) *OrganizationsGcpUserAccessBindingsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsGcpUserAccessBindingsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsGcpUserAccessBindingsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.gcpuseraccessbinding)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.organizations.gcpUserAccessBindings.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *OrganizationsGcpUserAccessBindingsPatchCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type ServicesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Returns a VPC-SC supported service based on the service name.
//
//   - name: The name of the service to get information about. The names must be
//     in the same format as used in defining a service perimeter, for example,
//     `storage.googleapis.com`.
func (r *ServicesService) Get(name string) *ServicesGetCall {
	c := &ServicesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ServicesGetCall) Fields(s ...googleapi.Field) *ServicesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ServicesGetCall) IfNoneMatch(entityTag string) *ServicesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ServicesGetCall) Context(ctx context.Context) *ServicesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ServicesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ServicesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/services/{name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.services.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *SupportedService.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ServicesGetCall) Do(opts ...googleapi.CallOption) (*SupportedService, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SupportedService{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type ServicesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all VPC-SC supported services.
func (r *ServicesService) List() *ServicesListCall {
	c := &ServicesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// PageSize sets the optional parameter "pageSize": This flag specifies the
// maximum number of services to return per page. Default is 100.
func (c *ServicesListCall) PageSize(pageSize int64) *ServicesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Token to start on a later
// page. Default is the first page.
func (c *ServicesListCall) PageToken(pageToken string) *ServicesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ServicesListCall) Fields(s ...googleapi.Field) *ServicesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ServicesListCall) IfNoneMatch(entityTag string) *ServicesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ServicesListCall) Context(ctx context.Context) *ServicesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ServicesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ServicesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/services")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accesscontextmanager.services.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListSupportedServicesResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ServicesListCall) Do(opts ...googleapi.CallOption) (*ListSupportedServicesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListSupportedServicesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ServicesListCall) Pages(ctx context.Context, f func(*ListSupportedServicesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}
