/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef DECOR_H
#define DECOR_H

/* decoration types (alignment) */
#define DA_FULL		0	/* the entire length of the edge */
#define DA_NEAR		1	/* the low vals. i.e. further to the left, or further to the top */
#define	DA_FAR		2	/* bigger vals of x/y. aligned down or aligned right */

/* decoration edges */
#define DE_TOP		0
#define DE_LEFT		1
#define DE_RIGHT	2
#define DE_BOTTOM	3

/* click_actions */
#define ACT_NONE	0
#define ACT_MOVE	1
#define ACT_RESIZE	2
#define ACT_DELETE	3
#define ACT_ICONIFY	5
#define ACT_ZOOM	6

/*
 * a unit of window decoration.  Each piece of decoration is implemented as an optionally
 * shaped child window of the window's frame.  Position and size are specified by the
 * edge and type members, with finer control provied by size/position mods (see below).  Window
 * decoration may have an operation to perform on mouse presses, or it may function as a
 * button, in which case the operation will be performed as the user would expect for a
 * button: when the mouse is released, if it's still inside the button decor.  Button 
 * decoration should also provide a pixmap that will be used when the button is pressed.
 *
 * note that the pressed button pixmap must have the same shape/size as the unpressed one.
 * perhaps at some point this will be changed, but not for now....if you use one that's not
 * the same size/shape yer gonna get some dumb crap happening.
 *
 * about the size pos modifification stuff:  to give finer control on how decoration is sized
 * or positioned these vals can be directly modified with numbers (lenmod and offset) or can
 * be modified by a multiplier on the total size of the edge (using lenmult and offsetmult).
 * for instance, to make something half as long as the edge, but indented by 1/8 of the size
 * of the edge plus 10 pixels, you would set lenmult to .5, offsetmult to .125, offset to 10
 * and lenmod to 0.  (you'd use DA_NEAR for type in this case).
 */
struct decor {
	char	*ident;		/* name of the decoration unit */
	int	type;		/* type of decoration */
	int	edge;		/* edge to put this decor on (top, left, etc) */

	int	offset;		/* pixel offset for placement in primary direction */
	int	soffset;	/* pixel offset in secondary direction; used to allow
				 * overlapping the client's area */
	int	lenmod;		/* modification to the length of decor */
	double	offsetmult;	/* factor * the total len of the edge to offset by */
	double	lenmult;	/* factor * the total len of the edge to size by */

	int	lclick_action;	/* what to do when pressed, or if button when clicked */
	int	rclick_action;	/* same for the right click */
	int	mclick_action;	/* same for the middle click */

	pixmap_t 	*pixmap;	/* pixmap struct for the decoration unit when focused */
	pixmap_t	*focpixmap;	/* pixmap for when unit is focused */
	pixmap_t	*pressedmap;	/* pixmap for the unit when it's pressed (for buttons) */

	struct {
		u_int shaped:1;	/* should we shape the decor win? (frame ShapeUnion'd w/ rect if 0) */
		u_int button:1;	/* does this piece of decoration have button functionality */
	} flags;

	SLIST_ENTRY(decor) d_list;
};

#ifdef I18N
extern XFontSet titlefont;
#else
extern XFontStruct *titlefont;
#endif

void decor_init();
void decor_shutdown();
void decor_add(decor_t *decor);
decor_t *decor_ident(char *ident);
void decor_shapesize(client_t *client);
void decor_decorate(client_t *client);
void decor_undecor(client_t *client);
void decor_handlepress(client_t *client, decor_t *decor, XButtonEvent *e);
void decor_focus(client_t *client);
void decor_unfocus(client_t *client);
void decor_expose(client_t *client, decor_t *decor, XExposeEvent *e);
void decor_titlechange(client_t *client);

#endif
