/*
 * Copyright (C) 2003 ViASIC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */

#include <string.h>
#include "util.h"
#include "utmem.h"

struct utMem_ *utMems;
U32 utfFreeMem, utfVirgMem_, utmMem;
struct utStack_ *utStacks;
U32 utsStack, utmStack;

static utStackRef utBotStack;
static U32 utMaxMem;
U32 utUsedMem;
U32 utMemSeed = 42;

/*--------------------------------------------------------------------------------------------------
  Create a new mem.
--------------------------------------------------------------------------------------------------*/
utMemRef utcMem (void)
{
   utMemRef mem;

   if (uttMemExists(utfFreeMem())) {
      mem = utfFreeMem();
      utfFreeMem() = utnFreeMem(mem);
   } else {
      if (utfVirgMem() == utmMem()) {
         utmMem() += utmMem()/2;
         utMems = (struct utMem_ *)mtRealloc((void *)utMems,
               utmMem(), sizeof(struct utMem_));
         if (!utMems) {
            utExit("utcMem: Can't allocate memory handle");
         }
      }
      mem = utfVirgMem()++;
   }
   uttMemUsed(mem) = true;
   return mem;
}

/*--------------------------------------------------------------------------------------------------
  Delete a mem (put it on the free list).
--------------------------------------------------------------------------------------------------*/
void UTEXPORT utdMem (
   utMemRef mem,
   bool farHeap,
   char *fileName,
   U32 line)
{
   utStackRef stack = utlStack();
   U8 rightPicket = *(((U8 *)utgMemPtr(mem)) + utgMemSize(mem) - 1);

   utUsedMem -= utgMemSize(mem);
   if (!uttMemUsed(mem)) {
      utExit("utdMem: Memory allocated in %s on line %u freed second time "
            "in %s on line %u", utSymGetName(utfMemNameSym(mem)),
            utgMemLine(mem), fileName, line);
   }
   if (utgMemPicket(mem) != rightPicket) {
      utExit("utdMem: Memory allocated in %s on line %u and freed in %s "
            "line %u lost right picket", utSymGetName(utfMemNameSym(mem)),
            utgMemLine(mem), fileName, line);
   }
   if (uttMemFarHeap(mem) && !farHeap) {
      utExit("utdMem: Far memory allocated in %s on line %u is freed in %s "
            "line %u to the near heap", utSymGetName(utfMemNameSym(mem)),
            utgMemLine(mem), fileName, line);
   }
   if (!uttMemFarHeap(mem) && farHeap) {
      utExit("utdMem: Near memory allocated in %s on line %u is freed in %s "
            "line %u to the far heap", utSymGetName(utfMemNameSym(mem)),
            utgMemLine(mem), fileName, line);
   }
   utdStackMem(stack, mem, fileName, line);
   utnFreeMem(mem) = utfFreeMem();
   utfFreeMem() = mem;
   uttMemUsed(mem) = false;
}

/*--------------------------------------------------------------------------------------------------
  Set the seed.  Set the signature to 0;
--------------------------------------------------------------------------------------------------*/
void UTEXPORT utMemInitSeed (
   U32 seed)
{
   if (seed == 0) {
      utMemSeed = 1;
   } else {
      utMemSeed = seed;
   }
}

/*--------------------------------------------------------------------------------------------------
  Return a random number between 1 and 2^23 - 1. Specific to
             utMem functions so that users utSeed is not affected.
--------------------------------------------------------------------------------------------------*/
U32 UTEXPORT utMemRand (void)
{
   utMemSeed = (((((utMemSeed << 5) ^ utMemSeed) & 0x7FFF80) >> 7) |
                (utMemSeed << 16)) & 0x7FFFFF;
   return utMemSeed;
}

/*--------------------------------------------------------------------------------------------------
  Create a new mem, filling in all fields.  Note that the calling
             routine is responsible for putting the pickets in memory.
--------------------------------------------------------------------------------------------------*/
utMemRef UTEXPORT utBuildMem (
   void *memPtr,
   U32 size,
   bool farHeap,
   char *file,
   U32 line)
{
   utMemRef mem = utcMem();
   utSym sym = utSymCreate(file);

   utgMemPtr(mem) = memPtr;
   utgMemSize(mem) = size;
   utUsedMem += size;
   if (utUsedMem > utMaxMem) {
      utMaxMem = utUsedMem;
   }
   utgMemLine(mem) = line;
   utfMemNameSym(mem) = sym;
   utaStackMem(utlStack(), mem);
   utgMemPicket(mem) = (U8)utMemRand();
   *(((U8 *)memPtr) + size - 1) = utgMemPicket(mem);
   uttMemFarHeap(mem) = farHeap;
   return mem;
}

/*--------------------------------------------------------------------------------------------------
  Find a mem with the same memory pointer.
--------------------------------------------------------------------------------------------------*/
utMemRef UTEXPORT utqMemPtr (
   void *memPtr)
{
   utStackRef stack = utlStack();
   utMemRef mem;

   for (mem = utfStackMem(stack); uttMemExists(mem);
         mem = utnStackMem(stack, mem)) {
      if (utgMemPtr(mem) == memPtr) {
         return mem;
      }
   }
   return ut0Mem;
}

/*--------------------------------------------------------------------------------------------------
  Create a new stack.
--------------------------------------------------------------------------------------------------*/
utStackRef UTEXPORT utcStackTrace (
   char *fileName,
   U32 line)
{
   utStackRef stack;
   utSym sym = utSymCreate(fileName);

   if (utsStack() == utmStack()) {
      utmStack() += utmStack()/2;
      utStacks = (struct utStack_ *)mtRealloc((void *)utStacks,
            utmStack(), sizeof(struct utStack_));
      if (!utStacks) {
         utExit("utcStackTrace: Can't allocate a new stack");
      }
   }
   stack = utsStack()++;
   utfStackMem(stack) = ut0Mem;
   utfStackNameSym(stack) = sym;
   utgStackLine(stack) = line;
   return stack;
}

/*--------------------------------------------------------------------------------------------------
  Delete a stack.  Check for memory errors.
--------------------------------------------------------------------------------------------------*/
void UTEXPORT utdStackTrace (
   utStackRef stack,
   char *fileName,
   U32 line)
{
   utMemRef mem;
   bool error = false;

   for (mem = utfStackMem(stack); uttMemExists(mem);
         mem = utnStackMem(stack, mem)) {
      error = true;
      utExit("UT0001: utdStack: Failed to free memory allocated in %s on line %u",
            utSymGetName(utfMemNameSym(mem)), utgMemLine(mem));
   }
   if (stack != utlStack()) {
      error = true;
      utExit("UT0002: utdStack: Stack allocated in %s on line %u is freed out of "
            "order in %s on line %u", utSymGetName(utfStackNameSym(stack)),
            utgStackLine(stack), fileName, line);
   }
   if (error) {
      utExit("utdStack: Punting due to memory errors");
   }
   utAssert(utsStack() > 0);
   utsStack()--;
}

/*--------------------------------------------------------------------------------------------------
  Add a mem to a stack.
--------------------------------------------------------------------------------------------------*/
void utaStackMem (
   utStackRef stack,
   utMemRef mem)
{
   utoStackMem(mem) = stack;
   utnStackMem(stack, mem) = utfStackMem(stack);
   utfStackMem(stack) = mem;
}

/*--------------------------------------------------------------------------------------------------
  Remove a mem from a stack.
--------------------------------------------------------------------------------------------------*/
void utdStackMem (
   utStackRef stack,
   utMemRef mem,
   char *fileName,
   U32 line)
{
   utMemRef pMem, nMem;

   pMem = ut0Mem;
   for (nMem = utfStackMem(stack);
        uttMemExists(nMem) && nMem != mem;
        nMem = utnStackMem(stack, nMem)) {
      pMem = nMem;
   }
   if (nMem != mem) {
      utExit("utdStackMem: Memory allocated in %s on line %u does not "
            "belong to the stack allocated in %s on line %u.  Look at "
            "%s line %u", utSymGetName(utfMemNameSym(mem)), utgMemLine(mem),
            utSymGetName(utfStackNameSym(stack)), utgStackLine(stack),
            fileName, line);
   }
   if (uttMemExists(pMem)) {
      utnStackMem(stack, pMem) = utnStackMem(stack, mem);
   } else {
      utfStackMem(stack) = utnStackMem(stack, mem);
   }
}

/*--------------------------------------------------------------------------------------------------
  Allocate initail memory for mems.
--------------------------------------------------------------------------------------------------*/
static void allocateMems (void)
{
   utfFreeMem() = ut0Mem;
   utfVirgMem() = 0;
   utmMem() = 100;
   utMems = (struct utMem_ *)mtCalloc(utmMem(), sizeof(struct utMem_));
   if (!utMems) {
      utExit("allocateMems: Out of memory");
   }
}

/*--------------------------------------------------------------------------------------------------
  Allocate initail memory for stacks.
--------------------------------------------------------------------------------------------------*/
static void allocateStacks (void)
{
   utsStack() = 0;
   utmStack() = 100;
   utStacks = (struct utStack_ *)mtCalloc(utmStack(),
         sizeof(struct utStack_));
   if (!utStacks) {
      utExit("allocateStacks: Out of memory");
   }
}

/*--------------------------------------------------------------------------------------------------
  Initialize the memory checker module;
--------------------------------------------------------------------------------------------------*/
void UTEXPORT utMemInit (void)
{
   allocateMems();
   allocateStacks();
   utBotStack = utcStack();
   utUsedMem = 0;
   utMaxMem = 0;
}

/*--------------------------------------------------------------------------------------------------
  Free memory used by the memory checker.
--------------------------------------------------------------------------------------------------*/
void UTEXPORT utMemClose (void)
{
   utdStack(utBotStack);
   if (!mtFree((void *)utMems)) {
      utExit("utCloseMem: mems");
   }
   if (!mtFree((void *)utStacks)) {
      utExit("utCloseMem: stacks");
   }
   if (utMaxMem > (U32) 1024*1024) {
       utLogMessage("Used %.5g MB of memory", utMaxMem/(1024.0*1024.0));
   } else if (utMaxMem > (U32) 1024) {
       utLogMessage("Used %.5g KB of memory", utMaxMem/1024.0);
   } else {
       utLogMessage("Used %lu bytes of memory", utMaxMem);
   }
}

/*--------------------------------------------------------------------------------------------------
  Perform a memory check.  Basically, check all the pickets.
--------------------------------------------------------------------------------------------------*/
void UTEXPORT utMemCheckTrace (
   char *fileName,
   U32 line)
{
   utStackRef stack;
   utMemRef mem;
   U8 picket;

   for (stack = utfStack(); uttStackExists(stack); stack = utnStack(stack)) {
      for (mem = utfStackMem(stack); uttMemExists(mem);
            mem = utnStackMem(stack, mem)) {
         picket = *(((U8 *)utgMemPtr(mem)) + utgMemSize(mem) - 1);
         if (picket != utgMemPicket(mem)) {
            utExit("utMemCheck: Invoked from %s, line %u: Picket hosed for "
                  "memory allocated in %s, " "line %u", fileName, line,
                  utSymGetName(utfMemNameSym(mem)), utgMemLine(mem));
         }
      }
   }
}

