/*
 * Copyright (C) 2003 ViASIC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */

#include <ctype.h>
#include <string.h>
#include "util.h"

struct _utHtblRootType _utHtblRoot;

/*--------------------------------------------------------------------------------------------------
Find the first Htbl in the database.
--------------------------------------------------------------------------------------------------*/
utHtblRef utfHtbl (void)
{
    utHtblRef Htbl, nHtbl;
    
    for (Htbl = _utfUsedHtbl(); uttHtblExists(Htbl); Htbl = nHtbl) {
        if (_uttHtblUsed(Htbl)) {
            return Htbl;
        }
        nHtbl = _utnHtbl(Htbl);
        _utfUsedHtbl() = _utnHtbl(Htbl);
        _utnHtbl(Htbl) = _utfFreeHtbl();
        _utfFreeHtbl() = Htbl;
        _utsUnusedHtbl()--;
        if (_utlUsedHtbl() == Htbl) {
            _utlUsedHtbl() = ut0Htbl;
        }
    }
    return ut0Htbl;
} /* utfHtbl */

/*--------------------------------------------------------------------------------------------------
  Find the next Htbl in the database.
--------------------------------------------------------------------------------------------------*/
utHtblRef utnHtbl (
    utHtblRef Htbl)
{
    utHtblRef pHtbl = Htbl, nHtbl;
    
    for (Htbl = _utnHtbl(Htbl); uttHtblExists(Htbl); Htbl = nHtbl) {
        if (_uttHtblUsed(Htbl)) {
            return Htbl;
        }
        nHtbl = _utnHtbl(Htbl);
        _utnHtbl(pHtbl) = nHtbl;
        _utnHtbl(Htbl) = _utfFreeHtbl();
        _utfFreeHtbl() = Htbl;
        _utsUnusedHtbl()--;
        if (_utlUsedHtbl() == Htbl) {
            _utlUsedHtbl() = pHtbl;
        }
    }
    return ut0Htbl;
} /* utnHtbl */

/*--------------------------------------------------------------------------------------------------
  Allocate a new memory block for the Htbl.
--------------------------------------------------------------------------------------------------*/
static utBlockRef utBuildHtblBlock (void)
{
    utBlockRef block = utcBlock();
    utHtblRef Htbl;
    U16 x, sHtbl;
    
    utaHeapBlock(_utHtblRoot.HtblHeap, block);
    sHtbl = UTHEAPSIZE/sizeof(struct _utHtbl);
    _utfFreeHtbl() =(struct _utHtbl *)utgBlockMem(block);
    Htbl = _utfFreeHtbl();
    for (x = 0; x < sHtbl; x++) {
        _utnHtbl(Htbl) = Htbl + 1;
        Htbl++;
    }
    _utnHtbl(--Htbl) = ut0Htbl;
    return block;
} /* utBuildHtblBlock */

/*--------------------------------------------------------------------------------------------------
  Create a new Htbl.
--------------------------------------------------------------------------------------------------*/
utHtblRef utcHtbl (void)
{
    utHtblRef Htbl;
    
    if (!uttHtblExists(_utfFreeHtbl())) {
        if (_utsUnusedHtbl() > ((utsHeapBlock(_utHtblRoot.HtblHeap) >> 3) + 1)*
            UTHEAPSIZE/sizeof(struct _utHtbl)) {
            for (Htbl = utfHtbl(); uttHtblExists(Htbl); Htbl = utnHtbl(Htbl));
        } else {
            utBuildHtblBlock();
        }
    }
    Htbl = _utfFreeHtbl();
    _utfFreeHtbl() = _utnHtbl(Htbl);
    if (uttHtblExists(_utlUsedHtbl())) {
        _utnHtbl(_utlUsedHtbl()) = Htbl;
    } else {
        _utfUsedHtbl() = Htbl;
    }
    _utlUsedHtbl() = Htbl;
    _utnHtbl(Htbl) = ut0Htbl;
    _uttHtblUsed(Htbl) = true;
    utfHtblArray(Htbl) = ut0Array;
    utfHtblEntry(Htbl) = ut0Entry;
    utlHtblEntry(Htbl) = ut0Entry;
    utgHtblNameCounter(Htbl) = 1;
    return Htbl;
} /* utcHtbl */

/*--------------------------------------------------------------------------------------------------
  Delete the Htbl.
--------------------------------------------------------------------------------------------------*/
void utdHtbl (
    utHtblRef Htbl)
{
    _uttHtblUsed(Htbl) = false;
    _utsUnusedHtbl()++;
} /* utdHtbl */

/*--------------------------------------------------------------------------------------------------
  Find the first Entry in the database.
--------------------------------------------------------------------------------------------------*/
utEntryRef utfEntry (void)
{
    utEntryRef Entry, nEntry;
    
    for (Entry = _utfUsedEntry(); uttEntryExists(Entry); Entry = nEntry) {
        if (_uttEntryUsed(Entry)) {
            return Entry;
        }
        nEntry = _utnEntry(Entry);
        _utfUsedEntry() = _utnEntry(Entry);
        _utnEntry(Entry) = _utfFreeEntry();
        _utfFreeEntry() = Entry;
        _utsUnusedEntry()--;
        if (_utlUsedEntry() == Entry) {
            _utlUsedEntry() = ut0Entry;
        }
    }
    return ut0Entry;
} /* utfEntry */

/*--------------------------------------------------------------------------------------------------
  Find the next Entry in the database.
--------------------------------------------------------------------------------------------------*/
utEntryRef utnEntry (
    utEntryRef Entry)
{
    utEntryRef pEntry = Entry, nEntry;
    
    for (Entry = _utnEntry(Entry); uttEntryExists(Entry); Entry = nEntry) {
        if (_uttEntryUsed(Entry)) {
            return Entry;
        }
        nEntry = _utnEntry(Entry);
        _utnEntry(pEntry) = nEntry;
        _utnEntry(Entry) = _utfFreeEntry();
        _utfFreeEntry() = Entry;
        _utsUnusedEntry()--;
        if (_utlUsedEntry() == Entry) {
            _utlUsedEntry() = pEntry;
        }
    }
    return ut0Entry;
} /* utnEntry */

/*--------------------------------------------------------------------------------------------------
  Allocate a new memory block for the Entry.
--------------------------------------------------------------------------------------------------*/
static utBlockRef utBuildEntryBlock (void)
{
    utBlockRef block = utcBlock();
    utEntryRef Entry;
    U16 x, sEntry;
    
    utaHeapBlock(_utHtblRoot.EntryHeap, block);
    sEntry = UTHEAPSIZE/sizeof(struct _utEntry);
    _utfFreeEntry() =(struct _utEntry *)utgBlockMem(block);
    Entry = _utfFreeEntry();
    for (x = 0; x < sEntry; x++) {
        _utnEntry(Entry) = Entry + 1;
        Entry++;
    }
    _utnEntry(--Entry) = ut0Entry;
    return block;
} /* utBuildEntryBlock */

/*--------------------------------------------------------------------------------------------------
  Create a new Entry.
--------------------------------------------------------------------------------------------------*/
utEntryRef utcEntry (void)
{
    utEntryRef Entry;
    
    if (!uttEntryExists(_utfFreeEntry())) {
        if (_utsUnusedEntry() > ((utsHeapBlock(_utHtblRoot.EntryHeap) >> 3) + 1)*
            UTHEAPSIZE/sizeof(struct _utEntry)) {
            for (Entry = utfEntry(); uttEntryExists(Entry); Entry = utnEntry(Entry));
            utAssert(uttEntryExists(_utfFreeEntry()) && _utsUnusedEntry() == 0);
        } else {
            utBuildEntryBlock();
            utAssert(uttEntryExists(_utfFreeEntry()));
        }
    }
    Entry = _utfFreeEntry();
    _utfFreeEntry() = _utnEntry(Entry);
    if (uttEntryExists(_utlUsedEntry())) {
        _utnEntry(_utlUsedEntry()) = Entry;
    } else {
        _utfUsedEntry() = Entry;
    }
    _utlUsedEntry() = Entry;
    _utnEntry(Entry) = ut0Entry;
    _uttEntryUsed(Entry) = true;
    utfEntryEntry(Entry) = ut0Entry;
    utoHtblEntry(Entry) = ut0Htbl;
    utgEntrySym(Entry) = utSymNull;
    return Entry;
} /* utcEntry */

/*--------------------------------------------------------------------------------------------------
  Delete the Entry.
--------------------------------------------------------------------------------------------------*/
void utdEntry (
    utEntryRef Entry)
{
    utAssert(_uttEntryUsed(Entry));
    _uttEntryUsed(Entry) = false;
    _utsUnusedEntry()++;
} /* utdEntry */

/*--------------------------------------------------------------------------------------------------
  Find the first Array in the database.
--------------------------------------------------------------------------------------------------*/
utArrayRef utfArray (void)
{
    utArrayRef Array, nArray;
    
    for (Array = _utfUsedArray(); uttArrayExists(Array); Array = nArray) {
        if (_uttArrayUsed(Array)) {
            return Array;
        }
        nArray = _utnArray(Array);
        _utfUsedArray() = _utnArray(Array);
        _utnArray(Array) = _utfFreeArray();
        _utfFreeArray() = Array;
        _utsUnusedArray()--;
        if (_utlUsedArray() == Array) {
            _utlUsedArray() = ut0Array;
        }
    }
    return ut0Array;
} /* utfArray */

/*--------------------------------------------------------------------------------------------------
  Find the next Array in the database.
--------------------------------------------------------------------------------------------------*/
utArrayRef utnArray (
    utArrayRef Array)
{
    utArrayRef pArray = Array, nArray;
    
    for (Array = _utnArray(Array); uttArrayExists(Array); Array = nArray) {
        if (_uttArrayUsed(Array)) {
            return Array;
        }
        nArray = _utnArray(Array);
        _utnArray(pArray) = nArray;
        _utnArray(Array) = _utfFreeArray();
        _utfFreeArray() = Array;
        _utsUnusedArray()--;
        if (_utlUsedArray() == Array) {
            _utlUsedArray() = pArray;
        }
    }
    return ut0Array;
} /* utnArray */

/*--------------------------------------------------------------------------------------------------
  Allocate space for the Entry relationship
  array of the Array.
--------------------------------------------------------------------------------------------------*/
static U32 allocArrayEntrys (
    U32 sEntry)
{
    utArrayRef Array;
    utEntryRef *Entry1;
    utEntryRef *Entry2;
    U32 sArrayEntry, ret;
    
    if (_utmArrayEntry() < _utsArrayEntry() + sEntry) {
        Entry1 = _utHtblRoot.ArrayEntrys;
        for (Array = utfArray(); uttArrayExists(Array); Array = utnArray(Array)) {
            Entry2 = _utHtblRoot.ArrayEntrys + _utfxArrayEntry(Array);
            _utfxArrayEntry(Array) = Entry1 - _utHtblRoot.ArrayEntrys;
            for (sArrayEntry = utsArrayEntry(Array); sArrayEntry; sArrayEntry--) {
                *Entry1++ = *Entry2++;
            }
        }
        _utsArrayEntry() = Entry1 - _utHtblRoot.ArrayEntrys;
        if (_utmArrayEntry()*3 < (_utsArrayEntry() + sEntry)*4) {
            _utmArrayEntry() += _utmArrayEntry()/2 + sEntry;
            _utHtblRoot.ArrayEntrys = (utEntryRef *)utRealloc((void *)
                _utHtblRoot.ArrayEntrys, _utmArrayEntry(), sizeof(utEntryRef));
            Entry1 = _utHtblRoot.ArrayEntrys + _utsArrayEntry();
        }
        while (Entry1 < _utHtblRoot.ArrayEntrys + _utmArrayEntry()) {
            *Entry1++ = ut0Entry;
        }
    }
    ret = _utsArrayEntry();
    _utsArrayEntry() += sEntry;
    return ret;
} /* allocArrayEntrys */

/*--------------------------------------------------------------------------------------------------
  Allocate a new memory block for the Array.
--------------------------------------------------------------------------------------------------*/
static utBlockRef utBuildArrayBlock (void)
{
    utBlockRef block = utcBlock();
    utArrayRef Array;
    U16 x, sArray;
    
    utaHeapBlock(_utHtblRoot.ArrayHeap, block);
    sArray = UTHEAPSIZE/sizeof(struct _utArray);
    _utfFreeArray() =(struct _utArray *)utgBlockMem(block);
    Array = _utfFreeArray();
    for (x = 0; x < sArray; x++) {
        _utnArray(Array) = Array + 1;
        Array++;
    }
    _utnArray(--Array) = ut0Array;
    return block;
} /* utBuildArrayBlock */

/*--------------------------------------------------------------------------------------------------
  Create a new Array.
--------------------------------------------------------------------------------------------------*/
utArrayRef utcArray (
    U32 sEntry)
{
    utArrayRef Array;
    
    if (!uttArrayExists(_utfFreeArray())) {
        if (_utsUnusedArray() > ((utsHeapBlock(_utHtblRoot.ArrayHeap) >> 3) + 1)*
            UTHEAPSIZE/sizeof(struct _utArray)) {
            for (Array = utfArray(); uttArrayExists(Array); Array = utnArray(Array));
        } else {
            utBuildArrayBlock();
        }
    }
    Array = _utfFreeArray();
    _utfFreeArray() = _utnArray(Array);
    if (uttArrayExists(_utlUsedArray())) {
        _utnArray(_utlUsedArray()) = Array;
    } else {
        _utfUsedArray() = Array;
    }
    _utlUsedArray() = Array;
    _utnArray(Array) = ut0Array;
    _uttArrayUsed(Array) = true;
    utsArrayEntry(Array) = 0;
    _utfxArrayEntry(Array) = allocArrayEntrys(sEntry);
    utsArrayEntry(Array) = sEntry;
    return Array;
} /* utcArray */

/*--------------------------------------------------------------------------------------------------
  Delete the Array.
--------------------------------------------------------------------------------------------------*/
void utdArray (
    utArrayRef Array)
{
    _uttArrayUsed(Array) = false;
    _utsUnusedArray()++;
} /* utdArray */

/*--------------------------------------------------------------------------------------------------
   Function: utfDynarray
   Purpose : Find the first Dynarray in the database.
--------------------------------------------------------------------------------------------------*/
utDynarrayRef utfDynarray (void)
{
   utDynarrayRef Dynarray, nDynarray;

   for (Dynarray = _utfUsedDynarray(); uttDynarrayExists(Dynarray); Dynarray = nDynarray) {
      if (_uttDynarrayUsed(Dynarray)) {
         return Dynarray;
      }
      nDynarray = _utnDynarray(Dynarray);
      _utfUsedDynarray() = _utnDynarray(Dynarray);
      _utnDynarray(Dynarray) = _utfFreeDynarray();
      _utfFreeDynarray() = Dynarray;
      _utsUnusedDynarray()--;
      if (_utlUsedDynarray() == Dynarray) {
         _utlUsedDynarray() = ut0Dynarray;
      }
   }
   return ut0Dynarray;
} /* utfDynarray */

/*--------------------------------------------------------------------------------------------------
   Function: utnDynarray
   Purpose : Find the next Dynarray in the database.
--------------------------------------------------------------------------------------------------*/
utDynarrayRef utnDynarray (
   utDynarrayRef Dynarray)
{
   utDynarrayRef pDynarray = Dynarray, nDynarray;

   for (Dynarray = _utnDynarray(Dynarray); uttDynarrayExists(Dynarray); Dynarray = nDynarray) {
      if (_uttDynarrayUsed(Dynarray)) {
         return Dynarray;
      }
      nDynarray = _utnDynarray(Dynarray);
      _utnDynarray(pDynarray) = nDynarray;
      _utnDynarray(Dynarray) = _utfFreeDynarray();
      _utfFreeDynarray() = Dynarray;
      _utsUnusedDynarray()--;
      if (_utlUsedDynarray() == Dynarray) {
         _utlUsedDynarray() = pDynarray;
      }
   }
   return ut0Dynarray;
} /* utnDynarray */

/*--------------------------------------------------------------------------------------------------
   Function: utBuildDynarrayBlock
   Purpose : Allocate a new memory block for the Dynarray.
--------------------------------------------------------------------------------------------------*/
static utBlockRef utBuildDynarrayBlock (void)
{
   utBlockRef block = utcBlock();
   utDynarrayRef Dynarray;
   U16 x, sDynarray;

   utaHeapBlock(_utHtblRoot.DynarrayHeap, block);
   sDynarray = UTHEAPSIZE/sizeof(struct _utDynarray);
   _utfFreeDynarray() =(struct _utDynarray *)utgBlockMem(block);
   Dynarray = _utfFreeDynarray();
   for (x = 0; x < sDynarray; x++) {
      _utnDynarray(Dynarray) = Dynarray + 1;
      Dynarray++;
   }
   _utnDynarray(--Dynarray) = ut0Dynarray;
   return block;
} /* utBuildDynarrayBlock */

/*--------------------------------------------------------------------------------------------------
   Function: utcDynarray_
   Purpose : Create a new Dynarray.
--------------------------------------------------------------------------------------------------*/
utDynarrayRef utcDynarray_ (void)
{
   utDynarrayRef Dynarray;

   if (!uttDynarrayExists(_utfFreeDynarray())) {
      if (_utsUnusedDynarray() > ((utsHeapBlock(_utHtblRoot.DynarrayHeap) >> 3) + 1)*
            UTHEAPSIZE/sizeof(struct _utDynarray)) {
         for (Dynarray = utfDynarray(); uttDynarrayExists(Dynarray);
              Dynarray = utnDynarray(Dynarray));
      } else {
         utBuildDynarrayBlock();
      }
   }
   Dynarray = _utfFreeDynarray();
   _utfFreeDynarray() = _utnDynarray(Dynarray);
   if (uttDynarrayExists(_utlUsedDynarray())) {
      _utnDynarray(_utlUsedDynarray()) = Dynarray;
   } else {
      _utfUsedDynarray() = Dynarray;
   }
   _utlUsedDynarray() = Dynarray;
   _utnDynarray(Dynarray) = ut0Dynarray;
   _uttDynarrayUsed(Dynarray) = true;
   utrDynarraySize(Dynarray, 0);
   utrDynarrayValueSize(Dynarray, 0);
   utrDynarrayValues(Dynarray, void, NULL);
   utrDynarrayNumUsed(Dynarray, 0);
   return Dynarray;
} /* utcDynarray */

/*--------------------------------------------------------------------------------------------------
   Function: utBuildDynarray
   Purpose : Build a new dynamic array.
--------------------------------------------------------------------------------------------------*/
utDynarrayRef utBuildDynarray_(
    U16 valueSize)
{
    utDynarrayRef dynarray = utcDynarray_();

    utrDynarrayValueSize(dynarray, valueSize);
    return dynarray;
}

/*--------------------------------------------------------------------------------------------------
   Function: utdDynarray
   Purpose : Delete the Dynarray.
--------------------------------------------------------------------------------------------------*/
void utdDynarray (
   utDynarrayRef Dynarray)
{
    utAssert(_uttDynarrayUsed(Dynarray));
    if(utgDynarrayValues_(Dynarray) != NULL) {
        mtFree(utgDynarrayValues_(Dynarray));
        utUsedMem -= utgDynarraySize(Dynarray);
    }
   _uttDynarrayUsed(Dynarray) = false;
   _utsUnusedDynarray()++;
} /* utdDynarray */

/*--------------------------------------------------------------------------------------------------
  Free memory used by the ut module.
--------------------------------------------------------------------------------------------------*/
void utHtblClose (void)
{
    uttRootInUse() = false;
    utFreeHeap(_utHtblRoot.DynarrayHeap);
    utFreeHeap(_utHtblRoot.HtblHeap);
    utFreeHeap(_utHtblRoot.EntryHeap);
    utFreeHeap(_utHtblRoot.ArrayHeap);
    if (!utFree((void *)_utHtblRoot.ArrayEntrys)) {
        utExit("freeArrayEntrys");
    }
} /* utHtblClose */

/*--------------------------------------------------------------------------------------------------
  Allocate initial memory for Htbls.
--------------------------------------------------------------------------------------------------*/
static void allocHtbls (void)
{
    _utHtblRoot.HtblHeap = utcHeap();
    _utlUsedHtbl() = ut0Htbl;
    _utfUsedHtbl() = ut0Htbl;
    utBuildHtblBlock();
} /* allocHtbls */

/*--------------------------------------------------------------------------------------------------
  Allocate initial memory for Entrys.
--------------------------------------------------------------------------------------------------*/
static void allocEntrys (void)
{
    _utHtblRoot.EntryHeap = utcHeap();
    _utlUsedEntry() = ut0Entry;
    _utfUsedEntry() = ut0Entry;
    utBuildEntryBlock();
} /* allocEntrys */

/*--------------------------------------------------------------------------------------------------
  Allocate initial memory for Arrays.
--------------------------------------------------------------------------------------------------*/
static void allocArrays (void)
{
    _utHtblRoot.ArrayHeap = utcHeap();
    _utlUsedArray() = ut0Array;
    _utfUsedArray() = ut0Array;
    utBuildArrayBlock();
    _utsArrayEntry() = 0;
    _utmArrayEntry() = 42;
    _utHtblRoot.ArrayEntrys = (utEntryRef *)utMalloc(_utmArrayEntry(), sizeof(utEntryRef));
    _utsArrayEntry() = 0;
    {
        utEntryRef *Entry = _utHtblRoot.ArrayEntrys;
        
        while (Entry < _utHtblRoot.ArrayEntrys + _utmArrayEntry()) {
            *Entry++ = ut0Entry;
        }
    }
} /* allocArrays */

/*--------------------------------------------------------------------------------------------------
   Function: allocDynarrays
   Purpose : Allocate initial memory for Dynarrays.
--------------------------------------------------------------------------------------------------*/
static void allocDynarrays (void)
{
   _utHtblRoot.DynarrayHeap = utcHeap();
   _utlUsedDynarray() = ut0Dynarray;
   _utfUsedDynarray() = ut0Dynarray;
   utBuildDynarrayBlock();
} /* allocDynarrays */

/*--------------------------------------------------------------------------------------------------
  Allocate memory used by the ut module.
--------------------------------------------------------------------------------------------------*/
void utHtblInit (void)
{
    if (uttRootInUse()) {
        utHtblClose();
    }
    uttRootInUse() = true;
    allocHtbls();
    allocEntrys();
    allocArrays();
    allocDynarrays();
} /* utHtblInit */

/*--------------------------------------------------------------------------------------------------
  Initialize a hash table.
--------------------------------------------------------------------------------------------------*/
void utInitHtbl(
    utHtblRef htbl)
{
    U32 sizeExp = 2; /* 4 entries to start with */
    U32 size = 1 << sizeExp;
    utArrayRef array = utcArray(size);
    
    utrHtblSizeExp(htbl, sizeExp);
    utrfHtblArray(htbl, array);
}

/*--------------------------------------------------------------------------------------------------
  Build a new hash table, with it's array.
--------------------------------------------------------------------------------------------------*/
utHtblRef utBuildHtbl(void)
{
    utHtblRef htbl = utcHtbl();

    utInitHtbl(htbl);
    return htbl;
}

/*--------------------------------------------------------------------------------------------------
  Destroy an empty hash table, along with its array.
--------------------------------------------------------------------------------------------------*/
void utRipHtbl(
    utHtblRef htbl)
{
    /* User must empty hash table first */
    utAssert(!uttEntryExists(utfHtblEntry(htbl)));
    utdArray(utfHtblArray(htbl));
    utdHtbl(htbl);
}

/*--------------------------------------------------------------------------------------------------
  Find an entry in the hash table with the given name.
--------------------------------------------------------------------------------------------------*/
utEntryRef utqEntryHtblSym(
    utHtblRef htbl,
    utSym sym)
{
    U32 mask = (1 << utgHtblSizeExp(htbl)) - 1;
    U32 hashValue = utSymGetHashValue(sym);
    utEntryRef entry = utxArrayEntry(utfHtblArray(htbl), hashValue & mask);
    
    while(uttEntryExists(entry)) {
        if(utgEntrySym(entry) == sym) {
            return entry;
        }
        entry = utfEntryEntry(entry);
    }
    return ut0Entry;
}

/*--------------------------------------------------------------------------------------------------
  Find a value in the hash table with the given name.
--------------------------------------------------------------------------------------------------*/
U32 utFindInHtbl(
    utHtblRef htbl,
    utSym sym)
{
    utEntryRef entry = utqEntryHtblSym(htbl, sym);
    
    if(uttEntryExists(entry)) {
        return utgEntryData(entry);
    }
    return U32_MAX;
}
              
/*--------------------------------------------------------------------------------------------------
  Make a new hash table array twice as large, and copy from the
  old one to the new one.
--------------------------------------------------------------------------------------------------*/
static void resizeHtbl(
   utHtblRef htbl)
{
   U16 sizeExp = utgHtblSizeExp(htbl);
   U32 oldSize = 1 << sizeExp;           
   U32 newSize = 1 << (sizeExp + 1);
   U32 mask = newSize - 1;
   U32 hashValue;
   utArrayRef newArray = utcArray(newSize);       
   utArrayRef oldArray = utfHtblArray(htbl);
   utEntryRef entry, nEntry;
   U32 xOldArray, xNewArray;
   
   utrfHtblArray(htbl, newArray);
   utrHtblSizeExp(htbl, sizeExp + 1);
   for (xOldArray = 0; xOldArray < oldSize; xOldArray++) {
      for (entry = utxArrayEntry(oldArray, xOldArray);
            uttEntryExists(entry); entry = nEntry) {
         nEntry = utfEntryEntry(entry);
         hashValue = utSymGetHashValue(utgEntrySym(entry));
         xNewArray = hashValue & mask;
         utrfEntryEntry(entry, utxArrayEntry(newArray, xNewArray));
         utrxArrayEntry(newArray, xNewArray, entry);
      }
   }
   utdArray(oldArray);
}

/*--------------------------------------------------------------------------------------------------
  Add an entry to the htbl.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
--------------------------------------------------------------------------------------------------*/
static void addHtblEntry(
    utHtblRef htbl,
    utEntryRef entry)
{
    utEntryRef nEntry;
    U32 mask, index, hashValue;
    utArrayRef array;

    utAssert(!uttEntryExists(utfEntryEntry(entry)));
    if (utgHtblNumEntries(htbl) > (U32)(1 << (utgHtblSizeExp(htbl) - 1))) {
        resizeHtbl(htbl);
    }
    mask = (1 << utgHtblSizeExp(htbl)) - 1;
    hashValue = utSymGetHashValue(utgEntrySym(entry));
    index = hashValue & mask;
    array = utfHtblArray(htbl);
    nEntry = utxArrayEntry(array, index);
    utAssert(!uttEntryExists(nEntry) || utfEntryEntry(nEntry) != entry);
    utrfEntryEntry(entry, nEntry);
    utrxArrayEntry(array, index, entry);
    utrHtblNumEntries(htbl, utgHtblNumEntries(htbl) + 1);
}

/*--------------------------------------------------------------------------------------------------
  Add an entry to the htbl.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.  Otherwise, return the
  current hash table.
--------------------------------------------------------------------------------------------------*/
void utInsertHtblEntry(
    utHtblRef htbl,
    utEntryRef entry)
{
    utAssert(!uttHtblExists(utoHtblEntry(entry)));
    if(utgEntrySym(entry) != utSymNull) {
        addHtblEntry(htbl, entry);
    }
    utroHtblEntry(entry, htbl);
    if(uttEntryExists(utfHtblEntry(htbl))) {
        utnHtblEntry(entry) = utfHtblEntry(htbl);
        utpHtblEntry(utfHtblEntry(htbl)) = entry;
    } else {
        utlHtblEntry(htbl) = entry;
        utnHtblEntry(entry) = ut0Entry;
    }
    utfHtblEntry(htbl) = entry;
    utpHtblEntry(entry) = ut0Entry;
}

/*--------------------------------------------------------------------------------------------------
  Append an entry to the htbl.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.  Otherwise, return the
  current hash table.
--------------------------------------------------------------------------------------------------*/
void utAppendHtblEntry(
    utHtblRef htbl,
    utEntryRef entry)
{
    utAssert(!uttHtblExists(utoHtblEntry(entry)));
    if(utgEntrySym(entry) != utSymNull) {
        addHtblEntry(htbl, entry);
    }
    utroHtblEntry(entry, htbl);
    if(uttEntryExists(utlHtblEntry(htbl))) {
        utnHtblEntry(utlHtblEntry(htbl)) = entry;
        utpHtblEntry(entry) = utlHtblEntry(htbl);
    } else {
        utfHtblEntry(htbl) = entry;
        utpHtblEntry(entry) = ut0Entry;
    }
    utlHtblEntry(htbl) = entry;
    utnHtblEntry(entry) = ut0Entry;
}

/*--------------------------------------------------------------------------------------------------
  Remove an entry from the hash table.
--------------------------------------------------------------------------------------------------*/
static void deleteHtblEntry (
   utHtblRef htbl,
   utEntryRef entry)
{
    U32 mask = (1 << utgHtblSizeExp(htbl)) - 1;
    U32 hashValue = utSymGetHashValue(utgEntrySym(entry));
    U32 index = hashValue & mask;
    utArrayRef array = utfHtblArray(htbl);
    utEntryRef pEntry, nEntry;
    
    utAssert(utoHtblEntry(entry) == htbl);
    nEntry = utxArrayEntry(array, index);
    if (nEntry == entry) {
        utrxArrayEntry(array, index, utfEntryEntry(nEntry));
    } else {            
        do {
            pEntry = nEntry;
            nEntry = utfEntryEntry(nEntry);
        } while (nEntry != entry);        
        utrfEntryEntry(pEntry, utfEntryEntry(entry));
    }
    utrHtblNumEntries(htbl, utgHtblNumEntries(htbl) - 1);
    utrfEntryEntry(entry, ut0Entry);
}

/*--------------------------------------------------------------------------------------------------
  Remove an entry from the hash table.
--------------------------------------------------------------------------------------------------*/
void utDeleteHtblEntry (
   utHtblRef htbl,
   utEntryRef entry)
{
    if(utgEntrySym(entry) != utSymNull) {
        deleteHtblEntry(htbl, entry);
    }
    if(uttEntryExists(utpHtblEntry(entry))) {
        utnHtblEntry(utpHtblEntry(entry)) = utnHtblEntry(entry);
    } else {
        utfHtblEntry(htbl) = utnHtblEntry(entry);
    }
    if(uttEntryExists(utnHtblEntry(entry))) {
        utpHtblEntry(utnHtblEntry(entry)) = utpHtblEntry(entry);
    } else {
        utlHtblEntry(htbl) = utpHtblEntry(entry);
    }
    utpHtblEntry(entry) = ut0Entry;
    utnHtblEntry(entry) = ut0Entry;
    utoHtblEntry(entry) = ut0Htbl;
}

/*--------------------------------------------------------------------------------------------------
  Build a new entry for a hash table.
--------------------------------------------------------------------------------------------------*/
utEntryRef utBuildEntry(
    U32 data)
{
    utEntryRef entry = utcEntry();

    utrEntrySym(entry, utSymNull);
    utrEntryData(entry, data);
    utnHtblEntry(entry) = ut0Entry;
    utpHtblEntry(entry) = ut0Entry;
    return entry;
}

/*--------------------------------------------------------------------------------------------------
  Delete an entry, removing it from its hash table.
--------------------------------------------------------------------------------------------------*/
void utRipEntry(
    utEntryRef entry)
{
    utHtblRef htbl = utoHtblEntry(entry);

    if(uttHtblExists(htbl)) {
        utDeleteHtblEntry(htbl, entry);
    }
    utdEntry(entry);
}

/*--------------------------------------------------------------------------------------------------
  Resize the dynamic array.
--------------------------------------------------------------------------------------------------*/
void utResizeDynarray(
    utDynarrayRef dynarray,
    U32 newSize)
{
    U32 *values;

    utAssert(newSize > 0);
    if(utgDynarraySize(dynarray) == 0) {
        values = mtMalloc(newSize, utgDynarrayValueSize(dynarray));
    } else {
        utUsedMem -= utgDynarraySize(dynarray);
        values = mtRealloc(utgDynarrayValues_(dynarray), newSize, sizeof(U32));
    }
    utUsedMem += newSize;
    utrDynarraySize(dynarray, newSize);
    utrDynarrayValues(dynarray, void, values);
}

/*--------------------------------------------------------------------------------------------------
  Make a unique name in the hash table.  Use the NameCounter field.
--------------------------------------------------------------------------------------------------*/
utSym utHtblMakeUniqueName(
    utHtblRef htbl,
    char *name)
{
    utSym sym = utSymCreate(name);
    U32 count = utgHtblNameCounter(htbl);

    if(utqEntryHtblSym(htbl, sym) == ut0Entry) {
        return sym;
    }
    do {
        sym = utSymCreateFormatted("%s%d", name, count++);
    } while(utqEntryHtblSym(htbl, sym) != ut0Entry);
    utgHtblNameCounter(htbl) = count;
    return sym;
}

/*--------------------------------------------------------------------------------------------------
  Determine if the name contains only letters, digits, and '_'.
--------------------------------------------------------------------------------------------------*/
static bool nameContainsNonAlnumChars(
    char *name)
{
    while(isalnum(*name) || *name == '_') {
        name++;
    }
    return *name != '\0';
}

/*--------------------------------------------------------------------------------------------------
  Replace all non-alphanumeric characters with '_', except for any trailing non-alnum char.
--------------------------------------------------------------------------------------------------*/
static char *mungeName(
    utHtblRef htbl,
    char *name)
{
    char *newName = utMakeString(strlen(name) + 42); /* Note: hard-wired 42 is bigger than needed */
    char *p = newName;
    char c;
    
    utDo {
        c = *name++;
    } utWhile(c != '\0') {
        if(isalnum(c) || c == '_') {
            *p++ = c;
        } else if(*name != '\0') {
            *p++ = '_'; /* Just leave off last non-alnum char */
        }
    } utRepeat;
    *p = '\0';
    return newName;
}

/*--------------------------------------------------------------------------------------------------
  Determine if the name contains only letters, digits, and '_'.
--------------------------------------------------------------------------------------------------*/
static void mungeEntryName(
    utHtblRef htbl,
    utEntryRef entry)
{
    char *name = utSymGetName(utgEntrySym(entry));
    char *newName = mungeName(htbl, name);

    utgEntrySym(entry) = utHtblMakeUniqueName(htbl, newName);
}

/*--------------------------------------------------------------------------------------------------
  Convert entry names in the hash table to contain only alpha-numeric letters, and '_'.
--------------------------------------------------------------------------------------------------*/
void utMungeHtblNames(
    utHtblRef htbl)
{
    utEntryRef entry;

    for(entry = utfHtblEntry(htbl); entry != ut0Entry; entry = utnHtblEntry(entry)) {
        if(utgEntrySym(entry) != utSymNull &&
                nameContainsNonAlnumChars(utSymGetName(utgEntrySym(entry)))) {
            deleteHtblEntry(htbl, entry);
            mungeEntryName(htbl, entry);
            addHtblEntry(htbl, entry);
        }
    }
}

