/*
 * Copyright (C) 2003 ViASIC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */

/*============================================================================
   Module : Type Definitions
   Purpose: Provide basic types used to build all objects.  These typedefs
            clarify what we mean when we use an integer type, and make the
            code more portable.
============================================================================*/

#ifndef UTTYPES_H
#define UTTYPES_H

#ifdef DDR_DEBUG
#define DDR_STRICT
#ifndef DDR_DEBUG_ORPHANS
  #define DDR_VALIDATE          /* select VALIDATE or DEBUG_ORPHANS, not both */
#endif
#endif

/* For all platforms */
typedef unsigned int UTUINT;
typedef int          UTINT;
#ifndef __cplusplus
/* Define C++ compatible Boolean types for use in C */
#define true 1
#define false 0
typedef char bool;
#define utIntToBool(x) ((bool)((x)? true : false))
#endif

#if defined(_WINDOWS) && !defined(_WIN32)
typedef unsigned long U32;    /* Unsigned >= 32 bits */
typedef unsigned int U16;     /* Unsigned >= 16 bits */
typedef unsigned char U8;     /* Unsigned >= 8 bits */
typedef signed long S32;      /* Signed >= 32 bits */
typedef signed int S16;       /* Signed >= 16 bits */
typedef signed char S8;       /* Signed >= 8 bits */

#define UTCDECL  _cdecl
#define UTEXPORT _export
#else
typedef unsigned int U32;     /* Unsigned >= 32 bits */
typedef unsigned short U16;   /* Unsigned >= 16 bits */
typedef unsigned char U8;     /* Unsigned >= 8 bits */
typedef signed int S32;       /* Signed >= 32 bits */
typedef signed short S16;     /* Signed >= 16 bits */
typedef signed char S8;       /* Signed >= 8 bits */

#define UTCDECL
#define UTEXPORT
#define _fmemcpy memcpy
#endif

typedef S16 (*UTFARPROC)(void);
#define U32_MAX ((U32)0xffffffff)
#define U16_MAX ((U16)0xffff)
#define U8_MAX ((U8)0xff)
#define S32_MAX ((S32)0x7fffffff)
#define S16_MAX ((S16)0x7fff)
#define S8_MAX (S8)0x7f)
#define S32_MIN ((S32)(-1 - S32_MAX))
#define S16_MIN ((S16)(-1 - S16_MAX))
#define S8_MIN ((S8)(-1 - S8_MAX))
#define DOUBLE_MAX ((double)1.7e308)

#if (defined(_WINDOWS) || defined(_WIN32))
#define UTDIRSEP '\\'
#define szUTDIRSEP "\\"
#define UTPATHSEP ';'
#else
#define UTDIRSEP '/'
#define szUTDIRSEP "/"
#define UTPATHSEP ':'
#endif                                                                         

/*--------------------------------------------------------------------------------------------------
  Dynarray headers for simple types
--------------------------------------------------------------------------------------------------*/

/* S32 Arrays */
typedef struct {char val;} *utS32ArrayRef;
#define utDynarray2S32Array(dynarray) ((utS32ArrayRef)(void *)(dynarray))
#define utS32Array2Dynarray(array) ((utDynarrayRef)(void *)(array))
#define ut0S32Array NULL
#define uttS32ArrayExists(array) uttDynarrayExists(utS32Array2Dynarray(array))
#define utmS32ArrayS32(array) utgDynarraySize(utS32Array2Dynarray(array))
#define utgS32ArrayS32Values(array) utgDynarrayValues(utS32Array2Dynarray(array), S32)
#define utsS32ArrayS32(array) utgDynarrayNumUsed(utS32Array2Dynarray(array))
#define utrsS32ArrayS32(array, _NumUsed) utrDynarrayNumUsed(utS32Array2Dynarray(array), (_NumUsed))
#define utxS32ArrayS32(array, x) utxDynarrayValue(utS32Array2Dynarray(array), S32, (x))
#define utrxS32ArrayS32(array, x, Value) utrxDynarrayValue(utS32Array2Dynarray(array), S32, (x), Value)
#define utcS32Array() utDynarray2S32Array(utBuildDynarray(S32))
#define utdS32Array(array) utdDynarray(utS32Array2Dynarray(array))
#define utResizeS32Array(array, newSize) utResizeDynarray(utS32Array2Dynarray(array), (newSize))
#define utaS32ArrayS32(array, Value) utaDynarrayValue(utS32Array2Dynarray(array), S32, Value)
#define utForEachS32ArrayS32(array, Value) utForEachDynarrayValue(utS32Array2Dynarray(array), S32, Value)
#define utEndS32ArrayS32 utEndDynarrayValue
#define utCopyS32Array(Dest, Source) utCopyDynarray(utS32Array2Dynarray(Dest), utS32Array2Dynarray(Source), S32)

/* U32 Arrays */
typedef struct {char val;} *utU32ArrayRef;
#define utDynarray2U32Array(dynarray) ((utU32ArrayRef)(void *)(dynarray))
#define utU32Array2Dynarray(array) ((utDynarrayRef)(void *)(array))
#define ut0U32Array NULL
#define uttU32ArrayExists(array) uttDynarrayExists(utU32Array2Dynarray(array))
#define utmU32ArrayU32(array) utgDynarraySize(utU32Array2Dynarray(array))
#define utgU32ArrayU32Values(array) utgDynarrayValues(utU32Array2Dynarray(array), U32)
#define utsU32ArrayU32(array) utgDynarrayNumUsed(utU32Array2Dynarray(array))
#define utrsU32ArrayU32(array, _NumUsed) utrDynarrayNumUsed(utU32Array2Dynarray(array), (_NumUsed))
#define utxU32ArrayU32(array, x) utxDynarrayValue(utU32Array2Dynarray(array), U32, (x))
#define utrxU32ArrayU32(array, x, Value) utrxDynarrayValue(utU32Array2Dynarray(array), U32, (x), Value)
#define utcU32Array() utDynarray2U32Array(utBuildDynarray(U32))
#define utdU32Array(array) utdDynarray(utU32Array2Dynarray(array))
#define utResizeU32Array(array, newSize) utResizeDynarray(utU32Array2Dynarray(array), (newSize))
#define utaU32ArrayU32(array, Value) utaDynarrayValue(utU32Array2Dynarray(array), U32, Value)
#define utForEachU32ArrayU32(array, Value) utForEachDynarrayValue(utU32Array2Dynarray(array), U32, Value)
#define utEndU32ArrayU32 utEndDynarrayValue
#define utCopyU32Array(Dest, Source) utCopyDynarray(utU32Array2Dynarray(Dest), utU32Array2Dynarray(Source), U32)

/*--------------------------------------------------------------------------------------------------
  Compiler optimization hints          example: if(utUnlikely(error)) {utExit("err");} 
// utLikely is implemented in functions: utWhile, utAssert, utIfDebug
//                          in datadraw: tExists, Valid, prev/next
//
// todo Likely: put in utbox?, utResizeDynarray,
//              in create, add, realloc, resize & extend functions
//              add in quad-tree type functions?
//              before conditions that generate utErrors/utExits  [only for debug code?]
//
// todo Prefetch: put in create functions where initialization are done.
--------------------------------------------------------------------------------------------------*/

#if defined(__GNUC__) && __GNUC__ > 3
#define utLikely(x)          __builtin_expect((x),1)    /* assumes gcc version >= 3.0 */
#define utUnlikely(x)        __builtin_expect((x),0)
#define utExpected(x,y)      __builtin_expect((x),(y))
/* Prefetch doesn't fault if addr is invalid. We don't use temporal locality parameter */
#define utPrefetchRead(x)    __builtin_prefetch((x), 0) 
#define utPrefetchWrite(x)   __builtin_prefetch((x), 1) 
/* also add markers for functions that never return (exit, error) using __attribute__ */
#else
#define utLikely(x)          (x)               /* unknown if ms c compiler supports this */
#define utUnlikely(x)        (x)
#define utExpected(x,y)      (x)
#define utPrefetchRead(x)    (x)
#define utPrefetchWrite(x)   (x)
#endif
#define utPrefetch(x)        utPrefetchRead(x)           

/*--------------------------------------------------------------------------------------------------
  handy macros
--------------------------------------------------------------------------------------------------*/
#define utMin(x, y) ((x) <= (y)? (x) : (y))
#define utMax(x, y) ((x) >= (y)? (x) : (y))
#define utAbs(x) ((x) > 0? (x) : -(x))

/* This loop structure allows loop initialization code to be written
   once (between utDo and utWhile), rather than having to be duplicated */
#define utDo do {
#define utWhile(cond) if(utUnlikely(!(cond))) break;
#define utRepeat } while(true);

/*--------------------------------------------------------------------------------------------------
  64 bit stuff
--------------------------------------------------------------------------------------------------*/
struct U64_
{
    U32 upper;
    U32 lower;
};

struct S64_
{
    S32 upper;
    U32 lower;
};

typedef struct U64_ U64;
typedef struct S64_ S64;

#define utgS64Upper(value) ((value).upper)
#define utgS64Lower(value) ((value).lower)
#define utgU64Upper(value) ((value).upper)
#define utgU64Lower(value) ((value).lower)

extern S64 S64_MAX;
extern S64 S64_MIN;
extern S64 S64_0;
extern U64 U64_MIN;
extern U64 U64_MAX;
extern U64 U64_0;
extern U64 utIncrementU64By1(U64 a);
extern U64 utAddU64(U64 a, U64 b);
extern S64 utAddS64(S64 a, S64 b);
extern U64 utSubtractU64(U64 a, U64 b);
extern S64 utSubtractS64(S64 a, S64 b);
extern U64 utConvertU32ToU64(U32 a);
extern U32 utConvertU64ToU32(U64 a);
extern S64 utConvertU32ToS64(U32 a);
extern U32 utConvertS64ToU32(S64 a);
extern S64 utConvertS32ToS64(S32 a);
extern S32 utConvertS64ToS32(S64 a);
extern U64 utConvertS64ToU64(S64 a);
extern S64 utConvertU64ToS64(U64 a);
extern S32 utConvertU64ToS32(U64 a);
extern U64 utConvertS32ToU64(S32 a);
extern double utConvertS64ToDouble(S64 a);

extern U64 utMakeU64(U32 a);
extern S64 utMakeS64(S32 a);
#define uttzeroU64(a) (((a).upper == 0) && ((a).lower == 0))
#define uttzeroS64(a) (((a).upper == 0) && ((a).lower == 0))
extern bool utgtU64(U64 a, U64 b);
extern bool utgtS64(S64 a, S64 b);
extern bool utltU64(U64 a, U64 b);
extern bool utltS64(S64 a, S64 b);
extern bool uteU64(U64 a, U64 b);
extern bool uteS64(S64 a, S64 b);
extern bool utgteU64(U64 a, U64 b);
extern bool utgteS64(S64 a, S64 b);
extern bool utlteU64(U64 a, U64 b);
extern bool utlteS64(S64 a, S64 b);
extern S64 utDivideS64By2(S64 a);
extern S64 utDivideS64ByS16(S64 a, S16 b, S16 *remainder);
extern S64 utDivideS64ByS32(S64 arga, S32 argb);
extern S64 utMultiplyS64(S64 a, S64 b);
extern bool utTestFor64Bit();
extern U64 utSquareRootU64(U64 a);
extern S64 utStrToS64(char *string, U32 length);
extern char *utS64ToStr(S64 value);

#endif

