/*
 * Copyright (C) 2004 ViASIC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */

#include <string.h>
#include <ctype.h>
#include "htext.h"

U32 htHashValue;

/*--------------------------------------------------------------------------------------------------
  Clear a hash table.
--------------------------------------------------------------------------------------------------*/
static void clearHtbl(
    htHtbl htbl)
{
    U32 size = 1 << htHtblGetSizeExp(htbl);
    U32 xEntry;

    for(xEntry = 0; xEntry < size; xEntry++) {
        htHtblSetiEntry(htbl, xEntry, htEntryNull);
    }
}

/*--------------------------------------------------------------------------------------------------
  Build a hash table.
--------------------------------------------------------------------------------------------------*/
htHtbl htHtblCreate(void)
{
    htHtbl htbl = htHtblAlloc();

    htHtblSetSizeExp(htbl, 8); /* htbls are powers of two in size */
    htHtblSetNumEntries(htbl, 0);
    htHtblAllocEntrys(htbl, 1 << htHtblGetSizeExp(htbl));
    clearHtbl(htbl);
    return htbl;
}

/*--------------------------------------------------------------------------------------------------
  Reset the hash key.
--------------------------------------------------------------------------------------------------*/
void htStartHashKey(void)
{
    htHashValue = 0L;
}

/*--------------------------------------------------------------------------------------------------
  Hash a single character.
--------------------------------------------------------------------------------------------------*/
#if 0    /* implemented as a macro for speed */
void htHashChar(
    char c)
{
    htHashValue = (htHashValue ^ c) * 1103515245 + 12345;
}
#endif

/*--------------------------------------------------------------------------------------------------
  Hash a single character.
--------------------------------------------------------------------------------------------------*/
void htHashBool(
    bool value)
{
    value = value ? 1 : 0;
    htHashValue = (htHashValue ^ value) * 1103515245 + 12345;
}

/*--------------------------------------------------------------------------------------------------
  Find a good hash id for a string.
--------------------------------------------------------------------------------------------------*/
void htHashString(
    char const *name)
{
    do {
        htHashValue = (htHashValue ^ *name) * 1103515245 + 12345;
    } while (*name++);
}

/*--------------------------------------------------------------------------------------------------
  Find a good hash id for a string.
--------------------------------------------------------------------------------------------------*/
void htHashStringWithoutCase(
    char const *name)
{
    do {
        htHashValue = (htHashValue ^ tolower(*name)) * 1103515245 + 12345;
    } while (*name++);
}

/*--------------------------------------------------------------------------------------------------
  Find a good hash id for a U16.
--------------------------------------------------------------------------------------------------*/
void htHashU16(
    U16 value)
{
    htHashChar((char)value);
    htHashChar((char)(value >> 8));
}

/*--------------------------------------------------------------------------------------------------
  Find a good hash id for a U32
--------------------------------------------------------------------------------------------------*/
#if 0    /* implemented as a macro for speed */
void htHashU32(
    U32 value)
{
    htHashChar((char)value);
    htHashChar((char)(value >> 8));
    htHashChar((char)(value >> 16));
    htHashChar((char)(value >> 24));
}
#endif

/*--------------------------------------------------------------------------------------------------
  Hash in a box's coordinates.
--------------------------------------------------------------------------------------------------*/
void htHashBox(
    utBox box)
{
    htHashS32(utBoxGetLeft(box));
    htHashS32(utBoxGetBottom(box));
    htHashS32(utBoxGetRight(box));
    htHashS32(utBoxGetTop(box));
}

/*--------------------------------------------------------------------------------------------------
  Find an entry in the hash table with the given name.
--------------------------------------------------------------------------------------------------*/
htEntry htHtblLookupEntry(
    htHtbl htbl,
    bool (*matchEntry)(htEntry entry))
{
    U32 mask = (1 << htHtblGetSizeExp(htbl)) - 1;
    htEntry entry = htHtblGetiEntry(htbl, htHashValue & mask);

    while(entry != htEntryNull) {
        if(matchEntry(entry)) {
            return entry;
        }
        entry = htEntryGetEntry(entry);
    }
    return htEntryNull;
}

/*--------------------------------------------------------------------------------------------------
  Find a value in the hash table with the given name.
--------------------------------------------------------------------------------------------------*/
U32 htHtblLookup(
    htHtbl htbl,
    bool (*matchEntry)(htEntry entry))
{
    htEntry entry = htHtblLookupEntry(htbl, matchEntry);

    if(entry != htEntryNull) {
        return htEntryGetData(entry);
    }
    return U32_MAX;
}

/*--------------------------------------------------------------------------------------------------
  Make a new hash table htbl twice as large, and copy from the old one to the new one.
  Note: We've violated the array abstraction here.
--------------------------------------------------------------------------------------------------*/
static void resizeHtbl(
    htHtbl htbl)
{
    U16 sizeExp = htHtblGetSizeExp(htbl);
    U32 oldSize = 1 << sizeExp;
    U32 newSize = 1 << (sizeExp + 1);
    U32 mask = newSize - 1;
    htEntry *oldEntries = htHtblGetEntrys(htbl);
    htEntry entry, nEntry;
    U32 xOldHtbl, xNewHtbl;

    htHtblSetSizeExp(htbl, sizeExp + 1);
    htHtblAllocEntrys(htbl, 1 << htHtblGetSizeExp(htbl));
    clearHtbl(htbl);
    for(xOldHtbl = 0; xOldHtbl < oldSize; xOldHtbl++) {
        for(entry = oldEntries[xOldHtbl]; entry != htEntryNull; entry = nEntry) {
            nEntry = htEntryGetEntry(entry);
            xNewHtbl = htEntryGetHashValue(entry) & mask;
            htEntrySetEntry(entry, htHtblGetiEntry(htbl, xNewHtbl));
            htHtblSetiEntry(htbl, xNewHtbl, entry);
        }
    }
    utFree(oldEntries);
}

/*--------------------------------------------------------------------------------------------------
  Add an entry to the htbl.  If the table is near full, build a new one twice as big, delete the
  old one, and return the new one.  Otherwise, return the current hash table.
--------------------------------------------------------------------------------------------------*/
htEntry htHtblAdd(
    htHtbl htbl,
    U32 data)
{
    U32 mask;
    U32 index;
    htEntry entry = htEntryAlloc();

    htEntrySetHashValue(entry, htHashValue);
    htEntrySetData(entry, data);
    if(htHtblGetNumEntries(htbl) > (U32)(1 << (htHtblGetSizeExp(htbl) - 1))) {
        resizeHtbl(htbl);
    }
    mask = (1 << htHtblGetSizeExp(htbl)) - 1;
    index = htHashValue & mask;
    htEntrySetEntry(entry, htHtblGetiEntry(htbl, index));
    htHtblSetiEntry(htbl, index, entry);
    htHtblSetNumEntries(htbl, htHtblGetNumEntries(htbl) + 1);
    return entry;
}

/*--------------------------------------------------------------------------------------------------
  Delete an entry in the hash table.
--------------------------------------------------------------------------------------------------*/
void htDeleteHtblEntry(
    htHtbl htbl,
    htEntry entry)
{
    U32 mask = (1 << htHtblGetSizeExp(htbl)) - 1;
    U32 index = htEntryGetHashValue(entry) & mask;
    htEntry pEntry, nEntry;

    nEntry = htHtblGetiEntry(htbl, index);
    if(nEntry == entry) {
        htHtblSetiEntry(htbl, index, htEntryGetEntry(nEntry));
    } else {            
        do {
            pEntry = nEntry;
            nEntry = htEntryGetEntry(nEntry);
        } while (nEntry != entry);        
        htEntrySetEntry(pEntry, htEntryGetEntry(entry));
    }
    htEntryFree(entry);
    htHtblSetNumEntries(htbl, htHtblGetNumEntries(htbl) - 1);
}

/*--------------------------------------------------------------------------------------------------
  Initialize hash table memory.
--------------------------------------------------------------------------------------------------*/
void htHtblAddSym(
    htHtbl htbl,
    utSym sym)
{
    htStartHashKey();
    htHashU32(utgxSym(sym));
    htHtblAdd(htbl, utgxSym(sym));
}

static utSym htSym;
/*--------------------------------------------------------------------------------------------------
  Check that the symbol matches.
--------------------------------------------------------------------------------------------------*/
static bool matchSym(
    htEntry entry)
{
    return htEntryGetData(entry) == utgxSym(htSym);
}

/*--------------------------------------------------------------------------------------------------
  Find the symbol in the hash table, and return its data, or U32_MAX if not found.
--------------------------------------------------------------------------------------------------*/
utSym htHtblLookupSym(
    htHtbl htbl,
    utSym sym)
{
    htStartHashKey();
    htHashU32(utgxSym(sym));
    htSym = sym;
    return utxSym(htHtblLookup(htbl, matchSym));
}

/*--------------------------------------------------------------------------------------------------
  Initialize hash table memory.
--------------------------------------------------------------------------------------------------*/
void htStart(void)
{
    htDDRStart();
}

/*--------------------------------------------------------------------------------------------------
  Free hash table memory.
--------------------------------------------------------------------------------------------------*/
void htStop(void)
{
    htDDRStop();
}

