// Copyright (c) 2021 Oasis Labs Inc
// Copyright (c) 2023 Yawning Angel
//
// SPDX-License-Identifier: BSD-3-Clause

package h2c

import (
	"crypto"
	"crypto/subtle"
	"errors"
	"math"
)

const oversizeDST = "H2C-OVERSIZE-DST-"

var (
	errInvalidDigestSize = errors.New("secp256k1/secec/h2c: b_in_bytes insufficiently larged")
	errInvalidOutputSize = errors.New("secp256k1/secec/h2c: len_in_bytes out of range")
	errInvalidDomainSep  = errors.New("secp256k1/secec/h2c: invalid domain separator")
	errEllOutOfRange     = errors.New("secp256k1/secec/h2c: ell out of range")
)

// expandMessageXMD implements expand_message_xmd, overwriting out with
// uniformly random data generated by the provided hash function, domain
// separation tag, and message.
func expandMessageXMD(out []byte, hFunc crypto.Hash, domainSeparator, message []byte) error {
	lenInBytes := len(out)
	bInBytes := hFunc.Size()

	h := hFunc.New()
	rInBytes := h.BlockSize()

	// 0. Ensure parameters are sensible.
	//
	// The RFC allows for 0-length output, we do not, because that is stupid.
	// This does not cause compatibility issues, as `len(out)` is always
	// `ell * 2`.
	if bInBytes < 2*kay/8 {
		return errInvalidDigestSize
	}
	if lenInBytes == 0 || lenInBytes > math.MaxUint16 {
		return errInvalidOutputSize
	}

	// 5.3.3 Using DSTs longer than 255 bytes.
	DST := domainSeparator
	lenDST := len(domainSeparator)
	switch {
	case lenDST == 0:
		// Per "3.1. Domain separation requirements", "Tags MUST have nonzero length."
		// This is not currently reflected in the procedure (See issue #353).
		return errInvalidDomainSep
	case lenDST > math.MaxUint8:
		// DST = H("H2C-OVERSIZE-DST-" || a_very_long_DST)
		_, _ = h.Write([]byte(oversizeDST))
		_, _ = h.Write(DST)

		DST = h.Sum(nil)
		lenDST = len(DST)

		h.Reset()
	}

	// 1. ell = ceil(len_in_bytes / b_in_bytes)
	ell := (lenInBytes + bInBytes - 1) / bInBytes

	// 2. ABORT if ell > 255 or len_in_bytes > 65535 or len(DST) > 255
	//
	// Note: len_in_bytes checks are done already.
	if ell > 255 {
		return errEllOutOfRange
	}

	// 7. b_0 = H(msg_prime)
	_, _ = h.Write(make([]byte, rInBytes))                             // Z_pad (I2OSP(0, r_in_bytes))
	_, _ = h.Write(message)                                            // msg
	_, _ = h.Write([]byte{byte(lenInBytes >> 8), byte(lenInBytes), 0}) // l_i_b_str || I2OSP(0, 1)
	_, _ = h.Write(DST)                                                // DST
	_, _ = h.Write([]byte{byte(lenDST)})                               // I2OSP(len(DST), 1)
	b0 := h.Sum(nil)

	// 8. b_1 = H(b_0 || I2OSP(1, 1) || DST_prime)
	h.Reset()
	_, _ = h.Write(b0)                   // b_0
	_, _ = h.Write([]byte{1})            // I2OSP(1, 1)
	_, _ = h.Write(DST)                  // DST
	_, _ = h.Write([]byte{byte(lenDST)}) // I2OSP(len(DST), 1)
	b1 := h.Sum(nil)

	// We attempt to be somewhat more efficient about the remaining steps
	// by:
	//  * Seeing if we can service the request just with b_1 (len_in_bytes <= b_in_bytes)
	//  * Allocating a b_in_bytes sized temporary buffer to store:
	//     * b_(i - 1) (Initialized to b_1)
	//     * strxor(b_0, b_(i - 1))
	//     * b_i, which becomes b_(i - 1)

	// Special case: if len_in_bytes <= b_in_bytes, we can return output
	// from b_1 and terminate.
	if lenInBytes <= bInBytes {
		copy(out, b1[:lenInBytes])
		return nil
	}

	// Reuse a temporary buffer to hold both the xored portion of the hash
	// input and b_(i - 1).
	xorBuf := make([]byte, 0, bInBytes)
	xorBuf = append(xorBuf, b1...)

	// Append b_1 to the output, since we know we need all of it.
	copy(out, b1) // 11. uniform_bytes = b_1 || ...
	outOff := len(b1)

	// 9. for i in (2, ..., ell):
	for i, wanted := 2, lenInBytes-bInBytes; wanted > 0; i++ {
		// 10. b_i = H(strxor(b_0, b_(i - 1)) || I2OSP(i, 1) || DST_prime)
		subtle.XORBytes(xorBuf, xorBuf, b0)

		h.Reset()
		_, _ = h.Write(xorBuf)               // strxor(b_0, b_(i - 1))
		_, _ = h.Write([]byte{byte(i)})      // I2OSP(i, 1)
		_, _ = h.Write(DST)                  // DST
		_, _ = h.Write([]byte{byte(lenDST)}) // I2OSP(len(DST), 1)
		h.Sum(xorBuf[:0])                    // xorBuf = b_i

		// Append up to b_in_bytes from b_i (this handles the substr)
		toAppend := wanted
		if wanted > bInBytes {
			toAppend = bInBytes
		}

		copy(out[outOff:], xorBuf[:toAppend])
		outOff += toAppend
		wanted -= toAppend
	}

	return nil
}
