package otr3

import (
	"bufio"
	"bytes"
	"crypto/rand"
	"os"
	"syscall"
	"testing"
)

var (
	serializedPublicKey = []byte{
		// key type for DSA
		0x00, 0x00,

		// len of p MPI
		0x00, 0x00, 0x00, 0x80,
		// p MPI
		0xF2, 0x48, 0x43, 0xF9, 0x44, 0x7B, 0x62, 0x13, 0x8A, 0xE4, 0x9B, 0xF8, 0x31, 0x88, 0xD1, 0x35, 0x3A, 0xDA, 0x5C, 0xAC, 0x11, 0x88, 0x90, 0xCF, 0xDE, 0xC0, 0x1B, 0xF3, 0x49, 0xD7, 0x5E, 0x88, 0x7B, 0x19, 0xC2, 0x21, 0x66, 0x5C, 0x78, 0x57, 0xCA, 0xD5, 0x83, 0xAF, 0x65, 0x6C, 0x67, 0xFB, 0x04, 0xA9, 0x9F, 0xD8, 0xF8, 0xD6, 0x9D, 0x09, 0xC9, 0x52, 0x9C, 0x6C, 0x14, 0xD4, 0x26, 0xF1, 0xE3, 0x92, 0x4D, 0xC9, 0x24, 0x3A, 0xF2, 0x97, 0x0E, 0x3E, 0x4B, 0x04, 0xA2, 0x34, 0x89, 0xA0, 0x9E, 0x8A, 0x90, 0xE7, 0xE8, 0x1E, 0xBA, 0x76, 0x3A, 0xD4, 0xF0, 0x63, 0x6B, 0x8A, 0x43, 0x41, 0x5B, 0x6F, 0xC1, 0x6A, 0x02, 0xC3, 0x62, 0x4C, 0xE7, 0x62, 0x72, 0xFA, 0x00, 0x78, 0x3C, 0x8D, 0xB8, 0x50, 0xD3, 0xA9, 0x96, 0xB5, 0x81, 0x36, 0xF7, 0xA0, 0xEB, 0x80, 0xAE, 0x0B, 0xC6, 0x13,

		// len of q MPI
		0x00, 0x00, 0x00, 0x14,
		// q MPI
		0xD1, 0x6B, 0x26, 0x07, 0xFC, 0xBC, 0x0E, 0xDC, 0x63, 0x9F, 0x76, 0x3A, 0x54, 0xF3, 0x44, 0x75, 0xB1, 0xCC, 0x84, 0x73,

		// len of g MPI
		0x00, 0x00, 0x00, 0x80,
		// g MPI
		0xB1, 0x5A, 0xFE, 0xF5, 0xF9, 0x6E, 0xFE, 0xE4, 0x10, 0x06, 0xF1, 0x36, 0xC2, 0x3A, 0x18, 0x84, 0x9D, 0xA8, 0x13, 0x30, 0x69, 0xA8, 0x79, 0xD0, 0x83, 0xF7, 0xC7, 0xAA, 0x36, 0x2E, 0x18, 0x7D, 0xAE, 0x3E, 0xD0, 0xC4, 0xF3, 0x72, 0xD0, 0xD4, 0xE3, 0xAA, 0xE5, 0x67, 0x00, 0x8A, 0x18, 0x72, 0xA6, 0xE8, 0x5D, 0x8F, 0x84, 0xE5, 0x3A, 0x3F, 0xE1, 0xB3, 0x52, 0xAF, 0x0B, 0x4E, 0x2F, 0x0C, 0xB0, 0x33, 0xA6, 0xD3, 0x42, 0x85, 0xEC, 0xD3, 0xE4, 0xA9, 0x36, 0x53, 0xBD, 0xE9, 0x9C, 0x3A, 0x8D, 0x84, 0x0D, 0x9D, 0x35, 0xF8, 0x2A, 0xC2, 0xFA, 0x85, 0x39, 0xDB, 0x6C, 0x7F, 0x7A, 0x1D, 0xAD, 0x77, 0xFE, 0xEC, 0xD6, 0x28, 0x03, 0x75, 0x7F, 0xF1, 0xE2, 0xDE, 0x4C, 0xEC, 0x4A, 0x5A, 0x2A, 0xD6, 0x43, 0x27, 0x15, 0x14, 0xDD, 0xEE, 0xEF, 0x3D, 0x00, 0x8F, 0x66, 0xFB, 0xF9, 0xDB,

		// len of y MPI
		0x00, 0x00, 0x00, 0x80,
		// y MPI
		0x01, 0xF9, 0xBE, 0x7D, 0xA0, 0xE4, 0xE8, 0x47, 0x74, 0x04, 0x80, 0x58, 0xB5, 0x32, 0x02, 0xB2, 0x70, 0x4B, 0xF6, 0x88, 0xA3, 0x06, 0x09, 0x2E, 0xD5, 0x33, 0xA5, 0x5E, 0x68, 0xEA, 0xBA, 0x81, 0x4C, 0x8D, 0x62, 0xF4, 0x5A, 0xAD, 0x8F, 0xF3, 0x0C, 0x30, 0x55, 0xDC, 0xA4, 0x61, 0xB7, 0xDB, 0xA6, 0xB7, 0x89, 0x38, 0xFC, 0x4D, 0x69, 0x78, 0x0A, 0x83, 0x0C, 0x64, 0x57, 0xCC, 0x10, 0x7F, 0x3D, 0x27, 0x5C, 0x21, 0xD0, 0x0E, 0x53, 0x14, 0x7C, 0x14, 0x16, 0x21, 0x76, 0xC7, 0x71, 0x69, 0xD3, 0xBC, 0xA5, 0x86, 0xDC, 0x30, 0xF1, 0x5F, 0x4B, 0x48, 0x21, 0x60, 0xE2, 0x76, 0x86, 0x9A, 0xA3, 0x36, 0xF3, 0x8A, 0xF7, 0xFC, 0x36, 0x86, 0xA7, 0x64, 0xAB, 0x5A, 0x02, 0xC7, 0x51, 0xD9, 0x21, 0xA4, 0x2B, 0x8B, 0x9A, 0xE8, 0xE0, 0x69, 0x18, 0x05, 0x9C, 0xD7, 0x3C, 0x42, 0x41, 0x54,
	}
	serializedPrivateKey = []byte{
		// key type for DSA
		0x00, 0x00,

		// len of p MPI
		0x00, 0x00, 0x00, 0x80,
		// p MPI
		0xF2, 0x48, 0x43, 0xF9, 0x44, 0x7B, 0x62, 0x13, 0x8A, 0xE4, 0x9B, 0xF8, 0x31, 0x88, 0xD1, 0x35, 0x3A, 0xDA, 0x5C, 0xAC, 0x11, 0x88, 0x90, 0xCF, 0xDE, 0xC0, 0x1B, 0xF3, 0x49, 0xD7, 0x5E, 0x88, 0x7B, 0x19, 0xC2, 0x21, 0x66, 0x5C, 0x78, 0x57, 0xCA, 0xD5, 0x83, 0xAF, 0x65, 0x6C, 0x67, 0xFB, 0x04, 0xA9, 0x9F, 0xD8, 0xF8, 0xD6, 0x9D, 0x09, 0xC9, 0x52, 0x9C, 0x6C, 0x14, 0xD4, 0x26, 0xF1, 0xE3, 0x92, 0x4D, 0xC9, 0x24, 0x3A, 0xF2, 0x97, 0x0E, 0x3E, 0x4B, 0x04, 0xA2, 0x34, 0x89, 0xA0, 0x9E, 0x8A, 0x90, 0xE7, 0xE8, 0x1E, 0xBA, 0x76, 0x3A, 0xD4, 0xF0, 0x63, 0x6B, 0x8A, 0x43, 0x41, 0x5B, 0x6F, 0xC1, 0x6A, 0x02, 0xC3, 0x62, 0x4C, 0xE7, 0x62, 0x72, 0xFA, 0x00, 0x78, 0x3C, 0x8D, 0xB8, 0x50, 0xD3, 0xA9, 0x96, 0xB5, 0x81, 0x36, 0xF7, 0xA0, 0xEB, 0x80, 0xAE, 0x0B, 0xC6, 0x13,

		// len of q MPI
		0x00, 0x00, 0x00, 0x14,
		// q MPI
		0xD1, 0x6B, 0x26, 0x07, 0xFC, 0xBC, 0x0E, 0xDC, 0x63, 0x9F, 0x76, 0x3A, 0x54, 0xF3, 0x44, 0x75, 0xB1, 0xCC, 0x84, 0x73,

		// len of g MPI
		0x00, 0x00, 0x00, 0x80,
		// g MPI
		0xB1, 0x5A, 0xFE, 0xF5, 0xF9, 0x6E, 0xFE, 0xE4, 0x10, 0x06, 0xF1, 0x36, 0xC2, 0x3A, 0x18, 0x84, 0x9D, 0xA8, 0x13, 0x30, 0x69, 0xA8, 0x79, 0xD0, 0x83, 0xF7, 0xC7, 0xAA, 0x36, 0x2E, 0x18, 0x7D, 0xAE, 0x3E, 0xD0, 0xC4, 0xF3, 0x72, 0xD0, 0xD4, 0xE3, 0xAA, 0xE5, 0x67, 0x00, 0x8A, 0x18, 0x72, 0xA6, 0xE8, 0x5D, 0x8F, 0x84, 0xE5, 0x3A, 0x3F, 0xE1, 0xB3, 0x52, 0xAF, 0x0B, 0x4E, 0x2F, 0x0C, 0xB0, 0x33, 0xA6, 0xD3, 0x42, 0x85, 0xEC, 0xD3, 0xE4, 0xA9, 0x36, 0x53, 0xBD, 0xE9, 0x9C, 0x3A, 0x8D, 0x84, 0x0D, 0x9D, 0x35, 0xF8, 0x2A, 0xC2, 0xFA, 0x85, 0x39, 0xDB, 0x6C, 0x7F, 0x7A, 0x1D, 0xAD, 0x77, 0xFE, 0xEC, 0xD6, 0x28, 0x03, 0x75, 0x7F, 0xF1, 0xE2, 0xDE, 0x4C, 0xEC, 0x4A, 0x5A, 0x2A, 0xD6, 0x43, 0x27, 0x15, 0x14, 0xDD, 0xEE, 0xEF, 0x3D, 0x00, 0x8F, 0x66, 0xFB, 0xF9, 0xDB,

		// len of y MPI
		0x00, 0x00, 0x00, 0x80,
		// y MPI
		0x01, 0xF9, 0xBE, 0x7D, 0xA0, 0xE4, 0xE8, 0x47, 0x74, 0x04, 0x80, 0x58, 0xB5, 0x32, 0x02, 0xB2, 0x70, 0x4B, 0xF6, 0x88, 0xA3, 0x06, 0x09, 0x2E, 0xD5, 0x33, 0xA5, 0x5E, 0x68, 0xEA, 0xBA, 0x81, 0x4C, 0x8D, 0x62, 0xF4, 0x5A, 0xAD, 0x8F, 0xF3, 0x0C, 0x30, 0x55, 0xDC, 0xA4, 0x61, 0xB7, 0xDB, 0xA6, 0xB7, 0x89, 0x38, 0xFC, 0x4D, 0x69, 0x78, 0x0A, 0x83, 0x0C, 0x64, 0x57, 0xCC, 0x10, 0x7F, 0x3D, 0x27, 0x5C, 0x21, 0xD0, 0x0E, 0x53, 0x14, 0x7C, 0x14, 0x16, 0x21, 0x76, 0xC7, 0x71, 0x69, 0xD3, 0xBC, 0xA5, 0x86, 0xDC, 0x30, 0xF1, 0x5F, 0x4B, 0x48, 0x21, 0x60, 0xE2, 0x76, 0x86, 0x9A, 0xA3, 0x36, 0xF3, 0x8A, 0xF7, 0xFC, 0x36, 0x86, 0xA7, 0x64, 0xAB, 0x5A, 0x02, 0xC7, 0x51, 0xD9, 0x21, 0xA4, 0x2B, 0x8B, 0x9A, 0xE8, 0xE0, 0x69, 0x18, 0x05, 0x9C, 0xD7, 0x3C, 0x42, 0x41, 0x54,

		// len of x MPI
		0x00, 0x00, 0x00, 0x14,
		// y MPI
		0x14, 0xD0, 0x34, 0x5A, 0x35, 0x62, 0xC4, 0x80, 0xA0, 0x39, 0xE3, 0xC7, 0x27, 0x64, 0xF7, 0x2D, 0x79, 0x04, 0x32, 0x16,
	}
)

func inp(s string) *bufio.Reader {
	return bufio.NewReader(bytes.NewBuffer([]byte(s)))
}

func Test_readParameter_willReturnTheParameterRead(t *testing.T) {
	tag, value, _, _ := readParameter(inp(`(p #00FC07ABCF0DC916AFF6E9A0D450A9B7A857#)`))
	assertDeepEquals(t, tag, "p")
	assertDeepEquals(t, value, bnFromHex("00FC07ABCF0DC916AFF6E9A0D450A9B7A857"))
}

func Test_readParameter_willReturnAnotherParameterRead(t *testing.T) {
	tag, value, _, _ := readParameter(inp(`(quux #00FC07ABCF0DC916AFF6E9A0D450A9B7A858#)`))
	assertDeepEquals(t, tag, "quux")
	assertDeepEquals(t, value, bnFromHex("00FC07ABCF0DC916AFF6E9A0D450A9B7A858"))
}

func Test_readParameter_willReturnNotOKIfAskedToParseATooShortList(t *testing.T) {
	_, _, _, ok := readParameter(inp(`()`))
	assertDeepEquals(t, ok, false)

	_, _, _, ok = readParameter(inp(`(quux)`))
	assertDeepEquals(t, ok, false)
}

func Test_readParameter_willReturnNotOKIfAskedToParseSomethingOfTheWrongType(t *testing.T) {
	_, _, _, ok := readParameter(inp(`("quux" #00FC07ABCF0DC916AFF6E9A0D450A9B7A858#)`))
	assertDeepEquals(t, ok, false)

	_, _, _, ok = readParameter(inp(`(quux "00FC07ABCF0DC916AFF6E9A0D450A9B7A858")`))
	assertDeepEquals(t, ok, false)
}

func Test_readDSAPrivateKey_willReturnADSAPrivateKey(t *testing.T) {
	from := inp(`(dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043216#)
  )`)
	k, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, k.P, bnFromHex("00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857"))
	assertDeepEquals(t, k.Q, bnFromHex("00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081"))
	assertDeepEquals(t, k.G, bnFromHex("535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26"))
	assertDeepEquals(t, k.X, bnFromHex("14D0345A3562C480A039E3C72764F72D79043216"))
	assertDeepEquals(t, k.Y, bnFromHex("0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF"))
	assertDeepEquals(t, ok, true)
}

func Test_readDSAPrivateKey_willReturnNotOKForNoList(t *testing.T) {
	from := inp(`dsa`)
	_, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readDSAPrivateKey_willReturnNotOKForListWithNoEntries(t *testing.T) {
	from := inp(`()`)
	_, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readDSAPrivateKey_willReturnNotOKForListWithNoEnding(t *testing.T) {
	from := inp(`(dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043216#)
  `)
	_, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readDSAPrivateKey_willReturnNotOKForListWithTheWrongTag(t *testing.T) {
	from := inp(`(dsax
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043216#)
  `)
	_, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readDSAPrivateKey_willReturnNotOKForListWithInvalidTypeOfTag(t *testing.T) {
	from := inp(`("dsa"
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043216#)
  `)
	_, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readDSAPrivateKey_willReturnNotOKWhenPParameterIsInvalid(t *testing.T) {
	from := inp(`(dsa
  (px #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043216#))
  `)
	_, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readDSAPrivateKey_willReturnNotOKWhenQParameterIsInvalid(t *testing.T) {
	from := inp(`(dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (qx #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043216#))
  `)
	_, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readDSAPrivateKey_willReturnNotOKWhenGParameterIsInvalid(t *testing.T) {
	from := inp(`(dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (gx #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043216#))
  `)
	_, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readDSAPrivateKey_willReturnNotOKWhenYParameterIsInvalid(t *testing.T) {
	from := inp(`(dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (yx #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043216#))
  `)
	_, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readDSAPrivateKey_willReturnNotOKWhenXParameterIsInvalid(t *testing.T) {
	from := inp(`(dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (xx #14D0345A3562C480A039E3C72764F72D79043216#))
  `)
	_, ok := readDSAPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readPrivateKey_willReturnAPrivateKey(t *testing.T) {
	from := inp(`(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  ))`)
	k, ok := readPrivateKey(from)
	assertDeepEquals(t, k.(*DSAPrivateKey).PrivateKey.P, bnFromHex("00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857"))
	assertDeepEquals(t, k.(*DSAPrivateKey).PrivateKey.Q, bnFromHex("00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081"))
	assertDeepEquals(t, k.(*DSAPrivateKey).PrivateKey.G, bnFromHex("535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26"))
	assertDeepEquals(t, k.(*DSAPrivateKey).PrivateKey.X, bnFromHex("14D0345A3562C480A039E3C72764F72D79043217"))
	assertDeepEquals(t, k.(*DSAPrivateKey).PrivateKey.Y, bnFromHex("0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF"))
	assertDeepEquals(t, ok, true)
}

func Test_readPrivateKey_willReturnNotOKForSomethingNotAList(t *testing.T) {
	from := inp(`one`)
	_, ok := readPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readPrivateKey_willReturnNotOKForAListThatIsNotEnded(t *testing.T) {
	from := inp(`(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  )`)
	_, ok := readPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readPrivateKey_willReturnNotOKForAnInvalidDSAKey(t *testing.T) {
	from := inp(`(private-key (dsax
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  ))`)
	_, ok := readPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readPrivateKey_willReturnNotOKForAnInvalidTag(t *testing.T) {
	from := inp(`(private-keyx (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  ))`)
	_, ok := readPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readPrivateKey_willReturnNotOKForATagOfWrongType(t *testing.T) {
	from := inp(`("private-key" (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  ))`)
	_, ok := readPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readPrivateKey_willReturnNotOKForNoTag(t *testing.T) {
	from := inp(`("private-key" (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  ))`)
	_, ok := readPrivateKey(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccount_willReturnAnAccount(t *testing.T) {
	from := inp(`(account
(name "foo")
(protocol libpurple-Jabber)
(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  )))`)
	k, ok, _ := readAccount(from)
	assertDeepEquals(t, k.Name, "foo")
	assertDeepEquals(t, k.Protocol, "libpurple-Jabber")
	assertDeepEquals(t, k.Key.(*DSAPrivateKey).PrivateKey.P, bnFromHex("00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857"))
	assertDeepEquals(t, ok, true)
}

func Test_readAccount_willReturnNotOKForSomethingNotAList(t *testing.T) {
	from := inp(`account`)
	_, ok, atEnd := readAccount(from)
	assertDeepEquals(t, ok, true)
	assertDeepEquals(t, atEnd, true)
}

func Test_readAccount_willReturnNotOKForAListThatIsNotEnded(t *testing.T) {
	from := inp(`(account
(name "foo")
(protocol libpurple-Jabber)
(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  ))`)
	_, ok, _ := readAccount(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccount_willReturnNotOKForAMissingName(t *testing.T) {
	from := inp(`(account
(protocol libpurple-Jabber)
(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  )))`)
	_, ok, _ := readAccount(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccount_willReturnNotOKForAMissingProtocol(t *testing.T) {
	from := inp(`(account
(name "foo")
(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  )))`)
	_, ok, _ := readAccount(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccount_willReturnNotOKForAMissingPrivateKey(t *testing.T) {
	from := inp(`(account
(name "foo")
(protocol libpurple-Jabber)
)`)
	_, ok, _ := readAccount(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccount_willReturnNotOKForAnIncorrectName(t *testing.T) {
	from := inp(`(account
(namex "foo")
(protocol libpurple-Jabber)
(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  )))`)
	_, ok, _ := readAccount(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccount_willReturnNotOKForAnIncorrectProtocol(t *testing.T) {
	from := inp(`(account
(name "foo")
(protocolx libpurple-Jabber)
(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  )))`)
	_, ok, _ := readAccount(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccount_willReturnNotOKForAnIncorrectPrivateKey(t *testing.T) {
	from := inp(`(account
(name "foo")
(protocol libpurple-Jabber)
(private-keyx (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043217#)
  )))`)
	_, ok, _ := readAccount(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccounts_willReturnTheAccountRead(t *testing.T) {
	from := inp(`(privkeys (account
(name "foo2")
(protocol libpurple-Jabberx)
(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A858#)
  ))))`)
	k, ok := readAccounts(from)
	assertDeepEquals(t, k[0].Name, "foo2")
	assertDeepEquals(t, k[0].Protocol, "libpurple-Jabberx")
	assertDeepEquals(t, k[0].Key.(*DSAPrivateKey).PrivateKey.P, bnFromHex("00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A858"))
	assertDeepEquals(t, ok, true)
}

func Test_readAccounts_willReturnZeroAccountsIfNoAccountsThere(t *testing.T) {
	from := inp(`(privkeys)`)
	k, ok := readAccounts(from)
	assertDeepEquals(t, len(k), 0)
	assertDeepEquals(t, ok, true)
}

func Test_readAccounts_willReturnNotOKForNoList(t *testing.T) {
	from := inp(`privkeys`)
	_, ok := readAccounts(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccounts_willReturnNotOKForNonFinishedList(t *testing.T) {
	from := inp(`(privkeys`)
	_, ok := readAccounts(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccounts_willReturnNotOKForIncorrectTag(t *testing.T) {
	from := inp(`(privkeysx)`)
	_, ok := readAccounts(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccounts_willReturnNotOKForTagWithWrongType(t *testing.T) {
	from := inp(`("privkeys")`)
	_, ok := readAccounts(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccounts_willReturnNotOKForAccountThatIsNotOK(t *testing.T) {
	from := inp(`(privkeys
(accountx
	(name "2")
	(protocol libpurple-jabber-gtalk)
	(private-key
	 (dsa
	  (p #00F24843F9447B62138AE49BF83188D1353ADA5CAC118890CFDEC01BF349D75E887B19C221665C7857CAD583AF656C67FB04A99FD8F8D69D09C9529C6C14D426F1E3924DC9243AF2970E3E4B04A23489A09E8A90E7E81EBA763AD4F0636B8A43415B6FC16A02C3624CE76272FA00783C8DB850D3A996B58136F7A0EB80AE0BC613#)
	  (q #00D16B2607FCBC0EDC639F763A54F34475B1CC8473#)
	  (g #00B15AFEF5F96EFEE41006F136C23A18849DA8133069A879D083F7C7AA362E187DAE3ED0C4F372D0D4E3AAE567008A1872A6E85D8F84E53A3FE1B352AF0B4E2F0CB033A6D34285ECD3E4A93653BDE99C3A8D840D9D35F82AC2FA8539DB6C7F7A1DAD77FEECD62803757FF1E2DE4CEC4A5A2AD643271514DDEEEF3D008F66FBF9DB#)
	  (y #01F9BE7DA0E4E84774048058B53202B2704BF688A306092ED533A55E68EABA814C8D62F45AAD8FF30C3055DCA461B7DBA6B78938FC4D69780A830C6457CC107F3D275C21D00E53147C14162176C77169D3BCA586DC30F15F4B482160E276869AA336F38AF7FC3686A764AB5A02C751D921A42B8B9AE8E06918059CD73C424154#)
	  (x #00943480B228FC0D3D7ADFC91F680FC415E1306333#)
	  )
	 )
	 ))`)
	_, ok := readAccounts(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccounts_willReturnMoreThanOneAccount(t *testing.T) {
	from := inp(`(privkeys (account
(name "foo2")
(protocol libpurple-Jabberx)
(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A858#)
  )))
	(account
	(name "2")
	(protocol libpurple-jabber-gtalk)
	(private-key
	 (dsa
	  (p #00F24843F9447B62138AE49BF83188D1353ADA5CAC118890CFDEC01BF349D75E887B19C221665C7857CAD583AF656C67FB04A99FD8F8D69D09C9529C6C14D426F1E3924DC9243AF2970E3E4B04A23489A09E8A90E7E81EBA763AD4F0636B8A43415B6FC16A02C3624CE76272FA00783C8DB850D3A996B58136F7A0EB80AE0BC613#)
	  (q #00D16B2607FCBC0EDC639F763A54F34475B1CC8473#)
	  (g #00B15AFEF5F96EFEE41006F136C23A18849DA8133069A879D083F7C7AA362E187DAE3ED0C4F372D0D4E3AAE567008A1872A6E85D8F84E53A3FE1B352AF0B4E2F0CB033A6D34285ECD3E4A93653BDE99C3A8D840D9D35F82AC2FA8539DB6C7F7A1DAD77FEECD62803757FF1E2DE4CEC4A5A2AD643271514DDEEEF3D008F66FBF9DB#)
	  (y #01F9BE7DA0E4E84774048058B53202B2704BF688A306092ED533A55E68EABA814C8D62F45AAD8FF30C3055DCA461B7DBA6B78938FC4D69780A830C6457CC107F3D275C21D00E53147C14162176C77169D3BCA586DC30F15F4B482160E276869AA336F38AF7FC3686A764AB5A02C751D921A42B8B9AE8E06918059CD73C424154#)
	  (x #00943480B228FC0D3D7ADFC91F680FC415E1306333#)
	  )
	 )
	 )
	)`)
	k, ok := readAccounts(from)
	assertDeepEquals(t, k[0].Name, "foo2")
	assertDeepEquals(t, k[0].Protocol, "libpurple-Jabberx")
	assertDeepEquals(t, k[0].Key.(*DSAPrivateKey).PrivateKey.P, bnFromHex("00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A858"))
	assertDeepEquals(t, k[1].Name, "2")
	assertDeepEquals(t, k[1].Protocol, "libpurple-jabber-gtalk")
	assertDeepEquals(t, k[1].Key.(*DSAPrivateKey).PrivateKey.Q, bnFromHex("00D16B2607FCBC0EDC639F763A54F34475B1CC8473"))
	assertDeepEquals(t, ok, true)
}

func Test_PublicKey_parse_ParsePofAPublicKeyCorrectly(t *testing.T) {
	pk := &DSAPublicKey{}
	_, ok := pk.Parse(serializedPublicKey)

	assertDeepEquals(t, pk.P, bnFromHex("00F24843F9447B62138AE49BF83188D1353ADA5CAC118890CFDEC01BF349D75E887B19C221665C7857CAD583AF656C67FB04A99FD8F8D69D09C9529C6C14D426F1E3924DC9243AF2970E3E4B04A23489A09E8A90E7E81EBA763AD4F0636B8A43415B6FC16A02C3624CE76272FA00783C8DB850D3A996B58136F7A0EB80AE0BC613"))
	assertDeepEquals(t, ok, true)
}

func Test_PublicKey_parse_ReturnsNotOKIfThereIsTooLittleDataForTheKeyTypeTag(t *testing.T) {
	pk := &DSAPublicKey{}
	_, ok := pk.Parse([]byte{0x00})

	assertDeepEquals(t, ok, false)
}

func Test_PublicKey_parse_ReturnsNotOKIfTheTypeTagIsNotCorrect(t *testing.T) {
	pk := &DSAPublicKey{}
	_, ok := pk.Parse([]byte{
		// key type for something else
		0x00, 0x01,

		// len of p MPI
		0x00, 0x00, 0x00, 0x80,
		// p MPI
		0xF2, 0x48, 0x43, 0xF9, 0x44, 0x7B, 0x62, 0x13, 0x8A, 0xE4, 0x9B, 0xF8, 0x31, 0x88, 0xD1, 0x35, 0x3A, 0xDA, 0x5C, 0xAC, 0x11, 0x88, 0x90, 0xCF, 0xDE, 0xC0, 0x1B, 0xF3, 0x49, 0xD7, 0x5E, 0x88, 0x7B, 0x19, 0xC2, 0x21, 0x66, 0x5C, 0x78, 0x57, 0xCA, 0xD5, 0x83, 0xAF, 0x65, 0x6C, 0x67, 0xFB, 0x04, 0xA9, 0x9F, 0xD8, 0xF8, 0xD6, 0x9D, 0x09, 0xC9, 0x52, 0x9C, 0x6C, 0x14, 0xD4, 0x26, 0xF1, 0xE3, 0x92, 0x4D, 0xC9, 0x24, 0x3A, 0xF2, 0x97, 0x0E, 0x3E, 0x4B, 0x04, 0xA2, 0x34, 0x89, 0xA0, 0x9E, 0x8A, 0x90, 0xE7, 0xE8, 0x1E, 0xBA, 0x76, 0x3A, 0xD4, 0xF0, 0x63, 0x6B, 0x8A, 0x43, 0x41, 0x5B, 0x6F, 0xC1, 0x6A, 0x02, 0xC3, 0x62, 0x4C, 0xE7, 0x62, 0x72, 0xFA, 0x00, 0x78, 0x3C, 0x8D, 0xB8, 0x50, 0xD3, 0xA9, 0x96, 0xB5, 0x81, 0x36, 0xF7, 0xA0, 0xEB, 0x80, 0xAE, 0x0B, 0xC6, 0x13,

		// len of q MPI
		0x00, 0x00, 0x00, 0x14,
		// q MPI
		0xD1, 0x6B, 0x26, 0x07, 0xFC, 0xBC, 0x0E, 0xDC, 0x63, 0x9F, 0x76, 0x3A, 0x54, 0xF3, 0x44, 0x75, 0xB1, 0xCC, 0x84, 0x73,

		// len of g MPI
		0x00, 0x00, 0x00, 0x80,
		// g MPI
		0xB1, 0x5A, 0xFE, 0xF5, 0xF9, 0x6E, 0xFE, 0xE4, 0x10, 0x06, 0xF1, 0x36, 0xC2, 0x3A, 0x18, 0x84, 0x9D, 0xA8, 0x13, 0x30, 0x69, 0xA8, 0x79, 0xD0, 0x83, 0xF7, 0xC7, 0xAA, 0x36, 0x2E, 0x18, 0x7D, 0xAE, 0x3E, 0xD0, 0xC4, 0xF3, 0x72, 0xD0, 0xD4, 0xE3, 0xAA, 0xE5, 0x67, 0x00, 0x8A, 0x18, 0x72, 0xA6, 0xE8, 0x5D, 0x8F, 0x84, 0xE5, 0x3A, 0x3F, 0xE1, 0xB3, 0x52, 0xAF, 0x0B, 0x4E, 0x2F, 0x0C, 0xB0, 0x33, 0xA6, 0xD3, 0x42, 0x85, 0xEC, 0xD3, 0xE4, 0xA9, 0x36, 0x53, 0xBD, 0xE9, 0x9C, 0x3A, 0x8D, 0x84, 0x0D, 0x9D, 0x35, 0xF8, 0x2A, 0xC2, 0xFA, 0x85, 0x39, 0xDB, 0x6C, 0x7F, 0x7A, 0x1D, 0xAD, 0x77, 0xFE, 0xEC, 0xD6, 0x28, 0x03, 0x75, 0x7F, 0xF1, 0xE2, 0xDE, 0x4C, 0xEC, 0x4A, 0x5A, 0x2A, 0xD6, 0x43, 0x27, 0x15, 0x14, 0xDD, 0xEE, 0xEF, 0x3D, 0x00, 0x8F, 0x66, 0xFB, 0xF9, 0xDB,

		// len of y MPI
		0x00, 0x00, 0x00, 0x80,
		// y MPI
		0x01, 0xF9, 0xBE, 0x7D, 0xA0, 0xE4, 0xE8, 0x47, 0x74, 0x04, 0x80, 0x58, 0xB5, 0x32, 0x02, 0xB2, 0x70, 0x4B, 0xF6, 0x88, 0xA3, 0x06, 0x09, 0x2E, 0xD5, 0x33, 0xA5, 0x5E, 0x68, 0xEA, 0xBA, 0x81, 0x4C, 0x8D, 0x62, 0xF4, 0x5A, 0xAD, 0x8F, 0xF3, 0x0C, 0x30, 0x55, 0xDC, 0xA4, 0x61, 0xB7, 0xDB, 0xA6, 0xB7, 0x89, 0x38, 0xFC, 0x4D, 0x69, 0x78, 0x0A, 0x83, 0x0C, 0x64, 0x57, 0xCC, 0x10, 0x7F, 0x3D, 0x27, 0x5C, 0x21, 0xD0, 0x0E, 0x53, 0x14, 0x7C, 0x14, 0x16, 0x21, 0x76, 0xC7, 0x71, 0x69, 0xD3, 0xBC, 0xA5, 0x86, 0xDC, 0x30, 0xF1, 0x5F, 0x4B, 0x48, 0x21, 0x60, 0xE2, 0x76, 0x86, 0x9A, 0xA3, 0x36, 0xF3, 0x8A, 0xF7, 0xFC, 0x36, 0x86, 0xA7, 0x64, 0xAB, 0x5A, 0x02, 0xC7, 0x51, 0xD9, 0x21, 0xA4, 0x2B, 0x8B, 0x9A, 0xE8, 0xE0, 0x69, 0x18, 0x05, 0x9C, 0xD7, 0x3C, 0x42, 0x41, 0x54,
	})

	assertDeepEquals(t, ok, false)
}

func Test_PublicKey_parse_ReturnsNotOKIfPCannotBeParsedCorrectly(t *testing.T) {
	pk := &DSAPublicKey{}
	_, ok := pk.Parse([]byte{
		// key type for DSA
		0x00, 0x00,

		// len of p MPI
		0x00, 0x00, 0x10, 0x80,
		// p MPI
		0xF2, 0x48, 0x43, 0xF9, 0x44, 0x7B, 0x62, 0x13, 0x8A, 0xE4, 0x9B, 0xF8, 0x31, 0x88, 0xD1, 0x35, 0x3A, 0xDA, 0x5C, 0xAC, 0x11, 0x88, 0x90, 0xCF, 0xDE, 0xC0, 0x1B, 0xF3, 0x49, 0xD7, 0x5E, 0x88, 0x7B, 0x19, 0xC2, 0x21, 0x66, 0x5C, 0x78, 0x57, 0xCA, 0xD5, 0x83, 0xAF, 0x65, 0x6C, 0x67, 0xFB, 0x04, 0xA9, 0x9F, 0xD8, 0xF8, 0xD6, 0x9D, 0x09, 0xC9, 0x52, 0x9C, 0x6C, 0x14, 0xD4, 0x26, 0xF1, 0xE3, 0x92, 0x4D, 0xC9, 0x24, 0x3A, 0xF2, 0x97, 0x0E, 0x3E, 0x4B, 0x04, 0xA2, 0x34, 0x89, 0xA0, 0x9E, 0x8A, 0x90, 0xE7, 0xE8, 0x1E, 0xBA, 0x76, 0x3A, 0xD4, 0xF0, 0x63, 0x6B, 0x8A, 0x43, 0x41, 0x5B, 0x6F, 0xC1, 0x6A, 0x02, 0xC3, 0x62, 0x4C, 0xE7, 0x62, 0x72, 0xFA, 0x00, 0x78, 0x3C, 0x8D, 0xB8, 0x50, 0xD3, 0xA9, 0x96, 0xB5, 0x81, 0x36, 0xF7, 0xA0, 0xEB, 0x80, 0xAE, 0x0B, 0xC6, 0x13,

		// len of q MPI
		0x00, 0x00, 0x00, 0x14,
		// q MPI
		0xD1, 0x6B, 0x26, 0x07, 0xFC, 0xBC, 0x0E, 0xDC, 0x63, 0x9F, 0x76, 0x3A, 0x54, 0xF3, 0x44, 0x75, 0xB1, 0xCC, 0x84, 0x73,

		// len of g MPI
		0x00, 0x00, 0x00, 0x80,
		// g MPI
		0xB1, 0x5A, 0xFE, 0xF5, 0xF9, 0x6E, 0xFE, 0xE4, 0x10, 0x06, 0xF1, 0x36, 0xC2, 0x3A, 0x18, 0x84, 0x9D, 0xA8, 0x13, 0x30, 0x69, 0xA8, 0x79, 0xD0, 0x83, 0xF7, 0xC7, 0xAA, 0x36, 0x2E, 0x18, 0x7D, 0xAE, 0x3E, 0xD0, 0xC4, 0xF3, 0x72, 0xD0, 0xD4, 0xE3, 0xAA, 0xE5, 0x67, 0x00, 0x8A, 0x18, 0x72, 0xA6, 0xE8, 0x5D, 0x8F, 0x84, 0xE5, 0x3A, 0x3F, 0xE1, 0xB3, 0x52, 0xAF, 0x0B, 0x4E, 0x2F, 0x0C, 0xB0, 0x33, 0xA6, 0xD3, 0x42, 0x85, 0xEC, 0xD3, 0xE4, 0xA9, 0x36, 0x53, 0xBD, 0xE9, 0x9C, 0x3A, 0x8D, 0x84, 0x0D, 0x9D, 0x35, 0xF8, 0x2A, 0xC2, 0xFA, 0x85, 0x39, 0xDB, 0x6C, 0x7F, 0x7A, 0x1D, 0xAD, 0x77, 0xFE, 0xEC, 0xD6, 0x28, 0x03, 0x75, 0x7F, 0xF1, 0xE2, 0xDE, 0x4C, 0xEC, 0x4A, 0x5A, 0x2A, 0xD6, 0x43, 0x27, 0x15, 0x14, 0xDD, 0xEE, 0xEF, 0x3D, 0x00, 0x8F, 0x66, 0xFB, 0xF9, 0xDB,

		// len of y MPI
		0x00, 0x00, 0x00, 0x80,
		// y MPI
		0x01, 0xF9, 0xBE, 0x7D, 0xA0, 0xE4, 0xE8, 0x47, 0x74, 0x04, 0x80, 0x58, 0xB5, 0x32, 0x02, 0xB2, 0x70, 0x4B, 0xF6, 0x88, 0xA3, 0x06, 0x09, 0x2E, 0xD5, 0x33, 0xA5, 0x5E, 0x68, 0xEA, 0xBA, 0x81, 0x4C, 0x8D, 0x62, 0xF4, 0x5A, 0xAD, 0x8F, 0xF3, 0x0C, 0x30, 0x55, 0xDC, 0xA4, 0x61, 0xB7, 0xDB, 0xA6, 0xB7, 0x89, 0x38, 0xFC, 0x4D, 0x69, 0x78, 0x0A, 0x83, 0x0C, 0x64, 0x57, 0xCC, 0x10, 0x7F, 0x3D, 0x27, 0x5C, 0x21, 0xD0, 0x0E, 0x53, 0x14, 0x7C, 0x14, 0x16, 0x21, 0x76, 0xC7, 0x71, 0x69, 0xD3, 0xBC, 0xA5, 0x86, 0xDC, 0x30, 0xF1, 0x5F, 0x4B, 0x48, 0x21, 0x60, 0xE2, 0x76, 0x86, 0x9A, 0xA3, 0x36, 0xF3, 0x8A, 0xF7, 0xFC, 0x36, 0x86, 0xA7, 0x64, 0xAB, 0x5A, 0x02, 0xC7, 0x51, 0xD9, 0x21, 0xA4, 0x2B, 0x8B, 0x9A, 0xE8, 0xE0, 0x69, 0x18, 0x05, 0x9C, 0xD7, 0x3C, 0x42, 0x41, 0x54,
	})

	assertDeepEquals(t, ok, false)
}

func Test_PublicKey_parse_ReturnsNotOKIfQCannotBeParsedCorrectly(t *testing.T) {
	pk := &DSAPublicKey{}
	_, ok := pk.Parse([]byte{
		// key type for DSA
		0x00, 0x00,

		// len of p MPI
		0x00, 0x00, 0x00, 0x80,
		// p MPI
		0xF2, 0x48, 0x43, 0xF9, 0x44, 0x7B, 0x62, 0x13, 0x8A, 0xE4, 0x9B, 0xF8, 0x31, 0x88, 0xD1, 0x35, 0x3A, 0xDA, 0x5C, 0xAC, 0x11, 0x88, 0x90, 0xCF, 0xDE, 0xC0, 0x1B, 0xF3, 0x49, 0xD7, 0x5E, 0x88, 0x7B, 0x19, 0xC2, 0x21, 0x66, 0x5C, 0x78, 0x57, 0xCA, 0xD5, 0x83, 0xAF, 0x65, 0x6C, 0x67, 0xFB, 0x04, 0xA9, 0x9F, 0xD8, 0xF8, 0xD6, 0x9D, 0x09, 0xC9, 0x52, 0x9C, 0x6C, 0x14, 0xD4, 0x26, 0xF1, 0xE3, 0x92, 0x4D, 0xC9, 0x24, 0x3A, 0xF2, 0x97, 0x0E, 0x3E, 0x4B, 0x04, 0xA2, 0x34, 0x89, 0xA0, 0x9E, 0x8A, 0x90, 0xE7, 0xE8, 0x1E, 0xBA, 0x76, 0x3A, 0xD4, 0xF0, 0x63, 0x6B, 0x8A, 0x43, 0x41, 0x5B, 0x6F, 0xC1, 0x6A, 0x02, 0xC3, 0x62, 0x4C, 0xE7, 0x62, 0x72, 0xFA, 0x00, 0x78, 0x3C, 0x8D, 0xB8, 0x50, 0xD3, 0xA9, 0x96, 0xB5, 0x81, 0x36, 0xF7, 0xA0, 0xEB, 0x80, 0xAE, 0x0B, 0xC6, 0x13,

		// len of q MPI
		0x00, 0x00, 0x10, 0x14,
		// q MPI
		0xD1, 0x6B, 0x26, 0x07, 0xFC, 0xBC, 0x0E, 0xDC, 0x63, 0x9F, 0x76, 0x3A, 0x54, 0xF3, 0x44, 0x75, 0xB1, 0xCC, 0x84, 0x73,

		// len of g MPI
		0x00, 0x00, 0x00, 0x80,
		// g MPI
		0xB1, 0x5A, 0xFE, 0xF5, 0xF9, 0x6E, 0xFE, 0xE4, 0x10, 0x06, 0xF1, 0x36, 0xC2, 0x3A, 0x18, 0x84, 0x9D, 0xA8, 0x13, 0x30, 0x69, 0xA8, 0x79, 0xD0, 0x83, 0xF7, 0xC7, 0xAA, 0x36, 0x2E, 0x18, 0x7D, 0xAE, 0x3E, 0xD0, 0xC4, 0xF3, 0x72, 0xD0, 0xD4, 0xE3, 0xAA, 0xE5, 0x67, 0x00, 0x8A, 0x18, 0x72, 0xA6, 0xE8, 0x5D, 0x8F, 0x84, 0xE5, 0x3A, 0x3F, 0xE1, 0xB3, 0x52, 0xAF, 0x0B, 0x4E, 0x2F, 0x0C, 0xB0, 0x33, 0xA6, 0xD3, 0x42, 0x85, 0xEC, 0xD3, 0xE4, 0xA9, 0x36, 0x53, 0xBD, 0xE9, 0x9C, 0x3A, 0x8D, 0x84, 0x0D, 0x9D, 0x35, 0xF8, 0x2A, 0xC2, 0xFA, 0x85, 0x39, 0xDB, 0x6C, 0x7F, 0x7A, 0x1D, 0xAD, 0x77, 0xFE, 0xEC, 0xD6, 0x28, 0x03, 0x75, 0x7F, 0xF1, 0xE2, 0xDE, 0x4C, 0xEC, 0x4A, 0x5A, 0x2A, 0xD6, 0x43, 0x27, 0x15, 0x14, 0xDD, 0xEE, 0xEF, 0x3D, 0x00, 0x8F, 0x66, 0xFB, 0xF9, 0xDB,

		// len of y MPI
		0x00, 0x00, 0x00, 0x80,
		// y MPI
		0x01, 0xF9, 0xBE, 0x7D, 0xA0, 0xE4, 0xE8, 0x47, 0x74, 0x04, 0x80, 0x58, 0xB5, 0x32, 0x02, 0xB2, 0x70, 0x4B, 0xF6, 0x88, 0xA3, 0x06, 0x09, 0x2E, 0xD5, 0x33, 0xA5, 0x5E, 0x68, 0xEA, 0xBA, 0x81, 0x4C, 0x8D, 0x62, 0xF4, 0x5A, 0xAD, 0x8F, 0xF3, 0x0C, 0x30, 0x55, 0xDC, 0xA4, 0x61, 0xB7, 0xDB, 0xA6, 0xB7, 0x89, 0x38, 0xFC, 0x4D, 0x69, 0x78, 0x0A, 0x83, 0x0C, 0x64, 0x57, 0xCC, 0x10, 0x7F, 0x3D, 0x27, 0x5C, 0x21, 0xD0, 0x0E, 0x53, 0x14, 0x7C, 0x14, 0x16, 0x21, 0x76, 0xC7, 0x71, 0x69, 0xD3, 0xBC, 0xA5, 0x86, 0xDC, 0x30, 0xF1, 0x5F, 0x4B, 0x48, 0x21, 0x60, 0xE2, 0x76, 0x86, 0x9A, 0xA3, 0x36, 0xF3, 0x8A, 0xF7, 0xFC, 0x36, 0x86, 0xA7, 0x64, 0xAB, 0x5A, 0x02, 0xC7, 0x51, 0xD9, 0x21, 0xA4, 0x2B, 0x8B, 0x9A, 0xE8, 0xE0, 0x69, 0x18, 0x05, 0x9C, 0xD7, 0x3C, 0x42, 0x41, 0x54,
	})

	assertDeepEquals(t, ok, false)
}

func Test_PublicKey_parse_ReturnsNotOKIfGCannotBeParsedCorrectly(t *testing.T) {
	pk := &DSAPublicKey{}
	_, ok := pk.Parse([]byte{
		// key type for DSA
		0x00, 0x00,

		// len of p MPI
		0x00, 0x00, 0x00, 0x80,
		// p MPI
		0xF2, 0x48, 0x43, 0xF9, 0x44, 0x7B, 0x62, 0x13, 0x8A, 0xE4, 0x9B, 0xF8, 0x31, 0x88, 0xD1, 0x35, 0x3A, 0xDA, 0x5C, 0xAC, 0x11, 0x88, 0x90, 0xCF, 0xDE, 0xC0, 0x1B, 0xF3, 0x49, 0xD7, 0x5E, 0x88, 0x7B, 0x19, 0xC2, 0x21, 0x66, 0x5C, 0x78, 0x57, 0xCA, 0xD5, 0x83, 0xAF, 0x65, 0x6C, 0x67, 0xFB, 0x04, 0xA9, 0x9F, 0xD8, 0xF8, 0xD6, 0x9D, 0x09, 0xC9, 0x52, 0x9C, 0x6C, 0x14, 0xD4, 0x26, 0xF1, 0xE3, 0x92, 0x4D, 0xC9, 0x24, 0x3A, 0xF2, 0x97, 0x0E, 0x3E, 0x4B, 0x04, 0xA2, 0x34, 0x89, 0xA0, 0x9E, 0x8A, 0x90, 0xE7, 0xE8, 0x1E, 0xBA, 0x76, 0x3A, 0xD4, 0xF0, 0x63, 0x6B, 0x8A, 0x43, 0x41, 0x5B, 0x6F, 0xC1, 0x6A, 0x02, 0xC3, 0x62, 0x4C, 0xE7, 0x62, 0x72, 0xFA, 0x00, 0x78, 0x3C, 0x8D, 0xB8, 0x50, 0xD3, 0xA9, 0x96, 0xB5, 0x81, 0x36, 0xF7, 0xA0, 0xEB, 0x80, 0xAE, 0x0B, 0xC6, 0x13,

		// len of q MPI
		0x00, 0x00, 0x00, 0x14,
		// q MPI
		0xD1, 0x6B, 0x26, 0x07, 0xFC, 0xBC, 0x0E, 0xDC, 0x63, 0x9F, 0x76, 0x3A, 0x54, 0xF3, 0x44, 0x75, 0xB1, 0xCC, 0x84, 0x73,

		// len of g MPI
		0x00, 0x00, 0x10, 0x80,
		// g MPI
		0xB1, 0x5A, 0xFE, 0xF5, 0xF9, 0x6E, 0xFE, 0xE4, 0x10, 0x06, 0xF1, 0x36, 0xC2, 0x3A, 0x18, 0x84, 0x9D, 0xA8, 0x13, 0x30, 0x69, 0xA8, 0x79, 0xD0, 0x83, 0xF7, 0xC7, 0xAA, 0x36, 0x2E, 0x18, 0x7D, 0xAE, 0x3E, 0xD0, 0xC4, 0xF3, 0x72, 0xD0, 0xD4, 0xE3, 0xAA, 0xE5, 0x67, 0x00, 0x8A, 0x18, 0x72, 0xA6, 0xE8, 0x5D, 0x8F, 0x84, 0xE5, 0x3A, 0x3F, 0xE1, 0xB3, 0x52, 0xAF, 0x0B, 0x4E, 0x2F, 0x0C, 0xB0, 0x33, 0xA6, 0xD3, 0x42, 0x85, 0xEC, 0xD3, 0xE4, 0xA9, 0x36, 0x53, 0xBD, 0xE9, 0x9C, 0x3A, 0x8D, 0x84, 0x0D, 0x9D, 0x35, 0xF8, 0x2A, 0xC2, 0xFA, 0x85, 0x39, 0xDB, 0x6C, 0x7F, 0x7A, 0x1D, 0xAD, 0x77, 0xFE, 0xEC, 0xD6, 0x28, 0x03, 0x75, 0x7F, 0xF1, 0xE2, 0xDE, 0x4C, 0xEC, 0x4A, 0x5A, 0x2A, 0xD6, 0x43, 0x27, 0x15, 0x14, 0xDD, 0xEE, 0xEF, 0x3D, 0x00, 0x8F, 0x66, 0xFB, 0xF9, 0xDB,

		// len of y MPI
		0x00, 0x00, 0x00, 0x80,
		// y MPI
		0x01, 0xF9, 0xBE, 0x7D, 0xA0, 0xE4, 0xE8, 0x47, 0x74, 0x04, 0x80, 0x58, 0xB5, 0x32, 0x02, 0xB2, 0x70, 0x4B, 0xF6, 0x88, 0xA3, 0x06, 0x09, 0x2E, 0xD5, 0x33, 0xA5, 0x5E, 0x68, 0xEA, 0xBA, 0x81, 0x4C, 0x8D, 0x62, 0xF4, 0x5A, 0xAD, 0x8F, 0xF3, 0x0C, 0x30, 0x55, 0xDC, 0xA4, 0x61, 0xB7, 0xDB, 0xA6, 0xB7, 0x89, 0x38, 0xFC, 0x4D, 0x69, 0x78, 0x0A, 0x83, 0x0C, 0x64, 0x57, 0xCC, 0x10, 0x7F, 0x3D, 0x27, 0x5C, 0x21, 0xD0, 0x0E, 0x53, 0x14, 0x7C, 0x14, 0x16, 0x21, 0x76, 0xC7, 0x71, 0x69, 0xD3, 0xBC, 0xA5, 0x86, 0xDC, 0x30, 0xF1, 0x5F, 0x4B, 0x48, 0x21, 0x60, 0xE2, 0x76, 0x86, 0x9A, 0xA3, 0x36, 0xF3, 0x8A, 0xF7, 0xFC, 0x36, 0x86, 0xA7, 0x64, 0xAB, 0x5A, 0x02, 0xC7, 0x51, 0xD9, 0x21, 0xA4, 0x2B, 0x8B, 0x9A, 0xE8, 0xE0, 0x69, 0x18, 0x05, 0x9C, 0xD7, 0x3C, 0x42, 0x41, 0x54,
	})

	assertDeepEquals(t, ok, false)
}

func Test_PublicKey_parse_ReturnsNotOKIfYCannotBeParsedCorrectly(t *testing.T) {
	pk := &DSAPublicKey{}
	_, ok := pk.Parse([]byte{
		// key type for DSA
		0x00, 0x00,

		// len of p MPI
		0x00, 0x00, 0x00, 0x80,
		// p MPI
		0xF2, 0x48, 0x43, 0xF9, 0x44, 0x7B, 0x62, 0x13, 0x8A, 0xE4, 0x9B, 0xF8, 0x31, 0x88, 0xD1, 0x35, 0x3A, 0xDA, 0x5C, 0xAC, 0x11, 0x88, 0x90, 0xCF, 0xDE, 0xC0, 0x1B, 0xF3, 0x49, 0xD7, 0x5E, 0x88, 0x7B, 0x19, 0xC2, 0x21, 0x66, 0x5C, 0x78, 0x57, 0xCA, 0xD5, 0x83, 0xAF, 0x65, 0x6C, 0x67, 0xFB, 0x04, 0xA9, 0x9F, 0xD8, 0xF8, 0xD6, 0x9D, 0x09, 0xC9, 0x52, 0x9C, 0x6C, 0x14, 0xD4, 0x26, 0xF1, 0xE3, 0x92, 0x4D, 0xC9, 0x24, 0x3A, 0xF2, 0x97, 0x0E, 0x3E, 0x4B, 0x04, 0xA2, 0x34, 0x89, 0xA0, 0x9E, 0x8A, 0x90, 0xE7, 0xE8, 0x1E, 0xBA, 0x76, 0x3A, 0xD4, 0xF0, 0x63, 0x6B, 0x8A, 0x43, 0x41, 0x5B, 0x6F, 0xC1, 0x6A, 0x02, 0xC3, 0x62, 0x4C, 0xE7, 0x62, 0x72, 0xFA, 0x00, 0x78, 0x3C, 0x8D, 0xB8, 0x50, 0xD3, 0xA9, 0x96, 0xB5, 0x81, 0x36, 0xF7, 0xA0, 0xEB, 0x80, 0xAE, 0x0B, 0xC6, 0x13,

		// len of q MPI
		0x00, 0x00, 0x00, 0x14,
		// q MPI
		0xD1, 0x6B, 0x26, 0x07, 0xFC, 0xBC, 0x0E, 0xDC, 0x63, 0x9F, 0x76, 0x3A, 0x54, 0xF3, 0x44, 0x75, 0xB1, 0xCC, 0x84, 0x73,

		// len of g MPI
		0x00, 0x00, 0x00, 0x80,
		// g MPI
		0xB1, 0x5A, 0xFE, 0xF5, 0xF9, 0x6E, 0xFE, 0xE4, 0x10, 0x06, 0xF1, 0x36, 0xC2, 0x3A, 0x18, 0x84, 0x9D, 0xA8, 0x13, 0x30, 0x69, 0xA8, 0x79, 0xD0, 0x83, 0xF7, 0xC7, 0xAA, 0x36, 0x2E, 0x18, 0x7D, 0xAE, 0x3E, 0xD0, 0xC4, 0xF3, 0x72, 0xD0, 0xD4, 0xE3, 0xAA, 0xE5, 0x67, 0x00, 0x8A, 0x18, 0x72, 0xA6, 0xE8, 0x5D, 0x8F, 0x84, 0xE5, 0x3A, 0x3F, 0xE1, 0xB3, 0x52, 0xAF, 0x0B, 0x4E, 0x2F, 0x0C, 0xB0, 0x33, 0xA6, 0xD3, 0x42, 0x85, 0xEC, 0xD3, 0xE4, 0xA9, 0x36, 0x53, 0xBD, 0xE9, 0x9C, 0x3A, 0x8D, 0x84, 0x0D, 0x9D, 0x35, 0xF8, 0x2A, 0xC2, 0xFA, 0x85, 0x39, 0xDB, 0x6C, 0x7F, 0x7A, 0x1D, 0xAD, 0x77, 0xFE, 0xEC, 0xD6, 0x28, 0x03, 0x75, 0x7F, 0xF1, 0xE2, 0xDE, 0x4C, 0xEC, 0x4A, 0x5A, 0x2A, 0xD6, 0x43, 0x27, 0x15, 0x14, 0xDD, 0xEE, 0xEF, 0x3D, 0x00, 0x8F, 0x66, 0xFB, 0xF9, 0xDB,

		// len of y MPI
		0x00, 0x00, 0x10, 0x80,
		// y MPI
		0x01, 0xF9, 0xBE, 0x7D, 0xA0, 0xE4, 0xE8, 0x47, 0x74, 0x04, 0x80, 0x58, 0xB5, 0x32, 0x02, 0xB2, 0x70, 0x4B, 0xF6, 0x88, 0xA3, 0x06, 0x09, 0x2E, 0xD5, 0x33, 0xA5, 0x5E, 0x68, 0xEA, 0xBA, 0x81, 0x4C, 0x8D, 0x62, 0xF4, 0x5A, 0xAD, 0x8F, 0xF3, 0x0C, 0x30, 0x55, 0xDC, 0xA4, 0x61, 0xB7, 0xDB, 0xA6, 0xB7, 0x89, 0x38, 0xFC, 0x4D, 0x69, 0x78, 0x0A, 0x83, 0x0C, 0x64, 0x57, 0xCC, 0x10, 0x7F, 0x3D, 0x27, 0x5C, 0x21, 0xD0, 0x0E, 0x53, 0x14, 0x7C, 0x14, 0x16, 0x21, 0x76, 0xC7, 0x71, 0x69, 0xD3, 0xBC, 0xA5, 0x86, 0xDC, 0x30, 0xF1, 0x5F, 0x4B, 0x48, 0x21, 0x60, 0xE2, 0x76, 0x86, 0x9A, 0xA3, 0x36, 0xF3, 0x8A, 0xF7, 0xFC, 0x36, 0x86, 0xA7, 0x64, 0xAB, 0x5A, 0x02, 0xC7, 0x51, 0xD9, 0x21, 0xA4, 0x2B, 0x8B, 0x9A, 0xE8, 0xE0, 0x69, 0x18, 0x05, 0x9C, 0xD7, 0x3C, 0x42, 0x41, 0x54,
	})

	assertDeepEquals(t, ok, false)
}

func Test_PublicKey_parse_ReturnsNotOKIfParametersAreMissing(t *testing.T) {
	pk := &DSAPublicKey{}
	_, ok := pk.Parse([]byte{
		// key type for DSA
		0x00, 0x00,

		// len of p MPI
		0x00, 0x00, 0x00, 0x80,
		// p MPI
		0xF2, 0x48, 0x43, 0xF9, 0x44, 0x7B, 0x62, 0x13, 0x8A, 0xE4, 0x9B, 0xF8, 0x31, 0x88, 0xD1, 0x35, 0x3A, 0xDA, 0x5C, 0xAC, 0x11, 0x88, 0x90, 0xCF, 0xDE, 0xC0, 0x1B, 0xF3, 0x49, 0xD7, 0x5E, 0x88, 0x7B, 0x19, 0xC2, 0x21, 0x66, 0x5C, 0x78, 0x57, 0xCA, 0xD5, 0x83, 0xAF, 0x65, 0x6C, 0x67, 0xFB, 0x04, 0xA9, 0x9F, 0xD8, 0xF8, 0xD6, 0x9D, 0x09, 0xC9, 0x52, 0x9C, 0x6C, 0x14, 0xD4, 0x26, 0xF1, 0xE3, 0x92, 0x4D, 0xC9, 0x24, 0x3A, 0xF2, 0x97, 0x0E, 0x3E, 0x4B, 0x04, 0xA2, 0x34, 0x89, 0xA0, 0x9E, 0x8A, 0x90, 0xE7, 0xE8, 0x1E, 0xBA, 0x76, 0x3A, 0xD4, 0xF0, 0x63, 0x6B, 0x8A, 0x43, 0x41, 0x5B, 0x6F, 0xC1, 0x6A, 0x02, 0xC3, 0x62, 0x4C, 0xE7, 0x62, 0x72, 0xFA, 0x00, 0x78, 0x3C, 0x8D, 0xB8, 0x50, 0xD3, 0xA9, 0x96, 0xB5, 0x81, 0x36, 0xF7, 0xA0, 0xEB, 0x80, 0xAE, 0x0B, 0xC6, 0x13,

		// len of q MPI
		0x00, 0x00, 0x00, 0x14,
		// q MPI
		0xD1, 0x6B, 0x26, 0x07, 0xFC, 0xBC, 0x0E, 0xDC, 0x63, 0x9F, 0x76, 0x3A, 0x54, 0xF3, 0x44, 0x75, 0xB1, 0xCC, 0x84, 0x73,
	})

	assertDeepEquals(t, ok, false)
}

func Test_PublicKey_parse_ParseQofAPublicKeyCorrectly(t *testing.T) {
	pk := &DSAPublicKey{}
	pk.Parse(serializedPublicKey)

	assertDeepEquals(t, pk.Q, bnFromHex("00D16B2607FCBC0EDC639F763A54F34475B1CC8473"))
}

func Test_PublicKey_parse_ParseGofAPublicKeyCorrectly(t *testing.T) {
	pk := &DSAPublicKey{}
	pk.Parse(serializedPublicKey)

	assertDeepEquals(t, pk.G, bnFromHex("00B15AFEF5F96EFEE41006F136C23A18849DA8133069A879D083F7C7AA362E187DAE3ED0C4F372D0D4E3AAE567008A1872A6E85D8F84E53A3FE1B352AF0B4E2F0CB033A6D34285ECD3E4A93653BDE99C3A8D840D9D35F82AC2FA8539DB6C7F7A1DAD77FEECD62803757FF1E2DE4CEC4A5A2AD643271514DDEEEF3D008F66FBF9DB"))
}

func Test_PublicKey_parse_ParseYofAPublicKeyCorrectly(t *testing.T) {
	pk := &DSAPublicKey{}
	pk.Parse(serializedPublicKey)

	assertDeepEquals(t, pk.Y, bnFromHex("01F9BE7DA0E4E84774048058B53202B2704BF688A306092ED533A55E68EABA814C8D62F45AAD8FF30C3055DCA461B7DBA6B78938FC4D69780A830C6457CC107F3D275C21D00E53147C14162176C77169D3BCA586DC30F15F4B482160E276869AA336F38AF7FC3686A764AB5A02C751D921A42B8B9AE8E06918059CD73C424154"))
}

func Test_PrivateKey_parse_ParsePublicKeyofAPrivateKeyCorrectly(t *testing.T) {
	priv := &DSAPrivateKey{}
	pk := &DSAPublicKey{}
	priv.Parse(serializedPrivateKey)
	pk.Parse(serializedPublicKey)

	assertDeepEquals(t, priv.PublicKey(), pk)
}

func Test_PrivateKey_parse_ParseXofAPrivateKeyCorrectly(t *testing.T) {
	priv := &DSAPrivateKey{}
	_, ok := priv.Parse(serializedPrivateKey)
	assertDeepEquals(t, priv.X, bnFromHex("14D0345A3562C480A039E3C72764F72D79043216"))
	assertDeepEquals(t, ok, true)
}

func Test_PrivateKey_parse_ReturnsNotOKIfPublicKeyIsNotOK(t *testing.T) {
	priv := &DSAPrivateKey{}
	_, ok := priv.Parse([]byte{
		// key type for DSA
		0x00, 0x00,

		// len of p MPI
		0x00, 0x00, 0x00, 0x80,
		// p MPI
		0xF2, 0x48, 0x43, 0xF9, 0x44, 0x7B, 0x62, 0x13, 0x8A, 0xE4, 0x9B, 0xF8, 0x31, 0x88, 0xD1, 0x35, 0x3A, 0xDA, 0x5C, 0xAC, 0x11, 0x88, 0x90, 0xCF, 0xDE, 0xC0, 0x1B, 0xF3, 0x49, 0xD7, 0x5E, 0x88, 0x7B, 0x19, 0xC2, 0x21, 0x66, 0x5C, 0x78, 0x57, 0xCA,

		// len of x MPI
		0x00, 0x00, 0x00, 0x14,
		// x MPI
		0x14, 0xD0, 0x34, 0x5A, 0x35, 0x62, 0xC4, 0x80, 0xA0, 0x39, 0xE3, 0xC7, 0x27, 0x64, 0xF7, 0x2D, 0x79, 0x04, 0x32, 0x16,
	})
	assertDeepEquals(t, ok, false)
}

func Test_PrivateKey_parse_ReturnsNotOKIfPrivateKeyIsNotOK(t *testing.T) {
	priv := &DSAPrivateKey{}
	_, ok := priv.Parse([]byte{
		// key type for DSA
		0x00, 0x00,

		// len of p MPI
		0x00, 0x00, 0x00, 0x80,
		// p MPI
		0xF2, 0x48, 0x43, 0xF9, 0x44, 0x7B, 0x62, 0x13, 0x8A, 0xE4, 0x9B, 0xF8, 0x31, 0x88, 0xD1, 0x35, 0x3A, 0xDA, 0x5C, 0xAC, 0x11, 0x88, 0x90, 0xCF, 0xDE, 0xC0, 0x1B, 0xF3, 0x49, 0xD7, 0x5E, 0x88, 0x7B, 0x19, 0xC2, 0x21, 0x66, 0x5C, 0x78, 0x57, 0xCA, 0xD5, 0x83, 0xAF, 0x65, 0x6C, 0x67, 0xFB, 0x04, 0xA9, 0x9F, 0xD8, 0xF8, 0xD6, 0x9D, 0x09, 0xC9, 0x52, 0x9C, 0x6C, 0x14, 0xD4, 0x26, 0xF1, 0xE3, 0x92, 0x4D, 0xC9, 0x24, 0x3A, 0xF2, 0x97, 0x0E, 0x3E, 0x4B, 0x04, 0xA2, 0x34, 0x89, 0xA0, 0x9E, 0x8A, 0x90, 0xE7, 0xE8, 0x1E, 0xBA, 0x76, 0x3A, 0xD4, 0xF0, 0x63, 0x6B, 0x8A, 0x43, 0x41, 0x5B, 0x6F, 0xC1, 0x6A, 0x02, 0xC3, 0x62, 0x4C, 0xE7, 0x62, 0x72, 0xFA, 0x00, 0x78, 0x3C, 0x8D, 0xB8, 0x50, 0xD3, 0xA9, 0x96, 0xB5, 0x81, 0x36, 0xF7, 0xA0, 0xEB, 0x80, 0xAE, 0x0B, 0xC6, 0x13,

		// len of q MPI
		0x00, 0x00, 0x00, 0x14,
		// q MPI
		0xD1, 0x6B, 0x26, 0x07, 0xFC, 0xBC, 0x0E, 0xDC, 0x63, 0x9F, 0x76, 0x3A, 0x54, 0xF3, 0x44, 0x75, 0xB1, 0xCC, 0x84, 0x73,

		// len of g MPI
		0x00, 0x00, 0x00, 0x80,
		// g MPI
		0xB1, 0x5A, 0xFE, 0xF5, 0xF9, 0x6E, 0xFE, 0xE4, 0x10, 0x06, 0xF1, 0x36, 0xC2, 0x3A, 0x18, 0x84, 0x9D, 0xA8, 0x13, 0x30, 0x69, 0xA8, 0x79, 0xD0, 0x83, 0xF7, 0xC7, 0xAA, 0x36, 0x2E, 0x18, 0x7D, 0xAE, 0x3E, 0xD0, 0xC4, 0xF3, 0x72, 0xD0, 0xD4, 0xE3, 0xAA, 0xE5, 0x67, 0x00, 0x8A, 0x18, 0x72, 0xA6, 0xE8, 0x5D, 0x8F, 0x84, 0xE5, 0x3A, 0x3F, 0xE1, 0xB3, 0x52, 0xAF, 0x0B, 0x4E, 0x2F, 0x0C, 0xB0, 0x33, 0xA6, 0xD3, 0x42, 0x85, 0xEC, 0xD3, 0xE4, 0xA9, 0x36, 0x53, 0xBD, 0xE9, 0x9C, 0x3A, 0x8D, 0x84, 0x0D, 0x9D, 0x35, 0xF8, 0x2A, 0xC2, 0xFA, 0x85, 0x39, 0xDB, 0x6C, 0x7F, 0x7A, 0x1D, 0xAD, 0x77, 0xFE, 0xEC, 0xD6, 0x28, 0x03, 0x75, 0x7F, 0xF1, 0xE2, 0xDE, 0x4C, 0xEC, 0x4A, 0x5A, 0x2A, 0xD6, 0x43, 0x27, 0x15, 0x14, 0xDD, 0xEE, 0xEF, 0x3D, 0x00, 0x8F, 0x66, 0xFB, 0xF9, 0xDB,

		// len of y MPI
		0x00, 0x00, 0x00, 0x80,
		// y MPI
		0x01, 0xF9, 0xBE, 0x7D, 0xA0, 0xE4, 0xE8, 0x47, 0x74, 0x04, 0x80, 0x58, 0xB5, 0x32, 0x02, 0xB2, 0x70, 0x4B, 0xF6, 0x88, 0xA3, 0x06, 0x09, 0x2E, 0xD5, 0x33, 0xA5, 0x5E, 0x68, 0xEA, 0xBA, 0x81, 0x4C, 0x8D, 0x62, 0xF4, 0x5A, 0xAD, 0x8F, 0xF3, 0x0C, 0x30, 0x55, 0xDC, 0xA4, 0x61, 0xB7, 0xDB, 0xA6, 0xB7, 0x89, 0x38, 0xFC, 0x4D, 0x69, 0x78, 0x0A, 0x83, 0x0C, 0x64, 0x57, 0xCC, 0x10, 0x7F, 0x3D, 0x27, 0x5C, 0x21, 0xD0, 0x0E, 0x53, 0x14, 0x7C, 0x14, 0x16, 0x21, 0x76, 0xC7, 0x71, 0x69, 0xD3, 0xBC, 0xA5, 0x86, 0xDC, 0x30, 0xF1, 0x5F, 0x4B, 0x48, 0x21, 0x60, 0xE2, 0x76, 0x86, 0x9A, 0xA3, 0x36, 0xF3, 0x8A, 0xF7, 0xFC, 0x36, 0x86, 0xA7, 0x64, 0xAB, 0x5A, 0x02, 0xC7, 0x51, 0xD9, 0x21, 0xA4, 0x2B, 0x8B, 0x9A, 0xE8, 0xE0, 0x69, 0x18, 0x05, 0x9C, 0xD7, 0x3C, 0x42, 0x41, 0x54,

		// len of x MPI
		0x00, 0x00, 0x10, 0x14,
		// x MPI
		0x14, 0xD0, 0x34, 0x5A, 0x35, 0x62, 0xC4, 0x80, 0xA0, 0x39, 0xE3, 0xC7, 0x27, 0x64, 0xF7, 0x2D, 0x79, 0x04, 0x32, 0x16,
	})
	assertDeepEquals(t, ok, false)
}

func Test_PublicKey_serialize_willSerializeAPublicKeyCorrectly(t *testing.T) {
	pk := &DSAPublicKey{}
	pk.Parse(serializedPublicKey)
	result := pk.serialize()
	assertDeepEquals(t, result, serializedPublicKey)
}

func Test_PublicKey_serialize_returnsEmptyForNil(t *testing.T) {
	pk := &DSAPublicKey{}
	result := pk.serialize()
	assertNil(t, result)
}

func Test_PrivateKey_roundTripGeneratesCorrectValue(t *testing.T) {
	pk := &DSAPrivateKey{}
	pk.Parse(serializedPrivateKey)
	result := pk.serialize()
	assertDeepEquals(t, result, serializedPrivateKey)
}

func Test_PublicKey_fingerprint_willGenerateACorrectFingerprint(t *testing.T) {
	priv := parseIntoPrivateKey("000000000080c81c2cb2eb729b7e6fd48e975a932c638b3a9055478583afa46755683e30102447f6da2d8bec9f386bbb5da6403b0040fee8650b6ab2d7f32c55ab017ae9b6aec8c324ab5844784e9a80e194830d548fb7f09a0410df2c4d5c8bc2b3e9ad484e65412be689cf0834694e0839fb2954021521ffdffb8f5c32c14dbf2020b3ce7500000014da4591d58def96de61aea7b04a8405fe1609308d000000808ddd5cb0b9d66956e3dea5a915d9aba9d8a6e7053b74dadb2fc52f9fe4e5bcc487d2305485ed95fed026ad93f06ebb8c9e8baf693b7887132c7ffdd3b0f72f4002ff4ed56583ca7c54458f8c068ca3e8a4dfa309d1dd5d34e2a4b68e6f4338835e5e0fb4317c9e4c7e4806dafda3ef459cd563775a586dd91b1319f72621bf3f00000080b8147e74d8c45e6318c37731b8b33b984a795b3653c2cd1d65cc99efe097cb7eb2fa49569bab5aab6e8a1c261a27d0f7840a5e80b317e6683042b59b6dceca2879c6ffc877a465be690c15e4a42f9a7588e79b10faac11b1ce3741fcef7aba8ce05327a2c16d279ee1b3d77eb783fb10e3356caa25635331e26dd42b8396c4d00000001420bec691fea37ecea58a5c717142f0b804452f57")
	expectedFingerprint := bytesFromHex("0bb01c360424522e94ee9c346ce877a1a4288b2f")
	assertDeepEquals(t, priv.PublicKey().Fingerprint(), expectedFingerprint)
}

func Test_PublicKey_fingerprint_generatesForEmpty(t *testing.T) {
	pk := &DSAPublicKey{}
	result := pk.Fingerprint()
	assertNil(t, result)
}

func Test_PublicKey_Verify_willReturnOK(t *testing.T) {
	pk := &DSAPublicKey{}
	pk.Parse(bytesFromHex("000000000080a5138eb3d3eb9c1d85716faecadb718f87d31aaed1157671d7fee7e488f95e8e0ba60ad449ec732710a7dec5190f7182af2e2f98312d98497221dff160fd68033dd4f3a33b7c078d0d9f66e26847e76ca7447d4bab35486045090572863d9e4454777f24d6706f63e02548dfec2d0a620af37bbc1d24f884708a212c343b480d00000014e9c58f0ea21a5e4dfd9f44b6a9f7f6a9961a8fa9000000803c4d111aebd62d3c50c2889d420a32cdf1e98b70affcc1fcf44d59cca2eb019f6b774ef88153fb9b9615441a5fe25ea2d11b74ce922ca0232bd81b3c0fcac2a95b20cb6e6c0c5c1ace2e26f65dc43c751af0edbb10d669890e8ab6beea91410b8b2187af1a8347627a06ecea7e0f772c28aae9461301e83884860c9b656c722f0000008065af8625a555ea0e008cd04743671a3cda21162e83af045725db2eb2bb52712708dc0cc1a84c08b3649b88a966974bde27d8612c2861792ec9f08786a246fcadd6d8d3a81a32287745f309238f47618c2bd7612cb8b02d940571e0f30b96420bcd462ff542901b46109b1e5ad6423744448d20a57818a8cbb1647d0fea3b664e"))
	hashed := bytesFromHex("122773a99f5eafbaaa04b419b5c417b9949ce11bf199ea1bee3586619b94bb29")
	sig := bytesFromHex("2e9e1c92773e6e51541f47f674f17e24138c48af2c86aea3e5689bd9116b5d6d28562c0aeb84a989")
	result, ok := pk.Verify(hashed, sig)
	assertEquals(t, ok, true)
	assertDeepEquals(t, result, sig[20*2:])
}

func Test_PublicKey_Verify_willReturnNotOK(t *testing.T) {
	pk := &DSAPublicKey{}
	pk.Parse(serializedPublicKey)
	hashed := bytesFromHex("122773a99f5eafbaaa04b419b5c417b9949ce11bf199ea1bee3586619b94bb29")
	sig := bytesFromHex("2e9e1c92773e6e51541f47f674f17e24138c48af2c86aea3e5689bd9116b5d6d28562c0aeb84a989")
	result, ok := pk.Verify(hashed, sig)
	assertEquals(t, ok, false)
	assertDeepEquals(t, result, sig[20*2:])
}

func Test_readAccountName_willSignalNotOKIfNoListIsGiven(t *testing.T) {
	from := inp(`name`)
	_, ok := readAccountName(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountName_willSignalNotOKIfNoCompleteListIsGiven(t *testing.T) {
	from := inp(`(name "foo"`)
	_, ok := readAccountName(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountName_willSignalNotOKIfNoNameValueIsGiven(t *testing.T) {
	from := inp(`(name)`)
	_, ok := readAccountName(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountName_willSignalNotOKIfNoTagIsGiven(t *testing.T) {
	from := inp(`()`)
	_, ok := readAccountName(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountName_willSignalNotOKIfTagIsTheWrongType(t *testing.T) {
	from := inp(`("blarg" "foo")`)
	_, ok := readAccountName(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountName_willSignalNotOKIfTagIsNotTheSymbolName(t *testing.T) {
	from := inp(`(namex "foo")`)
	_, ok := readAccountName(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountName_willSignalNotOKIfValueIsTheWrongType(t *testing.T) {
	from := inp(`(name #42)`)
	_, ok := readAccountName(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountName_willSignalOKIfTagAndValueIsCorrect(t *testing.T) {
	from := inp(`(name "foo")`)
	_, ok := readAccountName(from)
	assertDeepEquals(t, ok, true)
}

func Test_readAccountName_willSignalOKIfTagAndValueAsSymbolIsCorrect(t *testing.T) {
	from := inp(`(name foo)`)
	_, ok := readAccountName(from)
	assertDeepEquals(t, ok, true)
}

func Test_readAccountProtocol_willSignalNotOKIfNoListIsGiven(t *testing.T) {
	from := inp(`protocol`)
	_, ok := readAccountProtocol(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountProtocol_willSignalNotOKIfNoCompleteListIsGiven(t *testing.T) {
	from := inp(`(protocol libpurple`)
	_, ok := readAccountProtocol(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountProtocol_willSignalNotOKIfNoProtocolValueIsGiven(t *testing.T) {
	from := inp(`(protocol)`)
	_, ok := readAccountProtocol(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountProtocol_willSignalNotOKIfNoTagIsGiven(t *testing.T) {
	from := inp(`()`)
	_, ok := readAccountProtocol(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountProtocol_willSignalNotOKIfTagIsTheWrongType(t *testing.T) {
	from := inp(`("protocol" libpurple)`)
	_, ok := readAccountProtocol(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountProtocol_willSignalNotOKIfTagIsNotTheSymbolProtocol(t *testing.T) {
	from := inp(`(protocolx libpurple)`)
	_, ok := readAccountProtocol(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountProtocol_willSignalNotOKIfValueIsTheWrongType(t *testing.T) {
	from := inp(`(protocol "libpurple")`)
	_, ok := readAccountProtocol(from)
	assertDeepEquals(t, ok, false)
}

func Test_readAccountProtocol_willSignalOKIfTagAndValueIsCorrect(t *testing.T) {
	from := inp(`(protocol libpurple)`)
	_, ok := readAccountProtocol(from)
	assertDeepEquals(t, ok, true)
}

func Test_ImportKeys_willReturnARelevantErrorForIncorrectData(t *testing.T) {
	from := bytes.NewBuffer([]byte(`(privkeys (account
(name "foo2")
(protocol libpurple-Jabberx)
(private-key (dsa
  (px #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A858#)
  ))))`))
	_, err := ImportKeys(from)
	assertDeepEquals(t, err, newOtrError("couldn't import data into private key"))
}

func Test_ImportKeys_willReturnTheParsedAccountInformation(t *testing.T) {
	from := bytes.NewBuffer([]byte(`(privkeys (account
(name "foo2")
(protocol libpurple-Jabberx)
(private-key (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A858#)
  ))))`))
	res, err := ImportKeys(from)
	assertDeepEquals(t, len(res), 1)
	assertDeepEquals(t, err, nil)
}

func Test_ImportKeysFromFile_willReturnAnErrorIfAskedToReadAFileNameThatDoesntExist(t *testing.T) {
	_, err := ImportKeysFromFile("this_file_doesnt_exist.asc")
	assertDeepEquals(t, err, &os.PathError{
		Op:   "open",
		Path: "this_file_doesnt_exist.asc",
		Err:  syscall.Errno(0x02)})
}

func Test_ImportKeysFromFile_willReturnAValidAccountReadFromAFile(t *testing.T) {
	res, err := ImportKeysFromFile("test_resources/valid_key.asc")
	assertDeepEquals(t, len(res), 1)
	assertDeepEquals(t, err, nil)
}

func Test_ImportKeysFromFile_willReturnAnErrorIfTheFileIsinvalid(t *testing.T) {
	_, err := ImportKeysFromFile("test_resources/invalid_key.asc")
	assertDeepEquals(t, err, newOtrError("couldn't import data into private key"))
}

func Test_PrivateKey_ImportWithoutError(t *testing.T) {
	const libOTRPrivateKey = `(privkeys
 (account
(name "foo@example.com")
(protocol prpl-jabber)
(private-key
 (dsa
  (p #00FC07ABCF0DC916AFF6E9AE47BEF60C7AB9B4D6B2469E436630E36F8A489BE812486A09F30B71224508654940A835301ACC525A4FF133FC152CC53DCC59D65C30A54F1993FE13FE63E5823D4C746DB21B90F9B9C00B49EC7404AB1D929BA7FBA12F2E45C6E0A651689750E8528AB8C031D3561FECEE72EBB4A090D450A9B7A857#)
  (q #00997BD266EF7B1F60A5C23F3A741F2AEFD07A2081#)
  (g #535E360E8A95EBA46A4F7DE50AD6E9B2A6DB785A66B64EB9F20338D2A3E8FB0E94725848F1AA6CC567CB83A1CC517EC806F2E92EAE71457E80B2210A189B91250779434B41FC8A8873F6DB94BEA7D177F5D59E7E114EE10A49CFD9CEF88AE43387023B672927BA74B04EB6BBB5E57597766A2F9CE3857D7ACE3E1E3BC1FC6F26#)
  (y #0AC8670AD767D7A8D9D14CC1AC6744CD7D76F993B77FFD9E39DF01E5A6536EF65E775FCEF2A983E2A19BD6415500F6979715D9FD1257E1FE2B6F5E1E74B333079E7C880D39868462A93454B41877BE62E5EF0A041C2EE9C9E76BD1E12AE25D9628DECB097025DD625EF49C3258A1A3C0FF501E3DC673B76D7BABF349009B6ECF#)
  (x #14D0345A3562C480A039E3C72764F72D79043216#)
  )
 )
 )
)`
	priv := &DSAPrivateKey{}
	ok := priv.Import([]byte(libOTRPrivateKey))
	assertEquals(t, ok, true)
}

func Test_PrivateKey_GenerateWithoutError(t *testing.T) {
	priv := &DSAPrivateKey{}
	err := priv.Generate(rand.Reader)
	assertEquals(t, err, nil)
}

func Test_PrivateKey_GenerateErrorWhenGenerateParams(t *testing.T) {
	priv := &DSAPrivateKey{}
	err := priv.Generate(fixedRand([]string{"ABCDEF"}))
	assertEquals(t, err.Error(), "unexpected EOF")
}

func Test_notHex(t *testing.T) {
	var isNotHex bool
	testString := "o0"
	isNotHex = notHex(rune(testString[0]))
	assertEquals(t, isNotHex, true)
	isNotHex = notHex(rune(testString[1]))
	assertEquals(t, isNotHex, false)
}

func Test_exportAccounts_exportsAccounts(t *testing.T) {
	priv := &DSAPrivateKey{}
	priv.Parse(serializedPrivateKey)
	acc := Account{Name: "hello", Protocol: "go-xmpp", Key: priv}
	bt := bytes.NewBuffer(make([]byte, 0, 200))
	exportAccounts([]*Account{&acc}, bt)
	assertDeepEquals(t, bt.String(),
		`(privkeys
  (account
    (name "hello")
    (protocol go-xmpp)
    (private-key
      (dsa
        (p #F24843F9447B62138AE49BF83188D1353ADA5CAC118890CFDEC01BF349D75E887B19C221665C7857CAD583AF656C67FB04A99FD8F8D69D09C9529C6C14D426F1E3924DC9243AF2970E3E4B04A23489A09E8A90E7E81EBA763AD4F0636B8A43415B6FC16A02C3624CE76272FA00783C8DB850D3A996B58136F7A0EB80AE0BC613#)
        (q #D16B2607FCBC0EDC639F763A54F34475B1CC8473#)
        (g #B15AFEF5F96EFEE41006F136C23A18849DA8133069A879D083F7C7AA362E187DAE3ED0C4F372D0D4E3AAE567008A1872A6E85D8F84E53A3FE1B352AF0B4E2F0CB033A6D34285ECD3E4A93653BDE99C3A8D840D9D35F82AC2FA8539DB6C7F7A1DAD77FEECD62803757FF1E2DE4CEC4A5A2AD643271514DDEEEF3D008F66FBF9DB#)
        (y #1F9BE7DA0E4E84774048058B53202B2704BF688A306092ED533A55E68EABA814C8D62F45AAD8FF30C3055DCA461B7DBA6B78938FC4D69780A830C6457CC107F3D275C21D00E53147C14162176C77169D3BCA586DC30F15F4B482160E276869AA336F38AF7FC3686A764AB5A02C751D921A42B8B9AE8E06918059CD73C424154#)
        (x #14D0345A3562C480A039E3C72764F72D79043216#)
      )
    )
  )
)
`)
}

func Test_ExportKeysToFile_exportsKeysToAFile(t *testing.T) {
	priv := &DSAPrivateKey{}
	priv.Parse(serializedPrivateKey)
	acc := &Account{Name: "hello", Protocol: "go-xmpp", Key: priv}

	err := ExportKeysToFile([]*Account{acc}, "test_resources/test_export_of_keys.blah")
	assertNil(t, err)

	res, err2 := ImportKeysFromFile("test_resources/test_export_of_keys.blah")

	defer os.Remove("test_resources/test_export_of_keys.blah")

	assertNil(t, err2)
	assertDeepEquals(t, res[0].Key, acc.Key)
}

func Test_ExportKeysToFile_returnsAnErrorIfSomethingGoesWrong(t *testing.T) {
	priv := &DSAPrivateKey{}
	priv.Parse(serializedPrivateKey)
	acc := &Account{Name: "hello", Protocol: "go-xmpp", Key: priv}

	err := ExportKeysToFile([]*Account{acc}, "non_existing_directory/test_export_of_keys.blah")
	assertMatches(t, err.Error(), "open non_existing_directory/test_export_of_keys.blah: (no such file or directory|The system cannot find the path specified.)")
}
