package io

import (
	"encoding/json"
	"fmt"
	"io"

	"git.sr.ht/~charles/rq/util"
	"github.com/alecthomas/chroma/v2/quick"
)

func init() {
	registerOutputHandler("ndjson", func() OutputHandler { return &NDJSONOutputHandler{} })
}

// Declare conformance with OutputHandler interface.
var _ OutputHandler = &NDJSONOutputHandler{}

// NDJSONOutputHandler handles serializing NDJSON data.
type NDJSONOutputHandler struct {
	indent      string
	style       string
	colorize    bool
	pretty      bool
	initialized bool
}

func (j *NDJSONOutputHandler) init() {
	if j.initialized {
		return
	}

	j.colorize = false
	j.pretty = false
	j.indent = "\t"
	j.style = "native"
	j.initialized = true
}

// Name implements OutputHandler.Name().
func (j *NDJSONOutputHandler) Name() string {
	return "ndjson"
}

// SetOption implements OutputHandler.SetOption().
func (j *NDJSONOutputHandler) SetOption(name string, value string) error {
	j.init()

	switch name {
	case "output.colorize":
		j.colorize = util.StringToValue(value).(bool)
	case "output.style":
		j.style = value
	}

	return nil
}

// Format implements OutputHandler.Format()
func (j *NDJSONOutputHandler) Format(writer io.Writer, data interface{}) error {
	j.init()

	dataList, ok := data.([]interface{})
	if !ok {
		return fmt.Errorf("ndjson output must be an array, not %T", data)
	}

	for i, elem := range dataList {
		encoded, err := json.Marshal(elem)
		if err != nil {
			return fmt.Errorf("failed to encode array element at index %d: %w", i, err)
		}

		if j.colorize {
			err := quick.Highlight(writer, string(encoded), "json", "terminal", j.style)
			if err != nil {
				return err
			}
		} else {
			_, err := writer.Write(encoded)
			if err != nil {
				return err
			}
		}
		_, err = writer.Write([]byte("\n"))
		if err != nil {
			return nil
		}
	}

	return nil
}
