package builtins

import (
	"encoding/json"
	"fmt"
	"math"
	"math/rand"
	"os"
	"path/filepath"
	"runtime"
	"testing"

	"git.sr.ht/~charles/rq/internal"
	"git.sr.ht/~charles/rq/util"
	"github.com/jaswdr/faker"
	"github.com/open-policy-agent/opa/v1/ast"
	"github.com/open-policy-agent/opa/v1/topdown"
	"github.com/stretchr/testify/assert"
)

func makeOperands(command []string, options map[string]interface{}) []*ast.Term {
	return []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(command)},
		&ast.Term{Value: ast.MustInterfaceToValue(options)},
	}

}

func runCommand(command []string, options map[string]interface{}) (map[string]interface{}, error) {
	var iface interface{}
	var err error

	iter := func(t *ast.Term) error {
		iface, err = ast.ValueToInterface(t.Value, &dummyResolver{})
		return nil
	}
	err2 := builtinRun(topdown.BuiltinContext{}, makeOperands(command, options), iter)
	if err != nil {
		return nil, err
	}
	if err2 != nil {
		return nil, err2
	}

	if res, ok := iface.(map[string]interface{}); ok {
		return res, nil
	}
	return nil, fmt.Errorf("%v is not an object", iface)
}

func TestRun(t *testing.T) {
	// Test that we can send in a string and read it back out via `cat`.

	cases := []struct {
		command  []string
		options  map[string]interface{}
		expected map[string]interface{}
		mustErr  bool
	}{
		{
			command: []string{"cat"},
			options: map[string]interface{}{
				"stdin": "hello, test 1!",
			},
			expected: map[string]interface{}{
				"exitcode": json.Number("0"),
				"stdout":   "hello, test 1!",
				"stderr":   "",
			},
		},
		{
			command: []string{"cat"},
			options: map[string]interface{}{
				"stdin": map[string]interface{}{"foo": "bar"},
			},
			expected: map[string]interface{}{
				"exitcode": json.Number("0"),
				"stdout":   "{\"foo\":\"bar\"}",
				"stderr":   "",
			},
		},
		{
			command: []string{"cat"},
			options: map[string]interface{}{
				"stdin":      map[string]interface{}{"foo": "bar"},
				"stdin_spec": "yaml:",
			},
			expected: map[string]interface{}{
				"exitcode": json.Number("0"),
				"stdout":   "foo: bar\n",
				"stderr":   "",
			},
		},
		{
			command: []string{"cat"},
			options: map[string]interface{}{
				"stdin": map[string]interface{}{"foo": "bar"},
				"stdin_spec": map[string]interface{}{
					"format": "yaml",
				},
			},
			expected: map[string]interface{}{
				"exitcode": json.Number("0"),
				"stdout":   "foo: bar\n",
				"stderr":   "",
			},
		},
		{
			command: []string{"cat"},
			options: map[string]interface{}{
				"stdin": map[string]interface{}{"foo": "bar"},
				"stdin_spec": map[string]interface{}{
					"format": "yaml",
				},
				"stdout_spec": "yaml:",
			},
			expected: map[string]interface{}{
				"exitcode": json.Number("0"),
				"stdout": map[string]interface{}{
					"foo": "bar",
				},
				"stderr": "",
			},
		},
		{
			command: []string{"sh", "-c", "cat 1>&2"},
			options: map[string]interface{}{
				"stdin": map[string]interface{}{"foo": "bar"},
				"stderr_spec": map[string]interface{}{
					"format": "yaml",
				},
				"stderr": "yaml:",
			},
			expected: map[string]interface{}{
				"exitcode": json.Number("0"),
				"stderr": map[string]interface{}{
					"foo": "bar",
				},
				"stdout": "",
			},
		},
		{
			// Make sure that if stdin is provided, and the
			// FilePath is provided for the stdin_spec, we throw an
			// error.
			command: []string{"echo", `{"foo": "bar"}`},
			options: map[string]interface{}{
				"stdin": "foo",
				"stdin_spec": map[string]interface{}{
					"file_path": "/nonexistant",
				},
			},
			mustErr: true,
		},
		{
			// Providing a non-empty filepath for the stdout_spec
			// should error.
			command: []string{"echo", `{"foo": "bar"}`},
			options: map[string]interface{}{
				"stdout_spec": map[string]interface{}{
					"file_path": "/nonexistant",
				},
			},
			mustErr: true,
		},
		{
			// Providing a non-empty filepath for the stderr_spec
			// should error.
			command: []string{"echo", `{"foo": "bar"}`},
			options: map[string]interface{}{
				"stderr_spec": map[string]interface{}{
					"file_path": "/nonexistant",
				},
			},
			mustErr: true,
		},
		{
			command: []string{"sh", "-c", "sleep 1"},
			options: map[string]interface{}{
				"timeout": 0.05, // 50ms
			},
			expected: map[string]interface{}{
				"exitcode": json.Number("-1"),
				"stderr":   "",
				"stdout":   "",
				"timedout": true,
			},
		},
		{
			command: []string{"sh", "-c", "true"},
			options: map[string]interface{}{
				"timeout": 0.05, // 50ms
			},
			expected: map[string]interface{}{
				"exitcode": json.Number("0"),
				"stderr":   "",
				"stdout":   "",
				"timedout": false,
			},
		},
	}

	for i, c := range cases {
		t.Logf("-------- test %d\n%+v\n", i, c)

		result, err := runCommand(c.command, c.options)
		if c.mustErr {
			assert.NotNil(t, err)
		} else {
			assert.Nil(t, err)
		}

		assert.Equal(t, c.expected, result)
	}
}

func getRqPath(t *testing.T) string {
	_, filename, _, ok := runtime.Caller(0)
	if !ok {
		t.Fatal()
	}
	// We're in ./builtins
	rqDir := filepath.Dir(filepath.Dir(filename))

	return rqDir
}

func getTestPath(t *testing.T) string {
	testPath := filepath.Join(getRqPath(t), "test")
	return testPath
}

func runTree(path string, options map[string]interface{}) (map[string]interface{}, error) {
	var iface interface{}
	var err error

	iter := func(t *ast.Term) error {
		iface, err = ast.ValueToInterface(t.Value, &dummyResolver{})
		return nil
	}

	operands := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(path)},
		&ast.Term{Value: ast.MustInterfaceToValue(options)},
	}

	// We can re-use makeOperands from the rq.run() stuff, since
	// we happen to have the same shape for our operands.
	err2 := builtinTree(topdown.BuiltinContext{}, operands, iter)
	if err != nil {
		return nil, err
	}
	if err2 != nil {
		return nil, err2
	}

	if res, ok := iface.(map[string]interface{}); ok {
		return res, nil
	}
	return nil, fmt.Errorf("%v is not an object", iface)
}

func TestTree1(t *testing.T) {
	result, err := runTree(filepath.Join(getTestPath(t), "tree1"), map[string]interface{}{})
	assert.Nil(t, err)

	testPath := getTestPath(t)

	resultJSONBytes, err := json.MarshalIndent(result, "", "\t")
	assert.Nil(t, err)
	t.Logf("output is: %s", string(resultJSONBytes))

	file2I, ok := result[filepath.Join(testPath, "tree1/dir1/file2.txt")]
	assert.True(t, ok)
	file2, ok := file2I.(map[string]interface{})
	assert.True(t, ok)

	assert.Equal(t, "2", util.ValueToString(file2["depth"]))
	assert.Equal(t, "false", util.ValueToString(file2["is_dir"]))
	assert.Equal(t, "txt", util.ValueToString(file2["ext"]))
	assert.Equal(t, "dir1/file2.txt", util.ValueToString(file2["rel"]))
	assert.Equal(t, "file2.txt", util.ValueToString(file2["base"]))
	assert.Equal(t, "0", util.ValueToString(file2["size"]))

}

func runEnv() (map[string]interface{}, error) {
	var iface interface{}
	var err error

	iter := func(t *ast.Term) error {
		iface, err = ast.ValueToInterface(t.Value, &dummyResolver{})
		return nil
	}

	operands := []*ast.Term{}

	// We can re-use makeOperands from the rq.run() stuff, since
	// we happen to have the same shape for our operands.
	err2 := builtinEnv(topdown.BuiltinContext{}, operands, iter)
	if err != nil {
		return nil, err
	}
	if err2 != nil {
		return nil, err2
	}

	if res, ok := iface.(map[string]interface{}); ok {
		return res, nil
	}
	return nil, fmt.Errorf("%v is not an object", iface)
}

func TestEnv1(t *testing.T) {
	err := os.Setenv("RQ_TEST_ENV_VAR_1", "TestEnv1")
	assert.Nil(t, err)

	result, err := runEnv()
	assert.Nil(t, err)

	iface, ok := result["RQ_TEST_ENV_VAR_1"]
	assert.True(t, ok)
	assert.Equal(t, "TestEnv1", util.ValueToString(iface))
}

func runError(message string) error {
	iter := func(t *ast.Term) error {
		return nil
	}

	return builtinError(topdown.BuiltinContext{}, []*ast.Term{&ast.Term{Value: ast.MustInterfaceToValue(message)}}, iter)
}

func TestError1(t *testing.T) {
	err := runError("this is the error message")
	assert.NotNil(t, err)
	assert.Equal(t, "this is the error message", err.Error())
}

func runDecode(input string, spec string) (interface{}, error) {
	var result interface{}
	iter := func(t *ast.Term) error {
		var err error
		result, err = ast.ValueToInterface(t.Value, &dummyResolver{})
		if err != nil {
			return err
		}
		return nil
	}

	args := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(input)},
		&ast.Term{Value: ast.MustInterfaceToValue(spec)},
	}

	err := builtinDecode(topdown.BuiltinContext{}, args, iter)
	return result, err
}

func TestDecode(t *testing.T) {
	cases := []struct {
		input     string
		spec      string
		expect    interface{}
		mustError bool
	}{
		{
			input: `{"foo": "bar"}`,
			spec:  `json:`,
			expect: map[string]interface{}{
				"foo": "bar",
			},
		},
		{
			input: "col1|col2\nval1|val2\nval3|val4",
			spec:  `csv:csv.comma=|;csv.headers=true:`,
			expect: []interface{}{
				map[string]interface{}{"col1": "val1", "col2": "val2"},
				map[string]interface{}{"col1": "val3", "col2": "val4"},
			},
		},
		{
			spec:      "json:foo=bar",
			mustError: true,
		},
	}

	for _, c := range cases {
		actual, err := runDecode(c.input, c.spec)
		if c.mustError {
			assert.NotNil(t, err)
		} else {
			assert.Nil(t, err)
		}
		assert.Equal(t, c.expect, actual)
	}
}

func runEncode(input interface{}, spec string) (string, error) {
	var result string
	iter := func(t *ast.Term) error {
		resultI, err := ast.ValueToInterface(t.Value, &dummyResolver{})
		if err != nil {
			return err
		}
		result = util.ValueToString(resultI)
		return nil
	}

	args := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(input)},
		&ast.Term{Value: ast.MustInterfaceToValue(spec)},
	}

	err := builtinEncode(topdown.BuiltinContext{}, args, iter)
	return result, err
}

func TestEncode(t *testing.T) {
	cases := []struct {
		input     interface{}
		spec      string
		expect    string
		mustError bool
	}{
		{
			input:  map[string]interface{}{"foo": "bar"},
			spec:   `json:output.colorize=false;output.pretty=false:`,
			expect: `{"foo":"bar"}`,
		},
		{
			spec:      "json:foo=bar",
			mustError: true,
		},
	}

	for _, c := range cases {
		actual, err := runEncode(c.input, c.spec)
		if c.mustError {
			assert.NotNil(t, err)
		} else {
			assert.Nil(t, err)
		}
		assert.Equal(t, c.expect, actual)
	}
}

func TestParseDate(t *testing.T) {
	cases := []struct {
		input  string
		expect json.Number
	}{
		{
			input:  "December 4th, 2022",
			expect: json.Number("1670112000000000000"),
		},
		{
			input:  "Fri Jul 03 2015 18:04:07 GMT+0100 (GMT Daylight Time)",
			expect: json.Number("1435943047000000000"),
		},
	}

	for _, c := range cases {
		var actual json.Number
		err := builtinParsedate(
			topdown.BuiltinContext{},
			[]*ast.Term{&ast.Term{Value: ast.MustInterfaceToValue(c.input)}},
			func(t *ast.Term) error {
				resultI, err := ast.ValueToInterface(t.Value, &dummyResolver{})
				if err != nil {
					return err
				}
				actual = resultI.(json.Number)
				return nil
			},
		)

		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)
	}
}

func runConvert(amount float64, from, to string) (float64, error) {
	var result float64
	iter := func(t *ast.Term) error {
		resultI, err := ast.ValueToInterface(t.Value, &dummyResolver{})
		if err != nil {
			return err
		}
		resultN := resultI.(json.Number)
		result, err = resultN.Float64()
		return err
	}

	args := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(amount)},
		&ast.Term{Value: ast.MustInterfaceToValue(from)},
		&ast.Term{Value: ast.MustInterfaceToValue(to)},
	}

	err := builtinConvert(topdown.BuiltinContext{}, args, iter)
	return result, err
}

func TestConvert(t *testing.T) {
	cases := []struct {
		from   string
		to     string
		amount float64
		expect float64
	}{
		{
			from:   "celsius",
			to:     "fahrenheit",
			amount: 25.5,
			expect: 77.9,
		},
		{
			from:   "bytes",
			to:     "kilobytes",
			amount: 1000,
			expect: 1,
		},
		{
			from:   "kilometers",
			to:     "miles",
			amount: 37.9,
			expect: 23.55,
		},
	}

	for _, c := range cases {
		actual, err := runConvert(c.amount, c.from, c.to)
		assert.Nil(t, err)
		if math.Abs(actual-c.expect) > 0.0001 { // account for FP precision
			assert.Equal(t, c.expect, actual)
		}
	}
}

func runBase(path string) (string, error) {
	var result string
	iter := func(t *ast.Term) error {
		resultI, err := ast.ValueToInterface(t.Value, &dummyResolver{})
		result = resultI.(string)
		return err
	}

	args := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(path)},
	}

	err := builtinBase(topdown.BuiltinContext{}, args, iter)
	return result, err
}

func TestBase(t *testing.T) {
	cases := []struct {
		path   string
		expect string
	}{
		{
			path:   "/foo/bar/baz",
			expect: "baz",
		},
	}

	for _, c := range cases {
		actual, err := runBase(c.path)
		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)
	}
}

func runExt(path string) (string, error) {
	var result string
	iter := func(t *ast.Term) error {
		resultI, err := ast.ValueToInterface(t.Value, &dummyResolver{})
		result = resultI.(string)
		return err
	}

	args := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(path)},
	}

	err := builtinExt(topdown.BuiltinContext{}, args, iter)
	return result, err
}

func TestExt(t *testing.T) {
	cases := []struct {
		path   string
		expect string
	}{
		{
			path:   "/foo/bar/baz.txt",
			expect: ".txt",
		},
	}

	for _, c := range cases {
		actual, err := runExt(c.path)
		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)
	}
}

func runDir(path string) (string, error) {
	var result string
	iter := func(t *ast.Term) error {
		resultI, err := ast.ValueToInterface(t.Value, &dummyResolver{})
		result = resultI.(string)
		return err
	}

	args := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(path)},
	}

	err := builtinDir(topdown.BuiltinContext{}, args, iter)
	return result, err
}

func TestDir(t *testing.T) {
	cases := []struct {
		path   string
		expect string
	}{
		{
			path:   "/foo/bar/baz.txt",
			expect: "/foo/bar",
		},
	}

	for _, c := range cases {
		actual, err := runDir(c.path)
		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)
	}
}

func runSplitPath(path string) ([]interface{}, error) {
	var result []interface{}
	iter := func(t *ast.Term) error {
		resultI, err := ast.ValueToInterface(t.Value, &dummyResolver{})
		result = resultI.([]interface{})
		return err
	}

	args := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(path)},
	}

	err := builtinSplitPath(topdown.BuiltinContext{}, args, iter)
	return result, err
}

func TestSplitPath(t *testing.T) {
	cases := []struct {
		path   string
		expect []interface{}
	}{
		{
			path:   "/foo/bar/baz.txt",
			expect: []interface{}{"foo", "bar", "baz.txt"},
		},
		{
			path:   "//foo/bar////baz.txt/",
			expect: []interface{}{"foo", "bar", "baz.txt"},
		},
	}

	for _, c := range cases {
		actual, err := runSplitPath(c.path)
		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)
	}
}

func runJoinPath(elements []string) (string, error) {
	var result string
	iter := func(t *ast.Term) error {
		resultI, err := ast.ValueToInterface(t.Value, &dummyResolver{})
		result = resultI.(string)
		return err
	}

	args := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(elements)},
	}

	err := builtinJoinPath(topdown.BuiltinContext{}, args, iter)
	return result, err
}

func TestJoinPath(t *testing.T) {
	cases := []struct {
		elements []string
		expect   string
	}{
		{
			elements: []string{"foo", "/bar/", "/baz"},
			expect:   "foo/bar/baz",
		},
	}

	for _, c := range cases {
		actual, err := runJoinPath(c.elements)
		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)
	}
}

func runTemplate(s string, data map[string]interface{}) (string, error) {
	var iface interface{}
	var result string
	var err error

	iter := func(t *ast.Term) error {
		iface, err = ast.ValueToInterface(t.Value, &dummyResolver{})
		if err != nil {
			return err
		}
		result = iface.(string)
		return nil
	}

	operands := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(s)},
		&ast.Term{Value: ast.MustInterfaceToValue(data)},
	}

	err2 := builtinTemplate(topdown.BuiltinContext{}, operands, iter)
	if err != nil {
		return "", err
	}
	if err2 != nil {
		return "", err2
	}

	return result, nil
}

func runQuote(s string) (string, error) {
	var iface interface{}
	var result string
	var err error

	iter := func(t *ast.Term) error {
		iface, err = ast.ValueToInterface(t.Value, &dummyResolver{})
		if err != nil {
			return err
		}
		result = iface.(string)
		return nil
	}

	operands := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(s)},
	}

	err2 := builtinQuote(topdown.BuiltinContext{}, operands, iter)
	if err != nil {
		return "", err
	}
	if err2 != nil {
		return "", err2
	}

	return result, nil
}

func TestQuote(t *testing.T) {
	cases := []struct {
		s      string
		expect string
	}{
		{
			s:      "hello, rq!",
			expect: `"hello, rq!"`,
		},
		{
			s:      `"hello, rq!"`,
			expect: `"\"hello, rq!\""`,
		},
	}

	for _, c := range cases {
		actual, err := runQuote(c.s)
		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)
	}
}

func runUnquote(s string) (string, error) {
	var iface interface{}
	var result string
	var err error

	iter := func(t *ast.Term) error {
		iface, err = ast.ValueToInterface(t.Value, &dummyResolver{})
		if err != nil {
			return err
		}
		result = iface.(string)
		return nil
	}

	operands := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(s)},
	}

	err2 := builtinUnquote(topdown.BuiltinContext{}, operands, iter)
	if err != nil {
		return "", err
	}
	if err2 != nil {
		return "", err2
	}

	return result, nil
}

func TestUnquote(t *testing.T) {
	cases := []struct {
		s      string
		expect string
	}{
		{
			s:      "hello, rq!",
			expect: `hello, rq!`,
		},
		{
			s:      `"hello, rq!"`,
			expect: `hello, rq!`,
		},
		{
			s:      `'hello, rq!'`,
			expect: `'hello, rq!'`,
		},
	}

	for _, c := range cases {
		actual, err := runUnquote(c.s)
		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)
	}
}

func TestTemplate(t *testing.T) {
	internal.FakeOptions.Faker = faker.NewWithSeed(rand.NewSource(42))
	util.TemplateFakeOptions.Faker = faker.NewWithSeed(rand.NewSource(42))

	cases := []struct {
		s      string
		data   map[string]interface{}
		expect string
	}{
		{
			s:      "hello {{.x}}",
			data:   map[string]interface{}{"x": "world"},
			expect: "hello world",
		},
		{
			s:      `{{lower "aaa BBB cCc"}}`,
			data:   map[string]interface{}{},
			expect: "aaa bbb ccc",
		},
		{
			s:      `{{upper "aaa BBB cCc"}}`,
			data:   map[string]interface{}{},
			expect: "AAA BBB CCC",
		},
		{
			s:      `{{replace "aaa BBB cCc" "aaa" "xxx"}}`,
			data:   map[string]interface{}{},
			expect: "xxx BBB cCc",
		},
		{
			s:      `{{reverse "abc"}}`,
			data:   map[string]interface{}{},
			expect: "cba",
		},
		{
			s:      `{{trim "abc aaa bbb aaa" " ab"}}`,
			data:   map[string]interface{}{},
			expect: "c",
		},
		{
			s:      `{{trim "  abc  " " "}}`,
			data:   map[string]interface{}{},
			expect: "abc",
		},
		{
			s:      `{{trim_left "aaabbbaaa" "a"}}`,
			data:   map[string]interface{}{},
			expect: "bbbaaa",
		},
		{
			s:      `{{trim_right "aaabbbaaa" "a"}}`,
			data:   map[string]interface{}{},
			expect: "aaabbb",
		},
		{
			s:      `{{trim_space "  abc    "}}`,
			data:   map[string]interface{}{},
			expect: "abc",
		},
		{
			s:      `{{trim_prefix "xyzabc" "xyz"}}`,
			data:   map[string]interface{}{},
			expect: "abc",
		},
		{
			s:      `{{trim_prefix "xyzabc" "yz"}}`,
			data:   map[string]interface{}{},
			expect: "xyzabc",
		},
		{
			s:      `{{trim_suffix "abcxyz" "xyz"}}`,
			data:   map[string]interface{}{},
			expect: "abc",
		},
		{
			s:      `{{trim_suffix "abcxyz" "xy"}}`,
			data:   map[string]interface{}{},
			expect: "abcxyz",
		},
		{
			s:      `{{repeat "abc" 3}}`,
			data:   map[string]interface{}{},
			expect: "abcabcabc",
		},
		{
			s:      `{{sprintf "%03d %2.2f" 7 5.678}}`,
			data:   map[string]interface{}{},
			expect: "007 5.68",
		},

		{
			s:      `{{fake "name"}}`,
			data:   map[string]interface{}{},
			expect: "Idell Denesik",
		},
		{
			s:      `{{fake "name"}}`,
			data:   map[string]interface{}{},
			expect: "Mr. Tyson Becker",
		},
		{
			s:      `{{sfake "name" "u1"}}`,
			data:   map[string]interface{}{},
			expect: "Idell Denesik",
		},
		{
			s:      `{{sfake "name" "u2"}}`,
			data:   map[string]interface{}{},
			expect: "Mr. Tyson Becker",
		},
		{
			s:      `{{sfake "name" "u1"}}`,
			data:   map[string]interface{}{},
			expect: "Idell Denesik",
		},
		{
			s:      `{{sfake "name" "u2"}}`,
			data:   map[string]interface{}{},
			expect: "Mr. Tyson Becker",
		},
	}

	// NOTE: I copied these to util/util_test.go to make the test coverage
	// report more accurate. It's good to exercise the builtin itself
	// as well, but any new cases should just be added to the one in util.

	for _, c := range cases {
		t.Logf("s %s expect %s", c.s, c.expect)
		actual, err := runTemplate(c.s, c.data)
		assert.Nil(t, err)
		if err != nil {
			t.Logf("err %s", err.Error())
		}
		assert.Equal(t, c.expect, actual)
	}
}

func runSlug(s string) (string, error) {
	var iface interface{}
	var result string
	var err error

	iter := func(t *ast.Term) error {
		iface, err = ast.ValueToInterface(t.Value, &dummyResolver{})
		if err != nil {
			return err
		}
		result = iface.(string)
		return nil
	}

	operands := []*ast.Term{
		&ast.Term{Value: ast.MustInterfaceToValue(s)},
	}

	err2 := builtinSlug(topdown.BuiltinContext{}, operands, iter)
	if err != nil {
		return "", err
	}
	if err2 != nil {
		return "", err2
	}

	return result, nil
}

func TestSlug(t *testing.T) {
	cases := []struct {
		s      string
		expect string
	}{
		{
			s:      "hello",
			expect: `hello`,
		},
		{
			s:      "Hello, World!",
			expect: `hello-world`,
		},
		{
			s:      "北京kožušček,abc",
			expect: "bei-jing-kozuscek-abc",
		},
		{
			s:      "東亜重工",
			expect: "dong-ya-zhong-gong",
		},
	}

	for _, c := range cases {
		actual, err := runSlug(c.s)
		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)
	}
}
