/* sor.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "primitives/sor.h"
#include "csgtree.h"
#include "utils.h"
#include "giramcursor.h"

#include "pixmaps/sor.xpm"

//#include "giramwidgets.h"

#include "tool_sor.h"

#include "giramintl.h"

static GtkWidget *SorDrawingArea;
static GtkWidget *SorLabelCoordX;
static GtkWidget *SorLabelCoordY;
static GtkWidget *SorOffsetXSpinButton;
static GtkWidget *SorOffsetYSpinButton;
static GtkWidget *SorSizeXSpinButton;
static GtkWidget *SorSizeYSpinButton;
static int SorSnapToGrid;
static int SorDrawGrid;
static int SorNbPoint;
static Vector *SorAllPoints = NULL;
static int SorSpline;
static double SorZoom;
static double SorXoffset, SorYoffset;
#define SOR_ADD_POINT    1
#define SOR_DELETE_POINT 2
#define SOR_MOVE_POINT   3
static int SorModeTool;
static int SorPointIndex;
#define SOR_NONE   1
#define SOR_MOVING 2
static int SOR_STATE = SOR_NONE;

/*****************************************************************************
*  SorSetModeAdd
******************************************************************************/
static void SorSetModeAdd(void)
{
  SorModeTool = SOR_ADD_POINT;
}

/*****************************************************************************
*  SorSetModeDelete
******************************************************************************/
static void SorSetModeDelete(void)
{
  SorModeTool = SOR_DELETE_POINT;
}

/*****************************************************************************
*  SorSetModeMove
******************************************************************************/
static void SorSetModeMove(void)
{
  SorModeTool = SOR_MOVE_POINT;
}

/*****************************************************************************
*  SorSetStraightLine
******************************************************************************/
static void SorSetStraightLine(void)
{
  SorSpline = FALSE;
  gtk_widget_queue_draw(SorDrawingArea);
}

/*****************************************************************************
*  SorSetSpline
******************************************************************************/
static void SorSetSpline(void)
{
  SorSpline = TRUE;
  gtk_widget_queue_draw(SorDrawingArea);
}

/*****************************************************************************
*  SorToggleDrawGrid
******************************************************************************/
static void SorToggleDrawGrid(GtkWidget *widget)
{
  SorDrawGrid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
  gtk_widget_queue_draw(SorDrawingArea);
}

/*****************************************************************************
*  SorToggleSnapToGrid
******************************************************************************/
static void SorToggleSnapToGrid(GtkWidget *widget)
{
  SorSnapToGrid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
}

/*****************************************************************************
*  SorDrawSnapGrid
******************************************************************************/
static void SorDrawSnapGrid(GtkWidget *Area)
{
  GdkColormap *colormap;
  GdkColor     ForeColor;
  GdkGC       *gc;
  double       x, y;
  double       StartX, StartY;
  double       EndX, EndY;
  double       Xinc, Yinc;

  colormap = gdk_drawable_get_colormap(Area->window);
  ForeColor.red = 32767;
  ForeColor.green = 32767;
  ForeColor.blue = 32767;
  gdk_colormap_alloc_color(colormap, &ForeColor, FALSE, TRUE);
  gc = gdk_gc_new(Area->window);
  gdk_gc_set_foreground(gc, &ForeColor);

  Xinc = gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorSizeXSpinButton));
  Yinc = gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorSizeYSpinButton));
  StartX = SorXoffset-Area->allocation.width/2.0/SorZoom;
  StartY = SorYoffset-Area->allocation.height/2.0/SorZoom;

  StartX = Xinc*((int)(StartX/Xinc))+gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorOffsetXSpinButton));
  StartY = Yinc*((int)(StartY/Yinc))+gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorOffsetYSpinButton));

  EndX = Area->allocation.width/2.0/SorZoom+SorXoffset;
  EndY = SorYoffset+Area->allocation.height/2.0/SorZoom;
  for (x=StartX; x<EndX ; x+=Xinc)
  {
    for (y=StartY ; y<EndY ; y+=Yinc)
    {
      gdk_draw_point(Area->window, gc,
                     Area->allocation.width/2+(x-SorXoffset)*SorZoom,
                     Area->allocation.height/2-(y-SorYoffset)*SorZoom);
    }
  }
  g_object_unref(gc);
}

/*****************************************************************************
*  SorDrawSpline
******************************************************************************/
static void SorDrawSpline(GtkWidget *Area, GdkGC *gc,
                   double x0, double y0, double x1, double y1,
                   double x00, double y00, double x11, double y11)
{
  TransformStruct Mat;
  Vector          ABCD, Ord;
  int             i, From, To;
  double          RealY, RealX;
  double          x;

  Mat.Direct[0][0] = x0*x0*x0;
  Mat.Direct[0][1] = x1*x1*x1;
  Mat.Direct[0][2] = 3.0*x0*x0;
  Mat.Direct[0][3] = 3.0*x1*x1;
  Mat.Direct[1][0] = x0*x0;
  Mat.Direct[1][1] = x1*x1;
  Mat.Direct[1][2] = 2*x0;
  Mat.Direct[1][3] = 2*x1;
  Mat.Direct[2][0] = x0;
  Mat.Direct[2][1] = x1;
  Mat.Direct[2][2] = 1.0;
  Mat.Direct[2][3] = 1.0;
  Mat.Direct[3][0] = 1.0;
  Mat.Direct[3][1] = 1.0;
  Mat.Direct[3][2] = 0.0;
  Mat.Direct[3][3] = 0.0;
  MInverse(Mat.Inverse, Mat.Direct);
  Ord[0] = y0*y0;
  Ord[1] = y1*y1;
  Ord[2] = 2*y0*(y1-y00)/(x1-x00);
  Ord[3] = 2*y1*(y11-y0)/(x11-x0);

  for (i=0 ; i<4 ; i++)
  {
    ABCD[i] = Ord[0] * Mat.Inverse[0][i] +
              Ord[1] * Mat.Inverse[1][i] +
              Ord[2] * Mat.Inverse[2][i] +
              Ord[3] * Mat.Inverse[3][i];
  }
  From = (int)(Area->allocation.height/2-(x1-SorXoffset)*SorZoom);
  To = (int)(Area->allocation.height/2-(x0-SorXoffset)*SorZoom);
  for (i=From ; i<To ; i++)
  {
    RealY = SorYoffset-(i-Area->allocation.height/2)/SorZoom;
    RealX = sqrt(((ABCD[0]*RealY+ABCD[1])*RealY+ABCD[2])*RealY+ABCD[3]);
    x = Area->allocation.width/2+(RealX-SorXoffset)*SorZoom;
    gdk_draw_point(Area->window, gc, x,i);
  }
}

/*****************************************************************************
*  ExposeNewSorDrawingArea
******************************************************************************/
static gboolean ExposeNewSorDrawingArea(GtkWidget *DrawingArea, GdkEventExpose *ev)
{
  GdkColormap *colormap;
  GdkColor     BackColor;
  GdkGC       *gc;
  GdkGC       *DashedGC;
  GdkColor     ForeColor;
  int          i;

  colormap = gdk_drawable_get_colormap(DrawingArea->window);
  BackColor.red = 65535;
  BackColor.green = 65535;
  BackColor.blue = 65535;
  gdk_colormap_alloc_color(colormap, &BackColor, FALSE, TRUE);
  gdk_window_set_background(DrawingArea->window, &BackColor);
  gdk_window_clear(DrawingArea->window);

  if (SorDrawGrid)
  {
    SorDrawSnapGrid(DrawingArea);
  }

  DashedGC = gdk_gc_new(DrawingArea->window);
  gdk_gc_set_line_attributes(DashedGC, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

  gc = gdk_gc_new(DrawingArea->window);
  ForeColor.red = 0;
  ForeColor.green = 0;
  ForeColor.blue = 0;
  gdk_colormap_alloc_color(colormap, &ForeColor, FALSE, TRUE);
  gdk_gc_set_foreground(gc, &ForeColor);


  /* First, we draw the axes */
  gdk_draw_line(DrawingArea->window, gc,
                DrawingArea->allocation.width/2-SorXoffset*SorZoom,
                0,
                DrawingArea->allocation.width/2-SorXoffset*SorZoom,
                DrawingArea->allocation.height-1);
  gdk_draw_line(DrawingArea->window, gc,
                0,
                DrawingArea->allocation.height/2+SorYoffset*SorZoom,
                DrawingArea->allocation.width-1,
                DrawingArea->allocation.height/2+SorYoffset*SorZoom);
  /* Then the lines between the dot */
  gdk_draw_line(DrawingArea->window, DashedGC,
                DrawingArea->allocation.width/2+(SorAllPoints[0][0]-SorXoffset)*SorZoom,
                DrawingArea->allocation.height/2-(SorAllPoints[0][1]-SorYoffset)*SorZoom,
                DrawingArea->allocation.width/2+(SorAllPoints[1][0]-SorXoffset)*SorZoom,
                DrawingArea->allocation.height/2-(SorAllPoints[1][1]-SorYoffset)*SorZoom);
  if (SorSpline)
  {
    for (i=1 ; i<SorNbPoint-2 ; i++)
    {
      SorDrawSpline(DrawingArea, gc,
                    SorAllPoints[i][1], SorAllPoints[i][0],
                    SorAllPoints[i+1][1], SorAllPoints[i+1][0],
                    SorAllPoints[i-1][1], SorAllPoints[i-1][0],
                    SorAllPoints[i+2][1], SorAllPoints[i+1][0]);
    }
  } else
  {
    for (i=1 ; i<SorNbPoint-2 ; i++)
    {
      gdk_draw_line(DrawingArea->window, gc,
                    DrawingArea->allocation.width/2+(SorAllPoints[i][0]-SorXoffset)*SorZoom,
                    DrawingArea->allocation.height/2-(SorAllPoints[i][1]-SorYoffset)*SorZoom,
                    DrawingArea->allocation.width/2+(SorAllPoints[i+1][0]-SorXoffset)*SorZoom,
                    DrawingArea->allocation.height/2-(SorAllPoints[i+1][1]-SorYoffset)*SorZoom);
    }
  }
  gdk_draw_line(DrawingArea->window, DashedGC,
                DrawingArea->allocation.width/2+(SorAllPoints[SorNbPoint-2][0]-SorXoffset)*SorZoom,
                DrawingArea->allocation.height/2-(SorAllPoints[SorNbPoint-2][1]-SorYoffset)*SorZoom,
                DrawingArea->allocation.width/2+(SorAllPoints[SorNbPoint-1][0]-SorXoffset)*SorZoom,
                DrawingArea->allocation.height/2-(SorAllPoints[SorNbPoint-1][1]-SorYoffset)*SorZoom);
  /* Draw the plots */
  gdk_draw_rectangle(DrawingArea->window, gc, FALSE,
                     DrawingArea->allocation.width/2+
                              (SorAllPoints[0][0]-SorXoffset)*SorZoom-5,
                     DrawingArea->allocation.height/2-
                              (SorAllPoints[0][1]-SorYoffset)*SorZoom-5,
                     11, 11);
  for (i=1 ; i<SorNbPoint-1 ; i++)
  {
    gdk_draw_rectangle(DrawingArea->window, gc, TRUE,
                       DrawingArea->allocation.width/2+
                                (SorAllPoints[i][0]-SorXoffset)*SorZoom-5,
                       DrawingArea->allocation.height/2-
                                (SorAllPoints[i][1]-SorYoffset)*SorZoom-5,
                       11, 11);
  }
  gdk_draw_rectangle(DrawingArea->window, gc, FALSE,
                     DrawingArea->allocation.width/2+
                              (SorAllPoints[i][0]-SorXoffset)*SorZoom-5,
                     DrawingArea->allocation.height/2-
                              (SorAllPoints[i][1]-SorYoffset)*SorZoom-5,
                     11, 11);
  return FALSE;
}

/*****************************************************************************
*  SorUpdateCursorPosition
******************************************************************************/
static void SorUpdateCursorPosition(double x, double y)
{
  double  RealX, RealY;
  gchar  *buf;

  RealX = (x-SorDrawingArea->allocation.width/2)/SorZoom+SorXoffset;
  RealY = SorYoffset-(y-SorDrawingArea->allocation.height/2)/SorZoom;
  buf = g_strdup_printf("x: %g", RealX);
  gtk_label_set_text(GTK_LABEL(SorLabelCoordX), buf);
  g_free(buf);
  buf = g_strdup_printf("y: %g", RealY);
  gtk_label_set_text(GTK_LABEL(SorLabelCoordY), buf);
  g_free(buf);
}

/*****************************************************************************
*  SorAddPoint
******************************************************************************/
static void SorAddPoint(GtkWidget *Area, GdkEventButton *bevent)
{
  double RealX, RealY;
  int i  ;

  RealX = (bevent->x-Area->allocation.width/2)/SorZoom+SorXoffset;
  RealY = SorYoffset-(bevent->y-Area->allocation.height/2)/SorZoom;

  if (SorSnapToGrid)
  {
    double Xinc, Yinc;
    Xinc = gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorSizeXSpinButton));
    Yinc = gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorSizeYSpinButton));
    RealX = Xinc*((int)(RealX/Xinc))+gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorOffsetXSpinButton));
    RealY = Yinc*((int)(RealY/Yinc))+gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorOffsetYSpinButton));
  }
  SorAllPoints = g_realloc(SorAllPoints, (SorNbPoint+1)*sizeof(Vector));
  for (i=SorNbPoint ; (i>0) && (SorAllPoints[i-1][1]>RealY) ; i--)
  {
    SorAllPoints[i][0] = SorAllPoints[i-1][0];
    SorAllPoints[i][1] = SorAllPoints[i-1][1];
  }
  if (RealX <= 0.0) RealX = 0.0001; /* for sanity */
  SorAllPoints[i][0] = RealX;
  SorAllPoints[i][1] = RealY;
  SorNbPoint++;
  gtk_widget_queue_draw(Area);
}

/*****************************************************************************
*  SorBeginMovePoint
******************************************************************************/
static void SorBeginMovePoint(GtkWidget *Area, GdkEventButton *bevent)
{
  gint    i;
  gdouble XD, YD;
  /* Search for the index of the point if it exists */
  SorPointIndex = -1;
  for (i=0 ; i<SorNbPoint ; i++)
  {
    XD = Area->allocation.width/2+(SorAllPoints[i][0]-SorXoffset)*SorZoom;
    YD = Area->allocation.height/2-(SorAllPoints[i][1]-SorYoffset)*SorZoom;
    if (fabs(XD-bevent->x)<5.0 && fabs(YD-bevent->y)<5.0)
      SorPointIndex = i;
  }
  if (SorPointIndex != -1)
  {
    SOR_STATE = SOR_MOVING;
    gdk_pointer_grab(Area->window, FALSE,
                     GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                     NULL, NULL, bevent->time);
  }
}

/*****************************************************************************
*  SorMoveMovePoint
******************************************************************************/
static void SorMoveMovePoint(GtkWidget *Area, GdkEventMotion *mevent)
{
  double RealX, RealY;

  RealX = (mevent->x-Area->allocation.width/2)/SorZoom+SorXoffset;
  RealY = SorYoffset-(mevent->y-Area->allocation.height/2)/SorZoom;
  if (SorPointIndex == 0)
  {
    RealY = MIN(RealY, SorAllPoints[1][1]-1/SorZoom);
  } else if (SorPointIndex == SorNbPoint-1)
  {
    RealY = MAX(RealY, SorAllPoints[SorNbPoint-2][1]+1/SorZoom);
  } else
  {
    RealY = MIN(RealY, SorAllPoints[SorPointIndex+1][1]-1/SorZoom);
    RealY = MAX(RealY, SorAllPoints[SorPointIndex-1][1]+1/SorZoom);
  }
  if (SorSnapToGrid)
  {
    double Xinc, Yinc;
    Xinc = gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorSizeXSpinButton));
    Yinc = gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorSizeYSpinButton));
    RealX = Xinc*((int)(RealX/Xinc))+gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorOffsetXSpinButton));
    RealY = Yinc*((int)(RealY/Yinc))+gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorOffsetYSpinButton));
  }
  SorAllPoints[SorPointIndex][0] = RealX;
  SorAllPoints[SorPointIndex][1] = RealY;
  gtk_widget_queue_draw(Area);
}

/*****************************************************************************
*  SorEndMovePoint
******************************************************************************/
static void SorEndMovePoint(GtkWidget *Area, GdkEventButton *bevent)
{
  double RealX, RealY;

  RealX = (bevent->x-Area->allocation.width/2)/SorZoom+SorXoffset;
  RealY = SorYoffset-(bevent->y-Area->allocation.height/2)/SorZoom;
  if (SorPointIndex == 0)
  {
    RealY = MIN(RealY, SorAllPoints[1][1]-1/SorZoom);
  } else if (SorPointIndex == SorNbPoint-1)
  {
    RealY = MAX(RealY, SorAllPoints[SorNbPoint-2][1]+1/SorZoom);
  } else
  {
    RealY = MIN(RealY, SorAllPoints[SorPointIndex+1][1]-1/SorZoom);
    RealY = MAX(RealY, SorAllPoints[SorPointIndex-1][1]+1/SorZoom);
  }
  if (SorSnapToGrid)
  {
    double Xinc, Yinc;
    Xinc = gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorSizeXSpinButton));
    Yinc = gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorSizeYSpinButton));
    RealX = Xinc*((int)(RealX/Xinc))+gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorOffsetXSpinButton));
    RealY = Yinc*((int)(RealY/Yinc))+gtk_spin_button_get_value(GTK_SPIN_BUTTON(SorOffsetYSpinButton));
  }
  if (RealX <= 0.0) RealX = 0.0001; /* for sanity */
  SorAllPoints[SorPointIndex][0] = RealX;
  SorAllPoints[SorPointIndex][1] = RealY;
  SOR_STATE = SOR_NONE;
  gdk_pointer_ungrab(bevent->time);

  gtk_widget_queue_draw(Area);
}

/*****************************************************************************
*  SorDeletePoint
******************************************************************************/
static void SorDeletePoint(GtkWidget *Area, GdkEventButton *bevent)
{
  int    index, i;
  double XD, YD;
  /* Search for the index of the point if it exists */
  if (SorNbPoint < 5)
    return;
  index = -1;
  for (i=0 ; i<SorNbPoint ; i++)
  {
    XD = Area->allocation.width/2+(SorAllPoints[i][0]-SorXoffset)*SorZoom;
    YD = Area->allocation.height/2-(SorAllPoints[i][1]-SorYoffset)*SorZoom;
    if (fabs(XD-bevent->x)<5.0 && fabs(YD-bevent->y)<5.0)
      index = i;
  }
  if (index != -1)
  {
    for (i=index ; i<SorNbPoint-1 ; i++)
    {
      SorAllPoints[i][0] = SorAllPoints[i+1][0];
      SorAllPoints[i][1] = SorAllPoints[i+1][1];
    }
    SorNbPoint--;
    SorAllPoints = g_realloc(SorAllPoints, SorNbPoint*sizeof(Vector));
  }
  gtk_widget_queue_draw(Area);
}

/*****************************************************************************
*  SorDrawingAreaEventDispatcher
******************************************************************************/
static gboolean SorDrawingAreaEventDispatcher(GtkWidget *DrawingArea, GdkEvent *event)
{
  GdkEventButton *bevent = (GdkEventButton *) event;
  GdkEventMotion *mevent = (GdkEventMotion *) event;
  switch (event->type)
  {
    case GDK_BUTTON_PRESS:
      if (bevent->button == 2) /* Middle Button ==> Begin a pan */
      {
        /* FIXME */
      }
      else if (bevent->button == 1)
      { /* Left Button */
        switch (SorModeTool)
        {
          case SOR_ADD_POINT:
            SorAddPoint(DrawingArea, bevent);
            break;
          case SOR_DELETE_POINT:
            SorDeletePoint(DrawingArea, bevent);
            break;
          case SOR_MOVE_POINT:
            SorBeginMovePoint(DrawingArea, bevent);
            break;
        }
      }
      return TRUE;
    case GDK_MOTION_NOTIFY:
      SorUpdateCursorPosition(mevent->x, mevent->y);
      if ((SorModeTool == SOR_MOVE_POINT) && (SOR_STATE == SOR_MOVING))
      {
        SorMoveMovePoint(DrawingArea, mevent);
      }
      return TRUE;
    case GDK_BUTTON_RELEASE:
      if ((SorModeTool == SOR_MOVE_POINT) && (SOR_STATE == SOR_MOVING))
      {
        SorEndMovePoint(DrawingArea, bevent);
      }
      return TRUE;
    default:
      return FALSE;
  }
  return FALSE;
}

/*****************************************************************************
*  NewSorOK
******************************************************************************/
static void NewSorOK(GtkWidget *dialog)
{
  ViewStruct   *view_data;
  GSList       *tmp_list;
  ViewStruct   *TmpView;
  ObjectStruct *sor;

  view_data = get_current_view_data();

  sor = giram_sor_new(SorNbPoint, SorAllPoints);
  sor->name = create_uniq_object_name(view_data->frame, _("SoR"));
  view_data->frame->all_objects = g_slist_append(view_data->frame->all_objects,
                                                 sor);
  sor->frame = view_data->frame;
  giram_object_build_triangle_mesh(sor);
  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_create_tree_model(view_data->frame);
  gtk_widget_destroy(dialog);
}

/*****************************************************************************
*  tool_sor_dialog_response
******************************************************************************/
static void tool_sor_dialog_response(GtkWidget *dialog, gint response)
{
  if (response == GTK_RESPONSE_ACCEPT)
    NewSorOK(dialog);
  else
    gtk_widget_destroy(dialog);
}

/*****************************************************************************
*  tool_sor_button_press
******************************************************************************/
static void tool_sor_button_press(GtkWidget *DrawingArea, GdkEventButton *bevent)
{
  GtkWidget     *dialog, *Button, *Table, *Frame, *VBox;
  GSList        *RadioGroup;
  GtkWidget     *Alignment, *Pixmap, *HBox, *RenderVBox;
  GtkWidget     *GridVBox, *Label, *CheckButton;
  GtkAdjustment *adj;

  dialog = gtk_dialog_new_with_buttons(_("New Sor"), NULL, 0,
                                       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                                       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
                                       NULL);
  g_signal_connect(G_OBJECT(dialog), "response",
                   G_CALLBACK(tool_sor_dialog_response), NULL);

  /* The main table */
  Table = gtk_table_new(2, 2, FALSE);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), Table);
  /* The Drawing Area */
  SorDrawingArea = gtk_drawing_area_new();
  gtk_widget_set_size_request(SorDrawingArea, 200, 300);
  gtk_table_attach(GTK_TABLE(Table), SorDrawingArea, 0,1, 0,1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK, 5, 5);
  g_signal_connect(G_OBJECT(SorDrawingArea), "event",
                   G_CALLBACK(SorDrawingAreaEventDispatcher), NULL);

  g_signal_connect(G_OBJECT(SorDrawingArea), "expose_event",
                   G_CALLBACK(ExposeNewSorDrawingArea), NULL);
  gtk_widget_set_events(SorDrawingArea, GDK_EXPOSURE_MASK
                                      | GDK_BUTTON_PRESS_MASK
                                      | GDK_BUTTON_RELEASE_MASK
                                      | GDK_POINTER_MOTION_MASK
                                      | GDK_KEY_PRESS_MASK);
  /* The Frame */
  Frame = gtk_frame_new("SOR parameters");
  gtk_table_attach(GTK_TABLE(Table), Frame, 1,2, 0,1,
                   GTK_FILL,
                   GTK_FILL | GTK_EXPAND, 5, 5);
  /* The Vertical box into the big Frame */
  VBox = gtk_vbox_new(FALSE, 10);
  gtk_container_add(GTK_CONTAINER(Frame), VBox);
  /* The table for the mode Button */
  Table = gtk_table_new(1,3, TRUE);
  gtk_box_pack_start(GTK_BOX(VBox), Table, FALSE, TRUE, 10);
  RadioGroup = NULL;
  SorModeTool = SOR_ADD_POINT;
    /* The "Add point" button */
    Button = gtk_radio_button_new(RadioGroup);
    gtk_container_set_border_width(GTK_CONTAINER(Button), 0);
    RadioGroup = gtk_radio_button_get_group(GTK_RADIO_BUTTON(Button));
    gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(Button), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(Button), TRUE);
    g_signal_connect(G_OBJECT(Button), "toggled",
                     G_CALLBACK(SorSetModeAdd), NULL);
    gtk_table_attach(GTK_TABLE(Table), Button, 0,1, 0,1,
                     GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                     GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
    Alignment = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
    gtk_container_set_border_width(GTK_CONTAINER(Alignment), 0);
    gtk_container_add(GTK_CONTAINER(Button), Alignment);
    Pixmap = gtk_label_new("Add Point");
    gtk_container_add(GTK_CONTAINER(Alignment), Pixmap);
    /* The "Delete point" button */
    Button = gtk_radio_button_new(RadioGroup);
    gtk_container_set_border_width(GTK_CONTAINER(Button), 0);
    RadioGroup = gtk_radio_button_get_group(GTK_RADIO_BUTTON(Button));
    gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(Button), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(Button), FALSE);
    g_signal_connect(G_OBJECT(Button), "toggled",
                     G_CALLBACK(SorSetModeDelete), NULL);
    gtk_table_attach(GTK_TABLE(Table), Button, 1,2, 0,1,
                     GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                     GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
    Alignment = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
    gtk_container_set_border_width(GTK_CONTAINER(Alignment), 0);
    gtk_container_add(GTK_CONTAINER(Button), Alignment);
    Pixmap = gtk_label_new("Delete Point");
    gtk_container_add(GTK_CONTAINER(Alignment), Pixmap);
    /* The "Move point" button */
    Button = gtk_radio_button_new(RadioGroup);
    gtk_container_set_border_width(GTK_CONTAINER(Button), 0);
    RadioGroup = gtk_radio_button_get_group(GTK_RADIO_BUTTON(Button));
    gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(Button), FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(Button), FALSE);
    g_signal_connect(G_OBJECT(Button), "toggled",
                     G_CALLBACK(SorSetModeMove), NULL);
    gtk_table_attach(GTK_TABLE(Table), Button, 2,3, 0,1,
                     GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                     GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
    Alignment = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
    gtk_container_set_border_width(GTK_CONTAINER(Alignment), 0);
    gtk_container_add(GTK_CONTAINER(Button), Alignment);
    Pixmap = gtk_label_new("Move Point");
    gtk_container_add(GTK_CONTAINER(Alignment), Pixmap);
  /* The Horizontal Box that contain the two Labels for the pointer coords */
  HBox = gtk_hbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(VBox), HBox, TRUE, TRUE, 0);
    /* The Label that shows the X coordinate of the pointer in the Area */
    SorLabelCoordX = gtk_label_new("x:     ");
    gtk_box_pack_start(GTK_BOX(HBox), SorLabelCoordX, TRUE, TRUE, 0);
    /* The Label that shows the Y coordinate of the pointer in the Area */
    SorLabelCoordY = gtk_label_new("y:     ");
    gtk_box_pack_start(GTK_BOX(HBox), SorLabelCoordY, TRUE, TRUE, 0);
  /* The Frame for the Rendering */
  Frame = gtk_frame_new("Rendering Options");
  gtk_box_pack_start(GTK_BOX(VBox), Frame, FALSE, TRUE, 0);
  RenderVBox = gtk_vbox_new(TRUE, 0);
  gtk_container_add(GTK_CONTAINER(Frame), RenderVBox);
  RadioGroup = NULL;
    /* The Straight line rendering radio button */
    Button = gtk_radio_button_new_with_label(RadioGroup, "Straight Line");
    RadioGroup = gtk_radio_button_get_group(GTK_RADIO_BUTTON(Button));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(Button), TRUE);
    gtk_box_pack_start(GTK_BOX(RenderVBox), Button, TRUE, TRUE, 0);
    g_signal_connect(G_OBJECT(Button), "toggled",
                     G_CALLBACK(SorSetStraightLine), NULL);
    /* The Spline rendering radio button */
    Button = gtk_radio_button_new_with_label(RadioGroup, "Spline");
    RadioGroup = gtk_radio_button_get_group(GTK_RADIO_BUTTON(Button));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(Button), FALSE);
    gtk_box_pack_start(GTK_BOX(RenderVBox), Button, TRUE, TRUE, 0);
    g_signal_connect(G_OBJECT(Button), "toggled",
                     G_CALLBACK(SorSetSpline), NULL);
    SorSpline = FALSE;
  /* The Frame for the Grid Parameter */
  Frame = gtk_frame_new("Grid Parameter");
  gtk_box_pack_start(GTK_BOX(VBox), Frame, TRUE, TRUE, 0);
  GridVBox = gtk_vbox_new(TRUE, 0);
  gtk_container_add(GTK_CONTAINER(Frame), GridVBox);
    /* The Table for the offset and size parameters */
    Table = gtk_table_new(2, 4, FALSE);
    gtk_box_pack_start(GTK_BOX(GridVBox), Table, TRUE, TRUE, 0);
      /* The "offset x" label */
      Label = gtk_label_new("offset x");
      gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 0,1,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
      /* The "offset x" spin */
      adj = (GtkAdjustment *) gtk_adjustment_new(0.0, 0.00, 5.00, 0.05, 0.25, 0.0);
      SorOffsetXSpinButton = gtk_spin_button_new(adj, 0.2, 2);
      gtk_table_attach(GTK_TABLE(Table), SorOffsetXSpinButton, 1,2, 0,1,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
      /* The "offset y" label */
      Label = gtk_label_new("offset y");
      gtk_table_attach(GTK_TABLE(Table), Label, 2,3, 0,1,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
      /* The "offset y" spin */
      adj = (GtkAdjustment *) gtk_adjustment_new(0.00, 0.00, 5.00, 0.05, 0.25, 0.0);
      SorOffsetYSpinButton = gtk_spin_button_new(adj, 0.2, 2);
      gtk_table_attach(GTK_TABLE(Table), SorOffsetYSpinButton, 3,4, 0,1,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
      /* The "size x" label */
      Label = gtk_label_new("size x");
      gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 1,2,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
      /* The "size x" spin */
      adj = (GtkAdjustment *) gtk_adjustment_new(0.25, 0.05, 5.00, 0.05, 0.25, 0.0);
      SorSizeXSpinButton = gtk_spin_button_new(adj, 0.2, 2);
      gtk_table_attach(GTK_TABLE(Table), SorSizeXSpinButton, 1,2, 1,2,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
      /* The "size y" label */
      Label = gtk_label_new("size y");
      gtk_table_attach(GTK_TABLE(Table), Label, 2,3, 1,2,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
      /* The "size y" spin */
      adj = (GtkAdjustment *) gtk_adjustment_new(0.25, 0.05, 5.00, 0.05, 0.25, 0.0);
      SorSizeYSpinButton = gtk_spin_button_new(adj, 0.2, 2);
      gtk_table_attach(GTK_TABLE(Table), SorSizeYSpinButton, 3,4, 1,2,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                       GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
    /* The check button for the "show grid" option */
    CheckButton = gtk_check_button_new_with_label("Show grid");
    SorDrawGrid = 0;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CheckButton), FALSE);
    g_signal_connect(G_OBJECT(CheckButton), "toggled",
                     G_CALLBACK(SorToggleDrawGrid), NULL);
    gtk_box_pack_start_defaults(GTK_BOX(GridVBox), CheckButton);
    /* The check button for the "snap to grid" option */
    CheckButton = gtk_check_button_new_with_label("Snap to grid");
    SorSnapToGrid = 0;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CheckButton), FALSE);
    g_signal_connect(G_OBJECT(CheckButton), "toggled",
                     G_CALLBACK(SorToggleSnapToGrid), NULL);
    gtk_box_pack_start_defaults(GTK_BOX(GridVBox), CheckButton);

  gtk_widget_show_all(dialog);
  SorXoffset = SorYoffset = 0.0;
  SorZoom = 50.0;

  SorNbPoint = 4;
  if (SorAllPoints)
    g_free(SorAllPoints);
  SorAllPoints = g_new(Vector, 4);
  SorAllPoints[0][0] = 1.0;
  SorAllPoints[0][1] = -1.5;
  SorAllPoints[1][0] = 1.0;
  SorAllPoints[1][1] = -0.5;
  SorAllPoints[2][0] = 1.0;
  SorAllPoints[2][1] = 0.5;
  SorAllPoints[3][0] = 1.0;
  SorAllPoints[3][1] = 1.5;
}

/****************************************************************************
*  tool_sor_cursor_update
*****************************************************************************/
static void tool_sor_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;

  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_sor_register
*****************************************************************************/
GiramTool *giram_tool_sor_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("New Sor");
  tool->Icon           = sor_icon;
  tool->Path           = "<ToolBar/Lathes>";
  tool->ID             = MT_NEW_SOR;
  tool->OptionsFunc    = NULL;
  tool->button_press   = tool_sor_button_press;
  tool->motion         = NULL;
  tool->button_release = NULL;
  tool->cursor_update  = tool_sor_cursor_update;

  return tool;
}

