/* tool_scale.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <math.h>
#include "giram.h"
#include "object.h"
#include "trimesh.h"
#include "view.h"
#include "snap.h"
#include "utils.h"
#include "csgtree.h"
#include "giramobjecteditor.h"
#include "giramcursor.h"
#include "pixmaps/scale.xpm"
#include "widgets/giramviewshell.h"
#include "tool_scale.h"

#include "giramintl.h"

TriangleListStruct *AllSelectionTri, *InitialSelectionTri;
double Xorg, Yorg, OrgNorme;
guint id;

static GdkPixmap *backpixmap = NULL;

#define NONE   1
#define MOVING 2
int SCALE_STATE = NONE;

gboolean EvenlyScaleFlag;

/*****************************************************************************
*  ToggleEvenlyScale
******************************************************************************/
static void ToggleEvenlyScale(void)
{
  EvenlyScaleFlag = !EvenlyScaleFlag;
}

/*****************************************************************************
*  BuildScaleOptions
******************************************************************************/
static GtkWidget *BuildScaleOptions(GtkWidget *VBox)
{
  GtkWidget *vbox;
  GtkWidget *CheckEvenly;
  GtkWidget *label;

  /*  the main vbox  */
  vbox = gtk_vbox_new (FALSE, 1);
  /*  the main label  */
  label = gtk_label_new(_("Scale"));
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  CheckEvenly = gtk_check_button_new_with_label(_("Scale evenly"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CheckEvenly), EvenlyScaleFlag);
  g_signal_connect(G_OBJECT(CheckEvenly), "toggled",
                   G_CALLBACK(ToggleEvenlyScale), NULL);
  gtk_box_pack_start(GTK_BOX(vbox), CheckEvenly, FALSE, FALSE, 0);
  gtk_widget_show(CheckEvenly);

  gtk_box_pack_start(GTK_BOX(VBox), vbox, TRUE, TRUE, 0);
  gtk_widget_show(vbox);
  return vbox;
}

/*****************************************************************************
*  ToolInitScaleSelection
******************************************************************************/
static void ToolInitScaleSelection(GtkWidget *DrawingArea, GdkEventButton *bevent)
{
  ViewStruct *view_data;
  double      Zoom, Xoff, Yoff;
  GSList     *tmp_list;
  ViewStruct *TmpView;
  GtkWidget  *StatusBar;

  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  id = gtk_statusbar_get_context_id(GTK_STATUSBAR(StatusBar), "scale");
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  MouseToReal(bevent->x, bevent->y, DrawingArea, &Xorg, &Yorg);

  if (bevent->state & 4) /* Ctrl */
  { /* The user only want to move the Scale Center */
    switch (view_data->camera_style)
    {
      case ORTHO_XY_CAMERA:
        view_data->frame->ScaleCenter[0] = Xorg;
        view_data->frame->ScaleCenter[1] = Yorg;
        break;

      case ORTHO_XZ_CAMERA:
        view_data->frame->ScaleCenter[0] = Xorg;
        view_data->frame->ScaleCenter[2] = Yorg;
        break;

      case ORTHO_ZY_CAMERA:
        view_data->frame->ScaleCenter[2] = Xorg;
        view_data->frame->ScaleCenter[1] = Yorg;
        break;
    }
    for (tmp_list = view_data->frame->all_views ;
         tmp_list ;
         tmp_list = tmp_list->next)
    {
      TmpView = tmp_list->data;
      gtk_widget_queue_draw(TmpView->canvas);
    }
    return;
  }
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      Xorg -= view_data->frame->ScaleCenter[0];
      Yorg -= view_data->frame->ScaleCenter[1];
      break;

    case ORTHO_XZ_CAMERA:
      Xorg -= view_data->frame->ScaleCenter[0];
      Yorg -= view_data->frame->ScaleCenter[2];
      break;

    case ORTHO_ZY_CAMERA:
      Xorg -= view_data->frame->ScaleCenter[2];
      Yorg -= view_data->frame->ScaleCenter[1];
      break;
  }
  OrgNorme = sqrt(Xorg*Xorg + Yorg*Yorg);
  if (EvenlyScaleFlag && OrgNorme == 0.0)
    return;
  if ((!EvenlyScaleFlag) && (Xorg == 0.0 || Yorg == 0.0))
    return;
  if (view_data->frame->selection == NULL)
    return;
  if (backpixmap)
      g_object_unref(backpixmap);

  backpixmap = gdk_pixmap_new(DrawingArea->window,
                              DrawingArea->allocation.width,
                              DrawingArea->allocation.height,
                              -1);

  draw_unselected(DrawingArea, backpixmap, view_data->camera_style,
                  DrawingArea->allocation.width,
                  DrawingArea->allocation.height);
  gdk_window_set_back_pixmap(DrawingArea->window, backpixmap, FALSE);
  gdk_window_clear(DrawingArea->window);

  AllSelectionTri = ComputeSelectionTriangleMesh(view_data->frame->selection);
  InitialSelectionTri = ComputeSelectionTriangleMesh(view_data->frame->selection);

  DrawSelection(DrawingArea, AllSelectionTri, giram_purple_gc, Xoff, Yoff, Zoom);
  SCALE_STATE = MOVING;
  gtk_statusbar_push(GTK_STATUSBAR(StatusBar), id, _("Scale Factor: "));
  gdk_pointer_grab(DrawingArea->window, FALSE,
                   GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                   NULL, NULL, bevent->time);
}

/*****************************************************************************
*  ToolMotionScaleSelection
******************************************************************************/
static void ToolMotionScaleSelection(GtkWidget *DrawingArea, GdkEventMotion *Mev)
{
  ViewStruct         *view_data;
  GtkWidget          *StatusBar;
  double              Zoom, Xoff, Yoff;
  double              XTrans, YTrans;
  double              Norme, Ratio;
  Matrix              MatTrans;
  TriangleListStruct *TmpTri, *TmpOrigTri;
  gchar              *Message;

  if (SCALE_STATE != MOVING)
    return;

  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  gtk_statusbar_pop(GTK_STATUSBAR(StatusBar), id);
  gdk_window_clear(DrawingArea->window);
  MouseToReal(Mev->x, Mev->y, DrawingArea, &XTrans, &YTrans);

  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      XTrans -= view_data->frame->ScaleCenter[0];
      YTrans -= view_data->frame->ScaleCenter[1];
      break;

    case ORTHO_XZ_CAMERA:
      XTrans -= view_data->frame->ScaleCenter[0];
      YTrans -= view_data->frame->ScaleCenter[2];
      break;

    case ORTHO_ZY_CAMERA:
      XTrans -= view_data->frame->ScaleCenter[2];
      YTrans -= view_data->frame->ScaleCenter[1];
      break;
  }

  if (Mev->state & GDK_SHIFT_MASK)
  {
    if (fabs(YTrans-Yorg) > fabs(XTrans-Xorg))
      XTrans = Xorg;
    else
      YTrans = Yorg;
  }

  Norme = sqrt(XTrans*XTrans + YTrans*YTrans);
  if (Norme == 0.0)
    Ratio = 1.0;
  else
    Ratio = Norme / OrgNorme;
  MIdentity(MatTrans);
  if (EvenlyScaleFlag && !(Mev->state & GDK_SHIFT_MASK))
  {
    MatTrans[0][0] = MatTrans[1][1] = MatTrans[2][2] = Ratio;
  }
  else
  {
    switch (view_data->camera_style)
    {
      case ORTHO_XY_CAMERA:
        MatTrans[0][0] = XTrans / Xorg;
        if (MatTrans[0][0] == 0.0) MatTrans[0][0] = 1.0;
        MatTrans[1][1] = YTrans / Yorg;
        if (MatTrans[1][1] == 0.0) MatTrans[1][1] = 1.0;
        MatTrans[2][2] = 1.0;
        break;

      case ORTHO_XZ_CAMERA:
        MatTrans[0][0] = XTrans / Xorg;
        if (MatTrans[0][0] == 0.0) MatTrans[0][0] = 1.0;
        MatTrans[1][1] = 1.0;
        MatTrans[2][2] = YTrans / Yorg;
        if (MatTrans[2][2] == 0.0) MatTrans[2][2] = 1.0;
        break;

      case ORTHO_ZY_CAMERA:
        MatTrans[0][0] = 1.0;
        MatTrans[1][1] = YTrans / Yorg;
        if (MatTrans[1][1] == 0.0) MatTrans[1][1] = 1.0;
        MatTrans[2][2] = XTrans / Xorg;
        if (MatTrans[2][2] == 0.0) MatTrans[2][2] = 1.0;
        break;
    }
  }
  MatTrans[3][0] = (1.0-MatTrans[0][0])*view_data->frame->ScaleCenter[0];
  MatTrans[3][1] = (1.0-MatTrans[1][1])*view_data->frame->ScaleCenter[1];
  MatTrans[3][2] = (1.0-MatTrans[2][2])*view_data->frame->ScaleCenter[2];
  Message = g_strdup_printf(_("Scale Factor: <%g, %g, %g>"),
                            MatTrans[0][0], MatTrans[1][1], MatTrans[2][2]);
  gtk_statusbar_push(GTK_STATUSBAR(StatusBar), id, Message);
  g_free(Message);
  for (TmpTri = AllSelectionTri, TmpOrigTri = InitialSelectionTri ;
       TmpTri ;
       TmpTri = TmpTri->Next, TmpOrigTri = TmpOrigTri->Next)
  {
    TmpTri->P1[0] = TmpOrigTri->P1[0]*MatTrans[0][0] + TmpOrigTri->P1[1]*MatTrans[1][0] + TmpOrigTri->P1[2]*MatTrans[2][0] + MatTrans[3][0];
    TmpTri->P1[1] = TmpOrigTri->P1[0]*MatTrans[0][1] + TmpOrigTri->P1[1]*MatTrans[1][1] + TmpOrigTri->P1[2]*MatTrans[2][1] + MatTrans[3][1];
    TmpTri->P1[2] = TmpOrigTri->P1[0]*MatTrans[0][2] + TmpOrigTri->P1[1]*MatTrans[1][2] + TmpOrigTri->P1[2]*MatTrans[2][2] + MatTrans[3][2];
    TmpTri->P2[0] = TmpOrigTri->P2[0]*MatTrans[0][0] + TmpOrigTri->P2[1]*MatTrans[1][0] + TmpOrigTri->P2[2]*MatTrans[2][0] + MatTrans[3][0];
    TmpTri->P2[1] = TmpOrigTri->P2[0]*MatTrans[0][1] + TmpOrigTri->P2[1]*MatTrans[1][1] + TmpOrigTri->P2[2]*MatTrans[2][1] + MatTrans[3][1];
    TmpTri->P2[2] = TmpOrigTri->P2[0]*MatTrans[0][2] + TmpOrigTri->P2[1]*MatTrans[1][2] + TmpOrigTri->P2[2]*MatTrans[2][2] + MatTrans[3][2];
    TmpTri->P3[0] = TmpOrigTri->P3[0]*MatTrans[0][0] + TmpOrigTri->P3[1]*MatTrans[1][0] + TmpOrigTri->P3[2]*MatTrans[2][0] + MatTrans[3][0];
    TmpTri->P3[1] = TmpOrigTri->P3[0]*MatTrans[0][1] + TmpOrigTri->P3[1]*MatTrans[1][1] + TmpOrigTri->P3[2]*MatTrans[2][1] + MatTrans[3][1];
    TmpTri->P3[2] = TmpOrigTri->P3[0]*MatTrans[0][2] + TmpOrigTri->P3[1]*MatTrans[1][2] + TmpOrigTri->P3[2]*MatTrans[2][2] + MatTrans[3][2];
  }
  DrawSelection(DrawingArea, AllSelectionTri, giram_purple_gc, Xoff, Yoff, Zoom);
}

/*****************************************************************************
*  ToolReleaseScaleSelection
******************************************************************************/
static void ToolReleaseScaleSelection(GtkWidget *DrawingArea, GdkEventButton *Bev)
{
  ViewStruct        *view_data;
  GtkWidget         *StatusBar;
  double             Zoom, Xoff, Yoff;
  double             XTrans, YTrans;
  GList             *selection;
  ObjectStruct      *tmp_object;
  GSList            *tmp_list;
  ViewStruct        *TmpView;
  double             Norme, Ratio;
  Vector             ScaleVect, TransVect;
/*  GiramObjectEditor *goe;*/

  if (SCALE_STATE != MOVING)
    return;

  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  gtk_statusbar_pop(GTK_STATUSBAR(StatusBar), id);
  MouseToReal(Bev->x, Bev->y, DrawingArea, &XTrans, &YTrans);

  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      XTrans -= view_data->frame->ScaleCenter[0];
      YTrans -= view_data->frame->ScaleCenter[1];
      break;

    case ORTHO_XZ_CAMERA:
      XTrans -= view_data->frame->ScaleCenter[0];
      YTrans -= view_data->frame->ScaleCenter[2];
      break;

    case ORTHO_ZY_CAMERA:
      XTrans -= view_data->frame->ScaleCenter[2];
      YTrans -= view_data->frame->ScaleCenter[1];
      break;
  }

  if (Bev->state & GDK_SHIFT_MASK)
  {
    if (fabs(YTrans-Yorg) > fabs(XTrans-Xorg))
      XTrans = Xorg;
    else
      YTrans = Yorg;
  }

  Norme = sqrt(XTrans*XTrans+YTrans*YTrans);
  if (Norme == 0.0)
    Ratio = 1.0;
  else
    Ratio = Norme / OrgNorme;
  if (EvenlyScaleFlag  && !(Bev->state & GDK_SHIFT_MASK))
  {
    ScaleVect[0] = ScaleVect[1] = ScaleVect[2] = Ratio;
  } else
  {
    switch (view_data->camera_style)
    {
      case ORTHO_XY_CAMERA:
        ScaleVect[0] = XTrans / Xorg;
        if (ScaleVect[0] == 0.0) ScaleVect[0] = 1.0;
        ScaleVect[1] = YTrans / Yorg;
        if (ScaleVect[1] == 0.0) ScaleVect[1] = 1.0;
        ScaleVect[2] = 1.0;
        break;

      case ORTHO_XZ_CAMERA:
        ScaleVect[0] = XTrans / Xorg;
        if (ScaleVect[0] == 0.0) ScaleVect[0] = 1.0;
        ScaleVect[1] = 1.0;
        ScaleVect[2] = YTrans / Yorg;
        if (ScaleVect[2] == 0.0) ScaleVect[2] = 1.0;
        break;

      case ORTHO_ZY_CAMERA:
        ScaleVect[0] = 1.0;
        ScaleVect[1] = YTrans / Yorg;
        if (ScaleVect[1] == 0.0) ScaleVect[1] = 1.0;
        ScaleVect[2] = XTrans / Xorg;
        if (ScaleVect[2] == 0.0) ScaleVect[2] = 1.0;
        break;
    }
  }
  TransVect[0] = (1.0-ScaleVect[0])*view_data->frame->ScaleCenter[0];
  TransVect[1] = (1.0-ScaleVect[1])*view_data->frame->ScaleCenter[1];
  TransVect[2] = (1.0-ScaleVect[2])*view_data->frame->ScaleCenter[2];
  for (selection = view_data->frame->selection ;
       selection ;
       selection = g_list_next(selection) )
  {
    tmp_object = selection->data;
    giram_object_scale(tmp_object, ScaleVect);
    giram_object_translate(tmp_object, TransVect);
    giram_object_build_triangle_mesh(tmp_object);
  }

  SCALE_STATE = NONE;
  { /* Freeing the two temporary mesh */
    TriangleListStruct *Tmp1, *Tmp2;
    Tmp1 = AllSelectionTri;
    AllSelectionTri = NULL;
    while(Tmp1)
    {
      Tmp2=Tmp1->Next;
      g_free(Tmp1);
      Tmp1=Tmp2;
    }
    Tmp1 = InitialSelectionTri;
    InitialSelectionTri = NULL;
    while(Tmp1)
    {
      Tmp2=Tmp1->Next;
      g_free(Tmp1);
      Tmp1=Tmp2;
    }
  }
  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_object_editor_update(csgtree_object_editor);
  giram_create_tree_model(view_data->frame);

  gdk_pointer_ungrab(Bev->time);
  g_object_unref(backpixmap);
  backpixmap = NULL;
}

/****************************************************************************
*  tool_scale_cursor_update
*****************************************************************************/
static void tool_scale_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;

  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_scale_register
*****************************************************************************/
GiramTool *giram_tool_scale_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("Scale");
  tool->Icon           = scale_icon;
  tool->Path           = "<ToolBar>";
  tool->ID             = MT_SCALE;
  tool->OptionsFunc    = BuildScaleOptions;
  tool->button_press   = ToolInitScaleSelection;
  tool->motion         = ToolMotionScaleSelection;
  tool->button_release = ToolReleaseScaleSelection;
  tool->cursor_update  = tool_scale_cursor_update;

  return tool;
}

