/* snap.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <math.h>
#include "giram.h"
#include "utils.h"
#include "snap.h"

#include "giramintl.h"

GtkWidget *SnapDialog = NULL;

/****************************************************************************
*  SnapDialogOK
*****************************************************************************/
static void SnapDialogOK(GtkWidget *dialog)
{
  ViewStruct  *view_data;
  FrameStruct *Frame;
  GSList      *tmp_list;
  ViewStruct  *TmpView;
  GtkWidget   *spin, *button;

  view_data = g_object_get_data(G_OBJECT(dialog), "ViewData");
  Frame = view_data->frame;

  spin = g_object_get_data(G_OBJECT(dialog), "snap_grid_1");
  Frame->SnapGrid[0] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));
  spin = g_object_get_data(G_OBJECT(dialog), "snap_grid_2");
  Frame->SnapGrid[1] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));
  spin = g_object_get_data(G_OBJECT(dialog), "snap_grid_3");
  Frame->SnapGrid[2] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));

  button = g_object_get_data(G_OBJECT(dialog), "snap_to_grid");
  Frame->SnapToGrid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
  button = g_object_get_data(G_OBJECT(dialog), "show_grid");
  Frame->ShowGrid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));

  for (tmp_list = Frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  gtk_widget_destroy(dialog);
}

/****************************************************************************
*  SnapX10
* Multiply all grid values by 10
*****************************************************************************/
static void SnapX10(GtkWidget *dialog)
{
  ViewStruct  *view_data;
  FrameStruct *Frame;
  GtkWidget   *spin;

  view_data = g_object_get_data(G_OBJECT(dialog), "ViewData");
  Frame = view_data->frame;
  /* Get the grid spin button values and multiply by 10 */
  spin = g_object_get_data(G_OBJECT(dialog), "snap_grid_1");
  Frame->SnapGrid[0] = 10.0 * gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), Frame->SnapGrid[0]);

  spin = g_object_get_data(G_OBJECT(dialog), "snap_grid_2");
  Frame->SnapGrid[1] = 10.0 * gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), Frame->SnapGrid[1]);

  spin = g_object_get_data(G_OBJECT(dialog), "snap_grid_3");
  Frame->SnapGrid[2] = 10.0 * gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), Frame->SnapGrid[2]);
}

/****************************************************************************
*  SnapDiv10
* Divide all grid values by 10
*****************************************************************************/
static void SnapDiv10(GtkWidget *dialog)
{
  ViewStruct  *view_data;
  FrameStruct *Frame;
  GtkWidget   *spin;

  view_data = g_object_get_data(G_OBJECT(dialog), "ViewData");
  Frame = view_data->frame;
  /* Get the grid spin button values and divide by 10 */
  spin = g_object_get_data(G_OBJECT(dialog), "snap_grid_1");
  Frame->SnapGrid[0] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin)) / 10.0;
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), Frame->SnapGrid[0]);

  spin = g_object_get_data(G_OBJECT(dialog), "snap_grid_2");
  Frame->SnapGrid[1] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin)) / 10.0;
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), Frame->SnapGrid[1]);

  spin = g_object_get_data(G_OBJECT(dialog), "snap_grid_3");
  Frame->SnapGrid[2] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin)) / 10.0;
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), Frame->SnapGrid[2]);
}

/*************************************************************************
*  snap_dialog_response
**************************************************************************/
static void snap_dialog_response(GtkWidget *dialog, gint response)
{
  switch (response)
  {
    case GTK_RESPONSE_ACCEPT:
      SnapDialogOK(dialog);
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(dialog);
      break;
  }
}

/****************************************************************************
*  SnapDialogShow
*****************************************************************************/
void SnapDialogShow(void)
{
  GtkWidget     *Table, *Label, *Spin, *CheckButton, *Button;
  GtkAdjustment *adj;
  ViewStruct    *view_data;
  FrameStruct   *Frame;

  view_data = get_current_view_data();
  Frame = view_data->frame;
  if (SnapDialog)
  { /* If it's already on screen, juste raise it... */
    gdk_window_raise(SnapDialog->window);
  } else
  {
    SnapDialog = gtk_dialog_new_with_buttons(_("Snap Grid"), NULL, 0,
                                             GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                                             GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
                                             NULL);
    gtk_window_set_position(GTK_WINDOW(SnapDialog), GTK_WIN_POS_MOUSE);
    g_signal_connect(G_OBJECT(SnapDialog), "response",
                     G_CALLBACK(snap_dialog_response), NULL);

    g_object_set_data(G_OBJECT(SnapDialog), "ViewData", view_data);
    /* Put SnapDialog Back to NULL when it gets destroyed */
    g_signal_connect(G_OBJECT(SnapDialog), "destroy",
                     G_CALLBACK(gtk_widget_destroyed), &SnapDialog);
    Table = gtk_table_new(7, 2, FALSE);
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(SnapDialog)->vbox), Table);
    /* The X spin button */
    Label = gtk_label_new(_("X Snap Size: "));
    gtk_table_attach_defaults(GTK_TABLE(Table), Label, 0,1, 0,1);
    adj = (GtkAdjustment *)gtk_adjustment_new(Frame->SnapGrid[0],
                                              0.00, 50.00, 0.05, 0.25, 0.0);
    Spin = gtk_spin_button_new(adj, 0.2, 2);
    gtk_table_attach_defaults(GTK_TABLE(Table), Spin, 1,2, 0,1);
    g_object_set_data(G_OBJECT(SnapDialog), "snap_grid_1", Spin);

    /* The Y spin button */
    Label = gtk_label_new(_("Y Snap Size: "));
    gtk_table_attach_defaults(GTK_TABLE(Table), Label, 0,1, 1,2);
    adj = (GtkAdjustment *)gtk_adjustment_new(Frame->SnapGrid[1],
                                              0.00, 50.00, 0.05, 0.25, 0.0);
    Spin = gtk_spin_button_new(adj, 0.2, 2);
    gtk_table_attach_defaults(GTK_TABLE(Table), Spin, 1,2, 1,2);
    g_object_set_data(G_OBJECT(SnapDialog), "snap_grid_2", Spin);

    /* The Z spin button */
    Label = gtk_label_new(_("Z Snap Size: "));
    gtk_table_attach_defaults(GTK_TABLE(Table), Label, 0,1, 2,3);
    adj = (GtkAdjustment *)gtk_adjustment_new(Frame->SnapGrid[2],
                                              0.00, 50.00, 0.05, 0.25, 0.0);
    Spin = gtk_spin_button_new(adj, 0.2, 2);
    gtk_table_attach_defaults(GTK_TABLE(Table), Spin, 1,2, 2,3);
    g_object_set_data(G_OBJECT(SnapDialog), "snap_grid_3", Spin);

    /* The Snap to Grid Check Button */
    CheckButton = gtk_check_button_new_with_label(_("Snap to Grid"));
    gtk_table_attach_defaults(GTK_TABLE(Table), CheckButton, 0,2, 3,4);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CheckButton),
                                 Frame->SnapToGrid);
    g_object_set_data(G_OBJECT(SnapDialog), "snap_to_grid", CheckButton);

    /* The Show Grid Check Button */
    CheckButton = gtk_check_button_new_with_label(_("Show Grid"));
    gtk_table_attach_defaults(GTK_TABLE(Table), CheckButton, 0,2, 4,5);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CheckButton),
                                 Frame->ShowGrid);
    g_object_set_data(G_OBJECT(SnapDialog), "show_grid", CheckButton);

    Button = gtk_button_new_with_label(_("x10"));
    gtk_table_attach_defaults(GTK_TABLE(Table), Button, 0,1, 6,7);
    g_signal_connect_swapped(G_OBJECT(Button), "clicked",
                             G_CALLBACK(SnapX10), SnapDialog);
    Button = gtk_button_new_with_label(_("/10"));
    gtk_table_attach_defaults(GTK_TABLE(Table), Button, 1,2, 6,7);
    g_signal_connect_swapped(G_OBJECT(Button), "clicked",
                             G_CALLBACK(SnapDiv10), SnapDialog);

    /* Show the whole thing */
    gtk_widget_show_all(SnapDialog);
  }
}

/****************************************************************************
*  MouseToReal
*****************************************************************************/
void MouseToReal(double mx, double my, GtkWidget *Area, double *px, double *py)
{
  ViewStruct *view_data;
  double      Zoom, Xoff, Yoff;
  double      xs = 0.0, ys = 0.0;

  view_data = g_object_get_data(G_OBJECT(Area), "ViewData");
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  *px = Xoff+(mx-Area->allocation.width /2)/Zoom;
  *py = Yoff-(my-Area->allocation.height/2)/Zoom;
  if (!(view_data->frame->SnapToGrid))
    return;
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      xs = view_data->frame->SnapGrid[0];
      ys = view_data->frame->SnapGrid[1];
      break;

    case ORTHO_XZ_CAMERA:
      xs = view_data->frame->SnapGrid[0];
      ys = view_data->frame->SnapGrid[2];
      break;

    case ORTHO_ZY_CAMERA:
      xs = view_data->frame->SnapGrid[2];
      ys = view_data->frame->SnapGrid[1];
      break;
  }
  if (xs != 0.0)
    *px = xs*(floor((*px)/xs+0.5*xs));
  if (ys != 0.0)
    *py = ys*(floor((*py)/ys+0.5*xs));
}

