/* mesh.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "giram.h"
#include "mesh.h"
#include "trimesh.h"

#include "giramintl.h"

static void giram_mesh_build_triangle_mesh(ObjectStruct *mesh);
static gboolean giram_mesh_inside(ObjectStruct *mesh, double x, double y, double z);
static gboolean giram_mesh_is_intersection(ObjectStruct *mesh, Vector origin, Vector direction);
static gboolean giram_mesh_find_intersection_segment(ObjectStruct *mesh,
                                                     Vector in_point, Vector out_point,
                                                     Vector inter_point, Vector inter_norm);

/*****************************************************************************
*  giram_mesh_build_triangle_mesh
******************************************************************************/
static void giram_mesh_build_triangle_mesh(ObjectStruct *mesh)
{ /* XXX: clipped_by */
  Vector              p1, p2, p3, n1, n2, n3;
  TriangleListStruct *TmpTri;
  MeshStruct         *mmesh = (MeshStruct *)mesh;

  if (mesh->FirstTriangle)
    DestroyObjectTriangleMesh(mesh);
  for (TmpTri=mmesh->FirstTriangle ; TmpTri ; TmpTri=TmpTri->Next)
  {
    V3Dcopy(p1, TmpTri->P1);V3Dcopy(n1, TmpTri->N1);
    V3Dcopy(p2, TmpTri->P2);V3Dcopy(n2, TmpTri->N2);
    V3Dcopy(p3, TmpTri->P3);V3Dcopy(n3, TmpTri->N3);
    if (mesh->Trans)
    {
      MEvaluatePoint(p1, mesh->Trans, p1);
      MEvaluatePoint(p2, mesh->Trans, p2);
      MEvaluatePoint(p3, mesh->Trans, p3);
      MEvaluateVector(n1, mesh->Trans, n1);
      MEvaluateVector(n2, mesh->Trans, n2);
      MEvaluateVector(n3, mesh->Trans, n3);
    }
    AddTriangleToObjectMesh(mesh, p1,p2,p3, n1,n2,n3);
  }
}

/*****************************************************************************
*  giram_mesh_inside
******************************************************************************/
static gboolean giram_mesh_inside(ObjectStruct *mesh, double x, double y, double z)
{
  return FALSE; /* We cannot be "Inside" a mesh */
}

/*****************************************************************************
*  giram_mesh_is_intersection
******************************************************************************/
static gboolean giram_mesh_is_intersection(ObjectStruct *mesh, Vector origin, Vector direction)
{ /* XXX: clipped_by */
  Vector              org, dir;
  int                 i;
  MeshStruct         *mmesh = (MeshStruct *)mesh;
  TriangleListStruct *TmpTri;
  Matrix              Mat, InvMat;
  double              D, u, v;

  if (mesh->Trans)
  {
    MInverseEvaluatePoint(org, mesh->Trans, origin);
    MInverseEvaluateVector(dir, mesh->Trans, direction);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      org[i] = origin[i];
      dir[i] = direction[i];
    }
  }
  for (TmpTri = mmesh->FirstTriangle ; TmpTri ; TmpTri=TmpTri->Next)
  {
    Mat[0][0] = TmpTri->P2[0] - TmpTri->P1[0];
    Mat[0][1] = TmpTri->P2[1] - TmpTri->P1[1];
    Mat[0][2] = TmpTri->P2[2] - TmpTri->P1[2];
    Mat[1][0] = TmpTri->P3[0] - TmpTri->P1[0];
    Mat[1][1] = TmpTri->P3[1] - TmpTri->P1[1];
    Mat[1][2] = TmpTri->P3[2] - TmpTri->P1[2];
    Mat[2][0] = -dir[0];
    Mat[2][1] = -dir[1];
    Mat[2][2] = -dir[2];
    D = Mat[0][0]*Mat[1][1]*Mat[2][2]+Mat[0][1]*Mat[1][2]*Mat[2][0]+
        Mat[0][2]*Mat[1][0]*Mat[2][1]-Mat[2][0]*Mat[1][1]*Mat[0][2]-
        Mat[2][1]*Mat[1][2]*Mat[0][0]-Mat[2][2]*Mat[1][0]*Mat[0][1];
    if (D != 0.0)
    {
      InvMat[0][0] =  (Mat[1][1]*Mat[2][2]-Mat[2][1]*Mat[1][2])/D;
      InvMat[0][1] = -(Mat[0][1]*Mat[2][2]-Mat[2][1]*Mat[0][2])/D;
      InvMat[0][2] =  (Mat[0][1]*Mat[1][2]-Mat[1][1]*Mat[0][2])/D;
      InvMat[1][0] = -(Mat[1][0]*Mat[2][2]-Mat[2][0]*Mat[1][2])/D;
      InvMat[1][1] =  (Mat[0][0]*Mat[2][2]-Mat[2][0]*Mat[0][2])/D;
      InvMat[1][2] = -(Mat[0][0]*Mat[1][2]-Mat[1][0]*Mat[0][2])/D;
      InvMat[2][0] =  (Mat[1][0]*Mat[2][1]-Mat[2][0]*Mat[1][1])/D;
      InvMat[2][1] = -(Mat[0][0]*Mat[2][1]-Mat[2][0]*Mat[0][1])/D;
      InvMat[2][2] =  (Mat[0][0]*Mat[1][1]-Mat[1][0]*Mat[0][1])/D;
      u = InvMat[0][0]*(org[0]-TmpTri->P1[0])+
          InvMat[1][0]*(org[1]-TmpTri->P1[1])+
          InvMat[2][0]*(org[2]-TmpTri->P1[2]);
      v = InvMat[0][1]*(org[0]-TmpTri->P1[0])+
          InvMat[1][1]*(org[1]-TmpTri->P1[1])+
          InvMat[2][1]*(org[2]-TmpTri->P1[2]);
      if ((u>=0.0) && (v>=0.0) && (u+v<=1.0))
        return TRUE;
    }
  }
  return FALSE;
}

/*****************************************************************************
*  giram_mesh_find_intersection_segment
******************************************************************************/
static gboolean giram_mesh_find_intersection_segment(ObjectStruct *mesh,
                                                     Vector in_point, Vector out_point,
                                                     Vector inter_point, Vector inter_norm)
{
  return FALSE;
}

/*****************************************************************************
*  giram_mesh_add_triangle
******************************************************************************/
void giram_mesh_add_triangle(ObjectStruct *Mesh, Vector P1, Vector P2, Vector P3)
{
  TriangleListStruct *TmpTri;
  MeshStruct         *MMesh = (MeshStruct *)Mesh;
  Vector              V1, V2;

  if (Mesh == NULL)
    return;
  TmpTri = g_new(TriangleListStruct, 1);
  TmpTri->Next = MMesh->FirstTriangle;
  MMesh->FirstTriangle = TmpTri;
  V3Dcopy(TmpTri->P1, P1);
  V3Dcopy(TmpTri->P2, P2);
  V3Dcopy(TmpTri->P3, P3);

  V1[0] = TmpTri->P2[0] - TmpTri->P1[0];
  V1[1] = TmpTri->P2[1] - TmpTri->P1[1];
  V1[2] = TmpTri->P2[2] - TmpTri->P1[2];
  V2[0] = TmpTri->P3[0] - TmpTri->P1[0];
  V2[1] = TmpTri->P3[1] - TmpTri->P1[1];
  V2[2] = TmpTri->P3[2] - TmpTri->P1[2];
  VCross(TmpTri->N1, V1, V2);
  VCross(TmpTri->N2, V1, V2);
  VCross(TmpTri->N3, V1, V2);
  AddTriangleToObjectMesh(Mesh, TmpTri->P1, TmpTri->P2, TmpTri->P3,
                                TmpTri->N1, TmpTri->N2, TmpTri->N3);
}

/*****************************************************************************
*  giram_mesh_add_smooth_triangle
******************************************************************************/
void giram_mesh_add_smooth_triangle(ObjectStruct *Mesh,
                                    Vector P1, Vector P2, Vector P3,
                                    Vector N1, Vector N2, Vector N3)
{
  TriangleListStruct *TmpTri;
  MeshStruct         *MMesh = (MeshStruct *)Mesh;

  if (Mesh == NULL)
    return;
  TmpTri = g_new(TriangleListStruct, 1);
  TmpTri->Next = MMesh->FirstTriangle;
  MMesh->FirstTriangle = TmpTri;
  V3Dcopy(TmpTri->P1, P1);
  V3Dcopy(TmpTri->P2, P2);
  V3Dcopy(TmpTri->P3, P3);
  V3Dcopy(TmpTri->N1, N1);
  V3Dcopy(TmpTri->N2, N2);
  V3Dcopy(TmpTri->N3, N3);

  AddTriangleToObjectMesh(Mesh, TmpTri->P1, TmpTri->P2, TmpTri->P3,
                                TmpTri->N1, TmpTri->N2, TmpTri->N3);
}

/*****************************************************************************
*  giram_mesh_new
******************************************************************************/
ObjectStruct *giram_mesh_new(void)
{
  ObjectStruct *mesh;
  MeshStruct *mmesh;
  static GiramObjectClass *mesh_klass = NULL;

  if (mesh_klass == NULL)
  {
    mesh_klass = giram_object_class_new();

    mesh_klass->name                      = _("Mesh");
    mesh_klass->build_triangle_mesh       = giram_mesh_build_triangle_mesh;
    mesh_klass->inside                    = giram_mesh_inside;
    mesh_klass->is_intersection           = giram_mesh_is_intersection;
    mesh_klass->find_intersection_segment = giram_mesh_find_intersection_segment;
  }
  mmesh = g_new(MeshStruct, 1);
  mesh = (ObjectStruct *)mmesh;
  InitObject(mesh);
  mesh->Type = MESH_OBJECT;
  mesh->klass = mesh_klass;
  mmesh->FirstTriangle = NULL;
  return mesh;
}

