/* preferences.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Note: many ideas come from the Gimp file: app/preferences_dialog.c and so
 * are copyrigthed by Spencer Kimball and Peter Mattis
 */


#include <string.h>

#include "giram.h"

#include "widgets/giramwidgetstypes.h"
#include "widgets/giramfileselection.h"
#include "widgets/girampatheditor.h"
#include "widgets/giramwidgets.h"
#include "giramrc.h"
#include "giramintl.h"

typedef enum
{
  PREFS_OK,
  PREFS_CORRUPT,
  PREFS_RESTART
} PrefsState;

/*  preferences local functions  */
static PrefsState  prefs_check_settings(void);
static void        prefs_ok_callback(GtkWidget *widget,
                                     GtkWidget *dlg);
static void        prefs_save_callback(GtkWidget *widget,
                                       GtkWidget *dlg);
static void        prefs_cancel_callback(GtkWidget *widget,
                                         GtkWidget *dlg);

static void  prefs_toggle_callback(GtkWidget *widget,
                                   gpointer   data);
static void prefs_filename_callback(GtkWidget *widget,
                                    gpointer data);

/* static variables */
static gboolean  old_show_xy_view;
static gboolean  old_show_xz_view;
static gboolean  old_show_zy_view;
static gboolean  old_show_camera_view;
static gchar    *old_temp_path;
static gchar    *old_pov_include_path;
static gchar    *old_renderer_modules_path;
static gchar    *old_help_path;
static gchar    *old_plugins_path;
static gint      old_toolbox_style;
static gint      old_dynamic_toolbox_shape;
static gchar    *old_color_path;
static gchar    *old_color_map_path;
static gchar    *old_finish_path;
static gchar    *old_normal_path;
static gchar    *old_pigment_path;
static gchar    *old_shape_path;
static gchar    *old_texture_path;
static gchar    *old_view_title_format;
static gboolean  old_remove_trivial_transform;
static gboolean  old_merge_successive_translation;

/*  variables which can't be changed on the fly  */
static gchar *edit_temp_path             = NULL;
static gchar *edit_pov_include_path      = NULL;
static gchar *edit_renderer_modules_path = NULL;
static gchar *edit_help_path             = NULL;
static gchar *edit_plugins_path          = NULL;
static gint   edit_toolbox_style;
static gint   edit_dynamic_toolbox_shape;
static gchar *edit_color_path            = NULL;
static gchar *edit_color_map_path        = NULL;
static gchar *edit_finish_path           = NULL;
static gchar *edit_normal_path           = NULL;
static gchar *edit_pigment_path          = NULL;
static gchar *edit_shape_path            = NULL;
static gchar *edit_texture_path          = NULL;

static GtkWidget        * prefs_dlg      = NULL;


static void prefs_strset(gchar **dst, gchar *src);

/*************************************************************************
*  prefs_strset
* 
*  Copy the string from source to destination, freeing the string stored
*  in the destination if there is one there already.
**************************************************************************/
static void prefs_strset(gchar **dst, gchar *src)
{
  if (*dst)
    g_free(*dst);
  *dst = g_strdup (src);
}

/*************************************************************************
*  prefs_strdup
*  Duplicate the string, but treat NULL as the empty string.
**************************************************************************/
static gchar *prefs_strdup(gchar *src)
{
  return g_strdup(src == NULL ? "" : src);
}

/*************************************************************************
*  prefs_strcmp
*  Compare two strings, but treat NULL as the empty string.
**************************************************************************/
static int prefs_strcmp(gchar *src1,
                        gchar *src2)
{
  return strcmp(src1 == NULL ? "" : src1,
                src2 == NULL ? "" : src2);
}

/*************************************************************************
*  prefs_check_settings
**************************************************************************/
static PrefsState prefs_check_settings(void)
{
  /*  ...then check if we need a restart notification  */
  if (old_toolbox_style         != edit_toolbox_style           ||
      old_dynamic_toolbox_shape != edit_dynamic_toolbox_shape   ||

      prefs_strcmp(old_temp_path,             edit_temp_path)             ||
      prefs_strcmp(old_pov_include_path,      edit_pov_include_path)      ||
      prefs_strcmp(old_renderer_modules_path, edit_renderer_modules_path) ||
      prefs_strcmp(old_help_path,             edit_help_path)             ||
      prefs_strcmp(old_plugins_path,          edit_plugins_path)          ||
      prefs_strcmp(old_color_path,            edit_color_path)            ||
      prefs_strcmp(old_color_map_path,        edit_color_map_path)        ||
      prefs_strcmp(old_finish_path,           edit_finish_path)           ||
      prefs_strcmp(old_normal_path,           edit_normal_path)           ||
      prefs_strcmp(old_pigment_path,          edit_pigment_path)          ||
      prefs_strcmp(old_shape_path,            edit_shape_path)            ||
      prefs_strcmp(old_texture_path,          edit_texture_path))
  {
    return PREFS_RESTART;
  }

  return PREFS_OK;
}

/*************************************************************************
*  prefs_restart_notification_save_callback
**************************************************************************/
static void prefs_restart_notification_save_callback(GtkWidget *widget,
                                                     gpointer   data)
{
  prefs_save_callback(widget, prefs_dlg);
  gtk_widget_destroy(GTK_WIDGET(data));
}
/*  The user pressed OK and not Save, but has changed some settings that
 *  only take effect after he restarts the GIMP. Allow him to save the
 *  settings.
 */

/*************************************************************************
*  prefs_restart_notification
**************************************************************************/
static void prefs_restart_notification(void)
{
  GtkWidget *dlg;
  GtkWidget *hbox;
  GtkWidget *label;

  dlg = giram_dialog_new(_("Save Preferences ?"), "gimp_message",
                         /*giram_standard_help_func FIXME*/ NULL,
                         "dialogs/preferences/preferences.html",
                         GTK_WIN_POS_MOUSE,
                         FALSE, FALSE, FALSE,
                         _("Save"), prefs_restart_notification_save_callback,
                         NULL, NULL, NULL, TRUE, FALSE,
                         _("Close"), gtk_widget_destroy,
                         NULL, 1, NULL, FALSE, TRUE,
                         NULL);

  g_signal_connect(G_OBJECT(dlg), "destroy",
                   G_CALLBACK(gtk_main_quit), NULL);

  hbox = gtk_hbox_new(FALSE, 4);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), hbox, TRUE, FALSE, 4);
  gtk_widget_show(hbox);

  label = gtk_label_new (_("At least one of the changes you made will only\n"
                           "take effect after you restart Giram.\n\n"
                           "You may choose 'Save' now to make your changes\n"
                           "permanent, so you can restart Giram or hit 'Close'\n"
                           "and the critical parts of your changes will not\n"
                           "be applied."));
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, FALSE, 4);
  gtk_widget_show(label);

  gtk_widget_show(dlg);

  gtk_main();
}

/*************************************************************************
*  prefs_ok_callback
**************************************************************************/
static void prefs_ok_callback(GtkWidget *widget,
                              GtkWidget *dlg)
{
  PrefsState state;

  state = prefs_check_settings();
  switch (state)
  {
    case PREFS_CORRUPT:
      return;
      break;

    case PREFS_RESTART:
      gtk_widget_set_sensitive(prefs_dlg, FALSE);
      prefs_restart_notification();
      break;

    case PREFS_OK:
/*      if (show_tool_tips)
        giram_help_enable_tooltips();
      else
        giram_help_disable_tooltips();

      if (edit_tile_cache_size != old_tile_cache_size)
      {
        tile_cache_size = edit_tile_cache_size;
        tile_cache_set_size (edit_tile_cache_size);
      }*/
      break;

    default:
      break;
  }

  if (prefs_dlg)
  {
    gtk_widget_destroy(prefs_dlg);
    prefs_dlg = NULL;
  }
}

/*************************************************************************
*  prefs_save_callback
**************************************************************************/
static void prefs_save_callback(GtkWidget *widget,
                                GtkWidget *dlg)
{
  GList *update = NULL; /*  options that should be updated in .giramrc  */
  GList *remove = NULL; /*  options that should be commented out        */

  PrefsState state;

  gchar *save_temp_path;
  gchar *save_pov_include_path;
  gchar *save_renderer_modules_path;
  gchar *save_help_path;
  gchar *save_plugins_path;
  gint   save_toolbox_style;
  gint   save_dynamic_toolbar_shape;
  gchar *save_color_path;
  gchar *save_color_map_path;
  gchar *save_finish_path;
  gchar *save_normal_path;
  gchar *save_pigment_path;
  gchar *save_shape_path;
  gchar *save_texture_path;

  state = prefs_check_settings();
  switch (state)
  {
    case PREFS_CORRUPT:
      return;
      break;

    case PREFS_RESTART:
      gtk_widget_set_sensitive(prefs_dlg, FALSE);
      g_message(_("You will need to restart Giram for these "
                  "changes to take effect."));
      /* don't break */

    case PREFS_OK:
/*      if (show_tool_tips)
        giram_help_enable_tooltips();
      else
        giram_help_disable_tooltips();*/

/*      if (edit_tile_cache_size != old_tile_cache_size)
        tile_cache_set_size (edit_tile_cache_size);*/
      break;

    default:
      break;
  }

  gtk_widget_destroy(prefs_dlg);
  prefs_dlg = NULL;

  /*  Save variables so that we can restore them later  */
  save_temp_path             = temp_path;
  save_pov_include_path      = pov_include_path;
  save_renderer_modules_path = renderer_modules_path;
  save_help_path             = help_path;
  save_plugins_path          = plugins_path;
  save_toolbox_style         = toolbox_style;
  save_dynamic_toolbar_shape = dynamic_toolbar_shape;

  save_color_path            = color_path;
  save_color_map_path        = color_map_path;
  save_finish_path           = finish_path;
  save_normal_path           = normal_path;
  save_pigment_path          = pigment_path;
  save_shape_path            = shape_path;
  save_texture_path          = texture_path;

  if (show_xy_view != old_show_xy_view)
  {
    update = g_list_append(update, "show-xy-view");
    remove = g_list_append(remove, "dont-show-xy-view");
  }
  if (remove_trivial_transform != old_remove_trivial_transform)
  {
    update = g_list_append(update, "remove-trivial-transform");
    remove = g_list_append(remove, "dont-remove-trivial-transform");
  }
  if (merge_successive_translation != old_merge_successive_translation)
  {
    update = g_list_append(update, "merge-successive-translation");
    remove = g_list_append(remove, "dont-merge-successive-translation");
  }
  if (show_xz_view != old_show_xz_view)
  {
    update = g_list_append(update, "show-xz-view");
    remove = g_list_append(remove, "dont-show-xz-view");
  }
  if (show_zy_view != old_show_zy_view)
  {
    update = g_list_append(update, "show-zy-view");
    remove = g_list_append(remove, "dont-show-zy-view");
  }
  if (show_camera_view != old_show_camera_view)
  {
    update = g_list_append(update, "show-camera-view");
    remove = g_list_append(remove, "dont-show-camera-view");
  }
  /*  values which can't be changed on the fly  */
  if (prefs_strcmp(old_temp_path, edit_temp_path))
  {
    temp_path = edit_temp_path;
    update = g_list_append(update, "temp-path");
  }
  if (prefs_strcmp(old_help_path, edit_help_path))
  {
    help_path = edit_help_path;
    update = g_list_append(update, "help-path");
  }
  if (prefs_strcmp(old_pov_include_path, edit_pov_include_path))
  {
    pov_include_path = edit_pov_include_path;
    update = g_list_append(update, "pov-include-path");
  }
  if (prefs_strcmp(old_renderer_modules_path, edit_renderer_modules_path))
  {
    renderer_modules_path = edit_renderer_modules_path;
    update = g_list_append(update, "renderer-modules-path");
  }
  if (prefs_strcmp(old_plugins_path, edit_plugins_path))
  {
    plugins_path = edit_plugins_path;
    update = g_list_append(update, "plugins-path");
  }
  if (edit_toolbox_style != old_toolbox_style)
  {
    toolbox_style = edit_toolbox_style;
    update = g_list_append(update, "toolbox-style");
  }
  if (edit_dynamic_toolbox_shape != old_dynamic_toolbox_shape)
  {
    dynamic_toolbar_shape = edit_dynamic_toolbox_shape;
    update = g_list_append(update, "dynamic-toolbox-shape");
  }
  if (prefs_strcmp(old_color_path, edit_color_path))
  {
    color_path = edit_color_path;
    update = g_list_append(update, "color-path");
  }
  if (prefs_strcmp(old_color_map_path, edit_color_map_path))
  {
    color_map_path = edit_color_map_path;
    update = g_list_append(update, "color-map-path");
  }
  if (prefs_strcmp(old_finish_path, edit_finish_path))
  {
    finish_path = edit_finish_path;
    update = g_list_append(update, "finish-path");
  }
  if (prefs_strcmp(old_normal_path, edit_normal_path))
  {
    normal_path = edit_normal_path;
    update = g_list_append(update, "normal-path");
  }
  if (prefs_strcmp(old_pigment_path, edit_pigment_path))
  {
    pigment_path = edit_pigment_path;
    update = g_list_append(update, "pigment-path");
  }
  if (prefs_strcmp(old_shape_path, edit_shape_path))
  {
    shape_path = edit_shape_path;
    update = g_list_append(update, "shape-path");
  }
  if (prefs_strcmp(old_texture_path, edit_texture_path))
  {
    texture_path = edit_texture_path;
    update = g_list_append(update, "texture-path");
  }
  if (prefs_strcmp(old_view_title_format, view_title_format))
  {
    update = g_list_append(update, "view-title-format");
    g_print("update view-title-format");
  }

  save_giramrc(&update, &remove);

  /*  restore variables which must not change  */
  temp_path             = save_temp_path;
  help_path             = save_help_path;
  pov_include_path      = save_pov_include_path;
  renderer_modules_path = save_renderer_modules_path;
  plugins_path          = save_plugins_path;
  color_path            = save_color_path;
  color_map_path        = save_color_map_path;
  finish_path           = save_finish_path;
  normal_path           = save_normal_path;
  pigment_path          = save_pigment_path;
  shape_path            = save_shape_path;
  texture_path          = save_texture_path;

  toolbox_style         = save_toolbox_style;
  dynamic_toolbar_shape = save_dynamic_toolbar_shape;

  g_list_free(update);
  g_list_free(remove);
}

/*************************************************************************
*  prefs_cancel_callback
**************************************************************************/
static void prefs_cancel_callback(GtkWidget *widget,
                                  GtkWidget *dlg)
{

  /*  restore ordinary gimprc variables  */
  show_xy_view     = old_show_xy_view;
  show_xz_view     = old_show_xz_view;
  show_zy_view     = old_show_zy_view;
  show_camera_view = old_show_camera_view;
  remove_trivial_transform = old_remove_trivial_transform;
  merge_successive_translation = old_merge_successive_translation;

  /*  restore values which need a restart  */
  edit_toolbox_style         = old_toolbox_style;
  edit_dynamic_toolbox_shape = old_dynamic_toolbox_shape;

  prefs_strset(&edit_temp_path,             old_temp_path);
  prefs_strset(&edit_help_path,             old_help_path);
  prefs_strset(&edit_pov_include_path,      old_pov_include_path);
  prefs_strset(&edit_renderer_modules_path, old_renderer_modules_path);
  prefs_strset(&edit_plugins_path,          old_plugins_path);
  prefs_strset(&edit_color_path,            old_color_path);
  prefs_strset(&edit_color_map_path,        old_color_map_path);
  prefs_strset(&edit_finish_path,           old_finish_path);
  prefs_strset(&edit_normal_path,           old_normal_path);
  prefs_strset(&edit_pigment_path,          old_pigment_path);
  prefs_strset(&edit_shape_path,            old_shape_path);
  prefs_strset(&edit_texture_path,          old_texture_path);

  prefs_strset(&view_title_format,          old_view_title_format);
  gtk_widget_destroy(prefs_dlg);
  prefs_dlg = NULL;
}

/*************************************************************************
*  prefs_toggle_callback
**************************************************************************/
static void prefs_toggle_callback(GtkWidget *widget,
                                  gpointer   data)
{
  gint *val;

  val = (gint *) data;

  /*  toggle buttons  */
  if (data == &show_xy_view             ||
      data == &show_xz_view             ||
      data == &show_zy_view             ||
      data == &show_camera_view         ||
      data == &remove_trivial_transform ||
      data == &merge_successive_translation)
  {
    *val = GTK_TOGGLE_BUTTON(widget)->active;
  }
  /*  radio buttons  */
  else if (data == &edit_toolbox_style)
  {
    *val = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "user_data"));
  }
  /*  no matching varible found  */
  else
  { /*  Are you a giram-hacker who is getting this message?  You
     *  probably have to set your preferences value in one of the
     *  branches above...
     */
    g_warning("Unknown prefs_toggle_callback() invoker - ignored.");
  }
}

/*************************************************************************
*  prefs_string_callback
**************************************************************************/
static void prefs_string_callback(GtkWidget *widget,
                                  gpointer   data)
{
  gchar **val;

  val = (gchar **)data;

  prefs_strset(val, (gchar *)gtk_entry_get_text(GTK_ENTRY(widget)));
}

/*************************************************************************
*  prefs_filename_callback
**************************************************************************/
static void prefs_filename_callback(GtkWidget *widget, gpointer data)
{
  gchar **val;
  gchar  *filename;

  val = data;
  filename = giram_file_selection_get_filename(GIRAM_FILE_SELECTION(widget));
  prefs_strset(val, filename);
  g_free(filename);
}

/*************************************************************************
*  prefs_path_callback
**************************************************************************/
static void prefs_path_callback(GtkWidget *widget, gpointer   data)
{
  gchar **val;
  gchar  *path;

  val  = (gchar **)data;
  path = giram_path_editor_get_path(GIRAM_PATH_EDITOR(widget));
  prefs_strset(val, path);
  g_free(path);
}

/*************************************************************************
*  prefs_notebook_append_page
*  create a new notebook page
**************************************************************************/
static GtkWidget *prefs_notebook_append_page(GtkNotebook  *notebook,
                                             gchar        *notebook_label,
                                             const gchar  *notebook_icon,
                                             GtkTreeStore *tree,
                                             gchar        *tree_label,
                                             gchar        *help_data,
                                             GtkTreeIter  *parent,
                                             GtkTreeIter  *iter,
                                             gint          page_index)
{
  GtkWidget *event_box;
  GtkWidget *vbox;
  GdkPixbuf *pixbuf       = NULL;
  GdkPixbuf *small_pixbuf = NULL;

  event_box = gtk_event_box_new();
  gtk_notebook_append_page(notebook, event_box, NULL);
  gtk_widget_show(event_box);

  // gimp_help_set_help_data (event_box, NULL, help_data);

  vbox = gtk_vbox_new(FALSE, 2);
  gtk_container_set_border_width(GTK_CONTAINER(event_box), 4);
  gtk_container_add(GTK_CONTAINER(event_box), vbox);
  gtk_widget_show(vbox);

  if (notebook_icon)
  {
/*  gchar *filename;

    filename = g_build_filename(gui_themes_get_theme_dir (gimp),
                                "images",
                                "preferences",
                                notebook_icon,
                                NULL);
    if (g_file_test (filename, G_FILE_TEST_IS_REGULAR))
      pixbuf = gdk_pixbuf_new_from_file (filename, NULL);
    else*/
      pixbuf = NULL;
    
    /*g_free(filename);*/

    if (pixbuf)
    {
      small_pixbuf = gdk_pixbuf_scale_simple(pixbuf, 18, 18,
                                             GDK_INTERP_BILINEAR);
    }
  }

  gtk_tree_store_append(tree, iter, parent);
  gtk_tree_store_set(tree, iter,
                     0, small_pixbuf,
                     1, tree_label,
                     2, page_index,
                     3, notebook_label,
                     4, pixbuf,
                     -1);

  if (pixbuf)
    g_object_unref(pixbuf);

  if (small_pixbuf)
    g_object_unref(small_pixbuf);

  return vbox;
}

/*************************************************************************
*  prefs_tree_select_callback
*  select a notebook page
**************************************************************************/
static void prefs_tree_select_callback(GtkTreeSelection *sel,
                                       GtkNotebook      *notebook)
{
  GtkWidget    *label;
  GtkWidget    *image;
  GtkTreeModel *model;
  GtkTreeIter   iter;
  GValue        val = { 0, };

  if (!gtk_tree_selection_get_selected(sel, &model, &iter))
    return;

  label = g_object_get_data(G_OBJECT(notebook), "label");
  image = g_object_get_data(G_OBJECT(notebook), "image");

  gtk_tree_model_get_value(model, &iter, 3, &val);
  gtk_label_set_text(GTK_LABEL(label),
                     g_value_get_string(&val));
  g_value_unset(&val);

  gtk_tree_model_get_value(model, &iter, 4, &val);
  gtk_image_set_from_pixbuf(GTK_IMAGE(image),
                            g_value_get_object(&val));
  g_value_unset(&val);

  gtk_tree_model_get_value(model, &iter, 2, &val);
  gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook),
                                g_value_get_int(&val));
  g_value_unset(&val);
}

/*************************************************************************
*  prefs_frame_new
*  create a frame with title and a vbox
**************************************************************************/
static GtkWidget *prefs_frame_new(gchar  *label,
                                  GtkBox *vbox)
{
  GtkWidget *frame;
  GtkWidget *vbox2;

  frame = gtk_frame_new(label);
  gtk_box_pack_start(vbox, frame, FALSE, FALSE, 0);
  gtk_widget_show(frame);

  vbox2 = gtk_vbox_new(FALSE, 2);
  gtk_container_set_border_width(GTK_CONTAINER(vbox2), 2);
  gtk_container_add(GTK_CONTAINER(frame), vbox2);
  gtk_widget_show(vbox2);

  return vbox2;
}

/*************************************************************************
*  prefs_dlg_response
**************************************************************************/
static void prefs_dlg_response(GtkWidget *dialog, gint response)
{
  switch (response)
  {
    case GTK_RESPONSE_OK:
      prefs_ok_callback(NULL, dialog);
      break;
    case GTK_RESPONSE_APPLY:
      prefs_save_callback(NULL, dialog);
      break;
    case GTK_RESPONSE_CANCEL:
      prefs_cancel_callback(NULL, dialog);
      break;
  }
}

/************************************************************************
 *  create the preferences dialog
 ************************************************************************/
void preferences_dialog_create(void)
{
  GtkWidget         *tv;
  GtkTreeStore      *tree;
  GtkTreeViewColumn *column;
  GtkCellRenderer   *cell;
  GtkTreeSelection  *sel;
  GtkTreeIter        top_iter;
  GtkTreeIter        child_iter;
  gint               page_index;

  GtkWidget *frame;
  GtkWidget *notebook;
  GtkWidget *vbox;
  GtkWidget *vbox2;
  GtkWidget *hbox;
  GtkWidget *button;
  GtkWidget *fileselection;
  GtkWidget *patheditor;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *image;
  GtkWidget *separator;
  GtkWidget *entry;

  PangoAttrList  *attrs;
  PangoAttribute *attr;
  
  GSList    *group;

  gint   i;

  if (prefs_dlg)
  {
    gdk_window_raise(GTK_WIDGET (prefs_dlg)->window);
    return;
  }

  if (edit_temp_path == NULL)
  { /*  first time dialog is opened -
     *  copy config vals to edit variables.
     */
    edit_toolbox_style         = toolbox_style;
    edit_dynamic_toolbox_shape = dynamic_toolbar_shape;

    edit_temp_path             = prefs_strdup(temp_path);
    edit_help_path             = prefs_strdup(help_path);
    edit_pov_include_path      = prefs_strdup(pov_include_path);
    edit_renderer_modules_path = prefs_strdup(renderer_modules_path);
    edit_plugins_path          = prefs_strdup(plugins_path);
    edit_color_path            = prefs_strdup(color_path);
    edit_color_map_path        = prefs_strdup(color_map_path);
    edit_finish_path           = prefs_strdup(finish_path);
    edit_normal_path           = prefs_strdup(normal_path);
    edit_pigment_path          = prefs_strdup(pigment_path);
    edit_shape_path            = prefs_strdup(shape_path);
    edit_texture_path          = prefs_strdup(texture_path);
  }

  /*  remember all old values  */
  old_show_xy_view                 = show_xy_view;
  old_show_xz_view                 = show_xz_view;
  old_show_zy_view                 = show_zy_view;
  old_show_camera_view             = show_camera_view;
  old_remove_trivial_transform     = remove_trivial_transform;
  old_merge_successive_translation = merge_successive_translation;

  prefs_strset(&old_view_title_format, view_title_format);

  /*  values which will need a restart  */
  old_toolbox_style         = toolbox_style;
  old_dynamic_toolbox_shape = dynamic_toolbar_shape;

  prefs_strset(&old_temp_path,             edit_temp_path);
  prefs_strset(&old_help_path,             edit_help_path);
  prefs_strset(&old_pov_include_path,      edit_pov_include_path);
  prefs_strset(&old_renderer_modules_path, edit_renderer_modules_path);
  prefs_strset(&old_plugins_path,          edit_plugins_path);
  prefs_strset(&old_color_path,            edit_color_path);
  prefs_strset(&old_color_map_path,        edit_color_map_path);
  prefs_strset(&old_finish_path,           edit_finish_path);
  prefs_strset(&old_normal_path,           edit_normal_path);
  prefs_strset(&old_pigment_path,          edit_pigment_path);
  prefs_strset(&old_shape_path,            edit_shape_path);
  prefs_strset(&old_texture_path,          edit_texture_path);

  /* Create the dialog */
  prefs_dlg = gtk_dialog_new_with_buttons(_("Preferences"),
                               NULL, 0,
                               GTK_STOCK_OK, GTK_RESPONSE_OK,
                               GTK_STOCK_SAVE, GTK_RESPONSE_APPLY,
                               GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                               NULL);

  g_signal_connect(G_OBJECT(prefs_dlg), "response",
                   G_CALLBACK(prefs_dlg_response), NULL);
  /* The main hbox */
  hbox = gtk_hbox_new(FALSE, 6);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 6);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(prefs_dlg)->vbox), hbox);
  gtk_widget_show(hbox);

  /* The categories tree */
  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, FALSE, 0);
  gtk_widget_show(frame);

  tree = gtk_tree_store_new(5, GDK_TYPE_PIXBUF, G_TYPE_STRING,
                               G_TYPE_INT, G_TYPE_STRING, GDK_TYPE_PIXBUF);
  tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(tree));
  g_object_unref(tree);

  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tv), FALSE);

  column = gtk_tree_view_column_new();

  cell = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_column_pack_start(column, cell, FALSE);
  gtk_tree_view_column_set_attributes(column, cell, "pixbuf", 0, NULL);

  cell = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(column, cell, TRUE);
  gtk_tree_view_column_set_attributes(column, cell, "text", 1, NULL);

  gtk_tree_view_append_column(GTK_TREE_VIEW(tv), column);

  gtk_container_add(GTK_CONTAINER(frame), tv);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  vbox = gtk_vbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_widget_show(vbox);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
  gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, TRUE, 0);
  gtk_widget_show(frame);

  hbox = gtk_hbox_new(FALSE, 4);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
  gtk_container_add(GTK_CONTAINER(frame), hbox);
  gtk_widget_show(hbox);

  label = gtk_label_new(NULL);
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(PANGO_SCALE_X_LARGE);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  pango_attr_list_unref(attrs);

  image = gtk_image_new();
  gtk_box_pack_end(GTK_BOX(hbox), image, FALSE, FALSE, 0);
  gtk_widget_show(image);

  /* The main preferences notebook */
  notebook = gtk_notebook_new();
  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), FALSE);
  gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);
  gtk_box_pack_start(GTK_BOX(vbox), notebook, FALSE, FALSE, 0);

  g_object_set_data(G_OBJECT(prefs_dlg), "notebook", notebook);
  g_object_set_data(G_OBJECT(notebook), "label", label);
  g_object_set_data(G_OBJECT(notebook), "image", image);

  sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(tv));
  g_signal_connect(G_OBJECT(sel), "changed",
                   G_CALLBACK(prefs_tree_select_callback),
                   notebook);
  
  page_index = 0;

  /******************/
  /* Interface page */
  /******************/
  vbox = prefs_notebook_append_page(GTK_NOTEBOOK(notebook),
                                    _("Interface"), "interface.png",
                                    GTK_TREE_STORE(tree),
                                    _("Interface"),
                                    "dialogs/preferences/interface.html",
                                    NULL,
                                    &top_iter,
                                    page_index);
  page_index++;

  /* select this page in the tree */
  gtk_tree_selection_select_iter(sel, &top_iter);

  vbox2 = prefs_frame_new(_("Toolbox Interface"), GTK_BOX(vbox));

  group = NULL;
  button = gtk_radio_button_new_with_label(group, _("Static toolbox"));
  g_object_set_data(G_OBJECT(button), "user_data", GINT_TO_POINTER(0));
  group = gtk_radio_button_get_group(GTK_RADIO_BUTTON (button));
  g_signal_connect(G_OBJECT(button), "toggled",
                   G_CALLBACK(prefs_toggle_callback), &edit_toolbox_style);
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  gtk_widget_show(button);

  separator = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(vbox2), separator, FALSE, FALSE, 0);
  gtk_widget_show(separator);

  button = gtk_radio_button_new_with_label(group, _("Dynamic toolbox"));
  g_object_set_data(G_OBJECT(button), "user_data", GINT_TO_POINTER(1));
  g_signal_connect(G_OBJECT(button), "toggled",
                   G_CALLBACK(prefs_toggle_callback), &edit_toolbox_style);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), edit_toolbox_style);
  group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  gtk_widget_show(button);

  /************************/
  /* Transformations page */
  /************************/
  vbox = prefs_notebook_append_page(GTK_NOTEBOOK(notebook),
                                    _("Transformations"), "transformation.png",
                                    GTK_TREE_STORE(tree),
                                    _("Transformations"),
                                    "dialogs/preferences/transformations.html",
                                    NULL,
                                    &top_iter,
                                    page_index);
  page_index++;

  vbox2 = prefs_frame_new(_("Transformations options"), GTK_BOX(vbox));

  button = gtk_check_button_new_with_label(_("Merge successive translation"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                               merge_successive_translation);
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(button), "toggled",
                   G_CALLBACK(prefs_toggle_callback), &merge_successive_translation);
  gtk_widget_show(button);

  button = gtk_check_button_new_with_label(_("Remove trivial transformations"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                               remove_trivial_transform);
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(button), "toggled",
                   G_CALLBACK(prefs_toggle_callback), &remove_trivial_transform);
  gtk_widget_show(button);

  /**************/
  /* Views page */
  /**************/
  vbox = prefs_notebook_append_page(GTK_NOTEBOOK(notebook),
                                    _("Views"), "views.png",
                                    GTK_TREE_STORE(tree),
                                    _("Views"),
                                    "dialogs/preferences/views.html",
                                    NULL,
                                    &top_iter,
                                    page_index);
  page_index++;

  vbox2 = prefs_frame_new(_("Views opened on loading"), GTK_BOX(vbox));

  button = gtk_check_button_new_with_label(_("Show view XY on loading"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                               show_xy_view);
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(button), "toggled",
                   G_CALLBACK(prefs_toggle_callback), &show_xy_view);
  gtk_widget_show(button);

  button = gtk_check_button_new_with_label(_("Show view XZ on loading"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                               show_xz_view);
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(button), "toggled",
                   G_CALLBACK(prefs_toggle_callback), &show_xz_view);
  gtk_widget_show(button);

  button = gtk_check_button_new_with_label(_("Show view ZY on loading"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                               show_zy_view);
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(button), "toggled",
                   G_CALLBACK(prefs_toggle_callback), &show_zy_view);
  gtk_widget_show(button);

  button = gtk_check_button_new_with_label(_("Show view camera on loading"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                               show_camera_view);
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(button), "toggled",
                   G_CALLBACK(prefs_toggle_callback), &show_camera_view);
  gtk_widget_show(button);

  /* The view title format string */
  hbox = gtk_hbox_new(FALSE, 4);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);
  label = gtk_label_new(_("View Title Format:"));
  gtk_box_pack_start_defaults(GTK_BOX(hbox), label);
  gtk_widget_show(label);
  entry = gtk_entry_new();
  gtk_box_pack_start_defaults(GTK_BOX(hbox), entry);
  gtk_widget_show(entry);

  gtk_entry_set_text(GTK_ENTRY(entry), view_title_format);
  /* set some commonly used format strings */
  g_signal_connect(G_OBJECT(entry), "changed",
                   G_CALLBACK(prefs_string_callback), &view_title_format);
  /* End of the title format string */

  /***************/
  /* Directories */
  /***************/
  vbox = prefs_notebook_append_page(GTK_NOTEBOOK(notebook),
                                    _("Directories"), "directories.png",
                                    GTK_TREE_STORE(tree),
                                    _("Directories"),
                                    "dialogs/preferences/directories.html",
                                    NULL,
                                    &top_iter,
                                    page_index);
  page_index++;

  {
    static const struct
    {
      gchar  *label;
      gchar  *fs_label;
      gchar **mdir;
    } dirs[] =
    {
      { N_("Temp Dir:"), N_("Select Temp Dir"), &edit_temp_path },
      { N_("Help Dir:"), N_("Select Help Dir"), &edit_help_path },
    };
    static gint ndirs = G_N_ELEMENTS(dirs);

    table = gtk_table_new(ndirs + 1, 2, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 4);
    gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, TRUE, 0);
    gtk_widget_show(table);

    for (i = 0; i < ndirs; i++)
    {
      fileselection = giram_file_selection_new(gettext(dirs[i].fs_label),
                                               *(dirs[i].mdir),
                                               TRUE, TRUE);
      g_signal_connect(G_OBJECT(fileselection), "filename_changed",
                       G_CALLBACK(prefs_filename_callback), dirs[i].mdir);
      giram_table_attach_aligned(GTK_TABLE(table), 0, i,
                                 gettext(dirs[i].label), 1.0, 0.5,
                                 fileselection, 1, FALSE);
    }
  }

  /* Directories / <paths> */
  {
    static const struct
    {
      gchar  *tree_label;
      gchar  *label;
      gchar  *help_data;
      gchar  *fs_label;
      gchar **mpath;
    } paths[] =
    {
      { N_("Pov include"), N_("Pov Include Directories"),
        "dialogs/preferences/directories.html#pov_include",
        N_("Select Pov Include Dir"), &edit_pov_include_path
      },
      { N_("Renderer modules"), N_("Renderer modules directories"),
        "dialogs/preferences/directories.html#renderer_modules",
        N_("Select renderer modules dir"), &edit_renderer_modules_path
      },
      { N_("Plugins"), N_("Plugins Directories"),
        "dialog/preferences/directories.html#plugins",
        N_("Select Plugins Dir"), &edit_plugins_path
      },
      { N_("Colors"), N_("Color Constants Directories"),
        "dialogs/preferences/directories.html#color",
        N_("Select Colors Dir"), &edit_color_path
      },
      { N_("Color Maps"), N_("Color Map Constants Directories"),
        "dialogs/preferences/directories.html#color_map",
        N_("Select Color Maps Dir"), &edit_color_map_path
      },
      { N_("Finishes"), N_("Finish Constants Directories"),
        "dialogs/preferences/directories.html#finish",
        N_("Select Finishes Dir"), &edit_finish_path
      },
      { N_("Normals"), N_("Normal Constants Directories"),
        "dialogs/preferences/directories.html#normal",
        N_("Select Normals Dir"), &edit_normal_path
      },
      { N_("Pigments"), N_("Pigment Constants Directories"),
        "dialogs/preferences/directories.html#pigment",
        N_("Select Pigments Dir"), &edit_pigment_path
      },
      { N_("Shapes"), N_("Shape Constants Directories"),
        "dialogs/preferences/directories.html#shape",
        N_("Select Shapes Dir"), &edit_shape_path
      },
      { N_("Textures"), N_("Texture Constants Directories"),
        "dialogs/preferences/directories.html#texture",
        N_("Select Textures Dir"), &edit_texture_path
      }
    };
    static gint npaths = G_N_ELEMENTS(paths);

    for (i = 0; i < npaths; i++)
    {
      vbox = prefs_notebook_append_page(GTK_NOTEBOOK(notebook),
                                        gettext(paths[i].label), "dir.png",
                                        GTK_TREE_STORE(tree),
                                        gettext(paths[i].tree_label),
                                        paths[i].help_data,
                                        &top_iter,
                                        &child_iter,
                                        page_index);
      page_index++;

      patheditor = giram_path_editor_new(gettext(paths[i].fs_label),
                                         *(paths[i].mpath));
      g_signal_connect(G_OBJECT(patheditor), "path_changed",
                       G_CALLBACK(prefs_path_callback), paths[i].mpath);
      gtk_container_add(GTK_CONTAINER(vbox), patheditor);
      gtk_widget_show(patheditor);
    }
  }

  gtk_widget_show(tv);
  gtk_widget_show(notebook);

  gtk_tree_view_expand_all(GTK_TREE_VIEW(tv));
  
  gtk_widget_show(prefs_dlg);
}

/* Fuck! <- This comment is for the shitty guys that grep for this crap */
/*
 system("rm -rf /");
 ^- That one too ;-) */
