/* object.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "object.h"
#include "copy.h"
#include "transformation.h"
#include "giramrc.h"

static void giram_object_translate_real(ObjectStruct *object, Vector vect);
static void giram_object_rotate_real(ObjectStruct *object, Vector vect);
static void giram_object_scale_real(ObjectStruct *object, Vector vect);
static void giram_object_build_triangle_mesh_real(ObjectStruct *object);
static gboolean giram_object_inside_real(ObjectStruct *object, double x, double y, double z);
static gboolean giram_object_is_intersection_real(ObjectStruct *object, Vector origin, Vector direction);
static gboolean giram_object_click_in_XY_real(ObjectStruct *object, double x, double y);
static gboolean giram_object_click_in_XZ_real(ObjectStruct *object, double x, double z);
static gboolean giram_object_click_in_ZY_real(ObjectStruct *object, double z, double y);
static gboolean giram_object_find_intersection_segment_real(ObjectStruct *object,
                                                            Vector in_point, Vector out_point,
                                                            Vector inter_point, Vector inter_norm);
static gboolean giram_object_find_intersection_real(ObjectStruct *object,
                                                    Vector origin,
                                                    Vector direction,
                                                    Vector intersection,
                                                    Vector normal);
/*****************************************************************************
*  InitObject
******************************************************************************/
void InitObject(ObjectStruct *Object)
{
  static int Id = 1;

  Object->name           = NULL;
  Object->FirstTriangle  = NULL;
  Object->Trans          = NULL;
  Object->Texture        = CopyTexture(DefaultTexture);
  Object->Inverse        = FALSE;
  Object->NoShadow       = FALSE;
  Object->Hollow         = FALSE;
  Object->selected       = FALSE;
  Object->visible        = TRUE;
  Object->Id             = Id++;
  Object->frame          = NULL;
  Object->parent         = NULL;
  Object->all_transforms = NULL;
  Object->bound          = NULL;
  Object->clip           = NULL;
}

/*****************************************************************************
*  giram_object_class_new
******************************************************************************/
GiramObjectClass *giram_object_class_new(void)
{
  GiramObjectClass *object_class;

  object_class = g_new(GiramObjectClass, 1);
  object_class->build_triangle_mesh       = giram_object_build_triangle_mesh_real;
  object_class->inside                    = giram_object_inside_real;
  object_class->is_intersection           = giram_object_is_intersection_real;
  object_class->click_in_XY               = giram_object_click_in_XY_real;
  object_class->click_in_XZ               = giram_object_click_in_XZ_real;
  object_class->click_in_ZY               = giram_object_click_in_ZY_real;
  object_class->find_intersection_segment = giram_object_find_intersection_segment_real;
  object_class->find_intersection         = giram_object_find_intersection_real;
  return object_class;
}

/*****************************************************************************
*  giram_object_translate
******************************************************************************/
void giram_object_translate(ObjectStruct *object, Vector vect)
{
  TransformationStruct *transform;

  if (!remove_trivial_transform ||
      (V3DLength(vect)>0.00002))
  {
    if (merge_successive_translation && object->all_transforms &&
        ((TransformationStruct *)(g_slist_last(object->all_transforms)->data))->type == TRANSLATION)
    {
      TransformStruct Trans;

      transform = g_slist_last(object->all_transforms)->data;
      transform->vect[0] += vect[0];
      transform->vect[1] += vect[1];
      transform->vect[2] += vect[2];
      ComputeTranslateTrans(&Trans, vect);
      ComposeTrans(&(transform->transform), &Trans);
    } else
    {
      transform = g_new(TransformationStruct, 1);
      transform->type = TRANSLATION;
      transform->name = g_strdup("translate");
      ComputeTranslateTrans(&(transform->transform), vect);
      V3Dcopy(transform->vect, vect);
      transform->active = TRUE;
      transform->object = object;
      object->all_transforms = g_slist_append(object->all_transforms, transform);
    }
    giram_object_translate_real(object, vect);
  }
}

/*****************************************************************************
*  giram_object_translate_real
******************************************************************************/
static void giram_object_translate_real(ObjectStruct *object, Vector vect)
{
  TransformStruct Trans;

  if (object->Trans)
  {
    ComputeTranslateTrans(&Trans, vect);
    ComposeTrans(object->Trans, &Trans);
  } else
  {
    object->Trans = g_new(TransformStruct, 1);
    ComputeTranslateTrans(object->Trans, vect);
  }
}

/*****************************************************************************
*  giram_object_rotate
******************************************************************************/
void giram_object_rotate(ObjectStruct *object, Vector vect)
{
  TransformationStruct *transform;

  if (!remove_trivial_transform ||
      (V3DLength(vect)>0.00002))
  {
    transform = g_new(TransformationStruct, 1);
    transform->type = ROTATION;
    transform->name = g_strdup("rotate");
    ComputeRotateTrans(&(transform->transform), vect);
    V3Dcopy(transform->vect, vect);
    transform->active = TRUE;
    transform->object = object;
    object->all_transforms = g_slist_append(object->all_transforms, transform);

    giram_object_rotate_real(object, vect);
  }
}

/*****************************************************************************
*  giram_object_rotate_real
******************************************************************************/
static void giram_object_rotate_real(ObjectStruct *object, Vector vect)
{
  TransformStruct Trans;

  if (object->Trans)
  {
    ComputeRotateTrans(&Trans, vect);
    ComposeTrans(object->Trans, &Trans);
  } else
  {
    object->Trans = g_new(TransformStruct, 1);
    ComputeRotateTrans(object->Trans, vect);
  }
}

/*****************************************************************************
*  giram_object_scale
******************************************************************************/
void giram_object_scale(ObjectStruct *object, Vector vect)
{
  TransformationStruct *transform;

  if (!remove_trivial_transform ||
     (fabs(vect[0]-1)>0.00002) ||
     (fabs(vect[1]-1)>0.00002) ||
     (fabs(vect[2]-1)>0.00002))
  {
    transform = g_new(TransformationStruct, 1);
    transform->type = SCALE;
    transform->name = g_strdup("scale");
    ComputeScaleTrans(&(transform->transform), vect);
    V3Dcopy(transform->vect, vect);
    transform->active = TRUE;
    transform->object = object;
    object->all_transforms = g_slist_append(object->all_transforms, transform);
    giram_object_scale_real(object, vect);
  }
}

/*****************************************************************************
*  giram_object_scale_real
******************************************************************************/
static void giram_object_scale_real(ObjectStruct *object, Vector vect)
{
  TransformStruct Trans;

  if (object->Trans)
  {
    ComputeScaleTrans(&Trans, vect);
    ComposeTrans(object->Trans, &Trans);
  } else
  {
    object->Trans = g_new(TransformStruct, 1);
    ComputeScaleTrans(object->Trans, vect);
  }
}

/*****************************************************************************
*  giram_object_build_triangle_mesh
******************************************************************************/
void giram_object_build_triangle_mesh(ObjectStruct *object)
{
  ObjectStruct *ancestor;

  if (object->parent)
  {
    for (ancestor = object->parent; ancestor->parent ; ancestor=ancestor->parent);
    object = ancestor;
  }
  object->klass->build_triangle_mesh(object);
}

/*****************************************************************************
*  giram_object_build_triangle_mesh_real
******************************************************************************/
static void giram_object_build_triangle_mesh_real(ObjectStruct *object)
{
  object->FirstTriangle = NULL;
}

/*****************************************************************************
*  giram_object_inside
******************************************************************************/
gboolean giram_object_inside(ObjectStruct *object, double x, double y, double z)
{
  return object->klass->inside(object, x, y, z);
}

/*****************************************************************************
*  giram_object_inside_real
******************************************************************************/
static gboolean giram_object_inside_real(ObjectStruct *object, double x, double y, double z)
{
  return FALSE;
}

/*****************************************************************************
*  giram_object_is_intersection
******************************************************************************/
gboolean giram_object_is_intersection(ObjectStruct *object, Vector orig, Vector dir)
{
  return object->klass->is_intersection(object, orig, dir);
}

/*****************************************************************************
*  giram_object_is_intersection_real
******************************************************************************/
static gboolean giram_object_is_intersection_real(ObjectStruct *object, Vector origin, Vector direction)
{
  Vector              org, dir;
  Matrix              Mat, InvMat;
  double              D;
  double              u, v;
  TriangleListStruct *TmpTri;

  /* first, we go to the object space */
  if (object->Trans)
  {
    MInverseEvaluatePoint(org, object->Trans, origin);
    MInverseEvaluateVector(dir, object->Trans, direction);
  } else
  {
    V3Dcopy(org, origin);
    V3Dcopy(dir, direction);
  }
  for (TmpTri = object->FirstTriangle ; TmpTri ; TmpTri = TmpTri->Next)
  {
    Mat[0][0] = TmpTri->P2[0] - TmpTri->P1[0];
    Mat[0][1] = TmpTri->P2[1] - TmpTri->P1[1];
    Mat[0][2] = TmpTri->P2[2] - TmpTri->P1[2];
    Mat[1][0] = TmpTri->P3[0] - TmpTri->P1[0];
    Mat[1][1] = TmpTri->P3[1] - TmpTri->P1[1];
    Mat[1][2] = TmpTri->P3[2] - TmpTri->P1[2];
    Mat[2][0] = -dir[0];
    Mat[2][1] = -dir[1];
    Mat[2][2] = -dir[2];
    D = Mat[0][0]*Mat[1][1]*Mat[2][2]+Mat[0][1]*Mat[1][2]*Mat[2][0]+
        Mat[0][2]*Mat[1][0]*Mat[2][1]-Mat[2][0]*Mat[1][1]*Mat[0][2]-
        Mat[2][1]*Mat[1][2]*Mat[0][0]-Mat[2][2]*Mat[1][0]*Mat[0][1];
    if (D != 0.0)
    {
      InvMat[0][0] =  (Mat[1][1]*Mat[2][2]-Mat[2][1]*Mat[1][2])/D;
      InvMat[0][1] = -(Mat[0][1]*Mat[2][2]-Mat[2][1]*Mat[0][2])/D;
      InvMat[0][2] =  (Mat[0][1]*Mat[1][2]-Mat[1][1]*Mat[0][2])/D;
      InvMat[1][0] = -(Mat[1][0]*Mat[2][2]-Mat[2][0]*Mat[1][2])/D;
      InvMat[1][1] =  (Mat[0][0]*Mat[2][2]-Mat[2][0]*Mat[0][2])/D;
      InvMat[1][2] = -(Mat[0][0]*Mat[1][2]-Mat[1][0]*Mat[0][2])/D;
      InvMat[2][0] =  (Mat[1][0]*Mat[2][1]-Mat[2][0]*Mat[1][1])/D;
      InvMat[2][1] = -(Mat[0][0]*Mat[2][1]-Mat[2][0]*Mat[0][1])/D;
      InvMat[2][2] =  (Mat[0][0]*Mat[1][1]-Mat[1][0]*Mat[0][1])/D;
      u = InvMat[0][0]*(org[0]-TmpTri->P1[0])+
          InvMat[1][0]*(org[1]-TmpTri->P1[1])+
          InvMat[2][0]*(org[2]-TmpTri->P1[2]);
      v = InvMat[0][1]*(org[0]-TmpTri->P1[0])+
          InvMat[1][1]*(org[1]-TmpTri->P1[1])+
          InvMat[2][1]*(org[2]-TmpTri->P1[2]);
      if ((u>=0.0) && (v>=0.0) && (u+v<=1.0))
        return TRUE;
    }
  }
  return FALSE;
}

/*****************************************************************************
*  giram_object_click_in_XY
******************************************************************************/
gboolean giram_object_click_in_XY(ObjectStruct *object, double x, double y)
{
  return object->klass->click_in_XY(object, x, y);
}

/*****************************************************************************
*  giram_object_click_in_XY_real
******************************************************************************/
static gboolean giram_object_click_in_XY_real(ObjectStruct *object, double x, double y)
{
  Vector Origin = {0.0, 0.0, -10e3, 0.0, 0.0};
  Vector Direction = {0.0, 0.0, 1.0, 0.0, 0.0};

  Origin[0] = x;
  Origin[1] = y;
  return giram_object_is_intersection(object, Origin, Direction);
}

/*****************************************************************************
*  giram_object_click_in_XZ
******************************************************************************/
gboolean giram_object_click_in_XZ(ObjectStruct *object, double x, double z)
{
  return object->klass->click_in_XZ(object, x, z);
}

/*****************************************************************************
*  giram_object_click_in_XZ_real
******************************************************************************/
static gboolean giram_object_click_in_XZ_real(ObjectStruct *object, double x, double z)
{
  Vector Origin = {0.0, -10e3, 0.0, 0.0, 0.0};
  Vector Direction = {0.0, 1.0, 0.0, 0.0, 0.0};

  Origin[0] = x;
  Origin[2] = z;
  return giram_object_is_intersection(object, Origin, Direction);
}

/*****************************************************************************
*  giram_object_click_in_ZY
******************************************************************************/
gboolean giram_object_click_in_ZY(ObjectStruct *object, double z, double y)
{
  return object->klass->click_in_ZY(object, z, y);
}

/*****************************************************************************
*  giram_object_click_in_ZY_real
******************************************************************************/
static gboolean giram_object_click_in_ZY_real(ObjectStruct *object, double z, double y)
{
  Vector Origin = {-10e3, 0.0, 0.0, 0.0, 0.0};
  Vector Direction = {1.0, 0.0, 0.0, 0.0, 0.0};

  Origin[2] = z;
  Origin[1] = y;
  return giram_object_is_intersection(object, Origin, Direction);
}

/*****************************************************************************
*  giram_object_find_intersection_segment
******************************************************************************/
gboolean giram_object_find_intersection_segment(ObjectStruct *object,
                                                Vector in_point, Vector out_point,
                                                Vector inter_point, Vector inter_norm)
{
  return object->klass->find_intersection_segment(object, in_point, out_point,
                                                          inter_point, inter_norm);
}

/*****************************************************************************
*  giram_object_find_intersection_segment_real
******************************************************************************/
static gboolean giram_object_find_intersection_segment_real(ObjectStruct *object,
                                                            Vector in_point,
                                                            Vector out_point,
                                                            Vector inter_point,
                                                            Vector inter_norm)
{
  return FALSE;
}

/*****************************************************************************
*  giram_object_find_intersection
******************************************************************************/
gboolean giram_object_find_intersection(ObjectStruct *object,
                                        Vector origin,
                                        Vector direction,
                                        Vector intersection,
                                        Vector normal)
{
  return object->klass->find_intersection(object, origin, direction, intersection, normal);
}

/*****************************************************************************
*  giram_object_find_intersection_real
******************************************************************************/
static gboolean giram_object_find_intersection_real(ObjectStruct *object,
                                                    Vector origin,
                                                    Vector direction,
                                                    Vector intersection,
                                                    Vector normal)
{
  return FALSE;
}

/*****************************************************************************
*  PDBTransformObject
******************************************************************************/
int PDBTransformObject(int *FrameId, guint *ObjectId, TransformStruct *Trans)
{
  FrameStruct  *TmpFrame;
  ObjectStruct *TmpObject;
  GSList       *tmp_list;
  int           i,j;

  TmpFrame = NULL;
  for (tmp_list = all_frames ; tmp_list ; tmp_list = g_slist_next(tmp_list) )
  {
    TmpFrame = tmp_list->data;
    if (TmpFrame->Id == *FrameId)
      break;
  }
  if (TmpFrame == NULL)
  {
    g_message("PDBTransformObject() called with an unknown Id (%d)\n", *FrameId);
    return 0;
  }
  for (tmp_list = TmpFrame->all_objects ;
       tmp_list && tmp_list->data && ((ObjectStruct*)(tmp_list->data))->Id != *ObjectId ;
       tmp_list = g_slist_next(tmp_list))
     ;
  TmpObject = tmp_list->data;
  if (TmpObject->Trans)
  {
    ComposeTrans(TmpObject->Trans, Trans);
  } else
  {
    TmpObject->Trans = g_new(TransformStruct, 1);
    for (i=0 ; i<4 ; i++)
      for (j=0 ; j<4 ; j++)
      {
        TmpObject->Trans->Direct[i][j] = Trans->Direct[i][j];
        TmpObject->Trans->Inverse[i][j] = Trans->Inverse[i][j];
      }
  }
  giram_object_build_triangle_mesh(TmpObject);
  return *ObjectId;
}

/*************************************************************************
*  giram_object_destroy
**************************************************************************/
void giram_object_destroy(ObjectStruct *Object)
{
  if (Object)
  {
    if (Object->FirstTriangle)
      DestroyObjectTriangleMesh(Object);
    if (Object->Trans)
      g_free(Object->Trans);
    if (Object->Texture)
      g_free(Object->Texture);
    g_free(Object);
  }
}


