/* SavePov.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 David Odin <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "giram.h"

#include "primitives/bicubic_patch.h"
#include "primitives/box.h"
#include "primitives/cone.h"
#include "csg.h"
#include "primitives/cylinder.h"
#include "primitives/disc.h"
#include "primitives/heightfield.h"
#include "primitives/mesh.h"
#include "primitives/plane.h"
#include "primitives/quadric.h"
#include "primitives/sor.h"
#include "primitives/sphere.h"
#include "primitives/superellipsoid.h"
#include "primitives/torus.h"
#include "primitives/triangle.h"

#include "camera.h"

#include "color_map.h"
#include "view.h"
#include "giramintl.h"

/****************************************************************************
*  save_pov_transformation
*****************************************************************************/
static void save_pov_transformation(FILE *out_file, GSList *all_transforms, guint indentation)
{
  GSList *tmp_list;
  gchar  *s;

  s = g_strnfill(indentation*2, ' ');
  
  for (tmp_list = all_transforms ;tmp_list ;tmp_list=tmp_list->next)
  {
    TransformationStruct *transform = tmp_list->data;

    switch (transform->type)
    {
      case TRANSLATION:
         fprintf(out_file, "%s  translate <%g, %g, %g>\n", s,
                 transform->vect[0], transform->vect[1], transform->vect[2]);
         break;
      case ROTATION:
         fprintf(out_file, "%s  rotate <%g, %g, %g>\n", s,
                 transform->vect[0], transform->vect[1], transform->vect[2]);
         break;
      case SCALE:
         fprintf(out_file, "%s  scale <%g, %g, %g>\n", s,
                 transform->vect[0], transform->vect[1], transform->vect[2]);
         break;
      case MATRIX:
        fprintf(out_file, "%smatrix  <%g, %g, %g,\n", s,
                transform->transform.Direct[0][0],
                transform->transform.Direct[0][1],
                transform->transform.Direct[0][2]);
        fprintf(out_file, "%s   %g, %g, %g,\n", s,
                transform->transform.Direct[1][0],
                transform->transform.Direct[1][1],
                transform->transform.Direct[1][2]);
        fprintf(out_file, "%s   %g, %g, %g,\n", s,
                transform->transform.Direct[2][0],
                transform->transform.Direct[2][1],
                transform->transform.Direct[2][2]);
        fprintf(out_file, "%s   %g, %g, %g>\n", s,
                transform->transform.Direct[3][0],
                transform->transform.Direct[3][1],
                transform->transform.Direct[3][2]);
        break;
    }
  }
}

/*************************************************************************
*  save_pov_color_map
**************************************************************************/
static void save_pov_color_map(FILE *OutFile, GList *color_map, guint indentation)
{
  gchar           *s;
  GList           *tmp;
  color_map_entry *entry;

  s = g_strnfill(indentation*2, ' ');

  fprintf(OutFile, "%scolor_map\n%s{\n", s, s);
  for (tmp = color_map ; tmp ; tmp = g_list_next(tmp) )
  {
    entry = tmp->data;
    fprintf(OutFile, "  %s[%g  color <%g, %g, %g>]\n", s,
            entry->position, entry->color[0],
            entry->color[1], entry->color[2]);
  }
  fprintf(OutFile, "%s}\n", s);
}

/*****************************************************************************
*  SavePovPigment
******************************************************************************/
static void SavePovPigment(FILE *OutFile, PigmentStruct *Pigment, guint indentation)
{
  gchar *s;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%spigment\n%s{\n", s, s);
  switch (Pigment->Type)
  {
    case PAT_SOLID_COLOR:
      fprintf(OutFile, "%s  color rgbft <%g, %g, %g,  %g, %g>\n", s,
              Pigment->Color[0], Pigment->Color[1], Pigment->Color[2],
              Pigment->Color[3], Pigment->Color[4]);
      break;

    case PAT_BRICK:
      fprintf(OutFile, "%s  brick rgbft <%g, %g, %g,  %g, %g>, rgbft <%g, %g, %g,  %g, %g>\n", s,
              Pigment->Color[0], Pigment->Color[1], Pigment->Color[2],
              Pigment->Color[3], Pigment->Color[4],
              Pigment->Color2[0], Pigment->Color2[1], Pigment->Color2[2],
              Pigment->Color2[3], Pigment->Color2[4]);
      break;

    case PAT_CHECKER:
      fprintf(OutFile, "%s  checker rgbft <%g, %g, %g,  %g, %g>, rgbft <%g, %g, %g,  %g, %g>\n", s,
              Pigment->Color[0], Pigment->Color[1], Pigment->Color[2],
              Pigment->Color[3], Pigment->Color[4],
              Pigment->Color2[0], Pigment->Color2[1], Pigment->Color2[2],
              Pigment->Color2[3], Pigment->Color2[4]);
      break;

    case PAT_HEXAGON:
      fprintf(OutFile, "%s  hexagon rgbft <%g, %g, %g,  %g, %g>, rgbft <%g, %g, %g,  %g, %g>, rgbft <%g, %g, %g,  %g, %g>\n", s,
              Pigment->Color[0], Pigment->Color[1], Pigment->Color[2],
              Pigment->Color[3], Pigment->Color[4],
              Pigment->Color2[0], Pigment->Color2[1], Pigment->Color2[2],
              Pigment->Color2[3], Pigment->Color2[4],
              Pigment->Color3[0], Pigment->Color3[1], Pigment->Color3[2],
              Pigment->Color3[3], Pigment->Color3[4]);
      break;

    case PAT_ONION:
          fprintf(OutFile, "%s  onion\n", s);
          save_pov_color_map(OutFile, Pigment->color_map, indentation+1);
          break;

    case PAT_LEOPARD:
          fprintf(OutFile, "%s  leopard\n", s);
          save_pov_color_map(OutFile, Pigment->color_map, indentation+1);
          break;

    case PAT_LAST:
      break;
  }
/*  if (Pigment->Trans)
    SavePovTrans(OutFile, Pigment->Trans, indentation+1);*/
  fprintf(OutFile, "%s}\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovFinish
******************************************************************************/
static void SavePovFinish(FILE *OutFile, FinishStruct *finish, guint indentation)
{
  gchar *s;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%sfinish\n%s{\n", s, s);
  fprintf(OutFile, "%s  ambient <%g, %g, %g>\n", s,
          finish->ambient[0], finish->ambient[1], finish->ambient[2]);
  fprintf(OutFile, "%s  diffuse %g\n", s, finish->diffuse);
  if (V3DLength(finish->reflection)>0.001)
    fprintf(OutFile, "%s  reflection <%g, %g, %g>\n", s,
            finish->reflection[0], finish->reflection[1], finish->reflection[2]);
  if (finish->brilliance != 1.0)
    fprintf(OutFile, "%s  brilliance %g\n", s, finish->brilliance);
  if (finish->crand != 0.0)
    fprintf(OutFile, "%s  crand %g\n", s, finish->crand);
  if (finish->phong != 0.0)
  {
    fprintf(OutFile, "%s  phong %g", s, finish->phong);
    if (finish->phong_size != 40)
      fprintf(OutFile, " phong_size %g\n", finish->phong_size);
    else
      fprintf(OutFile, "\n");
  }
  if (finish->specular != 0.0)
  {
    fprintf(OutFile, "%s  specular %g", s, finish->specular);
    if (finish->roughness!=0.05)
      fprintf(OutFile, " roughness %g\n", finish->roughness);
    else
      fprintf(OutFile, "\n");
  }
  fprintf(OutFile, "%s}\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovTexture
******************************************************************************/
static void SavePovTexture(FILE *OutFile, TextureStruct *Texture, guint indentation)
{
  gchar *s;

  if (!Texture)
    return;
  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%stexture\n%s{\n", s, s);
  if (Texture->Pigment)
  {
    SavePovPigment(OutFile, Texture->Pigment, indentation+1);
  }
  if (Texture->finish)
  {
    SavePovFinish(OutFile, Texture->finish, indentation+1);
  }
/*  if (Texture->Trans)
    SavePovTrans(OutFile, Texture->Trans, indentation+1);*/
  fprintf(OutFile, "%s}\n", s);
  g_free(s);
}

/*************************************************************************
*  SavePovBicubicPatch
**************************************************************************/
static void SavePovBicubicPatch(FILE *OutFile, ObjectStruct *bicubic_patch, guint indentation)
{
  gchar              *s;
  BicubicPatchStruct *bbicubic_patch = (BicubicPatchStruct *)bicubic_patch;
  gint                i;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%sbicubic_patch\n%s{\n%s  type 1 flatness 0.0 u_steps %d v_steps %d",
          s, s, s, bbicubic_patch->u_steps, bbicubic_patch->v_steps);
  for (i=0 ; i<16 ; i++)
  {
    fprintf(OutFile, "%s<%g, %g, %g>", s, bbicubic_patch->control_points[i][0],
                                          bbicubic_patch->control_points[i][1],
                                          bbicubic_patch->control_points[i][2]);
    if (i!=15) fprintf(OutFile, ",");
    fprintf(OutFile, "\n");
  }
  if (bicubic_patch->Texture)
  {
    SavePovTexture(OutFile, bicubic_patch->Texture, indentation+1);
  }
  if (bicubic_patch->all_transforms)
    save_pov_transformation(OutFile, bicubic_patch->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovBox
******************************************************************************/
static void SavePovBox(FILE *OutFile, ObjectStruct *Box, guint indentation)
{
  gchar     *s;
  BoxStruct *BBox = (BoxStruct *)Box;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%sbox\n%s{\n%s  <%g, %g, %g>, <%g, %g, %g>\n", s, s, s,
                   BBox->MinCorner[0], BBox->MinCorner[1], BBox->MinCorner[2],
                   BBox->MaxCorner[0], BBox->MaxCorner[1], BBox->MaxCorner[2]);
  if (Box->Texture)
  {
    SavePovTexture(OutFile, Box->Texture, indentation+1);
  }
  if (Box->all_transforms)
    save_pov_transformation(OutFile, Box->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovCone
******************************************************************************/
static void SavePovCone(FILE *OutFile, ObjectStruct *Cone, guint indentation)
{
  gchar      *s;
  ConeStruct *CCone = (ConeStruct *)Cone;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%scone\n%s{\n%s  <0, %g, 0>, %g, <0, %g, 0>, %g\n",
          s, s, s, CCone->Radius1, CCone->Radius1,
                   CCone->Radius2, CCone->Radius2);
  if (CCone->Open)
    fprintf(OutFile, "%s  open\n",s);
  if (Cone->Texture)
  {
    SavePovTexture(OutFile, Cone->Texture, indentation+1);
  }
  if (Cone->all_transforms)
    save_pov_transformation(OutFile, Cone->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovCylinder
******************************************************************************/
static void SavePovCylinder(FILE *OutFile, ObjectStruct *Cylinder, guint indentation)
{
  gchar          *s;
  CylinderStruct *CCylinder = (CylinderStruct *)Cylinder;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%scylinder\n%s{\n%s  <%g, %g, %g>, <%g, %g, %g>, %g\n", s, s, s,
                   0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 1.0);
  if (CCylinder->Open)
    fprintf(OutFile, "%s  open\n",s);
  if (Cylinder->Texture)
  {
    SavePovTexture(OutFile, Cylinder->Texture, indentation+1);
  }
  if (Cylinder->all_transforms)
    save_pov_transformation(OutFile, Cylinder->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovDisc
******************************************************************************/
static void SavePovDisc(FILE *OutFile, ObjectStruct *Disc, guint indentation)
{
  gchar      *s;
  DiscStruct *DDisc = (DiscStruct *)Disc;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%sdisc\n%s{\n%s  <0, 0, 0>, <0, 1, 0>, %g",
          s, s, s, DDisc->Radius);
  if (DDisc->HoleRadius != 0.0)
    fprintf(OutFile, ", %g\n", DDisc->HoleRadius);
  else
    fprintf(OutFile, "\n");
  if (Disc->Texture)
  {
    SavePovTexture(OutFile, Disc->Texture, indentation+1);
  }
  if (Disc->all_transforms)
    save_pov_transformation(OutFile, Disc->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovHeightField
******************************************************************************/
static void SavePovHeightField(FILE *OutFile, ObjectStruct  *HeightField, guint indentation)
{
  gchar             *s;
  HeightFieldStruct *HHeightField = (HeightFieldStruct *)HeightField;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%sheight_field\n%s{\n", s, s);
  switch (HHeightField->type)
  {
    case IMAGE_TYPE_GIF:
      fprintf(OutFile, "%s  gif ",s);
      break;
    case IMAGE_TYPE_TGA:
      fprintf(OutFile, "%s  tga ",s);
      break;
    case IMAGE_TYPE_POT:
      fprintf(OutFile, "%s  pot ",s);
      break;
    case IMAGE_TYPE_PNG:
      fprintf(OutFile, "%s  png ",s);
      break;
    case IMAGE_TYPE_PGM:
      fprintf(OutFile, "%s  pgm ",s);
      break;
    case IMAGE_TYPE_PPM:
      fprintf(OutFile, "%s  ppm ",s);
      break;
    case IMAGE_TYPE_SYS:
      fprintf(OutFile, "%s  sys ",s);
      break;
  }
  fprintf(OutFile, "\"%s\"\n", HHeightField->filename);
  if (HHeightField->smooth)
    fprintf(OutFile, "%s  smooth\n",s);
  fprintf(OutFile, "%s  water_level %g\n", s, HHeightField->waterlevel);
  if (HeightField->Texture)
  {
    SavePovTexture(OutFile, HeightField->Texture, indentation+1);
  }
  if (HeightField->all_transforms)
    save_pov_transformation(OutFile, HeightField->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovMesh
******************************************************************************/
static void SavePovMesh(FILE *OutFile, ObjectStruct *Mesh, guint indentation)
{
  gchar              *s;
  MeshStruct         *MMesh = (MeshStruct *)Mesh;
  TriangleListStruct *TmpTri;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%smesh\n%s{\n", s, s);
  for (TmpTri = MMesh->FirstTriangle ; TmpTri ; TmpTri=TmpTri->Next)
  {
    fprintf(OutFile, "%s  smooth_triangle { <%g, %g, %g>, <%g, %g, %g>,\n", s,
            TmpTri->P1[0], TmpTri->P1[1], TmpTri->P1[2],
            TmpTri->N1[0], TmpTri->N1[1], TmpTri->N1[2]);
    fprintf(OutFile, "%s                  <%g, %g, %g>, <%g, %g, %g>,\n", s,
            TmpTri->P2[0], TmpTri->P2[1], TmpTri->P2[2],
            TmpTri->N2[0], TmpTri->N2[1], TmpTri->N2[2]);
    fprintf(OutFile, "%s                  <%g, %g, %g>, <%g, %g, %g> }\n", s,
            TmpTri->P3[0], TmpTri->P3[1], TmpTri->P3[2],
            TmpTri->N3[0], TmpTri->N3[1], TmpTri->N3[2]);

  }
  if (Mesh->Texture)
  {
    SavePovTexture(OutFile, Mesh->Texture, indentation+1);
  }
  if (Mesh->all_transforms)
    save_pov_transformation(OutFile, Mesh->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovPlane
******************************************************************************/
static void SavePovPlane(FILE *OutFile, ObjectStruct *Plane, guint indentation)
{
  gchar       *s;
  PlaneStruct *PPlane = (PlaneStruct *)Plane;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%splane\n%s{\n%s  <%g, %g, %g>, %g\n", s, s, s,
                   PPlane->Normal[0], PPlane->Normal[1], PPlane->Normal[2],
                   PPlane->Distance);
  if (Plane->Texture)
  {
    SavePovTexture(OutFile, Plane->Texture, indentation+1);
  }
  if (Plane->all_transforms)
    save_pov_transformation(OutFile, Plane->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovQuadric
******************************************************************************/
static void SavePovQuadric(FILE *OutFile, ObjectStruct *Quadric, guint indentation)
{
  gchar         *s;
  QuadricStruct *QQuadric = (QuadricStruct *)Quadric;

  s =  g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%squadric\n%s{\n%s  <%g, %g, %g>,\n"
                                   "%s  <%g, %g, %g>,\n"
                                   "%s  <%g, %g, %g>,\n"
                                   "%s  %g\n", s, s,
          s, QQuadric->A, QQuadric->B, QQuadric->C,
          s, QQuadric->D, QQuadric->E, QQuadric->F,
          s, QQuadric->G, QQuadric->H, QQuadric->I,
          s, QQuadric->J);
  if (Quadric->Texture)
  {
    SavePovTexture(OutFile, Quadric->Texture, indentation+1);
  }
  if (Quadric->all_transforms)
    save_pov_transformation(OutFile, Quadric->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovSor
******************************************************************************/
static void SavePovSor(FILE *OutFile, ObjectStruct *Sor, guint indentation)
{
  gint       i;
  gchar     *s;
  SorStruct *SSor = (SorStruct *)Sor;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%ssor\n%s{\n%s  %d,\n", s, s, s, SSor->NumberOfPoints);
  for (i=0 ; i<SSor->NumberOfPoints-1 ; i++)
    fprintf(OutFile, "%s  <%g, %g>,\n", s, SSor->Point[i][0], SSor->Point[i][1]);
  fprintf(OutFile, "%s  <%g, %g>\n", s, SSor->Point[i][0], SSor->Point[i][1]);
  if (Sor->Texture)
  {
    SavePovTexture(OutFile, Sor->Texture, indentation+1);
  }
  if (Sor->all_transforms)
    save_pov_transformation(OutFile, Sor->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovSphere
******************************************************************************/
static void SavePovSphere(FILE *OutFile, ObjectStruct *Sphere, guint indentation)
{
  gchar        *s;
  SphereStruct *SSphere = (SphereStruct *)Sphere;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%ssphere\n%s{\n%s  <%g, %g, %g>, %g\n", s, s, s,
                   SSphere->Center[0], SSphere->Center[1], SSphere->Center[2],
                   SSphere->Radius);
  if (Sphere->Texture)
  {
    SavePovTexture(OutFile, Sphere->Texture, indentation+1);
  }
  if (Sphere->all_transforms)
    save_pov_transformation(OutFile, Sphere->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovSuperEllipsoid
******************************************************************************/
static void SavePovSuperEllipsoid(FILE *OutFile, ObjectStruct *SuperEllipsoid, guint indentation)
{
  gchar                *s;
  SuperEllipsoidStruct *SSuperEllipsoid = (SuperEllipsoidStruct *)SuperEllipsoid;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%ssuperellipsoid\n%s{\n%s  <%g, %g>\n", s, s, s,
          SSuperEllipsoid->E, SSuperEllipsoid->N);
  if (SuperEllipsoid->Texture)
  {
    SavePovTexture(OutFile, SuperEllipsoid->Texture, indentation+1);
  }
  if (SuperEllipsoid->all_transforms)
    save_pov_transformation(OutFile, SuperEllipsoid->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovTorus
******************************************************************************/
static void SavePovTorus(FILE *OutFile, ObjectStruct *Torus, guint indentation)
{
  gchar       *s;
  TorusStruct *TTorus = (TorusStruct *)Torus;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%storus\n%s{\n%s  %g, %g\n", s, s, s,
                   TTorus->Major, TTorus->Minor);
  if (Torus->Texture)
  {
    SavePovTexture(OutFile, Torus->Texture, indentation+1);
  }
  if (Torus->all_transforms)
    save_pov_transformation(OutFile, Torus->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovTriangle
******************************************************************************/
static void SavePovTriangle(FILE *OutFile, ObjectStruct *Triangle, guint indentation)
{
  gchar          *s;
  TriangleStruct *TTriangle = (TriangleStruct *)Triangle;

  s = g_strnfill(indentation*2, ' ');
  fprintf(OutFile, "%ssmooth_triangle\n%s{\n", s, s);
  fprintf(OutFile, "%s  <%g, %g, %g>, <%g, %g, %g>,\n", s,
          TTriangle->P1[0], TTriangle->P1[1], TTriangle->P1[2],
          TTriangle->N1[0], TTriangle->N1[1], TTriangle->N1[2]);
  fprintf(OutFile, "%s  <%g, %g, %g>, <%g, %g, %g>,\n", s,
          TTriangle->P2[0], TTriangle->P2[1], TTriangle->P2[2],
          TTriangle->N2[0], TTriangle->N2[1], TTriangle->N2[2]);
  fprintf(OutFile, "%s  <%g, %g, %g>, <%g, %g, %g>\n", s,
          TTriangle->P3[0], TTriangle->P3[1], TTriangle->P3[2],
          TTriangle->N3[0], TTriangle->N3[1], TTriangle->N3[2]);
  if (Triangle->Texture)
  {
    SavePovTexture(OutFile, Triangle->Texture, indentation+1);
  }
  if (Triangle->all_transforms)
    save_pov_transformation(OutFile, Triangle->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovCSG
******************************************************************************/
static void SavePovCSG(FILE *OutFile, ObjectStruct *CSG, guint indentation)
{
  gchar        *s;
  ObjectStruct *TmpObject;
  CSGStruct    *CCSG = (CSGStruct *)CSG;
  GSList       *tmp_list;

  s = g_strnfill(indentation*2, ' ');
  switch (CCSG->Type)
  {
    case CSG_DIFFERENCE:
      fprintf(OutFile, "%sdifference\n%s{\n", s, s);
      break;

    case CSG_INTERSECTION:
      fprintf(OutFile, "%sintersection\n%s{\n", s, s);
      break;

    case CSG_MERGE:
      fprintf(OutFile, "%smerge\n%s{\n", s, s);
      break;

    case CSG_UNION:
      fprintf(OutFile, "%sunion\n%s{\n", s, s);
      break;
  }
  for (tmp_list = CCSG->all_objects ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list))
  {
    TmpObject = (ObjectStruct*)(tmp_list->data);
    switch (TmpObject->Type)
    {
      case BICUBIC_PATCH_OBJECT:
        SavePovBicubicPatch(OutFile, TmpObject, indentation+1);
        break;

      case BOX_OBJECT:
        SavePovBox(OutFile, TmpObject, indentation+1);
        break;

      case CONE_OBJECT:
        SavePovCone(OutFile, TmpObject, indentation+1);
        break;

      case CSG_OBJECT:
        SavePovCSG(OutFile, TmpObject, indentation+1);
        break;

      case CYLINDER_OBJECT:
        SavePovCylinder(OutFile, TmpObject, indentation+1);
        break;

      case DISC_OBJECT:
        SavePovDisc(OutFile, TmpObject, indentation+1);
        break;

      case HEIGHT_FIELD_OBJECT:
        SavePovHeightField(OutFile, TmpObject,  indentation+1);
        break;

      case MESH_OBJECT:
        SavePovMesh(OutFile, TmpObject, indentation+1);
        break;

      case PLANE_OBJECT:
        SavePovPlane(OutFile, TmpObject, indentation+1);
        break;

      case QUADRIC_OBJECT:
        SavePovQuadric(OutFile, TmpObject, indentation+1);
        break;

      case SOR_OBJECT:
        SavePovSor(OutFile, TmpObject, indentation+1);
        break;

      case SPHERE_OBJECT:
        SavePovSphere(OutFile, TmpObject, indentation+1);
        break;

      case SUPERELLIPSOID_OBJECT:
        SavePovSuperEllipsoid(OutFile, TmpObject, indentation+1);
        break;

      case TORUS_OBJECT:
        SavePovTorus(OutFile, TmpObject, indentation+1);
        break;

      case TRIANGLE_OBJECT:
        SavePovTriangle(OutFile, TmpObject, indentation+1);
        break;
    }
  }
  if (CSG->Texture)
  {
    SavePovTexture(OutFile, CSG->Texture, indentation+1);
  }
  if (CSG->all_transforms)
    save_pov_transformation(OutFile, CSG->all_transforms, indentation+1);
  fprintf(OutFile, "%s}\n\n", s);
  g_free(s);
}

/*****************************************************************************
*  SavePovLightSource
******************************************************************************/
static void SavePovLightSource(FILE *OutFile, GSList *all_light_sources)
{ /* FIXME: far from complete... */
  GSList            *tmp_list;
  LightSourceStruct *TmpLS;

  for (tmp_list = all_light_sources ;
       tmp_list ;
       tmp_list = g_slist_next(tmp_list) )
  {
    TmpLS = tmp_list->data;
    fprintf(OutFile, "light_source\n{\n");
    fprintf(OutFile, "  <%g, %g, %g>\n", TmpLS->Location[0],
                                         TmpLS->Location[1],
                                         TmpLS->Location[2]);
    fprintf(OutFile, "  color rgb <%g, %g, %g>\n", TmpLS->Color[0],
                                               TmpLS->Color[1],
                                               TmpLS->Color[2]);
    fprintf(OutFile, "}\n");
  }
  fprintf(OutFile, "\n");
}

/*****************************************************************************
*  SavePovCamera
******************************************************************************/
static void SavePovCamera(FILE *OutFile, CameraStruct *Camera)
{

  if (Camera == NULL)
    return;
  fprintf(OutFile, "camera\n{\n");
  switch (Camera->Type)
  {
    case CAMERA_PERSPECTIVE:
      fprintf(OutFile, "  perspective\n");
      break;

    case CAMERA_ORTHOGRAPHIC:
      fprintf(OutFile, "  orthographic\n");
      break;

    case CAMERA_FISHEYE:
      fprintf(OutFile, "  fisheye\n");
      break;

    case CAMERA_ULTRA_WIDE_ANGLE:
      fprintf(OutFile, "  ultra_wide_angle\n");
      break;

    case CAMERA_OMNIMAX:
      fprintf(OutFile, "  omnimax\n");
      break;

    case CAMERA_PANORAMIC:
      fprintf(OutFile, "  panoramic\n");
      break;

    case CAMERA_CYLINDER_1:
      fprintf(OutFile, "  cylinder 1\n");
      break;

    case CAMERA_CYLINDER_2:
      fprintf(OutFile, "  cylinder 2\n");
      break;

    case CAMERA_CYLINDER_3:
      fprintf(OutFile, "  cylinder 3\n");
      break;

    case CAMERA_CYLINDER_4:
      fprintf(OutFile, "  cylinder 4\n");
      break;

    default:
      /* panic */
  }
  fprintf(OutFile, "  location <%g, %g, %g>\n", Camera->Location[0],
                                                Camera->Location[1],
                                                Camera->Location[2]);
  fprintf(OutFile, "  right <%.12g, %.12g, %.12g>\n", Camera->Right[0],
                                             Camera->Right[1],
                                             Camera->Right[2]);
  fprintf(OutFile, "  up <%.12g, %.12g, %.12g>\n", Camera->Up[0],
                                          Camera->Up[1],
                                          Camera->Up[2]);
  fprintf(OutFile, "  direction <%.12g, %.12g, %.12g>\n", Camera->Direction[0],
                                                 Camera->Direction[1],
                                                 Camera->Direction[2]);
  fprintf(OutFile, "  sky <%g, %g, %g>\n", Camera->Sky[0],
                                           Camera->Sky[1],
                                           Camera->Sky[2]);
  if (Camera->BlurSamples)
  {
    fprintf(OutFile, "  blur_samples %d\n", Camera->BlurSamples);
    fprintf(OutFile, "  aperture %g\n", Camera->Aperture);
    fprintf(OutFile, "  focal_point <%g, %g, %g>\n", Camera->FocalPoint[0],
                                                     Camera->FocalPoint[1],
                                                     Camera->FocalPoint[2]);
    fprintf(OutFile, "  confidence %g\n", Camera->Confidence);
    fprintf(OutFile, "  variance %g\n", Camera->Variance);
  }
  fprintf(OutFile, "}\n\n");
}

/*****************************************************************************
*  SavePovByName
******************************************************************************/
void SavePovByName(FrameStruct *frame, char *FileName)
{
  GSList       *tmp_list;
  ObjectStruct *TmpObject;
  FILE         *OutFile;

  OutFile = fopen(FileName, "w");

  fprintf(OutFile, _("// Persistence of Vision 3.0 File.\n"));
  fprintf(OutFile, _("// This file has been generated by giram.\n"));
  fprintf(OutFile, _("// If you found any error, please contact David@dindinx.org\n\n"));
  fprintf(OutFile, _("// The scene global parameters:\n"));
  fprintf(OutFile, "background { rgb <%g, %g, %g> }\n\n",
          frame->background_color[0],
          frame->background_color[1],
          frame->background_color[2]);
  fprintf(OutFile, _("// The Camera:\n"));
  SavePovCamera(OutFile, frame->all_cameras->data);
  fprintf(OutFile, _("// The Scene Light Sources:\n"));
  SavePovLightSource(OutFile, frame->all_light_sources);
  fprintf(OutFile, _("// The Scene Objects:\n"));
  for (tmp_list = frame->all_objects ; tmp_list ; tmp_list = g_slist_next(tmp_list))
  {
    TmpObject = tmp_list->data;
    switch (TmpObject->Type)
    {
      case BICUBIC_PATCH_OBJECT:
        SavePovBicubicPatch(OutFile, TmpObject, 0);
        break;

      case BOX_OBJECT:
        SavePovBox(OutFile, TmpObject, 0);
        break;

      case CONE_OBJECT:
        SavePovCone(OutFile, TmpObject, 0);
        break;

      case CSG_OBJECT:
        SavePovCSG(OutFile, TmpObject, 0);
        break;

      case CYLINDER_OBJECT:
        SavePovCylinder(OutFile, TmpObject, 0);
        break;

      case DISC_OBJECT:
        SavePovDisc(OutFile, TmpObject, 0);
        break;

      case HEIGHT_FIELD_OBJECT:
        SavePovHeightField(OutFile, TmpObject, 0);
        break;

      case MESH_OBJECT:
        SavePovMesh(OutFile, TmpObject, 0);
        break;

      case PLANE_OBJECT:
        SavePovPlane(OutFile, TmpObject, 0);
        break;

      case QUADRIC_OBJECT:
        SavePovQuadric(OutFile, TmpObject, 0);
        break;

      case SOR_OBJECT:
        SavePovSor(OutFile, TmpObject, 0);
        break;

      case SPHERE_OBJECT:
        SavePovSphere(OutFile, TmpObject, 0);
        break;

      case SUPERELLIPSOID_OBJECT:
        SavePovSuperEllipsoid(OutFile, TmpObject, 0);
        break;

      case TORUS_OBJECT:
        SavePovTorus(OutFile, TmpObject, 0);
        break;

      case TRIANGLE_OBJECT:
        SavePovTriangle(OutFile, TmpObject, 0);
        break;
    }
  }
  fclose(OutFile);
}

/*****************************************************************************
*  SavePov
******************************************************************************/
void SavePov(GtkWidget *w, GtkFileSelection *FS)
{
  FrameStruct     *LocalFrame;
  GSList          *tmp_list;
  ViewStruct      *tmp_view;
  char            *FileName;

  FileName = (gchar *)gtk_file_selection_get_filename(GTK_FILE_SELECTION(FS));
  LocalFrame = g_object_get_data(G_OBJECT(FS), "frame");
  SavePovByName(LocalFrame, FileName);

  if (LocalFrame->file_name)
    g_free(LocalFrame->file_name);
  LocalFrame->file_name = g_strdup(FileName);
  /* update the title of every windows */
  for (tmp_list = LocalFrame->all_views ; tmp_list ; tmp_list = tmp_list->next)
  {
    tmp_view = tmp_list->data;
    giram_view_set_title(tmp_view);
  }
}

