/* dxf_write.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2001 DindinX <David@dindinx.org>
 * Copyright (C) 1999-2001 Noah Davis of VIEWS Net, Inc. <noah@viewsnet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * This file is from Noah Davis.
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include "giram.h"
#include "object.h"
#include "csgtree.h"
#include "camera.h"
#include "frame.h"
#include "primitives/cylinder.h"

#include "dxf.h"

#include "widgets/giramwidgets.h"

#include "giramintl.h"

/*****************************************************************************
*  dxf_save_file
*  Dumps out Giram triangle data in an ASCII (for now) DXF file
******************************************************************************/
int dxf_save_file(FrameStruct *frame, char *fname, int binary, int SelectedOnly,
                  char *Comment)
{
  FILE         *fp;
  GSList       *tmp_list;
  ObjectStruct *iterObject;
  GList        *selection;

  if (binary)
  {
    fprintf(stderr, "DXF_SAVE: Binary writes not implemented yet.\n");
    return 1;
  }

  if (SelectedOnly)
  {
    if (frame->selection == NULL)
    {
      fprintf(stderr, "DXF_SAVE: Nothing to write!\n");
      return 1;
    }
  } else
  {
    if (frame->all_objects == NULL)
    {
      fprintf(stderr, "DXF_SAVE: Nothing to write!\n");
      return 1;
    }
  }
  fp = fopen(fname,"w");

  if (fp == NULL)
  {
    fprintf(stderr, "DXF_SAVE: Couldn't open %s for writing.\n", fname);
    return 1;
  }

  /* Okay, we have an open file, and at least one thing to write to it.
   * Lets start dumping...
   */


  /* First, the header, eschewing all but the ENTITIES section */
  if (!dxf_write_header(fp, binary, Comment))
  {
    fclose(fp);
    return 1;
  }

  if (SelectedOnly)
  {
    for (selection = frame->selection ;
         selection ;
         selection = g_list_next(selection))
    {
      iterObject = selection->data;
      if (!dxf_write_object(fp, iterObject))
      {
        fclose(fp);
        return 1;
      }
    }
  } else
  {
    for (tmp_list = frame->all_objects ;
         tmp_list ;
         tmp_list = g_slist_next(tmp_list))
    {
      iterObject = tmp_list->data;
      if (!dxf_write_object(fp, iterObject))
      {
        fclose(fp);
        return 1;
      }
    }
  }

  if (!dxf_write_end(fp, binary))
  {
    fclose(fp);
    return 1;
  }

  fclose (fp);
  return 0;
}

/*****************************************************************************
*  dxf_write_header
******************************************************************************/
int dxf_write_header(FILE *fp, int bin, char *Comment)
{
  int strpos = 0;
  char CommentLine[256];
#if 0
  if (bin) ; /* Nothing for now. */
#endif
  /* The blank HEADER section */
  fprintf(fp, "0\nSECTION\n2\nHEADER\n0\nENDSEC\n");

  /* Comments out, 256 Bytes per line */
  if (Comment)
  {
    while (Comment[strpos])
    {
      int i;
      for (i=0;
           (Comment[strpos]) && (Comment[strpos]!='\n') && (i < 255) ;
           i++, strpos++)
      {
        CommentLine[i] = Comment[strpos];
      }
      if (i<255) strpos++;
      CommentLine[i] = 0;
      fprintf(fp, "999\n%s\n", CommentLine);
    }
  }

  /* The blank HEADER section */
  fprintf(fp, "0\nSECTION\n2\nHEADER\n0\nENDSEC\n");

  /* Start the ENTITIES section */
  fprintf(fp, "0\nSECTION\n2\nENTITIES\n");

  return TRUE;
}

/*****************************************************************************
*  dxf_write_object
******************************************************************************/
int dxf_write_object(FILE *fp, ObjectStruct *obj)
{
  TriangleListStruct *tri;

  if ((tri=obj->FirstTriangle) == NULL)
    return TRUE;   /* Nothing to write */

  for ( ; tri ; tri=tri->Next)
  {
    /* The entity.. always a 3dface for now */
    fprintf(fp, "0\n3DFACE\n");

    /* The layer, blank until Giram supports it */
    fprintf(fp, "8\nGiramLayer\n");

    fprintf(fp, "10\n%f\n", tri->P1[0]);
    fprintf(fp, "20\n%f\n", tri->P1[1]);
    fprintf(fp, "30\n%f\n", tri->P1[2]);

    fprintf(fp, "11\n%f\n", tri->P2[0]);
    fprintf(fp, "21\n%f\n", tri->P2[1]);
    fprintf(fp, "31\n%f\n", tri->P2[2]);

    fprintf(fp, "12\n%f\n", tri->P3[0]);
    fprintf(fp, "22\n%f\n", tri->P3[1]);
    fprintf(fp, "32\n%f\n", tri->P3[2]);

    fprintf(fp, "13\n%f\n", tri->P3[0]);
    fprintf(fp, "23\n%f\n", tri->P3[1]);
    fprintf(fp, "33\n%f\n", tri->P3[2]);
  }
  return TRUE;
}

/*****************************************************************************
*  dxf_write_end
******************************************************************************/
int dxf_write_end(FILE *fp, int bin)
{
#if 0
  if (bin) ; /* GNDN */
#endif
  fprintf(fp, "0\nENDSEC\n0\nEOF\n");

  return TRUE;
}

/*****************************************************************************
*  SaveDXFOK
******************************************************************************/
static void SaveDXFOK(GtkWidget *Dialog)
{
  FrameStruct *frame;
  gchar       *file_name;
  gboolean     selected_only;
  GtkWidget   *text;
  gboolean     include_comment;

  frame     = g_object_get_data(G_OBJECT(Dialog), "frame");
  file_name = g_object_get_data(G_OBJECT(Dialog), "file_name");
  selected_only = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(Dialog),
                                  "selected_only"));
  text = g_object_get_data(G_OBJECT(Dialog), "text");
  include_comment = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(Dialog),
                                    "include_comment"));

  if (include_comment)
    dxf_save_file(frame, file_name, 0, selected_only,
                  gtk_editable_get_chars(GTK_EDITABLE(text),0,-1) );
  else
    dxf_save_file(frame, file_name, 0, selected_only, NULL);

  gtk_widget_destroy(Dialog);
  g_free(file_name);
}

/*****************************************************************************
*  SaveDXFCancel
******************************************************************************/
static void SaveDXFCancel(GtkWidget *dialog)
{
  gchar *file_name;

  file_name = g_object_get_data(G_OBJECT(dialog), "file_name");
  g_free(file_name);
  gtk_widget_destroy(dialog);
}

/*****************************************************************************
*  ToggleDXFIncludeComment
******************************************************************************/
static void ToggleDXFIncludeComment(GObject *Dialog)
{
  gboolean include_comment;

  include_comment = GPOINTER_TO_INT(g_object_get_data(Dialog,
                                                        "include_comment"));
  include_comment = !include_comment;
  g_object_set_data(Dialog, "include_comment",
                      GINT_TO_POINTER(include_comment));
}

/*****************************************************************************
*  SaveDXFDialog
******************************************************************************/
void SaveDXFDialog(FrameStruct *Frame, gchar *FileName, gboolean SelectedOnly)
{
  GtkWidget *Dialog, *CheckButton, *frame, *Table/*, *Text, *Vscrollbar*/;

/*  gchar *Comment = _("This file was Generated by Giram.\n"
                     "If you experience any error, you can reach the\n"
                     "author of the Giram's DXF exporter at <adoyle@rimmer.viewsnet.com>\n");*/
  /* The Dialog Box for the Optionnal Comment */
  Dialog = giram_dialog_new(_("Save as DXF"), "giram_dialog",
                            NULL /* FIXME */,
                            "dialogs/save_as_dxf.html",
                            GTK_WIN_POS_MOUSE,
                            FALSE, TRUE, FALSE,
                            _("OK"), SaveDXFOK,
                            NULL, 1, NULL, TRUE, FALSE,
                            _("Cancel"), SaveDXFCancel,
                            NULL, 1, NULL, FALSE, TRUE,
                            NULL);
   g_object_set_data(G_OBJECT(Dialog), "frame", Frame);
   g_object_set_data(G_OBJECT(Dialog), "file_name",
                     g_strdup(FileName));
   g_object_set_data(G_OBJECT(Dialog), "selected_only",
                     GINT_TO_POINTER(SelectedOnly));

  /* The check button for enabling/disabling comment */
  CheckButton = gtk_check_button_new_with_label(_("Save with comment"));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Dialog)->vbox), CheckButton, FALSE, FALSE, 2);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CheckButton), TRUE);
  g_object_set_data(G_OBJECT(Dialog), "include_comment",
                    GINT_TO_POINTER(TRUE));
  g_signal_connect_swapped(G_OBJECT(CheckButton), "toggled",
                           G_CALLBACK(ToggleDXFIncludeComment), Dialog);

  frame = gtk_frame_new(_("Comments:"));
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_set_border_width(GTK_CONTAINER(frame), 10);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Dialog)->vbox), frame, TRUE, TRUE, 0);

  Table = gtk_table_new(1, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(Table), 10);
  gtk_container_add(GTK_CONTAINER(frame), Table);

#if 0
  Text = gtk_text_new(NULL, NULL);
  gtk_text_set_editable(GTK_TEXT(Text), TRUE);
  gtk_text_insert(GTK_TEXT(Text), NULL, NULL, NULL, Comment, -1);
  gtk_table_attach(GTK_TABLE(Table), Text, 0, 1, 0, 1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_object_set_data(GTK_OBJECT(Dialog), "text", Text);

  /* Add a vertical scrollbar to the GtkText widget */
  Vscrollbar = gtk_vscrollbar_new(GTK_TEXT(Text)->vadj);
  gtk_table_attach(GTK_TABLE(Table), Vscrollbar, 1, 2, 0, 1,
                   GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
#endif
  gtk_widget_show_all(Dialog);
}

/*****************************************************************************
*  SaveDXF
******************************************************************************/
void SaveDXF(GtkWidget *w, GtkFileSelection *FS)
{
  FrameStruct *LocalFrame;
  gchar *FileName;

  FileName = (gchar *)gtk_file_selection_get_filename(GTK_FILE_SELECTION(FS));
  LocalFrame = g_object_get_data(G_OBJECT(FS), "frame");
  SaveDXFDialog(LocalFrame, FileName, FALSE);
}

/*****************************************************************************
*  SaveSelectedDXF
******************************************************************************/
void SaveSelectedDXF(GtkWidget *w, GtkFileSelection *FS)
{
  FrameStruct *LocalFrame;
  gchar *FileName;

  FileName = (gchar *)gtk_file_selection_get_filename(GTK_FILE_SELECTION(FS));
  LocalFrame = g_object_get_data(G_OBJECT(FS), "frame");
  SaveDXFDialog(LocalFrame, FileName, TRUE);
}

