/* dxf_entity.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2001 DindinX <David@dindinx.org>
 * Copyright (C) 1999-2001 Noah Davis of VIEWS Net, Inc. <noah@viewsnet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * This file is from Noah Davis.
 */

/*
 * Parse and implement the important ENTITIES section
 * Noah Davis
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef DXF_STANDALONE
#include "giram.h"
#endif

#include "dxf.h"

int dxf_parse_entity(FILE *fp, DXF_Data *dat)
{
  int err;
  char value[DXF_LINE_MAX];
  int code;

#ifdef DXF_DEBUG
   fprintf(stderr, "Parsing ENTITY section...\n");
#endif

  /* Parse the different entities */
  while ((err = dxf_getpair(fp, &code, value, dat)))
  {
    dat->lineno++;

    switch (code)
	  {
      case DXF_CODE_ENTITY:
        if (strstr(value, "ENDSEC"))
          return (TRUE);
        else if (strstr(value, "POLYLINE"))
          dxf_add_polyline(dxf_parse_polyline(fp, dat), dat);
        else if (strstr(value, "LINE"))
          dxf_add_line(dxf_parse_line(fp, dat), dat);
        else if (strstr(value, "3DFACE"))
          dxf_add_3dface(dxf_parse_3dface(fp, dat), dat);
        else if (strstr(value, "VERTEX"))
          dxf_add_vertex(dxf_parse_vertex(fp, dat), dat);
        else if (strstr(value, "ARC"))
          dxf_add_arc(dxf_parse_arc(fp, dat), dat);
        else if (strstr(value, "CIRCLE"))
          dxf_add_circle(dxf_parse_circle(fp, dat), dat);
        else if (strstr(value, "SOLID"))
          dxf_add_solid(dxf_parse_solid(fp, dat), dat);
        else if (strstr(value, "POINT"))
          dxf_add_point(dxf_parse_point(fp, dat), dat);
#ifdef DXF_DEBUG
        else
        {
          cr_strip(value);
          fprintf(stderr,"Unrecognized entity: %s\n",value);
        }
#endif
        break;
    }
    dat->lineno++;
  }
  if (err)
    return (TRUE);
  else
    return (FALSE);
}

/* A polyline. This is a PITA. AC seems to use polylines as some sort of
 * mutant workhorse.
 */
DXF_Polyline *dxf_parse_polyline(FILE *fp, DXF_Data *dat)
{
  int err = 0;
  char value[DXF_LINE_MAX];
  int code;
  int good = TRUE;
  DXF_Polyline *pline;

  /* Malloc() one */
  pline = malloc(sizeof(DXF_Polyline));
  if (pline == NULL)        /* Couldn't allocate, out of memory! */
    return (NULL);

  bzero(pline,sizeof(DXF_Polyline));

  while (good == TRUE)
  {
    err = dxf_getpair(fp, &code, value, dat);
    if (!err)
      break;

    dat->lineno++;

    switch (code)
    {
      case DXF_CODE_ENTITY:
        if (strstr(value, "SEQEND"))
        {
          good = FALSE;
          break;
        }
        if (strstr(value, "VERTEX")) /* Found a vertex for this pline */
	 {
	    dxf_add_vertex_to_polyline(dxf_parse_vertex(fp, dat), pline);
	    break;
	 }
#ifdef DXF_DEBUG
        if (strstr(value, "FACE"))
          fprintf(stderr, "FOUND Pline FACE\n");
#endif
       else fprintf (stderr,"Fould polyline entity: %s\n",value);
        break;
      case 8:        /* Layer */
        cr_strip(value);
        pline->layer = dxf_add_layer(value, dat);
        break;
      case 30:       /* Elevation */
        pline->elevation = strtod(value, NULL);
        break;
      case 39:       /* Thickness */
        pline->thickness = strtod(value, NULL);
        break;
      case 70:       /* Polyline flags */
        pline->flags = atoi(value);
        break;
      case 71:       /* M count */
        pline->mcount = atoi(value);
        break;
      case 72:       /* N Count */
        pline->ncount = atoi(value);
        break;
      case 73:       /* M Density */
        pline->mdensity = atoi(value);
        break;
      case 74:       /* N Density */
        pline->ndensity = atoi(value);
        break;
      case 210:      /* Extrusion vector X */
        pline->extx = strtod(value, NULL);
        break;
      case 220:      /* Extrusion vector Y */
        pline->exty=strtod(value,NULL);
        break;
      case 230:      /* Extrusion vector Z */
        pline->extz=strtod(value,NULL);
        break;
    }
    dat->lineno++;
  }
  if (!err)
  {
    free(pline);
    return (NULL);
  }
  return (pline);
}

/* A  line.
 */
DXF_Line *dxf_parse_line(FILE *fp, DXF_Data *dat)
{
  int err = 0;
  char value[DXF_LINE_MAX];
  int code;
  DXF_Line *dline;
  int good = TRUE;

  /* Malloc() one */
  dline = malloc(sizeof(DXF_Line));
  if (dline == NULL)
    return NULL;

  bzero (dline,sizeof(DXF_Line));

  while (good)
  {
    err = dxf_getpair(fp, &code, value, dat);
    if (!err) break;
    dat->lineno++;

    switch (code)
    {
      case 0:    /* Oops, start of new entity */
        good = FALSE;
        /* Now to undo the last read... oh boy */
        dxf_getpair(fp, NULL, NULL, dat);
        /* That's right, NULL's... force a rewind in getpair() */
        break;
      case 8:    /* Layer */
        cr_strip(value);
        dline->layer = dxf_add_layer(value, dat);
        break;
      case 39:
        dline->thickness = strtod(value, NULL);
        break;
      case 10:
        dline->x1 = strtod(value, NULL);
        break;
      case 20:
        dline->y1 = strtod(value, NULL);
        break;
      case 30:
        dline->z1 = strtod(value, NULL);
        break;
      case 11:
        dline->x2 = strtod(value, NULL);
        break;
      case 21:
        dline->y2 = strtod(value, NULL);
        break;
      case 31:
        dline->z2 = strtod(value, NULL);
        break;
      case 210:  /* Extrusion */
        dline->extx = strtod(value, NULL);
        break;
      case 220:
        dline->exty = strtod(value, NULL);
        break;
      case 230:
        dline->extz = strtod(value, NULL);
        break;
      default:
        break;
    }
    dat->lineno++;
  }

  if (!err) /* Couldn't parse it! */
  {
    free(dline);
    return NULL;
  }
  /* Check for degenerate */
  if ((dline->x1==dline->x2) && (dline->y1==dline->y2) &&
      (dline->z1==dline->z2))
  {
    dxf_err("Degenerate line.", dat);
    free(dline);
    return(NULL);
  }

  /* Update the bounding box */
  dxf_dobox(dline->x1, dline->y1, dline->z1, dat);
  dxf_dobox(dline->x2, dline->y2, dline->z2, dat);

  return dline;
}

DXF_Vertex *dxf_parse_vertex(FILE *fp, DXF_Data *dat)
{
  int err = 0;
  char value[DXF_LINE_MAX];
  int code;
  DXF_Vertex *vert;
  int good = TRUE;

  /* Malloc() one */
  vert = malloc(sizeof(DXF_Vertex));
  if (vert == NULL)        /* Couldn't allocate, out of memory! */
    return NULL;

  bzero (vert,sizeof(DXF_Vertex));

  while (good)
  {
    err = dxf_getpair(fp, &code, value, dat);
    if (!err) break;

    dat->lineno++;

    switch (code)
    {
      case 0: /* Oops, start of new entity */
        good = FALSE;
        /* Now to undo the last read... oh boy */
        dxf_getpair(fp, NULL, NULL, dat);
        /* That's right, NULL's... force a rewind in getpair() */
        break;
      case 8:        /* Layer */
        cr_strip(value);
        vert->layer = dxf_add_layer(value, dat);
        break;
      case 10:
        vert->x = strtod(value, NULL);
        break;
      case 20:
        vert->y = strtod(value, NULL);
        break;
      case 30:
        vert->z = strtod(value, NULL);
        break;
      case 70: /* Flags */
        vert->flags = atoi(value);
        break;
      case 71: /* Polyface mesh vertex index */
        vert->vert_index1=atoi(value);
        break;
      case 72: /* Polyface mesh vertex index */
        vert->vert_index2=atoi(value);
        break;
      case 73: /* Polyface mesh vertex index */
        vert->vert_index3=atoi(value);
        break;
      case 74: /* Polyface mesh vertex index */
        vert->vert_index4=atoi(value);
        break;
      default:
        break;
    }
    dat->lineno++;
  }

  if (!err) /* Couldn't parse it! */
  {
    free(vert);
    return NULL;
  }

  /* Update the bounding box */
  dxf_dobox(vert->x, vert->y, vert->z,dat);

  return vert;
}

/* Parse a 3d-face... you get the idea by now
 */
DXF_3DFace *dxf_parse_3dface(FILE *fp, DXF_Data *dat)
{
  int err = 0;
  char value[DXF_LINE_MAX];
  int code;
  DXF_3DFace *face;
  int good = TRUE;

  /* Malloc() one */
  face = malloc(sizeof(DXF_3DFace));
  if (face == NULL)        /* Couldn't allocate, out of memory! */
    return NULL;

  bzero (face,sizeof(DXF_3DFace));

  while (good)
  {
    err = dxf_getpair(fp, &code, value, dat);
    if (!err) break;

    dat->lineno++;

    switch (code)
    {
      case 0: /* Oops, start of new entity */
        good = FALSE;
        /* Now to undo the last read... oh boy */
        dxf_getpair(fp, NULL, NULL, dat);
        /* That's right, NULL's... force a rewind in getpair() */
        break;
      case 8:        /* Layer */
        cr_strip(value);
        face->layer = dxf_add_layer(value, dat);
        break;
      case 10:   /* X point 1 */
        face->x1 = strtod(value, NULL);
        break;
      case 11:
        face->x2 = strtod(value, NULL);
        break;
      case 12:
        face->x3 = strtod(value, NULL);
        break;
      case 13:
        face->x4 = strtod(value, NULL);
        break;
      case 20:
        face->y1 = strtod(value, NULL);
        break;
      case 21:
        face->y2 = strtod(value, NULL);
        break;
      case 22:
        face->y3 = strtod(value, NULL);
        break;
      case 23:
        face->y4 = strtod(value, NULL);
        break;
      case 30:
        face->z1 = strtod(value, NULL);
        break;
      case 31:
        face->z2 = strtod(value, NULL);
        break;
      case 32:
        face->z3 = strtod(value, NULL);
        break;
      case 33:
        face->z4=strtod(value,NULL);
        break;
    }
    dat->lineno++;
  }

  if (!err) /* Couldn't parse it! */
  {
    free(face);
    return (NULL);
  }

  /* Update the bounding box */
  dxf_dobox(face->x1,face->y1,face->z1,dat);
  dxf_dobox(face->x2,face->y2,face->z2,dat);
  dxf_dobox(face->x3,face->y3,face->z3,dat);
  dxf_dobox(face->x4,face->y4,face->z4,dat);

  return (face);
}

/* Parse an ARC
 */
DXF_Arc *dxf_parse_arc(FILE *fp, DXF_Data *dat)
{
  int err = 0;
  char value[DXF_LINE_MAX];
  int code;
  DXF_Arc *arc;
  int good = TRUE;

  /* Malloc() one */
  arc = malloc(sizeof(DXF_Arc));
  if (arc==NULL)        /* Couldn't allocate, out of memory! */
    return NULL;

  bzero (arc,sizeof(DXF_Arc));

  while (good)
  {
    err = dxf_getpair(fp, &code, value, dat);
    if (!err) break;

    dat->lineno++;

    switch (code)
    {
      case 0: /* Oops, start of new entity */
        good = FALSE;
        /* Now to undo the last read... oh boy */
        dxf_getpair(fp, NULL, NULL, dat);
        /* That's right, NULL's... force a rewind in getpair() */
        break;
      case 8:        /* Layer */
        cr_strip(value);
        arc->layer = dxf_add_layer(value, dat);
        break;
      case 10:   /* X point 1 */
        arc->x = strtod(value, NULL);
        break;
      case 20:
        arc->y = strtod(value, NULL);
        break;
      case 30:
        arc->z = strtod(value, NULL);
        break;
      case 40:
        arc->radius = strtod(value, NULL);
        break;
      case 50:
        arc->start_angle = strtod(value, NULL);
        break;
      case 51:
        arc->end_angle = strtod(value, NULL);
        break;
      case 210:
        arc->extx = strtod(value, NULL);
        break;
      case 220:
        arc->exty = strtod(value, NULL);
        break;
      case 230:
        arc->extz = strtod(value, NULL);
        break;
    }
    dat->lineno++;
  }

  if (!err) /* Couldn't parse it! */
  {
    free(arc);
    return (NULL);
  }

  /* Update the bounding box */
  dxf_dobox(arc->x+arc->radius, arc->y+arc->radius, arc->z,dat);
  dxf_dobox(arc->x-arc->radius, arc->y-arc->radius, arc->z,dat);

  return (arc);
}

/* POINT Parser block */
DXF_Point *dxf_parse_point(FILE *fp,DXF_Data *dat)
{
  int err = 0;
  char value[DXF_LINE_MAX];
  int code;
  DXF_Point *point;
  int good = TRUE;

  /* Malloc() one */
  point = malloc(sizeof(DXF_Point));
  if (point == NULL)        /* Couldn't allocate, out of memory! */
    return NULL;

  bzero (point,sizeof(DXF_Point));

  while (good)
  {
    err = dxf_getpair(fp, &code, value, dat);
    if (!err) break;

    dat->lineno++;

    switch (code)
    {
      case 0: /* Oops, start of new entity */
        good = FALSE;
        /* Now to undo the last read... oh boy */
        dxf_getpair(fp, NULL, NULL, dat);
        /* That's right, NULL's... force a rewind in getpair() */
        break;
      case 8:       /* Layer */
        cr_strip(value);
        point->layer = dxf_add_layer(value, dat);
        break;
      case 10:
        point->x = strtod(value, NULL);
        break;
      case 20:
        point->y = strtod(value, NULL);
        break;
      case 30:
        point->z = strtod(value, NULL);
        break;
      case 39:
        point->thickness = strtod(value, NULL);
        break;
      case 210: /* Extrusion */
        point->extx = strtod(value, NULL);
        break;
      case 220:
        point->exty = strtod(value, NULL);
        break;
      case 230:
        point->extz = strtod(value, NULL);
        break;
    }
    dat->lineno++;
  }

  if (!err) /* Couldn't parse it! */
  {
    free(point);
    return NULL;
  }

  /* Update the bounding box */
  dxf_dobox(point->x, point->y, point->z, dat);

  return point;
}

/* CIRCLE parser block */
DXF_Circle *dxf_parse_circle(FILE *fp, DXF_Data *dat)
{
  int err = 0;
  char value[DXF_LINE_MAX];
  int code;
  DXF_Circle *circle;
  int good = TRUE;

  /* Malloc() one */
  circle = malloc(sizeof(DXF_Circle));
  if (circle == NULL)        /* Couldn't allocate, out of memory! */
    return (NULL);

  bzero (circle,sizeof(DXF_Circle));

  while (good)
  {
    err = dxf_getpair(fp, &code, value, dat);
    if (!err) break;

    dat->lineno++;

    switch (code)
    {
      case 0: /* Oops, start of new entity */
        good = FALSE;
        /* Now to undo the last read... oh boy */
        dxf_getpair(fp, NULL, NULL, dat);
        /* That's right, NULL's... force a rewind in getpair() */
        break;
      case 8:        /* Layer */
        cr_strip(value);
        circle->layer=dxf_add_layer(value,dat);
        break;
      case 10:   /* X point 1 */
        circle->x = strtod(value, NULL);
        break;
      case 20:
        circle->y = strtod(value, NULL);
        break;
      case 30:
        circle->z = strtod(value, NULL);
        break;
      case 40:
        circle->radius = strtod(value, NULL);
        break;
      case 210:
        circle->extx = strtod(value, NULL);
        break;
      case 220:
        circle->exty = strtod(value, NULL);
        break;
      case 230:
        circle->extz = strtod(value, NULL);
        break;
    }
    dat->lineno++;
  }

  if (!err) /* Couldn't parse it! */
  {
    free(circle);
    return NULL;
  }

  /* Update the bounding box */
//   dxf_dobox(arc->x+arc->radius,arc->y+arc->radius,arc->z,dat);

  return circle;
}

/* SOLID parsing block */
DXF_Solid *dxf_parse_solid(FILE *fp, DXF_Data *dat)
{
  int err = 0;
  char value[DXF_LINE_MAX];
  int code;
  DXF_Solid *solid;
  int good = TRUE;

  /* Malloc() one */
  solid = malloc(sizeof(DXF_Solid));
  if (solid == NULL)        /* Couldn't allocate, out of memory! */
    return NULL;

  bzero (solid,sizeof(DXF_Solid));

  while (good)
  {
    err = dxf_getpair(fp, &code, value, dat);
    if (!err) break;

    dat->lineno++;

    switch (code)
    {
      case 0: /* Oops, start of new entity */
        good = FALSE;
        /* Now to undo the last read... oh boy */
        dxf_getpair(fp, NULL, NULL, dat);
        /* That's right, NULL's... force a rewind in getpair() */
        break;
      case 8:        /* Layer */
        cr_strip(value);
        solid->layer = dxf_add_layer(value, dat);
        break;
      case 10:   /* X point 1 */
        solid->x1 = strtod(value, NULL);
        break;
      case 11:
        solid->x2 = strtod(value, NULL);
        break;
      case 12:
        solid->x3 = strtod(value, NULL);
        break;
      case 13:
        solid->x4 = strtod(value, NULL);
        break;
      case 20:
        solid->y1 = strtod(value, NULL);
        break;
      case 21:
        solid->y2 = strtod(value, NULL);
        break;
      case 22:
        solid->y3 = strtod(value, NULL);
        break;
      case 23:
        solid->y4 = strtod(value, NULL);
        break;
      case 30:
        solid->z1 = strtod(value, NULL);
        break;
      case 31:
        solid->z2 = strtod(value, NULL);
        break;
      case 32:
        solid->z3 = strtod(value, NULL);
        break;
      case 33:
        solid->z4 = strtod(value, NULL);
        break;
      case 39:
        solid->thickness = strtod(value, NULL);
        break;
      case 210:
        solid->extx = strtod(value, NULL);
        break;
      case 220:
        solid->exty = strtod(value, NULL);
        break;
      case 230:
        solid->extz=strtod(value,NULL);
        break;
    }
    dat->lineno++;
  }
  if (!err) /* Couldn't parse it! */
  {
    free(solid);
    return (NULL);
  }

   /* Update the bounding box */
//   dxf_dobox(arc->x+arc->radius,arc->y+arc->radius,arc->z,dat);

  return (solid);
}

