/* color_map.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "config.h"

#include <math.h>

#include <gtk/gtk.h>

//#include "giramlib/vectors.h"
#include "color_map.h"

#define Epsilon 1e-8

/*************************************************************************
*  color_map_check_valid
*  return TRUE if color_map is a valid color map
**************************************************************************/
gboolean color_map_check_valid(GList *color_map)
{
  
  if (g_list_length(color_map)<2 || g_list_length(color_map)>256)
    return FALSE;
  
  return TRUE;
}

/*************************************************************************
*  color_map_new
**************************************************************************/
GList *color_map_new(void)
{
  return NULL; /* XXX... */
}

/*************************************************************************
*  color_map_add_point
**************************************************************************/
GList *color_map_add_point(GList *color_map, gdouble position, gdouble *color)
{
  color_map_entry *entry, *new_entry;
  GList           *tmp;
  guint            index;
  
  new_entry = g_new(color_map_entry, 1);
  new_entry->position = position;
  new_entry->color[0] = color[0];
  new_entry->color[1] = color[1];
  new_entry->color[2] = color[2];

  if (!color_map || position < ((color_map_entry*)(color_map->data))->position)
  {
    return g_list_prepend(color_map, new_entry);
  }
  
  for (tmp = color_map, index=1 ; tmp ; tmp = g_list_next(tmp), index++)
  {
    entry = tmp->data;
    if ((position > entry->position) && 
        ((!tmp->next) || ( ((color_map_entry*)(tmp->next->data))->position >= position)))
    {
      return g_list_insert(color_map, new_entry, index);
    }
  }
  g_print("here! (should never happen)\n");
  return g_list_append(color_map, new_entry);
}

/*************************************************************************
*  color_map_display
**************************************************************************/
void color_map_display(GList *color_map)
{
  GList           *tmp;
  color_map_entry *entry;

  for (tmp=color_map ; tmp ; tmp = g_list_next(tmp) )
  {
    entry = tmp->data;
    g_print("%.4g <%g, %g, %g>\n", entry->position,
            entry->color[0], entry->color[1], entry->color[2]);
  }
}

/*************************************************************************
*  color_map_change_color
**************************************************************************/
gboolean color_map_change_color(GList *color_map, gdouble position, gdouble *color)
{
  GList           *tmp;
  color_map_entry *entry;

  for ( tmp=color_map ; tmp ; tmp = g_list_next(tmp) )
  {
    entry = tmp->data;
    if (fabs(entry->position-position) < Epsilon)
    {
      entry->color[0] = color[0];
      entry->color[1] = color[1];
      entry->color[2] = color[2];
      return TRUE;
    }
  }
  return FALSE;
}

/*************************************************************************
*  color_map_remove_point
**************************************************************************/
GList *color_map_remove_point(GList *color_map, gdouble position)
{
  GList           *tmp;
  color_map_entry *entry;

  for ( tmp=color_map ; tmp ; tmp = g_list_next(tmp) )
  {
    entry = tmp->data;
    if (fabs(entry->position-position) < Epsilon)
    {
      return g_list_remove_link(color_map, tmp);
    }
  }
  return color_map;
}

/*************************************************************************
*  color_map_remove_point_index
**************************************************************************/
GList *color_map_remove_point_index(GList *color_map, guint index)
{
  return g_list_remove_link(color_map, g_list_nth(color_map, index));
}

/*************************************************************************
*  color_map_evaluate
**************************************************************************/
void color_map_evaluate(GList *color_map, gdouble position, gdouble *color)
{
  color_map_entry *entry, *left_entry, *right_entry;
  GList           *tmp;

  if (!color_map)
  {
    color[0] = color[1] = color[2] = 1.0;
    return;
  }

  /* before first and after last are special cased */
  entry = color_map->data;
  if (position <= entry->position)
  {
    color[0] = entry->color[0];
    color[1] = entry->color[1];
    color[2] = entry->color[2];
    return;
  }
  entry = g_list_nth(color_map, g_list_length(color_map) - 1)->data;
  if (position >= entry->position)
  {
    color[0] = entry->color[0];
    color[1] = entry->color[1];
    color[2] = entry->color[2];
  }

  /* else, we interpolate */ 
  for (tmp = color_map ; tmp->next ; tmp = g_list_next(tmp))
  {
    left_entry = tmp->data;
    right_entry = tmp->next->data;
    if ((position > left_entry->position) && right_entry->position >= position)
    {
      gdouble alpha, beta;

      alpha = (right_entry->position-position) / (right_entry->position - left_entry->position);
      beta  = (position-left_entry->position) / (right_entry->position - left_entry->position);
      color[0] = alpha*left_entry->color[0] + beta*right_entry->color[0];
      color[1] = alpha*left_entry->color[1] + beta*right_entry->color[1];
      color[2] = alpha*left_entry->color[2] + beta*right_entry->color[2];
      return;
    }
  }
}

/*************************************************************************
*  color_map_copy
**************************************************************************/
GList *color_map_copy(GList *color_map)
{
  GList           *new_color_map = NULL, *tmp;
  color_map_entry *entry, *new_entry;

  for (tmp = color_map ; tmp ; tmp = g_list_next(tmp) )
  {
    entry = tmp->data;
    new_entry = g_new(color_map_entry, 1);
    new_entry->position = entry->position;
    new_entry->color[0] = entry->color[0];
    new_entry->color[1] = entry->color[1];
    new_entry->color[2] = entry->color[2];
    new_color_map = g_list_append(new_color_map, new_entry);
  }
  
  return new_color_map;
}

