/* camera.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2000 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "giram.h"
#include "utils.h"
#include "camera.h"

#include "giramintl.h"

/*****************************************************************************
*  InitCamera
******************************************************************************/
void InitCamera(CameraStruct *Camera)
{
  Camera->Type = CAMERA_PERSPECTIVE;
  V3Deq(Camera->Location,  0.0 , 0.0, -5.0);
  V3Deq(Camera->LookAt,    0.0 , 0.0,  1.0);
  V3Deq(Camera->Right,     1.33, 0.0,  0.0);
  V3Deq(Camera->Up,        0.0,  1.0,  0.0);
  V3Deq(Camera->Direction, 0.0,  0.0,  1.0);
  V3Deq(Camera->Sky,       0.0,  1.0,  0.0);
  Camera->Angle = 90.0;
  Camera->BlurSamples = 0;
  Camera->Aperture = 0.0;
  V3Deq(Camera->FocalPoint, 0.0, 0.0, 1.0);
  Camera->Confidence = 0.9;
  Camera->Variance = 1.0/128.0;
  Camera->name = g_strdup(_("Camera 1"));
}

/*****************************************************************************
*  ComputeCamera
******************************************************************************/
void ComputeCamera(CameraStruct *Camera)
{
  double Norme;

  if (Camera == NULL)
    return;
  Camera->x0 = Camera->Location[0];
  Camera->y0 = Camera->Location[1];
  Camera->z0 = Camera->Location[2];
  Camera->PX = Camera->Location[0] + Camera->Direction[0];
  Camera->PY = Camera->Location[1] + Camera->Direction[1];
  Camera->PZ = Camera->Location[2] + Camera->Direction[2];
  Camera->A = -(Camera->Up[1]*Camera->Right[2]-Camera->Up[2]*Camera->Right[1]);
  Camera->B = -(Camera->Up[2]*Camera->Right[0]-Camera->Up[0]*Camera->Right[2]);
  Camera->C = -(Camera->Up[0]*Camera->Right[1]-Camera->Up[1]*Camera->Right[0]);
  Norme = sqrt(Camera->A*Camera->A+Camera->B*Camera->B+Camera->C*Camera->C);
  Camera->A /= Norme;
  Camera->B /= Norme;
  Camera->C /= Norme;
  Camera->D = -(Camera->A*Camera->PX+Camera->B*Camera->PY+Camera->C*Camera->PZ);
  /* Precalculate the P.. series */
  V3Deq(Camera->P00, Camera->Location[0]+Camera->Direction[0]-
                     Camera->Up[0]/2.0-Camera->Right[0]/2.0,
                     Camera->Location[1]+Camera->Direction[1]-
                     Camera->Up[1]/2.0-Camera->Right[1]/2.0,
                     Camera->Location[2]+Camera->Direction[2]-
                     Camera->Up[2]/2.0-Camera->Right[2]/2.0);

  Camera->P01[0] = Camera->Location[0]+Camera->Direction[0]+
                   Camera->Up[0]/2.0-Camera->Right[0]/2.0;
  Camera->P01[1] = Camera->Location[1]+Camera->Direction[1]+
                   Camera->Up[1]/2.0-Camera->Right[1]/2.0;
  Camera->P01[2] = Camera->Location[2]+Camera->Direction[2]+
                   Camera->Up[2]/2.0-Camera->Right[2]/2.0;

  Camera->P10[0] = Camera->Location[0]+Camera->Direction[0]-
                   Camera->Up[0]/2.0+Camera->Right[0]/2.0;
  Camera->P10[1] = Camera->Location[1]+Camera->Direction[1]-
                   Camera->Up[1]/2.0+Camera->Right[1]/2.0;
  Camera->P10[2] = Camera->Location[2]+Camera->Direction[2]-
                   Camera->Up[2]/2.0+Camera->Right[2]/2.0;

  Camera->P11[0] = Camera->Location[0]+Camera->Direction[0]+
                   Camera->Up[0]/2.0+Camera->Right[0]/2.0;
  Camera->P11[1] = Camera->Location[1]+Camera->Direction[1]+
                   Camera->Up[1]/2.0+Camera->Right[1]/2.0;
  Camera->P11[2] = Camera->Location[2]+Camera->Direction[2]+
                   Camera->Up[2]/2.0+Camera->Right[2]/2.0;

  Camera->PA[0] = Camera->P00[0]-Camera->P10[0]+Camera->P11[0]-Camera->P01[0];
  Camera->PA[1] = Camera->P00[1]-Camera->P10[1]+Camera->P11[1]-Camera->P01[1];
  Camera->PA[2] = Camera->P00[2]-Camera->P10[2]+Camera->P11[2]-Camera->P01[2];
  Camera->PB[0] = Camera->P10[0]-Camera->P00[0];
  Camera->PB[1] = Camera->P10[1]-Camera->P00[1];
  Camera->PB[2] = Camera->P10[2]-Camera->P00[2];
  Camera->PC[0] = Camera->P01[0]-Camera->P00[0];
  Camera->PC[1] = Camera->P01[1]-Camera->P00[1];
  Camera->PC[2] = Camera->P01[2]-Camera->P00[2];
  Camera->PD[0] = Camera->P00[0];
  Camera->PD[1] = Camera->P00[1];
  Camera->PD[2] = Camera->P00[2];

  Camera->Na[0]=Camera->PA[1]*Camera->C-Camera->PA[2]*Camera->B;
  Camera->Na[1]=Camera->PA[2]*Camera->A-Camera->PA[0]*Camera->C;
  Camera->Na[2]=Camera->PA[0]*Camera->B-Camera->PA[1]*Camera->A;

  Camera->Nb[0]=Camera->PB[1]*Camera->C-Camera->PB[2]*Camera->B;
  Camera->Nb[1]=Camera->PB[2]*Camera->A-Camera->PB[0]*Camera->C;
  Camera->Nb[2]=Camera->PB[0]*Camera->B-Camera->PB[1]*Camera->A;

  Camera->Nc[0]=Camera->PC[1]*Camera->C-Camera->PC[2]*Camera->B;
  Camera->Nc[1]=Camera->PC[2]*Camera->A-Camera->PC[0]*Camera->C;
  Camera->Nc[2]=Camera->PC[0]*Camera->B-Camera->PC[1]*Camera->A;

  Camera->Du0 = Camera->Nc[0]*Camera->PD[0]+
                Camera->Nc[1]*Camera->PD[1]+
                Camera->Nc[2]*Camera->PD[2];
  Camera->Du1 = Camera->Na[0]*Camera->PD[0]+
                Camera->Na[1]*Camera->PD[1]+
                Camera->Na[2]*Camera->PD[2]+
                Camera->Nc[0]*Camera->PB[0]+
                Camera->Nc[1]*Camera->PB[1]+
                Camera->Nc[2]*Camera->PB[2];
  Camera->Du2 = Camera->Na[0]*Camera->PB[0]+
                Camera->Na[1]*Camera->PB[1]+
                Camera->Na[2]*Camera->PB[2];
  Camera->Dv0 = Camera->Nb[0]*Camera->PD[0]+
                Camera->Nb[1]*Camera->PD[1]+
                Camera->Nb[2]*Camera->PD[2];
  Camera->Dv1 = Camera->Na[0]*Camera->PD[0]+
                Camera->Na[1]*Camera->PD[1]+
                Camera->Na[2]*Camera->PD[2]+
                Camera->Nb[0]*Camera->PC[0]+
                Camera->Nb[1]*Camera->PC[1]+
                Camera->Nb[2]*Camera->PC[2];
  Camera->Dv2 = Camera->Na[0]*Camera->PC[0]+
                Camera->Na[1]*Camera->PC[1]+
                Camera->Na[2]*Camera->PC[2];
  if (fabs(Camera->Du2)>1e-6)
  {
    V3Deq(Camera->Qux, Camera->Na[0]/(2.0*Camera->Du2),
                     Camera->Na[1]/(2.0*Camera->Du2),
                     Camera->Na[2]/(2.0*Camera->Du2));
    Camera->Dux = -Camera->Du1/(2.0*Camera->Du2);
    Camera->Quy[0]=-Camera->Nc[0]/Camera->Du2;
    Camera->Quy[1]=-Camera->Nc[1]/Camera->Du2;
    Camera->Quy[2]=-Camera->Nc[2]/Camera->Du2;
    Camera->Duy=Camera->Du0/Camera->Du2;
  }
  if (fabs(Camera->Dv2)>1e-6)
  {
    Camera->Qvx[0]=Camera->Na[0]/(2.0*Camera->Dv2);
    Camera->Qvx[1]=Camera->Na[1]/(2.0*Camera->Dv2);
    Camera->Qvx[2]=Camera->Na[2]/(2.0*Camera->Dv2);
    Camera->Dvx = -Camera->Dv1/(2.0*Camera->Dv2);
    Camera->Qvy[0]=-Camera->Nc[0]/Camera->Dv2;
    Camera->Qvy[1]=-Camera->Nc[1]/Camera->Dv2;
    Camera->Qvy[2]=-Camera->Nc[2]/Camera->Dv2;
    Camera->Dvy=Camera->Dv0/Camera->Dv2;
  }
}

static int LocalCameraType; /* FIXME: shouldn't be a global varible but should
                               be embedded in an object... */
/*****************************************************************************
*  ChangeCameraType
******************************************************************************/
void ChangeCameraType(GtkWidget *Menu, int Type)
{
  LocalCameraType = Type;
}

/*****************************************************************************
*  EditCameraOK
******************************************************************************/
void EditCameraOK(GtkWidget *Dialog)
{
  GtkWidget    **Data;
  ViewStruct    *view_data;
  CameraStruct  *Camera;
  GSList        *tmp_list;
  ViewStruct    *TmpView;

  Data = g_object_get_data(G_OBJECT(Dialog), "WidgetData");
  view_data = get_current_view_data();
  Camera = view_data->frame->all_cameras->data;
  switch (LocalCameraType)
  {
    case 0:
      Camera->Type = CAMERA_PERSPECTIVE;
      break;
    case 1:
      Camera->Type = CAMERA_ORTHOGRAPHIC;
      break;
    case 2:
      Camera->Type = CAMERA_FISHEYE;
      break;
    case 3:
      Camera->Type = CAMERA_ULTRA_WIDE_ANGLE;
      break;
    case 4:
      Camera->Type = CAMERA_OMNIMAX;
      break;
    case 5:
      Camera->Type = CAMERA_PANORAMIC;
      break;
    case 6:
      Camera->Type = CAMERA_CYLINDER_1;
      break;
    case 7:
      Camera->Type = CAMERA_CYLINDER_2;
      break;
    case 8:
      Camera->Type = CAMERA_CYLINDER_3;
      break;
    case 9:
      Camera->Type = CAMERA_CYLINDER_4;
      break;
  }
  Camera->Location[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[0])));
  Camera->Location[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[1])));
  Camera->Location[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[2])));
  Camera->Direction[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[3])));
  Camera->Direction[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[4])));
  Camera->Direction[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[5])));
  Camera->Right[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[6])));
  Camera->Right[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[7])));
  Camera->Right[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[8])));
  Camera->Up[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[9])));
  Camera->Up[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[10])));
  Camera->Up[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[11])));
  Camera->Sky[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[12])));
  Camera->Sky[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[13])));
  Camera->Sky[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[14])));
  Camera->LookAt[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[15])));
  Camera->LookAt[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[16])));
  Camera->LookAt[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[17])));
  Camera->Angle = atof(gtk_entry_get_text(GTK_ENTRY(Data[18])));
  Camera->BlurSamples = atoi(gtk_entry_get_text(GTK_ENTRY(Data[19])));
  Camera->Aperture = atof(gtk_entry_get_text(GTK_ENTRY(Data[20])));
  Camera->FocalPoint[0] = atof(gtk_entry_get_text(GTK_ENTRY(Data[21])));
  Camera->FocalPoint[1] = atof(gtk_entry_get_text(GTK_ENTRY(Data[22])));
  Camera->FocalPoint[2] = atof(gtk_entry_get_text(GTK_ENTRY(Data[23])));
  Camera->Confidence = atof(gtk_entry_get_text(GTK_ENTRY(Data[24])));
  Camera->Variance = atof(gtk_entry_get_text(GTK_ENTRY(Data[25])));

  ComputeCamera(Camera);

  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  gtk_widget_destroy(Dialog);
}

/*************************************************************************
*  edit_camera_response
**************************************************************************/
static void edit_camera_response(GtkWidget *dialog, gint response)
{
  switch (response)
  {
    case GTK_RESPONSE_ACCEPT:
      EditCameraOK(dialog);
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(dialog);
      break;
  }
}

/*****************************************************************************
*  EditCamera
******************************************************************************/
void EditCamera(GtkWidget *DrawingArea)
{
  ViewStruct    *view_data;
  CameraStruct  *Camera;
  GtkWidget     *Dialog, *Table, *Label;
  GtkWidget     *OptionMenu, *Menu, *Entry;
  GSList        *MenuGroup;
  GtkWidget     *MenuItem, *Frame;
  GtkWidget    **Data;
  gchar          Text[50];
  guint          i;
  char *CameraType[] =
  {
    "Perspective",
    "Orthographic",
    "FishEye",
    "Ultra Wide Angle",
    "Omnimax",
    "Panoramic",
    "Cylinder 1",
    "Cylinder 2",
    "Cylinder 3",
    "Cylinder 4"
  };

  view_data = get_current_view_data();
  Camera = view_data->frame->all_cameras->data;
  Dialog = gtk_dialog_new_with_buttons(_("Edit Camera"), NULL, 0,
                                       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                                       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
                                       NULL);
  gtk_window_set_position(GTK_WINDOW(Dialog), GTK_WIN_POS_MOUSE);
  g_signal_connect(G_OBJECT(Dialog), "response",
                   G_CALLBACK(edit_camera_response), NULL);

  Data = g_new(GtkWidget*, 30);
  g_object_set_data(G_OBJECT(Dialog), "WidgetData", Data);

  /* The "Camera Type" Options Menu */
  Table = gtk_table_new(2, 1, FALSE);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Dialog)->vbox), Table, TRUE, TRUE, 0);

  Label = gtk_label_new(_("Camera Type:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

  OptionMenu = gtk_option_menu_new();
  Menu = gtk_menu_new();
  MenuGroup = NULL;
  for (i=0 ; i<10 ; i++)
  {
    MenuItem = gtk_radio_menu_item_new_with_label(MenuGroup, CameraType[i]);
    g_signal_connect(G_OBJECT(MenuItem), "activate",
                     G_CALLBACK(ChangeCameraType), GINT_TO_POINTER(i));
    MenuGroup = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(MenuItem));
    gtk_menu_shell_append(GTK_MENU_SHELL(Menu), MenuItem);
    if (Camera->Type-1 == i)
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(MenuItem), TRUE);
    else
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(MenuItem), FALSE);
    gtk_widget_show(MenuItem);
  }
  gtk_option_menu_set_menu(GTK_OPTION_MENU(OptionMenu), Menu);
  gtk_table_attach(GTK_TABLE(Table), OptionMenu, 1,2, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

  /* All the "Vectors" are in a table */
  Table = gtk_table_new(4, 7, FALSE);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Dialog)->vbox), Table, TRUE, TRUE, 0);

  Label = gtk_label_new(_("x:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 1,2, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  Label = gtk_label_new(_("y:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 2,3, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  Label = gtk_label_new(_("z:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 3,4, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  /* The Location Point */
  Label = gtk_label_new(_("Location:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 1,2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 1,2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Location[0]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[0] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 2,3, 1,2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Location[1]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[1] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 3,4, 1,2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Location[2]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[2] = Entry;
  /* The Direction Vector */
  Label = gtk_label_new(_("Direction:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 2,3,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 2,3,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Direction[0]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[3] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 2,3, 2,3,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Direction[1]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[4] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 3,4, 2,3,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Direction[2]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[5] = Entry;
  /* The Right Vector */
  Label = gtk_label_new(_("Right:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 3,4,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 3,4,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Right[0]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[6] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 2,3, 3,4,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Right[1]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[7] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 3,4, 3,4,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Right[2]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[8] = Entry;
  /* The Up Vector */
  Label = gtk_label_new(_("Up:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 4,5,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 4,5,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Up[0]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[9] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 2,3, 4,5,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Up[1]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[10] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 3,4, 4,5,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Up[2]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[11] = Entry;
  /* The Sky Vector */
  Label = gtk_label_new(_("Sky:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 5,6,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 5,6,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Sky[0]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[12] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 2,3, 5,6,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Sky[1]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[13] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 3,4, 5,6,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Sky[2]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[14] = Entry;
  /* The "Look At" Point */
  Label = gtk_label_new(_("Look At:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 6,7,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 6,7,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->LookAt[0]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[15] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 2,3, 6,7,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->LookAt[1]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[16] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 3,4, 6,7,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->LookAt[2]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[17] = Entry;

  /* The Angle */
  Table = gtk_table_new(2, 1, FALSE);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Dialog)->vbox), Table, TRUE, TRUE, 0);

  Label = gtk_label_new(_("Angle:"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  sprintf(Text, "%g", Camera->Angle);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[18] = Entry;
  /* The "Depth of view" features */
  Frame = gtk_frame_new(_("Depth of view"));
  gtk_container_set_border_width(GTK_CONTAINER(Frame), 10);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Dialog)->vbox), Frame, TRUE, TRUE, 0);

  Table = gtk_table_new(4, 3, FALSE);
  gtk_container_add(GTK_CONTAINER(Frame), Table);
  /* The Number of Blur Samples */
  Label = gtk_label_new(_("Blur Samples"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%d", Camera->BlurSamples);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[19] = Entry;
  /* The Aperture */
  Label = gtk_label_new(_("Aperture"));
  gtk_table_attach(GTK_TABLE(Table), Label, 2,3, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 3,4, 0,1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Aperture);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[20] = Entry;
  /* The Focal Point */
  Label = gtk_label_new(_("Focal Point"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 1,2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 1,2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->FocalPoint[0]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[21] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 2,3, 1,2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->FocalPoint[1]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[22] = Entry;
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 3,4, 1,2,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->FocalPoint[2]);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[23] = Entry;
  /* The Confidence */
  Label = gtk_label_new(_("Confidence"));
  gtk_table_attach(GTK_TABLE(Table), Label, 0,1, 2,3,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 1,2, 2,3,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Confidence);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[24] = Entry;
  /* The Variance */
  Label = gtk_label_new(_("Variance"));
  gtk_table_attach(GTK_TABLE(Table), Label, 2,3, 2,3,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  Entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(Table), Entry, 3,4, 2,3,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 5, 5);
  sprintf(Text, "%g", Camera->Variance);
  gtk_entry_set_text(GTK_ENTRY(Entry), Text);
  gtk_widget_set_size_request(Entry, 80, -1);
  Data[25] = Entry;
  Data[26] = DrawingArea;
  /* The Normal perturbation */
  gtk_widget_show_all(Dialog);
}

