/* povfront.c
 * PovFront - A GPLed Front-end for POVray raytracer
 * Copyright (C) 1999 Philippe DAVID <clovis1@club-internet.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Please, report the bugs to clovis1@club-internet.fr
 */


/****************************************************************
* Name : 		povfront.c
* Parameters :
* Author : Philippe DAVID
* Update (mm/dd/yy) : 10/16/1999
* Return : 0 or 1 if an error occurs
* Comments : A graphical interface for povray raytracing engine
****************************************************************/

#warning "fixme: GTK_DISABLE_DEPRECATED"
#undef GTK_DISABLE_DEPRECATED

#include "povfront.h"

/****************************************************************
 * Name : 		init_programme
 * parameters :       
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 9/04/1999
 * Return : void
 * Comments : Initialise some global variables
 *            This is call after gtk_init call and before creating widget
 ****************************************************************/
void
init_programme (void)
{
  /*** set options if the default file cannot be read (first use)  */
  strcpy (g_aa_methode, "1");
  strcpy (g_palette, "G");
  g_out_file_mode = 1;
  g_quality_value = 9;
  g_quality_threshold = 0.3;
  g_bounding_threshold_value = 25;
  g_quality_gamma = 2.2;
  g_quality_depth = 3;
  strcpy (g_command_to_invoque, "x-povray");
  strcpy (g_entry_save_path, "");
  strcpy (g_entry_pov_version, "3.1");
  g_invoque_command_number = 1;
  g_local_invoque_command_number = g_invoque_command_number;
  g_pvm_nice_value = 10;

  g_substartx_value = 20;
  g_subendx_value = 70;
  g_substarty_value = 10;
  g_subendy_value = 40;

  g_tooltips_background_color.pixel = 65535;
  g_tooltips_background_color.red = 65500;
  g_tooltips_background_color.green = 65500;
  g_tooltips_background_color.blue = 0;

  g_tooltips_foreground_color.pixel = 65535;
  g_tooltips_foreground_color.red = 0;
  g_tooltips_foreground_color.green = 0;
  g_tooltips_foreground_color.blue = 0;

  g_jobs_list = NULL;

#ifdef ENABLE_NLS
  setlocale (LC_ALL, "");
  textdomain ("giram");    /* DDX: changed this for giram compatibility */
  /*bindtextdomain ("povfront", GNOMELOCALEDIR);  */
#endif

  /* Get the HOME environment variable  */
  strcpy (g_home_directory, getenv ("HOME"));
  sprintf (g_default_config_file, "%s/.povfront", g_home_directory);
  if (access (g_default_config_file, F_OK) == -1)
    {
      /* if (mkdir (g_default_config_file, (mode_t) ACCESSPERMS) == -1) */
      /* ACCESSPERMS is not define on some plateform */
      if (mkdir (g_default_config_file, (mode_t) 0777) == -1)
	{
	  exit (-1);
	}
      else
	{
	  sprintf (g_default_config_file, "%s/.povfront/default.txt", g_home_directory);
	  return; /* I don't try to read the default file which will be created by the user  */
	}
    }
  else /* The directory ~/.povfront exist  */
    {
      sprintf (g_default_config_file, "%s/.povfront/default.txt", g_home_directory);
      if (access (g_default_config_file, F_OK) == -1)
	{
	  return;
	}
      else
	{
	  g_config_file_to_parse = g_strdup (g_default_config_file);
	}
    }
  /* Create the job control window  */
  control ();
}
/*** END OF init_programme */

/****************************************************************
 * Name : 		delete_event
 * parameters :       
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/06/1998
 * Return : 
 * Comments :
 ****************************************************************/
gint
delete_event (GtkWidget *p_widget, GdkEvent *p_event, gpointer p_data)
{
  return (FALSE);
}
/*** END OF delete_event */

/****************************************************************
 * Name : 		destroy
 * parameters : Aucun
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/06/1998
 * Return :
 * Comments :
 ****************************************************************/
void
destroy (GtkWidget *p_widget, gpointer p_data)
{
  if (g_save_settings_exit)
    {
      write_config_file (g_default_config_file);
    }
  /* free some space if necessary (no error if null with g_free()  */
  /* Gnome is handling itself this
     but with GTK+ WE have made some g_strdup so we must free ourself  */
  g_free (g_image_file_to_compute);
  g_free (g_pov_file_to_parse);
  g_free (g_redirect_command_to_use);
  g_free (g_config_file_to_parse);

  gtk_main_quit ();
}
/*** END OF destroy */

/****************************************************************
 * Name : 		invoque_command
 * parameters : 
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 02/13/1999
 * Return :
 * Comments : (callback) Apply command string in configure window
 ***************************************************************/
gint
invoque_command (GtkWidget *p_widget, gchar *p_command_to_invoque)
{
  if (strcmp (p_command_to_invoque, "1") == 0)
    {
      g_local_invoque_command_number = 1;
    }
  else
    {
      g_local_invoque_command_number = 2;
    }
  return (FALSE);
}
/*** END OF invoque_command */

/****************************************************************
 * Name : 		image_size
 * parameters : # of the radio button (image size in the output page)
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 03/17/1999
 * Return :
 * Comments : (call_back)
 ****************************************************************/
gint
image_size (GtkWidget *p_widget, gchar *p_data)
{
  g_return_val_if_fail (p_data != NULL, TRUE);
  g_return_val_if_fail (p_widget != NULL, TRUE);

  g_size_type = atoi (p_data);

  return (FALSE);
}
/*** END OF image_size */

/****************************************************************
 * Name : 		aa_methode
 * parameters : 
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 02/21/1999
 * Return :
 * Comments : (call_back)
 ****************************************************************/
gint
aa_methode (GtkWidget *p_widget, gpointer p_data)
{
  g_return_val_if_fail (p_data != NULL, TRUE);

  strcpy (g_aa_methode, p_data);
  return (FALSE);
}
/*** END OF aa_methode */

/****************************************************************
 * Name : 		apply_options
 * parameters : 
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 9/16/1999
 * Return :
 * Comments : Apply the modified options from the configure window
 *            Callback for the apply button
 ****************************************************************/
gint
apply_options (GtkWidget *p_widget, gpointer p_data)
{
  strcpy (g_command_to_invoque, gtk_entry_get_text (GTK_ENTRY (g_entry_command)));
  strcpy (g_alt_command_to_invoque, gtk_entry_get_text (GTK_ENTRY (g_entry_altcommand)));
  g_invoque_command_number = g_local_invoque_command_number;
  g_show_tooltips = g_show_tooltips_local;
  if (g_show_tooltips)
    {
      gtk_tooltips_enable (GTK_TOOLTIPS (g_tooltips));
    }
  else
    {
      gtk_tooltips_disable (GTK_TOOLTIPS (g_tooltips));
    }
  g_test_overwrite = g_test_overwrite_local;
  g_save_settings_exit = g_save_settings_exit_local;
  g_beep = g_beep_local;
  return (FALSE);
}
/*** END OF apply_options */

/****************************************************************
 * Name : 		options
 * parameters : 
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/15/1999
 * Return :
 * Comments :(callback) Display configure window for option setting
 ****************************************************************/
gint
options (GtkWidget *p_widget, gpointer p_data)
{
  GtkWidget *l_check_tooltips;
  GtkWidget *l_check_overwrite;
  GtkWidget *l_check_save_settings_exit;
  GtkWidget *l_check_beep;
  GtkWidget *l_radio_command;
  GtkWidget *l_radio_altcommand;
  GtkWidget *l_hbox;
  GtkWidget *l_vbox_radio;
  GtkWidget *l_vbox_entry;
  GtkWidget *l_button_apply;
  GtkWidget *l_button_cancel;
  GtkBox    *l_dialog;
  GSList    *l_grp_radio;

  if (g_configure_window != NULL)
    {
      if (g_configure_window->window != NULL)
	{
	  gdk_window_show (g_configure_window->window);
	  gdk_window_raise (g_configure_window->window);
	  return (FALSE);
	}
    }
  g_configure_window = gtk_dialog_new();
  gtk_container_set_border_width(GTK_CONTAINER(g_configure_window), 5);
  gtk_window_set_title (GTK_WINDOW (g_configure_window), _("PovFront: Options setting"));
  l_dialog = GTK_BOX(GTK_DIALOG(g_configure_window)->vbox);

  l_button_apply = gtk_button_new_with_label(_("Apply"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_configure_window)->action_area), l_button_apply, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(l_button_apply), "clicked", G_CALLBACK(apply_options), NULL);
  g_signal_connect_swapped(G_OBJECT(l_button_apply), "clicked",
                           G_CALLBACK(gtk_widget_destroy), g_configure_window);

  l_button_cancel = gtk_button_new_with_label (_("Cancel"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_configure_window)->action_area), l_button_cancel, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(l_button_cancel), "clicked",
                           G_CALLBACK(gtk_widget_destroy), g_configure_window);
  /* gtk_widget_grab_default (l_button_cancel);*/
  /* Show tool tips option  */
  l_check_tooltips = gtk_check_button_new_with_label (_("Show tooltips"));
  gtk_box_pack_start (l_dialog, l_check_tooltips, FALSE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(l_check_tooltips), g_show_tooltips);
  g_show_tooltips_local = g_show_tooltips;
  g_signal_connect(G_OBJECT(l_check_tooltips), "clicked",
                   G_CALLBACK(switch_option), &g_show_tooltips_local);

  /* Overwrite option  */
  l_check_overwrite = gtk_check_button_new_with_label (_("Test overwrite of output image"));
  gtk_box_pack_start(l_dialog, l_check_overwrite, FALSE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(l_check_overwrite), g_test_overwrite);
  g_test_overwrite_local = g_test_overwrite;
  g_signal_connect(G_OBJECT(l_check_overwrite), "clicked",
                   G_CALLBACK(switch_option), &g_test_overwrite_local);

  /* Save settings as default when exit option  */
  l_check_save_settings_exit = gtk_check_button_new_with_label (_("Save settings as default when exit"));
  gtk_box_pack_start (l_dialog, l_check_save_settings_exit, FALSE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(l_check_save_settings_exit), g_save_settings_exit);
  g_save_settings_exit_local = g_save_settings_exit;
  g_signal_connect(G_OBJECT(l_check_save_settings_exit), "clicked",
                   G_CALLBACK(switch_option), &g_save_settings_exit_local);

  /* Beep when a job is finished  */
  l_check_beep = gtk_check_button_new_with_label (_("Beep when a job is finished"));
  gtk_box_pack_start(l_dialog, l_check_beep, FALSE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(l_check_beep), g_beep);
  g_beep_local = g_beep;
  g_signal_connect(G_OBJECT(l_check_beep), "clicked",
                   G_CALLBACK(switch_option), &g_beep_local);

  /* Radio buttons  */
  l_hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (l_dialog, l_hbox, FALSE, FALSE, 0);

  l_vbox_radio = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox), l_vbox_radio, TRUE, TRUE, 0);

  l_radio_command = gtk_radio_button_new_with_label (NULL, _("Command :"));
  gtk_box_pack_start (GTK_BOX (l_vbox_radio), l_radio_command, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(l_radio_command), "pressed",
                   G_CALLBACK(invoque_command), "1");
  l_grp_radio = gtk_radio_button_get_group(GTK_RADIO_BUTTON(l_radio_command));

  l_radio_altcommand = gtk_radio_button_new_with_label (l_grp_radio, _("Alt. command :"));
  gtk_box_pack_start (GTK_BOX (l_vbox_radio), l_radio_altcommand, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(l_radio_altcommand), "pressed",
                   G_CALLBACK(invoque_command), "2");

  if (g_invoque_command_number == 1)
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (l_radio_command), TRUE);
    }
  else
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (l_radio_altcommand), TRUE);
    }

  /* Entry command  */
  l_vbox_entry = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox), l_vbox_entry, TRUE, TRUE, 0);

  g_entry_command = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_entry), g_entry_command, FALSE, FALSE, 0);
  gtk_entry_set_text (GTK_ENTRY (g_entry_command), g_command_to_invoque);

  g_entry_altcommand = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_entry), g_entry_altcommand, FALSE, FALSE, 0);
  gtk_entry_set_text (GTK_ENTRY (g_entry_altcommand), g_alt_command_to_invoque);

  gtk_widget_show_all (g_configure_window);
  return (FALSE);
}
/*** END OF options */

/****************************************************************
 * Name : 	      adjustment_value_changed
 * parameters : 
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 02/29/1999
 * Return :
 * Comments : (callback)
 ****************************************************************/
gint
adjustment_value_changed (GtkAdjustment *p_adjustment, gfloat *p_value)
{
  g_return_val_if_fail (p_value != NULL, TRUE);

  *p_value = (gfloat) p_adjustment->value;
  return (FALSE);
}
/*** END OF adjustment_value_changed */

/****************************************************************
 * Name : 		load_config_ok
 * parameters : 
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/16/1999
 * Return :
 * Comments : (callback)
 ****************************************************************/
gint
load_config_ok (GtkWidget *p_widget, GtkFileSelection *p_file_selection)
{
  gchar *l_selected_file;

  g_return_val_if_fail (p_file_selection != NULL, TRUE);

  /* Get the selected file  */
  l_selected_file =  (char  *)gtk_file_selection_get_filename (GTK_FILE_SELECTION (p_file_selection));
  if (load_config_file (l_selected_file))
    {
      strcpy (g_entry_load_path, l_selected_file);
    }
  gtk_widget_destroy (GTK_WIDGET (p_file_selection));
  return (FALSE);
}
/*** END OF load_config_ok */

/****************************************************************
 * Name : 	      load_config
 * parameters : callback
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 03/10/1999
 * Return :
 * Comments : callback for the menu option "load config."
 ****************************************************************/
gint
load_config (GtkWidget *p_widget, gpointer p_data)
{
  GtkWidget *l_file_selection;

  l_file_selection = gtk_file_selection_new (_("Load configuration"));
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (l_file_selection), g_entry_load_path);
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION (l_file_selection)->ok_button),
                  "clicked",
                  G_CALLBACK(load_config_ok), l_file_selection );
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(l_file_selection)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), l_file_selection);
  gtk_widget_show(l_file_selection);
  return FALSE;
}
/*** END OF load_config */

/****************************************************************
 * Name : 	      read_line
 * parameters : P_FILE_HANDLE is the file descriptor to use for reading
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 05/13/1999
 * Return :   Pointer to the first char. of the string read
 * Comments : read the line until the end of the file
 ****************************************************************/
gint
read_line (FILE *p_file_handle, gchar *p_line)
{
  gint l_char;
  gint i = 0;
  gchar l_line[256];

  g_return_val_if_fail (p_file_handle != NULL, TRUE);
  g_return_val_if_fail (p_line != NULL, TRUE);

  l_char = fgetc (p_file_handle);
  while ((l_char != '\n') && (l_char != '\r') && (l_char != EOF))
    {
      l_line[i] = (gchar) l_char;
      ++i;
      if (i == 255) /* If the line is too long, it is supsect  */
	{
	  povfront_error (_("ERROR: this is not a configuration file!\n"));
	  return 0;
	}
      l_char = fgetc (p_file_handle);
    }
  l_line [i]=0;
  strcpy (p_line, l_line);
  if (l_char == EOF)
    {
      return 0;
    }
  else
    {
      return 1;
    }
}
/*** END OF read_line */

/****************************************************************
 * Name : 	      load_config_file
 * parameters : P_FILE is the file to load
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 9/26/1999
 * Return : return 0 if succes and 1 if it fail
 * Comments : Load a configuration file and set the options in the
 * 		interface. Load the default config. file if P_FILE = NULL
 ****************************************************************/
gint
load_config_file (gchar *p_file)
{
  FILE  *l_file_handle;
  gchar  l_car;
  gchar  l_option [256];
  gfloat l_value;
  gchar  l_string [256];
  gchar *l_row_data;
  gint   l_row;
  gchar  l_one_line [256];
  gint   l_line_number = 0;   /* Used to report error while parsing the file  */
  gint   l_nbr_error = 0;     /* Nbr of error while parsing the file  */
  gchar  l_message[256];
  gint   i;

  g_return_val_if_fail (p_file != NULL, TRUE);

  /* First, we clean the pvm host list  */
  for (i=0; (l_row_data = (gchar *) gtk_clist_get_row_data (GTK_CLIST (g_gtklist_hosts), i)) != NULL; i++)
    {
      g_free (l_row_data);
    }
  gtk_clist_clear (GTK_CLIST (g_gtklist_hosts));

  if (p_file == NULL) /* NULL if no config. file in the command line  */
    {
      return (load_config_file (g_default_config_file)); /* Recursive call  */
    }
  else
    {
      l_file_handle = fopen (p_file, "r");
      if (l_file_handle == NULL)
	{
	  sprintf (l_message, _("ERROR: load_config_file: cannot open the file %s\n"), p_file);
	  povfront_error (l_message);
	  return 0;
	}
      while (1)
	{
	  l_string [0] = 0;  /* To abort side effect  */
	  if (!read_line (l_file_handle, l_one_line))
	      {
		return 1;
	      }
	  l_car = sscanf (l_one_line, "%s\t%f\t%s\n", l_option, &l_value, l_string);
	  ++l_line_number;
	  if (strcmp ("command", l_option) == 0)
	    {
	      strcpy (g_command_to_invoque, l_string);
	    }
	  else if (strcmp ("altcommand", l_option) == 0)
	    {
	      strcpy (g_alt_command_to_invoque, l_string);
	    }
	  else if (strcmp ("path", l_option) == 0)
	    {
	      strcpy (g_entry_save_path, l_string);
	    }
	  else if (strcmp ("exit", l_option) == 0)
	    {
	      g_save_settings_exit = l_value;
	    }
        else if (strcmp ("beep", l_option) == 0)
          {
            g_beep = l_value;
          }
	  else if (strcmp ("showtooltips", l_option) == 0)
	    {
	      g_show_tooltips = l_value;
	    }
	  else if (strcmp ("testoverwrite", l_option) == 0)
	    {
	      g_test_overwrite = l_value;
	    }
	  else if (strcmp ("windowtitle", l_option) == 0)
	    {
            gtk_window_set_title (GTK_WINDOW (g_main_window), l_string);
	    }
	  else if (strcmp ("title", l_option) == 0)
	    {
            gtk_label_set_text(GTK_LABEL(g_plugin_title), l_string);
	    }
	  else if (strcmp ("subtitle", l_option) == 0)
	    {
            gtk_label_set_text(GTK_LABEL(g_plugin_subtitle), l_string);
	    }
	  else if (strcmp ("povfile", l_option) == 0)
	    {
            /* In plugin mode, one can't modify the pov file name  */
	      if (!g_plugin)
		{
		  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_pov_file), (gint) l_value);
		  gtk_entry_set_text (GTK_ENTRY (g_entry_pov_file), l_string);
		}
	      else if (g_pov_file_to_parse)
		{
		  gtk_entry_set_text (GTK_ENTRY (g_entry_pov_file), g_pov_file_to_parse);
		}
	    }
	  else if (strcmp ("sizeX1", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_xsize1), l_string);
	    }
	  else if (strcmp ("sizeY1", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_ysize1), l_string);
	    }
	  else if (strcmp ("sizeX2", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_xsize2), l_string);
	    }
	  else if (strcmp ("sizeY2", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_ysize2), l_string);
	    }
	  else if (strcmp ("size", l_option) == 0)
	    {
	      g_size_type = l_value;
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_radio_out_size [g_size_type]), TRUE);
	    }
	  else if (strcmp ("subrender", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_sub), (gint) l_value);
	    }
	  else if (strcmp ("substartX", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_5), l_value);
	    }
	  else if (strcmp ("subendX", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_6), l_value);
	    }
	  else if (strcmp ("substartY", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_7), l_value);
	    }
	  else if (strcmp ("subendY", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_8), l_value);
	    }
	  else if (strcmp ("previewstart", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_out_preview), (gint) l_value);
	      gtk_entry_set_text (GTK_ENTRY (g_entry_out_preview1), l_string);
	    }
	  else if (strcmp ("previewend", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_out_preview2), l_string);
	    }
	  else if (strcmp ("inioutput", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_inifile), (gint) l_value);
	      gtk_entry_set_text (GTK_ENTRY (g_entry_inifile), l_string);
	    }
	  else if (strcmp ("imageoutput", l_option) == 0)
	    {
            /* In plugin mode, one can't modify the image ouput  */
	      if (!g_plugin)
		{
		  gtk_entry_set_text (GTK_ENTRY (g_entry_outfile), l_string);
		}
	      else if (g_image_file_to_compute)
		{
		  gtk_entry_set_text (GTK_ENTRY (g_entry_outfile), g_image_file_to_compute);
		}
	    }
	  else if (strcmp ("grep", l_option) == 0)
	    {
            /* In plugin mode, one can't modify the redirection  */
	      if (!g_plugin)
		{
		  gtk_entry_set_text (GTK_ENTRY (g_entry_outgrep), l_string);
		}
	      else if (g_redirect_command_to_use)
		{
		  gtk_entry_set_text (GTK_ENTRY (g_entry_outgrep), g_redirect_command_to_use);
		}
	    }
        /* Chek if valid when in plugin mode  */
	  else if (strcmp ("format", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (g_combo_out_file)->entry), l_string);
	    }
	  else if (strcmp ("abort", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_out1), (gint) l_value);
	    }
	  else if (strcmp ("continue", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_out2), (gint) l_value);
	    }
	  else if (strcmp ("pause", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_out3), (gint) l_value);
	    }
	  else if (strcmp ("verbose", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_out4), (gint) l_value);
	    }
	  else if (strcmp ("drawvista", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_out5), (gint) l_value);
	    }
	  else if (strcmp ("outfilemode", l_option) == 0)
	    {
          /* Check if valid in plugin mode  */
	      g_out_file_mode = l_value;
	      if (l_value == 1)
		{
		  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_radio_out_file1), TRUE);
		}
	      else if (l_value == 2)
		{
		  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_radio_out_file2), TRUE);
		}
	      else if (l_value == 3)
		{
		  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_radio_out_file3), TRUE);
		}
	    }
	  else if (strcmp ("display", l_option) == 0)
	    {
	      strcpy (g_palette, l_string);
	      if (strcmp ("G", l_string) == 0)
		{
		  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_radio_out_color1), TRUE);
		}
	      else if (strcmp ("H", l_string) == 0)
		{
		  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_radio_out_color2), TRUE);
		}
	      else if (strcmp ("T", l_string) == 0)
		{
		  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_radio_out_color3), TRUE);
		}
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_out6), (gint) l_value);
	    }
	  else if (strcmp ("alphaout", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_out7), (gint) l_value);
	    }
	  else if (strcmp ("antialiasing", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_antialiasing), (gint) l_value);
	    }
	  else if (strcmp ("threshold", l_option) == 0)
	    {
	      gtk_adjustment_set_value(GTK_ADJUSTMENT (g_adjust_threshold), l_value);
	    }
	  else if (strcmp ("gamma", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_gamma), l_value);
	    }
	  else if (strcmp ("jitter", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_jitter), (gint) l_value);
	    }
	  else if (strcmp ("jittervalue", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_jitter), l_value);
	    }
	  else if (strcmp ("refresh", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_refresh), (gint) l_value);
	    }
	  else if (strcmp ("refreshvalue", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_refresh), l_value);
	    }
	  else if (strcmp ("methode", l_option) == 0)
	    {
	      strcpy (g_aa_methode, l_string);
	      if (strcmp ("1", l_string) == 0)
		{
		  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_radio_aa1), TRUE);
		}
	      else
		{
		  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_radio_aa2), TRUE);
		}
	    }
	  else if (strcmp ("comchoice", l_option) == 0) /* commande to invoque  */
	    {
		  g_invoque_command_number = (int) l_value;
	    }
	  else if (strcmp ("aadepth", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_depth), l_value);
	    }
	  else if (strcmp ("quality", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_quality), l_value);
	    }
	  else if (strcmp ("radiosity", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_quality1), (gint) l_value);
	    }
	  else if (strcmp ("boundingbox", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_quality2), (gint) l_value);
	    }
	  else if (strcmp ("boundingthreshold", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_bounding_threshold), l_value);
	    }
	  else if (strcmp ("lightbuffer", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_quality3), (gint) l_value);
	    }
	  else if (strcmp ("vistabuffer", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_quality4), (gint) l_value);
	    }
	  else if (strcmp ("removebounding", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_quality5), (gint) l_value);
	    }
	  else if (strcmp ("splitunion", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_quality6), (gint) l_value);
	    }
	  else if (strcmp ("fieldrender", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_anim_field), (gint) l_value);
	    }
	  else if (strcmp ("cyclic", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_anim_cyclic), (gint) l_value);
	    }
	  else if (strcmp ("oddfield", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_anim_odd), (gint) l_value);
	    }
	  else if (strcmp ("prescenecommand", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_pre_scene), l_string);
	    }
	  else if (strcmp ("preframecommand", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_pre_frame), l_string);
	    }
	  else if (strcmp ("postscenecommand", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_post_scene), l_string);
	    }
	  else if (strcmp ("postframecommand", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_post_frame), l_string);
	    }
	  else if (strcmp ("userabortcommand", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_user_abort), l_string);
	    }
	  else if (strcmp ("fatalerrorcommand", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_fatal_error), l_string);
	    }
	  else if (strcmp ("library1", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library [0]), l_string);
	    }
	  else if (strcmp ("library2", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library [1]), l_string);
	    }
	  else if (strcmp ("library3", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library [2]), l_string);
	    }
	  else if (strcmp ("library4", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library [3]), l_string);
	    }
	  else if (strcmp ("library5", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library [4]), l_string);
	    }
	  else if (strcmp ("library6", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library [5]), l_string);
	    }
	  else if (strcmp ("library7", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library [6]), l_string);
	    }
	  else if (strcmp ("library8", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library [7]), l_string);
	    }
	  else if (strcmp ("library9", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library [8]), l_string);
	    }
	  else if (strcmp ("library10", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library [9]), l_string);
	    }
	  else if (strcmp ("version", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_library_version), l_string);
	    }
	  else if (strcmp ("debugstream", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_debug_stream), (gint) l_value);
	      gtk_entry_set_text (GTK_ENTRY (g_entry_debug_stream), l_string);
	    }
	  else if (strcmp ("fatalstream", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_fatal_stream), (gint) l_value);
	      gtk_entry_set_text (GTK_ENTRY (g_entry_fatal_stream), l_string);
	    }
	  else if (strcmp ("renderstream", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_render_stream), (gint) l_value);
	      gtk_entry_set_text (GTK_ENTRY (g_entry_render_stream), l_string);
	    }
	  else if (strcmp ("statstream", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_stat_stream), (gint) l_value);
	      gtk_entry_set_text (GTK_ENTRY (g_entry_stat_stream), l_string);
	    }
	  else if (strcmp ("warningstream", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_warning_stream), (gint) l_value);
	      gtk_entry_set_text (GTK_ENTRY (g_entry_warning_stream), l_string);
	    }
	  else if (strcmp ("allstream", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_all_stream), (gint) l_value);
	      gtk_entry_set_text (GTK_ENTRY (g_entry_all_stream), l_string);
	    }
	  else if (strcmp ("extanimloop", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_anim_clock), l_string);
	    }
	  else if (strcmp ("initframe", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_anim_frame_ini), l_string);
	    }
	  else if (strcmp ("finalframe", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_anim_frame_fin), l_string);
	    }
	  else if (strcmp ("initclock", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_anim_clock_ini), l_string);
	    }
	  else if (strcmp ("finalclock", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_anim_clock_fin), l_string);
	    }
	  else if (strcmp ("initialsub", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_anim_subset_ini), l_string);
	    }
	  else if (strcmp ("finalsub", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_anim_subset_fin), l_string);
	    }
	  else if (strcmp ("outputfile", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_histo_file), (gint) l_value);
	      gtk_entry_set_text (GTK_ENTRY (g_entry_histo_file), l_string);
	    }
	  else if (strcmp ("histosizeX", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_history_xsize), l_string);
	    }
	  else if (strcmp ("histosizeY", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_history_ysize), l_string);
	    }
	  else if (strcmp ("histoformat", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (g_combo_histo_type)->entry), l_string);
	    }
	  else if (strcmp ("usepvm", l_option) == 0)
	    {
	      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_pvm), (gint) l_value);
	    }
	  else if (strcmp ("pvmnbrtasks", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_pvm_num), l_string);
	    }
	  else if (strcmp ("pvmcmd", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_pvm_cmd), l_string);
	    }
	  else if (strcmp ("pvmdir", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_pvm_dir), l_string);
	    }
	  else if (strcmp ("pvmheight", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_pvm_height), l_string);
	    }
	  else if (strcmp ("pvmwidth", l_option) == 0)
	    {
	      gtk_entry_set_text (GTK_ENTRY (g_entry_pvm_width), l_string);
	    }
	  else if (strcmp ("pvmhosts", l_option) == 0)
	    {
	      l_row_data = (gchar *) malloc (strlen (l_string) * sizeof (gchar));
	      strcpy (l_row_data, l_string);
	      l_row = gtk_clist_append (GTK_CLIST (g_gtklist_hosts), &l_row_data);
	      gtk_clist_set_row_data (GTK_CLIST (g_gtklist_hosts), l_row, l_row_data);
	    }
	  else if (strcmp ("pvmnice", l_option) == 0)
	    {
	      gtk_adjustment_set_value (GTK_ADJUSTMENT (g_adjust_pvm_nice), l_value);
	    }
  	  else
	    {
	      sprintf (l_message, "Line %d : %s (unrecognized word)\n", l_line_number, l_string);
	      povfront_error (l_message);
            l_nbr_error++;
            if (l_nbr_error == 10)  /* 10 errors maximum  */
              {
                povfront_message (_("Too much errors to continue\n"));
	          return 1;
              }
	    }
	}
      fclose (l_file_handle);
    }
  if (l_nbr_error)
    {
      return 1;
    }
  else
    {
      return 0;
    }
}
/*** END OF load_config_file */

/****************************************************************
 * Name : 		write_config_file
 * parameters : 
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/16/1999
 * Return :
 * Comments : Write a configuration file on the disk
 ****************************************************************/
gint
write_config_file (gchar *p_selected_file)
{
  FILE *l_conf_file_handler;
  gchar l_message[256];
  gint i;
  gchar *l_host;

  g_return_val_if_fail (p_selected_file != NULL, TRUE);

  l_conf_file_handler = fopen (p_selected_file, "w");
  if (l_conf_file_handler == NULL)
    {
      sprintf (l_message, _("ERROR: write_config: file %s cannot be opened\n"), p_selected_file);
      povfront_error (l_message);
      return 0;
    }

  fprintf (l_conf_file_handler, "command\t0\t%s\n", g_command_to_invoque);
  fprintf (l_conf_file_handler, "altcommand\t0\t%s\n", g_alt_command_to_invoque);
  fprintf (l_conf_file_handler, "comchoice\t%d\n", g_invoque_command_number);
  fprintf (l_conf_file_handler, "path\t0\t%s\n", g_entry_save_path);
  fprintf (l_conf_file_handler, "exit\t%d\n", g_save_settings_exit);
  fprintf (l_conf_file_handler, "beep\t%d\n", g_beep);
  fprintf (l_conf_file_handler, "showtooltips\t%d\n", g_show_tooltips);
  fprintf (l_conf_file_handler, "testoverwrite\t%d\n", g_test_overwrite);
  /* Input POV File  */
  /* I don't write file name in plugin mode  (check for g_pov_file)*/
  if (!g_plugin)
    {
      fprintf (l_conf_file_handler, "povfile\t%d\t%s\n", g_input_pov_file,
	       gtk_entry_get_text (GTK_ENTRY (g_entry_pov_file)));
    }
  /* Image size  */
  fprintf (l_conf_file_handler, "sizeX1\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY(g_entry_xsize1)));
  fprintf (l_conf_file_handler, "sizeY1\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY(g_entry_ysize1)));
  fprintf (l_conf_file_handler, "sizeX2\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY(g_entry_xsize2)));
  fprintf (l_conf_file_handler, "sizeY2\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY(g_entry_ysize2)));
  fprintf (l_conf_file_handler, "size\t%d\t\n", g_size_type);
  /* rendering Window flags  */
  fprintf (l_conf_file_handler, "subrender\t%d\n", g_sub_render);
  fprintf (l_conf_file_handler, "substartX\t%f\n", g_substartx_value);
  fprintf (l_conf_file_handler, "subendX\t%f\n", g_subendx_value);
  fprintf (l_conf_file_handler, "substartY\t%f\n", g_substarty_value);
  fprintf (l_conf_file_handler, "subendY\t%f\n", g_subendy_value);
  /* Preview flags  */
  fprintf (l_conf_file_handler, "previewstart\t%d\t%s\n", g_preview, gtk_entry_get_text (GTK_ENTRY(g_entry_out_preview1)));
  fprintf (l_conf_file_handler, "previewend\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY(g_entry_out_preview2)));
  /* Output .ini file  */
  fprintf (l_conf_file_handler, "inioutput\t%d\t%s\n", g_output_inifile, gtk_entry_get_text (GTK_ENTRY (g_entry_inifile)));
   /* Output image file  */
  /* In plugin mode, I don't write the image name  */
  if (!g_plugin)
    {
      fprintf (l_conf_file_handler, "imageoutput\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_outfile)));
    }
  /* Redirection command  */
  if (!g_plugin)
    {
      fprintf (l_conf_file_handler, "grep\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_outgrep)));
    }
  /* Image format  */
  fprintf (l_conf_file_handler, "format\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_out_file)->entry)));
  /* Test abort flag  */
  fprintf (l_conf_file_handler, "abort\t%d\n", g_test_abort);
  /* Continue flag  */
  fprintf (l_conf_file_handler, "continue\t%d\n", g_continue);
   /* Pause flag  */
  fprintf (l_conf_file_handler, "pause\t%d\n", g_pause);
  /* Verbose flag  */
  fprintf (l_conf_file_handler, "verbose\t%d\n", g_verbose);
  /* Draw vista flag  */
  fprintf (l_conf_file_handler, "drawvista\t%d\n", g_draw_vista);
  /* Display Image flag  */
  fprintf (l_conf_file_handler, "display\t%d\t%s\n", g_display_image, g_palette);
  /* output mode  */
  fprintf (l_conf_file_handler, "outfilemode\t%d\n", g_out_file_mode);
  /* Output Alpha flag  */
  fprintf (l_conf_file_handler, "alphaout\t%d\n", g_alpha_out);
  /* Antialiasing flag with threshold  */
  fprintf (l_conf_file_handler, "antialiasing\t%d\n", g_antialiasing);
  fprintf (l_conf_file_handler, "threshold\t%f\n", g_quality_threshold);
  /* Gamma  */
  fprintf (l_conf_file_handler, "gamma\t%f\n", g_quality_gamma);
  /* Jitter  */
  fprintf (l_conf_file_handler, "jitter\t%d\n", g_jitter);
  fprintf (l_conf_file_handler, "jittervalue\t%f\n", g_jitter_value);
  /* Refresh  */
  fprintf (l_conf_file_handler, "refresh\t%d\n", g_refresh);
  fprintf (l_conf_file_handler, "refreshvalue\t%f\n", g_refresh_value);
  /* Antialiasing methode and depth  */
  fprintf (l_conf_file_handler, "methode\t0\t%s\n", g_aa_methode);
  fprintf (l_conf_file_handler, "aadepth\t%f\n", g_quality_depth);
  /* Quality level  */
  fprintf (l_conf_file_handler, "quality\t%f\n", g_quality_value);
  /* Radiosity flag  */
  fprintf (l_conf_file_handler, "radiosity\t%d\n", g_radiosity);
  /* Bounding flag  */
  fprintf (l_conf_file_handler, "boundingbox\t%d\n", g_bounding_box);
  /* Bounding threshold  */
  fprintf (l_conf_file_handler, "boundingthreshold\t%d\n", (int) g_bounding_threshold_value);
  /* Light Buffer flag  */
  fprintf (l_conf_file_handler, "lightbuffer\t%d\n", g_light_buffer);
  /* Vista Buffer flag  */
  fprintf (l_conf_file_handler, "vistabuffer\t%d\n", g_vista_buffer);
  /* Remove bounds flag  */
  fprintf (l_conf_file_handler, "removebounding\t%d\n", g_remove_bounding);
  /* Split Union flag  */
  fprintf (l_conf_file_handler, "splitunion\t%d\n", g_split);
  /* Field render flag  */
  fprintf (l_conf_file_handler, "fieldrender\t%d\n", g_field_render);
  /* Cyclic animation flag  */
  fprintf (l_conf_file_handler, "cyclic\t%d\n", g_cyclic);
  /* Odd fieldflag  */
  fprintf (l_conf_file_handler, "oddfield\t%d\n", g_odd_field);
  /* pre scene command  */
  fprintf (l_conf_file_handler, "prescenecommand\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_pre_scene)));
  /* pre frame command  */
  fprintf (l_conf_file_handler, "preframecommand\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_pre_frame)));
  /* post scene command  */
  fprintf (l_conf_file_handler, "postscenecommand\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_post_scene)));
  /* Post frame command  */
  fprintf (l_conf_file_handler, "postframecommand\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_post_frame)));
  /* User abort command  */
  fprintf (l_conf_file_handler, "userabortcommand\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_user_abort)));
  /* Fatal Error command  */
  fprintf (l_conf_file_handler, "fatalerrorcommand\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_fatal_error)));
  /* Library path option  */
  for (i=0; i<10; i++)
    {
      fprintf (l_conf_file_handler, "library%d\t0\t%s\n", i+1, gtk_entry_get_text (GTK_ENTRY (g_entry_library [i])));
    }

  /* POV file version  */
  fprintf (l_conf_file_handler, "version\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_library_version)));

  /* Stream flag  */
  fprintf (l_conf_file_handler, "debugstream\t%d\t%s\n", g_debug_stream
	   ,gtk_entry_get_text (GTK_ENTRY (g_entry_debug_stream)));
  fprintf (l_conf_file_handler, "fatalstream\t%d\t%s\n", g_fatal_stream
	   ,gtk_entry_get_text (GTK_ENTRY (g_entry_fatal_stream)));
  fprintf (l_conf_file_handler, "renderstream\t%d\t%s\n", g_render_stream
	   ,gtk_entry_get_text (GTK_ENTRY (g_entry_render_stream)));
  fprintf (l_conf_file_handler, "statstream\t%d\t%s\n", g_stat_stream
	   ,gtk_entry_get_text (GTK_ENTRY (g_entry_stat_stream)));
  fprintf (l_conf_file_handler, "warningstream\t%d\t%s\n", g_warning_stream
	   ,gtk_entry_get_text (GTK_ENTRY (g_entry_warning_stream)));
  fprintf (l_conf_file_handler, "allstream\t%d\t%s\n", g_all_stream
	   ,gtk_entry_get_text (GTK_ENTRY (g_entry_all_stream)));

  /* External animation loop  */
  fprintf (l_conf_file_handler, "extanimloop\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_anim_clock)));
  /* Initial frame for internal animation  */
  fprintf (l_conf_file_handler, "initframe\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_anim_frame_ini)));
  /* Final frame for internal animation  */
  fprintf (l_conf_file_handler, "finalframe\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_anim_frame_fin)));
  /* Initial clock value for internal animation  */
  fprintf (l_conf_file_handler, "initclock\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_anim_clock_ini)));
  /* Final clock value for internal animation  */
  fprintf (l_conf_file_handler, "finalclock\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_anim_clock_fin)));
  /* Initial subset of frame for animation  */
  fprintf (l_conf_file_handler, "initialsub\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_anim_subset_ini)));
  /* Final subset of frame for animation  */
  fprintf (l_conf_file_handler, "finalsub\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_anim_subset_fin)));
  /* Histogram option  */
  fprintf (l_conf_file_handler, "outputfile\t%d\t%s\n", g_output_histo_file
	   ,gtk_entry_get_text (GTK_ENTRY (g_entry_histo_file)));
  fprintf (l_conf_file_handler, "histoformat\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_histo_type)->entry)));
  fprintf (l_conf_file_handler, "histosizeX\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_history_xsize)));
  fprintf (l_conf_file_handler, "histosizeY\t0\t%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_history_ysize)));
  /* PVM options  */
  fprintf (l_conf_file_handler, "usepvm\t%d\n",  g_pvm);
  fprintf (l_conf_file_handler, "pvmnbrtasks\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_pvm_num)));
  fprintf (l_conf_file_handler, "pvmcmd\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_pvm_cmd)));
  fprintf (l_conf_file_handler, "pvmdir\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_pvm_dir)));
  fprintf (l_conf_file_handler, "pvmheight\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_pvm_height)));
  fprintf (l_conf_file_handler, "pvmwidth\t0\t%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_pvm_width)));
  for (i=0; (l_host = (gchar *) gtk_clist_get_row_data (GTK_CLIST (g_gtklist_hosts), i)) != NULL; i++)
    {
      fprintf (l_conf_file_handler, "pvmhosts\t0\t%s\n",  l_host);
    }
  fprintf (l_conf_file_handler, "pvmnice\t%f\n",  g_pvm_nice_value);

  fclose (l_conf_file_handler);
  return 1;
}
/*** END OF write_config_file */

/****************************************************************
 * Name : 		write_default_config
 * parameters : Not used
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 02/28/1999
 * Return : FALSE
 * Comments : (callback) Write the default configuration file
 ****************************************************************/
gint
write_default_config (GtkWidget *p_widget, gpointer *p_data)
{
  write_config_file (g_default_config_file);
  return (FALSE);
}
/*** END OF write_default_config */

/****************************************************************
 * Name : 		save_config_ok
 * parameters : p_file_selection : The widget for selection of a file
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 02/28/1999
 * Return :
 * Comments : Callback for the ok button 
 ****************************************************************/
gint
save_config_ok (GtkWidget *p_widget, GtkFileSelection *p_file_selection)
{
  gchar *l_selected_file;

  g_return_val_if_fail (p_file_selection != NULL, TRUE);

  /* Get the selected file  */
  l_selected_file =  (gchar *)gtk_file_selection_get_filename (GTK_FILE_SELECTION (p_file_selection));
  if (write_config_file (l_selected_file))
    {
      strcpy (g_entry_save_path, l_selected_file);
    }
  gtk_widget_destroy (GTK_WIDGET (p_file_selection));
  return (FALSE);
}
/*** END OF SAVE_config_ok */

/****************************************************************
 * Name : 		save_config
 * parameters : 
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 02/28/1999
 * Return : FALSE
 * Comments : Open as selection file widget for saving config.
 ****************************************************************/
gint
save_config (GtkWidget *p_widget, gpointer p_data)
{
  GtkWidget *l_file_selection;

  l_file_selection = gtk_file_selection_new (_("Save configuration"));
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (l_file_selection), g_entry_save_path);
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION (l_file_selection)->ok_button), "clicked",
                   G_CALLBACK(save_config_ok), l_file_selection );
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(l_file_selection)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), l_file_selection);
  gtk_widget_show (l_file_selection);
  return (FALSE);
}
/*** END OF save_config */

/****************************************************************
 * Name : 		a_propos
 * parameters : 
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/8/1999
 * Return :
 * Comments : (callback) Display The about window
 ****************************************************************/
gint
a_propos (GtkWidget *widget, gpointer p_data)
{
  GtkWidget *l_button_close;
  GtkWidget *l_label;
  GtkWidget *l_separator;
  /*  To avoid multiple about window creation  */
  if (g_a_propos_window != NULL)
    {
      if (g_a_propos_window->window != NULL)
	{
	  gdk_window_show (g_a_propos_window->window);
	  gdk_window_raise (g_a_propos_window->window);
	  return (FALSE);
	}
    }
  g_a_propos_window = gtk_dialog_new ();
  gtk_container_set_border_width(GTK_CONTAINER (g_a_propos_window), 5);
  gtk_window_set_title (GTK_WINDOW (g_a_propos_window), _("PovFront: about ..."));

  l_label = gtk_label_new ("PovFront version 1.3.2");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_a_propos_window)->vbox), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("A front end for POV-Ray"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_a_propos_window)->vbox), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("This program is free software (GPL)"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_a_propos_window)->vbox), l_label, TRUE, TRUE, 0);

  l_separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_a_propos_window)->vbox), l_separator, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("Author: Philippe P.E. DAVID"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_a_propos_window)->vbox), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new ("Clovis1@club-internet.fr");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_a_propos_window)->vbox), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("If you use it, tell it to me by E-Mail"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_a_propos_window)->vbox), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("and make me some comments about PovFront"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_a_propos_window)->vbox), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("Look at http://perso.club-internet.fr/clovis1 for the last version"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_a_propos_window)->vbox), l_label, TRUE, TRUE, 0);

  l_button_close = gtk_button_new_with_label ("Close");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_a_propos_window)->action_area), l_button_close, FALSE, FALSE, 10);
  g_signal_connect_swapped(G_OBJECT(l_button_close), "clicked",
                           G_CALLBACK(gtk_widget_destroy), g_a_propos_window);
  /*gtk_widget_grab_default (l_button_close);*/
  gtk_window_set_wmclass (GTK_WINDOW (g_a_propos_window), "about", "PovFront");
  gtk_widget_show_all (g_a_propos_window);
  return (FALSE);
}
/*** END OF a_propos */

/****************************************************************
 * Name : 		povfront_message
 * parameters : p_data : The message to display
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 9/8/1999
 * Return :
 * Comments : Display a window with the message p_data
 ****************************************************************/
void
povfront_message (gchar *p_data)
{
  GtkWidget *l_window;
  GtkWidget *l_button_close;
  GtkWidget *l_label;

  g_return_if_fail (p_data != NULL);

  l_window = gtk_dialog_new ();
  gtk_container_set_border_width(GTK_CONTAINER(l_window), 5);
  gtk_window_set_title (GTK_WINDOW (l_window), "PovFront");

  l_label = gtk_label_new (p_data);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->vbox), l_label, TRUE, TRUE, 0);

  l_button_close = gtk_button_new_with_label (_("Close"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->action_area), l_button_close, FALSE, FALSE, 10);
  g_signal_connect_swapped(G_OBJECT(l_button_close), "clicked",
                           G_CALLBACK(gtk_widget_destroy), l_window);
  /*gtk_widget_grab_default (l_button_close);*/
  gtk_widget_show_all (l_window);
}
/*** END OF povfront_message */

/****************************************************************
 * Name : 		povfront_warning
 * parameters : p_data : The warning message to display
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 9/8/1999
 * Return : void
 * Comments : Display a window with a warning message
 ****************************************************************/
void
povfront_warning (gchar *p_data)
{
  GtkWidget *l_window;
  GtkWidget *l_button_close;
  GtkWidget *l_label;

  g_return_if_fail (p_data != NULL);

  l_window = gtk_dialog_new();
  gtk_container_set_border_width(GTK_CONTAINER(l_window), 5);
  gtk_window_set_title(GTK_WINDOW(l_window), "PovFront");

  l_label = gtk_label_new(p_data);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->vbox), l_label, TRUE, TRUE, 0);

  l_button_close = gtk_button_new_with_label (_("Close"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->action_area), l_button_close, FALSE, FALSE, 10);
  g_signal_connect_swapped(G_OBJECT(l_button_close), "clicked",
                           G_CALLBACK(gtk_widget_destroy), l_window);
  /*gtk_widget_grab_default (l_button_close);*/
  gtk_widget_show_all(l_window);
}
/*** END OF povfront_warning  */

/****************************************************************
 * Name : 		povfront_error
 * parameters : p_data : the error message to display
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 9/8/1999
 * Return :
 * Comments : Display a window with an error message
 ****************************************************************/
void
povfront_error (gchar *p_data)
{
  GtkWidget *l_window;
  GtkWidget *l_button_close;
  GtkWidget *l_label;

  g_return_if_fail (p_data != NULL);

  l_window = gtk_dialog_new ();
  gtk_container_set_border_width (GTK_CONTAINER (l_window), 5);
  gtk_window_set_title (GTK_WINDOW (l_window), "PovFront");

  l_label = gtk_label_new (p_data);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->vbox), l_label, TRUE, TRUE, 0);

  l_button_close = gtk_button_new_with_label (_("Close"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->action_area), l_button_close, FALSE, FALSE, 10);
  g_signal_connect_swapped(G_OBJECT(l_button_close), "clicked",
                           G_CALLBACK(gtk_widget_destroy), l_window);
  gtk_widget_show_all (l_window);
  /*gtk_widget_grab_default (l_button_close);*/
}
/*** END OF povfront_error  */

/****************************************************************
 * Name : 		gnome_overwrite
 * parameters : p_reply : the reply comming from the question window
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 9/4/1999
 * Return : void
 * Comments : Display the overwrite message and ask for confirmation
 *            It is a gnome only function
 ****************************************************************/
void
gnome_overwrite (gint p_reply, gpointer p_data)
{
  if (p_reply == 0)
    {
      render2 (NULL, NULL);
    }
}
/*** END OF gnome_overwrite */

/****************************************************************
 * Name : 		povfront_message_overwrite
 * parameters : p_data :  The file name to be overwritten
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 9/8/1999
 * Return : void
 * Comments : Display the overwrite message and ask for confirmation
 ****************************************************************/
void
povfront_message_overwrite (gchar *p_data)
{
  GtkWidget *l_window;
  GtkWidget *l_button_overwrite;
  GtkWidget *l_button_cancel;
  GtkWidget *l_label;

  g_return_if_fail (p_data != NULL);

  l_window = gtk_dialog_new ();
  gtk_container_set_border_width(GTK_CONTAINER(l_window), 5);
  gtk_window_set_title (GTK_WINDOW (l_window), "PovFront");

  l_label = gtk_label_new (p_data);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->vbox), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new ("Overwrite this file?");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->vbox), l_label, TRUE, TRUE, 0);

  l_button_overwrite = gtk_button_new_with_label (_("Overwrite"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->action_area), l_button_overwrite, FALSE, FALSE, 10);
  g_signal_connect(G_OBJECT(l_button_overwrite), "clicked",
                   G_CALLBACK(render2), NULL);
  g_signal_connect_swapped(G_OBJECT(l_button_overwrite), "clicked",
                           G_CALLBACK(gtk_widget_destroy), l_window);

  l_button_cancel = gtk_button_new_with_label (_("Cancel"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->action_area), l_button_cancel, FALSE, FALSE, 10);
  g_signal_connect_swapped(G_OBJECT(l_button_cancel), "clicked",
                           G_CALLBACK(gtk_widget_destroy), l_window);
  /*gtk_widget_grab_default (l_button_cancel);*/

  gtk_widget_show_all (l_window);
}
/*** END OF povfront_message_overwrite */

/****************************************************************
 * Name : 		render
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/4/1999
 * Return : FALSE
 * Comments : (callback) Make some test and then render
 ****************************************************************/
gint
render (GtkWidget *p_widget, gpointer p_data)
{
  gchar *l_image_file_name;
  FILE  *l_output_image_handle;

  /* If no display and no output to the disk  */
  if ((g_out_file_mode == 3) && (g_display_image == 0))
    {
      povfront_warning (_("You don't have a rendering display nor disk output!"));
    }
  /* Test overwrite of output images if necessary  */
  if (g_out_file_mode == 1) /* 1 is for output to disk  */
    {
      l_image_file_name = (gchar *)gtk_entry_get_text (GTK_ENTRY (g_entry_outfile));
      /* if (access (l_image_file_name, F_OK) == -1)  */
      l_output_image_handle = fopen (l_image_file_name, "r");
      if (l_output_image_handle != NULL)
      {
        fclose(l_output_image_handle); /* I don't want to read but to see if the file exist  */
        if (g_test_overwrite)
        {
	      /* Then the image already exist  */
	      if (!g_continue)
		{
		  povfront_message_overwrite (l_image_file_name);
		}
	      else
		{
		  render2 (NULL, NULL);
		}
	    }
	  else
	    {
	      render2 (NULL, NULL);
	    }
	}
      else
	{
	  if (g_continue)
	    {
	      povfront_warning (_("You specify the CONTINUE flag, but the file does not exist; rendering in any way"));
	    }
	  render2 (NULL, NULL);
	}
    }
  else
    {
      render2 (NULL, NULL);
    }
  return (FALSE);
}
/*** END OF render */

/****************************************************************
 * Name : 		render2
 * parameters :
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 08/22/1999
 * Return :
 * Comments : launch rendering after writing ini file in ~/.povfront/inifile
 ****************************************************************/
gint
render2 (GtkWidget *p_widget, gpointer p_data)
{
  gchar  l_command_line [1000];
  gchar *l_cursor;
  gchar *l_tampon;
  gchar *l_text;
  gchar *l_arg[ARG_MAX];
  gint   l_index_argument = 0;
  pid_t  l_pid_pov;
  job   *l_the_last_job;
  gchar  l_time_char[31];
  time_t l_job_time;
  struct tm *l_job_time_struct;
  gchar  l_ini_file_name[512];
  FILE  *l_ini_file_handle;
  gchar  l_sh_file_name[512];
  FILE  *l_sh_file_handle;
  gint   l_return;
  gint   i;
  gchar *l_host;

  /* Preparing the ini file contening the command to invoque script page  */
  sprintf (l_ini_file_name, "%s/.povfront/inifile", g_home_directory);
  l_ini_file_handle = fopen (l_ini_file_name, "w");
  if (l_ini_file_handle == NULL)
    {
      povfront_error (_("ERROR: opening inifile"));
	/* Why exit here and not return  */
      exit (1);
    }
  /* Header of the ini file  */
  fprintf (l_ini_file_handle, ";Generated by povfront Don't modify please\n");
  /* Script commands  */
  fprintf (l_ini_file_handle, "Pre_Scene_Command=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_pre_scene)));
  fprintf (l_ini_file_handle, "Pre_Frame_Command=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_pre_frame)));
  fprintf (l_ini_file_handle, "Post_Scene_Command=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_post_scene)));
  fprintf (l_ini_file_handle, "Post_Frame_Command=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_post_frame)));
  fprintf (l_ini_file_handle, "User_Abort_Command=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_user_abort)));
  fprintf (l_ini_file_handle, "Fatal_Error_Command=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_fatal_error)));
  /* Alpha out  */
  if (g_alpha_out) fprintf (l_ini_file_handle, "OutputAlpha=On\n");
  /* Histogram  */
  if (g_output_histo_file)
    {
      if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_histo_type)->entry)), "CSV") == 0)
	{
	  fprintf (l_ini_file_handle, "Histogram_Type=C\n");
	}
      else if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_histo_type)->entry)), "PNG") == 0)
	{
	  fprintf (l_ini_file_handle, "Histogram_Type=N\n");
	}
      else if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_histo_type)->entry)), "PPM") == 0)
	{
	  fprintf (l_ini_file_handle, "Histogram_Type=P\n");
	}
      else if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_histo_type)->entry)), "BMP") == 0)
	{
	  fprintf (l_ini_file_handle, "Histogram_Type=S\n");
	}
      else if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_histo_type)->entry)), "TGA") == 0)
	{
	  fprintf (l_ini_file_handle, "Histogram_Type=T ; TARGA\n");
	}
      fprintf (l_ini_file_handle, "Histogram_Grid_Size=%s.%s\nHistogram_Name=%s\n",
	       gtk_entry_get_text (GTK_ENTRY (g_entry_history_xsize)),
	       gtk_entry_get_text (GTK_ENTRY (g_entry_history_ysize)),
	       gtk_entry_get_text (GTK_ENTRY (g_entry_histo_file)));
    }

  /* Input POV File  */
  if (g_input_pov_file || g_plugin)
    {
      fprintf (l_ini_file_handle, "+I%s \n", gtk_entry_get_text (GTK_ENTRY (g_entry_pov_file)));
    }

  /* Image size  */
  if (g_size_type == 8)
    {
      fprintf (l_ini_file_handle, "+W%s\n", gtk_entry_get_text (GTK_ENTRY(g_entry_xsize1)));
      fprintf (l_ini_file_handle, "+H%s\n" ,gtk_entry_get_text (GTK_ENTRY(g_entry_ysize1)));
    }
  else
    {
      if (g_size_type == 9)
	{
	  fprintf (l_ini_file_handle, "+W%s\n", gtk_entry_get_text (GTK_ENTRY(g_entry_xsize2)));
	  fprintf (l_ini_file_handle, "+H%s\n" ,gtk_entry_get_text (GTK_ENTRY(g_entry_ysize2)));
	}
      else
	{
	  fprintf (l_ini_file_handle, "+W%d\n+H%d\n",
		   g_predef_size [g_size_type] [0],
		   g_predef_size [g_size_type] [1]);
	}
    }
  
  /* Rendering Window flags  */
  if (g_sub_render)
    {
      fprintf (l_ini_file_handle, "+SC%f\n", g_substartx_value);
      fprintf (l_ini_file_handle, "+EC%f\n" , g_subendx_value);
      fprintf (l_ini_file_handle, "+SR%f\n" , g_substarty_value);
      fprintf (l_ini_file_handle, "+ER%f\n" , g_subendy_value);
    }

  /* Preview flags  */
  if (g_preview)
    {
      fprintf (l_ini_file_handle, "+SP%s\n", gtk_entry_get_text (GTK_ENTRY(g_entry_out_preview1)));
      fprintf (l_ini_file_handle, "+EP%s\n", gtk_entry_get_text (GTK_ENTRY(g_entry_out_preview2)));
    }

  /* Output ini file  */
  if (g_output_inifile)
    {
      if (strcmp (gtk_entry_get_text (GTK_ENTRY (g_entry_inifile)), "") == 0)
	{
	  fprintf (l_ini_file_handle, "Create_Ini=on\n");
	}
      else
	{
	  fprintf (l_ini_file_handle, "Create_Ini=%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_inifile)));
	}
    }
  else
    {
      fprintf (l_ini_file_handle, "Create_Ini=off\n");
    }

  /* Output image file  */
  if (g_out_file_mode ==1)
    {
      if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_out_file)->entry)), "CSV") == 0)
	{
	  fprintf (l_ini_file_handle, "+FC\n");
	}
      else if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_out_file)->entry)), "PNG") == 0)
	{
	  fprintf (l_ini_file_handle, "+FN\n");
	}
      else if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_out_file)->entry)), "PPM") == 0)
	{
	  fprintf (l_ini_file_handle, "+FP\n");
	}
      else if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_out_file)->entry)), "BMP") == 0)
	{
	  fprintf (l_ini_file_handle, "+FS\n");
	}
      else if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (g_combo_out_file)->entry)), "TGA") == 0)
	{
	  fprintf (l_ini_file_handle, "+FT\n");
	}
      fprintf (l_ini_file_handle, "Output_File_Name=%s\n",  gtk_entry_get_text (GTK_ENTRY (g_entry_outfile)));
    }
  else if (g_out_file_mode ==2) /* Output to the standard output  */
    {
      fprintf (l_ini_file_handle, "Output_File_Name=-\n");
    }
  else if (g_out_file_mode ==3)
    {
      fprintf (l_ini_file_handle, "-F\n"); /* Set to no output at all  */
    }

  /* Test abort flag  */
  if (g_test_abort)
    {
      fprintf (l_ini_file_handle, "+X\n");
    }
  else
    {
      fprintf (l_ini_file_handle, "-X\n");
    }

  /* Continue flag  */
  if (g_continue) fprintf (l_ini_file_handle, "+C\n");
  else fprintf (l_ini_file_handle, "-C\n");

  /* Pause flag  */
  if (g_pause) fprintf (l_ini_file_handle, "+P \n");
  else fprintf (l_ini_file_handle, "-P \n");

  /* Verbose flag  */
  if (g_verbose) fprintf (l_ini_file_handle, "+V \n");
  else fprintf (l_ini_file_handle, "-V \n");

  /* Draw vista flag  */
  if (g_draw_vista) fprintf (l_ini_file_handle, "+UD \n");
  else fprintf (l_ini_file_handle, "-UD \n");

  /* Display Image flag  */
  if (g_display_image) fprintf (l_ini_file_handle, "+D0%s \n", g_palette);
  else fprintf (l_ini_file_handle, "-D0%s \n", g_palette);

  /* Antialiasing flag with threshold  */
  if (g_antialiasing) fprintf (l_ini_file_handle, "+A%f \n", g_quality_threshold);
  else fprintf (l_ini_file_handle, "-A%f \n", g_quality_threshold);

  /* Gamma  */
  fprintf (l_ini_file_handle, "Display_Gamma=%f \n", g_quality_gamma);

  /* Jitter  */
  if (g_jitter) fprintf (l_ini_file_handle, "+J%f \n", g_jitter_value);
  else fprintf (l_ini_file_handle, "-J%f \n",  g_jitter_value);

  /* Antialiasing methode  */
  fprintf (l_ini_file_handle, "+AM%s\n", g_aa_methode);
  fprintf (l_ini_file_handle, "+R%d\n", (gint) g_quality_depth);

  /* Quality level  */
  fprintf (l_ini_file_handle, "+Q%d \n", (gint) g_quality_value);

  /* Radiosity flag  */
  if (g_radiosity) fprintf (l_ini_file_handle, "+QR \n");
  else fprintf (l_ini_file_handle, "-QR \n");

/*** why */
  /* Bounding flag  */
  if (g_bounding_box) fprintf (l_ini_file_handle, "+MB%d \n", (int) g_bounding_threshold_value);
  else fprintf (l_ini_file_handle, "-MB%d \n", (int) g_bounding_threshold_value);

  /* Light Buffer flag  */
  if (g_light_buffer) fprintf (l_ini_file_handle, "+UL \n");
  else fprintf (l_ini_file_handle, "-UL \n");

  /* vista Buffer flag  */
  if (g_vista_buffer) fprintf (l_ini_file_handle, "+UV \n");
  else fprintf (l_ini_file_handle, "-UV \n");

  /* Remove bounds flag  */
  if (g_remove_bounding) fprintf (l_ini_file_handle, "+UR \n");
  else fprintf (l_ini_file_handle, "-UR \n");

  /* Split Union flag  */
  if (g_remove_bounding) fprintf (l_ini_file_handle, "+SU \n");
  else fprintf (l_ini_file_handle, "-SU \n");

  /* Field render flag  */
  if (g_field_render) fprintf (l_ini_file_handle, "+UF \n");
  else fprintf (l_ini_file_handle, "-UF \n");

  /* Odd field flag  */
  if (g_odd_field) fprintf (l_ini_file_handle, "+UO \n");
  else fprintf (l_ini_file_handle, "-UO \n");

  /* Library path option  */
  for (i=0; i<10; i++)
    {
      l_tampon = (gchar *)gtk_entry_get_text (GTK_ENTRY (g_entry_library [i]));
      if (strlen (l_tampon) != 0)
	{
	  fprintf (l_ini_file_handle, "+L%s \n", l_tampon);
	}
    }
  /* POV file version  */
  l_tampon = (gchar  *)gtk_entry_get_text (GTK_ENTRY (g_entry_library_version));
  if (strlen (l_tampon) != 0)
    {
      fprintf (l_ini_file_handle, "+MV%s \n", l_tampon);
    }

  /* Stream flag  */
  if (g_debug_stream) fprintf (l_ini_file_handle, "-GD%s \n", gtk_entry_get_text (GTK_ENTRY (g_entry_debug_stream)));
  else fprintf (l_ini_file_handle, "-GD \n");

  if (g_fatal_stream) fprintf (l_ini_file_handle, "-GF%s \n", gtk_entry_get_text (GTK_ENTRY (g_entry_fatal_stream)));
  else fprintf (l_ini_file_handle, "-GF \n");

  if (g_render_stream) fprintf (l_ini_file_handle, "-GR%s \n", gtk_entry_get_text (GTK_ENTRY (g_entry_render_stream)));
  else fprintf (l_ini_file_handle, "-GR \n");

  if (g_stat_stream) fprintf (l_ini_file_handle, "-GS%s \n", gtk_entry_get_text (GTK_ENTRY (g_entry_stat_stream)));
  else fprintf (l_ini_file_handle, "-GS \n");

  if (g_warning_stream) fprintf (l_ini_file_handle, "-GW%s \n", gtk_entry_get_text (GTK_ENTRY (g_entry_warning_stream)));
  else fprintf (l_ini_file_handle, "-GW \n");

  if (g_all_stream) fprintf (l_ini_file_handle, "-GA%s \n", gtk_entry_get_text (GTK_ENTRY (g_entry_all_stream)));
  else fprintf (l_ini_file_handle, "-GA \n");

  /* External animation loop  */
  if ((l_text = (gchar *)gtk_entry_get_text (GTK_ENTRY (g_entry_anim_clock))) != NULL)
    {
      fprintf (l_ini_file_handle, "+K%s \n",  l_text);
    }

  /* Initial frame for internal animation  */
  if ((l_text = (gchar *)gtk_entry_get_text (GTK_ENTRY (g_entry_anim_frame_ini))) != NULL)
    {
      fprintf (l_ini_file_handle, "+KFI%s \n",  l_text);
    }

  /* Final frame for internal animation  */
  if ((l_text = (gchar *)gtk_entry_get_text (GTK_ENTRY (g_entry_anim_frame_fin))) != NULL)
    {
      fprintf (l_ini_file_handle, "+KFF%s \n",  l_text);
    }

  /* Initial clock value for internal animation  */
  if ((l_text = (gchar *)gtk_entry_get_text (GTK_ENTRY (g_entry_anim_clock_ini))) != NULL)
    {
      fprintf (l_ini_file_handle, "+KI%s \n",  l_text);
    }

  /* Final clock value for internal animation  */
  if ((l_text = (gchar *)gtk_entry_get_text (GTK_ENTRY (g_entry_anim_clock_fin))) != NULL)
    {
      fprintf (l_ini_file_handle, "+KF%s \n",  l_text);
    }

  /* Initial subset of frame for animation  */
  if ((l_text = (gchar *)gtk_entry_get_text (GTK_ENTRY (g_entry_anim_subset_ini))) != NULL)
    {
      fprintf (l_ini_file_handle, "+SF%s \n",  l_text);
    }

  /* Final subset of frame for animation  */
  if ((l_text = (gchar *)gtk_entry_get_text (GTK_ENTRY (g_entry_anim_subset_fin))) != NULL)
    {
      fprintf (l_ini_file_handle, "+EF%s \n",  l_text);
    }

  /* PVM options  */
  /* pvm=on if PVM is used but nothing if PVM is not used (so no pvm=off)
     because I want the program to be used without PVM  */
  if (g_pvm)
    {
      fprintf (l_ini_file_handle, "pvm=on\n");
      fprintf (l_ini_file_handle, "pvm_tasks=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_pvm_num)));
      fprintf (l_ini_file_handle, "pvm_nice=%d\n", (gint) g_pvm_nice_value);
      fprintf (l_ini_file_handle, "pvm_width=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_pvm_width)));
      fprintf (l_ini_file_handle, "pvm_height=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_pvm_height)));
      fprintf (l_ini_file_handle, "pvm_slave=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_pvm_cmd)));
      fprintf (l_ini_file_handle, "pvm_wd=%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_pvm_dir)));
      fprintf (l_ini_file_handle, "pvm_hosts=");
      for (i=0; (l_host = (gchar *) gtk_clist_get_row_data (GTK_CLIST (g_gtklist_hosts), i)) != NULL; i++)
	{
	  if (i) fprintf (l_ini_file_handle, ",");
	  fprintf (l_ini_file_handle, "%s\n",  l_host);
	}
    }
  fclose (l_ini_file_handle);

  /* Choice of the right command to invoque  */
  sprintf (l_sh_file_name, "%s/.povfront/process", g_home_directory);
  l_sh_file_handle = fopen (l_sh_file_name, "w");
  if (l_sh_file_handle == NULL)
    {
      povfront_error (_("ERROR: Unable to open the process file\n"));
      return (FALSE);
    }
  fprintf (l_sh_file_handle, "#! /bin/sh\n");
  if (g_invoque_command_number == 1)
    {
      sprintf (l_command_line, "%s ", g_command_to_invoque);
      l_arg [l_index_argument++] = g_command_to_invoque;
      fprintf (l_sh_file_handle, "%s ", g_command_to_invoque);
    }
  else
    {
      sprintf (l_command_line, "%s ", g_alt_command_to_invoque);
      l_arg [l_index_argument++] = g_alt_command_to_invoque;
      fprintf (l_sh_file_handle, "%s ", g_alt_command_to_invoque);
    }
  l_cursor = l_command_line + strlen (l_command_line);

  /* Add the ini file to parse on the command line  */
  sprintf (l_cursor, "%s ", l_ini_file_name);
  l_arg [l_index_argument++] = l_cursor;
  l_cursor += strlen (l_cursor) + 1;
  fprintf (l_sh_file_handle, "%s ", l_ini_file_name);
  l_arg [l_index_argument] = NULL;
  
  /* Add the redirection command if necessary  */
  if (g_out_file_mode == 2)
    {
      fprintf (l_sh_file_handle, "%s\n", gtk_entry_get_text (GTK_ENTRY (g_entry_outgrep)));
    }
  fclose (l_sh_file_handle);
  if ( chmod (l_sh_file_name, S_IXUSR | S_IRUSR | S_IWUSR) == -1)
    {
      povfront_error (_("chmod on the process file impossible!"));
      return (FALSE);
    }

  l_pid_pov = fork ();
  if (l_pid_pov == (pid_t) 0)
    {
      if (g_out_file_mode == 2)
	{
	  /* I launch the $HOME/.povfront/process script  */
	  l_return = execve (l_sh_file_name, NULL, g_pf_environ);
	}
      else
	{
	  l_return = execvp (l_arg[0], l_arg);
	}
      if (l_return == -1)
	{
        /* povfront_error does'nt work int the sohn process  */
	  printf (_("Cannot launch rendering\n"));
	  _exit (1);
        
	}
      else
	{
        /* why it is WORKING */
	  _exit (1);
	}
    }
  else
    {
      if (l_pid_pov == -1)
	{
	  povfront_error (_("Not enough memory to launch a new process\n"));
	  return (FALSE);
	}
      else
	{
	  g_pid_last_process = l_pid_pov;
	  l_the_last_job = (job *) malloc (sizeof (job));
	  if (l_the_last_job == NULL)
	    {
	      povfront_error (_("Cannot allocate memory"));
	      return (FALSE);
	    }
	  else
	    {
	      l_job_time = time (NULL);
	      l_job_time_struct = gmtime (&l_job_time);
	      strftime (l_time_char, (size_t) 30, "%a %X", l_job_time_struct);
	      /*l_time_char =  ctime (&l_job_time);*/
	      l_the_last_job->time = (gchar *) malloc (strlen (l_time_char));
	      strcpy (l_the_last_job->time, l_time_char);
	      l_the_last_job->pid = l_pid_pov;
	      l_the_last_job->state = POVF_RUNNING;
	      l_the_last_job->file = malloc (strlen (gtk_entry_get_text (GTK_ENTRY (g_entry_pov_file))));
	      if (l_the_last_job->file == NULL)
		{
		  povfront_error (_("Cannot allocate memory"));
		  return (FALSE);
		}
	      else
		{
		  strcpy (l_the_last_job->file, gtk_entry_get_text (GTK_ENTRY (g_entry_pov_file)));
		}
	      l_the_last_job->image = malloc (strlen (gtk_entry_get_text (GTK_ENTRY (g_entry_outfile))));
	      if (l_the_last_job == NULL)
		{
		  povfront_error (_("ERROR: Cannot allocate memory"));
		  return (FALSE);
		}
	      else
		{
		  strcpy (l_the_last_job->image, gtk_entry_get_text (GTK_ENTRY (g_entry_outfile)));
		}
	      g_jobs_list = g_list_append (g_jobs_list, l_the_last_job);
	    }
	}
    }
  return (FALSE);
}
/*** END OF render2  */

/****************************************************************
 * Name : 		pf_abort
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 03/20/1999
 * Return :
 * Comments : Abort the last job launched
 ****************************************************************/
gint
pf_abort (GtkWidget *p_widget, gpointer p_data)
{
  if (kill (g_pid_last_process, SIGKILL) != 0)
    {
      povfront_error (_("BUG: Cannot kill the process"));
    }
  return (FALSE);
}
/*** END OF pf_abort */

/****************************************************************
 * Name : 		help
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 05/13/1999
 * Return :
 * Comments :
 ****************************************************************/
gint
help (GtkWidget *p_widget, gpointer p_data)
{
/*
  if (g_help_window != NULL)
    {
      gdk_window_show (g_help_window->window);
      gdk_window_raise (g_help_window->window);
      return (FALSE);
    }
*/
  povfront_message (_("Help : look at http://perso.club-internet.fr/clovis1\n"));
  return (FALSE);
}
/*** END OF help */

/****************************************************************
 * Name : 		refresh
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 03/27/1999
 * Return :
 * Comments :
 ****************************************************************/
gint
refresh (GtkWidget *p_widget, void *p_gtklist)
{
  refresh_list (g_gtklist_jobs);
  return (FALSE);
}
/* END OF refresh */

/****************************************************************
 * Name : 		control_button_up
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/8/1999
 * Return : FALSE
 * Comments : Let the user activate some button of the job control
 *            It is called when the user select a line in the job list
 ****************************************************************/

gint
control_button_up (GtkWidget *p_widget, void *p_data)

{
/*
  gtk_widget_set_sensitive (g_control_stop_button, TRUE);
  gtk_widget_set_sensitive (g_control_continue_button, TRUE);
  gtk_widget_set_sensitive (g_control_abort_button, TRUE);
  gtk_widget_set_sensitive (g_control_delete_button, TRUE);
*/
  return (FALSE);

}


/* END OF control_button_up */

/****************************************************************
 * Name : 		control_button_down
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/8/1999
 * Return : FALSE
 * Comments : dim the buttons in the job control (to be called by refresh)
 ****************************************************************/

gint
control_button_down (void)
{
/*
  gtk_widget_set_sensitive (g_control_stop_button, FALSE);
  gtk_widget_set_sensitive (g_control_continue_button, FALSE);
  gtk_widget_set_sensitive (g_control_abort_button, FALSE);
  gtk_widget_set_sensitive (g_control_delete_button, FALSE);
*/
  return (FALSE);

}

/* END OF control_button_down */

/****************************************************************
 * Name : 		control_on
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/8/1999
 * Return :
 * Comments : Show the control window
 ****************************************************************/
gint
control_on (GtkWidget *p_widget, void *p_data)
{
  gtk_widget_show (g_control_window);
  gdk_window_raise (g_control_window->window);
  return (FALSE);
}
/* END OF control_on */

/****************************************************************
 * Name : 		control_off
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/4/1999
 * Return :
 * Comments : Show the control window
 ****************************************************************/
gint
control_off (GtkWidget *p_widget, void *p_data)
{
  gtk_widget_hide (p_data);
  return (FALSE);
}
/* END OF control_off */

/****************************************************************
 * Name : 		switch_refresh
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/4/1999
 * Return : FALSE
 * Comments : Switch between auto refresh and manual refresh
 ****************************************************************/
gint
switch_refresh (GtkWidget *p_widget, void *p_data)
{
  if (g_refresh)
    {
      g_timeout_tag = gtk_timeout_add (1000*g_refresh_value, (GtkFunction) refresh_list ,g_gtklist_jobs);
    }
  else
    {
      gtk_timeout_remove (g_timeout_tag);
    }
  return (FALSE);
}
/* END OF switch_refresh */

/****************************************************************
 * Name : 		adjust_refresh_time
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/4/1999
 * Return : FALSE
 * Comments : Call back for the refresh scale
 ****************************************************************/
gint
adjust_refresh_time (GtkWidget *p_widget, void *p_data)
{
  if (g_refresh)
    {
      gtk_timeout_remove (g_timeout_tag);
      g_timeout_tag = gtk_timeout_add (1000*g_refresh_value, (GtkFunction) refresh_list ,g_gtklist_jobs);
    }
  return (FALSE);
}
/* END OF adjust_refresh_time */

/****************************************************************
 * Name : 		refresh_list
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/04/1999
 * Return : FALSE
 * Comments : Update the list of process in the job control window
 ****************************************************************/
gint
refresh_list (GtkWidget *p_gtklist)
{
  job *l_a_job;
  gchar *l_row [5];
  gint i;
  gint l_line_number = 0;
  gint l_status;
  gint l_list_length;
  GList *l_jobs_list;
  GdkColor l_red_color;
  GdkColor l_gris_color;
  GdkColor l_yellow_color;

  if (p_gtklist == NULL)
    {
      return (TRUE);
    }

  l_red_color.pixel = 65535;
  l_red_color.red = 65500;
  l_red_color.green = 0;
  l_red_color.blue = 0;

  l_gris_color.pixel = 65535;
  l_gris_color.red = 32000;
  l_gris_color.green = 32000;
  l_gris_color.blue = 32000;

  l_yellow_color.pixel = 65535;
  l_yellow_color.red = 65535;
  l_yellow_color.green = 65535;
  l_yellow_color.blue = 0;

  l_jobs_list = g_jobs_list;         /* Use a local pointer for the list  */
  gtk_clist_clear (GTK_CLIST (p_gtklist));
  if (l_jobs_list != NULL)
    {
      l_jobs_list = g_list_first (l_jobs_list);
      l_list_length = g_list_length (l_jobs_list);
      /*** If there is no selection, I dim the button in the window */
      if (l_list_length == 0)
        {
          control_button_down ();
        }
      for (i=1; i <= l_list_length; i++)
	{
	  l_a_job = l_jobs_list->data;

	  l_row [0] = "";  /* Setting default values  */
	  l_row [2] = l_a_job->time;
	  l_row [3] = l_a_job->file;
	  l_row [4] = l_a_job->image;
	  l_status = 1; /* l_status must be != 0 to receive informations from waitpid  */

	  if (l_a_job->state == POVF_ERROR)
	    {
	      l_row [1] = _("Error");
	    }
	  else if (l_a_job->state == POVF_DONE)
	    {
	      l_row [1] = _("Done");
	    }
	  else if (l_a_job->state == POVF_ABORTED)
	    {
	      l_row [1] = _("Aborted");
	    }
	  else if (l_a_job->state == POVF_HALTED)
	    {
	      l_row [1] = _("Halted"); /* This state is refresh by the continue call back  */
	      l_row [0] = malloc (15);
	      sprintf (l_row [0], "%d", (int) l_a_job->pid);
	    }
	  /* We just refresh some process (running and finished) */
	  else if (waitpid (l_a_job->pid, &l_status, WNOHANG|WUNTRACED) == -1)
	    {
	      if (errno == ECHILD)
		{
		  if (l_a_job->state == POVF_RUNNING)
		    {
		      l_a_job->state = POVF_ABORTED;
		      l_row [1] = _("Aborted");
		    }
		  else
		    {
		      l_a_job->state = POVF_DONE;
		      l_row [1] = _("Done");
		    }
		}
	      else
		{
		  /* I don't know  */
		}
	    }
	  else /* The job is running or is a zombie  */
	    {
	      if (WIFEXITED (l_status))
		{
		  if (WEXITSTATUS (l_status) == 0) /* 0 is the return code of the process in case of success  */
		    {
		      l_a_job->state = POVF_FINISHED;
		      l_row [1] = _("Finished");
		      gdk_beep ();
		    }
		  else
		    {
		      l_a_job->state = POVF_ERROR;
		      l_row [1] = _("Error");
		    }
		}
	      else if (WIFSTOPPED (l_status))
		{
		  l_a_job->state = POVF_HALTED;
		  l_row [1] = _("Halted");
		  l_row [0] = malloc (15);
		  sprintf (l_row [0], "%d", (int) l_a_job->pid);
		}
	      else
		{
		  l_a_job->state = POVF_RUNNING;
		  l_row [1] = _("Running");
		  l_row [0] = malloc (15);
		  sprintf (l_row [0], "%d", (int) l_a_job->pid);
		}
	    }
	  gtk_clist_append (GTK_CLIST (p_gtklist), l_row);
	  if (l_a_job->state == POVF_ERROR)
	    {
	      gtk_clist_set_foreground (GTK_CLIST (p_gtklist), l_line_number, &l_red_color);
	    }
	  else if (l_a_job->state == POVF_DONE)
	    {
	      gtk_clist_set_foreground (GTK_CLIST (p_gtklist), l_line_number, &l_gris_color);
	    }
	  else if (l_a_job->state == POVF_ABORTED)
	    {
	      gtk_clist_set_foreground (GTK_CLIST (p_gtklist), l_line_number, &l_gris_color);
	    }
	  else if (l_a_job->state == POVF_HALTED)
	    {
	      gtk_clist_set_foreground (GTK_CLIST (p_gtklist), l_line_number, &l_yellow_color);
	    }
	  l_line_number++;
	  g_free (l_row [0]);
	  l_jobs_list = g_list_next (l_jobs_list);
	}
    }
  return (TRUE);
}
/*** END OF refresh_list  */

/****************************************************************
 * Name : 		stop_jobs
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 9/5/1999
 * Return :
 * Comments :
 ****************************************************************/
void
stop_jobs (GtkObject *p_jobs_list)
{
  GList *l_list;
  guint l_data;
  gint i, l_length;
  GList *l_process;
  pid_t l_process_pid;
  job *l_a_job;

  l_list = GTK_CLIST (g_gtklist_jobs)->selection;
  l_list = g_list_first (l_list);
  l_length = g_list_length (l_list);
  for (i=1; i<= l_length; i++)
    {
      l_data = (guint) l_list->data;
      l_process = (GList *) g_list_nth (g_jobs_list, l_data);
      l_a_job = (job *) l_process->data;
      l_process_pid = l_a_job->pid;
      if (kill (l_process_pid, SIGSTOP) == -1)
	{
	  povfront_error (_("ERROR: Unable to send the signal to the proc."));
	}
      l_list = g_list_next (l_list);
    }
}
/*** END OF stop_jobs  */

/****************************************************************
 * Name : 		continue_jobs
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/5/1999
 * Return :
 * Comments :
 ****************************************************************/
void
continue_jobs (GtkObject *p_jobs_list)
{
  GList *l_list;
  guint l_data;
  gint i, l_length;
  GList *l_process;
  pid_t l_process_pid;
  job *l_a_job;

  l_list = GTK_CLIST (g_gtklist_jobs)->selection;
  l_list = g_list_first (l_list);
  l_length = g_list_length (l_list);
  for (i=1; i<= l_length; i++)
    {
      l_data = (guint) l_list->data;
      l_process = (GList *) g_list_nth (g_jobs_list, l_data);
      l_a_job = (job *) l_process->data;
      l_process_pid = l_a_job->pid;
      if (kill (l_process_pid, SIGCONT) == -1)
	{
	  povfront_error (_("ERROR: Unable to send the signal to the proc."));
	}
      else
	{
	  l_a_job->state = POVF_RUNNING;
	}
      l_list = g_list_next (l_list);
    }
}
/*** END OF continue_jobs  */

/****************************************************************
 * Name : 		kill_jobs
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/5/1999
 * Return :
 * Comments :
 ****************************************************************/
void
kill_jobs (GtkWidget *p_widget, void *p_jobs_list)
{
  GList *l_list;
  guint l_data;
  gint i, l_length;
  GList *l_process;
  pid_t l_process_pid;
  job *l_a_job;

  l_list = GTK_CLIST (g_gtklist_jobs)->selection;
  l_list = g_list_first (l_list);
  l_length = g_list_length (l_list);
  for (i=1; i<= l_length; i++)
    {
      l_data = (guint) l_list->data;
      l_process = (GList *) g_list_nth (g_jobs_list, l_data);
      l_a_job = (job *) l_process->data;
      l_process_pid = l_a_job->pid;
      if (kill (l_process_pid, SIGKILL) == -1)
	{
	  povfront_error (_("ERROR: Unable to kill the proc."));
	}
      else
	{
	  l_a_job->state = POVF_FINISHED;
	}
      l_list = g_list_next (l_list);
    }
}
/*** END OF kill_jobs  */

/****************************************************************
 * Name : 		greater_than
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 04/03/1999
 * Return :
 * Comments : Used to sort the list of process
 ****************************************************************/
gint
greater_than (gint *p_a, gint *p_b)
{
  if (*p_a > *p_b) return 1;
  else return 0;
}
/*** END OF greater_than  */

/****************************************************************
 * Name : 		delete_jobs
 * parameters :
 * Author : Philippe DAVID
 * Update (dd/mm/yy) : 09/5/1999
 * Return :
 * Comments : Delete  the selected jobs in the list from the job list
 ****************************************************************/
void
delete_jobs (GtkWidget *p_widget, void *p_jobs_list)
{
  GList *l_list;
  GList *l_process;
  GList *l_copy_of_list = NULL;  /* Usefull because gtk_clist_remove reset the selection list  */
  job *l_a_job;
  guint l_process_state;  /* State of the process to delete from the list  */
  guint *l_data;
  guint *l_range;
  guint i, l_length;

  l_list = GTK_CLIST (g_gtklist_jobs)->selection;
  l_length = g_list_length (l_list);
  /* We begin to copy the selection list because the first call to gtk_clist_remove delete it  */
  for (i=1; i<=l_length; i++)
    {
      l_data = (guint *) malloc (sizeof (gint));
      *l_data = (guint) l_list->data;
      l_copy_of_list = g_list_insert_sorted (l_copy_of_list, l_data, (GCompareFunc) greater_than);
      l_list = g_list_next (l_list);
    }
  l_copy_of_list = g_list_last (l_copy_of_list);
  for (i=1; i<=l_length; i++)
    {
      l_range = (guint *) l_copy_of_list->data;
      l_process = (GList *) g_list_nth (g_jobs_list, *l_range);
      l_a_job = (job *) l_process->data;
      l_process_state = l_a_job->state;
      /* We remove just some process  */
      if ((l_process_state == POVF_DONE) || (l_process_state == POVF_ERROR)|| (l_process_state == POVF_ABORTED))
	{
	  g_free (l_a_job->time);
	  g_free (l_a_job->file);
	  g_free (l_a_job->image);
	  /* Remove from the job control list  */
	  gtk_clist_remove (GTK_CLIST (g_gtklist_jobs), *l_range);
	  /* Remove from the process list  */
	  g_jobs_list = g_list_remove (g_jobs_list, l_a_job);
	}
      l_copy_of_list = g_list_previous (l_copy_of_list);
    }
  g_list_free (l_copy_of_list);
}
/*** END OF delete_jobs  */

/****************************************************************
 * Name : 		control
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/5/1999
 * Return :
 * Comments :
 ****************************************************************/
void
control ()
{
  GtkWidget   *l_hbox_refresh;
  GtkWidget   *l_scrolled_window;
  GtkWidget   *l_hscale_refresh;
  GtkBox      *l_dialog;
  gchar       *l_title[5] = {_("PID"),
		                 _("State"),
		                 _("Launch time"),
		                 _("File"),
		                 _("Image")};


  /* Dialog window creation  */
  GtkWidget *l_button_stop;
  GtkWidget *l_button_continue;
  GtkWidget *l_button_kill;
  GtkWidget *l_button_delete;
  GtkWidget *l_button_refresh;
  GtkWidget *l_button_close;
  GtkTooltips *l_button_tip;

  g_control_window = gtk_dialog_new ();
  gtk_container_set_border_width (GTK_CONTAINER (g_control_window), 5);
  gtk_window_set_title (GTK_WINDOW (g_control_window), _("PovFront : Jobs Control"));

  l_dialog = GTK_BOX (GTK_DIALOG (g_control_window)->vbox);

  l_button_stop = gtk_button_new_with_label (_("Stop"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_control_window)->action_area), l_button_stop, TRUE, FALSE, 10);
  g_signal_connect_swapped(G_OBJECT(l_button_stop), "clicked",
                           G_CALLBACK(stop_jobs), g_gtklist_jobs);
  l_button_tip = gtk_tooltips_new();
  gtk_tooltips_set_tip (l_button_tip, l_button_stop, _("Stop the selected process"), NULL);
  gtk_widget_show (l_button_stop);

  l_button_continue = gtk_button_new_with_label (_("Continue"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_control_window)->action_area), l_button_continue, TRUE, FALSE, 10);
  g_signal_connect_swapped(G_OBJECT(l_button_continue), "clicked",
                           G_CALLBACK(continue_jobs), g_gtklist_jobs);
  gtk_tooltips_set_tip (l_button_tip, l_button_continue, _("Make the selected process continue"), NULL);
  gtk_widget_show (l_button_continue);

  l_button_kill = gtk_button_new_with_label (_("Abort"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_control_window)->action_area), l_button_kill, TRUE, FALSE, 10);
  g_signal_connect(G_OBJECT (l_button_kill), "clicked",
                   G_CALLBACK(kill_jobs), g_gtklist_jobs);
  gtk_tooltips_set_tip (l_button_tip, l_button_kill, _("Kill the selected process"), NULL);
  gtk_widget_show (l_button_kill);

  l_button_delete = gtk_button_new_with_label (_("Delete"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_control_window)->action_area), l_button_delete, TRUE, FALSE, 10);
  g_signal_connect(G_OBJECT(l_button_delete), "clicked",
                   G_CALLBACK(delete_jobs), g_gtklist_jobs);
  gtk_tooltips_set_tip (l_button_tip, l_button_delete, _("Delete the selected 'Done or Error' process"), NULL);
  gtk_widget_show (l_button_delete);

/* To dim buttons while no line is selected in the job list
  gtk_widget_set_sensitive (g_control_stop_button, FALSE);
  gtk_widget_set_sensitive (g_control_continue_button, FALSE);
  gtk_widget_set_sensitive (g_control_abort_button, FALSE);
  gtk_widget_set_sensitive (g_control_delete_button, FALSE);
*/
  l_button_refresh = gtk_button_new_with_label (_("Refresh"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_control_window)->action_area), l_button_refresh, TRUE, FALSE, 10);
  g_signal_connect(G_OBJECT(l_button_refresh), "clicked",
                   G_CALLBACK(refresh), g_gtklist_jobs);
  gtk_tooltips_set_tip (l_button_tip, l_button_refresh, _("Refresh the list"), NULL);
  gtk_widget_show (l_button_refresh);
  /*gtk_widget_grab_default (l_button_refresh);*/

  l_button_close = gtk_button_new_with_label (_("Close"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (g_control_window)->action_area), l_button_close, TRUE, FALSE, 10);
  g_signal_connect(G_OBJECT(l_button_close), "clicked",
                   G_CALLBACK(control_off), g_control_window);
  gtk_tooltips_set_tip (l_button_tip, l_button_close, _("Close the job control"), NULL);
  gtk_widget_show (l_button_close);

  /* this is the scolled window to put the GtkList widget inside */
  gtk_window_set_wmclass (GTK_WINDOW (g_control_window), "configure", "PovFront");
  l_scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_set_usize (l_scrolled_window, 300, 150);
  gtk_box_pack_start (l_dialog, l_scrolled_window, TRUE, TRUE, 0);
  gtk_widget_show (l_scrolled_window);

  /* Create the GtkList widget  */
  g_gtklist_jobs = gtk_clist_new_with_titles (5, l_title);
  gtk_container_add (GTK_CONTAINER (l_scrolled_window), g_gtklist_jobs);

  refresh_list (g_gtklist_jobs);

  gtk_clist_set_column_width (GTK_CLIST (g_gtklist_jobs), 0, 30);
  gtk_clist_set_column_width (GTK_CLIST (g_gtklist_jobs), 1, 40);
  gtk_clist_set_column_width (GTK_CLIST (g_gtklist_jobs), 2, 125);
  gtk_clist_set_column_width (GTK_CLIST (g_gtklist_jobs), 3, 200);
  gtk_clist_set_column_width (GTK_CLIST (g_gtklist_jobs), 4, 200);
  gtk_clist_set_selection_mode (GTK_CLIST (g_gtklist_jobs), GTK_SELECTION_MULTIPLE);
  /* gtk_clist_set_auto_sort (GTK_CLIST (g_gtklist_jobs), TRUE);*/
  gtk_clist_column_titles_passive  (GTK_CLIST (g_gtklist_jobs));

/*
mettre ici le call back pour une slection de ligne
  gtk_signal_connect (GTK_OBJECT (g_gtklist_jobs), "clicked", GTK_SIGNAL_FUNC (control_button_up), g_control_window);

*/
  gtk_widget_show (g_gtklist_jobs);

  /* Buttons creation  */
  l_hbox_refresh = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (l_dialog, l_hbox_refresh, TRUE, TRUE, 0);
  gtk_widget_show (l_hbox_refresh);

  g_check_refresh = gtk_check_button_new_with_label (_("Auto Refresh (s)"));
  gtk_box_pack_start (GTK_BOX (l_hbox_refresh), g_check_refresh, TRUE, TRUE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_refresh), g_refresh);
  g_signal_connect(G_OBJECT(g_check_refresh), "clicked",
                   G_CALLBACK(switch_option), &g_refresh );
  g_signal_connect(G_OBJECT(g_check_refresh), "clicked",
                   G_CALLBACK(switch_refresh), &g_refresh );
  gtk_widget_show (g_check_refresh);

  g_adjust_refresh = gtk_adjustment_new (1.0, 0.5, 30, 0.5, 0.5, 0);
  l_hscale_refresh = gtk_hscale_new (GTK_ADJUSTMENT (g_adjust_refresh));
  gtk_box_pack_start (GTK_BOX (l_hbox_refresh), l_hscale_refresh, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(g_adjust_refresh), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_refresh_value);
  g_signal_connect(G_OBJECT(g_adjust_refresh), "value_changed",
                   G_CALLBACK(adjust_refresh_time), &g_refresh_value);
  gtk_widget_show (l_hscale_refresh);
  
  gtk_widget_realize (g_control_window);

  g_timeout_tag = gtk_timeout_add (1000*g_refresh_value, (GtkFunction) refresh_list ,g_gtklist_jobs);
}
/*** END OF control  */

/****************************************************************
 * Name : 		switch_option
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 02/13/1999
 * Return :
 * Comments : Attention  l'initialisation
 ****************************************************************/
gint
switch_option (GtkWidget *p_widget, gint *p_switch)
{
  g_return_val_if_fail (p_switch != NULL, TRUE);

  *p_switch = (*p_switch)? FALSE: TRUE;
  return (FALSE);
}
/*** END OF switch_option  */

/****************************************************************
 * Name : 		switch_out_file
 * parameters :
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 5/29/1999
 * Return :
 * Comments : switch the file output mode
 ****************************************************************/
gint
switch_out_file (GtkWidget *p_widget, gchar *p_new_code)
{
  g_return_val_if_fail (p_new_code != NULL, TRUE);

  g_out_file_mode = atoi(p_new_code);
  return (FALSE);
}
/*** END OF switch_out_file  */

/****************************************************************
 * Name : 		switch_palette
 * parameters :
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 02/27/1999
 * Return :
 * Comments : switch the palette code
 ****************************************************************/
gint
switch_palette (GtkWidget *p_widget, gchar *p_new_code)
{
  g_return_val_if_fail (p_new_code != NULL, TRUE);

  strcpy (g_palette, p_new_code);
  return (FALSE);
}
/*** END OF switch_palette  */

/****************************************************************
 * Name : 		file_selection_ok
 * parameters :        
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 02/21/1999
 * Return :
 * Comments : callback for the Ok button
 ****************************************************************/
gint
file_selection_ok (GtkWidget *p_widget, GtkFileSelection *p_file_selection)
{
  gchar *l_selected_file;
  GtkWidget *l_entry;

  g_return_val_if_fail (p_file_selection != NULL, TRUE);

  l_selected_file = (gchar *)gtk_file_selection_get_filename (GTK_FILE_SELECTION (p_file_selection));
  l_entry = g_object_get_data(G_OBJECT(p_file_selection), "Entry");
  gtk_entry_set_text(GTK_ENTRY(l_entry), (gchar*) l_selected_file);
  gtk_widget_destroy(GTK_WIDGET(p_file_selection));
  return FALSE;
}
/*** END OF file_selection_ok  */

/****************************************************************
 * Name : 		file_selection
 * parameters :       
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 02/21/1999
 * Return :
 * Comments :
 ****************************************************************/
gint
file_selection (GtkWidget *p_widget, gpointer p_entry)
{
  GtkWidget *l_file_selection;
  gchar *l_current_path;
  gchar *l_window_name;

  g_return_val_if_fail (p_widget != NULL, TRUE);
  g_return_val_if_fail (p_entry != NULL, TRUE);

  l_window_name = g_object_get_data(G_OBJECT(p_widget), "FileSelect");
  l_file_selection = gtk_file_selection_new (l_window_name);
  g_object_set_data(G_OBJECT(l_file_selection), "Entry", p_entry);
  l_current_path = (gchar  *)gtk_entry_get_text (GTK_ENTRY (p_entry));
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (l_file_selection), l_current_path);
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION (l_file_selection)->ok_button),
                   "clicked", G_CALLBACK(file_selection_ok), l_file_selection );
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(l_file_selection)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), l_file_selection);
  gtk_widget_show(l_file_selection);
  return FALSE;
}

/****************************************************************
 * Name : 		file_view
 * parameters : p_file_name : The file name
 * Author : Philippe DAVID
 * Update (mm/dd/yy) : 09/05/1999
 * Return : always FALSE
 * Comments : A window which display the text of a file
 ****************************************************************/
gint
file_view (GtkWidget *p_widget, gpointer *p_file_name)
{
/*  GtkWidget *l_text_window;*/
  gchar     *l_window_name;
  FILE      *l_file_handle;
  GtkWidget *l_hbox;
  GtkWidget *l_window;
  GtkWidget *l_button_close;
/*  GtkWidget *l_vscrollbar;*/
  gchar      l_buffer[1024];
  gint       l_length;
  GtkBox    *l_dialog;

  g_return_val_if_fail (p_file_name != NULL, TRUE);

  l_file_handle = fopen (gtk_entry_get_text (GTK_ENTRY (p_file_name)), "r");
  if (l_file_handle == NULL)
    {
      povfront_warning (_("This file does not exist (yet?)"));
      return (FALSE);
    }
  else
    {
      l_window_name = g_object_get_data(G_OBJECT(p_widget), "FileSelect");
      l_window = gtk_dialog_new ();
      gtk_window_set_title (GTK_WINDOW (l_window), l_window_name);
      l_dialog = GTK_BOX (GTK_DIALOG (l_window)->vbox);

      l_button_close = gtk_button_new_with_label (_("Close"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (l_window)->action_area), l_button_close, TRUE, FALSE, 10);
      g_signal_connect_swapped(G_OBJECT(l_button_close), "clicked",
                               G_CALLBACK(gtk_widget_destroy), l_window);
      //gtk_widget_grab_default (l_button_close);
      //gtk_widget_show (l_button_close);
      l_hbox = gtk_hbox_new (FALSE, FALSE);
      gtk_box_pack_start (l_dialog, l_hbox, TRUE, TRUE, 0);
      //gtk_widget_show (l_hbox);

/*      l_text_window = gtk_text_new (NULL, NULL);*/
      /*gtk_text_set_editable (GTK_TEXT (l_text_window), FALSE);*/
/*      gtk_widget_set_usize (l_text_window, 400, 400);
      gtk_box_pack_start (GTK_BOX (l_hbox), l_text_window, TRUE, TRUE, 0);*/

      /*l_vscrollbar = gtk_vscrollbar_new (GTK_TEXT(l_text_window)->vadj);*/
/*      gtk_box_pack_start (GTK_BOX (l_hbox), l_vscrollbar, FALSE, FALSE, 0);*/
      //gtk_widget_show (l_vscrollbar);

      do
	{
	  l_length = fread (l_buffer, 1, 1024, l_file_handle);
/*	  gtk_text_insert (GTK_TEXT (l_text_window), NULL, NULL, NULL, l_buffer, l_length);*/
	}
      while (l_length == 1024);
      fclose (l_file_handle);

      //gtk_widget_show (l_text_window);
      gtk_widget_show_all (l_window);
    }
  return (FALSE);
}

/****************************************************************
 * Name : 		povf_parse_options
 * parameters : 
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 9/12/1999
 * Return : 0 if error >0 otherwhile
 * Comments :
 ****************************************************************/
gint
povf_parse_options (int argc, char *argv[])
{
  int i;
  FILE *l_pov_file_handler;
  gchar l_message[256];
  gchar *tmp;

  for (i=1; i<argc; i++)
    {
      if (strcmp ("--version", argv[i]) == 0)
	{
	  printf (_("povfront version 1.3.2 is a GNU's GPL utility\nfrom Ph. P. E. DAVID\n"));
	  exit (0);
	}
      else if (strcmp ("--help", argv[i]) == 0)
	{
	  printf (_("\nUsage: povfront [-p] [-c FILE] [-i IMAGE] [-r CMD] [-s SOURCE]\n"));
	  printf (_("povfront is a front end for POVRAY ray tracing engine\n\n"));
	  printf (_("  -p\t\tfor plugin mode\n"));
	  printf (_("  -c FILE\twhere FILE specify a configuration file\n"));
	  printf (_("  -i IMAGE\twhere IMAGE specify the image file to compute\n"));
	  printf (_("  -r CMD\twhere CMD is the command to redirect output console\n"));
	  printf (_("  -s SOURCE\twhere SOURCE is the pov file to parse\n"));
	  printf (_("\t--version\tdisplay the name and the version of the software\n\n"));
	  printf (_("\t--help\t\tdisplay this text\n\n"));
	  printf (_("please report bug to clovis1@club-internet.fr\n"));
	  printf ("\n");
	  exit (0);
	}
      else if (strcmp ("-c", argv[i]) == 0)
	{
	  ++i;
	  g_config_file_to_parse = g_strdup (argv[i]);
	}
      else if (strcmp ("-i", argv[i]) == 0)
	{
	  ++i;
	  tmp = g_strdup (argv[i]);
	  g_image_file_to_compute = tmp;
	}
      else if (strcmp ("-r", argv[i]) == 0)
	{
	  ++i;
	  g_redirect_command_to_use = g_strdup (argv[i]);
	}
      else if (strcmp ("-s", argv[i]) == 0)
	{
	  ++i;
	  g_pov_file_to_parse = g_strdup (argv[i]);
    printf("g_pov_file_to_parse : %s\n", g_pov_file_to_parse); /* FIXME */
	  l_pov_file_handler = fopen (g_pov_file_to_parse, "r");
	  if (l_pov_file_handler == NULL)
	    {
	      sprintf (l_message, "%s", g_pov_file_to_parse);
	      perror (l_message);
	      exit (1);
	    }
	  else /* The file can be opened  */
	    {
	      fclose (l_pov_file_handler);
	    }
	}
      else if (strcmp ("-p", argv[i]) == 0)
	{
/*	  ++i;*/
	  g_plugin = 1;       /* indicate to the soft that the file name cannot be changed  */
	}
    }
  return 0;
}
/*** END OF povf_parse_options  */

/****************************************************************
 * Name : 		povf_parse_gnome_options
 * parameters : 
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 09/12/1999
 * Return :
 * Comments :
 ****************************************************************/
gint
povf_parse_gnome_options (gchar ** p_args)
{
  FILE *l_pov_file_handler;
  gchar l_message[256];

  if (p_args != NULL)
    {
      strcpy (g_pov_file, p_args[0]);
      l_pov_file_handler = fopen (g_pov_file, "r");
      if (l_pov_file_handler == NULL)
	{
	  sprintf (l_message, "%s", g_pov_file);
	  perror (l_message);
	  exit (1);
	}
      else /* The file can be opened  */
	{
	  fclose (l_pov_file_handler);
	}
    }
  return 0;
}
/*** END OF povf_parse_gnome_options  */

/****************************************************************
 * Name : 		add_pvm_host
 * parameters :
 * Author : Philippe DAVID
 * Update (dd/mm/yy) : 08/22/1999
 * Return :
 * Comments : add a host to the list of host
 ****************************************************************/
gint
add_pvm_host (GtkObject *p_hosts_list, GtkWidget *p_host_name)
{
  gchar *l_host_name;
  gchar *l_row_data;
  gint   l_row;

  g_return_val_if_fail (p_host_name != NULL, TRUE);

  l_host_name = (gchar *)gtk_entry_get_text (GTK_ENTRY (p_host_name));
  l_row_data = (gchar *) malloc (strlen (l_host_name) * sizeof (gchar));
  strcpy (l_row_data, l_host_name);
  l_row = gtk_clist_append (GTK_CLIST (g_gtklist_hosts), &l_host_name);
  gtk_clist_set_row_data (GTK_CLIST (g_gtklist_hosts), l_row, l_row_data);
  return (FALSE);
}
/*** END OF add_pvm_host  */

/****************************************************************
 * Name : 		delete_hosts
 * parameters :
 * Author : Philippe DAVID
 * Update (dd/mm/yy) : 08/21/1999
 * Return :
 * Comments : Delete the selected hosts from the host list
 ****************************************************************/
void
delete_hosts (GtkObject *p_jobs_list)
{
  GList *l_list;
  GList *l_copy_of_list = NULL;  /* Usefull because gtk_clist_remove reset the selection list  */
  guint *l_range;
  guint *l_data;
  guint i, l_length;
  gchar *l_row_data;

  l_list = GTK_CLIST (g_gtklist_hosts)->selection;
  l_length = g_list_length (l_list);
  /* We begin to copy the selection list because the first call to gtk_clist_remove delete it  */
  for (i=1; i<=l_length; i++)
    {
      l_data = (guint *) malloc (sizeof (guint));
      *l_data = (guint) l_list->data;
      l_copy_of_list = g_list_insert_sorted (l_copy_of_list, l_data, (GCompareFunc) greater_than);
      l_list = g_list_next (l_list);
    }
  l_copy_of_list = g_list_last (l_copy_of_list);
  for (i=1; i<=l_length; i++)
    {
      l_range = (guint *) l_copy_of_list->data;
      l_row_data = gtk_clist_get_row_data (GTK_CLIST (g_gtklist_hosts), *l_range);
      g_free (l_row_data);/* To be sure that removing, delete the row data too  */
      gtk_clist_remove (GTK_CLIST (g_gtklist_hosts), *l_range);
      l_copy_of_list = g_list_previous (l_copy_of_list);
    }
  g_list_free (l_copy_of_list);
}
/*** END OF delete_hosts  */

/****************************************************************
 * Name : 		create_output_page
 * parameters :
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 09/25/1999
 * Return : 
 * Comments :
 ****************************************************************/
void
create_output_page (GtkWidget *p_note_book)
{
  GtkWidget *l_separator;
  GtkWidget *l_label;
  GtkWidget *l_vbox_out_main;
  GtkWidget *l_vbox1_output;
  GtkWidget *l_vbox2_output;
  GtkWidget *l_hbox_inifile;
  GtkWidget *l_hbox_out2;
  GtkWidget *l_hbox_out3;
  GtkWidget *l_hbox_output;
  GSList    *l_grp_radio_color;
  GSList    *l_grp_radio = NULL;
  GtkWidget *l_button_inifile;
  GtkWidget *l_button_ini_view;
  GtkWidget *l_hbox_output_preview; 
  GtkWidget *l_entry_out_depth;
  GtkWidget *l_table_out_sub;
  GtkWidget *l_hbox_output_depth;
  GtkWidget *l_entry_substartx;          /* sub size of the output image  */
  GtkWidget *l_entry_subendx;            /* sub size of the output image  */
  GtkWidget *l_entry_substarty;          /* sub size of the output image  */
  GtkWidget *l_entry_subendy;            /* sub size of the output image  */
  GtkObject *g_adjust_;
  GtkObject *g_adjust_2;
  GtkObject *g_adjust_3;
  GtkObject *g_adjust_4;
  GtkObject *g_adjust_9;
  GtkObject *g_adjust_10;
  GtkObject *g_adjust_11;
  GList     *l_list_file_type;
  gint i;
  gchar l_size_label[15];   /* The radio button label  */
  gchar *l_radio_number [8] = {"0", "1", "2", "3", "4", "5", "6", "7"};

  GtkStyle  *l_style;
  GdkPixmap *l_subset_pixmap;
  GtkWidget *l_subset_pixmap_wid;
  GdkBitmap *l_mask_subset;

  g_return_if_fail (p_note_book != NULL);

  l_label = gtk_label_new (_("Output"));
  l_vbox_out_main = gtk_vbox_new (FALSE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (p_note_book), l_vbox_out_main, l_label);
  l_hbox_output = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox_out_main), l_hbox_output, TRUE, TRUE, 0);
  l_vbox1_output = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_output), l_vbox1_output, TRUE, TRUE, 0);

  /* Create all the buttons for pre-defined sizes  */
  for (i=0 ; i<8; ++i)
    {
      sprintf (l_size_label, "%d x %d", g_predef_size [i][0], g_predef_size [i][1]);
      g_radio_out_size [i] = gtk_radio_button_new_with_label ((i==0)?NULL:l_grp_radio, l_size_label);
      gtk_box_pack_start (GTK_BOX (l_vbox1_output), g_radio_out_size [i], TRUE, TRUE, 0);
      g_signal_connect(G_OBJECT(g_radio_out_size [i]), "pressed",
                       G_CALLBACK(image_size), l_radio_number[i]);
      l_grp_radio = gtk_radio_button_get_group(GTK_RADIO_BUTTON(g_radio_out_size[i]));
    }
  /*** Customized size output 1  */
  l_hbox_out2 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox1_output), l_hbox_out2, TRUE, TRUE, 0);

  g_radio_out_size [8] = gtk_radio_button_new_with_label (l_grp_radio, ":");
  gtk_box_pack_start (GTK_BOX (l_hbox_out2), g_radio_out_size [8], FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(g_radio_out_size[8]), "pressed",
                   G_CALLBACK(image_size), "8");
  l_grp_radio = gtk_radio_button_group (GTK_RADIO_BUTTON (g_radio_out_size [8]));

  g_adjust_ = gtk_adjustment_new (100, 1, 10000, 1, 10, 50);
  g_entry_xsize1 = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_), 4, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_out2), g_entry_xsize1, TRUE, TRUE, 0);

  g_adjust_2 = gtk_adjustment_new (100, 1, 10000, 1, 10, 50);
  g_entry_ysize1 = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_2), 4, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_out2), g_entry_ysize1, TRUE, TRUE, 0);

  /*** Customized size outpout 2  */
  l_hbox_out3 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox1_output), l_hbox_out3, TRUE, TRUE, 0);

  g_radio_out_size [9] = gtk_radio_button_new_with_label (l_grp_radio, ":");
  gtk_box_pack_start (GTK_BOX (l_hbox_out3), g_radio_out_size [9], FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(g_radio_out_size[9]), "pressed",
                   G_CALLBACK(image_size), "9");
  
  g_adjust_3 = gtk_adjustment_new(50, 1, 10000, 1, 10, 50);
  g_entry_xsize2 = gtk_spin_button_new(GTK_ADJUSTMENT (g_adjust_3), 4, 0);
  gtk_box_pack_start(GTK_BOX(l_hbox_out3), g_entry_xsize2, TRUE, TRUE, 0);

  g_adjust_4 = gtk_adjustment_new (50, 1, 10000, 1, 10, 50);
  g_entry_ysize2 = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_4), 4, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_out3), g_entry_ysize2, TRUE, TRUE, 0);
  
  l_separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox1_output), l_separator, TRUE, TRUE, 5);
  
  /*** Table for subset of rendering  */
  l_table_out_sub = gtk_table_new (3, 3, TRUE);
  gtk_box_pack_start (GTK_BOX (l_vbox1_output), l_table_out_sub, TRUE, TRUE, 0);

  g_check_sub = gtk_check_button_new_with_label (_("Sub"));
  gtk_table_attach_defaults (GTK_TABLE(l_table_out_sub), g_check_sub, 0, 1, 0, 1);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_sub), g_sub_render);
  g_signal_connect(G_OBJECT(g_check_sub), "clicked",
                   G_CALLBACK(switch_option), &g_sub_render);
  
  /* Pixmap for the subset of the image  */
  l_style = gtk_widget_get_style(g_main_window);
  gtk_widget_realize (g_main_window);
  l_subset_pixmap = gdk_pixmap_create_from_xpm_d (g_main_window->window, &l_mask_subset,
					      &l_style->bg[GTK_STATE_NORMAL], (gchar **) subset_xpm);
  l_subset_pixmap_wid = gtk_pixmap_new (l_subset_pixmap, l_mask_subset);
  gtk_table_attach_defaults (GTK_TABLE (l_table_out_sub), l_subset_pixmap_wid, 1, 3, 1, 3);
  
  /* Spin button for the subset rendering  */
  g_adjust_5 = gtk_adjustment_new (g_substartx_value, 0, 10000, 1, 10, 50);
  g_signal_connect(G_OBJECT(g_adjust_5), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_substartx_value);
  l_entry_substartx = gtk_spin_button_new(GTK_ADJUSTMENT(g_adjust_5), 4, 0);
  gtk_table_attach_defaults(GTK_TABLE(l_table_out_sub), l_entry_substartx, 1, 2, 0, 1);

  g_adjust_6 = gtk_adjustment_new (g_subendx_value, 0, 10000, 1, 10, 50);
  g_signal_connect(G_OBJECT(g_adjust_6), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_subendx_value);
  l_entry_subendx = gtk_spin_button_new(GTK_ADJUSTMENT(g_adjust_6), 4, 0);
  gtk_table_attach_defaults(GTK_TABLE(l_table_out_sub), l_entry_subendx, 2, 3, 0, 1);
  
  g_adjust_7 = gtk_adjustment_new (g_substarty_value, 0, 10000, 1, 10, 50);
  g_signal_connect(G_OBJECT(g_adjust_7), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_substarty_value);
  l_entry_substarty = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_7), 4, 0);
  gtk_table_attach_defaults (GTK_TABLE(l_table_out_sub), l_entry_substarty, 0, 1, 1, 2);
  
  g_adjust_8 = gtk_adjustment_new (g_subendy_value, 0, 10000, 1, 10, 50);
  g_signal_connect(G_OBJECT(g_adjust_8), "value_changed",
		      GTK_SIGNAL_FUNC (adjustment_value_changed), &g_subendy_value);
  l_entry_subendy = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_8), 4, 0);
  gtk_table_attach_defaults (GTK_TABLE(l_table_out_sub), l_entry_subendy, 0, 1, 2, 3);
  
  l_separator = gtk_vseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_hbox_output), l_separator, TRUE, TRUE, 5);
  gtk_widget_show (l_separator);

  /*** Right vertical column of options */
  l_vbox2_output = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_output), l_vbox2_output, TRUE, TRUE, 0);
  
  g_check_out1 = gtk_check_button_new_with_label (_("Test abort"));
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_check_out1, TRUE, TRUE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_out1), g_test_abort);
  g_signal_connect(G_OBJECT(g_check_out1), "clicked",
                   G_CALLBACK(switch_option), &g_test_abort);
  
  g_check_out2 = gtk_check_button_new_with_label (_("Continue"));
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_check_out2, TRUE, TRUE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_out2), g_continue);
  g_signal_connect(G_OBJECT(g_check_out2), "clicked",
                   G_CALLBACK(switch_option), &g_continue);
  
  g_check_out3 = gtk_check_button_new_with_label (_("Pause"));
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_check_out3, TRUE, TRUE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_out3), g_pause);
  g_signal_connect(G_OBJECT(g_check_out3), "clicked",
                   G_CALLBACK(switch_option), &g_pause);
  
  g_check_out4 = gtk_check_button_new_with_label (_("Verbose"));
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_check_out4, TRUE, TRUE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_out4), g_verbose);
  g_signal_connect(G_OBJECT(g_check_out4), "clicked",
                   G_CALLBACK(switch_option), &g_verbose);
  
  g_check_out5 = gtk_check_button_new_with_label (_("Draw vista"));
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_check_out5, TRUE, TRUE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_out5), g_draw_vista);
  g_signal_connect(G_OBJECT(g_check_out5), "clicked",
                   G_CALLBACK(switch_option), &g_draw_vista);
  
  g_check_out6 = gtk_check_button_new_with_label (_("Display Image"));
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_check_out6, TRUE, TRUE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_out6), g_display_image);
  g_signal_connect(G_OBJECT(g_check_out6), "clicked",
                   G_CALLBACK(switch_option), &g_display_image);
  
  g_check_out7 = gtk_check_button_new_with_label (_("Alpha out"));
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_check_out7, TRUE, TRUE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_out7), g_alpha_out);
  g_signal_connect(G_OBJECT(g_check_out7), "clicked",
                   G_CALLBACK(switch_option), &g_alpha_out);

  l_separator = gtk_hseparator_new();
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), l_separator, TRUE, TRUE, 0);
  
  g_combo_out_file = gtk_combo_new ();
  gtk_entry_set_editable (GTK_ENTRY (GTK_COMBO (g_combo_out_file)->entry), FALSE);
  l_list_file_type = NULL;
  l_list_file_type  = g_list_prepend (l_list_file_type, "CSV");
  l_list_file_type  = g_list_prepend (l_list_file_type, "PNG");
  l_list_file_type  = g_list_prepend (l_list_file_type, "PPM");
  l_list_file_type  = g_list_prepend (l_list_file_type, "BMP");
  l_list_file_type  = g_list_prepend (l_list_file_type, "TGA");
  gtk_combo_set_popdown_strings (GTK_COMBO (g_combo_out_file), l_list_file_type);
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_combo_out_file, TRUE, FALSE, 0);

  l_separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), l_separator, TRUE, TRUE, 0);
  
  g_radio_out_color1 = gtk_radio_button_new_with_label (NULL, _("Gray scale"));
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_radio_out_color1, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(g_radio_out_color1), "clicked",
                   G_CALLBACK(switch_palette), "G");
  l_grp_radio_color = gtk_radio_button_group (GTK_RADIO_BUTTON (g_radio_out_color1));
  
  g_radio_out_color2 = gtk_radio_button_new_with_label (l_grp_radio_color, _("High color"));
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_radio_out_color2, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(g_radio_out_color2), "clicked",
                   G_CALLBACK(switch_palette), "H");
  l_grp_radio_color = gtk_radio_button_group (GTK_RADIO_BUTTON (g_radio_out_color2));
  
  g_radio_out_color3 = gtk_radio_button_new_with_label (l_grp_radio_color, _("True color"));
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), g_radio_out_color3, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(g_radio_out_color3), "clicked",
                   G_CALLBACK(switch_palette), "T");
  
  l_separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), l_separator, TRUE, TRUE, 5);
  
  l_hbox_output_preview = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), l_hbox_output_preview, TRUE, TRUE, 0);
  
  g_check_out_preview = gtk_check_button_new_with_label (_("Preview"));
  gtk_box_pack_start (GTK_BOX (l_hbox_output_preview), g_check_out_preview, TRUE, TRUE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_out_preview), g_preview);
  g_signal_connect(G_OBJECT(g_check_out_preview), "clicked",
                   G_CALLBACK(switch_option), &g_preview);
  
  g_adjust_9 = gtk_adjustment_new (8, 2, 64, 2, 8, 4);
  g_entry_out_preview1 = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_9), 2, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_output_preview), g_entry_out_preview1, TRUE, TRUE, 0);
  
  l_label = gtk_label_new ("to");
  gtk_box_pack_start (GTK_BOX (l_hbox_output_preview), l_label, TRUE, TRUE, 0);
  
  g_adjust_10 = gtk_adjustment_new (8, 2, 64, 2, 8, 4);
  g_entry_out_preview2 = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_10), 2,0);
  gtk_box_pack_start (GTK_BOX (l_hbox_output_preview), g_entry_out_preview2, TRUE, TRUE, 0);

  l_hbox_output_depth = gtk_hbox_new (TRUE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox2_output), l_hbox_output_depth, TRUE, TRUE, 0);
  l_label = gtk_label_new (_("Depth"));
  gtk_box_pack_start (GTK_BOX (l_hbox_output_depth), l_label, TRUE, TRUE, 0);
  
  g_adjust_11 = gtk_adjustment_new (16, 2, 32, 2, 8, 4);
  l_entry_out_depth = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_11), 2,0);
  gtk_box_pack_start (GTK_BOX (l_hbox_output_depth), l_entry_out_depth, FALSE, FALSE, 0);

  l_separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_out_main), l_separator, TRUE, TRUE, 0);

  /*** Boite de selection du fichier INI en sortie  */
  l_hbox_inifile = gtk_hbox_new (FALSE,0);
  gtk_box_pack_start (GTK_BOX (l_vbox_out_main), l_hbox_inifile, TRUE, FALSE, 0);

  g_check_inifile = gtk_check_button_new_with_label (_("Output .ini file"));
  gtk_box_pack_start (GTK_BOX (l_hbox_inifile), g_check_inifile, FALSE, FALSE, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_inifile), g_output_inifile);
  g_signal_connect(G_OBJECT(g_check_inifile), "clicked",
                   G_CALLBACK(switch_option), &g_output_inifile );

  g_entry_inifile = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_hbox_inifile), g_entry_inifile, TRUE, TRUE, 0);

  l_button_inifile = gtk_button_new_with_label (_("Browse"));
  gtk_box_pack_start (GTK_BOX (l_hbox_inifile), l_button_inifile, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(l_button_inifile), "FileSelect", _("INI File Selection"));
  g_signal_connect(G_OBJECT(l_button_inifile), "clicked",
                   G_CALLBACK(file_selection), g_entry_inifile);

  l_button_ini_view = gtk_button_new_with_label (_("View"));
  g_object_set_data(G_OBJECT(l_button_ini_view), "FileSelect", _("View .ini file"));
  g_signal_connect(G_OBJECT(l_button_ini_view), "clicked",
                   G_CALLBACK(file_view), g_entry_inifile);
  gtk_box_pack_start (GTK_BOX (l_hbox_inifile), l_button_ini_view, TRUE, FALSE, 0);
  gtk_widget_show (l_vbox_out_main);
}
/*** END OF create_output_page */

/****************************************************************
 * Name : 		create_quality_page
 * parameters : Aucun
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 09/7/1999
 * Return : 
 * Comments :
 ****************************************************************/
void
create_quality_page (GtkWidget *p_note_book)
{
  GtkWidget *l_separator;
  GtkWidget *l_label;
  GtkWidget *l_vbox_quality;
  GtkWidget *l_hscale_quality;
  GtkWidget *l_hscale_bounding_threshold;
  GtkWidget *l_table_quality;
  GtkWidget *l_hbox_aa;
  GtkWidget *l_vbox_methode_aa;
  GtkWidget *l_vbox_labels_aa;
  GtkWidget *l_vbox_scales_aa;
  GtkWidget *l_hscale_jitter;
  GtkWidget *l_hscale_depth;
  GtkWidget *l_hscale_threshold;
  GtkWidget *l_hscale_gamma;
  GSList    *l_grp_radio_aa;
  GtkWidget *l_hbox_histo_file;
  GtkWidget *l_hbox_histo_type;
  GtkWidget *l_hbox_history_size;
  GtkWidget *l_button_histo_file;
  GList     *l_list_file_type;
  GtkObject *l_histo_adjust;
  GtkObject *l_histo_adjust2;

  g_return_if_fail (p_note_book != NULL);

  l_list_file_type = NULL;
  l_list_file_type  = g_list_prepend (l_list_file_type, "CSV");
  l_list_file_type  = g_list_prepend (l_list_file_type, "PNG");
  l_list_file_type  = g_list_prepend (l_list_file_type, "PPM");
  l_list_file_type  = g_list_prepend (l_list_file_type, "BMP");
  l_list_file_type  = g_list_prepend (l_list_file_type, "TGA");

  l_label = gtk_label_new (_("Quality"));
  l_vbox_quality = gtk_vbox_new (FALSE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (p_note_book), l_vbox_quality, l_label);

  l_table_quality = gtk_table_new (6, 2, FALSE);
  gtk_box_pack_start (GTK_BOX (l_vbox_quality), l_table_quality, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("Quality"));
  gtk_table_attach_defaults (GTK_TABLE(l_table_quality), l_label, 0, 1, 0, 1);

  g_adjust_quality = gtk_adjustment_new (g_quality_value, 0, 11, 1, 1, 0);
  l_hscale_quality = gtk_hscale_new (GTK_ADJUSTMENT (g_adjust_quality));
  gtk_scale_set_digits (GTK_SCALE (l_hscale_quality), 0);
  gtk_table_attach_defaults (GTK_TABLE(l_table_quality), l_hscale_quality, 1, 2, 0, 1);
  g_signal_connect(G_OBJECT(g_adjust_quality), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_quality_value);
  gtk_tooltips_set_tip (g_tooltips, l_hscale_quality, _("0 : Just show quick colors, full ambient lighting only\n1 : Quick colors at 5 or below\n2 : Specified diffuse and ambient light\n3 :\n4 : Shadows but no ext. lights\n5 : Shadows, including ext. lights\n6 : Texture patterns\n7 :\n8 : Reflected, refracted, and transmitted rays\n9 : Media\n10 : Radiosity, no media\n11 : Radiosity and media"), NULL);

  g_check_quality1 = gtk_check_button_new_with_label (_("Radiosity"));
  gtk_table_attach_defaults (GTK_TABLE(l_table_quality), g_check_quality1, 0, 1, 2, 3);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_quality1), g_radiosity);
  g_signal_connect(G_OBJECT(g_check_quality1), "clicked",
                   G_CALLBACK(switch_option), &g_radiosity);

  g_check_quality2 = gtk_check_button_new_with_label (_("Bounding Box"));
  gtk_table_attach_defaults (GTK_TABLE(l_table_quality), g_check_quality2, 0, 1, 5, 6);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_quality2), g_bounding_box);
  g_signal_connect(G_OBJECT(g_check_quality2), "clicked",
                   G_CALLBACK(switch_option), &g_bounding_box);

  g_adjust_bounding_threshold = gtk_adjustment_new (g_bounding_threshold_value, 0, 50, 1, 5, 0);
  l_hscale_bounding_threshold = gtk_hscale_new (GTK_ADJUSTMENT (g_adjust_bounding_threshold));
  gtk_scale_set_digits (GTK_SCALE (l_hscale_bounding_threshold), 0);
  gtk_table_attach_defaults (GTK_TABLE(l_table_quality), l_hscale_bounding_threshold, 1, 2, 5, 6);
  g_signal_connect(G_OBJECT(g_adjust_bounding_threshold), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_bounding_threshold_value);

  /* Display Gamma  */
  l_label = gtk_label_new (_("Gamma"));
  gtk_table_attach_defaults (GTK_TABLE(l_table_quality), l_label, 0, 1, 6, 7);

  g_adjust_gamma = gtk_adjustment_new (2.2, 1.6, 2.7, 0.1, 0.1, 0.1);
  l_hscale_gamma = gtk_hscale_new (GTK_ADJUSTMENT (g_adjust_gamma));
  gtk_table_attach_defaults (GTK_TABLE(l_table_quality), l_hscale_gamma, 1, 2, 6, 7);
  g_signal_connect(G_OBJECT(g_adjust_gamma), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_quality_gamma);

  g_check_quality3 = gtk_check_button_new_with_label (_("Light buffer"));
  gtk_table_attach_defaults (GTK_TABLE(l_table_quality), g_check_quality3, 0, 1, 3, 4);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_quality3), g_light_buffer);
  g_signal_connect(G_OBJECT(g_check_quality3), "clicked",
                   G_CALLBACK(switch_option), &g_light_buffer);

  g_check_quality4 = gtk_check_button_new_with_label (_("Vista buffer"));
  gtk_table_attach_defaults (GTK_TABLE(l_table_quality), g_check_quality4, 1, 2, 2, 3);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_quality4), g_vista_buffer);
  g_signal_connect(G_OBJECT(g_check_quality4), "clicked",
                   G_CALLBACK(switch_option), &g_vista_buffer);

  g_check_quality5 = gtk_check_button_new_with_label (_("Remove bounding"));
  gtk_table_attach_defaults (GTK_TABLE(l_table_quality), g_check_quality5, 1, 2, 3, 4);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_quality5), g_remove_bounding);
  g_signal_connect(G_OBJECT(g_check_quality5), "clicked",
                   G_CALLBACK(switch_option), &g_remove_bounding);

  g_check_quality6 = gtk_check_button_new_with_label (_("Split"));
  gtk_table_attach_defaults(GTK_TABLE(l_table_quality), g_check_quality6, 1, 2, 4, 5);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_quality6), g_split);
  g_signal_connect(G_OBJECT(g_check_quality6), "clicked",
                   G_CALLBACK(switch_option), &g_split);

  l_separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_quality), l_separator, FALSE, FALSE, 0);

  l_label = gtk_label_new (_("Antialiasing options"));
  gtk_box_pack_start (GTK_BOX (l_vbox_quality), l_label, FALSE, FALSE, 0);

  l_hbox_aa = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox_quality), l_hbox_aa, FALSE, FALSE, 0);

  l_vbox_methode_aa = gtk_vbox_new (TRUE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_aa), l_vbox_methode_aa, TRUE, TRUE, 0);

  g_check_antialiasing = gtk_check_button_new_with_label (_("Antialiasing"));
  gtk_box_pack_start (GTK_BOX (l_vbox_methode_aa), g_check_antialiasing, FALSE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_antialiasing), g_antialiasing);
  g_signal_connect(G_OBJECT(g_check_antialiasing), "clicked",
                   G_CALLBACK(switch_option), &g_antialiasing );

  g_radio_aa1 = gtk_radio_button_new_with_label (NULL, _("Methode 1"));
  gtk_box_pack_start (GTK_BOX (l_vbox_methode_aa), g_radio_aa1, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(g_radio_aa1), "pressed",
                   G_CALLBACK(aa_methode), "1");
  l_grp_radio_aa = gtk_radio_button_get_group(GTK_RADIO_BUTTON(g_radio_aa1));
  
  g_radio_aa2 = gtk_radio_button_new_with_label (l_grp_radio_aa, _("Methode 2"));
  gtk_box_pack_start (GTK_BOX (l_vbox_methode_aa), g_radio_aa2, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(g_radio_aa2), "pressed",
                   G_CALLBACK(aa_methode), "2");

  l_separator = gtk_vseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_hbox_aa), l_separator, TRUE, TRUE, 0);

  l_vbox_labels_aa  = gtk_vbox_new (TRUE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_aa), l_vbox_labels_aa, TRUE, TRUE, 0);

  g_check_jitter = gtk_check_button_new_with_label (_("Jitter"));
  gtk_box_pack_start (GTK_BOX (l_vbox_labels_aa), g_check_jitter, TRUE, TRUE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_jitter), g_jitter);
  g_signal_connect(G_OBJECT(g_check_jitter), "clicked",
                   G_CALLBACK(switch_option), &g_jitter );

  l_label = gtk_label_new (_("Depth"));
  gtk_box_pack_start (GTK_BOX (l_vbox_labels_aa), l_label, TRUE, TRUE, 0);
  
  l_label = gtk_label_new (_("Threshold"));
  gtk_box_pack_start (GTK_BOX (l_vbox_labels_aa), l_label, TRUE, TRUE, 0);

  gtk_widget_show (l_vbox_labels_aa);

  l_vbox_scales_aa = gtk_vbox_new (TRUE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_aa), l_vbox_scales_aa, TRUE, TRUE, 0);

  g_adjust_jitter = gtk_adjustment_new (1.0, 0, 1, 0.1, 0.1, 0);
  l_hscale_jitter = gtk_hscale_new (GTK_ADJUSTMENT (g_adjust_jitter));
  gtk_box_pack_start (GTK_BOX (l_vbox_scales_aa), l_hscale_jitter, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(g_adjust_jitter), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_jitter_value);
  
  g_adjust_depth = gtk_adjustment_new (3, 1, 9, 1, 1, 0);
  l_hscale_depth = gtk_hscale_new (GTK_ADJUSTMENT (g_adjust_depth));
  gtk_scale_set_digits (GTK_SCALE (l_hscale_depth), 0);
  gtk_box_pack_start (GTK_BOX (l_vbox_scales_aa), l_hscale_depth, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(g_adjust_depth), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_quality_depth);
  
  g_adjust_threshold = gtk_adjustment_new (0.3, 0, 1, 0.1, 0.1, 0.1);
  l_hscale_threshold = gtk_hscale_new (GTK_ADJUSTMENT (g_adjust_threshold));
  gtk_box_pack_start (GTK_BOX (l_vbox_scales_aa), l_hscale_threshold, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(g_adjust_threshold), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_quality_threshold);

  l_separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_quality), l_separator, TRUE, TRUE, 0);

  /* Histogram options  */
  l_label = gtk_label_new (_("Histogram options"));
  gtk_box_pack_start (GTK_BOX (l_vbox_quality), l_label, TRUE, TRUE, 0);
  
  l_hbox_histo_file = gtk_hbox_new (FALSE,0);
  gtk_box_pack_start (GTK_BOX (l_vbox_quality), l_hbox_histo_file, TRUE, FALSE, 0);
  
  g_check_histo_file = gtk_check_button_new_with_label (_("Histo. file"));
  gtk_box_pack_start (GTK_BOX (l_hbox_histo_file), g_check_histo_file, FALSE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_histo_file), g_output_histo_file);
  g_signal_connect(G_OBJECT(g_check_histo_file), "clicked",
                   G_CALLBACK(switch_option), &g_output_histo_file );
  gtk_tooltips_set_tip (g_tooltips, g_check_histo_file, _("Enable/Disable outpout of histogram file"), NULL);
  
  g_entry_histo_file = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_hbox_histo_file), g_entry_histo_file, TRUE, TRUE, 0);
  
  l_button_histo_file = gtk_button_new_with_label (_("Browse"));
  gtk_box_pack_start (GTK_BOX (l_hbox_histo_file), l_button_histo_file, FALSE, FALSE, 0);
  g_object_set_data(G_OBJECT(l_button_histo_file), "FileSelect", _("Histogram File Selection"));
  g_signal_connect(G_OBJECT(l_button_histo_file), "clicked",
                   G_CALLBACK(file_selection), g_entry_histo_file);

  /* Histogram file type  */
  l_hbox_histo_type = gtk_hbox_new (FALSE,0);
  gtk_box_pack_start (GTK_BOX (l_vbox_quality), l_hbox_histo_type, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("Histogram file type"));
  gtk_box_pack_start (GTK_BOX (l_hbox_histo_type), l_label, TRUE, TRUE, 0);

  g_combo_histo_type = gtk_combo_new ();
  gtk_entry_set_editable (GTK_ENTRY (GTK_COMBO (g_combo_histo_type)->entry), FALSE);
  gtk_combo_set_popdown_strings (GTK_COMBO (g_combo_histo_type), l_list_file_type);
  gtk_box_pack_start (GTK_BOX (l_hbox_histo_type), g_combo_histo_type, FALSE, TRUE, 0);
  
  /* Histogram image size  */
  l_hbox_history_size = gtk_hbox_new (FALSE,0);
  gtk_box_pack_start (GTK_BOX (l_vbox_quality), l_hbox_history_size, TRUE, TRUE, 0);
  
  l_label = gtk_label_new (_("Histogram image size"));
  gtk_box_pack_start (GTK_BOX (l_hbox_history_size), l_label, TRUE, TRUE, 0);
  
  l_histo_adjust = gtk_adjustment_new (100, 0, 10000, 1, 10, 50);
  g_entry_history_xsize = gtk_spin_button_new (GTK_ADJUSTMENT (l_histo_adjust), 4, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_history_size), g_entry_history_xsize, FALSE, FALSE, 0);
  
  l_histo_adjust2 = gtk_adjustment_new (100, 0, 10000, 1, 10, 50);
  g_entry_history_ysize = gtk_spin_button_new (GTK_ADJUSTMENT (l_histo_adjust2), 4, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_history_size), g_entry_history_ysize, FALSE, FALSE, 0);
}
/*** END OF create_quality_page */

/****************************************************************
 * Name : 		create_animation_page
 * parameters : Aucun
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 09/7/1999
 * Return : 
 * Comments :
 ****************************************************************/
void
create_animation_page (GtkWidget *p_note_book)
{
  GtkWidget *l_separator;
  GtkWidget *l_label;
  GtkWidget *l_vbox_animation;
  GtkWidget *l_hbox_anim_clock;
  GtkWidget *l_hbox_internal_animation;
  GtkWidget *l_vbox_internal_animation1;
  GtkWidget *l_vbox_internal_animation2;

  g_return_if_fail (p_note_book != NULL);

  l_label = gtk_label_new (_("Animation"));
  l_vbox_animation = gtk_vbox_new (FALSE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (p_note_book), l_vbox_animation, l_label);
 /* Clock option of external animation  */
  l_label = gtk_label_new (_("External animation Loop"));
  gtk_box_pack_start (GTK_BOX (l_vbox_animation), l_label, TRUE, TRUE, 0);
  
  l_hbox_anim_clock = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox_animation), l_hbox_anim_clock, TRUE, TRUE, 0);
  l_label = gtk_label_new (_("Clock"));
  gtk_box_pack_start (GTK_BOX (l_hbox_anim_clock), l_label, TRUE, TRUE, 0);

  g_entry_anim_clock = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_hbox_anim_clock), g_entry_anim_clock, FALSE, FALSE, 0);
  /* Initial and final values for _clock, frames and subset  */
  l_separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_animation), l_separator, TRUE, TRUE, 0);
  gtk_widget_show (l_separator);
  l_label = gtk_label_new (_("Internal animation Loop"));
  gtk_box_pack_start (GTK_BOX (l_vbox_animation), l_label, TRUE, TRUE, 0);
  
  l_hbox_internal_animation = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox_animation), l_hbox_internal_animation, TRUE, TRUE, 0);

  l_vbox_internal_animation1 = gtk_vbox_new (TRUE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_internal_animation), l_vbox_internal_animation1, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("Initial frame"));
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation1), l_label, TRUE, TRUE, 5);

  l_label = gtk_label_new (_("Final frame"));
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation1), l_label, TRUE, TRUE, 5);

  l_label = gtk_label_new (_("Initial clock"));
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation1), l_label, TRUE, TRUE, 5);

  l_label = gtk_label_new (_("Final clock"));
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation1), l_label, TRUE, TRUE, 5);

  l_label = gtk_label_new (_("Initial subset"));
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation1), l_label, TRUE, FALSE, 5);

  l_label = gtk_label_new (_("Final subset"));
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation1), l_label, TRUE, FALSE, 5);
  
  l_vbox_internal_animation2 = gtk_vbox_new (TRUE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_internal_animation), l_vbox_internal_animation2, TRUE, TRUE, 0);

  g_entry_anim_frame_ini = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation2), g_entry_anim_frame_ini, TRUE, TRUE, 5);
  
  g_entry_anim_frame_fin = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation2), g_entry_anim_frame_fin, TRUE, TRUE, 5);
  
  g_entry_anim_clock_ini = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation2), g_entry_anim_clock_ini, TRUE, TRUE, 5);
  
  g_entry_anim_clock_fin = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation2), g_entry_anim_clock_fin, TRUE, TRUE, 5);
  
  g_entry_anim_subset_ini = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation2), g_entry_anim_subset_ini, TRUE, FALSE, 5);
  
  g_entry_anim_subset_fin = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_internal_animation2), g_entry_anim_subset_fin, TRUE, FALSE, 5);
  
  g_check_anim_cyclic = gtk_check_button_new_with_label (_("Cyclic animation"));
  gtk_box_pack_start (GTK_BOX (l_vbox_animation), g_check_anim_cyclic, TRUE, FALSE, 0);
  g_signal_connect(G_OBJECT(g_check_anim_cyclic), "clicked",
                   G_CALLBACK(switch_option), &g_cyclic);
 
  g_check_anim_field = gtk_check_button_new_with_label (_("Field rendering"));
  gtk_box_pack_start (GTK_BOX (l_vbox_animation), g_check_anim_field, TRUE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_anim_field), g_field_render);
  g_signal_connect(G_OBJECT(g_check_anim_field), "clicked",
                   G_CALLBACK(switch_option), &g_field_render);
  
  g_check_anim_odd = gtk_check_button_new_with_label (_("Odd field"));
  gtk_box_pack_start (GTK_BOX (l_vbox_animation), g_check_anim_odd, TRUE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_anim_odd), g_odd_field);
  g_signal_connect(G_OBJECT(g_check_anim_odd), "clicked",
                   G_CALLBACK(switch_option), &g_odd_field);
}
/*** END OF create_animation_page */

/****************************************************************
 * Name : 		create_script_page
 * parameters : Aucun
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 09/7/1999
 * Return : 
 * Comments :
 ****************************************************************/
void
create_script_page (GtkWidget *p_note_book)
{
  GtkWidget *l_label;
  GtkWidget *l_hbox_script;
  GtkWidget *l_vbox_script_label;
  GtkWidget *l_vbox_script_entry;

  g_return_if_fail (p_note_book != NULL);

  l_label = gtk_label_new (_("Script"));
  l_hbox_script = gtk_hbox_new (FALSE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (p_note_book), l_hbox_script, l_label);
  l_vbox_script_label = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_script), l_vbox_script_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("Pre scene command"));
  gtk_box_pack_start (GTK_BOX (l_vbox_script_label), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("Pre frame command"));
  gtk_box_pack_start (GTK_BOX (l_vbox_script_label), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("Post scene command"));
  gtk_box_pack_start (GTK_BOX (l_vbox_script_label), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("Post frame command"));
  gtk_box_pack_start (GTK_BOX (l_vbox_script_label), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("User abort command"));
  gtk_box_pack_start (GTK_BOX (l_vbox_script_label), l_label, TRUE, TRUE, 0);

  l_label = gtk_label_new (_("Fatal error command"));
  gtk_box_pack_start (GTK_BOX (l_vbox_script_label), l_label, TRUE, TRUE, 0);

  l_vbox_script_entry = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_script), l_vbox_script_entry, TRUE, TRUE, 0);

  g_entry_pre_scene = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_script_entry), g_entry_pre_scene, TRUE, TRUE, 0);

  g_entry_pre_frame = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_script_entry), g_entry_pre_frame, TRUE, TRUE, 0);

  g_entry_post_scene = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_script_entry), g_entry_post_scene, TRUE, TRUE, 0);

  g_entry_post_frame = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_script_entry), g_entry_post_frame, TRUE, TRUE, 0);

  g_entry_user_abort = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_script_entry), g_entry_user_abort, TRUE, TRUE, 0);

  g_entry_fatal_error = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_script_entry), g_entry_fatal_error, TRUE, TRUE, 0);
}
/*** END OF create_script_page */

/****************************************************************
 * Name : 		create_message_page
 * parameters : Aucun
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 09/7/1999
 * Return : 
 * Comments :
 ****************************************************************/
void
create_message_page (GtkWidget *p_note_book)
{
  GtkWidget *l_label;
  GtkWidget *l_hbox_messages;
  GtkWidget *l_vbox_messages_check;
  GtkWidget *l_vbox_messages_entry;
  GtkWidget *l_vbox_messages_browse;
  GtkWidget *l_vbox_messages_view;
  GtkWidget *l_button_debug_stream;
  GtkWidget *l_button_fatal_stream;
  GtkWidget *l_button_render_stream;
  GtkWidget *l_button_stat_stream;
  GtkWidget *l_button_warning_stream;
  GtkWidget *l_button_all_stream;
  GtkWidget *l_button_debug_view;
  GtkWidget *l_button_fatal_view;
  GtkWidget *l_button_render_view;
  GtkWidget *l_button_stat_view;
  GtkWidget *l_button_warning_view;
  GtkWidget *l_button_all_view;

  g_return_if_fail (p_note_book != NULL);

  l_label = gtk_label_new (_("Messages"));
  l_hbox_messages = gtk_hbox_new (FALSE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (p_note_book), l_hbox_messages, l_label);
 l_vbox_messages_check = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_messages), l_vbox_messages_check, FALSE, FALSE, 0);
  l_vbox_messages_entry = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_messages), l_vbox_messages_entry, TRUE, TRUE, 0);
  l_vbox_messages_browse = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_messages), l_vbox_messages_browse, FALSE, FALSE, 0);
  l_vbox_messages_view = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_messages), l_vbox_messages_view, FALSE, FALSE, 0);
  
  /* Check button for messages options  */
  g_check_debug_stream = gtk_check_button_new_with_label (_("Debug"));
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_check), g_check_debug_stream, TRUE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_debug_stream), g_debug_stream);
  g_signal_connect(G_OBJECT(g_check_debug_stream), "clicked",
                   G_CALLBACK(switch_option), &g_debug_stream);

  g_check_fatal_stream = gtk_check_button_new_with_label (_("Fatal"));
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_check), g_check_fatal_stream, TRUE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_fatal_stream), g_fatal_stream);
  g_signal_connect(G_OBJECT(g_check_fatal_stream), "clicked",
                   G_CALLBACK(switch_option), &g_fatal_stream);

  g_check_render_stream = gtk_check_button_new_with_label (_("Render"));
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_check), g_check_render_stream, TRUE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_render_stream), g_render_stream);
  g_signal_connect(G_OBJECT(g_check_render_stream), "clicked",
                   G_CALLBACK(switch_option), &g_render_stream);
  
  g_check_stat_stream = gtk_check_button_new_with_label (_("Stat"));
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_check), g_check_stat_stream, TRUE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_check_stat_stream), g_stat_stream);
  g_signal_connect(G_OBJECT(g_check_stat_stream), "clicked",
                   G_CALLBACK(switch_option), &g_stat_stream);
  
  g_check_warning_stream = gtk_check_button_new_with_label (_("Warning"));
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_check), g_check_warning_stream, TRUE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_warning_stream), g_warning_stream);
  g_signal_connect(G_OBJECT(g_check_warning_stream), "clicked",
                   G_CALLBACK(switch_option), &g_warning_stream);
  
  g_check_all_stream = gtk_check_button_new_with_label (_("All"));
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_check), g_check_all_stream, TRUE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (g_check_all_stream), g_all_stream);
  g_signal_connect(G_OBJECT(g_check_all_stream), "clicked",
                   G_CALLBACK(switch_option), &g_all_stream);
  
  /* Entry field for messages options  */
  g_entry_debug_stream = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_entry), g_entry_debug_stream, TRUE, TRUE, 0);
  
  g_entry_fatal_stream = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_entry), g_entry_fatal_stream, TRUE, TRUE, 0);
  
  g_entry_render_stream = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_entry), g_entry_render_stream, TRUE, TRUE, 0);
  
  g_entry_stat_stream = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_entry), g_entry_stat_stream, TRUE, TRUE, 0);
  
  g_entry_warning_stream = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_entry), g_entry_warning_stream, TRUE, TRUE, 0);
  
  g_entry_all_stream = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_entry), g_entry_all_stream, TRUE, TRUE, 0);
  
  /* Browse buttons for file selection  */
  l_button_debug_stream = gtk_button_new_with_label (_("Browse"));
  g_object_set_data(G_OBJECT (l_button_debug_stream), "FileSelect", _("Debug stream file Selection"));
  g_signal_connect(G_OBJECT(l_button_debug_stream), "clicked",
                   G_CALLBACK(file_selection), g_entry_debug_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_browse), l_button_debug_stream, TRUE, FALSE, 0);
  
  l_button_fatal_stream = gtk_button_new_with_label (_("Browse"));
  g_object_set_data(G_OBJECT(l_button_fatal_stream), "FileSelect", _("Fatal stream file Selection"));
  g_signal_connect(G_OBJECT(l_button_fatal_stream), "clicked",
                   G_CALLBACK(file_selection), g_entry_fatal_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_browse), l_button_fatal_stream, TRUE, FALSE, 0);
  
  l_button_render_stream = gtk_button_new_with_label (_("Browse"));
  g_object_set_data(G_OBJECT (l_button_render_stream), "FileSelect", _("Render stream file Selection"));
  g_signal_connect(G_OBJECT(l_button_render_stream), "clicked",
                   G_CALLBACK(file_selection), g_entry_render_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_browse), l_button_render_stream, TRUE, FALSE, 0);
  
  l_button_stat_stream = gtk_button_new_with_label (_("Browse"));
  g_object_set_data(G_OBJECT(l_button_stat_stream), "FileSelect", _("Stat stream file Selection"));
  g_signal_connect(G_OBJECT(l_button_stat_stream), "clicked",
                   G_CALLBACK(file_selection), g_entry_stat_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_browse), l_button_stat_stream, TRUE, FALSE, 0);
  
  l_button_warning_stream = gtk_button_new_with_label (_("Browse"));
  g_object_set_data(G_OBJECT(l_button_warning_stream), "FileSelect", _("Warning stream file Selection"));
  g_signal_connect(G_OBJECT(l_button_warning_stream), "clicked",
                   G_CALLBACK(file_selection), g_entry_warning_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_browse), l_button_warning_stream, TRUE, FALSE, 0);
  
  l_button_all_stream = gtk_button_new_with_label (_("Browse"));
  g_object_set_data(G_OBJECT (l_button_all_stream), "FileSelect", _("All stream file Selection"));
  g_signal_connect(G_OBJECT(l_button_all_stream), "clicked",
                   G_CALLBACK(file_selection), g_entry_all_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_browse), l_button_all_stream, TRUE, FALSE, 0);

  /* View buttons for the messages files  */
  l_button_debug_view = gtk_button_new_with_label (_("View"));
  g_object_set_data(G_OBJECT(l_button_debug_view), "FileSelect", _("View Debug stream file"));
  g_signal_connect(G_OBJECT(l_button_debug_view), "clicked",
                   G_CALLBACK(file_view), g_entry_debug_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_view), l_button_debug_view, TRUE, FALSE, 0);
  
  l_button_fatal_view = gtk_button_new_with_label (_("View"));
  g_object_set_data(G_OBJECT(l_button_fatal_view), "FileSelect", _("View fatal stream file"));
  g_signal_connect(G_OBJECT(l_button_fatal_view), "clicked",
                   G_CALLBACK(file_view), g_entry_fatal_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_view), l_button_fatal_view, TRUE, FALSE, 0);
  
  l_button_render_view = gtk_button_new_with_label (_("View"));
  g_object_set_data(G_OBJECT(l_button_render_view), "FileSelect", _("View render stream file"));
  g_signal_connect(G_OBJECT(l_button_render_view), "clicked",
                   G_CALLBACK(file_view), g_entry_render_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_view), l_button_render_view, TRUE, FALSE, 0);
  
  l_button_stat_view = gtk_button_new_with_label (_("View"));
  g_object_set_data(G_OBJECT(l_button_stat_view), "FileSelect", _("View stat stream file"));
  g_signal_connect(G_OBJECT(l_button_stat_view), "clicked",
                   G_CALLBACK(file_view), g_entry_stat_stream);
  gtk_box_pack_start (GTK_BOX(l_vbox_messages_view), l_button_stat_view, TRUE, FALSE, 0);
  
  l_button_warning_view = gtk_button_new_with_label (_("View"));
  g_object_set_data(G_OBJECT (l_button_warning_view), "FileSelect", _("View warning stream file"));
  g_signal_connect(G_OBJECT(l_button_warning_view), "clicked",
                   G_CALLBACK(file_view), g_entry_warning_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_view), l_button_warning_view, TRUE, FALSE, 0);
  
  l_button_all_view = gtk_button_new_with_label (_("View"));
  g_object_set_data(G_OBJECT(l_button_all_view), "FileSelect", _("View all stream file"));
  g_signal_connect(G_OBJECT(l_button_all_view), "clicked",
                   G_CALLBACK(file_view), g_entry_all_stream);
  gtk_box_pack_start (GTK_BOX (l_vbox_messages_view), l_button_all_view, TRUE, FALSE, 0);
}
/*** END OF create_message_page */

/****************************************************************
 * Name : 		create_library_page
 * parameters : Aucun
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 09/7/1999
 * Return : 
 * Comments :
 ****************************************************************/
void
create_library_page (GtkWidget *p_note_book)
{
  GtkWidget *l_label;
  GtkWidget *l_vbox_library;
  GtkWidget *l_hbox_library;
  GtkWidget *l_vbox_library_entry;
  GtkWidget *l_vbox_library_browse;
  GtkWidget *l_button_library_browse;
  GtkWidget *l_hbox_library_version;
  gint i;

  g_return_if_fail (p_note_book != NULL);

  l_label = gtk_label_new ("Library");

  l_vbox_library = gtk_vbox_new (FALSE,0);
  gtk_notebook_append_page (GTK_NOTEBOOK (p_note_book),l_vbox_library, l_label);

  l_hbox_library = gtk_hbox_new (FALSE,0);
  gtk_box_pack_start (GTK_BOX (l_vbox_library), l_hbox_library, TRUE, TRUE, 0);

  l_vbox_library_entry = gtk_vbox_new (TRUE, 3);   /* Library entry box  */
  gtk_box_pack_start (GTK_BOX (l_hbox_library), l_vbox_library_entry, TRUE, TRUE, 0);

  l_vbox_library_browse = gtk_vbox_new (TRUE, 3);  /* Browse button box  */
  gtk_box_pack_start (GTK_BOX (l_hbox_library), l_vbox_library_browse, FALSE, TRUE, 0);

  for (i=0; i<10; i++)
    {
      g_entry_library [i] = gtk_entry_new ();
      gtk_box_pack_start (GTK_BOX (l_vbox_library_entry), g_entry_library [i], TRUE, TRUE, 0);

      l_button_library_browse = gtk_button_new_with_label (_("Browse"));
      g_object_set_data(G_OBJECT(l_button_library_browse), "FileSelect", _("Library file Selection"));
      g_signal_connect(G_OBJECT(l_button_library_browse), "clicked",
                       G_CALLBACK(file_selection), g_entry_library [i]);
      gtk_box_pack_start (GTK_BOX (l_vbox_library_browse), l_button_library_browse, TRUE, FALSE, 0);
    }
  
  l_hbox_library_version = gtk_hbox_new (FALSE, 0);  /* Pov file version box  */
  gtk_box_pack_start (GTK_BOX (l_vbox_library),l_hbox_library_version , TRUE, FALSE, 0);
  
  l_label = gtk_label_new (_("POV File Version"));
  gtk_box_pack_start (GTK_BOX (l_hbox_library_version), l_label, TRUE, FALSE, 0);
  
  g_entry_library_version = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (g_entry_library_version), g_entry_pov_version);
  gtk_box_pack_start (GTK_BOX (l_hbox_library_version), g_entry_library_version, TRUE, FALSE, 0);
}
/*** END OF create_library_page */

/****************************************************************
 * Name : 		create_pvm_page
 * parameters : Aucun
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 09/7/1999
 * Return : 
 * Comments :
 ****************************************************************/
void
create_pvm_page (GtkWidget *p_note_book)
{
  GtkWidget *l_label;
  GtkWidget *l_vbox_pvm;
  GtkWidget *l_button_add_host;
  GtkWidget *l_button_remove_host;
  GtkWidget *l_scrolled_window_pvm;
  GtkWidget *l_hbox_pvm_num;
  GtkWidget *l_hbox_pvm_size;
  GtkWidget *l_hbox_pvm_host_add;
  GtkWidget *l_entry_pvm_host;
  gchar     *l_title[4] = {_("Host")};
  GtkWidget *l_hscale_pvm_nice;

  g_return_if_fail (p_note_book != NULL);

  l_label = gtk_label_new ("PVM");
  l_vbox_pvm = gtk_vbox_new (FALSE,0);
  gtk_notebook_append_page (GTK_NOTEBOOK (p_note_book),l_vbox_pvm, l_label);
  g_check_pvm = gtk_check_button_new_with_label (_("PVM"));
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm), g_check_pvm, TRUE, FALSE, 0);
  g_signal_connect(G_OBJECT(g_check_pvm), "clicked",
                   G_CALLBACK(switch_option), &g_pvm);

  /* A horizontal box for the block size  */
  l_hbox_pvm_size = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm), l_hbox_pvm_size, TRUE, FALSE, 0);

  l_label = gtk_label_new (_("Height :"));
  gtk_box_pack_start (GTK_BOX (l_hbox_pvm_size),l_label , TRUE, FALSE, 0);

  g_adjust_pvm_height = gtk_adjustment_new (32, 0, 10000, 1, 10, 50);
  g_entry_pvm_height = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_pvm_height), 4, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_pvm_size), g_entry_pvm_height, TRUE, FALSE, 0);

  l_label = gtk_label_new (_("Width :"));
  gtk_box_pack_start (GTK_BOX (l_hbox_pvm_size),l_label , TRUE, FALSE, 0);

  g_adjust_pvm_width = gtk_adjustment_new (32, 0, 10000, 1, 10, 50);
  g_entry_pvm_width = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_pvm_width), 4, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_pvm_size), g_entry_pvm_width, TRUE, FALSE, 0);
 
  /* A horizontal box for the number of task  */
  l_hbox_pvm_num = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm), l_hbox_pvm_num, TRUE, FALSE, 0);

  l_label = gtk_label_new (_("Number of tasks"));
  gtk_box_pack_start (GTK_BOX (l_hbox_pvm_num),l_label , TRUE, FALSE, 0);

  g_adjust_pvm_num = gtk_adjustment_new (32, 0, 10000, 1, 10, 50);
  g_entry_pvm_num = gtk_spin_button_new (GTK_ADJUSTMENT (g_adjust_pvm_num), 4, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_pvm_num), g_entry_pvm_num, TRUE, FALSE, 0);

  /* Nice option  */
  l_label = gtk_label_new (_("Nice"));
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm), l_label, TRUE, TRUE, 0);

  g_adjust_pvm_nice = gtk_adjustment_new (g_pvm_nice_value, -20, 19, 1, 1, 1);
  l_hscale_pvm_nice = gtk_hscale_new (GTK_ADJUSTMENT (g_adjust_pvm_nice));
  gtk_scale_set_digits (GTK_SCALE (l_hscale_pvm_nice), 0);
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm), l_hscale_pvm_nice, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(g_adjust_pvm_nice), "value_changed",
                   G_CALLBACK(adjustment_value_changed), &g_pvm_nice_value);

  /* Entry field for the command option of pvm  */
  l_label = gtk_label_new (_("Command"));
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm),l_label , TRUE, FALSE, 0);

  g_entry_pvm_cmd = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm), g_entry_pvm_cmd, TRUE, FALSE, 0);

  /* Entry field for the directory option of pvm  */
  l_label = gtk_label_new (_("Directory"));
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm),l_label , TRUE, FALSE, 0);

  g_entry_pvm_dir = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm), g_entry_pvm_dir, TRUE, FALSE, 0);

  /* List of the hosts  */
  l_scrolled_window_pvm = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_set_usize (l_scrolled_window_pvm, 300, 150);
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm), l_scrolled_window_pvm, TRUE, TRUE, 0);

  /* Create the GtkList widget  */
  g_gtklist_hosts = gtk_clist_new_with_titles (1, l_title);
  gtk_container_add (GTK_CONTAINER (l_scrolled_window_pvm), g_gtklist_hosts);

  gtk_clist_set_column_width (GTK_CLIST (g_gtklist_hosts), 0, 35);
  gtk_clist_set_selection_mode (GTK_CLIST (g_gtklist_hosts), GTK_SELECTION_MULTIPLE);

  /* A horizontal box for adding an Entry in the list  */
  l_hbox_pvm_host_add = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm), l_hbox_pvm_host_add, TRUE, FALSE, 0);

  /* Entry field to add a host to the list  */
  l_label = gtk_label_new (_("Host to add :"));
  gtk_box_pack_start (GTK_BOX (l_hbox_pvm_host_add), l_label, TRUE, FALSE, 0);

  l_entry_pvm_host = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_hbox_pvm_host_add), l_entry_pvm_host, TRUE, FALSE, 0);

  /* Buttons creation for the list managment  */
  l_button_add_host = gtk_button_new_with_label (_("Add"));
  gtk_box_pack_start (GTK_BOX (l_hbox_pvm_host_add), l_button_add_host, TRUE, FALSE, 0);
  g_signal_connect(G_OBJECT(l_button_add_host), "clicked",
                   G_CALLBACK(add_pvm_host), l_entry_pvm_host);
  gtk_tooltips_set_tip (g_tooltips, l_button_add_host, _("Add a host to the list"), NULL);

  l_button_remove_host = gtk_button_new_with_label (_("Remove the selected host(s)"));
  gtk_box_pack_start (GTK_BOX (l_vbox_pvm), l_button_remove_host, TRUE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(l_button_remove_host), "clicked",
                           G_CALLBACK(delete_hosts), g_gtklist_hosts);
  gtk_tooltips_set_tip (g_tooltips, l_button_remove_host, _("Remove a host from the list"), NULL);
}
/*** END OF create_pvm_page */

/****************************************************************
 * Name : 		image_changed
 * parameters :
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 10/11/1999
 * Return : 
 * Comments :
 ****************************************************************/
void
image_changed (GtkWidget *p_note_book)
{
  printf ("bonjour\n");
}
/*** END OF image_changed  */

/****************************************************************
 * Name : 		main
 * parameters : p_pf_environ: table of char pointer on the environment variablles
 * Author : Philippe DAVID
 * Update (MM/DD/YY) : 10/16/1999
 * Return : 
 * Comments :
 ****************************************************************/
int
main (int argc, char *argv[], char *p_pf_environ[])
{
  GtkWidget *l_separator;

  GtkWidget *l_vbox;             /* The main box of the application */
  GtkWidget *l_hbox_buttons;
  GSList    *l_grp_radio_file;
  GtkWidget *l_note_book;

  /*** Variables for the header  */
  GtkWidget *l_hbox_head;
  GtkWidget *l_vbox_head_check;
  GtkWidget *l_vbox_head_entry;
  GtkWidget *l_vbox_head_browse;

  GtkWidget *l_button_outfile;
  GtkWidget *l_button_pov_file;

  GtkWidget *l_button_render;
  GtkWidget *l_button_abort;
  GtkWidget *l_button_control;

  GtkWidget *l_menu_bar;

  GtkWidget *l_file_menu;
  GtkWidget *l_file_item;
  GtkWidget *l_file_menu_item_file;
  GtkWidget *l_file_menu_item_save;
  GtkWidget *l_file_menu_item_save_as;
  GtkWidget *l_file_menu_separator;
  GtkWidget *l_file_menu_item_quit;

  GtkWidget *l_option_menu;
  GtkWidget *l_option_item;
  GtkWidget *l_option_menu_item_option;
  GtkWidget *l_option_menu_item_default;

  GtkWidget *l_help_menu;
  GtkWidget *l_help_item;
  GtkWidget *l_help_menu_item_apropos;
  GtkWidget *l_help_menu_item_help;

  GtkWidget *l_jobs_menu;
  GtkWidget *l_jobs_item;
  GtkWidget *l_jobs_menu_item_control;

  gtk_init (&argc, &argv);

  gtk_rc_parse ("~/.povfront/.povfrontrc");
  g_pf_environ = p_pf_environ;
  init_programme ();                        /* Some variables to set  */

  /*** Main window creation  */
  povf_parse_options (argc, argv);
  g_main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_wmclass (GTK_WINDOW (g_main_window), "povfront", "PovFront");
  gtk_widget_set_name (g_main_window, "main window");

  g_signal_connect(G_OBJECT(g_main_window), "delete_event",
                   G_CALLBACK(delete_event), NULL);
  g_signal_connect(G_OBJECT(g_main_window), "destroy",
                   G_CALLBACK(destroy), NULL);
  gtk_container_set_border_width(GTK_CONTAINER(g_main_window), 3);
  gtk_window_set_title(GTK_WINDOW(g_main_window), _("PovFront : a front end for povray"));

  /*** declaration of the main verticale box of the window  */
  l_vbox = gtk_vbox_new (FALSE, 0);

  gtk_container_add (GTK_CONTAINER (g_main_window), l_vbox);

  /*** Menu creation  */
  /* Menu _file  */
  l_file_menu = gtk_menu_new ();
  
  l_file_menu_item_file = gtk_menu_item_new_with_label (_("Load Config."));
  g_object_set_data(G_OBJECT(l_file_menu_item_file), "FileSelect", _("Load configuration file"));
  g_signal_connect_swapped(G_OBJECT(l_file_menu_item_file), "activate",
                           G_CALLBACK(load_config), NULL);
  gtk_menu_append (GTK_MENU (l_file_menu), l_file_menu_item_file);
  gtk_widget_show (l_file_menu_item_file);

  l_file_menu_item_save = gtk_menu_item_new_with_label (_("Save Config."));
  g_object_set_data(G_OBJECT(l_file_menu_item_save), "FileSelect", _("Save configuration file"));
  g_signal_connect_swapped(G_OBJECT(l_file_menu_item_save), "activate",
                           G_CALLBACK(save_config), NULL);
  gtk_menu_append (GTK_MENU (l_file_menu), l_file_menu_item_save);
  gtk_widget_show (l_file_menu_item_save);

  l_file_menu_item_save_as = gtk_menu_item_new_with_label (_("Save Config. as"));
  g_object_set_data(G_OBJECT(l_file_menu_item_save_as), "FileSelect", _("Save configuration file"));
  g_signal_connect_swapped(G_OBJECT(l_file_menu_item_save_as), "activate",
                           G_CALLBACK(save_config), NULL);
  gtk_menu_append (GTK_MENU (l_file_menu), l_file_menu_item_save_as);
  gtk_widget_show (l_file_menu_item_save_as);

  l_file_menu_separator = gtk_menu_item_new();
  gtk_menu_append (GTK_MENU (l_file_menu), l_file_menu_separator);
  gtk_widget_show (l_file_menu_separator);

  l_file_menu_item_quit = gtk_menu_item_new_with_label (_("Quit"));
  g_signal_connect_swapped(G_OBJECT(l_file_menu_item_quit), "activate",
                           G_CALLBACK(destroy), g_main_window);
  gtk_menu_append (GTK_MENU (l_file_menu), l_file_menu_item_quit);
  gtk_widget_show (l_file_menu_item_quit);

  /* Option menu  */
  l_option_menu = gtk_menu_new ();

  l_option_menu_item_option = gtk_menu_item_new_with_label (_("Configure"));
  g_signal_connect_swapped(G_OBJECT(l_option_menu_item_option), "activate",
                           G_CALLBACK(options), NULL);
  gtk_menu_append (GTK_MENU (l_option_menu), l_option_menu_item_option);
  gtk_widget_show (l_option_menu_item_option);

  l_option_menu_item_default = gtk_menu_item_new_with_label (_("Save default"));
  g_signal_connect_swapped(G_OBJECT(l_option_menu_item_default), "activate",
                           G_CALLBACK(write_default_config), NULL);
  gtk_menu_append (GTK_MENU (l_option_menu), l_option_menu_item_default);
  gtk_widget_show (l_option_menu_item_default);

  /* Jobs menu  */
  l_jobs_menu = gtk_menu_new ();

  l_jobs_menu_item_control = gtk_menu_item_new_with_label (_("Control"));
  g_signal_connect_swapped(G_OBJECT(l_jobs_menu_item_control), "activate",
                           G_CALLBACK(control_on), NULL);
  gtk_menu_append (GTK_MENU (l_jobs_menu), l_jobs_menu_item_control);
  gtk_widget_show (l_jobs_menu_item_control);

  /* Help menu  */
  l_help_menu = gtk_menu_new ();

  l_help_menu_item_apropos = gtk_menu_item_new_with_label (_("About..."));
  g_signal_connect_swapped(G_OBJECT(l_help_menu_item_apropos), "activate",
                           G_CALLBACK(a_propos), NULL);
  gtk_menu_append (GTK_MENU (l_help_menu), l_help_menu_item_apropos);
  gtk_widget_show (l_help_menu_item_apropos);

  l_help_menu_item_help = gtk_menu_item_new_with_label (_("Help"));
  g_signal_connect_swapped(G_OBJECT(l_help_menu_item_help), "activate",
                           G_CALLBACK(help), NULL);
  gtk_menu_append (GTK_MENU (l_help_menu), l_help_menu_item_help);
  gtk_widget_show (l_help_menu_item_help);

  /* Menu bar  */
  l_menu_bar = gtk_menu_bar_new ();

  l_file_item = gtk_menu_item_new_with_label (_("File"));
  gtk_menu_bar_append (GTK_MENU_BAR (l_menu_bar), l_file_item);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (l_file_item), l_file_menu);
  gtk_widget_show (l_file_item);

  l_option_item = gtk_menu_item_new_with_label (_("Options"));
  gtk_menu_bar_append (GTK_MENU_BAR (l_menu_bar), l_option_item);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (l_option_item), l_option_menu);
  gtk_widget_show (l_option_item);

  l_jobs_item = gtk_menu_item_new_with_label (_("Jobs"));
  gtk_menu_bar_append (GTK_MENU_BAR (l_menu_bar), l_jobs_item);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (l_jobs_item), l_jobs_menu);
  gtk_widget_show (l_jobs_item);

  l_help_item = gtk_menu_item_new_with_label (_("Help"));
  gtk_menu_item_right_justify (GTK_MENU_ITEM (l_help_item));
  gtk_menu_bar_append (GTK_MENU_BAR (l_menu_bar), l_help_item);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (l_help_item), l_help_menu);
  gtk_widget_show (l_help_item);

  gtk_box_pack_start (GTK_BOX (l_vbox), l_menu_bar, FALSE, FALSE, 0);
  gtk_widget_show (l_menu_bar );

  /* POV file selection  */
  l_hbox_head = gtk_hbox_new (FALSE, 0);            /* Box of the _header  */
  l_vbox_head_check = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_head), l_vbox_head_check, FALSE, TRUE, 0);
  l_vbox_head_entry = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_head), l_vbox_head_entry, TRUE, TRUE, 0);
  l_vbox_head_browse = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (l_hbox_head), l_vbox_head_browse, FALSE, FALSE, 0);

  /* Now we put the boxes in the header if it's not the plugin mode  */
  if (!g_plugin)
    {
      gtk_box_pack_start (GTK_BOX (l_vbox), l_hbox_head, TRUE, TRUE, 0);
    }
  else
    {
      g_plugin_title = gtk_label_new ("");
      gtk_box_pack_start (GTK_BOX (l_vbox), g_plugin_title, TRUE, TRUE, 0);
      g_plugin_subtitle = gtk_label_new ("");
      gtk_box_pack_start (GTK_BOX (l_vbox), g_plugin_subtitle, TRUE, TRUE, 0);
    }
  g_check_pov_file = gtk_check_button_new_with_label (_("Pov file"));
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (g_check_pov_file), g_input_pov_file);
  gtk_box_pack_start (GTK_BOX (l_vbox_head_check), g_check_pov_file, FALSE, FALSE, 0);
  g_tooltips = gtk_tooltips_new ();
  gtk_tooltips_set_tip (g_tooltips, g_check_pov_file, _("Enable/Disable parsing a pov file as input"), NULL);
  g_signal_connect(G_OBJECT(g_check_pov_file), "clicked",
                   G_CALLBACK(switch_option), &g_input_pov_file );

  g_entry_pov_file = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_head_entry), g_entry_pov_file, FALSE, FALSE, 0);
  if (!g_plugin)
    {
      l_button_pov_file = gtk_button_new_with_label (_("Browse"));
      gtk_box_pack_start (GTK_BOX (l_vbox_head_browse), l_button_pov_file, FALSE, FALSE, 0);
      g_object_set_data(G_OBJECT(l_button_pov_file), "FileSelect", _("POV file Selection"));
      g_signal_connect(G_OBJECT(l_button_pov_file), "clicked",
                       G_CALLBACK(file_selection), g_entry_pov_file);
    }

  g_radio_out_file1 = gtk_radio_button_new_with_label (NULL, _("Output Image"));
  gtk_box_pack_start (GTK_BOX (l_vbox_head_check), g_radio_out_file1, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(g_radio_out_file1), "clicked",
                   G_CALLBACK(switch_out_file), "1");
  l_grp_radio_file = gtk_radio_button_group (GTK_RADIO_BUTTON (g_radio_out_file1));
  
  g_radio_out_file2 = gtk_radio_button_new_with_label (l_grp_radio_file, _("Redirect."));
  gtk_box_pack_start (GTK_BOX (l_vbox_head_check), g_radio_out_file2, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(g_radio_out_file2), "clicked",
                   G_CALLBACK(switch_out_file), "2");
  l_grp_radio_file = gtk_radio_button_group (GTK_RADIO_BUTTON (g_radio_out_file2));
  
  g_radio_out_file3 = gtk_radio_button_new_with_label (l_grp_radio_file, _("None"));
  gtk_box_pack_start (GTK_BOX (l_vbox_head_check), g_radio_out_file3, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(g_radio_out_file3), "clicked",
                   G_CALLBACK(switch_out_file), "3");

  g_entry_outfile = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_head_entry), g_entry_outfile, FALSE, FALSE, 0);

  if (!g_plugin)
    {
      l_button_outfile = gtk_button_new_with_label (_("Browse"));
      g_object_set_data(G_OBJECT(l_button_outfile), "FileSelect", _("Image File Selection"));
      gtk_box_pack_start (GTK_BOX (l_vbox_head_browse), l_button_outfile, FALSE, FALSE, 0);
      g_signal_connect(G_OBJECT(l_button_outfile), "clicked",
                       G_CALLBACK(file_selection), g_entry_outfile);
    }
  /* Entry for redirection of output image  */
  g_entry_outgrep = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox_head_entry), g_entry_outgrep, FALSE, FALSE, 0);

  l_separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox), l_separator, TRUE, TRUE, 5);

  l_hbox_buttons = gtk_hbox_new (TRUE, 0);
  /* l_hbox_buttons = gtk_handle_box_new ();*/
  gtk_widget_set_name (l_hbox_buttons, "main buttons");
  gtk_box_pack_start (GTK_BOX (l_vbox), l_hbox_buttons, TRUE, TRUE, 0);

  l_button_render = gtk_button_new_with_label (_("Render"));
  gtk_widget_set_name (l_button_render, "render");
  gtk_box_pack_start (GTK_BOX (l_hbox_buttons), l_button_render, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(l_button_render), "clicked",
                   G_CALLBACK(render), NULL );

  l_button_abort = gtk_button_new_with_label (_("Abort the last job"));
  gtk_widget_set_name (l_button_abort, "abort");
  gtk_box_pack_start (GTK_BOX (l_hbox_buttons), l_button_abort, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(l_button_abort), "clicked",
                   G_CALLBACK(pf_abort), NULL );

  l_button_control = gtk_button_new_with_label (_("Job control"));
  gtk_widget_set_name (l_button_abort, "control");
  gtk_box_pack_start (GTK_BOX (l_hbox_buttons), l_button_control, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(l_button_control), "clicked",
                   G_CALLBACK(control_on), NULL );

  l_separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox), l_separator, TRUE, TRUE, 5);

  /*** NoteBook creation  */
  l_note_book = gtk_notebook_new ();
  gtk_box_pack_start (GTK_BOX (l_vbox), l_note_book, TRUE, TRUE, 0);

  create_output_page (l_note_book);

  create_quality_page (l_note_book);
  
  create_animation_page (l_note_book);

  create_script_page (l_note_book);
  
  create_message_page (l_note_book);
  
  create_library_page (l_note_book);
  
  create_pvm_page (l_note_book);

  /***  END of notebook creation  */

  if (strlen (g_config_file_to_parse) == 0)
    {
      povfront_warning (_("You don't have a default configuration file, don't forget to save one\n"));
    }
  else
    {
      load_config_file (g_config_file_to_parse);
    }

  gtk_widget_show_all (g_main_window);
  gtk_main ();

  return 0;
}
/*** END OF povfront.c  */
