
/* Quad precision square root calculation
 *
 * Copyright (c) 2008, Andrew Vaught
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * * The name of Andrew Vaught may not be used to endorse or promote
 *   products derived from this software without specific prior
 *   written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE. */

#include "runtime.h"

#define power_r16_i4 prefix(power_r16_i4)


/* int_power16()-- Core of integer powers. */

static int_power16(unpacked16 *base, long long e0, unpacked16 *result) {
int negative, negative_exp;
unsigned long long exp;
unpacked16 one;

    if (e0 >= 0) {
	negative_exp = 0;
	exp = e0;

    } else {
	exp = -e0;
	negative_exp = 1;
    }

    result->sign = 0;
    result->exp  = 16383;

    result->m[0] = 0x10000;
    result->m[1] = 0;
    result->m[2] = 0;
    result->m[3] = 0;

    if (((base->exp == 0  && base->m[0] == 0  && base->m[1] == 0 &&
	  base->m[2] == 0 && base->m[3] == 0) && exp == 0) || exp == 0 ||
	(base->exp == 16383 && !base->sign     && base->m[0] == 0x10000 &&
	 base->m[1] == 0    && base->m[2] == 0 && base->m[3] == 0))
	return;

    if (base->exp == 16383 && base->sign &&
	base->m[0] == 0x10000 && base->m[1] == 0 &&
	base->m[2] == 0       && base->m[3] == 0) {

	result->sign = exp & 1;
	return;
    }

    negative = base->sign;
    base->sign = 0;

    for(;;) {
	if (exp & 1)
	    multiply_unpacked(result, base, result);

	exp = exp >> 1;
	if (exp == 0)
	    break;

	multiply_unpacked(base, base, base);
    }

    if (negative)
	result->sign = !result->sign;

    if (negative_exp) {
	one.sign = 0;
	one.exp  = 16383;

	one.m[0] = 0x10000;
	one.m[1] = 0;
	one.m[2] = 0;
	one.m[3] = 0;

	divide_unpacked(result, &one, &one);
	*result = one;
    }
}



#if QUAD_POINTER

#define power_r16_i1 prefix(power_r16_i1)

void power_r16_i1(packed16 *result, void *base, char *exp) {
unpacked16 b, p;

    unpack_quad(base, &b.m, &b.exp, &b.sign);

    int_power16(&b, *exp, &p);

    pack_real_16(result, p.m, &p.exp, &p.sign);
}

#define power_r16_i2 prefix(power_r16_i2)

void power_r16_i2(packed16 *result, void *base, short *exp) {
unpacked16 b, p;

    unpack_quad(base, &b.m, &b.exp, &b.sign);

    int_power16(&b, *exp, &p);

    pack_real_16(result, p.m, &p.exp, &p.sign);
}

#define power_r16_i4 prefix(power_r16_i4)

void power_r16_i4(packed16 *result, void *base, int *exp) {
unpacked16 b, p;

    unpack_quad(base, &b.m, &b.exp, &b.sign);

    int_power16(&b, *exp, &p);

    pack_real_16(result, p.m, &p.exp, &p.sign);
}

#define power_r16_i8 prefix(power_r16_i8)

void power_r16_i8(packed16 *result, void *base, long long *exp) {
unpacked16 b, p;

    unpack_quad(base, &b.m, &b.exp, &b.sign);

    int_power16(&b, *exp, &p);

    pack_real_16(result, p.m, &p.exp, &p.sign);
}

#else

#define power_r16_i1 prefix(power_r16_i1)

packed16 power_r16_i1(void *base, char *exp) {
packed16 result;
unpacked16 b, p;

    unpack_quad(base, b.m, &b.exp, &b.sign);

    int_power16(&b, *exp, &p);

    pack_real_16(&result, p.m, &p.exp, &p.sign);
    return result;
}

#define power_r16_i2 prefix(power_r16_i2)

packed16 power_r16_i2(void *base, short *exp) {
packed16 result;
unpacked16 b, p;

    unpack_quad(base, b.m, &b.exp, &b.sign);

    int_power16(&b, *exp, &p);

    pack_real_16(&result, p.m, &p.exp, &p.sign);
    return result;
}

#define power_r16_i4 prefix(power_r16_i4)

packed16 power_r16_i4(void *base, int *exp) {
packed16 result;
unpacked16 b, p;

    unpack_quad(base, b.m, &b.exp, &b.sign);

    int_power16(&b, *exp, &p);

    pack_real_16(&result, p.m, &p.exp, &p.sign);
    return result;
}

#define power_r16_i8 prefix(power_r16_i8)

packed16 power_r16_i8(void *base, long long *exp) {
packed16 result;
unpacked16 b, p;

    unpack_quad(base, b.m, &b.exp, &b.sign);

    int_power16(&b, *exp, &p);

    pack_real_16(&result, p.m, &p.exp, &p.sign);
    return result;
}

#endif

