
/*  References: "An Overview of Software Development for Special
                Functions", W. J. Cody, Lecture Notes in Mathematics,
                506, Numerical Analysis Dundee, 1975, G. A. Watson
                (ed.), Springer Verlag, Berlin, 1976.

                Computer Approximations, Hart, Et. Al., Wiley and
                sons, New York, 1968.

    Latest modification: October 12, 1989

    Authors: W. J. Cody and L. Stoltz
            Applied Mathematics Division
            Argonne National Laboratory
            Argonne, IL 60439

*/

#include <math.h>
#include "runtime.h"

#define EPS 2.22e-16
#define PI 3.141592653589793234

#define gamma_8 prefix(gamma_8)

double gamma_8(double *xp) {
double fact, x, y, y1, y2, ysq, z, xnum, xden, sum, res;
int i, n, parity;

static double p[8] = {
    -1.71618513886549492533811e+0,   2.47656508055759199108314e+1,
    -3.79804256470945635097577e+2,   6.29331155312818442661052e+2,
    8.66966202790413211295064e+2,   -3.14512729688483675254357e+4,
    -3.61444134186911729807069e+4,   6.64561438202405440627855e+4 },

    q[8] = {
	-3.08402300119738975254353e+1,   3.15350626979604161529144e+2,
	-1.01515636749021914166146e+3,  -3.10777167157231109440444e+3,
	2.25381184209801510330112e+4,   4.75584627752788110767815e+3,
	-1.34659959864969306392456e+5,  -1.15132259675553483497211e+5 },

    c[7] = {
	-1.910444077728e-03,             8.4171387781295e-04,
	-5.952379913043012e-04,          7.93650793500350248e-04,
	-2.777777777777681622553e-03,    8.333333333333333331554247e-02,
	5.7083835261e-03 };

    x = *xp;
    parity = 0;
    fact = 1.0;
    n = 0;
    y = x;

    if (y <= 0.0) {
	y = -x;
	y1 = y;

	y1 = (int) y1;
	//trunc_8(&y1);

	res = y - y1;
	if (res != 0.0) {
	    y2 = y1 * 0.5;

	    y2 = (int) y2;
	    //trunc_8(&y2);

	    y2 = 2.0 * y2;

	    if (y1 != y2)
		parity = 1;

	    fact = -PI / sin(PI*res);
	    y = y + 1.0;

	} else {
	    build_infinity(0, (char *) &res, 8);
	    return res;
	}
    }

    /* Positive argument */

    if (y <= 2.23e-308)
	build_infinity(0, (char *) &res, 8);

    else if (y <= EPS)
	res = 1.0 / y;

    else if (y <= 12.0) {
	y1 = y;
	if (y < 1.0) {
	    z = y;
	    y = y + 1.0;

	} else {
	    n = ((int) y) - 1;
	    y = y - n;
	    z = y - 1.0;
	}

	/* 1.0 <= y <= 2.0 */

	xnum = 0.0;
	xden = 1.0;

	for(i=0; i<8; i++) {
	    xnum = (xnum + p[i])*z;
	    xden = xden*z + q[i];
	}

	res = (xnum / xden) + 1.0;

	if (y1 < y)
	    res = res / y1;

	else if (y1 > y) {
	    for(i=0; i<n; i++) {
		res = res * y;
		y = y + 1.0;
	    }
	}

    } else {
	ysq = y*y;
	sum = c[6];

	for(i=0; i<6; i++)
	    sum = sum / ysq + c[i];

	sum = sum/y - y + 0.9189385332046727417803297;
	sum = sum + (y-0.5)*log(y);

	res = exp(sum);
    }

    if (parity)
	res = -res;

    if (fact != 1.0)
	res = fact / res;

    return res;
}



#define gamma_4 prefix(gamma_4)

float gamma_4(float *p) {
double x;

    x = (double) *p;
    return gamma_8(&x);
}



#define log_gamma_8 prefix(log_gamma_8)

double log_gamma_8(double *xp) {
double y, res, corr, xm1, xm2, xm4, xden, xnum, ysq;
int i;

static double p1[8] = {
      4.945235359296727046734888e+0,  2.018112620856775083915565e+2,
      2.290838373831346393026739e+3,  1.131967205903380828685045e+4,
      2.855724635671635335736389e+4,  3.848496228443793359990269e+4,
      2.637748787624195437963534e+4,  7.225813979700288197698961e+3 },
    q1[8] = {
	6.748212550303777196073036e+1,  1.113332393857199323513008e+3,
	7.738757056935398733233834e+3,  2.763987074403340708898585e+4,
	5.499310206226157329794414e+4,  6.161122180066002127833352e+4,
	3.635127591501940507276287e+4,  8.785536302431013170870835e+3 },

    p2[8] = {
	4.974607845568932035012064e+0,  5.424138599891070494101986e+2,
	1.550693864978364947665077e+4,  1.847932904445632425417223e+5,
	1.088204769468828767498470e+6,  3.338152967987029735917223e+6,
	5.106661678927352456275255e+6,  3.074109054850539556250927e+6 },

    q2[8] = {
	1.830328399370592604055942e+2,  7.765049321445005871323047e+3,
	1.331903827966074194402448e+5,  1.136705821321969608938755e+6,
	5.267964117437946917577538e+6,  1.346701454311101692290052e+7,
	1.782736530353274213975932e+7,  9.533095591844353613395747e+6 },

    p4[8] = {
	1.474502166059939948905062e+4,   2.426813369486704502836312e+6,
	1.214755574045093227939592e+8,   2.663432449630976949898078e+9,
	2.940378956634553899906876e+10,  1.702665737765398868392998e+11,
	4.926125793377430887588120e+11,  5.606251856223951465078242e+11 },

    q4[8] = {
	2.690530175870899333379843e+3,   6.393885654300092398984238e+5,
	4.135599930241388052042842e+7,   1.120872109616147941376570e+9,
	1.488613728678813811542398e+10,  1.016803586272438228077304e+11,
	3.417476345507377132798597e+11,  4.463158187419713286462081e+11 },

    c[7] = {
	-1.910444077728e-03,            8.4171387781295e-04,
	-5.952379913043012e-04,         7.93650793500350248e-04,
	-2.777777777777681622553e-03,   8.333333333333333331554247e-02,
	5.7083835261e-03 };

    y = *xp; 

    if (y < 0.0)
	build_infinity(0, (char *) &res, 8);
   
    else if (y < 2.22e-16)
	res = -log(y);

    else if (y < 1.5) {
	if (y < 0.6796875) {
	    corr = -log(y);
	    xm1 = y;
	} else {
	    corr = 0.0;
	    xm1 = (y - 0.5) - 0.5;
	}
    
	xden = 1.0;
	xnum = 0.0;

	if (y < 0.5 || y > 0.6796875) {
	    for(i=0; i<8; i++) {
		xnum = xnum * xm1 + p1[i];
		xden = xden * xm1 + q1[i];
	    }

	    res = corr +
                 (xm1 * (-5.772156649015328605195174e-1 + xm1*(xnum/xden)));

	} else {
	    xm2 = (y - 0.5) - 0.5;

	    for(i=0; i<8; i++) {
		xnum = xnum * xm2 + p2[i];
		xden = xden * xm2 + q2[i];
	    }

	    res = corr +
                  xm2 * (4.227843350984671393993777e-1 + xm2*(xnum/xden));
	}

    } else if (y < 4.0) {
	xm2 = y - 2.0;
	xnum = 0.0;
	xden = 1.0;

	for(i=0; i<8; i++) {
	    xnum = xnum * xm2 + p2[i];
	    xden = xden * xm2 + q2[i];
	}

	res = xm2 * (4.227843350984671393993777e-1 + xm2*(xnum/xden));

    } else if (y < 12.0) {
	xm4 = y - 4.0;
	xden = -1.0;
	xnum = 0.0;

	for(i=0; i<8; i++) {
	    xnum = xnum * xm4 + p4[i];
	    xden = xden * xm4 + q4[i];
	}

	res = 1.791759469228055000094023 + xm4*(xnum/xden);

    } else {  /* y > 12.0 */
	res = 0.0;
	if (y < 2.25e+76) {
	    res = c[6];
	    ysq = y*y;

	    for(i=0; i<6; i++)
		res = res/ysq + c[i];

	    res = res / y;
	    corr = log(y);

	    res = res + 0.9189385332046727417803297 - 0.5*corr;
	    res = res + y*(corr-1.0);
	}
    }

    return res;
}


#define log_gamma_4 prefix(log_gamma_4)

float log_gamma_4(float *p) {
double x;

    x = (double) *p;
    return log_gamma_8(&x);
}

