
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/

/* Write integers */


#ifndef FIRST
#define FIRST 1

#include <string.h>
#include "runtime.h"

/* Result of ascii conversions.  The width must hold than the widest
 * binary output from the widest integer. */

static char num_buffer[128+2];

/* write_int()-- Write an integer in one of several formats */

static void write_int(fnode *f, char *buffer, int need_sign, int sign_n) {
int w, m, digits, nsign, nzero, nblank;
sign_t sign;
char *p;

    w = f->u.integer.w;
    m = f->u.integer.m;

    /* Special case */

    if (m == 0 && sign_n == 0) {
	if (w == 0) w = 1;

	p = write_block(w);
	if (p == NULL)
	    return;

	memset(p, ' ', w);
	return;
    }

    sign = need_sign ? calculate_sign(sign_n < 0) : SIGN_NONE;

    nsign = (sign == SIGN_NONE) ? 0 : 1;
    digits = strlen(buffer);

    /* Select a width if none was specified.  The idea here is to
     * always print something. */

    if (w == 0)
	w = ((digits < m) ? m : digits) + nsign;

    p = write_block(w);
    if (p == NULL)
	return;

    nzero = 0;
    if (digits < m)
	nzero = m - digits;

    /* See if things will work */

    nblank = w - (nsign + nzero + digits);

    if (nblank < 0) {
	star_fill(p, w);
	return;
    }

    memset(p, ' ', nblank);
    p += nblank;

    switch(sign) {
    case SIGN_PLUS:   *p++ = '+';   break;
    case SIGN_MINUS:  *p++ = '-';   break;
    case SIGN_NONE:                 break;
    }

    memset(p, '0', nzero);
    p += nzero;

    memcpy(p, buffer, digits);
}

#endif


#ifdef SUBROUTINE_B

static char *SUBROUTINE_B(unsigned TYPE *v, int *sign) {
unsigned TYPE n;
char *p;

    n = *v;
    if (n == 0) {
	*sign = 0;
	num_buffer[0] = '0';
	num_buffer[1] = '\0';
	return num_buffer;
    }

    *sign = 1;

    p = num_buffer + sizeof(num_buffer) - 1;
    *p-- = '\0';

    while(n != 0) {
	*p-- = '0' + (n & 1);
	n >>= 1;
    }

    return ++p;
}


static char *SUBROUTINE_O(unsigned TYPE *v, int *sign) {
unsigned TYPE n;
char *p;

    n = *v;
    if (n == 0) {
	*sign = 0;
	num_buffer[0] = '0';
	num_buffer[1] = '\0';
	return num_buffer;
    }

    *sign = 1;

    p = num_buffer + sizeof(num_buffer) - 1;
    *p-- = '\0';

    while(n != 0) {
	*p-- = '0' + (n & 0x07);
	n >>= 3;
    }

    return ++p;
}


static char *SUBROUTINE_Z(unsigned TYPE *v, int *sign) {
unsigned TYPE n;
int digit;
char *p;

    n = *v;
    if (n == 0) {
	*sign = 0;
	num_buffer[0] = '0';
	num_buffer[1] = '\0';
	return num_buffer;
    }

    *sign = 1;

    p = num_buffer + sizeof(num_buffer) - 1;
    *p-- = '\0';

    while(n != 0) {
	digit = n & 0xF;
	if (digit > 9)
	    digit += 'A' - '0' - 10;

	*p-- = '0' + digit;
	n >>= 4;
    }

    return ++p;
}


static char *SUBROUTINE_U(TYPE *v, int *sign) {
unsigned TYPE u;
TYPE v0;
char *p;

    v0 = *v;

    *sign = (v0 == 0)
	? 0
	: ((v0 > 0) ? 1 : -1);

    if (v0 == 0) { 
	num_buffer[0] = '0';
	num_buffer[1] = '\0';
	return num_buffer;
    }

    u = (v0 > 0) ? v0 : -v0;
    p = num_buffer + sizeof(num_buffer) - 1;
    *p-- = '\0';

    while(u != 0) {
	*p-- = '0' + (u % 10);
	u /= 10;
    }

    return ++p;
}


static char *SUBROUTINE_I(TYPE *v) {
signed TYPE v0;
unsigned TYPE u;
int negative;
char *p;

    v0 = *v;

    if (v0 == 0) { 
	num_buffer[0] = '0';
	num_buffer[1] = '\0';
	return num_buffer;
    }

    if (v0 >= 0) {
	negative = 0;
	u = v0;
    } else {
	negative = 1;
	u = -v0;
    }

    p = num_buffer + sizeof(num_buffer) - 1;
    *p-- = '\0';

    while(u != 0) {
	*p-- = '0' + (u % 10);
	u /= 10;
    }

    if (negative)
	*p-- = '-';

    return ++p;
}

#undef TYPE
#undef SUBROUTINE_B
#undef SUBROUTINE_O
#undef SUBROUTINE_Z
#undef SUBROUTINE_U
#undef SUBROUTINE_L
#undef SUBROUTINE_I

#else

#define TYPE G95_INT1
#define SUBROUTINE_B btoa_1
#define SUBROUTINE_O otoa_1
#define SUBROUTINE_Z ztoa_1
#define SUBROUTINE_U utoa_1
#define SUBROUTINE_L extract_logical_1
#define SUBROUTINE_I itoa_1
#include "iwrite.c"

#define TYPE G95_INT2
#define SUBROUTINE_B btoa_2
#define SUBROUTINE_O otoa_2
#define SUBROUTINE_Z ztoa_2
#define SUBROUTINE_U utoa_2
#define SUBROUTINE_L extract_logical_2
#define SUBROUTINE_I itoa_2
#include "iwrite.c"

#define TYPE G95_INT4
#define SUBROUTINE_B btoa_4
#define SUBROUTINE_O otoa_4
#define SUBROUTINE_Z ztoa_4
#define SUBROUTINE_U utoa_4
#define SUBROUTINE_L extract_logical_4
#define SUBROUTINE_I itoa_4
#include "iwrite.c"

#ifdef G95_INT8
#define TYPE G95_INT8
#define SUBROUTINE_B btoa_8
#define SUBROUTINE_O otoa_8
#define SUBROUTINE_Z ztoa_8
#define SUBROUTINE_U utoa_8
#define SUBROUTINE_L extract_logical_8
#define SUBROUTINE_I itoa_8
#include "iwrite.c"
#endif

#ifdef G95_INT16
#define TYPE G95_INT16
#define SUBROUTINE_B btoa_16
#define SUBROUTINE_O otoa_16
#define SUBROUTINE_Z ztoa_16
#define SUBROUTINE_U utoa_16
#define SUBROUTINE_L extract_logical_16
#define SUBROUTINE_I itoa_16
#include "iwrite.c"
#endif


/* write_b()-- Write with the B format */

void write_b(fnode *f, void *p, int len) {
int sign;
char *q;

    switch(len) {
    case 1: q = btoa_1(p, &sign); break;
    case 2: q = btoa_2(p, &sign); break;
    case 4: q = btoa_4(p, &sign); break;

#ifdef G95_INT8
    case 8: q = btoa_8(p, &sign); break;
#endif

#ifdef G95_INT16
    case 16: q = btoa_16(p, &sign); break;
#endif

    default:
	internal_error("write_b(): Bad kind");
	q = NULL;
	sign = 0;
    }

    write_int(f, q, 0, sign);
}


/* write_o()-- Write with the O format */

void write_o(fnode *f, void *p, int len) {
int sign;
char *q;

    switch(len) {
    case 1: q = otoa_1(p, &sign); break;
    case 2: q = otoa_2(p, &sign); break;
    case 4: q = otoa_4(p, &sign); break;

#ifdef G95_INT8
    case 8: q = otoa_8(p, &sign); break;
#endif

#ifdef G95_INT16
    case 16: q = otoa_16(p, &sign); break;
#endif

    default:
	internal_error("write_b(): Bad kind");
	q = NULL;
	sign = 0;
    }

    write_int(f, q, 0, sign);
}


/* write_z()-- Write with the Z format */

void write_z(fnode *f, void *p, int len) {
int sign;
char *q;

    switch(len) {
    case 1: q = ztoa_1(p, &sign); break;
    case 2: q = ztoa_2(p, &sign); break;
    case 4: q = ztoa_4(p, &sign); break;

#ifdef G95_INT8
    case 8: q = ztoa_8(p, &sign); break;
#endif

#ifdef G95_INT16
    case 16: q = ztoa_16(p, &sign); break;
#endif

    default:
	internal_error("write_b(): Bad kind");
	q = NULL;
	sign = 0;
    }

    write_int(f, q, 0, sign);
}


/* write_i()-- Write with the I format */

void write_i(fnode *f, void *p, int len) {
int sign;
char *q;

    switch(len) {
    case 1: q = utoa_1(p, &sign); break;
    case 2: q = utoa_2(p, &sign); break;
    case 4: q = utoa_4(p, &sign); break;

#ifdef G95_INT8
    case 8: q = utoa_8(p, &sign); break;
#endif

#ifdef G95_INT16
    case 16: q = utoa_16(p, &sign); break;
#endif

    default:
	internal_error("write_b(): Bad kind");
	q = NULL;
	sign = 0;
    }

    write_int(f, q, 1, sign);
}


/* write_integer()-- Write a list-directed integer value. */

void write_integer(void *p, int len) {
char *q, *r;
int m;

    switch(len) {
    case 1: q = itoa_1(p); break;
    case 2: q = itoa_2(p); break;
    case 4: q = itoa_4(p); break;
   
#ifdef G95_INT8
    case 8: q = itoa_8(p); break;
#endif

#ifdef G95_INT16
    case 16: q = itoa_16(p); break;
#endif

    default:
	internal_error("write_integer(): Bad kind");
	q = NULL;
    }

    m = strlen(q);
    r = write_block(m);
    if (r != NULL)
	memcpy(r, q, m);
}


/* int_to_a()-- Convert an int to a string. */

char *int_to_a(G95_DINT v) {
int negative;
G95_UINT n;
char *p;

    if (v == 0) { 
	num_buffer[0] = '0';
	num_buffer[1] = '\0';
	return num_buffer;
    }

    if (v >= 0) {
	negative = 0;
	n = v;
    } else {
	negative = 1;
	n = -v;
    }

    p = num_buffer + sizeof(num_buffer) - 1;
    *p-- = '\0';

    while(n != 0) {
	*p-- = '0' + (n % 10);
	n /= 10;
    }

    if (negative)
	*p-- = '-';

    return ++p;
}



/* long_to_a()-- Convert a long to a string. */

char *long_to_a(G95_AINT v) {
unsigned G95_AINT n;
int negative;
char *p;

    if (v == 0) { 
	num_buffer[0] = '0';
	num_buffer[1] = '\0';
	return num_buffer;
    }

    if (v >= 0) {
	negative = 0;
	n = v;
    } else {
	negative = 1;
	n = -v;
    }

    p = num_buffer + sizeof(num_buffer) - 1;
    *p-- = '\0';

    while(n != 0) {
	*p-- = '0' + (n % 10);
	n /= 10;
    }

    if (negative)
	*p-- = '-';

    return ++p;
}


/* mint_to_a()-- Convert a G95_MINT to a string. */

char *mint_to_a(G95_MINT v) {
int negative;
char *p;

    if (v == 0) { 
	num_buffer[0] = '0';
	num_buffer[1] = '\0';
	return num_buffer;
    }

    if (v >= 0)
	negative = 0;

    else {
	negative = 1;
	v = -v;
    }

    p = num_buffer + sizeof(num_buffer) - 1;
    *p-- = '\0';

    while(v != 0) {
	*p-- = '0' + (v % 10);
	v /= 10;
    }

    if (negative)
	*p-- = '-';

    return ++p;
}


char *hex_int(unsigned G95_DINT n) {
int digit;
char *p;

    if (n == 0) {
	num_buffer[0] = '0';
	num_buffer[1] = '\0';
	return num_buffer;
    }

    p = num_buffer + sizeof(num_buffer) - 1;
    *p-- = '\0';

    while(n != 0) {
	digit = n & 0xF;
	if (digit > 9)
	    digit += 'A' - '0' - 10;

	*p-- = '0' + digit;
	n >>= 4;
    }

    return ++p;
}

#endif
