/*
 * videolib.c
 *
 * Routines to convert images from one format to another.
 *
 * (C) 1999 Randall Hopper
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*      ******************** Include Files                ************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "videolib.h"

/*      ******************** Local defines                ************** */

/*  Note: you should pass UNSIGNED integers to these macros  */
#define SWAP2(x) ((((x) >>  8) & 0x000000ff)  |  \
                  (((x) <<  8) & 0x0000ff00))

#define SWAP3(x) ((((x) >> 16) & 0x000000ff)  |  \
                  ( (x)        & 0x0000ff00)  |  \
                  (((x) << 16) & 0x00ff0000))

#define SWAP4(x) ( ((x) >> 24)                |  \
                  (((x) >>  8) & 0x0000ff00)  |  \
                  (((x) <<  8) & 0x00ff0000)  |  \
                   ((x) << 24) )

#define IN_RANGE(x,a,b) ((x) >= (a) && (x) <= (b))

#define SHIFT_AND_MASK(val,lshf,msk)                     \
           (((lshf) >= 0) ? (((val) <<  (lshf)) & (msk)) \
                          : (((val) >> -(lshf)) & (msk)))

#define MIN(x,y) ((x)<(y)?(x):(y))
#define MAX(x,y) ((x)>(y)?(x):(y))

/*      ******************** Forward declarations         ************** */
/*      ******************** Private variables            ************** */

/*  15bpp 555 direct-color to 8bpp pseudo-color conversion table  */
static VL_BOOL    S_15d_to_8p_inited  = VL_FALSE;
static VL_UINT32  S_15d_to_8p_cmap_id;           /*  Cmap we're set up for  */
static VL_UINT32  S_15d_to_8p_size[3];           /*  Sides of color cube    */
static VL_UINT8   S_15d_to_8p[ 65536 ];

/*  16bpp direct-color to 16bpp direct-color conversion table */
static VL_BOOL    S_16d_to_16d_inited = VL_FALSE;
static VL_UINT32  S_16d_to_16d_src_masks[3],
               S_16d_to_16d_dst_masks[3];        /*  R,G,B pixel masks  */
static VL_UINT16  S_16d_to_16d[ 65536 ];

#define CNV16dTO16d(pix) S_16d_to_16d[ (pix) ]
#define CNV15dTO8p(pix)  S_15d_to_8p [ (pix) ]

/*      ******************** Function Definitions         ************** */

/**@BEGINFUNC**************************************************************

    Prototype  : VL_COLORMAP *VIDEOLIBNewColormap(
                      VL_UINT32 num_colors )

    Purpose    : Allocate a colormap definition for use with VideoLib

    Programmer : 20-Oct-99  Randall Hopper

    Parameters : num_colors - I: num colors to allocate space for

    Returns    : colormap

    Globals    : None.

 **@ENDFUNC*****************************************************************/

VL_COLORMAP *VIDEOLIBNewColormap( VL_UINT32 num_colors )
{
  static VL_UINT32  S_unique_id = 0x0000fade;
  VL_COLORMAP      *cmap;
  VL_COLOR         *color_list;

  cmap       = calloc( 1, sizeof( *cmap ) );
  color_list = calloc( num_colors, sizeof( *color_list ) );
  if ( !cmap && !color_list ) {
    fprintf( stderr, "Out of memory\n" );
    exit(1);
  }
  cmap->id    = S_unique_id++;
  cmap->color = color_list;
  cmap->len   = num_colors;
  return cmap;
}



/**@BEGINFUNC**************************************************************

    Prototype  : static void VIDEOLIBGetPixelGeomInfo(
                      VL_UINT32       rel_mask[3],
                      int          bits    [3],
                      int          shift   [3],
                      int          mask    [3] )

    Purpose    : Get Pixel RGB component shifts, masks, and bit counts
                 for the specified visual.

                 (mask[0] << shift[0]) | (mask[1] << shift[1]) | 
                 (mask[2] << shift[2])

    Programmer : 07-Mar-97  Randall Hopper

    Parameters : rel_mask- I: R/G/B shifted mask for each compoent
                 bits    - O: R/G/B bpp
                 shift   - O: R/G/B left shift cnt to align component in pixel
                 mask    - O: R/G/B mask to keep only desired bits

       E.g.:  INPUT : rel_mask = FF0000,00FF00,0000FF
              OUTPUT: bits     = 8,8,8
                      shift    = 16,8,0
                      mask     = FF,FF,FF

    Returns    : None.

    Globals    : None.

 **@ENDFUNC*****************************************************************/

static void VIDEOLIBGetPixelGeomInfo( 
                VL_UINT32       rel_mask[3],
                int          bits    [3],
                int          shift   [3],
                int          mask    [3] )
{
  int    i;
  VL_UINT32 rel[3];

  memcpy( rel, rel_mask, sizeof( rel ) );

  for ( i = 0; i < 3; i++ ) {
    if ( !rel[i] ) {
      fprintf( stderr, "VIDEOLIBGetPixelGeomInfo: Bad X visual mask\n");
      exit(1);
    }
    bits[i] = shift[i] = mask[i] = 0;

    while ( !(rel[i] & 0x01) ) 
      rel[i] >>= 1, shift[i]++;

    mask[i] = rel[i];

    while (  (rel[i] & 0x01) )
      rel[i] >>= 1, bits[i]++;
  }
}

/**@BEGINFUNC**************************************************************

    Prototype  : void VIDEOLIBGetPixelConvInfo(
                      VL_UINT32 src_rel_mask[3],
                      VL_UINT32 dst_rel_mask[3],
                      VL_INT32  shift       [3],
                      VL_UINT32 mask        [3] )

    Purpose    : Given a source pixel mask and a destination pixel mask,
                 determines the shifts and masks necessary to map the
                 source pixel geometry to the destination.

    Programmer : 29-Mar-97  Randall Hopper

    Parameters : src_rel_mask - I: source relative pixel masks
                 dst_rel_mask - I: dest   relative pixel masks
                 shift        - O: initial pixel shifts
                 mask         - O: final   pixel masks

       E.g.:  INPUT : src_rel_mask = 00FC00, 0003e0, 00001f
                      dst_rel_mask = FF0000, 00FF00, 0000FF
                      
              OUTPUT: shift        =      8,      6,      3
                      mask         = FC0000, 00F800, 0000F8

    Returns    : None.

    Globals    : None.

 **@ENDFUNC*****************************************************************/

void VIDEOLIBGetPixelConvInfo(
         VL_UINT32 src_rel_mask[3],
         VL_UINT32 dst_rel_mask[3],
         VL_INT32  shift       [3],
         VL_UINT32 mask        [3] )
{
  int   src_bits[3], src_shf[3], src_msk[3], src_soff[3],
        dst_bits[3], dst_shf[3], dst_msk[3], dst_soff[3];
  VL_INT32 i;

  /*  Get R/G/B shifts, masks, and bit counts  */
  VIDEOLIBGetPixelGeomInfo( src_rel_mask, src_bits, src_shf, src_msk );
  VIDEOLIBGetPixelGeomInfo( dst_rel_mask, dst_bits, dst_shf, dst_msk );

  /*  Calculate R/G/B left-most bit offsets  */
  for ( i = 0; i < 3; i++ )
      src_soff[i] = src_shf[i] + src_bits[i] - 1,
      dst_soff[i] = dst_shf[i] + dst_bits[i] - 1;

  /*  Finally, calculate net shifts and masks  */
  for ( i = 0; i < 3; i++ ) {
    shift[i] = dst_soff[i] - src_soff[i];
    if ( shift[i] >= 0 )
      mask [i] = ( src_rel_mask[i] << shift[i] ) & dst_rel_mask[i];
    else
      mask [i] = ( src_rel_mask[i] >> -shift[i] ) & dst_rel_mask[i];
  }
}


/**@BEGINFUNC**************************************************************

    Prototype  : void VIDEOLIBPrep15dto8pTable(
                      VL_COLORMAP *cmap )

    Purpose    : Sets up a speedy table for fast table lookup-based 
                 15-bit direct color -to- 8-bit pseudo color quantization.

                 NOTE:  We presume a color cube of the specified size has
                 been allocated -- R,G,B in row-major order and stored in
                 cmap.

    Programmer : 08-Mar-97  Randall Hopper

    Parameters : cmap     - I: the color cube pixel indicies

    Returns    : None.

    Globals    : None.

 **@ENDFUNC*****************************************************************/

void VIDEOLIBPrep15dto8pTable( VL_COLORMAP *cmap )
{
  VL_UINT32 i,r,g,b,r_i,g_i,b_i,idx,
            r_dim,g_dim,b_dim,
            r_min,g_min,b_min,
            r_max,g_max,b_max;

  assert( cmap && cmap->color && cmap->len );

  if ( S_15d_to_8p_inited && ( S_15d_to_8p_cmap_id == cmap->id ) )
    return;

  /*  FIXME:  Limitation  */
  assert( cmap->type == COLORMAP_PREDEF_CUBE );

  assert( cmap->dim[0]*cmap->dim[1]*cmap->dim[2] <= cmap->len );
  assert( IN_RANGE( cmap->dim[0], 2, 1024 ) );
  assert( IN_RANGE( cmap->dim[1], 2, 1024 ) );
  assert( IN_RANGE( cmap->dim[2], 2, 1024 ) );

  r_dim = cmap->dim[0], g_dim = cmap->dim[1], b_dim = cmap->dim[2];
  r_min = cmap->corners[0][0];
  g_min = cmap->corners[0][1];
  b_min = cmap->corners[0][2];
  r_max = cmap->corners[1][0];
  g_max = cmap->corners[1][1];
  b_max = cmap->corners[1][2];

  /*  Scale min/max to input R,G,B intensity range  */
  r_min = r_min * 0x1f / 255, r_max = r_max * 0x1f / 255;
  g_min = g_min * 0x1f / 255, g_max = g_max * 0x1f / 255;
  b_min = b_min * 0x1f / 255, b_max = b_max * 0x1f / 255;

  /*  Fill in the lookup table  */
  for ( i = 0; i < 32768; i++ ) {
    r   = (i >> 10) & 0x1f;
    g   = (i >>  5) & 0x1f;
    b   =  i        & 0x1f;

    r = MAX( r_min, MIN( r_max, r ) );
    g = MAX( g_min, MIN( g_max, g ) );
    b = MAX( b_min, MIN( b_max, b ) );

    r_i = (r-r_min) * (r_dim-1) / (r_max-r_min);
    g_i = (g-g_min) * (g_dim-1) / (g_max-g_min);
    b_i = (b-b_min) * (b_dim-1) / (b_max-b_min);
    idx = r_i*g_dim*b_dim + g_i*b_dim + b_i;

    S_15d_to_8p[i] = cmap->color[ idx ].pixel;
  }
  memcpy( &S_15d_to_8p[ 32768 ], &S_15d_to_8p[ 0 ],
          sizeof( S_15d_to_8p[0] ) * 32768 );

  S_15d_to_8p_cmap_id = cmap->id;
  S_15d_to_8p_size[0] = r_dim;
  S_15d_to_8p_size[1] = g_dim;
  S_15d_to_8p_size[2] = b_dim;
  S_15d_to_8p_inited  = VL_TRUE;
}


/**@BEGINFUNC**************************************************************

    Prototype  : void VIDEOLIBPrep16dto16dTable(
                      VL_UINT32        src_rel_mask[3],
                      VL_UINT32        dst_rel_mask[3] )

    Purpose    : Creates a lookup-table for doing fast 16-bit to 16-bit
                 pixel weighting conversions.

    Programmer : 30-Mar-97  Randall Hopper

    Parameters : src_rel_mask  - I: src pixel geom  (e.g. 7C00,03e0,001F)
                 dst_rel_mask  - I: dest pixel geom (e.g. F800,07e0,001F)

    Returns    : None.

    Globals    : None.

 **@ENDFUNC*****************************************************************/

void VIDEOLIBPrep16dto16dTable( VL_UINT32        src_rel_mask[3],
                                VL_UINT32        dst_rel_mask[3] )
{
  VL_INT32  i;
  VL_INT32  shf[3];
  VL_UINT32 msk[3];

  if ( S_16d_to_16d_inited &&
     ( memcmp( S_16d_to_16d_src_masks, src_rel_mask,
               sizeof( S_16d_to_16d_src_masks ) ) == 0 ) &&
     ( memcmp( S_16d_to_16d_dst_masks, dst_rel_mask,
               sizeof( S_16d_to_16d_dst_masks ) ) == 0 ) )
    return;

  VIDEOLIBGetPixelConvInfo( src_rel_mask, dst_rel_mask, shf, msk);

  for ( i = 0; i < 65536; i++ )
    S_16d_to_16d[i] = SHIFT_AND_MASK( i, shf[0], msk[0] ) |
                      SHIFT_AND_MASK( i, shf[1], msk[1] ) |
                      SHIFT_AND_MASK( i, shf[2], msk[2] );

  memcpy( S_16d_to_16d_src_masks, src_rel_mask,
          sizeof( S_16d_to_16d_src_masks ) );
  memcpy( S_16d_to_16d_dst_masks, dst_rel_mask,
          sizeof( S_16d_to_16d_dst_masks ) );

  S_16d_to_16d_inited = VL_TRUE;
}


/**@BEGINFUNC**************************************************************

    Prototype  : void VIDEOLIBNewFrameHdlr(
                      VL_IMAGE *img )

    Purpose    : Called to convert RGB images to RGB images.

    Programmer : 07-Mar-97  Randall Hopper

    Parameters : src - I: source image
                 dst - I: dest image

    Returns    : None.

    Globals    : None.

 **@ENDFUNC*****************************************************************/

static void VIDEOLIBConvertRGBToRGB( VL_IMAGE *src, VL_IMAGE *dst )
{
  VL_GEOM             *geom     = &src->geom;
  VL_UINT32            dst_mask[3];
  VL_INT32             i,
                       j;
  VL_UINT32            src_line_pad,
                       dst_line_pad;
  VL_UINT8            *src8     = (VL_UINT8  *) src->buf;
  VL_UINT16           *src16    = (VL_UINT16 *) src->buf;
  VL_UINT32           *src32    = (VL_UINT32 *) src->buf;

  /*  Current conversion assumptions/limitations  */
  assert( src->buf && dst->buf );        /*  For RGB, alloc B_per_line * h  */
  assert( src->geom.w == dst->geom.w );
  assert( src->geom.h == dst->geom.h );
  assert( src->pix_geom.type == VL_PIXELTYPE_RGB );
  assert( dst->pix_geom.type == VL_PIXELTYPE_RGB );
  assert( src->pix_geom.rgb.direct_color );
  assert( dst->pix_geom.rgb.direct_color ||
          (( src->pix_geom.rgb.Bpp == 2 ) && 
           ( dst->pix_geom.rgb.Bpp == 1 ) &&
           ( src->pix_geom.rgb.mask[0] == 0x7c00 ) &&
           ( src->pix_geom.rgb.mask[1] == 0x03e0 ) &&
           ( src->pix_geom.rgb.mask[2] == 0x001f )) );
  assert( dst->pix_geom.rgb.direct_color ||
          dst->pix_geom.rgb.colormap &&
          ( dst->pix_geom.rgb.colormap->type == COLORMAP_PREDEF_CUBE ) );

    /*  FIXME:  Generalize so we don't have to have a perfect NxNxN cube  */
    /*          Also support odd start/end colormaps.                     */

  /*  Sanity checks  */
  assert( dst->pix_geom.rgb.direct_color || dst->pix_geom.rgb.colormap );

  /*  FIXME:  24bpp or 15 bpp not sufficient for bcopy; must also  */
  /*          have RGB positioned correctly  */

  /*  Calc ximage scanline pad  */
  src_line_pad = src->geom.bytes_per_line - 
                 ( src->geom.w * src->pix_geom.rgb.Bpp );
  dst_line_pad = dst->geom.bytes_per_line - 
                 ( dst->geom.w * dst->pix_geom.rgb.Bpp );

  assert( src_line_pad == 0 );

  /*  Check active visual.  If TrueColor, convert pixels (IF necessary). */
  /*     If Pseudo-color, quantize.                                      */
  if ( dst->pix_geom.rgb.direct_color ) {

    /*  FIXME:  Create a fallback TrueColor handler that always works.  */
    /*          Brute force whatever->(R,G,B)->whatever.                */

    VL_BOOL dst_swap_b = dst->pix_geom.rgb.swap_bytes,
            dst_swap_s = dst->pix_geom.rgb.swap_shorts;

    dst_mask[0] = dst->pix_geom.rgb.mask[0];
    dst_mask[1] = dst->pix_geom.rgb.mask[1];
    dst_mask[2] = dst->pix_geom.rgb.mask[2];

    /*  CASE: No conversion required  */
    if (( src->pix_geom.rgb.Bpp == dst->pix_geom.rgb.Bpp ) &&
      ( memcmp( src->pix_geom.rgb.mask, 
                dst->pix_geom.rgb.mask,
                sizeof( src->pix_geom.rgb.mask ) ) == 0 ) && 
      (( src->pix_geom.rgb.swap_bytes  == dst_swap_b ) &&
       ( src->pix_geom.rgb.swap_shorts == dst_swap_s )) &&
      ( src->geom.bytes_per_line == dst->geom.bytes_per_line )) {

      bcopy( src->buf, dst->buf, src->geom.bytes_per_line * geom->h );
    }

    /*  CASE: 2 Bpp -> 2Bpp conversions  */
    else if (( src->pix_geom.rgb.Bpp == 2 ) && 
             ( dst->pix_geom.rgb.Bpp == 2 )) {

      VL_UINT16 *dst_line = (VL_UINT16 *) dst->buf,
                *p;
      VL_BOOL    src_padded = VL_FALSE;

      /*printf( "FRAME: 2Bpp->2Bpp conv\n" );*/

      VIDEOLIBPrep16dto16dTable( src->pix_geom.rgb.mask, dst_mask );

      if ( !src->pix_geom.rgb.swap_bytes ) {        /*  This is ugly  */
        for ( j = 0; j < geom->h; j++ ) {
          p = dst_line;
          for ( i = geom->w; i > 0; i-- ) {
            *(p++) = (*src16 >> 8) | (*src16 << 8);
            src16++;
          }
          for ( i = dst_line_pad; i > 0; i-- )
            *(((VL_UINT8 *)p)++) = 0x00;
          (VL_UINT8 *)dst_line += dst->geom.bytes_per_line;
        }

        src16 = dst_line = (VL_UINT16 *) dst->buf;
        src_padded = VL_TRUE;
      }

      for ( j = 0; j < geom->h; j++ ) {
        p = dst_line;
        for ( i = geom->w; i >= 16; i -= 16 ) {
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
          *(p++) = CNV16dTO16d( *(src16++) );
        }
        for ( ; i > 0; i-- )
          *(p++) = CNV16dTO16d( *(src16++) );
        if ( src_padded ) 
          src16 = (VL_UINT16 *)( (VL_UINT8 *)src16 + dst_line_pad );
        else
          for ( i = dst_line_pad; i > 0; i-- )
            *(((VL_UINT8 *)dst)++) = 0x00;
        (VL_UINT8 *)dst_line += dst->geom.bytes_per_line;
      }

      if ( !dst_swap_b ) {                       /*  More ugliness  */
        src16 = dst_line = (VL_UINT16 *) dst->buf;

        for ( j = 0; j < geom->h; j++ ) {
          p = dst_line;
          for ( i = geom->w; i > 0; i-- ) {
            *(p++) = (*src16 >> 8) | (*src16 << 8);
            src16++;
          }
          (VL_UINT8 *)dst_line += dst->geom.bytes_per_line;
          (VL_UINT8 *)src16    += dst->geom.bytes_per_line;
        }
      }
    }

    /*  CASE: 2/3/4 Bpp -> 3/4 Bpp conversion  */
    else if ( IN_RANGE( src->pix_geom.rgb.Bpp, 2, 4 ) && 
              IN_RANGE( dst->pix_geom.rgb.Bpp, 3, 4 )) {
      VL_INT32   shf[3];
      VL_UINT32  msk[3];
      VL_BOOL    shorts = (src->pix_geom.rgb.Bpp == 2),
                 quads  = (src->pix_geom.rgb.Bpp == 4);
      VL_UINT8  *dst_line = (VL_UINT8 *) dst->buf;
      VL_UINT8  *dst3 = NULL;
      VL_UINT32 *dst4 = NULL;

      /*printf( "FRAME: 2/3/4Bpp->3/4Bpp conv\n" );*/

      /*  Brute force any -to-3/4-Bpp TrueColor conversions  */
      VIDEOLIBGetPixelConvInfo( src->pix_geom.rgb.mask, dst_mask, shf, msk);

      for ( j = 0; j < geom->h; j++ ) {
        if ( dst->pix_geom.rgb.Bpp == 3 ) dst3 = (VL_UINT8  *) dst_line;
        else                     dst4 = (VL_UINT32 *) dst_line;

        for ( i = geom->w; i > 0; i-- ) {
          register VL_UINT32 pix;

          /*  Read pixel in right byte order  */
          if ( shorts ) {
            pix = *(src16++);
            if ( !src->pix_geom.rgb.swap_bytes )
                pix = SWAP2(pix);
          }
          else if ( quads ) {
            pix = *(src32++);
            if ( !(src->pix_geom.rgb.swap_bytes && 
                   src->pix_geom.rgb.swap_shorts) )
                pix = SWAP4(pix);
          }
          else {
            pix  = *(src8++);
            pix |= ((VL_INT32)(*(src8++))) <<  8;
            pix |= ((VL_INT32)(*(src8++))) << 16;
            if ( !src->pix_geom.rgb.swap_bytes )
              pix = SWAP3(pix);
          }

          pix       = SHIFT_AND_MASK( pix, shf[0], msk[0] ) |
                      SHIFT_AND_MASK( pix, shf[1], msk[1] ) |
                      SHIFT_AND_MASK( pix, shf[2], msk[2] );

          if ( dst->pix_geom.rgb.Bpp == 3 ) {
            if ( !dst_swap_b )
              pix = SWAP3(pix);
            *(dst3++) =  pix        & 0xff;
            *(dst3++) = (pix >>  8) & 0xff;
            *(dst3++) = (pix >> 16) & 0xff;
          }
          else {
            if ( !(dst_swap_b && dst_swap_s) )
                pix = SWAP4(pix);
            *(dst4++) = pix;
          }
        }

        /*  Write EOL pad  */
        if ( dst->pix_geom.rgb.Bpp == 4 )     
          dst3 = (VL_UINT8 *)dst4;
        for ( i = dst_line_pad; i > 0; i-- )
          *(dst3++) = 0;

        dst_line += dst->geom.bytes_per_line;
      }
    }
    else {
      fprintf( stderr, 
               "VIDEOLIBConvertRGBToRGB() - Unsupported TrueColor conv\n");
      fprintf( stderr,
               "\tFrom: %ld Bpp (R %.6lx, G %.6lx, B %.6lx)\n"
               "\tTo  : %ld Bpp (R %.6lx, G %.6lx, B %.6lx)\n",
               src->pix_geom.rgb.Bpp, src->pix_geom.rgb.mask[0], 
                                      src->pix_geom.rgb.mask[1],
                                      src->pix_geom.rgb.mask[2], 
               dst->pix_geom.rgb.Bpp, dst->pix_geom.rgb.mask[0], 
                                      dst->pix_geom.rgb.mask[1],
                                      dst->pix_geom.rgb.mask[2] );
      exit(1);
    }
  }
  else {
    VL_UINT8 *dst_line = (VL_UINT8 *) dst->buf,
             *p;

    VIDEOLIBPrep15dto8pTable( dst->pix_geom.rgb.colormap );

    for ( j = 0; j < geom->h; j++ ) {
      p = dst_line;

      for ( i = geom->w; i >= 16; i -= 16 ) {
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
        *(p++) = CNV15dTO8p( *(src16++) );
      }
      for ( ; i > 0; i-- )
        *(p++) = CNV15dTO8p( *(src16++) );
      for ( i = dst_line_pad; i > 0; i-- )
        *(p++) = CNV15dTO8p( 0x0000 );

      dst_line += dst->geom.bytes_per_line;
    }
  }
}


/**@BEGINFUNC**************************************************************

    Prototype  : void VIDEOLIBConvertImage(
                      VL_IMAGE *src,
                      VL_IMAGE *dst )

    Purpose    : Called to handle display of a new frame when we're not 
                 in direct-video mode.

                 This is where any conversion or quantization that needs 
                 to be performed for display occurs.

    Programmer : 07-Mar-97  Randall Hopper

    Parameters : src - I: source image
                 dst - I: dest image

    Returns    : None.

    Globals    : None.

 **@ENDFUNC*****************************************************************/

void VIDEOLIBConvertImage( VL_IMAGE *src, VL_IMAGE *dst )
{
  /*  Current conversion assumptions/limitations  */
  assert( src->buf && dst->buf );        /*  For RGB, alloc B_per_line * h  */
  assert( src->geom.w == dst->geom.w );
  assert( src->geom.h == dst->geom.h );
  assert( src->pix_geom.type != VL_PIXELTYPE_YUV );
  assert( dst->pix_geom.type != VL_PIXELTYPE_YUV );
  assert( src->pix_geom.rgb.direct_color );

  /*  FIXME:  More assertion checks on what we do and do not support  */

  VIDEOLIBConvertRGBToRGB( src, dst );
}
