from dataclasses import dataclass

from fritzconnection.core.exceptions import (
    FritzActionError,
    FritzServiceError,
    FritzArrayIndexError,
)


@dataclass
class FCService:
    actions: list[str]


def create_fc_services(services_mock):
    services = {}
    for svc, actions in services_mock.items():
        services[svc] = FCService(actions)
    return services


def call_action_mock(service, action, **kwargs):
    _ = kwargs

    call_action_responses = {
        ("DeviceInfo1", "GetInfo"): {
            "NewSerialNumber": "1234567890",
            "NewModelName": "Fritz!MockBox 9790",
            "NewSoftwareVersion": "1.2",
            "NewUpTime": 1234,
        },
        ("DeviceConfig1", "GetPersistentData"): {},
        ("Hosts1", "GetHostNumberOfEntries"): {"NewHostNumberOfEntries": 3},
        ("UserInterface1", "GetInfo"): {
            "NewUpgradeAvailable": 1,
            "NewX_AVM-DE_Version": "1.3",
        },
        ("LANEthernetInterfaceConfig1", "GetInfo"): {
            "NewEnable": 1,
            "NewStatus": "Up",
        },
        ("LANEthernetInterfaceConfig1", "GetStatistics"): {
            "NewBytesReceived": 1234,
            "NewBytesSent": 9876,
            "NewPacketsReceived": 123,
            "NewPacketsSent": 987,
        },
        ("WANDSLInterfaceConfig1", "GetInfo"): {
            "NewEnable": 1,
            "NewStatus": "Up",
            "NewUpstreamCurrRate": 500,
            "NewDownstreamCurrRate": 100,
            "NewUpstreamMaxRate": 567,
            "NewDownstreamMaxRate": 123,
            "NewUpstreamNoiseMargin": 56,
            "NewDownstreamNoiseMargin": 67,
            "NewUpstreamAttenuation": 12,
            "NewDownstreamAttenuation": 23,
        },
        ("WANDSLInterfaceConfig1", "X_AVM-DE_GetDSLInfo"): {
            "NewFECErrors": 12,
            "NewCRCErrors": 23,
        },
        ("WANPPPConnection1", "GetStatusInfo"): {
            "NewConnectionStatus": "Connected",
            "NewUptime": 12345,
            "NewLastConnectionError": "Timeout",
        },
        ("WANCommonInterfaceConfig1", "GetCommonLinkProperties"): {
            "NewWANAccessType": "PPPoE",
            "NewLayer1UpstreamMaxBitRate": 10000,
            "NewLayer1DownstreamMaxBitRate": 10001,
            "NewPhysicalLinkStatus": "Up",
        },
        ("WANCommonInterfaceConfig1", "GetTotalBytesReceived"): {"NewTotalBytesReceived": 1234567},
        ("WANCommonInterfaceConfig1", "GetTotalBytesSent"): {"NewTotalBytesSent": 234567},
        ("WANCommonIFC1", "GetAddonInfos"): {
            "NewByteReceiveRate": 12345,
            "NewByteSendRate": 23456,
        },
        ("WANCommonInterfaceConfig1", "GetTotalPacketsReceived"): {
            "NewTotalPacketsReceived": 12345
        },
        ("WANCommonInterfaceConfig1", "GetTotalPacketsSent"): {"NewTotalPacketsSent": 2345},
        ("WLANConfiguration1", "GetInfo"): {
            "NewStatus": "Up",
            "NewEnable": 1,
            "NewStandard": "802.11xe",
            "NewSSID": "SomeSSID-1",
            "NewChannel": "42",
        },
        ("WLANConfiguration1", "GetTotalAssociations"): {"NewTotalAssociations": 56},
        ("WLANConfiguration1", "GetPacketStatistics"): {
            "NewTotalPacketsReceived": 123456,
            "NewTotalPacketsSent": 2345,
        },
        ("WLANConfiguration2", "GetInfo"): {
            "NewStatus": "Up",
            "NewEnable": 1,
            "NewStandard": "802.11xe2",
            "NewSSID": "SomeSSID-2",
            "NewChannel": "23",
        },
        ("WLANConfiguration2", "GetTotalAssociations"): {"NewTotalAssociations": 43},
        ("WLANConfiguration2", "GetPacketStatistics"): {
            "NewTotalPacketsReceived": 1234560,
            "NewTotalPacketsSent": 23450,
        },
        ("WLANConfiguration3", "GetInfo"): {
            "NewStatus": "Up",
            "NewEnable": 1,
            "NewStandard": "802.11xe3",
            "NewSSID": "SomeSSID-3",
            "NewChannel": "69",
        },
        ("WLANConfiguration3", "GetTotalAssociations"): {"NewTotalAssociations": 82},
        ("WLANConfiguration3", "GetPacketStatistics"): {
            "NewTotalPacketsReceived": 1234561,
            "NewTotalPacketsSent": 23451,
        },
        ("Hosts1", "GetGenericHostEntry"): {
            "NewIPAddress": "192.168.178.42",
            "NewMACAddress": "01:02:03:04:05:06",
            "NewHostName": "generichost",
            "NewActive": 1,
        },
        ("Hosts1", "X_AVM-DE_GetSpecificHostEntryByIP"): {
            "NewInterfaceType": "eth",
            "NewX_AVM-DE_Port": "LAN1",
            "NewX_AVM-DE_Model": "Mockgear",
            "NewX_AVM-DE_Speed": 1000,
        },
        ("X_AVM-DE_Homeauto1", "GetInfo"): {
            "NewAllowedCharsAIN": "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ",
            "MaxCharsAIN": 64,
            "MinCharsAIN": 12,
            "MaxCharsDeviceName": 32,
            "MinCharsDeviceName": 10,
        },
        ("X_AVM-DE_Homeauto1", "GetGenericDeviceInfos"): {
            "NewAIN": "123456789012",
            "NewDeviceId": 123,
            "NewFunctionBitMask": 1,
            "NewFirmwareVersion": "1.2",
            "NewManufacturer": "AVM",
            "NewProductName": "MockDevice",
            "NewDeviceName": "MockDeviceName",
            "NewPresent": "CONNECTED",
            "NewMultimeterIsEnabled": "ENABLED",
            "NewMultimeterIsValid": "VALID",
            "NewMultimeterPower": 1234,
            "NewMultimeterEnergy": 12345,
            "NewTemperatureIsEnabled": "ENABLED",
            "NewTemperatureIsValid": "VALID",
            "NewTemperatureCelsius": 234,
            "NewTemperatureOffset": 0,
            "NewSwitchIsEnabled": "ENABLED",
            "NewSwitchIsValid": "VALID",
            "NewSwitchState": "ON",
            "NewSwitchMode": "MANUAL",
            "NewSwitchLock": False,
            "NewHkrIsEnabled": "ENABLED",
            "NewHkrIsValid": "VALID",
            "NewHkrIsTemperature": 245,
            "NewHkrSetVentilStatus": "OPEN",
            "NewHkrSetTemperature": 234,
            "NewHkrReduceVentilStatus": "CLOSED",
            "NewHkrReduceTemperature": 234,
            "NewHkrComfortVentilStatus": "OPEN",
            "NewHkrComfortTemperature": 234,
        },
    }

    if service == "X_AVM-DE_Homeauto1" and action == "GetGenericDeviceInfos":
        if kwargs["NewIndex"] == 0:
            return call_action_responses[(service, action)]
        else:
            raise FritzArrayIndexError
    return call_action_responses[(service, action)]


def call_action_no_basic_mock(service, action, **kwargs):
    _ = kwargs

    if service == "DeviceInfo1" and action == "GetInfo":
        raise FritzServiceError

    call_action_responses = {
        ("Hosts1", "GetHostNumberOfEntries"): {"NewHostNumberOfEntries": 3},
        ("UserInterface1", "GetInfo"): {
            "NewUpgradeAvailable": 1,
            "NewX_AVM-DE_Version": "1.3",
        },
        ("LANEthernetInterfaceConfig1", "GetInfo"): {
            "NewEnable": 1,
            "NewStatus": "Up",
        },
        ("LANEthernetInterfaceConfig1", "GetStatistics"): {
            "NewBytesReceived": 1234,
            "NewBytesSent": 9876,
            "NewPacketsReceived": 123,
            "NewPacketsSent": 987,
        },
        ("WANDSLInterfaceConfig1", "GetInfo"): {
            "NewEnable": 1,
            "NewStatus": "Up",
            "NewUpstreamCurrRate": 500,
            "NewDownstreamCurrRate": 100,
            "NewUpstreamMaxRate": 567,
            "NewDownstreamMaxRate": 123,
            "NewUpstreamNoiseMargin": 56,
            "NewDownstreamNoiseMargin": 67,
            "NewUpstreamAttenuation": 12,
            "NewDownstreamAttenuation": 23,
        },
        ("WANDSLInterfaceConfig1", "X_AVM-DE_GetDSLInfo"): {
            "NewFECErrors": 12,
            "NewCRCErrors": 23,
        },
        ("WANPPPConnection1", "GetStatusInfo"): {
            "NewConnectionStatus": "Connected",
            "NewUptime": 12345,
            "NewLastConnectionError": "Timeout",
        },
        ("WANCommonInterfaceConfig1", "GetCommonLinkProperties"): {
            "NewWANAccessType": "PPPoE",
            "NewLayer1UpstreamMaxBitRate": 10000,
            "NewLayer1DownstreamMaxBitRate": 10001,
            "NewPhysicalLinkStatus": "Up",
        },
        ("WANCommonInterfaceConfig1", "GetTotalBytesReceived"): {"NewTotalBytesReceived": 1234567},
        ("WANCommonInterfaceConfig1", "GetTotalBytesSent"): {"NewTotalBytesSent": 234567},
        ("WANCommonIFC1", "GetAddonInfos"): {
            "NewByteReceiveRate": 12345,
            "NewByteSendRate": 23456,
        },
        ("WANCommonInterfaceConfig1", "GetTotalPacketsReceived"): {
            "NewTotalPacketsReceived": 12345
        },
        ("WANCommonInterfaceConfig1", "GetTotalPacketsSent"): {"NewTotalPacketsSent": 2345},
        ("WLANConfiguration1", "GetInfo"): {
            "NewStatus": "Up",
            "NewEnable": 1,
            "NewStandard": "802.11xe",
            "NewSSID": "SomeSSID-1",
            "NewChannel": "42",
        },
        ("WLANConfiguration1", "GetTotalAssociations"): {"NewTotalAssociations": 56},
        ("WLANConfiguration1", "GetPacketStatistics"): {
            "NewTotalPacketsReceived": 123456,
            "NewTotalPacketsSent": 2345,
        },
        ("WLANConfiguration2", "GetInfo"): {
            "NewStatus": "Up",
            "NewEnable": 1,
            "NewStandard": "802.11xe2",
            "NewSSID": "SomeSSID-2",
            "NewChannel": "23",
        },
        ("WLANConfiguration2", "GetTotalAssociations"): {"NewTotalAssociations": 43},
        ("WLANConfiguration2", "GetPacketStatistics"): {
            "NewTotalPacketsReceived": 1234560,
            "NewTotalPacketsSent": 23450,
        },
        ("WLANConfiguration3", "GetInfo"): {
            "NewStatus": "Up",
            "NewEnable": 1,
            "NewStandard": "802.11xe3",
            "NewSSID": "SomeSSID-3",
            "NewChannel": "69",
        },
        ("WLANConfiguration3", "GetTotalAssociations"): {"NewTotalAssociations": 82},
        ("WLANConfiguration3", "GetPacketStatistics"): {
            "NewTotalPacketsReceived": 1234561,
            "NewTotalPacketsSent": 23451,
        },
        ("Hosts1", "GetGenericHostEntry"): {
            "NewIPAddress": "192.168.178.42",
            "NewMACAddress": "01:02:03:04:05:06",
            "NewHostName": "generichost",
            "NewActive": 1,
        },
        ("Hosts1", "X_AVM-DE_GetSpecificHostEntryByIP"): {
            "NewInterfaceType": "eth",
            "NewX_AVM-DE_Port": "LAN1",
            "NewX_AVM-DE_Model": "Mockgear",
            "NewX_AVM-DE_Speed": 1000,
        },
        ("X_AVM-DE_Homeauto1", "GetGenericDeviceInfos"): {
            "NewAIN": "123456789012",
            "NewDeviceId": 123,
            "NewFunctionBitMask": 1,
            "NewFirmwareVersion": "1.2",
            "NewManufacturer": "AVM",
            "NewProductName": "MockDevice",
            "NewDeviceName": "MockDeviceName",
            "NewPresent": "CONNECTED",
            "NewMultimeterIsEnabled": "ENABLED",
            "NewMultimeterIsValid": "VALID",
            "NewMultimeterPower": 1234,
            "NewMultimeterEnergy": 12345,
            "NewTemperatureIsEnabled": "ENABLED",
            "NewTemperatureIsValid": "VALID",
            "NewTemperatureCelsius": 234,
            "NewTemperatureOffset": 0,
            "NewSwitchIsEnabled": "ENABLED",
            "NewSwitchIsValid": "VALID",
            "NewSwitchState": "ON",
            "NewSwitchMode": "MANUAL",
            "NewSwitchLock": False,
            "NewHkrIsEnabled": "ENABLED",
            "NewHkrIsValid": "VALID",
            "NewHkrIsTemperature": 245,
            "NewHkrSetVentilStatus": "OPEN",
            "NewHkrSetTemperature": 234,
            "NewHkrReduceVentilStatus": "CLOSED",
            "NewHkrReduceTemperature": 234,
            "NewHkrComfortVentilStatus": "OPEN",
            "NewHkrComfortTemperature": 234,
        },
    }

    return call_action_responses[(service, action)]


def call_action_no_basic_action_error_mock(service, action, **kwargs):
    _ = kwargs

    if service == "DeviceInfo1" and action == "GetInfo":
        raise FritzActionError

    call_action_responses = {
        ("Hosts1", "GetHostNumberOfEntries"): {"NewHostNumberOfEntries": 3},
        ("UserInterface1", "GetInfo"): {
            "NewUpgradeAvailable": 1,
            "NewX_AVM-DE_Version": "1.3",
        },
        ("LANEthernetInterfaceConfig1", "GetInfo"): {
            "NewEnable": 1,
            "NewStatus": "Up",
        },
        ("LANEthernetInterfaceConfig1", "GetStatistics"): {
            "NewBytesReceived": 1234,
            "NewBytesSent": 9876,
            "NewPacketsReceived": 123,
            "NewPacketsSent": 987,
        },
        ("WANDSLInterfaceConfig1", "GetInfo"): {
            "NewEnable": 1,
            "NewStatus": "Up",
            "NewUpstreamCurrRate": 500,
            "NewDownstreamCurrRate": 100,
            "NewUpstreamMaxRate": 567,
            "NewDownstreamMaxRate": 123,
            "NewUpstreamNoiseMargin": 56,
            "NewDownstreamNoiseMargin": 67,
            "NewUpstreamAttenuation": 12,
            "NewDownstreamAttenuation": 23,
        },
        ("WANDSLInterfaceConfig1", "X_AVM-DE_GetDSLInfo"): {
            "NewFECErrors": 12,
            "NewCRCErrors": 23,
        },
        ("WANPPPConnection1", "GetStatusInfo"): {
            "NewConnectionStatus": "Connected",
            "NewUptime": 12345,
            "NewLastConnectionError": "Timeout",
        },
        ("WANCommonInterfaceConfig1", "GetCommonLinkProperties"): {
            "NewWANAccessType": "PPPoE",
            "NewLayer1UpstreamMaxBitRate": 10000,
            "NewLayer1DownstreamMaxBitRate": 10001,
            "NewPhysicalLinkStatus": "Up",
        },
        ("WANCommonInterfaceConfig1", "GetTotalBytesReceived"): {"NewTotalBytesReceived": 1234567},
        ("WANCommonInterfaceConfig1", "GetTotalBytesSent"): {"NewTotalBytesSent": 234567},
        ("WANCommonIFC1", "GetAddonInfos"): {
            "NewByteReceiveRate": 12345,
            "NewByteSendRate": 23456,
        },
        ("WANCommonInterfaceConfig1", "GetTotalPacketsReceived"): {
            "NewTotalPacketsReceived": 12345
        },
        ("WANCommonInterfaceConfig1", "GetTotalPacketsSent"): {"NewTotalPacketsSent": 2345},
        ("WLANConfiguration1", "GetInfo"): {
            "NewStatus": "Up",
            "NewEnable": 1,
            "NewStandard": "802.11xe",
            "NewSSID": "SomeSSID-1",
            "NewChannel": "42",
        },
        ("WLANConfiguration1", "GetTotalAssociations"): {"NewTotalAssociations": 56},
        ("WLANConfiguration1", "GetPacketStatistics"): {
            "NewTotalPacketsReceived": 123456,
            "NewTotalPacketsSent": 2345,
        },
        ("WLANConfiguration2", "GetInfo"): {
            "NewStatus": "Up",
            "NewEnable": 1,
            "NewStandard": "802.11xe2",
            "NewSSID": "SomeSSID-2",
            "NewChannel": "23",
        },
        ("WLANConfiguration2", "GetTotalAssociations"): {"NewTotalAssociations": 43},
        ("WLANConfiguration2", "GetPacketStatistics"): {
            "NewTotalPacketsReceived": 1234560,
            "NewTotalPacketsSent": 23450,
        },
        ("WLANConfiguration3", "GetInfo"): {
            "NewStatus": "Up",
            "NewEnable": 1,
            "NewStandard": "802.11xe3",
            "NewSSID": "SomeSSID-3",
            "NewChannel": "69",
        },
        ("WLANConfiguration3", "GetTotalAssociations"): {"NewTotalAssociations": 82},
        ("WLANConfiguration3", "GetPacketStatistics"): {
            "NewTotalPacketsReceived": 1234561,
            "NewTotalPacketsSent": 23451,
        },
        ("Hosts1", "GetGenericHostEntry"): {
            "NewIPAddress": "192.168.178.42",
            "NewMACAddress": "01:02:03:04:05:06",
            "NewHostName": "generichost",
            "NewActive": 1,
        },
        ("Hosts1", "X_AVM-DE_GetSpecificHostEntryByIP"): {
            "NewInterfaceType": "eth",
            "NewX_AVM-DE_Port": "LAN1",
            "NewX_AVM-DE_Model": "Mockgear",
            "NewX_AVM-DE_Speed": 1000,
        },
        ("X_AVM-DE_Homeauto1", "GetGenericDeviceInfos"): {
            "NewAIN": "123456789012",
            "NewDeviceId": 123,
            "NewFunctionBitMask": 1,
            "NewFirmwareVersion": "1.2",
            "NewManufacturer": "AVM",
            "NewProductName": "MockDevice",
            "NewDeviceName": "MockDeviceName",
            "NewPresent": "CONNECTED",
            "NewMultimeterIsEnabled": "ENABLED",
            "NewMultimeterIsValid": "VALID",
            "NewMultimeterPower": 1234,
            "NewMultimeterEnergy": 12345,
            "NewTemperatureIsEnabled": "ENABLED",
            "NewTemperatureIsValid": "VALID",
            "NewTemperatureCelsius": 234,
            "NewTemperatureOffset": 0,
            "NewSwitchIsEnabled": "ENABLED",
            "NewSwitchIsValid": "VALID",
            "NewSwitchState": "ON",
            "NewSwitchMode": "MANUAL",
            "NewSwitchLock": False,
            "NewHkrIsEnabled": "ENABLED",
            "NewHkrIsValid": "VALID",
            "NewHkrIsTemperature": 245,
            "NewHkrSetVentilStatus": "OPEN",
            "NewHkrSetTemperature": 234,
            "NewHkrReduceVentilStatus": "CLOSED",
            "NewHkrReduceTemperature": 234,
            "NewHkrComfortVentilStatus": "OPEN",
            "NewHkrComfortTemperature": 234,
        },
    }

    return call_action_responses[(service, action)]


DEVICE_INFO1 = [
    "GetInfo",
    "SetProvisioningCode",
    "GetDeviceLog",
    "GetSecurityPort",
]

fc_services_no_basic_info = {
    "UserInterface1": [
        "GetInfo",
        "X_AVM-DE_CheckUpdate",
        "X_AVM-DE_DoUpdate",
        "X_AVM-DE_DoPrepareCGI",
        "X_AVM-DE_DoManualUpdate",
        "X_AVM-DE_GetInternationalConfig",
        "X_AVM-DE_SetInternationalConfig",
        "X_AVM-DE_GetInfo",
        "X_AVM-DE_SetConfig",
    ]
}

fc_services_capabilities: dict[str, dict] = {}
fc_services_capabilities["DeviceInfo"] = {
    "DeviceInfo1": DEVICE_INFO1,
}
fc_services_capabilities["HostNumberOfEntries"] = {
    "Hosts1": [
        "GetHostNumberOfEntries",
    ],
}
fc_services_capabilities["UserInterface"] = {
    "UserInterface1": [
        "GetInfo",
        "X_AVM-DE_CheckUpdate",
        "X_AVM-DE_DoUpdate",
        "X_AVM-DE_DoPrepareCGI",
        "X_AVM-DE_DoManualUpdate",
        "X_AVM-DE_GetInternationalConfig",
        "X_AVM-DE_SetInternationalConfig",
        "X_AVM-DE_GetInfo",
        "X_AVM-DE_SetConfig",
    ],
}
fc_services_capabilities["LanInterfaceConfig"] = {
    "LANEthernetInterfaceConfig1": ["GetInfo"],
}
fc_services_capabilities["LanInterfaceConfigStatistics"] = {
    "LANEthernetInterfaceConfig1": ["GetStatistics"],
}
fc_services_capabilities["WanDSLInterfaceConfig"] = {
    "WANDSLInterfaceConfig1": [
        "GetInfo",
    ],
}
fc_services_capabilities["WanDSLInterfaceConfigAVM"] = {
    "WANDSLInterfaceConfig1": [
        "X_AVM-DE_GetDSLInfo",
    ],
}
fc_services_capabilities["WanPPPConnectionStatus"] = {
    "WANPPPConnection1": [
        "GetInfo",
        "GetConnectionTypeInfo",
        "SetConnectionType",
        "GetStatusInfo",
        "GetUserName",
        "SetUserName",
        "SetPassword",
        "GetNATRSIPStatus",
        "SetConnectionTrigger",
        "ForceTermination",
        "RequestConnection",
        "GetGenericPortMappingEntry",
        "GetSpecificPortMappingEntry",
        "AddPortMapping",
        "DeletePortMapping",
        "GetExternalIPAddress",
        "X_GetDNSServers",
        "GetLinkLayerMaxBitRates",
        "GetPortMappingNumberOfEntries",
        "SetRouteProtocolRx",
        "SetIdleDisconnectTime",
        "X_AVM-DE_GetAutoDisconnectTimeSpan",
        "X_AVM-DE_SetAutoDisconnectTimeSpan",
    ],
}
fc_services_capabilities["WanCommonInterfaceConfig"] = {
    "WANCommonInterfaceConfig1": [
        "GetCommonLinkProperties",
    ],
}
fc_services_capabilities["WanCommonInterfaceDataBytes"] = {
    "WANCommonInterfaceConfig1": [
        "GetTotalBytesSent",
        "GetTotalBytesReceived",
    ],
}
fc_services_capabilities["WanCommonInterfaceDataPackets"] = {
    "WANCommonInterfaceConfig1": [
        "GetTotalPacketsSent",
        "GetTotalPacketsReceived",
    ],
}
fc_services_capabilities["WanCommonInterfaceByteRate"] = {
    "WANCommonIFC1": [
        "GetCommonLinkProperties",
        "GetTotalBytesSent",
        "GetTotalBytesReceived",
        "GetTotalPacketsSent",
        "GetTotalPacketsReceived",
        "GetAddonInfos",
        "X_AVM_DE_GetDsliteStatus",
        "X_AVM_DE_GetIPTVInfos",
    ],
}
fc_services_capabilities["WlanConfigurationInfo"] = {
    "WLANConfiguration1": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
    "WLANConfiguration2": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
}
fc_services_capabilities["HostInfo"] = {
    "Hosts1": [
        "GetHostNumberOfEntries",
        "GetGenericHostEntry",
        "X_AVM-DE_GetSpecificHostEntryByIP",
    ],
}

fc_services_capabilities["HomeAutomation"] = {
    "X_AVM-DE_Homeauto1": [
        "GetInfo",
        "GetGenericDeviceInfos",
    ],
}

fc_services_devices = {}
fc_services_devices["FritzBox 7590"] = {
    "DeviceConfig1": [
        "GetPersistentData",
        "SetPersistentData",
        "ConfigurationStarted",
        "ConfigurationFinished",
        "FactoryReset",
        "Reboot",
        "X_GenerateUUID",
        "X_AVM-DE_GetConfigFile",
        "X_AVM-DE_SetConfigFile",
        "X_AVM-DE_CreateUrlSID",
        "X_AVM-DE_SendSupportData",
        "X_AVM-DE_GetSupportDataInfo",
    ],
    "DeviceInfo1": [
        "GetInfo",
        "SetProvisioningCode",
        "GetDeviceLog",
        "GetSecurityPort",
    ],
    "Hosts1": [
        "GetHostNumberOfEntries",
        "GetSpecificHostEntry",
        "GetGenericHostEntry",
        "X_AVM-DE_GetChangeCounter",
        "X_AVM-DE_SetHostNameByMACAddress",
        "X_AVM-DE_GetAutoWakeOnLANByMACAddress",
        "X_AVM-DE_SetAutoWakeOnLANByMACAddress",
        "X_AVM-DE_WakeOnLANByMACAddress",
        "X_AVM-DE_GetSpecificHostEntryByIP",
        "X_AVM-DE_HostsCheckUpdate",
        "X_AVM-DE_HostDoUpdate",
        "X_AVM-DE_GetHostListPath",
        "X_AVM-DE_GetMeshListPath",
    ],
    "LANConfigSecurity1": [
        "GetInfo",
        "X_AVM-DE_GetCurrentUser",
        "X_AVM-DE_GetAnonymousLogin",
        "SetConfigPassword",
        "X_AVM-DE_GetUserList",
    ],
    "LANEthernetInterfaceConfig1": ["SetEnable", "GetInfo", "GetStatistics"],
    "LANHostConfigManagement1": [
        "GetInfo",
        "SetDHCPServerEnable",
        "SetIPInterface",
        "GetAddressRange",
        "SetAddressRange",
        "GetIPRoutersList",
        "SetIPRouter",
        "GetSubnetMask",
        "SetSubnetMask",
        "GetDNSServers",
        "GetIPInterfaceNumberOfEntries",
    ],
    "Layer3Forwarding1": [
        "SetDefaultConnectionService",
        "GetDefaultConnectionService",
        "GetForwardNumberOfEntries",
        "AddForwardingEntry",
        "DeleteForwardingEntry",
        "GetSpecificForwardingEntry",
        "GetGenericForwardingEntry",
        "SetForwardingEntryEnable",
    ],
    "ManagementServer1": [
        "GetInfo",
        "SetManagementServerURL",
        "SetManagementServerUsername",
        "SetManagementServerPassword",
        "SetPeriodicInform",
        "SetConnectionRequestAuthentication",
        "SetUpgradeManagement",
        "X_SetTR069Enable",
        "X_AVM-DE_GetTR069FirmwareDownloadEnabled",
        "X_AVM-DE_SetTR069FirmwareDownloadEnabled",
    ],
    "Time1": ["GetInfo", "SetNTPServers"],
    "UserInterface1": [
        "GetInfo",
        "X_AVM-DE_CheckUpdate",
        "X_AVM-DE_DoUpdate",
        "X_AVM-DE_DoPrepareCGI",
        "X_AVM-DE_DoManualUpdate",
        "X_AVM-DE_GetInternationalConfig",
        "X_AVM-DE_SetInternationalConfig",
        "X_AVM-DE_GetInfo",
        "X_AVM-DE_SetConfig",
    ],
    "WANCommonIFC1": [
        "GetCommonLinkProperties",
        "GetTotalBytesSent",
        "GetTotalBytesReceived",
        "GetTotalPacketsSent",
        "GetTotalPacketsReceived",
        "GetAddonInfos",
        "X_AVM_DE_GetDsliteStatus",
        "X_AVM_DE_GetIPTVInfos",
    ],
    "WANCommonInterfaceConfig1": [
        "GetCommonLinkProperties",
        "GetTotalBytesSent",
        "GetTotalBytesReceived",
        "GetTotalPacketsSent",
        "GetTotalPacketsReceived",
        "X_AVM-DE_SetWANAccessType",
        "X_AVM-DE_GetOnlineMonitor",
    ],
    "WANDSLInterfaceConfig1": [
        "GetInfo",
        "GetStatisticsTotal",
        "X_AVM-DE_GetDSLDiagnoseInfo",
        "X_AVM-DE_GetDSLInfo",
    ],
    "WANDSLLinkC1": [
        "SetDSLLinkType",
        "GetDSLLinkInfo",
        "GetAutoConfig",
        "GetModulationType",
        "SetDestinationAddress",
        "GetDestinationAddress",
        "SetATMEncapsulation",
        "GetATMEncapsulation",
        "SetFCSPreserved",
        "GetFCSPreserved",
    ],
    "WANDSLLinkConfig1": [
        "GetInfo",
        "SetEnable",
        "GetAutoConfig",
        "SetDSLLinkType",
        "GetDSLLinkInfo",
        "SetDestinationAddress",
        "GetDestinationAddress",
        "SetATMEncapsulation",
        "GetATMEncapsulation",
        "GetStatistics",
    ],
    "WANEthernetLinkConfig1": ["GetEthernetLinkStatus"],
    "WANIPConn1": [
        "SetConnectionType",
        "GetConnectionTypeInfo",
        "GetAutoDisconnectTime",
        "GetIdleDisconnectTime",
        "RequestConnection",
        "RequestTermination",
        "ForceTermination",
        "GetStatusInfo",
        "GetNATRSIPStatus",
        "GetGenericPortMappingEntry",
        "GetSpecificPortMappingEntry",
        "AddPortMapping",
        "DeletePortMapping",
        "GetExternalIPAddress",
        "X_AVM_DE_GetExternalIPv6Address",
        "X_AVM_DE_GetIPv6Prefix",
        "X_AVM_DE_GetDNSServer",
        "X_AVM_DE_GetIPv6DNSServer",
    ],
    "WANIPConnection1": [
        "GetInfo",
        "GetConnectionTypeInfo",
        "SetConnectionType",
        "GetStatusInfo",
        "GetNATRSIPStatus",
        "SetConnectionTrigger",
        "ForceTermination",
        "RequestConnection",
        "GetGenericPortMappingEntry",
        "GetSpecificPortMappingEntry",
        "AddPortMapping",
        "DeletePortMapping",
        "GetExternalIPAddress",
        "X_GetDNSServers",
        "GetPortMappingNumberOfEntries",
        "SetRouteProtocolRx",
        "SetIdleDisconnectTime",
    ],
    "WANIPv6Firewall1": [
        "GetFirewallStatus",
        "GetOutboundPinholeTimeout",
        "AddPinhole",
        "UpdatePinhole",
        "DeletePinhole",
        "GetPinholePackets",
        "CheckPinholeWorking",
    ],
    "WANPPPConnection1": [
        "GetInfo",
        "GetConnectionTypeInfo",
        "SetConnectionType",
        "GetStatusInfo",
        "GetUserName",
        "SetUserName",
        "SetPassword",
        "GetNATRSIPStatus",
        "SetConnectionTrigger",
        "ForceTermination",
        "RequestConnection",
        "GetGenericPortMappingEntry",
        "GetSpecificPortMappingEntry",
        "AddPortMapping",
        "DeletePortMapping",
        "GetExternalIPAddress",
        "X_GetDNSServers",
        "GetLinkLayerMaxBitRates",
        "GetPortMappingNumberOfEntries",
        "SetRouteProtocolRx",
        "SetIdleDisconnectTime",
        "X_AVM-DE_GetAutoDisconnectTimeSpan",
        "X_AVM-DE_SetAutoDisconnectTimeSpan",
    ],
    "WLANConfiguration1": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
    "WLANConfiguration2": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
    "WLANConfiguration3": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
    "X_AVM-DE_AppSetup1": [
        "GetInfo",
        "GetConfig",
        "GetAppMessageFilter",
        "RegisterApp",
        "SetAppVPN",
        "SetAppVPNwithPFS",
        "SetAppMessageFilter",
        "SetAppMessageReceiver",
        "ResetEvent",
        "GetAppRemoteInfo",
    ],
    "X_AVM-DE_Auth1": ["GetInfo", "GetState", "SetConfig"],
    "X_AVM-DE_Dect1": [
        "GetNumberOfDectEntries",
        "GetGenericDectEntry",
        "GetSpecificDectEntry",
        "DectDoUpdate",
        "GetDectListPath",
    ],
    "X_AVM-DE_Filelinks1": [
        "GetNumberOfFilelinkEntries",
        "GetGenericFilelinkEntry",
        "GetSpecificFilelinkEntry",
        "NewFilelinkEntry",
        "SetFilelinkEntry",
        "DeleteFilelinkEntry",
        "GetFilelinkListPath",
    ],
    "X_AVM-DE_Homeauto1": [
        "GetInfo",
        "GetGenericDeviceInfos",
        "GetSpecificDeviceInfos",
        "SetDeviceName",
        "SetSwitch",
    ],
    "X_AVM-DE_Homeplug1": [
        "GetNumberOfDeviceEntries",
        "GetGenericDeviceEntry",
        "GetSpecificDeviceEntry",
        "DeviceDoUpdate",
    ],
    "X_AVM-DE_HostFilter1": [
        "MarkTicket",
        "GetTicketIDStatus",
        "DiscardAllTickets",
        "DisallowWANAccessByIP",
        "GetWANAccessByIP",
    ],
    "X_AVM-DE_MyFritz1": [
        "GetInfo",
        "GetNumberOfServices",
        "GetServiceByIndex",
        "SetServiceByIndex",
        "DeleteServiceByIndex",
    ],
    "X_AVM-DE_OnTel1": [
        "GetInfo",
        "SetEnable",
        "SetConfig",
        "GetInfoByIndex",
        "SetEnableByIndex",
        "SetConfigByIndex",
        "DeleteByIndex",
        "GetNumberOfEntries",
        "GetCallList",
        "GetPhonebookList",
        "GetPhonebook",
        "AddPhonebook",
        "DeletePhonebook",
        "GetPhonebookEntry",
        "GetPhonebookEntryUID",
        "SetPhonebookEntry",
        "SetPhonebookEntryUID",
        "DeletePhonebookEntry",
        "DeletePhonebookEntryUID",
        "GetCallBarringEntry",
        "GetCallBarringEntryByNum",
        "GetCallBarringList",
        "SetCallBarringEntry",
        "DeleteCallBarringEntryUID",
        "GetDECTHandsetList",
        "GetDECTHandsetInfo",
        "SetDECTHandsetPhonebook",
        "GetNumberOfDeflections",
        "GetDeflection",
        "GetDeflections",
        "SetDeflectionEnable",
    ],
    "X_AVM-DE_RemoteAccess1": [
        "GetInfo",
        "SetConfig",
        "SetEnable",
        "GetDDNSInfo",
        "GetDDNSProviders",
        "SetDDNSConfig",
    ],
    "X_AVM-DE_Speedtest1": ["GetInfo", "SetConfig"],
    "X_AVM-DE_Storage1": [
        "GetInfo",
        "RequestFTPServerWAN",
        "SetFTPServer",
        "SetFTPServerWAN",
        "SetSMBServer",
        "GetUserInfo",
        "SetUserConfig",
    ],
    "X_AVM-DE_TAM1": [
        "GetInfo",
        "SetEnable",
        "GetMessageList",
        "MarkMessage",
        "DeleteMessage",
        "GetList",
    ],
    "X_AVM-DE_UPnP1": ["GetInfo", "SetConfig"],
    "X_AVM-DE_WebDAVClient1": ["GetInfo", "SetConfig"],
    "X_VoIP1": [
        "GetInfoEx",
        "X_AVM-DE_AddVoIPAccount",
        "X_AVM-DE_GetVoIPAccount",
        "X_AVM-DE_DelVoIPAccount",
        "GetInfo",
        "SetConfig",
        "GetMaxVoIPNumbers",
        "GetExistingVoIPNumbers",
        "X_AVM-DE_GetNumberOfClients",
        "X_AVM-DE_GetClient",
        "X_AVM-DE_GetClient2",
        "X_AVM-DE_SetClient",
        "X_AVM-DE_SetClient2",
        "X_AVM-DE_GetClient3",
        "X_AVM-DE_GetClientByClientId",
        "X_AVM-DE_SetClient3",
        "X_AVM-DE_SetClient4",
        "X_AVM-DE_GetClients",
        "X_AVM-DE_GetNumberOfNumbers",
        "X_AVM-DE_GetNumbers",
        "X_AVM-DE_DeleteClient",
        "X_AVM-DE_DialGetConfig",
        "X_AVM-DE_DialSetConfig",
        "X_AVM-DE_DialNumber",
        "X_AVM-DE_DialHangup",
        "X_AVM-DE_GetPhonePort",
        "X_AVM-DE_SetDelayedCallNotification",
        "GetVoIPCommonCountryCode",
        "X_AVM-DE_GetVoIPCommonCountryCode",
        "SetVoIPCommonCountryCode",
        "X_AVM-DE_SetVoIPCommonCountryCode",
        "GetVoIPEnableCountryCode",
        "SetVoIPEnableCountryCode",
        "GetVoIPCommonAreaCode",
        "X_AVM-DE_GetVoIPCommonAreaCode",
        "SetVoIPCommonAreaCode",
        "X_AVM-DE_SetVoIPCommonAreaCode",
        "GetVoIPEnableAreaCode",
        "SetVoIPEnableAreaCode",
        "X_AVM-DE_GetAlarmClock",
        "X_AVM-DE_SetAlarmClockEnable",
        "X_AVM-DE_GetNumberOfAlarmClocks",
    ],
    "any1": [],
}

fc_services_devices["FritzRepeater 2400"] = {
    "DeviceConfig1": [
        "GetPersistentData",
        "SetPersistentData",
        "ConfigurationStarted",
        "ConfigurationFinished",
        "FactoryReset",
        "Reboot",
        "X_GenerateUUID",
        "X_AVM-DE_GetConfigFile",
        "X_AVM-DE_SetConfigFile",
        "X_AVM-DE_CreateUrlSID",
        "X_AVM-DE_SendSupportData",
        "X_AVM-DE_GetSupportDataInfo",
    ],
    "DeviceInfo1": [
        "GetInfo",
        "SetProvisioningCode",
        "GetDeviceLog",
        "GetSecurityPort",
    ],
    "Hosts1": [
        "GetHostNumberOfEntries",
        "GetSpecificHostEntry",
        "GetGenericHostEntry",
        "X_AVM-DE_GetChangeCounter",
        "X_AVM-DE_SetHostNameByMACAddress",
        "X_AVM-DE_GetAutoWakeOnLANByMACAddress",
        "X_AVM-DE_SetAutoWakeOnLANByMACAddress",
        "X_AVM-DE_WakeOnLANByMACAddress",
        "X_AVM-DE_GetSpecificHostEntryByIP",
        "X_AVM-DE_HostsCheckUpdate",
        "X_AVM-DE_HostDoUpdate",
        "X_AVM-DE_GetHostListPath",
        "X_AVM-DE_GetMeshListPath",
    ],
    "UserInterface1": [
        "GetInfo",
        "X_AVM-DE_CheckUpdate",
        "X_AVM-DE_DoUpdate",
        "X_AVM-DE_DoPrepareCGI",
        "X_AVM-DE_DoManualUpdate",
        "X_AVM-DE_GetInternationalConfig",
        "X_AVM-DE_SetInternationalConfig",
        "X_AVM-DE_GetInfo",
        "X_AVM-DE_SetConfig",
    ],
    "WLANConfiguration1": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
    "WLANConfiguration2": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
    "WLANConfiguration3": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
}

fc_services_devices["FritzRepeater 1750E"] = {
    "DeviceConfig1": [
        "GetPersistentData",
        "SetPersistentData",
        "ConfigurationStarted",
        "ConfigurationFinished",
        "FactoryReset",
        "Reboot",
        "X_GenerateUUID",
        "X_AVM-DE_GetConfigFile",
        "X_AVM-DE_SetConfigFile",
        "X_AVM-DE_CreateUrlSID",
        "X_AVM-DE_SendSupportData",
        "X_AVM-DE_GetSupportDataInfo",
    ],
    "DeviceInfo1": [
        "GetInfo",
        "SetProvisioningCode",
        "GetDeviceLog",
        "GetSecurityPort",
    ],
    "Hosts1": [
        "GetHostNumberOfEntries",
        "GetSpecificHostEntry",
        "GetGenericHostEntry",
        "X_AVM-DE_GetChangeCounter",
        "X_AVM-DE_SetHostNameByMACAddress",
        "X_AVM-DE_GetAutoWakeOnLANByMACAddress",
        "X_AVM-DE_SetAutoWakeOnLANByMACAddress",
        "X_AVM-DE_WakeOnLANByMACAddress",
        "X_AVM-DE_GetSpecificHostEntryByIP",
        "X_AVM-DE_HostsCheckUpdate",
        "X_AVM-DE_HostDoUpdate",
        "X_AVM-DE_GetHostListPath",
        "X_AVM-DE_GetMeshListPath",
    ],
    "UserInterface1": [
        "GetInfo",
        "X_AVM-DE_CheckUpdate",
        "X_AVM-DE_DoUpdate",
        "X_AVM-DE_DoPrepareCGI",
        "X_AVM-DE_DoManualUpdate",
        "X_AVM-DE_GetInternationalConfig",
        "X_AVM-DE_SetInternationalConfig",
        "X_AVM-DE_GetInfo",
        "X_AVM-DE_SetConfig",
    ],
    "WLANConfiguration1": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
    "WLANConfiguration2": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
    "WLANConfiguration3": [
        "SetEnable",
        "GetInfo",
        "SetConfig",
        "SetSecurityKeys",
        "GetSecurityKeys",
        "SetDefaultWEPKeyIndex",
        "GetDefaultWEPKeyIndex",
        "SetBasBeaconSecurityProperties",
        "GetBasBeaconSecurityProperties",
        "GetStatistics",
        "GetPacketStatistics",
        "GetBSSID",
        "GetSSID",
        "SetSSID",
        "GetBeaconType",
        "SetBeaconType",
        "GetChannelInfo",
        "SetChannel",
        "GetBeaconAdvertisement",
        "SetBeaconAdvertisement",
        "GetTotalAssociations",
        "GetGenericAssociatedDeviceInfo",
        "GetSpecificAssociatedDeviceInfo",
        "X_AVM-DE_GetSpecificAssociatedDeviceInfoByIp",
        "X_AVM-DE_GetWLANDeviceListPath",
        "X_AVM-DE_SetStickSurfEnable",
        "X_AVM-DE_GetIPTVOptimized",
        "X_AVM-DE_SetIPTVOptimized",
        "X_AVM-DE_GetNightControl",
        "X_AVM-DE_GetWLANHybridMode",
        "X_AVM-DE_SetWLANHybridMode",
        "X_AVM-DE_GetWLANExtInfo",
        "X_AVM-DE_GetWPSInfo",
        "X_AVM-DE_SetWPSConfig",
        "X_AVM-DE_SetWPSEnable",
        "X_AVM-DE_SetWLANGlobalEnable",
        "X_AVM-DE_GetWLANConnectionInfo",
    ],
}
