#include "common.h"
#include <glade/glade.h>

/* Makefile creates this file, define DATADIR
 * where frameworks.glade is */
#include "config.h"

#define FPS_FRAMES 10  /* number of elpased times to retain for fps calculation */
#define FWKS_OVERLAY_OPACITY 60  /* between 0..255, the opacity of the overlays */
#define FWKS_OPACITY_FULL 255  /* full opacity */

gpointer test_thread(gpointer data);

gpointer display_thread(gpointer data);
void initialize(struct Interface *interface, struct Camera *camera);
void init_display_mode_vars(struct Interface *interface);

void init_gc(struct Interface *interface);

void update_display(struct Interface *interface);
void render_overlays(struct Interface *interface);
gboolean render_on_pixbuf(GdkPixbuf *src, GdkPixbuf *dest, gint opacity);

void update_fps(struct Interface *interface);
void update_statusbar(struct Interface *interface, const gchar *context, const gchar *msg);

void draw_guides(struct Interface *interface);
GdkPixbuf *sync_rawbuf_pixbuf(const struct RawBuf *buf, GdkPixbuf *pixbuf);
void sync_pixbuf_drawingarea(const GdkPixbuf *pixbuf, const GtkWidget *drawingarea);
void raw2gdkpixbuf(struct RawBuf *raw, GdkPixbuf *pixbuf);

void update_prefs_display_mode_live(struct Interface *interface, gboolean sensitivity);
void update_prefs_display_mode_continuous(struct Interface *interface, gboolean sensitivity);

/* This function does not return until the program is quit. */
struct Interface *frameworks_interface_run(struct Camera *camera)
{
    static struct Interface interface;

    gdk_threads_enter();
    initialize(&interface, camera);
    gdk_threads_leave();

    /* start a thread to update the display */
    (&interface)->display_thread = g_thread_create(display_thread, &interface, TRUE, NULL);

/*    g_thread_create(test_thread, &interface, TRUE, NULL); */

    /* run gtk_main, returns when program is quit by the user */
    gdk_threads_enter();
    gtk_main();
    gdk_threads_leave();

    return &interface;
}

gpointer frameworks_interface_quit(struct Interface *interface)
{
    /* tell our threads to quit */
    fwks_quit = TRUE;

    gdk_threads_leave();
    if (interface->display_thread != NULL
        && interface->camera->open == TRUE
        && g_thread_join(interface->display_thread))

    if (interface->put_thread != NULL
        && g_thread_join(interface->put_thread))
        ;
    gdk_threads_enter();

    gtk_main_quit();
    return NULL;
}

gpointer test_thread(gpointer data)
{
    struct Interface *interface;
    int i;
    interface = (struct Interface *) data;

    /* frame avg test */
    printf("starting in 15 seconds...\n");
    sleep(15);
    printf("started...\n");

    g_mutex_lock(interface->grab_mutex);
    for (i=0; i<1000; i++) {
        test_average(interface, 1);
    }
    printf("done.\n");
    g_mutex_unlock(interface->grab_mutex);

    return NULL;
}

gpointer display_thread(gpointer data)
{
    struct Interface *interface;
    struct Camera *camera;
    gint current_frame, i;
    GdkPixbuf *pixbuf;

    gboolean limit_fps;  /* should we limit live video fps? */
    gfloat max_fps;  /* limit live video to this framerate */

    gfloat speed;  /* continuous display parameters */
    gfloat hold;
    gfloat duration;
    gint frames;  /* calculated number of frames */
    gint release;  /* frame to release the mutex during continuous */

    interface = (struct Interface *) data;
    camera = interface->camera;

    while (fwks_quit == FALSE) {
        /* Update paramter values.  We must store our own cached
         * values because we have to check that they're not zero, and
         * they might change between the time we check and the time we
         * divide */
        limit_fps = interface->limit_fps;
        max_fps = interface->max_fps;
        speed = interface->speed;
        duration = interface->duration;
        hold = interface->hold;
        if (duration > 0 && speed > 0)
            frames = duration * speed;
        else
            frames = 0;

        if (frames > 8)
            release = 8;
        else
            release = frames;

        /* wait for the next available frame */
        g_mutex_lock(camera->buf_b_mutex);
        while (!camera->buf_b_ready)
            g_cond_wait(camera->buf_b_cond, camera->buf_b_mutex);

        /* lock mutex to be unlocked below by each display mode.  The
         * locking is a bit tricky, prone to freezes, and all around
         * poorly done and buggy.  Just be sure not to sleep() while
         * it's locked */
        g_mutex_lock(interface->pixbufs_mutex);

        /* make sure interface->display is the right size */
        interface->display =
            sync_rawbuf_pixbuf(camera->buf_b, interface->display);

        /* render the live frame to interface->display */
        raw2gdkpixbuf(camera->buf_b, interface->display);

        camera->buf_b_ready = FALSE;
        /* now we're done with "next available frame", but we might
         * not want to let the camera read another one just yet.  If
         * we're doing continuous preview, we'd like to have it read a
         * new frame so it's ready just in the nick of time (otherwise
         * it's a stale image from a few seconds ago) */

        if (interface->display_mode == DISPLAY_LIVE) {

            /* allow the camera thread to continue */
            g_mutex_unlock(camera->buf_b_mutex);

            /* render and update display */
            gdk_threads_enter();
            render_overlays(interface);
            update_display(interface);
            g_mutex_unlock(interface->pixbufs_mutex);
            update_fps(interface);
            gdk_flush();
            gdk_threads_leave();

            /* optional speed cap, not super accurate */
            if (limit_fps && max_fps > 0)
                    g_usleep( (1/max_fps) * G_USEC_PER_SEC);
        }
        else {  /* interface->display_mode == DISPLAY_CONTINUOUS_PREVIEW) */

            gdk_threads_enter();
            update_statusbar(interface, "display_thread", " Continuous preview...");
            /* update display for live frame */
            update_display(interface);
            gdk_threads_leave();
            g_mutex_unlock(interface->pixbufs_mutex);

            /* Display the live frame for a bit; hold should be at
             * least 1/fps */
            if (speed > 0 && hold < 1.0/speed)
                hold = 1.0/speed;
            g_usleep(hold * G_USEC_PER_SEC);

            current_frame = interface->current_frame;
            for (i=frames; i>0; i--) {
                pixbuf = frameworks_animation_get_pixbuf(interface->animation, current_frame - i);

                g_mutex_lock(interface->pixbufs_mutex);
                if (render_on_pixbuf(pixbuf, interface->display, FWKS_OPACITY_FULL) == TRUE)
                {
                    gdk_threads_enter();
                    update_display(interface);
                    gdk_threads_leave();

                    g_mutex_unlock(interface->pixbufs_mutex);

                    /* not super accurate framerate, but passable */
                    g_usleep( (1/speed) * G_USEC_PER_SEC);
                } else {
                    g_mutex_unlock(interface->pixbufs_mutex);
                }

                if (i == release) 
                    /* allow the camera thread to continue */
                    g_mutex_unlock(camera->buf_b_mutex);

            }
            if (frames == 0) { /* we didn't go through the for loop */
                /* allow the camera thread to continue */
                g_mutex_unlock(camera->buf_b_mutex);

                /* render and update display */
                g_mutex_lock(interface->pixbufs_mutex);
                gdk_threads_enter();
                update_display(interface);
                gdk_flush();
                gdk_threads_leave();
                g_mutex_unlock(interface->pixbufs_mutex);
            }
        }
    }
    return NULL;
}

void initialize(struct Interface *interface, struct Camera *camera)
{
    GladeXML *xml;
    GtkWidget *tree_view;
    GString *filename;
    GString *datadir;
    GtkCellRenderer *renderer;
    GtkTreeIter iter;
    GtkTreeSelection *selection;
    struct Animation *animation;

    animation = g_malloc(sizeof(struct Animation));
    interface->pixbufs_mutex = g_mutex_new();
    interface->grab_mutex = g_mutex_new();

    interface->elapsed_time = g_array_new(FALSE, TRUE, sizeof(gdouble));
    g_array_set_size(interface->elapsed_time, FPS_FRAMES);
    interface->fps_timer = g_timer_new();
    interface->total_frames = 0;

    /* Threads */
    interface->display_thread = NULL;
    interface->put_thread = NULL;

    /* find our glade interface file */
    datadir = g_string_new(DATADIR);
    if (datadir->str != "" || datadir->str != NULL)
        datadir = g_string_append(datadir, "/");
    filename = g_string_new("frameworks.glade");

    if (g_file_test(filename->str, G_FILE_TEST_EXISTS) == TRUE) {
        xml = glade_xml_new(filename->str, NULL, NULL);
        if (xml == NULL) {
            printf("Failed to build interface from frameworks.glade\n");
            exit(1);
        }
    }
    else if ( (filename = g_string_prepend(filename, datadir->str))
             && g_file_test(filename->str, G_FILE_TEST_EXISTS)) {
        xml = glade_xml_new(filename->str, NULL, NULL);
        if (xml == NULL) {
            printf("Failed to build interface from frameworks.glade\n");
            exit(1);
        }
    }
    else {
        printf("Failed to find frameworks.glade\n");
        exit(1);
    }
    
    interface->gladexml_filename = filename;
    interface->xml = xml;

    interface->window = glade_xml_get_widget(xml, "main_window");
    interface->drawingarea = glade_xml_get_widget(xml, "drawingarea");
    interface->guides_checkbutton = glade_xml_get_widget(xml, "guides_checkbutton");

    interface->display = NULL;

    /* setup gc's */
    interface->gc_frame = NULL;
    interface->gc_overlays = NULL;
    interface->gc_guides = NULL;
    init_gc(interface);

    interface->frame = NULL;
    interface->guides = NULL;
    interface->display_mode = DISPLAY_LIVE;

    interface->limit_fps = FALSE;
    interface->max_fps = 3;

    interface->pixbuf_averaged = NULL;
    interface->camera = camera;
    interface->animation = animation;
    frameworks_animation_initialize(animation);

    /* for each scale (camera controls) set the initial value.  Do
     * this before we connect the signal handlers. */
    frameworks_interface_sync_sliders(interface);

    /* set sensitivity of display mode widgets */
    frameworks_interface_update_display_mode(interface);
    /* set the display mode variables */
    init_display_mode_vars(interface);

    /* connect the signals */
    frameworks_interface_connect_signals(interface);

    /* set the size of the drawing area */
    frameworks_interface_resize_drawing_area(interface);

    /* set up our GtkTreeView that holds thumbnails of all the frames */
    interface->liststore = gtk_list_store_new(N_COLUMNS, G_TYPE_STRING, GDK_TYPE_PIXBUF);

    tree_view = glade_xml_get_widget(interface->xml, "treeview_frames");
    gtk_tree_view_set_model(GTK_TREE_VIEW(tree_view),
                            GTK_TREE_MODEL(interface->liststore));

    renderer = gtk_cell_renderer_text_new();
    interface->column_label = gtk_tree_view_column_new_with_attributes("Frame", renderer,
                                                                       "text", COLUMN_LABEL, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view),
                                GTK_TREE_VIEW_COLUMN(interface->column_label));

    renderer = gtk_cell_renderer_pixbuf_new();
    interface->column_thumb = gtk_tree_view_column_new_with_attributes("Thumbnail", renderer,
                                                                       "pixbuf", COLUMN_THUMB, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view),
                                GTK_TREE_VIEW_COLUMN(interface->column_thumb));
    /* make a label for the current frame (frame 1) */
    gtk_list_store_append(interface->liststore, &iter);
    gtk_list_store_set(interface->liststore, &iter,
                       COLUMN_LABEL, "Frame 1",
                       -1);

    /* hook up the selection callback */
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
    g_signal_connect(G_OBJECT(selection), "changed",
                     G_CALLBACK(on_treeview_frames_selection_changed), interface);
    frameworks_interface_set_current_frame(interface, 1, TRUE);

}

void frameworks_interface_sync_sliders(struct Interface *interface)
{

    GtkWidget *widget;
    gint value;

    /* brightness */
    widget = glade_xml_get_widget(interface->xml, "scale_brightness");
                                    
    value = frameworks_camera_set_brightness(interface->camera, -1);
    gtk_range_set_value(GTK_RANGE(widget), value/256);

    /* FIXME, this should be visible for greyscale cams */
    /* whitebalance - greyscale only */
    /* hide it for now */
    widget = glade_xml_get_widget(interface->xml, "scale_whitebalance");
    value = frameworks_camera_set_whitebalance(interface->camera, -1);
    gtk_range_set_value(GTK_RANGE(widget), value/256);
    gtk_widget_hide_all(widget);
    widget = glade_xml_get_widget(interface->xml, "label_whitebalance");
    gtk_widget_hide_all(widget);

    /* contrast */
    widget = glade_xml_get_widget(interface->xml, "scale_contrast");
    value = frameworks_camera_set_contrast(interface->camera, -1);
    gtk_range_set_value(GTK_RANGE(widget), value/256);
    
    /* hue */
    widget = glade_xml_get_widget(interface->xml, "scale_hue");
    value = frameworks_camera_set_hue(interface->camera, -1);
    gtk_range_set_value(GTK_RANGE(widget), value/256);
    
    /* color */
    widget = glade_xml_get_widget(interface->xml, "scale_color");
    value = frameworks_camera_set_color(interface->camera, -1);
    gtk_range_set_value(GTK_RANGE(widget), value/256);
}

void frameworks_interface_resize_drawing_area(struct Interface *interface)
{
    gint width, height;
    width = frameworks_camera_get_max_width(interface->camera);
    height = frameworks_camera_get_max_height(interface->camera);

    if (width < 1)
        width = 320;
    if (height < 1)
        height = 240;

    gtk_drawing_area_size(GTK_DRAWING_AREA(interface->drawingarea),
			  width, height);
}

void frameworks_interface_connect_signals(struct Interface *interface)
{
    GladeXML *xml;
    
    xml = interface->xml;

    glade_xml_signal_connect_data(xml, "on_drawingarea_expose_event",
                                  G_CALLBACK(frameworks_interface_draw_frame),
                                  interface);

    /* menus */
    glade_xml_signal_connect_data(xml, "on_main_window_delete",
                                  G_CALLBACK(on_main_window_delete),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_quit_activate",
                                  G_CALLBACK(on_quit_activate),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_guides_checkbutton_toggled",
                                  G_CALLBACK(on_guides_checkbutton_toggled),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_about_activate",
                                  G_CALLBACK(on_about_activate),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_open_new_camera_activate",
                                  G_CALLBACK(on_open_new_camera_activate),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_prefs_activate",
                                  G_CALLBACK(on_prefs_activate),
                                  interface);
    /* open new camera */
    glade_xml_signal_connect(xml, "on_fileselection_camera_cancel",
                             G_CALLBACK(on_fileselection_camera_cancel));
    glade_xml_signal_connect_data(xml, "on_fileselection_camera_ok",
                                  G_CALLBACK(on_fileselection_camera_ok),
                                  interface);
    /* prefs */
    glade_xml_signal_connect_data(xml, "on_prefs_delete_event",
                             G_CALLBACK(on_prefs_delete_event),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_checkbutton_pref_bgr_toggled",
                                  G_CALLBACK(on_checkbutton_pref_bgr_toggled),
                                  interface);

    glade_xml_signal_connect_data(xml, "on_checkbutton_frameavg_toggled",
                                  G_CALLBACK(on_checkbutton_frameavg_toggled),
                                  interface);

    glade_xml_signal_connect_data(xml, "on_prefs_display_mode_changed",
                                  G_CALLBACK(on_prefs_display_mode_changed),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_display_mode_changed",
                                  G_CALLBACK(on_display_mode_changed),
                                  interface);

    glade_xml_signal_connect_data(xml, "on_checkbutton_limit_fps_toggled",
                                  G_CALLBACK(on_checkbutton_limit_fps_toggled),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_spinbutton_speed_changed",
                                  G_CALLBACK(on_spinbutton_speed_changed),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_spinbutton_continuous_speed_changed",
                                  G_CALLBACK(on_spinbutton_continuous_speed_changed),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_spinbutton_continuous_hold_changed",
                                  G_CALLBACK(on_spinbutton_continuous_hold_changed),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_spinbutton_continuous_duration_changed",
                                  G_CALLBACK(on_spinbutton_continuous_duration_changed),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_prefs_guides_checkbutton_toggled",
                                  G_CALLBACK(on_prefs_guides_checkbutton_toggled),
                                  interface);
    /* about dialog */
    glade_xml_signal_connect(xml, "on_about_close",
                             G_CALLBACK(on_about_close));
    /* fileselection frames */
    glade_xml_signal_connect_data(xml, "on_fileselection_frames_activate",
                                  G_CALLBACK(on_fileselection_frames_activate),
                                  interface);
    glade_xml_signal_connect(xml, "on_fileselection_frames_cancel",
                             G_CALLBACK(on_fileselection_frames_cancel));
    glade_xml_signal_connect_data(xml, "on_fileselection_frames_ok",
                                  G_CALLBACK(on_fileselection_frames_ok),
                                  interface);
    /* Camera tab: brightness etc. scales and picture size */
    glade_xml_signal_connect_data(xml, "on_scale_brightness_value_changed",
                                  G_CALLBACK(frameworks_interface_change_brightness),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_scale_whitebalance_value_changed",
                                  G_CALLBACK(frameworks_interface_change_whitebalance),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_scale_contrast_value_changed",
                                  G_CALLBACK(frameworks_interface_change_contrast),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_scale_hue_value_changed",
                                  G_CALLBACK(frameworks_interface_change_hue),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_scale_color_value_changed",
                                  G_CALLBACK(frameworks_interface_change_color),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_radiobutton_size_full_toggled",
                                  G_CALLBACK(on_radiobutton_size_full_toggled),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_radiobutton_size_half_toggled",
                                  G_CALLBACK(on_radiobutton_size_half_toggled),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_radiobutton_size_quarter_toggled",
                                  G_CALLBACK(on_radiobutton_size_quarter_toggled),
                                  interface);
    glade_xml_signal_connect_data(xml, "on_radiobutton_size_min_toggled",
                                  G_CALLBACK(on_radiobutton_size_min_toggled),
                                  interface);
    /* animation tab */
    glade_xml_signal_connect_data(xml, "on_grabframe_clicked",
                                  G_CALLBACK(frameworks_interface_put),
                                  interface);
/*    glade_xml_signal_connect_data(xml, "on_checkbutton_frameoverlay_toggled",
                                  G_CALLBACK(on_checkbutton_frameoverlay_toggled),
                                  interface); */
    glade_xml_signal_connect_data(xml, "on_spinbutton_currentframe_value_changed",
                                  G_CALLBACK(on_spinbutton_currentframe_value_changed),
                                  interface);
}

void init_gc(struct Interface *interface)
{
    if (interface->gc_frame != NULL) {
	g_object_unref(interface->gc_frame);
	interface->gc_frame = NULL;
    }
    interface->gc_frame = gdk_gc_new(interface->drawingarea->window);

    if (interface->gc_overlays != NULL) {
        g_object_unref(interface->gc_overlays);
        interface->gc_overlays = NULL;
    }
    interface->gc_overlays = gdk_gc_new(interface->drawingarea->window);
    
    if (interface->gc_guides != NULL) {
        g_object_unref(interface->gc_guides);
        interface->gc_guides = NULL;
    }
    interface->gc_guides = gdk_gc_new(interface->drawingarea->window);
    gdk_gc_set_line_attributes(interface->gc_guides, 1, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
    gdk_gc_set_function(interface->gc_guides, GDK_INVERT);
}

/* Make a gdkpixbuf the same dimensions as a rawbuf (not necesarily
 * the same bpp).  It will modify the pixbuf pointer. */
GdkPixbuf * sync_rawbuf_pixbuf(const struct RawBuf *buf, GdkPixbuf *pixbuf)
{
    if (buf != NULL
        && (pixbuf == NULL
            || buf->width != gdk_pixbuf_get_width(pixbuf)
            || buf->height != gdk_pixbuf_get_height(pixbuf)
            )
        )
    {
        if (pixbuf != NULL) g_object_unref(pixbuf);
        pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB,
                                FALSE, 8,
                                buf->width,
                                buf->height);
    }
    return pixbuf;
}

void sync_pixbuf_drawingarea(const GdkPixbuf *pixbuf, const GtkWidget *drawingarea)
{
    if (pixbuf != NULL) {
        gtk_drawing_area_size(GTK_DRAWING_AREA(drawingarea),
                              gdk_pixbuf_get_width(pixbuf),
                              gdk_pixbuf_get_height(pixbuf));
    }
}

/* copy the raw format read from the camera into a Gdk pixbuf */
void raw2gdkpixbuf(struct RawBuf *buf, GdkPixbuf *pixbuf)
{
    int width, height;
    gint pixbuf_rowstride;
    gpointer pixbuf_start;
    gpointer raw;
    int i = 0;
    int j = 0;

    /* just return if the buffers aren't the same */
    if (buf == NULL
        || pixbuf == NULL
        || buf->width != gdk_pixbuf_get_width(pixbuf)
        || buf->height != gdk_pixbuf_get_height(pixbuf))
        return;

    width = buf->width;
    height = buf->height;
    raw = buf->data;

    /* GDK pixbuf is block of 3 byte pixels, where each row may be
     * longer than necessary (length equal to rowstride) */
    pixbuf_start = gdk_pixbuf_get_pixels(pixbuf);
    pixbuf_rowstride = gdk_pixbuf_get_rowstride(pixbuf);

    /* copy the data from buf into pixbuf, converting to rgb24 */
    colorspace_rawbuf_to_rgb24(buf, pixbuf_start, pixbuf_rowstride);
}

void render_overlays(struct Interface *interface)
{
    GtkWidget *w;  /* generic widget pointer */
    gint i, past, future, current_frame;

    w = glade_xml_get_widget(interface->xml, "checkbutton_frameoverlay");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)) == TRUE) {

        w = glade_xml_get_widget(interface->xml, "spinbutton_pastframes");
        past = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));
        w = glade_xml_get_widget(interface->xml, "spinbutton_futureframes");
        future = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));

        current_frame = interface->current_frame;
        for (i = 1; i <= past; i++) {
            render_on_pixbuf(frameworks_animation_get_pixbuf(interface->animation, current_frame - i),
                             interface->display,
                             FWKS_OVERLAY_OPACITY);
        }
        for (i = 1; i <= future; i++) {
            render_on_pixbuf(frameworks_animation_get_pixbuf(interface->animation, current_frame + i),
                             interface->display,
                             FWKS_OVERLAY_OPACITY);
        }
    }
}

/* renders a src pixbuf onto another at given opacity, unrefs the src pixbuf */
gboolean render_on_pixbuf(GdkPixbuf *src, GdkPixbuf *dest, gint opacity)
{
    gint width_src, height_src;  /* dimensions of the src (overlayed) pixbuf */
    gint width_dest, height_dest;  /* dimensions of the destination pixbuf */

    if (src != NULL && dest != NULL) {

        width_dest = gdk_pixbuf_get_width(dest);
        height_dest = gdk_pixbuf_get_height(dest);

        width_src = gdk_pixbuf_get_width(src);
        height_src = gdk_pixbuf_get_height(src);

        if (width_src > width_dest)
            width_src = 0;
        if (height_src > height_dest)
            height_src = 0;

        gdk_pixbuf_composite(src, dest,
                             0, 0,
                             width_src, height_src,
                             0, 0, 1, 1, GDK_INTERP_BILINEAR, opacity);
        g_object_unref(src);
        return TRUE;
    } else {
        return FALSE;
    }
}

/* render interface->display to the drawingarea*/
void update_display(struct Interface *interface)
{
    /* make sure stuff's the right size */
    sync_pixbuf_drawingarea(interface->display, interface->drawingarea);

    /*
     * tell gtk the widget needs to be redrawn, gtk runs the
     * "expose_event" callback when it's ready to be drawn (the
     * callback renders the frame to the drawingarea and draws
     * guides); this avoid flicker.
     */

 /* NOTE: this is unecessary because the above function triggers the
 * callback anyway */

/*    gdk_threads_enter();
    gtk_widget_queue_draw(interface->drawingarea);
    gdk_threads_leave(); */

}

#define OLDEST 0
void update_fps(struct Interface *interface)
{
    gulong musec;
    gdouble elapsed_time;
    gdouble total_time = 0;

    gchar *status;  /* status bar message */

    gint i;

    /* update fps count.  We use a g_array to hold the last 10 elapsed
     * times, and each time around we discard the earliest time so we
     * always have the ten latest elapsed times to average for a
     * smooth fps counter.  Before the array fills up, the values are
     * zero. */
    interface->total_frames++;

    elapsed_time = g_timer_elapsed(interface->fps_timer, &musec);
    g_timer_start(interface->fps_timer);  /* reset timer to zero */

    g_array_append_val(interface->elapsed_time, elapsed_time);
    g_array_remove_index(interface->elapsed_time, OLDEST); /* remove the oldest value */

    /* keep total_frames in sync with the array */
    if (interface->total_frames > FPS_FRAMES)
        interface->total_frames--;

    for (i=0; i<FPS_FRAMES; i++) {
        total_time += g_array_index(interface->elapsed_time, gdouble, i);
    }

    status = g_strdup_printf(" Live FPS: %.1f", interface->total_frames / total_time);
    update_statusbar(interface, "display_thread", status);
    g_free(status);  /* is this right? i guess so...*/
}

/* Remove the previous statusbar message, and set a new one, or remove
 * one if the memssage is "".  Remember to remove every message you
 * are done with. */
void update_statusbar(struct Interface *interface, const gchar *context, const gchar *msg)
{
    guint context_id;
    GtkWidget *statusbar;

    statusbar = glade_xml_get_widget(interface->xml, "statusbar");
    context_id = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar), context);

    if (strcmp(msg, "") != 0) {
        gtk_statusbar_pop(GTK_STATUSBAR(statusbar), context_id);
        gtk_statusbar_push(GTK_STATUSBAR(statusbar), context_id, msg);
    } else {
        gtk_statusbar_pop(GTK_STATUSBAR(statusbar), context_id);
    }
}

gboolean frameworks_interface_draw_frame(GtkWidget * widget,
					 GdkEventExpose * event,
					 struct Interface *interface)
{
    gint width_win, height_win;  /* dimensions of drawing area */
    gint width_display, height_display;  /* dimensions of interface->display */
    gint left, right, top, bottom;

    width_win = interface->drawingarea->allocation.width;
    height_win = interface->drawingarea->allocation.height;

    /* make it black (or whatever the gc's bg color is) */
    gdk_draw_rectangle(interface->drawingarea->window, interface->gc_frame,
                       TRUE,  /* filled */
                       0, 0,
                       width_win, height_win);

    /* We must give up the gdk lock before attempting to lock the
     * pixbufs_mutex because the display thread needs the gdk lock for
     * a while before it can unlock the pixbufs_mutex */
    gdk_threads_leave();
    g_mutex_lock(interface->pixbufs_mutex);
    gdk_threads_enter();

    if (interface->camera->open == TRUE
        && interface->display != NULL) {

        width_display = gdk_pixbuf_get_width(interface->display);
        height_display = gdk_pixbuf_get_height(interface->display);

        /* make sure our pixbuf fits in the drawingarea */
        if (width_display > width_win)
            width_display = 0;
        if (height_display > height_win)
            height_display = 0;

        left = (width_win - width_display) / 2;
        right = (width_win + width_display) / 2;
        top = (height_win - height_display) / 2;
        bottom = (height_win + height_display) / 2;
                                 
        /* draw pixbuf centered in window */
        gdk_draw_pixbuf(interface->drawingarea->window, interface->gc_frame,
                        interface->display,
                        0, 0,
                        left, top,
                        width_display, height_display,
                        GDK_RGB_DITHER_NORMAL, 0, 0);

        /* draw guides on window */
        if (gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(interface->guides_checkbutton)) == TRUE) {
            gdk_draw_line(interface->drawingarea->window, interface->gc_guides,
                          width_win / 2, top,
                          width_win / 2, bottom);
            gdk_draw_line(interface->drawingarea->window, interface->gc_guides,
                          left, height_win / 2,
                          right, height_win / 2);
            gdk_draw_arc(interface->drawingarea->window, interface->gc_guides, FALSE,
                         width_win/2 - 15, height_win/2 - 15,
                         30, 30,
                         0, 360 * 64);
        }

    } else if (interface->camera->open == FALSE) {
        /* draw "no camera" message on window */
        gdk_draw_string(interface->drawingarea->window,
                      gdk_font_load("-*-helvetica-medium-r-normal--*-120-*-*-*-*-iso8859-1"),
                      interface->gc_guides,
                      width_win / 2 - 80, height_win / 2,
                      "No Camera Device");
    }
    /* else, camera is open but pixbuf is currently NULL; do nothing */

    g_mutex_unlock(interface->pixbufs_mutex);
    return TRUE;
}

void frameworks_interface_update_progressbar(struct Interface *interface,
                                             const gchar *text,
                                             gdouble fraction)
{
    GtkWidget *pbar;

    gdk_threads_enter();
    pbar = glade_xml_get_widget(interface->xml, "progressbar");
    gtk_progress_bar_set_text(GTK_PROGRESS_BAR(pbar), text);
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(pbar), fraction);
    gdk_flush();
    gdk_threads_leave();
}

void frameworks_interface_change_brightness(GtkRange * range,
					    struct Interface *interface)
{
    int value;
    value = gtk_range_get_value(range);
    value = frameworks_camera_set_brightness(interface->camera, value * 256);
    gtk_range_set_value(range, value/256);
}

void frameworks_interface_change_whitebalance(GtkRange * range,
                                              struct Interface *interface)
{
    int value;
    value = gtk_range_get_value(range);
    value = frameworks_camera_set_whitebalance(interface->camera, value * 256);
    gtk_range_set_value(range, value/256);
}

void frameworks_interface_change_contrast(GtkRange *range,
                                          struct Interface *interface)
{
    int value;
    value = gtk_range_get_value(range);
    value = frameworks_camera_set_contrast(interface->camera, value * 256);
    gtk_range_set_value(range, value/256);
}

void frameworks_interface_change_hue(GtkRange *range,
                                     struct Interface *interface)
{
    int value;
    value = gtk_range_get_value(range);
    value = frameworks_camera_set_hue(interface->camera, value * 256);
    gtk_range_set_value(range, value/256);
}

void frameworks_interface_change_color(GtkRange *range,
                                       struct Interface *interface)
{
    int value;
    value = gtk_range_get_value(range);
    value = frameworks_camera_set_color(interface->camera, value * 256);
    gtk_range_set_value(range, value/256);
}

void frameworks_interface_set_current_frame(struct Interface *interface, gint current_frame, gboolean scroll)
{
    GtkTreePath *path;
    GtkWidget *spin;
    GtkWidget *treeview;
    GtkTreeSelection *selection;
    gint last_frame;
    
    if (interface->current_frame == current_frame) {
        return;
    }

    interface->current_frame = current_frame;

    spin = glade_xml_get_widget(interface->xml, "spinbutton_currentframe");
    last_frame = frameworks_animation_get_last_frame(interface->animation);
    if (last_frame < current_frame) {
        gtk_spin_button_set_range(GTK_SPIN_BUTTON(spin),
                                  1, last_frame + 1);
    }
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), current_frame);

    treeview = glade_xml_get_widget(interface->xml, "treeview_frames");
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    path = gtk_tree_path_new_from_indices(current_frame - 1, -1);
    if (scroll == TRUE) 
        gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(treeview),
                                     path, NULL, TRUE, 0.667, 0);

    gtk_tree_selection_select_path(selection, path);
    gtk_tree_path_free(path);
    
    frameworks_animation_set_current_frame(interface->animation, current_frame);
}

/* callback to save stuff */
int frameworks_interface_put(GtkButton *button, struct Interface *interface)
{
    if (interface->put_thread == NULL)  /* only create one at a time */
        interface->put_thread =
            g_thread_create(frameworks_interface_put_thread, interface, TRUE, NULL);
    return 1;
}

gpointer frameworks_interface_put_thread(gpointer data)
{
    struct Interface *interface;
    GtkWidget *spin;
    GtkWidget *button;
    GString *string;
    gint current_frame;
    gint num;  /* number of frames to average */

    GdkPixbuf *pixbuf;
    GdkPixbuf *thumb;
    GtkTreePath *path;
    GtkTreeIter iter;

    interface = (struct Interface *) data;

    /* Only run one of these threads at once */
    if (g_mutex_trylock(interface->grab_mutex) == FALSE)
        return NULL;

    frameworks_interface_update_progressbar(interface, "Grabbing...", 0);

    gdk_threads_enter();
    spin = glade_xml_get_widget(interface->xml, "spinbutton_frameavg");
    button = glade_xml_get_widget(interface->xml, "checkbutton_frameavg");

    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))  /* frame averaging is enabled */
        num = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
    else
        num = 1;  /* averaging one frame is the same as no averaging */

    gdk_threads_leave();

    frameworks_camera_read_averaged(interface->camera, num, interface);
    interface->pixbuf_averaged = sync_rawbuf_pixbuf(interface->camera->buf_avg, interface->pixbuf_averaged);
    raw2gdkpixbuf(interface->camera->buf_avg, interface->pixbuf_averaged);
    frameworks_animation_put(interface->animation, interface->pixbuf_averaged);

    current_frame = interface->current_frame + 1;

    /* add thumbnail to box_frames */
    gdk_threads_enter();

    /* NOTE: The list of frames is a one-indexed list; furthermore,
     * current_frame - 1 is the frame we just acquired while current
     * frame is the next frame which we need to add a label for.  The
     * liststore is a zero-indexed list, so current_frame - 2 is the
     * one we just acquired. */

    /* add the pixbuf for the frame just aquired */
#define FWKS_THUMB_HEIGHT 30
    gint w, h;
    pixbuf = frameworks_animation_get_pixbuf(interface->animation, current_frame - 1);
    w = gdk_pixbuf_get_width(pixbuf);
    h = gdk_pixbuf_get_height(pixbuf);
    thumb = gdk_pixbuf_scale_simple(pixbuf,
                                    FWKS_THUMB_HEIGHT*w/h,  /* keep aspect ratio */
                                    FWKS_THUMB_HEIGHT,
                                    GDK_INTERP_BILINEAR);
    path = gtk_tree_path_new_from_indices(current_frame - 2, -1);
    gtk_tree_model_get_iter(GTK_TREE_MODEL(interface->liststore), &iter, path);
    gtk_list_store_set(interface->liststore, &iter,
                       COLUMN_THUMB, thumb,
                       -1);
    /* make sure it updates incase the thumb already existed */
    gtk_tree_model_row_changed(GTK_TREE_MODEL(interface->liststore), path, &iter);
    gtk_tree_path_free(path);

    /* Add a label for the current frame (which won't have a thumb yet) */
    string = g_string_new("Frame ");
    g_string_append_printf(string, "%d", current_frame);

    path = gtk_tree_path_new_from_indices(current_frame - 1, -1);
    if (gtk_tree_model_get_iter(GTK_TREE_MODEL(interface->liststore), &iter, path) == TRUE)
        /* do nothing because it's already there */;
    else { /* we must be at the end of the list; append a new one */
        gtk_list_store_append(interface->liststore, &iter);
        gtk_list_store_set(interface->liststore, &iter,
                           COLUMN_LABEL, string->str,
                           -1);
    }
    gtk_tree_path_free(path);
    gdk_threads_leave();

    /* select current frame in treeview */
    gdk_threads_enter();
    frameworks_interface_set_current_frame(interface, current_frame, TRUE);
    gdk_threads_leave();

    /* text must be at least a blank, or height will shrink */
    frameworks_interface_update_progressbar(interface, " ", 0);

    g_mutex_unlock(interface->grab_mutex);
    interface->put_thread = NULL;
    return NULL;
}

/* initialize the display mode variables (speed, hold, etc.) */
void init_display_mode_vars(struct Interface *interface)
{
#define SET(i) (spin = glade_xml_get_widget(interface->xml, i))
#define VALUE gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin))

    GtkWidget *spin;

    spin = glade_xml_get_widget(interface->xml, "checkbutton_limit_fps");
    interface->limit_fps = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(spin));
    SET("spinbutton_speed");
    interface->max_fps = VALUE;

    SET("spinbutton_continuous_speed");
    interface->speed = VALUE;
    SET("spinbutton_continuous_hold");
    interface->hold = VALUE;
    SET("spinbutton_continuous_duration");
    interface->duration = VALUE;
}

/* set sensitivity of widgets based on the display mode set in the main window */
void frameworks_interface_update_display_mode(struct Interface *interface)
{
    GtkWidget *button;

    /* set the overlapping buttons in the prefs dialog */
    button = glade_xml_get_widget(interface->xml, "checkbutton_frameoverlay");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(interface->xml,
                                                                        "checkbutton_prefs_frameoverlay")),
                                 gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)));
    button = glade_xml_get_widget(interface->xml, "radiobutton_display_live");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(interface->xml,
                                                                        "radiobutton_prefs_display_live")),
                                 gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)));
    button = glade_xml_get_widget(interface->xml, "radiobutton_display_continuous");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(interface->xml,
                                                                        "radiobutton_prefs_display_continuous")),
                                 gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)));

    button = glade_xml_get_widget(interface->xml, "radiobutton_display_live");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button))) {
        /* display mode is live */
        interface->display_mode = DISPLAY_LIVE;
        update_prefs_display_mode_live(interface, TRUE);
        update_prefs_display_mode_continuous(interface, FALSE);
    } else {
        /* display mode is continuous preview */
        interface->display_mode = DISPLAY_CONTINUOUS_PREVIEW;
        update_prefs_display_mode_live(interface, FALSE);
        update_prefs_display_mode_continuous(interface, TRUE);
    }
}

/* set sensitivity of widgets based on the display mode set in the prefs window */
void frameworks_interface_prefs_update_display_mode(struct Interface *interface)
{
    GtkWidget *button;

    /* set the overlapping buttons in the main window */
    button = glade_xml_get_widget(interface->xml, "checkbutton_prefs_frameoverlay");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(interface->xml,
                                                                        "checkbutton_frameoverlay")),
                                 gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)));
    button = glade_xml_get_widget(interface->xml, "radiobutton_prefs_display_live");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(interface->xml,
                                                                        "radiobutton_display_live")),
                                 gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)));
    button = glade_xml_get_widget(interface->xml, "radiobutton_prefs_display_continuous");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(interface->xml,
                                                                        "radiobutton_display_continuous")),
                                 gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)));

    button = glade_xml_get_widget(interface->xml, "radiobutton_prefs_display_live");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button))) {
        /* display mode is live */
        interface->display_mode = DISPLAY_LIVE;
        update_prefs_display_mode_live(interface, TRUE);
        update_prefs_display_mode_continuous(interface, FALSE);
    } else {
        /* display mode is continuous preview */
        interface->display_mode = DISPLAY_CONTINUOUS_PREVIEW;
        update_prefs_display_mode_live(interface, FALSE);
        update_prefs_display_mode_continuous(interface, TRUE);
    }
}

void update_prefs_display_mode_live(struct Interface *interface, gboolean sensitivity)
{
    GtkWidget *widget;
    GtkWidget *button;

    if (sensitivity == FALSE) {
        /* desensitize everything */
        widget = glade_xml_get_widget(interface->xml, "checkbutton_frameoverlay");
        gtk_widget_set_sensitive(widget, FALSE);

        widget = glade_xml_get_widget(interface->xml, "checkbutton_limit_fps");
        gtk_widget_set_sensitive(widget, FALSE);
        widget = glade_xml_get_widget(interface->xml, "label_speed");
        gtk_widget_set_sensitive(widget, FALSE);
        widget = glade_xml_get_widget(interface->xml, "spinbutton_speed");
        gtk_widget_set_sensitive(widget, FALSE);
        widget = glade_xml_get_widget(interface->xml, "label_speed_units");

        widget = glade_xml_get_widget(interface->xml, "checkbutton_prefs_frameoverlay");
        gtk_widget_set_sensitive(widget, FALSE);
        widget = glade_xml_get_widget(interface->xml, "label_pastframes");
        gtk_widget_set_sensitive(widget, FALSE);
        widget = glade_xml_get_widget(interface->xml, "spinbutton_pastframes");
        gtk_widget_set_sensitive(widget, FALSE);
        widget = glade_xml_get_widget(interface->xml, "label_futureframes");
        gtk_widget_set_sensitive(widget, FALSE);
        widget = glade_xml_get_widget(interface->xml, "spinbutton_futureframes");
        gtk_widget_set_sensitive(widget, FALSE);
    } else {
        /* speed */
        button = glade_xml_get_widget(interface->xml, "checkbutton_limit_fps");
        gtk_widget_set_sensitive(button, TRUE);
        if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button))) {
            sensitivity = TRUE;
        }
        else
            sensitivity = FALSE;

        widget = glade_xml_get_widget(interface->xml, "label_speed");
        gtk_widget_set_sensitive(widget, sensitivity);
        widget = glade_xml_get_widget(interface->xml, "spinbutton_speed");
        gtk_widget_set_sensitive(widget, sensitivity);
        widget = glade_xml_get_widget(interface->xml, "label_speed_units");
        gtk_widget_set_sensitive(widget, sensitivity);

        /* frame overlay */
        button = glade_xml_get_widget(interface->xml, "checkbutton_frameoverlay");
        gtk_widget_set_sensitive(button, TRUE);
        button = glade_xml_get_widget(interface->xml, "checkbutton_prefs_frameoverlay");
        gtk_widget_set_sensitive(button, TRUE);
        if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))
            sensitivity = TRUE;
        else
            sensitivity = FALSE;

        widget = glade_xml_get_widget(interface->xml, "label_pastframes");
        gtk_widget_set_sensitive(widget, sensitivity);
        widget = glade_xml_get_widget(interface->xml, "label_futureframes");
        gtk_widget_set_sensitive(widget, sensitivity);
        widget = glade_xml_get_widget(interface->xml, "spinbutton_pastframes");
        gtk_widget_set_sensitive(widget, sensitivity);
        widget = glade_xml_get_widget(interface->xml, "spinbutton_futureframes");
        gtk_widget_set_sensitive(widget, sensitivity);
    }
}

void update_prefs_display_mode_continuous(struct Interface *interface, gboolean sensitivity)
{
    GtkWidget *widget;

    widget = glade_xml_get_widget(interface->xml, "label_continuous_speed");
    gtk_widget_set_sensitive(widget, sensitivity);
    widget = glade_xml_get_widget(interface->xml, "spinbutton_continuous_speed");
    gtk_widget_set_sensitive(widget, sensitivity);
    widget = glade_xml_get_widget(interface->xml, "label_continuous_speed_units");
    gtk_widget_set_sensitive(widget, sensitivity);

    widget = glade_xml_get_widget(interface->xml, "label_continuous_duration");
    gtk_widget_set_sensitive(widget, sensitivity);
    widget = glade_xml_get_widget(interface->xml, "spinbutton_continuous_duration");
    gtk_widget_set_sensitive(widget, sensitivity);
    widget = glade_xml_get_widget(interface->xml, "label_continuous_duration_units");

    widget = glade_xml_get_widget(interface->xml, "label_continuous_hold");
    gtk_widget_set_sensitive(widget, sensitivity);
    widget = glade_xml_get_widget(interface->xml, "spinbutton_continuous_hold");
    gtk_widget_set_sensitive(widget, sensitivity);
    widget = glade_xml_get_widget(interface->xml, "label_continuous_hold_units");
    gtk_widget_set_sensitive(widget, sensitivity);
}


gpointer test_average(struct Interface *interface, gint num)
{
    GtkWidget *spin;
    GtkWidget *button;
    GString *string;
    gint current_frame;

    GdkPixbuf *pixbuf;
    GdkPixbuf *thumb;
    GtkTreePath *path;
    GtkTreeIter iter;

    frameworks_interface_update_progressbar(interface, "Grabbing...", 0);

    spin = glade_xml_get_widget(interface->xml, "spinbutton_frameavg");
    button = glade_xml_get_widget(interface->xml, "checkbutton_frameavg");

    frameworks_camera_read_averaged(interface->camera, num, interface);
    interface->pixbuf_averaged = sync_rawbuf_pixbuf(interface->camera->buf_avg, interface->pixbuf_averaged);
    raw2gdkpixbuf(interface->camera->buf_avg, interface->pixbuf_averaged);
    frameworks_animation_put(interface->animation, interface->pixbuf_averaged);

    current_frame = interface->current_frame + 1;

    /* add thumbnail to box_frames */
    gdk_threads_enter();

    /* NOTE: The list of frames is a one-indexed list; furthermore,
     * current_frame - 1 is the frame we just acquired while current
     * frame is the next frame which we need to add a label for.  The
     * liststore is a zero-indexed list, so current_frame - 2 is the
     * one we just acquired. */

    /* add the pixbuf for the frame just aquired */
    pixbuf = frameworks_animation_get_pixbuf(interface->animation, current_frame - 1);
    thumb = gdk_pixbuf_scale_simple(pixbuf,
                                    50, 30, GDK_INTERP_BILINEAR);
    path = gtk_tree_path_new_from_indices(current_frame - 2, -1);
    gtk_tree_model_get_iter(GTK_TREE_MODEL(interface->liststore), &iter, path);
    gtk_list_store_set(interface->liststore, &iter,
                       COLUMN_THUMB, thumb,
                       -1);
    /* make sure it updates incase the thumb already existed */
    gtk_tree_model_row_changed(GTK_TREE_MODEL(interface->liststore), path, &iter);
    gtk_tree_path_free(path);

    /* Add a label for the current frame (which won't have a thumb yet) */
    string = g_string_new("Frame ");
    g_string_append_printf(string, "%d", current_frame);

    path = gtk_tree_path_new_from_indices(current_frame - 1, -1);
    if (gtk_tree_model_get_iter(GTK_TREE_MODEL(interface->liststore), &iter, path) == TRUE)
        /* do nothing because it's already there */;
    else { /* we must be at the end of the list; append a new one */
        gtk_list_store_append(interface->liststore, &iter);
        gtk_list_store_set(interface->liststore, &iter,
                           COLUMN_LABEL, string->str,
                           -1);
    }
    gtk_tree_path_free(path);
    gdk_threads_leave();

    /* select current frame in treeview */
    gdk_threads_enter();
    frameworks_interface_set_current_frame(interface, current_frame, TRUE);
    gdk_threads_leave();

    /* text must be at least a blank, or height will shrink */
    frameworks_interface_update_progressbar(interface, " ", 0);

    return NULL;
}
