// 
// hash.c -- simple hash support
//

//
// Copyright (c) 2004 Dean Gaudet <dean@arctic.org>
// 
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
// 

// $Id: hash.c,v 1.1 2004-11-08 01:54:59 dean Exp $

#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include "hash.h"

typedef struct hash_elm_str {
	struct hash_elm_str *next;
	void *key;
	size_t key_len;
	void *value;
} hash_elm_t;

struct hash_str {
	size_t n_bits;
	size_t n;
	size_t n_elts;
	hash_elm_t *table[0];
};

hash_t *hash_create(size_t n_bits)
{
	size_t n = 1<<n_bits;
	hash_t *h;

	h = malloc(sizeof(*h) + n * sizeof(hash_elm_t));
	h->n_bits = n_bits;
	h->n = n;
	h->n_elts = 0;
	memset(h->table, 0, n*sizeof(hash_elm_t));
	return h;
}


void hash_destroy(hash_t *h)
{
	size_t i;

	for (i = 0; i < h->n; ++i) {
		hash_elm_t *he = h->table[i];
		while (he) {
			hash_elm_t *next = he->next;
			free(he->key);
			free(he);
			he = next;
		}
	}
	free(h);
}


void hash_destroy_and_free_value(hash_t *h)
{
	size_t i;

	for (i = 0; i < h->n; ++i) {
		hash_elm_t *he = h->table[i];
		while (he) {
			hash_elm_t *next = he->next;
			free(he->key);
			free(he->value);
			free(he);
			he = next;
		}
	}
	free(h);
}


size_t hash_n_elts(hash_t *h)
{
	return h->n_elts;
}


static uint32_t hash_key(const void *vkey, size_t key_len, size_t n)
{
	const uint32_t *key = vkey;
	size_t i;
	uint32_t h = 0;

	key_len /= sizeof(uint32_t);
	for (i = 0; i < key_len; ++i) {
		h = (h << 4) | (h >> 28);
		h ^= key[i];
	}
	h ^= (h >> 12) ^ (h >> 22);
	return h & (n - 1);
}


void *hash_lookup(hash_t *h, const void *key, size_t key_len)
{
	hash_elm_t **phe;
	uint32_t k;

	k = hash_key(key, key_len, h->n);
	phe = &h->table[k];
	while (*phe) {
		if ((*phe)->key_len == key_len
			&& memcmp((*phe)->key, key, key_len) == 0) {
			// move to front
			hash_elm_t *tmp_he = *phe;
			*phe = tmp_he->next;
			tmp_he->next = h->table[k];
			h->table[k] = tmp_he;
			return tmp_he->value;
		}
		phe = &(*phe)->next;
	}
	return NULL;
}


// insert key into hash
void hash_insert(hash_t *h, const void *key, size_t key_len, void *value)
{
	hash_elm_t *he;
	size_t k;

	++h->n_elts;
	k = hash_key(key, key_len, h->n);
	he = malloc(sizeof(hash_elm_t));
	he->next = h->table[k];
	h->table[k] = he;
	he->key = malloc(key_len);
	memcpy(he->key, key, key_len);
	he->key_len = key_len;
	he->value = value;
}


hash_linear_t *hash_linearize(hash_t *h)
{
	hash_linear_t *l;
	size_t i;
	size_t k;

	l = malloc(sizeof(*l) * h->n_elts);
	i = 0;
	for (k = 0; k < h->n; ++k) {
		hash_elm_t *he = h->table[k];
		while (he) {
			l[i].key = he->key;
			l[i].key_len = he->key_len;
			l[i].value = he->value;
			++i;
			he = he->next;
		}
	}
	assert(i == h->n_elts);
	return l;
}
