/*
   convert.c
   
   By Bill Kendrick
   CS 495 Fall 1995
   October 14, 1995 - October 15, 1995
   November 4, 1997
*/

/* Debugging info flags */

#define VERBOSE 0
#define DUMP 0
#define MASKDUMP 0


/* Max size defines */

#define WIDTH 4096
#define HEIGHT 256
#define SIZE WIDTH * HEIGHT / 8


#include <stdio.h>

char progname[128];
int TwoToThe[8];


/* prototypes: */

void render(unsigned char *dest, int width, int height, unsigned char *source);
void load(FILE *fi, int w, int h, unsigned char *buf, char *file);
void save(FILE *fi, int w, int h, unsigned char *buf);
void usage(char *error);
void help(void);


int main(int argc, char *argv[])
{
  char infile[128][80], inmask[128][80];
  char outfile[80], type[200], sbuf[200];
  unsigned char buf[SIZE], buf2[SIZE];
  FILE *fi, *fo;
  int i, ins, mask, step, width, height, back, allwidth, allheight;
  
  TwoToThe[0] = 128;
  TwoToThe[1] = 64;
  TwoToThe[2] = 32;
  TwoToThe[3] = 16;
  TwoToThe[4] = 8;
  TwoToThe[5] = 4;
  TwoToThe[6] = 2;
  TwoToThe[7] = 1;

  strcpy(progname, argv[0]);
  
  /* Check for "-h" help argument */
  
  if (argc == 2 && strcmp(argv[1], "-h") == 0)
    help();
  
  
  /* minimum 5 arguments: "prog -m file -o out" */
  
  if (argc < 5)
    if (argc == 2)
      {
	sprintf(sbuf, "Unknown switch: %s", argv[1]);
	usage(sbuf);
      }
    else
      usage("Not enough parameters");
  
  
  /* First argument should specify conversion mode: */
  
  step = 1;
  back = 0;
  
  if (strcmp(argv[1], "-x") == 0)         /* Solid rectangles for masks */
    mask = 0;
  else if (strcmp(argv[1], "-f") == 0)    /* Use mask-files for masks */
    {
      mask = 1;
      step = 2;
    }
  else if (strcmp(argv[1], "-r") == 0)    /* Render the masks */
    mask = 2;
  else if (strcmp(argv[1], "-b") ==0 )    /* Background */
    back = 1;
  else
    {
      sprintf(sbuf,"Unknown conversion mode: %s",argv[1]);
      usage(sbuf);
    }
  
  
  /* You must specify a mask file for every file if you use file-based masks */
  
  if (step == 2 && ((argc - 3) % 2) != 1)
    {
      sprintf(sbuf, 
	      "You must specify a mask file for every file in \"%s\" mode",
	      argv[1]);
      usage(sbuf);
    }
  
  /* Last arguments should specify out file: */
  
  if (strcmp(argv[argc - 2], "-o") != 0)
    usage("No out-file specified");
  strcpy(outfile, argv[argc - 1]);
  
  
  /* Determine all of the in-files */
  
  ins = 0;
  
  for (i = 2; i < argc - 2; i = i + step)
    {
      strcpy(infile[ins], argv[i]);
      if (step == 2)
	strcpy(inmask[ins], argv[i + 1]);
      ins = ins + 1;
    }
  
  
  /* Open the out file */
  
  fo = fopen(outfile, "w");
  if (fo == NULL)
    {
      sprintf(sbuf,"Error creating output file \"%s\"",outfile);
      perror(sbuf);
      exit(1);
    }
  
  if (back == 0)
    fprintf(fo, "obj\n");
  else
    fprintf(fo, "bkg\n");
  
  fprintf(fo, "%d\n", ins);
  
  
  if (VERBOSE == 1)
    printf("Files: %d\n", ins);
  
  
  /* We shall find out the width/height once we open the first file */
  /* (All of the other files had better be the same width/height or */
  /* we'll get mad) */
  
  allwidth = -1;
  allheight = -1;
  

  /* Go through the bitmap or pixmap files and do the work! */
  
  for (i=0; i < ins; i++)
    {
      /* Open file: */
      
      fi = fopen(infile[i], "r");
      
      if (fi == NULL)
	{
	  sprintf(sbuf, "Can't open input file \"%s\"", infile[i]);
	  perror(sbuf);
	  fclose(fo);
	  exit(1);
	}
      
      
      /* Get file type (should be "P4" for binary-saved PBM's   */
      /*                      and "P6" for binary-saved PPM's)  */
      
      fscanf(fi, "%s", type);
      
      if ((back == 0 && strcmp(type, "P4") != 0) || 
	  (back == 1 && strcmp(type, "P6") != 0))
	{
	  fprintf(stderr,
		  "\nBad file header \"%s\" in file \"%s\"\n\n", type,
		  infile[i]);
	  exit(1);
	}
      
      
      /* Get width and height */
      
      fscanf(fi, "%d", &width);
      fscanf(fi, "%d", &height);
      
      
      /* Has it been stored in the output file yet? */
      
      if (allwidth == -1 && allheight == -1)
	{
	  if (width < 1 || height < 1)
	    {
	      fprintf(stderr,"\nBad image size in file \"%s\"\n\n", infile[i]);
	      exit(1);
	    }

	  fprintf(fo, "%d\n%d\n", width, height);
	  allwidth = width;
	  allheight = height;
	  
	  if (VERBOSE == 1)
	    printf("Size: %d x %d\n", allwidth, allheight);


	  /* Do we have enough space for this stuff? */

	  if (back == 0)
	    {
	      if (allwidth / 8 * allheight * ins > SIZE)
		{
		  fprintf(stderr,
			  "Error: Not enough memory (alloc'd:%d, req'd:%d)\n",
			  SIZE, allwidth / 8 * allheight * ins);
		  exit(1);
		}
	    }
	  else
	    {
	      /* Eat maxbits value from PPM file */

	      fscanf(fi, "%s", sbuf);
	      
	      if (allwidth * allheight > SIZE)
		{
		  fprintf(stderr,
			  "Error: Not enough memory (alloc'd:%d, req'd:%d)\n",
			  SIZE, allwidth * allheight);
		  exit(1);
		}
	    }
	}
      else if (width != allwidth || height != allheight)
	{
	  /* Oops!  Conflicting sizes! */
	  
	  fprintf(stderr, "\nAll images must be the same size (%d x %d) ",
		  allwidth, allheight);
	  fprintf(stderr, "but \"%s\" is not\n\n", infile[i]);
	  exit(1);
	}
      
      if (VERBOSE == 1)
	printf("Adding %d: \"%s\"\n", i + 1, infile[i]);
      

      /* Load the data */
      
      if (back == 0) 
	load(fi, width / 8, height, buf + (i * width / 8 * height), infile[i]);
      else
	{
	  load(fi, width, height, buf, infile[i]);
	  save(fo, width, height, buf);
	}
      
      fclose(fi);
    }
  
  
  /* Background file is now complete. */

  if (back == 1)
    {
      fclose(fo);
      exit(0);
    }


  /* Now for the masks. */

  if (mask == 0)
    {
      /* solid rectangle */

      if (VERBOSE == 1)
	printf("Using solid rectangle masks\n");

      for (i = 0; i < width / 8 * height * ins ; i++)
	buf2[i] = 0xFF;
    }
  else if (mask == 1)
    {
      /* files for masks */

      /* Go through the bitmap (mask) files and do the work! */
      
      for (i=0; i < ins; i++)
	{
	  /* Open file: */
	  
	  fi = fopen(inmask[i], "r");
	  
	  if (fi == NULL)
	    {
	      sprintf(sbuf, "Can't open input file \"%s\"", inmask[i]);
	      perror(sbuf);
	      fclose(fo);
	      exit(1);
	    }
	  
	  
	  /* Get file type (should be "P4" for binary-saved PBM's */
	  
	  fscanf(fi, "%s", type);
	  
	  if (strcmp(type, "P4") != 0)
	    {
	      fprintf(stderr,
		      "\nBad file header \"%s\" in file \"%s\"\n\n", type,
		      inmask[i]);
	      exit(1);
	    }
	  
	  
	  /* Get width and height */
	  
	  fscanf(fi, "%d", &width);
	  fscanf(fi, "%d", &height);
	  
	  
	  if (width != allwidth || height != allheight)
	    {
	      /* Oops!  Conflicting sizes! */
	      
	      fprintf(stderr, "\nAll images must be the same size (%d x %d) ",
		      allwidth, allheight);
	      fprintf(stderr, "but \"%s\" is not\n\n", inmask[i]);
	      exit(1);
	    }
	  
	  if (VERBOSE == 1)
	    printf("Adding mask %d: \"%s\"\n", i + 1, inmask[i]);
	  
	  
	  /* Load the data */
	  
	  load(fi, width / 8, height, buf2 + (i * width / 8 * height),
	       inmask[i]);
	  fclose(fi);
	}
    }
  else if (mask == 2)
    {
      /* render the masks */

      for (i=0; i < ins; i++)
	{
	  if (VERBOSE == 1)
	    printf("Rendering mask for sprite %d\n", i + 1);
	  render(buf2 + ((allwidth / 8) * height * i), allwidth, allheight,
		 buf + ((allwidth / 8) * height * i));
	}
    }

  for (i = 0; i < ins; i++)
    {
      if (VERBOSE == 1)
	printf("Saving frame %d\n", i + 1);
      save(fo, allwidth / 8, allheight, buf + (i * allwidth / 8 * allheight));
      save(fo, allwidth / 8, allheight, buf2 + (i * allwidth / 8 * allheight));
    }

  fclose(fo);
  
  exit(0);
}

void render(unsigned char *dest, int width, int height, unsigned char *source)
{
  unsigned char destbits[WIDTH][HEIGHT];
  int x, y, xx;
  unsigned char z;

  /* Clear buffers */
  
  for (y = 0; y < height; y++)
    {
      for (x = 0; x < width; x++)
	{
	  destbits[x][y] = 0;
	  dest[(y * width) + (x / 8)] = 0;
	}
    }
  
  /* Convert to single bits (destbits[]) and "thicken" */

  for (y = 1; y < height - 1; y++)
    for (x = 0; x < width / 8; x++)
      {
	z = source[(y * width / 8) + x];
	for (xx = 0; xx < 8; xx++)
	  if ((z & TwoToThe[xx]) != 0)
	    {
	      destbits[x * 8 + xx][y] = 1;
	      destbits[x * 8 + xx + 1][y] = 1;
	      destbits[x * 8 + xx - 1][y] = 1;
	      destbits[x * 8 + xx][y - 1] = 1;
	      destbits[x * 8 + xx][y + 1] = 1;
	    }
      }
  

  /* Search through the bits for "stray" blanks */

  for (y = 1; y < height - 1; y++)
    for (x = 1; x < width - 1; x++)
      {
	if (destbits[x][y] == 0)
	  {
	    z = destbits[x - 1][y - 1] + destbits[x][y - 1]
	      + destbits[x + 1][y - 1] + destbits[x - 1][y]
		+ destbits[x + 1][y] + destbits[x - 1][y + 1]
		  + destbits[x][y + 1] + destbits[x + 1][y + 1];
	    if (z > 4)
	      destbits[x][y] = 1;
	  }
      }

  
  /* Back to bytes */
  
  for (y = 0; y < height; y++)
    {
      for (x = 0; x < width ; x++)
	{
	  if (destbits[x][y] == 1)
	    {
	      dest[(y * width / 8) + (x / 8)] += TwoToThe[(x % 8)];
	      if (MASKDUMP == 1)
		printf("*");
	    }
	  else
	    if (MASKDUMP == 1)
	      printf(",");
	}
      if (MASKDUMP == 1)
	{
	  printf(" = ");
	  for (x = 0; x < width / 8; x++)
	    {
	      printf("%.2x", dest[(y * width / 8) + x]);
	    }
	  printf("\n");
	}
    }
}

void load(FILE *fi, int w, int h, unsigned char *buf, char *file)
{
  int z;

  fgetc(fi);

  for (z = 0; z < w * h; z++)
    {
      buf[z] = fgetc(fi);
      if (feof(fi) != 0)
	{
	  fprintf(stderr, "\nUnexpected end of file at %d in \"%s\"\n", z,
		  file);
	  exit(1);
	}
    }
}

void save(FILE *fo, int w, int h, unsigned char *buf)
{
  int z, o, p, flip;

  for (z = 0; z < w * h; z++)
    {
      o = buf[z];
      p = 0;
      for (flip = 0; flip < 8; flip++)
	if (o & TwoToThe[flip])
	  p += TwoToThe[7 - flip];
      fprintf(fo,"%c",p);
      if (DUMP == 1)
	printf("%4x",o);
    }
  if (DUMP == 1)
    printf("\n");
}

void usage(char * error)
{
  fprintf(stderr, "\nError: %s\n",error);
  fprintf(stderr, "Usage:\t%s [-r | -x] file1.pbm... -o file.obj\n", progname);
  fprintf(stderr, "\t%s -f file1.pbm filemask.pbm... -o file.obj\n",
	  progname);
  fprintf(stderr, "\t%s -b file1.ppm... -o file.bkg\n", progname);
  fprintf(stderr, "\t%s -h\n\n", progname);
  exit(1);
}

void help(void)
{
  printf("%s: Convert\n\n", progname);
  printf("...\n");
  exit(0);
}
