/* #includes */ /*{{{C}}}*//*{{{*/
#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef BROKEN_REALLOC
#define realloc(s,l) myrealloc(s,l)
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "libfe.h"
/*}}}*/
/* #defines */ /*{{{*/
#define KILL_LEN 128
/*}}}*/

/* types */ /*{{{*/
struct Emdisplay
{
  struct Display *d;
  enum { YANKMARK=1, APPENDKILL=2, AUTOINDENT=4 } flags;
};
/*}}}*/
/* variables */ /*{{{*/
static char *filehist[HISTORY_LEN];
static char *gotohist[HISTORY_LEN];
static char *searchhist[HISTORY_LEN];
static char *cmdhist[HISTORY_LEN];
static struct Buffer *killbf[KILL_LEN];
static int kill_cur, kill_len;
static struct Emdisplay **emdisplay=(struct Emdisplay**)0;
static unsigned int esz=0;
static unsigned int ce=0;
static int counter=0, increment=1;
/*}}}*/

/* em_name       -- name a buffer */ /*{{{*/
static int em_name(struct Emdisplay *e, char *history[HISTORY_LEN])
{
  char path[_POSIX_PATH_MAX];
  size_t x,offx;
    
  x=offx=0;
  path[0]='\0';
  if (editline(path,sizeof(path),_("File name?"),&x,&offx,history)==0) return -1;
  if (bf_name(e->d->buffer,path)==0) return 0;
  return 1;
}
/*}}}*/
/* em_save       -- save a file */ /*{{{*/
static int em_save(struct Emdisplay *e)
{
  unsigned int chars;

  if (e->d->buffer->name==(char*)0)
  {
    switch (em_name(e,filehist))
    {
      case -1: return 0; /* abort by user */
      case 0: showmsg(_("[Naming file failed (%s)]"),strerror(errno)); return 0;
      case 1: break; /* ok */
      default: assert(0);
    }
  }
  if (bf_save(e->d->buffer,e->d->buffer->name,"{{{","}}}",&chars))
  {
    showmsg(_("[Wrote %d characters]"),chars);
    return 1;
  }
  else
  {
    showmsg(_("[File not written!]"));
    return 0;
  }
}
/*}}}*/
/* em_saveas     -- save as another file */ /*{{{*/
static void em_saveas(struct Emdisplay *e)
{
  char file[_POSIX_PATH_MAX];
  size_t x,offx;
  unsigned int chars;
  
  x=offx=0;
  file[0]='\0';
  if (editline(file,sizeof(file),_("Save as file?"),&x,&offx,filehist)==1 && file[0])
  {
    if (ok(_("Save without fold marks:"),0)==1)
    {
      if (bf_write(e->d->buffer,file,&chars)==1) showmsg(_("[Wrote %d characters]"),chars);
      else showmsg(_("[File not written!]"));
    }
    else
    {
      if (bf_save(e->d->buffer,file,"{{{","}}}",&chars)==1) showmsg(_("[Wrote %d characters]"),chars);
      else showmsg(_("[File not written!]"));
    }
  }
}
/*}}}*/
/* em_load       -- load a file */ /*{{{*/
static void em_load(void)
{
  emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
  if (emdisplay[ce]->d->buffer->refcnt>1 || !(emdisplay[ce]->d->buffer->mode&CHANGED) || ok(_("Buffer changed, load new file anyway:"),0)==1)
  {
    struct Buffer *b;

    if ((b=bf_alloc())==(struct Buffer*)0) showmsg(_("[No file loaded]"));
    else
    {
      ds_realloc(emdisplay[ce]->d,b);
      if (em_name(emdisplay[ce],filehist)==1)
      {
        int i;

        for (i=0; i<MAXDISPLAYS; ++i)
        {
          if (buffers[i] && buffers[i]->name && strcmp(buffers[i]->name,emdisplay[ce]->d->buffer->name)==0 && emdisplay[ce]->d->buffer!=buffers[i])
          {
            ds_free(emdisplay[ce]->d);
            ds_realloc(emdisplay[ce]->d,buffers[i]);
            ds_current(emdisplay[ce]->d);
            showmsg(_("[Attached already loaded buffer]"));
            return;
          }
        }
        if (bf_load(emdisplay[ce]->d->buffer,(const char*)0,emdisplay[ce]->d->buffer->name,-1,"{{{","}}}"))
        {
          emdisplay[ce]->d->buffer->mode&=~CHANGED;
          ds_begin();
          return;
        }
        else showmsg(_("[Loading failed: %s]"),strerror(errno));
      }
    }
  }
  else showmsg(_("[No file loaded]"));
}
/*}}}*/
/* em_insert     -- insert a file */ /*{{{*/
static void em_insert(struct Emdisplay *e)
{
  struct Buffer *b;
  char file[_POSIX_PATH_MAX];
  size_t x,offx;
  
  e->flags&=~(YANKMARK|APPENDKILL);
  x=offx=0;
  file[0]='\0';
  if (editline(file,sizeof(file),_("Insert file?"),&x,&offx,filehist)==1 && file[0])
  {
    if ((b=bf_alloc())==(struct Buffer*)0)
    {
      showmsg(_("[Out of memory]"));
      return;
    }
    if (bf_load(b,(const char*)0,file,-1,"{{{","}}}")==0) showmsg(_("[Loading failed: %s]"),strerror(errno));
    ds_insertbuf(b,1);
    bf_free(b);
  }
}
/*}}}*/
/* em_gotoline   -- goto numbered line */ /*{{{*/
static void em_gotoline(struct Emdisplay *e, int argument)
{
  e->flags&=~(YANKMARK|APPENDKILL);
  if (argument>0)
  {
    if (ds_gotoline(argument-1)==0) showmsg(_("[No such line]"));
  }
  else
  {
    char buf[20],*bufp;
    size_t x,offx;
            
    buf[0]='\0';
    x=offx=(size_t)0;
    if (editline(buf,sizeof(buf),_("Go to line number?"),&x,&offx,gotohist)==1)
    {
      if ((argument=strtod(buf,&bufp))>0 && bufp>buf && *bufp=='\0')
      {
        if (ds_gotoline(argument-1)==0) showmsg(_("[No such line]"));
      }
      else showmsg(_("[This is not a valid number]"));
    }
  }
}
/*}}}*/
/* em_search     -- search */ /*{{{*/
static void em_search(struct Emdisplay *e, int forward)
{
  char buf[128];
  size_t x,offx;
            
  e->flags&=~(YANKMARK|APPENDKILL);
  buf[0]='\0';
  x=offx=(size_t)0;
  if (editline(buf,sizeof(buf),forward ? _("String to search forward?") : _("String to search backward?"),&x,&offx,searchhist)==1)
  {
    if (e->d->mode&MAGIC)
    {
      regex_t needle;
      int err;

      if ((err=regcomp(&needle,buf,REG_NEWLINE)))
      {
        size_t sz;
        char errstr[256];

        sz=regerror(err,&needle,errstr+1,sizeof(errstr)-2);
        errstr[0]='[';
        errstr[sz]=']';
        errstr[sz+1]='\0';
        showmsg("%s",errstr);
      }
      else if ((forward ? ds_regfsearch : ds_regbsearch)(&needle)==0) showmsg(_("[Not found]"));
      regfree(&needle);
    }
    else if ((forward ? ds_strfsearch : ds_strbsearch)(buf,strlen(buf))==0) showmsg(_("[Not found]"));
  }
}
/*}}}*/
/* em_page       -- move a page but stay into fold */ /*{{{*/
static void em_page(int argument, int forward)
{
  int dy;
  int hitstart;

  emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
  if (argument==0) argument=1;
  dy=emdisplay[ce]->d->y;
  if (argument<0)
  {
    forward=!forward;
    argument=-argument;
  }
  while (argument>0)
  {
    hitstart=(forward ? ds_nextpage() : ds_prevpage());
    if (hitstart==1) --argument;
    else if (hitstart==0)
    {
      showmsg(forward ? _("[You are in the last line]") : _("[You are in the first line]"));
      break;
    }
    else if (hitstart==2) break;
  }
  ds_torow(emdisplay[ce]->d,dy);
}
/*}}}*/
/* em_kill       -- store buffer in kill ring */ /*{{{*/
static int em_kill(struct Buffer *b, int append)
{
  if (kill_len==0) append=0;
  if (append)
  {
    char c,mark;
    unsigned int count;
    
    bf_end(killbf[kill_cur]);
    bf_begin(b);
    bf_lchar(killbf[kill_cur],&c,&mark);
    if (c!='\n' && bf_isfold(killbf[kill_cur],FOLDEND,LEFT) && bf_isfold(b,FOLDSTART,RIGHT))
    {
      bf_insert(killbf[kill_cur],'\n','\0');
      showmsg(_("[An extra newline has been appended to the kill buffer for consistency]"));
      count=1;
    }
    else count=0;
    while (bf_rchar(b,&c,&mark))
    {
      if (bf_insert(killbf[kill_cur],c,mark)) ++count;
      else
      {
        while (count>0) { bf_delete(killbf[kill_cur]); --count; }
        return 0;
      }
      bf_forward(b);
    }
    bf_free(b);
  }
  else
  {
    kill_cur=(kill_cur+1)%KILL_LEN;
    if (kill_len<KILL_LEN) ++kill_len;
    bf_free(killbf[kill_cur]);
    killbf[kill_cur]=b;
  }
  return 1;
}
/*}}}*/
/* em_itssearch  -- incremental search */ /*{{{*/
static void em_itssearch(struct Emdisplay *e, int direction, char *history[HISTORY_LEN])
{
  /* variables *//*{{{*/
  size_t bufsize;
  char buf[128];
  char *start;
  chtype c;
  int oops;
  int inhist=-1;
  const char *estr;
  char estrbuf[256];
  regex_t regex;
  /*}}}*/

  buf[0]='\0';
  bufsize=0;
  oops=0;
  estr=_("[search failed]");
  start=e->d->buffer->gapstart;
  do
  {
    showmsg(direction ? _("I-Search: %s%s") : _("R-Search: %s%s"),buf,oops ? estr : "");
    ds_refresh();
    switch (c=mc_get())
    {
      /* C-h, KEY_BACKSPACE */ /*{{{*/
      case '\010':
      case KEY_BACKSPACE:
      {
        if (bufsize) buf[--bufsize]='\0';
        break;
      }
      /*}}}*/
      /* C-s */ /*{{{*/
      case '\023':
      {
        if (direction==1 && !oops) bf_forward(e->d->buffer);
        direction=1;
        start=e->d->buffer->gapstart;
        break;
      }
      /*}}}*/
      /* C-r */ /*{{{*/
      case '\022':
      {
        if (direction==0 && !oops) bf_backward(e->d->buffer);
        direction=0;
        start=e->d->buffer->gapstart;
        break;  
      }
      /*}}}*/
      /* C-p */ /*{{{*/
      case KEY_UP:
      case '\020':
      {
        if (inhist<(HISTORY_LEN-1) && history[inhist+1]!=(char*)0)
        {
          strcpy(buf,history[++inhist]);
          bufsize=strlen(buf);
        }
        break;
      }
      /*}}}*/
      /* C-n -- next line in history */ /*{{{*/
      case KEY_DOWN:
      case '\016':
      {
        if (inhist>0)
        {
          strcpy(buf,history[--inhist]);
          bufsize=strlen(buf);
        }
        break;
      }
      /*}}}*/
      /* C-q -- quote character */ /*{{{*/
      case '\021': 
      {
        chtype c2;
        
        c=mc_get();
        if (c==(c&0xff))
        {
          if (ds_compose(c,0))
          {
            c2=mc_get();
            if (c2==(c2&0xff) && ds_compose(c,c2)) c=((unsigned char)ds_compose(c,c2));
            else break;
          }
        }
        else break;
      }
      /*}}}*/
      default:     /*     -- add to search string */ /*{{{*/
      {
        if (c==(c&0xff) && c>=(unsigned char)' ' && bufsize<(sizeof(buf)-1))
        {
          buf[bufsize]=(char)c;
          buf[++bufsize]='\0';
        }
      }
      /*}}}*/
    }
    ds_topos(e->d,start);
    if (e->d->mode&MAGIC)
    {
      if ((oops=regcomp(&regex,buf,REG_NEWLINE)))
      {
        size_t sz;

        estr=estrbuf;
        sz=regerror(oops,&regex,estrbuf+1,sizeof(estrbuf)-2);
        estrbuf[0]='[';
        estrbuf[sz]=']';
        estrbuf[sz+1]='\0';
      }
      else
      {
        if ((oops=!(direction ? ds_regfsearch(&regex) : ds_regbsearch(&regex)))) estr=_("[search failed]");
      }
      regfree(&regex);
    }
    else
    {
      if ((oops=!(direction ? ds_strfsearch(buf,bufsize) : ds_strbsearch(buf,bufsize))))
      estr=_("[search failed]");
    }
  } while (c!='\r' && c!='\n' && c!=KEY_ENTER && c!='\007' && c!=KEY_CANCEL);
  if (c=='\007' || c==KEY_CANCEL) ds_topos(e->d,start);
  else
  {
    if (history[HISTORY_LEN-1]) free(history[HISTORY_LEN-1]);
    memmove(history+1,history,sizeof(history[0])*(HISTORY_LEN-1));
    history[0]=mystrmalloc(buf);
  }
  showmsg((const char*)0);
}
/*}}}*/
/* em_replace    -- search and replace */ /*{{{*/
static void em_replace(struct Emdisplay *e, int argument, int query)
{
  /* variables *//*{{{*/
  char search[128],replace[128];
  size_t x,offx;
  MenuChoice menu[6];
  int replacements=0,oops;
  regex_t regex;
  /*}}}*/
            
  e->flags&=~(YANKMARK|APPENDKILL);
  search[0]='\0';
  x=offx=(size_t)0;
  if (editline(search,sizeof(search),argument<0 ? _("String to search backward?") : _("String to search forward?"),&x,&offx,searchhist)==1)
  {
    if ((e->d->mode&MAGIC) && (oops=regcomp(&regex,search,REG_NEWLINE)))
    {
      size_t sz;
      char estrbuf[256];

      sz=regerror(oops,&regex,estrbuf+1,sizeof(estrbuf)-2);
      estrbuf[0]='[';
      estrbuf[sz]=']';
      estrbuf[sz+1]='\0';
      showmsg("%s",estrbuf);
      return;
    }
    menu[0].str=_("yYes");                menu[0].c='\0';
    menu[1].str=_("nNo");                 menu[1].c='\0';
    menu[2].str=_("dD)o rest");           menu[2].c='!';
    menu[3].str=_("aA)bort");             menu[3].c='\0';
    menu[4].str=_("rR)eplace and abort"); menu[4].c='.';
    menu[5].str=(char*)0;
    replace[0]='\0';
    x=offx=(size_t)0;
    if (editline(replace,sizeof(replace),_("Replace by?"),&x,&offx,searchhist)==1)
    {
      int noarg=0;
      int doit=1;

      if (argument==0) noarg=1;
      while
      (
        (noarg || argument)
        &&
        (
          doit==1 ||
          (
            argument<0
            ? bf_backward(e->d->buffer)
            : bf_forward(e->d->buffer)
          )!=-1
        )
        &&
        (
          argument<0
          ? (e->d->mode&MAGIC?ds_regbsearch(&regex):ds_strbsearch(search,strlen(search)))
          : (e->d->mode&MAGIC?ds_regfsearch(&regex):ds_strfsearch(search,strlen(search)))
          ==1
        )
      )
      {
        /* variables *//*{{{*/
        struct Buffer *b;
        char *r,*s;
        /*}}}*/

        if (query) /*{{{*/
        {
          ds_refresh();
          switch (menuline(_("Replace:"),menu,0))
          {
            case 0: doit=1; break;
            case 1: doit=0; break;
            case 2: query=0; break;
            case 3: noarg=argument=doit=0; break;
            case 4: doit=1; noarg=argument=0; break;
            case -1: noarg=argument=doit=0; break;
            default: assert(0);
          }
        }
        /*}}}*/
        if (!query || doit) /* replace string *//*{{{*/
        {
          for (r=replace; *r; ++r) ds_insertch(*r);
          ds_mark();
          for (s=search; *s; ++s) bf_forward(e->d->buffer);
          b=ds_covebuf(e->d,1,1);
          if (b==(struct Buffer*)0)
          {
            for (s=search; *s; ++s) bf_backward(e->d->buffer);
            for (r=replace; *r; ++r) ds_deletech(1);
            break;
          }
          else bf_free(b);
          ++replacements;
        }
        /*}}}*/
        if (!noarg)
        {
          if (argument<0) ++argument;
          else if (argument>0) --argument;
        }
      }
      showmsg(_("[%d replacements]"),replacements);
    }
  }
}
/*}}}*/
/* em_filter     -- filter region */ /*{{{*/
static void em_filter(struct Emdisplay *e)
{
  /* variables */ /*{{{*/
  struct Buffer *old,*new=(struct Buffer*)0;
  char cmd[_POSIX_PATH_MAX];
  size_t x,offx;
  char *tmp;
  /*}}}*/

  e->flags&=~(YANKMARK|APPENDKILL);
  x=offx=0;
  cmd[0]='\0';
  if (editline(cmd,sizeof(cmd),_("Command?"),&x,&offx,cmdhist)==0) return;
  if (cmd[0])
  {
    if ((old=ds_covebuf(e->d,1,1))!=(struct Buffer*)0)
    {
      int ok=1;
      char tmpstr[L_tmpnam];

      tmp=mytmpnam(tmpstr);
      if (bf_save(old,tmp,"{{{","}}}",(unsigned int*)0))
      {
        if ((new=bf_alloc())!=(struct Buffer*)0)
        {
          if (bf_load(new,cmd,tmp,-1,"{{{","}}}"))
          {
            if (ds_insertbuf(new,1)==0)
            {
              showmsg(_("[The filtered region can not be inserted here]"));
              ok=0;
            }
          }
          else
          {
            showmsg(_("[The filtered region can not be loaded]"));
            ok=0;
          }
        }
        else
        {
          showmsg(_("[The filtered region can not be loaded]"));
          ok=0;
        }
      }
      else
      {
        showmsg(_("[The filtered region can not be saved]"));
        ok=0;
      }
      unlink(tmp);
      if (ok)
      {
        em_kill(old,0);
        em_kill(new,0);
        e->flags|=YANKMARK;
      }
      else
      {
        ds_insertbuf(old,1);
        bf_free(old);
        if (new) bf_free(new);
      }
    }
    else showmsg(_("[You can not filter this region]"));
  }
}
/*}}}*/
/* em_new        -- allocate new emdisplay by splitting current emdisplay */ /*{{{*/
static int em_new(void)
{
  int i;

  if (esz>0 && emdisplay[ce]->d->maxy<4) return 0;
  ++esz;
  emdisplay=realloc(emdisplay,esz*sizeof(struct Emdisplay *));
  for (i=(esz-1); i>(ce+1); --i) emdisplay[i]=emdisplay[i-1];
  if (esz==1) ce=0; else ++ce;
  emdisplay[ce]=malloc(sizeof(struct Emdisplay));
  if (esz==1) emdisplay[ce]->d=ds_alloc(bf_alloc(),0,0,COLS,LINES-1);
  else
  {
    int old_x,old_y,old_maxx,old_maxy;

    old_x=emdisplay[ce-1]->d->orix;
    old_y=emdisplay[ce-1]->d->oriy;
    old_maxx=emdisplay[ce-1]->d->maxx;
    old_maxy=emdisplay[ce-1]->d->maxy;
    ds_reshape(old_x,old_y,old_maxx,old_maxy/2);
    emdisplay[ce]->d=ds_alloc(emdisplay[ce-1]->d->buffer,old_x,old_y+old_maxy/2,old_maxx,old_maxy-old_maxy/2);
    emdisplay[ce]->flags=emdisplay[ce-1]->flags;
    emdisplay[ce]->d->mode=emdisplay[ce-1]->d->mode;
    emdisplay[ce]->d->tabsize=emdisplay[ce-1]->d->tabsize;
  }
  ds_current(emdisplay[ce]->d);
  return 1;
}
/*}}}*/
/* em_pipe       -- process region and put result in new emdisplay */ /*{{{*/
static void em_pipe(void)
{
  /* variables */ /*{{{*/
  struct Buffer *old,*new=(struct Buffer*)0;
  char cmd[_POSIX_PATH_MAX];
  size_t x,offx;
  char *tmp;
  /*}}}*/

  assert(esz>=1);
  emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
  x=offx=0;
  cmd[0]='\0';
  if (editline(cmd,sizeof(cmd),_("Command?"),&x,&offx,cmdhist)==0) return;
  if (cmd[0])
  {
    if ((old=ds_covebuf(emdisplay[ce]->d,0,1))!=(struct Buffer*)0)
    {
      int ok=1;
      char tmpstr[L_tmpnam];

      tmp=mytmpnam(tmpstr);
      if (bf_save(old,tmp,"{{{","}}}",(unsigned int*)0))
      {
        if ((new=bf_alloc())!=(struct Buffer*)0)
        {
          if (bf_load(new,cmd,tmp,-1,"{{{","}}}"))
          {
            if (emdisplay[ce]->d->maxy>=4)
            {
              int i;
              int old_x,old_y,old_maxx,old_maxy;

              ++esz;
              emdisplay=realloc(emdisplay,esz*sizeof(struct Emdisplay *));
              for (i=(esz-1); i>(ce+1); --i) emdisplay[i]=emdisplay[i-1];
              emdisplay[++ce]=malloc(sizeof(struct Emdisplay));
              old_x=emdisplay[ce-1]->d->orix;
              old_y=emdisplay[ce-1]->d->oriy;
              old_maxx=emdisplay[ce-1]->d->maxx;
              old_maxy=emdisplay[ce-1]->d->maxy;
              ds_reshape(old_x,old_y,old_maxx,old_maxy/2);
              emdisplay[ce]->d=ds_alloc(new,old_x,old_y+old_maxy/2,old_maxx,old_maxy-old_maxy/2);
              emdisplay[ce]->flags=emdisplay[ce-1]->flags;
              emdisplay[ce]->d->mode=emdisplay[ce-1]->d->mode;
              emdisplay[ce]->d->tabsize=emdisplay[ce-1]->d->tabsize;
              ds_current(emdisplay[ce]->d);
              ds_begin();
              unlink(tmp);
              bf_free(old);
              return;
            }
            else showmsg(_("[Window is too small to be split]"));
          }
          else showmsg(_("[The filtered region can not be loaded]"));
        }
        else showmsg(_("[The filtered region can not be loaded]"));
        bf_free(new);
      }
      else showmsg(_("[The filtered region can not be saved]"));
      unlink(tmp);
      bf_free(old);
    }
    else showmsg(_("[You can not filter this region]"));
  }
}
/*}}}*/
/* em_free       -- free emdisplay */ /*{{{*/
static void em_free(void)
{
  int i;

  if (ce==0)
  {
    if (esz>1)
    {
      ds_current(emdisplay[1]->d);
      ds_reshape(emdisplay[0]->d->orix,emdisplay[0]->d->oriy,emdisplay[0]->d->maxx,emdisplay[0]->d->maxy+emdisplay[1]->d->maxy);
    }
  }
  else
  {
    ds_current(emdisplay[ce-1]->d);
    ds_reshape(emdisplay[ce-1]->d->orix,emdisplay[ce-1]->d->oriy,emdisplay[ce-1]->d->maxx,emdisplay[ce-1]->d->maxy+emdisplay[ce]->d->maxy);
  }
  ds_free(emdisplay[ce]->d);
  free(emdisplay[ce]);
  for (i=ce; i<(esz-1); ++i) emdisplay[i]=emdisplay[i+1];
  --esz;
  if (esz) emdisplay=realloc(emdisplay,esz*sizeof(struct Emdisplay*));
  else free(emdisplay);
  if (ce>0) --ce;
}
/*}}}*/
/* em_reshape    -- reshape display */ /*{{{*/
static void em_reshape(void)
{
  MenuChoice menu[4];
  int i;

  menu[0].str=_("sS)hrink"); menu[0].c='-';
  menu[1].str=_("gG)grow");  menu[1].c='+';
  menu[2].str=_("qQ)uit");   menu[2].c='\0';
  menu[3].str=(char*)0;
  i=0;
  do
  {
    ds_refresh();
    switch (i=menuline(_("Reshape display:"),menu,i))
    {
      case 0: /* shrink */ /*{{{*/
      {
        if ((ce+1)<esz && emdisplay[ce]->d->maxy>2)
        {
          ds_reshape(emdisplay[ce]->d->orix,emdisplay[ce]->d->oriy,emdisplay[ce]->d->maxx,emdisplay[ce]->d->maxy-1);
          ds_current(emdisplay[ce+1]->d);
          ds_reshape(emdisplay[ce+1]->d->orix,emdisplay[ce+1]->d->oriy-1,emdisplay[ce+1]->d->maxx,emdisplay[ce+1]->d->maxy+1);
          ds_current(emdisplay[ce]->d);
        }
        else if ((ce+1)==esz && emdisplay[ce]->d->maxy>2)
        {
          ds_reshape(emdisplay[ce]->d->orix,emdisplay[ce]->d->oriy+1,emdisplay[ce]->d->maxx,emdisplay[ce]->d->maxy-1);
          ds_current(emdisplay[ce-1]->d);
          ds_reshape(emdisplay[ce-1]->d->orix,emdisplay[ce-1]->d->oriy,emdisplay[ce-1]->d->maxx,emdisplay[ce-1]->d->maxy+1);
          ds_current(emdisplay[ce]->d);
        }
        break;
      }
      /*}}}*/
      case 1: /* grow */ /*{{{*/
      {
        if ((ce+1)<esz && emdisplay[ce+1]->d->maxy>2)
        {
          ds_reshape(emdisplay[ce]->d->orix,emdisplay[ce]->d->oriy,emdisplay[ce]->d->maxx,emdisplay[ce]->d->maxy+1);
          ds_current(emdisplay[ce+1]->d);
          ds_reshape(emdisplay[ce+1]->d->orix,emdisplay[ce+1]->d->oriy+1,emdisplay[ce+1]->d->maxx,emdisplay[ce+1]->d->maxy-1);
          ds_current(emdisplay[ce]->d);
        }
        else
        if ((ce+1)==esz && emdisplay[ce-1]->d->maxy>2)
        {
          ds_current(emdisplay[ce-1]->d);
          ds_reshape(emdisplay[ce-1]->d->orix,emdisplay[ce-1]->d->oriy,emdisplay[ce-1]->d->maxx,emdisplay[ce-1]->d->maxy-1);
          ds_current(emdisplay[ce]->d);
          ds_reshape(emdisplay[ce]->d->orix,emdisplay[ce]->d->oriy-1,emdisplay[ce]->d->maxx,emdisplay[ce]->d->maxy+1);
        }
        break;
      }
      /*}}}*/
      case 2: break;
      case -1: break;
      default: assert(0);
    }
  } while (i!=2 && i!=-1);
}
/*}}}*/

int main(int argc, char *argv[]) /*{{{*/
{
  /* variables *//*{{{*/
  chtype ch;
  int o;
  int argument,argsign,clear_argument,line;
  char *options;
  struct Macro curmac;
  struct Buffer *b;
  /*}}}*/

  /* init buffer and parse options *//*{{{*/
  line=0;
#ifdef LC_MESSAGES
  setlocale(LC_MESSAGES,"");
#endif
  setlocale(LC_CTYPE,"");
#ifdef HAVE_GETTEXT
  bindtextdomain("fe",LOCALEDIR);
  textdomain("fe");
#endif
  while ((o=getopt(argc,argv,"+l:h?"))!=EOF) switch (o)
  {
    case 'l': line=strtol(optarg,(char**)0,10); break;
    default: fputs(_("Usage: fe [-l line] [file [file]]\n"),stderr); exit(1);
  }
  if (optind<argc && *argv[optind]=='+')
  {
    line=strtol(argv[optind],(char**)0,10);
    ++optind;
  }
  ds_init();
  em_new();
  mc_null(&curmac);
  if ((options=getenv("FE"))) while (*options) 
  {
    switch (*options)
    {
      case 'b': ds_addmode(SHOWBEGEND); break;
      case 'i': emdisplay[0]->flags|=AUTOINDENT; break;
      case 'o': ds_addmode(OVERWRITE); break;
      case 'p': ds_addmode(POSITION); break;
      case 'l': ds_addmode(LINENO); break;
      case 'R': 
#ifdef HAVE_TYPEAHEAD
                typeahead(-1);
#endif
                break;
      case 't': ds_addmode(TIME); break;
      case 's': ds_addmode(SIDESCROLL); break;
      case 'm': ds_addmode(MAGIC); break;
      case 'B': ds_addmode(BOLDMARKS); break;
    }
    ++options;
  }
  /* init history buffers *//*{{{*/
  for (o=0; o<HISTORY_LEN; ++o)
  {
    filehist[o]=(optind+o<argc ? mystrmalloc(argv[optind+o]) : (char*)0);
    gotohist[o]=(char*)0;
    searchhist[o]=(char*)0;
    cmdhist[o]=(char*)0;
  }
  /*}}}*/
  if (optind<argc)
  {
    bf_name(emdisplay[0]->d->buffer,argv[optind]);
    if (bf_load(emdisplay[0]->d->buffer,(const char*)0,argv[optind],-1,"{{{","}}}")==0) showmsg(_("[Loading failed: %s]"),strerror(errno));
    emdisplay[0]->d->buffer->mode&=~CHANGED;
    bf_begin(emdisplay[0]->d->buffer);
    if (line) ds_gotoline(line-1);
    ++optind;
    if (optind<argc && em_new())
    {
      ds_realloc(emdisplay[1]->d,bf_alloc());
      bf_name(emdisplay[1]->d->buffer,argv[optind]);
      if (bf_load(emdisplay[1]->d->buffer,(const char*)0,argv[optind],-1,"{{{","}}}")==0) showmsg(_("[Loading failed: %s]"),strerror(errno));
      emdisplay[1]->d->buffer->mode&=~CHANGED;
      bf_begin(emdisplay[1]->d->buffer);
    }
  }
  else showmsg(_("[Use M-x to get in the main menu]"));
  /*}}}*/
  /* init kill ring buffers *//*{{{*/
  for (o=0; o<KILL_LEN; ++o)
  {
    if ((killbf[o]=bf_alloc())==(struct Buffer*)0)
    {
      showmsg(_("[Out of memory]"));
      echo();
      noraw();
      endwin();
      write(1,"\n",1);
    }
    kill_len=0;
    kill_cur=-1;
  }
  /*}}}*/
  argument=0;
  argsign=0;
  clear_argument=0;
  do
  {
    /* do argument display and management *//*{{{*/
    if (clear_argument)
    {
      argument=0;
      argsign=0;
    }
    else if (argument) showmsg("M-%d",argument);
    else if (argsign) showmsg("M--");
    clear_argument=1;
    /*}}}*/
    ds_refresh();
    ch=mc_get();
    showmsg((const char*)0);
    /* process ch *//*{{{*/
    switch (ch)
    {
      case '\000':     /*{{{*/ /* C-@       -- set mark                           */
      {
        ds_mark();
        showmsg(_("[Mark is set]"));
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        break;
      }
      /*}}}*/
      case '\001':     /*{{{*/ /* C-a       -- beginning of line                  */
      case KEY_BEG:
      {
        ds_beginline();
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        break;
      }
      /*}}}*/
      case '\002':     /*{{{*/ /* C-b       -- backward character                 */
      case KEY_LEFT:
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_backward()==0)
          {
            showmsg(_("[You are at the beginning of the buffer]"));
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_forward()==0)
          {
            showmsg(_("[You are at end of the buffer]"));
            break;
          } while (++argument);
        }
        else if (ds_backward()==0) showmsg(_("[You are at the beginning of the buffer]"));
        break;
      }
      /*}}}*/
      case '\003':     /*{{{*/ /* C-c       -- close fold                         */
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_closefold()==0)
          {
            showmsg(_("[You are not on a fold]"));
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_openfold()==0)
          {
            showmsg(_("[You are not on a fold]"));
            break;
          } while (++argument);
        }
        else if (ds_closefold()==0) showmsg(_("[You are not on a fold]"));
        break;
      }
      /*}}}*/
      case '\004':     /*{{{*/ /* C-d       -- delete character under cursor      */
      case '\177':
      case KEY_DC:
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_deletech(0)==0)
          {
            showmsg(_("[You can not delete here]"));
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_deletech(1)==0)
          {
            showmsg(_("[You can not delete here]"));
            break;
          } while (++argument);
        }
        else if (ds_deletech(0)==0) showmsg(_("[You can not delete here]"));
        break;
      }
      /*}}}*/
      case '\005':     /*{{{*/ /* C-e       -- end of line                        */
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        ds_endline();
        break;
      }
      /*}}}*/
      case '\006':     /*{{{*/ /* C-f       -- forward character                  */
      case KEY_RIGHT:
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_forward()==0)
          {
            showmsg(_("[You are at end of the buffer]"));
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_backward()==0)
          {
            showmsg(_("[You are at the beginning of the buffer]"));
            break;
          } while (++argument);
        }
        else if (ds_forward()==0) showmsg(_("[You are at end of the buffer]"));
        break;
      }
      /*}}}*/
      case '\007':     /*{{{*/ /* C-g       -- abort character                    */
      {
        showmsg(_("[aborted]"));
        argument=0;
        break;
      }
      /*}}}*/
      case '\010':     /*{{{*/ /* C-h       -- delete previous character          */
      case KEY_BACKSPACE:
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_deletech(1)==0)
          {
            showmsg(_("[You can not delete here]"));
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_deletech(0)==0)
          {
            showmsg(_("[You can not delete here]"));
            break;
          } while (++argument);
        }
        else if (ds_deletech(1)==0) showmsg(_("[You can not delete here]"));
        break;
      }
      /*}}}*/
      case '\013':     /*{{{*/ /* C-k       -- kill to end of line                */
      {
        emdisplay[ce]->flags&=~YANKMARK;
        if ((b=ds_moveline()))
        {
          if (em_kill(b,emdisplay[ce]->flags&APPENDKILL)==0)
          {
            ds_insertbuf(b,1);
            showmsg(_("[Out of memory]"));
          }
          else emdisplay[ce]->flags|=APPENDKILL;
        }
        break;
      }
      /*}}}*/
      case '\014':     /*{{{*/ /* C-l       -- center display                     */
      {
        ds_torow(emdisplay[ce]->d,(emdisplay[ce]->d->maxy-1)/2);
        break;
      }
      /*}}}*/
      case '\016':     /*{{{*/ /* C-n       -- next line                          */
      case KEY_DOWN:
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_nextline()==0)
          {
            showmsg(_("[You are in the last line]"));
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_prevline()==0)
          {
            showmsg(_("[You are in the first line]"));
            break;
          } while (++argument);
        }
        else if (ds_nextline()==0) showmsg(_("[You are in the last line]"));
        break;
      }
      /*}}}*/
      case '\017':     /*{{{*/ /* C-o       -- open fold                          */
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_openfold()==0)
          {
            showmsg(_("[You are not on a fold]"));
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_closefold()==0)
          {
            showmsg(_("[You are not in a fold]"));
            break;
          } while (++argument);
        }
        else if (ds_openfold()==0) showmsg(_("[You are not on a fold]"));
        break;
      }
      /*}}}*/
      case '\020':     /*{{{*/ /* C-p       -- previous line                      */
      case KEY_UP:
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_prevline()==0)
          {
            showmsg(_("[You are in the first line]"));
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_nextline()==0)
          {
            showmsg(_("[You are in the last line]"));
            break;
          } while (++argument);
        }
        else if (ds_prevline()==0) showmsg(_("[You are in the first line]"));
        break;
      }
      /*}}}*/
      case '\021':     /*{{{*/ /* C-q       -- quote character                    */
      {
        chtype ch2;

        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        showmsg(_("Quote character?"));
        ch=mc_get();
        if (ch!=(ch&0xff))
        {
          showmsg(_("[Only single characters can be inserted]"));
          break;
        }
        if (ds_compose(ch,0))
        {
          ch2=mc_get();
          if (ch2!=(ch2&0xff))
          {
            showmsg(_("[Only single characters can be inserted]"));
            break;
          }
          if ((ch=ds_compose(ch,ch2))==0)
          {
            showmsg(_("[No character can be composed from this]"));
            break;
          }
        }
        if (argument>0)
        {
          do if (ds_insertch((char)ch)==0) break; while (--argument);
        }
        else if (ds_insertch((char)ch)) showmsg((const char*)0);
        break;
      }
      /*}}}*/
      case '\022':     /*{{{*/ /* C-r       -- reverse incremental search         */
      {
        em_itssearch(emdisplay[ce],0,searchhist);
        break;
      }
      /*}}}*/
      case '\023':     /*{{{*/ /* C-s       -- incremental search                 */
      {
        em_itssearch(emdisplay[ce],1,searchhist);
        break;
      }
      /*}}}*/
      case '\024':     /*{{{*/ /* C-t       -- transpose character                */
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        ds_transpose();
        break;
      }
      /*}}}*/
      case '\025':     /*{{{*/ /* C-u       -- unfold                             */
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_unfold()==0)
          {
            showmsg(_("[You are not on a fold]"));
            break;
          } while (--argument);
        }
        else if (ds_unfold()==0) showmsg(_("[You are not on a fold]"));
        break;
      }
      /*}}}*/
      case '\026':     /*{{{*/ /* C-v       -- page down                          */
      case KEY_NPAGE:
      {
        em_page(argument,1);
        break;
      }
      /*}}}*/
      case KEY_PPAGE:  /*{{{*/ /* KEY_PPAGE -- page up                            */
      {
        em_page(argument,0);
        break;
      }
      /*}}}*/
      case '\027':     /*{{{*/ /* C-w       -- wipe out region                    */
      {
        struct Buffer *b;
        
        emdisplay[ce]->flags&=~YANKMARK;
        b=ds_covebuf(emdisplay[ce]->d,1,1);
        if (b==(struct Buffer*)0) showmsg(_("[You can not wipe out this region]"));
        else if (em_kill(b,emdisplay[ce]->flags&APPENDKILL)==0)
        {
          ds_insertbuf(b,1);
          showmsg(_("[Out of memory]"));
        }
        emdisplay[ce]->flags|=APPENDKILL;
        break;
      }
      /*}}}*/
      case '\030':     /*{{{*/ /* C-x       -- C-x prefix                         */
      {
        emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
        showmsg("C-x ");
        ds_refresh();
        ch=mc_get();
        showmsg((const char*)0);
        switch (ch)
        {
          case '\002': /*{{{*/ /* C-x C-b -- name buffer                          */
          {
            if (em_name(emdisplay[ce],filehist)==0) showmsg(_("[Naming file failed (%s)]"),strerror(errno));
            break;
          }
          /*}}}*/
          case '\003': /*{{{*/ /* C-x C-c -- exit                                 */
          {
            int i;

            for (i=0; i<MAXDISPLAYS; ++i) if (buffers[i] && buffers[i]->mode&CHANGED)
            {
              if (ok(_("There are unsaved changes, exit anyway:"),0)==1) goto exit;
              else break;
            }
            if (i==MAXDISPLAYS) goto exit;
            break;
          }
          /*}}}*/
          case '\006': /*{{{*/ /* C-x C-f -- filter region                        */
          em_filter(emdisplay[ce]); break;
          /*}}}*/
          case '|':    /*{{{*/ /* C-x |   -- pipe region                          */
          em_pipe(); break;
          /*}}}*/
          case '.':    /*{{{*/ /* C-x .   -- fold region                          */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (ds_fold()==0) showmsg(_("[You can not fold this region]"));
            break;
          }
          /*}}}*/
          case '\011': /*{{{*/ /* C-x C-i -- insert file                          */
          em_insert(emdisplay[ce]); break;
          /*}}}*/
          case '\015': /*{{{*/ /* C-x C-m -- delete mode                          */
          {
            showmsg("C-x C-m ");
            ds_refresh();
            ch=mc_get();
            showmsg((const char*)0);
            switch (ch)
            {
              /* b         -- begin/end marks *//*{{{*/
              case 'b': ds_delmode(SHOWBEGEND); break;
              /*}}}*/
              /* i         -- auto indent *//*{{{*/
              case 'i':
              {
                emdisplay[ce]->flags&=~AUTOINDENT;
                break;
              }
              /*}}}*/
              /* m         -- magic *//*{{{*/
              case 'm': ds_delmode(MAGIC); break;
              /*}}}*/
              /* o         -- overwrite *//*{{{*/
              case 'o': ds_delmode(OVERWRITE); break;
              /*}}}*/
              /* p         -- position *//*{{{*/
              case 'p': ds_delmode(POSITION); break;
              /*}}}*/
              /* r         -- read-only */ /*{{{*/
              case 'r':
              {
                switch (ok(_("Check-out?"),1))
                {
                  case 1:ds_checkout(); break;
                  case 0: emdisplay[ce]->d->buffer->mode&=~READONLY; break;
                }
                break;
              }
              /*}}}*/
              /* R         -- redraw *//*{{{*/
              case 'R': 
#ifdef TYPE_AHEAD
                        typeahead(0);
#endif
                        showmsg(_("[Type-ahead switched on]")); break;
              /*}}}*/
              /* s         -- scroll sideways *//*{{{*/
              case 's': ds_delmode(SIDESCROLL); break;
              /*}}}*/
              /* l         -- display line numbers *//*{{{*/
              case 'l': ds_delmode(LINENO); break;
              /*}}}*/
              /* t         -- time *//*{{{*/
              case 't': ds_delmode(TIME); break;
              /*}}}*/
              /* B         -- bold marks *//*{{{*/
              case 'B': ds_delmode(BOLDMARKS); break;
              /*}}}*/
              /* default   -- key not bound *//*{{{*/
              default: showmsg(_("[key '%d'='%o' not bound]"),ch,ch);
              /*}}}*/
            }
            break;
          }
          /*}}}*/
          case '\022': /*{{{*/ /* C-x C-r -- load buffer from file                */
          {
            em_load();
            break;
          }
          /*}}}*/
          case '\023': /*{{{*/ /* C-x C-s -- save buffer to file                  */
          {
            (void)em_save(emdisplay[ce]);
            break;
          }
          /*}}}*/
          case '\027': /*{{{*/ /* C-x C-w -- save buffer to new file              */
          {
            em_saveas(emdisplay[ce]);
            break;
          }
          /*}}}*/
          case '\030': /*{{{*/ /* C-x C-x -- swap mark and cursor                 */
          {
            ds_swapmark();
            break;
          }
          /*}}}*/
          case '0':    /*{{{*/ /* C-x 0   -- remove display                       */
          {
            if (esz>1 && (emdisplay[ce]->d->buffer->refcnt>1 || !(emdisplay[ce]->d->buffer->mode&CHANGED) || ok(_("Buffer contains unsaved changes, delete it anyway?"),0)==1))
            {
              em_free();
            }
            break;
          }
          /*}}}*/
          case '2':    /*{{{*/ /* C-x 2   -- split display                        */
          {
            if (em_new()==0) showmsg(_("[Window is too small to be split]"));
            break;
          }
          /*}}}*/
          case '=':    /*{{{*/ /* C-x =   -- describe line                        */
          {
            ds_describe();
            break;
          }
          /*}}}*/
          case ':':    /*{{{*/ /* C-x :   -- set variable                         */
          {
            showmsg("C-x : ");
            ds_refresh();
            ch=mc_get();
            showmsg((const char*)0);
            switch (ch)
            {
              /* d         -- set display size */ /*{{{*/
              case 'd':
              {
                em_reshape();
                break;
              }
              /*}}}*/
              /* t         -- set tab size */ /*{{{*/
              case 't':
              {
                char buf[8],*end;
                size_t x,offx;
                int tabsize;

                sprintf(buf,"%d",ds_tabsize(-1));
                x=offx=0;
                while (editline(buf,sizeof(buf),_("Tab width?"),&x,&offx,(char**)0)!=0)
                {
                  tabsize=strtol(buf,&end,0);
                  if (end!=buf && *end=='\0' && tabsize>=0 && tabsize<=MAXTAB)
                  {
                    ds_tabsize(tabsize);
                    break;
                  }
                }
                break;
              }
              /*}}}*/
              /* l         -- set language */ /*{{{*/
              case 'l': ds_setlang(); break;
              /*}}}*/
              /* #         -- set counter and increment */ /*{{{*/
              case '#':
              {
                char buf[8],*end;
                size_t x,offx;
                int c,i;

                sprintf(buf,"%d",counter);
                x=offx=0;
                while (editline(buf,sizeof(buf),_("Counter?"),&x,&offx,(char**)0)!=0)
                {
                  c=strtol(buf,&end,0);
                  if (end!=buf && *end=='\0')
                  {
                    sprintf(buf,"%d",increment);
                    x=offx=0;
                    while (editline(buf,sizeof(buf),_("Increment?"),&x,&offx,(char**)0)!=0)
                    {
                      i=strtol(buf,&end,0);
                      if (end!=buf && *end=='\0')
                      {
                        counter=c;
                        increment=i;
                        break;
                      }
                    }
                    break;
                  }
                }
                break;
              }
              /*}}}*/
              /* default   -- key not bound */ /*{{{*/
              default: showmsg(_("[key '%d'='%o' not bound]"),ch,ch);
              /*}}}*/
            }
            break;
          }
          /*}}}*/
          case '(':    /*{{{*/ /* C-x (   -- start recording macro                */
          {
            if (mc_defstart()) showmsg(_("[You are recording a macro now]"));
            else showmsg(_("[You are already recording a macro]"));
            break;
          }
          /*}}}*/
          case ')':    /*{{{*/ /* C-x )   -- end recording macro                  */
          {
            if (mc_defend(&curmac,2)==0) showmsg(_("[You are not recording a macro]"));
            break;
          }
          /*}}}*/
          case '!':    /*{{{*/ /* C-x !   -- start shell command                  */
          {
            const char *msg;
            char cmd[255];
            size_t x,offx;
    
            x=offx=0;
            cmd[0]='\0';
            if (editline(cmd,sizeof(cmd),_("Command?"),&x,&offx,cmdhist)==0) break;
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if ((msg=ds_shell((const char*)0,cmd))) showmsg(_("[Sub-shell: %s]"),msg);
            break;
          }
          /*}}}*/
          case 'c':    /*{{{*/ /* C-x c   -- start sub shell                      */
          {
            const char *msg;
    
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if ((msg=ds_shell(_("[Sub-shell started]"),(const char*)0))) showmsg(_("[Sub-shell: %s]"),msg);
            break;
          }
          /*}}}*/
          case 'e':    /*{{{*/ /* C-x e   -- execute macro                        */
          {
            if (mc_start(&curmac,argument>0 ? argument : 1)==-1) showmsg(_("[You are already recording a macro]"));
            break;
          }
          /*}}}*/
          case 'm':    /*{{{*/ /* C-x m   -- add mode                             */
          {
            showmsg("C-x m ");
            ds_refresh();
            ch=mc_get();
            showmsg((const char*)0);
            switch (ch)
            {
              /* b         -- begin/end marks *//*{{{*/
              case 'b': ds_addmode(SHOWBEGEND); break;
              /*}}}*/
              /* i         -- auto indent *//*{{{*/
              case 'i':
              {
                emdisplay[ce]->flags|=AUTOINDENT;
                break;
              }
              /*}}}*/
              /* m         -- magic *//*{{{*/
              case 'm': ds_addmode(MAGIC); break;
              /*}}}*/
              /* o         -- overwrite *//*{{{*/
              case 'o': ds_addmode(OVERWRITE); break;
              /*}}}*/
              /* p         -- position *//*{{{*/
              case 'p': ds_addmode(POSITION); break;
              /*}}}*/
              /* r         -- read-only */ /*{{{*/
              case 'r':
              {
                switch (ok(_("Check-in?"),1))
                {
                  case 1: ds_checkin(); break;
                  case 0: emdisplay[ce]->d->buffer->mode|=READONLY; break;
                }
                break;
              }
              /*}}}*/
              /* R         -- redraw *//*{{{*/
              case 'R':
#ifdef HAVE_TYPEAHEAD
                              typeahead(-1);
#endif
                              showmsg(_("[Type-ahead switched off]")); break;
              /*}}}*/
              /* s         -- scroll sideways *//*{{{*/
              case 's': ds_addmode(SIDESCROLL); break;
              /*}}}*/
              /* l         -- display line numbers *//*{{{*/
              case 'l': ds_addmode(LINENO); break;
              /*}}}*/
              /* t         -- time *//*{{{*/
              case 't': ds_addmode(TIME); break;
              /*}}}*/
              /* B         -- bold marks *//*{{{*/
              case 'B': ds_addmode(BOLDMARKS); break;
              /*}}}*/
              /* default   -- key not bound *//*{{{*/
              default: showmsg(_("[key '%d'='%o' not bound]"),ch,ch);
              /*}}}*/
            }
            break;
          }
          /*}}}*/
          case 'o':    /*{{{*/ /* C-x o   -- other window                         */
          {
            if (++ce==esz) ce=0;
            ds_current(emdisplay[ce]->d);
            break;
          }
          /*}}}*/
          case '#':    /*{{{*/ /* C-x #   -- insert counter                       */
          {
            char buf[10],*s;

            sprintf(s=buf,"%d",counter);
            while (*s)
            {
              if (ds_insertch(*s)==0) break;
              ++s;
            }
            counter+=increment;
            break;
          }
          /*}}}*/
          default:     /*{{{*/ /* default -- key not bound                        */
          showmsg(_("[key '%d'='%o' not bound]"),ch,ch);
          /*}}}*/
        }
        break;
      }
      /*}}}*/
      case '\031':     /*{{{*/ /* C-y       -- yank current kill ring buffer back */
      {
        if (kill_len)
        {
          if (ds_insertbuf(killbf[kill_cur],1)) emdisplay[ce]->flags|=(YANKMARK|APPENDKILL);
        }
        else showmsg(_("[The kill ring is empty]"));
        break;
      }
      /*}}}*/
      case '\032':     /*{{{*/ /* C-z       -- suspend                            */
      case KEY_SUSPEND:
      ds_suspend(_("[Suspended]")); break;
      /*}}}*/
      case '\033':     /*{{{*/ /* M-        -- M- prefix                          */
      {
        showmsg("M-");
        ds_refresh();
        ch=mc_get();
        showmsg((const char*)0);
        switch (ch)
        {
          case '\000': /*{{{*/ /* M-C-@     -- fold region                        */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (ds_fold()==0) showmsg(_("[You can not fold this region]"));
            break;
          }
          /*}}}*/
          case '\004': /*{{{*/ /* M-C-d     -- delete word                        */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_forword(DELETE)==0)
              {
                showmsg(_("[You are at end of the buffer]"));
                break;
              } while (--argument);
            }
            else if (argument<0)
            {
              do if (ds_backword(DELETE)==0)
              {
                showmsg(_("[You are at the beginning of the buffer]"));
                break;
              } while (++argument);
            }
            else if (ds_forword(DELETE)==0) showmsg(_("[You are at end of the buffer]"));
            break;
          }
          /*}}}*/
          case '\010': /*{{{*/ /* M-C-h     -- delete word backward               */
          case KEY_BACKSPACE:
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument<0)
            {
              do if (ds_forword(DELETE)==0)
              {
                showmsg(_("[You are at end of the buffer]"));
                break;
              } while (++argument);
            }
            else if (argument>0)
            {
              do if (ds_backword(DELETE)==0)
              {
                showmsg(_("[You are at the beginning of the buffer]"));
                break;
              } while (--argument);
            }
            else if (ds_backword(DELETE)==0) showmsg(_("[You are at the beginning of the buffer]"));
            break;
          }
          /*}}}*/
          case '\014': /*{{{*/ /* M-C-l     -- redraw screen                      */
          {
            ds_redraw();
            break;
          }
          /*}}}*/
          case '\016': /*{{{*/ /* M-C-n     -- go to end of fold                  */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (ds_endfold()==0) showmsg(_("[You are not in a fold]"));
            break;
          }
          /*}}}*/
          case '\020': /*{{{*/ /* M-C-p     -- go to beginning of fold            */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (ds_beginfold()==0) showmsg(_("[You are not in a fold]"));
            break;
          }
          /*}}}*/
          case '\022': /*{{{*/ /* M-C-r     -- search backward                    */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            em_search(emdisplay[ce],0);
            break;
          }
          /*}}}*/
          case '\023': /*{{{*/ /* M-C-s     -- search forward                     */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            em_search(emdisplay[ce],1);
            break;
          }
          /*}}}*/
          case '.':    /*{{{*/ /* M-.       -- set mark                           */
          {
            ds_mark();
            showmsg(_("[Mark is set]"));
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            break;
          }
          /*}}}*/
          case '#':    /*{{{*/ /* M-#       -- go to matching fence               */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            ds_gotofence();
            break;
          }
          /*}}}*/
          case '<':    /*{{{*/ /* M-<       -- beginning of buffer                */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            ds_begin();
            break;
          }
          /*}}}*/
          case '>':    /*{{{*/ /* M->       -- end of buffer                      */
          { 
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            ds_end();
            break;
          }
          /*}}}*/
          case '-':    /*{{{*/ /* M--       -- set argument                       */
          {
            argsign=1;
            clear_argument=0;
            break;
          }
          /*}}}*/
          /* digit */  /*{{{*/ /* M-digit   -- set argument                       */
          case '0':
          case '1':
          case '2':
          case '3':
          case '4':
          case '5':
          case '6':
          case '7':
          case '8':
          case '9': argument=(argsign ? argument*10-(ch-'0') : argument*10+(ch-'0')); clear_argument=0; break;
          /*}}}*/
          case 'b':    /*{{{*/ /* M-b       -- backward word                      */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_backword(NOTHING)==0)
              {
                showmsg(_("[You are at the beginning of the buffer]"));
                break;
              } while (--argument);
            }
            else if (argument<0)
            {
              do if (ds_forword(NOTHING)==0)
              {
                showmsg(_("[You are at the beginning of the buffer]"));
                break;
              } while (++argument);
            }
            else if (ds_backword(NOTHING)==0) showmsg(_("[You are at the beginning of the buffer]"));
            break;
          }
          /*}}}*/
          case 'c':    /*{{{*/ /* M-c       -- capitalise word                    */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_forword(CAPITALISE)==0)
              {
                showmsg(_("[You are at end of the buffer]"));
                break;
              } while (--argument);
            }
            else if (ds_forword(CAPITALISE)==0) showmsg(_("[You are at end of the buffer]"));
            break;
          }
          /*}}}*/
          case 'f':    /*{{{*/ /* M-f       -- forward word                       */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_forword(NOTHING)==0)
              {
                showmsg(_("[You are at end of the buffer]"));
                break;
              } while (--argument);
            }
            else if (argument<0)
            {
              do if (ds_backword(NOTHING)==0)
              {
                showmsg(_("[You are at end of the buffer]"));
                break;
              } while (++argument);
            }
            else if (ds_forword(NOTHING)==0) showmsg(_("[You are at end of the buffer]"));
            break;
          }
          /*}}}*/
          case 'g':    /*{{{*/ /* M-g       -- go to line                         */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            em_gotoline(emdisplay[ce],argument);
            break;
          }
          /*}}}*/
          case 'l':    /*{{{*/ /* M-l       -- lower case word                    */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_forword(TOLOWER)==0)
              {
                showmsg(_("[You are at end of the buffer]"));
                break;
              } while (--argument);
            }
            else if (ds_forword(TOLOWER)==0) showmsg(_("[You are at end of the buffer]"));
            break;
          }
          /*}}}*/
          case 'R':    /*{{{*/ /* M-R       -- replace                            */
          {
            em_replace(emdisplay[ce],argument,0);
            break;
          }
          /*}}}*/
          case 'r':    /*{{{*/ /* M-r       -- query replace                      */
          {
            em_replace(emdisplay[ce],argument,1);
            break;
          }
          /*}}}*/
          case 'u':    /*{{{*/ /* M-u       -- upper case word                    */
          {
            emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_forword(TOUPPER)==0)
              {
                showmsg(_("[You are at end of the buffer]"));
                break;
              } while (--argument);
            }
            else if (ds_forword(TOUPPER)==0) showmsg(_("[You are at end of the buffer]"));
            break;
          }
          /*}}}*/
          case 'v':    /*{{{*/ /* M-v       -- previous page                      */
          case KEY_PPAGE:
          {
            em_page(argument,0);
            break;
          }
          /*}}}*/
          case 'w':    /*{{{*/ /* M-w       -- copy region to kill ring element   */
          {
            struct Buffer *b;
            
            emdisplay[ce]->flags&=~YANKMARK;
            b=ds_covebuf(emdisplay[ce]->d,0,1);
            if (b==(struct Buffer*)0) showmsg(_("[You can not copy this region]"));
            else if (em_kill(b,emdisplay[ce]->flags&APPENDKILL)==0)
            {
              ds_insertbuf(b,1);
              showmsg(_("[Out of memory]"));
            }
            emdisplay[ce]->flags|=APPENDKILL;
            break;
          }
          /*}}}*/
          case 'x':    /*{{{*/ /* M-x       -- main menu                          */
          {
            MenuChoice menu[6];
            
            menu[0].str=_("lL)oad"); menu[0].c='\0';
            menu[1].str=_("sS)ave"); menu[1].c='\0';
            menu[2].str=_("gG)oto"); menu[2].c='\0';
            menu[3].str=_("mM)ode"); menu[3].c='\0';
            menu[4].str=_("qQ)uit"); menu[4].c='\0';
            menu[5].str=(char*)0;
            switch (menuline(_("Main menu:"),menu,0))
            {
              /*      -1 -- abort menu *//*{{{*/
              case -1: break;
              /*}}}*/
              /*       0 -- load *//*{{{*/
              case 0:
              {
                em_load();
                break;
              }
              /*}}}*/
              /*       1 -- save *//*{{{*/
              case 1: (void)em_save(emdisplay[ce]); break;
              /*}}}*/
              /*       2 -- move menu *//*{{{*/
              case 2:
              {
                MenuChoice movemenu[5];

                movemenu[0].str=_("bB)egin");  movemenu[0].c='\0';
                movemenu[1].str=_("eE)nd");    movemenu[1].c='\0';
                movemenu[2].str=_("lL)ine");   movemenu[2].c='\0';
                movemenu[3].str=_("sS)tring"); movemenu[3].c='\0';
                movemenu[4].str=(char*)0;
                switch (menuline(_("Go to:"),movemenu,0))
                {
                  case -1: break;
                  case 0: emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL); ds_begin(); break;
                  case 1: emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL); ds_end(); break;
                  case 2: em_gotoline(emdisplay[ce],0); break;
                  case 3: em_search(emdisplay[ce],1); break;
                  default: assert(0);
                }
                break;
              }
              /*}}}*/
              /*       3 -- mode *//*{{{*/
              case 3:
              {
                MenuChoice modemenu[5];
                int add;
                
                modemenu[0].str=_("dD)elete"); modemenu[0].c='\0';
                modemenu[1].str=_("aA)dd");    modemenu[1].c='\0';
                modemenu[2].str=(char*)0;
                add=menuline(_("Mode:"),modemenu,0);
                if (add==-1) break;
                modemenu[0].str=_("aA)uto-indent"); modemenu[0].c='\0';
                modemenu[1].str=_("sS)croll");      modemenu[1].c='\0';
                modemenu[2].str=_("tT)ime");        modemenu[2].c='\0';
                modemenu[3].str=_("pP)osition");    modemenu[3].c='\0';
                modemenu[4].str=(char*)0;
                switch (menuline(add ? _("Add mode:") : _("Delete mode:"),modemenu,0))
                {
                  case -1: break;
                  case 0: emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL); if (add) emdisplay[ce]->flags|=AUTOINDENT; else emdisplay[ce]->flags&=~AUTOINDENT; break;
                  case 1: emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL); (add ? ds_addmode : ds_delmode)(SIDESCROLL); break;
                  case 2: emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL); (add ? ds_addmode : ds_delmode)(TIME); break;
                  case 3: emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL); (add ? ds_addmode : ds_delmode)(POSITION); break;
                  default: assert(0);
                }
                break;
              }
              /*}}}*/
              /*       4 -- quit *//*{{{*/
              case 4: if (!(emdisplay[ce]->d->buffer->mode&CHANGED) || ok(_("There are unsaved changes, exit anyway:"),0)==1) goto exit; break;
              /*}}}*/
              /* default -- should not happen *//*{{{*/
              default: assert(0);
              /*}}}*/
            }
            break;
          }
          /*}}}*/
          case 'y':    /*{{{*/ /* M-y       -- move backward in kill ring         */
          {
            emdisplay[ce]->flags&=~(APPENDKILL); 
            if (kill_len)
            {
              if (emdisplay[ce]->flags&YANKMARK)
              {
                b=ds_covebuf(emdisplay[ce]->d,1,1);
                if (argument>=0) do
                {
                  if (kill_cur==0) kill_cur=kill_len-1;
                  else --kill_cur;
                  if (argument==0) argument=1;
                } while (--argument);
                else if (argument<0) do
                {
                  if (kill_cur==(kill_len-1)) kill_cur=0;
                  else ++kill_cur;
                } while (++argument>0);
                if (ds_insertbuf(killbf[kill_cur],1)==0)
                {
                  ds_insertbuf(b,1);
                  showmsg(_("[You can not yank the current kill ring element here]"));
                }
                bf_free(b);
              }
              else showmsg(_("[The mark has not been set by yanking back text]"));
            }
            else showmsg(_("[The kill ring is empty]"));
            break;
          }
          /*}}}*/
          case 'z':    /*{{{*/ /* M-z       -- save and exit                      */
          {
            int i;

            for (i=0; i<esz; ++i) if (emdisplay[i]->d->buffer->mode&CHANGED)
            {
              if (em_save(emdisplay[i])) ds_refresh();
              else break;
            }
            if (i==esz) goto exit;
            break;
          }
          /*}}}*/
          default:     /*{{{*/ /* default   -- key not bound                      */
          showmsg(_("[key '%d'='%o' not bound]"),ch,ch);
          /*}}}*/
        }
        break;
      }
      /*}}}*/
      case KEY_IC:     /*{{{*/ /* KEY_IC    -- toggle insert mode                 */
      {
        if (emdisplay[ce]->d->mode&OVERWRITE) ds_delmode(OVERWRITE);
        else ds_addmode(OVERWRITE);
        break;
      }
      /*}}}*/
#ifdef KEY_RESIZE
      case KEY_RESIZE: /*{{{*/ /* KEY_RESIZE-- window resized                     */
      {
        ds_winch();
        break;
      }
      /*}}}*/
#endif
      default:         /*{{{*/ /* default   -- insert all text characters         */
      {
        if (ch==KEY_ENTER) ch='\r';
        if ((argument || argsign) && isdigit(ch))
        {
          argument=(argsign ? argument*10-(ch-'0') : argument*10+(ch-'0'));
          clear_argument=0;
        }
        else if (ch==(ch&0xff))
        {
          struct Buffer *b;

          if (!(emdisplay[ce]->d->mode&OVERWRITE) && ((emdisplay[ce]->flags&AUTOINDENT && ch=='\15') || (!(emdisplay[ce]->flags&AUTOINDENT) && ch=='\n')))
          {
            if ((b=ds_indent())==(struct Buffer*)0)
            {
              showmsg(_("[Out of memory]"));
              break;
            }
          }
          else b=(struct Buffer*)0;
          if (ch=='\015') ch='\n';
          emdisplay[ce]->flags&=~(YANKMARK|APPENDKILL); 
          if (argument<=0) argument=1;
          while (argument--)
          {
            if (ds_insertch((char)ch)==0) break;
            else if (b && ds_insertbuf(b,0)==0) break;
          }
          if (b) bf_free(b);
        }
        else showmsg(_("[key '%d'='%o' not bound]"),ch,ch);
      }
      /*}}}*/
    }
    /*}}}*/
  } while (1);
  exit:
  /* land nicely *//*{{{*/
  ds_exit(1);
  while (esz) em_free();
  for (o=0; o<HISTORY_LEN; ++o)
  {
    if (filehist[o]) free(filehist[o]);
    if (gotohist[o]) free(gotohist[o]);
    if (searchhist[o]) free(searchhist[o]);
    if (cmdhist[o]) free(cmdhist[o]);
  }
  for (o=0; o<KILL_LEN; ++o) bf_free(killbf[o]);
  /*}}}*/
  return 0;
}
/*}}}*/
