#include "prjlibs-c/standards.h"
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/vt.h>

#include <skalibs/stddjb.h>
#include "prjlibs-c/constants.h"
#include "prjlibs-c/diewarn.h"
#include "prjlibs-c/types.h"
#include "fdtools.h"

char const* PROG="vc-lock";

static int fd;

static void release_vt(int sig) {
  (void)sig;
  ioctl(fd, VT_RELDISP, 0);
}

static void acquire_vt(int sig) {
  (void)sig;
  ioctl(fd, VT_RELDISP, VT_ACKACQ);
}

#define RETRY(func, args) \
  do { if ((func args)==0) break; if (errno!=EINTR) DIE0(func); } while (1)

int main(int argc, char** argv) {
  struct vt_mode vtm, old_vtm;
  struct termios ti, old_ti;
  sigset_t set, old_set;
  type_bool scan;
  tain now, wake;

  if (argc<3) DIE_USAGE(" {fd|/dev/ttyN} program [arg ...]");

  fd=fd_scan_or_open(&scan, argv[1], O_RDWR|O_NOCTTY, 0);
  if (fd<0) {
    if (scan) DIE_MALFORMED_FD(argv[1]);
    else DIE1(open, argv[1]);
  }

  if (ioctl(fd, VT_GETMODE, &old_vtm)<0) DIE0(vt_getm);

  RETRY(tcgetattr, (fd, &old_ti));

  sig_catch(SIGUSR1, release_vt);
  sig_catch(SIGUSR2, acquire_vt);
  sigemptyset(&set);
  sigaddset(&set, SIGUSR1);
  sigaddset(&set, SIGUSR2);
  if (sigprocmask(SIG_UNBLOCK, &set, &old_set)!=0) DIE0(sigprocmask);

  ti=old_ti;
  ti.c_iflag&=~BRKINT;
  ti.c_iflag|= IGNBRK;
  ti.c_lflag&=~ISIG;
  RETRY(tcsetattr, (fd, TCSADRAIN, &ti));
  RETRY(tcdrain, (fd));

  vtm=old_vtm;
  vtm.mode=VT_PROCESS;
  vtm.relsig=SIGUSR1;
  vtm.acqsig=SIGUSR2;
  if (ioctl(fd, VT_SETMODE, &vtm)<0) DIE0(vt_setm);

  argv+=2;
  tain_now(&now);
  tain_addsec(&wake, &now, 0);
  for (;;) {
    pid_t const pid=fork();
    if (pid<0) {
      WARN0(fork);
    } else if (pid==0) {
      sigprocmask(SIG_SETMASK, &old_set, NULLP);
      mexec((char const**)argv);
      DIE1(exec, *argv);
    } else {
      int status;
      if (waitpid_nointr(pid, &status, 0)!=pid) WARN0(wait);
      else if (status==0) break;
    }
    tain_addsec(&wake, &wake, 1);
    deepsleepuntil(&wake, &now);
  }

  if (ioctl(fd, VT_SETMODE, &old_vtm)<0) DIE0(vt_setm);
  RETRY(tcsetattr, (fd, TCSADRAIN, &old_ti));
  RETRY(tcdrain, (fd));

  _exit(0);
}
