/* multitee.c: send multiple inputs to multiple outputs
Daniel J. Bernstein, brnstnd@nyu.edu.
No known patent problems.

This is a complete rewrite of multitee 2.0, as distributed in
comp.sources.unix volume 22.

XXX: should use optprogname
*/

#include "prjlibs-c/standards.h"
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>

#include <skalibs/stddjb.h>
#include "prjlibs-c/diewarn.h"
#include "prjlibs-c/intattr.h"
#include "sigsched.h"
#include "sod.h"

char const* PROG="multitee";

#define usage() DIE_USAGE(" [ -bsize ] [ -vQq ] [ fd-fd,fd,fd...  ] ...")

/*
//#ifndef EWOULDBLOCK
//#define EWOULDBLOCK 0
//#endif
//#ifndef EPIPE
//#define EPIPE 0
//#endif
*/

static unsigned int bufsize = 8192;

/*
We take pains to avoid all I/O copies. Each live buffer has an ending
position; the buffer is read into as long as the ending position isn't
the actual end of the buffer. Each live buffer also has an output
schedule count, and each output descriptor for the buffer has a starting
position. As soon as data is read in, it's scheduled to be written out
on each of the output descriptors which isn't already scheduled. The
schedule count is checked at this time and whenever it is decremented;
if it is 0, the buffer is emptied, and all starting positions are set to
0. Upon an output scheduling, the buffer is written from the starting
position to the ending position. (If this produces an error, die, except
in cases like EINTR.) The starting position is incremented by the write
count. If this brings it to the ending position, the schedule count is
decremented. Otherwise the write is rescheduled.
*/

int flagverbose;

#define s_descriptor "descriptor "
static void readerr(int fd)
{
 char buffer[sizeof s_descriptor+INTATTR_LEN(10, int)]=s_descriptor;
 if (!flagverbose)
   return;
 buffer[(sizeof s_descriptor)+int_fmt(buffer+sizeof s_descriptor, fd)]='\0';
 buffer[(sizeof s_descriptor)-1+int_fmt(buffer+(sizeof s_descriptor)-1, fd)]
   ='\0';
 WARN1(read, buffer);
}

static void writeerr(int fd)
{
 char buffer[sizeof s_descriptor+INTATTR_LEN(10, int)]=s_descriptor;
 if (!flagverbose)
   return;
 buffer[(sizeof s_descriptor)-1+int_fmt(buffer+(sizeof s_descriptor)-1, fd)]
   ='\0';
 WARN1(write, buffer);
}

struct iosc
 {
  int in;
  int out;
  int flagsched;
  unsigned int startpos;
  int breakpipe;
  int dead;
 }
;

SODdecl(intstack,int);
SODdecl(ioscstack,struct iosc);

ioscstack *fdio;
intstack *fdeof;
int *fdlive;
unsigned int *fdendpos;
int *fdschedcnt;
char **fdbuf;
int fdinmax;

static void nomorewrite(int fdout)
{
 int i;
 ioscstack iosc;
 for (i = 0;i <= fdinmax;++i)
   if (fdlive[i])
     for (iosc = fdio[i];iosc;iosc = SODnext(iosc))
       if (SODdata(iosc).out == fdout)
	 SODdata(iosc).dead = 1;
}

/* forward declarations of rescheduling routines */

static void donoread();
static void checksched();
static void doeof();
static void dowritesched();
static void dowriteunsched();

/* I/O routines */

static void readit(int fd)
{
 int r;
 int endpos;
 ioscstack iosc;

 endpos = fdendpos[fd];
 if (fdlive[fd])
  {
   r = read(fd,fdbuf[fd] + endpos,bufsize - endpos);
   /* XXX: this'll be slow if endpos is not aligned */
   if (r == -1)
    {
     if (errno == EWOULDBLOCK)
       return;
       /* sample case where this can happen: someone reads before we do */
     readerr(fd);
     r = 0;
    }
  }
 else
   r = 0;
 if (r == 0)
  {
   donoread(fd);
   doeof(fd);
   return;
  }
 fdendpos[fd] = endpos + r;
 if (fdio[fd])
  {
   for (iosc = fdio[fd];iosc;iosc = SODnext(iosc))
     if (SODdata(iosc).flagsched == 0)
       /* ``scheduled to be written out on each of the output descriptors'' */
       dowritesched(&(SODdata(iosc)));
   /* it says ``schedule count is checked at this time...'' */
   /* but as an optimization, since fdio[fd], schedcnt must be positive */
  }
 else
   fdendpos[fd] = 0; /* throwing it away immediately */
 if (fdendpos[fd] >= bufsize)
   donoread(fd); /* ``buffer is read into as long as...'' */
}

static int cont = 0;

static void sigcont(int sig)
{
 (void)sig;
 cont = 1;
}

static void writeit(ss_idptr p)
{
 struct iosc *ioscp;
 int in;
 int w;

 ioscp = p;
 in = ioscp->in;

 if (ioscp->dead)
   w = fdendpos[in] - ioscp->startpos;
 else
   w = write(ioscp->out,fdbuf[in] + ioscp->startpos,fdendpos[in] - ioscp->startpos);
 /*``buffer is written from the starting position to the ending position...''*/
 if (w == -1)
  {
   if (errno == EWOULDBLOCK)
     w = write(ioscp->out,fdbuf[in],1); /* XXX: kludge alert! */
   if (w == -1)
    {
     if ((errno == EPIPE) && ioscp->breakpipe) do {
       int sig=SIGPIPE;
       sigset_t oldmask;
       sigset_t allmask;
       sigset_t mask;
       struct sigaction oldact;
       struct sigaction act;
       struct sigaction contact;
       sigfillset(&allmask);
       sigprocmask(SIG_BLOCK, &allmask, &oldmask);
       /*oldmask = sigblock(0);*/
       /*sigblock(~0);*/
       /* now we won't receive any signals */
       act.sa_handler = SIG_DFL;
       sigfillset(&act.sa_mask);
       act.sa_flags = 0;
       if (sigaction(sig,&act,&oldact) == -1)
         break;
       act.sa_handler = sigcont;
       sigfillset(&act.sa_mask);
       act.sa_flags = 0;
       if (sigaction(SIGCONT,&act,&contact) == -1)
         break;
       cont = 0;
       if (kill(getpid(),sig) == -1)
         break;
       /* now a sig should be queued, and we have control over sig and CONT */
       /* exception: SIGSTOP and SIGKILL can't be blocked, so those signals
          might already have been delivered. in the SIGSTOP case, if we've
          reached this point, sigcont() might already have been run. that's
          why cont must be set to 0 before the kill(). */
       /* after this next bit we may receive sig and/or CONT */
       sigfillset(&mask);
       sigdelset(&mask, sig);
       sigdelset(&mask, SIGCONT);
       sigprocmask(SIG_BLOCK, &mask, NULL);
       /*sigsetmask(~(sigmask(sig) | sigmask(SIGCONT)));*/
       /* in the near future, sig will in fact be received */
       while (!cont) /* dead loop until we receive CONT */
         (void)0; /* XXX: there should be a syscall so we don't have to loop here */
       sigprocmask(SIG_BLOCK, &allmask, NULL);
       /*sigblock(~0);*/
       /* now we won't receive any signals */
       sigaction(sig,&oldact,&act); /* we don't care if it fails */
       sigaction(SIGCONT,&contact,&act);
       /* now signal handlers are back to normal */
       sigprocmask(SIG_BLOCK, &oldmask, NULL);
       /*sigsetmask(oldmask);*/
     } while (0);
     writeerr(ioscp->out);
     nomorewrite(ioscp->out);
     w = 0;
    }
  }
 if (w == 0)
   (void)0; /* XXX: wtf? */
 ioscp->startpos += w; /* ``starting pos is incremented by write count'' */

 if (ioscp->startpos == fdendpos[in])
   /* ``if this brings it to the ending position, schedcnt is decremented'' */
   dowriteunsched(ioscp);
}

/* rescheduling routines */

static void donoread(int fd)
{
 ss_unsched(ss_sigread(fd),readit,fd);
}

static void checksched(int fd)
{
 ioscstack iosc;
 /* ``if it is 0, buffer is emptied, and all startpos are set to 0.'' */
 if (!fdschedcnt[fd])
  {
   if (fdendpos[fd] == bufsize)
     if (ss_schedwait(ss_sigread(fd),readit,fd,1) == -1)
       DIE0(alloc);
   fdendpos[fd] = 0;
   for (iosc = fdio[fd];iosc;iosc = SODnext(iosc))
     SODdata(iosc).startpos = 0; /* schedcnt must be 0 */
  }
}

static void doeof(int fd)
{
 intstack eof;
 fdlive[fd] = 0; /* just in case */
 for (eof = fdeof[fd];eof;eof = SODnext(eof))
   if (fdlive[SODdata(eof)])
    {
     fdlive[SODdata(eof)] = 0;
     donoread(SODdata(eof));
     if (ss_schedonce(ss_asap(),readit,SODdata(eof)) == -1)
       DIE0(alloc);
    }
}

static void dowritesched(struct iosc *ioscp)
{
 if (!ioscp->flagsched)
   ++fdschedcnt[ioscp->in];
 ioscp->flagsched = 1;

 if (ss_schedvwait(ss_sigwrite(ioscp->out),writeit,0,0,ioscp,1) == -1)
   DIE0(alloc);
 /* Exercise for the reader: Show that this ``works'' for signals but */
 /* can fail miserably with interrupts. This is one situation where */
 /* the behavior of multiple threads scheduled upon one signal is */
 /* very important. */
}

static void dowriteunsched(struct iosc *ioscp)
{
 int in;

 ss_unschedv(ss_sigwrite(ioscp->out),writeit,0,0,ioscp);
 in = ioscp->in;
 ioscp->flagsched = 0;
 --fdschedcnt[in];
 checksched(in); /* ``schedule count is checked [when] decremented...'' */
}

/* and the function which pulls it all together */

SODdecl(opstack,struct { int fdin; int fd; int special; });

unsigned int scan_fd(char const* s, int* fd) {
  unsigned int tmp;
  unsigned int len=uint_scan(s, &tmp);
  *fd=tmp; /* XXX: overflow */
  return len;
}

int main(int argc,char** argv)
{
 int opt;
 char ch;
 char *t;
 int fd;
 intstack fdinhead;
 intstack fdin;
 opstack iohead;
 opstack eofhead;
 opstack op;
 ioscstack iosc;

 /* XXXXXXXXX: set non-blocking I/O? */

 fdinhead = 0;
 iohead = 0;
 eofhead = 0;

 flagverbose = 1;

 while ((opt = getopt(argc,argv,"b:vqQ")) != -1)
   switch(opt)
    {
     case 'b':
       if (optarg[uint_scan(optarg,&bufsize)])
	 (void)0; /* extra chars; no meaning at this time */
       break;
     case 'v':
       flagverbose = 2;
       break;
     case 'q':
       flagverbose = 0;
       break;
     case 'Q':
       flagverbose = 1;
       break;
     case '?': /*XXX*/
     default:
       usage();
    }
 argc -= optind;
 argv += optind;

 while (*argv)
  {
   unsigned int len;
   fdin = SODalloc(intstack,fdin,malloc);
   if (!fdin)
     DIE0(alloc);
   t = *argv;
   len = scan_fd(t,&(SODdata(fdin)));
   if (len == 0) usage();
   t += len;
   SODpush(fdinhead,fdin);
   for (;;)
    {
     ch = *t;
     if (!ch) break;
     op = SODalloc(opstack,op,malloc);
     if (!op)
       DIE0(alloc);
     SODdata(op).fdin = SODdata(fdin);
     SODdata(op).special = 0;
     ++t;
     len = scan_fd(t,&(SODdata(op).fd));
     if (len == 0) usage();
     t += len;
     switch(ch)
      {
       case ':':
	 SODdata(op).special = 1; /* falls through */
       case '-':
       case ',':
	 SODpush(iohead,op);
	 break;
       case 'e':
	 SODpush(eofhead,op);
	 break;
       /* XXX: here's the greatest possibilities for future development */
       default:
	 usage();
      }
    }
   ++argv;
  }

 if (!fdinhead)
   _exit(0); /* might as well take the easy way out */
 fdinmax = SODdata(fdinhead);
 for (fdin = fdinhead;fdin;fdin = SODnext(fdin))
  {
   fd = SODdata(fdin);
   if (fd > fdinmax)
     fdinmax = fd;
  }
 
 fdio = malloc((fdinmax + 1) * sizeof(ioscstack));
 if (!fdio) DIE0(alloc);
 for (fd = 0;fd <= fdinmax;++fd) fdio[fd] = 0;
 fdeof = malloc((fdinmax + 1) * sizeof(intstack));
 if (!fdeof) DIE0(alloc);
 for (fd = 0;fd <= fdinmax;++fd) fdeof[fd] = 0;
 fdlive = malloc((fdinmax + 1) * sizeof(int));
 if (!fdlive) DIE0(alloc);
 for (fd = 0;fd <= fdinmax;++fd) fdlive[fd] = 0;
 fdendpos = malloc((fdinmax + 1) * sizeof(int));
 if (!fdendpos) DIE0(alloc);
 for (fd = 0;fd <= fdinmax;++fd) fdendpos[fd] = 0;
 fdschedcnt = malloc((fdinmax + 1) * sizeof(int));
 if (!fdschedcnt) DIE0(alloc);
 for (fd = 0;fd <= fdinmax;++fd) fdschedcnt[fd] = 0;
 fdbuf = malloc((fdinmax + 1) * sizeof(char *));
 if (!fdbuf) DIE0(alloc);
 /* no fdbuf initialization needed */

 while (fdinhead)
  {
   SODpop(fdinhead,fdin);
   fdlive[SODdata(fdin)] = 1;
   SODfree(fdin,free);
  }

 for (fd = 0;fd <= fdinmax;++fd)
   if (fdlive[fd])
     if (!(fdbuf[fd] = malloc(bufsize)))
       DIE0(alloc);

 while (iohead)
  {
   SODpop(iohead,op);
   iosc = SODalloc(ioscstack,iosc,malloc);
   if (!iosc)
     DIE0(alloc);
   SODdata(iosc).in = SODdata(op).fdin;
   SODdata(iosc).out = SODdata(op).fd;
   SODdata(iosc).dead = 0;
   SODdata(iosc).breakpipe = SODdata(op).special; /*XXX*/
   SODdata(iosc).startpos = 0;
   SODdata(iosc).flagsched = 0;
   SODpush(fdio[SODdata(op).fdin],iosc);
   SODfree(op,free);
  }

 while (eofhead)
  {
   SODpop(eofhead,op);
   fdin = SODalloc(intstack,fdin,malloc);
   if (!fdin)
     DIE0(alloc);
   SODdata(fdin) = SODdata(op).fd;
   SODpush(fdeof[SODdata(op).fdin],fdin);
   SODfree(op,free);
  }

 for (fd = 0;fd <= fdinmax;++fd)
   if (fdlive[fd])
     ss_schedwait(ss_sigread(fd),readit,fd,1);

 ss_addsig(SIGPIPE);

 if (ss_exec() == -1)
   DIE0(alloc);

 _exit(0);
}
