package testlib;

use 5.014; # so srand returns the seed
use strict;
use warnings;
use Cwd qw(realpath);
use Digest::MD5;
use File::Basename;
use File::Compare;
use File::Find;
use File::Path qw(make_path remove_tree);
use File::Spec::Functions qw(:ALL);
use IO::Select;
use IPC::Open3;
use Symbol qw(gensym);
use Test::More;

our (@ISA, @EXPORT, @EXPORT_OK);
BEGIN {
    require Exporter;
    @ISA = qw(Exporter);
    @EXPORT = qw(
	TODO_add
	clean
	cmp_dummy_files
	create_directories
	create_dummy_files
	escape
	etcmanage
	etcmanage_with_input
	hash
	hash_file
	is_manifest
	is_num
	isnt_num
	ok_dummy_files
	ok_filestate
	open_etcmanage
	parse_manifest
	set_filestate
	state_icon
	sortlines
	vars
    );
}

sub hash {
    return Digest::MD5->new->add($_[0])->hexdigest;
}

sub hash_file {
    open(my $fh, "<", $_[0]) or die("unable to open $_[0] for reading: $!");
    binmode($fh);
    return Digest::MD5->new->addfile($fh)->hexdigest;
}

# the build tree
my $bd = $ENV{"abs_top_builddir"}
    or die("abs_top_builddir environment variable not set");
$bd = realpath($bd) if defined($bd);
# the source tree
my $sd = $ENV{"abs_top_srcdir"}
    or die("abs_top_srcdir environment variable not set");
$sd = realpath($sd) if defined($sd);
# directory containing this script
my $mydir = realpath(dirname($0));
# <build tree>/t
my $td = catdir($bd, abs2rel($mydir, $sd));

my %vars = ();
# path to the built etcmanage script
$vars{etcmanage} = catfile($bd, "etcmanage");
# where to stick temporary test files
$vars{test_dir} = catdir($td, basename($0, ".pl") . ".tmp");
# where to stick "live" test config files
$vars{live_dir} = $vars{test_dir};
# where to stick "upstream" test config files (relative to live_dir)
$vars{upstream_dir_in} = catdir(rootdir(), "usr", "netbsd-etc");
# where to stick "upstream" test config files (with live_dir)
$vars{upstream_dir_out} = catdir($vars{live_dir}, $vars{upstream_dir_in});
# where the temporary test database is created (relative to live_dir)
$vars{dbfile_in} = catfile(rootdir(), "var", "db", "etcmanage.db");
# where the temporary test database is created (with live_dir)
$vars{dbfile_out} = catfile($vars{live_dir}, $vars{dbfile_in});
# names of dummy config files (relative to live_dir)
$vars{dummy_files_nonetc} = [
    map(catfile(rootdir(), $_),
	"foo",
	"bar",
	"baz",
    )];
$vars{dummy_files_etc} = [
    map(catfile(rootdir(), "etc", $_),
	"bif",
	"services",
    )];
$vars{dummy_files} = [@{$vars{dummy_files_nonetc}}, @{$vars{dummy_files_etc}}];
# contents of the files in dummy_files
$vars{dummy_contents} = {
    map(($_ => "dummy contents for file $_\n"), @{$vars{dummy_files}})
};
# reference manifest for the files in dummy_files
$vars{dummy_manifest} = {
    map(($_ => hash($vars{dummy_contents}{$_})), @{$vars{dummy_files}})
};
# random number seed (for randomized testing)
$vars{seed} = srand;

# filenames that shouldn't be in the database
{
    my $etcfile = $vars{dummy_files_etc}->[0];
    my $etcfile_dir = dirname($etcfile);
    my $etcfile_base = basename($etcfile);
    $vars{abnormal_files} = [
	"",
	# what are relative paths relative to?
	abs2rel($etcfile, rootdir()),
	# pathnames with "..", ".", or extra slashes could result in
	# duplicate entries in the database (so could symlinks, but
	# they're harder to detect correctly)
	"$etcfile_dir/../etc/$etcfile_base",
	"$etcfile_dir//$etcfile_base",
	"$etcfile_dir/./$etcfile_base",
	# whitespace is used as a manifest field separator
	"$etcfile with_space",
	"$etcfile\fwith_formfeed",
	"$etcfile\nwith_newline",
	"$etcfile\rwith_return",
	"$etcfile\twith_tab",
	# octothorpe is used for comments in a manifest
	"#$etcfile",
	];
}

sub TODO_add {
    $::TODO = join("; ", (defined($::TODO) ? ($::TODO,) : ()), @_);
}

sub clean {
    remove_tree($vars{test_dir});
}

sub cmp_dummy_files {
    # find all files in the live dir and the upstream dir
    my (%live, %upstream);
    foreach (
	[\%live, $vars{live_dir}],
	[\%upstream, $vars{upstream_dir_out}],
	)
    {
	my ($h, $d) = @{$_};
	find(
	    {
		wanted => sub {
		    my $f = catdir(rootdir(), abs2rel($_, $d));
		    # skip over non-config files
		    if ($d eq $vars{live_dir}) {
			if ($f eq $vars{upstream_dir_in}) {
			    $File::Find::prune = 1;
			    return;
			}
			return if ($f eq $vars{dbfile_in});
		    }
		    # skip directories
		    return if -d $_;
		    # record the visited file
		    $h->{$f} = undef;
		},
		no_chdir => 1,
	    },
	    $d
	    );
    }

    # organize the files into the following buckets:
    #   * file is only found in the live dir
    #   * file is only found in the upstream dir
    #   * file is found in both places and are different
    #   * file is found in both places and they're the same
    my %ret = (
	only_live => [],
	only_upstream => [],
	diff => [],
	same => [],
    );
    foreach (keys(%live)) {
	if (exists($upstream{$_})) {
	    my $l = catfile($vars{live_dir}, $_);
	    my $u = catfile($vars{upstream_dir_out}, $_);
	    if (compare($l, $u) == 0) {
		push(@{$ret{same}}, $_);
	    } else {
		push(@{$ret{diff}}, $_);
	    }
	    delete $upstream{$_};
	} else {
	    push(@{$ret{only_live}}, $_);
	}
    }
    push(@{$ret{only_upstream}}, keys(%upstream));

    return \%ret;
}

sub create_directories {
    foreach (
	$vars{live_dir},
	$vars{upstream_dir_out},
	) {
	make_path($_);
    }
}

sub create_dummy_files {
    create_directories();
    foreach my $d ($vars{live_dir}, $vars{upstream_dir_out}) {
	foreach (@{$vars{dummy_files}}) {
	    my $f = catfile($d, $_);
	    make_path(dirname($f));
	    open(my $fh, ">", $f) or die("unable to open $f for writing: $!");
	    $fh->say("dummy contents for file " . $_);
	}
    }
}

# substitute whitespace for the equivalent perl escape sequence
sub escape {
    my %repl = ("\f" => "\\f", "\n" => "\\n", "\r" => "\\r", "\t" => "\\t");
    my $re = qr/[${[join("", keys(%repl))]}[0]]/;
    return $_[0] =~ s/($re)/$repl{$1}/gr
}

sub etcmanage {
    return etcmanage_with_input("", @_);
}

sub etcmanage_with_input {
    my $input = shift();
    make_path($vars{test_dir});
    my ($in, $out, $err);
    $err = gensym;
    my $pid = open_etcmanage($in, $out, $err, @_);
    my $output = '';
    my $errput = '';
    my $s_r = IO::Select->new($out, $err);
    my $s_w = IO::Select->new($in);
    open(my $input_fh, "<", \$input)
	or die("unable to open \$input as a file for reading");
    my $line = "";
    while (1) {
	last unless ($s_r->count() or $s_w->count());

	my ($ready_r, $ready_w, undef) =
	    IO::Select->select($s_r, $s_w, undef);

	foreach (@{$ready_r}) {
	    my $buf = '';
	    my $rv = $_->sysread($buf, 4 * 1024);
	    if (!defined($rv)) {
		die("error from sysread: $!");
	    } elsif ($rv == 0) {
		$s_r->remove($_);
		$_->close;
	    } elsif ($_ == $out) {
		$output .= $buf;
	    } elsif ($_ == $err) {
		$errput .= $buf;
	    } else {
		die("unexpected fd from select");
	    }
	}

	foreach (@{$ready_w}) {
	    $line = readline($input_fh) if ($line eq "");
	    if (!defined($line)) {
		$s_w->remove($_);
		$_->close();
		$input_fh->close();
		next;
	    }
	    my $rv = $_->syswrite($line);
	    if (!defined($rv)) {
		die("error for syswrite: $!");
	    } else {
		$line = substr($line, $rv);
	    }
	}
    }
    waitpid($pid, 0);
    return ($? >> 8, $output, $errput);
}

# args:
#   1. hashref to test
#   2. reference (expected) hashref
#   3. test name (optional)
sub is_manifest ($$;$) {
    my %test = %{shift()};
    my %exp = %{shift()};

    local $Test::Builder::Level = $Test::Builder::Level + 1;

    return subtest @_ => sub {
	foreach (keys(%exp)) {
	    # test 2n+1
	    ok(exists($test{$_}), "contains $_");

	    SKIP: {
		skip "file not in manifest", 1 unless exists($test{$_});

		# test 2n+2
		is($test{$_}, $exp{$_}, "hash for $_ matches");

		delete $test{$_};
	    };
	}

	# test 2n+3
	is_num(scalar keys(%test), 0, "no extra entries");

	done_testing((2 * keys(%exp)) + 1);
    };
}

sub is_num ($$;$) {
    my $tb = Test::More->builder;
    return $tb->is_num(@_);
}

sub isnt_num ($$;$) {
    my $tb = Test::More->builder;
    return $tb->isnt_num(@_);
}

# args:
#   1. test name (optional)
sub ok_dummy_files (;$) {

    local $Test::Builder::Level = $Test::Builder::Level + 1;

    return subtest @_ => sub {
	my $ret = cmp_dummy_files();

	# test 1
	is_num(@{$ret->{only_live}}, 0,
	       "no files only in live");

	# test 2
	is_num(@{$ret->{only_upstream}}, 0,
	       "no files only in upstream");

	# test 3
	is_num(@{$ret->{diff}}, 0,
	       "no files have different contents");

	# test 4
	is_deeply([sort(@{$ret->{same}})], [sort(@{$vars{dummy_files}})],
		  "all files have matching contents");

	done_testing(4);
    };
}

# Test if the state of the live file, db entry, and upstream file
# match an expected state.  Takes three or four arguments:
#   0. An arrayref describing the expected state of the file.  See the
#      description of argument 0 of set_filestate().
#   1. inside name of the file that should have the above state, or an
#      arrayref of such names
#   2. hashref of the manifest (parsed from "etcmanage --print")
#   3. (optional) test name
#
# See also:  set_filestate()
sub ok_filestate {
    my $state = shift();
    my @files = (shift());
    if (ref($files[0]) eq "ARRAY") {
	@files = @{$files[0]};
    }
    my $manifest = shift();

    local $Test::Builder::Level = $Test::Builder::Level + 1;

    my $subtest = sub {

	foreach my $f_in (@files) {
	    my $f_out = catfile($vars{live_dir}, $f_in);
	    my $up_out = catfile($vars{upstream_dir_out}, $f_in);
	    my $pfx = state_icon($state) . " $f_in";

	    # check the live file
	    if (defined($state->[0])) {
		## subtest 1
		ok((-f $f_out) && (hash($state->[0]) eq hash_file($f_out)),
		   "$pfx: live file exists and has the correct hash");
	    } else {
		## subtest 1
		ok(!(-e $f_out), "$pfx: live file doesn't exist");
	    }

	    # check the database
	    if (defined($state->[1])) {
		my $expected = ($state->[1] eq "manual")
		    ? "manual" : hash($state->[1]);
		## subtest 2
		ok(exists($manifest->{$f_in})
		   && ($manifest->{$f_in} eq $expected),
		   "$pfx: database entry exists with proper value");
	    } else {
		## subtest 2
		ok(!exists($manifest->{$f_in}),
		   "$pfx: database entry doesn't exist");
	    }

	    # check the upstream file
	    if (defined($state->[2])) {
		## subtest 3
		ok((-f $up_out) && (hash($state->[2]) eq hash_file($up_out)),
		   "$pfx: upstream file exists and has the correct hash");
	    } else {
		## subtest 3
		ok(!(-e $up_out), "$pfx: upstream file doesn't exist");
	    }
	}

	done_testing(3 * @files);
    };

    subtest($_[0] => $subtest);
}

sub open_etcmanage {
    make_path($vars{test_dir});
    return open3(@_[0 .. 2], $vars{etcmanage},
		 "--destdir", $vars{test_dir}, @_[3 .. $#_]);
}

sub parse_manifest {
    my @lines_with_newlines = split(/^/m, $_[0]);
    my @lines = map({chomp; $_} @lines_with_newlines);
    my @lines_no_comments = grep(!/^#/, @lines);
    my @file_hash_tuples = map(split(' ', $_, 2), @lines_no_comments);
    my %ret = @file_hash_tuples;
    return %ret;
}

# Set the state of the live file, db entry, and upstream file to match
# the given state description.  Takes two arguments:
#   0. An arrayref describing the state of the system that should be
#      established.  It has three entries:
#       0. contents of the installed ("live") version of the file or
#          undef if the installed file doesn't exist
#       1. database entry state, which is one of three possibilities:
#           * data whose hash is in the database
#           * undef if there is no entry in the database for the file
#           * the string "manual" if the file is marked as manually
#             maintained
#       2. contents of the upstream version of the file or undef if
#          there is no upstream version of the file
#   1. inside name of the config file that will have the above state,
#      or an arrayref of such names
#
# See also:  ok_filestate()
#
sub set_filestate {
    my $state = shift();
    my @files = (shift());
    if (ref($files[0]) eq "ARRAY") {
	@files = @{$files[0]};
    }

    foreach my $f_in (@files) {
	my $f_out = catfile($vars{live_dir}, $f_in);
	my $up_out = catfile($vars{upstream_dir_out}, $f_in);

	# create/remove the live file
	if (defined($state->[0])) {
	    make_path(dirname($f_out));
	    open(my $fh, ">", $f_out) or die("unable to open $f_out: $!");
	    print($fh $state->[0]);
	} elsif (-e $f_out) {
	    unlink($f_out) or die("unable to delete $f_out: $!");
	}

	# create/remove the db entry
	my @args;
	if (defined($state->[1])) {
	    if ($state->[1] eq "manual") {
		@args = ("--manual", $f_in);
	    } else {
		@args = ("--add", $f_in, "--md5", hash($state->[1]));
	    }
	} else {
	    @args = ("--remove", $f_in);
	}
	my ($ret) = etcmanage(@args);
	die("'etcmanage " . join(" ", @args) . "' failed") if ($ret != 0);

	# create/remove the upstream file
	make_path($vars{upstream_dir_out});
	if (defined($state->[2])) {
	    make_path(dirname($up_out));
	    open(my $fh, ">", $up_out) or die("unable to open $up_out: $!");
	    print($fh $state->[2]);
	} elsif (-e $up_out) {
	    unlink($up_out) or die("unable to delete $up_out: $!");
	}
    }
}

# return a short string representing the state of the live file, db,
# and upstream file (for naming tests).  e.g.:
#      state_icon([undef, "manual", "X"]) -> "(-,m,X)"
#
# see also:  set_filestate(), ok_filestate()
#
sub state_icon {
    my @abbr = map(defined($_) ? (($_ eq "manual") ? "m" : $_) : "-",
		   @{shift()});
    return "(" . join(",", @abbr) . ")"
}

sub sortlines {
    my $txt = $_[0];
    my $added_nl = 0;
    if ($txt !~ /\n\z/) {
	$txt .= "\n" if ($txt !~ /\n\z/);
	$added_nl = 1;
    }
    my $ret = join("", sort(split(/^/m, $txt)));
    $ret = "" if ($added_nl && ($ret eq "\n"));
    return $ret
}

sub vars {
    return %vars;
}
