my %vars = vars();
my @files;
my @dummy_files = @{$vars{dummy_files}};
my ($ret, $out, $err);
my %manifest;
my %dummy_manifest = %{$vars{dummy_manifest}};
my ($deleted_in, $deleted_out);
my $pfx;

my $etcfile = ${$vars{dummy_files_etc}}[0];
my $nonetcfile = ${$vars{dummy_files_nonetc}}[0];

# Function to test multiple update scenarios.
#
# The first argument is an arrayref of scenarios, each of which is a
# hashref with the following entries:
#   * before:  passed as argument 0 to set_filestate()
#   * after:  passed as argument 0 to ok_filestate()
#   * token:  The string that, along with the filename, should be
#     printed by 'etcmanage --update'.  undef if nothing should be
#     printed.
#   * err: if true, token should be printed to STDERR, STDOUT
#     otherwise
#   * todo (optional): an arrayref of TODO strings if this scenario
#     is expected to fail for both non-etc and etc config files
#   * files (optional): an arrayref of inside filenames to use for
#     testing the scenario.  each file is tested separately
#     ('etcmanage --update' is run once per file).  if unspecified,
#     sets it to [$nonetcfile, $etcfile].
#
# The second argument is a test name prefix (optional)
#
sub ok_scenarios {
    my $scenarios = shift();
    my $name = shift();
    my $pfx = join(": ", $name || (), "");
    my %todo = ();
    my %files = ();

    local $Test::Builder::Level = $Test::Builder::Level + 1;

    clean();

    # set up the scenarios
    foreach my $scenario (@{$scenarios}) {
	$todo{$_} = undef foreach (@{$scenario->{todo} or []});
	if (!exists($scenario->{files})) {
	    $scenario->{files} = [$nonetcfile, $etcfile];
	}
	foreach my $f_in (@{$scenario->{files}}) {
	    $files{$f_in}++
		and die("file $f_in already specified in another scenario");
	    set_filestate($scenario->{before}, $f_in);
	}
    }

    # run etcmanage --update
    ($ret, $out, $err) = etcmanage("--update", $vars{upstream_dir_in});

    my $subtest = sub {

	## subtest 1
	# check return value
	is_num($ret, 0, "${pfx}update doesn't fail");

	# check STDOUT/STDERR
	my @exp_out = ();
	my @exp_err = ();
	foreach my $scenario (@{$scenarios}) {
	    my $token = $scenario->{token};
	    next unless defined($token);
	    my $toerr = $scenario->{err};
	    foreach my $f_in (@{$scenario->{files}}) {
		my $str = "$token $f_in\n";
		if ($toerr) {
		    push(@exp_err, $str);
		} else {
		    push(@exp_out, $str);
		}
	    }
	}
	my $exp_out = join("", sort(@exp_out));
	my $exp_err = join("", sort(@exp_err));
	## subtest 2
	is(join("", sortlines($out)), $exp_out,
	   "${pfx}writes the expected messages to STDOUT");
	## subtest 3
	is(join("", sortlines($err)), $exp_err,
	   "${pfx}writes the expected messages to STDERR");

	# check the effects on the config files
	($ret, $out, $err) = etcmanage("--print");
	my %manifest = parse_manifest($out);
	my $ntests = 0;
	foreach my $scenario (@{$scenarios}) {
	    my $p = "$pfx"
		. state_icon($scenario->{before})
		. " -> "
		. state_icon($scenario->{after});
	    foreach my $f_in (@{$scenario->{files}}) {
		## subtest 4+
		ok_filestate($scenario->{after}, $f_in, \%manifest, "$p $f_in");
		++$ntests;
	    }
	}

	done_testing(3 + $ntests);
    };

    TODO: {
	local $TODO;
	TODO_add(keys(%todo));
	subtest $name => $subtest;
    };
}

my @scenarios = ();

## test 1
# no file to manage, nothing in database -> noop
push(@scenarios, {
    before => [ undef, undef, undef ],
    after  => [ undef, undef, undef ],
    token  => undef,
    err    => undef,
     });

## test 2
# new upstream file -> create and manage
push(@scenarios, {
    before => [ undef, undef, "X" ],
    after  => [ "X", "X", "X" ],
    token  => "NEW",
    err    => 0,
     });

## test 3
# file marked manual, doesn't exist either locally or upstream ->
# manual file that matches upstream
push(@scenarios, {
    before => [ undef, "manual", undef ],
    after  => [ undef, "manual", undef ],
    token  => "MANUAL_DELETED;UPSTREAM_WITHDRAWN",
    err    => 0,
     });

## test 4
# file marked manual, missing locally but exists upstream -> manual
# file that differs from upstream
push(@scenarios, {
    before => [ undef, "manual", "X" ],
    after  => [ undef, "manual", "X" ],
    token  => "MANUAL_DELETED",
    err    => 0,
     });

## test 5
# both live and upstream were deleted, but db still has a hash -> file
# doesn't match db so don't modify anything
push(@scenarios, {
    before => [ undef, "X", undef ],
    after  => [ undef, "X", undef ],
    token  => "MISSING;UPSTREAM_WITHDRAWN",
    err    => 0,
     });

## test 6
# local deleted the file, upstream hasn't done anything with the file
# -> report that it's missing
push(@scenarios, {
    before => [ undef, "X", "X" ],
    after  => [ undef, "X", "X" ],
    token  => "MISSING",
    err    => 0,
     });

## test 7
# local deleted the file, upstream changed the file -> report
# divergence
push(@scenarios, {
    before => [ undef, "X", "Y" ],
    after  => [ undef, "X", "Y" ],
    token  => "MISSING;UPSTREAM_DIFFERENT",
    err    => 0,
     });

## test 8
# local has a file that doesn't exist in upstream, isn't managed ->
# file outside scope of etcmanage, so noop
push(@scenarios, {
    before => [ "X", undef, undef ],
    after  => [ "X", undef, undef ],
    token  => undef,
    out    => undef,
     });

## test 9
# new upstream file happens to match locally created file -> don't
# auto manage the file (otherwise --remove is no longer an effective
# way to mark the file as unmanaged) but report the situation
push(@scenarios, {
    before => [ "X", undef, "X" ],
    after  => [ "X", undef, "X" ],
    token  => "UNMANAGED_EQ_UPSTREAM",
    err    => 0,
     });

## test 10
# new upstream file differs from locally created file -> don't auto
# manage the file (otherwise --remove is no longer an effective way to
# mark the file as unmanaged) but warn the user
push(@scenarios, {
    before => [ "X", undef, "Y" ],
    after  => [ "X", undef, "Y" ],
    token  => "UNMANAGED_NEQ_UPSTREAM",
    err    => 0,
     });

## test 11
# upstream deleted a manually managed file that still exists -> don't
# touch the file or db, but report the situation
push(@scenarios, {
    before => [ "X", "manual", undef ],
    after  => [ "X", "manual", undef ],
    token  => "MANUAL;UPSTREAM_WITHDRAWN",
    err    => 0,
     });

## test 12
# file marked as manual but it matches upstream -> don't touch db, but
# report the situation
push(@scenarios, {
    before => [ "X", "manual", "X" ],
    after  => [ "X", "manual", "X" ],
    token  => "MANUAL_EQ_UPSTREAM",
    err    => 0,
     });

## test 13
# file marked as manual but differs from upstream -> don't touch the
# file or db, but report the situation
push(@scenarios, {
    before => [ "X", "manual", "Y" ],
    after  => [ "X", "manual", "Y" ],
    token  => "MANUAL",
    err    => 0,
     });

## test 14
# managed file matches db and upstream deletes the file -> delete the
# local copy to match upstream
push(@scenarios, {
    before => [ "X", "X", undef ],
    after  => [ undef, undef, undef ],
    token  => "DELETED",
    err    => 0,
     });

## test 15
# managed file matches db and upstream hasn't changed the file ->
# nothing to do, so noop
push(@scenarios, {
    before => [ "X", "X", "X" ],
    after  => [ "X", "X", "X" ],
    token  => undef,
    out    => undef,
     });

## test 16
# local file is unmodified but upstream has changed the file -> update
# the local file and db
push(@scenarios, {
    before => [ "X", "X", "Y" ],
    after  => [ "Y", "Y", "Y" ],
    token  => "UPDATED",
    err    => 0,
     });

## test 17
# local has modified the file but upstream has deleted the file ->
# don't touch the db and warn the user
push(@scenarios, {
    before => [ "X", "Y", undef ],
    after  => [ "X", "Y", undef ],
    token  => "MODIFIED;UPSTREAM_WITHDRAWN",
    err    => 0,
     });

## test 18
# both live and upstream were modified in the same way -> file doesn't
# match db so don't modify anything
push(@scenarios, {
    before => [ "X", "Y", "X" ],
    after  => [ "X", "Y", "X" ],
    token  => "MODIFIED_EQ_UPSTREAM",
    err    => 0,
     });

## test 19
# local modified the file but upstream hasn't done anything -> warn
# the user
push(@scenarios, {
    before => [ "X", "Y", "Y" ],
    after  => [ "X", "Y", "Y" ],
    token  => "MODIFIED",
    err    => 0,
     });

## test 20
# both local and upstream have modified in different ways -> warn the
# user
push(@scenarios, {
    before => [ "X", "Y", "Z" ],
    after  => [ "X", "Y", "Z" ],
    token  => "MODIFIED;UPSTREAM_DIFFERENT",
    err    => 0,
     });

foreach (@scenarios) {
    my $icon = state_icon($_->{before});
    ok_scenarios([$_], "single scenario $icon");
}

######################################################################
### all scenarios at once

my $i = 0;
foreach (@scenarios) {
    $_->{files} = [map(catfile(dirname($_), sprintf("%02d", $i), basename($_)),
		       $nonetcfile, $etcfile)];
    ++$i;
}

## test 21
ok_scenarios(\@scenarios, "all scenarios at once");

######################################################################
### test updating when both live and upstream dirs are empty

clean();
create_directories();
($ret, $out, $err) = etcmanage("--update", $vars{upstream_dir_in});

## test 22
is_num($ret, 0, "empty dir: doesn't fail");

## test 23
is($err, "", "empty dir: doesn't write to STDERR");

## test 24
is($out, "", "empty dir: doesn't write to STDOUT");

######################################################################
### test suspiciously large number of new files

foreach my $enable_safety (1, 0) {

    $pfx = "too many new";
    $pfx .= " (disabled safety checks)" if (!$enable_safety);

    clean();

    set_filestate(["X", "X", "X"], \@dummy_files);
    ($ret, $out, $err) = etcmanage("--print");
    %dummy_manifest = parse_manifest($out);

    @files = map(("$nonetcfile-$_", "$etcfile-$_"), 1..5);
    set_filestate([undef, undef, "X"], \@files);

    ($ret, $out, $err) = etcmanage(
	$enable_safety ? () : ("--no-safety-checks",),
	"--update", $vars{upstream_dir_in});

    if ($enable_safety) {
	## test 25
	isnt_num($ret, 0, "$pfx: exits with non-0 status");

	## test 26
	like($err, qr/^(?:ERROR: .*\n)+\z/, "$pfx: prints an error message");

	## test 27
	is($out, "", "$pfx: does not log any new files");
    } else {
	## test 31
	is_num($ret, 0, "$pfx: exits with 0 status");

	## test 32
	like($err, qr/^(?:WARNING: .*\n)+\z/, "$pfx: prints a warning");

	## test 33
	is(sortlines($out),
	   join("", sort(map("NEW $_\n", @files))),
	   "$pfx: logs NEW messages")
    }

    ($ret, $out, $err) = etcmanage("--print");
    %manifest = parse_manifest($out);

    if ($enable_safety) {
	## test 28
	is_manifest(\%manifest, \%dummy_manifest, "$pfx: db untouched");

	## test 29
	ok_filestate([undef, undef, "X"], \@files, \%manifest,
		     "$pfx: new files untouched");
    } else {
	$dummy_manifest{$_} = hash("X") foreach (@files);

	## test 34
	is_manifest(\%manifest, \%dummy_manifest, "$pfx: db updated");

	## test 35
	ok_filestate(["X", "X", "X"], \@files, \%manifest,
		     "$pfx: new files installed");
    }

    ## test 30, test 36
    ok_filestate(["X", "X", "X"], \@dummy_files, \%manifest,
		 "$pfx: dummy files untouched");
}

######################################################################
### test suspiciously large number of withdrawn files

foreach my $enable_safety (1, 0) {

    $pfx = "too many withdrawn";
    $pfx .= " (disabled safety checks)" if (!$enable_safety);

    clean();

    @files = map(("$nonetcfile-$_", "$etcfile-$_"), 1..5);
    set_filestate(["X", "X", undef], \@files);
    set_filestate(["X", "X", "X"], \@dummy_files);
    ($ret, $out, $err) = etcmanage("--print");
    %dummy_manifest = parse_manifest($out);

    ($ret, $out, $err) = etcmanage(
	$enable_safety ? () : ("--no-safety-checks",),
	"--update", $vars{upstream_dir_in});

    if ($enable_safety) {
	## test 37
	isnt_num($ret, 0, "$pfx: exits with non-0 status");

	## test 38
	like($err, qr/^(?:ERROR: .*\n)+\z/, "$pfx: prints an error message");

	## test 39
	is($out, "", "$pfx: does not log any deleted files");
    } else {
	## test 43
	is_num($ret, 0, "$pfx: exits with 0 status");

	## test 44
	like($err, qr/^(?:WARNING: .*\n)+\z/, "$pfx: prints a warning");

	## test 45
	is(sortlines($out),
	   join("", sort(map("DELETED $_\n", @files))),
	   "$pfx: logs DELETED messages");
    }

    ($ret, $out, $err) = etcmanage("--print");
    %manifest = parse_manifest($out);

    if ($enable_safety) {
	## test 40
	is_manifest(\%manifest, \%dummy_manifest, "$pfx: db untouched");

	## test 41
	ok_filestate(["X", "X", undef], \@files, \%manifest,
		     "$pfx: withdrawn files untouched");
    } else {
	delete $dummy_manifest{$_} foreach (@files);

	## test 46
	is_manifest(\%manifest, \%dummy_manifest, "$pfx: db updated");

	## test 47
	ok_filestate([undef, undef, undef], \@files, \%manifest,
		     "$pfx: withdrawn files deleted");
    }

    ## test 42, test 48
    ok_filestate(["X", "X", "X"], \@dummy_files, \%manifest,
		 "$pfx: dummy files untouched");
}

######################################################################
### test abort on invalid directory

$pfx = "invalid dir";
clean();
($ret, $out, $err) = etcmanage(
    "--update", catfile(rootdir(), "foo", "bar", "baz"));

## test 49
isnt_num($ret, 0, "$pfx: exits with non-0 status");

## test 50
like($err, qr/^(?:ERROR: .*\n)\z/, "$pfx: prints an error message");

## test 51
is($out, "", "$pfx: doesn't write to STDOUT");

######################################################################
### rejects '..' with --destdir

$pfx = "rejects '..' with destdir";
clean();
create_dummy_files();
($ret, $out, $err) = etcmanage(
    "--update", catdir($vars{upstream_dir_in}, updir(),
		       basename($vars{upstream_dir_in})));

## test 52
isnt_num($ret, 0, "$pfx: exits with non-0 status");

## test 53
like($err, qr/^(?:ERROR: .*\n)\z/, "$pfx: prints an error message");

## test 54
is($out, "", "$pfx: doesn't write to STDOUT");

######################################################################
### properly handles "." and extra slashes

$pfx = "handles '.' and extra slashes";
clean();

{
    my $dir = dirname($vars{upstream_dir_in});
    my $base = basename($vars{upstream_dir_in});
    my $d = "/$dir/.///./$base/.//";
    local $vars{upstream_dir_in} = $d;

    ## test 55
    ok_scenarios(\@scenarios, $pfx);
}

done_testing(55);
