/*****************************************************************************
 *  ENTROPY - emerging network to reduce orwellian potency yield
 *
 *  Copyright (C) 2002 Juergen Buchmueller <pullmoll@stop1984.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *	$Id: store.h,v 1.2 2005/07/12 23:19:26 pullmoll Exp $
 *****************************************************************************/
#ifndef _store_h_
#define	_store_h_

#include "osd.h"
#include "ek5.h"
#include "sha1.h"
#include "key.h"

#if	HAVE_MMAP
#ifndef	HUGE_MMAP
#define	HUGE_MMAP	1
#endif
#else
#undef	HUGE_MMAP
#define	HUGE_MMAP	0
#endif

#define	SEEK_SLOTS	16
/* 1/LIMIT_FRACT below the average key count is the limit */
#define	LIMIT_FRACT	16

#define	NUM_ROUTES	16
#define	VER_SIZE	12

#define	DROP_MAX	8192
/* keys to be dropped, when (timeout) and what size */
typedef struct drop_key_s {
	sha1_digest_t sha1;
	time_t time;
	size_t size;
}	drop_key_t;


typedef struct store_s {
	osd_sem_t sem;					/* lock open files */
	size_t storecount;				/* current total number of keys */
	uint64_t storesize;			/* maximum size of the data store */
	uint64_t currsize;				/* current total size of the data store */
	size_t keycount[NUM_ROUTES];	/* number of keys per routing key */
	uint8_t version[VER_SIZE];		/* store version number */
	uint8_t destination[NUM_ROUTES];	/* destination fingerprint */
	uint8_t fingerprint[NUM_ROUTES];
	uint8_t avg;
	size_t seeks[SEEK_SLOTS+1];		/* how many seeks per distance */
	size_t tail;
	size_t head;
	drop_key_t drop[DROP_MAX];		/* list of keys to be dropped */
	sha1_digest_t missing[65536];	/* 64K slots for recently missing keys */
	sha1_digest_t available[65536];	/* 64K slots for recently available keys */
}	store_t;

typedef struct store_info_s {
	size_t storecount;
	uint64_t storesize;
	uint64_t currsize;
	size_t keycount[16];
	size_t seeks[SEEK_SLOTS+1];
}	store_info_t;

/* struct to keep the file age, path and name while purging the store */
typedef struct fileage_s {
	int marked;				/* non-zero if to be purged */
	size_t size;			/* size of this key (trailing zeroes stripped) */
	time_t age;				/* age (time() - atime in seconds) */
	uint8_t dist;			/* distance from our fingerprint */
	uint8_t tree1;			/* tree index 1 */
	uint8_t tree2;			/* tree index 2 */
	uint8_t tree3;			/* tree index 3 */
	uint8_t fpr;			/* fingerprint index */
	sha1_digest_t sha1;
}	fileage_t;

#define	STORE_LOCK() \
	if (0 != osd_sem_wait(&g_store->sem)) { \
		const char *errmsg = strerror(errno); \
		LOGS(L_STORE,L_ERROR,("locking store failed (%s)\n", \
			errmsg)); \
		die(1, "locking store failed (%s)", errmsg); \
	}
#define	STORE_UNLOCK()	osd_sem_post(&g_store->sem)

#ifdef	__cplusplus
extern "C" {
#endif
extern store_t *g_store;
extern sha1_digest_t null;

/* implemented by store.c */
int store_info(store_info_t *info);

/* called from store implementations after keycount[] changed */
int store_upd_fingerprint(void);

/* called from the store() functions to get/set version */
int store_get_version(void);
int store_put_version(int zap);

/* called from the store() functions to add/subtract a key from the stats */
int store_add_key(const sha1_digest_t *sha1, size_t size);
int store_sub_key(const sha1_digest_t *sha1, size_t size);

/* implemented by store types: mono, tree, mysql */
int store_put(const sha1_digest_t *sha1, const void *buff);
int store_get(const sha1_digest_t *sha1, void *buff);
int store_del(const sha1_digest_t *sha1);
int store_zap(const sha1_digest_t *sha1);
int store_check(const sha1_digest_t *sha1, int flags,
	size_t count, size_t required);
int store_purge(void);

void store_exit(int sig);
int store(void);

#if STORE_TYPE == 2
int store_connect_db(void);
void store_disconnect_db(void);
#endif

#ifdef	__cplusplus
}
#endif
#endif	/* !defined(_store_h_) */
