//-----------------------------------------------------------------------------
//
// $Id: doom_umi_lexer.c 1610 2023-10-06 09:40:13Z wesleyjohnson $
//
// Copyright (C) 2023 by DooM Legacy Team.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

/*
 * Lexer for DoomLegacy libdoom-umapinfo
 *
 * Copyright (c) 2022-2023 by the developers. See the LICENSE file for details.
 *
 *
 * Input data must be an octet stream.
 * Encoding of syntactic elements must be US-ASCII.
 * Arbitrary octets are accepted in quoted strings (except delimiter itself).
 */

#include "doomdef.h"
#ifdef ENABLE_UMAPINFO

// For FLEX
#include <stdlib.h>
#include <errno.h>

// For DoomLegacy
#include <ctype.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>

// Has int types, maxint, and everything for all platforms.
#include "doomtype.h"

#include "doom_umi.h"
#include "doom_umi_internal.h"
#include "doom_umi_lexer.h"

/* This header is generated by yacc */
#include "doom_umi_parser.h"

#include "m_misc.h"
  // dl_strncp


// Configuration Defines

#define DEBUG_UMAPINFO_LEXER

// If lexer recognizes negative numbers.
#define NEG_NUMBERS

/* C90 conformant replacement for SIZE_MAX */
// #define DOOM_UMI1_LEXER_MAX_LEN (~(size_t) 0)
// The size_t on some old machines, is a signed int, so it cannot be trusted.

// This is the max text and string that DoomLegacy will ever need to handle.
// If have anything longer, then something has broken.
// Limit of signed int is MAXINT (0x7fffffff).  Cannot prevent this from being stored in int.
#define UMI_LEXER_MAX_LEN  32000

#define LEXER_MSG_BUF_SIZE 256
static  const char * UMI_LEXER_PRINT_STR = "UMI LEXER";


// Public
unsigned int umi_lexer_lineno = 1;

// This indicates the position of a string or text.
// The lexer will not modify the string or text, so it will not
// have a string termination.  Lines may not have a line termination.
// The caller MUST use the text, using the length.
// The caller can change char of the text within the length.
// There is NO guarantee that there is any buffer after the text, so cannot append.
char * umi_lexer_text;
int umi_lexer_text_len;

// Quoted string
umi_qstring_t  umi_lexer_string;

// Number value
int  umi_lexer_value;

// Symbol and special char exposure.
char umi_lexer_char;


/* ======================================================================== */
/* Print message s1 for ASCII string s2 */
//   errcode: if <0, then print as error, 0=verbose, 1=debug
static
void umi_lexer_msg( int8_t errcode, const char * msg_str, const char * s2 )
{
    char buf[LEXER_MSG_BUF_SIZE];
    buf[0] = 0;  // in case of s2==NULL
    if( s2 ) {
        snprintf( buf, LEXER_MSG_BUF_SIZE-1, "'%s'", s2);
    }
    buf[LEXER_MSG_BUF_SIZE-1] = 0; // safety, due to unterminated snprintf
    umi_doom_print( errcode, UMI_LEXER_PRINT_STR, msg_str, buf );  // verbose or debug
}

static
void umi_lexer_msg_by_len( int8_t errcode, const char * msg_str, const char * s2, int s2_len )
{
    char buf[LEXER_MSG_BUF_SIZE];
    // Limit string to buffer, with a place for term 0.
    buf[0] = 0;  // empty string, in case of s2==NULL
    if( s2_len > LEXER_MSG_BUF_SIZE-1 )  s2_len = LEXER_MSG_BUF_SIZE-1;
    // It is necessary to put a term on the string, thus it must be buffered.
    // The alternative would be to alter the source string.
    // The s2 string may be at the end of the source buffer, so a term cannot be added past the s2_len.
    if( s2 ) {
        memcpy(buf, s2, s2_len);
        buf[s2_len] = 0;
    }
    buf[LEXER_MSG_BUF_SIZE-1] = 0;  // safety, due to paranoia
    umi_doom_print( errcode, UMI_LEXER_PRINT_STR, msg_str, buf );  // verbose or debug
}

// Error handling
#ifdef ENABLE_BUFFER_STACK
static
void umi_lexer_fatal_error( int8_t errcode, const char * msg1, const char * funcstr )
{
    char buf[LEXER_MSG_BUF_SIZE];
    umi_fatal_error = errcode;

    snprintf(buf, LEXER_MSG_BUF_SIZE-1, "%s, Function %s", msg1, funcstr );
    buf[LEXER_MSG_BUF_SIZE-1] = 0;
    umi_doom_print( errcode, UMI_LEXER_PRINT_STR, "fatal error", buf );  // verbose or debug
    // All fatal error must use DoomLegacy exits.
}
#endif


/* ======================================================================== */
#ifdef DEBUG_UMAPINFO_LEXER
static byte  EN_umi_lex_debug_out = 0;

/* Print debug message s1 for ASCII string s2 */
static
void umi_lexer_debug_msg( const char * msg_str, const char * s2 )
{
    umi_lexer_msg( 1, msg_str, s2 );
}

/* Print debug message s1 for quoted string s2 with length */
//  msg_str : message str
//  s2 : quoted string
//  s2_len : length of s2
static
void umi_lexer_debug_msg_qstring( const char * msg_str,
                            const char * s2, unsigned int s2_len )
{
    int np = 0;

    if( s2_len > UMI_LEXER_MAX_LEN )
        s2_len = UMI_LEXER_MAX_LEN;
   
    // This is DEBUG code, so must be independent of optional functions.
    size_t  buf_size = s2_len + strlen( msg_str ) + 64;
    char *  buf = (char *) malloc( buf_size );
   
    if( buf == NULL )
        return;

    size_t max_size = buf_size - 1;
    buf[np++] = '"'; // double_quote

    unsigned int i;
    for( i=0; i<s2_len; i++ )
    {
        char c = *s2++;

        if( np >= max_size )  goto print_out;  // should not happen
// [WDJ] This is questionable.  If a user has substituted text in their own language,
// they likely needed extended characters.  They also have likely provided
// font support for such on their computer, by unicode, or by loading a code page.
// These characters should be left alone.
// On the other hand, DoomLegacy screen fonts are very limited.
// DoomLegacy V_DrawCharacter is already protected.
// This is not a choice that is local to umapinfo.
#if 0
        /* Strip nonprintable US-ASCII characters */
        // [WDJ] Making all non-standard characters unrecognizable is not necessary.       
        // They are NOT unprintable, they just require a supporting font.
        if (c <= 0x1F || c >= 0x7F)
            c = '?';
#endif
#if 0
        /* Print extended US-ASCII characters as code */
        if (c <= 0x1F || c >= 0x7F)
        {
            int np2 = snprintf( &buf[np], max_size - np, "<%x>", c );
            if( np2 < 0 ) break;
            np += np2;
            continue;
        }
#endif
#if 1
        /* Print control codes, as code */
        if (c <= 0x1F)
        {
            int np2 = snprintf( &buf[np], max_size - np, "<%x>", c );
            if( np2 < 0 ) break;
            np += np2;
            continue;
        }
#endif
        buf[np++] = c;
    }
    buf[np++] = '"'; // double_quote
    buf[np++] = 0; // term

print_out:   
    buf[max_size] = 0;
    umi_doom_print( 1, UMI_LEXER_PRINT_STR, msg_str, buf );  // debug

    free(buf);
    return;
}

// Print umi_lexer_text
static
void umi_lexer_debug_msg_text( const char * msg_str )
{
    if( EN_umi_lex_debug_out )
    {
        umi_lexer_debug_msg_qstring( msg_str, umi_lexer_text, umi_lexer_text_len );
    }
}

// Print umi_lexer_text, with a value.
static
void umi_lexer_debug_msg_val( const char * msg_str, int val )
{
    if( EN_umi_lex_debug_out )
    {
        char buf[20];
        snprintf( buf, 19, "%d", val );
        buf[19] = 0;
        umi_lexer_debug_msg( msg_str, buf );
    }
}
#endif

// ==============================
// [WDJ] Lexer ought to have multiple keyword tables, which would be switched
// during parse according to which keywords were in effect.
// Umapinfo does not allow identifiers which overlap with keywords, so
// there is no need for keywords to be not in effect.
// Thus one keyword list would suffice.


/* ======================================================================== */
/* Case insensitive search for US-ASCII keyword */
// Return UMI_KEY_ value.
// Return UMI_KEY_NOT_FOUND when not found (neg value).
int  umi_keyword_lookup( const char * istr, unsigned int len,
                         const char * const * keyword_table, const unsigned int keyword_table_len )
{
    unsigned int keyi = keyword_table_len;

    while (keyi)  // search keyword list, from last to first
    {
        const char * kw  = keyword_table[--keyi];
        unsigned int  i;

        // this could use strncasecmp
        for (i = 0; i < len; i++)
        {
            // tolower may be a macro, or table lookup, requires unsigned
            const unsigned char ck = kw[i]; // always already lowercase
            if( ck == 0 )
                goto next_key; // end of table entry before end of istr

            unsigned char ci = istr[i];
            if (ci < 0x80)
            {
                ci = tolower(ci);
            }

            if (ck != ci)
                goto next_key;  // does not match
        }

        if( kw[len] == 0 )  // is end of table keyword too
            goto found;
        // partial match is not valid
next_key:
        continue;
    }
    return UMI_KEY_NOT_FOUND;  // neg value

found:
    return keyi;  // keyword UMI_KEY_ value equals table index
}


// Main umapinfo keywords.
// [WDJ] Seems like overkill for one keyword, but the comparision is buried in umi_keyword_lookup.
const char * const umi_main_keyword[ 1 ] =
{
    "map"  // TOK_map
};

const uint16_t umi_main_tok[ 1 ] =
{
    TOK_MAP
};

#define NUM_MAIN_TOK   (sizeof(umi_main_tok)/sizeof(uint16_t))

// Detects main keyword.
// Return TOK_ID when main keyword not found.
static inline
uint16_t  umi_detect_keywords( void )
{
    // Even when only 1 main keyword, need to use the case cmp with length parameter.
    int mti = umi_keyword_lookup( umi_lexer_text, umi_lexer_text_len,
                                  umi_main_keyword, NUM_MAIN_TOK );
    if( mti >= 0 )
        return umi_main_tok[mti];  // lookup TOK_xx

    return TOK_ID;  // identifier not found
}

// ==============================

// [WDJ] Umapinfo lexical scanner.
// This is a rewrite (replacement) of the FLEX lexer used with umapinfo_libdoom.
// That Flex lexer had several strange capabilities for which we have no use.
// This lexer is limited, as follows:
// * The only source of input is from DoomLegacy.  The Lexer will NOT switch files directly.
// FLEX capability of reading external files has been removed.
// The lexer source is entirely presented by DoomLegacy, usually from some lump.
// * Will not be interactive.
// * Will not automatic switch buffers.
// * Will not switch buffers in the middle of a token parsing.  Tokens must wholly be in one buffer.
// * This is parsed during wad read, and does not need the utmost speed.
//   Being easy to read, debug, and modify is more important.
// * FLEX functions replaced by DoomLegacy functions where possible.
// * Buffer stack is not necessary.  Other uses, or other users may have need of it.
// * The current state is always directly accessed, and static, independent of stack accesses.
// Access to any buffer stack would be rare, if ever done, as pushing, popping and changing buffers are exceptional events.

#ifdef ENABLE_BUFFER_STACK
// State
typedef enum {
  BFS_BUFF_empty,
  BFS_BUFF_scanning,
  BFS_BUFF_EOF_pending,
  BFS_BUFF_EOF_done,
  BFS_BUFF_deleted,
} buffer_status_e;
#endif

typedef struct umilex_buffer_state
{
    char * src_buffer;   // the source buffer being lexed, the input buffer
    char * buf_pos;   // current position in input buffer
    char * buf_end;   // after last char in input buffer 
    char * bol;  // BOL buffer position

    unsigned int num_char;  // source size in the buffer

#ifdef ENABLE_BUFFER_STACK
    // Only used when saving buffer state, pushed onto stack.
    unsigned int buf_lineno;  // current line number, in this buffer
    unsigned int buf_size_alloc; // allocated size of buffer
    byte buffer_status;  // buffer_status_e
#endif
} umilex_buffer_state_t;


static umilex_buffer_state_t   curbs;  // current buffer state
#ifdef ENABLE_BUFFER_STACK
static umilex_buffer_state_t * curbs_p = NULL; // current buffer state source
#endif

/* ======================================================================== */
#define  REPORT_BUF_LEN  20

void umi_lexer_report_line( int8_t errcode )
{
    char buf[REPORT_BUF_LEN];
    snprintf( buf, REPORT_BUF_LEN-1, "Line %d:", umi_lexer_lineno );  // most current
    buf[REPORT_BUF_LEN-1] = 0;
    umi_lexer_msg_by_len( errcode, buf, curbs.bol, curbs.buf_pos - curbs.bol );
}

#undef REPORT_BUF_LEN



#if 0
// Not used
/* ======================================================================== */
/* Create quoted string object */
static
umi_qstring_t  umi_lexer_make_qstring( const char * s, int ilen )
{
    umi_qstring_t res;

    res.octet = (const unsigned char *) s;
    // string output is limited to practical values, not max memory values.
    if( ilen > UMI_LEXER_MAX_LEN )  ilen = UMI_LEXER_MAX_LEN;
    res.len = ilen;

    return res;
}
#endif

/* ======================================================================== */
// Parser quick character classification.

// Classify char
typedef enum {
  LCC_0,
  LCC_x,  // control chars, and other unexpected junk
  LCC_EOL,
  LCC_space,  // space, TAB
  LCC_symbol,  // symbols that lexer just passes through
  LCC_token_symbol,  // symbols the lexer cares about
  LCC_digit,
  LCC_alpha,
} char_classify_e;


// indexed by src char
// returns ch_class  (char_classify_e)
static const byte lex_char_classify[128] = {
    LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, // 0..0x07
    LCC_x, // 0x08, BS
    LCC_space, // 0x09, TAB
    LCC_EOL,  // 0x0A, LF
    LCC_x, LCC_x, // 0x0B..0x0C
    LCC_EOL,  // 0x0D, CR
    LCC_x, LCC_x, // 0x0E..0x0F
    LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, // 0x10..0x17
    LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, LCC_x, // 0x18..0x1F
    LCC_space, // 0x20, space
    LCC_symbol, // 0x21
    LCC_token_symbol, // 0x22, double_quote
    LCC_symbol, LCC_symbol, LCC_symbol, LCC_symbol, // 0x23..0x26
    LCC_symbol, // 0x27, single_quote
    LCC_symbol, LCC_symbol, LCC_symbol, LCC_symbol, // 0x28..0x2B, comma
    LCC_token_symbol, // 0x2C, comma
    LCC_symbol, LCC_symbol, LCC_symbol, // 0x2D..0x2F
    LCC_digit, LCC_digit, LCC_digit, LCC_digit, LCC_digit, LCC_digit, LCC_digit, LCC_digit, LCC_digit, LCC_digit, // 0x30..0x39
    LCC_symbol, LCC_symbol, LCC_symbol, // 0x3A..0x3C,  equal
    LCC_token_symbol, // 0x3D, equal
    LCC_symbol, LCC_symbol, // 0x3E..0x3F
    LCC_symbol, // 0x40
    LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, // 0x41..0x47, A to G
    LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, // 0x48..0x4F, H to O
    LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, // 0x50..0x57, P to W
    LCC_alpha, LCC_alpha, LCC_alpha, // 0x58..0x5A, X to Z
    LCC_symbol, LCC_symbol, LCC_symbol, LCC_symbol, LCC_symbol, // 0x5B..0x5F
    LCC_symbol, // 0x60
    LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, // 0x61..0x67, a to g
    LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, // 0x68..0x6F, h to o
    LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, LCC_alpha, // 0x70..0x77, p to w
    LCC_alpha, LCC_alpha, LCC_alpha, // 0x78..0x7A, x to z
    LCC_token_symbol, // 0x7B, open_brace
    LCC_symbol, // 0x7C
    LCC_token_symbol, // 0x7D, close_brace
    LCC_symbol, // 0x7E
    LCC_x, // 0x7F, DEL
};


typedef enum {
   LG_none,
   LG_delim,
   LG_EOL,
   LG_EOB,
} lexer_get_e;


// Scan functions

// Get string from lexer buffer upto end delimiter.
//   bufp : buffer pos within curbs buffer
//   buf_end : end of curbs buffer
//   delim1 : end delimiter of string, usually double_quote
//   delim_EOL : stop at CR or LF, otherwise include line breaks in the string
// Return string in umi_lexer_string.
// Return status from lexer_get_e.
static
byte umi_lexer_scan_string( /*IN*/ char ** bufp, const char * buf_end,
                          /*IN*/ const char delim1, boolean delim_EOL )
{
    int rstat = LG_none;
    char * b_start = *bufp; // initial buffer position
    char * b = b_start;
    byte  non_inclusive_char_cnt = 0;

    while( b < buf_end )
    {
        char ch = *b++;
        if( ch == delim1 )
        {
            non_inclusive_char_cnt++;
            rstat = LG_delim;
            goto done;
        }
        if( delim_EOL && ( ch == 0x0D || ch == 0x0A ) )
        {
            // do not include CR, LF in the string
            non_inclusive_char_cnt++;
            rstat = LG_EOL;
            goto done;
        }
    }
    rstat = LG_EOB;

done:
    // Output
    umi_lexer_string.octet = (unsigned char*) b_start;
    umi_lexer_string.len = (b - b_start) - non_inclusive_char_cnt;

    // update caller
    *bufp = b;
    return rstat;
}

// [WDJ] Most of the routines do their own get_char inline.
// Avoid the parser needing "put_back", like it is the plague.
// If there is ever implemented a "put_back", then they will need a common
// get_char that looks at the "put_back" buffer first.
// A "back-up" does not require any side buffers for "put_back" and
// get_char can continue to be done inline.

#if 0
// Get one char from lexer buf.
int umi_lexer_scan_char( /*IN*/ const char ** bufp, const char * buf_end )
{
    char ch = 0;
    const char * b = *bufp;  // initial buffer position
    if( b < buf_end )
    {
        // get_char
        ch = *b++;
        if( ch > 127 )  ch = 0;

        // update caller
        *bufp = b;
    }
    return ch;
}
#endif

int umi_lexer_scan_number( /*IN*/ char ** bufp, const char * buf_end )
{
    char * b = *bufp; 
    umi_lexer_value = 0;
    while( b < buf_end )
    {
        // get_char
        unsigned char ch = *b++;
        if( ch < '0' || ch > '9' )
        {
            b--;  // something not a number, backup
            break;
        }
        umi_lexer_value = (umi_lexer_value * 10) + (ch - '0');
    }
    // update caller
    *bufp = b;
    
    return umi_lexer_value;
}



// Return token when match, otherwise 0.
uint16_t umi_lexer_scan_map_identifier( /*OUT*/ umi_mapid_t * out_mapid )
{
    uint16_t tok = 0;
    umi_mapid_t  mapid;
    char * buf = curbs.buf_pos;  // initial buffer position
    unsigned char chu;

    for(;;)  // skip space
    {
        if( buf >= curbs.buf_end )  goto fail;
        // get_char
        chu = *buf++;
        if( chu != ' ' ) break;  // end of space
    }

    if( chu > 127 )  goto fail;
    chu = toupper( chu ); // first char
    if( chu == 'M' )
    {
        char * buf1 = buf - 1;  // start of "MAP"
        // Check that will not hit EOB during strncasecmp
        buf = buf1 + 3; // strlen("map")
        if( buf >= curbs.buf_end )  goto fail;

        if( strncasecmp( buf1, "map", 3 ) == 0 )
        {
// MAP{DIGIT}+
// Examples: "MAP5", "MAP06", "MAP112"
            char * buf2 = buf;
            int map_num = umi_lexer_scan_number( &buf, curbs.buf_end );
            if( buf2 == buf )
                goto fail; // did not get a number	    
#ifdef DEBUG_UMAPINFO_LEXER
            if( EN_umi_lex_debug_out )
                 umi_lexer_debug_msg_val("Doom2 map name", map_num);
#endif
            mapid.episode = 0;
            mapid.map = map_num;
            tok = TOK_MAPNAME_DOOM2;
            goto accept;
        }
    }
    else if( chu == 'E' )
    {
// E{DIGIT}+M{DIGIT}+
// Examples: "E0M1", "E4M11"
        char * buf2 = buf;
        int epi_num = umi_lexer_scan_number( &buf, curbs.buf_end );
        if( buf2 == buf )
            goto fail; // did not get a number	    

        if( buf >= curbs.buf_end )  goto fail;
        chu = *buf++;
        if( chu > 127 )  goto fail;
        if( toupper(chu) != 'M' )  goto fail;

        char * buf3 = buf;
        int map_num = umi_lexer_scan_number( &buf, curbs.buf_end );
        if( buf3 == buf )
            goto fail; // did not get a number	    

#ifdef DEBUG_UMAPINFO_LEXER
        if( EN_umi_lex_debug_out )
            umi_lexer_debug_msg_val("Doom1 map name", epi_num*100 + map_num );
#endif
        mapid.episode = epi_num;
        mapid.map = map_num;
        tok = TOK_MAPNAME_DOOM1;
        goto accept;
    }
fail:
    // Not an error, does not match syntax.
    return 0;

    
accept:
    if( out_mapid )
        *out_mapid = mapid;  // return to caller

//    umi_token.epimap = mapid;  // through union
   
    // update lexer buffer    
    curbs.buf_pos = (char*) buf;
    
    return tok;
}


// [WDJ] main token scanner.
// Returns TOK_ value, unsigned.
int umi_lexer_get_token( void )
{
    char * cur_p = curbs.buf_pos;  // cur char position
    char * bot_p;  // begin of token position
    uint16_t tok = TOK_ERROR;  // token value
    byte ch_class;
    char ch;

retry:    
    bot_p = cur_p;  // start of token
    umi_lexer_text = bot_p;

    if( cur_p >= curbs.buf_end )
    {
        // End of buffer
        tok = TOK_EOB;
        goto tok_out;
    }

    // get_char
    ch = *cur_p++;
    if( ch > 127 )
    {
        umi_lexer_char = ch;
        tok = TOK_ERROR;
        goto tok_out;
    }

    ch_class = lex_char_classify[(unsigned char)ch];
    
    switch( ch_class )
    {
     case LCC_space:
        // one or more spaces
        // {SP}+
        goto retry;

     case LCC_EOL:
        // CR, LF
        curbs.bol = cur_p;
        if( ch == 0x0A )  umi_lexer_lineno++;  // LF
        goto retry;

     case LCC_token_symbol:
        if( ch == 0x22 )  // double_quote
            goto scan_quoted_string;

        // braces, equal, comma
        // {OBRACE}, open_brace
        // {CBRACE}, close_brace
        // <BLOCK>{EQUAL}, equal_sign
        // <BLOCK>{COMMA}
#ifdef DEBUG_UMAPINFO_LEXER
        if( EN_umi_lex_debug_out )
        {
            // The debug_msg will put quotes around the buf string.	   
            char buf[4];  // string with offending char

            buf[0] = ch;
            buf[1] = 0;
            umi_lexer_debug_msg("Symbol", buf);
        }
#endif
        tok = ch;  // the TOK_ for token_symbol matches the char value
        goto tok_out;

     case LCC_symbol:
        // unusual symbol
        umi_lexer_char = ch;
#ifdef NEG_NUMBERS
        if( ch == '-' )  // neg-number
        {
            ch = '0';
            goto scan_number;
        }
#endif       

        goto unusual_symbol_out;

     case LCC_digit:
        // Get number
        // <BLOCK>{DIGIT}+
#ifdef NEG_NUMBERS
        umi_lexer_char = '+';  // internal indicator of neg-number
#endif

scan_number:
        cur_p--;  // to get first digit of number
        umi_lexer_scan_number( &cur_p, curbs.buf_end );  // return umi_lexer_value

#ifdef NEG_NUMBERS
        if( umi_lexer_char == '-' )  // negative number
        {
            if( umi_lexer_value == 0 )  // no number, was '-' by itself
                goto unusual_symbol_out;

            umi_lexer_value = -umi_lexer_value;
        }
#endif       
#ifdef DEBUG_UMAPINFO_LEXER
        if( EN_umi_lex_debug_out )
            umi_lexer_debug_msg_val( "Number", umi_lexer_value );
#endif

#if 0
        // Special handling to use umi_token union
#ifdef NEG_NUMBERS
        umi_token.num =
            (umi_lexer_value < 0) ? 0 : (unsigned int) umi_lexer_value;
#else
        umi_token.num = (unsigned int) umi_lexer_value;
#endif
#endif

        tok = TOK_NUMBER;
        goto tok_out;

     case LCC_alpha:
        // Get identifier
        umi_lexer_text = bot_p;
        // For some reason, the umapinfo lexer will parse the identifier
        // into derivative tokens, without taking context into account.
        // Have provided scan functions that can be used by lexer, or by the parser.

        // Until encounter something not part of an identifier.
        // Umapinfo identifiers are all alpha, no digit, no underline.	??
        // Umapinfo syntax does not delineate identifier by digit or underline, either.
//        while( ch_class == LCC_alpha )
        while( ch_class == LCC_alpha || ch_class == LCC_digit || ch == '_' )
        {
            // next char
            if( cur_p >= curbs.buf_end )
                break;

            ch = *cur_p++;
            if( ch > 127 )  break;  // cannot be part of identifier
            ch_class = lex_char_classify[(unsigned char) ch];
        }
        cur_p--;  // to terminate identifier, must scan past it
        umi_lexer_text_len = cur_p - umi_lexer_text;
        tok = umi_detect_keywords(); // TOK_ID when not found

        if( tok == TOK_ID )
        {
#ifdef DEBUG_UMAPINFO_LEXER
            if( EN_umi_lex_debug_out )
                umi_lexer_debug_msg_text("Identifier");
#endif
//            umi_token.qstr = umi_lexer_make_qstring(umi_lexer_text, umi_lexer_text_len);
        }
        
        goto tok_out;

     default:
        // control char
        // CTL == 0..0x1F, 0x7F
        umi_lexer_char = ch;
        if( umi_verbose_mode )
        {
            char mbuf[4];  // string with offending char

            if( ch_class == LCC_x )
            {
                mbuf[0] = '^';
                mbuf[1] = ch ^ (char) 0x40;
            }
            else
            {
                mbuf[0] = ch;
                mbuf[1] = 0;
            }
            mbuf[2] = 0;
            umi_lexer_msg( 0, "unexpected char", mbuf);  // verbose
        }
//        goto unusual_symbol_out;
        goto retry; // ignore it
    }
    
tok_out:    
    curbs.buf_pos = cur_p;
    return tok;

unusual_symbol_out:
    tok = TOK_UNKNOWN_SYMBOL;
    goto tok_out;

scan_quoted_string:
    // <BLOCK>["]{STRING}["]
    {
        umi_lexer_text = cur_p; // for quotes, start of string
        byte dv = umi_lexer_scan_string( &cur_p, curbs.buf_end, 0x22, false );  // quotes
           // returns string in umi_lexer_string

        if( dv != LG_delim )
        {
            // delimiter not found
            int slen = curbs.buf_pos - bot_p;
            umi_lexer_msg_by_len( UMI_ERROR_SYNTAX, "unexpected end of string", bot_p, slen );
        }

#if 0
        /* The TOK_QSTRING token does not contain the delimiters */
        umi_token.qstr = umi_lexer_string;
//            umi_lexer_make_qstring(umi_lexer_text, umi_lexer_text_len);
#endif
       
#ifdef DEBUG_UMAPINFO_LEXER
        if( EN_umi_lex_debug_out )
            umi_lexer_debug_msg_qstring( "Quoted string", (char*) umi_lexer_string.octet, umi_lexer_string.len );
#endif
    }
    tok = TOK_QSTRING;
    goto tok_out;
}




// FLEX defines


#if 0
// This is covered by doomtype.h
#warning  C99 tests start

// In umapinfo lib this should be a separate file, used by all.

/* First, we deal with  platform-specific or compiler-specific issues. */

/* C99 systems have <inttypes.h>. Non-C99 systems may or may not. */

#if defined (__STDC_VERSION__) && __STDC_VERSION__ >= 199901L

/* C99 says to define __STDC_LIMIT_MACROS before including stdint.h,
 * if you want the limit (max/min) macros for int types. 
 */
#ifndef __STDC_LIMIT_MACROS
#define __STDC_LIMIT_MACROS 1
#endif

#include <inttypes.h>

#else
#endif  /* ! C99 */

#warning  C99 tests end
#endif // #if 0



// =====================

// [WDJ] BUFFER STACK is left-over from FLEX lexer.  It was never
// needed as there was only one source lump.

#ifdef ENABLE_BUFFER_STACK
void umi_flush_buffer( umilex_buffer_state_t * b);
void umi_delete_buffer( umilex_buffer_state_t * b );
void umi_switch_to_buffer( umilex_buffer_state_t * new_buffer );
void umi_push_buffer_state( umilex_buffer_state_t * new_buffer );
void umi_pop_buffer_state(void);
void umi_scan_buffer( char * base, unsigned int size );
#endif


#ifdef ENABLE_BUFFER_STACK
static
void umi_load_buffer_state(void)
{
    umi_lexer_lineno = curbs.buf_lineno;
    
    umi_lexer_text = curbs.buf_pos = curbs.buf_pos;
}

// save current state to a stack pushable, curbs_p
void save_buffer_state_to_curbs_p( int8_t enable_create )
{
    // Only make a dynamic buffer state, when one is needed for the stack.
    if( ! curbs_p && enable_create )
    {
        // Make a new pushable state.
        curbs_p = (umilex_buffer_state_t *) umi_malloc( sizeof(umilex_buffer_state_t) );
        if (! curbs_p)
            goto fatal_allocate;
    }
   
    if( curbs_p )
    {
        curbs.buf_lineno = umi_lexer_lineno;
        *curbs_p = curbs;
    }
    return;

fatal_allocate:
    umi_lexer_fatal_error( UMI_ERROR_MEMORY, "out of memory", "umi_lexer state" );
    return;  // must wind out of Lexer, detect fatal_error.
    // All error must use DoomLegacy exits.
}

/** Switch to a different input buffer. */
//  new_buffer: new input buffer.
void umi_switch_to_buffer( umilex_buffer_state_t * new_buffer )
{
    /* TODO. We should be able to replace this entire function body
     * with
     *              umi_pop_buffer_state();
     *              umi_push_buffer_state(new_buffer);
     */
    if (new_buffer == &curbs)
        return;

    if (new_buffer == curbs_p)
        return;

    // save state
    save_buffer_state_to_curbs_p( 0 );  // do not create

    // YY_CURRENT_BUFFER_LVALUE = new_buffer
    curbs_p = new_buffer;
    curbs = * new_buffer;

    umi_load_buffer_state();
}
#endif


#ifdef ENABLE_BUFFER_STACK
/** Destroy the buffer.
 * @param b a buffer created with umi_create_buffer_from_file()
 * 
 */
void umi_delete_buffer( umilex_buffer_state_t * b )
{
    if (!b)
        return;

    b->buffer_status = BFS_BUFF_deleted;
    
    if (b == curbs_p) /* Not sure if we should pop here. */
    {
        curbs_p = NULL;
        curbs.buffer_status = BFS_BUFF_deleted;
    }

    umi_free((void *) b);
}
#endif


#ifdef ENABLE_BUFFER_STACK
/** Discard all buffered characters. On the next scan, YY_INPUT will be called.
 * @param b the buffer state to be flushed, usually @c YY_CURRENT_BUFFER.
 * 
 */
void umi_flush_buffer( umilex_buffer_state_t * b )
{
    if (!b)
        return;

    b->num_char = 0;
    b->buf_pos = &b->src_buffer[0];
    b->buf_end = b->buf_pos;
    b->bol = b->buf_pos;  // is at bol

    b->buffer_status = BFS_BUFF_empty;

    if ( b == curbs_p )
        umi_load_buffer_state();
}
#endif


#ifdef ENABLE_BUFFER_STACK

/* Stack of input buffers. */
static unsigned int umilex_buffer_stack_top = 0; /* index of top of stack. */
static unsigned int umilex_buffer_stack_alloc = 0; /* capacity of stack. */
static umilex_buffer_state_t * umilex_buffer_stack = NULL;  /* Stack as an array. */

/* Allocates the stack if it does not exist.
 *  Guarantees space for at least one push.
 */
//  inc_num : additional
static
void umi_alloc_buffer_stack( unsigned int req_inc )
{
    unsigned int num_to_alloc = umilex_buffer_stack_alloc + req_inc;

    if( (umilex_buffer_stack_top + 1) >= umilex_buffer_stack_alloc )
    {
        umilex_buffer_stack = (umilex_buffer_state_t **)
           umi_realloc( umilex_buffer_stack, num_to_alloc * sizeof(umilex_buffer_state_t *) );

        if( ! umilex_buffer_stack )
            goto fatal_allocate;

        /* zero only the new slots. */
        memset( &umilex_buffer_stack[umilex_buffer_stack_alloc], 0,
               req_inc * sizeof(umilex_buffer_state_t *));

        umilex_buffer_stack_alloc = num_to_alloc;
    }
    return;

fatal_allocate;	  
    umi_lexer_fatal_error( UMI_ERROR_MEMORY, "out of memory", "umi_alloc_buffer_stack" );
    return;  // must wind out of Lexer
    // All fatal must use DoomLegacy exits.
}


/** Pushes the new state onto the stack. The new state becomes
 *  the current state. This function will allocate the stack
 *  if necessary.
 *  @param new_buffer The new state.
 *  
 */
void umi_push_buffer_state( umilex_buffer_state_t * new_buffer )
{
    if (new_buffer == NULL)
        return;

    if( (umilex_buffer_stack_top + 1) >= umilex_buffer_stack_alloc )
    {
        umi_alloc_buffer_stack( 4 );
    
        if( umi_fatal_error )  // allocate failure
            return;
    }

    // save state
    save_buffer_state_to_curbs_p( 1 );  // make curbs_p
   
    // Allowed to increment top by 1.
    if( umilex_buffer_stack )
    {
        umilex_buffer_stack[umilex_buffer_stack_top] = curbs_p;
        umilex_buffer_stack_top++;
    }

    curbs_p = new_buffer;
    curbs = * new_buffer;

    umi_load_buffer_state();
}

/** Removes and deletes the top of the stack, if present.
 *  The next element becomes the new top.
 *  
 */
void umi_pop_buffer_state(void)
{
    if( (! umilex_buffer_stack) || (umilex_buffer_stack_top == 0) )
        return;

    if( curbs_p )
        umi_delete_buffer( curbs_p );

    if( umilex_buffer_stack_top > 0 )
    {
        --umilex_buffer_stack_top;
    }

    curbs_p = umilex_buffer_stack[umilex_buffer_stack_top];
    umilex_buffer_stack[umilex_buffer_stack_top] = NULL;

    if( curbs_p )
    {
        curbs = *curbs_p;
        umi_load_buffer_state();
    }
    else
    {
        curbs.buffer_status = BFS_BUFF_deleted;
    }
}
#endif



#ifdef ENABLE_BUFFER_STACK
/** Setup the input buffer state to scan directly from a user-specified character buffer.
 * @param base the character buffer
 * @param size the size in bytes of the character buffer
 * The new scan object is curbs.
 */
void umi_scan_buffer( char * base, unsigned int base_size )
{
    // Push first if wanted to save current curbs.
    curbs.buf_size_alloc = base_size;
    curbs.num_char = base_size;
    curbs.buf_pos = curbs_p.src_buffer = base;
    curbs.bol = base;  // is at bol
    curbs.buf_end = &base[curbs_p.num_char];
    curbs.buffer_status = BFS_BUFF_empty;

    umi_load_buffer_state();

    // save state
//    save_buffer_state_to_curbs_p( 1 );  // create curbs_p
//    return curbs_p;
}
#endif

// =====================
// umi_lexer public support functions


static
int umi_lexer_init_globals(void)
{
    /* Initialization is the same as for the non-reentrant scanner.
     * This function is called from umi_lexer_destroy(), so don't allocate here.
     */
#ifdef ENABLE_BUFFER_STACK
    umilex_buffer_stack = NULL;
    umilex_buffer_stack_top = 0;
    umilex_buffer_stack_alloc = 0;
#endif
    curbs.buf_pos = NULL;
    curbs.buf_end = NULL;
    curbs.bol = NULL;

    /* For future reference: Set errno on error, since we are called by umi_lexer_init() */
    return 0;
}

/* umi_lexer_destroy is for both reentrant and non-reentrant scanners. */
void  umi_lexer_destroy(void)
{
#ifdef ENABLE_BUFFER_STACK
    /* Pop the buffer stack, destroying each element. */
    while( umilex_buffer_stack_top-- )
    {
        umilex_buffer_state_t * btop = umilex_buffer_stack[umilex_buffer_stack_top];
        umi_delete_buffer( btop );
        if( curbs_p == btop )  curbs_p = NULL;  // curbs_p might be on the stack
        umilex_buffer_stack[umilex_buffer_stack_top] = NULL;
    }

    if( curbs_p )
    {
       umi_delete_buffer( curbs_p );
       curbs_p = NULL;
    }
   
    /* Destroy the stack itself. */
    umi_free( umilex_buffer_stack );
    umilex_buffer_stack = NULL;
#endif

    umi_lexer_init_globals(); // clear state

    umi_fatal_error = 99;  // prevent usage
}


//  umapinfo_src : buffer with umapinfo source
//  src_size : size of umapinfo source
void umi_lexer_init( char * umapinfo_src, unsigned int src_size )
{
    
#ifdef DEBUG_UMAPINFO_LEXER
    EN_umi_lex_debug_out = ( umi_verbose_mode > 1 );
#endif

    umi_lexer_init_globals();

#ifdef ENABLE_BUFFER_STACK
    /*
     * Do not copy the input data. Our tokens use references to the buffer.
     *
     * Do not cast size to size_t. flex uses size_t.
     * Compiler should catch potential type mismatch for other generator.
     */
    umi_scan_buffer( umapinfo_src, src_size );
#else
    // One buffer, direct setup
    curbs.buf_pos = curbs.src_buffer = umapinfo_src;
    curbs.bol = umapinfo_src;  // is at bol
    curbs.buf_end = &umapinfo_src[src_size];
    curbs.num_char = src_size;
#endif

    umi_fatal_error = 0;
    umi_lexer_lineno = 1;
}

// =====================

// UMAPINFO end
#endif

/* EOF */

