/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*-
 *
 * Copyright (C) 2008 David Zeuthen <davidz@redhat.com>
 * Copyright (C) 2008 Kay Sievers <kay.sievers@vrfy.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#define _GNU_SOURCE 1
#define _BSD_SOURCE 1

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/utsname.h>
#include <net/if_arp.h>
#include <fcntl.h>
#include <signal.h>

#if defined(HAVE_SYS_DRVCTLIO_H) && defined(HAVE_PROP_PROPLIB_H)
#define _PATH_DRVCTL	"/dev/drvctl"
#include <machine/disklabel.h>
#include <sys/drvctlio.h>
#include <prop/proplib.h>
static char *	drvctl_subsystem_lookup (const char *);
#else
#error Platform not supported yet
#endif

#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "devkit-daemon.h"
#include "devkit-daemon-glue.h"
#include "devkit-marshal.h"

/*--------------------------------------------------------------------------------------------------------------*/

enum
{
        PROP_0,
        PROP_DAEMON_VERSION,
};

enum
{
        DEVICE_EVENT_SIGNAL,
        LAST_SIGNAL,
};

static guint signals[LAST_SIGNAL] = { 0 };

struct DevkitDaemonPrivate
{
        DBusGConnection   *system_bus_connection;
        DBusGProxy        *system_bus_proxy;

#ifdef HAVE_SYS_DRVCTLIO_H
	GIOChannel	  *drvctl_channel;
	int		  drvctl_fd;
#endif

        GList             *inhibitors;
        guint              killtimer_id;
        int                num_local_inhibitors;
        gboolean           no_exit;
};

static void     devkit_daemon_class_init  (DevkitDaemonClass *klass);
static void     devkit_daemon_init        (DevkitDaemon      *seat);
static void     devkit_daemon_finalize    (GObject     *object);

G_DEFINE_TYPE (DevkitDaemon, devkit_daemon, G_TYPE_OBJECT)

#define DEVKIT_DAEMON_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DEVKIT_TYPE_DAEMON, DevkitDaemonPrivate))

typedef struct
{
        char *cookie;
        char *system_bus_name;
} Inhibitor;

static void
inhibitor_free (Inhibitor *inhibitor)
{
        g_free (inhibitor->cookie);
        g_free (inhibitor->system_bus_name);
        g_free (inhibitor);
}

static void
inhibitor_list_changed (DevkitDaemon *daemon)
{
        devkit_daemon_reset_killtimer (daemon);
}

void
devkit_daemon_inhibit_killtimer (DevkitDaemon *daemon)
{
        daemon->priv->num_local_inhibitors++;
        devkit_daemon_reset_killtimer (daemon);
}

void
devkit_daemon_uninhibit_killtimer (DevkitDaemon *daemon)
{
        daemon->priv->num_local_inhibitors--;
        devkit_daemon_reset_killtimer (daemon);
}

void main_quit (void);

static gboolean
killtimer_do_exit (gpointer user_data)
{
        g_debug ("Exiting due to inactivity");
        main_quit ();
        return FALSE;
}

void
devkit_daemon_reset_killtimer (DevkitDaemon *daemon)
{
        /* remove existing timer */
        if (daemon->priv->killtimer_id > 0) {
                g_debug ("Removed existing killtimer");
                g_source_remove (daemon->priv->killtimer_id);
                daemon->priv->killtimer_id = 0;
        }

        /* someone on the bus is inhibiting us */
        if (g_list_length (daemon->priv->inhibitors) > 0)
                return;

        /* some local long running method is inhibiting us */
        if (daemon->priv->num_local_inhibitors > 0)
                return;

        if (daemon->priv->no_exit)
                return;

        g_debug ("Setting killtimer to 30 seconds");
        daemon->priv->killtimer_id = g_timeout_add (30 * 1000, killtimer_do_exit, NULL);
}

static DBusHandlerResult
_filter (DBusConnection *connection, DBusMessage *message, void *user_data)
{
        DevkitDaemon *daemon = DEVKIT_DAEMON (user_data);
        const char *interface;

        interface = dbus_message_get_interface (message);

        if (dbus_message_is_signal (message, DBUS_INTERFACE_DBUS, "NameOwnerChanged")) {
                GList *l;
                const char *system_bus_name;
                const char *old_service_name;
                const char *new_service_name;

                /* find the name */
		if (dbus_message_get_args (message, NULL,
                                           DBUS_TYPE_STRING, &system_bus_name,
                                           DBUS_TYPE_STRING, &old_service_name,
                                           DBUS_TYPE_STRING, &new_service_name,
                                           DBUS_TYPE_INVALID)) {

                        if (strlen (new_service_name) == 0) {
                                /* he exited.. remove from inhibitor list */
                                for (l = daemon->priv->inhibitors; l != NULL; l = l->next) {
                                        Inhibitor *inhibitor = l->data;
                                        if (strcmp (system_bus_name, inhibitor->system_bus_name) == 0) {
                                                daemon->priv->inhibitors = g_list_remove (daemon->priv->inhibitors,
                                                                                          inhibitor);
                                                inhibitor_list_changed (daemon);
                                                g_debug ("removed inhibitor %s %s (disconnected from the bus)",
                                                         inhibitor->cookie, inhibitor->system_bus_name);
                                                break;
                                        }
                                }
                        }
                }
        }

        /* other filters might want to process this message too */
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/*--------------------------------------------------------------------------------------------------------------*/

GQuark
devkit_daemon_error_quark (void)
{
        static GQuark ret = 0;

        if (ret == 0) {
                ret = g_quark_from_static_string ("devkit_daemon_error");
        }

        return ret;
}


#define ENUM_ENTRY(NAME, DESC) { NAME, "" #NAME "", DESC }

GType
devkit_daemon_error_get_type (void)
{
        static GType etype = 0;

        if (etype == 0)
        {
                static const GEnumValue values[] =
                        {
                                ENUM_ENTRY (DEVKIT_DAEMON_ERROR_GENERAL, "GeneralError"),
                                ENUM_ENTRY (DEVKIT_DAEMON_ERROR_NOT_SUPPORTED, "NotSupported"),
                                ENUM_ENTRY (DEVKIT_DAEMON_ERROR_NO_SUCH_DEVICE, "NoSuchDevice"),
                                { 0, 0, 0 }
                        };
                g_assert (DEVKIT_DAEMON_NUM_ERRORS == G_N_ELEMENTS (values) - 1);
                etype = g_enum_register_static ("DevkitDaemonError", values);
        }
        return etype;
}

static void
devkit_daemon_get_property (GObject         *object,
                            guint            prop_id,
                            GValue          *value,
                            GParamSpec      *pspec)
{
        switch (prop_id) {
        case PROP_DAEMON_VERSION:
                g_value_set_string (value, PACKAGE_VERSION);
                break;

        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
                break;
        }
}

static void
devkit_daemon_class_init (DevkitDaemonClass *klass)
{
        GObjectClass   *object_class = G_OBJECT_CLASS (klass);

        object_class->finalize = devkit_daemon_finalize;
        object_class->get_property = devkit_daemon_get_property;

        g_type_class_add_private (klass, sizeof (DevkitDaemonPrivate));

        signals[DEVICE_EVENT_SIGNAL] =
                g_signal_new ("device-event",
                              G_OBJECT_CLASS_TYPE (klass),
                              G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                              0,
                              NULL, NULL,
                              devkit_marshal_VOID__STRING_STRING_STRING_STRING_BOXED_BOXED,
                              G_TYPE_NONE,
                              6,
                              /* action */
                              G_TYPE_STRING,
                              /* subsystem */
                              G_TYPE_STRING,
                              /* native path */
                              G_TYPE_STRING,
                              /* device file */
                              G_TYPE_STRING,
                              /* device file symlinks */
                              dbus_g_type_get_collection ("GPtrArray", G_TYPE_STRING),
                              /* properties */
                              dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_STRING)
                        );


        g_object_class_install_property (object_class,
                                         PROP_DAEMON_VERSION,
                                         g_param_spec_string ("daemon-version",
                                                              "Daemon Version",
                                                              "The version of the running daemon",
                                                              NULL,
                                                              G_PARAM_READABLE));

        dbus_g_object_type_install_info (DEVKIT_TYPE_DAEMON, &dbus_glib_devkit_daemon_object_info);

        dbus_g_error_domain_register (DEVKIT_DAEMON_ERROR,
                                      NULL,
                                      DEVKIT_DAEMON_TYPE_ERROR);
}

static void
devkit_daemon_init (DevkitDaemon *daemon)
{
        daemon->priv = DEVKIT_DAEMON_GET_PRIVATE (daemon);
}

static void
devkit_daemon_finalize (GObject *object)
{
        DevkitDaemon *daemon;

        g_return_if_fail (object != NULL);
        g_return_if_fail (DEVKIT_IS_DAEMON (object));

        daemon = DEVKIT_DAEMON (object);

        g_return_if_fail (daemon->priv != NULL);

        if (daemon->priv->system_bus_proxy != NULL)
                g_object_unref (daemon->priv->system_bus_proxy);

        if (daemon->priv->system_bus_connection != NULL)
                dbus_g_connection_unref (daemon->priv->system_bus_connection);

        if (daemon->priv->inhibitors != NULL) {
                g_list_foreach (daemon->priv->inhibitors, (GFunc) inhibitor_free, NULL);
                g_list_free (daemon->priv->inhibitors);
        }

        if (daemon->priv->killtimer_id > 0) {
                g_source_remove (daemon->priv->killtimer_id);
        }

#ifdef HAVE_SYS_DRVCTLIO_H
        if (daemon->priv->drvctl_channel != NULL)
                g_io_channel_unref (daemon->priv->drvctl_channel);
	if (daemon->priv->drvctl_fd != -1)
		close (daemon->priv->drvctl_fd);
#endif

        G_OBJECT_CLASS (devkit_daemon_parent_class)->finalize (object);
}

static char *
_dupv8 (const char *s)
{
        const char *end_valid;

        if (s == NULL)
                return NULL;

        if (!g_utf8_validate (s,
                             -1,
                             &end_valid)) {
                g_warning ("The string '%s' is not valid UTF-8. Invalid characters begins at '%s'", s, end_valid);
                return g_strndup (s, end_valid - s);
        } else {
                return g_strdup (s);
        }
}

#ifdef HAVE_SYS_DRVCTLIO_H
static char *
drvctl_get_device_file (const char *driver_name, int device_unit, char *subsystem)
{
	if (strcmp (subsystem, "disks") == 0) {
		if (device_unit >= 0)
			return g_strdup_printf ("/dev/%s%u%c", driver_name, device_unit, RAW_PART + 'a');
		else
			return g_strdup_printf ("/dev/%s%c", driver_name, RAW_PART + 'a');
	} else {
		if (device_unit >= 0)
			return g_strdup_printf ("/dev/%s%u", driver_name, device_unit);
		else
			return g_strdup_printf ("/dev/%s", driver_name);
	}

}

static gboolean
receive_drvctl_data (GIOChannel *source, GIOCondition condition, gpointer user_data)
{
        DevkitDaemon *daemon = user_data;
        GPtrArray *device_file_symlinks = NULL;
        GHashTable *properties = NULL;
	prop_dictionary_t command_dict = NULL, args_dict = NULL, results_dict = NULL;
	prop_dictionary_t device_properties = NULL, ev = NULL;
	const char *event, *device;
	char *device_driver;
	const char *action;
	struct stat sb;
	char *device_file, *subsystem;
	uint16_t device_unit;
	uint8_t errorcode;
	int result;

	result = prop_dictionary_recv_ioctl (daemon->priv->drvctl_fd, DRVGETEVENT, &ev);
	if (result)
		goto out;

	prop_dictionary_get_cstring_nocopy (ev, "event", &event);
	prop_dictionary_get_cstring_nocopy (ev, "device", &device);

	g_warning ("device=%s event=%s", device, event);

	if (strcmp (event, "device-attach") == 0)
		action = "added";
	else if (strcmp (event, "device-detach") == 0)
		action = "removed";
	else
		goto out;

	command_dict = prop_dictionary_create ();
	args_dict = prop_dictionary_create ();

	prop_dictionary_set_cstring_nocopy (command_dict, "drvctl-command", "get-properties");
	prop_dictionary_set_cstring_nocopy (args_dict, "device-name", device);
	prop_dictionary_set (command_dict, "drvctl-arguments", args_dict);

	result = prop_dictionary_sendrecv_ioctl (command_dict, daemon->priv->drvctl_fd, DRVCTLCOMMAND, &results_dict);
	prop_object_release (command_dict);
	if (result)
		goto out;

	if (prop_dictionary_get_uint8 (results_dict, "drvctl-error", &errorcode) == false)
		goto out;

	device_properties = prop_dictionary_get (results_dict, "drvctl-result-data");
	if (device_properties == NULL) {
		char *ep = strpbrk (device, "0123456789");
		if (ep == NULL) {
			g_warning ("can't derive device-driver for %s", device);
			goto out;
		}
		device_driver = g_strndup (device, ep - device);
	} else
		prop_dictionary_get_cstring (device_properties, "device-driver", &device_driver);

	subsystem = drvctl_subsystem_lookup (device_driver);
        if (subsystem == NULL)
                subsystem = g_strdup ("kernel");

        properties = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
        if (properties == NULL)
                goto out;
#if notyet
        udev_list_entry_foreach (list_entry, udev_device_get_properties_list_entry (device))
               g_hash_table_insert (properties,
                                    _dupv8 (udev_list_entry_get_name (list_entry)),
                                    _dupv8 (udev_list_entry_get_value (list_entry)));
#endif

        device_file_symlinks = g_ptr_array_new ();
        if (device_file_symlinks == NULL)
                goto out;
#if notyet
        udev_list_entry_foreach (list_entry, udev_device_get_devlinks_list_entry (device))
                g_ptr_array_add (device_file_symlinks,
                                 _dupv8 (udev_list_entry_get_name (list_entry)));
#endif

	device_file = drvctl_get_device_file (device, -1, subsystem);
	if (stat(device_file, &sb) == -1) {
		g_free (device_file);
		device_file = NULL;
	}

        g_signal_emit (daemon, signals[DEVICE_EVENT_SIGNAL], 0,
                       action,
                       subsystem,
		       g_strdup (device),
		       device_file != NULL ? device_file : "",
                       device_file_symlinks,
                       properties,
                       NULL);
out:
	if (device_driver)
		g_free (device_driver);
	if (device_file)
		g_free (device_file);
	if (subsystem)
		g_free (subsystem);
	if (results_dict)
		prop_object_release (results_dict);
	if (args_dict)
		prop_object_release (args_dict);
	if (ev)
		prop_object_release (ev);
        g_ptr_array_foreach (device_file_symlinks, (GFunc) g_free, NULL);
        g_ptr_array_free (device_file_symlinks, TRUE);
        g_hash_table_destroy (properties);
        return TRUE;
}
#endif

static gboolean
register_daemon (DevkitDaemon *daemon)
{
        DBusConnection *connection;
        GError *error = NULL;
        DBusError dbus_error;

        error = NULL;
        daemon->priv->system_bus_connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
        if (daemon->priv->system_bus_connection == NULL) {
                if (error != NULL) {
                        g_critical ("error getting system bus: %s", error->message);
                        g_error_free (error);
                }
                goto error;
        }
        connection = dbus_g_connection_get_connection (daemon->priv->system_bus_connection);

        dbus_g_connection_register_g_object (daemon->priv->system_bus_connection,
                                             "/org/freedesktop/DeviceKit",
                                             G_OBJECT (daemon));

        daemon->priv->system_bus_proxy = dbus_g_proxy_new_for_name (daemon->priv->system_bus_connection,
                                                                    DBUS_SERVICE_DBUS,
                                                                    DBUS_PATH_DBUS,
                                                                    DBUS_INTERFACE_DBUS);

        /* need to listen to NameOwnerChanged */
        dbus_error_init (&dbus_error);
	dbus_bus_add_match (dbus_g_connection_get_connection (daemon->priv->system_bus_connection),
			    "type='signal'"
			    ",interface='"DBUS_INTERFACE_DBUS"'"
			    ",sender='"DBUS_SERVICE_DBUS"'"
			    ",member='NameOwnerChanged'",
			    &dbus_error);
        if (dbus_error_is_set (&dbus_error)) {
                g_warning ("Cannot add match rule: %s: %s", dbus_error.name, dbus_error.message);
                dbus_error_free (&dbus_error);
                goto error;
        }
        if (!dbus_connection_add_filter (dbus_g_connection_get_connection (daemon->priv->system_bus_connection),
                                         _filter,
                                         daemon,
                                         NULL)) {
                g_warning ("Cannot add D-Bus filter: %s: %s", dbus_error.name, dbus_error.message);
                dbus_error_free (&dbus_error);
                goto error;
        }

#ifdef HAVE_SYS_DRVCTLIO_H
	daemon->priv->drvctl_fd = open (_PATH_DRVCTL, O_RDWR);
	if (daemon->priv->drvctl_fd == -1) {
		g_warning ("Error opening drvctl device: %s", strerror (errno));
		goto error;
	}
	daemon->priv->drvctl_channel = g_io_channel_unix_new (daemon->priv->drvctl_fd);
        g_io_add_watch (daemon->priv->drvctl_channel, G_IO_IN, receive_drvctl_data, daemon);
        g_io_channel_unref (daemon->priv->drvctl_channel);
#endif

        devkit_daemon_reset_killtimer (daemon);

        return TRUE;

error:
        return FALSE;
}


DevkitDaemon *
devkit_daemon_new (gboolean no_exit)
{
        gboolean res;
        DevkitDaemon *daemon;

        daemon = DEVKIT_DAEMON (g_object_new (DEVKIT_TYPE_DAEMON, NULL));
        daemon->priv->no_exit = no_exit;

        res = register_daemon (DEVKIT_DAEMON (daemon));
        if (! res) {
                g_object_unref (daemon);
                return NULL;
        }

        return daemon;
}

/*--------------------------------------------------------------------------------------------------------------*/

#if 0
static gboolean
throw_error (DBusGMethodInvocation *context, int error_code, const char *format, ...)
{
        GError *error;
        va_list args;
        char *message;

        va_start (args, format);
        message = g_strdup_vprintf (format, args);
        va_end (args);

        error = g_error_new (DEVKIT_DAEMON_ERROR,
                             error_code,
                             message);
        dbus_g_method_return_error (context, error);
        g_error_free (error);
        g_free (message);
        return TRUE;
}
#endif

/*--------------------------------------------------------------------------------------------------------------*/
/* exported methods */

gboolean
devkit_daemon_inhibit_shutdown (DevkitDaemon     *daemon,
                                      DBusGMethodInvocation *context)
{
        GList *l;
        Inhibitor *inhibitor;
        const char *system_bus_name;

        system_bus_name = dbus_g_method_get_sender (context);

        inhibitor = g_new0 (Inhibitor, 1);
        inhibitor->system_bus_name = g_strdup (system_bus_name);
regen_cookie:
        inhibitor->cookie = g_strdup_printf ("%d", g_random_int_range (0, G_MAXINT));
        for (l = daemon->priv->inhibitors; l != NULL; l = l->next) {
                Inhibitor *i = l->data;
                if (strcmp (i->cookie, inhibitor->cookie) == 0) {
                        g_free (inhibitor->cookie);
                        goto regen_cookie;
                }
        }

        daemon->priv->inhibitors = g_list_prepend (daemon->priv->inhibitors, inhibitor);
        inhibitor_list_changed (daemon);

        g_debug ("added inhibitor %s %s", inhibitor->cookie, inhibitor->system_bus_name);

        dbus_g_method_return (context, inhibitor->cookie);
        return TRUE;
}

/*--------------------------------------------------------------------------------------------------------------*/

gboolean
devkit_daemon_uninhibit_shutdown (DevkitDaemon     *daemon,
                                        const char            *cookie,
                                        DBusGMethodInvocation *context)
{
        GList *l;
        GError *error;
        const char *system_bus_name;

        system_bus_name = dbus_g_method_get_sender (context);

        for (l = daemon->priv->inhibitors; l != NULL; l = l->next) {
                Inhibitor *inhibitor = l->data;
                if (strcmp (cookie, inhibitor->cookie) == 0 &&
                    strcmp (system_bus_name, inhibitor->system_bus_name) == 0) {

                        daemon->priv->inhibitors = g_list_remove (daemon->priv->inhibitors, inhibitor);
                        inhibitor_list_changed (daemon);

                        g_debug ("removed inhibitor %s %s", inhibitor->cookie, inhibitor->system_bus_name);
                        dbus_g_method_return (context);
                        goto out;
                }
        }

        error = g_error_new (DEVKIT_DAEMON_ERROR,
                             DEVKIT_DAEMON_ERROR_GENERAL,
                             "No such inhibitor");
        dbus_g_method_return_error (context, error);
        g_error_free (error);

out:
        return TRUE;
}

/*--------------------------------------------------------------------------------------------------------------*/

#define STRUCT_TYPE (dbus_g_type_get_struct ("GValueArray",             \
                                             G_TYPE_STRING,             \
                                             G_TYPE_STRING,             \
                                             G_TYPE_STRING,             \
                                             G_TYPE_STRV,                                        \
                                             dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_STRING), \
                                             G_TYPE_INVALID))

#ifdef HAVE_SYS_DRVCTLIO_H
static char *
drvctl_subsystem_lookup (const char *device_driver)
{
	prop_dictionary_t subsystem_map;
	const char *value = NULL;
	char *subsystem = NULL;

	subsystem_map = prop_dictionary_internalize_from_file (PACKAGE_SYSCONF_DIR "/devkitd-subsystem-map.plist");
	if (subsystem_map == NULL) {
		g_warning (PACKAGE_SYSCONF_DIR "/devkitd-subsystem-map.plist not found or invalid");
		return NULL;
	}

	prop_dictionary_get_cstring_nocopy (subsystem_map, device_driver, &value);
	if (value)
		subsystem = g_strdup (value);

	prop_object_release (subsystem_map);

	return subsystem;
}

static void
add_drvctl_device (GPtrArray *devices, prop_dictionary_t device_properties)
{
	const char *device_driver;
        char *subsystem;
        char *native_path;
        char *device_file;
	uint16_t device_unit;
        GPtrArray *device_file_symlinks;
        GHashTable *properties;
        GValue elem = {0};
	struct stat sb;

	g_return_if_fail (prop_dictionary_get_cstring_nocopy (device_properties, "device-driver", &device_driver));
	g_return_if_fail (prop_dictionary_get_uint16 (device_properties, "device-unit", &device_unit));

        subsystem = drvctl_subsystem_lookup (device_driver);
	if (subsystem == NULL)
		subsystem = g_strdup ("kernel");
        native_path = g_strdup_printf ("%s%u", device_driver, device_unit);
        g_return_if_fail (native_path != NULL);

	device_file = drvctl_get_device_file (device_driver, device_unit, subsystem);
	if (stat(device_file, &sb) == -1) {
		g_free (device_file);
		device_file = NULL;
	}
        device_file_symlinks = g_ptr_array_new ();
        g_value_init (&elem, STRUCT_TYPE);
        g_value_take_boxed (&elem, dbus_g_type_specialized_construct (STRUCT_TYPE));

        g_ptr_array_add (device_file_symlinks, NULL);

        properties = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);

        dbus_g_type_struct_set (&elem,
                                0, subsystem,
                                1, native_path,
                                2, device_file != NULL ? device_file : "",
                                3, device_file_symlinks->pdata,
                                4, properties,
                                G_MAXUINT);
        g_ptr_array_add (devices, g_value_get_boxed (&elem));

        g_ptr_array_foreach (device_file_symlinks, (GFunc) g_free, NULL);
        g_ptr_array_free (device_file_symlinks, TRUE);
        g_hash_table_destroy (properties);
	g_free (native_path);
	if (device_file != NULL)
		g_free (device_file);
	g_free (subsystem);
}

static void
drvctl_enumerate (DevkitDaemon *daemon, GPtrArray *devices, const char *device_name, const char **subsystems)
{
	char *device_subsystem;
	const char *device_driver;
	struct devlistargs laa;
	int result, children, i;
	uint8_t errorcode;
	prop_dictionary_t command_dict, args_dict, results_dict;
	prop_dictionary_t device_properties;

	g_message ("drvctl_enumerate: daemon=%p devices=%p device_name=%s subsystems=%p",
	    daemon, devices, device_name, subsystems);

	command_dict = prop_dictionary_create ();
	args_dict = prop_dictionary_create ();

	prop_dictionary_set_cstring_nocopy (command_dict, "drvctl-command", "get-properties");
	prop_dictionary_set_cstring_nocopy (args_dict, "device-name", device_name);
	prop_dictionary_set (command_dict, "drvctl-arguments", args_dict);
	prop_object_release (args_dict);

	result = prop_dictionary_sendrecv_ioctl (command_dict, daemon->priv->drvctl_fd, DRVCTLCOMMAND, &results_dict);
	prop_object_release (command_dict);
	if (result)
		return;

	if (prop_dictionary_get_int8 (results_dict, "drvctl-error", &errorcode) == false) {
		prop_object_release (results_dict);
		return;
	}

	device_properties = prop_dictionary_get (results_dict, "drvctl-result-data");
	if (device_properties == NULL)
		return;

	if (prop_dictionary_get_cstring_nocopy (device_properties, "device-driver", &device_driver) == false) {
		prop_object_release (device_properties);
		prop_object_release (results_dict);
		return;
	}

	device_subsystem = drvctl_subsystem_lookup (device_driver);
	if (subsystems != NULL && subsystems[0] != NULL) {
		if (device_subsystem == NULL) {
			prop_object_release (device_properties);
			prop_object_release (results_dict);
			return;
		}
		for (i = 0; subsystems[i] != NULL; i++)
			if (strcmp(subsystems[i], device_subsystem) == 0)
				break;
		if (subsystems[i] == NULL) {
			prop_object_release (device_properties);
			prop_object_release (results_dict);
			return;
		}
	}

	add_drvctl_device (devices, device_properties);
	prop_object_release (device_properties);
	//prop_object_release (results_dict);
	if (device_subsystem != NULL)
		g_free (device_subsystem);

	memset (&laa, 0, sizeof (laa));
	strlcpy (laa.l_devname, device_name, sizeof (laa.l_devname));

	result = ioctl (daemon->priv->drvctl_fd, DRVLISTDEV, &laa);
	if (result == -1 || laa.l_children == 0)
		return;

	laa.l_childname = calloc (1, laa.l_children * sizeof (laa.l_childname[0]));
	if (laa.l_childname == NULL) {
		g_warning ("couldn't allocate %d children for DRVLISTDEV", laa.l_children);
		return;
	}

	children = laa.l_children;
	result = ioctl (daemon->priv->drvctl_fd, DRVLISTDEV, &laa);
	if (result == -1)
		return;
	if (laa.l_children != children) {
		g_warning ("expected %d children from DRVLISTDEV, got %d", children, laa.l_children);
		return;
	}

	for (i = 0; i < laa.l_children; i++)
		drvctl_enumerate (daemon, devices, laa.l_childname[i], subsystems);

	free (laa.l_childname);
}


gboolean
devkit_daemon_enumerate_by_subsystem  (DevkitDaemon          *daemon,
                                       const char           **subsystems,
                                       DBusGMethodInvocation *context)
{
        GPtrArray *devices;
        int n;

        devices = dbus_g_type_specialized_construct (dbus_g_type_get_collection ("GPtrArray", STRUCT_TYPE));

	drvctl_enumerate (daemon, devices, "mainbus0", subsystems);

        dbus_g_method_return (context, devices);
        return TRUE;
}

gboolean
devkit_daemon_enumerate_by_native_path (DevkitDaemon          *daemon,
                                        const char           **native_paths,
                                        DBusGMethodInvocation *context)
{
        int n;
        GPtrArray *devices;

        devices = dbus_g_type_specialized_construct (dbus_g_type_get_collection ("GPtrArray", STRUCT_TYPE));

	for (n = 0; native_paths[n] != NULL; n++)
		drvctl_enumerate (daemon, devices, native_paths[n], NULL);

        dbus_g_method_return (context, devices);
        return TRUE;
}
#endif

/*--------------------------------------------------------------------------------------------------------------*/
