/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: depot.c,v 5.11 1994/11/01 15:56:39 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */


#include "depotlib.h"
#include "util.h"
#include "DepotErrorCodes.h"
#include "Lock.h"
#include "File.h"
#include "Hint.h"
#include "FileSystemImage.h"
#include "Preference.h"
#include "MasterDepotDB.h"
#include "HintService.h"
#include "config.h"
#include "Depot.h"

static char *Depot_TargetPath = DEPOT_DEFAULTTARGETPATH;
static char *Depot_ImagePath = NULL;
static unsigned Depot_UpdateMode = M_NULL;	/* options -n, -v & lock
						 * options */
static int Depot_LockAction = L_NULL;	/* options -l, L, u, U */

static void Usage();
static void HandleArgs();
static void Depot_CheckOptionsConsistency();

static char VersionString[] = "depot: $Revision: 5.11 $ $Date: 1994/11/01 15:56:39 $";
static char *UsageString = "Usage: depot [-LUhilnuvVQx] [-T targetdir] ";

static char *
DepotLongOptionList[] =
{
  "[--usemodtimes=true|false] ",
  "[--keepcopiedfileattribues=true|false]",
  "[--compresstargetdb=true|false]"
};



static struct DepotOption {
  char *option;
  char *description;
} DepotOptionList[] =
{
  {
    "-L", "Lock, by force if necessary"
  },
  {
    "-U", "Unlock, by force if necessary"
  },
  {
    "-h", "print Help message"
  },
  {
    "-i", "locking not required by owner of lock (Implicit locking)"
  },
  {
    "-l", "Lock"
  },
  {
    "-n", "pretend to run, show update actions without actually updating"
  },
  {
    "-u", "Unlock"
  },
  {
    "-v", "Verbose output, show update actions"
  },
  {
    "-V", "Extremely Verbose output; show processing"
  },
  {
    "-x", "print Help message"
  },
  {
    "-Q", "Quick update mode; use only after depot.db has been built"
  },
  {
    "-T targetdir", "use targetdir as target directory for depot"
  }
};

static void 
Usage(fp)
     FILE *fp;
{
  register unsigned i;
  register char **lop;
  register struct DepotOption *op;
  unsigned NDepotOptions =
    sizeof(DepotOptionList) / sizeof(DepotOptionList[0]);
  unsigned NDepotLongOptions =
    sizeof(DepotLongOptionList) / sizeof(char *);

  (void) fprintf(fp, "%s\n", VersionString);
  (void) fprintf(fp, "%s\n", getversion_libdepot());
  (void) fprintf(fp, "%s\n", UsageString);
  for (i = 0, lop = DepotLongOptionList; i < NDepotLongOptions; i++, lop++) {
    (void) fprintf(fp, "\t\t%s\n", *lop);
  }
  (void) fprintf(fp, "\n");
  for (i = 0, op = DepotOptionList; i < NDepotOptions; i++, op++) {
    (void) fprintf(fp, "  %-12s %s\n", op->option, op->description);
  }
  (void) fflush(fp);

  return;
}


int
main(argc, argv)
     int argc;
     char **argv;
{
  time_t now = 0;
  HandleArgs(argc, argv);

  if (PROGRAM_Verbose) {
    now = time(NULL);
    printf("Depot starting on %s", ctime(&now));
  }
  (void) Depot(Depot_TargetPath,

	       Depot_ImagePath,
	       Depot_UpdateMode,
	       Depot_LockAction);


  if (PROGRAM_Verbose) {
    now = time(NULL);
    if (PROGRAM_ErrorNo == E_NULL) {
      printf("\nDepot finished normally on %s", ctime(&now));
    } else {
      printf("\nDepot exited with error on %s", ctime(&now));
    }
  }
  exit(PROGRAM_ErrorNo);
}



static void 
HandleArgs(argc, argv)
     int argc;
     char **argv;
{
  register char *cp;
  Boolean eooptions;

  argv++, argc--;
  eooptions = FALSE;
  PROGRAM_Verbose = 0;

  while (!eooptions && (argc > 0)) {
    cp = *argv;
    if (*cp != '-') {
      eooptions = TRUE;
    } else {
      while (*(++cp) != '\0') {
	switch (*cp) {
	case 'T':
	  argv++, argc--;
	  if (argc <= 0) {
	    FatalError(E_NULLTARGET, "No Target Specified\n");
	  } else {
	    Depot_TargetPath = String(*argv, strlen(*argv));
	  }
	  break;
	case 'I':
	  argv++, argc--;
	  if (argc <= 0) {
	    FatalError(E_NULLTARGET, "No Image dir Specified\n");
	  } else {
	    Depot_ImagePath = String(*argv, strlen(*argv));
	  }
	  fprintf(stderr, "WARNING: the use of -I is ONLY for testing purposes. Other usage may\n");
	  fprintf(stderr, "         result in depot (and subsequently you) getting confused.\n");
	  break;
	case 'L':
	  Depot_LockAction = L_FORCELOCK;
	  Depot_UpdateMode |= M_LOCKONLY;
	  break;
	case 'U':
	  Depot_LockAction = L_FORCEUNLOCK;
	  Depot_UpdateMode |= M_LOCKONLY;
	  break;
	case 'l':
	  Depot_LockAction = L_LOCK;
	  Depot_UpdateMode |= M_LOCKONLY;
	  break;
	case 'u':
	  Depot_LockAction = L_UNLOCK;
	  Depot_UpdateMode |= M_LOCKONLY;
	  break;
	case 'n':
	  Depot_UpdateMode |= M_SHOWACTIONSONLY;
	  break;
	case 'v':
	  Depot_UpdateMode |= M_VERBOSE;
	  PROGRAM_Verbose++;
	  break;
	case 'V':
	  Depot_UpdateMode |= M_VERBOSE;
	  PROGRAM_Verbose = PROGRAM_TRACEVAL;
	  break;
	case 'i':
	  Depot_UpdateMode |= M_IMPLICITLOCKINGALLOWED;
	  break;
	case 'Q':
	  Depot_UpdateMode |= M_DELTA;
	  break;
	case 'h':
	case 'x':
	  Usage(stdout);
	  exit(E_USAGE);
	  break;
	case '-': /* process long options */
	  {
	    char *buf;
	    char *ans;
	    Boolean value;
	    cp++;

	    buf = strrchr(cp, '=');
	    if (buf == NULL) {
	      FatalError(E_USAGE, "No = found in processing long command option: %s\n",cp);
	    } 
	    if (PROGRAM_ErrorNo == E_NULL) {
	      *buf = '\0';
	      ans = buf+1;
	      if (*ans == '\0') {
		FatalError(E_USAGE, "No option specified after = mark in command string: %s\n",
			   cp);
	      }
	    }
	      
	    if (PROGRAM_ErrorNo == E_NULL) {
	      if (*ans == 'T' || *ans == 't') {
		value = TRUE;
	      } else if ((*ans == 'F') || (*ans == 'f')) {
		value = FALSE;
	      } else {
		FatalError(E_USAGE, "Please specify 'true' or 'false' after the = sign.\n");
	      }
	    }
	    
	    if (PROGRAM_ErrorNo == E_NULL) {
	      if (strcmp(cp, "usemodtimes") == 0) {
		Depot_UpdateMode |= (value) ? M_USEMODTIMES : M_NOMODTIMES;
	      } else if (strcmp(cp, "keepcopiedfileattributes") == 0) {
		Depot_UpdateMode |= (value) ? M_KEEPATTRIB : M_DITCHATTRIB;
	      } else if (strcmp(cp, "compresstargetdb") == 0) {
		Depot_UpdateMode |= (value) ? M_COMPRESSTDB : M_NOCOMPRESSTDB;
	      } else {
		FatalError(E_USAGE, "Invalid long option name: %s\n", cp);
	      }
	    }
	  }
	  *(cp+1) = '\0';
	  break;
	default:
	  FatalError(E_USAGE, "Bad command line option %c\n", *cp);
	  break;
	}
      }
      argv++, argc--;
    }
  }

  if (PROGRAM_ErrorNo != E_NULL) {	/* probably a usage error */
    Usage(stderr);
  }
  Depot_CheckOptionsConsistency();
  if (argc > 0) {
    FatalError(E_USAGE, "Too many command line options.\n");
    Usage(stderr);
  }
  if (PROGRAM_ErrorNo != E_NULL)
    exit(E_USAGE);

  return;
}


static void 
Depot_CheckOptionsConsistency()
{
  if ((Depot_UpdateMode & M_USEMODTIMES)
      && (Depot_UpdateMode & M_NOMODTIMES)) {
    FatalError(E_USAGE, "Cannot specify both 'true' and 'false' to usemodtimes!\n");
  }

  if ((Depot_UpdateMode & M_KEEPATTRIB)
      && (Depot_UpdateMode & M_DITCHATTRIB)) {
    FatalError(E_USAGE, "Cannot specify both 'true' and 'false' to keepcopiedfileattribues!\n");
  }

  if ((Depot_UpdateMode & M_COMPRESSTDB)
      && (Depot_UpdateMode &  M_NOCOMPRESSTDB)) {
    FatalError(E_USAGE, "Cannot specify both 'true' and 'false' to compresstargetdb!\n");
  }

  return;
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/022/src/cmd/depot/RCS/depot.c,v $ */
