import './style.scss';

import classNames from 'classnames';
import { isUndefined } from 'lodash-es';
import {
  type ComponentPropsWithoutRef,
  type ReactNode,
  useEffect,
  useMemo,
  useState,
} from 'react';

import SvgIconInfo from '../../svg/IconInfo';
import SvgIconInfoSuccess from '../../svg/IconInfoSuccess';
import SvgIconWarning from '../../svg/IconWarning';
import SvgIconX from '../../svg/IconX';
import { MessageBoxType } from './types';
import { readMessageBoxVisibility, writeMessageBoxVisibility } from './utils';

interface Props extends ComponentPropsWithoutRef<'div'> {
  message?: string | ReactNode;
  type?: MessageBoxType;
  dismissId?: string;
}

/**
 * Styled box with message.
 */
export const MessageBoxOld = ({
  message,
  className,
  dismissId,
  type = MessageBoxType.INFO,
  ...props
}: Props) => {
  const [visible, setVisible] = useState(true);

  const dismissible = !isUndefined(dismissId);

  const getClassName = useMemo(() => {
    return classNames('message-box-old', className, type.valueOf());
  }, [className, type]);

  const getIcon = useMemo(() => {
    switch (type) {
      case MessageBoxType.INFO:
        return <SvgIconInfo />;
      case MessageBoxType.SUCCESS:
        return <SvgIconInfoSuccess />;
      case MessageBoxType.WARNING:
        return <SvgIconInfo />;
      case MessageBoxType.ERROR:
        return <SvgIconWarning />;
    }
  }, [type]);

  const renderMessage = useMemo(() => {
    if (typeof message === 'string') {
      return <p>{message}</p>;
    }
    return message;
  }, [message]);

  useEffect(() => {
    if (dismissId?.length) {
      const visibility = readMessageBoxVisibility(dismissId);
      if (visible !== visibility) {
        setVisible(visibility);
      }
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [dismissId, visible]);

  if (!visible) return null;

  return (
    <div className={getClassName} {...props}>
      <div className="icon-container">{getIcon}</div>
      <div className="message">{renderMessage}</div>
      {dismissible && (
        <button
          className="dismiss"
          onClick={(e) => {
            e.preventDefault();
            e.stopPropagation();
            writeMessageBoxVisibility(dismissId);
            setVisible(false);
          }}
        >
          <SvgIconX />
        </button>
      )}
    </div>
  );
};
