/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : lfo.cc
|
| Object      : DSPKitLFO
|
| Description : DSPKitLFO
|
| (c) Richard Kent 1996
|
| $Id: lfo.cc,v 1.1 2003/09/10 00:06:25 rk Exp $
|
****************************************************************************/

static char lfo_cc [] = "$Id: lfo.cc,v 1.1 2003/09/10 00:06:25 rk Exp $";

#include "lfo.h"

int DSPKitLFO::setTypeAndFreq (
  int newType,
  double newInputFreq,
  double newOutputFreq, 
  double newDepth,
  double newPhase)
{
  depth                = newDepth;
  phase                = 0;
  inputFreq            = newInputFreq;
  outputFreq           = newOutputFreq;
  type                 = newType;
  samplesPerCycle      = (int) (inputFreq / outputFreq);
  cyclesPerSample      = 1.0 / samplesPerCycle;
  twoPiCyclesPerSample = 8.0 * atan (1.0) / samplesPerCycle;
  expPower = 4.0;
  
  while (newPhase < 0.0)   newPhase += 360.0;
  while (newPhase > 360.0) newPhase -= 360.0;
  phase = (int) samplesPerCycle * newPhase / 360.0;
  i = (int) phase;
  return 1;
}

int DSPKitLFO::getSample (double &outputSample)
{
  // type 0 = sine
  // type 1 = triangle
  // type 2 = exponential 1
  // type 3 = exponential 2
  // type 4 = square
  
  double x;
  
  switch (type)
  {
    case 0 :
    {
      outputSample = sin (i * twoPiCyclesPerSample);
      break;
    }
    
    case 1 :
    {
      x = i * cyclesPerSample;
      if (x < 0.0) x += 1.0;
      else if (x > 1.0) x -= 1.0;
      if (x < 0.25)
        outputSample = 4.0 * x;
      else if (x > 0.75)
        outputSample = 4.0 * (-1.0 + x);
      else
        outputSample = 4.0 * (0.5 - x);
      break;
    }
    
    case 2 :
    {
      x = i * cyclesPerSample;
      if (x < 0.0) x += 1.0;
      else if (x > 1.0) x -= 1.0;
      if (x < 0.25)
        outputSample = 1.0 - pow (4.0 * (0.25 - x),expPower);
      else if (x > 0.75)
        outputSample = -1.0 + pow (4.0 * (x - 0.75),expPower);
      else if (x < 0.5)
        outputSample = 1.0 - pow (4.0 * (x - 0.25),expPower);
      else
        outputSample = -1.0 + pow (4.0 * (0.75 - x),expPower);
      break;
    }
    
    case 3 :
    {
      x = i * cyclesPerSample;
      if (x < 0.0) x += 1.0;
      else if (x > 1.0) x -= 1.0;
      if (x < 0.25)
        outputSample = pow (4.0 * x,expPower);
      else if (x > 0.75)
        outputSample = -pow (4.0 * (1.0 - x),expPower);
      else if (x < 0.5)
        outputSample = pow (4.0 * (0.5 - x),expPower);
      else
        outputSample = -pow (4.0 * (x - 0.5),expPower);
      break;
    }
    
    case 4 :
    {
      x = i * cyclesPerSample;
      if (x < 0.0) x += 1.0;
      else if (x > 1.0) x -= 1.0;
      if (x < 0.5)
        outputSample = 1.0;
      else
        outputSample = -1.0;
      break;
    }
    
    default :
      return 0;
  }
  
  outputSample *= depth;
  i++;
  if (i >= samplesPerCycle) i = 0;
  return 1;
}

/***************************************************************************/
