/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPTich_prefs.cc
|
| Object      : None
|
| Description : Preferences code file
|
| (c) Richard Kent 1997
|
| $Id: DPTich_prefs.cc,v 1.1 2003/09/10 00:06:24 rk Exp $
|
****************************************************************************/

static char DPTich_prefs_cc [] = "$Id: DPTich_prefs.cc,v 1.1 2003/09/10 00:06:24 rk Exp $";

#include "DPTich_prefs.h"

int localPrefsUsed = FALSE;

/*---------------------------------------------------------------------------
| FUNCTION loadPreferences
---------------------------------------------------------------------------*/

int loadPreferences ()
{
  char *directory = 0;
  char *filename  = 0;
  FILE *file      = 0;
  int  prefsFound;
  
  prefsFound = FALSE;

  directory = getenv ("HOME");

  if (!localPrefsUsed && directory)
  {
    filename = new char [strlen (directory) + strlen (PREFSNAME) + 2];
    strcpy (filename,directory);
    if (strlen (filename) > 0 && filename [strlen (filename) - 1] != '/')
      strcat (filename,"/");
    strcat (filename,PREFSNAME);
    file = fopen (filename,"r");
    if (file)
      prefsFound = TRUE;
    else
      delete [] filename;
  }
  
  if (!prefsFound)
  {
    directory = ".";
    filename = new char [strlen (directory) + strlen (PREFSNAME) + 2];
    strcpy (filename,directory);
    if (strlen (filename) > 0 && filename [strlen (filename) - 1] != '/')
      strcat (filename,"/");
    strcat (filename,PREFSNAME);
    file = fopen (filename,"r");
    if (file)
    {
      prefsFound = TRUE;
      localPrefsUsed = TRUE;
    }
    else
      delete [] filename;
  }

  if (!prefsFound) return FALSE;
  
  // Read in prefs (into mainForm->sample)

  const int  tempStringLen = 128;
  const char delimiters [] = " \t\n";
  const char textDelimiters [] = "\n";
  char tempString [tempStringLen];
  char *token;
  
  while (readLine (file,tempString,tempStringLen))
  {
    if (!strlen (tempString)) continue;
    token = strtok (tempString,delimiters);
    if (!token) continue;
    if (!strcasecmp (token,"drawmode:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"dots"))
          fl_set_sample_drawmode (mainForm->sample,0);
        else if (!strcasecmp (token,"lines"))
          fl_set_sample_drawmode (mainForm->sample,1);
        else if (!strcasecmp (token,"filled"))
          fl_set_sample_drawmode (mainForm->sample,2);
      }
    }
    if (!strcasecmp (token,"drawaccuracy:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"quick"))
          fl_set_sample_drawaccuracy (mainForm->sample,0);
        else if (!strcasecmp (token,"accurate"))
          fl_set_sample_drawaccuracy (mainForm->sample,1);
      }
    }
    else if (!strcasecmp (token,"axismode:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"axis"))
          fl_set_sample_axis (mainForm->sample,1);
        else if (!strcasecmp (token,"noaxis"))
          fl_set_sample_axis (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"axisscale:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"frames"))
          fl_set_sample_frames (mainForm->sample,1);
        else if (!strcasecmp (token,"time"))
          fl_set_sample_frames (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"undobuffer:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          fl_set_sample_undomode (mainForm->sample,1);
        else if (!strcasecmp (token,"off"))
          fl_set_sample_undomode (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"autowindows:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          fl_set_sample_autowindows (mainForm->sample,1);
        else if (!strcasecmp (token,"off"))
          fl_set_sample_autowindows (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"autoglobals:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          fl_set_sample_autoglobals (mainForm->sample,1);
        else if (!strcasecmp (token,"off"))
          fl_set_sample_autoglobals (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"autoeditmode:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          fl_set_sample_autoeditmode (mainForm->sample,1);
        else if (!strcasecmp (token,"off"))
          fl_set_sample_autoeditmode (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"copyrightsave:"))
    {
      // backward compatible - use of copyright save deprecated
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          fl_set_sample_soxcompatible (mainForm->sample,0);
        else if (!strcasecmp (token,"off"))
          fl_set_sample_soxcompatible (mainForm->sample,1);
      }
    }
    else if (!strcasecmp (token,"soxcompatible:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          fl_set_sample_soxcompatible (mainForm->sample,1);
        else if (!strcasecmp (token,"off"))
          fl_set_sample_soxcompatible (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"tooltips:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          fl_set_sample_tooltips (mainForm->sample,1);
        else if (!strcasecmp (token,"off"))
          fl_set_sample_tooltips (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"tempdir:"))
    {
      token = strtok (NULL,textDelimiters);
      if (token)
      {
        int firstNonSpace = findNonSpace (token);
        fl_set_sample_tempdir (mainForm->sample,token + firstNonSpace);
      }
    }
    else if (!strcasecmp (token,"normalidle:"))
    {
      token = strtok (NULL,textDelimiters);
      if (token)
      {
        int firstNonSpace = findNonSpace (token);
        normalIdle = atoi (token + firstNonSpace);
        
        // Assume we are currently in normal idle mode
        fl_set_idle_delta (normalIdle);
      }
    }
    else if (!strcasecmp (token,"heavyidle:"))
    {
      token = strtok (NULL,textDelimiters);
      if (token)
      {
        int firstNonSpace = findNonSpace (token);
        heavyIdle = atoi (token + firstNonSpace);
      }
    }
    else if (!strcasecmp (token,"rangealternative:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          fl_set_sample_rangealt (mainForm->sample,1);
        else if (!strcasecmp (token,"off"))
          fl_set_sample_rangealt (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"rangecrossover:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          fl_set_sample_rangecross (mainForm->sample,1);
        else if (!strcasecmp (token,"off"))
          fl_set_sample_rangecross (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"verticalcontrols:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          switchVertControls (TRUE);
        else if (!strcasecmp (token,"off"))
          switchVertControls (FALSE);
      }
    }
    else if (!strcasecmp (token,"inputtextsize:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"small"))
          switchTextSize (FL_SMALL_SIZE);
        else if (!strcasecmp (token,"tiny"))
          switchTextSize (FL_TINY_SIZE);
      }
    }
    else if (!strcasecmp (token,"borderwidth:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        int firstNonSpace = findNonSpace (token);
        changeBorderWidth (atoi (token + firstNonSpace));
      }
    }
    else if (!strcasecmp (token,"basicwavs:"))
    {
      token = strtok (NULL,delimiters);
      if (token)
      {
        if (!strcasecmp (token,"on"))
          fl_set_sample_basicwavs (mainForm->sample,1);
        else if (!strcasecmp (token,"off"))
          fl_set_sample_basicwavs (mainForm->sample,0);
      }
    }
    else if (!strcasecmp (token,"audiodev:"))
    {
      token = strtok (NULL,textDelimiters);
      if (token)
      {
        int firstNonSpace = findNonSpace (token);
        fl_set_sample_audio_device (mainForm->sample,token + firstNonSpace);
      }
    }
    else if (!strcasecmp (token,"mixerdev:"))
    {
      token = strtok (NULL,textDelimiters);
      if (token)
      {
        int firstNonSpace = findNonSpace (token);
        fl_set_sample_mixer_device (mainForm->sample,token + firstNonSpace);
      }
    }
  }
  
  fclose (file);
  delete [] filename;
  updatePrefsForm ();
  updateSamplePrefs ();
  preferencesChanged = FALSE;
  return TRUE;
}  
  
/*---------------------------------------------------------------------------
| FUNCTION savePreferences
---------------------------------------------------------------------------*/

int savePreferences ()
{
  char *directory = 0;
  char *filename  = 0;
  FILE *file      = 0;
  int  prefsFound;
  
  prefsFound = FALSE;

  directory = getenv ("HOME");

  if (!localPrefsUsed && directory)
  {
    filename = new char [strlen (directory) + strlen (PREFSNAME) + 2];
    strcpy (filename,directory);
    if (strlen (filename) > 0 && filename [strlen (filename) - 1] != '/')
      strcat (filename,"/");
    strcat (filename,PREFSNAME);
    file = fopen (filename,"w");
    if (file)
      prefsFound = TRUE;
    else
      delete [] filename;
  }
  
  if (!prefsFound)
  {
    directory = "";
    filename = new char [strlen (directory) + strlen (PREFSNAME) + 2];
    strcpy (filename,directory);
    if (strlen (filename) > 0 && filename [strlen (filename) - 1] != '/')
      strcat (filename,"/");
    strcat (filename,PREFSNAME);
    file = fopen (filename,"w");
    if (file)
    {
      prefsFound = TRUE;
      localPrefsUsed = TRUE;
    }
    else
      delete [] filename;
  }

  if (!prefsFound) return FALSE;
  
  // Write out prefs (from mainForm->sample)

  int item = fl_get_sample_drawmode (mainForm->sample);
  
  switch (item)
  {
    case 0 :
      fprintf (file,"drawmode: dots\n"); 
      break;
    case 1 :
      fprintf (file,"drawmode: lines\n"); 
      break;
    case 2 :
      fprintf (file,"drawmode: filled\n"); 
      break;
    default :
      break;
  }

  item = fl_get_sample_drawaccuracy (mainForm->sample);
  
  switch (item)
  {
    case 0 :
      fprintf (file,"drawaccuracy: quick\n"); 
      break;
    case 1 :
      fprintf (file,"drawaccuracy: accurate\n"); 
      break;
    default :
      break;
  }

  item=fl_get_sample_axis (mainForm->sample);
  
  if (item)
    fprintf (file,"axismode: axis\n"); 
  else
    fprintf (file,"axismode: noaxis\n"); 

  item=fl_get_sample_frames (mainForm->sample);
  
  if (item)
    fprintf (file,"axisscale: frames\n"); 
  else
    fprintf (file,"axisscale: time\n"); 

  item=fl_get_sample_undomode (mainForm->sample);
  
  if (item)
    fprintf (file,"undobuffer: on\n"); 
  else
    fprintf (file,"undobuffer: off\n"); 

  item=fl_get_sample_autowindows (mainForm->sample);
  
  if (item)
    fprintf (file,"autowindows: on\n"); 
  else
    fprintf (file,"autowindows: off\n"); 

  item=fl_get_sample_autoglobals (mainForm->sample);
  
  if (item)
    fprintf (file,"autoglobals: on\n"); 
  else
    fprintf (file,"autoglobals: off\n"); 

  item=fl_get_sample_autoeditmode (mainForm->sample);
  
  if (item)
    fprintf (file,"autoeditmode: on\n"); 
  else
    fprintf (file,"autoeditmode: off\n"); 

  item=fl_get_sample_soxcompatible (mainForm->sample);
  
  if (item)
    fprintf (file,"soxcompatible: on\n"); 
  else
    fprintf (file,"soxcompatible: off\n"); 

  item=fl_get_sample_tooltips (mainForm->sample);
  
  if (item)
    fprintf (file,"tooltips: on\n"); 
  else
    fprintf (file,"tooltips: off\n"); 

  const char *textItem = fl_get_sample_tempdir (mainForm->sample);
  
  if (textItem)
    fprintf (file,"tempdir: %s\n",textItem);

  fprintf (file,"normalidle: %d\n",normalIdle);

  fprintf (file,"heavyidle: %d\n",heavyIdle);

  item=fl_get_sample_rangealt (mainForm->sample);
  
  if (item)
    fprintf (file,"rangealternative: on\n"); 
  else
    fprintf (file,"rangealternative: off\n"); 

  item=fl_get_sample_rangecross (mainForm->sample);
  
  if (item)
    fprintf (file,"rangecrossover: on\n"); 
  else
    fprintf (file,"rangecrossover: off\n"); 

  item=fl_get_sample_vertscroll (mainForm->sample);
  
  if (item)
    fprintf (file,"verticalcontrols: on\n"); 
  else
    fprintf (file,"verticalcontrols: off\n"); 

  item=fl_get_sample_inputtextsize (mainForm->sample);
  
  if (item == FL_SMALL_SIZE)
    fprintf (file,"inputtextsize: small\n"); 
  else /* Assume tiny */
    fprintf (file,"inputtextsize: tiny\n"); 

  item=fl_get_sample_borderwidth (mainForm->sample);
  
  fprintf (file,"borderwidth: %d\n",item);
  
  item=fl_get_sample_basicwavs (mainForm->sample);
  
  if (item)
    fprintf (file,"basicwavs: on\n"); 
  else
    fprintf (file,"basicwavs: off\n"); 
  
  textItem = fl_get_sample_audio_device (mainForm->sample);
  
  if (textItem)
    fprintf (file,"audiodev: %s\n",textItem);

  textItem = fl_get_sample_mixer_device (mainForm->sample);
  
  if (textItem)
    fprintf (file,"mixerdev: %s\n",textItem);

  fclose (file);
  delete [] filename;
  preferencesChanged = FALSE;
  return TRUE;
}  
  
/*---------------------------------------------------------------------------
| FUNCTION readLine
---------------------------------------------------------------------------*/

int readLine (FILE *file,char *buffer,int bufferLen)
{
  int i=0;
  int c;

  if (!buffer || bufferLen < 1) return FALSE;
  c = getc (file);
  if (c == EOF) return FALSE;
  while (c != EOF && c != '\n')
  {
    if (i < bufferLen - 1)
      buffer [i++] = c;
    c = getc (file);
  }
  buffer [i] = 0;
  return TRUE;
}

/*---------------------------------------------------------------------------
| CALLBACK prefsEmpty_cb
---------------------------------------------------------------------------*/

void prefsEmpty_cb (FL_OBJECT *,long)
{
}

/*---------------------------------------------------------------------------
| CALLBACK prefsOK_cb
---------------------------------------------------------------------------*/

void prefsOK_cb (FL_OBJECT *,long)
{
  fl_hide_form (prefsForm->prefsForm);
  updateSamplePrefs ();
  updatePrefsForm ();
  preferencesChanged = TRUE;
}

/*---------------------------------------------------------------------------
| CALLBACK prefsCancel_cb
---------------------------------------------------------------------------*/

void prefsCancel_cb (FL_OBJECT *,long)
{
  fl_hide_form (prefsForm->prefsForm);
  updatePrefsForm ();
}

/*---------------------------------------------------------------------------
| FUNCTION updatePrefsForm
---------------------------------------------------------------------------*/

void updatePrefsForm (void)
{
  int item = fl_get_sample_drawmode (mainForm->sample);
  
  switch (item)
  {
    case 0 :
      fl_set_button (prefsForm->dotsButton,TRUE);
      break;
    case 1 :
      fl_set_button (prefsForm->linesButton,TRUE);
      break;
    case 2 :
      fl_set_button (prefsForm->filledButton,TRUE);
      break;
    default :
      break;
  }

  item = fl_get_sample_drawaccuracy (mainForm->sample);
  
  switch (item)
  {
    case 0 :
      fl_set_button (prefsForm->quickButton,TRUE);
      break;
    case 1 :
      fl_set_button (prefsForm->accurateButton,TRUE);
      break;
    default :
      break;
  }

  item=fl_get_sample_axis (mainForm->sample);
  
  if (item)
  {
    fl_set_button (prefsForm->axisButton,TRUE);
  }
  else
  {
    fl_set_button (prefsForm->noaxisButton,TRUE);
  }

  item=fl_get_sample_frames (mainForm->sample);

  // Handle no range or no loop (sustain and release)
  // (range and loops get set due to text callbacks)
  DPSample *sample = fl_get_sample (mainForm->sample);
  int rangeValid  = sample->getRangeValid ();
  int susLoopMode = sample->getSusLoopMode ();
  int relLoopMode = sample->getRelLoopMode ();
  
  if (item)
  {
    fl_set_button (prefsForm->framesButton,TRUE);
    fl_hide_object (displayForm->floatInputs);
    fl_show_object (displayForm->intInputs);
    fl_hide_object (loopForm->floatInputs);
    fl_show_object (loopForm->intInputs);
    fl_hide_object (rangeForm->floatInputs);
    fl_show_object (rangeForm->intInputs);
  }
  else
  {
    fl_set_button (prefsForm->timeButton,TRUE);
    fl_hide_object (displayForm->intInputs);
    fl_show_object (displayForm->floatInputs);
    fl_hide_object (loopForm->intInputs);
    fl_show_object (loopForm->floatInputs);
    fl_hide_object (rangeForm->intInputs);
    fl_show_object (rangeForm->floatInputs);
  }

  // Restore range and loop settings
  if (!rangeValid)
    sample->rangeClear ();
  if (susLoopMode == AF_LOOP_MODE_NOLOOP)
    sample->setSusLoopMode (AF_LOOP_MODE_NOLOOP);
  if (relLoopMode == AF_LOOP_MODE_NOLOOP)
    sample->setRelLoopMode (AF_LOOP_MODE_NOLOOP);

  fl_redraw_object (mainForm->sample);
  updateRangeDetails ();
  updateLoopDetails ();

  item=fl_get_sample_undomode (mainForm->sample);
  
  if (item)
    fl_set_button (prefsForm->undoButton,TRUE);
  else
    fl_set_button (prefsForm->undoButton,FALSE);

  item=fl_get_sample_autowindows (mainForm->sample);
  
  if (item)
    fl_set_button (prefsForm->autowindowsButton,TRUE);
  else
    fl_set_button (prefsForm->autowindowsButton,FALSE);

  item=fl_get_sample_autoglobals (mainForm->sample);
  
  if (item)
    fl_set_button (prefsForm->autoglobalsButton,TRUE);
  else
    fl_set_button (prefsForm->autoglobalsButton,FALSE);

  item=fl_get_sample_autoeditmode (mainForm->sample);
  
  if (item)
    fl_set_button (prefsForm->autoeditmodeButton,TRUE);
  else
    fl_set_button (prefsForm->autoeditmodeButton,FALSE);

  item=fl_get_sample_soxcompatible (mainForm->sample);
  
  if (item)
    fl_set_button (prefsForm->soxcompatibleButton,TRUE);
  else
    fl_set_button (prefsForm->soxcompatibleButton,FALSE);

  item=fl_get_sample_tooltips (mainForm->sample);
  
  if (item)
    fl_set_button (prefsForm->tooltipsButton,TRUE);
  else
    fl_set_button (prefsForm->tooltipsButton,FALSE);
  
  const char *textItem = fl_get_sample_tempdir (mainForm->sample);
  
  if (textItem)
    fl_set_input (prefsForm->tempdir,textItem);
  else
    fl_set_input (prefsForm->tempdir,"");

  char tempString [80];
  
  sprintf (tempString,"%d",normalIdle);
  fl_set_input (prefsForm->normalidle,tempString);

  sprintf (tempString,"%d",heavyIdle);
  fl_set_input (prefsForm->heavyidle,tempString);

  item=fl_get_sample_rangealt (mainForm->sample);
  
  if (item)
    fl_set_button (prefsForm->rangealtButton,TRUE);
  else
    fl_set_button (prefsForm->rangealtButton,FALSE);

  item=fl_get_sample_rangecross (mainForm->sample);
  
  if (item)
    fl_set_button (prefsForm->rangecrossButton,TRUE);
  else
    fl_set_button (prefsForm->rangecrossButton,FALSE);

  item=fl_get_sample_vertscroll (mainForm->sample);
  
  if (item)
    fl_set_button (prefsForm->vertcontrolsButton,TRUE);
  else
    fl_set_button (prefsForm->vertcontrolsButton,FALSE);

  item=fl_get_sample_inputtextsize (mainForm->sample);
  
  if (item == FL_SMALL_SIZE)
    fl_set_button (prefsForm->textSmallButton,TRUE);
  else /* Assume tiny */
    fl_set_button (prefsForm->textTinyButton,TRUE);
  
  item=fl_get_sample_borderwidth (mainForm->sample);
  
  sprintf (tempString,"%d",item);
  fl_set_input (prefsForm->borderwidth,tempString);
  
  item=fl_get_sample_basicwavs (mainForm->sample);
  
  if (item)
    fl_set_button (prefsForm->basicwavsButton,TRUE);
  else
    fl_set_button (prefsForm->basicwavsButton,FALSE);
  
  textItem = fl_get_sample_audio_device (mainForm->sample);
  
  if (textItem)
    fl_set_input (prefsForm->audiodev,textItem);
  else
    fl_set_input (prefsForm->audiodev,"");

  textItem = fl_get_sample_mixer_device (mainForm->sample);
  
  if (textItem)
    fl_set_input (prefsForm->mixerdev,textItem);
  else
    fl_set_input (prefsForm->mixerdev,"");
}

/*---------------------------------------------------------------------------
| FUNCTION updateSamplePrefs
---------------------------------------------------------------------------*/

void updateSamplePrefs (void)
{
  if (fl_get_button (prefsForm->dotsButton))
    fl_set_sample_drawmode (mainForm->sample,0);
  else if (fl_get_button (prefsForm->linesButton))
    fl_set_sample_drawmode (mainForm->sample,1);
  else
    fl_set_sample_drawmode (mainForm->sample,2);

  if (fl_get_button (prefsForm->quickButton))
    fl_set_sample_drawaccuracy (mainForm->sample,0);
  else
    fl_set_sample_drawaccuracy (mainForm->sample,1);

  if (fl_get_button (prefsForm->axisButton))
    fl_set_sample_axis (mainForm->sample,1);
  else
    fl_set_sample_axis (mainForm->sample,0);

  if (fl_get_button (prefsForm->framesButton))
    fl_set_sample_frames (mainForm->sample,1);
  else
    fl_set_sample_frames (mainForm->sample,0);
  
  if (fl_get_button (prefsForm->undoButton))
    fl_set_sample_undomode (mainForm->sample,1);
  else
  {
    waitCursorOn ();
    clearBuffers ();
    waitCursorOff ();
    fl_set_sample_undomode (mainForm->sample,0);
  }

  if (fl_get_button (prefsForm->autowindowsButton))
    fl_set_sample_autowindows (mainForm->sample,1);
  else
    fl_set_sample_autowindows (mainForm->sample,0);

  if (fl_get_button (prefsForm->autoglobalsButton))
    fl_set_sample_autoglobals (mainForm->sample,1);
  else
    fl_set_sample_autoglobals (mainForm->sample,0);

  if (fl_get_button (prefsForm->autoeditmodeButton))
    fl_set_sample_autoeditmode (mainForm->sample,1);
  else
    fl_set_sample_autoeditmode (mainForm->sample,0);

  if (fl_get_button (prefsForm->soxcompatibleButton))
    fl_set_sample_soxcompatible (mainForm->sample,1);
  else
    fl_set_sample_soxcompatible (mainForm->sample,0);

  if (fl_get_button (prefsForm->tooltipsButton))
  {
    fl_set_sample_tooltips (mainForm->sample,1);
    tooltips_resume ();
  }
  else
  {
    fl_set_sample_tooltips (mainForm->sample,0);
    tooltips_suspend ();
  }
  
  const char *textItem = fl_safe_get_input (prefsForm->tempdir);
  fl_set_sample_tempdir (mainForm->sample,textItem);
  
  sscanf (fl_safe_get_input (prefsForm->normalidle),"%d",&normalIdle);
  sscanf (fl_safe_get_input (prefsForm->heavyidle),"%d",&heavyIdle);
  
  // Assume we are currently in normal idle mode
  fl_set_idle_delta (normalIdle);
    
  if (fl_get_button (prefsForm->rangealtButton))
    fl_set_sample_rangealt (mainForm->sample,1);
  else
    fl_set_sample_rangealt (mainForm->sample,0);

  if (fl_get_button (prefsForm->rangecrossButton))
    fl_set_sample_rangecross (mainForm->sample,1);
  else
    fl_set_sample_rangecross (mainForm->sample,0);

  if (fl_get_button (prefsForm->vertcontrolsButton))
    switchVertControls (TRUE);
  else
    switchVertControls (FALSE);
  
  if (fl_get_button (prefsForm->textSmallButton))
    switchTextSize (FL_SMALL_SIZE);
  else /* Assume tiny */
    switchTextSize (FL_TINY_SIZE);

  int width;  
  sscanf (fl_safe_get_input (prefsForm->borderwidth),"%d",&width);
  changeBorderWidth (width);
  
  if (fl_get_button (prefsForm->basicwavsButton))
    fl_set_sample_basicwavs (mainForm->sample,1);
  else
    fl_set_sample_basicwavs (mainForm->sample,0);

  int changed = FALSE;
  textItem = fl_safe_get_input (prefsForm->audiodev);
  if (strcmp (textItem,fl_get_sample_audio_device (mainForm->sample)))
    changed = TRUE;
  fl_set_sample_audio_device (mainForm->sample,textItem);
  
  textItem = fl_safe_get_input (prefsForm->mixerdev);
  if (strcmp (textItem,fl_get_sample_mixer_device (mainForm->sample)))
    changed = TRUE;
  fl_set_sample_mixer_device (mainForm->sample,textItem);
  
  // Tell user to restart if devices changed
  if (changed)
  {
     fl_show_alert ("Warning",
       "Changes to audio or mixer device will only",
       "take effect upon restarting DAP",TRUE);
  }
  
  fl_redraw_object (mainForm->sample);
}

/*---------------------------------------------------------------------------
| FUNCTION defaultSamplePrefs
---------------------------------------------------------------------------*/

void defaultSamplePrefs ()
{
  // Set prefs to default values
  fl_set_sample_scroll (mainForm->sample,0);
  fl_set_sample_axis (mainForm->sample,1);
  fl_set_sample_edit (mainForm->sample,2);
  fl_set_sample_frames (mainForm->sample,1);
  fl_set_sample_freehand (mainForm->sample,0);
  fl_set_sample_drawmode (mainForm->sample,1);
  fl_set_sample_drawaccuracy (mainForm->sample,0);
  fl_set_sample_undomode (mainForm->sample,1);
  fl_set_sample_autowindows (mainForm->sample,1);
  fl_set_sample_autoglobals (mainForm->sample,1);
  fl_set_sample_autoeditmode (mainForm->sample,1);
  fl_set_sample_soxcompatible (mainForm->sample,0);
  fl_set_sample_tooltips (mainForm->sample,1);
  fl_set_sample_rangealt (mainForm->sample,0);
  fl_set_sample_rangecross (mainForm->sample,0);
  fl_set_sample_tempdir (mainForm->sample,"/tmp");
  fl_set_sample_vertscroll (mainForm->sample,0);
  fl_set_sample_inputtextsize (mainForm->sample,FL_SMALL_SIZE);
  fl_set_sample_borderwidth (mainForm->sample,FL_BOUND_WIDTH);
  fl_set_sample_basicwavs (mainForm->sample,0);
  #ifdef NETBSD
  fl_set_sample_audio_device (mainForm->sample,"/dev/audio");
  fl_set_sample_mixer_device (mainForm->sample,"/dev/mixer");
  #else
  fl_set_sample_audio_device (mainForm->sample,"/dev/dsp");
  fl_set_sample_mixer_device (mainForm->sample,"/dev/mixer");
  #endif
  updatePrefsForm ();
  preferencesChanged = FALSE;
}

/*---------------------------------------------------------------------------
| FUNCTION switchVertControls
---------------------------------------------------------------------------*/

void switchVertControls (int on)
{
  if (on)
  {
    if (!fl_get_sample_vertscroll (mainForm->sample))
    {
      /* Switch vertical controls on */
      fl_set_sample_vertscroll (mainForm->sample,1);
      fl_show_object (mainForm->vertControlGroup);
      fl_set_object_position (mainForm->sample,
        mainForm->sample->x + 30,mainForm->sample->y);
      fl_set_object_position (mainForm->sampleFrame,
        mainForm->sampleFrame->x + 30,mainForm->sampleFrame->y);
      fl_set_object_size (mainForm->sample,
        mainForm->sample->w - 30,mainForm->sample->h);
      fl_set_object_size (mainForm->sampleFrame,
        mainForm->sampleFrame->w - 30,mainForm->sampleFrame->h);
          fl_set_sample_vertscroll (mainForm->sample,1);
    }
  }
  else
  {
    if (fl_get_sample_vertscroll (mainForm->sample))
    {
      /* Switch vertical controls off */
      fl_set_sample_vertscroll (mainForm->sample,0);
      fl_hide_object (mainForm->vertControlGroup);
      fl_set_object_position (mainForm->sample,
        mainForm->sample->x - 30,mainForm->sample->y);
      fl_set_object_position (mainForm->sampleFrame,
        mainForm->sampleFrame->x - 30,mainForm->sampleFrame->y);
      fl_set_object_size (mainForm->sample,
        mainForm->sample->w + 30,mainForm->sample->h);
      fl_set_object_size (mainForm->sampleFrame,
        mainForm->sampleFrame->w + 30,mainForm->sampleFrame->h);
      
      /* Set zoom back to default */
      fl_call_object_callback (mainForm->vertZoomAllButton);
    }
  }
}

/*---------------------------------------------------------------------------
| FUNCTION switchTextSize
---------------------------------------------------------------------------*/

void switchTextSize (int size)
{
  if (fl_get_sample_inputtextsize (mainForm->sample) != size)
  {
    fl_set_sample_inputtextsize (mainForm->sample,size);
    switchTextSizeForm (mainForm->mainForm,size);
    switchTextSizeForm (playForm->playForm,size);
    switchTextSizeForm (zoomForm->zoomForm,size);
    switchTextSizeForm (displayForm->displayForm,size);
    switchTextSizeForm (rangeForm->rangeForm,size);
    switchTextSizeForm (loopForm->loopForm,size);
    switchTextSizeForm (newForm->newForm,size);
    switchTextSizeForm (recordForm->recordForm,size);
    switchTextSizeForm (convertForm->convertForm,size);
    switchTextSizeForm (mixForm->mixForm,size);
    switchTextSizeForm (channelForm->channelForm,size);
    switchTextSizeForm (duplicateForm->duplicateForm,size);
    switchTextSizeForm (panoramaForm->panoramaForm,size);
    switchTextSizeForm (workspaceForm->workspaceForm,size);
    switchTextSizeForm (miscForm->miscForm,size);
    switchTextSizeForm (rateForm->rateForm,size);
    switchTextSizeForm (ampForm->ampForm,size);
    switchTextSizeForm (balanceForm->balanceForm,size);
    switchTextSizeForm (bounceForm->bounceForm,size);
    switchTextSizeForm (rampForm->rampForm,size);
    switchTextSizeForm (resampleForm->resampleForm,size);
    switchTextSizeForm (pitchForm->pitchForm,size);
    switchTextSizeForm (bufferForm->bufferForm,size);
    switchTextSizeForm (mixplayForm->mixplayForm,size);
    switchTextSizeForm (selectForm->selectForm,size);
    switchTextSizeForm (adjustDCForm->adjustDCForm,size);
    switchTextSizeForm (initialForm->initialForm,size);
    switchTextSizeForm (effectForm->effectForm,size);
    switchTextSizeForm (executeForm->executeForm,size);
    switchTextSizeForm (postForm->postForm,size);
    switchTextSizeForm (aboutForm->aboutForm,size);
    switchTextSizeForm (stretchForm->stretchForm,size);
    switchTextSizeForm (convert4to1Form->convert4to1Form,size);
    switchTextSizeForm (convert4to2Form->convert4to2Form,size);
    switchTextSizeForm (convert2to4Form->convert2to4Form,size);
    switchTextSizeForm (channel4Form->channel4Form,size);
    switchTextSizeForm (bounce4Form->bounce4Form,size);
    switchTextSizeForm (prefsForm->prefsForm,size);
    switchTextSizeForm (mixerForm->mixerForm,size);
    switchTextSizeForm (macroForm->macroForm,size);
    switchTextSizeForm (macroexecForm->macroexecForm,size);
    switchTextSizeForm (macroinfoForm->macroinfoForm,size);

    fl_set_object_lsize (displayForm->durationText,size);
    fl_set_object_lsize (displayForm->durationTimeText,size);
    fl_set_object_lsize (displayForm->positionText,size);
    fl_set_object_lsize (displayForm->timeText,size);
    fl_set_object_lsize (rangeForm->durationText,size);
    fl_set_object_lsize (rangeForm->durationTimeText,size);
    fl_set_object_lsize (loopForm->susDurationText,size);
    fl_set_object_lsize (loopForm->susDurationTimeText,size);
    fl_set_object_lsize (loopForm->relDurationText,size);
    fl_set_object_lsize (loopForm->relDurationTimeText,size);
    fl_set_object_lsize (loopForm->susBPM,size);
    fl_set_object_lsize (loopForm->relBPM,size);
    fl_set_object_lsize (mixForm->clipboardText,size);
    fl_set_object_lsize (mixForm->bufferText,size);
    fl_set_object_lsize (mixForm->framesText,size);
    fl_set_object_lsize (mixForm->timeText,size);
    fl_set_object_lsize (panoramaForm->amountText,size);
    fl_set_object_lsize (rampForm->framesText,size);
    fl_set_object_lsize (rampForm->timeText,size);
    fl_set_object_lsize (resampleForm->currentRate,size);
    fl_set_object_lsize (resampleForm->currentFrames,size);
    fl_set_object_lsize (resampleForm->currentTime,size);
    fl_set_object_lsize (resampleForm->currentSemi,size);
    fl_set_object_lsize (resampleForm->currentOct,size);
    fl_set_object_lsize (resampleForm->currentCent,size);
    fl_set_object_lsize (resampleForm->currentBigSemi,size);
    fl_set_object_lsize (resampleForm->pitchChangeDir,size);
    fl_set_object_lsize (resampleForm->newSemi,size);
    fl_set_object_lsize (resampleForm->newOct,size);
    fl_set_object_lsize (resampleForm->newCent,size);
    fl_set_object_lsize (resampleForm->newBigSemi,size);
    fl_set_object_lsize (resampleForm->currentBPM,size);
    fl_set_object_lsize (adjustDCForm->amountText,size);
    fl_set_object_lsize (executeForm->frames,size);
    fl_set_object_lsize (executeForm->time,size);
    fl_set_object_lsize (executeForm->elapsedTime,size);
    fl_set_object_lsize (executeForm->remainTime,size);
    fl_set_object_lsize (executeForm->percentComp,size);
    fl_set_object_lsize (stretchForm->percentComp,size);
  }
}

/*---------------------------------------------------------------------------
| FUNCTION changeBorderWidth
---------------------------------------------------------------------------*/

void changeBorderWidth (int width)
{
  if (fl_get_sample_borderwidth (mainForm->sample) != width)
  {
    fl_set_sample_borderwidth (mainForm->sample,width);
    changeBorderWidthForm (mainForm->mainForm,width);
    changeBorderWidthForm (playForm->playForm,width);
    changeBorderWidthForm (zoomForm->zoomForm,width);
    changeBorderWidthForm (displayForm->displayForm,width);
    changeBorderWidthForm (rangeForm->rangeForm,width);
    changeBorderWidthForm (loopForm->loopForm,width);
    changeBorderWidthForm (newForm->newForm,width);
    changeBorderWidthForm (recordForm->recordForm,width);
    changeBorderWidthForm (convertForm->convertForm,width);
    changeBorderWidthForm (mixForm->mixForm,width);
    changeBorderWidthForm (channelForm->channelForm,width);
    changeBorderWidthForm (duplicateForm->duplicateForm,width);
    changeBorderWidthForm (panoramaForm->panoramaForm,width);
    changeBorderWidthForm (workspaceForm->workspaceForm,width);
    changeBorderWidthForm (miscForm->miscForm,width);
    changeBorderWidthForm (rateForm->rateForm,width);
    changeBorderWidthForm (ampForm->ampForm,width);
    changeBorderWidthForm (balanceForm->balanceForm,width);
    changeBorderWidthForm (bounceForm->bounceForm,width);
    changeBorderWidthForm (rampForm->rampForm,width);
    changeBorderWidthForm (resampleForm->resampleForm,width);
    changeBorderWidthForm (pitchForm->pitchForm,width);
    changeBorderWidthForm (bufferForm->bufferForm,width);
    changeBorderWidthForm (mixplayForm->mixplayForm,width);
    changeBorderWidthForm (selectForm->selectForm,width);
    changeBorderWidthForm (adjustDCForm->adjustDCForm,width);
    changeBorderWidthForm (initialForm->initialForm,width);
    changeBorderWidthForm (effectForm->effectForm,width);
    changeBorderWidthForm (executeForm->executeForm,width);
    changeBorderWidthForm (postForm->postForm,width);
    changeBorderWidthForm (aboutForm->aboutForm,width);
    changeBorderWidthForm (stretchForm->stretchForm,width);
    changeBorderWidthForm (convert4to1Form->convert4to1Form,width);
    changeBorderWidthForm (convert4to2Form->convert4to2Form,width);
    changeBorderWidthForm (convert2to4Form->convert2to4Form,width);
    changeBorderWidthForm (channel4Form->channel4Form,width);
    changeBorderWidthForm (bounce4Form->bounce4Form,width);
    changeBorderWidthForm (prefsForm->prefsForm,width);
    changeBorderWidthForm (mixerForm->mixerForm,width);
    changeBorderWidthForm (macroForm->macroForm,width);
    changeBorderWidthForm (macroexecForm->macroexecForm,width);
    changeBorderWidthForm (macroinfoForm->macroinfoForm,width);
  }
}

/*---------------------------------------------------------------------------
| FUNCTION switchTextSizeForm
---------------------------------------------------------------------------*/

void switchTextSizeForm (FL_FORM *form,int size)
{
  FL_OBJECT *first    = form->first;
  FL_OBJECT *current  = first;
  
  while (current)
  {
    if (current->objclass == FL_INPUT)
    {
      fl_set_object_lsize (current,size);
    }
    current = current->next;
    if (current == first) current = 0;
  }
}

/*---------------------------------------------------------------------------
| FUNCTION changeBorderWidthForm
---------------------------------------------------------------------------*/

void changeBorderWidthForm (FL_FORM *form,int width)
{
  FL_OBJECT *first    = form->first;
  FL_OBJECT *current  = first;
  
  while (current)
  {
    fl_set_object_bw (current,width);
    current = current->next;
    if (current == first) current = 0;
  }
}

/***************************************************************************/
