/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPTich_DSP.cc
|
| Object      : None
|
| Description : Effects functions
|
| (c) Richard Kent 1996
|
| $Id: DPTich_DSP.cc,v 1.1 2003/09/10 00:06:24 rk Exp $
|
****************************************************************************/

static char DPTich_DSP_cc [] = "$Id: DPTich_DSP.cc,v 1.1 2003/09/10 00:06:24 rk Exp $";

#include "DPTich_DSP.h"

#define PARAM(n)  effects->effect [effectNo].param [n]
#define PARAMR(n) effects->effect [effectNo].param [n].realValue
#define PARAME(n) effects->effect [effectNo].param [n].enumValue

/*---------------------------------------------------------------------------
| FUNCTION DSPReverberation
---------------------------------------------------------------------------*/

char *DSPReverberation
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPReverberation
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPReverberation (
  DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int channel_a,int channel_b)
{
  int effectNo  = 0;
  double rate   = inputSample->getRate ();
  double rate_2 = rate / 2.0;
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitBWLowPassFilter LPF_L;
  DSPKitBWLowPassFilter LPF_R;
  DSPKitMux       splitter_L;
  DSPKitMux       splitter_R;
  DSPKitDelay     ER1_L;
  DSPKitDelay     ER1_R;
  DSPKitDelay     ER2_L;
  DSPKitDelay     ER2_R;
  DSPKitDelay     crossER_L;
  DSPKitDelay     crossER_R;
  DSPKitDelay     predelay_L;
  DSPKitDelay     predelay_R;
  DSPKitAmp       ER1amp_L;
  DSPKitAmp       ER1amp_R;
  DSPKitAmp       ER2amp_L;
  DSPKitAmp       ER2amp_R;
  DSPKitAmp       crossERamp_L;
  DSPKitAmp       crossERamp_R;
  DSPKitAmp       predelayamp_L;
  DSPKitAmp       predelayamp_R;
  DSPKitSchroederReverb schReverb_L;
  DSPKitSchroederReverb schReverb_R;
  DSPKitJAMReverb jamReverb_L;
  DSPKitJAMReverb jamReverb_R;
  DSPKitStdReverb stdReverb_L;
  DSPKitStdReverb stdReverb_R;
  DSPKitSum       summer_L;
  DSPKitSum       summer_R;
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,channel_a,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,channel_b,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections

  if (PARAMR(6) < 0.9999)
  {
    LPF_L.setInput (&reader_L);
    LPF_R.setInput (&reader_R);
    LPF_L.setCutOffFreq (rate_2 * PARAMR(6));
    LPF_R.setCutOffFreq (rate_2 * PARAMR(6));
    
    splitter_L.setInput (&LPF_L);
    splitter_R.setInput (&LPF_R);
  }
  else
  {
    splitter_L.setInput (&reader_L);
    splitter_R.setInput (&reader_R);
  }
  
  ER1_L.setInputAndDelayTime (&splitter_L,PARAMR(7)  / 1000.0);
  ER1_R.setInputAndDelayTime (&splitter_R,PARAMR(8)  / 1000.0);
  
  ER2_L.setInputAndDelayTime (&splitter_L,PARAMR(11) / 1000.0);
  ER2_R.setInputAndDelayTime (&splitter_R,PARAMR(12) / 1000.0);
  
  crossER_L.setInputAndDelayTime (&splitter_L,PARAMR(15) / 1000.0);
  crossER_R.setInputAndDelayTime (&splitter_R,PARAMR(16) / 1000.0);
  
  predelay_L.setInputAndDelayTime (&splitter_L,PARAMR(2)  / 1000.0);
  predelay_R.setInputAndDelayTime (&splitter_R,PARAMR(3)  / 1000.0);
  
  ER1amp_L.setInput (&ER1_L);
  ER1amp_R.setInput (&ER1_R);
  ER1amp_L.setGain (PARAMR(9)/100.0);
  ER1amp_R.setGain (PARAMR(10)/100.0);
  
  ER2amp_L.setInput (&ER2_L);
  ER2amp_R.setInput (&ER2_R);
  ER2amp_L.setGain (PARAMR(13)/100.0);
  ER2amp_R.setGain (PARAMR(14)/100.0);
  
  crossERamp_L.setInput (&crossER_L);
  crossERamp_R.setInput (&crossER_R);
  crossERamp_L.setGain (PARAMR(17)/100.0);
  crossERamp_R.setGain (PARAMR(18)/100.0);
  
  predelayamp_L.setInput (&predelay_L);
  predelayamp_R.setInput (&predelay_R);
  predelayamp_L.setGain (1.0);
  predelayamp_R.setGain (1.0);
  
  if (PARAME(0) == 0)
  {
    // Moorer
    stdReverb_L.setInput (&predelayamp_L,PARAMR(4),PARAMR(5));
    stdReverb_R.setInput (&predelayamp_R,PARAMR(4),PARAMR(5));
    stdReverb_L.setReverbTime (PARAMR(1));
    stdReverb_R.setReverbTime (PARAMR(1));
    
    summer_L.setInput (&stdReverb_L);
    summer_R.setInput (&stdReverb_R);
  }
  else if (PARAME(0) == 1)
  {
    // Schroeder
    schReverb_L.setInput (&predelayamp_L,PARAMR(4),PARAMR(5));
    schReverb_R.setInput (&predelayamp_R,PARAMR(4),PARAMR(5));
    schReverb_L.setReverbTime (PARAMR(1));
    schReverb_R.setReverbTime (PARAMR(1));
    
    summer_L.setInput (&schReverb_L);
    summer_R.setInput (&schReverb_R);
  }
  else
  {
    // Multi Tap
    jamReverb_L.setInput (&predelayamp_L,PARAMR(4),PARAMR(5));
    jamReverb_R.setInput (&predelayamp_R,PARAMR(4),PARAMR(5));
    jamReverb_L.setReverbTime (PARAMR(1));
    jamReverb_R.setReverbTime (PARAMR(1));
    
    summer_L.setInput (&jamReverb_L);
    summer_R.setInput (&jamReverb_R);
  }

  summer_L.addInput (&ER1amp_L);
  summer_L.addInput (&ER2amp_L);
  summer_L.addInput (&crossERamp_R);
  summer_R.addInput (&ER1amp_R);
  summer_R.addInput (&ER2amp_R);
  summer_R.addInput (&crossERamp_L);
  
  mainAmp_L.setInput (&summer_L);
  mainAmp_R.setInput (&summer_R);

  if (PARAME(0) == 0)
  {
    // Moorer
    mainAmp_L.setGain (1.0);
    mainAmp_R.setGain (1.0);
  }
  else if (PARAME(0) == 1)
  {
    // Schroeder
    mainAmp_L.setGain (0.8);
    mainAmp_R.setGain (0.8);
  }
  else
  {
    // Multi Tap
    mainAmp_L.setGain (0.4);
    mainAmp_R.setGain (0.4);
  }

  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }
  
  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPGatedReverberation
---------------------------------------------------------------------------*/

char *DSPGatedReverberation
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPGatedReverberation
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPGatedReverberation
  (DPSample *inputSample,DPSample *,int,int,
  int,int,int)
{
  //int effectNo  = 1;
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  return "Not yet implemented";
}

/*---------------------------------------------------------------------------
| FUNCTION DSPStereoDelay
---------------------------------------------------------------------------*/

char *DSPStereoDelay
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPStereoDelay
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPStereoDelay
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 2;
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitFeedback  summer_L;
  DSPKitFeedback  summer_R;
  DSPKitBWLowPassFilter  LPF_L;
  DSPKitBWLowPassFilter  LPF_R;
  DSPKitBWHighPassFilter HPF_L;
  DSPKitBWHighPassFilter HPF_R;
  DSPKitFBDelay   delay_L;
  DSPKitFBDelay   delay_R;
  DSPKitAmp       feedbackamp_L;
  DSPKitAmp       feedbackamp_R;
  DSPKitMux       splitter_L;
  DSPKitMux       splitter_R;
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections

  summer_L.setInput (&reader_L);
  summer_R.setInput (&reader_R);
  summer_L.setFeedbackInput (&feedbackamp_L);
  summer_R.setFeedbackInput (&feedbackamp_R);

  // Calculate feedback time
  
  double feedbackTime_L;
  double feedbackTime_R;
  double feedbackTime;

  feedbackTime_L = calcFeedbackTime (PARAMR(3),PARAMR(0));
  feedbackTime_R = calcFeedbackTime (PARAMR(4),PARAMR(1));

  if (feedbackTime_L > feedbackTime_R)
    feedbackTime = feedbackTime_L;
  else
    feedbackTime = feedbackTime_R;

  summer_L.setFeedbackTime (feedbackTime);
  summer_R.setFeedbackTime (feedbackTime);

  if (PARAMR(5) >= 19.9999 && PARAMR(6) >= 19.9999
  && PARAMR(7) <= 0.0201 && PARAMR(8) <= 0.0201)
  {
    delay_L.setInputAndDelayTime (&summer_L,PARAMR(0) / 1000.0);
    delay_R.setInputAndDelayTime (&summer_R,PARAMR(1) / 1000.0);
  }
  else
  {
    LPF_L.setInput (&summer_L);
    LPF_R.setInput (&summer_R);
    LPF_L.setCutOffFreq (PARAMR(5) * 1000.0);
    LPF_R.setCutOffFreq (PARAMR(6) * 1000.0);
    
    HPF_L.setInput (&LPF_L);
    HPF_R.setInput (&LPF_R);
    HPF_L.setCutOffFreq (PARAMR(7) * 1000.0);
    HPF_R.setCutOffFreq (PARAMR(8) * 1000.0);
    
    delay_L.setInputAndDelayTime (&HPF_L,PARAMR(0) / 1000.0);
    delay_R.setInputAndDelayTime (&HPF_R,PARAMR(1) / 1000.0);
  }

  splitter_L.setInput (&delay_L);
  splitter_R.setInput (&delay_R);
  
  if (PARAME(2) == 0)
  {
    feedbackamp_L.setInput (&splitter_L);
    feedbackamp_R.setInput (&splitter_R);
  }
  else
  {
    feedbackamp_L.setInput (&splitter_R);
    feedbackamp_R.setInput (&splitter_L);
  }

  feedbackamp_L.setGain (PARAMR(3) / 100.0);
  feedbackamp_R.setGain (PARAMR(4) / 100.0);

  mainAmp_L.setInput (&splitter_L);
  mainAmp_R.setInput (&splitter_R);
  mainAmp_L.setGain (0.8);
  mainAmp_R.setGain (0.8);
  
  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }

  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPModulatedDelay
---------------------------------------------------------------------------*/

char *DSPModulatedDelay
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPModulatedDelay
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPModulatedDelay
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 3;
  double rate   = inputSample->getRate ();
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitFeedback  summer_L;
  DSPKitFeedback  summer_R;
  DSPKitDelayInt  delay_L;
  DSPKitDelayInt  delay_R;
  DSPKitAmp       feedbackamp_L;
  DSPKitAmp       feedbackamp_R;
  DSPKitMux       splitter_L;
  DSPKitMux       splitter_R;
  DSPKitLFO       LFO_L;
  DSPKitLFO       LFO_R;
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections

  summer_L.setInput (&reader_L);
  summer_R.setInput (&reader_R);
  summer_L.setFeedbackInput (&feedbackamp_L);
  summer_R.setFeedbackInput (&feedbackamp_R);
  
  // Calculate feedback time

  double feedbackTime_L;
  double feedbackTime_R;
  double feedbackTime;

  feedbackTime_L = calcFeedbackTime (PARAMR(3),PARAMR(0));
  feedbackTime_R = calcFeedbackTime (PARAMR(4),PARAMR(1));
  
  if (feedbackTime_L > feedbackTime_R)
    feedbackTime = feedbackTime_L;
  else
    feedbackTime = feedbackTime_R;

  summer_L.setFeedbackTime (feedbackTime);
  summer_R.setFeedbackTime (feedbackTime);

  
  delay_L.setInputAndDelayTime (&summer_L,PARAMR(0) / 1000.0);
  delay_R.setInputAndDelayTime (&summer_R,PARAMR(1) / 1000.0);
  delay_L.setModInput (&LFO_L);
  delay_R.setModInput (&LFO_R);

  LFO_L.setTypeAndFreq
    (PARAME(10),rate,PARAMR(8),PARAMR(5) / 100.0,0.0);
  LFO_R.setTypeAndFreq
    (PARAME(11),rate,PARAMR(9),PARAMR(6) / 100.0,PARAMR(7));
  
  splitter_L.setInput (&delay_L);
  splitter_R.setInput (&delay_R);
  
  if (PARAME(2) == 0)
  {
    feedbackamp_L.setInput (&splitter_L);
    feedbackamp_R.setInput (&splitter_R);
  }
  else
  {
    feedbackamp_L.setInput (&splitter_R);
    feedbackamp_R.setInput (&splitter_L);
  }

  feedbackamp_L.setGain (PARAMR(3) / 100.0);
  feedbackamp_R.setGain (PARAMR(4) / 100.0);
  
  mainAmp_L.setInput (&splitter_L);
  mainAmp_R.setInput (&splitter_R);
  mainAmp_L.setGain (0.8);
  mainAmp_R.setGain (0.8);
  
  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }
  
  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPMultiTapDelay
---------------------------------------------------------------------------*/

char *DSPMultiTapDelay
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPMultiTapDelay
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPMultiTapDelay
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 4;
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitFeedback  summer_L;
  DSPKitFeedback  summer_R;
  DSPKitAmp       feedbackamp_L;
  DSPKitAmp       feedbackamp_R;
  DSPKitMux       splitter_L;
  DSPKitMux       splitter_R;
  
  DSPKitDelay     tap_L [6];
  DSPKitAmp       tapamp_L [6];
  DSPKitFBDelay   delay_L;
  DSPKitMux       delaysplitter_L;
  DSPKitAmp       delayamp_L;
  
  DSPKitDelay     tap_R [6];
  DSPKitAmp       tapamp_R [6];
  DSPKitFBDelay   delay_R;
  DSPKitMux       delaysplitter_R;
  DSPKitAmp       delayamp_R;
  
  DSPKitSum       delaysummer_L;
  DSPKitSum       delaysummer_R;
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections

  summer_L.setInput (&reader_L);
  summer_R.setInput (&reader_R);
  summer_L.setFeedbackInput (&feedbackamp_L);
  summer_R.setFeedbackInput (&feedbackamp_R);
  
  // Calculate feedback time
  
  double feedbackTime_L;
  double feedbackTime_R;
  double feedbackTime;

  feedbackTime_L = calcFeedbackTime (PARAMR(3),PARAMR(0));
  feedbackTime_R = calcFeedbackTime (PARAMR(4),PARAMR(1));
  
  if (feedbackTime_L > feedbackTime_R)
    feedbackTime = feedbackTime_L;
  else
    feedbackTime = feedbackTime_R;

  summer_L.setFeedbackTime (feedbackTime);
  summer_R.setFeedbackTime (feedbackTime);
  
  splitter_L.setInput (&summer_L);
  splitter_R.setInput (&summer_R);
  
  int i;
  
  for(i=0; i<6; i++)
  {
    tap_L [i].setInputAndDelayTime (&splitter_L,PARAMR(4*i + 5) / 1000.0);
    tap_R [i].setInputAndDelayTime (&splitter_R,PARAMR(4*i + 6) / 1000.0);
  }
  
  delay_L.setInputAndDelayTime (&splitter_L,PARAMR(0) / 1000.0);
  delay_R.setInputAndDelayTime (&splitter_R,PARAMR(1) / 1000.0);
  
  for (i=0; i<6; i++)
  {
    tapamp_L [i].setInput (&(tap_L [i]));
    tapamp_R [i].setInput (&(tap_R [i]));
    tapamp_L [i].setGain (PARAMR(4*i + 7) / 100.0);
    tapamp_R [i].setGain (PARAMR(4*i + 8) / 100.0);
  }

  delaysplitter_L.setInput (&delay_L);
  delaysplitter_R.setInput (&delay_R);
  
  delayamp_L.setInput (&delaysplitter_L);
  delayamp_R.setInput (&delaysplitter_R);
  delayamp_L.setGain (PARAMR(29) / 100.0);
  delayamp_R.setGain (PARAMR(30) / 100.0);
  
  delaysummer_L.setInput (&delayamp_L);
  delaysummer_R.setInput (&delayamp_R);
  
  for (i=0; i<6; i++)
  {
    delaysummer_L.addInput (&(tapamp_L [i]));
    delaysummer_R.addInput (&(tapamp_R [i]));
  }

  if (PARAME(2) == 0)
  {
    feedbackamp_L.setInput (&delaysplitter_L);
    feedbackamp_R.setInput (&delaysplitter_R);
  }
  else
  {
    feedbackamp_L.setInput (&delaysplitter_R);
    feedbackamp_R.setInput (&delaysplitter_L);
  }

  feedbackamp_L.setGain (PARAMR(3) / 100.0);
  feedbackamp_R.setGain (PARAMR(4) / 100.0);
  
  mainAmp_L.setInput (&delaysummer_L);
  mainAmp_R.setInput (&delaysummer_R);
  mainAmp_L.setGain (0.8);
  mainAmp_R.setGain (0.8);

  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }
  
  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPMultiDelay
---------------------------------------------------------------------------*/

char *DSPMultiDelay
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPMultiDelay
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPMultiDelay
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 5;
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitFeedback  summer_L [6];
  DSPKitFeedback  summer_R [6];
  DSPKitAmp       feedbackamp_L [6];
  DSPKitAmp       feedbackamp_R [6];
  DSPKitFBDelay   delay_L [6];
  DSPKitFBDelay   delay_R [6];
  DSPKitMux       splitter_L [6];
  DSPKitMux       splitter_R [6];
  
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections
  
  int i;
  
  for (i=0; i<6; i++)
  {
    if (i == 0)
    {
      summer_L [i].setInput (&reader_L);
      summer_R [i].setInput (&reader_R);
    }
    else
    {
      summer_L [i].setInput (&(splitter_L [i-1]));
      summer_R [i].setInput (&(splitter_R [i-1]));
    }
    
    summer_L [i].setFeedbackInput (&(feedbackamp_L [i]));
    summer_R [i].setFeedbackInput (&(feedbackamp_R [i]));
    
    // Calculate feedback time
    
    double feedbackTime_L;
    double feedbackTime_R;
    double feedbackTime;

    feedbackTime_L = calcFeedbackTime (PARAMR(5*i + 1),PARAMR(5*i + 3));
    feedbackTime_R = calcFeedbackTime (PARAMR(5*i + 2),PARAMR(5*i + 4));

    if (feedbackTime_L > feedbackTime_R)
      feedbackTime = feedbackTime_L;
    else
      feedbackTime = feedbackTime_R;

    summer_L [i].setFeedbackTime (feedbackTime);
    summer_R [i].setFeedbackTime (feedbackTime);

    delay_L [i].setInputAndDelayTime
      (&(summer_L [i]),PARAMR(5*i + 3) / 1000.0);
    delay_R [i].setInputAndDelayTime
      (&(summer_R [i]),PARAMR(5*i + 4) / 1000.0);
      
    splitter_L [i].setInput (&(delay_L [i]));
    splitter_R [i].setInput (&(delay_R [i]));
    
    if (PARAME(5*i + 0) == 0)
    {
      feedbackamp_L [i].setInput (&splitter_L [i]);
      feedbackamp_R [i].setInput (&splitter_R [i]);
    }
    else
    {
      feedbackamp_L [i].setInput (&splitter_R [i]);
      feedbackamp_R [i].setInput (&splitter_L [i]);
    }
    
    feedbackamp_L [i].setGain (PARAMR(5*i + 1) / 100.0);
    feedbackamp_R [i].setGain (PARAMR(5*i + 2) / 100.0);
  }

  mainAmp_L.setInput (&splitter_L [5]);
  mainAmp_R.setInput (&splitter_R [5]);
  mainAmp_L.setGain (0.8);
  mainAmp_R.setGain (0.8);

  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }
  
  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPFlanger
---------------------------------------------------------------------------*/

char *DSPFlanger
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPFlanger
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPFlanger
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 6;
  double rate   = inputSample->getRate ();
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitMux       splitter_L;
  DSPKitMux       splitter_R;
  DSPKitFeedback  summer_L;
  DSPKitFeedback  summer_R;
  DSPKitDelayInt  delay_L;
  DSPKitDelayInt  delay_R;
  DSPKitAmp       feedbackamp_L;
  DSPKitAmp       feedbackamp_R;
  DSPKitMux       delaysplitter_L;
  DSPKitMux       delaysplitter_R;
  DSPKitAmp       delaydryamp_L;
  DSPKitAmp       delaydryamp_R;
  DSPKitAmp       delaywetamp_L;
  DSPKitAmp       delaywetamp_R;
  DSPKitSum       delaymixer_L;
  DSPKitSum       delaymixer_R;
  DSPKitCombInt   flanger_L;
  DSPKitCombInt   flanger_R;
  DSPKitLFO       LFO_L;
  DSPKitLFO       LFO_R;
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections

  splitter_L.setInput (&reader_L);
  splitter_R.setInput (&reader_R);

  summer_L.setInput (&splitter_L);
  summer_R.setInput (&splitter_R);
  summer_L.setFeedbackInput (&feedbackamp_L);
  summer_R.setFeedbackInput (&feedbackamp_R);
  
  // Calculate feedback time

  double feedbackTime_L;
  double feedbackTime_R;
  double feedbackTime;

  feedbackTime_L = calcFeedbackTime (PARAMR(10),PARAMR(7));
  feedbackTime_R = calcFeedbackTime (PARAMR(11),PARAMR(8));

  if (feedbackTime_L > feedbackTime_R)
    feedbackTime = feedbackTime_L;
  else
    feedbackTime = feedbackTime_R;

  summer_L.setFeedbackTime (feedbackTime);
  summer_R.setFeedbackTime (feedbackTime);

  delay_L.setInputAndDelayTime (&summer_L,PARAMR(7) / 1000.0);
  delay_R.setInputAndDelayTime (&summer_R,PARAMR(8) / 1000.0);
  
  delaysplitter_L.setInput (&delay_L);
  delaysplitter_R.setInput (&delay_R);
  
  delaywetamp_L.setInput (&delaysplitter_L);
  delaywetamp_R.setInput (&delaysplitter_R);
  delaywetamp_L.setGain (PARAMR(16) / 100.0);
  delaywetamp_R.setGain (PARAMR(17) / 100.0);

  delaydryamp_L.setInput (&splitter_L);
  delaydryamp_R.setInput (&splitter_R);
  delaydryamp_L.setGain ((100.0 - PARAMR(16)) / 100.0);
  delaydryamp_R.setGain ((100.0 - PARAMR(17)) / 100.0);
  
  delaymixer_L.setInput (&delaydryamp_L);
  delaymixer_R.setInput (&delaydryamp_R);
  delaymixer_L.addInput (&delaywetamp_L);
  delaymixer_R.addInput (&delaywetamp_R);
  
  flanger_L.setInputAndDelayTime (&delaymixer_L,PARAMR(12) / 1000.0);
  flanger_R.setInputAndDelayTime (&delaymixer_R,PARAMR(13) / 1000.0);
  flanger_L.setG (PARAMR(14) / 100.0);
  flanger_R.setG (PARAMR(15) / 100.0);
  flanger_L.setModInput (&LFO_L);
  flanger_R.setModInput (&LFO_R);
  
  LFO_L.setTypeAndFreq
    (PARAME(5),rate,PARAMR(3),PARAMR(0) / 100.0,0.0);
  LFO_R.setTypeAndFreq
    (PARAME(6),rate,PARAMR(4),PARAMR(1) / 100.0,PARAMR(2));
  
  if (PARAME(9) == 0)
  {
    feedbackamp_L.setInput (&delaysplitter_L);
    feedbackamp_R.setInput (&delaysplitter_R);
  }
  else
  {
    feedbackamp_L.setInput (&delaysplitter_R);
    feedbackamp_R.setInput (&delaysplitter_L);
  }

  feedbackamp_L.setGain (PARAMR(10) / 100.0);
  feedbackamp_R.setGain (PARAMR(11) / 100.0);
  
  mainAmp_L.setInput (&flanger_L);
  mainAmp_R.setInput (&flanger_R);
  mainAmp_L.setGain (0.6);
  mainAmp_R.setGain (0.6);
  
  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }
  
  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPPhaser
---------------------------------------------------------------------------*/

char *DSPPhaser
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPPhaser
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPPhaser
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 7;
  double rate   = inputSample->getRate ();
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitMux       splitter_L;
  DSPKitMux       splitter_R;
  DSPKitFeedback  summer_L;
  DSPKitFeedback  summer_R;
  DSPKitDelayInt  delay_L;
  DSPKitDelayInt  delay_R;
  DSPKitAmp       feedbackamp_L;
  DSPKitAmp       feedbackamp_R;
  DSPKitMux       delaysplitter_L;
  DSPKitMux       delaysplitter_R;
  DSPKitAmp       delaydryamp_L;
  DSPKitAmp       delaydryamp_R;
  DSPKitAmp       delaywetamp_L;
  DSPKitAmp       delaywetamp_R;
  DSPKitSum       delaymixer_L;
  DSPKitSum       delaymixer_R;
  DSPKitAllPassNetworkInt phaser_L;
  DSPKitAllPassNetworkInt phaser_R;
  DSPKitLFO       LFO_L;
  DSPKitLFO       LFO_R;
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections

  splitter_L.setInput (&reader_L);
  splitter_R.setInput (&reader_R);

  summer_L.setInput (&splitter_L);
  summer_R.setInput (&splitter_R);
  summer_L.setFeedbackInput (&feedbackamp_L);
  summer_R.setFeedbackInput (&feedbackamp_R);
  
  // Calculate feedback time

  double feedbackTime_L;
  double feedbackTime_R;
  double feedbackTime;

  feedbackTime_L = calcFeedbackTime (PARAMR(10),PARAMR(7));
  feedbackTime_R = calcFeedbackTime (PARAMR(11),PARAMR(8));

  if (feedbackTime_L > feedbackTime_R)
    feedbackTime = feedbackTime_L;
  else
    feedbackTime = feedbackTime_R;

  summer_L.setFeedbackTime (feedbackTime);
  summer_R.setFeedbackTime (feedbackTime);

  delay_L.setInputAndDelayTime (&summer_L,PARAMR(7) / 1000.0);
  delay_R.setInputAndDelayTime (&summer_R,PARAMR(8) / 1000.0);
  
  delaysplitter_L.setInput (&delay_L);
  delaysplitter_R.setInput (&delay_R);
  
  delaywetamp_L.setInput (&delaysplitter_L);
  delaywetamp_R.setInput (&delaysplitter_R);
  delaywetamp_L.setGain (PARAMR(16) / 100.0);
  delaywetamp_R.setGain (PARAMR(17) / 100.0);

  delaydryamp_L.setInput (&splitter_L);
  delaydryamp_R.setInput (&splitter_R);
  delaydryamp_L.setGain ((100.0 - PARAMR(16)) / 100.0);
  delaydryamp_R.setGain ((100.0 - PARAMR(17)) / 100.0);
  
  delaymixer_L.setInput (&delaydryamp_L);
  delaymixer_R.setInput (&delaydryamp_R);
  delaymixer_L.addInput (&delaywetamp_L);
  delaymixer_R.addInput (&delaywetamp_R);
  
  phaser_L.setInputAndDelayTime (&delaymixer_L,PARAMR(12) / 1000.0);
  phaser_R.setInputAndDelayTime (&delaymixer_R,PARAMR(13) / 1000.0);
  phaser_L.setG (PARAMR(14) / 100.0);
  phaser_R.setG (PARAMR(15) / 100.0);
  phaser_L.setModInput (&LFO_L);
  phaser_R.setModInput (&LFO_R);
  
  LFO_L.setTypeAndFreq
    (PARAME(5),rate,PARAMR(3),PARAMR(0) / 100.0,0.0);
  LFO_R.setTypeAndFreq
    (PARAME(6),rate,PARAMR(4),PARAMR(1) / 100.0,PARAMR(2));
  
  if (PARAME(9) == 0)
  {
    feedbackamp_L.setInput (&delaysplitter_L);
    feedbackamp_R.setInput (&delaysplitter_R);
  }
  else
  {
    feedbackamp_L.setInput (&delaysplitter_R);
    feedbackamp_R.setInput (&delaysplitter_L);
  }

  feedbackamp_L.setGain (PARAMR(10) / 100.0);
  feedbackamp_R.setGain (PARAMR(11) / 100.0);
  
  // Tends to be quiet => double volume
  mainAmp_L.setInput (&phaser_L);
  mainAmp_R.setInput (&phaser_R);
  mainAmp_L.setGain (2.0);
  mainAmp_R.setGain (2.0);
  
  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }
  
  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPDistortion
---------------------------------------------------------------------------*/

char *DSPDistortion
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPDistortion
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPDistortion
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 8;
  double rate   = inputSample->getRate ();
  double rate_2 = rate / 2.0;
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;

  DSPKitDPReader   reader_L;
  DSPKitDPReader   reader_R;
  DSPKitDistortion distShape_L;
  DSPKitDistortion distShape_R;
  DSPKitBWLowPassFilter LPF_L;
  DSPKitBWLowPassFilter LPF_R;
  DSPKitAmp        inAmp_L;
  DSPKitAmp        inAmp_R;
  DSPKitWaveShaper distortion_L;
  DSPKitWaveShaper distortion_R;
  DSPKitAmp        mainAmp_L;
  DSPKitAmp        mainAmp_R;
  DSPKitDPWriter   writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set connections
  
  if (PARAMR(3) < 0.9999)
  {
    LPF_L.setInput (&reader_L);
    LPF_R.setInput (&reader_R);
    LPF_L.setCutOffFreq (rate_2 * PARAMR(3));
    LPF_R.setCutOffFreq (rate_2 * PARAMR(3));
    
    inAmp_L.setInput (&LPF_L);
    inAmp_R.setInput (&LPF_R);
  }
  else
  {
    inAmp_L.setInput (&reader_L);
    inAmp_R.setInput (&reader_R);
  }
  
  double gain_L = 1.0 + (PARAMR(0) / 100.0);
  double gain_R = 1.0 + (PARAMR(1) / 100.0);

  inAmp_L.setGain (gain_L);
  inAmp_R.setGain (gain_R);
  
  distShape_L.setType (PARAME(2));
  distShape_R.setType (PARAME(2));
  
  distortion_L.setInputAndBuffer (&inAmp_L,&distShape_L);
  distortion_R.setInputAndBuffer (&inAmp_R,&distShape_R);

  mainAmp_L.setInput (&distortion_L);
  mainAmp_R.setInput (&distortion_R);
  mainAmp_L.setGain (1.0);
  mainAmp_R.setGain (1.0);
  
  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }

  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPFilter
---------------------------------------------------------------------------*/

char *DSPFilter
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPFilter
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPFilter
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 9;
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitBWLowPassFilter  LPF_L;
  DSPKitBWLowPassFilter  LPF_R;
  DSPKitBWHighPassFilter HPF_L;
  DSPKitBWHighPassFilter HPF_R;
  DSPKitBWBandPassFilter BPF_L;
  DSPKitBWBandPassFilter BPF_R;
  DSPKitBWBandRejectFilter BRF_L;
  DSPKitBWBandRejectFilter BRF_R;
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections

  switch (PARAME(0))
  {
    case 0 :
    
      LPF_L.setInput (&reader_L);
      LPF_R.setInput (&reader_R);
      LPF_L.setCutOffFreq (PARAMR(1) * 1000.0);
      LPF_R.setCutOffFreq (PARAMR(1) * 1000.0);
      
      mainAmp_L.setInput (&LPF_L);
      mainAmp_R.setInput (&LPF_R);
      mainAmp_L.setGain (0.8);
      mainAmp_R.setGain (0.8);
      
      break;
    
    case 1 :
    
      HPF_L.setInput (&reader_L);
      HPF_R.setInput (&reader_R);
      HPF_L.setCutOffFreq (PARAMR(2) * 1000.0);
      HPF_R.setCutOffFreq (PARAMR(2) * 1000.0);
      
      mainAmp_L.setInput (&HPF_L);
      mainAmp_R.setInput (&HPF_R);
      mainAmp_L.setGain (0.8);
      mainAmp_R.setGain (0.8);
      
      break;
    
    case 2 :
    
      BPF_L.setInput (&reader_L);
      BPF_R.setInput (&reader_R);
      BPF_L.setCenterFreqAndBW (PARAMR(3) * 1000.0,PARAMR(4) * 1000.0);
      BPF_R.setCenterFreqAndBW (PARAMR(3) * 1000.0,PARAMR(4) * 1000.0);
      
      mainAmp_L.setInput (&BPF_L);
      mainAmp_R.setInput (&BPF_R);
      mainAmp_L.setGain (0.8);
      mainAmp_R.setGain (0.8);
      
      break;
    
    case 3 :
    
      BRF_L.setInput (&reader_L);
      BRF_R.setInput (&reader_R);
      BRF_L.setCenterFreqAndBW (PARAMR(3) * 1000.0,PARAMR(4) * 1000.0);
      BRF_R.setCenterFreqAndBW (PARAMR(3) * 1000.0,PARAMR(4) * 1000.0);
      
      mainAmp_L.setInput (&BRF_L);
      mainAmp_R.setInput (&BRF_R);
      mainAmp_L.setGain (0.8);
      mainAmp_R.setGain (0.8);
      
      break;
    
    default :
      return "Unknown filter type";
  }
  
  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }

  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPAutoPan
---------------------------------------------------------------------------*/

char *DSPAutoPan
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPAutoPan
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPAutoPan
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 10;
  double rate   = inputSample->getRate ();
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;

  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitMux       splitter_L;
  DSPKitMux       splitter_R;
  DSPKitFeedback  summer_L;
  DSPKitFeedback  summer_R;
  DSPKitDelayInt  delay_L;
  DSPKitDelayInt  delay_R;
  DSPKitAmp       feedbackamp_L;
  DSPKitAmp       feedbackamp_R;
  DSPKitMux       delaysplitter_L;
  DSPKitMux       delaysplitter_R;
  DSPKitAmp       delaydryamp_L;
  DSPKitAmp       delaydryamp_R;
  DSPKitAmp       delaywetamp_L;
  DSPKitAmp       delaywetamp_R;
  DSPKitSum       delaymixer_L;
  DSPKitSum       delaymixer_R;
  DSPKitPan       pan_L;
  DSPKitPan       pan_R;
  DSPKitLFO       LFO_L;
  DSPKitLFO       LFO_R;
  DSPKitSum       mainSum_L;
  DSPKitSum       mainSum_R;
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections

  splitter_L.setInput (&reader_L);
  splitter_R.setInput (&reader_R);

  summer_L.setInput (&splitter_L);
  summer_R.setInput (&splitter_R);
  summer_L.setFeedbackInput (&feedbackamp_L);
  summer_R.setFeedbackInput (&feedbackamp_R);
  
  // Calculate feedback time

  double feedbackTime_L;
  double feedbackTime_R;
  double feedbackTime;

  feedbackTime_L = calcFeedbackTime (PARAMR(10),PARAMR(7));
  feedbackTime_R = calcFeedbackTime (PARAMR(11),PARAMR(8));
  
  if (feedbackTime_L > feedbackTime_R)
    feedbackTime = feedbackTime_L;
  else
    feedbackTime = feedbackTime_R;

  summer_L.setFeedbackTime (feedbackTime);
  summer_R.setFeedbackTime (feedbackTime);

  delay_L.setInputAndDelayTime (&summer_L,PARAMR(7) / 1000.0);
  delay_R.setInputAndDelayTime (&summer_R,PARAMR(8) / 1000.0);
  
  delaysplitter_L.setInput (&delay_L);
  delaysplitter_R.setInput (&delay_R);
  
  delaywetamp_L.setInput (&delaysplitter_L);
  delaywetamp_R.setInput (&delaysplitter_R);
  delaywetamp_L.setGain (PARAMR(12) / 100.0);
  delaywetamp_R.setGain (PARAMR(13) / 100.0);

  delaydryamp_L.setInput (&splitter_L);
  delaydryamp_R.setInput (&splitter_R);
  delaydryamp_L.setGain ((100.0 - PARAMR(12)) / 100.0);
  delaydryamp_R.setGain ((100.0 - PARAMR(13)) / 100.0);
  
  delaymixer_L.setInput (&delaydryamp_L);
  delaymixer_R.setInput (&delaydryamp_R);
  delaymixer_L.addInput (&delaywetamp_L);
  delaymixer_R.addInput (&delaywetamp_R);
  
  pan_L.setInput (&delaymixer_L);
  pan_R.setInput (&delaymixer_R);
  pan_L.setModInput (&LFO_L);
  pan_R.setModInput (&LFO_R);
  
  LFO_L.setTypeAndFreq
    (PARAME(5),rate,PARAMR(3),PARAMR(0) / 100.0,0.0);
  LFO_R.setTypeAndFreq
    (PARAME(6),rate,PARAMR(4),PARAMR(1) / 100.0,PARAMR(2));
  
  if (PARAME(9) == 0)
  {
    feedbackamp_L.setInput (&delaysplitter_L);
    feedbackamp_R.setInput (&delaysplitter_R);
  }
  else
  {
    feedbackamp_L.setInput (&delaysplitter_R);
    feedbackamp_R.setInput (&delaysplitter_L);
  }

  feedbackamp_L.setGain (PARAMR(10) / 100.0);
  feedbackamp_R.setGain (PARAMR(11) / 100.0);
  
  mainSum_L.setInput (&pan_L);
  mainSum_L.addInput (&pan_R);
  mainSum_R.setInput (&pan_L);
  mainSum_R.addInput (&pan_R);
  
  mainAmp_L.setInput (&mainSum_L);
  mainAmp_R.setInput (&mainSum_R);
  mainAmp_L.setGain (0.8);
  mainAmp_R.setGain (0.8);
  
  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }
  
  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPEqualizer
---------------------------------------------------------------------------*/

char *DSPEqualizer
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPEqualizer
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPEqualizer
  (DPSample *inputSample,DPSample *,int,int,
  int,int,int)
{
  //int effectNo  = 11;
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  return "Not yet implemented";
}

/*---------------------------------------------------------------------------
| FUNCTION DSPGate
---------------------------------------------------------------------------*/

char *DSPGate
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPGate
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPGate
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 12;
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitDelay     predelay_L;
  DSPKitDelay     predelay_R;
  DSPKitGate      gate_L;
  DSPKitGate      gate_R;
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections

  predelay_L.setInputAndDelayTime (&reader_L,PARAMR(4) / 1000.0);
  predelay_R.setInputAndDelayTime (&reader_R,PARAMR(5) / 1000.0);
  
  gate_L.setInputAndEstimationTime (&predelay_L,PARAMR(0) / 1000.0);
  gate_R.setInputAndEstimationTime (&predelay_R,PARAMR(1) / 1000.0);
  gate_L.setThreshold (PARAMR(2) / 100.0);
  gate_R.setThreshold (PARAMR(3) / 100.0);
  
  mainAmp_L.setInput (&gate_L);
  mainAmp_R.setInput (&gate_R);
  mainAmp_L.setGain (1.0);
  mainAmp_R.setGain (1.0);
  
  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }

  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION DSPCompressor
---------------------------------------------------------------------------*/

char *DSPCompressor
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop)
{
  return DSPCompressor
    (inputSample,outputSample,susLoop,relLoop,TRUE,0,1);
}

char *DSPCompressor
  (DPSample *inputSample,DPSample *outputSample,int susLoop,int relLoop,
  int first,int,int)
{
  int effectNo  = 13;
  int channels;
  
  if (inputSample->getChannels () == 4)
    channels = 4;
  else
    channels = 2;
  
  DSPKitDPReader  reader_L;
  DSPKitDPReader  reader_R;
  DSPKitCompressor            comp_L;
  DSPKitCompressor            comp_R;
  DSPKitLimiter               limit_L;
  DSPKitLimiter               limit_R;
  DSPKitCompressorLimiter     complimit_L;
  DSPKitCompressorLimiter     complimit_R;
  DSPKitAmp       mainAmp_L;
  DSPKitAmp       mainAmp_R;
  DSPKitDPWriter  writer;
  
  if (reader_L.setReaderInput(inputSample,0,susLoop,relLoop) ||
      reader_R.setReaderInput(inputSample,1,susLoop,relLoop))
  {
    return "Unable to set reader inputs";
  }

  if (first)
  {
    if (!(outputSample->fresh (
      inputSample->getRate (),
      2,channels,                          // Always produce 16 bit
      reader_L.getSampleCount ())))
    {
      return "Unable to create output sample - out of memory";
    }

    if (!(outputSample->cloneSimple (*inputSample)))
    {
      return "Unable to clone input sample details - out of memory";
    }
  }
  
  // Set Connections

  switch (PARAME(0))
  {
    case 0 :
    
      comp_L.setInputAndEstimationTime (&reader_L,PARAMR(2) / 1000.0);
      comp_L.setRatio (PARAMR(1));
      comp_L.setThreshold (PARAMR(3) / 100.0);
      comp_R.setInputAndEstimationTime (&reader_R,PARAMR(2) / 1000.0);
      comp_R.setRatio (PARAMR(1));
      comp_R.setThreshold (PARAMR(3) / 100.0);
      
      mainAmp_L.setInput (&comp_L);
      mainAmp_R.setInput (&comp_R);
      mainAmp_L.setGain (1.0);
      mainAmp_R.setGain (1.0);
      
      break;
    
    case 1 :
    
      limit_L.setInputAndEstimationTime (&reader_L,PARAMR(4) / 1000.0);
      limit_L.setThreshold (PARAMR(5) / 100.0);
      limit_R.setInputAndEstimationTime (&reader_R,PARAMR(4) / 1000.0);
      limit_R.setThreshold (PARAMR(5) / 100.0);
      
      mainAmp_L.setInput (&limit_L);
      mainAmp_R.setInput (&limit_R);
      mainAmp_L.setGain (1.0);
      mainAmp_R.setGain (1.0);
      
      break;
    
    case 2 :
    
      complimit_L.setInputAndEstimationTime
        (&reader_L,PARAMR(2) / 1000.0,PARAMR(4) / 1000.0);
      complimit_L.setCompressionRatio (PARAMR(1));
      complimit_L.setCompressorThreshold (PARAMR(3) / 100.0);
      complimit_L.setLimiterThreshold (PARAMR(5) / 100.0);
      complimit_R.setInputAndEstimationTime
        (&reader_R,PARAMR(2) / 1000.0,PARAMR(4) / 1000.0);
      complimit_R.setCompressionRatio (PARAMR(1));
      complimit_R.setCompressorThreshold (PARAMR(3) / 100.0);
      complimit_R.setLimiterThreshold (PARAMR(5) / 100.0);
      
      mainAmp_L.setInput (&complimit_L);
      mainAmp_R.setInput (&complimit_R);
      mainAmp_L.setGain (1.0);
      mainAmp_R.setGain (1.0);
      
      break;
    
    default :
      return "Unknown mode";
  }
  
  if (writer.setInputOutput(&mainAmp_L,&mainAmp_R,outputSample))
  {
    return "Unable to set writer outputs";
  }

  if (first)
    writer.setChannels (0,1);
  else
    writer.setChannels (2,3);
  
  DSPTime     = time (NULL);
  DSPReader_L = &reader_L;
  DSPReader_R = &reader_R;
  DSPWriter   = &writer;
  DSPStop     = 0;
  DSPCancel   = 0;
  
  writer.run();

  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION calcFeedbackTime
---------------------------------------------------------------------------*/
double calcFeedbackTime (double feedback,double delayTime)
{
  // feedback in percent
  // delayTime in milliseconds
  
  double absFeedback = feedback < 0.0 ? -feedback : feedback;
  
  // No feedback time if less than 0.01 percent
  if (absFeedback < 0.01) return 0.0;
  
  if (absFeedback < 0.1) absFeedback = 0.1;
  else if (absFeedback > 99.9) absFeedback = 99.9;
  
  return
    (3.0 * (delayTime / 1000.0) / log10 (1.0 / (absFeedback / 100.0)));

//   if (absFeedback < 99.0)
//     return ((delayTime / 1000.0) / (1.0 - (absFeedback / 100.0)));
//   else
//     return (100.0 * (delayTime / 1000.0));

}

/***************************************************************************/
