/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : macros.c
|
| Object      : None
|
| Description : Macro routines for linux replacement code
|
| (c) Richard Kent 1997
|
| $Id: macros.c,v 1.1 2003/09/10 00:06:24 rk Exp $
|
****************************************************************************/

static char macros_c [] = "$Id: macros.c,v 1.1 2003/09/10 00:06:24 rk Exp $";

#include "macros.h"
#include "audiofile.h"

/* Note - the define BIGENDIAN states what the CPUs architecture is    */
/* whereas the bigEndian variable controls what we want to read/write. */

static int bigEndian = FALSE;

/*---------------------------------------------------------------------------
| FUNCTION setBigEndian
---------------------------------------------------------------------------*/
void setEndian (int newEndian)
{
#ifdef BIGENDIAN
  setBigEndian (newEndian == AF_BIG_ENDIAN_FORMAT ||
    newEndian == AF_DEF_ENDIAN_FORMAT);
#else
  setBigEndian (newEndian == AF_BIG_ENDIAN_FORMAT);
#endif
}

/*---------------------------------------------------------------------------
| FUNCTION setBigEndian
---------------------------------------------------------------------------*/
void setBigEndian (int newBigEndian)
{
  bigEndian = newBigEndian;
}

/*---------------------------------------------------------------------------
| FUNCTION getBigEndian
---------------------------------------------------------------------------*/
int getBigEndian (void)
{
  return bigEndian;
}

/*---------------------------------------------------------------------------
| FUNCTION mRead
---------------------------------------------------------------------------*/
int mRead (FHandle file,void *buf,int count)
{
  #ifdef DIRECT_FILE
  if (read (file,buf,count) != count) return 0;
  #else
  if (fread (buf,count,1,file) != 1) return 0;
  #endif
  
  return count;
}

int mReadF (FILE *file,void *buf,int count)
{
  if (fread (buf,count,1,file) != 1) return 0;
  return count;
}

/*---------------------------------------------------------------------------
| FUNCTION uiRead
---------------------------------------------------------------------------*/
int uiRead (FHandle file,unsigned int *ui)
{
  #ifdef DIRECT_FILE
  if (read (file,ui,sizeof(*ui)) != sizeof(*ui)) return 0;
  #else
  if (fread (ui,sizeof(*ui),1,file) != 1) return 0;
  #endif

  #ifndef BIGENDIAN
  if (bigEndian)
    *ui = swapbytes_uint (*ui);
  #else
  if (!bigEndian)
    *ui = swapbytes_uint (*ui);
  #endif
  
  return sizeof(*ui);
}

int uiReadF (FILE *file,unsigned int *ui)
{
  if (fread (ui,sizeof(*ui),1,file) != 1) return 0;

  #ifndef BIGENDIAN
  if (bigEndian)
    *ui = swapbytes_uint (*ui);
  #else
  if (!bigEndian)
    *ui = swapbytes_uint (*ui);
  #endif
  
  return sizeof(*ui);
}

/*---------------------------------------------------------------------------
| FUNCTION iRead
---------------------------------------------------------------------------*/
int iRead (FHandle file,int *i)
{
  #ifdef DIRECT_FILE
  if (read (file,i,sizeof(*i)) != sizeof(*i)) return 0;
  #else
  if (fread (i,sizeof(*i),1,file) != 1) return 0;
  #endif
  
  #ifndef BIGENDIAN
  if (bigEndian)
    *i = swapbytes_int (*i);
  #else
  if (!bigEndian)
    *i = swapbytes_int (*i);
  #endif
  
  return sizeof(*i);
}

int iReadF (FILE *file,int *i)
{
  if (fread (i,sizeof(*i),1,file) != 1) return 0;
  
  #ifndef BIGENDIAN
  if (bigEndian)
    *i = swapbytes_int (*i);
  #else
  if (!bigEndian)
    *i = swapbytes_int (*i);
  #endif
  
  return sizeof(*i);
}

/*---------------------------------------------------------------------------
| FUNCTION usRead
---------------------------------------------------------------------------*/
int usRead (FHandle file,unsigned short *us)
{
  #ifdef DIRECT_FILE
  if (read (file,us,sizeof(*us)) != sizeof(*us)) return 0;
  #else
  if (fread (us,sizeof(*us),1,file) != 1) return 0;
  #endif
  
  #ifndef BIGENDIAN
  if (bigEndian)
    *us = swapbytes_ushort (*us);
  #else
  if (!bigEndian)
    *us = swapbytes_ushort (*us);
  #endif
  
  return sizeof(*us);
}

int usReadF (FILE *file,unsigned short *us)
{
  if (fread (us,sizeof(*us),1,file) != 1) return 0;
  
  #ifndef BIGENDIAN
  if (bigEndian)
    *us = swapbytes_ushort (*us);
  #else
  if (!bigEndian)
    *us = swapbytes_ushort (*us);
  #endif
  
  return sizeof(*us);
}

/*---------------------------------------------------------------------------
| FUNCTION sRead
---------------------------------------------------------------------------*/
int sRead (FHandle file,short *s)
{
  #ifdef DIRECT_FILE
  if (read (file,s,sizeof(*s)) != sizeof(*s)) return 0;
  #else
  if (fread (s,sizeof(*s),1,file) != 1) return 0;
  #endif
  
  #ifndef BIGENDIAN
  if (bigEndian)
    *s = swapbytes_short (*s);
  #else
  if (!bigEndian)
    *s = swapbytes_short (*s);
  #endif
  
  return sizeof(*s);
}

int sReadF (FILE *file,short *s)
{
  if (fread (s,sizeof(*s),1,file) != 1) return 0;
  
  #ifndef BIGENDIAN
  if (bigEndian)
    *s = swapbytes_short (*s);
  #else
  if (!bigEndian)
    *s = swapbytes_short (*s);
  #endif
  
  return sizeof(*s);
}

/*---------------------------------------------------------------------------
| FUNCTION ucRead
---------------------------------------------------------------------------*/
int ucRead (FHandle file,unsigned char *uc)
{
  #ifdef DIRECT_FILE
  if (read (file,uc,sizeof(*uc)) != sizeof(*uc)) return 0;
  #else
  if (fread (uc,sizeof(*uc),1,file) != 1) return 0;
  #endif
  
  return sizeof(*uc);
}

int ucReadF (FILE *file,unsigned char *uc)
{
  if (fread (uc,sizeof(*uc),1,file) != 1) return 0;
  return sizeof(*uc);
}

/*---------------------------------------------------------------------------
| FUNCTION cRead
---------------------------------------------------------------------------*/
int cRead (FHandle file,signed char *c)
{
  #ifdef DIRECT_FILE
  if (read (file,c,sizeof(*c)) != sizeof(*c)) return 0;
  #else
  if (fread (c,sizeof(*c),1,file) != 1) return 0;
  #endif
  
  return sizeof(*c);
}

int cReadF (FILE *file,signed char *c)
{
  if (fread (c,sizeof(*c),1,file) != 1) return 0;
  return sizeof(*c);
}

/*---------------------------------------------------------------------------
| FUNCTION fRead
---------------------------------------------------------------------------*/
int fRead (FHandle file,float *f)
{
  #ifdef DIRECT_FILE
  if (read (file,f,sizeof(*f)) != sizeof(*f)) return 0;
  #else
  if (fread (f,sizeof(*f),1,file) != 1) return 0;
  #endif
  
  return sizeof(*f);
}

int fReadF (FILE *file,float *f)
{
  if (fread (f,sizeof(*f),1,file) != 1) return 0;
  return sizeof(*f);
}

/*---------------------------------------------------------------------------
| FUNCTION dRead
---------------------------------------------------------------------------*/
int dRead (FHandle file,double *d)
{
  #ifdef DIRECT_FILE
  if (read (file,d,sizeof(*d)) != sizeof(*d)) return 0;
  #else
  if (fread (d,sizeof(*d),1,file) != 1) return 0;
  #endif
  
  return sizeof(*d);
}

int dReadF (FILE *file,double *d)
{
  if (fread (d,sizeof(*d),1,file) != 1) return 0;
  return sizeof(*d);
}

/*---------------------------------------------------------------------------
| FUNCTION efRead
---------------------------------------------------------------------------*/
int efRead (FHandle file,float *f)
{
  unsigned char etemp [10];
  
  #ifdef DIRECT_FILE
  if (read (file,etemp,10) != 10) return 0;
  #else
  if (fread (etemp,10,1,file) != 1) return 0;
  #endif
  
  *f = (float) convertFromIeeeExtended (etemp);
  return 10;
}

int efReadF (FILE *file,float *f)
{
  unsigned char etemp [10];
  if (fread (etemp,10,1,file) != 1) return 0;
  *f = (float) convertFromIeeeExtended (etemp);
  return 10;
}

/*---------------------------------------------------------------------------
| FUNCTION edRead
---------------------------------------------------------------------------*/
int edRead (FHandle file,double *d)
{
  unsigned char etemp [10];
  
  #ifdef DIRECT_FILE
  if (read (file,etemp,10) != 10) return 0;
  #else
  if (fread (etemp,10,1,file) != 1) return 0;
  #endif
  
  *d = convertFromIeeeExtended (etemp);
  return 10;
}

int edReadF (FILE *file,double *d)
{
  unsigned char etemp [10];
  if (fread (etemp,10,1,file) != 1) return 0;
  *d = convertFromIeeeExtended (etemp);
  return 10;
}

/*---------------------------------------------------------------------------
| FUNCTION idRead
---------------------------------------------------------------------------*/
int idRead (FHandle file,char *c)
{
  #ifdef DIRECT_FILE
  if (read (file,c,4) != 4) return 0;
  #else
  if (fread (c,4,1,file) != 1) return 0;
  #endif
  
  return 4;
}

int idReadF (FILE *file,char *c)
{
  if (fread (c,4,1,file) != 1) return 0;
  return 4;
}

/*---------------------------------------------------------------------------
| FUNCTION pstringRead
---------------------------------------------------------------------------*/
int pstringRead (FHandle file,char **c)
{
  unsigned char stringsize;
  unsigned char temp;
  int count;
  *c = 0;  
  if (!ucRead (file,&stringsize)) return 0;
  *c = (char *) calloc (stringsize + 1,sizeof (char));
  if (!(*c)) return 0;
  if (stringsize > 0)
  {
    #ifdef DIRECT_FILE
    if (read (file,*c,stringsize) != stringsize) return 0;
    #else
    if (fread (*c,stringsize,1,file) != 1) return 0;
    #endif
  }
  (*c) [stringsize] = 0;
  count = 1 + stringsize;
  if (!(stringsize % 2))
  {
    if (!ucRead (file,&temp)) return 0;
    count++;
  }
  return count;
}

int pstringReadF (FILE *file,char **c)
{
  unsigned char stringsize;
  unsigned char temp;
  int count;
  *c = 0;
  if (!ucReadF (file,&stringsize)) return 0;
  *c = (char *) calloc (stringsize + 1,sizeof (char));
  if (!(*c)) return 0;
  if (stringsize > 0)
  {
    if (fread (*c,stringsize,1,file) != 1) return 0;
  }
  (*c) [stringsize] = 0;
  count = 1 + stringsize;
  if (!(stringsize % 2))
  {
    if (!ucReadF (file,&temp)) return 0;
    count++;
  }
  return count;
}

/*---------------------------------------------------------------------------
| FUNCTION mWrite
---------------------------------------------------------------------------*/
int mWrite (FHandle file,void *buf,int count)
{
  #ifdef DIRECT_FILE
  if (write (file,buf,count) != count) return 0;
  #else
  if (fwrite (buf,count,1,file) != 1) return 0;
  #endif
  
  return count;
}

int mWriteF (FILE *file,void *buf,int count)
{
  if (fwrite (buf,count,1,file) != 1) return 0;
  return count;
}

/*---------------------------------------------------------------------------
| FUNCTION uiWrite
---------------------------------------------------------------------------*/
int uiWrite (FHandle file,unsigned int ui)
{
  #ifndef BIGENDIAN
  if (bigEndian)
    ui = swapbytes_uint (ui);
  #else
  if (!bigEndian)
    ui = swapbytes_uint (ui);
  #endif
  
  #ifdef DIRECT_FILE
  if (write (file,&ui,sizeof(ui)) != sizeof(ui)) return 0;
  #else
  if (fwrite (&ui,sizeof(ui),1,file) != 1) return 0;
  #endif
  
  return sizeof(ui);
}

int uiWriteF (FILE *file,unsigned int ui)
{
  #ifndef BIGENDIAN
  if (bigEndian)
    ui = swapbytes_uint (ui);
  #else
  if (!bigEndian)
    ui = swapbytes_uint (ui);
  #endif
  
  if (fwrite (&ui,sizeof(ui),1,file) != 1) return 0;
  return sizeof(ui);
}

/*---------------------------------------------------------------------------
| FUNCTION iWrite
---------------------------------------------------------------------------*/
int iWrite (FHandle file,int i)
{
  #ifndef BIGENDIAN
  if (bigEndian)
    i = swapbytes_int (i);
  #else
  if (!bigEndian)
    i = swapbytes_int (i);
  #endif

  #ifdef DIRECT_FILE
  if (write (file,&i,sizeof(i)) != sizeof(i)) return 0;
  #else
  if (fwrite (&i,sizeof(i),1,file) != 1) return 0;
  #endif
  
  return sizeof(i);
}

int iWriteF (FILE *file,int i)
{
  #ifndef BIGENDIAN
  if (bigEndian)
    i = swapbytes_int (i);
  #else
  if (!bigEndian)
    i = swapbytes_int (i);
  #endif

  if (fwrite (&i,sizeof(i),1,file) != 1) return 0;
  return sizeof(i);
}

/*---------------------------------------------------------------------------
| FUNCTION usWrite
---------------------------------------------------------------------------*/
int usWrite (FHandle file,unsigned short us)
{
  #ifndef BIGENDIAN
  if (bigEndian)
    us = swapbytes_ushort (us);
  #else
  if (!bigEndian)
    us = swapbytes_ushort (us);
  #endif
  
  #ifdef DIRECT_FILE
  if (write (file,&us,sizeof(us)) != sizeof(us)) return 0;
  #else
  if (fwrite (&us,sizeof(us),1,file) != 1) return 0;
  #endif
  
  return sizeof(us);
}

int usWriteF (FILE *file,unsigned short us)
{
  #ifndef BIGENDIAN
  if (bigEndian)
    us = swapbytes_ushort (us);
  #else
  if (!bigEndian)
    us = swapbytes_ushort (us);
  #endif
  
  if (fwrite (&us,sizeof(us),1,file) != 1) return 0;
  return sizeof(us);
}

/*---------------------------------------------------------------------------
| FUNCTION sWrite
---------------------------------------------------------------------------*/
int sWrite (FHandle file,short s)
{
  #ifndef BIGENDIAN
  if (bigEndian)
    s = swapbytes_short (s);
  #else
  if (!bigEndian)
    s = swapbytes_short (s);
  #endif
  
  #ifdef DIRECT_FILE
  if (write (file,&s,sizeof(s)) != sizeof(s)) return 0;
  #else
  if (fwrite (&s,sizeof(s),1,file) != 1) return 0;
  #endif
  
  return sizeof(s);
}

int sWriteF (FILE *file,short s)
{
  #ifndef BIGENDIAN
  if (bigEndian)
    s = swapbytes_short (s);
  #else
  if (!bigEndian)
    s = swapbytes_short (s);
  #endif
  
  if (fwrite (&s,sizeof(s),1,file) != 1) return 0;
  return sizeof(s);
}

/*---------------------------------------------------------------------------
| FUNCTION ucWrite
---------------------------------------------------------------------------*/
int ucWrite (FHandle file,unsigned char uc)
{
  #ifdef DIRECT_FILE
  if (write (file,&uc,sizeof(uc)) != sizeof(uc)) return 0;
  #else
  if (fwrite (&uc,sizeof(uc),1,file) != 1) return 0;
  #endif
  
  return sizeof(uc);
}

int ucWriteF (FILE *file,unsigned char uc)
{
  if (fwrite (&uc,sizeof(uc),1,file) != 1) return 0;
  return sizeof(uc);
}

/*---------------------------------------------------------------------------
| FUNCTION cWrite
---------------------------------------------------------------------------*/
int cWrite (FHandle file,signed char c)
{
  #ifdef DIRECT_FILE
  if (write (file,&c,sizeof(c)) != sizeof(c)) return 0;
  #else
  if (fwrite (&c,sizeof(c),1,file) != 1) return 0;
  #endif
  
  return sizeof(c);
}

int cWriteF (FILE *file,signed char c)
{
  if (fwrite (&c,sizeof(c),1,file) != 1) return 0;
  return sizeof(c);
}

/*---------------------------------------------------------------------------
| FUNCTION fWrite
---------------------------------------------------------------------------*/
int fWrite (FHandle file,float f)
{
  #ifdef DIRECT_FILE
  if (write (file,&f,sizeof(f)) != sizeof(f)) return 0;
  #else
  if (fwrite (&f,sizeof(f),1,file) != 1) return 0;
  #endif
  
  return sizeof(f);
}

int fWriteF (FILE *file,float f)
{
  if (fwrite (&f,sizeof(f),1,file) != 1) return 0;
  return sizeof(f);
}

/*---------------------------------------------------------------------------
| FUNCTION dWrite
---------------------------------------------------------------------------*/
int dWrite (FHandle file,double d)
{
  #ifdef DIRECT_FILE
  if (write (file,&d,sizeof(d)) != sizeof(d)) return 0;
  #else
  if (fwrite (&d,sizeof(d),1,file) != 1) return 0;
  #endif
  
  return sizeof(d);
}

int dWriteF (FILE *file,double d)
{
  if (fwrite (&d,sizeof(d),1,file) != 1) return 0;
  return sizeof(d);
}

/*---------------------------------------------------------------------------
| FUNCTION efWrite
---------------------------------------------------------------------------*/
int efWrite (FHandle file,float f)
{
  unsigned char etemp [10];
  convertToIeeeExtended ((double) f,etemp);
  
  #ifdef DIRECT_FILE
  if (write (file,etemp,10) != 10) return 0;
  #else
  if (fwrite (etemp,10,1,file) != 1) return 0;
  #endif
  
  return 10;
}

int efWriteF (FILE *file,float f)
{
  unsigned char etemp [10];
  convertToIeeeExtended ((double) f,etemp);
  if (fwrite (etemp,10,1,file) != 1) return 0;
  return 10;
}

/*---------------------------------------------------------------------------
| FUNCTION edWrite
---------------------------------------------------------------------------*/
int edWrite (FHandle file,double d)
{
  unsigned char etemp [10];
  convertToIeeeExtended (d,etemp);
  
  #ifdef DIRECT_FILE
  if (write (file,etemp,10) != 10) return 0;
  #else
  if (fwrite (etemp,10,1,file) != 1) return 0;
  #endif
  
  return 10;
}

int edWriteF (FILE * file,double d)
{
  unsigned char etemp [10];
  convertToIeeeExtended (d,etemp);
  if (fwrite (etemp,10,1,file) != 1) return 0;
  return 10;
}

/*---------------------------------------------------------------------------
| FUNCTION idWrite
---------------------------------------------------------------------------*/
int idWrite (FHandle file,char *c)
{
  #ifdef DIRECT_FILE
  if (write (file,c,4) != 4) return 0;
  #else
  if (fwrite (c,4,1,file) != 1) return 0;
  #endif
  
  return 4;
}

int idWriteF (FILE *file,char *c)
{
  if (fwrite (c,4,1,file) != 1) return 0;
  return 4;
}

/*---------------------------------------------------------------------------
| FUNCTION pstringWrite
---------------------------------------------------------------------------*/
int pstringWrite (FHandle file,char *c)
{
  unsigned char stringsize;
  unsigned char temp;
  int count;
  
  if (c) stringsize = strlen (c);
  else stringsize = 0;
  if (!ucWrite (file,stringsize)) return 0;
  if (stringsize)
  {
    #ifdef DIRECT_FILE
    if (write (file,c,stringsize) != stringsize) return 0;
    #else
    if (fwrite (c,stringsize,1,file) != 1) return 0;
    #endif
  }
  count = 1 + stringsize;
  if (!(stringsize % 2))
  {
    temp = 0;
    if (!(ucWrite (file,temp))) return 0;
    count++;
  }
  return count;
}

int pstringWriteF (FILE *file,char *c)
{
  unsigned char stringsize;
  unsigned char temp;
  int count;
  
  if (c) stringsize = strlen (c);
  else stringsize = 0;
  if (!ucWriteF (file,stringsize)) return 0;
  if (stringsize)
  {
    if (fwrite (c,stringsize,1,file) != 1) return 0;
  }
  count = 1 + stringsize;
  if (!(stringsize % 2))
  {
    temp = 0;
    if (!(ucWriteF (file,temp))) return 0;
    count++;
  }
  return count;
}

/*---------------------------------------------------------------------------
| FUNCTION mOpenRead
---------------------------------------------------------------------------*/
FHandle mOpenRead (const char *filename)
{
  #ifdef DIRECT_FILE
  return open (filename,O_RDONLY);
  #else
  return fopen (filename,"rb");
  #endif
}

FILE *mOpenReadF (const char *filename)
{
  return fopen (filename,"rb");
}

/*---------------------------------------------------------------------------
| FUNCTION mOpenWrite
---------------------------------------------------------------------------*/
FHandle mOpenWrite (const char *filename)
{
  // NB decimal 438 = octal 666
  #ifdef DIRECT_FILE
  return open (filename,O_RDWR | O_CREAT | O_TRUNC,438);
  #else
  FILE *file = fopen (filename,"wb");
  if (file) fchmod (fileno(file),438);
  return file;
  #endif
}

FILE *mOpenWriteF (const char *filename)
{
  // NB decimal 438 = octal 666
  FILE *file = fopen (filename,"wb");
  if (file) fchmod (fileno(file),438);
  return file;
}

/*---------------------------------------------------------------------------
| FUNCTION mCheckFile
---------------------------------------------------------------------------*/
int mCheckFile (FHandle file)
{
  #ifdef DIRECT_FILE
  if (file == -1) return 0;
  return 1;
  #else
  if (!file) return 0;
  return 1;
  #endif
}

int mCheckFileF (FILE *file)
{
  if (!file) return 0;
  return 1;
}

/*---------------------------------------------------------------------------
| FUNCTION mClose
---------------------------------------------------------------------------*/
int mClose (FHandle file)
{
  #ifdef DIRECT_FILE
  return close (file);
  #else
  if (fclose (file) == EOF) return -1;
  return 0;
  #endif
}

int mCloseF (FILE *file)
{
  if (fclose (file) == EOF) return -1;
  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION mSeek
---------------------------------------------------------------------------*/
int mSeek (FHandle file,int offset,int where)
{
  #ifdef DIRECT_FILE
  return lseek (file,offset,where);
  #else
  return fseek (file,offset,where);
  #endif
}

int mSeekF (FILE *file,int offset,int where)
{
  return fseek (file,offset,where);
}

/*---------------------------------------------------------------------------
| FUNCTION mTell
---------------------------------------------------------------------------*/
int mTell (FHandle file)
{
  #ifdef DIRECT_FILE
  return lseek (file,0L,SEEK_CUR);
  #else
  return ftell (file);
  #endif
}

int mTellF (FILE *file)
{
  return ftell (file);
}

/*---------------------------------------------------------------------------
| FUNCTION mEof
---------------------------------------------------------------------------*/
int mEof (FHandle file)
{
  #ifdef DIRECT_FILE
  /* Cannot detect eof so always return false */
  return 0;
  #else
  return feof (file);
  #endif
}

int mEofF (FILE *file)
{
  return feof (file);
}

/*---------------------------------------------------------------------------
| FUNCTION mFlush
---------------------------------------------------------------------------*/
int mFlush (FHandle file)
{
  #ifdef DIRECT_FILE
  /* No need to flush so just return success */
  return 0;
  #else
  if (fflush (file) == EOF) return -1;
  return 0;
  #endif
}

int mFlushF (FILE *file)
{
  if (fflush (file) == EOF) return -1;
  return 0;
}

/*---------------------------------------------------------------------------
| FUNCTION mStat
---------------------------------------------------------------------------*/
int mStat (FHandle file,struct stat *buf)
{
  #ifdef DIRECT_FILE
  return fstat (file,buf);
  #else
  return fstat (fileno(file),buf);
  #endif  
}

int mStatF (FILE *file,struct stat *buf)
{
  return fstat (fileno(file),buf);
}

/*---------------------------------------------------------------------------
| FUNCTION swapbytes_type functions
---------------------------------------------------------------------------*/
void swapbytes_string (char *buffer,int n)
{
  int i;
  int j;
  char temp;
  
  if (n < 1) return;
  i = 0;
  j = n - 1;
  while (i < j)
  {
    temp = buffer [i];
    buffer [i] = buffer [j];
    buffer [j] = temp;
    i++;
    j--;
  }
}

unsigned int swapbytes_uint (unsigned int ui)
{
  unsigned int uiNew = ui;
  swapbytes_string (((char *) &uiNew),sizeof (uiNew));
  return uiNew;
}

int swapbytes_int (int i)
{
  int iNew = i;
  swapbytes_string (((char *) &iNew),sizeof (iNew));
  return iNew;
}

unsigned short swapbytes_ushort (unsigned short us)
{
  unsigned short usNew = us;
  swapbytes_string (((char *) &usNew),sizeof (usNew));
  return usNew;
}

short swapbytes_short (short s)
{
  short sNew = s;
  swapbytes_string (((char *) &sNew),sizeof (sNew));
  return sNew;
}

double swapbytes_double (double d)
{
  double dNew = d;
  swapbytes_string (((char *) &dNew),sizeof (dNew));
  return dNew;
}

/***************************************************************************/
