/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
	main.c   Cut Ogg/Vorbis files pagewise.
	The code is somewhat ugly, sorry! At least it is fast :-)
	(c) Jochen Puchalla 2006-2007  <mail@puchalla-online.de>
*/

#include "cutoggvorbis.h"

#define VERSION "0.2"
#define YEAR "2007"

/* general buffersize */
#define BUFFER 65535

	unsigned int silencelength=1000, silvol=1;

	FILE *oggfile;
	signed int bospage[BUFFER]; /* for beginning of stream */
	long filesize, inpoint, outpoint;
	signed int startoff=1, endoff=1; /* are -1 if seeking from end of file */
	int totalmins, totalsecs, audiobegin=0;
	long double totalseconds; /* total time in seconds */
	int channels=0, samplerate=0;
	int bitratemax=0, bitratemin=0;
	long startmin, endmin; /* for non-interactive cutting */
	double startsec, endsec; /* for non-interactive cutting */
	long double offset=0; /* time in seconds at start of audio data */
	signed int avbr=0; /* average bitrate */
	int playpages=4; /* pages to play for interactive mode */

	char *filename;
	char *forcedname;
	char *userin;
	char *userout;

	int mute=0, forced_file=0, overwrite=0;
	int nonint=0;
 	int stdoutwrite=0;

	int card=1;
	signed int genre=-1;
	int hastable=0;

	int debug=0; /* 1: seek forward
					2: pagesize
					3: pcmbytes */

void exitseq(int foobar)
{
	remove("/tmp/cut.ogg");
	remove("/tmp/timetable");

	exit(foobar);  /* now really exit */
}

void usage(char *text)
{
	printf("\ncutoggvorbis  version %s  \n", VERSION);
	printf("\nUsage:  cutoggvorbis -i file.ogg [-a inpoint] [-b outpoint] [-f timetable]");
	printf("\n                                 [-o outputprefix] [-q]\n");
	printf("\n  cutoggvorbis -i file.ogg -a 0:37 -b 3:57  copies valid data from 0:37 to 3:57");
	printf("\n  cutoggvorbis -i file.ogg -f timetable     copies valid data described in timetable");
	printf("\n  cutoggvorbis -i file.ogg -o song          writes the output to song0001.ogg, song0002.ogg,...");
	printf("\n  cutoggvorbis -i file.ogg -O song.ogg      writes the output to song.ogg");
	printf("\n  cutoggvorbis -i file.ogg -O -             writes the output to STDOUT");
	printf("\n  cutoggvorbis -i file.ogg -d 2             use second sound card");
	printf("\n  cutoggvorbis -I file.ogg                  prints file information\n\n");
	if (text && text[0]!='\0') printf("%s\n\n",text);
}

void help(void)
{
	printf("\n\ncutoggvorbis  version %s  \n", VERSION);
	printf("\n        Seek with 1234567890,.\n");
	printf("        Mark beginning with 'a', go there with 'A'\n");
	printf("        Mark       end with 'b', go there with 'B'\n");
	printf("        Save selection with 's'.\n");
	printf("        'o' turns on overwriting for next time.\n");
	printf("        'i' shows file info.\n\n");
	printf("        Replay with 'r'.\n");
	printf("        Toggle sound on/off with '#'.\n");
	printf("        Write configuration file with 'S'.\n\n");
	printf("        Press 'q' to quit.\n\n");
	printf("        HINT: 'w' is a shortcut for 'bsa'.\n");
	return;
}

/*  This function returns the pagesize and not a position */
long pagesize(long seekpos)
{
	unsigned char a,b,c,d;
	long i,skip=0;
	long lesen;

	if (seekpos>filesize-27) return 0; /* header not completely inside file */

	fseek(oggfile, seekpos, SEEK_SET);
	if (debug==2) printf(" seekpos=%li ",seekpos);

	a=fgetc(oggfile);
	b=fgetc(oggfile);
	c=fgetc(oggfile);
	d=fgetc(oggfile);
	if (a=='O' && b=='g' && c=='g' && d=='S')
	{
		fseek(oggfile, seekpos+26, SEEK_SET);
		lesen=fgetc(oggfile);
		if (seekpos+lesen > filesize) return 0; /* header not completely inside file */
		for (i=0;i<lesen;i++) skip=skip+fgetc(oggfile);
		if (seekpos+skip+27+lesen > filesize) return 0; /* page not completely inside file */
		{
			if (debug==2) printf(" pagesize=%li ",skip+27+lesen);
			return skip+27+lesen;
		}
	}
	else return 0;
}

unsigned long pcmbytes(long seekpos)
{
	unsigned long pcmb=0;
	int i;
	int posinfo[16];

	fseek(oggfile, seekpos+6, SEEK_SET);
	for (i=0; i<8; i++) {posinfo[i]=fgetc(oggfile);}
	for (i=7; i>=0; i--) {pcmb=pcmb*256+(long)posinfo[i];}

	if (debug==3) printf("%lu ",pcmb);

	return pcmb;
}


/* find next page at seekpos */
long nextpage(long seekpos)
{
	long oldseekpos=seekpos;   /* remember seekpos */
	unsigned char a,b,c,d;
	long i,skip=0;
	long lesen;

	if (seekpos>filesize-27) return filesize;

	fseek(oggfile, seekpos, SEEK_SET);
	if (debug==1) printf("nextpage 1 seekpos=%lu \n",seekpos);

	/* if seekpos is a header and pagesize is valid, jump to next header via pagesize, else move on one byte */
	a=fgetc(oggfile);
	b=fgetc(oggfile);
	c=fgetc(oggfile);
	d=fgetc(oggfile);
	if (a=='O' && b=='g' && c=='g' && d=='S')
	{
		seekpos=oldseekpos+26;
		fseek(oggfile, seekpos, SEEK_SET);
		lesen=fgetc(oggfile);
// 		printf("\nNumber of segments in table=%i",lesen);
		if (debug==1) printf("nextpage 1a seekpos=%lu \n",seekpos);
		if (seekpos+lesen>filesize-27) return filesize; /* header not completely inside file */
		for (i=0;i<lesen;i++) {skip=skip+fgetc(oggfile);}
		if (debug==1) printf("nextpage 1b seekpos=%lu \n",seekpos);
		seekpos=seekpos+1+lesen+skip;
		if (seekpos>filesize-27) seekpos=filesize; /* header not completely inside file */
		if (debug==1) printf("nextpage 1c seekpos=%lu \n",seekpos);
		return seekpos;
	}
	else seekpos=oldseekpos+1;
	if (debug==1) printf("nextpage 2 seekpos=%lu \n",seekpos);

	/* find next possible header, start right at seekpos */
	fseek(oggfile, seekpos, SEEK_SET);

	while(1)           /* loop till break */
	{
		if (seekpos>filesize-27) {seekpos=filesize; break;} /* header not completely inside file */
		fseek(oggfile, seekpos, SEEK_SET);
		a=fgetc(oggfile);  /* get next byte */
		if (a=='O')
		{
			if (debug==1) printf("nextpage 3a seekpos=%lu \n",seekpos);
			b=fgetc(oggfile);
			c=fgetc(oggfile);
			d=fgetc(oggfile);
			if (b=='g' && c=='g' && d=='S') break;  /* next header found */
		}
		seekpos++;
		if (debug==1) printf("nextpage 3b seekpos=%lu \n",seekpos);
	}

	return seekpos;
}


/* find previous page at seekpos */
long prevpage(long seekpos)
{
	unsigned char a,b,c,d;

	if (seekpos<audiobegin) return audiobegin;

	/* rewind to previous header */
	seekpos=seekpos-1-pagesize(seekpos)*0.8;
	while(1)           /* loop till break */
	{
		if (seekpos<audiobegin) break;
		fseek(oggfile, seekpos, SEEK_SET);
		a=fgetc(oggfile);
		if (a=='O')
		{
			b=fgetc(oggfile);
			c=fgetc(oggfile);
			d=fgetc(oggfile);
 			if (b=='g' && c=='g' && d=='S') break;     /* previous page found */
		}
		seekpos--;
	}
	if (seekpos<audiobegin) return audiobegin;
	return seekpos;
}

/* This function returns the total amount of seconds from pcmbytes at this point */
long double seconds (long bytes)
{
	return (long double)pcmbytes(bytes)/(long double)samplerate;
}

long fforward (long seekpos,long skiptime)
{
	/* skiptime is in msecs */
	long double oldsecs=seconds(seekpos);
	long pos=seekpos;

	if (pos>filesize-27) return filesize;
	pos=nextpage(pos);
	if (pos>filesize-27) return filesize;


	while ((long double)seconds(pos) - oldsecs < (long double)skiptime/1000 && pos<filesize-27)
	{
		if (debug==1) printf("fforward 1 pos=%lu \n",pos);
		pos=nextpage(pos);
		if (debug==1) printf("fforward 2 pos=%lu \n",pos);
	}

	if (pos>filesize-27) return filesize;

	if (seconds(pos) - seconds(prevpage(pos)) > 2) printf("  LEAP IN TIME DETECTED (%.1Lf seconds)               \n                    ",seconds(pos) - seconds(prevpage(pos)));

	return pos;
}

long frewind (long pos,long skiptime)
{
	/* skiptime is in msecs */
	long double oldsecs;

	if (pos<audiobegin) return audiobegin;
	if (pos==filesize) pos=prevpage(pos);
	oldsecs=(long double)pcmbytes(pos)/(long double)samplerate;
	if (pos<audiobegin) return audiobegin;

	printf("  seeking at           ");

	while (oldsecs - (long double)pcmbytes(pos)/(long double)samplerate < (long double)skiptime/1000 && pos>audiobegin)
	{
		pos=prevpage(pos);
		printf("\b\b\b\b\b\b\b\b\b\b%4u:%05.2f",showmins(pos),showsecs(pos));
	}

	if (pos<audiobegin) return audiobegin;
	printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b");

	if (seconds(nextpage(pos)) - seconds(pos) > 2) printf("  LEAP IN TIME DETECTED (%.1Lf seconds)               \n                    ",seconds(nextpage(pos)) - seconds(pos));

	return pos;
}

int showmins (long bytes)
{
	return ( ((long double)pcmbytes(bytes)/(long double)samplerate) - offset ) / 60;
}

double showsecs (long bytes)
{
	double temp;

	temp=bytes;
	temp=( ( ((long double)pcmbytes(bytes)/(long double)samplerate) - offset ) - (showmins(bytes)*60) );
	if (temp>59.99) return 59.99; /* 59.997 would be displayed as 60:00 without this */
	return temp;
}

/* This function writes the configuration file */
void writeconf(void)
{
	FILE *conffile;
	char filename1[8191];

	snprintf(filename1,8190,"%s/%s",getenv("HOME"),".cutoggrc");
	/* check for conffile */
	if (NULL== (conffile = fopen(filename1,"wb")))
	{
		printf("  configuration file could NOT be written!     \n");
		return;
	}

	fprintf(conffile, "leng=");
	fprintf(conffile, "%u",silencelength);
	fprintf(conffile, "\n");
	fprintf(conffile, "volu=");
	fprintf(conffile, "%u",silvol);
	fprintf(conffile, "\n");
	fprintf(conffile, "card=");
	fprintf(conffile, "%u",card);
	fprintf(conffile, "\n");
	fprintf(conffile, "mute=");
	fprintf(conffile, "%u",mute);
	fprintf(conffile, "\n");
	fclose(conffile);

	printf("\n");
// 	printf("\n     length of silence is %u milliseconds",silencelength);
// 	printf("\n     maximum volume during silence is %u ",silvol);
	printf("\n     using soundcard #%u ",card);
	printf("\n     sound is ");
	if (mute) printf("OFF \n");
	else printf("ON \n");
	printf("\n     This configuration has been saved.     \n");
	return;
}


/* This function plays sound up to the current position */
void playtoend(long int pos)
{
	int i;

	if (mute) return;
	printf("  Now playing up to endpoint.\n");
	for (i=0; i<playpages; i++)	/* seek back playpages pages */
	{
		fseek(oggfile, pos, SEEK_SET);
		pos=prevpage(pos);
	}
	playsel(pos);
	return;
}


/* This function plays sound at the current position */
void playsel(long int playpos)
{
	long bytesin, i;
	long pos=playpos;
	char outname[8191];
	int length, diffmins, diffsecs;
	FILE *outfile;

	snprintf(outname,20,"%s","/tmp/cut.ogg");
	remove(outname); /* delete old tempfile */

	if (!mute)
	{
		if (playpos < audiobegin) playpos=audiobegin;
		if (playpos > filesize-27) playpos=filesize;

		if (NULL== (outfile = fopen(outname,"w+b"))){perror("\nfailed writing /tmp/cut.ogg");exitseq(9);}

		/* write bos page: */
		for (i=0 ; i<audiobegin ; i++) fputc(bospage[i],outfile);

		for (i=0; i<playpages; i++)
		{
			length=pagesize(pos);
			fseek(oggfile, pos, SEEK_SET);
			for (bytesin=0; bytesin < length /* && pos+bytesin < filesize */; bytesin++) fputc(getc(oggfile),outfile);
			pos=pos+length;
		}
		fclose(outfile);
	}

	pos=playpos;

	if (pos>filesize-27)
	{
		printf("  End of file reached!     \n");
		pos=filesize;
		printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b");
		return;
	}

	if (!mute)	/* unsilent mode */
	{
		if (playpos >= inpoint)
		{
			if(showsecs(playpos)<showsecs(inpoint)){diffmins=showmins(playpos)-showmins(inpoint)-1;diffsecs=showsecs(playpos)-showsecs(inpoint)+60;}
			else {diffmins=showmins(playpos)-showmins(inpoint);diffsecs=showsecs(playpos)-showsecs(inpoint);}
			printf("  playing at %4u:%05.2f / %u:%02u  (%u:%02u after startpoint) \n",showmins(pos),showsecs(pos),totalmins,totalsecs,diffmins,diffsecs);
		}
		else
		{
			if(showsecs(playpos)>showsecs(inpoint)){diffmins=showmins(inpoint)-showmins(playpos)-1;diffsecs=showsecs(inpoint)-showsecs(playpos)+60;}
			else {diffmins=showmins(inpoint)-showmins(playpos);diffsecs=showsecs(inpoint)-showsecs(playpos);}
			printf("  playing at %4u:%05.2f / %u:%02u  (%u:%02u before startpoint) \n",showmins(pos),showsecs(pos),totalmins,totalsecs,diffmins,diffsecs);
		}

		if (card==2)
		system("ogg123 /tmp/cut.ogg -a /dev/dsp1 > /tmp/ogg123.log 2>&1 &");
		else if (card==3)
		system("ogg123 /tmp/cut.ogg -a /dev/dsp2 > /tmp/ogg123.log 2>&1 &");
		else if (card==4)
		system("ogg123 /tmp/cut.ogg -a /dev/dsp3 > /tmp/ogg123.log 2>&1 &");
		else if (card==5)
		system("ogg123 /tmp/cut.ogg -a /dev/dsp4 > /tmp/ogg123.log 2>&1 &");
		else
		system("ogg123 /tmp/cut.ogg > /tmp/ogg123.log 2>&1 &");
	}
	else	/* silent mode */
	{
		if (playpos >= inpoint)
		{
			if(showsecs(playpos)<showsecs(inpoint)){diffmins=showmins(playpos)-showmins(inpoint)-1;diffsecs=showsecs(playpos)-showsecs(inpoint)+60;}
			else {diffmins=showmins(playpos)-showmins(inpoint);diffsecs=showsecs(playpos)-showsecs(inpoint);}
			printf("  position is %4u:%05.2f / %u:%02u  (%u:%02u after startpoint) \n",showmins(pos),showsecs(pos),totalmins,totalsecs,diffmins,diffsecs);
		}
		else
		{
			if(showsecs(playpos)>showsecs(inpoint)){diffmins=showmins(inpoint)-showmins(playpos)-1;diffsecs=showsecs(inpoint)-showsecs(playpos)+60;}
			else {diffmins=showmins(inpoint)-showmins(playpos);diffsecs=showsecs(inpoint)-showsecs(playpos);}
			printf("  position is %4u:%05.2f / %u:%02u  (%u:%02u before startpoint) \n",showmins(pos),showsecs(pos),totalmins,totalsecs,diffmins,diffsecs);
		}
	}
return;
}

/* This function reads the configuration file, very ugly programming */
void readconf(void)
{
	FILE *conffile;
	long confsize=0,pos=0;
	unsigned char conf[BUFFER];
	char filename1[8191];

	snprintf(filename1,8190,"%s/%s",getenv("HOME"),".cutoggrc");  /* get name of conffile */
	if (NULL == (conffile = fopen(filename1,"rb"))) return;       /* check for conffile */
	fseek(conffile, 0, SEEK_END);
	confsize=ftell(conffile);         /* size of conffile */

	/* read conffile into buffer: */
	fseek(conffile, 0, SEEK_SET);
	for (pos=0 ; pos<BUFFER && pos<confsize ; pos++) conf[pos]=fgetc(conffile);
	fclose(conffile);

	pos=0;        /* reset pos */
	while(1)                              /* loop till break */
	{
		if (pos==confsize || pos==BUFFER) break;
		if (conf[pos]=='l')
		{
			pos++;
			if (conf[pos]=='e' && conf[pos+1]=='n' && conf[pos+2]=='g' && conf[pos+3]=='=')
			{
				pos=pos+4;
				silencelength=0;
				while(isdigit(conf[pos]))
				{
					silencelength = silencelength*10 + conf[pos]-48;
					pos++;
				}
			}
		}
		if (conf[pos]=='v')
		{
			pos++;
			if (conf[pos]=='o' && conf[pos+1]=='l' && conf[pos+2]=='u' && conf[pos+3]=='=')
			{
				pos=pos+4;
				silvol=0;
				while(isdigit(conf[pos]))
				{
					silvol = silvol*10 + conf[pos]-48;
					pos++;
				}
			}
		}
		if (conf[pos]=='c')
		{
			pos++;
			if (conf[pos]=='a' && conf[pos+1]=='r' && conf[pos+2]=='d' && conf[pos+3]=='=')
			{
				pos=pos+4;
				if (isdigit(conf[pos]))
				{
					card = conf[pos]-48;
					pos++;
				}
			}
		}
		if (conf[pos]=='m')
		{
			pos++;
			if (conf[pos]=='u' && conf[pos+1]=='t' && conf[pos+2]=='e' && conf[pos+3]=='=')
			{
				pos=pos+4;
				if (isdigit(conf[pos]))
				{
					mute = conf[pos]-48;
					pos++;
				}
			}
		}
		pos++;
	}
	return;
}


/* This function plays the file at program start only */
void playfirst(long int playpos)
{
	printf("\n1 <- 5 seeks backward");
	printf("\n6 -> 0 seeks forward");
	printf("\n ,  .  rewinds/skips one page");
	printf("\n     a sets startpoint, A goes there");
	printf("\n     b sets endpoint, B goes there");
	printf("\n     s saves");
	printf("\n     using soundcard #%u ",card);
	printf("\n     sound is ");
	if (mute) printf("OFF \n");
	else printf("ON \n");
	printf("\n     Press 'h' for more help\n");

	printf("\n[1..0,r,a,b,s,q] >  ");
	playsel(playpos);
}


/* This function saves the selection non-interactively */
void savesel(char *prefix)
{
	char *outname=malloc(8191);
	int number=1;
	long bytesin=0;
	int i;
	FILE *fp;
	FILE *outfile;

	/* title not known */
	{
		/* outname = prefix number . suffix */
		snprintf(outname,8190, "%s%04u.ogg",prefix,number);

		if (inpoint < audiobegin) inpoint=audiobegin;
		if (outpoint > filesize-27) outpoint=filesize;
		if (outpoint < inpoint)
		{
			printf("  endpoint (%u:%05.2f) must be after startpoint (%u:%05.2f)!  \n",showmins(outpoint),showsecs(outpoint),showmins(inpoint),showsecs(inpoint));
			return;
		}
		if (outpoint == inpoint)
		{
			printf("  endpoint must not be the same as the startpoint!  \n");
			return;
		}

		/* if file exists, increment number */
		fp = fopen(outname,"r");
		while(fp != NULL) /* test if file exists */
		{
			number++;
			fclose(fp);
			if (number>9999) usage("9999 files written. Please choose another prefix via '-o prefix.'");
			snprintf(outname,8190, "%s%04u.ogg",prefix,number);
			fp = fopen(outname,"r");
		}
		number=number-overwrite; /* step one number back in case of overwrite mode */
		if (number==0) number=1;
		snprintf(outname,8190, "%s%04u.ogg",prefix,number);
	}

	/* forced output file name used? */
	if (forced_file==1) snprintf (outname, 8190, forcedname);

	/* open outfile */
	if (NULL== (outfile = fopen(outname,"wb"))){perror("\ncannot not write output file! read-only filesystem?");exitseq(10);}

	/* COPY DATA HERE */
	/* write bos page: */
	for (i=0 ; i<audiobegin ; i++) fputc(bospage[i],outfile);

	/* write audio data: */
	fseek(oggfile, inpoint, SEEK_SET);
	for (bytesin=0 ; bytesin < outpoint-inpoint ; bytesin++)
	{
		if(stdoutwrite==1) putchar(getc(oggfile));	/* to STDOUT? */
		else fputc(getc(oggfile),outfile);
	}

	/* close outfile */
	fclose(outfile);

	if(stdoutwrite==1) /* if write to STDOUT, return without printing messages */
 	{
		overwrite=0;
		free(outname);
		return;
	}

	/* noninteractive cutting: */
	if (nonint==1) printf("  saved %i:%05.2f - %i:%05.2f to '%s'.  \n",showmins(inpoint),showsecs(inpoint),showmins(outpoint),showsecs(outpoint),outname);
	else
	/* interactive cutting: */
	printf("  saved %u:%05.2f - %u:%05.2f to '%s'.  \n",showmins(inpoint),showsecs(inpoint),showmins(outpoint),showsecs(outpoint),outname);

	overwrite=0;
	free(outname);
	return;
}


/* ################### KEY PRESS ######################## */

static struct termios  current,  /* new terminal settings             */
                       initial;  /* initial state for restoring later */

/* Restore term-settings to those saved when term_init was called */
void  term_restore  (void)  {
  tcsetattr(0, TCSANOW, &initial);
}  /* term_restore */

/* Clean up terminal; called on exit */
void  term_exit  ()  {
	term_restore();
	printf("\n\nbugreports to mail@puchalla-online.de\n\n");
	exitseq(0);
}	/* term_exit */

/* Will be called when ctrl-z is pressed, this correctly handles the terminal */
void  term_ctrlz  ()  {
	signal(SIGTSTP, term_ctrlz);
	term_restore();
	kill(getpid(), SIGSTOP);
}  /* term_ctrlz */

/* Will be called when application is continued after having been stopped */
void  term_cont  ()  {
	signal(SIGCONT, term_cont);
	tcsetattr(0, TCSANOW, &current);
}  /* term_cont */

/* Needs to be called to initialize the terminal */
void  term_init  (void)  {
	/* if stdin isn't a terminal this fails. But    */
	/* then so does tcsetattr so it doesn't matter. */

	tcgetattr(0, &initial);
	current = initial;                      /* Save a copy to work with later  */
	signal(SIGINT,  term_exit);             /* We _must_ clean up when we exit */
	signal(SIGQUIT, term_exit);
	signal(SIGTSTP, term_ctrlz);            /* Ctrl-Z must also be handled     */
	signal(SIGCONT, term_cont);
	//atexit(term_exit);
}	/* term_init */

/* Set character-by-character input mode */
void  term_character  (void)  {
	/* One or more characters are sufficient to cause a read to return */
	current.c_cc[VMIN]   = 1;
	current.c_cc[VTIME]  = 0;  /* No timeout; read forever until ready */
	current.c_lflag     &= ~ICANON;           /* Line-by-line mode off */
	tcsetattr(0, TCSANOW, &current);
}  /* term_character */

/* Return to line-by-line input mode */
void  term_line  (void)  {
	current.c_cc[VEOF]  = 4;
	current.c_lflag    |= ICANON;
	tcsetattr(0, TCSANOW, &current);
}  /* term_line */

/* Key pressed ? */
int  kbhit  (void)  {
	struct timeval  tv;
	fd_set          read_fd;

	/* Do not wait at all, not even a microsecond */
	tv.tv_sec  = 0;
	tv.tv_usec = 0;

	/* Must be done first to initialize read_fd */
	FD_ZERO(&read_fd);

	/* Makes select() ask if input is ready;  0 is file descriptor for stdin */
	FD_SET(0, &read_fd);

	/* The first parameter is the number of the largest fd to check + 1 */
	if (select(1, &read_fd,
		NULL,         /* No writes        */
		NULL,         /* No exceptions    */
		&tv)  == -1)
	return(0);                 /* an error occured */

	/* read_fd now holds a bit map of files that are readable. */
	/* We test the entry for the standard input (file 0).      */
	return(FD_ISSET(0, &read_fd)  ?  1  :  0);
}

/* ------------------------------------------------------------------------- */


/* This function writes a timetable when -a and -b is used */
void writetable(char *tablename)
{
	FILE *outfile;

	if (NULL== (outfile = fopen(tablename,"wb"))){perror("\nfailed writing /tmp/timetable");exitseq(11);}
	fprintf(outfile, userin);
	fprintf(outfile, " ");
	fprintf(outfile, userout);
	fprintf(outfile, "\n");
	fclose(outfile);
	return;
}

/* This function seeks a time from EOF by scanning forward and using pcmbytes() */
long seekfromend(double minn, double secc)
{
	long pos=audiobegin;
	long double time=0;
	long double endtime=seconds(prevpage(filesize));

	fseek(oggfile, audiobegin, SEEK_SET);
	while(1)				/* loop till break */
	{
		time=endtime-seconds(pos);	/* net time */
		if (time<=minn*60+secc) break;	/* in/outpoint reached */
		if (pos>=filesize-27) break;	/* end of file reached */
		pos=nextpage(pos);
	}
	return pos;
}


/* This function copies data by its time using the "timestamp" from pcmbytes */
void cutbytime(char *prefix)
{
	long pos=audiobegin;
	long double time=0;

	/* is endpoint inside the file? */
	if (endoff<0 && endmin*60+endsec>totalseconds)
	{
		printf("  The audio length is only %u:%02u, but your endpoint is %li:%02.0f from the end!  \n",totalmins,totalsecs,endmin,endsec);
		startsec=startmin=endsec=endmin=0;
		return;
	}

	/* first look for inpoint */

	if (startoff<0) inpoint=seekfromend(startmin,startsec);
	else
	{
		fseek(oggfile, audiobegin, SEEK_SET);
		while(1)                              /* loop till break */
		{
			time=seconds(pos)-offset;		/* net time */
			if (time>=startmin*60+startsec) break;	/* inpoint reached */
			if (pos>=filesize-27) break;		/* EOF reached */
			pos=nextpage(pos);
		}
		inpoint=pos;
	}

	/* inpoint reached, now search for outpoint */

	if (endoff<0) outpoint=seekfromend(endmin,endsec);
	else
	{
		while(1)                              /* loop till break */
		{
			time=seconds(pos)-offset;		/* net time */
			if (time>=endmin*60+endsec) break;	/* outpoint reached */
			if (pos>=filesize-27) break;		/* EOF reached */
			pos=nextpage(pos);
		}
		outpoint=pos;
	}

	/* outpoint reached, now write file */

	savesel(prefix);
//	printf("%u %u %u %u",startsec,endsec,inpoint,outpoint);
	startsec=startmin=endsec=endmin=0;
	return;
}


/* This function copies selections when using a timetable */
void cutfromtable(char *tablename, char *prefix)
{
	int i, stelle=1;
	double number;
	unsigned char ttable[BUFFER]; /* for timetable */
	FILE *timefile;

	startsec=startmin=endsec=endmin=0;

	timefile = fopen(tablename,"rb");
	/* Load BUFFER Bytes into unsigned buffer[] */
	for (i=0 ; i<BUFFER ; i++) ttable[i]=fgetc(timefile);
	fclose(timefile);

	i=0;
	do
	{
		/* stelle: startmin=1 startsek=2 startten=3 starthun=4 */
		/*             endmin=5   endsek=6   endten=7   endhun=8 */
		number=ttable[i]-48;
		if (isdigit(ttable[i]) && stelle==1) startmin=startmin*10+ttable[i]-48;
		if (isdigit(ttable[i]) && stelle==2) startsec=startsec*10+ttable[i]-48;
		if (isdigit(ttable[i]) && stelle==3)
		{
			startsec=startsec+number/10;
			stelle++;
			i++;
			number=ttable[i]-48;
		}
		if (isdigit(ttable[i]) && stelle==4) startsec=startsec+number/100;
		if (isdigit(ttable[i]) && stelle==5) endmin=endmin*10+ttable[i]-48;
		if (isdigit(ttable[i]) && stelle==6) endsec=endsec*10+ttable[i]-48;
		if (isdigit(ttable[i]) && stelle==7)
		{
			endsec=endsec+number/10;
			stelle++;
			i++;
			number=ttable[i]-48;
		}
		if (isdigit(ttable[i]) && stelle==8) endsec=endsec+number/100;

		/* '-' is offset from end */
		if (ttable[i]==45 && stelle < 5) startoff=-1;
		if (ttable[i]==45 && stelle > 4) endoff=-1;
		/* ':' and '.' increase stelle */
		if (ttable[i]==58) stelle++;
		if (ttable[i]==46) stelle++;

		/* space switches to read end or to cutbytime */
		if (isspace(ttable[i]) && stelle<5 && stelle>1) stelle=5;
		if (isspace(ttable[i]) && stelle>5)
		{
// 		printf("i=%u startmin=%lu startsec=%02.2f endmin=%lu endsec=%02.2f stelle=%u startoff=%i endoff=%i \n",i,startmin,startsec,endmin,endsec,stelle,startoff,endoff);
			cutbytime(prefix);
			stelle=1;
			startsec=startmin=endsec=endmin=0;
			startoff=0;endoff=0;
		}

		if (ttable[i]==255) i=BUFFER;
//		printf("i=%u %lu:%02.2f %lu:%02.2f stelle=%u\n",i,startmin,startsec,endmin,endsec,stelle);
		i++;
	}
	while (i<BUFFER);
}


void showfileinfo(int rawmode)
{
	unsigned char a,b,c,d,e,f,ll;
	long skip, seekpos, pos, secondpage, i, j;
	unsigned char vendor[256];

	if (rawmode==1)	exitseq(0);

	printf("\n\n* Properties of \"%s\":\n* \n",filename);
// 	if (audiobegin != 0) printf("* Audio data starts at %u Bytes\n",audiobegin);
	printf("*           %3u kBit/sec (nominal)\n",avbr/1000);
	printf("*           %3.0Lf kBit/sec (computed)\n",(long double)filesize*8/1000/ ((long double)pcmbytes(prevpage(filesize))-(long double)pcmbytes(audiobegin)) *(long double)samplerate);
	if (bitratemin!=0) printf("*           %3u kBit/sec (minimum)\n",bitratemin);
	if (bitratemax!=0) printf("*           %3u kBit/sec (maximum)\n",bitratemax);
	printf("*         %5u Hz ",samplerate);
	if (channels==2) printf("Stereo\n");
	if (channels==1) printf("Mono\n");
	printf("*\n* Filelength is %u minute(s) %u second(s)\n",totalmins,totalsecs);
	printf("* Timestamp of beginning is %u minute(s) %2.0f second(s)\n",showmins(2*offset),showsecs(2*offset));

	/* VORBIS COMMENTS: */
	printf("*\n* Vorbis Comments:\n");
	seekpos=secondpage=nextpage(0);
// 	printf(" 2ndpage=%li ",secondpage);
	fseek(oggfile, secondpage, SEEK_SET);

	while(1)           /* loop till break */
	{
		if (seekpos>nextpage(secondpage)) break;
		fseek(oggfile, seekpos, SEEK_SET);
// 		printf(" seekpos=%li ",seekpos);
		a=fgetc(oggfile);  /* get next byte */
		if (a=='v')
		{
			b=fgetc(oggfile);
			c=fgetc(oggfile);
			d=fgetc(oggfile);
			e=fgetc(oggfile);
			f=fgetc(oggfile);
			if (b=='o' && c=='r' && d=='b' && e=='i' && f=='s') break;  /* vorbis found */
		}
		seekpos++;
	}
	fseek(oggfile, seekpos+6, SEEK_SET);	/* seek to vendorlength */
	a=fgetc(oggfile);
	b=fgetc(oggfile);
	c=fgetc(oggfile);
	d=fgetc(oggfile);
// 	printf(" vendorlength %u %u %u %u ",a,b,c,d);
	skip=a+b*256+c*256*256+d*256*256*256;
	printf("*   Vendor String=");
	for (i=0; i<a; i++){vendor[i]=fgetc(oggfile);printf("%c",vendor[i]);}	/* cut to 8bit */
	printf("\n");

	fseek(oggfile, seekpos+10+skip, SEEK_SET);	/* seek to listlength */
	a=fgetc(oggfile); ll=a;
	b=fgetc(oggfile);
	c=fgetc(oggfile);
	d=fgetc(oggfile);
// 	printf(" listlength %u %u %u %u ",a,b,c,d);
// 	skip=a+b*256+c*256*256+d*256*256*256;	/* now at first entry */
	for (i=0; i<ll; i++)	/* cut to 8bit */
	{
		a=fgetc(oggfile);
		b=fgetc(oggfile);
		c=fgetc(oggfile);
		d=fgetc(oggfile);
		skip=a+b*256+c*256*256+d*256*256*256;
// 	 	printf(" skip %u %u %u %u ",a,b,c,d);
		pos=ftell(oggfile);
		printf("*   ");
		for (j=0; j<skip; j++)	/* cut to 8bit */
		{
			printf("%c",fgetc(oggfile));
		}
		printf("\n");
		fseek(oggfile, pos+skip, SEEK_SET);	/* seek to listlength */
	}
	printf("\n");
	return;
}


/* ------------------------------------------------------------------------- */
/*      */
/* MAIN */
/*      */
/* ------------------------------------------------------------------------- */

int main(int argc, char *argv[])
{
	int a,c,char1,showinfo=0,rawmode=0;
	long int pos, startpos;
	char *tablename=malloc(8191);
	char *prefix=malloc(8191);
	FILE *testfile;

/* ------------------------------------------------------------------------- */
/*                   */
/* parse commandline */
/*                   */
/* ------------------------------------------------------------------------- */

	readconf();       /* read configuration file first! */

	if (argc<2) {usage("Error: missing filename");exitseq(1);}
	else
	{
		while ((char1 = getopt(argc, argv, "hqf:a:b:i:o:O:I:d:D:")) != -1)
		{
			switch (char1)
			{
				case 'a':
					if (optarg!=0) userin=optarg;
//					else usage("Error: missing time argument");
					break;
				case 'b':
					if (optarg!=0) userout=optarg;
//					else usage("Error: missing time argument");
					break;
				case 'd':
					if (optarg!=0) card=atoi(optarg);
					break;
				case 'D':
					if (optarg!=0) debug=atoi(optarg);
					break;
				case 'f':
					if (NULL == (testfile = fopen(optarg,"rb") ) )
					{
						perror(optarg);
						exitseq(3);
					}
					else tablename=optarg; hastable=1;
					break;
				case 'h':
					usage("");
					exit(0);
				case 'q':
					mute=1;
					break;
				case 'i':
//					if (optarg==0) usage("Error: option requires an argument");
					if (NULL == (testfile = fopen(optarg,"rb") ) )
					{
						perror(optarg);
						exitseq(2);
					}
					else filename=optarg;
					break;
				case 'I':
//					if (optarg==0) usage("Error: option requires an argument");
					if (NULL == (testfile = fopen(optarg,"rb") ) )
					{
						perror(optarg);
						exitseq(2);
					}
					else { filename=optarg; showinfo=1; }
					break;
				case 'o':
					if (optarg!=0) snprintf(prefix,8190,optarg);
//					else usage("Error: missing outputprefix");
					break;
/* ThOr: let user define exactly one output file */
				case 'O':
					if (optarg!=0)
					{
						forcedname=optarg;
						forced_file=1;
						if(optarg[0]=='-') stdoutwrite=1;
					}
					break;
				default:
					exitseq(1);
					break;
			}
		}
	}

	if (strlen(prefix)==0) prefix="result";

	if (userin==0 && userout!=0) usage("Error: missing inpoint");
	if (userin!=0 && userout==0) usage("Error: missing outpoint");
	if (filename==0) {usage("Error: missing filename, use 'cutoggvorbis -i file.ogg [options]'");exitseq(1);}

	if (stdoutwrite==0) setvbuf(stdout,NULL,_IONBF,0); /* switch off buffered output to stdout when writing to file */

	oggfile = fopen(filename, "rb");

	/* get filesize and set outpoint to it */
	fseek(oggfile, 0, SEEK_END);
	outpoint=filesize=ftell(oggfile);

	audiobegin=nextpage(nextpage(0));
	while(1)
	{
		fseek(oggfile, audiobegin+5, SEEK_SET);
		a=fgetc(oggfile);
		a=a<<7;
// 		printf("\nheader type = %i",a);
		if (a==128) audiobegin=nextpage(audiobegin); // audio data starts at fresh page
		else break;
	}

// 	printf("\naudiobegin=%i",audiobegin);
	startpos=inpoint=audiobegin;

	/* write bos page to buffer: */
	fseek(oggfile, 0, SEEK_SET);
	for (pos=0 ; pos<audiobegin && pos<=filesize ; pos++) bospage[pos]=fgetc(oggfile);

	/* read vorbis header: */
	channels=bospage[39];
// 	printf("\nchannels=%u",channels);
	samplerate=bospage[40]+bospage[41]*256+bospage[42]*256*256+bospage[43]*256*256*256;
// 	printf("\nsample rate=%u",samplerate);
	bitratemax=bospage[44]+bospage[45]*256+bospage[46]*256*256+bospage[47]*256*256*256;
// 	printf("\nbitrate maximum=%u",bitratemax);
	avbr=bospage[48]+bospage[49]*256+bospage[50]*256*256+bospage[51]*256*256*256;
// 	printf("\naverage bitrate=%i",avbr);
	bitratemin=bospage[52]+bospage[53]*256+bospage[54]*256*256+bospage[55]*256*256*256;
// 	printf("\nminimum bitrate=%u",bitratemin);

	fseek(oggfile, audiobegin, SEEK_SET);

	/* get total time data */
	offset=seconds(audiobegin);
	totalseconds=seconds(prevpage(filesize))-offset;
	totalmins=showmins(prevpage(filesize));
	totalsecs=(int)showsecs(prevpage(filesize));

/********** End of global part, now do specific parts **********/

	/* Show file info only? */
	if (showinfo==1)
	{
		showfileinfo(rawmode);
		printf("\n");
		exitseq(0);
	}

	/* -a and -b used? */
	if (userin!=0 && userout!=0)
	{
		nonint=1;
		tablename="/tmp/timetable";
		writetable(tablename);
		cutfromtable(tablename,prefix);
		remove("/tmp/timetable");
		exitseq(0);
	}

	/* timetable used? */
	if (hastable==1)
	{
		nonint=1;
		cutfromtable(tablename,prefix);
		exitseq(0);
	}

	/*** now handle interactive mode ***/

	if (stdoutwrite==1)
	{
		printf("\n  Writing to STDOUT in interactive mode is a bad idea.\n  Switching to resultfile writing.\n");
		stdoutwrite=0;
		forced_file=0;
	}

// 	copytags=0;  /* -c is only for noninteractive mode */

	term_init();
	term_character();

	playfirst(audiobegin);

	while (1)
	{
// 		printf("pos=%li [1..0,r,a,b,s,q] > ",startpos);
		printf("[1..0,r,a,b,s,q] > ");
		c = getchar();
		if (c == 'q') {printf("\n\n");exitseq(0);}
		if (c == '1') {startpos=frewind(startpos,1000*600); playsel(startpos);}
		if (c == '2') {startpos=frewind(startpos,1000*60); playsel(startpos);}
		if (c == '3') {startpos=frewind(startpos,1000*10); playsel(startpos);}
		if (c == '4') {startpos=frewind(startpos,1000); playsel(startpos);}
		if (c == '5') {startpos=frewind(startpos,100); playsel(startpos);}
		/* previous page, when at EOF */
		if (c == ',') {startpos=prevpage(startpos); playsel(startpos);}
		if (c == '.') {startpos=nextpage(startpos); playsel(startpos);}
		if (c == '6') {startpos=fforward(startpos,100);playsel(startpos);}
		if (c == '7') {startpos=fforward(startpos,1000);playsel(startpos);}
		if (c == '8') {startpos=fforward(startpos,1000*10);playsel(startpos);}
		if (c == '9') {startpos=fforward(startpos,1000*60);playsel(startpos);}
		if (c == '0') {startpos=fforward(startpos,1000*600);playsel(startpos);}
		if (c == 'r') playsel(startpos);
		if (c == 's') savesel(prefix);
		if (c == 'S') writeconf();
		if (c == 'i') showfileinfo(rawmode);
		if (c == 'h') help();
// 		if (c == 'z') showdebug(startpos,prefix);
		if (c == 'a')
		{
			inpoint=startpos;
			printf("  startpoint set to %u:%05.2f  \n",showmins(inpoint),showsecs(inpoint));
		}
		if (c == 'b')
		{
			outpoint=startpos;
			playtoend(startpos);
			printf("  endpoint set to %u:%05.2f  \n",showmins(outpoint),showsecs(outpoint));
		}
		if (c == '#')
		{
			mute=1-mute;
			if (mute) printf("  sound off\n");
			else printf("  sound on\n");
		}
		if (c == 'o')
		{
			overwrite=1-overwrite;
			if (1-overwrite) printf("  overwrite off\n");
			else printf("  overwrite on (for next time only)\n");
		}
		if (c == 'w')
		{
			outpoint=startpos;printf("  endpoint set to %u:%05.2f  \n",showmins(outpoint),showsecs(outpoint));
			savesel(prefix);
			inpoint=startpos;printf("  startpoint set to %u:%05.2f  \n",showmins(inpoint),showsecs(inpoint));
		}
		if (c == 'A')
		{
			startpos=inpoint;
			playsel(startpos);
		}
		if (c == 'B')
		{
			startpos=outpoint;
			playsel(startpos);
		}
	printf("\b\b\b   \b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b");	//delete echo from undefined key
	}
	exitseq(0);
}


/*
OGG page format:

#byte: Inhalt
0-3: OggS
4: Version (=0)
5: Header Typ, Bit 0x01: packet fortgesetzt von page zuvor
               Bit 0x02: Beginning of stream flag
               Bit 0x04: End of stream flag
6-13: Granulare Position, optional!
14-17: Bitstream Seriennummer, identifiziert jeden logischen Bitstream
18-21: page Sequenznummer, erhht sich fr jeden Bitstream separat
22-25: CRC check der page, incl. Header mit leerem CRC
26: Lnge der segment table in Bytes
27-xx: segment table, jedes Byte nennt die Lnge der folgenden Segmente (in Bytes), danach folgt die nchste page

Lnge einer page in Bytes ist also: 27 + [Inhalt von #26] + [Summe der Inhalte von #27-xx]
                                  = 27 + [Summe der Inhalte von #26-xx]
*/

/*
Ogg encapsulation of a Vorbis packet stream is straightforward.
 The first Vorbis packet (the identification header), which uniquely identifies a stream as Vorbis audio, is placed alone in the first page of the logical Ogg stream. This results in a first Ogg page of exactly 58 bytes at the very beginning of the logical stream.
 This first page is marked 'beginning of stream' in the page flags.
 The second and third vorbis packets (comment and setup headers) may span one or more pages beginning on the second page of the logical stream. However many pages they span, the third header packet finishes the page on which it ends. The next (first audio) packet must begin on a fresh page.
 The granule position of these first pages containing only headers is zero.

* The first audio packet of the logical stream begins a fresh Ogg page.

 Packets are placed into ogg pages in order until the end of stream.
 The last page is marked 'end of stream' in the page flags.
 Vorbis packets may span page boundaries.
 The granule position of pages containing Vorbis audio is in units of PCM audio samples (per channel; a stereo stream's granule position does not increment at twice the speed of a mono stream).
 The granule position of a page represents the end PCM sample position of the last packet completed on that page. A page that is entirely spanned by a single packet (that completes on a subsequent page) has no granule position, and the granule position is set to '-1'.
 The granule (PCM) position of the first page need not indicate that the stream started at position zero. Although the granule position belongs to the last completed packet on the page and a valid granule position must be positive, by inference it may indicate that the PCM position of the beginning of audio is positive or negative.
 A positive starting value simply indicates that this stream begins at some positive time offset, potentially within a larger program. This is a common case when connecting to the middle of broadcast stream.
 A negative value indicates that output samples preceeding time zero should be discarded during decoding; this technique is used to allow sample-granularity editing of the stream start time of already-encoded Vorbis streams. The number of samples to be discarded must not exceed the overlap-add span of the first two audio packets.
 In both of these cases in which the initial audio PCM starting offset is nonzero, the second finished audio packet must flush the page on which it appears and the third packet begin a fresh page. This allows the decoder to always be able to perform PCM position adjustments before needing to return any PCM data from synthesis, resulting in correct positioning information without any aditional seeking logic.
Note
 Failure to do so should, at worst, cause a decoder implementation to return incorrect positioning information for seeking operations at the very beginning of the stream.
 A granule position on the final page in a stream that indicates less audio data than the final packet would normally return is used to end the stream on other than even frame boundaries. The difference between the actual available data returned and the declared amount indicates how many trailing samples to discard from the decoding process.
*/















// 		fseek(oggfile, oldseekpos+4, SEEK_SET);
//		printf("\nogg version=%li",fgetc(oggfile));

// 		fseek(oggfile, oldseekpos+5, SEEK_SET);
// 		printf("\nheader type = %li",a=fgetc(oggfile));
// 		if (a==1)printf(" = continued data from previous packet");
// 		if (a==2)printf(" = BOS packet");
// 		if (a==4)printf(" = EOS packet");
// 		if (a==0)printf(" = fresh packet");

// 		fseek(oggfile, oldseekpos+6, SEEK_SET);
// 		printf("\npos info=%i %i %i %i %i %i %i %i",posinfo[0],posinfo[1],posinfo[2],posinfo[3],posinfo[4],posinfo[5],posinfo[6],posinfo[7]);

// 		fseek(oggfile, oldseekpos+14, SEEK_SET);
//		printf("\nbitstream ID=%i %i %i %i",fgetc(oggfile),fgetc(oggfile),fgetc(oggfile),fgetc(oggfile));

// 		fseek(oggfile, oldseekpos+18, SEEK_SET);
// 		printf("\npage sequence number=%i %i %i %i",fgetc(oggfile),fgetc(oggfile),fgetc(oggfile),fgetc(oggfile));

// 		fseek(oggfile, oldseekpos+22, SEEK_SET);
// 		printf("\nCRC=%i %i %i %i",fgetc(oggfile),fgetc(oggfile),fgetc(oggfile),fgetc(oggfile));

// 		fseek(oggfile, oldseekpos+26, SEEK_SET);
// 		lesen=fgetc(oggfile);
// 		for (i=0;i<lesen;i++) {skip=skip+fgetc(oggfile);}
