/****************************************************************************
 *
 *  Filename: cpia2_usb.c
 *
 *  Copyright 2001, STMicrolectronics, Inc.
 *      Contact:  steve.miller@st.com
 *
 *  Description:
 *     This is a USB driver for CPia2 based video cameras.
 *     The infrastructure of this driver is based on the cpia usb driver by
 *     Jochen Scharrlach and Johannes Erdfeldt.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/
/* $Id: cpia2_usb.c,v 1.14 2002/07/19 21:57:44 sbertin Exp $ */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/usb.h>

//#define _CPIA2_DEBUG_

#include "cpia2.h"
#include "cpia2_jpeg.h"

static void process_frame(struct camera_data *cam);
static int check_for_app2_segment(unsigned char *buffer);
static int parse_app2_segment(struct camera_data *cam, int app2_offset);
static int insert_tables(struct camera_data *cam,
                         unsigned long app2_len,
                         unsigned int dht_pos,
                         unsigned short squeeze,
                         unsigned short dc_squeeze,
                         unsigned short dbl_squeeze);
static void cpia2_usb_complete(struct urb *urb);
static void cpia2_usb_free_resources(struct camera_data *cam);
static void free_sbufs(struct camera_data *cam);
static void cpia2_usb_disconnect(struct usb_device *udev, void *ptr);
static void *cpia2_usb_probe(struct usb_device *udev,
			     unsigned int ifnum,
			     const struct usb_device_id *id);
static void add_APPn(struct camera_data *cam);
static void add_COM(struct camera_data *cam);

static struct usb_device_id cpia2_id_table[] = {
	{USB_DEVICE(0x0553, 0x0100)},
	{USB_DEVICE(0x0553, 0x0140)},
	{}			/* Terminating entry */
};

static struct usb_driver cpia2_driver = {
	name:"cpia2",
	probe:cpia2_usb_probe,
	disconnect:cpia2_usb_disconnect,
	id_table:cpia2_id_table
};

/******************************************************************************
 *
 *  process_frame
 *
 *****************************************************************************/
static void process_frame(struct camera_data *cam)
{
	static int frame_count = 0;
	int app2_offset;

	unsigned char *inbuff = cam->workbuff->data;

	if(cam->workbuff->length > cam->workbuff->max_length)
		cam->workbuff->max_length = cam->workbuff->length;

	if ((inbuff[0] == 0xFF) && (inbuff[1] == 0xD8)) {
		frame_count++;
	} else {
		cam->workbuff->status = FRAME_ERROR;
		DBG("Start of frame not found\n");
		return;
	}

	/***
	 * Now the output buffer should have a JPEG image in it.
	 ***/
	if (cam->workbuff->length > 3 &&
	    (app2_offset=check_for_app2_segment(cam->workbuff->data)) > 0) {
		if(parse_app2_segment(cam, app2_offset)) {
			if(cam->mmapped &&
			   cam->workbuff->length < cam->workbuff->max_length) {
				/* No junk in the buffers */
				memset(cam->curbuff->data+cam->curbuff->length,
				       0, cam->curbuff->max_length-
				          cam->curbuff->length);
			}
			cam->workbuff->max_length = cam->workbuff->length;
			cam->workbuff->status = FRAME_READY;
			
			if(!cam->mmapped && cam->num_frames > 2) {
				/* During normal reading, the most recent
				 * frame will be read.  If the current frame
				 * hasn't started reading yet, it will never
				 * be read, so mark it empty.  If the buffer is
				 * mmapped, or we have few buffers, we need to
				 * wait for the user to free the buffer.  
				 *
				 * NOTE: This is not entirely foolproof with 3
				 * buffers, but it would take an EXTREMELY
				 * overloaded system to cause problems (possible
				 * image data corruption).  Basically, it would
				 * need to take more time to execute cpia2_read
				 * than it would for the camera to send
				 * cam->num_frames-2 frames before problems
				 * could occur.
				 */
				cam->curbuff->status = FRAME_EMPTY;
			}
			cam->curbuff = cam->workbuff;
			cam->workbuff = cam->workbuff->next;
			return;
		}
		DBG("parse_app2_segment failed\n");
	} else {
		DBG("App2 segment not found\n");
	}

	cam->workbuff->status = FRAME_ERROR;
	return;
}

/******************************************************************************
 *
 *  check_for_app2_segment
 *
 *****************************************************************************/
static int check_for_app2_segment(unsigned char *buffer)
{
	unsigned char *buf;
	
	if (!buffer) {
		ERR(__FUNCTION__": got null buffer pointer\n");
		return -1;
	}

	buf = buffer;
	while(buf[0] == 0xFF) {
		switch(buf[1]) {
			case JPEG_APP_2:
			    return buf-buffer;
			case JPEG_SOI:
			case JPEG_EOI:
			case JPEG_TEM:
			case JPEG_RST_0:
			case JPEG_RST_1:
			case JPEG_RST_2:
			case JPEG_RST_3:
			case JPEG_RST_4:
			case JPEG_RST_5:
			case JPEG_RST_6:
			case JPEG_RST_7:
			    buf += 2;
			    break;
			default:
			    buf += buf[2]*256 + buf[3] + 2;
			    break;
		}
	}

	return -1;
}

/******************************************************************************
 *
 *  parse_app2_segment
 *
 *****************************************************************************/
static int parse_app2_segment(struct camera_data *cam, int app2_offset)
{
	unsigned char *buffer = cam->workbuff->data;
	jpeg_vvl_app2 *app2 = (jpeg_vvl_app2 *) & buffer[app2_offset];
	unsigned long app2_len;
	unsigned char *data;
	unsigned int dht_pos = 0;
	unsigned long seg_len;
	unsigned short squeeze = 0;
	unsigned short dc_squeeze = 0;
	int result = true;
	int dbl_squeeze = false;

	app2_len = (app2->app_len[0] * 256) + app2->app_len[1];
	
	if (2+app2_len <= cam->workbuff->length &&
	    app2->app_vvl[0] == 'V' &&
	    app2->app_vvl[1] == 'V' && app2->app_vvl[2] == 'L') {
		squeeze = app2->app_ac_squeeze;
		dc_squeeze = app2->app_dc_squeeze;

		if (app2->app_version & 0x80) {
			dbl_squeeze = true;
		} else {
			dbl_squeeze = false;
		}


		for (data = buffer + 2 + app2_len;
		     data < buffer + cam->workbuff->length;
		     data++) {
			if (0xFF == *data) {
				switch (*(data + 1)) {
				case JPEG_DHT:
					dht_pos = data - buffer;
					/* Fall through */
				case JPEG_SOF_0:
					seg_len = (*(data+2)*256) + *(data+3);
					if ((seg_len + data) <
					    (buffer + cam->workbuff->length)) {
						data += seg_len;
					} else {
						data = buffer +
						       cam->workbuff->length;
					}
					break;
				case JPEG_SOS:
				case JPEG_EOI:
					data = buffer + cam->workbuff->length;
					break;
				default:
					break;
				}
			}
		}
		result = insert_tables(cam,
		                       app2_len,
		                       dht_pos,
		                       squeeze,
		                       dc_squeeze,
		                       dbl_squeeze);
	} else
		result = -1;
	
	return result;
}

/******************************************************************************
 *
 * insert_tables 
 *
 *****************************************************************************/
static int insert_tables(struct camera_data *cam,
                         unsigned long app2_len,
                         unsigned int dht_pos,
                         unsigned short squeeze,
                         unsigned short dc_squeeze,
                         unsigned short dbl_squeeze)
{
	unsigned char *buffer = cam->workbuff->data;
	int result = true;
	int data_size = 0;
	int table_size = 0;
	int pos_after_app2 = 0;
	unsigned char *buffer_ptr = 0;

	pos_after_app2 = app2_len + sizeof(JPEG_SOI);	/* APP2 length + SOI length */
	if (dht_pos == 0) {
		table_size = sizeof(jpeg_vvl_dhtseg);
		data_size = cam->workbuff->length - pos_after_app2;

		DBG("tableSize %d, dataSize %d, imageSize %d, posAfterApp2 %d\n",
		    table_size, data_size, cam->workbuff->length, pos_after_app2);
		
		if(data_size > 0 &&
		   pos_after_app2 + table_size + data_size <= cam->frame_size) {
			/* Move image forward in buffer to make room for tables */
			memmove(buffer + pos_after_app2 + table_size,
				buffer + pos_after_app2, data_size);

			/* Point to end of app2 segment (where tables go). */
			buffer_ptr = buffer + pos_after_app2;

			memcpy(buffer_ptr, jpeg_vvl_dhtseg, table_size);

			cam->workbuff->length += table_size;
		} else 
			result = -1;
	}
	return result;
}


/******************************************************************************
 *
 *  add_APPn
 *
 *  Adds a user specified APPn record
 *****************************************************************************/
static void add_APPn(struct camera_data *cam)
{
	if(cam->APP_len > 0) {
		cam->workbuff->data[cam->workbuff->length++] = 0xFF;
		cam->workbuff->data[cam->workbuff->length++] = 0xE0+cam->APPn;
		cam->workbuff->data[cam->workbuff->length++] = 0;
		cam->workbuff->data[cam->workbuff->length++] = cam->APP_len+2;
		memcpy(cam->workbuff->data+cam->workbuff->length,
		       cam->APP_data, cam->APP_len);
		cam->workbuff->length += cam->APP_len;
	}
}

/******************************************************************************
 *
 *  add_COM
 *
 *  Adds a user specified COM record
 *****************************************************************************/
static void add_COM(struct camera_data *cam)
{
	if(cam->COM_len > 0) {
		cam->workbuff->data[cam->workbuff->length++] = 0xFF;
		cam->workbuff->data[cam->workbuff->length++] = 0xFE;
		cam->workbuff->data[cam->workbuff->length++] = 0;
		cam->workbuff->data[cam->workbuff->length++] = cam->COM_len+2;
		memcpy(cam->workbuff->data+cam->workbuff->length,
		       cam->COM_data, cam->COM_len);
		cam->workbuff->length += cam->COM_len;
	}
}

/******************************************************************************
 *
 *  cpia2_usb_complete
 *
 *  callback when incoming packet is received
 *****************************************************************************/
static void cpia2_usb_complete(struct urb *urb)
{
	int i;
	unsigned char *cdata;
	static int frame_ready = false;
	struct camera_data *cam;

	DBG("Enter\n");
	if (!urb || !urb->context)
		return;

	cam = (struct camera_data *) urb->context;


	if (!cam->dev ||
	    ((cam->streaming != 1) && (cam->streaming != 3)) ||
	    !cam->present || cam->open_count == 0) {
		DBG("Error: streaming = %d\n", cam->streaming);
		return;
	}

	if (!cam->buffers) {
		DBG("Error: buffers not allocated\n");
		return;
	}

	/***
	 * Packet collater
	 ***/
	DBG("Collating %d packets\n", urb->number_of_packets);
	for (i = 0; i < urb->number_of_packets; i++) {
		u16 checksum, iso_checksum;
		int j;
		int n = urb->iso_frame_desc[i].actual_length;
		int st = urb->iso_frame_desc[i].status;

		if(cam->workbuff->status == FRAME_READY) {
			/* Try to find an available buffer */
			DBG("workbuff full, searching\n");
			for(j=0; j<cam->num_frames; ++j) {
				if(cam->buffers[j].status != FRAME_READY) {
					cam->workbuff = &cam->buffers[j];
					break;
				}
			}
		}

		if (cam->workbuff->status == FRAME_EMPTY ||
		    cam->workbuff->status == FRAME_ERROR) {
			cam->workbuff->status = FRAME_READING;
			cam->workbuff->length = 0;
		}

		DBG("   Packet %d length = %d, status = %d\n", i, n, st);
		cdata = urb->transfer_buffer + urb->iso_frame_desc[i].offset;

		if (st) {
			LOG("cpia2 data error: [%d] len=%d, status = %d\n",
			    i, n, st);
			if(!ALLOW_CORRUPT)
				cam->workbuff->status = FRAME_ERROR;
			continue;
		}
		
		if(n<=2)
			continue;

		checksum = 0;
		for(j=0; j<n-2; ++j)
			checksum += cdata[j];
		iso_checksum = cdata[j] + cdata[j+1]*256;
		if(checksum != iso_checksum) {
			LOG("checksum mismatch: [%d] len=%d, calculated = %x, checksum = %x\n",
			    i, n, (int)checksum, (int)iso_checksum);
			if(!ALLOW_CORRUPT) {
				cam->workbuff->status = FRAME_ERROR;
				continue;
			}
		}
		n -= 2;

		if(cam->workbuff->status != FRAME_READING) {
			if((0xFF == cdata[0] && 0xD8 == cdata[1]) ||
			   (0xD8 == cdata[0] && 0xFF == cdata[1] &&
			    0 != cdata[2])) {
				/* frame is skipped, but increment total
				 * frame count anyway */
				cam->frame_count++;
			}
			DBG("workbuff not reading, status=%d\n",
			    cam->workbuff->status);
			continue;
		}

		if (cam->frame_size < cam->workbuff->length + n) {
			ERR("buffer overflow! length: %d, n: %d\n",
			    cam->workbuff->length, n);
			cam->workbuff->status = FRAME_ERROR;
			if(cam->workbuff->length > cam->workbuff->max_length)
				cam->workbuff->max_length =
					cam->workbuff->length;
			continue;
		}

		if (cam->workbuff->length == 0) {
			if ((0xD8 == cdata[0]) && (0xFF == cdata[1])) {
				do_gettimeofday(&cam->workbuff->timestamp);
				cam->workbuff->seq = cam->frame_count++;
				cam->workbuff->data[0] = 0xFF;
				cam->workbuff->data[1] = 0xD8;
				cam->workbuff->length = 2;
				add_APPn(cam);
				add_COM(cam);
				memcpy(cam->workbuff->data+cam->workbuff->length,
				       cdata+1, n-1);
				DBG("Start of frame pattern found\n");
				cam->workbuff->length += n-1;
			} else if((0xFF == cdata[0]) && (0xD8 == cdata[1])
			          && (0xFF == cdata[2])) {
				do_gettimeofday(&cam->workbuff->timestamp);
				cam->workbuff->seq = cam->frame_count++;
				cam->workbuff->data[0] = 0xFF;
				cam->workbuff->data[1] = 0xD8;
				cam->workbuff->length = 2;
				add_APPn(cam);
				add_COM(cam);
				memcpy(cam->workbuff->data+cam->workbuff->length,
				       cdata+2, n-2);
				DBG("Start of frame pattern found\n");
				cam->workbuff->length += n-2;
			} else {
				DBG("Ignoring packet, not beginning!\n");
			}
		} else if (cam->workbuff->length > 0) {
			memcpy(cam->workbuff->data + cam->workbuff->length,
			       cdata, n);
			cam->workbuff->length += n;
		}

		if ((cam->workbuff->length >= 3) &&
		    (cam->workbuff->data[cam->workbuff->length - 3] == 0xFF) &&
		    (cam->workbuff->data[cam->workbuff->length - 2] == 0xD9) &&
		    (cam->workbuff->data[cam->workbuff->length - 1] == 0xFF)) {
			frame_ready = true;
			cam->workbuff->data[cam->workbuff->length - 1] = 0;
			cam->workbuff->length -= 1;
		} else if ((cam->workbuff->length >= 2) &&
		   (cam->workbuff->data[cam->workbuff->length - 2] == 0xFF) &&
		   (cam->workbuff->data[cam->workbuff->length - 1] == 0xD9)) {
			frame_ready = true;
		}

		if ((frame_ready) || (cam->streaming == 3)) {
			DBG("Workbuff image size = %d\n",cam->workbuff->length);
			process_frame(cam);

			frame_ready = false;

			if (cam->streaming == 3)
				cam->streaming = 0;

			if (waitqueue_active(&cam->wq_stream))
				wake_up_interruptible(&cam->wq_stream);
		}
	}
}

/******************************************************************************
 *
 *  cpia2_usb_open
 *
 *****************************************************************************/
int cpia2_usb_open(struct camera_data *cam)
{
	urb_t *urb;
	int ret;
	int fx;
	int err;
	int i;

	if (!cam) {
		ERR("NULL pointer passed to "__FUNCTION__"\n");
		return -EINVAL;
	}

	for(i=0; i<NUM_SBUF; ++i) {
		cam->sbuf[i].data =
		    kmalloc(FRAMES_PER_DESC * FRAME_SIZE_PER_DESC, GFP_KERNEL);
		if (!cam->sbuf[i].data) {
			free_sbufs(cam);
			return -EINVAL;
		}
	}

	ret = cpia2_usb_set_alternate(cam, USBIF_CMDONLY);
	if (ret < 0) {
		ERR(__FUNCTION__": usb_set_interface error (ret = %d)\n",
		       ret);
		free_sbufs(cam);
		return -EBUSY;
	}

	/* We double buffer the Isoc lists */
	for(i=0; i<NUM_SBUF; ++i) {
		urb = usb_alloc_urb(FRAMES_PER_DESC);
		if (!urb) {
			ERR("unable to allocate urb %d\n",i);
			free_sbufs(cam);
			return -ENOMEM;
		}

		cam->sbuf[i].urb = urb;
		urb->dev = cam->dev;
		urb->context = cam;
		urb->pipe = usb_rcvisocpipe(cam->dev, 1);
		urb->transfer_flags = USB_ISO_ASAP;
		urb->transfer_buffer = cam->sbuf[i].data;
		urb->complete = cpia2_usb_complete;
		urb->number_of_packets = FRAMES_PER_DESC;
		urb->transfer_buffer_length =
			FRAME_SIZE_PER_DESC * FRAMES_PER_DESC;

		for (fx = 0; fx < FRAMES_PER_DESC; fx++) {
			urb->iso_frame_desc[fx].offset =
				FRAME_SIZE_PER_DESC * fx;
			urb->iso_frame_desc[fx].length = FRAME_SIZE_PER_DESC;
		}
	}

	cam->sbuf[0].urb->next = cam->sbuf[NUM_SBUF-1].urb;
	for(i=1; i<NUM_SBUF; ++i)
		cam->sbuf[i].urb->next = cam->sbuf[i-1].urb;

	for(i=0; i<NUM_SBUF; ++i) {
		err = usb_submit_urb(cam->sbuf[i].urb);
		if (err) {
			ERR("usb_submit_urb %d ret %d\n", i, err);
			for(--i; i>=0; --i)
				usb_unlink_urb(cam->sbuf[i].urb);
			free_sbufs(cam);
			return err;
		}
	}

	cam->streaming = 2;
	cam->mmapped = false;
	cam->frame_count = 0;

	return 0;
}

/******************************************************************************
 *
 * cpia2_usb_set_alternate
 *
 *****************************************************************************/
int cpia2_usb_set_alternate(struct camera_data *cam, unsigned int alt)
{
	int ret;

	DBG("New alt = %d\n", alt);
	if(alt != cam->cur_alt) {
		ret = usb_set_interface(cam->dev, cam->iface, USBIF_CMDONLY);

		/* Hardcode configuration, 1 is the only configuration
		 * for this camera */
		usb_set_configuration(cam->dev, 1);

		ret = usb_set_interface(cam->dev, cam->iface, alt);
		if (ret < 0)
			return -EBUSY;
	}
	
	cam->old_alt = cam->cur_alt;
	cam->cur_alt = alt;
	cam->state.current_alternate = alt;

	return 0;
}

/******************************************************************************
 *
 * free_sbufs
 *
 * free 0 to last_buf sbuf's.  Only called from the open function.
 *****************************************************************************/
static void free_sbufs(struct camera_data *cam)
{
	int i;

	for (i = 0; i < NUM_SBUF; i++) {
		if(cam->sbuf[i].urb) {
			usb_free_urb(cam->sbuf[i].urb);
			cam->sbuf[i].urb = NULL;
		}
		if(cam->sbuf[i].data) {
			kfree(cam->sbuf[i].data);
			cam->sbuf[i].data = NULL;
		}
	}
}

/*******
* Convenience functions
*******/
/****************************************************************************
 *
 *  write_packet
 *
 ***************************************************************************/
static int write_packet(struct usb_device *udev,
		        u8 request, u8 * registers, u16 start, size_t size)
{
	if (!registers || size <= 0)
		return -EINVAL;

	return usb_control_msg(udev,
	                       usb_sndctrlpipe(udev, 0),
	                       request,
	                       USB_TYPE_VENDOR | USB_RECIP_DEVICE,
	                       start,	/* value */
	                       0,	/* index */
	                       registers,	/* buffer */
	                       size,
	                       HZ);
}

/****************************************************************************
 *
 *  read_packet
 *
 ***************************************************************************/
static int read_packet(struct usb_device *udev,
		       u8 request, u8 * registers, u16 start, size_t size)
{
	if (!registers || size <= 0)
		return -EINVAL;

	return usb_control_msg(udev,
	                       usb_rcvctrlpipe(udev, 0),
	                       request,
	                       USB_DIR_IN|USB_TYPE_VENDOR|USB_RECIP_DEVICE,
	                       start,	/* value */
	                       0,	/* index */
	                       registers,	/* buffer */
	                       size,
	                       HZ);
}

/******************************************************************************
 *
 *  cpia2_usb_transfer_cmd
 *
 *****************************************************************************/
int cpia2_usb_transfer_cmd(struct camera_data *cam,
			   void *registers,
			   u8 request, u8 start, u8 count, u8 direction)
{
	int err = 0;
	struct usb_device *udev = cam->dev;

	if (!udev) {
		ERR(__FUNCTION__": Internal driver error: udev is NULL\n");
		return -EINVAL;
	}

	if (!registers) {
		ERR(__FUNCTION__": Internal driver error: register array is NULL\n");
		return -EINVAL;
	}

	if (direction == TRANSFER_READ) {
		err = read_packet(udev, request, (u8 *)registers, start, count);
		if (err > 0)
			err = 0;
	} else if (direction == TRANSFER_WRITE) {
		err =write_packet(udev, request, (u8 *)registers, start, count);
		if (err < 0) {
			LOG("Control message failed, err val = %d\n", err);
			LOG("Message: request = 0x%0X, start = 0x%0X\n",
			    request, start);
			LOG("Message: count = %d, register[0] = 0x%0X\n",
			    count, ((unsigned char *) registers)[0]);
		} else
			err=0;
	} else {
		LOG("Unexpected first byte of direction: %d\n",
		       direction);
		return -EINVAL;
	}

	if(err != 0)
		LOG("Unexpected error: %d\n", err);
	return err;
}


/******************************************************************************
 *
 *  cpia2_usb_camera_start
 *
 *  Restart streaming, possibly with an alternate if.
 *****************************************************************************/
int cpia2_usb_camera_start(struct camera_data *cam, unsigned int alternate)
{
	static unsigned char iso_regs[8][4] = {
		{0x00, 0x00, 0x00, 0x00},
		{0x00, 0x00, 0x00, 0x00},
		{0xB9, 0x00, 0x00, 0x7E},
		{0xB9, 0x00, 0x01, 0x7E},
		{0xB9, 0x00, 0x02, 0x7E},
		{0xB9, 0x00, 0x02, 0xFE},
		{0xB9, 0x00, 0x03, 0x7E},
		{0xB9, 0x00, 0x03, 0xFD}
	};
	struct cpia2_command cmd;
	unsigned char reg;
	unsigned int old_streaming = cam->streaming;

	/* suspend any running collection */
	if (cam->streaming == 1)
		cam->streaming = 2;

	/***
	 * Write the isoc registers according to the alternate selected
	 ***/
	cmd.direction = TRANSFER_WRITE;
	cmd.buffer.block_data[0] = iso_regs[alternate][0];
	cmd.buffer.block_data[1] = iso_regs[alternate][1];
	cmd.buffer.block_data[2] = iso_regs[alternate][2];
	cmd.buffer.block_data[3] = iso_regs[alternate][3];
	cmd.req_mode = CAMERAACCESS_TYPE_BLOCK | CAMERAACCESS_VC;
	cmd.start = CPIA2_VC_USB_ISOLIM;
	cmd.reg_count = 4;
	cpia2_send_command(cam, &cmd);

	/***
	 * Enable relevant streams before starting polling.
	 * First read USB Stream Config Register.
	 ***/
	cmd.direction = TRANSFER_READ;
	cmd.req_mode = CAMERAACCESS_TYPE_BLOCK | CAMERAACCESS_VC;
	cmd.start = CPIA2_VC_USB_STRM;
	cmd.reg_count = 1;
	cpia2_send_command(cam, &cmd);
	reg = cmd.buffer.block_data[0];

	/* Clear iso, bulk, and int */
	reg &= ~(CPIA2_VC_USB_STRM_BLK_ENABLE |
		 CPIA2_VC_USB_STRM_ISO_ENABLE |
		 CPIA2_VC_USB_STRM_INT_ENABLE);

	if (alternate == 1) {
		DBG("Enabling bulk xfer\n");
		reg |= CPIA2_VC_USB_STRM_BLK_ENABLE;	/* Enable Bulk */
		cam->xfer_mode = XFER_BULK;
	} else if (alternate > 1) {
		DBG("Enabling ISOC xfer\n");
		reg |= CPIA2_VC_USB_STRM_ISO_ENABLE;
		cam->xfer_mode = XFER_ISOC;
	}

	cmd.buffer.block_data[0] = reg;
	cmd.direction = TRANSFER_WRITE;
	cmd.start = CPIA2_VC_USB_STRM;
	cmd.reg_count = 1;
	cmd.req_mode = CAMERAACCESS_TYPE_BLOCK | CAMERAACCESS_VC;
	cpia2_send_command(cam, &cmd);

	if (alternate == 0) {
		cam->streaming = 2;
	} else {
		cam->streaming = old_streaming;
		cam->state.current_alternate = alternate;
	}

	return 0;
}

/******************************************************************************
 *
 *  cpia2_usb_stream_start
 *
 *****************************************************************************/
int cpia2_usb_stream_start(struct camera_data *cam, unsigned int alternate)
{
	DBG("Enter\n");
	if (cam->flush) {
		int i;
		DBG("Flushing buffers\n");
		for(i=0; i<cam->num_frames; ++i) {
			cam->buffers[i].status = FRAME_EMPTY;
			cam->buffers[i].length = 0;
		}
		cam->curbuff = &cam->buffers[0];
		cam->workbuff = cam->curbuff->next;
		cam->flush = false;
	}

	if (cam->old_alt == 0) {
		cpia2_usb_camera_start(cam, alternate);
	}

	if (cpia2_usb_set_alternate(cam, alternate) < 0) {
		ERR("Failed to set alternate in streamStart\n");
		return -1;
	}

	cam->streaming = 1;

	return 0;
}

/******************************************************************************
 *
 *  cpia2_usb_stream_stop
 *
 *****************************************************************************/
int cpia2_usb_stream_stop(struct camera_data *cam, int flush)
{
	if (flush) {
		cam->streaming = 3;
		return 0;
	}

	cam->streaming = 2;
	return 0;
}

/******************************************************************************
 *
 *  cpia2_usb_free_resources
 *
 *****************************************************************************/
static void cpia2_usb_free_resources(struct camera_data *cam)
{
	int i;
	
	if (!cam->streaming) {
		DBG("cam->streaming = false\n");
		return;
	}

	cam->streaming = 0;

	/* Set packet size to 0 */
	if (cam->present) {
		int ret;

		ret = usb_set_interface(cam->dev, cam->iface, USBIF_CMDONLY);
		if (ret < 0) {
			LOG("usb_set_interface error (ret = %d)\n", ret);
		}
	}

	/* Unschedule all of the iso td's */
	for(i=NUM_SBUF-1; i>=0; --i) {
		if (cam->sbuf[i].urb) {
			usb_unlink_urb(cam->sbuf[i].urb);
			usb_free_urb(cam->sbuf[i].urb);
			cam->sbuf[i].urb = NULL;
		}
	}
}

/******************************************************************************
 *
 *  cpia2_usb_close
 *
 *****************************************************************************/
int cpia2_usb_close(struct camera_data *cam)
{
	cpia2_usb_free_resources(cam);

	return 0;
}

/******************************************************************************
 *
 *  cpia2_usb_probe
 *
 *  Probe and initialize.
 *****************************************************************************/
static void *cpia2_usb_probe(struct usb_device *udev,
			     unsigned int ifnum,
			     const struct usb_device_id *id)
{
	struct usb_interface_descriptor *interface;
	struct camera_data *cam;
	int ret;

	interface = &udev->actconfig->interface[ifnum].altsetting[0];

	if (ifnum != 0) {	/* Only interface available for CPia2 */
		ERR(__FUNCTION__": Invalid interface number %d\n", ifnum);
		return NULL;
	}

	/* If we get to this point, we found a CPiA2 camera */
	LOG("USB CPiA2 camera found\n");

	if((cam = cpia2_init_camera_struct()) == NULL)
		return NULL;
	
	cam->dev = udev;
	cam->iface = interface->bInterfaceNumber;

	/* Hardcode, 1 is the only configuration for this camera */
	usb_set_configuration(udev, 1);
	ret = cpia2_usb_set_alternate(cam, USBIF_CMDONLY);
	if (ret < 0) {
		ERR(__FUNCTION__": usb_set_interface error (ret = %d)\n",
		       ret);
		kfree(cam);
		return NULL;
	}
	
	if (cpia2_register_camera(cam) != 0) {
		ERR(__FUNCTION__": failed to cpia2_register_camera\n");
		kfree(cam);
		return NULL;
	}


	if(cpia2_init_camera(cam) != 0) {
		ERR(__FUNCTION__": failed to cpia2_init_camera\n");
		cpia2_unregister_camera(cam);
		kfree(cam);
		return NULL;
	}
	LOG("  CPiA Version: %d.%02d (%d.%d)\n",
	       cam->params.version.firmware_revision_hi,
	       cam->params.version.firmware_revision_lo,
	       cam->params.version.asic_id,
	       cam->params.version.asic_rev);
	LOG("  CPiA PnP-ID: %04x:%04x:%04x\n",
	       cam->params.pnp_id.vendor,
	       cam->params.pnp_id.product,
	       cam->params.pnp_id.device_revision);
	LOG("  SensorID: %d.(version %d)\n",
	       cam->params.version.sensor_flags,
	       cam->params.version.sensor_rev);

	return cam;
}

/******************************************************************************
 *
 *  cpia2_disconnect
 *
 *****************************************************************************/
static void cpia2_usb_disconnect(struct usb_device *udev, void *ptr)
{
	struct camera_data *cam = (struct camera_data *) ptr;

	cam->present = 0;

	cpia2_usb_free_resources(cam);

	cpia2_unregister_camera(cam);

	if(cam->buffers) {
		cam->curbuff->status = FRAME_READY;
		cam->curbuff->length = 0;
		if (waitqueue_active(&cam->wq_stream))
			wake_up_interruptible(&cam->wq_stream);
	}
	
	usb_driver_release_interface(&cpia2_driver,
				     &udev->actconfig->interface[0]);

	if (cam->open_count == 0) {
		kfree(cam);
	}
}

/******************************************************************************
 *
 *  usb_cpia2_init
 *
 *****************************************************************************/
int cpia2_usb_init(void)
{
	return usb_register(&cpia2_driver);
}

/******************************************************************************
 *
 *  usb_cpia_cleanup
 *
 *****************************************************************************/
void cpia2_usb_cleanup(void)
{
	schedule_timeout(2 * HZ);
	usb_deregister(&cpia2_driver);
}
