package gui

import (
	"bytes"
	"encoding/base64"
	"io/ioutil"
	"net/http"
	"os"
	"path"
	"sync"
	"time"
)

type _escLocalFS struct{}

var _escLocal _escLocalFS

type _escStaticFS struct{}

var _escStatic _escStaticFS

type _escDirectory struct {
	fs   http.FileSystem
	name string
}

type _escFile struct {
	compressed string
	size       int64
	modtime    int64
	local      string
	isDir      bool

	once sync.Once
	data []byte
	name string
}

func (_escLocalFS) Open(name string) (http.File, error) {
	f, present := _escData[path.Clean(name)]
	if !present {
		return nil, os.ErrNotExist
	}
	return os.Open(f.local)
}

func (_escStaticFS) prepare(name string) (*_escFile, error) {
	f, present := _escData[path.Clean(name)]
	if !present {
		return nil, os.ErrNotExist
	}
	var err error
	f.once.Do(func() {
		f.name = path.Base(name)
		if f.size == 0 {
			return
		}
		b64 := base64.NewDecoder(base64.StdEncoding, bytes.NewBufferString(f.compressed))
		f.data, err = ioutil.ReadAll(b64)
	})
	if err != nil {
		return nil, err
	}
	return f, nil
}

func (fs _escStaticFS) Open(name string) (http.File, error) {
	f, err := fs.prepare(name)
	if err != nil {
		return nil, err
	}
	return f.File()
}

func (dir _escDirectory) Open(name string) (http.File, error) {
	return dir.fs.Open(dir.name + name)
}

func (f *_escFile) File() (http.File, error) {
	type httpFile struct {
		*bytes.Reader
		*_escFile
	}
	return &httpFile{
		Reader:   bytes.NewReader(f.data),
		_escFile: f,
	}, nil
}

func (f *_escFile) Close() error {
	return nil
}

func (f *_escFile) Readdir(count int) ([]os.FileInfo, error) {
	return nil, nil
}

func (f *_escFile) Stat() (os.FileInfo, error) {
	return f, nil
}

func (f *_escFile) Name() string {
	return f.name
}

func (f *_escFile) Size() int64 {
	return f.size
}

func (f *_escFile) Mode() os.FileMode {
	return 0
}

func (f *_escFile) ModTime() time.Time {
	return time.Unix(f.modtime, 0)
}

func (f *_escFile) IsDir() bool {
	return f.isDir
}

func (f *_escFile) Sys() interface{} {
	return f
}

// FS returns a http.Filesystem for the embedded assets. If useLocal is true,
// the filesystem's contents are instead used.
func FS(useLocal bool) http.FileSystem {
	if useLocal {
		return _escLocal
	}
	return _escStatic
}

// Dir returns a http.Filesystem for the embedded assets on a given prefix dir.
// If useLocal is true, the filesystem's contents are instead used.
func Dir(useLocal bool, name string) http.FileSystem {
	if useLocal {
		return _escDirectory{fs: _escLocal, name: name}
	}
	return _escDirectory{fs: _escStatic, name: name}
}

// FSByte returns the named file from the embedded assets. If useLocal is
// true, the filesystem's contents are instead used.
func FSByte(useLocal bool, name string) ([]byte, error) {
	if useLocal {
		f, err := _escLocal.Open(name)
		if err != nil {
			return nil, err
		}
		b, err := ioutil.ReadAll(f)
		f.Close()
		return b, err
	}
	f, err := _escStatic.prepare(name)
	if err != nil {
		return nil, err
	}
	return f.data, nil
}

// FSMustByte is the same as FSByte, but panics if name is not present.
func FSMustByte(useLocal bool, name string) []byte {
	b, err := FSByte(useLocal, name)
	if err != nil {
		panic(err)
	}
	return b
}

// FSString is the string version of FSByte.
func FSString(useLocal bool, name string) (string, error) {
	b, err := FSByte(useLocal, name)
	return string(b), err
}

// FSMustString is the string version of FSMustByte.
func FSMustString(useLocal bool, name string) string {
	return string(FSMustByte(useLocal, name))
}

var _escData = map[string]*_escFile{

	"/definitions/AccountDetails.xml": {
		local:   "definitions/AccountDetails.xml",
		size:    39293,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0U3RvcmUiIGlkPSJwcm94aWVzLW1vZGVs
Ij4KICAgIDxjb2x1bW5zPgogICAgICA8IS0tIHByb3h5IC0tPgogICAgICA8Y29sdW1uIHR5cGU9Imdj
aGFyYXJyYXkiLz4KICAgICAgPCEtLSByZWFsIHByb3h5IGRhdGEgLS0+CiAgICAgIDxjb2x1bW4gdHlw
ZT0iZ2NoYXJhcnJheSIvPgogICAgPC9jb2x1bW5zPgogIDwvb2JqZWN0PgogIDxvYmplY3QgY2xhc3M9
Ikd0a0xpc3RTdG9yZSIgaWQ9InBpbnMtbW9kZWwiPgogICAgPGNvbHVtbnM+CiAgICAgIDwhLS0gU3Vi
amVjdCAtLT4KICAgICAgPGNvbHVtbiB0eXBlPSJnY2hhcmFycmF5Ii8+CiAgICAgIDwhLS0gSXNzdWVy
IC0tPgogICAgICA8Y29sdW1uIHR5cGU9ImdjaGFyYXJyYXkiLz4KICAgICAgPCEtLSBGaW5nZXJwcmlu
dCAtLT4KICAgICAgPGNvbHVtbiB0eXBlPSJnY2hhcmFycmF5Ii8+CiAgICA8L2NvbHVtbnM+CiAgPC9v
YmplY3Q+CiAgPG9iamVjdCBjbGFzcz0iR3RrRGlhbG9nIiBpZD0iQWNjb3VudERldGFpbHMiPgogICAg
PHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5h
bWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+QWNjb3VudCBEZXRhaWxzPC9wcm9wZXJ0eT4KICAg
IDxzaWduYWwgbmFtZT0iY2xvc2UiIGhhbmRsZXI9Im9uX2NhbmNlbCIgLz4KICAgIDxwcm9wZXJ0eSBu
YW1lPSJ0eXBlX2hpbnQiPm5vcm1hbDwvcHJvcGVydHk+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9
InZib3giPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJWYm94Ij4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkIGludGVy
bmFsLWNoaWxkPSJhY3Rpb25fYXJlYSI+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25C
b3giIGlkPSJidXR0b25fYm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbiI+MTA8
L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtC
dXR0b24iIGlkPSJjYW5jZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0
cmFuc2xhdGFibGU9InllcyI+Q2FuY2VsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX2NhbmNlbCIgc3dhcHBlZD0ibm8iLz4KICAg
ICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtCdXR0b24iIGlkPSJzYXZlIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJs
YWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNhdmU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImNhbl9kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX3NhdmUiIHN3YXBwZWQ9Im5vIi8+
CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTm90
ZWJvb2siIGlkPSJub3RlYm9vazEiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzaG93LWJvcmRlciI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFnZSI+MDwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tYm90dG9tIj4xMDwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBp
ZD0ibm90aWZpY2F0aW9uLWFyZWEiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bWFyZ2luLXRvcCI+NTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im1hcmdpbi1sZWZ0Ij41PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ibWFyZ2luLXJpZ2h0Ij41PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9WRVJUSUNBTDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxzdHlsZT4KICAgICAgICAgICAgICAgICAgICAgIDxjbGFzcyBuYW1l
PSJub3RpZmljYXRpb25zIi8+CiAgICAgICAgICAgICAgICAgICAgPC9zdHlsZT4KICAgICAgICAgICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4K
ICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrR3JpZCIgaWQ9ImdyaWQiPgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tdG9wIj4xNTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1ib3R0b20iPjEwPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXN0YXJ0Ij4xMDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1lbmQiPjEwPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93LXNwYWNpbmciPjEyPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY29sdW1uLXNwYWNpbmci
PjY8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMi
PkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9IkFjY291bnRNZXNzYWdlTGFiZWwiPgog
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNs
YXRhYmxlPSJ5ZXMiPllvdXIgYWNjb3VudCYjeEE7KGV4YW1wbGU6IGtpbTQyQGphYmJlci5vdHIuaW0p
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnki
PkdUS19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0VudHJ5
IiBpZD0iYWNjb3VudCI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZh
dGUiIGhhbmRsZXI9Im9uX3NhdmUiIC8+CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9
IlBhc3N3b3JkTGFiZWwiPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2Fu
X2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlBhc3N3b3JkPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX0VORDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2gi
PjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0
dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRyeSIgaWQ9InBhc3N3b3JkIj4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2liaWxpdHkiPmZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25f
c2F2ZSIgLz4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0
dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0
b3AtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAg
ICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0iZGlzcGxheU5hbWVMYWJl
bCI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0
cmFuc2xhdGFibGU9InllcyI+QWNjb3VudCBuYW1lIChvcHRpb25hbCk8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fRU5EPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAg
PHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFj
aCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3At
YXR0YWNoIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAg
ICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAg
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0VudHJ5IiBpZD0iZGlzcGxheU5hbWUiPgogICAg
ICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9zYXZl
IiAvPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNo
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1h
dHRhY2giPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAg
ICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJzaG93T3RoZXJTZXR0aW5ncyI+
CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFu
c2xhdGFibGU9InllcyI+RGlzcGxheSBhbGwgc2V0dGluZ3M8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+R1RLX0pVU1RJRllfUklHSFQ8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJ
R05fRU5EPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ0b3AtYXR0YWNoIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NoZWNrQnV0dG9uIiBpZD0i
b3RoZXJTZXR0aW5ncyI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0idG9nZ2xl
ZCIgaGFuZGxlcj0ib25fdG9nZ2xlX290aGVyX3NldHRpbmdzIiAvPgogICAgICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjM8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAg
ICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZCB0eXBlPSJ0YWIi
PgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ibGFiZWwtdGFiMSI+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5BY2Nv
dW50PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idGFiLWZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4K
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtHcmlkIiBpZD0ib3RoZXJPcHRpb25zR3JpZCI+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXRvcCI+MTU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1ib3R0b20iPjEwPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tc3RhcnQiPjEwPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tZW5kIj4xMDwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93LXNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1bW4tc3BhY2luZyI+NjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xh
YmVsIiBpZD0ic2VydmVyTGFiZWwiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJj
YW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5TZXJ2ZXIgKGxlYXZlIGVtcHR5IGZvciBkZWZhdWx0
KTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdU
S19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJs
ZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InRvcC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAg
ICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRyeSIgaWQ9InNlcnZlciI+CiAgICAgICAgICAgICAgICAgICAg
PHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25fc2F2ZSIgc3dhcHBlZD0ibm8iLz4KICAg
ICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9
InBvcnRMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIg
dHJhbnNsYXRhYmxlPSJ5ZXMiPlBvcnQgKGxlYXZlIGVtcHR5IGZvciBkZWZhdWx0KTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX0VORDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4xPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrRW50cnkiIGlkPSJwb3J0Ij4KICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2
YXRlIiBoYW5kbGVyPSJvbl9zYXZlIiAvPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imxl
ZnRfYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dG9wX2F0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAg
ICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0icGlubmluZ1BvbGljeUxhYmVsIj4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+UGlu
bmluZyBwb2xpY3k8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJo
YWxpZ24iPkdUS19BTElHTl9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0idG9wLWF0dGFjaCI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4K
ICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NvbWJvQm94VGV4dCIgaWQ9InBpbm5pbmdQb2xpY3lWYWx1
ZSI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxpdGVtcz4KICAgICAgICAgICAgICAgICAgICAgIDxp
dGVtIHRyYW5zbGF0YWJsZT0ieWVzIiBpZD0ibm9uZSI+Tm9uZTwvaXRlbT4KICAgICAgICAgICAgICAg
ICAgICAgIDxpdGVtIHRyYW5zbGF0YWJsZT0ieWVzIiBpZD0iZGVueSI+RGVueTwvaXRlbT4KICAgICAg
ICAgICAgICAgICAgICAgIDxpdGVtIHRyYW5zbGF0YWJsZT0ieWVzIiBpZD0iYWRkIj5BbHdheXMgYWRk
PC9pdGVtPgogICAgICAgICAgICAgICAgICAgICAgPGl0ZW0gdHJhbnNsYXRhYmxlPSJ5ZXMiIGlkPSJh
ZGQtZmlyc3QtYXNrLXJlc3QiPkFkZCB0aGUgZmlyc3QsIGFzayBmb3IgdGhlIHJlc3Q8L2l0ZW0+CiAg
ICAgICAgICAgICAgICAgICAgICA8aXRlbSB0cmFuc2xhdGFibGU9InllcyIgaWQ9ImFkZC1maXJzdC1k
ZW55LXJlc3QiPkFkZCB0aGUgZmlyc3QsIGRlbnkgdGhlIHJlc3Q8L2l0ZW0+CiAgICAgICAgICAgICAg
ICAgICAgICA8aXRlbSB0cmFuc2xhdGFibGU9InllcyIgaWQ9ImFzayI+QWx3YXlzIGFzazwvaXRlbT4K
ICAgICAgICAgICAgICAgICAgICA8L2l0ZW1zPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idG9wLWF0dGFjaCI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a1BhbmVkIiBpZD0iaHBhbmVkUGlucyI+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW4tZm9jdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTc1PC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtT
Y3JvbGxlZFdpbmRvdyIgaWQ9InNjcm9sbGVkd2luZG93UGlucyI+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJoc2Nyb2xsYmFyLXBvbGljeSI+R1RLX1BPTElDWV9BVVRPTUFUSUM8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idnNjcm9sbGJh
ci1wb2xpY3kiPkdUS19QT0xJQ1lfQVVUT01BVElDPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoLXJlcXVlc3QiPjE3MDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZWlnaHQtcmVxdWVzdCI+MTUwPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbiI+NTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW4tZm9jdXMiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhw
YW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iY2FuLWZvY3VzIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InNoYWRvdy10eXBlIj5pbjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtUcmVl
VmlldyIgaWQ9InBpbnMtdmlldyI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibW9kZWwiPnBpbnMtbW9kZWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbi1mb2N1cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGVhZGVycy12aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzaG93
LWV4cGFuZGVycyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InJlb3JkZXJhYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0ic2VsZWN0aW9uIj4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrVHJlZVNlbGVjdGlvbiIgaWQ9InBpbnMtc2Vs
ZWN0aW9uIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibW9k
ZSI+R1RLX1NFTEVDVElPTl9TSU5HTEU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a1RyZWVWaWV3Q29sdW1uIiBpZD0icGlucy1zdWJqZWN0LWNvbHVtbiI+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIj5TdWJqZWN0
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDZWxsUmVuZGVyZXJUZXh0
IiBpZD0icGlucy1zdWJqZWN0LWNvbHVtbi1yZW5kZXJlZCIvPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxhdHRyaWJ1dGUgbmFtZT0idGV4dCI+MDwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrVHJlZVZpZXdDb2x1
bW4iIGlkPSJwaW5zLWlzc3Vlci1jb2x1bW4iPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSI+SXNzdWVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtDZWxsUmVuZGVyZXJUZXh0IiBpZD0icGlucy1pc3N1ZXItY29sdW1uLXJlbmRl
cmVkIi8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRlcz4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ0ZXh0Ij4xPC9hdHRy
aWJ1dGU+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtUcmVlVmlld0NvbHVtbiIgaWQ9InBpbnMtZnByLWNvbHVtbiI+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIj5GaW5nZXJwcmlu
dDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2VsbFJlbmRlcmVyVGV4
dCIgaWQ9InBpbnMtZnByLWNvbHVtbi1yZW5kZXJlZCIvPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxh
dHRyaWJ1dGUgbmFtZT0idGV4dCI+MjwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hp
bGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJyZXNpemUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ic2hyaW5rIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlk
PSJ2Ym94LXBpbmJ1dHRvbnMiPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bWFyZ2luIj41PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iY2FuLWZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCdXR0b24iIGlkPSJyZW1vdmVfcGluX2J1dHRvbiI+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5fUmVtb3ZlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJj
YW4tZm9jdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InJlY2VpdmVzLWRlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InVzZV91bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJv
bl9yZW1vdmVfcGluIiAvPgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVzaXplIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzaHJpbmsiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnRfYXR0YWNo
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wX2F0dGFj
aCI+NDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoIj4y
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrU2Nyb2xsZWRXaW5kb3ciIGlkPSJwaW5uaW5nUG9saWN5U2Nyb2xsZWRXaW5kb3ciPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2hhZG93X3R5cGUiPmluPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWF4X2NvbnRlbnRfaGVpZ2h0Ij4y
MDA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwcm9wYWdhdGVf
bmF0dXJhbF93aWR0aCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InByb3BhZ2F0ZV9uYXR1cmFsX2hlaWdodCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrVmll
d3BvcnQiIGlkPSJwaW5uaW5nUG9saWN5Vmlld1BvcnQiPgogICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0xhYmVsIiBpZD0icGlubmluZ1BvbGljeUluc3RydWN0aW9ucyI+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5z
bGF0YWJsZT0ieWVzIj4mbHQ7YiZndDtDZXJ0aWZpY2F0ZSBwaW5uaW5nIHBvbGljeSZsdDsvYiZndDsK
ClRoZSBwaW5uaW5nIHBvbGljeSBnb3Zlcm5zIHdoZXRoZXIgd2Ugd2lsbCBjb25zaWRlciBzYXZpbmcg
aW5mb3JtYXRpb24gYWJvdXQgY2VydGlmaWNhdGVzIHRoYXQgd2UgaGF2ZSBzZWVuIGJlZm9yZSwgYW5k
IGhvdyB3ZSB3aWxsIHJlYWN0IGluIGRpZmZlcmVudCBjYXNlcy4gQSBwaW5uZWQgY2VydGlmaWNhdGUg
aW1wbGllcyB0aGF0IHdlIHRydXN0IHRoYXQgY2VydGlmaWNhdGUgdG8gc2VjdXJlIHRoZSBjb25uZWN0
aW9uLiBUaGUgYXZhaWxhYmxlIG9wdGlvbnMgYW5kIHRoZWlyIG1lYW5pbmcgYXJlIGRlc2NyaWJlZCBi
ZWxvdzoKCiZsdDtiJmd0O05vbmUmbHQ7L2ImZ3Q7OiBEaXNhYmxlIGNlcnRpZmljYXRlIHBpbm5pbmcu
CgombHQ7YiZndDtEZW55Jmx0Oy9iJmd0OzogRG8gbm90IHBpbiBuZXcgY2VydGlmaWNhdGVzIGFuZCBv
bmx5IHVzZSB0aGUgZXhpc3RpbmcgcGlubmVkIGNlcnRpZmljYXRlcy4gSW4gY2FzZSBhIGNlcnRpZmlj
YXRlIG9mZmVyZWQgYnkgdGhlIHNlcnZlciBpcyBub3QgZm91bmQsIHRoZSBjb25uZWN0aW9uIGlzIHJl
ZnVzZWQuCgombHQ7YiZndDtBbHdheXMgYWRkJmx0Oy9iJmd0OzogQXV0b21hdGljYWxseSBwaW4gbmV3
IGNlcnRpZmljYXRlcyBmcm9tIHRoZSBzZXJ2ZXIuIEEgdXNlIGNhc2UgaXMgd2hlbiBjb25uZWN0aW5n
IHRvIG11bHRpcGxlIHNlcnZlcnMgaW5pdGlhbGx5IHRvIHN0b3JlIHRoZWlyIGNlcnRpZmljYXRlLCB0
aGVuIGNoYW5nZSB0aGUgb3B0aW9uIGxhdGVyIHRvIGEgbW9yZSByZXN0cmljdGVkIG9uZSBhbmQgY3Vy
YXRlIHRoZW0gbWFudWFsbHkuCgombHQ7aSZndDtOT1RFIC0gVGhpcyBvcHRpb24gbXVzdCBiZSB1c2Vk
IHdpdGggY2F1dGlvbi4mbHQ7L2kmZ3Q7CgombHQ7YiZndDtBZGQgdGhlIGZpcnN0LCBhc2sgZm9yIHRo
ZSByZXN0Jmx0Oy9iJmd0OzogIEF1dG9tYXRpY2FsbHkgcGluIHRoZSBmaXJzdCBjZXJ0aWZpY2F0ZSB3
ZSBldmVyIHNlZSBhbmQgdGhlbiBmb3IgZnV0dXJlIHVucGlubmVkIGNlcnRpZmljYXRlcyB3ZSBlbmNv
dW50ZXIgd2UgYXNrIHRoZSB1c2VyIHdoZXRoZXIgdG8gYWNjZXB0IG9yIHRvIHJlamVjdC4KCiZsdDti
Jmd0O0FkZCB0aGUgZmlyc3QsIGRlbnkgdGhlIHJlc3QmbHQ7L2ImZ3Q7OiBBdXRvbWF0aWNhbGx5IHBp
biB0aGUgZmlyc3QgY2VydGlmaWNhdGUgd2UgZXZlciBzZWUgYW5kIHRoZW4gZm9yIGZ1dHVyZSB1bnBp
bm5lZCBjZXJ0aWZpY2F0ZXMgd2UgcmVqZWN0IHRoZW0gYWxsLgoKJmx0O2ImZ3Q7QWx3YXlzIGFzayZs
dDsvYiZndDs6IEFzayB0aGUgdXNlciBldmVyeSB0aW1lIHdlIGVuY291bnRlciBhIG5ldyB1bnBpbm5l
ZCBjZXJ0aWZpY2F0ZSB0byBhY2NlcHQgb3IgdG8gcmVqZWN0IGl0LjwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX21hcmt1cCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWF4X3dp
ZHRoX2NoYXJzIj41MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImxlZnRfYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0idG9wX2F0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9IndpZHRoIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAg
ICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICA8Y2hpbGQgdHlwZT0idGFiIj4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJl
bCIgaWQ9ImxhYmVsLXRhYjIiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0
cmFuc2xhdGFibGU9InllcyI+U2VydmVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idGFiLWZpbGwiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlk
PSJ2Ym94MSI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luIj41PC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW4tZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjY8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtQ
YW5lZCIgaWQ9ImhwYW5lZDEiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNp
YmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2Fu
LWZvY3VzIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cG9zaXRpb24iPjE3NTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2Nyb2xsZWRXaW5kb3ciIGlkPSJzY3JvbGxl
ZHdpbmRvdzEiPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaHNjcm9sbGJh
ci1wb2xpY3kiPkdUS19QT0xJQ1lfTkVWRVI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idnNjcm9sbGJhci1wb2xpY3kiPkdUS19QT0xJQ1lfQVVUT01BVElDPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoLXJlcXVl
c3QiPjE3MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJo
ZWlnaHQtcmVxdWVzdCI+MjMwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Im1hcmdpbiI+NTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW4tZm9jdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2hhZG93LXR5cGUiPmluPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a1RyZWVWaWV3IiBpZD0icHJveGllcy12aWV3Ij4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtb2RlbCI+cHJveGllcy1tb2RlbDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWZv
Y3VzIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJoZWFkZXJzLXZpc2libGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzaG93LWV4cGFuZGVycyI+RmFsc2U8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlb3JkZXJhYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0icm93LWFjdGl2
YXRlZCIgaGFuZGxlcj0ib25fZWRpdF9hY3RpdmF0ZV9wcm94eSIgLz4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0ic2VsZWN0aW9uIj4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrVHJlZVNlbGVjdGlvbiIgaWQ9InNlbGVjdGlv
biI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1vZGUiPkdU
S19TRUxFQ1RJT05fU0lOR0xFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtUcmVlVmlld0NvbHVtbiIgaWQ9InByb3h5LW5hbWUtY29sdW1uIj4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiPnByb3h5LW5hbWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NlbGxSZW5kZXJlclRleHQiIGlk
PSJwcm94eS1uYW1lLWNvbHVtbi1yZW5kZXJlZCIvPgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxhdHRy
aWJ1dGUgbmFtZT0idGV4dCI+MDwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJyZXNpemUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ic2hyaW5rIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJ2
Ym94MyI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4iPjU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW4tZm9j
dXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIg
aWQ9ImFkZF9idXR0b24iPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+X0FkZC4uLjwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWZvY3VzIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlcy1k
ZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ1c2VfdW5kZXJsaW5lIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fYWRkX3Byb3h5IiAvPgogICAg
ICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L3Bh
Y2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0
b24iIGlkPSJyZW1vdmVfYnV0dG9uIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPl9SZW1vdmUuLi48L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbi1mb2N1cyI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cmVjZWl2ZXMtZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX3JlbW92ZV9w
cm94eSIgLz4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrQnV0dG9uIiBpZD0iZWRpdF9idXR0b24iPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+X0VkaXQuLi48L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNh
bi1mb2N1cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icmVjZWl2ZXMtZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlLXVuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9u
X2VkaXRfcHJveHkiIC8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tp
bmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZXNpemUiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNocmluayI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAg
ICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQgdHlwZT0idGFi
Ij4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9ImxhYmVsLXRhYjMiPgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+UHJv
eGllczwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRhYi1maWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iZW5jcnlwdGlvbk9wdGlvbnNC
b3giPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlci13aWR0aCI+MTA8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhvbW9nZW5lb3VzIj5mYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklF
TlRBVElPTl9WRVJUSUNBTDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0iZmluZ2VycHJpbnRzTWVzc2FnZSI+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xF
RlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdU
S19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtHcmlkIiBpZD0iZW5jcnlwdGlvbkdyaWQiPgogICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWJvdHRvbSI+MTA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tc3RhcnQiPjEwPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWVuZCI+MTA8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyb3ctc3BhY2luZyI+MTI8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1bW4tc3BhY2luZyI+NjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJlbmNyeXB0aW9uSW1wb3J0SW5zdHJ1Y3Rpb25zIj4KICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJh
bnNsYXRhYmxlPSJ5ZXMiPlRoZSBiZWxvdyBidXR0b25zIGFsbG93IHlvdSB0byBpbXBvcnQgcHJpdmF0
ZSBrZXlzIGFuZCBmaW5nZXJwcmludHMuIEJvdGggb2YgdGhlbSBzaG91bGQgYmUgaW4gdGhlIFBpZGdp
bi9saWJvdHIgZm9ybWF0LiBJZiB5b3UgaW1wb3J0IHByaXZhdGUga2V5cywgeW91ciBleGlzdGluZyBw
cml2YXRlIGtleXMgd2lsbCBiZSBkZWxldGVkLCBzaW5jZSBjdXJyZW50bHkgdGhlcmUgaXMgbm8gd2F5
IHRvIGNob29zZSB3aGljaCBrZXkgdG8gdXNlIGZvciBlbmNyeXB0ZWQgY2hhdC4KClRoZXJlIGFyZSBz
ZXZlcmFsIGFwcGxpY2F0aW9ucyB0aGF0IHVzZSB0aGUgbGlib3RyIGZvcm1hdCwgc3VjaCBhcyBQaWRn
aW4sIEFkaXVtIG9yIFRvciBNZXNzZW5nZXIuIERlcGVuZGluZyBvbiB5b3VyIHBsYXRmb3JtLCB0aGVz
ZSBmaWxlcyBjYW4gYmUgZm91bmQgaW4gc2V2ZXJhbCBkaWZmZXJlbnQgcGxhY2VzLiBSZWZlciB0byB0
aGUgZG9jdW1lbnRhdGlvbiBmb3IgdGhlIGFwcGxpY2F0aW9uIGluIHF1ZXN0aW9uIHRvIGZpbmQgb3V0
IHdoZXJlIHRoZSBmaWxlcyBhcmUgbG9jYXRlZCBmb3IgeW91ciBwbGF0Zm9ybS4gVGhlIGZpbGVuYW1l
cyB0byBsb29rIGZvciBhcmUgIm90ci5maW5nZXJwcmludHMiIGFuZCAib3RyLnByaXZhdGVfa2V5Ii48
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFw
Ij50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1h
eC13aWR0aC1jaGFycyI+NTA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0id2lkdGgiPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9u
IiBpZD0iaW1wb3J0X2tleV9idXR0b24iPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5JbXBvcnQgUHJpdmF0ZSBfS2V5cy4uLjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbi1mb2N1
cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJy
ZWNlaXZlcy1kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InVzZV91bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX2ltcG9ydF9rZXkiIC8+CiAg
ICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5n
PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+
MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJpbXBvcnRfZnByX2J1dHRvbiI+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMi
PkltcG9ydCBfRmluZ2VycHJpbnRzLi4uPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWZvY3VzIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzLWRlZmF1bHQiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX3VuZGVybGluZSI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIg
aGFuZGxlcj0ib25faW1wb3J0X2ZwciIgLz4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBp
ZD0iZW5jcnlwdGlvbkV4cG9ydEluc3RydWN0aW9ucyI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+VGhlIGJlbG93IGJ1
dHRvbnMgYWxsb3cgeW91IHRvIGV4cG9ydCBwcml2YXRlIGtleXMgYW5kIGZpbmdlcnByaW50cy4gQmUg
Y2FyZWZ1bCB3aXRoIHRoZSBmaWxlcyB0aGF0IGNvbWUgb3V0IG9mIHRoaXMgcHJvY2VzcyBhcyB0aGV5
IGNvbnRhaW4gcG90ZW50aWFsbHkgc2Vuc2l0aXZlIGRhdGEuIFRoZSBleHBvcnQgd2lsbCBiZSBpbiB0
aGUgUGlkZ2luL2xpYm90ciBmb3JtYXQuPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXgtd2lkdGgtY2hhcnMiPjUwPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4yPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoIj4yPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAg
ICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImV4cG9ydF9rZXlfYnV0dG9uIj4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+RXhw
b3J0IFByaXZhdGUgS2V5cy4uLjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImNhbi1mb2N1cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlcy1kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9leHBv
cnRfa2V5IiAvPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQt
YXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InRvcC1hdHRhY2giPjM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iZXhwb3J0X2Zwcl9i
dXR0b24iPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5z
bGF0YWJsZT0ieWVzIj5FeHBvcnQgRmluZ2VycHJpbnRzLi4uPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWZvY3VzIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzLWRlZmF1bHQiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQi
IGhhbmRsZXI9Im9uX2V4cG9ydF9mcHIiIC8+CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICA8Y2hpbGQgdHlwZT0idGFiIj4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9ImxhYmVsLXRhYjQiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+RW5jcnlwdGlvbjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icG9zaXRpb24iPjM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InRhYi1maWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAg
ICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9j
aGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPHN0eWxlPgogICAgICA8Y2xhc3Mg
bmFtZT0iY295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/AccountRegistration.xml": {
		local:   "definitions/AccountRegistration.xml",
		size:    3856,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtBc3Npc3RhbnQiIGlkPSJhc3Npc3RhbnQiPgog
ICAgPHNpZ25hbCBuYW1lPSJjbG9zZSIgaGFuZGxlcj0ib25fY2FuY2VsIiAvPgogICAgPHNpZ25hbCBu
YW1lPSJjYW5jZWwiIGhhbmRsZXI9Im9uX2NhbmNlbCIgLz4KICAgIDxzaWduYWwgbmFtZT0icHJlcGFy
ZSIgaGFuZGxlcj0ib25fcHJlcGFyZSIgLz4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrQm94IiBpZD0iZGlhbG9nIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luIj4xMDwvcHJv
cGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEwPC9wcm9wZXJ0eT4KCSAgICA8
Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtHcmlkIj4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Im1hcmdpbi1ib3R0b20iPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InJvdy1zcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJjb2x1bW4tc3BhY2luZyI+NjwvcHJvcGVydHk+CgkgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InNlcnZlci1sYWJlbCI+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5SZWNvbW1lbmRlZCBzZXJ2
ZXJzOiA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdU
S19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJz
ZWxlY3RhYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+
MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ29tYm9Cb3hUZXh0
IiBpZD0ic2VydmVyIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYXMtZW50cnkiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5n
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CgkgICAgICA8L29iamVjdD4K
ICAgICAgICA8L2NoaWxkPgogICAgICAgIDxzdHlsZT4KICAgICAgICAgIDxjbGFzcyBuYW1lPSJyZWdp
c3RyYXRpb24iLz4KICAgICAgICAgIDxjbGFzcyBuYW1lPSJzZXJ2ZXJTZWxlY3Rpb24iLz4KICAgICAg
ICA8L3N0eWxlPgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+MS4gQ2hvb3NlIGEgc2VydmVyPC9wcm9wZXJ0
eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFnZS10eXBlIj5HVEtfQVNTSVNUQU5UX1BBR0VfSU5U
Uk88L3Byb3BlcnR5PgoJICAgIDxwcm9wZXJ0eSBuYW1lPSJjb21wbGV0ZSI+dHJ1ZTwvcHJvcGVydHk+
CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0JveCIgaWQ9ImZvcm0iPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0ltYWdlIiBpZD0iZm9ybUltYWdlIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1h
cmdpbl9yaWdodCI+MTU8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGls
ZD4KCSAgICA8Y2hpbGQ+CgkgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9ImZvcm1NZXNz
YWdlIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi10b3AiPjIwPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CgkgICAgICA8
L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a1NwaW5uZXIiIGlkPSJzcGlubmVyIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2lu
LXRvcCI+NDA8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5G
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJhY3RpdmUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtHcmlkIiBpZD0iZm9ybUdyaWQiPgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWJvdHRvbSI+MTA8L3Byb3BlcnR5PgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0icm93LXNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImNvbHVtbi1zcGFjaW5nIj42PC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPHN0eWxlPgogICAgICAgICAgPGNsYXNzIG5hbWU9InJlZ2lz
dHJhdGlvbiIvPgogICAgICAgICAgPGNsYXNzIG5hbWU9ImZvcm0iLz4KICAgICAgICA8L3N0eWxlPgog
ICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InRpdGxl
IiB0cmFuc2xhdGFibGU9InllcyI+Mi4gQ3JlYXRlIGFjY291bnQ8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwYWdlLXR5cGUiPkdUS19BU1NJU1RBTlRfUEFHRV9DT05GSVJNPC9wcm9wZXJ0
eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrQm94Ij4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJ
bWFnZSIgaWQ9ImRvbmVJbWFnZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fcmln
aHQiPjE1PC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgoJICAgIDwvY2hpbGQ+CgkgICAgPGNo
aWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJkb25lTWVzc2FnZSI+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPHN0eWxlPgogICAgICAgICAgPGNs
YXNzIG5hbWU9InJlZ2lzdHJhdGlvbiIvPgogICAgICAgIDwvc3R5bGU+CiAgICAgIDwvb2JqZWN0Pgog
ICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFnZS10eXBlIj5HVEtfQVNTSVNU
QU5UX1BBR0VfU1VNTUFSWTwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAg
ICA8c3R5bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9zdHlsZT4KICA8L29iamVj
dD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/AddContact.xml": {
		local:   "definitions/AddContact.xml",
		size:    9250,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtUZXh0QnVmZmVyIiBpZD0ic3Vic2NyaXB0aW9u
QXNrTWVzc2FnZSI+CiAgICA8cHJvcGVydHkgbmFtZT0idGV4dCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkkg
d291bGQgbGlrZSB0byBhZGQgeW91IHRvIG15IGNvbnRhY3QgbGlzdC48L3Byb3BlcnR5PgogIDwvb2Jq
ZWN0PgogIDxvYmplY3QgY2xhc3M9Ikd0a0RpYWxvZyIgaWQ9IkFkZENvbnRhY3QiPgogICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPmZhbHNlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aW5k
b3ctcG9zaXRpb24iPkdUS19XSU5fUE9TX0NFTlRFUjwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0iYm9yZGVyX3dpZHRoIj43PC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJh
bnNsYXRhYmxlPSJ5ZXMiPkFkZCBuZXcgY29udGFjdDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0icmVzaXphYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJkZWZhdWx0LWhl
aWdodCI+NDAwPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJkZWZhdWx0LXdpZHRoIj41MDA8
L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImRlc3Ryb3ktd2l0aC1wYXJlbnQiPnRydWU8L3By
b3BlcnR5PgogICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrQm94IiBpZD0iVmJveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRydWU8
L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJob21vZ2VuZW91cyI+ZmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX1ZF
UlRJQ0FMPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVy
dHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPCEtLVRPRE86IHRoaXMgc2VlbXMgdG8gcmVuZGVy
IHdlaXJkbHkgLS0+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJub3RpZmljYXRp
b24tYXJlYSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5mYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tdG9wIj41PC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1sZWZ0Ij41PC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1yaWdodCI+NTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX1ZFUlRJQ0FMPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAgIDxjbGFzcyBuYW1lPSJub3RpZmljYXRp
b25zIi8+CiAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2lu
Zz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0dyaWQiIGlkPSJncmlkIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXRvcCI+MTU8L3By
b3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWJvdHRvbSI+MTA8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXN0YXJ0Ij4xMDwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tZW5kIj4xMDwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyb3ctc3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iY29sdW1uLXNwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJhY2NvdW50c0xh
YmVsIiA+CgkJICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5BY2Nv
dW50OjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+R1RL
X0pVU1RJRllfUklHSFQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imhh
bGlnbiI+R1RLX0FMSUdOX0VORDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRh
Y2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2gi
PjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NvbWJvQm94
IiBpZD0iYWNjb3VudHMiPgoJCSAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imhhcy1mb2N1cyI+dHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0NlbGxSZW5kZXJlclRleHQiIGlkPSJhY2NvdW50LW5hbWUtcmVuZGVyZWQiLz4KICAgICAgICAg
ICAgICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ0
ZXh0Ij4wPC9hdHRyaWJ1dGU+CiAgICAgICAgICAgICAgICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tp
bmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjE8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0iYWNjb3VudExhYmVs
IiA+CgkJICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5Db250YWN0
IHRvIGFkZDo8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnki
PkdUS19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJoYWxpZ24iPkdUS19BTElHTl9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQt
YXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0
YWNoIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hp
bGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRy
eSIgaWQ9ImFkZHJlc3MiPgoJCSAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBsYWNlaG9sZGVyLXRleHQiPnNv
bWVvbmVAamFiYmVyLm9yZzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
aGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2
YXRlIiBoYW5kbGVyPSJvbl9zYXZlIiAvPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4x
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4yPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9Im5p
Y2tuYW1lTGFiZWwiID4KCQkgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5
ZXMiPk5pY2tuYW1lOjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVz
dGlmeSI+R1RLX0pVU1RJRllfUklHSFQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX0VORDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRv
cC1hdHRhY2giPjM8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
PC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0VudHJ5IiBpZD0ibmlja25hbWUiPgoJCSAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0i
b25fc2F2ZSIgLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MzwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+dHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj50cnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0NoZWNrQnV0dG9uIiBpZD0iYXV0b19hdXRob3JpemVfY2hlY2tidXR0b24i
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFfbGxvdyB0aGlz
IGNvbnRhY3QgdG8gdmlldyBteSBzdGF0dXM8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0idXNlX2FjdGlvbl9hcHBlYXJhbmNlIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibm9fc2hvd19hbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYWN0
aXZlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImRyYXdfaW5kaWNh
dG9yIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1Nj
cm9sbGVkV2luZG93IiBpZD0ibWVzc2FnZV9zY3JvbGxlZHdpbmRvdyI+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImNhbl9mb2N1cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJu
b19zaG93X2FsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3Jk
ZXJfd2lkdGgiPjY8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2hhZG93X3R5
cGUiPmV0Y2hlZC1pbjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtaW5fY29u
dGVudF9oZWlnaHQiPjU8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrVGV4dFZpZXciIGlkPSJtZXNzYWdlX3RleHR2aWV3Ij4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9IndyYXBfbW9kZSI+d29yZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iYnVmZmVyIj5zdWJzY3JpcHRpb25Bc2tNZXNzYWdlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MzwvcHJvcGVydHk+CiAg
ICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InN1YnNjcmlwdGlvbkluc3RydWN0aW9ucyI+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+V2FybmluZzogVGhpcyBt
ZXNzYWdlIHdpbGwgbm90IGJlIGVuZC10by1lbmQgZW5jcnlwdGVkLiBEbyBub3QgcmV2ZWFsIGFueSBz
ZW5zaXRpdmUgaW5mb3JtYXRpb24gaW4gaXQuPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9IndyYXAiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWF4
LXdpZHRoLWNoYXJzIj42NTwvcHJvcGVydHk+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2No
aWxkPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9
ImFjdGlvbl9hcmVhIj4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94IiBpZD0iYnV0dG9u
X2JveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRydWU8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX0hPUklaT05UQUw8
L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRv
biIgaWQ9ImJ1dHRvbl9jYW5jZWwiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCI+X0NhbmNlbDwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2UtdW5kZXJsaW5lIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9jYW5j
ZWwiIC8+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4K
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImJ1dHRvbl9vayI+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+QWRkPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InVzZS11bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWRlZmF1bHQiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX3NhdmUiIC8+CiAgICAgICAgICA8L29i
amVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8c3R5
bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9zdHlsZT4KICA8L29iamVjdD4KPC9p
bnRlcmZhY2U+Cg==
`,
	},

	"/definitions/AnswerSMPQuestion.xml": {
		local:   "definitions/AnswerSMPQuestion.xml",
		size:    6371,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRyeUJ1ZmZlciIgaWQ9InBpbl9lbnRyeV9i
dWZmZXIiPgogICAgPHNpZ25hbCBuYW1lPSJpbnNlcnRlZC10ZXh0IiBoYW5kbGVyPSJ0ZXh0X2NoYW5n
aW5nIi8+CiAgICA8c2lnbmFsIG5hbWU9ImRlbGV0ZWQtdGV4dCIgaGFuZGxlcj0idGV4dF9jaGFuZ2lu
ZyIvPgogIDwvb2JqZWN0PgogIDxvYmplY3QgY2xhc3M9Ikd0a0RpYWxvZyIgaWQ9ImRpYWxvZyI+CiAg
ICA8cHJvcGVydHkgbmFtZT0id2luZG93LXBvc2l0aW9uIj5HVEtfV0lOX1BPU19DRU5URVI8L3Byb3Bl
cnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlci13aWR0aCI+NzwvcHJvcGVydHk+CiAgICA8Y2hp
bGQgaW50ZXJuYWwtY2hpbGQ9InZib3giPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJi
b3giPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3JkZXItd2lkdGgiPjEwPC9wcm9wZXJ0eT4KICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9WRVJUSUNBTDwv
cHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBp
ZD0ibm90aWZpY2F0aW9uLWFyZWEiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRp
b24iPkdUS19PUklFTlRBVElPTl9WRVJUSUNBTDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJzcGFjaW5nIj4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxzdHlsZT4KICAgICAgICAg
ICAgICA8Y2xhc3MgbmFtZT0ibm90aWZpY2F0aW9ucyIvPgogICAgICAgICAgICA8L3N0eWxlPgogICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImV4cGFuZCI+ZmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwv
cHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICA8b2JqZWN0ICBjbGFzcz0iR3RrSW1hZ2UiIGlkPSJzbXBfaW1hZ2UiPgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2JvdHRvbSI+MTA8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3RvcCI+MTA8L3Byb3BlcnR5PgogICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtHcmlkIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdO
X0NFTlRFUjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InF1ZXN0aW9uX2Zyb21fcGVlciI+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIi8+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ibWFyZ2luLWJvdHRvbSI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9j
aGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtHcmlkIj4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX0NFTlRFUjwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0ICBjbGFzcz0iR3RrTGFiZWwi
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+
QW5zd2VyOiA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnki
PkdUS19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFj
aCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+
MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id2lkdGgiPjI8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgIGNsYXNzPSJHdGtFbnRyeSIgaWQ9ImFuc3dl
ciI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYnVmZmVyIj5waW5fZW50cnlfYnVmZmVy
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tYm90dG9tIj4x
MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MDwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxl
PSJ5ZXMiPllvdXIgcGVlciBtdXN0IGhhdmUgcHJldmlvdXNseSBzaGFyZWQgaXQgd2l0aCB5b3UuIElm
IG5vdCwgdHJ5IHRvOjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4t
dG9wIj41PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1ib3R0b20i
PjEwPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
PGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrR3JpZCI+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJjb2x1bW4tc3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJyb3ctc3BhY2luZyI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9InBhZGxvY2tfaW1hZ2UxIi8+CiAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRh
Y2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2gi
PjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIj4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFzayB0
aGVtIGluIHBlcnNvbjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVz
dGlmeSI+R1RLX0pVU1RJRllfTEVGVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0
b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtJbWFnZSIgaWQ9InBhZGxvY2tfaW1hZ2UyIi8+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFzayB0aGVtIGluIGEgZGlmZmVyZW50IGVuY3J5
cHRlZCBjaGFubmVsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0
aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJoYWxpZ24iPkdUS19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGVmdC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRv
cC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
PC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0ltYWdlIiBpZD0iYWxlcnRfaW1hZ2UiLz4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+QXNrIHRoZW0gb3ZlciB0aGUgcGhvbmU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdO
X1NUQVJUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MjwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlv
bl9hcmVhIj4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCIgaWQ9ImJ1dHRvbl9i
b3giPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElP
Tl9IT1JJWk9OVEFMPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImJ1dHRvbl9zdWJtaXQiPgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+VmVyaWZ5IEFuc3dlcjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9ImNsb3Nl
X3NoYXJlX3BpbiIvPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2No
aWxkPgogICAgPGFjdGlvbi13aWRnZXRzPgogICAgICA8YWN0aW9uLXdpZGdldCByZXNwb25zZT0ib2si
PmJ1dHRvbl9zdWJtaXQ8L2FjdGlvbi13aWRnZXQ+CiAgICA8L2FjdGlvbi13aWRnZXRzPgogICAgPHN0
eWxlPgogICAgICA8Y2xhc3MgbmFtZT0iY295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+Cjwv
aW50ZXJmYWNlPgo=
`,
	},

	"/definitions/AskForPassword.xml": {
		local:   "definitions/AskForPassword.xml",
		size:    4811,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJBc2tGb3JQYXNzd29yZCI+
CiAgICA8cHJvcGVydHkgbmFtZT0id2luZG93LXBvc2l0aW9uIj5HVEtfV0lOX1BPU19DRU5URVI8L3By
b3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+RW50ZXIg
eW91ciBwYXNzd29yZDwvcHJvcGVydHk+CiAgICA8c2lnbmFsIG5hbWU9ImRlbGV0ZS1ldmVudCIgaGFu
ZGxlcj0ib25fY2FuY2VsX3Bhc3N3b3JkIiAvPgogICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94
Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iVmJveCI+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9Im1hcmdpbiI+MTA8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ibWVzc2FnZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9IndyYXAiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibWF4LXdpZHRoLWNoYXJzIj41MDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJzZWxlY3RhYmxlIj5UUlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im1hcmdpbi1ib3R0b20iPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrR3JpZCIgaWQ9
ImdyaWQiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWJvdHRvbSI+MTA8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93LXNwYWNpbmciPjEyPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNvbHVtbi1zcGFjaW5nIj42PC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0i
YWNjb3VudE1lc3NhZ2UiID4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJh
bnNsYXRhYmxlPSJ5ZXMiPkFjY291bnQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0id2VpZ2h0
IiB2YWx1ZT0iYm9sZCIvPgogICAgICAgICAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtMYWJlbCIgaWQ9ImFjY291bnROYW1lIiA+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0
b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtMYWJlbCIgaWQ9InBhc3NNZXNzYWdlIiA+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5QYXNzd29yZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+R1RLX0pVU1RJRllfUklHSFQ8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX0VORDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwv
cGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0VudHJ5IiBpZD0icGFzc3dvcmQiPgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9Imhhcy1mb2N1cyI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJpbGl0eSI+ZmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25fZW50ZXJlZF9wYXNzd29yZCIgLz4KICAg
ICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJzYXZlUGFzc3dvcmRMYWJlbCI+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5TYXZlIHBhc3N3b3Jk
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVT
VElGWV9SSUdIVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWdu
Ij5HVEtfQUxJR05fRU5EPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+
MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+Mjwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2hlY2tCdXR0b24i
IGlkPSJzYXZlUGFzc3dvcmQiLz4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGls
ZD4KICAgICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCIgaWQ9ImJ1dHRvbl9ib3giPgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9IT1JJWk9OVEFMPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRv
biIgaWQ9ImJ1dHRvbl9jYW5jZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVs
Ij5DYW5jZWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBo
YW5kbGVyPSJvbl9jYW5jZWxfcGFzc3dvcmQiIC8+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtCdXR0b24iIGlkPSJidXR0b25fb2siPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Q29ubmVjdDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWRlZmF1bHQiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9lbnRlcmVkX3Bhc3N3b3JkIiAv
PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPHN0
eWxlPgogICAgICA8Y2xhc3MgbmFtZT0iY295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+Cjwv
aW50ZXJmYWNlPgo=
`,
	},

	"/definitions/AskToEncrypt.xml": {
		local:   "definitions/AskToEncrypt.xml",
		size:    725,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtNZXNzYWdlRGlhbG9nIiBpZD0iQXNrVG9FbmNy
eXB0Ij4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aW5kb3ctcG9zaXRpb24iPkdUS19XSU5fUE9TX0NFTlRF
UjwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0ibW9kYWwiPnRydWU8L3Byb3BlcnR5PgogICAg
PHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+NzwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5FbmNyeXB0IHlvdXIgYWNjb3VudCdzIGluZm9ybWF0
aW9uPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0ZXh0IiB0cmFuc2xhdGFibGU9InllcyI+
CldvdWxkIHlvdSBsaWtlIHRvIGVuY3J5cHQgeW91ciBhY2NvdW50J3MgaW5mb3JtYXRpb24/CiAgICA8
L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InNlY29uZGFyeV90ZXh0IiB0cmFuc2xhdGFibGU9
InllcyI+WW91IHdpbGwgbm90IGJlIGFibGUgdG8gYWNjZXNzIHlvdXIgYWNjb3VudCdzIGluZm9ybWF0
aW9uIGZpbGUgaWYgeW91IGxvc2UgdGhlCgpwYXNzd29yZC4gWW91IHdpbGwgb25seSBiZSBhc2tlZCBm
b3IgaXQgd2hlbiBDb3lJTSBzdGFydHMuCiAgICA8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9
ImJ1dHRvbnMiPkdUS19CVVRUT05TX1lFU19OTzwvcHJvcGVydHk+CiAgPC9vYmplY3Q+CjwvaW50ZXJm
YWNlPgo=
`,
	},

	"/definitions/AuthorizeSubscription.xml": {
		local:   "definitions/AuthorizeSubscription.xml",
		size:    355,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtNZXNzYWdlRGlhbG9nIiBpZD0iZGlhbG9nIj4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlN1YnNjcmlwdGlvbiBy
ZXF1ZXN0PC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aW5kb3ctcG9zaXRpb24iPkdUS19X
SU5fUE9TX0NFTlRFUjwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0ibWVzc2FnZS10eXBlIj5H
VEtfTUVTU0FHRV9RVUVTVElPTjwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0iYnV0dG9ucyI+
R1RLX0JVVFRPTlNfWUVTX05PPC9wcm9wZXJ0eT4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/BackgroundColorDetectionInvisibleListBox.xml": {
		local:   "definitions/BackgroundColorDetectionInvisibleListBox.xml",
		size:    255,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICAgIDxyZXF1aXJlcyBsaWI9Imd0aysiIHZlcnNpb249IjMuMjAiLz4KICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0xpc3RCb3giIGlkPSJiZy1jb2xvci1kZXRlY3Rpb24taW52aXNpYmxlLWxp
c3Rib3giPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICA8L29iamVj
dD4KPC9pbnRlcmZhY2U+
`,
	},

	"/definitions/CannotVerifyWithSMP.xml": {
		local:   "definitions/CannotVerifyWithSMP.xml",
		size:    2134,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJzbXAtZXJyb3ItaW5mb2JhciI+
CiAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICA8cHJvcGVy
dHkgbmFtZT0ic3BhY2luZyI+Mjc8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtJbWFnZSIgaWQ9InNtcC1lcnJvci1pbWFnZSI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMi
PkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2xlZnQiPjEwPC9w
cm9wZXJ0eT4KICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0
eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrTGFiZWwiIGlkPSJzbXAtZXJyb3ItbGFiZWwiPgogICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3Vz
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3RhcnQ8L3By
b3BlcnR5PgogICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ3ZWln
aHQiIHZhbHVlPSJib2xkIi8+CiAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICA8L29iamVjdD4KICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2No
aWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJzbXAtZXJy
b3ItYnV0dG9uIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
PHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJtYXJnaW5fcmlnaHQiPjEwPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibWFyZ2luX3RvcCI+MTQ8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJn
aW5fYm90dG9tIj4xMjwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrTGFiZWwiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9sZWZ0Ij4xNTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fcmlnaHQiPjE1PC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+VHJ5IGFnYWlu
IGxhdGVyPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAg
IDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5G
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwYWNrX3R5cGUiPmVuZDwvcHJvcGVydHk+CiAgICAgICAg
PHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAg
PC9jaGlsZD4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/CaptureInitialMainPassword.xml": {
		local:   "definitions/CaptureInitialMainPassword.xml",
		size:    2787,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJDYXB0dXJlSW5pdGlhbE1h
aW5QYXNzd29yZCI+CiAgICA8cHJvcGVydHkgbmFtZT0id2luZG93LXBvc2l0aW9uIj5HVEtfV0lOX1BP
U19DRU5URVI8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+NzwvcHJv
cGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5Db25maWd1
cmUgeW91ciBtYWluIHBhc3N3b3JkPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJkZWZhdWx0
LXdpZHRoIj4zMDA8L3Byb3BlcnR5PgogICAgPHNpZ25hbCBuYW1lPSJjbG9zZSIgaGFuZGxlcj0ib25f
Y2FuY2VsIiAvPgogICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrQm94IiBpZD0iVmJveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbiI+MTA8
L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMDwvcHJvcGVydHk+CiAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImhvbW9nZW5lb3VzIj5mYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
PHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fVkVSVElDQUw8L3Byb3Bl
cnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0i
cGFzc01lc3NhZ2UiID4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFi
bGU9InllcyI+UGxlYXNlIGVudGVyIHRoZSBtYWluIHBhc3N3b3JkIGZvciB0aGUgY29uZmlndXJhdGlv
biBmaWxlIHR3aWNlLjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj50
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPmZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImZpbGwiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrRW50cnkiIGlk
PSJwYXNzd29yZCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYXMtZm9jdXMiPnRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJpbGl0eSI+ZmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9zYXZlIiAv
PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImV4cGFuZCI+ZmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZmlsbCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MTwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRyeSIgaWQ9InBhc3N3b3JkMiI+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmlsaXR5Ij5mYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUiIGhhbmRsZXI9Im9uX3NhdmUiIC8+CiAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZXhwYW5kIj5mYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9w
cm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVj
dD4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94IiBpZD0iYnV0dG9uX2JveCI+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRBTDwvcHJvcGVy
dHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0i
Y2FuY2VsIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9Inll
cyI+Q2FuY2VsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5k
bGVyPSJvbl9jYW5jZWwiIC8+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAg
ICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9InNhdmUiPgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5PSzwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fc2F2ZSIg
Lz4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAg
PC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAgICA8L3N0eWxl
PgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/CertificateDialog.xml": {
		local:   "definitions/CertificateDialog.xml",
		size:    21582,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkaWFsb2ciPgogICAgPHBy
b3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+UHJvYmxlbSB3aXRoIHNlY3VyZSBj
b25uZWN0aW9uIGZvciBBQ0NPVU5UX05BTUU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im1v
ZGFsIj50cnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij40ODA8
L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTA8L3Byb3BlcnR5Pgog
ICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94
IiBpZD0iYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyLXdpZHRoIj4xMDwvcHJvcGVy
dHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhvbW9nZW5lb3VzIj5mYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fVkVSVElDQUw8
L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVs
IiBpZD0ibWVzc2FnZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRh
YmxlPSJ5ZXMiPjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdU
S19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5
Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Indy
YXAiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWF4LXdpZHRoLWNo
YXJzIj44MDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5U
UlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1ib3R0b20iPjEw
PC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNo
aWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJpc3N1ZWRUb01lc3NhZ2Ui
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5Jc3N1
ZWQgdG86PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FM
SUdOX1NUQVJUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdU
S19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0
YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAgICAg
IDxhdHRyaWJ1dGUgbmFtZT0id2VpZ2h0IiB2YWx1ZT0iUEFOR09fV0VJR0hUX0JPTEQiLz4KICAgICAg
ICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtHcmlkIiBpZD0iaXNzdWVkVG9H
cmlkIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi10b3AiPjI8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWJvdHRvbSI+MTA8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXN0YXJ0Ij4xMDwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tZW5kIj4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJyb3ctc3BhY2luZyI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJjb2x1bW4tc3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJpc3N1ZWRUb0NuTGFiZWwiPgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Q29t
bW9uIE5hbWUgKENOKTo8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imhh
bGlnbiI+R1RLX0FMSUdOX1NUQVJUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UUlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dG9wLWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAg
ICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTGFiZWwiIGlkPSJpc3N1ZWRUb0NuVmFsdWUiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+R1RLX0pVU1RJRllfTEVGVDwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ic25pc0xhYmVsIj4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPk90aGVyIG5hbWVzOjwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+R1RLX0FMSUdO
X1NUQVJUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5H
VEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJz
ZWxlY3RhYmxlIj5UUlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+
MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJz
bmlzVmFsdWUiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFi
bGU9InllcyI+PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24i
PkdUS19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
anVzdGlmeSI+R1RLX0pVU1RJRllfTEVGVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ic2VsZWN0YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0id3JhcCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRh
Y2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2gi
PjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBp
ZD0iaXNzdWVkVG9PTGFiZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0
cmFuc2xhdGFibGU9InllcyI+T3JnYW5pemF0aW9uIChPKTo8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX1NUQVJUPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UUlVFPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJpc3N1ZWRUb09WYWx1ZSI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj48L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX1NUQVJUPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElG
WV9MRUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxl
Ij5UUlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MjwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJpc3N1ZWRUb09V
TGFiZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9
InllcyI+T3JnYW5pemF0aW9uYWwgVW5pdCAoT1UpOjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3Bh
Y2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9Imlzc3VlZFRvT1VWYWx1ZSI+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj48L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX1NUQVJUPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9M
RUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5U
UlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MzwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJTTkxhYmVsIj4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNlcmlh
bCBOdW1iZXI6PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24i
PkdUS19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
anVzdGlmeSI+R1RLX0pVU1RJRllfTEVGVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ic2VsZWN0YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVm
dC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1h
dHRhY2giPjQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xh
YmVsIiBpZD0iU05WYWx1ZSI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRy
YW5zbGF0YWJsZT0ieWVzIj48L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImhhbGlnbiI+R1RLX0FMSUdOX1NUQVJUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UUlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idG9wLWF0dGFjaCI+NDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAg
ICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9Imlzc3VlZEJ5TWVzc2Fn
ZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPklz
c3VlZCBieTo8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtf
QUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+
R1RLX0pVU1RJRllfTEVGVDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxl
Y3RhYmxlIj5UUlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAg
ICAgPGF0dHJpYnV0ZSBuYW1lPSJ3ZWlnaHQiIHZhbHVlPSJQQU5HT19XRUlHSFRfQk9MRCIvPgogICAg
ICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgog
ICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0dyaWQiIGlkPSJpc3N1ZWRC
eUdyaWQiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXRvcCI+MjwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tYm90dG9tIj4xMDwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tc3RhcnQiPjEwPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1lbmQiPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InJvdy1zcGFjaW5nIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImNvbHVtbi1zcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4K
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9Imlzc3VlZEJ5Q25MYWJlbCI+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5D
b21tb24gTmFtZSAoQ04pOjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
aGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9Imlzc3VlZEJ5Q25WYWx1ZSI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj48L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX1NUQVJUPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UUlVFPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJpc3N1ZWRCeU9MYWJlbCI+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5Pcmdhbml6YXRpb24g
KE8pOjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtf
QUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3Rp
ZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0
YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNo
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIg
aWQ9Imlzc3VlZEJ5T1ZhbHVlIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIg
dHJhbnNsYXRhYmxlPSJ5ZXMiPjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ0b3AtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtMYWJlbCIgaWQ9Imlzc3VlZEJ5T1VMYWJlbCI+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5Pcmdhbml6YXRpb25hbCBVbml0IChPVSk6
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElH
Tl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+
R1RLX0pVU1RJRllfTEVGVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
c2VsZWN0YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2gi
PjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjI8
L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0i
aXNzdWVkQnlPVVZhbHVlIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJh
bnNsYXRhYmxlPSJ5ZXMiPjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
aGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ0b3AtYXR0YWNoIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxj
aGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0idmFsaWRpdHlNZXNzYWdl
Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+VmFs
aWRpdHk6PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FM
SUdOX1NUQVJUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdU
S19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0
YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAgICAg
IDxhdHRyaWJ1dGUgbmFtZT0id2VpZ2h0IiB2YWx1ZT0iUEFOR09fV0VJR0hUX0JPTEQiLz4KICAgICAg
ICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtHcmlkIiBpZD0idmFsaWRpdHlH
cmlkIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi10b3AiPjI8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWJvdHRvbSI+MTA8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXN0YXJ0Ij4xMDwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tZW5kIj4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJyb3ctc3BhY2luZyI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJjb2x1bW4tc3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJpc3N1ZWRPbkxhYmVsIj4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPklzc3Vl
ZCBvbjo8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RL
X0FMSUdOX1NUQVJUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0
aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJzZWxlY3RhYmxlIj5UUlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0
dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFj
aCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwi
IGlkPSJpc3N1ZWRPblZhbHVlIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIg
dHJhbnNsYXRhYmxlPSJ5ZXMiPjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtMYWJlbCIgaWQ9ImV4cGlyZXNPbkxhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkV4cGlyZXMgb246PC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9TVEFSVDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+R1RLX0pVU1RJRllfTEVG
VDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0YWJsZSI+VFJV
RTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxk
PgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0iZXhwaXJlc09uVmFsdWUi
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElH
Tl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+
R1RLX0pVU1RJRllfTEVGVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
c2VsZWN0YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2gi
PjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjE8
L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrR3JpZCIgaWQ9ImZpbmdlcnByaW50c0dyaWQiPgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ibWFyZ2luLXRvcCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJtYXJnaW4tYm90dG9tIj4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJtYXJnaW4tc3RhcnQiPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im1hcmdpbi1lbmQiPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJvdy1z
cGFjaW5nIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNvbHVtbi1zcGFj
aW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InNoYTFGaW5nZXJwcmludExhYmVsIj4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNIQS0xIGZpbmdlcnByaW50
OjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJ
R05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnki
PkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNo
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9
InNoYTFGaW5nZXJwcmludFZhbHVlIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJl
bCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0iZm9udC1k
ZXNjIiB2YWx1ZT0iTW9ub3NwYWNlIi8+CiAgICAgICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAg
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2lu
Zz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ic2hhMjU2RmluZ2VycHJpbnRMYWJlbCI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5TSEEtMjU2IGZp
bmdlcnByaW50OjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWdu
Ij5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imxl
ZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3At
YXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtM
YWJlbCIgaWQ9InNoYTI1NkZpbmdlcnByaW50VmFsdWUiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+R1RLX0pVU1RJRllfTEVGVDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0YWJsZSI+VFJVRTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8YXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBu
YW1lPSJmb250LWRlc2MiIHZhbHVlPSJNb25vc3BhY2UiLz4KICAgICAgICAgICAgICAgIDwvYXR0cmli
dXRlcz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJzaGEzXzI1NkZpbmdlcnByaW50TGFi
ZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9Inll
cyI+U0hBMy0yNTYgZmluZ2VycHJpbnQ6PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+R1RLX0pVU1RJRllfTEVGVDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InRvcC1hdHRhY2giPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4K
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ic2hhM18yNTZGaW5nZXJwcmludFZhbHVlIj4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPjwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZ
X0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUi
PlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAg
ICAgIDxhdHRyaWJ1dGUgbmFtZT0iZm9udC1kZXNjIiB2YWx1ZT0iTW9ub3NwYWNlIi8+CiAgICAgICAg
ICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
PHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjI8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJidXR0b25JbnN0cnVjdGlvbnNNZXNzYWdlIj4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+RGVwZW5kaW5nIG9uIHdo
ZXRoZXIgeW91IGp1ZGdlIHRoZSBhYm92ZSBpbmZvcm1hdGlvbiBjb3JyZWN0LCBvciBob3cgbXVjaCBy
aXNrIHlvdSBhcmUgd2lsbGluZyB0byB0YWtlLCB5b3UgaGF2ZSB0aHJlZSBjaG9pY2VzOiBEZWNpZGUg
dG8gbm90IGNvbm5lY3QsIGNvbm5lY3QgdXNpbmcgdGhpcyBpbmZvcm1hdGlvbiB0aGlzIHRpbWUgYnV0
IHJlcXVlc3QgdGhhdCB3ZSBhc2sgYWdhaW4gbmV4dCB0aW1lLCBvciBjb25uZWN0IHVzaW5nIHRoaXMg
aW5mb3JtYXRpb24gYW5kIHJlbWVtYmVyIGl0IGZvciB0aGUgZnV0dXJlLjwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJtYXgtd2lkdGgtY2hhcnMiPjgwPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9Im1hcmdpbi1ib3R0b20iPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJhY3Rpb25f
YXJlYSI+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJidXR0b25fYm94
Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05f
SE9SSVpPTlRBTDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJidXR0b25fZG9udCI+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iY2FuLWRlZmF1bHQiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+RG9uJ3QgY29ubmVjdDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJidXR0b25f
dGVtcF9vayI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJs
ZT0ieWVzIj5BY2NlcHQgdGVtcG9yYXJpbHk8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iYnV0dG9uX3Blcm1fb2siPgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+QWNjZXB0IHBlcm1hbmVudGx5
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGls
ZD4KICAgIDxhY3Rpb24td2lkZ2V0cz4KICAgICAgPGFjdGlvbi13aWRnZXQgcmVzcG9uc2U9ImNhbmNl
bCIgZGVmYXVsdD0idHJ1ZSI+YnV0dG9uX2RvbnQ8L2FjdGlvbi13aWRnZXQ+CiAgICAgIDxhY3Rpb24t
d2lkZ2V0IHJlc3BvbnNlPSJvayI+YnV0dG9uX3RlbXBfb2s8L2FjdGlvbi13aWRnZXQ+CiAgICAgIDxh
Y3Rpb24td2lkZ2V0IHJlc3BvbnNlPSJhY2NlcHQiPmJ1dHRvbl9wZXJtX29rPC9hY3Rpb24td2lkZ2V0
PgogICAgPC9hY3Rpb24td2lkZ2V0cz4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWlt
Ii8+CiAgICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/ChangePassword.xml": {
		local:   "definitions/ChangePassword.xml",
		size:    11268,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJDaGFuZ2VQYXNzd29yZCI+
CiAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICA8cHJvcGVy
dHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5DaGFuZ2UgYWNjb3VudCBwYXNzd29yZDwv
cHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0id2luZG93X3Bvc2l0aW9uIj5jZW50ZXI8L3Byb3Bl
cnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImRlc3Ryb3lfd2l0aF9wYXJlbnQiPlRydWU8L3Byb3BlcnR5
PgogICAgPHByb3BlcnR5IG5hbWU9InR5cGVfaGludCI+bm9ybWFsPC9wcm9wZXJ0eT4KICAgIDxwcm9w
ZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgiPjc8L3Byb3BlcnR5PgogICAgPHNpZ25hbCBuYW1lPSJkZWxl
dGUtZXZlbnQiIGhhbmRsZXI9Im9uX2NhbmNlbF9wYXNzd29yZCIgc3dhcHBlZD0ibm8iLz4KICAgIDxj
aGlsZCBpbnRlcm5hbC1jaGlsZD0idmJveCI+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9
IlZib3giPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9WRVJUSUNB
TDwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJhY3Rpb25fYXJlYSI+CiAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJidXR0b24tYm94Ij4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRB
TDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCdXR0b24iIGlkPSJidXR0b24tY2FuY2VsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsYWJlbCI+Q2FuY2VsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJyZWNlaXZlc19kZWZhdWx0Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8c2lnbmFs
IG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX2NhbmNlbCIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCdXR0b24iIGlkPSJidXR0b24tY2hhbmdlIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNoYW5nZSBQYXNzd29yZDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9jaGFuZ2UiIHN3YXBw
ZWQ9Im5vIi8+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iYnV0dG9uLW9rIj4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPk9rPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9vayIgc3dhcHBlZD0ibm8i
Lz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+
CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5n
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAg
ICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtC
b3giIGlkPSJmb3JtLWJveCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0dyaWQiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl90b3AiPjEwPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyb3dfc3BhY2luZyI+NjwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY29sdW1uX3NwYWNpbmciPjY8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5l
bmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJh
bnNsYXRhYmxlPSJ5ZXMiPkVudGVyIG5ldyBwYXNzd29yZDo8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5yaWdodDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdF9hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxj
aGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrRW50cnkiIGlkPSJuZXctcGFz
c3dvcmQtZW50cnkiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3Vz
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJp
bGl0eSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJp
bnZpc2libGVfY2hhciI+KjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGVmdF9hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ0b3BfYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5Db25maXJtIG5ldyBwYXNzd29yZDo8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5yaWdodDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdF9hdHRhY2giPjA8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4xPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrRW50cnkiIGlkPSJyZXBlYXQtcGFzc3dvcmQtZW50cnkiPgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmlsaXR5Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImludmlzaWJsZV9jaGFyIj4qPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcF9hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InNhdmUt
bmV3LXBhc3N3b3JkLWxhYmVsIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFi
ZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5TYXZlIG5ldyBwYXNzd29yZCA/PC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+R1RLX0pVU1RJRllfUklHSFQ8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElH
Tl9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNo
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFj
aCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a0NoZWNrQnV0dG9uIiBpZD0ic2F2ZS1uZXctcGFzc3dvcmQtY2hlY2tib3giLz4KICAg
ICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idG9wLWF0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
PHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxk
PgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9Im1lc3NhZ2VzLWJveCI+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4zPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fdG9wIj4xMDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2JvdHRvbSI+MTA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIg
aWQ9ImZvcm0taW1hZ2UiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2Zv
Y3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1h
cmdpbl9sZWZ0Ij41MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhw
YW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3Np
dGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a1NwaW5uZXIiIGlkPSJjaGFuZ2UtcGFzc3dvcmQtc3Bpbm5lciI+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3JpZ2h0Ij40PC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+RmFsc2U8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0i
Zm9ybS1ib3gtbGFiZWwiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2Zv
Y3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imhh
bGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5sYWJlbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPnJpZ2h0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3Bh
Y2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj40PC9wcm9w
ZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4K
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0dyaWQiIGlkPSJjYWxsYmFjay1ncmlkIj4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgiPjM8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJjYWxsYmFjay1sYWJlbCI+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idG9wX2F0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrSW1hZ2UiIGlkPSJjYWxsYmFjay1pbWFnZSI+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibWFyZ2luX3JpZ2h0Ij4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGVmdF9hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRv
cF9hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
PC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cG9zaXRpb24iPjQ8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+
CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9
ImNveWltIi8+CiAgICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/ChangePasswordMessageResponse.xml": {
		local:   "definitions/ChangePasswordMessageResponse.xml",
		size:    3625,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJDaGFuZ2VQYXNzd29yZE1l
c3NhZ2VSZXNwb25zZSI+CiAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5DaGFuZ2UgcGFz
c3dvcmQ8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im1vZGFsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aW5kb3dfcG9zaXRpb24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICA8
cHJvcGVydHkgbmFtZT0iZGVzdHJveV93aXRoX3BhcmVudCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJv
cGVydHkgbmFtZT0idHlwZV9oaW50Ij5ub3JtYWw8L3Byb3BlcnR5PgogICAgPHNpZ25hbCBuYW1lPSJk
ZWxldGUtZXZlbnQiIGhhbmRsZXI9Im9uX2NhbmNlbF9wYXNzd29yZCIgc3dhcHBlZD0ibm8iLz4KICAg
IDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0idmJveCI+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIg
aWQ9IlZib3giPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0
eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2xlZnQiPjIwPC9wcm9wZXJ0eT4KICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3JpZ2h0Ij4yMDwvcHJvcGVydHk+CiAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Im1hcmdpbl90b3AiPjE1PC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bWFyZ2luX2JvdHRvbSI+MTU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRh
dGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj42
PC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCIgaWQ9ImJ1dHRvbl9ib3giPgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJidXR0b25f
b2siPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9Inll
cyI+T2s8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9kZWZhdWx0
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19k
ZWZhdWx0Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNr
ZWQiIGhhbmRsZXI9Im9uX29rX2NoYW5nZSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtHcmlkIiBpZD0iZ3JpZCI+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9ib3R0b20iPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InJvd19zcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJjb2x1bW5fc3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4K
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9ImNoYW5nZVBhc3N3b3JkTWVz
c2FnZUxhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5yaWdodDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8YXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgPGF0dHJp
YnV0ZSBuYW1lPSJmb3JlZ3JvdW5kIiB2YWx1ZT0iI2NjY2MwMDAwMDAwMCIvPgogICAgICAgICAgICAg
ICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnRfYXR0YWNoIj4wPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4wPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4K
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0iY2hhbmdlUGFzc3dvcmRNZXNzYWdlIj4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4K
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwv
Y2hpbGQ+CiAgICA8c3R5bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9zdHlsZT4K
ICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/ChooseKeyToImport.xml": {
		local:   "definitions/ChooseKeyToImport.xml",
		size:    2071,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkaWFsb2ciPgogICAgPHBy
b3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+Q2hvb3NlIGEga2V5IHRvIGltcG9y
dDwvcHJvcGVydHk+CiAgICA8c2lnbmFsIG5hbWU9ImNsb3NlIiBoYW5kbGVyPSJvbl9jYW5jZWwiIC8+
CiAgICA8c2lnbmFsIG5hbWU9ImRlbGV0ZS1ldmVudCIgaGFuZGxlcj0ib25fY2FuY2VsIiAvPgogICAg
PGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBp
ZD0iVmJveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbiI+MTA8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMDwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJtZXNzYWdlIj4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+VGhlIGZpbGUgeW91IGNob3Nl
IGNvbnRhaW5zIG1vcmUgdGhhbiBvbmUgcHJpdmF0ZSBrZXkuIENob29zZSBmcm9tIHRoZSBsaXN0IGJl
bG93IHRoZSBrZXkgeW91IHdvdWxkIGxpa2UgdG8gaW1wb3J0LjwvcHJvcGVydHk+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPmZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPnRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ29tYm9Cb3hUZXh0IiBpZD0ia2V5cyI+CiAgICAgICAgICA8
L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0iYWN0
aW9uX2FyZWEiPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94IiBpZD0iYmJveCI+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX0hP
UklaT05UQUw8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iYnRuLWNhbmNlbCI+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5DYW5jZWw8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9jYW5jZWwiIC8+CiAgICAg
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4K
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJidG4taW1wb3J0Ij4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkltcG9y
dDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWRlZmF1bHQiPnRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVy
PSJvbl9pbXBvcnQiIC8+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwv
Y2hpbGQ+CiAgICA8YWN0aW9uLXdpZGdldHM+CiAgICAgIDxhY3Rpb24td2lkZ2V0IHJlc3BvbnNlPSJj
YW5jZWwiPmJ0bi1jYW5jZWw8L2FjdGlvbi13aWRnZXQ+CiAgICAgIDxhY3Rpb24td2lkZ2V0IHJlc3Bv
bnNlPSJvayIgZGVmYXVsdD0idHJ1ZSI+YnRuLWltcG9ydDwvYWN0aW9uLXdpZGdldD4KICAgIDwvYWN0
aW9uLXdpZGdldHM+CiAgICA8c3R5bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9z
dHlsZT4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/ConfirmAccountRemoval.xml": {
		local:   "definitions/ConfirmAccountRemoval.xml",
		size:    557,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtNZXNzYWdlRGlhbG9nIiBpZD0iUmVtb3ZlQWNj
b3VudCI+CiAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5Db25maXJt
IGFjY291bnQgcmVtb3ZhbDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRo
Ij43PC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0ZXh0IiB0cmFuc2xhdGFibGU9InllcyI+
QXJlIHlvdSBzdXJlIHlvdSB3YW50IHRvIHJlbW92ZSB0aGlzIGFjY291bnQ/PC9wcm9wZXJ0eT4KICAg
IDxwcm9wZXJ0eSBuYW1lPSJ3aW5kb3ctcG9zaXRpb24iPkdUS19XSU5fUE9TX0NFTlRFUjwvcHJvcGVy
dHk+CiAgICA8cHJvcGVydHkgbmFtZT0ibW9kYWwiPnRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5
IG5hbWU9Im1lc3NhZ2UtdHlwZSI+R1RLX01FU1NBR0VfUVVFU1RJT048L3Byb3BlcnR5PgogICAgPHBy
b3BlcnR5IG5hbWU9ImJ1dHRvbnMiPkdUS19CVVRUT05TX1lFU19OTzwvcHJvcGVydHk+CiAgPC9vYmpl
Y3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/ConnectingAccountInfo.xml": {
		local:   "definitions/ConnectingAccountInfo.xml",
		size:    873,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtJbmZvQmFyIiBpZD0iaW5mb2JhciI+CiAgICA8
cHJvcGVydHkgbmFtZT0ibWVzc2FnZS10eXBlIj5HVEtfTUVTU0FHRV9JTkZPPC9wcm9wZXJ0eT4KICAg
IDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0iY29udGVudF9hcmVhIj4KICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrQm94IiBpZD0iYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iaG9tb2dlbmVvdXMiPmZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRB
VElPTl9IT1JJWk9OVEFMPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtMYWJlbCIgaWQ9Im1lc3NhZ2UiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZWxsaXBzaXplIj5QQU5HT19FTExJUFNJWkVfRU5EPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9IndyYXAiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
PC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtTcGlubmVy
IiBpZD0ic3Bpbm5lciI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJhY3RpdmUiPnRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fRU5EPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPnRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8
L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/ConnectionFailureNotification.xml": {
		local:   "definitions/ConnectionFailureNotification.xml",
		size:    3100,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtJbmZvQmFyIiBpZD0iaW5mb2JhciI+CiAgICA8
cHJvcGVydHkgbmFtZT0ibWVzc2FnZS10eXBlIj5HVEtfTUVTU0FHRV9FUlJPUjwvcHJvcGVydHk+CiAg
ICA8cHJvcGVydHkgbmFtZT0ic2hvdy1jbG9zZS1idXR0b24iPmZhbHNlPC9wcm9wZXJ0eT4KICAgIDxj
aGlsZCBpbnRlcm5hbC1jaGlsZD0iY29udGVudF9hcmVhIj4KICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
Qm94IiBpZD0iYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MDwvcHJvcGVydHk+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhvbW9nZW5lb3VzIj5mYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRBTDwv
cHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tc3RhcnQiPjA8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJtYXJnaW4tZW5kIj4wPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9Im1lc3NhZ2UiPgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ic2VsZWN0YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJlbGxpcHNpemUiPlBBTkdPX0VMTElQU0laRV9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ3cmFwIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im1hcmdpbi1yaWdodCI+MTA8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAg
ICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJidXR0b25fbW9yZV9pbmZvIj4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImhleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJl
bGllZiI+bm9uZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNl
bnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgiPjA8
L3Byb3BlcnR5PgogICAgICAgICAgICA8c3R5bGU+CiAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9InJh
aXNlZCIvPgogICAgICAgICAgICAgIDxjbGFzcyBuYW1lPSJjbG9zZSIvPgogICAgICAgICAgICA8L3N0
eWxlPgogICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX21vcmVfaW5m
byIgLz4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0lt
YWdlIiBpZD0ibW9yZS1pbmZvLWJ0biI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNv
bi1uYW1lIj5oZWxwLWZhcTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAg
ICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtC
dXR0b24iIGlkPSJidXR0b25fY2xvc2UiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2Zv
Y3VzIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+
MjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWxpZWYiPm5vbmU8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8c3R5bGU+CiAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9InJhaXNlZCIvPgogICAg
ICAgICAgICAgIDxjbGFzcyBuYW1lPSJjbG9zZSIvPgogICAgICAgICAgICA8L3N0eWxlPgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iaGFsaWduIj5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNs
aWNrZWQiIGhhbmRsZXI9Im9uX2Nsb3NlIiAvPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiIGlkPSJjbG9zZV9pbWFnZSI+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iaWNvbl9uYW1lIj53aW5kb3ctY2xvc2Utc3ltYm9saWM8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3By
b3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0
PgogICAgPC9jaGlsZD4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/ConnectionInformation.xml": {
		local:   "definitions/ConnectionInformation.xml",
		size:    6456,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJDb25uZWN0aW9uSW5mb3Jt
YXRpb24iPgogICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+Q29ubmVj
dGlvbiBJbmZvcm1hdGlvbjwvcHJvcGVydHk+CiAgICA8c2lnbmFsIG5hbWU9ImNsb3NlIiBoYW5kbGVy
PSJvbl9jbG9zZSIgLz4KICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0idmJveCI+CiAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0JveCIgaWQ9IlZib3giPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4i
PjEwPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtH
cmlkIiBpZD0iY29ubmVjdGlvbkluZm9HcmlkIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1h
cmdpbi10b3AiPjI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWJv
dHRvbSI+MTA8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXN0YXJ0
Ij4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tZW5kIj4xMDwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyb3ctc3BhY2luZyI+MjwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1bW4tc3BhY2luZyI+MTI8L3Byb3BlcnR5
PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwi
IGlkPSJzZXJ2ZXJMYWJlbCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRy
YW5zbGF0YWJsZT0ieWVzIj5TZXJ2ZXIgY29ubmVjdGVkIHRvOjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InNlcnZlclZhbHVlIj4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPjwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZ
X0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUi
PlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxj
aGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InRsc1ZlcnNpb25M
YWJlbCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0i
eWVzIj5TZWN1cmUgY29ubmVjdGlvbiB2ZXJzaW9uOjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3Bh
Y2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InRsc1ZlcnNpb25WYWx1ZSI+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj48L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX1NUQVJUPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9M
RUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5U
UlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJ0bHNBbGdvTGFiZWwi
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+
U2VjdXJlIGNvbm5lY3Rpb24gYWxnb3JpdGhtOjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InRsc0FsZ29WYWx1ZSI+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj48L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX1NUQVJUPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UUlVFPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJ0bHNGaW5nZXJwcmludExhYmVs
Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMi
PlNlY3VyZSBjb25uZWN0aW9uIGZpbmdlcnByaW50OjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX1NUQVJUPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UUlVFPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MzwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJ0bHNGaW5nZXJwcmludFZhbHVlIj4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPjwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05f
U1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdU
S19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNl
bGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGF0dHJpYnV0ZXM+CiAgICAg
ICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0iZm9udC1kZXNjIiB2YWx1ZT0iTW9ub3NwYWNlIi8+
CiAgICAgICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRh
Y2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2gi
PjM8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIg
aWQ9InBpbi1jZXJ0Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNs
YXRhYmxlPSJ5ZXMiPlBpbiBjZXJ0aWZpY2F0ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8c2ln
bmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX3BpbiIvPgogICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imxl
ZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3At
YXR0YWNoIj40PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4K
ICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94IiBpZD0iYnV0dG9uX2JveCI+CiAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRBTDwvcHJvcGVydHk+
CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iY2xv
c2UiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5D
bG9zZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0i
b25fY2xvc2UiLz4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2Jq
ZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAg
ICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/ContactPopupMenu.xml": {
		local:   "definitions/ContactPopupMenu.xml",
		size:    2549,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51IiBpZD0iY29udGFjdE1lbnUiPgogICAg
PGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51SXRlbSIgaWQ9ImVkaXRDb250YWN0TWVu
dUl0ZW0iPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkVk
aXQuLi48L3Byb3BlcnR5PgogICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUiIGhhbmRsZXI9Im9u
X2VkaXRfY29udGFjdCIgLz4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgog
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51SXRlbSIgaWQ9InJlbW92ZUNvbnRhY3RNZW51SXRlbSI+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+UmVtb3ZlPC9w
cm9wZXJ0eT4KICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9yZW1vdmVf
Y29udGFjdCIgLz4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtTZXBhcmF0b3JNZW51SXRlbSIgLz4KICAgIDwvY2hpbGQ+CiAgICA8Y2hp
bGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnVJdGVtIiBpZD0icmVzb3VyY2VzTWVudUl0ZW0i
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5UYWxrIHRvLi4uPC9wcm9w
ZXJ0eT4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtNZW51SXRlbSIgaWQ9InNlbmRGaWxlTWVudUl0ZW0iPgogICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNlbmQgZmlsZS4uLjwvcHJvcGVydHk+CiAg
ICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25fc2VuZF9maWxlX3RvX2NvbnRh
Y3QiIC8+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrTWVudUl0ZW0iIGlkPSJzZW5kRGlyZWN0b3J5TWVudUl0ZW0iPgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNlbmQgZGlyZWN0b3J5Li4uPC9w
cm9wZXJ0eT4KICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9zZW5kX2Rp
cmVjdG9yeV90b19jb250YWN0IiAvPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8Y2hp
bGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1NlcGFyYXRvck1lbnVJdGVtIiAvPgogICAgPC9jaGls
ZD4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0iIGlkPSJhc2tDb250
YWN0VG9TZWVTdGF0dXNNZW51SXRlbSI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFu
c2xhdGFibGU9InllcyI+QXNrIGNvbnRhY3QgdG8gc2VlIHRoZWlyIHN0YXR1czwvcHJvcGVydHk+CiAg
ICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25fYXNrX2NvbnRhY3RfdG9fc2Vl
X3N0YXR1cyIgLz4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtNZW51SXRlbSIgaWQ9ImFsbG93Q29udGFjdFRvU2VlU3RhdHVzTWVudUl0
ZW0iPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFsbG93
IGNvbnRhY3QgdG8gc2VlIG15IHN0YXR1czwvcHJvcGVydHk+CiAgICAgICAgPHNpZ25hbCBuYW1lPSJh
Y3RpdmF0ZSIgaGFuZGxlcj0ib25fYWxsb3dfY29udGFjdF90b19zZWVfc3RhdHVzIiAvPgogICAgICA8
L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01l
bnVJdGVtIiBpZD0iZm9yYmlkQ29udGFjdFRvU2VlU3RhdHVzTWVudUl0ZW0iPgogICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkZvcmJpZCBjb250YWN0IHRvIHNlZSBt
eSBzdGF0dXM8L3Byb3BlcnR5PgogICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUiIGhhbmRsZXI9
Im9uX2ZvcmJpZF9jb250YWN0X3RvX3NlZV9zdGF0dXMiIC8+CiAgICAgIDwvb2JqZWN0PgogICAgPC9j
aGlsZD4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2VwYXJhdG9yTWVudUl0ZW0i
IC8+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51SXRl
bSIgaWQ9ImR1bXBJbmZvTWVudUl0ZW0iPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJh
bnNsYXRhYmxlPSJ5ZXMiPkR1bXAgaW5mbzwvcHJvcGVydHk+CiAgICAgICAgPHNpZ25hbCBuYW1lPSJh
Y3RpdmF0ZSIgaGFuZGxlcj0ib25fZHVtcF9pbmZvIiAvPgogICAgICA8L29iamVjdD4KICAgIDwvY2hp
bGQ+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/Conversation.xml": {
		local:   "definitions/Conversation.xml",
		size:    717,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtXaW5kb3ciIGlkPSJjb252ZXJzYXRpb24iPgog
ICAgPHByb3BlcnR5IG5hbWU9IndpbmRvdy1wb3NpdGlvbiI+R1RLX1dJTl9QT1NfTk9ORTwvcHJvcGVy
dHk+CiAgICA8cHJvcGVydHkgbmFtZT0iZGVmYXVsdC1oZWlnaHQiPjUwMDwvcHJvcGVydHk+CiAgICA8
cHJvcGVydHkgbmFtZT0iZGVmYXVsdC13aWR0aCI+NDAwPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBu
YW1lPSJkZXN0cm95LXdpdGgtcGFyZW50Ij50cnVlPC9wcm9wZXJ0eT4KICAgIDxzaWduYWwgbmFtZT0i
ZW5hYmxlIiBoYW5kbGVyPSJvbl9jb25uZWN0IiAvPgogICAgPHNpZ25hbCBuYW1lPSJkaXNhYmxlIiBo
YW5kbGVyPSJvbl9kaXNjb25uZWN0IiAvPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCb3giIGlkPSJib3giPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9w
ZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iaG9tb2dlbmVvdXMiPmZhbHNlPC9wcm9wZXJ0eT4K
ICAgIAk8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9WRVJUSUNBTDwv
cHJvcGVydHk+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNs
YXNzIG5hbWU9ImNveWltIi8+CiAgICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/ConversationPane.xml": {
		local:   "definitions/ConversationPane.xml",
		size:    15117,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJib3giPgogICAgPHByb3BlcnR5
IG5hbWU9InZpc2libGUiPnRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImhvbW9nZW5l
b3VzIj5mYWxzZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19P
UklFTlRBVElPTl9WRVJUSUNBTDwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a01lbnVCYXIiIGlkPSJtZW51YmFyIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJs
ZSI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTWVudUl0ZW0iIGlkPSJjb252ZXJzYXRpb25NZW51Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJtZW51VGFnIj4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNlY3VyZSBDaGF0PC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2UtdW5kZXJsaW5lIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAg
PGNoaWxkIHR5cGU9InN1Ym1lbnUiPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnUi
IGlkPSJjb252ZXJzYXRpb25NZW51U3VibWVudSI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnVJdGVtIiBpZD0ic3RhcnRPVFJNZW51Ij4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+
U3RhcnQgZW5jcnlwdGVkIGNoYXQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxzaWduYWwg
bmFtZT0iYWN0aXZhdGUiIGhhbmRsZXI9Im9uX3N0YXJ0X290ciIgLz4KICAgICAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51SXRlbSIgaWQ9ImVuZE9UUk1lbnUi
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0i
eWVzIj5FbmQgZW5jcnlwdGVkIGNoYXQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxzaWdu
YWwgbmFtZT0iYWN0aXZhdGUiIGhhbmRsZXI9Im9uX2VuZF9vdHIiIC8+CiAgICAgICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4K
ICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0iIGlkPSJ2ZXJpZnlGaW5n
ZXJNZW51Ij4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xh
dGFibGU9InllcyI+VmVyaWZ5IGZpbmdlcnByaW50PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl92ZXJpZnlfZnAiIC8+CiAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0i
IGlkPSJ0cmFuc2Zlck1lbnUiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9InRyYW5zZmVyTGFiZWxUYWciPgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+VHJhbnNmZXI8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InVzZS11bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQgdHlw
ZT0ic3VibWVudSI+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudSIgaWQ9InRyYW5z
ZmVyTWVudVN1Ym1lbnUiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtNZW51SXRlbSIgaWQ9InNlbmRGaWxlTWVudUl0ZW0iPgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5TZW5kIEZpbGU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUiIGhhbmRs
ZXI9Im9uX3NlbmRfZmlsZV90b19jb250YWN0IiAvPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnVJdGVtIiBpZD0ic2VuZERpck1lbnVJdGVtIj4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+
U2VuZCBEaXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZh
dGUiIGhhbmRsZXI9Im9uX3NlbmRfZGlyX3RvX2NvbnRhY3QiIC8+CiAgICAgICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAg
IDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5m
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPnRydWU8L3Byb3BlcnR5
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwYWNrLXR5cGUiPkdUS19QQUNLX1NUQVJUPC9wcm9wZXJ0
eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICA8L3Bh
Y2tpbmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3gi
IGlkPSJub3RpZmljYXRpb24tYXJlYSI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9w
ZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9W
RVJUSUNBTDwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrQm94IiBpZD0ic2VjdXJpdHktd2FybmluZyI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9j
dXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjM3
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0ltYWdlIiBpZD0iaW1hZ2Utc2VjdXJpdHktd2FybmluZyI+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibWFyZ2luX2xlZnQiPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBh
bmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJs
YWJlbC1zZWN1cml0eS13YXJuaW5nIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNp
YmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9j
dXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAg
ICAgICAgICA8YXR0cmlidXRlIG5hbWU9IndlaWdodCIgdmFsdWU9ImJvbGQiIC8+CiAgICAgICAgICAg
ICAgICA8L2F0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBh
Y2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImJ1dHRvbi1zZWN1cml0eS13
YXJuaW5nIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fcmlnaHQiPjEwPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fdG9wIj4xNDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2JvdHRvbSI+MTI8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9ImJ1dHRvbi1sYWJlbC1zZWN1cml0eS13YXJuaW5nIj4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fbGVmdCI+MjM8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fcmlnaHQiPjIzPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVz
Ij5TZWN1cmUgQ2hhdDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFu
ZGxlcj0ib25fc3RhcnRfb3RyIiAvPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
IDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBhY2tfdHlwZSI+ZW5kPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwYWNrX3R5cGUiPnN0YXJ0PC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAg
IDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0JveCIgaWQ9ImZpbGUtdHJhbnNmZXIiPgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0idmlzaWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
Y2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFj
aW5nIj4zNzwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtJbWFnZSIgaWQ9ImltYWdlLWZpbGUtdHJhbnNmZXIiPgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9Im1hcmdpbl9sZWZ0Ij4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlk
PSJpbmZvLWZpbGUtdHJhbnNmZXIiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2li
bGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1
cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl90
b3AiPjE0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlv
biI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9ImxhYmVsLWZpbGUtdHJhbnNmZXIiPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3RhcnQ8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgICAgPGF0
dHJpYnV0ZSBuYW1lPSJ3ZWlnaHQiIHZhbHVlPSJib2xkIiAvPgogICAgICAgICAgICAgICAgICAgIDwv
YXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
UHJvZ3Jlc3NCYXIiIGlkPSJiYXItZmlsZS10cmFuc2ZlciI+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3RvcCI+NjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InNob3dfdGV4dCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8
L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFu
ZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fdG9wIj4xNDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9u
IiBpZD0iYnV0dG9uLWZpbGUtdHJhbnNmZXIiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iY2FuX2ZvY3VzIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibWFyZ2luX3JpZ2h0Ij4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9Im1hcmdpbl90b3AiPjc8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJtYXJnaW5fYm90dG9tIj4yMDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlk
PSJidXR0b24tbGFiZWwtZmlsZS10cmFuc2ZlciI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2xlZnQiPjM3PC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9yaWdodCI+Mzc8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3RvcCI+NTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fYm90dG9tIj41
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5k
bGVyPSJvbl9kZXN0cm95X2ZpbGVfdHJhbnNmZXIiIC8+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFja190eXBlIj5lbmQ8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwYWNrX3R5cGUiPmVuZDwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8c3R5bGU+CiAgICAgICAgICA8
Y2xhc3MgbmFtZT0ibm90aWZpY2F0aW9ucyIgLz4KICAgICAgICA8L3N0eWxlPgogICAgICA8L29iamVj
dD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+ZmFsc2U8L3By
b3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5mYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InBhY2stdHlwZSI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hp
bGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1Njcm9sbGVkV2luZG93IiBpZD0i
aGlzdG9yeVNjcm9sbCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRydWU8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2c2Nyb2xsYmFyLXBvbGljeSI+R1RLX1BPTElDWV9B
VVRPTUFUSUM8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoc2Nyb2xsYmFyLXBvbGlj
eSI+R1RLX1BPTElDWV9BVVRPTUFUSUM8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a1RleHRWaWV3IiBpZD0iaGlzdG9yeSI+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9IndyYXAtbW9kZSI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJlZGl0
YWJsZSI+ZmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY3Vyc29yLXZp
c2libGUiPmZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBpeGVscy1i
ZWxvdy1saW5lcyI+NTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LW1h
cmdpbiI+NTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyaWdodC1tYXJnaW4i
PjU8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9v
YmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPnRydWU8
L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj50cnVlPC9wcm9wZXJ0eT4KICAg
ICAgICA8cHJvcGVydHkgbmFtZT0icGFjay10eXBlIj5HVEtfUEFDS19FTkQ8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4K
ICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1Njcm9sbGVkV2lu
ZG93IiBpZD0icGVuZGluZ1Njcm9sbCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZzY3JvbGxiYXIt
cG9saWN5Ij5HVEtfUE9MSUNZX0FVVE9NQVRJQzwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImhzY3JvbGxiYXItcG9saWN5Ij5HVEtfUE9MSUNZX0FVVE9NQVRJQzwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im1pbi1jb250ZW50LWhlaWdodCI+NjA8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzaGFkb3ctdHlwZSI+aW48L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4K
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RleHRWaWV3IiBpZD0icGVuZGluZyI+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9IndyYXAtbW9kZSI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJlZGl0YWJsZSI+ZmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
Y3Vyc29yLXZpc2libGUiPmZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBpeGVscy1iZWxvdy1saW5lcyI+NTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsZWZ0LW1hcmdpbiI+NTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyaWdo
dC1tYXJnaW4iPjU8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4K
ICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBh
bmQiPmZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+dHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBhY2stdHlwZSI+R1RLX1BBQ0tfRU5EPC9wcm9w
ZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICA8
L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtT
Y3JvbGxlZFdpbmRvdyIgaWQ9Im1lc3NhZ2VTY3JvbGwiPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idnNjcm9sbGJhci1w
b2xpY3kiPkdUS19QT0xJQ1lfQVVUT01BVElDPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iaHNjcm9sbGJhci1wb2xpY3kiPkdUS19QT0xJQ1lfQVVUT01BVElDPC9wcm9wZXJ0eT4KICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ic2hhZG93LXR5cGUiPmluPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtUZXh0VmlldyIgaWQ9Im1lc3NhZ2UiPgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJoYXMtZm9jdXMiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0id3JhcC1tb2RlIj5HVEtfV1JBUF9XT1JEX0NIQVI8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZWRpdGFibGUiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGVmdC1tYXJnaW4iPjM8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icmlnaHQtbWFyZ2luIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
IDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkg
bmFtZT0iZXhwYW5kIj5mYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwi
PnRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwYWNrLXR5cGUiPkdUS19QQUNL
X0VORDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0
eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+
`,
	},

	"/definitions/EditProxy.xml": {
		local:   "definitions/EditProxy.xml",
		size:    6721,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJFZGl0UHJveHkiPgogICAg
PHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+RWRpdCBQcm94eTwvcHJvcGVy
dHk+CiAgICA8cHJvcGVydHkgbmFtZT0ibW9kYWwiPlRydWU8L3Byb3BlcnR5PgogICAgPHNpZ25hbCBu
YW1lPSJjbG9zZSIgaGFuZGxlcj0ib25fY2FuY2VsIiAvPgogICAgPGNoaWxkIGludGVybmFsLWNoaWxk
PSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iYm94Ij4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibWFyZ2luIj4xMDwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrR3JpZCIgaWQ9ImdyaWQiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJn
aW4tYm90dG9tIj4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyb3ctc3Bh
Y2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY29sdW1uLXNwYWNp
bmciPjY8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrTGFiZWwiIGlkPSJ0eXBlTGFiZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+UHJveHkgdHlwZTo8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9FTkQ8L3By
b3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAg
ICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDb21ib0JveFRleHQiIGlkPSJwcm90b2NvbC10eXBl
Ij4KICAgICAgICAgICAgICAgIDxpdGVtcz4KICAgICAgICAgICAgICAgIDwvaXRlbXM+CiAgICAgICAg
ICAgICAgICA8c2lnbmFsIG5hbWU9ImNoYW5nZWQiIGhhbmRsZXI9Im9uX3Byb3RvY29sX3R5cGVfY2hh
bmdlZCIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJ1c2VyTGFi
ZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9Inll
cyI+VXNlcjo8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnki
PkdUS19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJoYWxpZ24iPkdUS19BTElHTl9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQt
YXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0
YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hp
bGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRy
eSIgaWQ9InVzZXIiPgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxl
cj0ib25fc2F2ZSIgLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJwYXNzTGFiZWwiPgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+UGFz
c3dvcmQ6PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5H
VEtfSlVTVElGWV9SSUdIVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
aGFsaWduIj5HVEtfQUxJR05fRU5EPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0
dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFj
aCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrRW50cnki
IGlkPSJwYXNzd29yZCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJpbGl0eSI+
ZmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFu
ZGxlcj0ib25fc2F2ZSIgLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MjwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJzZXJ2ZXJMYWJl
bCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVz
Ij5TZXJ2ZXI6PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5
Ij5HVEtfSlVTVElGWV9SSUdIVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iaGFsaWduIj5HVEtfQUxJR05fRU5EPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0
LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0
dGFjaCI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2No
aWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrRW50
cnkiIGlkPSJzZXJ2ZXIiPgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFu
ZGxlcj0ib25fc2F2ZSIgLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MzwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJwb3J0TGFiZWwi
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+
UG9ydDo8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdU
S19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJo
YWxpZ24iPkdUS19BTElHTl9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0
YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNo
Ij40PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRyeSIg
aWQ9InBvcnQiPgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0i
b25fc2F2ZSIgLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+NDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJwYXRoTGFiZWwiPgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+UGF0aDo8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNU
SUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24i
PkdUS19BTElHTl9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj41PC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRyeSIgaWQ9InBh
dGgiPgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25fc2F2
ZSIgLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+NTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCIgaWQ9ImJ1dHRvbl9ib3giPgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9IT1JJWk9OVEFM
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0J1dHRvbiIgaWQ9ImNhbmNlbCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFi
ZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5DYW5jZWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHNp
Z25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9jYW5jZWwiLz4KICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9InNhdmUiPgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+U2F2ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWRlZmF1bHQiPnRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9zYXZlIi8+CiAgICAgICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8c3R5bGU+CiAgICAg
IDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9zdHlsZT4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+
Cg==
`,
	},

	"/definitions/ErrorNotification.xml": {
		local:   "definitions/ErrorNotification.xml",
		size:    382,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJpbmZvYmFyIj4KICAgIDxwcm9w
ZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX0hPUklaT05UQUw8L3Byb3BlcnR5
PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9Im1lc3NhZ2UiPgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibWFyZ2luLWxlZnQiPjEwPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bWFyZ2luLXJpZ2h0Ij4xMDwvcHJvcGVydHk+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICA8
L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/Feedback.xml": {
		local:   "definitions/Feedback.xml",
		size:    2889,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkaWFsb2ciPgogICAgPHBy
b3BlcnR5IG5hbWU9IndpbmRvdy1wb3NpdGlvbiI+R1RLX1dJTl9QT1NfQ0VOVEVSPC9wcm9wZXJ0eT4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPldlIHdvdWxkIGxpa2Ug
dG8gcmVjZWl2ZSB5b3VyIGZlZWRiYWNrPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJib3Jk
ZXJfd2lkdGgiPjc8L3Byb3BlcnR5PgogICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
Ym9yZGVyLXdpZHRoIj4xMDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0
aW9uIj5HVEtfT1JJRU5UQVRJT05fVkVSVElDQUw8L3Byb3BlcnR5PgoJICAgIDxjaGlsZD4KICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNh
bl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2lu
LWJvdHRvbSI+MzwvcHJvcGVydHk+CgkgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNs
YXRhYmxlPSJ5ZXMiPgogICAgICAgICAgICAgIFZpc2l0IHRoZSB3ZWJzaXRlIHRvIGdldCBpbiB0b3Vj
aCB3aXRoIHVzOgogICAgICAgICAgICA8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8
L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KCSAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtMYWJlbCIgaWQ9Im1lc3NhZ2UiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2Fu
X2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4t
dG9wIj4xMDwvcHJvcGVydHk+CgkgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRh
YmxlPSJ5ZXMiPmh0dHBzOi8vY295LmltPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InNlbGVjdGFibGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8YXR0cmlidXRlcz4KICAg
ICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9ImZvbnQtZGVzYyIgdmFsdWU9IiZsdDtFbnRlciBWYWx1
ZSZndDsgMTQiLz4KCSAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9IndlaWdodCIgdmFsdWU9InNlbWli
b2xkIi8+CiAgICAgICAgICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrTGFiZWwiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tdG9wIj4xMDwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tYm90dG9tIj4xMDwvcHJvcGVy
dHk+CgkgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPgogICAg
ICAgICAgICAgIExldCB1cyBrbm93IHdoYXQgeW91IHRoaW5rIG9mIENveUlNLgoKICAgICAgICAgICAg
ICBUaGlzIGlzIHRoZSBvbmx5IHdheSB3ZSBjYW4gY3JlYXRlIGEgYmV0dGVyIHByaXZhY3kgdG9vbC4K
ICAgICAgICAgICAgPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tp
bmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxjaGlsZCBpbnRl
cm5hbC1jaGlsZD0iYWN0aW9uX2FyZWEiPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3gi
IGlkPSJidXR0b25fYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19P
UklFTlRBVElPTl9IT1JJWk9OVEFMPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJjbG9zZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNsb3NlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9jbG9zZSIgLz4KICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHls
ZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAgICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2lu
dGVyZmFjZT4K
`,
	},

	"/definitions/FeedbackInfo.xml": {
		local:   "definitions/FeedbackInfo.xml",
		size:    1128,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtJbmZvQmFyIiBpZD0iZmVlZGJhY2tJbmZvIj4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJtZXNzYWdlLXR5cGUiPkdUS19NRVNTQUdFX09USEVSPC9wcm9wZXJ0
eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJzaG93LWNsb3NlLWJ1dHRvbiI+dHJ1ZTwvcHJvcGVydHk+CiAg
ICA8c2lnbmFsIG5hbWU9InJlc3BvbnNlIiBoYW5kbGVyPSJoYW5kbGVSZXNwb25zZSIgLz4KICAgIDxj
aGlsZCBpbnRlcm5hbC1jaGlsZD0iY29udGVudF9hcmVhIj4KICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
Qm94IiBpZD0iYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iaG9tb2dlbmVvdXMiPmZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElP
Tl9IT1JJWk9OVEFMPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtMYWJlbCIgaWQ9ImZlZWRiYWNrTWVzc2FnZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFyZSB5b3UgbGlraW5nIGl0PzwvcHJvcGVydHk+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwv
Y2hpbGQ+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrQm94IiBpZD0iYnV0dG9uX2JveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9y
aWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRBTDwvcHJvcGVydHk+CiAgICAgICAgPGNo
aWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iZmVlZGJhY2tCdXR0b24i
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5GZWVk
YmFjazwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElH
Tl9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAg
PC9vYmplY3Q+CiAgICA8L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/FileTransferAskToReceive.xml": {
		local:   "definitions/FileTransferAskToReceive.xml",
		size:    465,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtNZXNzYWdlRGlhbG9nIiBpZD0iRmlsZVRyYW5z
ZmVyQXNrVG9SZWNlaXZlIj4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aW5kb3ctcG9zaXRpb24iPkdUS19X
SU5fUE9TX0NFTlRFUl9BTFdBWVM8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im1vZGFsIj5m
YWxzZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj43PC9wcm9wZXJ0
eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkZpbGUgdHJhbnNm
ZXI8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InNlY29uZGFyeV90ZXh0IiB0cmFuc2xhdGFi
bGU9InllcyI+PC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJidXR0b25zIj5HVEtfQlVUVE9O
U19ZRVNfTk88L3Byb3BlcnR5PgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/FileTransferContinueWithoutEncryption.xml": {
		local:   "definitions/FileTransferContinueWithoutEncryption.xml",
		size:    498,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtNZXNzYWdlRGlhbG9nIiBpZD0iRmlsZVRyYW5z
ZmVyQ29udGludWVXaXRob3V0RW5jcnlwdGlvbiI+CiAgICA8cHJvcGVydHkgbmFtZT0id2luZG93LXBv
c2l0aW9uIj5HVEtfV0lOX1BPU19DRU5URVJfQUxXQVlTPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBu
YW1lPSJtb2RhbCI+ZmFsc2U8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0
aCI+NzwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVz
Ij5GaWxlIHRyYW5zZmVyIHdpdGhvdXQgZW5jcnlwdGlvbj88L3Byb3BlcnR5PgogICAgPHByb3BlcnR5
IG5hbWU9InNlY29uZGFyeV90ZXh0IiB0cmFuc2xhdGFibGU9InllcyI+PC9wcm9wZXJ0eT4KICAgIDxw
cm9wZXJ0eSBuYW1lPSJidXR0b25zIj5HVEtfQlVUVE9OU19ZRVNfTk88L3Byb3BlcnR5PgogIDwvb2Jq
ZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/FileTransferNotification.xml": {
		local:   "definitions/FileTransferNotification.xml",
		size:    1885,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJhcmVhLWZpbGUtdHJhbnNmZXIt
aW5mbyI+CiAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJv
cGVydHkgbmFtZT0iaG9tb2dlbmVvdXMiPmZhbHNlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1l
PSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX0hPUklaT05UQUw8L3Byb3BlcnR5PgogICAgPHBy
b3BlcnR5IG5hbWU9InNwYWNpbmciPjk8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtMYWJlbCIgaWQ9Im5hbWUtZmlsZS10cmFuc2Zlci1pbmZvIj4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24i
PnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiPnN0YXJ0PC9wcm9w
ZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXRvcCI+NDwvcHJvcGVydHk+CiAgICAg
ICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9ImZvbnQtZGVzYyIgdmFsdWU9
IiZsdDtFbnRlciBWYWx1ZSZndDsgOSIvPgogICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ3ZWlnaHQi
IHZhbHVlPSJ0aGluIi8+CiAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICA8L29iamVjdD4KICAgICAg
PHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxk
PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFdmVudEJveCI+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8c2lnbmFsIG5hbWU9ImJ1dHRvbi1w
cmVzcy1ldmVudCIgaGFuZGxlcj0ib25fZGVzdHJveV9zaW5nbGVfZmlsZV90cmFuc2ZlciIgc3dhcHBl
ZD0ibm8iLz4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIg
aWQ9ImltYWdlLWZpbGUtdHJhbnNmZXItaW5mbyI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1
cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGl4ZWxfc2l6ZSI+
MTI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9uYW1lIj53aW5kb3ct
Y2xvc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9zaXplIj4xPC9w
cm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0
PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJwYWNrLXR5cGUiPmVuZDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4K
ICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/FirstAccountDialog.xml": {
		local:   "definitions/FirstAccountDialog.xml",
		size:    2989,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkaWFsb2ciPgogICAgPHBy
b3BlcnR5IG5hbWU9IndpbmRvdy1wb3NpdGlvbiI+R1RLX1dJTl9QT1NfQ0VOVEVSPC9wcm9wZXJ0eT4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNldHVwIHlvdXIgZmly
c3QgYWNjb3VudDwvcHJvcGVydHk+CiAgICA8c2lnbmFsIG5hbWU9ImRlbGV0ZS1ldmVudCIgaGFuZGxl
cj0ib25fY2FuY2VsIiAvPgogICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrQm94IiBpZD0iVmJveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVu
dGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CgkgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbiI+MjA8
L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fdG9wIj4zMDwvcHJvcGVydHk+
CgkgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjI8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4K
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ibWVzc2FnZSI+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj50cnVlPC9wcm9wZXJ0eT4KCSAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+V2VsY29tZSB0byBDb3lJTSE8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2JvdHRvbSI+MTA8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8YXR0cmlidXRlcz4KICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9ImZvbnQtZGVz
YyIgdmFsdWU9IiZsdDtFbnRlciBWYWx1ZSZndDsgMTQiLz4KICAgICAgICAgICAgICA8YXR0cmlidXRl
IG5hbWU9InN0eWxlIiB2YWx1ZT0ibm9ybWFsIi8+CiAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1l
PSJ3ZWlnaHQiIHZhbHVlPSJzZW1pYm9sZCIvPgogICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZXhwYW5kIj5mYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9w
cm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGls
ZCBpbnRlcm5hbC1jaGlsZD0iYWN0aW9uX2FyZWEiPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
QnV0dG9uQm94IiBpZD0iYnV0dG9uX2JveCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmll
bnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX1ZFUlRJQ0FMPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImhvbW9nZW5lb3VzIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Im1hcmdpbl90b3AiPjIwPC9wcm9wZXJ0eT4KCSAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImJ1dHRvbl9yZWdpc3RlciI+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5DcmVhdGUg
YSBuZXcgYWNjb3VudDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFy
Z2luX2JvdHRvbSI+NTwvcHJvcGVydHk+CgkJICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhh
bmRsZXI9Im9uX3JlZ2lzdGVyIiAvPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8
L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
QnV0dG9uIiBpZD0iYnV0dG9uX2V4aXN0aW5nIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFkZCBhbiBleGlzdGluZyBhY2NvdW50PC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fYm90dG9tIj41PC9wcm9wZXJ0
eT4KCQkgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fZXhpc3RpbmciIC8+
CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxj
aGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJidXR0b25faW1w
b3J0Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5
ZXMiPkltcG9ydCBhY2NvdW50IGZyb20geW91ciBjb21wdXRlcjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2JvdHRvbSI+NTwvcHJvcGVydHk+CgkJICAgICAgICA8
c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX2ltcG9ydCIgLz4KICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KCSAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImJ1dHRvbl9jYW5jZWwiPgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIj5DYW5jZWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9Im1hcmdpbl9ib3R0b20iPjU8L3Byb3BlcnR5PgoJCSAgICAgICAgPHNpZ25h
bCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9jYW5jZWwiIC8+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxk
PgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8c3R5bGU+CiAgICAgIDxjbGFzcyBuYW1l
PSJjb3lpbSIvPgogICAgPC9zdHlsZT4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/GTKColumnTypes.xml": {
		local:   "definitions/GTKColumnTypes.xml",
		size:    189,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGlkPSJzdG9yZU9mQ29sdW1uVHlwZXMiIGNsYXNzPSJHdGtMaXN0
U3RvcmUiPgogICAgPGNvbHVtbnM+CiAgICAgIDwhLS0gY29sdW1uLW5hbWUgcGl4YnVmIC0tPgogICAg
ICA8Y29sdW1uIHR5cGU9Ikdka1BpeGJ1ZiIvPgogICAgPC9jb2x1bW5zPgogIDwvb2JqZWN0Pgo8L2lu
dGVyZmFjZT4K
`,
	},

	"/definitions/GeneratePIN.xml": {
		local:   "definitions/GeneratePIN.xml",
		size:    4938,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkaWFsb2ciPgogICAgPHBy
b3BlcnR5IG5hbWU9IndpbmRvdy1wb3NpdGlvbiI+R1RLX1dJTl9QT1NfQ0VOVEVSPC9wcm9wZXJ0eT4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJib3JkZXItd2lkdGgiPjc8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5
IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+PC9wcm9wZXJ0eT4KICAgIDxjaGlsZCBpbnRl
cm5hbC1jaGlsZD0idmJveCI+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9Im5vdGlmaWNh
dGlvbi1hcmVhIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyLXdpZHRoIj4xMDwvcHJvcGVy
dHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fVkVS
VElDQUw8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0JveCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFU
SU9OX1ZFUlRJQ0FMPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmci
PjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0ltYWdlIiBpZD0ic21wX2ltYWdlIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJtYXJnaW5fdG9wIj4xMDwvcHJvcGVydHk+CgkJICAgICAgPC9vYmplY3Q+CgkgICAgICAgIDwvY2hp
bGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJl
bCIgaWQ9InByb21wdCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXRvcCI+
NTwvcHJvcGVydHk+CgkJICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0i
eWVzIj48L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwi
IGlkPSJwaW4iPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrR3JpZCI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iY29sdW1uLXNwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJvdy1zcGFjaW5nIj4yPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiIGlk
PSJwYWRsb2NrX2ltYWdlMSIvPgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiPgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5TaGFy
ZSBpbiBwZXJzb248L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJq
dXN0aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdlIiBpZD0icGFkbG9ja19p
bWFnZTIiLz4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwv
cGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIj4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+U2hhcmUgdGhyb3VnaCBh
IGRpZmZlcmVudCBlbmNyeXB0ZWQgY2hhbm5lbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX0xFRlQ8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9TVEFSVDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4xPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1h
Z2UiIGlkPSJhbGVydF9pbWFnZSIvPgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4yPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5T
aGFyZSBpbiBhIHBob25lIGNhbGw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9MRUZUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICA8L2NoaWxkPgogICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0iYWN0aW9uX2FyZWEiPgogICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94IiBpZD0iYnV0dG9uX2JveCI+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX0hPUklaT05UQUw8
L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrQnV0dG9uIiBpZD0iYnV0dG9uX2ZpbmlzaGVkIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJjYW4tZGVmYXVsdCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5DbG9zZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9ImNsb3NlX3NoYXJlX3BpbiIvPgogICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8c3R5bGU+CiAgICAgICAgICA8Y2xhc3MgbmFtZT0ibm90
aWZpY2F0aW9ucyIvPgogICAgICAgIDwvc3R5bGU+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4K
ICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAgICA8L3N0eWxlPgogIDwvb2Jq
ZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/GlobalPreferences.xml": {
		local:   "definitions/GlobalPreferences.xml",
		size:    19985,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0U3RvcmUiIGlkPSJub3RpZmljYXRpb24t
dHlwZS1tb2RlbCI+CiAgICA8Y29sdW1ucz4KICAgICAgPCEtLSBub3RpZmljYXRpb24gbmFtZSAtLT4K
ICAgICAgPGNvbHVtbiB0eXBlPSJnY2hhcmFycmF5Ii8+CiAgICAgIDwhLS0gbm90aWZpY2F0aW9uIGlk
IC0tPgogICAgICA8Y29sdW1uIHR5cGU9ImdjaGFyYXJyYXkiLz4KICAgIDwvY29sdW1ucz4KICAgIDxk
YXRhPgogICAgICA8cm93PgogICAgICAgIDxjb2wgaWQ9IjAiIHRyYW5zbGF0YWJsZT0ieWVzIj5ObyBu
b3RpZmljYXRpb25zPC9jb2w+CiAgICAgICAgPGNvbCBpZD0iMSI+b2ZmPC9jb2w+CiAgICAgIDwvcm93
PgogICAgICA8cm93PgogICAgICAgIDxjb2wgaWQ9IjAiIHRyYW5zbGF0YWJsZT0ieWVzIj5Pbmx5IHNo
b3cgdGhhdCBhIG5ldyBtZXNzYWdlIGFycml2ZWQ8L2NvbD4KICAgICAgICA8Y29sIGlkPSIxIj5vbmx5
LXByZXNlbmNlLW9mLW5ldy1pbmZvcm1hdGlvbjwvY29sPgogICAgICA8L3Jvdz4KICAgICAgPHJvdz4K
ICAgICAgICA8Y29sIGlkPSIwIiB0cmFuc2xhdGFibGU9InllcyI+U2hvdyB3aG8gc2VudCB0aGUgbWVz
c2FnZTwvY29sPgogICAgICAgIDxjb2wgaWQ9IjEiPndpdGgtYXV0aG9yLWJ1dC1uby1jb250ZW50PC9j
b2w+CiAgICAgIDwvcm93PgogICAgICA8cm93PgogICAgICAgIDxjb2wgaWQ9IjAiIHRyYW5zbGF0YWJs
ZT0ieWVzIj5TaG93IG1lc3NhZ2U8L2NvbD4KICAgICAgICA8Y29sIGlkPSIxIj53aXRoLWNvbnRlbnQ8
L2NvbD4KICAgICAgPC9yb3c+CiAgICA8L2RhdGE+CiAgPC9vYmplY3Q+CiAgPG9iamVjdCBjbGFzcz0i
R3RrQWRqdXN0bWVudCIgaWQ9ImFkanVzdG1lbnQxIj4KICAgIDxwcm9wZXJ0eSBuYW1lPSJsb3dlciI+
MDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0idXBwZXIiPjM2MDA8L3Byb3BlcnR5PgogICAg
PHByb3BlcnR5IG5hbWU9InN0ZXBfaW5jcmVtZW50Ij42MDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkg
bmFtZT0icGFnZV9pbmNyZW1lbnQiPjYwPC9wcm9wZXJ0eT4KICA8L29iamVjdD4KICA8b2JqZWN0IGNs
YXNzPSJHdGtEaWFsb2ciIGlkPSJHbG9iYWxQcmVmZXJlbmNlcyI+CiAgICA8cHJvcGVydHkgbmFtZT0i
dGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5BY2NvdW50IERldGFpbHM8L3Byb3BlcnR5PgogICAgPHNp
Z25hbCBuYW1lPSJjbG9zZSIgaGFuZGxlcj0ib25fY2FuY2VsIiAvPgogICAgPGNoaWxkIGludGVybmFs
LWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iVmJveCI+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbiI+MTA8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9Im5vdGlmaWNhdGlvbi1hcmVhIj4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9WRVJUSUNBTDwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxzdHlsZT4KICAgICAgICAgICAgICA8Y2xhc3MgbmFtZT0ibm90aWZpY2F0
aW9ucyIvPgogICAgICAgICAgICA8L3N0eWxlPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9j
aGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtOb3RlYm9vayIg
aWQ9Im5vdGVib29rMSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNob3ctYm9yZGVyIj5GYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwYWdlIj4wPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1ib3R0b20iPjEwPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0dyaWQiIGlkPSJnZW5lcmFs
R3JpZCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXRvcCI+MTU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1ib3R0b20iPjEwPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tc3RhcnQiPjEwPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tZW5kIj4xMDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93LXNwYWNpbmciPjEyPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1bW4tc3BhY2luZyI+NjwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0xhYmVsIiBpZD0ic2luZ2xlV2luZG93TGFiZWwiPgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlVuaWZ5IGNvbnZlcnNhdGlvbnMgaW4g
b25lIHdpbmRvdzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1
c3RpZnkiPkdUS19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fRU5EPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxk
PgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDaGVja0J1dHRvbiIgaWQ9InNpbmds
ZVdpbmRvdyI+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4wPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTGFiZWwiIGlkPSJzZW5kV2l0aFNoaWZ0RW50ZXJMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+U2VuZCBtZXNzYWdlcyB3aXRo
IFNoaWZ0LUVudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
anVzdGlmeSI+R1RLX0pVU1RJRllfUklHSFQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NoZWNrQnV0dG9uIiBpZD0ic2Vu
ZFdpdGhTaGlmdEVudGVyIj4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFj
aCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRh
Y2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAg
ICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InNob3dFbXB0eUdyb3Vwc0xhYmVsIj4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5EaXNwbGF5IGVtcHR5
IGdyb3VwczwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3Rp
ZnkiPkdUS19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fRU5EPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8
L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDaGVja0J1dHRvbiIgaWQ9InNob3dFbXB0
eUdyb3VwcyI+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4yPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTGFiZWwiIGlkPSJlbWFjc0tleWJvYXJkTGFiZWwiPgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlVzZSBFbWFjcyBrZXkgYmluZGluZ3M8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtf
SlVTVElGWV9SSUdIVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImhhbGlnbiI+R1RLX0FMSUdOX0VORDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ0b3AtYXR0YWNoIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAg
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2hlY2tCdXR0b24iIGlkPSJlbWFjc0tleWJvYXJkIj4K
ICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjM8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQgdHlw
ZT0idGFiIj4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9ImxhYmVsLWdl
bmVyYWwtdGFiIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRh
YmxlPSJ5ZXMiPkdlbmVyYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0YWItZmlsbCI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0dyaWQiIGlkPSJub3Rp
ZmljYXRpb25zR3JpZCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXRvcCI+
MTU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1ib3R0b20i
PjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tc3RhcnQi
PjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tZW5kIj4x
MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93LXNwYWNpbmciPjEy
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1bW4tc3BhY2luZyI+
NjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ibm90aWZpY2F0aW9uVHlwZUluc3RydWN0aW9ucyI+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Q295
SU0gc3VwcG9ydHMgbm90aWZ5aW5nIHlvdSB3aGVuIGEgbmV3IG1lc3NhZ2UgYXJyaXZlcyAtIHRoZXNl
IG5vdGlmaWNhdGlvbnMgY2FuIGJlIHR1cm5lZCBvZmYsIG9yIHNldCB0byBzaG93IGRpZmZlcmVudCBh
bW91bnRzIG9mIGluZm9ybWF0aW9uLiBJdCdzIGltcG9ydGFudCB0byByZW1lbWJlciB0aGF0IHNob3dp
bmcgdGhlIHRleHQgb2YgdGhlIG1lc3NhZ2UsIG9yIGV2ZW4gdGhlIGF1dGhvciwgY291bGQgcG90ZW50
aWFsbHkgYmUgYSBwcml2YWN5IGxlYWsgaWYgeW91J3JlIGV2ZXIgaW4gc2V0dGluZ3Mgd2hlcmUgc29t
ZW9uZSBjYW4gc2VlIHlvdXIgc2NyZWVuLjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ3cmFwIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibWF4LXdpZHRoLWNoYXJzIj41MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0id2lkdGgiPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3Bh
Y2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9Im5vdGlmaWNhdGlvblR5cGVM
YWJlbCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFi
bGU9InllcyI+Tm90aWZpY2F0aW9uIHR5cGU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9SSUdIVDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX0VORDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4xPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ29tYm9C
b3giIGlkPSJub3RpZmljYXRpb25zVHlwZSI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9Im1vZGVsIj5ub3RpZmljYXRpb24tdHlwZS1tb2RlbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Imhhcy1mb2N1cyI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Nl
bGxSZW5kZXJlclRleHQiIGlkPSJub3RpZmljYXRpb24tbmFtZS1yZW5kZXJlZCIvPgogICAgICAgICAg
ICAgICAgICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGUg
bmFtZT0idGV4dCI+MDwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAgPC9hdHRyaWJ1dGVz
PgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ibGVmdC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ0b3AtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJub3RpZmljYXRpb25VcmdlbmN5SW5zdHJ1
Y3Rpb25zIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0
YWJsZT0ieWVzIj5Ob3RpZmljYXRpb25zIGNhbiBiZSBzZXQgdG8gZGlzcGxheSB1cmdlbnRseSAtIHRo
aXMgaXMgdXNlZnVsIGlmIHlvdSB3b3JrIGluIGZ1bGxzY3JlZW4gbW9kZS4gSWYgdGhlIG5vdGlmaWNh
dGlvbiBpcyBub3QgdXJnZW50LCBpdCB3aWxsIG5vdCBkaXNwbGF5IG92ZXIgYSBmdWxsc2NyZWVuIHdp
bmRvdy48L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+
dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1heC13aWR0
aC1jaGFycyI+NTA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQt
YXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9w
LWF0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Indp
ZHRoIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAg
ICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJub3RpZmljYXRpb25VcmdlbnRMYWJlbCI+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+U2hvdWxkIG5v
dGlmaWNhdGlvbnMgYmUgZGlzcGxheWVkIHVyZ2VudGx5PzwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fRU5EPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjM8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtDaGVja0J1dHRvbiIgaWQ9Im5vdGlmaWNhdGlvblVyZ2VudCI+CiAgICAgICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJub3RpZmljYXRpb25FeHBp
cnlJbnN0cnVjdGlvbnMiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIg
dHJhbnNsYXRhYmxlPSJ5ZXMiPk5vdGlmaWNhdGlvbnMgY2FuIHN0YXkgb24gdGhlIHNjcmVlbiB1bnRp
bCB5b3UndmUgZ29uZSBiYWNrIHRvIENveUlNLCBvciB0aGV5IGNhbiBleHBpcmUgYWZ0ZXIgYSB3aGls
ZS4gVGhlIGJlbG93IHNldHRpbmcgY2hhbmdlcyB0aGlzIGJlaGF2aW9yLjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj50cnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWF4LXdpZHRoLWNoYXJzIj41MDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj40PC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id2lkdGgiPjI8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9
Im5vdGlmaWNhdGlvbkV4cGlyZXNMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+U2hvdWxkIG5vdGlmaWNhdGlvbnMgZXhwaXJlPzwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPkdUS19K
VVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
aGFsaWduIj5HVEtfQUxJR05fRU5EPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InRvcC1hdHRhY2giPjU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+
CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDaGVja0J1dHRvbiIgaWQ9Im5vdGlmaWNhdGlvbkV4cGly
ZXMiPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+NTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAg
ICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xh
YmVsIiBpZD0ibm90aWZpY2F0aW9uQ29tbWFuZEluc3RydWN0aW9ucyI+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+WW91IGNhbiBzcGVjaWZ5
IGEgY3VzdG9tIGNvbW1hbmQgdG8gcnVuIHdoZW5ldmVyIGEgbWVzc2FnZSBpcyByZWNlaXZlZC4gSWYg
c3BlY2lmaWVkLCB0aGlzIGNvbW1hbmQgd2lsbCBydW4gb24gZXZlcnkgbWVzc2FnZXMgZXhjZXB0IGl0
IHdpbGwgd2FpdCBmb3IgYSB0aW1lb3V0IHBlcmlvZCBiZWZvcmUgcnVubmluZyB0aGUgbmV4dCB0aW1l
LiBUaGUgY29tbWFuZCBhbmQgdGltZW91dCBpcyBzcGVjaWZpZWQgYmVsb3cuIDwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPnRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj50cnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWF4LXdpZHRoLWNoYXJzIj41MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj42PC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id2lkdGgiPjI8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIg
aWQ9Im5vdGlmaWNhdGlvbkNvbW1hbmRMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Tm90aWZpY2F0aW9uIGNvbW1hbmQ8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElG
WV9SSUdIVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGln
biI+R1RLX0FMSUdOX0VORDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ0b3AtYXR0YWNoIj43PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrRW50cnkiIGlkPSJub3RpZmljYXRpb25Db21tYW5kIj4KICAgICAg
ICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9zYXZlIiAvPgog
ICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+NzwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBp
ZD0ibm90aWZpY2F0aW9uVGltZW91dExhYmVsIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5NaW5pbXVtIHRpbWUgYmV0d2VlbiBub3RpZmlj
YXRpb25zIGluIHNlY29uZHM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9SSUdIVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX0VORDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj44PC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU3BpbkJ1dHRvbiIgaWQ9
Im5vdGlmaWNhdGlvblRpbWVvdXQiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJj
bGltYi1yYXRlIj42MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImFkanVzdG1lbnQiPmFkanVzdG1lbnQxPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8c2ln
bmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9zYXZlIiAvPgogICAgICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+ODwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwv
cGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZCB0eXBlPSJ0YWIiPgogICAgICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ibGFiZWwtbm90aWZpY2F0aW9ucy10YWIiPgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Tm90
aWZpY2F0aW9uczwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJs
ZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBh
Y2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRhYi1maWxsIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrR3JpZCIgaWQ9ImRlYnVnZ2luZ0dy
aWQiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi10b3AiPjE1PC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tYm90dG9tIj4xMDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXN0YXJ0Ij4xMDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWVuZCI+MTA8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJvdy1zcGFjaW5nIj4xMjwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY29sdW1uLXNwYWNpbmciPjY8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtMYWJlbCIgaWQ9InJhd0xvZ0ZpbGVJbnN0cnVjdGlvbnMiPgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPklmIHlvdSBzZXQgdGhpcyBwcm9w
ZXJ0eSB0byBhIGZpbGUgbmFtZSwgbG93IGxldmVsIGluZm9ybWF0aW9uIHdpbGwgYmUgbG9nZ2VkIHRo
ZXJlLiBCZSB2ZXJ5IGNhcmVmdWwgLSB0aGlzIGluZm9ybWF0aW9uIGlzIHNlbnNpdGl2ZSBhbmQgY291
bGQgcG90ZW50aWFsbHkgY29udGFpbiB2ZXJ5IHByaXZhdGUgaW5mb3JtYXRpb24uIE9ubHkgdHVybiB0
aGlzIHNldHRpbmcgb24gaWYgeW91IGFic29sdXRlbHkgbmVlZCBpdCBmb3IgZGVidWdnaW5nLiBUaGlz
IGZpbGUgd2lsbCBzcGVjaWZpY2FsbHkgbG9nIFhNUFAgdHJhZmZpYyBpbmZvcm1hdGlvbi4gVGhpcyBz
ZXR0aW5nIHdpbGwgb25seSB0YWtlIGVmZmVjdCBhZnRlciBhIHJlc3RhcnQgb2YgQ295SU0uPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXAiPnRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXgtd2lkdGgtY2hhcnMiPjUw
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjA8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aCI+MjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0xhYmVsIiBpZD0icmF3TG9nRmlsZUxhYmVsIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5SYXcgbG9nIGZpbGU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5HVEtfSlVTVElGWV9SSUdIVDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FM
SUdOX0VORDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRh
Y2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0
YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAg
ICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrRW50cnkiIGlkPSJyYXdMb2dGaWxlIj4KICAgICAgICAgICAgICAgICAgICA8c2ln
bmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9zYXZlIiAvPgogICAgICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwv
cGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZCB0eXBlPSJ0YWIiPgogICAgICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ibGFiZWwtZGVidWdnaW5nLXRhYiI+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5EZWJ1Z2dp
bmc8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0YWItZmlsbCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJhY3Rpb25fYXJlYSI+
CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJidXR0b25fYm94Ij4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpP
TlRBTDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtCdXR0b24iIGlkPSJjYW5jZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Q2FuY2VsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fY2FuY2VsIi8+CiAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJzYXZlIj4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNhdmU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbi1kZWZhdWx0Ij50cnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fc2F2ZSIvPgogICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPHN0eWxlPgog
ICAgICA8Y2xhc3MgbmFtZT0iY295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+CjwvaW50ZXJm
YWNlPgo=
`,
	},

	"/definitions/GroupDetails.xml": {
		local:   "definitions/GroupDetails.xml",
		size:    2138,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkaWFsb2ciPgogICAgPHBy
b3BlcnR5IG5hbWU9IndpbmRvdy1wb3NpdGlvbiI+R1RLX1dJTl9QT1NfQ0VOVEVSPC9wcm9wZXJ0eT4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFkZCBHcm91cDwvcHJv
cGVydHk+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9InZib3giPgogICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCb3giIGlkPSJWYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iaG9tb2dlbmVvdXMiPmZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklF
TlRBVElPTl9WRVJUSUNBTDwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJncm91cC1uYW1lLWxhYmVsIiA+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkdyb3VwIG5hbWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImV4cGFuZCI+ZmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+
MTwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRyeSIgaWQ9Imdyb3VwLW5hbWUiPgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFzLWZvY3VzIj50cnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImFjdGl2YXRlcy1kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJleHBhbmQiPmZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8
L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNo
aWxkIGludGVybmFsLWNoaWxkPSJhY3Rpb25fYXJlYSI+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCdXR0b25Cb3giIGlkPSJiYm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0
aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRBTDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJidG4tY2FuY2VsIj4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNh
bmNlbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24i
IGlkPSJidG4tb2siPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xh
dGFibGU9InllcyI+T0s8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNh
bi1kZWZhdWx0Ij50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2Jq
ZWN0PgogICAgPC9jaGlsZD4KICAgIDxhY3Rpb24td2lkZ2V0cz4KICAgICAgPGFjdGlvbi13aWRnZXQg
cmVzcG9uc2U9ImNhbmNlbCI+YnRuLWNhbmNlbDwvYWN0aW9uLXdpZGdldD4KICAgICAgPGFjdGlvbi13
aWRnZXQgcmVzcG9uc2U9Im9rIiBkZWZhdWx0PSJ0cnVlIj5idG4tb2s8L2FjdGlvbi13aWRnZXQ+CiAg
ICA8L2FjdGlvbi13aWRnZXRzPgogICAgPHN0eWxlPgogICAgICA8Y2xhc3MgbmFtZT0iY295aW0iLz4K
ICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/Importer.xml": {
		local:   "definitions/Importer.xml",
		size:    4911,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0idXRmLTgiPz4KPGludGVyZmFjZT4KICA8b2JqZWN0
IGNsYXNzPSJHdGtMaXN0U3RvcmUiIGlkPSJpbXBvcnRBY2NvdW50c1N0b3JlIj4KICAgIDxjb2x1bW5z
PgogICAgICA8Y29sdW1uIHR5cGU9ImdjaGFyYXJyYXkiLz4KICAgICAgPGNvbHVtbiB0eXBlPSJnY2hh
cmFycmF5Ii8+CiAgICAgIDxjb2x1bW4gdHlwZT0iZ2Jvb2xlYW4iLz4KICAgIDwvY29sdW1ucz4KICA8
L29iamVjdD4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJpbXBvcnRlcldpbmRvdyI+CiAg
ICA8cHJvcGVydHkgbmFtZT0id2luZG93LXBvc2l0aW9uIj5HVEtfV0lOX1BPU19DRU5URVI8L3Byb3Bl
cnR5PgogICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+SW1wb3J0IEFj
Y291bnRzPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij40NTA8L3By
b3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImhlaWdodF9yZXF1ZXN0Ij42MDA8L3Byb3BlcnR5Pgog
ICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTA8L3Byb3BlcnR5PgogICAgPGNoaWxkIGlu
dGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iYm94Ij4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iaG9tb2dlbmVvdXMiPmZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9WRVJUSUNBTDwvcHJv
cGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlk
PSJsYWJlbCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5
ZXMiPkNob29zZSB0aGUgYWNjb3VudHMgeW91IHdvdWxkIGxpa2UgdG8gaW1wb3J0IGFuZCBzdGFydCB1
c2luZyBmcm9tIENveUlNLjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFw
Ij50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXAtbW9kZSI+UEFO
R09fV1JBUF9XT1JEX0NIQVI8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+ZmFsc2U8L3Byb3BlcnR5
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtTY3JvbGxlZFdpbmRvdyIgaWQ9ImltcG9ydGVyU2Nyb2xsIj4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZzY3JvbGxiYXItcG9saWN5Ij5HVEtfUE9MSUNZX0FVVE9NQVRJQzwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoc2Nyb2xsYmFyLXBvbGljeSI+R1RLX1BPTElDWV9B
VVRPTUFUSUM8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrVHJlZVZpZXciIGlkPSJpbXBvcnRlclRyZWVWaWV3Ij4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJtb2RlbCI+aW1wb3J0QWNjb3VudHNTdG9yZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVW
aWV3Q29sdW1uIiBpZD0iaW1wb3J0LXRoaXMtYWNjb3VudC1jb2x1bW4iPgogICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkltcG9ydCB0aGlzPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic29ydF9jb2x1bW5faWQi
PjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NlbGxSZW5kZXJlclRvZ2dsZSIgaWQ9ImltcG9ydC10aGlzLWFj
Y291bnQtcmVuZGVyZXIiPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYWN0
aXZhdGFibGUiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgICAgICAgPGF0
dHJpYnV0ZSBuYW1lPSJhY3RpdmUiPjI8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAgICAgIDwv
YXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVWaWV3Q29sdW1uIiBpZD0iZnJvbS1h
cHBsaWNhdGlvbi1jb2x1bW4iPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRs
ZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkZyb20gYXBwbGljYXRpb248L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzb3J0X2NvbHVtbl9pZCI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrQ2VsbFJlbmRlcmVyVGV4dCIgaWQ9ImZyb20tYXBwbGljYXRpb24tcmVuZGVyZXIiLz4KICAgICAg
ICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgICAgICA8YXR0cmli
dXRlIG5hbWU9InRleHQiPjA8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAgICAgIDwvYXR0cmli
dXRlcz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVWaWV3Q29sdW1uIiBpZD0iYWNjb3VudC1uYW1l
LWNvbHVtbiI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xh
dGFibGU9InllcyI+QWNjb3VudCBuYW1lPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ic29ydF9jb2x1bW5faWQiPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NlbGxSZW5kZXJl
clRleHQiIGlkPSJhY2NvdW50LW5hbWUtcmVuZGVyZXIiLz4KICAgICAgICAgICAgICAgICAgICAgIDxh
dHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9InRleHQiPjE8
L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAgICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZXhwYW5kIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwi
PnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3By
b3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxk
IGludGVybmFsLWNoaWxkPSJhY3Rpb25fYXJlYSI+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtC
dXR0b25Cb3giIGlkPSJidXR0b25fYm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVu
dGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRBTDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJtYXJnaW4tdG9wIj4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4K
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJidXR0b25fY2FuY2VsIj4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNh
bmNlbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24i
IGlkPSJidXR0b25fb2siPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFu
c2xhdGFibGU9InllcyI+SW1wb3J0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJjYW4tZGVmYXVsdCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAg
ICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8YWN0aW9uLXdpZGdldHM+CiAgICAgIDxhY3Rpb24t
d2lkZ2V0IHJlc3BvbnNlPSJjYW5jZWwiPmJ1dHRvbl9jYW5jZWw8L2FjdGlvbi13aWRnZXQ+CiAgICAg
IDxhY3Rpb24td2lkZ2V0IHJlc3BvbnNlPSJvayIgZGVmYXVsdD0idHJ1ZSI+YnV0dG9uX29rPC9hY3Rp
b24td2lkZ2V0PgogICAgPC9hY3Rpb24td2lkZ2V0cz4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5h
bWU9ImNveWltIi8+CiAgICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/InfoBar.xml": {
		local:   "definitions/InfoBar.xml",
		size:    4204,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtJbmZvQmFyIiBpZD0iaW5mb2JhciI+CiAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0i
cmV2ZWFsZWQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgIDxzaWduYWwgbmFtZT0icmVzcG9uc2UiIGhhbmRs
ZXI9ImhhbmRsZS1yZXNwb25zZSIgc3dhcHBlZD0ibm8iLz4KICAgIDxjaGlsZCBpbnRlcm5hbC1jaGls
ZD0iYWN0aW9uX2FyZWEiPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJhY3Rp
b25zIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ibm9fc2hvd19hbGwiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9Im1hcmdpbl9yaWdodCI+MTI8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJzcGFjaW5nIj42PC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGF5b3V0X3N0
eWxlIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgPHN0eWxlPgogICAgICAgICAgPGNsYXNzIG5hbWU9
ImFjdGlvbnMiLz4KICAgICAgICA8L3N0eWxlPgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICAgIDxj
aGlsZCBpbnRlcm5hbC1jaGlsZD0iY29udGVudF9hcmVhIj4KICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
Qm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0ltYWdlIiBpZD0iaWNvbi1pbWFnZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InN0b2Nr
Ij5ndGstbWlzc2luZy1pbWFnZTwvcHJvcGVydHk+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
IDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+LTE8L3Byb3BlcnR5PgogICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVs
IiBpZD0idGl0bGUtbGFiZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2Vu
dGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0cmFja192aXNpdGVkX2xpbmtz
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAgICAgICA8Y2xhc3MgbmFtZT0i
dGl0bGUiLz4KICAgICAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJ0aW1l
LWxhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJub19zaG93X2FsbCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxzdHlsZT4KICAgICAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9InRpbWUiLz4KICAg
ICAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
PHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJzcGlubmVyLWJveCI+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAgICAgICA8
L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8c3R5bGU+CiAgICAgICAgICA8Y2xhc3Mg
bmFtZT0iY29udGVudCIvPgogICAgICAgIDwvc3R5bGU+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFj
a2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBhZGRpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjA8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogICAgPHN0eWxlPgogICAg
ICA8Y2xhc3MgbmFtZT0iaW5mb2JhciIvPgogICAgPC9zdHlsZT4KICA8L29iamVjdD4KPC9pbnRlcmZh
Y2U+Cg==
`,
	},

	"/definitions/InfoBarHighlightAttributes.xml": {
		local:   "definitions/InfoBarHighlightAttributes.xml",
		size:    539,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9ImxhYmVsTmlja25hbWUiPgog
ICAgPGF0dHJpYnV0ZXM+CiAgICAgIDxhdHRyaWJ1dGUgbmFtZT0id2VpZ2h0IiB2YWx1ZT0iYm9sZCIv
PgogICAgPC9hdHRyaWJ1dGVzPgogIDwvb2JqZWN0PgogIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBp
ZD0ibGFiZWxBZmZpbGlhdGlvbiI+CiAgICA8YXR0cmlidXRlcz4KICAgICAgPGF0dHJpYnV0ZSBuYW1l
PSJ3ZWlnaHQiIHZhbHVlPSJib2xkIi8+CiAgICAgIDxhdHRyaWJ1dGUgbmFtZT0ic3R5bGUiIHZhbHVl
PSJpdGFsaWMiLz4KICAgIDwvYXR0cmlidXRlcz4KICA8L29iamVjdD4KICAgPG9iamVjdCBjbGFzcz0i
R3RrTGFiZWwiIGlkPSJsYWJlbFJvbGUiPgogICAgPGF0dHJpYnV0ZXM+CiAgICAgIDxhdHRyaWJ1dGUg
bmFtZT0id2VpZ2h0IiB2YWx1ZT0iYm9sZCIvPgogICAgICA8YXR0cmlidXRlIG5hbWU9InN0eWxlIiB2
YWx1ZT0iaXRhbGljIi8+CiAgICA8L2F0dHJpYnV0ZXM+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/LoadingOverlay.xml": {
		local:   "definitions/LoadingOverlay.xml",
		size:    4563,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtPdmVybGF5IiBpZD0ibG9hZGluZy1vdmVybGF5
Ij4KICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2ZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrQm94IiBpZD0ibG9hZGluZy1vdmVybGF5LWJveCI+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5k
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmV4cGFuZCI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVy
dHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0ibG9h
ZGluZy1vdmVybGF5LWNvbnRlbnQiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPmNlbnRlcjwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJsb2Fk
aW5nLW92ZXJsYXktc3Bpbm5lci1jb250ZW50Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJo
YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFs
aWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVu
dGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1NwaW5uZXIiIGlkPSJsb2FkaW5nLW92ZXJsYXktc3Bp
bm5lciI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX3JlcXVlc3QiPjQ4
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGVpZ2h0X3JlcXVl
c3QiPjQ4PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJs
ZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGln
biI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFs
aWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJo
ZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImFjdGl2ZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhw
YW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3Np
dGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
IDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9ImxvYWRpbmctb3ZlcmxheS1ib2R5Ij4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJs
b2FkaW5nLW92ZXJsYXktdGl0bGUiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJo
YWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Inlh
bGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5G
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8
L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtMYWJlbCIgaWQ9ImxvYWRpbmctb3ZlcmxheS1kZXNjcmlwdGlvbiI+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1heF93aWR0aF9jaGFycyI+NDA8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+
MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9v
YmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpbmRleCI+LTE8L3By
b3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFj
ZT4K
`,
	},

	"/definitions/MUCCreateRoomDialog.xml": {
		local:   "definitions/MUCCreateRoomDialog.xml",
		size:    2254,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJjcmVhdGUtcm9vbS1kaWFs
b2ciPgogICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX3JlcXVlc3QiPjU4MDwvcHJvcGVydHk+CiAgICA8
cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5DcmVhdGUgY2hhdCByb29tPC9w
cm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJyZXNpemFibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
IDxwcm9wZXJ0eSBuYW1lPSJkZXN0cm95X3dpdGhfcGFyZW50Ij5UcnVlPC9wcm9wZXJ0eT4KICAgIDxw
cm9wZXJ0eSBuYW1lPSJ0eXBlX2hpbnQiPmRpYWxvZzwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0iZ3Jhdml0eSI+bm9ydGgtZWFzdDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVy
X3dpZHRoIj4wPC9wcm9wZXJ0eT4KICAgIDxzaWduYWwgbmFtZT0iY2xvc2UiIGhhbmRsZXI9Im9uX2Ns
b3NlX3dpbmRvdyIgc3dhcHBlZD0ibm8iLz4KICAgIDxzaWduYWwgbmFtZT0iZGVzdHJveSIgaGFuZGxl
cj0ib25fY2xvc2Vfd2luZG93IiBzd2FwcGVkPSJubyIvPgogICAgPGNoaWxkIGludGVybmFsLWNoaWxk
PSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0
aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjI8
L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0iYWN0aW9uX2FyZWEiPgogICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94IiBpZD0iYnV0dG9ucyI+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsYXlvdXRfc3R5bGUiPmVuZDwvcHJvcGVydHk+CiAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJub3RpZmljYXRpb25zIj4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iY3JlYXRlLXJvb20tY29udGVudCI+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmll
bnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPHN0eWxl
PgogICAgICA8Y2xhc3MgbmFtZT0ibXVjIi8+CiAgICAgIDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAg
PC9zdHlsZT4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCCreateRoomForm.xml": {
		local:   "definitions/MUCCreateRoomForm.xml",
		size:    12007,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJjcmVhdGUtcm9vbS1mb3JtIj4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij41MDA8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im9yaWVu
dGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0JveCIgaWQ9Im5vdGlmaWNhdGlvbi1hcmVhLWJveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fYm90
dG9tIj4xMjwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3By
b3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3Bl
cnR5PgogICAgICAgIDxzdHlsZT4KICAgICAgICAgIDxjbGFzcyBuYW1lPSJub3RpZmljYXRpb25zIi8+
CiAgICAgICAgPC9zdHlsZT4KICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4w
PC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iY3JlYXRlLXJvb20tY29udGVudCI+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93
aWR0aCI+MTI8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVy
dGljYWw8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0dyaWQiIGlkPSJjb250cm9scyI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJvd19zcGFjaW5nIj4xMjwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1bW5fc3BhY2luZyI+MTI8L3Byb3BlcnR5
PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwi
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5BY2NvdW50PC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5yaWdodDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdF9hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcF9hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NvbWJvQm94IiBpZD0iYWNjb3VudHMiPgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDZWxsUmVuZGVyZXJU
ZXh0IiBpZD0iYWNjb3VudC1uYW1lLXJlbmRlcmVkIi8+CiAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1
dGVzPgogICAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0idGV4dCI+MDwvYXR0cmlidXRl
PgogICAgICAgICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxlZnRfYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3Bh
Y2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFs
aWduIj5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+
Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJh
bnNsYXRhYmxlPSJ5ZXMiPlJvb208L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9Imp1c3RpZnkiPnJpZ2h0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFj
aCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wX2F0dGFjaCI+
MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrRW50cnkiIGlk
PSJyb29tLW5hbWUtZW50cnkiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBsYWNlaG9sZGVyX3Rl
eHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5FbnRlciB0aGUgcm9vbSBuYW1lPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2hhbmdlZCIgaGFuZGxlcj0ib25fcm9vbV9uYW1lX2NoYW5n
ZSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUiIGhh
bmRsZXI9Im9uX2NyZWF0ZSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0
dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wX2F0dGFj
aCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ29tYm9C
b3hUZXh0IiBpZD0iY2hhdC1zZXJ2aWNlcy1saXN0Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJoZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJo
YXNfZW50cnkiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkIGludGVybmFsLWNo
aWxkPSJlbnRyeSI+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0VudHJ5IiBpZD0i
Y2hhdC1zZXJ2aWNlcy1lbnRyeSI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBs
YWNlaG9sZGVyX3RleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5FbnRlciB0aGUgbG9jYXRpb24gb3Igc2Vs
ZWN0IG9uZSBmcm9tIHRoZSBsaXN0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8c2lnbmFs
IG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9jcmVhdGUiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAg
ICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjaGFuZ2VkIiBoYW5kbGVyPSJvbl9jaGF0c2VydmljZV9l
bnRyeV9jaGFuZ2UiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wX2F0dGFjaCI+MjwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5XaGVyZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+cmlnaHQ8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImxlZnRfYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+
CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9u
Ij52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2lu
ZyI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0NoZWNrQnV0dG9uIiBpZD0iYXV0b2pvaW4tY2hlY2stYnV0dG9uIj4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5f
QXV0b21hdGljYWxseSBqb2luIHRoZSByb29tPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9vbHRpcF90ZXh0IiB0cmFuc2xhdGFibGU9InllcyI+
QXV0b21hdGljYWxseSBqb2luIHRoZSByb29tIHdoZW4gY3JlYXRlZDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InVzZV91bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJkcmF3X2luZGljYXRvciI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJ0b2dnbGVkIiBoYW5kbGVy
PSJvbl9yb29tX2F1dG9qb2luX3RvZ2dsZWQiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NoZWNrQnV0dG9uIiBpZD0iY29uZmlnLXJvb20t
Y2hlY2stYnV0dG9uIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRy
YW5zbGF0YWJsZT0ieWVzIj5BZGQgY3VzdG9tIHJvb20gY29uZmlndXJhdGlvbjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3RhcnQ8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRl
cjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImRyYXdfaW5kaWNh
dG9yIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9InRvZ2ds
ZWQiIGhhbmRsZXI9Im9uX3Jvb21fY29uZmlnX3RvZ2dsZWQiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGVmdF9hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRv
cF9hdHRhY2giPjM8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
PC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxk
PgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4
cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAg
ICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCdXR0b25Cb3giIGlkPSJidXR0b25zIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+ZW5kPC9wcm9wZXJ0
eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGF5b3V0X3N0eWxlIj5lbmQ8L3Byb3BlcnR5PgogICAg
ICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3JkZXJf
d2lkdGgiPjA8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrQm94IiBpZD0ic3Bpbm5lci1ib3giPgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94Ij4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImNhbmNlbC1i
dXR0b24iPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCI+X0NhbmNlbDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZh
dWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNl
X3VuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1l
PSJjbGlja2VkIiBoYW5kbGVyPSJvbl9jYW5jZWwiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJub25faG9tb2dlbmVvdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJjcmVh
dGUtcm9vbS1idXR0b24iPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIg
dHJhbnNsYXRhYmxlPSJ5ZXMiPl9DcmVhdGUgUm9vbTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzZW5zaXRpdmUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2VfdW5kZXJsaW5lIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9u
X2NyZWF0ZSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBh
bmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwYWNr
X3R5cGUiPmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBv
c2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibm9u
X2hvbW9nZW5lb3VzIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAg
ICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFja190eXBlIj5lbmQ8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFja190eXBlIj5lbmQ8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJw
YWNrX3R5cGUiPmVuZDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4y
PC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICA8L29iamVjdD4KPC9pbnRl
cmZhY2U+Cg==
`,
	},

	"/definitions/MUCCreateRoomSuccess.xml": {
		local:   "definitions/MUCCreateRoomSuccess.xml",
		size:    3502,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJjcmVhdGVSb29tU3VjY2VzcyI+
CiAgICA8cHJvcGVydHkgbmFtZT0iaGVpZ2h0X3JlcXVlc3QiPjMwMDwvcHJvcGVydHk+CiAgICA8cHJv
cGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Imhl
eHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InZleHBhbmQiPlRydWU8L3By
b3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+
CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9ImNyZWF0ZVJvb21TdWNj
ZXNzQ29udGVudCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0
eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkg
bmFtZT0idmV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVu
dGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmci
PjIwPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJ
bWFnZSIgaWQ9ImNyZWF0ZVJvb21TdWNjZXNzSW1hZ2UiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0icGl4ZWxfc2l6ZSI+NjA8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNv
bl9uYW1lIj5kaWFsb2ctb2s8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNl
X2ZhbGxiYWNrIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25f
c2l6ZSI+MzwvcHJvcGVydHk+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAg
ICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVs
IiBpZD0iY3JlYXRlUm9vbVN1Y2Nlc3NNZXNzYWdlIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Q29uZ3JhdHVsYXRpb25zLCB0aGUgcm9vbSBoYXMgYmVl
biBjcmVhdGVkPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0
dG9uQm94IiBpZD0iY3JlYXRlUm9vbVN1Y2Nlc3NCdXR0b25zIj4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImhhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InNwYWNpbmciPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxheW91dF9z
dHlsZSI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImNyZWF0ZU5ld1Jvb21CdXR0b24iPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Q3JlYXRlIGFub3Ro
ZXIgcm9vbTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNf
ZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNr
ZWQiIGhhbmRsZXI9Im9uX2NyZWF0ZVJvb21fY2xpY2tlZCIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCdXR0b24iIGlkPSJqb2luQ3JlYXRlZFJvb21CdXR0b24iPgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Sm9pbjwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX2pvaW5Sb29tX2Ns
aWNrZWQiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
PHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFj
a2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAg
PC9vYmplY3Q+CjwvaW50ZXJmYWNlPg==
`,
	},

	"/definitions/MUCJoinRoomDialog.xml": {
		local:   "definitions/MUCJoinRoomDialog.xml",
		size:    10397,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJqb2luLXJvb20tZGlhbG9n
Ij4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij41ODA8L3Byb3BlcnR5PgogICAgPHBy
b3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+Sm9pbiBhIGNoYXQgcm9vbTwvcHJv
cGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0icmVzaXphYmxlIj5GYWxzZTwvcHJvcGVydHk+CiAgICA8
cHJvcGVydHkgbmFtZT0iZGVzdHJveV93aXRoX3BhcmVudCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJv
cGVydHkgbmFtZT0idHlwZV9oaW50Ij5kaWFsb2c8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9
ImdyYXZpdHkiPm5vcnRoLWVhc3Q8L3Byb3BlcnR5PgogICAgPHNpZ25hbCBuYW1lPSJjbG9zZSIgaGFu
ZGxlcj0ib25fY2xvc2Vfd2luZG93IiBzd2FwcGVkPSJubyIvPgogICAgPHNpZ25hbCBuYW1lPSJkZXN0
cm95IiBoYW5kbGVyPSJvbl9jbG9zZV93aW5kb3ciIHN3YXBwZWQ9Im5vIi8+CiAgICA8Y2hpbGQgaW50
ZXJuYWwtY2hpbGQ9InZib3giPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgIDxjaGls
ZCBpbnRlcm5hbC1jaGlsZD0iYWN0aW9uX2FyZWEiPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
QnV0dG9uQm94IiBpZD0iYnV0dG9ucyI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+ZW5kPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGF5b3V0X3N0eWxlIj5lbmQ8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0i
c3Bpbm5lci1ib3giPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrQnV0dG9uQm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3Jk
ZXJfd2lkdGgiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFj
aW5nIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iY2FuY2VsLWJ1dHRvbiI+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+X0NhbmNlbDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX3Vu
ZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJj
bGlja2VkIiBoYW5kbGVyPSJvbl9jYW5jZWwiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJub25faG9tb2dlbmVvdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJqb2luLXJv
b20tYnV0dG9uIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5z
bGF0YWJsZT0ieWVzIj5fQ29udGludWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ic2Vuc2l0aXZlIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2VfdW5kZXJsaW5lIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaW1hZ2VfcG9zaXRpb24iPnJpZ2h0PC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9u
X2pvaW4iIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwi
PkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFja190
eXBlIj5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3Np
dGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im5vbl9o
b21vZ2VuZW91cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
PHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+RmFsc2U8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBhY2tfdHlwZSI+ZW5kPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJub3RpZmljYXRpb24tYXJlYS1ib3giPgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJtYXJnaW5fYm90dG9tIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJqb2luLXJvb20tY29udGVudCI+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImJvcmRlcl93aWR0aCI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3Jp
ZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNw
YWNpbmciPjU8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrR3JpZCIgaWQ9ImNvbnRyb2xzIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
aGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93
X3NwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1
bW5fc3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InJvb20tbmFtZS1sYWJlbCI+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPmVuZDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Um9v
bTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdF9hdHRhY2giPjA8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4x
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrRW50cnkiIGlkPSJyb29tLW5hbWUtZW50cnkiPgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBsYWNlaG9sZGVyX3RleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5FbnRlciB0aGUg
cm9vbSBuYW1lPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNoYW5n
ZWQiIGhhbmRsZXI9Im9uX3Jvb21uYW1lX2NoYW5nZWQiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAg
ICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25fam9pbiIgc3dhcHBlZD0i
bm8iLz4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5n
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcF9hdHRhY2giPjE8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtM
YWJlbCIgaWQ9ImFjY291bnRMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJoYWxpZ24iPmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+QWNjb3VudDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3RpZnkiPnJpZ2h0PC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InRvcF9hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDb21ib0JveCIgaWQ9ImFjY291
bnRzIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0NlbGxSZW5kZXJlclRleHQiIGlkPSJhY2NvdW50LW5hbWUtcmVuZGVyZWQi
Lz4KICAgICAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgICAg
ICA8YXR0cmlidXRlIG5hbWU9InRleHQiPjA8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAgICAg
IDwvYXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
ICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImxlZnRfYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idG9wX2F0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
IDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0iY2hhdC1zZXJ2aWNl
LWxhYmVsIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0
YWJsZT0ieWVzIj5XaGVyZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGVmdF9hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ0b3BfYXR0YWNoIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iY2hhdC1zZXJ2aWNlcy1ib3giPgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
Q29tYm9Cb3hUZXh0IiBpZD0iY2hhdC1zZXJ2aWNlcy1saXN0Ij4KICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFzX2VudHJ5Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJlbnRyeSI+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrRW50cnkiIGlkPSJjaGF0LXNlcnZpY2VzLWVudHJ5
Ij4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwbGFj
ZWhvbGRlcl90ZXh0IiB0cmFuc2xhdGFibGU9InllcyI+RW50ZXIgb3Igc2VsZWN0IHRoZSBzZXJ2aWNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZh
dGUiIGhhbmRsZXI9Im9uX2pvaW4iIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAg
ICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdF9hdHRhY2giPjE8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4yPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAg
ICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hp
bGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5h
bWU9Im11YyIvPgogICAgICA8Y2xhc3MgbmFtZT0iY295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmpl
Y3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCPublicRoomsDialog.xml": {
		local:   "definitions/MUCPublicRoomsDialog.xml",
		size:    18096,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJwdWJsaWMtcm9vbXMtZGlh
bG9nIj4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNoYXQgcm9v
bXM8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImRlZmF1bHRfd2lkdGgiPjYwMDwvcHJvcGVy
dHk+CiAgICA8cHJvcGVydHkgbmFtZT0iZGVmYXVsdF9oZWlnaHQiPjQwMDwvcHJvcGVydHk+CiAgICA8
cHJvcGVydHkgbmFtZT0iZGVzdHJveV93aXRoX3BhcmVudCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJv
cGVydHkgbmFtZT0idHlwZV9oaW50Ij5ub3JtYWw8L3Byb3BlcnR5PgogICAgPHNpZ25hbCBuYW1lPSJj
bG9zZSIgaGFuZGxlcj0ib25fY2xvc2Vfd2luZG93IiBzd2FwcGVkPSJubyIvPgogICAgPHNpZ25hbCBu
YW1lPSJkZXN0cm95IiBoYW5kbGVyPSJvbl9jbG9zZV93aW5kb3ciIHN3YXBwZWQ9Im5vIi8+CiAgICA8
Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9InZib3giPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlk
PSJsYXlvdXQiPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAg
ICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAgICAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a0J1dHRvbkJveCIgaWQ9ImJ1dHRvbnMiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxp
Z24iPmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmVuZDwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCdXR0b25Cb3giPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTI8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjY8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCdXR0b24iIGlkPSJidXR0b25fY2FuY2VsIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiPl9DYW5jZWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iY2FuX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InVzZV91bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fY2FuY2Vs
IiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrQnV0dG9uIiBpZD0iam9pbi1yb29tLWJ1dHRvbiI+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Sm9pbjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX3VuZGVybGlu
ZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2Vk
IiBoYW5kbGVyPSJvbl9qb2luIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAg
ICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJub25faG9tb2dlbmVvdXMiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBh
bmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0
eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9Im5vdGlmaWNhdGlvbnMtYXJlYSI+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8c3R5bGU+CiAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9Im5vdGlmaWNhdGlvbnMiLz4K
ICAgICAgICAgICAgPC9zdHlsZT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrR3Jp
ZCIgaWQ9ImdyaWQiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgiPjEyPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJvd19zcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1bW5fc3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJh
Y2NvdW50c0xhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPmVuZDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9Inll
cyI+QWNjb3VudDo8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imp1c3Rp
ZnkiPnJpZ2h0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wX2F0dGFjaCI+MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ29tYm9Cb3giIGlkPSJhY2Nv
dW50cyI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFzX2ZvY3VzIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Nl
bGxSZW5kZXJlclRleHQiIGlkPSJhY2NvdW50LW5hbWUtcmVuZGVyZWQiLz4KICAgICAgICAgICAgICAg
ICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ0ZXh0Ij4w
PC9hdHRyaWJ1dGU+CiAgICAgICAgICAgICAgICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdF9hdHRhY2giPjE8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcF9hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9InJlZnJlc2gtYnV0dG9uIj4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkxp
c3QgYWxsIHJvb21zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNp
YmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZl
c19kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNr
ZWQiIGhhbmRsZXI9Im9uX3JlZnJlc2giIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGVmdF9hdHRhY2giPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRv
cF9hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
PC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0xhYmVsIiBpZD0iY3VzdG9tLXNlcnZpY2UtbGFiZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFi
ZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5TZXJ2aWNlOjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+cmlnaHQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxlZnRfYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0
b3BfYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtFbnRyeSIgaWQ9ImN1c3RvbS1zZXJ2aWNlLWVudHJ5Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cGxhY2Vob2xkZXJfdGV4dCI+Y2hhdC1zZXJ2aWNlLmV4YW1wbGUub3JnPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUiIGhhbmRsZXI9Im9uX2N1c3RvbV9zZXJ2aWNl
IiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnRfYXR0YWNoIj4xPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4xPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxj
aGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJsaXN0LXJvb21z
LWJ1dHRvbiI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJs
ZT0ieWVzIj5MaXN0IHJvb21zIGluIHNlcnZpY2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxz
aWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fY3VzdG9tX3NlcnZpY2UiIHN3YXBwZWQ9Im5v
Ii8+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdF9hdHRhY2giPjI8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InRvcF9hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAg
IDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrQm94IiBpZD0icHVibGljLXJvb21zLXRyZWUtY29udGFpbmVyIj4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxk
PgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1NlcGFyYXRvciIgaWQ9InRvcC1zZXBhcmF0
b3IiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+
CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtPdmVybGF5Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNp
YmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5k
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2ZXhwYW5kIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlz
aWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imhl
eHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
ZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
b3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibWFyZ2luX3N0YXJ0Ij4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9Im1hcmdpbl9lbmQiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtTY3JvbGxlZFdpbmRv
dyIgaWQ9InB1YmxpYy1yb29tcy12aWV3Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2ZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InNoYWRvd190eXBlIj5ldGNoZWQtaW48L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrVHJlZVZpZXciIGlkPSJwdWJsaWMtcm9vbXMtdHJlZSI+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VhcmNoX2NvbHVtbiI+MDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZW5hYmxlX3Ry
ZWVfbGluZXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InRvb2x0aXBfY29sdW1uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxzaWduYWwgbmFtZT0icm93LWFjdGl2YXRlZCIgaGFuZGxlcj0ib25fYWN0aXZhdGVfcm9v
bV9yb3ciIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQgaW50
ZXJuYWwtY2hpbGQ9InNlbGVjdGlvbiI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a1RyZWVTZWxlY3Rpb24iPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxzaWduYWwgbmFtZT0iY2hhbmdlZCIgaGFuZGxlcj0ib25fc2VsZWN0aW9uX2NoYW5nZWQiIHN3YXBw
ZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrVHJlZVZpZXdD
b2x1bW4iIGlkPSJuYW1lLWNvbHVtbiI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InJlc2l6YWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNpemluZyI+YXV0b3NpemU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaXhlZF93aWR0aCI+MjAw
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bWluX3dpZHRoIj4xNTA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ0cnVlIj5OYW1lPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2xp
Y2thYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ic29ydF9pbmRpY2F0b3IiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzb3J0X2NvbHVtbl9pZCI+MTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2VsbFJlbmRlcmVyVGV4dCIgaWQ9Im5hbWUt
Y29sdW1uLXJlbmRlcmVkIi8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8YXR0cmli
dXRlcz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJt
YXJrdXAiPjE8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvYXR0
cmlidXRlcz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVWaWV3Q29sdW1uIiBpZD0iamlkLWNvbHVt
biI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlc2l6YWJs
ZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InNpemluZyI+YXV0b3NpemU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaXhlZF93aWR0aCI+MTUwPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWluX3dpZHRoIj41MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0
cmFuc2xhdGFibGU9InRydWUiPklEPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iY2xpY2thYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic29ydF9pbmRpY2F0b3IiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzb3J0
X2NvbHVtbl9pZCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2Vs
bFJlbmRlcmVyVGV4dCIgaWQ9ImppZC1jb2x1bW4tcmVuZGVyZWQiLz4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8YXR0cmlidXRlIG5hbWU9InRleHQiPjA8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVWaWV3
Q29sdW1uIiBpZD0iZGVzY3JpcHRpb24tY29sdW1uIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0icmVzaXphYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2l6aW5nIj5hdXRvc2l6ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpeGVkX3dp
ZHRoIj4yMDA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJtaW5fd2lkdGgiPjEwMDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InRydWUiPkRlc2NyaXB0aW9u
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iY2xpY2thYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic29ydF9pbmRpY2F0b3IiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzb3J0X2NvbHVtbl9p
ZCI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2VsbFJlbmRlcmVy
VGV4dCIgaWQ9ImRlc2NyaXB0aW9uLWNvbHVtbi1yZW5kZXJlZCIvPgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxhdHRyaWJ1dGUgbmFtZT0ibWFya3VwIj4zPC9hdHRyaWJ1dGU+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtUcmVlVmll
d0NvbHVtbiIgaWQ9Im9jY3VwYW50cy1jb2x1bW4iPgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJyZXNpemFibGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzaXppbmciPmF1dG9zaXplPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZml4ZWRfd2lk
dGgiPjEwMDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9Im1pbl93aWR0aCI+NTA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ0cnVlIj5QZW9wbGU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjbGlja2Fi
bGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJzb3J0X2luZGljYXRvciI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNvcnRfY29sdW1uX2lkIj40PC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDZWxsUmVuZGVyZXJUZXh0IiBpZD0ib2NjdXBhbnRz
LWNvbHVtbi1yZW5kZXJlZCIvPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGF0dHJp
YnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0i
dGV4dCI+NDwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9hdHRy
aWJ1dGVzPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icGFkZGluZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icGFzc190aHJvdWdoIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZCB0eXBlPSJv
dmVybGF5Ij4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrT3ZlcmxheSIgaWQ9InNw
aW5uZXItb3ZlcmxheSI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InNwaW5uZXItYm94Ij4KICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX3JlcXVlc3QiPjMyPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhlaWdodF9yZXF1ZXN0Ij4zMjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+
Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZh
bGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iaW5kZXgiPi0xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
ICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImluZGV4Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtTZXBhcmF0b3IiIGlkPSJib3R0
b20tc2VwYXJhdG9yIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8c3R5
bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJtdWMiLz4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAg
ICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomAffiliationDialog.xml": {
		local:   "definitions/MUCRoomAffiliationDialog.xml",
		size:    30156,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJhZmZpbGlhdGlvbi1kaWFs
b2ciPgogICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX3JlcXVlc3QiPjYwMDwvcHJvcGVydHk+CiAgICA8
cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5DaGFuZ2UgcG9zaXRpb248L3By
b3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InJlc2l6YWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAg
PHByb3BlcnR5IG5hbWU9Im1vZGFsIj5UcnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0
eXBlX2hpbnQiPmRpYWxvZzwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0idXJnZW5jeV9oaW50
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgIDxzaWduYWwgbmFtZT0ia2V5LXByZXNzLWV2ZW50IiBoYW5kbGVy
PSJvbl9rZXlfcHJlc3MiIHN3YXBwZWQ9Im5vIi8+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9InZi
b3giPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJhZmZpbGlhdGlvbi1jb250ZW50LWJv
eCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+
CiAgICAgICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJhY3Rpb25fYXJlYSI+CiAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJhZmZpbGlhdGlvbi1idXR0b25zIj4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxheW91dF9zdHlsZSI+ZXhwYW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giPgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlk
PSJhZmZpbGlhdGlvbi1jYW5jZWwtYnV0dG9uIj4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Q2FuY2VsPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIg
aGFuZGxlcj0ib25fY2FuY2VsIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAgICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1
dHRvbiIgaWQ9ImFmZmlsaWF0aW9uLWFwcGx5LWJ1dHRvbiI+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFwcGx5PC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2Vuc2l0aXZlIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZl
c19kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBu
YW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9hcHBseSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAg
ICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBhY2tfdHlwZSI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBv
c2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgog
ICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9ImFmZmlsaWF0
aW9uLWNvbnRlbnQiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtC
b3giPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl90b3AiPjEyPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
Qm94IiBpZD0iYWZmaWxpYXRpb24tdHlwZSI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iYWZmaWxp
YXRpb24tdHlwZS1pbmZvIj4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZp
c2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwi
IGlkPSJhZmZpbGlhdGlvbi10aXRsZS1sYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5D
aGFuZ2UgcG9zaXRpb248L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9ImFmZmlsaWF0aW9uLW9wdGlvbnMtZGlz
YWJsZWQtbGFiZWwiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZp
c2libGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFzIGFuIGFkbWluaXN0cmF0b3IgeW91IGNh
bid0IG1ha2Ugb3RoZXJzIGludG8gYWRtaW5pc3RyYXRvcnMgb3Igb3duZXJzLjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4wPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8Y2xhc3MgbmFtZT0iaGVscC10ZXh0Ii8+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8L3N0eWxlPgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtGcmFtZSIgaWQ9ImFmZmlsaWF0aW9uLW9wdGlvbnMtZnJhbWUiPgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsX3hhbGlnbiI+MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWxfeWFsaWdu
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJz
aGFkb3dfdHlwZSI+aW48L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxk
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0Qm94IiBp
ZD0iYWZmaWxpYXRpb24tb3B0aW9ucy1saXN0Ij4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGlvbl9tb2RlIj5ub25lPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYWN0aXZhdGVf
b25fc2luZ2xlX2NsaWNrIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrTGlzdEJveFJvdyIgaWQ9ImFmZmlsaWF0aW9uLW9wdGlvbi1vd25lciI+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImFj
dGl2YXRhYmxlIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJzcGFjaW5nIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrUmFkaW9CdXR0b24iIGlkPSJhZmZpbGlhdGlvbi1vd25lciI+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0
cmFuc2xhdGFibGU9InllcyI+T3duZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cmVjZWl2ZXNfZGVmYXVsdCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2VfdW5kZXJsaW5lIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZHJh
d19pbmRpY2F0b3IiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJncm91cCI+YWZmaWxpYXRpb24tbm9uZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1l
PSJ0b2dnbGVkIiBoYW5kbGVyPSJvbl90b2dnbGVkIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRh
YmxlPSJ5ZXMiPkFuIG93bmVyIGlzIGFsbG93ZWQgdG8gY2hhbmdlIHRoZSByb29tIGNvbmZpZ3VyYXRp
b24gYW5kIGRlc3Ryb3kgdGhlIHJvb20sIGluIGFkZGl0aW9uIHRvIGFsbCBhZG1pbmlzdHJhdG9yIHBy
aXZpbGVnZXMuPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX2NoYXJzIj4wPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibWF4X3dpZHRoX2NoYXJzIj41MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxpbmVzIj4yPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
eGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8c3R5bGU+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8Y2xhc3MgbmFtZT0iaGVscC10ZXh0Ii8+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9zdHlsZT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGlzdEJveFJvdyI+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImFjdGl2YXRhYmxlIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5GYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtTZXBhcmF0b3IiPgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
PC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0Qm94Um93IiBpZD0iYWZm
aWxpYXRpb24tb3B0aW9uLWFkbWluIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYWN0aXZhdGFibGUiPkZhbHNlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVj
dGFibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgi
PjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjY8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtSYWRpb0J1dHRv
biIgaWQ9ImFmZmlsaWF0aW9uLWFkbWluIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5BZG1pbmlz
dHJhdG9yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQi
PkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZHJhd19pbmRpY2F0b3IiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJncm91cCI+YWZmaWxpYXRpb24t
bm9uZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
PHNpZ25hbCBuYW1lPSJ0b2dnbGVkIiBoYW5kbGVyPSJvbl90b2dnbGVkIiBzd2FwcGVkPSJubyIvPgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJl
bCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFuIGFkbWluaXN0cmF0b3IgaXMgcmVzcG9uc2libGUgZm9yIG1l
bWJlcnNoaXBzIGZvciB0aGUgcm9vbSwgYW5kIGF1dG9tYXRpY2FsbHkgYmVjb21lcyBhIG1vZGVyYXRv
ciB3aGVuIGVudGVyaW5nIGl0LjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXAiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9j
aGFycyI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im1heF93aWR0aF9jaGFycyI+NTA8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsaW5lcyI+Mjwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9ImhlbHAtdGV4dCIvPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xpc3RC
b3hSb3ciPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJhY3RpdmF0YWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0YWJsZSI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2VwYXJhdG9yIj4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNp
YmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGlzdEJveFJv
dyIgaWQ9ImFmZmlsaWF0aW9uLW9wdGlvbi1tZW1iZXIiPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJhY3RpdmF0YWJsZSI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ic2VsZWN0YWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJv
cmRlcl93aWR0aCI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+
NjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a1JhZGlvQnV0dG9uIiBpZD0iYWZmaWxpYXRpb24tbWVtYmVyIj4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0i
eWVzIj5NZW1iZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVm
YXVsdCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50
ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJkcmF3X2luZGljYXRvciI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imdyb3VwIj5hZmZpbGlh
dGlvbi1ub25lPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8c2lnbmFsIG5hbWU9InRvZ2dsZWQiIGhhbmRsZXI9Im9uX3RvZ2dsZWQiIHN3YXBwZWQ9Im5v
Ii8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIj4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+QSBwZXJzb24gd2l0aCB0aGlzIHBvc2l0aW9uIGNhbiBl
bnRlciBhIG1lbWJlcnMtb25seSByb29tLCBhbmQgY2FuIGFsc28gaW1tZWRpYXRlbHkgc3BlYWsgaW4g
YWxsIHJvb21zIHRoZXkgYXJlIGEgbWVtYmVyIG9mLjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXAiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ3aWR0aF9jaGFycyI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1heF93aWR0aF9jaGFycyI+NTA8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsaW5lcyI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHN0eWxlPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9ImhlbHAt
dGV4dCIvPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvc3R5bGU+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0xpc3RCb3hSb3ciPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJhY3RpdmF0YWJsZSI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0YWJs
ZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
U2VwYXJhdG9yIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTGlzdEJveFJvdyIgaWQ9ImFmZmlsaWF0aW9uLW9wdGlvbi1ub25lIj4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYWN0aXZh
dGFibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJib3JkZXJfd2lkdGgiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InNwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtSYWRpb0J1dHRvbiIgaWQ9ImFmZmlsaWF0aW9uLW5vbmUiPgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNs
YXRhYmxlPSJ5ZXMiPk5vbmU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2
ZXNfZGVmYXVsdCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWdu
Ij5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJhY3RpdmUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJkcmF3X2luZGljYXRvciI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
PHNpZ25hbCBuYW1lPSJ0b2dnbGVkIiBoYW5kbGVyPSJvbl90b2dnbGVkIiBzd2FwcGVkPSJubyIvPgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJl
bCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPldpdGhvdXQgYSBwb3NpdGlvbiwgYSBwZXJzb24gY2FuJ3QgZW50
ZXIgYSBtZW1iZXJzLW9ubHkgcm9vbS48L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id2lk
dGhfY2hhcnMiPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXhfd2lkdGhfY2hhcnMiPjUwPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGluZXMi
PjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxzdHlsZT4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjbGFzcyBuYW1lPSJoZWxwLXRleHQiLz4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3N0eWxlPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3Np
dGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iYWZmaWxpYXRpb24tcmVhc29u
Ij4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNh
bDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjY8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0iYWZmaWxpYXRpb24tcmVhc29uLWxhYmVsIj4KICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJs
ZT0ieWVzIj5IZXJlIHlvdSBjYW4gcHJvdmlkZSBhbiBvcHRpb25hbCByZWFzb24gZm9yIHRoZSBjaGFu
Z2UuIEV2ZXJ5b25lIGluIHRoZSByb29tIHdpbGwgc2VlIHRoaXMgcmVhc29uLjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX2NoYXJzIj4wPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1heF93aWR0aF9jaGFy
cyI+NDA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGlu
ZXMiPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFs
aWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Inlh
bGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1Njcm9sbGVkV2luZG93Ij4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaHNjcm9sbGJhcl9wb2xp
Y3kiPm5ldmVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InNoYWRvd190eXBlIj5pbjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4K
ICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtUZXh0VmlldyIgaWQ9ImFm
ZmlsaWF0aW9uLXJlYXNvbi1lbnRyeSI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icGl4ZWxzX2luc2lkZV93cmFwIj4zPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfbWFyZ2luIj4zPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3R0b21fbWFyZ2luIj4z
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpbmRl
bnQiPjY8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImFjY2VwdHNfdGFiIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAg
ICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjQ8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFkZGlu
ZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAg
ICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8
L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2Jq
ZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAg
ICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomCloseWindowConfirm.xml": {
		local:   "definitions/MUCRoomCloseWindowConfirm.xml",
		size:    6625,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtXaW5kb3ciIGlkPSJyb29tLWNsb3NlLWNvbmZp
cm0td2luZG93Ij4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij40MDA8L3Byb3BlcnR5
PgogICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+Q2xvc2Ugcm9vbTwv
cHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0icmVzaXphYmxlIj5GYWxzZTwvcHJvcGVydHk+CiAg
ICA8cHJvcGVydHkgbmFtZT0ibW9kYWwiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9
InR5cGVfaGludCI+ZGlhbG9nPC9wcm9wZXJ0eT4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrQm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTI8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2li
bGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8
L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrQm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij40Mjwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGVpZ2h0X3JlcXVlc3QiPjQy
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPnN0YXJ0PC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtJbWFnZSIgaWQ9InJvb20tY2xvc2UtY29uZmlybS1pY29uIj4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBpeGVsX3NpemUiPjQyPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9uYW1lIj5kaWFsb2ctcXVlc3Rpb248L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpY29uX3NpemUiPjY8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCb3giPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNh
bDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0xhYmVsIiBpZD0icm9vbS1jbG9zZS1jb25maXJtLW1lc3NhZ2UiPgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5EbyB5b3Ugd2Fu
dCB0byBsZWF2ZSB0aGUgcm9vbT88L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4
cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJm
aWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9z
aXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAg
ICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtDaGVja0J1dHRvbiIgaWQ9InJvb20tY2xvc2UtY29uZmlybS1sZWF2ZS1jaGVj
a2JveCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmb2N1c19vbl9jbGlj
ayI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNl
aXZlc19kZWZhdWx0Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImRyYXdfaW5kaWNhdG9yIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
c2lnbmFsIG5hbWU9InRvZ2dsZWQiIGhhbmRsZXI9Im9uX2xlYXZlX3Jvb21fY2hlY2tfY2hhbmdlZCIg
c3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+WWVzLCBsZWF2ZSB0aGUgcm9v
bTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ4YWxpZ24i
PjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWdu
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAg
ICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hp
bGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBhZGRpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2No
aWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCI+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImxheW91dF9zdHlsZSI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxk
PgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9InJvb20tY2xvc2UtY2Fu
Y2VsLWJ1dHRvbiI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0
YWJsZT0ieWVzIj5DYW5jZWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJl
Y2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1l
PSJjbGlja2VkIiBoYW5kbGVyPSJvbl9jYW5jZWwiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9z
aXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im5vbl9ob21v
Z2VuZW91cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAg
ICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrQnV0dG9uIiBpZD0icm9vbS1jbG9zZS1jb25maXJtLWJ1dHRvbiI+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5DbG9zZSBSb29tPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fY29u
ZmlybSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibm9uX2hvbW9nZW5lb3VzIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4K
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwv
Y2hpbGQ+CiAgICA8c3R5bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9zdHlsZT4K
ICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomConfigAssistant.xml": {
		local:   "definitions/MUCRoomConfigAssistant.xml",
		size:    778,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtBc3Npc3RhbnQiIGlkPSJyb29tLWNvbmZpZy1h
c3Npc3RhbnQiPgogICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX3JlcXVlc3QiPjQwMDwvcHJvcGVydHk+
CiAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5Sb29tIGNvbmZpZ3Vy
YXRpb248L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im1vZGFsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aW5kb3dfcG9zaXRpb24iPmNlbnRlci1vbi1wYXJlbnQ8L3Byb3Bl
cnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImRlZmF1bHRfd2lkdGgiPjYwMDwvcHJvcGVydHk+CiAgICA8
cHJvcGVydHkgbmFtZT0iZGVmYXVsdF9oZWlnaHQiPjUwMDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkg
bmFtZT0iZGVzdHJveV93aXRoX3BhcmVudCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0idXNlX2hlYWRlcl9iYXIiPjE8L3Byb3BlcnR5PgogICAgPHNpZ25hbCBuYW1lPSJjYW5jZWwiIGhh
bmRsZXI9Im9uX2NhbmNlbCIgc3dhcHBlZD0ibm8iLz4KICAgIDxzaWduYWwgbmFtZT0icHJlcGFyZSIg
aGFuZGxlcj0ib25fcGFnZV9jaGFuZ2VkIiBzd2FwcGVkPSJubyIvPgogICAgPHNpZ25hbCBuYW1lPSJh
cHBseSIgaGFuZGxlcj0ib25fYXBwbHkiIHN3YXBwZWQ9Im5vIi8+CiAgICA8c3R5bGU+CiAgICAgIDxj
bGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9zdHlsZT4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomConfigAssistantNavigation.xml": {
		local:   "definitions/MUCRoomConfigAssistantNavigation.xml",
		size:    710,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1hc3Npc3Rh
bnQtbmF2aWdhdGlvbi1jb250ZW50Ij4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij4x
NTA8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJv
cGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xpc3RCb3giIGlkPSJyb29t
LWNvbmZpZy1hc3Npc3RhbnQtbmF2aWdhdGlvbi1saXN0Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHNpZ25hbCBuYW1lPSJyb3ctYWN0aXZhdGVk
IiBoYW5kbGVyPSJvbl9yb3dfYWN0aXZhdGVkIiBzd2FwcGVkPSJubyIvPgogICAgICAgIDxzaWduYWwg
bmFtZT0icm93LXNlbGVjdGVkIiBoYW5kbGVyPSJvbl9yb3dfc2VsZWN0ZWQiIHN3YXBwZWQ9Im5vIi8+
CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhw
YW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3By
b3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
IDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomConfigAssistantNavigationItem.xml": {
		local:   "definitions/MUCRoomConfigAssistantNavigationItem.xml",
		size:    1276,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0Qm94Um93IiBpZD0icm9vbS1jb25maWct
YXNzaXN0YW50LW5hdmlnYXRpb24taXRlbS1yb3ciPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25fYWN0
aXZhdGUiIHN3YXBwZWQ9Im5vIi8+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Jv
eCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgiPjg8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4K
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdlIiBpZD0icm9vbS1jb25maWctYXNzaXN0YW50
LW5hdmlnYXRpb24taXRlbS1pY29uIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9s
ZWZ0Ij40PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGls
ZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InJv
b20tY29uZmlnLWFzc2lzdGFudC1uYXZpZ2F0aW9uLWl0ZW0tbGFiZWwiPgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
PHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICA8L29iamVj
dD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomConfigAssistantePage.xml": {
		local:   "definitions/MUCRoomConfigAssistantePage.xml",
		size:    591,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtTY3JvbGxlZFdpbmRvdyIgaWQ9InJvb20tY29u
ZmlnLWFzc2lzdGFudC1wYWdlIj4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJoc2Nyb2xsYmFyX3BvbGljeSI+bmV2ZXI8L3Byb3BlcnR5
PgogICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5
IG5hbWU9InZleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtWaWV3cG9ydCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzaGFkb3dfdHlwZSI+bm9uZTwvcHJvcGVydHk+
CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0icm9vbS1j
b25maWctYXNzaXN0YW50LXBhZ2UtY29udGVudCI+PC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAg
ICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomConfigFieldPositions.xml": {
		local:   "definitions/MUCRoomConfigFieldPositions.xml",
		size:    12811,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1wb3NpdGlv
bnMtY29udGVudCI+CiAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgIDxwcm9w
ZXJ0eSBuYW1lPSJzcGFjaW5nIj4yNDwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a0JveCIgaWQ9InJvb20tY29uZmlnLXBvc2l0aW9ucy1mb3JtIj4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlv
biI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4yNDwv
cHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4yNDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlv
biI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ic3BhY2luZyI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAg
ICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InJvb20tY29u
ZmlnLXBvc2l0aW9uLWhlYWRlciI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlk
PSJyb29tLWNvbmZpZy1wb3NpdGlvbi1kZXNjcmlwdGlvbiI+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5Qb3NpdGlvbiBkZXNj
cmlwdGlvbiBoZXJlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8Y2xhc3MgbmFtZT0iaGVscC10ZXh0Ii8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
L3N0eWxlPgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0J1dHRvbkJveCI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNw
YWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxheW91dF9zdHlsZSI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0i
cm9vbS1jb25maWctcG9zaXRpb24tYWRkIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2VfdW5kZXJsaW5lIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjM8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdlIj4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9uYW1l
Ij5saXN0LWFkZC1zeW1ib2xpYzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5G
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIj4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5BZGQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im5vbl9o
b21vZ2VuZW91cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9u
IiBpZD0icm9vbS1jb25maWctcG9zaXRpb24tcmVtb3ZlIj4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZW5zaXRpdmUiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1
c2VfdW5kZXJsaW5lIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxj
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InNwYWNpbmciPjM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdl
Ij4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2li
bGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaWNvbl9uYW1lIj5saXN0LXJlbW92ZS1zeW1ib2xpYzwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0xhYmVsIiBpZD0icm9vbS1jb25maWctcG9zaXRpb24tcmVtb3ZlLWxhYmVsIj4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5SZW1vdmU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRp
b24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im5vbl9ob21vZ2VuZW91cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAg
ICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJw
b3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8
L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
Qm94IiBpZD0icm9vbS1jb25maWctcG9zaXRpb25zLWxpc3QtY29udGVudCI+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2Nyb2xsZWRX
aW5kb3ciPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZleHBhbmQiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvdmVybGF5LXNjcm9s
bGluZyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJt
YXgtY29udGVudC1oZWlnaHQiPjgwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibWluLWNvbnRlbnQtaGVpZ2h0Ij44MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InNoYWRvd190eXBlIj5ldGNoZWQtaW48L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a1RyZWVWaWV3IiBpZD0icm9vbS1jb25maWctcG9zaXRpb25zLWxpc3QiPiAgICAgICAgICAgICAgICAg
ICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZWFkZXJz
X3Zpc2libGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImVuYWJsZV9zZWFyY2giPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InNlYXJjaF9jb2x1bW4iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2hvd19leHBhbmRlcnMiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImVuYWJsZV9ncmlkX2xpbmVzIj5o
b3Jpem9udGFsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkIGludGVybmFs
LWNoaWxkPSJzZWxlY3Rpb24iPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a1RyZWVTZWxlY3Rpb24iPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9Im1vZGUiPm11bHRpcGxlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVW
aWV3Q29sdW1uIiBpZD0icm9vbS1jb25maWctcG9zaXRpb24tamlkLWNvbHVtbiI+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5B
Y2NvdW50IGFkZHJlc3M8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxk
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDZWxsUmVuZGVy
ZXJUZXh0IiBpZD0icm9vbS1jb25maWctcG9zaXRpb24tamlkLXRleHQtcmVuZGVyZXIiPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJlZGl0YWJsZSI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJlZGl0ZWQi
IGhhbmRsZXI9Im9uX2ppZF9lZGl0ZWQiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRl
cz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9InRleHQiPjA8
L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5G
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3Bl
cnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0Pgog
ICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9j
aGlsZD4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomConfigFormFieldAdvancedOptionsContainer.xml": {
		local:   "definitions/MUCRoomConfigFormFieldAdvancedOptionsContainer.xml",
		size:    1847,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmll
bnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtGcmFtZSI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbF94YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzaGFkb3dfdHlwZSI+bm9uZTwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxk
PgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQWxpZ25tZW50Ij4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrRXhwYW5kZXIiIGlkPSJyb29tLWNvbmZpZy1maWVsZC1h
ZHZhbmNlZC1leHBhbmRlciI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWFkdmFuY2VkLW9wdGlvbnMt
Y29udGVudC1ib3giPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRp
b24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bWFyZ2luX3RvcCI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJzcGFjaW5nIj4yNDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZCB0eXBlPSJsYWJlbCI+CiAgICAg
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0icm9vbS1jb25maWctYWR2YW5j
ZWQtZmllbGQtbGFiZWwiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwi
IHRyYW5zbGF0YWJsZT0ieWVzIj5BZHZhbmNlZCBvcHRpb25zPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8c3R5bGU+CiAgICAgICAgICAgICAgICAgICAgICA8Y2xhc3MgbmFtZT0iZXhwYW5kZXIt
dGV4dCIvPgogICAgICAgICAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+ICAgICAgCiAg
ICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgIDwv
cGFja2luZz4KICAgIDwvY2hpbGQ+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomConfigFormFieldBoolean.xml": {
		local:   "definitions/MUCRoomConfigFormFieldBoolean.xml",
		size:    4017,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmll
bnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMaXN0Qm94Um93IiBpZD0icm9vbS1jb25maWctYm9vbGVhbi1maWVsZCI+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9IndpZHRoX3JlcXVlc3QiPjEwMDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImhlaWdodF9yZXF1ZXN0Ij44MDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJhY3RpdmF0YWJs
ZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrR3Jp
ZCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWJvb2xlYW4tZ3JpZCI+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRl
ci13aWR0aCI+MTg8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93LXNwYWNp
bmciPjY8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY29sdW1uLXNwYWNpbmci
PjI0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a1N3aXRjaCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWJvb2xlYW4iPgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1h
dHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRh
Y2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhlaWdodCI+Mjwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiIGlkPSJpY29u
LWltYWdlIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+RmFsc2U8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50
ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJub19zaG93X2Fs
bCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5G
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8
L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtMYWJlbCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxhYmVsIj4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+cmlnaHQ8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3Bh
Y2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0
YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNo
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIg
aWQ9InJvb20tY29uZmlnLWZpZWxkLWRlc2NyaXB0aW9uIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ3aWR0aC1yZXF1ZXN0Ij4zMDA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9Im5vX3Nob3dfYWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ3cmFwIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGln
biI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8c3R5bGU+CiAgICAgICAgICAgICAgICAgIDxj
bGFzcyBuYW1lPSJjb25maWctZmllbGQtaGVscCIvPgogICAgICAgICAgICAgICAgPC9zdHlsZT4KICAg
ICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGls
ZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomConfigFormFieldBooleanContainer.xml": {
		local:   "definitions/MUCRoomConfigFormFieldBooleanContainer.xml",
		size:    784,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmll
bnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtGcmFtZSI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbF94YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzaGFkb3dfdHlwZSI+aW48L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4K
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xpc3RCb3giIGlkPSJyb29tLWNvbmZpZy1ib29sZWFu
LWZpZWxkcy1jb250ZW50Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0aW9uX21vZGUiPm5vbmU8
L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYWN0aXZhdGVfb25fc2luZ2xlX2Ns
aWNrIj5GYWxzZTwvcHJvcGVydHk+ICAgICAgICAgICAKICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
IDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICA8L29iamVjdD4KPC9pbnRlcmZh
Y2U+Cg==
`,
	},

	"/definitions/MUCRoomConfigFormFieldLanguage.xml": {
		local:   "definitions/MUCRoomConfigFormFieldLanguage.xml",
		size:    2830,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0JveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4zPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9Imljb24taW1hZ2UiPgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJu
b19zaG93X2FsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxhYmVsIj4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVz
dGlmeSI+cmlnaHQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8
L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICA8L3BhY2tp
bmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDb21ib0Jv
eFRleHQiIGlkPSJyb29tLWxhbmd1YWdlLWNvbWJvYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Imhhc19lbnRyeSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJlbnRyeSI+CiAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRyeSIgaWQ9InJvb20tbGFuZ3VhZ2UtZW50cnkiPgogICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNr
aW5nPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICAg
IDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJyb29tLWNvbmZpZy1maWVs
ZC1kZXNjcmlwdGlvbiI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX3JlcXVlc3QiPjMwMDwv
cHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Inhh
bGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVy
dHk+CiAgICAgICAgPHN0eWxlPgogICAgICAgICAgPGNsYXNzIG5hbWU9ImNvbmZpZy1maWVsZC1oZWxw
Ii8+CiAgICAgICAgPC9zdHlsZT4KICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkg
bmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4yPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICA8L29iamVjdD4KPC9p
bnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomConfigFormFieldLinkButton.xml": {
		local:   "definitions/MUCRoomConfigFormFieldLinkButton.xml",
		size:    417,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtMaW5rQnV0dG9uIiBpZD0icm9vbS1jb25maWct
bGluay1idXR0b24tZmllbGQiPgogICAgPHByb3BlcnR5IG5hbWU9ImZvY3VzX29uX2NsaWNrIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+RmFsc2U8L3By
b3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3RhcnQ8L3Byb3BlcnR5PgogICAgPHBy
b3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJy
ZWxpZWYiPm5vbmU8L3Byb3BlcnR5PgogICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZS1saW5rIiBoYW5k
bGVyPSJnb190b19wYWdlIiBzd2FwcGVkPSJubyIvPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomConfigFormFieldList.xml": {
		local:   "definitions/MUCRoomConfigFormFieldList.xml",
		size:    3339,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0JveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4zPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9Imljb24taW1hZ2UiPgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJu
b19zaG93X2FsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxhYmVsIj4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVz
dGlmeSI+cmlnaHQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8
L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICA8L3BhY2tp
bmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InNw
YWNpbmciPjM8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0NvbWJvQm94IiBpZD0icm9vbS1jb25maWctZmllbGQtbGlzdCI+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBvcHVwX2ZpeGVkX3dpZHRoIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJpZF9jb2x1bW4iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2VsbFJlbmRlcmVyVGV4dCIgaWQ9ImxhYmVsIi8+CiAgICAg
ICAgICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9InRleHQi
PjE8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAgICAgICAgIDwvY2hp
bGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0
aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAg
ICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICA8L3Bh
Y2tpbmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJl
bCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWRlc2NyaXB0aW9uIj4KICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+
c3RhcnQ8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJv
cGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im5vX3Nob3dfYWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+cmlnaHQ8L3Byb3BlcnR5PgogICAgICA8L29i
amVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8
L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+
CiAgICA8L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomConfigFormFieldListEntry.xml": {
		local:   "definitions/MUCRoomConfigFormFieldListEntry.xml",
		size:    3556,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0JveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4zPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9Imljb24taW1hZ2UiPgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJu
b19zaG93X2FsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxhYmVsIj4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVz
dGlmeSI+cmlnaHQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8
L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICA8L3BhY2tp
bmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InNw
YWNpbmciPjM8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0NvbWJvQm94VGV4dCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxpc3QiPgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0id2lkdGhfcmVxdWVzdCI+MzAwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJy
ZXNpemVfbW9kZSI+aW1tZWRpYXRlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Imhhc19lbnRyeSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGls
ZD0iZW50cnkiPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0VudHJ5IiBpZD0icm9vbS1j
b25maWctZmllbGQtbGlzdC1lbnRyeSI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaW5w
dXRfcHVycG9zZSI+bnVtYmVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0i
Y2hhbmdlZCIgaGFuZGxlcj0ib25fZmllbGRfZW50cnlfY2hhbmdlIiBzd2FwcGVkPSJubyIvPgogICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNr
aW5nPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICAg
IDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJyb29tLWNvbmZpZy1maWVs
ZC1kZXNjcmlwdGlvbiI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJub19zaG93X2FsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
Imp1c3RpZnkiPnJpZ2h0PC9wcm9wZXJ0eT4KICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBv
c2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICA8L29iamVj
dD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomConfigFormFieldListMulti.xml": {
		local:   "definitions/MUCRoomConfigFormFieldListMulti.xml",
		size:    4194,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0JveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4zPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9Imljb24taW1hZ2UiPgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJu
b19zaG93X2FsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxhYmVsIj4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVz
dGlmeSI+cmlnaHQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8
L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICA8L3BhY2tp
bmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtGcmFtZSI+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsYWJlbC14YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJzaGFkb3ctdHlwZSI+aW48L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a1RyZWVWaWV3IiBpZD0icm9vbS1jb25maWctZmllbGQtdHJlZXZpZXciPgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJoZWFkZXJzX3Zpc2libGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJzZWxlY3Rpb24iPgogICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a1RyZWVTZWxlY3Rpb24iLz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVWaWV3Q29sdW1uIiBp
ZD0icm9vbS1jb25maWctZmllbGQtdHJlZXZpZXctdGV4dC1jb2x1bW4iPgogICAgICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDZWxsUmVuZGVyZXJUb2dn
bGUiIGlkPSJyb29tLWNvbmZpZy1maWVsZC10cmVldmlldy10b2dnbGUtcmVuZGVyZXIiPgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aCI+MzA8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5cGFkIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8c2lnbmFsIG5hbWU9InRvZ2dsZWQiIGhhbmRsZXI9Im9uX2NoZWNrX2NoYW5nZWQiIHN3YXBwZWQ9
Im5vIi8+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8YXR0cmli
dXRlcz4KICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9ImFjdGl2ZSI+MjwvYXR0cmli
dXRlPgogICAgICAgICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtDZWxsUmVuZGVyZXJUZXh0IiBpZD0icm9vbS1jb25maWctZmllbGQtdHJlZXZpZXctdGV4dC1yZW5k
ZXJlciI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InhwYWQiPjEyPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieXBhZCI+NjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8YXR0cmlidXRlcz4KICAg
ICAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9InRleHQiPjE8L2F0dHJpYnV0ZT4KICAgICAg
ICAgICAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJv
cGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0icm9vbS1jb25maWctZmllbGQtZGVzY3JpcHRpb24iPgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij4zMDA8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJub19zaG93X2FsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImhhbGlnbiI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFw
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4wPC9wcm9wZXJ0
eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICA8c3R5
bGU+CiAgICAgICAgICA8Y2xhc3MgbmFtZT0iY29uZmlnLWZpZWxkLWhlbHAiLz4KICAgICAgICA8L3N0
eWxlPgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5Pgog
ICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomConfigFormFieldPassword.xml": {
		local:   "definitions/MUCRoomConfigFormFieldPassword.xml",
		size:    5730,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0dyaWQiPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93LXNwYWNpbmciPjY8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1bW4tc3BhY2luZyI+NjwvcHJvcGVydHk+CiAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ic3BhY2luZyI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9Imljb24taW1hZ2UiPgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibm9fc2hvd19hbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwi
IGlkPSJyb29tLWNvbmZpZy1maWVsZC1sYWJlbCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
aGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFi
ZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5FbnRlciB0aGUgcm9vbSBwYXNzd29yZDwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0VudHJ5IiBpZD0icGFzc3dvcmQtZW50cnkiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhw
YW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2liaWxpdHki
PkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImludmlzaWJsZV9jaGFy
Ij7il488L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaW5wdXRfcHVycG9zZSI+
cGFzc3dvcmQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNoYW5nZWQiIGhhbmRs
ZXI9Im9uX3Bhc3N3b3JkX2NoYW5nZSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+
MDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4xPC9wcm9w
ZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4K
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9InBhc3N3b3JkLXNob3ctYnV0dG9u
Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+U2hv
dzwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX3Nob3df
cGFzc3dvcmRfY2xpY2tlZCIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MTwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4xPC9wcm9wZXJ0eT4K
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNp
YmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhvbW9nZW5lb3Vz
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a0xhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRh
YmxlPSJ5ZXMiPkNvbmZpcm0gdGhlIHJvb20gcGFzc3dvcmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBv
c2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtM
YWJlbCIgaWQ9InBhc3N3b3JkLW1hdGNoLWVycm9yIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2aXNpYmxlIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ibm9fc2hvd19hbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImhhbGlnbiI+R1RLX0FMSUdOX0VORDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5QYXNzd29yZHMgZG9uJ3QgbWF0Y2g8L3By
b3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3Bh
Y2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtFbnRyeSIgaWQ9InBhc3N3b3JkLWNvbmZpcm1hdGlvbi1lbnRyeSI+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJpbGl0eSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iaW52aXNpYmxlX2NoYXIiPuKXjzwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJpbnB1dF9wdXJwb3NlIj5wYXNzd29yZDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxzaWdu
YWwgbmFtZT0iY2hhbmdlZCIgaGFuZGxlcj0ib25fY29uZmlybV9wYXNzd29yZF9jaGFuZ2UiIHN3YXBw
ZWQ9Im5vIi8+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MzwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAg
ICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9w
cm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJyb29tLWNvbmZpZy1maWVsZC1kZXNjcmlwdGlvbiI+CiAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibm9fc2hvd19hbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICA8L29iamVjdD4KICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2No
aWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomConfigFormFieldText.xml": {
		local:   "definitions/MUCRoomConfigFormFieldText.xml",
		size:    2769,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0JveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4zPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9Imljb24taW1hZ2UiPgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJu
b19zaG93X2FsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxhYmVsIj4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVz
dGlmeSI+cmlnaHQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8
L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICA8L3BhY2tp
bmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRyeSIg
aWQ9InJvb20tY29uZmlnLXRleHQtZmllbGQtZW50cnkiPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50
ZXI8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICA8c2lnbmFsIG5hbWU9ImNoYW5nZWQiIGhhbmRsZXI9Im9uX2ZpZWxkX2VudHJ5X2No
YW5nZSIgc3dhcHBlZD0ibm8iLz4KICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkg
bmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4xPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJyb29tLWNvbmZpZy1maWVsZC1kZXNjcmlwdGlv
biI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX3JlcXVlc3QiPjMwMDwvcHJvcGVydHk+CiAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ3cmFwIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFs
aWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWduIj4wPC9wcm9wZXJ0
eT4KICAgICAgICA8c3R5bGU+CiAgICAgICAgICA8Y2xhc3MgbmFtZT0iY29uZmlnLWZpZWxkLWhlbHAi
Lz4KICAgICAgICA8L3N0eWxlPgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjI8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2lu
dGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomConfigFormFieldTextMulti.xml": {
		local:   "definitions/MUCRoomConfigFormFieldTextMulti.xml",
		size:    3308,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0JveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4zPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9Imljb24taW1hZ2UiPgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJu
b19zaG93X2FsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxhYmVsIj4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVz
dGlmeSI+cmlnaHQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8
L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICA8L3BhY2tp
bmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtTY3JvbGxl
ZFdpbmRvdyI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoc2Nyb2xsYmFyX3BvbGljeSI+bmV2ZXI8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzaGFkb3dfdHlwZSI+aW48L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJtaW5fY29udGVudF9oZWlnaHQiPjUwPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtUZXh0VmlldyIgaWQ9InJvb20tY29uZmlnLXRl
eHQtbXVsdGktZmllbGQtdGV4dHZpZXciPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJs
ZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXBfbW9kZSI+d29yZDwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X21hcmdpbiI+NjwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyaWdodF9tYXJnaW4iPjY8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wX21hcmdpbiI+NjwvcHJvcGVydHk+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJib3R0b21fbWFyZ2luIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImFjY2VwdHMtdGFiIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICA8L29iamVj
dD4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjE8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWRlc2NyaXB0aW9u
Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0id2lkdGhfcmVxdWVzdCI+MzAwPC9wcm9wZXJ0eT4KICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgPHByb3Bl
cnR5IG5hbWU9IndyYXAiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ4YWxp
Z24iPjA8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5
PgogICAgICAgIDxzdHlsZT4KICAgICAgICAgIDxjbGFzcyBuYW1lPSJjb25maWctZmllbGQtaGVscCIv
PgogICAgICAgIDwvc3R5bGU+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+
MjwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAgPC9vYmplY3Q+CjwvaW50
ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomConfigListAddDialog.xml": {
		local:   "definitions/MUCRoomConfigListAddDialog.xml",
		size:    9399,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJyb29tLWNvbmZpZy1saXN0
LWFkZC1kaWFsb2ciPgogICAgPHByb3BlcnR5IG5hbWU9InJlc2l6YWJsZSI+RmFsc2U8L3Byb3BlcnR5
PgogICAgPHByb3BlcnR5IG5hbWU9Im1vZGFsIj5UcnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBu
YW1lPSJ0eXBlX2hpbnQiPmRpYWxvZzwvcHJvcGVydHk+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9
InZib3giPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZCBpbnRlcm5hbC1j
aGlsZD0iYWN0aW9uX2FyZWEiPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94Ij4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxheW91dF9zdHlsZSI+ZW5kPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJy
b29tLWNvbmZpZy1saXN0LWFkZC1jYW5jZWwiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNhbmNlbDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJv
bl9jYW5jZWwiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhw
YW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0
aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAg
ICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0icm9vbS1jb25maWctbGlzdC1hZGQtYXBwbHkiPgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFkZDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZW5zaXRpdmUiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNf
ZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InVzZV91bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxzaWduYWwg
bmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fYXBwbHkiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBh
bmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibm9uX2hvbW9nZW5lb3VzIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJub3RpZmljYXRpb24t
Ym94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4K
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjA8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
PGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9Im1hcmdpbl90b3AiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1h
cmdpbl9sZWZ0Ij4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5f
cmlnaHQiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEy
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0JveCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0xhYmVsIiBpZD0icm9vbS1jb25maWctbGlzdC1hZGQtdGl0bGUiPgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im1heF93aWR0aF9jaGFycyI+NjA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0
b25Cb3giPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYXlvdXRfc3R5bGUiPmVu
ZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0icm9vbS1jb25maWctbGlzdC1yZW1vdmUtYWxs
LWJ1dHRvbiI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbnNp
dGl2ZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0icmVjZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ1c2VfdW5kZXJsaW5lIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImFsd2F5c19zaG93X2ltYWdlIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJv
bl9yZW1vdmVfYWxsIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj42PC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9uYW1lIj5saXN0LXJlbW92
ZS1hbGwtc3ltYm9saWM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9z
aXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZp
c2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNsZWFyPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+
CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrRnJhbWUiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ibGFiZWxfeGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibGFiZWxfeWFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ic2hhZG93X3R5cGUiPmluPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNv
bmZpZy1saXN0LWFkZC1ib2R5Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjE8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwv
b2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+
CiAgICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomConfigListAddForm.xml": {
		local:   "definitions/MUCRoomConfigListAddForm.xml",
		size:    1295,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1saXN0LWZv
cm0iPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InZleHBh
bmQiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNh
bDwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0dyaWQiPgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icm93X3NwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJj
b2x1bW5fc3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0VudHJ5IiBpZD0icm9vbS1jb25maWctbGlzdC1qaWQiPgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBsYWNlaG9sZGVyX3RleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5BY2NvdW50IGFkZHJlc3M8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9maWVsZF9h
Y3RpdmF0ZSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjaGFuZ2VkIiBo
YW5kbGVyPSJvbl9maWVsZF9jaGFuZ2VkIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnRfYXR0YWNo
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcF9hdHRhY2giPjA8L3By
b3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0
PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAg
IDwvY2hpbGQ+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomConfigListAddFormItem.xml": {
		local:   "definitions/MUCRoomConfigListAddFormItem.xml",
		size:    2429,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1saXN0LWFk
ZC1pdGVtLWJveCI+CiAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFt
ZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCb3giIGlkPSJyb29tLWNvbmZpZy1saXN0LWFkZC1pdGVtLWZvcm0tYm94Ij4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgIDwvb2JqZWN0PgogICAgICA8
cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4K
ICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94IiBpZD0icm9vbS1jb25m
aWctbGlzdC1hZGQtaXRlbS1idXR0b25zIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjY8L3Byb3BlcnR5
PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9InJv
b20tY29uZmlnLWxpc3QtYWRkLWl0ZW0tYnV0dG9uIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InNlbnNpdGl2ZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVj
ZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
aWNvbl9uYW1lIj5saXN0LWFkZC1zeW1ib2xpYzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9Im5vbl9ob21vZ2VuZW91cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+
CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCdXR0b24iIGlkPSJyb29tLWNvbmZpZy1saXN0LXJlbW92ZS1pdGVtLWJ1dHRvbiI+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzZW5zaXRpdmUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Imljb25fbmFtZSI+bGlzdC1yZW1vdmUtc3ltYm9saWM8L3Byb3BlcnR5
PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFu
ZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJub25faG9tb2dlbmVvdXMiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAg
ICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hp
bGQ+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomConfigPage.xml": {
		local:   "definitions/MUCRoomConfigPage.xml",
		size:    6404,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtPdmVybGF5IiBpZD0icm9vbS1jb25maWctcGFn
ZS1vdmVybGF5Ij4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij43MDA8L3Byb3BlcnR5
PgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPGNoaWxkPgog
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtPdmVybGF5Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlz
aWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrQm94IiBpZD0icm9vbS1jb25maWctcGFnZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJub3RpZmljYXRpb25zLWJveCI+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0
aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hp
bGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtTY3Jv
bGxlZFdpbmRvdyIgaWQgPSJyb29tLWNvbmZpZy1wYWdlLXNjcm9sbGVkLXdpbmRvdyI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1ZpZXdwb3J0Ij4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2li
bGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
b3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InNwYWNpbmciPjI0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MjQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
TGFiZWwiIGlkPSJyb29tLWNvbmZpZy1wYWdlLWhlYWRlci1sYWJlbCI+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id2lkdGhfcmVxdWVzdCI+MzAwPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJub19zaG93X2Fs
bCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2xhc3MgbmFt
ZT0iY29uZmlnLWZpZWxkLWludHJvIi8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3N0eWxl
PgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4
cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
ICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCb3giIGlkPSJyb29tLWNvbmZpZy1wYWdlLWNvbnRlbnQiPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MjQ8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJrZXktcHJl
c3MtZXZlbnQiIGhhbmRsZXI9Im9uX2tleV9wcmVzcyIgc3dhcHBlZD0ibm8iIC8+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJw
b3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InJvb20tY29uZmlnLWF1
dG9qb2luLWNvbnRlbnQiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im5vX3Nob3dfYWxs
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlv
biI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJrZXktcHJl
c3MtZXZlbnQiIGhhbmRsZXI9Im9uX2tleV9wcmVzcyIgc3dhcHBlZD0ibm8iIC8+CiAgICAgICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1NlcGFyYXRvciI+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5lbmQ8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4yNDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQ2hlY2tCdXR0b24iIGlkPSJyb29tLWNvbmZpZy1hdXRvam9pbiI+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5
ZXMiPkF1dG9tYXRpY2FsbHkgam9pbiB0aGUgcm9vbTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPkZhbHNlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImRyYXdfaW5kaWNhdG9y
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJ0b2dn
bGVkIiBoYW5kbGVyPSJvbl9hdXRvam9pbl90b2dnbGVkIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5n
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29i
amVjdD4KICAgIDwvY2hpbGQ+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomConfigSummaryAdvancedOptionsField.xml": {
		local:   "definitions/MUCRoomConfigSummaryAdvancedOptionsField.xml",
		size:    4987,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmll
bnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMaXN0Qm94Um93IiBpZD0icm9vbS1jb25maWctZmllbGQiPgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ3aWR0aF9yZXF1ZXN0Ij4xMDA8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJh
Y3RpdmF0YWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3Rh
YmxlIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTI8L3By
b3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJyb29t
LWNvbmZpZy1maWVsZC1sYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imxh
YmVsIiB0cmFuc2xhdGFibGU9InllcyI+QWR2YW5jZWQgb3B0aW9uczwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHN0eWxlPgogICAgICAgICAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9InN1bW1hcnktZmll
bGQtbmFtZSIvPgogICAgICAgICAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAg
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJyb29tLWNvbmZpZy1maWVsZC12YWx1
ZSI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5lbmQ8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxzdHlsZT4KICAgICAgICAgICAgICAgICAgICAgIDxjbGFzcyBuYW1l
PSJyb29tLWNvbmZpZy1zdW1tYXJ5LXZhbHVlIi8+CiAgICAgICAgICAgICAgICAgICAgPC9zdHlsZT4K
ICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0i
cm9vbS1jb25maWctZmllbGQtbGlzdC1idXR0b24iPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX3Nob3dfbGlzdCIgc3dhcHBlZD0i
bm8iLz4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxpc3QtYnV0dG9uLWltYWdl
Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9uYW1lIj5w
YW4tdXAtc3ltYm9saWM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBv
c2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0icm9vbS1jb25maWctZmllbGQtYWR2YW5j
ZWQtb3B0aW9ucy1ib3giPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRl
cjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZl
cnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrRnJhbWUiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ibGFiZWxfeGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ic2hhZG93X3R5cGUiPmluPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0Qm94IiBpZD0icm9vbS1j
b25maWctZmllbGQtYWR2YW5jZWQtb3B0aW9ucy1jb250ZW50LWJveCI+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGlvbl9tb2RlIj5ub25lPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImFjdGl2YXRlX29uX3NpbmdsZV9j
bGljayI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjM8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICA8L29iamVjdD4KPC9pbnRl
cmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomConfigSummaryField.xml": {
		local:   "definitions/MUCRoomConfigSummaryField.xml",
		size:    6936,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmll
bnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMaXN0Qm94Um93IiBpZD0icm9vbS1jb25maWctZmllbGQiPgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ3aWR0aF9yZXF1ZXN0Ij4xMDA8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJh
Y3RpdmF0YWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3Rh
YmxlIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTI8L3By
b3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJyb29t
LWNvbmZpZy1maWVsZC1sYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZh
bGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
d3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Inhh
bGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGln
biI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAgICAg
ICAgICAgPGNsYXNzIG5hbWU9InN1bW1hcnktZmllbGQtbmFtZSIvPgogICAgICAgICAgICAgICAgICAg
IDwvc3R5bGU+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFi
ZWwiIGlkPSJyb29tLWNvbmZpZy1maWVsZC12YWx1ZSI+CiAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxzdHlsZT4KICAg
ICAgICAgICAgICAgICAgICAgIDxjbGFzcyBuYW1lPSJyb29tLWNvbmZpZy1zdW1tYXJ5LXZhbHVlIi8+
CiAgICAgICAgICAgICAgICAgICAgPC9zdHlsZT4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0icm9vbS1jb25maWctZmllbGQtbGlzdC1idXR0b24i
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJub19zaG93X2FsbCI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGln
biI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNr
ZWQiIGhhbmRsZXI9Im9uX3Nob3dfbGlzdCIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9
InJvb20tY29uZmlnLWZpZWxkLWxpc3QtYnV0dG9uLWltYWdlIj4KICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9uYW1lIj5wYW4tZG93bi1zeW1ib2xpYzwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICA8L2No
aWxkPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLXRleHQtbXVsdGktdmFsdWUiPgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ibm9fc2hvd19hbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxzdHlsZT4KICAgICAg
ICAgICAgICAgICAgPGNsYXNzIG5hbWU9InN1bW1hcnktZmllbGQtbXVsdGktdmFsdWUiLz4KICAgICAg
ICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBh
Y2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxp
c3QtY29udGVudCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZl
cnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJub19zaG93X2Fs
bCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtTZXBhcmF0b3IiPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtUcmVlVmlldyIgaWQ9InJvb20tY29uZmlnLWZpZWxkLWxpc3QtdmFsdWVz
LXRyZWUiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZWFkZXJzX3Zpc2libGUi
PkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGVhZGVy
c19jbGlja2FibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iZW5hYmxlX3NlYXJjaCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJzaG93X2V4cGFuZGVycyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJlbmFibGVfZ3JpZF9saW5lcyI+aG9yaXpvbnRhbDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0ic2VsZWN0aW9uIj4K
ICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVTZWxlY3Rpb24iPgogICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibW9kZSI+bm9uZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a1RyZWVWaWV3Q29sdW1uIiBpZD0iYWRtaW4tamlkIj4KICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+QWNjb3VudCBhZGRyZXNz
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NlbGxSZW5kZXJlclRleHQiIGlkPSJhZG1pbi1qaWQt
cmVuZGVyZXIiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InhwYWQi
PjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5
cGFkIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxhdHRyaWJ1dGUgbmFtZT0idGV4dCI+MDwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAg
ICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3Np
dGlvbiI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2No
aWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAg
ICA8L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomConfigSummaryFieldContainer.xml": {
		local:   "definitions/MUCRoomConfigSummaryFieldContainer.xml",
		size:    776,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZC1i
b3giPgogICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmll
bnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtGcmFtZSI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbF94YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzaGFkb3dfdHlwZSI+aW48L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4K
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xpc3RCb3giIGlkPSJyb29tLWNvbmZpZy1maWVsZHMt
Y29udGVudCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGlvbl9tb2RlIj5ub25lPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImFjdGl2YXRlX29uX3NpbmdsZV9jbGljayI+RmFs
c2U8L3Byb3BlcnR5PiAgICAgICAgICAgCiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxk
PgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomConfirmationRoomDialog.xml": {
		local:   "definitions/MUCRoomConfirmationRoomDialog.xml",
		size:    5100,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJjb25maXJtYXRpb24tZGlh
bG9nIj4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNhbmNlbCBy
b29tIGNyZWF0aW9uPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJyZXNpemFibGUiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJtb2RhbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8
cHJvcGVydHkgbmFtZT0iZGVzdHJveV93aXRoX3BhcmVudCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJv
cGVydHkgbmFtZT0idHlwZV9oaW50Ij5kaWFsb2c8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9
InVyZ2VuY3lfaGludCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9InZi
b3giPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJv
cmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGls
ZD0iYWN0aW9uX2FyZWEiPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94Ij4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTI8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsYXlvdXRfc3R5bGUiPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiI+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5ObzwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXJpZ2h0Ij42PC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fbm9fY2xpY2tlZCIgc3dh
cHBlZD0ibm8iIC8+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im5vbl9ob21vZ2VuZW91cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIj4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlllczwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX3llc19jbGlja2Vk
IiBzd2FwcGVkPSJubyIgLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibm9uX2hvbW9nZW5lb3VzIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3RvcCI+MTI8
L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrQm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0ltYWdlIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImRvdWJsZV9idWZmZXJl
ZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxp
Z24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFs
aWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imlj
b25fbmFtZSI+ZGlhbG9nLXdhcm5pbmc8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJpY29uX3NpemUiPjU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAg
ICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9ImNvbmZpcm1hdGlvbi1kaWFsb2ctaGVhZGVy
Ij4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0i
eWVzIj5BcmUgeW91IHN1cmUgeW91IHdhbnQgdG8gY2FuY2VsIHRoZSByb29tIGNyZWF0aW9uPyA8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0cmFja192aXNpdGVk
X2xpbmtzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Inlh
bGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5G
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+
MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJwYWRkaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4K
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAg
PGNsYXNzIG5hbWU9ImNveWltIi8+CiAgICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomConversation.xml": {
		local:   "definitions/MUCRoomConversation.xml",
		size:    8840,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWNvbnZlcnNhdGlvbiI+
CiAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkg
bmFtZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0idmV4cGFuZCI+
VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9w
cm9wZXJ0eT4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2ZXhwYW5k
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRp
Y2FsPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtT
Y3JvbGxlZFdpbmRvdyIgaWQ9ImNoYXQtc2Nyb2xsZWQtd2luZG93Ij4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0id2luZG93X3BsYWNlbWVudF9zZXQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImtpbmV0aWNfc2Nyb2xsaW5nIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJvdmVybGF5X3Njcm9sbGluZyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtUZXh0VmlldyIgaWQ9ImNoYXQt
dGV4dC12aWV3Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwaXhlbHNfYmVsb3dfbGluZXMi
PjU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImVkaXRhYmxlIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcF9tb2RlIj53b3Jk
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X21hcmdpbiI+NTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmlnaHRfbWFyZ2luIj41PC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfbWFyZ2luIj41PC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3R0b21fbWFyZ2luIj41PC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjdXJzb3JfdmlzaWJsZSI+RmFsc2U8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImFjY2VwdHNfdGFiIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAg
ICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8
L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2Jq
ZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgog
ICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2VwYXJhdG9yIj4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgIDwvb2Jq
ZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4K
ICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1ZpZXdwb3J0Ij4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InNoYWRvd190eXBlIj5ub25lPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRh
dGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0ibWVzc2FnZS12aWV3Ij4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2Nyb2xsZWRXaW5kb3ciIGlkPSJt
ZXNzYWdlLXNjcm9sbGVkLXdpbmRvdyI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJib3JkZXJfd2lkdGgiPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2c2Nyb2xsYmFyLXBvbGljeSI+R1RLX1BPTElDWV9BVVRPTUFUSUM8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoc2Nyb2xsYmFyLXBvbGljeSI+R1RLX1BPTElD
WV9BVVRPTUFUSUM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3
aW5kb3dfcGxhY2VtZW50X3NldCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJraW5ldGljX3Njcm9sbGluZyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXhfY29udGVudF9oZWlnaHQiPjYwPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icHJvcGFnYXRlX25hdHVyYWxfaGVpZ2h0
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtUZXh0VmlldyIgaWQ9Im1lc3NhZ2UtdGV4dC12aWV3Ij4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGl4ZWxzX2Fib3ZlX2xp
bmVzIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBp
eGVsc19iZWxvd19saW5lcyI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJwaXhlbHNfaW5zaWRlX3dyYXAiPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcC1tb2RlIj5HVEtfV1JBUF9XT1JEX0NIQVI8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdF9tYXJnaW4iPjEy
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJpZ2h0X21h
cmdpbiI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dG9wX21hcmdpbiI+NjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJib3R0b21fbWFyZ2luIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImFjY2VwdHNfdGFiIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxzaWduYWwgbmFtZT0ia2V5LXByZXNzLWV2ZW50IiBoYW5kbGVyPSJvbl9rZXlfcHJlc3Mi
IHN3YXBwZWQ9Im5vIiAvPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2No
aWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCdXR0b25Cb3giIGlkPSJtZXNzYWdlLWJ1dHRvbnMiPgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibGF5b3V0X3N0eWxlIj5leHBhbmQ8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1
dHRvbiIgaWQ9Im1lc3NhZ2Utc2VuZC1idXR0b24iPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InVzZV91bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9u
X3NlbmRfbWVzc2FnZSIgc3dhcHBlZD0ibm8iIC8+CiAgICAgICAgICAgICAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9sZWZ0Ij42
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJn
aW5fcmlnaHQiPjY8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InNwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCI+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNlbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhY2tp
bmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9Im1lc3NhZ2Utc2VuZC1pY29uIj4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+
Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ic3RvY2siPmd0ay1taXNzaW5nLWltYWdlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5n
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAg
ICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBh
Y2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFja190
eXBlIj5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhw
YW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3Np
dGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8c3R5bGU+CiAgICAgICAgICAgICAgICAgIDxjbGFz
cyBuYW1lPSJtZXNzYWdlLXZpZXciIC8+CiAgICAgICAgICAgICAgICA8L3N0eWxlPgogICAgICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCb3giIGlkPSJub3RpZmljYXRpb24tYm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFu
ZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tp
bmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFt
ZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogIDwv
b2JqZWN0Pgo8L2ludGVyZmFjZT4=
`,
	},

	"/definitions/MUCRoomDestroyDialog.xml": {
		local:   "definitions/MUCRoomDestroyDialog.xml",
		size:    15824,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkZXN0cm95LXJvb20tZGlh
bG9nIj4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij40MDA8L3Byb3BlcnR5PgogICAg
PHByb3BlcnR5IG5hbWU9InJlc2l6YWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5h
bWU9Im1vZGFsIj5UcnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJkZWZhdWx0X3dpZHRo
Ij4zMjA8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InR5cGVfaGludCI+ZGlhbG9nPC9wcm9w
ZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ1cmdlbmN5X2hpbnQiPlRydWU8L3Byb3BlcnR5PgogICAg
PGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAg
ICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAgICAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a0J1dHRvbkJveCIgaWQ9ImRlc3Ryb3ktcm9vbS1idXR0b25zIj4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxheW91dF9zdHlsZSI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9y
ZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3Bh
Y2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJjYW5jZWwtYnV0dG9uIj4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5DYW5jZWw8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVs
dCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2Vk
IiBoYW5kbGVyPSJvbl9jYW5jZWwiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iZGVzdHJveS1yb29tLWJ1dHRvbiI+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+
RGVzdHJveSBSb29tPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxzaWdu
YWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fZGVzdHJveSIgc3dhcHBlZD0ibm8iLz4KICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAg
IDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9Im5vdGlmaWNhdGlvbi1h
cmVhIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+
MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJkZXN0cm95LXJvb20tY29u
dGVudCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdlIj4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ic3RvY2siPmd0ay1kZWxldGU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpY29uX3NpemUiPjY8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8
L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0xhYmVsIj4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWdu
Ij5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJs
YWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPllvdSBhcmUgYWJvdXQgdG8gcGVybWFuZW50bHkgZGVzdHJv
eSB0aGlzIHJvb20uPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9IndyYXAiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ieGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIj4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0
YWJsZT0ieWVzIj5Zb3UgbWF5IHdhbnQgdG8gcHJvdmlkZSB0aGUgZm9sbG93aW5nIGluZm9ybWF0aW9u
OjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ4YWxpZ24i
PjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWdu
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InNwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9ImRlc3Ryb3ktcm9vbS1yZWFz
b24tbGFiZWwiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5z
bGF0YWJsZT0ieWVzIj5FbnRlciB0aGUgcmVhc29uIGZvciBkZXN0cm95aW5nIHRoZSByb29tPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4wPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWduIj4wPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2Nyb2xsZWRXaW5k
b3ciPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2hhZG93X3R5cGUiPmluPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtUZXh0VmlldyIgaWQ9ImRlc3Ryb3ktcm9vbS1yZWFzb24tZW50cnkiPgog
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwaXhlbHNfaW5zaWRlX3dy
YXAiPjM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9w
X21hcmdpbiI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJib3R0b21fbWFyZ2luIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImluZGVudCI+NjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJhY2NlcHRzX3RhYiI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9Im1hcmdpbl90b3AiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDaGVj
a0J1dHRvbiIgaWQ9ImRlc3Ryb3ktcm9vbS1hbHRlcm5hdGl2ZS1jaGVjayI+CiAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFkZCBpbmZv
cm1hdGlvbiBhYm91dCB0aGUgbmV3IHJvb208L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5GYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2VfdW5kZXJsaW5lIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImRyYXdfaW5kaWNh
dG9yIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJ0
b2dnbGVkIiBoYW5kbGVyPSJvbl9hbHRlcm5hdGl2ZV9yb29tX3RvZ2dsZWQiIHN3YXBwZWQ9Im5vIi8+
CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2No
aWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJkZXN0cm95LXJvb20tYWx0ZXJuYXRpdmUtYm94Ij4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtHcmlkIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVy
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93X3NwYWNpbmci
PjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY29sdW1uX3Nw
YWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9ImRlc3Ryb3ktcm9vbS1uYW1lLWxh
YmVsIj4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5l
bmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWdu
Ij5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5Sb29tPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0VudHJ5IiBpZD0iZGVzdHJveS1yb29tLW5hbWUtZW50cnkiPgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25fZGVzdHJv
eSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsZWZ0X2F0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ0b3BfYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0iZGVzdHJv
eS1yb29tLXNlcnZpY2UtbGFiZWwiPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJoYWxpZ24iPmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPldoZXJlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tp
bmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNo
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NvbWJvQm94VGV4dCIgaWQ9ImNoYXQtc2VydmljZXMtbGlz
dCI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imhhc19lbnRyeSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGls
ZD0iZW50cnkiPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0VudHJ5
IiBpZD0iY2hhdC1zZXJ2aWNlcy1lbnRyeSI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0icGxhY2Vob2xkZXJfdGV4dCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkVu
dGVyIG9yIHNlbGVjdCB0aGUgc2VydmljZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9kZXN0cm95IiBzd2FwcGVkPSJu
byIvPgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAg
ICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
ICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imxl
ZnRfYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InRvcF9hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJkZXN0cm95LXJv
b20tcGFzc3dvcmQtbGFiZWwiPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJoYWxpZ24iPmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlBhc3N3b3JkPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tp
bmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNo
Ij4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0VudHJ5IiBpZD0iZGVzdHJveS1yb29tLXBhc3N3b3JkLWVu
dHJ5Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaW5wdXRfcHVy
cG9zZSI+cGFzc3dvcmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5h
bWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9kZXN0cm95IiBzd2FwcGVkPSJubyIvPgogICAgICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnRfYXR0YWNoIj4xPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcF9hdHRhY2giPjI8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAg
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwYWRkaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3Np
dGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAg
ICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPHN0eWxlPgogICAgICA8Y2xhc3MgbmFtZT0iY295
aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomDestroyDialogError.xml": {
		local:   "definitions/MUCRoomDestroyDialogError.xml",
		size:    5955,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJyb29tLWRlc3RvcnktZXJy
b3ItZGlhbG9nIj4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlJv
b20gZGVzdHJ1Y3Rpb24gZXJyb3I8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InJlc2l6YWJs
ZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im1vZGFsIj5UcnVlPC9wcm9wZXJ0
eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJkZXN0cm95X3dpdGhfcGFyZW50Ij5UcnVlPC9wcm9wZXJ0eT4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0eXBlX2hpbnQiPmRpYWxvZzwvcHJvcGVydHk+CiAgICA8cHJvcGVy
dHkgbmFtZT0idXJnZW5jeV9oaW50Ij5UcnVlPC9wcm9wZXJ0eT4KICAgIDxjaGlsZCBpbnRlcm5hbC1j
aGlsZD0idmJveCI+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImJvcmRlcl93aWR0aCI+MTI8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGls
ZD0iYWN0aW9uX2FyZWEiPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94Ij4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxheW91dF9zdHlsZSI+ZW5kPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCI+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGF5b3V0X3N0eWxlIj5zdGFydDwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5lbmQ8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24i
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5
ZXMiPkNhbmNlbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZp
c2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJy
ZWNlaXZlc19kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9Im1hcmdpbi1yaWdodCI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fY2FuY2VsIiBzd2FwcGVkPSJubyIv
PgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJub25faG9tb2dlbmVvdXMiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtC
dXR0b24iPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRh
YmxlPSJ5ZXMiPlJldHJ5PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ1c2VfdW5kZXJsaW5lIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX3JldHJ5IiBzd2FwcGVkPSJubyIv
PgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJub25faG9tb2dlbmVvdXMiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAg
ICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdlIj4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzdG9jayI+Z3RrLWRlbGV0ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9zaXplIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fcmlnaHQiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3Np
dGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2No
aWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94
Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTGFiZWwiIGlkPSJyb29tLWRlc3Ryb3ktZXJyb3ItdGl0bGUiPgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ0cmFja192aXNpdGVkX2xpbmtzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0icm9vbS1k
ZXN0cm95LWVycm9yLW1lc3NhZ2UiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
d3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imhh
bGlnbiI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0
cmFja192aXNpdGVkX2xpbmtzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4K
ICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBhZGRpbmciPjY8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAg
IDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAgICA8L3N0eWxlPgogIDwvb2JqZWN0
Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomDialogErrorComponent.xml": {
		local:   "definitions/MUCRoomDialogErrorComponent.xml",
		size:    6740,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJyb29tLWVycm9yLWRpYWxv
ZyI+CiAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4wPC9wcm9wZXJ0eT4KICAgIDxwcm9w
ZXJ0eSBuYW1lPSJyZXNpemFibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJt
b2RhbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0iZGVzdHJveV93aXRoX3BhcmVu
dCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0idHlwZV9oaW50Ij5kaWFsb2c8L3By
b3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InVyZ2VuY3lfaGludCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9InZib3giPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3gi
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5Pgog
ICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0iYWN0aW9uX2FyZWEiPgogICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrQnV0dG9uQm94Ij4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJs
ZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dp
ZHRoIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJoYWxpZ24iPmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImhvbW9nZW5lb3VzIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGF5b3V0X3N0eWxlIj5leHBhbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiI+
CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCI+Z3RrLW9rPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNf
ZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ1c2VfdW5kZXJsaW5lIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InVzZV9zdG9jayI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fb2siIHN3YXBwZWQ9Im5vIi8+
CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJub25f
aG9tb2dlbmVvdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0Pgog
ICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0icGFjay10eXBlIj5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
PHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFja190eXBlIj5lbmQ8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3RvcCI+MTI8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94
Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdl
Ij4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImRvdWJsZV9idWZmZXJlZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0
YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5z
dGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25fbmFt
ZSI+ZGlhbG9nLWVycm9yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iaWNvbl9zaXplIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBv
c2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlz
aWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZh
bGlnbiI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJv
cmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJzcGFjaW5nIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InJvb20tZXJyb3It
ZGlhbG9nLWhlYWRlciI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNp
YmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
IndyYXAiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idHJhY2tfdmlzaXRlZF9saW5rcyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0xhYmVsIiBpZD0icm9vbS1lcnJvci1kaWFsb2ctbWVzc2FnZSI+CiAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXAiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWF4X3dpZHRoX2NoYXJzIj42MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0cmFja192aXNpdGVkX2xp
bmtzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ieWFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBhZGRpbmciPjEyPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxk
PgogICAgPHN0eWxlPgogICAgICA8Y2xhc3MgbmFtZT0iY295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9v
YmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomLobby.xml": {
		local:   "definitions/MUCRoomLobby.xml",
		size:    7480,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJtYWluLWNvbnRlbnQiPgogICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9
Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a0JveCIgaWQ9Im5vdGlmaWNhdGlvbnMtYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0
aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgPHN0eWxlPgogICAgICAgICAgPGNsYXNzIG5h
bWU9Im5vdGlmaWNhdGlvbnMiLz4KICAgICAgICA8L3N0eWxlPgogICAgICA8L29iamVjdD4KICAgICAg
PHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxk
PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4
cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZleHBhbmQiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4iPjI0PC9wcm9wZXJ0eT4KICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0id2lkdGhfcmVxdWVzdCI+NTAwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iaGFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFs
aWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRp
b24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0icm9vbS1uYW1lLXZhbHVlIj4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJoYWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibWFyZ2luX2JvdHRvbSI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtHcmlkIiBp
ZD0iZm9ybS1ncmlkIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fdG9wIj41PC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fYm90dG9tIj41PC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyb3dfc3BhY2luZyI+NjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY29sdW1uX3NwYWNpbmciPjEwPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJiYXNlbGluZV9yb3ciPjU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24i
PmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+
Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwi
IHRyYW5zbGF0YWJsZT0ieWVzIj5OaWNrbmFtZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGVmdF9hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ0b3BfYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrRW50cnkiIGlkPSJuaWNrbmFtZS1lbnRyeSI+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2Fwc19sb2NrX3dhcm5pbmciPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBo
YW5kbGVyPSJvbl9qb2luIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAgICAgICAgIDxzaWduYWwg
bmFtZT0iY2hhbmdlZCIgaGFuZGxlcj0ib25fbmlja25hbWVfY2hhbmdlZCIgc3dhcHBlZD0ibm8iLz4K
ICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0X2F0dGFjaCI+MTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcF9hdHRhY2giPjA8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIg
aWQ9InBhc3N3b3JkLWxhYmVsIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFs
aWduIj5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxp
Z24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imxh
YmVsIiB0cmFuc2xhdGFibGU9InllcyI+UGFzc3dvcmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImxlZnRfYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idG9wX2F0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
IDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0VudHJ5IiBpZD0icGFzc3dvcmQtZW50
cnkiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJpbGl0eSI+RmFsc2U8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUiIGhhbmRs
ZXI9Im9uX2pvaW4iIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1l
PSJjaGFuZ2VkIiBoYW5kbGVyPSJvbl9wYXNzd29yZF9jaGFuZ2VkIiBzd2FwcGVkPSJubyIvPgogICAg
ICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnRfYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wX2F0dGFjaCI+MjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iZmlsbCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCdXR0b25Cb3giIGlkPSJidXR0b25zLWJveCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
aGFsaWduIj5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdp
bl90b3AiPjU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9i
b3R0b20iPjU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmci
PjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYXlvdXRfc3R5bGUi
PnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iY2FuY2VsLWJ1dHRvbiI+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+X0NhbmNlbDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX3Vu
ZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJj
bGlja2VkIiBoYW5kbGVyPSJvbl9jYW5jZWwiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iam9pbi1idXR0b24iPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPl9K
b2luPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbnNpdGl2
ZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNl
aXZlc19kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHNp
Z25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9qb2luIiBzd2FwcGVkPSJubyIvPgogICAgICAg
ICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjA8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwv
b2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5n
PgogICAgPC9jaGlsZD4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomMain.xml": {
		local:   "definitions/MUCRoomMain.xml",
		size:    3122,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLXZpZXctYm94Ij4KICAg
IDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij40MDA8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5
IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InZleHBhbmQi
PlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwv
cHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1BhbmVkIiBpZD0icm9v
bS12aWV3LW1haW4iPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwb3NpdGlvbl9zZXQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3
aWRlX2hhbmRsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2Fs
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0JveCIgaWQ9InJvb20tdmlldy10b3AiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imhl
eHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVu
dGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5
PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InJvb20tdmlldy1j
b250ZW50Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij41MDA8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZleHBhbmQiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBh
Y2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZXNpemUiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNocmluayI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
ICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrVmlld3BvcnQiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id2lkdGhf
cmVxdWVzdCI+MjAwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2hhZG93X3R5cGUiPm5v
bmU8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrQm94IiBpZD0icm9vbS12aWV3LXBhbmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ3aWR0aF9yZXF1ZXN0Ij4zMDA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVzaXplIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJzaHJpbmsiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MjwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAgICA8c3R5bGU+CiAg
ICAgIDxjbGFzcyBuYW1lPSJtdWMiLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNl
Pgo=
`,
	},

	"/definitions/MUCRoomMessageBoxNotification.xml": {
		local:   "definitions/MUCRoomMessageBoxNotification.xml",
		size:    2128,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtJbmZvQmFyIiBpZD0ibm90aWZpY2F0aW9uLWJh
ciI+CiAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVy
dHkgbmFtZT0ibWVzc2FnZV90eXBlIj5vdGhlcjwvcHJvcGVydHk+CiAgICA8Y2hpbGQgaW50ZXJuYWwt
Y2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94Ij4KICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImxheW91dF9zdHlsZSI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICA8L29iamVjdD4KICAgICAg
PHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGls
ZD4KICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0iY29udGVudF9hcmVhIj4KICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrQm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xNjwvcHJvcGVydHk+CiAgICAgICAg
PGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3Jp
ZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ibWVzc2FnZSI+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+WW91IGRvbid0IGhhdmUgcGVybWlzc2lv
biB0byBzZW5kIG1lc3NhZ2VzLjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idHJhY2tfdmlzaXRlZF9saW5rcyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ieWFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFu
ZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4K
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgICAg
PHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGls
ZD4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomOccupantActionDialog.xml": {
		local:   "definitions/MUCRoomOccupantActionDialog.xml",
		size:    11387,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJvY2N1cGFudC1hY3Rpb24t
ZGlhbG9nIj4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij40MDA8L3Byb3BlcnR5Pgog
ICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+UmVtb3ZlIHBlcnNvbiBm
cm9tIHRoZSByb29tPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJyZXNpemFibGUiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJtb2RhbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8
cHJvcGVydHkgbmFtZT0iZGVmYXVsdF93aWR0aCI+MzIwPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBu
YW1lPSJ0eXBlX2hpbnQiPmRpYWxvZzwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0idXJnZW5j
eV9oaW50Ij5UcnVlPC9wcm9wZXJ0eT4KICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0idmJveCI+CiAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0
aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJhY3Rp
b25fYXJlYSI+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJkZXN0cm95
LXJvb20tYnV0dG9ucyI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPmVuZDwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYXlvdXRfc3R5bGUiPmV4cGFuZDwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTI8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94
Ij4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJjYW5jZWwtYnV0dG9uIj4KICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Q2FuY2VsPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVm
YXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0i
Y2xpY2tlZCIgaGFuZGxlcj0ib25fY2FuY2VsIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAgICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0J1dHRvbiIgaWQ9ImtpY2stb2NjdXBhbnQtYnV0dG9uIj4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+T2s8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZh
dWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJj
bGlja2VkIiBoYW5kbGVyPSJvbl9vayIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJub3RpZmljYXRpb24t
YXJlYSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjA8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
PGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iZGVzdHJveS1yb29tLWNv
bnRlbnQiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl90b3AiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJzcGFjaW5nIj4yNDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiPgogICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPnN0YXJ0PC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBpeGVsX3NpemUiPjMy
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25fbmFt
ZSI+ZGlhbG9nLXdhcm5pbmc8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iaWNvbl9zaXplIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAg
ICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94
Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50
ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50
YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxk
PgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ib2Nj
dXBhbnQtYWN0aW9uLWhlYWRlciI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5Zb3UgYXJlIGFi
b3V0IHRvIHRlbXBvcmFyaWx5IHJlbW92ZSB0aGUgcGVyc29uIGZyb20gdGhlIHJvb20uPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXhfd2lk
dGhfY2hhcnMiPjQwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ib2NjdXBhbnQtYWN0aW9u
LW1lc3NhZ2UiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2li
bGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+VGhleSB3aWxsIGJlIGFibGUgdG8gam9pbiB0aGUg
cm9vbSBhZ2Fpbi4gQXJlIHlvdSBzdXJlIHlvdSB3YW50IHRvIGNvbnRpbnVlPzwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWF4X3dpZHRoX2No
YXJzIj41MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ieGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8
L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAg
ICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBv
c2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAg
ICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFu
ZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRp
b24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAg
ICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50
YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAg
ICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ib2NjdXBhbnQtYWN0aW9u
LXJlYXNvbi1sYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNp
YmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+SGVyZSB5b3UgY2FuIHByb3ZpZGUgYW4gb3B0aW9uYWwg
cmVhc29uIGZvciByZW1vdmluZyB0aGUgcGVyc29uLiBFdmVyeW9uZSBpbiB0aGUgcm9vbSB3aWxsIHNl
ZSB0aGlzIHJlYXNvbi48L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJtYXhfd2lkdGhfY2hhcnMiPjQwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtTY3JvbGxlZFdpbmRvdyI+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InNoYWRvd190eXBlIj5pbjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxj
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtUZXh0VmlldyIg
aWQ9Im9jY3VwYW50LWFjdGlvbi1yZWFzb24tZW50cnkiPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBpeGVsc19pbnNpZGVfd3JhcCI+MzwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wX21hcmdpbiI+Mzwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm90dG9t
X21hcmdpbiI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iaW5kZW50Ij42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJhY2NlcHRzX3RhYiI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5n
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAg
ICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwYWRkaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAg
PC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJw
b3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4K
ICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPHN0eWxlPgogICAgICA8Y2xhc3MgbmFtZT0i
Y295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomPositionsDialog.xml": {
		local:   "definitions/MUCRoomPositionsDialog.xml",
		size:    4999,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtXaW5kb3ciIGlkPSJwb3NpdGlvbnMtd2luZG93
Ij4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlBvc2l0aW9uczwv
cHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0ibW9kYWwiPlRydWU8L3Byb3BlcnR5PgogICAgPHBy
b3BlcnR5IG5hbWU9IndpbmRvd19wb3NpdGlvbiI+Y2VudGVyLW9uLXBhcmVudDwvcHJvcGVydHk+CiAg
ICA8cHJvcGVydHkgbmFtZT0iZGVmYXVsdF93aWR0aCI+NjAwPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0
eSBuYW1lPSJyZXNpemFibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJkZXN0
cm95X3dpdGhfcGFyZW50Ij5UcnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0eXBlX2hp
bnQiPmRpYWxvZzwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Jv
eCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgIDxj
aGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImJvcmRlcl93aWR0aCI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3Jp
ZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNw
YWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0xhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0
YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNs
YXRhYmxlPSJ5ZXMiPkhlcmUgeW91IGNhbiBkZWZpbmUgd2hvIHRoZSBiYW5uZWQgcGVvcGxlIGFyZS48
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXAiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxzdHlsZT4KICAgICAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9ImNvbmZpZy1maWVsZC1p
bnRybyIvPgogICAgICAgICAgICAgICAgPC9zdHlsZT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQi
PkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iY29udGVu
dCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4yNDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+
CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgog
ICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1NlcGFyYXRvciI+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBh
bmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5
PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZp
c2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrQnV0dG9uQm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxp
Z24iPmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRo
Ij4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJjYW5jZWwtYnV0dG9uIj4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5DYW5jZWw8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVy
PSJvbl9jYW5jZWwiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBv
c2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iYXBwbHktYnV0dG9uIj4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5BcHBseTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZW5zaXRpdmUiPkZhbHNlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBo
YW5kbGVyPSJvbl9hcHBseSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAg
ICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+
CiAgICA8c3R5bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9zdHlsZT4KICA8L29i
amVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomRoleDialog.xml": {
		local:   "definitions/MUCRoomRoleDialog.xml",
		size:    22939,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJyb2xlLWRpYWxvZyI+CiAg
ICA8cHJvcGVydHkgbmFtZT0id2lkdGhfcmVxdWVzdCI+NjAwPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0
eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNoYW5nZSByb2xlPC9wcm9wZXJ0eT4KICAg
IDxwcm9wZXJ0eSBuYW1lPSJyZXNpemFibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBu
YW1lPSJtb2RhbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0idHlwZV9oaW50Ij5k
aWFsb2c8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InVyZ2VuY3lfaGludCI+VHJ1ZTwvcHJv
cGVydHk+CiAgICA8c2lnbmFsIG5hbWU9ImtleS1wcmVzcy1ldmVudCIgaGFuZGxlcj0ib25fa2V5X3By
ZXNzIiBzd2FwcGVkPSJubyIvPgogICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0icm9sZS1jb250ZW50LWJveCI+CiAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkIGlu
dGVybmFsLWNoaWxkPSJhY3Rpb25fYXJlYSI+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0
b25Cb3giIGlkPSJyb2xlLWJ1dHRvbnMiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWdu
Ij5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8
L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGF5b3V0X3N0eWxlIj5leHBhbmQ8
L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrQm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgiPjEyPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0J1dHRvbkJveCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4x
MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0icm9sZS1jYW5jZWwtYnV0dG9uIj4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+
Q2FuY2VsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZp
c2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0icmVjZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fY2FuY2VsIiBzd2FwcGVkPSJubyIvPgog
ICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAg
ICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9InJvbGUtYXBwbHktYnV0dG9uIj4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+
QXBwbHk8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlz
aWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJzZW5zaXRpdmUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX2FwcGx5IiBzd2FwcGVkPSJu
byIvPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0
aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAg
ICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAg
ICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8
L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBh
Y2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InBhY2tfdHlwZSI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Jv
eCIgaWQ9InJvbGUtY29udGVudCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3RvcCI+MTI8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNh
bDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0JveCIgaWQ9InJvbGUtdHlwZSI+CiAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2YWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBp
ZD0icm9sZS10eXBlLWxhYmVsIj4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iaGFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNoYW5nZSByb2xlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tp
bmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrRnJhbWUiPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsYWJlbF94YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2hhZG93X3R5cGUiPmluPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0xpc3RCb3giPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InNlbGVjdGlvbl9tb2RlIj5ub25lPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJhY3RpdmF0ZV9vbl9zaW5nbGVfY2xpY2siPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGlzdEJveFJvdyI+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX3JlcXVlc3QiPjEwMDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhlaWdodF9y
ZXF1ZXN0Ij44MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJhY3RpdmF0YWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNh
bl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxj
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Jv
eCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNp
YmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNw
YWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
UmFkaW9CdXR0b24iIGlkPSJyb2xlLW1vZGVyYXRvciI+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5Nb2Rl
cmF0b3I8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZHJhd19pbmRpY2F0
b3IiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9Imdyb3VwIj5yb2xlLXBhcnRpY2lwYW50PC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0idG9nZ2xlZCIgaGFuZGxl
cj0ib25fcm9sZV9vcHRpb25fY2hhbmdlZCIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVs
IiB0cmFuc2xhdGFibGU9InllcyI+QSBtb2RlcmF0b3IgY2FuIGtpY2sgdXNlcnMsIGdyYW50IG9yIHJl
dm9rZSB2b2ljZSBhbmQgYXNzaWduIG1vZGVyYXRvcnNoaXAuIEFkbWluaXN0cmF0b3JzIGFuZCBvd25l
cnMgd2lsbCBhdXRvbWF0aWNhbGx5IGJlY29tZSBtb2RlcmF0b3JzIHdoZW4gdGhleSBlbnRlciBhIHJv
b20sIGJ1dCB0aGV5IGNhbiBhbHNvIG1ha2Ugb3RoZXIgb2NjdXBhbnRzIGludG8gbW9kZXJhdG9ycy48
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9IndyYXAiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX2NoYXJzIj4wPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXhfd2lkdGhfY2hh
cnMiPjQwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsaW5lcyI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHN0eWxlPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2xhc3MgbmFtZT0iaGVscC10ZXh0
Ii8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3N0eWxlPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0xpc3RCb3hSb3ciPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrU2VwYXJhdG9yIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0
Qm94Um93Ij4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id2lk
dGhfcmVxdWVzdCI+MTAwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iaGVpZ2h0X3JlcXVlc3QiPjgwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImFjdGl2YXRhYmxlIj5G
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InNlbGVjdGFibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmll
bnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtSYWRpb0J1dHRvbiIgaWQ9InJvbGUtcGFydGljaXBhbnQiPgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVs
IiB0cmFuc2xhdGFibGU9InllcyI+UGFydGljaXBhbnQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InJlY2VpdmVzX2RlZmF1bHQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24i
PmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iYWN0aXZlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJkcmF3X2luZGljYXRvciI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZ3JvdXAiPnJvbGUtcGFydGljaXBhbnQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJ0b2dnbGVkIiBoYW5kbGVyPSJvbl9yb2xlX29w
dGlvbl9jaGFuZ2VkIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJs
ZT0ieWVzIj5JbiBhIG1vZGVyYXRlZCByb29tLCBvbmx5IHBhcnRpY2lwYW50cyBhbmQgbW9kZXJhdG9y
cyBjYW4gc2VuZCBtZXNzYWdlcyB0byB0aGUgcm9vbS4gSW4gbm9uLW1vZGVyYXRlZCByb29tcywgdGhl
cmUncyBubyBkaWZmZXJlbmNlIGJldHdlZW4gYSBwYXJ0aWNpcGFudCBhbmQgYSB2aXNpdG9yLjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0id2lkdGhfY2hhcnMiPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1heF93aWR0aF9jaGFycyI+
NDA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImxpbmVzIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8c3R5bGU+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjbGFzcyBuYW1lPSJoZWxwLXRleHQiLz4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTGlzdEJveFJvdyI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtTZXBhcmF0b3IiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xpc3RCb3hS
b3ciPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9y
ZXF1ZXN0Ij4xMDA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJoZWlnaHRfcmVxdWVzdCI+ODA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYWN0aXZhdGFibGUiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
c2VsZWN0YWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgiPjEyPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0
aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a1JhZGlvQnV0dG9uIiBpZD0icm9sZS12aXNpdG9yIj4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNs
YXRhYmxlPSJ5ZXMiPlZpc2l0b3I8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2Rl
ZmF1bHQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZHJhd19pbmRpY2F0b3IiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imdyb3VwIj5yb2xlLXBhcnRpY2lwYW50PC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0i
dG9nZ2xlZCIgaGFuZGxlcj0ib25fcm9sZV9vcHRpb25fY2hhbmdlZCIgc3dhcHBlZD0ibm8iLz4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2No
aWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+SW4gYSBtb2RlcmF0ZWQgcm9vbSwgYSB2
aXNpdG9yIHdpbGwgbm90IGJlIGFibGUgdG8gc2VuZCBtZXNzYWdlcywgYnV0IGNhbiBzdGlsbCByZWFk
IHRoZSBjb252ZXJzYXRpb24uIEluIGFuIHVubW9kZXJhdGVkIHJvb20gdGhlcmUgaXMgbm8gZGlmZmVy
ZW5jZSBiZXR3ZWVuIGEgcGFydGljaXBhbnQgYW5kIGEgdmlzaXRvci48L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXAiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9IndpZHRoX2NoYXJzIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXhfd2lkdGhfY2hhcnMiPjQwPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsaW5l
cyI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ieGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8Y2xhc3MgbmFtZT0iaGVscC10ZXh0Ii8+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8L3N0eWxlPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0icGFkZGluZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
IDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InJvbGUtcmVhc29uIj4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjY8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0icm9sZS1yZWFzb24tbGFiZWwiPgogICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkhl
cmUgeW91IGNhbiBwcm92aWRlIGFuIG9wdGlvbmFsIHJlYXNvbiBmb3IgdGhlIGNoYW5nZS4gRXZlcnlv
bmUgaW4gdGhlIHJvb20gd2lsbCBzZWUgdGhpcyByZWFzb24uPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXAiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id2lkdGhfY2hhcnMiPjA8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWF4X3dpZHRoX2NoYXJzIj40MDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsaW5lcyI+MjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ4YWxpZ24iPjA8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWduIj4wPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQi
PkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2Nyb2xsZWRXaW5kb3ciPgogICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoc2Nyb2xsYmFyX3BvbGljeSI+bmV2ZXI8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2hhZG93X3R5
cGUiPmluPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RleHRWaWV3IiBpZD0icm9sZS1yZWFzb24t
ZW50cnkiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBpeGVsc19pbnNpZGVfd3JhcCI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idG9wX21hcmdpbiI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm90dG9tX21hcmdpbiI+MzwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaW5kZW50Ij42PC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJhY2NlcHRzX3RhYiI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICA8
L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBhZGRpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+
CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFkZGluZyI+MTI8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+
CiAgICA8c3R5bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9zdHlsZT4KICA8L29i
amVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomRoster.xml": {
		local:   "definitions/MUCRoomRoster.xml",
		size:    5652,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb3N0ZXItdmlldyI+CiAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0i
aGV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InZleHBhbmQiPlRydWU8
L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVy
dHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InJvc3Rlci1tYWlu
LXBhbmVsIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2Nyb2xsZWRXaW5kb3ciIGlkPSJy
b3N0ZXItc2Nyb2xsZWQtd2luZG93Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaHNjcm9sbGJhcl9wb2xp
Y3kiPm5ldmVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImtpbmV0aWNfc2Ny
b2xsaW5nIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtUcmVlVmlldyIgaWQ9InJvc3Rlci10cmVlLXZpZXciPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iaGVhZGVyc192aXNpYmxlIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGVhZGVyc19jbGlja2FibGUiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmRlcl9jb2x1bW4iPmV4cGFuZGVyLWNv
bHVtbjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VhcmNoX2NvbHVt
biI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZW5hYmxlX2dyaWRf
bGluZXMiPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0
b29sdGlwX2NvbHVtbiI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
YWN0aXZhdGVfb25fc2luZ2xlX2NsaWNrIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxz
aWduYWwgbmFtZT0icm93LWFjdGl2YXRlZCIgaGFuZGxlcj0ib25fb2NjdXBhbnRfdHJlZV92aWV3X3Jv
d19hY3RpdmF0ZWQiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICA8Y2hpbGQgaW50ZXJuYWwt
Y2hpbGQ9InNlbGVjdGlvbiI+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVT
ZWxlY3Rpb24iPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtb2RlIj5ub25lPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtUcmVlVmlld0NvbHVtbiIgaWQ9ImV4cGFuZGVyLWNvbHVtbiI+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5FeHBhbmRlcjwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImFsaWdubWVudCI+MC41PC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtDZWxsUmVuZGVyZXJQaXhidWYiIGlkPSJoaWRkZW4tZXhwYW5kZXItcmVuZGVy
ZXIiPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+RmFsc2U8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAg
ICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICA8L2No
aWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtUcmVlVmlld0NvbHVtbiIgaWQ9Im9jY3VwYW50LWNvbHVtbiI+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+T2NjdXBhbnQ8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0NlbGxSZW5kZXJlclBpeGJ1ZiIgaWQ9ImV4cGFuZGVyLXJlbmRlcmVyIj4KICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoIj4zMjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ4cGFkIj4xMjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ4YWxpZ24iPjE8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8YXR0cmli
dXRlcz4KICAgICAgICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJjZWxsLWJhY2tncm91
bmQiPjY8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ2
aXNpYmxlIj4xMjwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5h
bWU9Imljb24tbmFtZSI+MTE8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAgICAgIDwvYXR0cmli
dXRlcz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NlbGxSZW5kZXJlclBpeGJ1
ZiIgaWQ9ImltYWdlLXJlbmRlcmVyIj4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9IndpZHRoIj41ODwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ4cGFkIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InhhbGlnbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgICAgICA8YXR0
cmlidXRlIG5hbWU9ImNlbGwtYmFja2dyb3VuZCI+NjwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAg
ICAgICAgICA8YXR0cmlidXRlIG5hbWU9InZpc2libGUiPjg8L2F0dHJpYnV0ZT4KICAgICAgICAgICAg
ICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJwaXhidWYiPjA8L2F0dHJpYnV0ZT4KICAgICAgICAg
ICAgICAgICAgICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0NlbGxSZW5kZXJlclRleHQiIGlkPSJ0ZXh0LXJlbmRlcmVyIj4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InlwYWQiPjM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZWxsaXBzaXplIj5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2luZ2xlX3BhcmFncmFwaF9tb2RlIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAg
PGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0iYmFja2dy
b3VuZCI+NjwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9
ImZvcmVncm91bmQiPjU8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0
ZSBuYW1lPSJ0ZXh0Ij4xMDwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAgICA8YXR0cmli
dXRlIG5hbWU9IndlaWdodCI+NDwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAgPC9hdHRy
aWJ1dGVzPgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2VsbFJlbmRlcmVyVGV4
dCIgaWQ9InJvbGUtcmVuZGVyZXIiPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ieHBhZCI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ieGFsaWduIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgICAgIDxhdHRy
aWJ1dGUgbmFtZT0idmlzaWJsZSI+OTwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgICAgICA8
YXR0cmlidXRlIG5hbWU9ImJhY2tncm91bmQiPjY8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgICAg
ICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJmb3JlZ3JvdW5kIj43PC9hdHRyaWJ1dGU+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0idGV4dCI+MjwvYXR0cmlidXRlPgogICAgICAgICAg
ICAgICAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAg
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAg
ICA8L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3Bh
Y2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRp
b24iPjE8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogIDwvb2JqZWN0Pgo8
L2ludGVyZmFjZT4K
`,
	},

	"/definitions/MUCRoomRosterInfo.xml": {
		local:   "definitions/MUCRoomRosterInfo.xml",
		size:    36583,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtQb3BvdmVyIiBpZD0ib2NjdXBhbnQtaW5mby1w
b3BvdmVyIj4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij4yMDA8L3Byb3BlcnR5Pgog
ICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVy
X3dpZHRoIj4xMjwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52
ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjQ8L3Byb3Bl
cnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9Im5p
Y2tuYW1lLXBvcG92ZXItYm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFn
ZSI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9vbHRpcF90ZXh0IiB0cmFuc2xhdGFibGU9
InllcyI+Tmlja25hbWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZh
bGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJn
aW5fdG9wIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpY29uX25h
bWUiPmF2YXRhci1kZWZhdWx0LXN5bWJvbGljPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ1c2VfZmFsbGJhY2siPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIg
aWQ9Im5pY2tuYW1lLXBvcG92ZXItbGFiZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRv
b2x0aXBfdGV4dCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPk5pY2tuYW1lPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idHJhY2tfdmlzaXRlZF9saW5rcyI+RmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ieWFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxhdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9IndlaWdodCIgdmFs
dWU9ImJvbGQiLz4KICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJzaXplIiB2YWx1ZT0i
MTQwMDAiLz4KICAgICAgICAgICAgICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+
MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+
MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJ1c2VyLWppZC1wb3BvdmVy
LWJveCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InRvb2x0aXBfdGV4dCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFjY291bnQg
YWRkcmVzczwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5j
ZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl90b3Ai
PjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25fbmFtZSI+Z29h
LXBhbmVsLXN5bWJvbGljPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1
c2VfZmFsbGJhY2siPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
ICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InVzZXItamlk
LXBvcG92ZXItbGFiZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvb2x0aXBfdGV4dCIg
dHJhbnNsYXRhYmxlPSJ5ZXMiPkFjY291bnQgYWRkcmVzczwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9IndyYXAiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9IndyYXBfbW9kZSI+Y2hhcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0idHJhY2tfdmlzaXRlZF9saW5rcyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ieWFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxzdHlsZT4KICAg
ICAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9Im9jY3VwYW50LWppZCIvPgogICAgICAgICAgICAgICAg
PC9zdHlsZT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tp
bmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCb3giIGlkPSJzdGF0dXMtcG9wb3Zlci1ib3giPgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFj
aW5nIj42PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a0ltYWdlIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b29sdGlwX3RleHQi
IHRyYW5zbGF0YWJsZT0ieWVzIj5TdGF0dXM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJtYXJnaW5fdG9wIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJpY29uX25hbWUiPmVtb2ppLWZsYWdzLXN5bWJvbGljPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2VfZmFsbGJhY2siPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtMYWJlbCIgaWQ9InN0YXR1cy1wb3BvdmVyLWxhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idHJhY2tfdmlzaXRlZF9saW5rcyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+
CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9u
Ij4yPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAg
IDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InN0YXR1cy1tZXNzYWdl
LXBvcG92ZXItYm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSI+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9vbHRpcF90ZXh0IiB0cmFuc2xhdGFibGU9InllcyI+
U3RhdHVzPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcGFjaXR5Ij4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3RvcCI+MjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9uYW1lIj5lbW9qaS1mbGFn
cy1zeW1ib2xpYzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX2Zh
bGxiYWNrIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAg
ICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJzdGF0dXMtbWVzc2Fn
ZS1wb3BvdmVyLWxhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRl
cjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idHJhY2tfdmlzaXRlZF9s
aW5rcyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAg
ICAgICA8Y2xhc3MgbmFtZT0ic3RhdHVzLW1lc3NhZ2UiLz4KICAgICAgICAgICAgICAgIDwvc3R5bGU+
CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4K
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjM8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICA8L29iamVjdD4KICA8b2Jq
ZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb3N0ZXItaW5mby1ib3giPgogICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52
ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIg
aWQ9InJvc3Rlci1pbmZvLXRvcC1ib3giPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2Fs
PC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3gi
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGVpZ2h0X3JlcXVlc3QiPjYxPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJy
b3N0ZXItaW5mby1hY3Rpb25zLWJveCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsYXlvdXRfc3R5bGUiPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iPgogICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3Rh
cnQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQi
IGhhbmRsZXI9Im9uX2hpZGUiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgICAgICAgIDxj
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjM8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSI+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpY29uX25hbWUiPmdvLXBy
ZXZpb3VzLXN5bWJvbGljPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBv
c2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTGFiZWwiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5CYWNrPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Im5vbl9ob21vZ2VuZW91cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3Bh
Y2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJvY2N1cGFudC1pbmZvLWFjdGlv
bnMtYm94Ij4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVy
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+Njwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrTWVudUJ1dHRvbiIgaWQ9Im9jY3VwYW50LWluZm8tbWVudS1idXR0b24i
PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmb2N1c19vbl9jbGlj
ayI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
cmVjZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ0b29sdGlwX3RleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5JbmZvcm1hdGlvbjwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3BvdmVyIj5v
Y2N1cGFudC1pbmZvLXBvcG92ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25fbmFtZSI+
ZGlhbG9nLWluZm9ybWF0aW9uLXN5bWJvbGljPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDwvY2hp
bGQ+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVj
dCBjbGFzcz0iR3RrTWVudUJ1dHRvbiIgaWQ9Im9jY3VwYW50LWFjdGlvbnMtbWVudS1idXR0b24iPgog
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmb2N1c19vbl9jbGljayI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVj
ZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJwb3B1cCI+cm9zdGVyLWluZm8tb2NjdXBhbnQtbWVudTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2VfcG9wb3ZlciI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25fbmFtZSI+b3Blbi1tZW51LXN5bWJvbGljPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAg
ICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAg
ICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBh
bmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJw
b3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tp
bmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2VwYXJhdG9yIj4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBh
bmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwv
cHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJvY2N1cGFudC1pbmZvLWJveCI+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTI8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9Im9jY3VwYW50
LWF2YXRhci1ib3giPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0
aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0ltYWdlIiBpZD0iYXZhdGFyLWltYWdlIj4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fdG9wIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2JvdHRvbSI+MjwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBpeGVsX3NpemUiPjgwPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9uYW1lIj5ndGstbWlzc2luZy1pbWFnZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25fc2l6ZSI+Njwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a0JveCIgaWQ9Im9jY3VwYW50LWJhc2ljLWluZm8iPgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ibmlja25hbWUtbGFi
ZWwiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5jZW50ZXI8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJqdXN0aWZ5Ij5jZW50ZXI8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0cmFja192aXNpdGVk
X2xpbmtzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZXM+CiAg
ICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9IndlaWdodCIgdmFsdWU9ImJvbGQiLz4K
ICAgICAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0ic2l6ZSIgdmFsdWU9IjE0MDAwIi8+
CiAgICAgICAgICAgICAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgICAgIDxzdHls
ZT4KICAgICAgICAgICAgICAgICAgICAgIDxjbGFzcyBuYW1lPSJvY2N1cGFudC1uaWNrbmFtZSIvPgog
ICAgICAgICAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJw
b3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ic3RhdHVzLWxhYmVsIj4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InRvb2x0aXBfdGV4dCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlN0YXR1czwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+Y2VudGVy
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+Y2Vu
dGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idHJhY2tfdmlz
aXRlZF9saW5rcyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxzdHlsZT4KICAg
ICAgICAgICAgICAgICAgICAgIDxjbGFzcyBuYW1lPSJvY2N1cGFudC1zdGF0dXMiLz4KICAgICAgICAg
ICAgICAgICAgICA8L3N0eWxlPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAg
ICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8
L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InBhZGRpbmciPjE1PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAg
ICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9w
cm9wZXJ0eT4KICAgICAgPC9wYWNraW5nPgogICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrU2VwYXJhdG9yIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwv
cHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a1Njcm9sbGVkV2luZG93IiBpZD0icm9zdGVyLWluZm8tc2Nyb2xsZWQtd2lu
ZG93Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrVmlld3BvcnQiIGlkPSJyb3N0ZXIt
aW5mby12aWV3cG9ydCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNoYWRvd190eXBlIj5ub25lPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Jv
eCIgaWQ9InJvc3Rlci1pbmZvLWNvbnRlbnQtYm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2ZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJv
cmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb3N0ZXItaW5mby1ib2R5Ij4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MTI8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGlj
YWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ZyYW1lIj4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ibGFiZWxfeGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InNoYWRvd190eXBlIj5pbjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMaXN0Qm94Ij4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJzZWxlY3Rpb25fbW9kZSI+bm9uZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8c2lnbmFsIG5hbWU9InJvdy1hY3RpdmF0ZWQiIGhhbmRsZXI9Im9uX29jY3VwYW50X2Fj
dGlvbiIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGlzdEJveFJvdyIgaWQ9
ImNoYW5nZS1hZmZpbGlhdGlvbi1saXN0Ym94LXJvdyI+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoX3JlcXVlc3QiPjEwMDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b29sdGlw
X3RleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5BIHBvc2l0aW9uIGlzIGEgbG9uZy1saXZlZCBhc3NvY2lh
dGlvbiB3aXRoIHRoZSByb29tLiBUaGUgcG9zaXRpb24gb2YgdGhlIHBlcnNvbiBpcyBtYWludGFpbmVk
IGJldHdlZW4gdmlzaXRzLjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2lu
ZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCb3giPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MzwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwi
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+UG9zaXRpb248L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj42PC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9ImN1cnJlbnQtYWZmaWxpYXRpb24iPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJoYWxpZ24iPmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5
ZXMiPkFkbWluaXN0cmF0b3I8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9ImNoYW5nZS1hZmZpbGlhdGlvbi1h
Y3Rpb24taW1hZ2UiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpY29uX25hbWUiPmdv
LW5leHQtc3ltYm9saWM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9j
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJm
aWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2VwYXJh
dG9yIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0Qm94Um93IiBpZD0iY2hh
bmdlLXJvbGUtbGlzdGJveC1yb3ciPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij4xMDA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0YWJsZSI+RmFsc2U8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0
b29sdGlwX3RleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5BIHJvbGUgY29udHJvbHMgd2hhdCBhIHBlcnNv
biBjYW4gZG8gd2l0aGluIHRoZSByb29tLiBBIHJvbGUgb25seSBsYXN0cyBmb3IgdGhlIGR1cmF0aW9u
IG9mIHRoZSB2aXNpdC48L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxj
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Jv
eCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNp
YmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZhbGlnbiI+c3RhcnQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlv
biI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0xhYmVsIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3RhcnQ8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlJvbGU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj42PC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtMYWJlbCIgaWQ9ImN1cnJlbnQtcm9sZSI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImhhbGlnbiI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+T3du
ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9ImNoYW5nZS1yb2xlLWFjdGlvbi1pbWFnZSI+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpY29u
X25hbWUiPmdvLW5leHQtc3ltYm9saWM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTGFiZWwiIGlkPSJjaGFuZ2Utcm9sZS1kaXNhYmxlZCI+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3BhY2l0
eSI+MC41PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkFkbWluaXN0cmF0b3JzIGFuZCBv
d25lcnMgd2lsbCBhdXRvbWF0aWNhbGx5IGJlIG1vZGVyYXRvcnMgZm9yIGEgcm9vbS4gVGhlcmVmb3Jl
LCB0aGUgcm9sZSBvZiBKdWFuaXRvIGNhbid0IGJlIGNoYW5nZWQuPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ0cmFja192aXNpdGVkX2xpbmtzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ieGFsaWduIj4wPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxp
Z24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGF0
dHJpYnV0ZXM+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1
dGUgbmFtZT0ic3R5bGUiIHZhbHVlPSJpdGFsaWMiLz4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxzdHlsZT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNs
YXNzIG5hbWU9Im9jY3VwYW50LXJvbGUtZGlzYWJsZWQtaGVscCIvPgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPC9zdHlsZT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAg
IDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNr
aW5nPgogICAgPC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9InJvc3Rlci1pbmZv
LXBhbmVsIi8+CiAgICA8L3N0eWxlPgogIDwvb2JqZWN0PgogIDxvYmplY3QgY2xhc3M9Ikd0a01lbnUi
IGlkPSJyb3N0ZXItaW5mby1vY2N1cGFudC1tZW51Ij4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJtZW51X3R5cGVfaGludCI+bWVudTwv
cHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnVJdGVtIiBpZD0i
a2ljay1vY2N1cGFudCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkV4cGVs
PC9wcm9wZXJ0eT4KICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9raWNr
IiBzd2FwcGVkPSJubyIvPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnVJdGVtIiBpZD0iYmFuLW9jY3VwYW50Ij4KICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+QmFuPC9wcm9wZXJ0eT4KICAgICAgICA8c2lnbmFs
IG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9iYW4iIHN3YXBwZWQ9Im5vIi8+CiAgICAgIDwvb2Jq
ZWN0PgogICAgPC9jaGlsZD4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MUCRoomToolbar.xml": {
		local:   "definitions/MUCRoomToolbar.xml",
		size:    21431,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51IiBpZD0icm9vbS1tZW51Ij4KICAgIDxj
aGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0iIGlkPSJzZWN1cml0eS1wcm9wZXJ0
aWVzLW1lbnUtaXRlbSI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNlY3Vy
aXR5IHByb3BlcnRpZXM8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2VfdW5kZXJs
aW5lIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVy
PSJvbl9zaG93X3NlY3VyaXR5X3Byb3BlcnRpZXMiIHN3YXBwZWQ9Im5vIi8+CiAgICAgIDwvb2JqZWN0
PgogICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0i
IGlkPSJyb29tLWNvbmZpZ3VyYXRpb24tbWVudS1pdGVtIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJh
bnNsYXRhYmxlPSJ5ZXMiPkNvbmZpZ3VyZSByb29tPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkg
bmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHNpZ25hbCBuYW1lPSJh
Y3RpdmF0ZSIgaGFuZGxlcj0ib25fY29uZmlndXJlX3Jvb20iIHN3YXBwZWQ9Im5vIi8+CiAgICAgIDwv
b2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVu
dUl0ZW0iIGlkPSJtb2RpZnktcG9zaXRpb24tbGlzdHMtbWVudS1pdGVtIj4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Imxh
YmVsIiB0cmFuc2xhdGFibGU9InllcyI+TW9kaWZ5IHBvc2l0aW9uczwvcHJvcGVydHk+CiAgICAgICAg
PHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25fbW9kaWZ5X3Bvc2l0aW9uX2xpc3RzIiBz
d2FwcGVkPSJubyIvPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a1NlcGFyYXRvck1lbnVJdGVtIiBpZD0iYWRtaW4tYWN0aW9uLXNlcGFy
YXRvciI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a01lbnVJdGVtIiBpZD0iZGVzdHJveS1yb29tLW1lbnUtaXRlbSI+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIg
dHJhbnNsYXRhYmxlPSJ5ZXMiPkRlc3Ryb3kgcm9vbTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9InVzZV91bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxzaWduYWwgbmFtZT0i
YWN0aXZhdGUiIGhhbmRsZXI9Im9uX2Rlc3Ryb3lfcm9vbSIgc3dhcHBlZD0ibm8iLz4KICAgICAgPC9v
YmplY3Q+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51
SXRlbSIgaWQ9ImxlYXZlLXJvb20tbWVudS1pdGVtIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlz
aWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xh
dGFibGU9InllcyI+TGVhdmUgcm9vbTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InVz
ZV91bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUi
IGhhbmRsZXI9Im9uX2xlYXZlX3Jvb20iIHN3YXBwZWQ9Im5vIi8+CiAgICAgIDwvb2JqZWN0PgogICAg
PC9jaGlsZD4KICA8L29iamVjdD4KICA8b2JqZWN0IGNsYXNzPSJHdGtUZXh0QnVmZmVyIiBpZD0icm9v
bS1zdWJqZWN0LXRleHR2aWV3LWJ1ZmZlciI+CiAgICA8c2lnbmFsIG5hbWU9ImNoYW5nZWQiIGhhbmRs
ZXI9Im9uX3N1YmplY3RfY2hhbmdlZCIgc3dhcHBlZD0ibm8iLz4KICA8L29iamVjdD4KICA8b2JqZWN0
IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLXZpZXctdG9vbGJhciI+CiAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0
eT4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0icm9vbS12aWV3LXRv
b2xiYXItY29udGVudCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5
PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InRvb2xi
YXIiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGVpZ2h0X3JlcXVlc3QiPjYwPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5zdGFydDwvcHJvcGVydHk+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InRvb2xiYXItdG9wIj4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCb3giPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0icm9vbS1pbmZvLWJveCI+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrSW1hZ2UiIGlkPSJyb29tLXN0YXR1cy1pY29uIj4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbl9u
YW1lIj5hcHBsaWNhdGlvbnMtY2hhdDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25fc2l6ZSI+NjwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAg
IDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFu
ZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICA8
L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtC
b3giIGlkPSJyb29tLWRldGFpbHMtYm94Ij4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InJvb20tbmFtZS1sYWJlbCI+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPnN0YXJ0
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZWxsaXBzaXplIj5lbmQ8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0cmFja192aXNpdGVkX2xp
bmtzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InlhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9InJvb20tc3ViamVjdC1idXR0b24iPgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZm9jdXNfb25f
Y2xpY2siPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50
ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJyZWxpZWYiPm5vbmU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ0b29sdGlwX3RleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5TaG93IHJvb20gc3Vi
amVjdDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1l
PSJjbGlja2VkIiBoYW5kbGVyPSJvbl90b2dnbGVfcm9vbV9zdWJqZWN0IiBzd2FwcGVkPSJubyIvPgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdlIiBpZD0icm9vbS1zdWJqZWN0LWJ1dHRv
bi1pbWFnZSI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25fbmFtZSI+Z28tZG93bi1zeW1i
b2xpYzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2lu
Zz4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
ICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtC
b3giIGlkPSJyb29tLW1lbnUtYm94Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a01lbnVCdXR0b24iIGlkPSJyb29tLW1lbnUtYnV0dG9uIj4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3B1cCI+cm9v
bS1tZW51PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ1c2VfcG9wb3ZlciI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFn
ZSI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJpY29uX25hbWUiPm9wZW4tbWVudS1zeW1ib2xpYzwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAg
ICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxk
PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1NlcGFyYXRvciI+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3Bl
cnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0Pgog
ICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwv
Y2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1JldmVhbGVyIiBpZD0icm9v
bS1zdWJqZWN0LXJldmVhbGVyIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCb3giPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxp
Z24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAg
ICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+Y2VudGVyPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4x
MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0icm9vbS1zdWJqZWN0
LWxhYmVsIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlz
aWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9IndyYXAiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwLW1vZGUiPmNoYXI8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idHJhY2stdmlz
aXRlZC1saW5rcyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ4YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBh
Y2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFu
ZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgog
ICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAg
IDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFu
ZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgIDwv
cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCb3giPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxp
Z24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxj
aGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlk
PSJyb29tLWVkaXQtc3ViamVjdC1idXR0b24iPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmb2N1c19vbl9jbGljayI+RmFsc2U8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZh
bGlnbiI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJyZWxpZWYiPm5vbmU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InRvb2x0aXBfdGV4dCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkVkaXQgc3ViamVj
dDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNr
ZWQiIGhhbmRsZXI9Im9uX2VkaXRfcm9vbV9zdWJqZWN0IiBzd2FwcGVkPSJubyIvPgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtJbWFnZSI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25fbmFtZSI+ZG9jdW1lbnQtZWRp
dC1zeW1ib2xpYzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+
MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAg
ICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICA8
L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAg
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrU2Nyb2xsZWRXaW5kb3ciIGlkPSJyb29tLXN1YmplY3Qt
ZWRpdGFibGUtY29udGVudCI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZzY3Jv
bGxiYXItcG9saWN5Ij5HVEtfUE9MSUNZX0FVVE9NQVRJQzwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImhzY3JvbGxiYXItcG9saWN5Ij5HVEtfUE9MSUNZX0FVVE9NQVRJ
QzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNoYWRvd190eXBl
Ij5pbjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1pbl9jb250
ZW50X2hlaWdodCI+NTA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAg
ICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RleHRWaWV3IiBpZD0icm9vbS1zdWJqZWN0
LXRleHR2aWV3Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4
cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ3cmFwLW1vZGUiPkdUS19XUkFQX0NIQVI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibGVmdF9tYXJnaW4iPjY8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icmlnaHRfbWFyZ2luIj42PC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcF9tYXJnaW4iPjY8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm90dG9tX21hcmdpbiI+NjwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJidWZmZXIiPnJvb20tc3Vi
amVjdC10ZXh0dmlldy1idWZmZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0icm9vbS1lZGl0LXN1YmplY3QtYnV0dG9ucy1jb250
YWluZXIiPgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5lbmQ8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0icm9vbS1lZGl0LXN1YmplY3QtY2FuY2VsLWJ1dHRvbiI+
CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxl
PSJ5ZXMiPkNhbmNlbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX2NhbmNlbF9yb29tX3N1Ympl
Y3RfZWRpdCIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAg
ICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICA8
L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICA8
Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJy
b29tLWVkaXQtc3ViamVjdC1hcHBseS1idXR0b24iPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5BcHBseTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbnNpdGl2ZSI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVm
YXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0i
Y2xpY2tlZCIgaGFuZGxlcj0ib25fYXBwbHlfcm9vbV9zdWJqZWN0X2VkaXQiIHN3YXBwZWQ9Im5vIi8+
CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAg
ICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+Mjwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hp
bGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2lu
Zz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a1NlcGFyYXRvciI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAg
ICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MTwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAgPC9vYmplY3Q+Cjwv
aW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomWarningsDialog.xml": {
		local:   "definitions/MUCRoomWarningsDialog.xml",
		size:    12707,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtXaW5kb3ciIGlkPSJyb29tLXdhcm5pbmdzLWRp
YWxvZyI+CiAgICA8cHJvcGVydHkgbmFtZT0id2lkdGhfcmVxdWVzdCI+NjQwPC9wcm9wZXJ0eT4KICAg
IDxwcm9wZXJ0eSBuYW1lPSJyZXNpemFibGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBu
YW1lPSJtb2RhbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0id2luZG93X3Bvc2l0
aW9uIj5jZW50ZXItb24tcGFyZW50PC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJkZWZhdWx0
X3dpZHRoIj42NDA8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImRlZmF1bHRfaGVpZ2h0Ij40
MDA8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImRlc3Ryb3lfd2l0aF9wYXJlbnQiPlRydWU8
L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InR5cGVfaGludCI+ZGlhbG9nPC9wcm9wZXJ0eT4K
ICAgIDxjaGlsZCB0eXBlPSJ0aXRsZWJhciI+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0hlYWRlckJh
ciI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+U2VjdXJp
dHkgcHJvcGVydGllcyB3YXJuaW5nczwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InNo
b3dfY2xvc2VfYnV0dG9uIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxk
PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtPdmVybGF5Ij4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2li
bGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2ZXhwYW5kIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+MjQ8L3Byb3BlcnR5
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjI0PC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4yNDwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MjQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aF9yZXF1ZXN0Ij42NDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0ltYWdlIiBpZD0icm9vbS13YXJuaW5ncy1jdXJyZW50LWljb24iPgogICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPmNlbnRlcjwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPnN0YXJ0PC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBpeGVsX3NpemUi
PjY0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25f
bmFtZSI+ZGlhbG9nLXdhcm5pbmctc3ltYm9saWM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjEyPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtMYWJlbCIgaWQ9InJvb20td2FybmluZ3MtY3VycmVudC10aXRsZSI+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+
Q29tbXVuaWNhdGlvbiBpbiB0aGlzIHJvb20gaXMgbm90IGVuY3J5cHRlZDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3cmFwIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idHJhY2tfdmlzaXRlZF9s
aW5rcyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ieGFsaWduIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InlhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxzdHlsZT4KICAg
ICAgICAgICAgICAgICAgICAgICAgICA8Y2xhc3MgbmFtZT0id2FybmluZy10aXRsZSIvPgogICAgICAg
ICAgICAgICAgICAgICAgICA8L3N0eWxlPgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9
InJvb20td2FybmluZ3MtY3VycmVudC1kZXNjcmlwdGlvbiI+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+UGxlYXNlIGJlIGF3
YXJlIHRoYXQgY29tbXVuaWNhdGlvbiBpbiBjaGF0IHJvb21zIGlzIG5vdCBlbmNyeXB0ZWQgLSBhbnlv
bmUgdGhhdCBjYW4gaW50ZXJjZXB0IGNvbW11bmljYXRpb24gYmV0d2VlbiB5b3UgYW5kIHRoZSBzZXJ2
ZXIgLSBhbmQgdGhlIHNlcnZlciBpdHNlbGYgLSB3aWxsIGJlIGFibGUgdG8gc2VlIHdoYXQgeW91IGFy
ZSBzYXlpbmcgaW4gdGhpcyBjaGF0IHJvb20uIE9ubHkgam9pbiB0aGlzIHJvb20gYW5kIGNvbW11bmlj
YXRlIGhlcmUgaWYgeW91IHRydXN0IHRoZSBzZXJ2ZXIgdG8gbm90IGJlIGhvc3RpbGUuPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXAiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0YWJsZSI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXhfd2lk
dGhfY2hhcnMiPjUwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InRyYWNrX3Zpc2l0ZWRfbGlua3MiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgICA8c3R5bGU+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9Indhcm5p
bmctZGVzY3JpcHRpb24iLz4KICAgICAgICAgICAgICAgICAgICAgICAgPC9zdHlsZT4KICAgICAgICAg
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2lu
Zz4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAg
ICAgICAgICAgIDxzdHlsZT4KICAgICAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9Indhcm5pbmctY29u
dGVudCIvPgogICAgICAgICAgICAgICAgPC9zdHlsZT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtTZXBhcmF0b3IiPgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNwYWNpbmciPjI0PC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtMYWJlbCIgaWQ9InJvb20td2FybmluZ3MtY3VycmVudC1pbmZvIj4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0id3JhcCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRyYWNrX3Zpc2l0ZWRfbGlu
a3MiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InhhbGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ5YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8c3R5bGU+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgPGNsYXNzIG5hbWU9Indhcm5pbmctY3VycmVudC1pbmZvIi8+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9u
Qm94IiBpZD0icm9vbS13YXJuaW5ncy1tb3ZlLWJ1dHRvbnMiPgogICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCdXR0b24iIGlkPSJyb29tLXdhcm5pbmdzLW1vdmUtcHJldmlvdXMtYnV0dG9uIj4KICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ0b29sdGlwX3RleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5HbyB0byBwcmV2aW91cyBbQ3RybCArIGxl
ZnRdPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xp
Y2tlZCIgaGFuZGxlcj0ib25fd2FybmluZ19nb19wcmV2aW91c19jbGlja2VkIiBzd2FwcGVkPSJubyIv
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSI+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpY29uX25hbWUiPmdvLXByZXZpb3VzLXN5
bWJvbGljPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAg
ICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAg
ICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJyb29tLXdhcm5pbmdz
LW1vdmUtbmV4dC1idXR0b24iPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvb2x0aXBfdGV4dCIgdHJhbnNsYXRhYmxlPSJ5
ZXMiPkdvIHRvIG5leHQgW0N0cmwgKyByaWdodF08L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl93YXJuaW5nX2dvX25leHRf
Y2xpY2tlZCIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4K
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGl4
ZWxfc2l6ZSI+MTY8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJpY29uX25hbWUiPmdvLW5leHQtc3ltYm9saWM8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
ICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPkZhbHNlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgICAgICA8
L2NoaWxkPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgPHBhY2tp
bmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9InBvc2l0aW9uIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxzdHlsZT4KICAgICAgICAgICAgICA8Y2xhc3MgbmFt
ZT0id2FybmluZy1jb250ZW50Ii8+CiAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaW5kZXgiPi0x
PC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxj
aGlsZCB0eXBlPSJvdmVybGF5Ij4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiI+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0idG9vbHRpcF90ZXh0IiB0cmFuc2xhdGFibGU9InllcyI+Q2xvc2Ug
ZGlhbG9nPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+ZW5kPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZhbGlnbiI+c3RhcnQ8L3Byb3BlcnR5
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4xMjwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fZGlhbG9nX2Nsb3NlIiBz
d2FwcGVkPSJubyIvPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrSW1hZ2UiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb25fbmFtZSI+d2luZG93
LWNsb3NlLXN5bWJvbGljPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgPC9jaGlsZD4KICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAgIDxjbGFzcyBuYW1lPSJ3
YXJuaW5ncy1kaWFsb2ctY2xvc2UiLz4KICAgICAgICAgICAgPC9zdHlsZT4KICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHls
ZT4KICAgICAgPGNsYXNzIG5hbWU9Indhcm5pbmdzLWRpYWxvZyIvPgogICAgICA8Y2xhc3MgbmFtZT0i
Y295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomWarningsOverlay.xml": {
		local:   "definitions/MUCRoomWarningsOverlay.xml",
		size:    5558,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtSZXZlYWxlciIgaWQ9InJldmVhbGVyIj4KICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1l
PSJ0cmFuc2l0aW9uX3R5cGUiPnNsaWRlLXJpZ2h0PC9wcm9wZXJ0eT4KICAgIDxjaGlsZD4KICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwv
cHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2xlZnQiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fcmlnaHQiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2ZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94Ij4KICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2ZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTGFiZWwiPgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxp
Z24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPldhcm5pbmdzIHJlbGF0ZWQgdG8gcm9vbSBjb25maWd1
cmF0aW9uPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Inhh
bGlnbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ5
YWxpZ24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCI+CiAgICAg
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxheW91dF9zdHlsZSI+c3RhcnQ8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAg
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIj4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNsb3NlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNl
aXZlc19kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxz
aWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib25fY2xvc2UiIHN3YXBwZWQ9Im5vIi8+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPHBh
Y2tpbmc+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJm
aWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJm
aWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwYWRkaW5n
Ij4xMjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8
L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1NlcGFyYXRvciI+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a1Njcm9sbGVkV2luZG93Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrVmlld3BvcnQiPgogICAgICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ic2hhZG93X3R5cGUiPm5vbmU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9Indh
cm5pbmdzQm94Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGV4
cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2ZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9w
ZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4K
ICAgIDwvY2hpbGQ+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/MUCRoomWindow.xml": {
		local:   "definitions/MUCRoomWindow.xml",
		size:    3756,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtXaW5kb3ciIGlkPSJyb29tLXdpbmRvdyI+CiAg
ICA8cHJvcGVydHkgbmFtZT0id2lkdGhfcmVxdWVzdCI+OTAwPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0
eSBuYW1lPSJoZWlnaHRfcmVxdWVzdCI+NjAwPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3
aW5kb3dfcG9zaXRpb24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0iZ3Jhdml0
eSI+Y2VudGVyPC9wcm9wZXJ0eT4KICAgIDxzaWduYWwgbmFtZT0iZGVsZXRlLWV2ZW50IiBoYW5kbGVy
PSJvbl9iZWZvcmVfZGVsZXRlIiBzd2FwcGVkPSJubyIvPgogICAgPHNpZ25hbCBuYW1lPSJkZXN0cm95
IiBoYW5kbGVyPSJvbl9kZXN0cm95X3dpbmRvdyIgc3dhcHBlZD0ibm8iLz4KICAgIDxjaGlsZD4KICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrT3ZlcmxheSIgaWQ9InJvb20tb3ZlcmxheSI+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InJvb20tY29udGVudC1vdmVybGF5LWJveCI+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLXBy
aXZhY3ktd2FybmluZ3MtYm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlv
biI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
IDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLWxvYWRpbmctbm90
aWZpY2F0aW9uLWJveCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVy
dGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAg
ICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJyb29tLW1haW4tYm94Ij4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpbmRleCI+LTE8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkIHR5cGU9Im92ZXJsYXkiPgogICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrT3ZlcmxheSIgaWQ9InJvb20tbm90aWZpY2F0aW9ucy1vdmVybGF5
Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0icm9v
bS1ub3RpZmljYXRpb25zLW92ZXJsYXktYm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2
YWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmll
bnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtSZXZlYWxlciIgaWQ9InJvb20tbm90aWZpY2F0aW9u
cy1yZXZlYWxlciI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAg
PHBhY2tpbmc+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgICAgICA8L2NoaWxk
PgogICAgICAgICAgICAgICAgPHN0eWxlPgogICAgICAgICAgICAgICAgICA8Y2xhc3MgbmFtZT0icm9v
bS1ub3RpZmljYXRpb25zLXdyYXBwZXIiLz4KICAgICAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaW5kZXgiPi0xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAg
ICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFzc190aHJvdWdoIj5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImluZGV4Ij4zPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8c3R5
bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJtdWMiLz4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAg
ICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/Main.xml": {
		local:   "definitions/Main.xml",
		size:    12838,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtBcHBsaWNhdGlvbldpbmRvdyIgaWQ9Im1haW5X
aW5kb3ciPgogICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAg
PHByb3BlcnR5IG5hbWU9InRpdGxlIj5Db3lJTTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0i
ZGVmYXVsdF93aWR0aCI+MjAwPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJkZWZhdWx0X2hl
aWdodCI+NjAwPC9wcm9wZXJ0eT4KICAgIDxzaWduYWwgbmFtZT0iZGVzdHJveSIgaGFuZGxlcj0ib25f
Y2xvc2Vfd2luZG93IiBzd2FwcGVkPSJubyIvPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCb3giIGlkPSJIYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5
PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9IlZib3gi
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVydGljYWw8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUJhciIg
aWQ9Im1lbnViYXIiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtNZW51SXRlbSIgaWQ9IkNvbnRhY3RzTWVudSI+CiAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPl9Db250YWN0czwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InVzZV91bmRlcmxpbmUi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZCB0eXBlPSJzdWJtZW51Ij4K
ICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnUiIGlkPSJtZW51Ij4KICAg
ICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0iIGlkPSJhZGRNZW51Ij4KICAgICAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxl
PSJ5ZXMiPkFkZC4uLjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8c2lnbmFs
IG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9hZGRfY29udGFjdF93aW5kb3ciIHN3YXBwZWQ9Im5v
Ii8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAg
ICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51SXRlbSIgaWQ9Im5ld0NvbnZNZW51Ij4KICAg
ICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIg
dHJhbnNsYXRhYmxlPSJ5ZXMiPk5ldyBDb252ZXJzYXRpb24uLi48L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25fbmV3X2Nv
bnZlcnNhdGlvbiIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2Jq
ZWN0PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0iIGlkPSJBY2NvdW50c01lbnUiPgog
ICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0i
eWVzIj5fQWNjb3VudHM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ1c2VfdW5kZXJsaW5lIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51SXRlbSIgaWQ9IkNoYXRSb29tc01lbnUiPgogICAg
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVz
Ij5DaGF0IF9Sb29tczwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InVzZV91bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZCB0
eXBlPSJzdWJtZW51Ij4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnUi
IGlkPSJDaGF0Um9vbXNTdWJtZW51Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0iIGlk
PSJtdWMtY3JlYXRlLWNoYXQtcm9vbS1tZW51Ij4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkNyZWF0ZSBS
b29tPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1
c2VfdW5kZXJsaW5lIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxz
aWduYWwgbmFtZT0iYWN0aXZhdGUiIGhhbmRsZXI9Im9uX2NyZWF0ZV9jaGF0X3Jvb20iIHN3YXBwZWQ9
Im5vIi8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51SXRlbSIgaWQ9InNob3dQdWJsaWNSb29t
c01lbnVJdGVtIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5f
Zm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNob3cgcHVibGljIHJvb21zLi4uPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUiIGhh
bmRsZXI9Im9uX211Y19zaG93X3B1YmxpY19yb29tcyIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAg
ICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a01lbnVJdGVtIiBpZD0iam9pblJvb21NZW51SXRlbSI+CiAgICAgICAgICAgICAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJs
ZT0ieWVzIj5Kb2luIGEgUm9vbTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
c2lnbmFsIG5hbWU9ImFjdGl2YXRlIiBoYW5kbGVyPSJvbl9tdWNfc2hvd19qb2luX3Jvb20iIHN3YXBw
ZWQ9Im5vIi8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAg
ICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
ICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a01lbnVJdGVtIiBpZD0iVmlld01lbnUiPgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5fVmlldzwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InVzZV91bmRlcmxpbmUiPlRydWU8L3By
b3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZCB0eXBlPSJzdWJtZW51Ij4KICAgICAgICAg
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnUiIGlkPSJtZW51MiI+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0NoZWNrTWVudUl0ZW0iIGlkPSJDaGVja0l0ZW1NZXJnZSI+CiAgICAgICAgICAg
ICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0
YWJsZT0ieWVzIj5NZXJnZSBBY2NvdW50czwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8c2lnbmFsIG5hbWU9InRvZ2dsZWQiIGhhbmRsZXI9Im9uX3RvZ2dsZWRfY2hlY2tfSXRlbV9N
ZXJnZSIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxk
PgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NoZWNrTWVudUl0ZW0i
IGlkPSJDaGVja0l0ZW1TaG93T2ZmbGluZSI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5TaG93IE9mZmxp
bmUgQ29udGFjdHM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBu
YW1lPSJ0b2dnbGVkIiBoYW5kbGVyPSJvbl90b2dnbGVkX2NoZWNrX0l0ZW1fU2hvd19PZmZsaW5lIiBz
d2FwcGVkPSJubyIvPgogICAgICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2hlY2tNZW51SXRlbSIgaWQ9IkNo
ZWNrSXRlbVNob3dXYWl0aW5nIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNob3cgV2FpdGluZyBDb250
YWN0czwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9InRv
Z2dsZWQiIGhhbmRsZXI9Im9uX3RvZ2dsZWRfY2hlY2tfSXRlbV9TaG93X1dhaXRpbmciIHN3YXBwZWQ9
Im5vIi8+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDaGVja01lbnVJdGVtIiBpZD0iQ2hlY2tJdGVt
U29ydFN0YXR1cyI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2Fu
X2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5Tb3J0IEJ5IFN0YXR1czwvcHJvcGVydHk+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9InRvZ2dsZWQiIGhhbmRsZXI9
Im9uX3RvZ2dsZWRfY2hlY2tfSXRlbV9Tb3J0X0J5X1N0YXR1cyIgc3dhcHBlZD0ibm8iLz4KICAgICAg
ICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgIDwvY2hp
bGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVu
dUl0ZW0iIGlkPSJPcHRpb25zTWVudSI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPl9PcHRpb25zPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPGNoaWxkIHR5cGU9InN1Ym1lbnUiPgogICAgICAgICAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudSIgaWQ9Im9wdGlvbnNfc3VibWVudSI+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0NoZWNrTWVudUl0ZW0iIGlkPSJFbmNyeXB0Q29uZmlndXJhdGlvbkZpbGVDaGVj
a01lbnVJdGVtIj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5f
Zm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkVuY3J5cHQgY29uZmlndXJhdGlvbiBmaWxl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0idG9nZ2xl
ZCIgaGFuZGxlcj0ib25fdG9nZ2xlZF9lbmNyeXB0X2NvbmZpZ3VyYXRpb25fZmlsZSIgc3dhcHBlZD0i
bm8iLz4KICAgICAgICAgICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAg
ICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnVJdGVtIiBpZD0icHJlZmVyZW5jZXNNZW51
SXRlbSI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3Vz
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5QcmVmZXJlbmNlcy4uLjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImFjdGl2YXRl
IiBoYW5kbGVyPSJvbl9wcmVmZXJlbmNlcyIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgICAgICAg
ICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAg
IDxjaGlsZD4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0iIGlkPSJI
ZWxwTWVudSI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJh
bnNsYXRhYmxlPSJ5ZXMiPl9IZWxwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0idXNlX3VuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAg
PGNoaWxkIHR5cGU9InN1Ym1lbnUiPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrTWVudSIgaWQ9Im1lbnUzIj4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8Y2hpbGQ+
CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0iIGlkPSJm
ZWVkYmFja01lbnUiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNh
bl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+RmVlZGJhY2s8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib25f
ZmVlZGJhY2tfZGlhbG9nIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAgICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTWVu
dUl0ZW0iIGlkPSJhYm91dE1lbnUiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+QWJvdXQ8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxl
cj0ib25fYWJvdXRfZGlhbG9nIiBzd2FwcGVkPSJubyIvPgogICAgICAgICAgICAgICAgICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4
cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4w
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJz
ZWFyY2gtYm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+dmVy
dGljYWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtTZWFyY2hCYXIiIGlkPSJzZWFyY2gtYXJlYSI+CiAgICAgICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Vu
dHJ5IiBpZD0ic2VhcmNoLWVudHJ5Ij4KICAgICAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzaGFkb3dfdHlwZSI+ZXRjaGVkLWluPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlY29uZGFyeV9pY29uX25hbWUiPmVkaXQtZmlu
ZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwbGFjZWhv
bGRlcl90ZXh0IiB0cmFuc2xhdGFibGU9InllcyI+U2VhcmNoLi4uPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmpl
Y3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
ZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFja190eXBl
Ij5lbmQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4x
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJu
b3RpZmljYXRpb24tYXJlYSI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3Vz
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRp
b24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxzdHlsZT4KICAgICAgICAgICAg
ICAgICAgPGNsYXNzIG5hbWU9Im5vdGlmaWNhdGlvbnMiLz4KICAgICAgICAgICAgICAgIDwvc3R5bGU+
CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBhY2tfdHlwZSI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgog
ICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAg
ICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8c3R5bGU+CiAgICAgIDxj
bGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9zdHlsZT4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/MainPassword.xml": {
		local:   "definitions/MainPassword.xml",
		size:    2376,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJNYWluUGFzc3dvcmQiPgog
ICAgPHByb3BlcnR5IG5hbWU9IndpbmRvdy1wb3NpdGlvbiI+R1RLX1dJTl9QT1NfQ0VOVEVSPC9wcm9w
ZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPkVudGVyIG1h
aW4gcGFzc3dvcmQ8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImRlZmF1bHQtd2lkdGgiPjMw
MDwvcHJvcGVydHk+CiAgICA8c2lnbmFsIG5hbWU9ImRlbGV0ZS1ldmVudCIgaGFuZGxlcj0ib25fY2Fu
Y2VsIiAvPgogICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrQm94IiBpZD0iVmJveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbiI+MTA8L3By
b3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xMDwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImhvbW9nZW5lb3VzIj5mYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fVkVSVElDQUw8L3Byb3BlcnR5
PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0icGFz
c01lc3NhZ2UiID4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9
InllcyI+UGxlYXNlIGVudGVyIHRoZSBtYWluIHBhc3N3b3JkIGZvciB0aGUgY29uZmlndXJhdGlvbiBm
aWxlLiBZb3Ugd2lsbCBub3QgYmUgYXNrZWQgZm9yIHRoaXMgcGFzc3dvcmQgYWdhaW4gdW50aWwgeW91
IHJlc3RhcnQgQ295SU0uPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndyYXAi
PnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+ZmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
PC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFbnRyeSIg
aWQ9InBhc3N3b3JkIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imhhcy1mb2N1cyI+dHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmlsaXR5Ij5mYWxzZTwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUiIGhhbmRsZXI9Im9uX3NhdmUi
IC8+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZXhwYW5kIj5mYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJmaWxsIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0
aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAg
ICA8L29iamVjdD4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9h
cmVhIj4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94IiBpZD0iYnV0dG9uX2JveCI+CiAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRB
TDwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0
dG9uIiBpZD0iY2FuY2VsIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xh
dGFibGU9InllcyI+Q2FuY2VsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlj
a2VkIiBoYW5kbGVyPSJvbl9jYW5jZWwiIC8+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2No
aWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9
InNhdmUiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVz
Ij5PSzwvcHJvcGVydHk+CiAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0i
b25fc2F2ZSIgLz4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2Jq
ZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAg
ICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/NewCustomConversation.xml": {
		local:   "definitions/NewCustomConversation.xml",
		size:    4760,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0U3RvcmUiIGlkPSJhY2NvdW50cy1tb2Rl
bCI+CiAgICA8Y29sdW1ucz4KICAgICAgPCEtLSBhY2NvdW50IG5hbWUgLS0+CiAgICAgIDxjb2x1bW4g
dHlwZT0iZ2NoYXJhcnJheSIvPgogICAgICA8IS0tIGFjY291bnQgaWQgLS0+CiAgICAgIDxjb2x1bW4g
dHlwZT0iZ2NoYXJhcnJheSIvPgogICAgPC9jb2x1bW5zPgogIDwvb2JqZWN0PgogIDxvYmplY3QgY2xh
c3M9Ikd0a0RpYWxvZyIgaWQ9Ik5ld0N1c3RvbUNvbnZlcnNhdGlvbiI+CiAgICA8cHJvcGVydHkgbmFt
ZT0id2luZG93LXBvc2l0aW9uIj5HVEtfV0lOX1BPU19DRU5URVI8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+NjwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0idGl0
bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5TdGFydCBuZXcgY29udmVyc2F0aW9uPC9wcm9wZXJ0eT4KICAg
IDxwcm9wZXJ0eSBuYW1lPSJyZXNpemFibGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5h
bWU9ImRlZmF1bHQtaGVpZ2h0Ij4yMDA8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImRlZmF1
bHQtd2lkdGgiPjUwMDwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0iZGVzdHJveS13aXRoLXBh
cmVudCI+dHJ1ZTwvcHJvcGVydHk+CiAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9InZib3giPgogICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJWYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
aG9tb2dlbmVvdXMiPmZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50
YXRpb24iPkdUS19PUklFTlRBVElPTl9WRVJUSUNBTDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9InNwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0dyaWQiIGlkPSJncmlkIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1h
cmdpbi10b3AiPjE1PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1i
b3R0b20iPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1zdGFy
dCI+MTA8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWVuZCI+MTA8
L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93LXNwYWNpbmciPjEyPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNvbHVtbi1zcGFjaW5nIj42PC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVs
IiBpZD0iYWNjb3VudHNMYWJlbCIgPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVs
IiB0cmFuc2xhdGFibGU9InllcyI+RnJvbTo8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Imp1c3RpZnkiPkdUS19KVVNUSUZZX1JJR0hUPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9FTkQ8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+
CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtDb21ib0JveCIgaWQ9ImFjY291bnRzIj4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJtb2RlbCI+YWNjb3VudHMtbW9kZWw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9Imhhcy1mb2N1cyI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0iaGV4cGFuZCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0NlbGxSZW5kZXJlclRleHQiIGlk
PSJhY2NvdW50LW5hbWUtcmVuZGVyZWQiLz4KICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZXM+CiAg
ICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ0ZXh0Ij4wPC9hdHRyaWJ1dGU+CiAgICAg
ICAgICAgICAgICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InRvcC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4K
ICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0icGVlckxhYmVsIiA+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5XaXRoOjwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ianVzdGlmeSI+R1RLX0pVU1RJRllfUklHSFQ8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX0VORDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0VudHJ5IiBpZD0iYWRkcmVzcyI+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0icGxhY2Vob2xkZXItdGV4dCI+c29tZW9uZUBqYWJiZXIub3JnPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iYWN0aXZhdGUiIGhhbmRsZXI9Im9uX3N0
YXJ0IiAvPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAg
ICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj50cnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPnRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAg
PC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJh
Y3Rpb25fYXJlYSI+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJidXR0
b25fYm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5U
QVRJT05fSE9SSVpPTlRBTDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJidXR0b25fY2FuY2VsIj4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCI+X0NhbmNlbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0idXNlLXVuZGVybGluZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX2Nsb3NlIiAvPgogICAgICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iYnV0dG9uX29rIj4KICAgICAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlN0YXJ0PC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ1c2UtdW5kZXJsaW5lIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW4tZGVmYXVsdCI+dHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9u
X3N0YXJ0IiAvPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxk
PgogICAgPHN0eWxlPgogICAgICA8Y2xhc3MgbmFtZT0iY295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9v
YmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/Password.xml": {
		local:   "definitions/Password.xml",
		size:    1307,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJwYXNzd29yZC1ib3giPgogICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9
InNwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtF
bnRyeSIgaWQ9InBhc3N3b3JkLWVudHJ5Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmlsaXR5Ij5GYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImludmlzaWJsZV9jaGFyIj7il488L3Byb3BlcnR5PgogICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJwbGFjZWhvbGRlcl90ZXh0IiB0cmFuc2xhdGFibGU9InllcyI+UGFzc3dv
cmQ8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpbnB1dF9wdXJwb3NlIj5wYXNzd29y
ZDwvcHJvcGVydHk+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJv
cGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9InBhc3N3b3JkLXRvZ2dsZS1idXR0b24iPgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlNob3c8L3Byb3BlcnR5PgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHNpZ25hbCBu
YW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9wYXNzd29yZF90b2dnbGUiIHN3YXBwZWQ9Im5vIi8+CiAg
ICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5k
Ij5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgIDwv
cGFja2luZz4KICAgIDwvY2hpbGQ+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/PeerDetails.xml": {
		local:   "definitions/PeerDetails.xml",
		size:    10054,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0U3RvcmUiIGlkPSJjdXJyZW50LWdyb3Vw
cyI+CiAgICA8Y29sdW1ucz4KICAgICAgPGNvbHVtbiB0eXBlPSJnY2hhcmFycmF5Ii8+CiAgICA8L2Nv
bHVtbnM+CiAgPC9vYmplY3Q+CiAgPG9iamVjdCBjbGFzcz0iR3RrTWVudUl0ZW0iIGlkPSJhZGRHcm91
cCI+CiAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5OZXcgR3JvdXAuLi48L3Byb3BlcnR5Pgog
ICAgPHNpZ25hbCBuYW1lPSJhY3RpdmF0ZSIgaGFuZGxlcj0ib24tYWRkLW5ldy1ncm91cCIgLz4KICA8
L29iamVjdD4KICA8b2JqZWN0IGNsYXNzPSJHdGtNZW51IiBpZD0iZ3JvdXBzLW1lbnUiPgogICAgPHBy
b3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogIDwvb2JqZWN0PgogIDxvYmplY3Qg
Y2xhc3M9Ikd0a0RpYWxvZyIgaWQ9ImRpYWxvZyI+CiAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRy
YW5zbGF0YWJsZT0ieWVzIj5FZGl0IGNvbnRhY3Q8L3Byb3BlcnR5PgogICAgPHNpZ25hbCBuYW1lPSJj
bG9zZSIgaGFuZGxlcj0ib24tY2FuY2VsIiAvPgogICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94
Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iVmJveCI+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9Im1hcmdpbiI+MTA8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0dyaWQiIGlkPSJncmlkIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1h
cmdpbi10b3AiPjE1PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1i
b3R0b20iPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1zdGFy
dCI+MTA8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLWVuZCI+MTA8
L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93LXNwYWNpbmciPjEyPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNvbHVtbi1zcGFjaW5nIj42PC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVs
IiBpZD0iYWNjb3VudC1sYWJlbCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwi
IHRyYW5zbGF0YWJsZT0ieWVzIj5BY2NvdW50OjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fRU5EPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UUlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0idG9wLWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAg
ICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFz
cz0iR3RrTGFiZWwiIGlkPSJhY2NvdW50LW5hbWUiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImhhbGlnbiI+R1RLX0FMSUdOX1NUQVJUPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJzZWxlY3RhYmxlIj5UUlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJs
ZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9w
LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8
L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
TGFiZWwiIGlkPSJqaWQtbGFiZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVs
IiB0cmFuc2xhdGFibGU9InllcyI+Q29udGFjdDo8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImhhbGlnbiI+R1RLX0FMSUdOX0VORDwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InRvcC1hdHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAg
ICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0xhYmVsIiBpZD0iamlkIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxp
Z24iPkdUS19BTElHTl9TVEFSVDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0ic2VsZWN0YWJsZSI+VFJVRTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRh
Y2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2gi
PjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBp
ZD0ibmlja25hbWUtbGFiZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0
cmFuc2xhdGFibGU9InllcyI+Tmlja25hbWU6PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJoYWxpZ24iPkdUS19BTElHTl9FTkQ8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ0b3AtYXR0YWNoIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtFbnRyeSIgaWQ9Im5pY2tuYW1lIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJh
Y3RpdmF0ZXMtZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1h
dHRhY2giPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRh
Y2giPjI8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVs
IiBpZD0icmVxdWlyZS1lbmNyeXB0aW9uLWxhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPlJlcXVpcmUgZW5jcnlwdGlvbiB3aXRoIHRoaXMg
cGVlcjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtf
QUxJR05fRU5EPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzZWxlY3Rh
YmxlIj5UUlVFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8
cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MzwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
ICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2hlY2tCdXR0b24iIGlkPSJy
ZXF1aXJlLWVuY3J5cHRpb24iLz4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+MzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNr
aW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJncm91cHMiPgogICAgICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+R3JvdXBzOjwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fU1RBUlQ8L3Byb3BlcnR5
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNlbGVjdGFibGUiPlRSVUU8L3Byb3BlcnR5
PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj40PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aCI+MjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8IS0tIFRPRE86IFRoaXMgc2hvdWxkIGJl
IGluIGEgc2Nyb2xsZWQgd2luZG93IHdpdGggbWluL21heCBzaXplICAtLT4KICAgICAgICAgICAgPGNo
aWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVWaWV3IiBpZD0iZ3JvdXBzLXZp
ZXciPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1vZGVsIj5jdXJyZW50LWdyb3Vwczwv
cHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWZvY3VzIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZWFkZXJzLXZpc2libGUiPkZhbHNl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzaG93LWV4cGFuZGVycyI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlb3JkZXJhYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0ic2Vs
ZWN0aW9uIj4KICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrVHJlZVNlbGVjdGlvbiIg
aWQ9InNlbGVjdGlvbiI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1vZGUiPkdU
S19TRUxFQ1RJT05fU0lOR0xFPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8c2lnbmFsIG5h
bWU9ImNoYW5nZWQiIGhhbmRsZXI9Im9uLWdyb3VwLXNlbGVjdGlvbi1jaGFuZ2VkIiBzd2FwcGVkPSJu
byIvPgogICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAg
ICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1Ry
ZWVWaWV3Q29sdW1uIiBpZD0iZ3JvdXAtbmFtZS1jb2x1bW4iPgogICAgICAgICAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ0aXRsZSI+Z3JvdXAtbmFtZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAg
ICAgPGNoaWxkPgogICAgICAgICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQ2VsbFJlbmRl
cmVyVGV4dCIgaWQ9Imdyb3VwLW5hbWUtY29sdW1uLXJlbmRlcmVkIi8+CiAgICAgICAgICAgICAgICAg
ICAgICA8YXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ0
ZXh0Ij4wPC9hdHRyaWJ1dGU+CiAgICAgICAgICAgICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAg
ICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgIDxwYWNr
aW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxlZnQtYXR0YWNoIj4wPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3AtYXR0YWNoIj41PC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aWR0aCI+MjwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94IiBpZD0iZ3JvdXBzLWJ1dHRvbnMi
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJ
T05fSE9SSVpPTlRBTDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a01lbnVCdXR0b24iIGlkPSJhZGQtYnRuIj4KICAgICAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvcHVwIj5ncm91cHMtbWVudTwvcHJvcGVydHk+CiAg
ICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImRpcmVjdGlvbiI+R1RLX0FSUk9XX0RPV048
L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdlIiBpZD0iYWRkLWJ0bi1pbWciPgogICAgICAgICAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iaWNvbi1uYW1lIj5saXN0LWFkZDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAg
ICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9
InJlbW92ZS1idG4iPgogICAgICAgICAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFu
ZGxlcj0ib24tcmVtb3ZlLWdyb3VwIi8+CiAgICAgICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAg
ICAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiIGlkPSJyZW1vdmUtYnRuLWltZyI+
CiAgICAgICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpY29uLW5hbWUiPmxpc3QtcmVt
b3ZlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAg
ICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAg
PC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsZWZ0LWF0dGFjaCI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idG9wLWF0dGFjaCI+NjwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9hcmVhIj4K
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCIgaWQ9ImJ1dHRvbl9ib3giPgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9IT1JJWk9O
VEFMPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a0J1dHRvbiIgaWQ9ImJ0bi1jYW5jZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+Q2FuY2VsPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0ib24tY2FuY2VsIi8+CiAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJidG4tc2F2ZSI+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5TYXZlPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW4tZGVmYXVsdCI+dHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uLXNhdmUi
Lz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPCEtLSBUT0RPOiBUaGlzIHNob3VsZCBiZSBp
biBhIHNjcm9sbGVkIHdpbmRvdyB3aXRoIG1pbi9tYXggc2l6ZSAgLS0+CiAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0iYm94Ij4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImJvcmRlci13aWR0aCI+MTA8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iaG9tb2dlbmVvdXMiPmZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fVkVSVElDQUw8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJmaW5n
ZXJwcmludHNJbmZvcm1hdGlvbiI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2VsZWN0
YWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtH
cmlkIiBpZD0iZmluZ2VycHJpbnRzR3JpZCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bWFyZ2luLXRvcCI+MTU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1h
cmdpbi1ib3R0b20iPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJt
YXJnaW4tc3RhcnQiPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJt
YXJnaW4tZW5kIj4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93
LXNwYWNpbmciPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1
bW4tc3BhY2luZyI+NjwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVj
dD4KICAgIDwvY2hpbGQ+CiAgICA8c3R5bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAg
PC9zdHlsZT4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/PeerRequestsSMP.xml": {
		local:   "definitions/PeerRequestsSMP.xml",
		size:    4283,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJzbXAtcmVxdWVzdGVkLWluZm9i
YXIiPgogICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgPHBy
b3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InNtcC1yZXF1ZXN0ZWQtY2xvc2UtaW5mb2JhciI+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KCSAgICA8Y2hpbGQ+CiAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtFdmVudEJveCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9m
b2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8c2lnbmFsIG5hbWU9ImJ1dHRvbi1wcmVz
cy1ldmVudCIgaGFuZGxlcj0ib25fcHJlc3NfY2xvc2VfaW1hZ2UiIHN3YXBwZWQ9Im5vIi8+CgkgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0icGl4ZWxfc2l6ZSI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Imljb25fbmFtZSI+d2luZG93LWNsb3NlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJpY29uX3NpemUiPjM8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9yaWdodCI+MzwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3RvcCI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2JvdHRvbSI+MTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
IDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InBhY2tfdHlwZSI+ZW5kPC9wcm9wZXJ0eT4KCSAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3Bl
cnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJm
aWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3By
b3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJzbXAtcmVxdWVzdGVkLW5vdGlmaWNhdGlvbiI+CiAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3Bh
Y2luZyI+MzM8L3Byb3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0ltYWdlIiBpZD0ic21wLXJlcXVlc3RlZC1pbWFnZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNh
bl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2lu
X2xlZnQiPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0icG9zaXRpb24iPjA8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFi
ZWwiIGlkPSJzbXAtcmVxdWVzdGVkLW1lc3NhZ2UiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9j
dXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3Rh
cnQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8YXR0cmlidXRlcz4KICAgICAgICAgICAgICA8YXR0cmli
dXRlIG5hbWU9IndlaWdodCIgdmFsdWU9ImJvbGQiLz4KICAgICAgICAgICAgPC9hdHRyaWJ1dGVzPgoJ
ICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmls
bCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwv
cHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJzbXAtcmVxdWVzdGVkLWJ1
dHRvbiI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5UcnVlPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9yaWdodCI+MTA8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3RvcCI+MTQ8L3Byb3BlcnR5PgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0ibWFyZ2luX2JvdHRvbSI+MTI8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hp
bGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiPgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9Im1hcmdpbl9sZWZ0Ij4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibWFyZ2luX3JpZ2h0Ij4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5GaW5pc2ggVmVyaWZpY2F0aW9uPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KCSAgICAgICAgPC9jaGlsZD4KICAgICAgICAg
IDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFja190eXBlIj5lbmQ8L3By
b3BlcnR5PgoJICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFj
a2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAg
PC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/RegistrationForm.xml": {
		local:   "definitions/RegistrationForm.xml",
		size:    2228,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkaWFsb2ciPgogICAgPHNp
Z25hbCBuYW1lPSJyZXNwb25zZSIgaGFuZGxlcj0ib25fcmVzcG9uc2UiIC8+CiAgICA8c2lnbmFsIG5h
bWU9ImNsb3NlIiBoYW5kbGVyPSJvbl9jbG9zZSIgLz4KICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0i
dmJveCI+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9IlZib3giPgogICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bWFyZ2luLWJvdHRvbSI+MTA8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4i
PjEwPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtH
cmlkIiBpZD0iZ3JpZCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1ib3R0b20iPjEwPC9wcm9w
ZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJvdy1zcGFjaW5nIj4xMjwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjb2x1bW4tc3BhY2luZyI+NjwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9
Imluc3RydWN0aW9ucyI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+dHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+
CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGVmdC1hdHRhY2giPjA8L3Byb3BlcnR5Pgog
ICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRvcC1hdHRhY2giPjA8L3Byb3BlcnR5PgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9IndpZHRoIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAg
ICA8L2NoaWxkPgogICAgICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0iYWN0aW9uX2FyZWEiPgogICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uQm94IiBpZD0iYmJveCI+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5
IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRBTDwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlk
PSJidG4tY2FuY2VsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj50cnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRh
YmxlPSJ5ZXMiPkNhbmNlbDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAg
ICAgIDwvY2hpbGQ+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtCdXR0b24iIGlkPSJidG4tcmVnaXN0ZXIiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+UmVnaXN0ZXI8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImNhbi1kZWZhdWx0Ij50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICA8L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAg
IDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxhY3Rpb24td2lkZ2V0cz4K
ICAgICAgPGFjdGlvbi13aWRnZXQgcmVzcG9uc2U9ImNhbmNlbCI+YnRuLWNhbmNlbDwvYWN0aW9uLXdp
ZGdldD4KICAgICAgPGFjdGlvbi13aWRnZXQgcmVzcG9uc2U9ImFwcGx5IiBkZWZhdWx0PSJ0cnVlIj5i
dG4tcmVnaXN0ZXI8L2FjdGlvbi13aWRnZXQ+CiAgICA8L2FjdGlvbi13aWRnZXRzPgogICAgPHN0eWxl
PgogICAgICA8Y2xhc3MgbmFtZT0iY295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+CjwvaW50
ZXJmYWNlPgo=
`,
	},

	"/definitions/Roster.xml": {
		local:   "definitions/Roster.xml",
		size:    2006,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtTY3JvbGxlZFdpbmRvdyIgaWQ9InJvc3RlciI+
CiAgICA8cHJvcGVydHkgbmFtZT0iaHNjcm9sbGJhci1wb2xpY3kiPkdUS19QT0xJQ1lfTkVWRVI8L3By
b3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9InZzY3JvbGxiYXItcG9saWN5Ij5HVEtfUE9MSUNZX0FV
VE9NQVRJQzwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1RyZWVW
aWV3IiBpZD0icm9zdGVyLXRyZWUiPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZWFkZXJzLXZpc2li
bGUiPmZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2hvdy1leHBhbmRlcnMi
PmZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGV2ZWwtaW5kZW50YXRpb24i
PjM8L3Byb3BlcnR5PgogICAgICAgIDwhLS0gVE9ETyByZW1vdmUgbWFnaWMgbnVtYmVyIC0tPgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b29sdGlwLWNvbHVtbiI+NjwvcHJvcGVydHk+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InNlYXJjaC1jb2x1bW4iPjY8L3Byb3BlcnR5PgogICAgICAgIDxzaWduYWwgbmFt
ZT0icm93LWFjdGl2YXRlZCIgaGFuZGxlcj0ib25fYWN0aXZhdGVfYnVkZHkiIC8+CiAgICAgICAgPHNp
Z25hbCBuYW1lPSJidXR0b24tcHJlc3MtZXZlbnQiIGhhbmRsZXI9Im9uX2J1dHRvbl9wcmVzcyIgLz4K
ICAgICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9InNlbGVjdGlvbiI+CiAgICAgICAgICA8b2JqZWN0
IGNsYXNzPSJHdGtUcmVlU2VsZWN0aW9uIiBpZD0ic2VsZWN0aW9uIj4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9Im1vZGUiPkdUS19TRUxFQ1RJT05fU0lOR0xFPC9wcm9wZXJ0eT4KICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrVHJlZVZpZXdDb2x1bW4iIGlkPSJpY29uLWNvbHVtbiI+CiAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDZWxsUmVuZGVyZXJQaXhidWYiIGlkPSJp
Y29uLWNvbHVtbi1yZW5kZXJlZCIvPgogICAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAg
ICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJwaXhidWYiPjc8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAg
IDxhdHRyaWJ1dGUgbmFtZT0iY2VsbC1iYWNrZ3JvdW5kIj40PC9hdHRyaWJ1dGU+CiAgICAgICAgICAg
ICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAg
ICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtU
cmVlVmlld0NvbHVtbiIgaWQ9Im5hbWUtY29sdW1uIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InRpdGxlIj5uYW1lPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxv
YmplY3QgY2xhc3M9Ikd0a0NlbGxSZW5kZXJlclRleHQiIGlkPSJuYW1lLWNvbHVtbi1yZW5kZXJlZCIv
PgogICAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAgICAgICAgPCEtLSBUT0RPIHJlbW92
ZSBtYWdpYyBudW1iZXJzIC0tPgogICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ0ZXh0Ij4x
PC9hdHRyaWJ1dGU+CiAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9ImZvcmVncm91bmQiPjM8
L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0iYmFja2dyb3VuZCI+NDwv
YXR0cmlidXRlPgogICAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ3ZWlnaHQiPjU8L2F0dHJp
YnV0ZT4KICAgICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICA8L29iamVjdD4KICAgIDwvY2hpbGQ+
CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/SimpleNotification.xml": {
		local:   "definitions/SimpleNotification.xml",
		size:    359,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtNZXNzYWdlRGlhbG9nIiBpZD0iZGlhbG9nIj4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aW5kb3ctcG9zaXRpb24iPkdUS19XSU5fUE9TX0NFTlRFUjwvcHJv
cGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0ibW9kYWwiPnRydWU8L3Byb3BlcnR5PgogICAgPHByb3Bl
cnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+NzwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0ibWVz
c2FnZS10eXBlIj5HVEtfTUVTU0FHRV9JTkZPPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJi
dXR0b25zIj5HVEtfQlVUVE9OU19PSzwvcHJvcGVydHk+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/Test.xml": {
		local:   "definitions/Test.xml",
		size:    333,
		modtime: 1489449600,
		compressed: `
CjxpbnRlcmZhY2U+CiAgPG9iamVjdCBjbGFzcz0iR3RrV2luZG93IiBpZD0iY29udmVyc2F0aW9uIj4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJkZWZhdWx0LWhlaWdodCI+NjAwPC9wcm9wZXJ0eT4KICAgIDxwcm9w
ZXJ0eSBuYW1lPSJkZWZhdWx0LXdpZHRoIj41MDA8L3Byb3BlcnR5PgogICAgPGNoaWxkPgoJICA8b2Jq
ZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJ2Ym94Ij4KCSAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRp
b24iPkdUS19PUklFTlRBVElPTl9WRVJUSUNBTDwvcHJvcGVydHk+ICAKCSAgPC9vYmplY3Q+CiAgICA8
L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/TorHelper.xml": {
		local:   "definitions/TorHelper.xml",
		size:    631,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtNZXNzYWdlRGlhbG9nIiBpZD0iVG9ySGVscGVy
Ij4KICAgIDxwcm9wZXJ0eSBuYW1lPSJ3aW5kb3ctcG9zaXRpb24iPkdUS19XSU5fUE9TX0NFTlRFUl9B
TFdBWVM8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9Im1vZGFsIj50cnVlPC9wcm9wZXJ0eT4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJib3JkZXJfd2lkdGgiPjc8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5
IG5hbWU9InRleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5XYXJuaW5nITwvcHJvcGVydHk+CiAgICA8cHJv
cGVydHkgbmFtZT0ic2Vjb25kYXJ5X3RleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj4KV2UgY291bGQgbm90
IGZpbmQgYSBydW5uaW5nIGluc3RhbGxhdGlvbiBvZiBUb3IuCgpJbiBvcmRlciB0byBoYXZlIHRoZSBz
YWZlc3QgZXhwZXJpZW5jZSwgd2UgcmVjb21tZW5kIHlvdSB0byBpbnN0YWxsIGl0LgoKVG9yIGlzIGEg
dG9vbCBmb3IgYW5vbnltaXR5LgoKV291bGQgeW91IGxpa2UgdG8gaW5zdGFsbCBpdD8KICAgIDwvcHJv
cGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0iYnV0dG9ucyI+R1RLX0JVVFRPTlNfWUVTX05PPC9wcm9w
ZXJ0eT4KICA8L29iamVjdD4KPC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/TorInstallHelper.xml": {
		local:   "definitions/TorInstallHelper.xml",
		size:    3879,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkaWFsb2ciPgogICAgPHBy
b3BlcnR5IG5hbWU9IndpbmRvdy1wb3NpdGlvbiI+R1RLX1dJTl9QT1NfQ0VOVEVSPC9wcm9wZXJ0eT4K
ICAgIDxwcm9wZXJ0eSBuYW1lPSJib3JkZXItd2lkdGgiPjc8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5
IG5hbWU9InRpdGxlIiB0cmFuc2xhdGFibGU9InllcyI+PC9wcm9wZXJ0eT4KICAgIDxjaGlsZCBpbnRl
cm5hbC1jaGlsZD0idmJveCI+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9ImJveCI+CiAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlci13aWR0aCI+MTA8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX1ZFUlRJQ0FMPC9wcm9wZXJ0
eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCI+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi10b3AiPjU8L3Byb3BlcnR5PgoJICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj4KICAgICAgICAgICAgICBJbnN0YWxsaW5n
IFRvciBmb3IgQ295SU06CiAgICAgICAgICAgIDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxhdHRyaWJ1
dGVzPgogICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0iZm9udC1kZXNjIiB2YWx1ZT0iJmx0O0Vu
dGVyIFZhbHVlJmd0OyAxNCIvPgoJICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0id2VpZ2h0IiB2YWx1
ZT0ic2VtaWJvbGQiLz4KICAgICAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICAgICAgPC9vYmplY3Q+
CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAg
ICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9j
dXMiPkZhbHNlPC9wcm9wZXJ0eT4KCSAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xh
dGFibGU9InllcyI+CgogICAgICAgICAgICAgIDEuIEdvIHRvIGh0dHBzOi8vd3d3LnRvcnByb2plY3Qu
b3JnLwoKICAgICAgICAgICAgICAyLiBEb3dubG9hZCBhbmQgaW5zdGFsbCBUb3IgKG9yIHRoZSBUb3Ig
QnJvd3NlciBCdW5kbGUpCgogICAgICAgICAgICAgIDMuIFN0YXJ0IFRvciAob3IgdGhlIFRvciBCcm93
c2VyKQogICAgICAgICAgICA8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic2Vs
ZWN0YWJsZSI+VHJ1ZTwvcHJvcGVydHk+CgkgICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5n
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHBy
b3BlcnR5IG5hbWU9InBvc2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAg
ICAgICA8L2NoaWxkPgoJICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0V2ZW50
Qm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgICAgIDxzaWduYWwgbmFtZT0iYnV0dG9uLXByZXNzLWV2ZW50IiBoYW5kbGVyPSJvbl9wcmVzc19s
YWJlbCIgc3dhcHBlZD0ibm8iLz4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0xhYmVsIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9j
dXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4t
dG9wIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tYm90
dG9tIj4xMDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXJp
Z2h0Ij42MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRy
YW5zbGF0YWJsZT0ieWVzIj40LiBDbGljayBoZXJlIHRvIGNoZWNrIGlmIFRvciBpcyBub3cgcnVubmlu
ZzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8YXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAg
PGF0dHJpYnV0ZSBuYW1lPSJ1bmRlcmxpbmUiIHZhbHVlPSJUcnVlIi8+CiAgICAgICAgICAgICAgICA8
L2F0dHJpYnV0ZXM+CgkgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAg
ICAgICA8L29iamVjdD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFt
ZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4yPC9w
cm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGls
ZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9InRvci1ydW5uaW5nLW5vdGlmaWNh
dGlvbiI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbi1ib3R0b20iPjEwPC9wcm9wZXJ0eT4KICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjM8L3By
b3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0
PgogICAgPC9jaGlsZD4KICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0iYWN0aW9uX2FyZWEiPgogICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJidXR0b25fYm94Ij4KICAgICAgICA8cHJv
cGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9IT1JJWk9OVEFMPC9wcm9wZXJ0
eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJj
bG9zZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMi
PkNsb3NlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVy
PSJvbl9jbG9zZSIgLz4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwv
b2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+
CiAgICA8L3N0eWxlPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/TorNotRunningNotification.xml": {
		local:   "definitions/TorNotRunningNotification.xml",
		size:    2904,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtJbmZvQmFyIiBpZD0iaW5mb2JhciI+CiAgICA8
cHJvcGVydHkgbmFtZT0ibWVzc2FnZS10eXBlIj5HVEtfTUVTU0FHRV9FUlJPUjwvcHJvcGVydHk+CiAg
ICA8cHJvcGVydHkgbmFtZT0ic2hvdy1jbG9zZS1idXR0b24iPmZhbHNlPC9wcm9wZXJ0eT4KICAgIDxj
aGlsZCBpbnRlcm5hbC1jaGlsZD0iY29udGVudF9hcmVhIj4KICAgICAgPG9iamVjdCBjbGFzcz0iR3Rr
Qm94IiBpZD0iYm94Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MDwvcHJvcGVydHk+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhvbW9nZW5lb3VzIj5mYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRBTDwv
cHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tc3RhcnQiPjA8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJtYXJnaW4tZW5kIj4wPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9Im1lc3NhZ2UiPgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZWxsaXBzaXplIj5QQU5HT19FTExJUFNJWkVfRU5EPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImhleHBhbmQiPlRydWU8L3Byb3BlcnR5PgoJICAgICAgICA8cHJv
cGVydHkgbmFtZT0id3JhcCI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJtYXJnaW4tcmlnaHQiPjEwPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBp
ZD0iYnV0dG9uX21vcmVfaW5mbyI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5G
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoYWxpZ24iPmVuZDwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWxpZWYiPm5vbmU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmFsaWduIj5jZW50ZXI8L3Byb3BlcnR5PgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHN0eWxlPgogICAgICAgICAgICAgIDxjbGFzcyBuYW1lPSJyYWlzZWQiLz4KICAgICAgICAgICAgICA8
Y2xhc3MgbmFtZT0iY2xvc2UiLz4KICAgICAgICAgICAgPC9zdHlsZT4KICAgICAgICAgICAgPHNpZ25h
bCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9tb3JlX2luZm8iIC8+CiAgICAgICAgICAgIDxjaGls
ZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtJbWFnZSIgaWQ9Im1vcmUtaW5mby1idG4i
PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imljb24tbmFtZSI+ZGlhbG9nLXF1ZXN0aW9u
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBv
c2l0aW9uIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgog
ICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImJ1dHRv
bl9jbG9zZSI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPjE8L3Byb3BlcnR5
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj4yPC9wcm9wZXJ0eT4KICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9InJlbGllZiI+bm9uZTwvcHJvcGVydHk+CiAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNlbnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxzdHls
ZT4KICAgICAgICAgICAgICA8Y2xhc3MgbmFtZT0icmFpc2VkIi8+CiAgICAgICAgICAgICAgPGNsYXNz
IG5hbWU9ImNsb3NlIi8+CiAgICAgICAgICAgIDwvc3R5bGU+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJoZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJo
YWxpZ24iPmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2YWxpZ24iPmNl
bnRlcjwvcHJvcGVydHk+CiAgICAgICAgICAgIDxzaWduYWwgbmFtZT0iY2xpY2tlZCIgaGFuZGxlcj0i
b25fY2xvc2UiIC8+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNz
PSJHdGtJbWFnZSIgaWQ9ImNsb3NlX2ltYWdlIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJp
Y29uX25hbWUiPndpbmRvdy1jbG9zZS1zeW1ib2xpYzwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAgICAg
ICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgog
IDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/TorRunningNotification.xml": {
		local:   "definitions/TorRunningNotification.xml",
		size:    1086,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJpbmZvYmFyIj4KICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5f
Zm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj4xODwvcHJv
cGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdlIiBpZD0iaW1hZ2Ui
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8
cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9Im1hcmdpbl9sZWZ0Ij4xNTwvcHJvcGVydHk+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFj
a2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAg
ICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ibWVzc2FnZSI+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
bWFyZ2luX3RvcCI+NTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9ib3R0
b20iPjU8L3Byb3BlcnR5PgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHBy
b3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8
L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVy
ZmFjZT4K
`,
	},

	"/definitions/UnifiedLayout.xml": {
		local:   "definitions/UnifiedLayout.xml",
		size:    8546,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtMaXN0U3RvcmUiIGlkPSJsaXN0c3RvcmUiPgog
ICAgPGNvbHVtbnM+CiAgICAgIDwhLS0gY29sdW1uLW5hbWUgcGFnZV9pbmRleCAtLT4KICAgICAgPGNv
bHVtbiB0eXBlPSJnaW50Ii8+CiAgICAgIDwhLS0gY29sdW1uLW5hbWUgbmFtZSAtLT4KICAgICAgPGNv
bHVtbiB0eXBlPSJnY2hhcmFycmF5Ii8+CiAgICAgIDwhLS0gY29sdW1uLW5hbWUgamlkIC0tPgogICAg
ICA8Y29sdW1uIHR5cGU9ImdjaGFyYXJyYXkiLz4KICAgICAgPCEtLSBjb2x1bW4tbmFtZSBjb2xvciAt
LT4KICAgICAgPGNvbHVtbiB0eXBlPSJnY2hhcmFycmF5Ii8+CiAgICAgIDwhLS0gY29sdW1uLW5hbWUg
YmFja2dyb3VuZCAtLT4KICAgICAgPGNvbHVtbiB0eXBlPSJnY2hhcmFycmF5Ii8+CiAgICAgIDwhLS0g
Y29sdW1uLW5hbWUgd2VpZ2h0IC0tPgogICAgICA8Y29sdW1uIHR5cGU9ImdpbnQiLz4KICAgICAgPCEt
LSBjb2x1bW4tbmFtZSB0b29sdGlwIC0tPgogICAgICA8Y29sdW1uIHR5cGU9ImdjaGFyYXJyYXkiLz4K
ICAgICAgPCEtLSBjb2x1bW4tbmFtZSBzdGF0dXMgLS0+CiAgICAgIDxjb2x1bW4gdHlwZT0iR2RrUGl4
YnVmIi8+CiAgICAgIDwhLS0gY29sdW1uLW5hbWUgdW5kZXJsaW5lIC0tPgogICAgICA8Y29sdW1uIHR5
cGU9ImdpbnQiLz4KICAgIDwvY29sdW1ucz4KICA8L29iamVjdD4KICA8b2JqZWN0IGNsYXNzPSJHdGtC
b3giIGlkPSJyaWdodCI+CiAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+
CiAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICA8cHJvcGVy
dHkgbmFtZT0ibm9fc2hvd19hbGwiPlRydWU8L3Byb3BlcnR5PgogICAgPGNoaWxkPgogICAgICA8b2Jq
ZWN0IGNsYXNzPSJHdGtTZXBhcmF0b3IiIGlkPSJzZXBhcmF0b3IxIj4KICAgICAgICA8cHJvcGVydHkg
bmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9m
b2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+
dmVydGljYWw8L3Byb3BlcnR5PgogICAgICA8L29iamVjdD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24i
PjA8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAg
ICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJib3gxIj4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+
RmFsc2U8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZXhwYW5kIj5UcnVlPC9wcm9w
ZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0
eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJoZWFk
ZXJfYm94Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5
PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAg
ICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9
ImhlYWRlcl9sYWJlbCI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxz
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8YXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAg
PGF0dHJpYnV0ZSBuYW1lPSJ3ZWlnaHQiIHZhbHVlPSJib2xkIi8+CiAgICAgICAgICAgICAgICA8L2F0
dHJpYnV0ZXM+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgPHBhY2tpbmc+CiAg
ICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgPC9w
YWNraW5nPgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAg
ICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iY2xvc2VfYnV0dG9uIj4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
IDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9InJlY2VpdmVzX2RlZmF1bHQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InJlbGllZiI+bm9uZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICA8c2lnbmFsIG5hbWU9ImNsaWNrZWQiIGhhbmRsZXI9Im9uX2NsaWNrZWQiIHN3YXBwZWQ9Im5vIi8+
CiAgICAgICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0
a0ltYWdlIiBpZD0iaW1hZ2UxIj4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlz
aWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNh
bl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJpY29uX25hbWUiPndpbmRvdy1jbG9zZS1zeW1ib2xpYzwvcHJvcGVydHk+CiAgICAgICAgICAgICAg
ICAgIDwvb2JqZWN0PgogICAgICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJl
eHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxs
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwYWNrX3R5cGUi
PmVuZDwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8
L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvcGFja2luZz4KICAgICAgICAgICAgPC9jaGlsZD4KICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjA8
L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNo
aWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTm90ZWJvb2siIGlkPSJub3RlYm9vayI+CiAg
ICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJzY3JvbGxhYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3Bl
cnR5IG5hbWU9InNob3dfdGFicyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0ic2hvd19ib3JkZXIiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9Im1hcmdpbl9zdGFydCI+NTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJt
YXJnaW5fZW5kIj41PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJzd2l0Y2gtcGFn
ZSIgaGFuZGxlcj0ib25fc3dpdGNoX3BhZ2UiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICA8L29iamVj
dD4KICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5U
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAg
ICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFj
a2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAg
ICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a1NlcGFyYXRvciIgaWQ9InNlcGFyYXRvcjIi
PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8
cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5
IG5hbWU9Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgIDwvb2JqZWN0PgogICAg
ICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MjwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hp
bGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0JveCIgaWQ9ImJveDMiPgogICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im9yaWVudGF0aW9uIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAg
PG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiIGlkPSJsYWJlbDEiPgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJj
YW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVs
IiB0cmFuc2xhdGFibGU9InllcyI+Q29udmVyc2F0aW9uczwvcHJvcGVydHk+CiAgICAgICAgICAgIDxh
dHRyaWJ1dGVzPgogICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0id2VpZ2h0IiB2YWx1ZT0iYm9s
ZCIvPgogICAgICAgICAgICA8L2F0dHJpYnV0ZXM+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAg
IDxwYWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgIDwvcGFj
a2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xh
c3M9Ikd0a1RyZWVWaWV3IiBpZD0idHJlZXZpZXciPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
dmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9j
dXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaHNjcm9sbF9wb2xp
Y3kiPm5hdHVyYWw8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibW9kZWwiPmxp
c3RzdG9yZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJoZWFkZXJzX3Zpc2li
bGUiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImVuYWJsZV9zZWFy
Y2giPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImFjdGl2YXRlX29u
X3NpbmdsZV9jbGljayI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxzaWduYWwgbmFtZT0icm93
LWFjdGl2YXRlZCIgaGFuZGxlcj0ib25fYWN0aXZhdGUiIHN3YXBwZWQ9Im5vIi8+CiAgICAgICAgICAg
IDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0ic2VsZWN0aW9uIj4KICAgICAgICAgICAgICA8b2JqZWN0IGNs
YXNzPSJHdGtUcmVlU2VsZWN0aW9uIiBpZD0idHJlZXZpZXctc2VsZWN0aW9uMSIvPgogICAgICAgICAg
ICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0i
R3RrVHJlZVZpZXdDb2x1bW4iIGlkPSJpY29uIj4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJ0aXRsZSIgdHJhbnNsYXRhYmxlPSJ5ZXMiPmljb248L3Byb3BlcnR5PgogICAgICAgICAgICAgICAg
PGNoaWxkPgogICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDZWxsUmVuZGVyZXJQaXhi
dWYiIGlkPSJpY29uLWNvbHVtbi1yZW5kZXJlZCIvPgogICAgICAgICAgICAgICAgICA8YXR0cmlidXRl
cz4KICAgICAgICAgICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9ImNlbGwtYmFja2dyb3VuZCI+NDwv
YXR0cmlidXRlPgogICAgICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGUgbmFtZT0icGl4YnVmIj43PC9h
dHRyaWJ1dGU+CiAgICAgICAgICAgICAgICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAg
ICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtUcmVlVmlld0NvbHVtbiIg
aWQ9InRyZWV2aWV3Y29sdW1uMSI+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2lu
ZyI+ODwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWluX3dpZHRoIj4x
MjA8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InRpdGxlIiB0cmFuc2xh
dGFibGU9InllcyI+Y29sdW1uPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJleHBhbmQiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtDZWxsUmVuZGVyZXJUZXh0IiBpZD0idWlkLWNvbHVtbi1y
ZW5kZXJlciIvPgogICAgICAgICAgICAgICAgICA8YXR0cmlidXRlcz4KICAgICAgICAgICAgICAgICAg
ICA8YXR0cmlidXRlIG5hbWU9ImJhY2tncm91bmQiPjQ8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAg
ICAgICA8YXR0cmlidXRlIG5hbWU9ImZvcmVncm91bmQiPjM8L2F0dHJpYnV0ZT4KICAgICAgICAgICAg
ICAgICAgICA8YXR0cmlidXRlIG5hbWU9InRleHQiPjE8L2F0dHJpYnV0ZT4KICAgICAgICAgICAgICAg
ICAgICA8YXR0cmlidXRlIG5hbWU9IndlaWdodCI+NTwvYXR0cmlidXRlPgogICAgICAgICAgICAgICAg
ICAgIDxhdHRyaWJ1dGUgbmFtZT0idW5kZXJsaW5lIj44PC9hdHRyaWJ1dGU+CiAgICAgICAgICAgICAg
ICAgIDwvYXR0cmlidXRlcz4KICAgICAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICAgIDwvY2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgIDxw
YWNraW5nPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5n
PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8
cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+
MzwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAgPC9vYmplY3Q+CiAgPG9i
amVjdCBjbGFzcz0iR3RrU2l6ZUdyb3VwIiBpZD0ic2l6ZWdyb3VwIj4KICAgIDxwcm9wZXJ0eSBuYW1l
PSJtb2RlIj52ZXJ0aWNhbDwvcHJvcGVydHk+CiAgICA8d2lkZ2V0cz4KICAgICAgPHdpZGdldCBuYW1l
PSJoZWFkZXJfYm94Ii8+CiAgICAgIDx3aWRnZXQgbmFtZT0ibGFiZWwxIi8+CiAgICA8L3dpZGdldHM+
CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/UnverifiedWarning.xml": {
		local:   "definitions/UnverifiedWarning.xml",
		size:    4232,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJ2ZXJpZnktaW5mb2JhciI+CiAg
ICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkg
bmFtZT0ib3JpZW50YXRpb24iPnZlcnRpY2FsPC9wcm9wZXJ0eT4KICAgIDxjaGlsZD4KICAgICAgPG9i
amVjdCBjbGFzcz0iR3RrQm94IiBpZD0idmVyaWZ5LWNsb3NlLWluZm9iYXIiPgogICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0i
Y2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CgkgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBj
bGFzcz0iR3RrRXZlbnRCb3giPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJidXR0b24tcHJlc3MtZXZlbnQiIGhhbmRs
ZXI9Im9uX3ByZXNzX2ltYWdlIiBzd2FwcGVkPSJubyIvPgoJICAgICAgICA8Y2hpbGQ+CiAgICAgICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrSW1hZ2UiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
InBpeGVsX3NpemUiPjEyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJp
Y29uX25hbWUiPndpbmRvdy1jbG9zZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0iaWNvbl9zaXplIj4zPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJtYXJnaW5fcmlnaHQiPjM8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
Im1hcmdpbl90b3AiPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1h
cmdpbl9ib3R0b20iPjE8L3Byb3BlcnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAg
ICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJwYWNrX3R5cGUiPmVuZDwvcHJvcGVydHk+CgkgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlv
biI+MDwvcHJvcGVydHk+CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAg
PC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBvc2l0aW9uIj4wPC9wcm9wZXJ0eT4KICAgICAgPC9wYWNr
aW5nPgogICAgPC9jaGlsZD4KICAgIDxjaGlsZD4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBp
ZD0idmVyaWZ5LW5vdGlmaWNhdGlvbiI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9w
ZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0ic3BhY2luZyI+MzM8L3Byb3BlcnR5PgogICAgICAg
IDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0ltYWdlIiBpZD0idmVyaWZ5LWltYWdl
Ij4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW5fbGVmdCI+MTA8L3Byb3BlcnR5PgogICAgICAgICAgPC9v
YmplY3Q+CiAgICAgICAgICA8cGFja2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFu
ZCI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJvcGVydHk+
CiAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQ+CiAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCIgaWQ9InZlcmlmeS1tZXNzYWdlIj4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJv
cGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJoYWxpZ24iPnN0YXJ0PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGF0dHJpYnV0ZXM+CiAg
ICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ3ZWlnaHQiIHZhbHVlPSJib2xkIi8+CiAgICAgICAg
ICAgIDwvYXR0cmlidXRlcz4KCSAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAg
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAg
PHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBp
ZD0idmVyaWZ5LWJ1dHRvbiI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVl
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+VHJ1ZTwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJyZWNlaXZlc19kZWZhdWx0Ij5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9yaWdodCI+MTA8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3RvcCI+MTQ8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2JvdHRvbSI+MTI8L3Byb3BlcnR5PgogICAg
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrTGFiZWwiPgogICAg
ICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2libGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFsc2U8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9sZWZ0Ij4xMDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX3JpZ2h0Ij4xMDwvcHJvcGVydHk+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5WYWxpZGF0ZSBD
aGFubmVsPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29iamVjdD4KCSAgICAgICAgPC9jaGlsZD4K
ICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBhY2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJleHBhbmQiPkZhbHNlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icGFja190eXBl
Ij5lbmQ8L3Byb3BlcnR5PgoJICAgICAgICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3Bl
cnR5PgogICAgICAgICAgPC9wYWNraW5nPgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0Pgog
ICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVy
dHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZpbGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MTwvcHJvcGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwv
Y2hpbGQ+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/VerificationFailed.xml": {
		local:   "definitions/VerificationFailed.xml",
		size:    2387,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJ2ZXJpZi1mYWlsdXJlLWRp
YWxvZyI+CiAgICA8cHJvcGVydHkgbmFtZT0id2luZG93LXBvc2l0aW9uIj5HVEtfV0lOX1BPU19DRU5U
RVI8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlci13aWR0aCI+NzwvcHJvcGVydHk+
CiAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj48L3Byb3BlcnR5Pgog
ICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94
Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyLXdpZHRoIj4xMDwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fVkVSVElDQUw8L3By
b3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0dyaWQiPgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iaGFsaWduIj5HVEtfQUxJR05fQ0VOVEVSPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVs
Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tdG9wIj41PC9wcm9wZXJ0eT4K
ICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tYm90dG9tIj4xNTwvcHJvcGVydHk+
CgkJICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5WZXJpZmlj
YXRpb24gRmFpbGVkPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAg
ICAgICAgICAgICA8YXR0cmlidXRlIG5hbWU9ImZvbnQtZGVzYyIgdmFsdWU9IiZsdDtFbnRlciBWYWx1
ZSZndDsgMTYiLz4KCSAgICAgICAgICAgICAgPGF0dHJpYnV0ZSBuYW1lPSJ3ZWlnaHQiIHZhbHVlPSJz
ZW1pYm9sZCIvPgogICAgICAgICAgICAgICAgPC9hdHRyaWJ1dGVzPgoJICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICAgIDxwYWNraW5nPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Imxl
ZnQtYXR0YWNoIj4wPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ0b3At
YXR0YWNoIj4xPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L3BhY2tpbmc+CiAgICAgICAgICAgIDwv
Y2hpbGQ+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4K
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0idmVyaWYtZmFpbHVyZS1sYWJlbCI+
CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPjwvcHJv
cGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJtYXJnaW4tYm90dG9tIj4xMDwvcHJvcGVy
dHk+CiAgICAgICAgICA8L29iamVjdD4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0dyaWQiPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0i
Y29sdW1uLXNwYWNpbmciPjY8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0icm93
LXNwYWNpbmciPjI8L3Byb3BlcnR5PgogICAgICAgICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFj
dGlvbl9hcmVhIj4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJi
dXR0b25fYm94Ij4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RL
X09SSUVOVEFUSU9OX0hPUklaT05UQUw8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPGNoaWxkPgog
ICAgICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJ2ZXJpZi1mYWlsdXJl
LWJ1dHRvbiI+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbi1kZWZhdWx0Ij50
cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXRv
cCI+NTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0
cmFuc2xhdGFibGU9InllcyI+VHJ5IGFnYWluIGxhdGVyPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAg
ICAgPC9vYmplY3Q+CiAgICAgICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICAgIDwvb2JqZWN0
PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4K
ICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPHN0eWxlPgogICAgICA8Y2xhc3MgbmFtZT0i
Y295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/VerificationSucceeded.xml": {
		local:   "definitions/VerificationSucceeded.xml",
		size:    1313,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJ2ZXJpZi1zdWNjZXNzLWRp
YWxvZyI+CiAgICA8cHJvcGVydHkgbmFtZT0id2luZG93LXBvc2l0aW9uIj5HVEtfV0lOX1BPU19DRU5U
RVI8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlci13aWR0aCI+NzwvcHJvcGVydHk+
CiAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj48L3Byb3BlcnR5Pgog
ICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94
Ij4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyLXdpZHRoIj4xMDwvcHJvcGVydHk+CiAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fVkVSVElDQUw8L3By
b3BlcnR5PgogICAgICAgIDxjaGlsZD4KICAgICAgICAgIDxvYmplY3QgIGNsYXNzPSJHdGtJbWFnZSIg
aWQ9InZlcmlmLXN1Y2Nlc3MtaW1hZ2UiPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgICA8cGFj
a2luZz4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InBhZGRpbmciPjIwPC9wcm9wZXJ0eT4KICAg
ICAgICAgIDwvcGFja2luZz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0idmVyaWYtc3VjY2Vzcy1sYWJlbCIvPgogICAgICAg
IDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkIGludGVybmFsLWNoaWxkPSJhY3Rpb25fYXJlYSI+CiAgICAg
ICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b25Cb3giIGlkPSJidXR0b25fYm94Ij4KICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fSE9SSVpPTlRBTDwv
cHJvcGVydHk+CiAgICAgICAgICAgIDxjaGlsZD4KICAgICAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJH
dGtCdXR0b24iIGlkPSJ2ZXJpZi1zdWNjZXNzLWJ1dHRvbiI+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5PSzwvcHJvcGVydHk+CiAgICAgICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luLXRvcCI+MTA8L3Byb3BlcnR5PgoJICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9j
aGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAgPHN0eWxlPgogICAgICA8Y2xhc3Mg
bmFtZT0iY295aW0iLz4KICAgIDwvc3R5bGU+CiAgPC9vYmplY3Q+CjwvaW50ZXJmYWNlPgo=
`,
	},

	"/definitions/VerifyFingerprint.xml": {
		local:   "definitions/VerifyFingerprint.xml",
		size:    1941,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkaWFsb2ciPgogICAgPHBy
b3BlcnR5IG5hbWU9IndpbmRvdy1wb3NpdGlvbiI+R1RLX1dJTl9QT1NfQ0VOVEVSPC9wcm9wZXJ0eT4K
ICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0idmJveCI+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Jv
eCIgaWQ9ImJveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlci13aWR0aCI+MTA8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJob21vZ2VuZW91cyI+ZmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX1ZFUlRJQ0FM
PC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJl
bCIgaWQ9Im1lc3NhZ2UiLz4KICAgICAgICA8L2NoaWxkPgogICAgICAgIDxjaGlsZD4KICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0xhYmVsIj4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InZpc2li
bGUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5G
YWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRh
YmxlPSJ5ZXMiPlRhZyAob3B0aW9uYWwgbmFtZSBmb3IgdGhpcyBmaW5nZXJwcmludCk6PC9wcm9wZXJ0
eT4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgPG9iamVjdCBjbGFzcz0iR3RrRW50cnkiIGlkPSJ0YWctZW50cnkiPgogICAgICAgICAgICA8
cHJvcGVydHkgbmFtZT0idmlzaWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJjYW5fZm9jdXMiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAg
ICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAg
ICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCIgaWQ9ImJ1dHRvbl9ib3giPgogICAgICAg
ICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRBVElPTl9IT1JJWk9OVEFM
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a0J1dHRvbiIgaWQ9ImJ1dHRvbl9jYW5jZWwiPgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5h
bWU9ImNhbi1kZWZhdWx0Ij50cnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPk5vPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8
L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAg
ICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImJ1dHRvbl92ZXJpZnkiPgogICAgICAgICAg
ICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+WWVzPC9wcm9wZXJ0
eT4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwv
b2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAgPC9jaGlsZD4KICAgIDxh
Y3Rpb24td2lkZ2V0cz4KICAgICAgPGFjdGlvbi13aWRnZXQgcmVzcG9uc2U9Im5vIiBkZWZhdWx0PSJ0
cnVlIj5idXR0b25fY2FuY2VsPC9hY3Rpb24td2lkZ2V0PgogICAgICA8YWN0aW9uLXdpZGdldCByZXNw
b25zZT0ieWVzIj5idXR0b25fdmVyaWZ5PC9hY3Rpb24td2lkZ2V0PgogICAgPC9hY3Rpb24td2lkZ2V0
cz4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAgICA8L3N0eWxlPgogIDwv
b2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/VerifyFingerprintUnknown.xml": {
		local:   "definitions/VerifyFingerprintUnknown.xml",
		size:    1153,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtEaWFsb2ciIGlkPSJkaWFsb2ciPgogICAgPHBy
b3BlcnR5IG5hbWU9IndpbmRvdy1wb3NpdGlvbiI+R1RLX1dJTl9QT1NfQ0VOVEVSPC9wcm9wZXJ0eT4K
ICAgIDxjaGlsZCBpbnRlcm5hbC1jaGlsZD0idmJveCI+CiAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0Jv
eCIgaWQ9ImJveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlci13aWR0aCI+MTA8L3Byb3Bl
cnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJob21vZ2VuZW91cyI+ZmFsc2U8L3Byb3BlcnR5Pgog
ICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJvcmllbnRhdGlvbiI+R1RLX09SSUVOVEFUSU9OX1ZFUlRJQ0FM
PC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtMYWJl
bCIgaWQ9Im1lc3NhZ2UiIC8+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQgaW50ZXJuYWwt
Y2hpbGQ9ImFjdGlvbl9hcmVhIj4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCIg
aWQ9ImJ1dHRvbl9ib3giPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdU
S19PUklFTlRBVElPTl9IT1JJWk9OVEFMPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAg
ICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImJ1dHRvbl9vayI+CiAgICAgICAg
ICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWRlZmF1bHQiPnRydWU8L3Byb3BlcnR5PgogICAgICAg
ICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImxhYmVsIiB0cmFuc2xhdGFibGU9InllcyI+T0s8L3Byb3Bl
cnR5PgogICAgICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAg
PC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAgPC9vYmplY3Q+CiAgICA8L2NoaWxkPgogICAg
PGFjdGlvbi13aWRnZXRzPgogICAgICA8YWN0aW9uLXdpZGdldCByZXNwb25zZT0ib2siIGRlZmF1bHQ9
InRydWUiPmJ1dHRvbl9vazwvYWN0aW9uLXdpZGdldD4KICAgIDwvYWN0aW9uLXdpZGdldHM+CiAgICA8
c3R5bGU+CiAgICAgIDxjbGFzcyBuYW1lPSJjb3lpbSIvPgogICAgPC9zdHlsZT4KICA8L29iamVjdD4K
PC9pbnRlcmZhY2U+Cg==
`,
	},

	"/definitions/WaitingSMPComplete.xml": {
		local:   "definitions/WaitingSMPComplete.xml",
		size:    2133,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtCb3giIGlkPSJzbXAtd2FpdGluZy1pbmZvYmFy
Ij4KICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMiPkZhbHNlPC9wcm9wZXJ0eT4KICAgIDxwcm9w
ZXJ0eSBuYW1lPSJzcGFjaW5nIj4yNzwvcHJvcGVydHk+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmplY3Qg
Y2xhc3M9Ikd0a0ltYWdlIiBpZD0ic21wLXdhaXRpbmctaW1hZ2UiPgogICAgICAgIDxwcm9wZXJ0eSBu
YW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2Zv
Y3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9sZWZ0Ij4x
MDwvcHJvcGVydHk+CiAgICAgIDwvb2JqZWN0PgogICAgICA8cGFja2luZz4KICAgICAgICA8cHJvcGVy
dHkgbmFtZT0iZXhwYW5kIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImZp
bGwiPlRydWU8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJwb3NpdGlvbiI+MDwvcHJv
cGVydHk+CiAgICAgIDwvcGFja2luZz4KICAgIDwvY2hpbGQ+CiAgICA8Y2hpbGQ+CiAgICAgIDxvYmpl
Y3QgY2xhc3M9Ikd0a0xhYmVsIiBpZD0ic21wLXdhaXRpbmctbGFiZWwiPgogICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2Fu
X2ZvY3VzIj5GYWxzZTwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhhbGlnbiI+c3Rh
cnQ8L3Byb3BlcnR5PgogICAgICAgIDxhdHRyaWJ1dGVzPgogICAgICAgICAgPGF0dHJpYnV0ZSBuYW1l
PSJ3ZWlnaHQiIHZhbHVlPSJib2xkIi8+CiAgICAgICAgPC9hdHRyaWJ1dGVzPgogICAgICA8L29iamVj
dD4KICAgICAgPHBhY2tpbmc+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImV4cGFuZCI+RmFsc2U8L3By
b3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJmaWxsIj5UcnVlPC9wcm9wZXJ0eT4KICAgICAg
ICA8cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjE8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAg
ICA8L2NoaWxkPgogICAgPGNoaWxkPgogICAgICA8b2JqZWN0IGNsYXNzPSJHdGtCdXR0b24iIGlkPSJz
bXAtd2FpdGluZy1idXR0b24iPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxlIj5UcnVlPC9w
cm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuX2ZvY3VzIj5UcnVlPC9wcm9wZXJ0eT4K
ICAgICAgICA8cHJvcGVydHkgbmFtZT0icmVjZWl2ZXNfZGVmYXVsdCI+VHJ1ZTwvcHJvcGVydHk+CiAg
ICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9yaWdodCI+MTA8L3Byb3BlcnR5PgogICAgICAgIDxw
cm9wZXJ0eSBuYW1lPSJtYXJnaW5fdG9wIj4xNDwvcHJvcGVydHk+CiAgICAgICAgPHByb3BlcnR5IG5h
bWU9Im1hcmdpbl9ib3R0b20iPjEyPC9wcm9wZXJ0eT4KICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8
b2JqZWN0IGNsYXNzPSJHdGtMYWJlbCI+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJ2aXNpYmxl
Ij5UcnVlPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9ImNhbl9mb2N1cyI+RmFs
c2U8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWFyZ2luX2xlZnQiPjM1PC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9Im1hcmdpbl9yaWdodCI+MzU8L3Byb3Bl
cnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibGFiZWwiIHRyYW5zbGF0YWJsZT0ieWVzIj5D
YW5jZWw8L3Byb3BlcnR5PgogICAgICAgICAgPC9vYmplY3Q+CiAgICAgICAgPC9jaGlsZD4KICAgICAg
PC9vYmplY3Q+CiAgICAgIDxwYWNraW5nPgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPkZh
bHNlPC9wcm9wZXJ0eT4KICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9InBhY2tfdHlwZSI+ZW5kPC9wcm9wZXJ0eT4KICAgICAgICA8
cHJvcGVydHkgbmFtZT0icG9zaXRpb24iPjI8L3Byb3BlcnR5PgogICAgICA8L3BhY2tpbmc+CiAgICA8
L2NoaWxkPgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/XMLConsole.xml": {
		local:   "definitions/XMLConsole.xml",
		size:    3087,
		modtime: 1489449600,
		compressed: `
PGludGVyZmFjZT4KICA8b2JqZWN0IGNsYXNzPSJHdGtUZXh0QnVmZmVyIiBpZD0iY29uc29sZUNvbnRl
bnQiPgogICAgPHByb3BlcnR5IG5hbWU9InRleHQiIHRyYW5zbGF0YWJsZT0ieWVzIj5IZWxsbyB3b3Js
ZDwvcHJvcGVydHk+CiAgPC9vYmplY3Q+CiAgPG9iamVjdCBjbGFzcz0iR3RrRGlhbG9nIiBpZD0iWE1M
Q29uc29sZSI+CiAgICA8cHJvcGVydHkgbmFtZT0id2luZG93LXBvc2l0aW9uIj5HVEtfV0lOX1BPU19D
RU5URVI8L3Byb3BlcnR5PgogICAgPHByb3BlcnR5IG5hbWU9ImJvcmRlcl93aWR0aCI+NjwvcHJvcGVy
dHk+CiAgICA8cHJvcGVydHkgbmFtZT0idGl0bGUiIHRyYW5zbGF0YWJsZT0ieWVzIj5YTVBQIENvbnNv
bGU6IEFDQ09VTlRfTkFNRTwvcHJvcGVydHk+CiAgICA8cHJvcGVydHkgbmFtZT0icmVzaXphYmxlIj5U
cnVlPC9wcm9wZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJkZWZhdWx0LWhlaWdodCI+NDAwPC9wcm9w
ZXJ0eT4KICAgIDxwcm9wZXJ0eSBuYW1lPSJkZWZhdWx0LXdpZHRoIj41MDA8L3Byb3BlcnR5PgogICAg
PHByb3BlcnR5IG5hbWU9ImRlc3Ryb3ktd2l0aC1wYXJlbnQiPnRydWU8L3Byb3BlcnR5PgogICAgPGNo
aWxkIGludGVybmFsLWNoaWxkPSJ2Ym94Ij4KICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQm94IiBpZD0i
VmJveCI+CiAgICAgICAgPHByb3BlcnR5IG5hbWU9ImhvbW9nZW5lb3VzIj5mYWxzZTwvcHJvcGVydHk+
CiAgICAgICAgPHByb3BlcnR5IG5hbWU9Im9yaWVudGF0aW9uIj5HVEtfT1JJRU5UQVRJT05fVkVSVElD
QUw8L3Byb3BlcnR5PgogICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJzcGFjaW5nIj42PC9wcm9wZXJ0eT4K
ICAgICAgICA8Y2hpbGQ+CiAgICAgICAgICA8b2JqZWN0IGNsYXNzPSJHdGtTY3JvbGxlZFdpbmRvdyIg
aWQ9Im1lc3NhZ2Vfc2Nyb2xsZWR3aW5kb3ciPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idmlz
aWJsZSI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJjYW5fZm9jdXMi
PlRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibm9fc2hvd19hbGwiPlRy
dWU8L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iYm9yZGVyX3dpZHRoIj42PC9w
cm9wZXJ0eT4KICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9InNoYWRvd190eXBlIj5ldGNoZWQtaW48
L3Byb3BlcnR5PgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ibWluX2NvbnRlbnRfaGVpZ2h0Ij41
PC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAgIDxvYmplY3QgY2xhc3M9
Ikd0a1RleHRWaWV3IiBpZD0ibWVzc2FnZV90ZXh0dmlldyI+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0idmlzaWJsZSI+dHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0id3JhcC1tb2RlIj4yPC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1l
PSJlZGl0YWJsZSI+ZmFsc2U8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHByb3BlcnR5IG5hbWU9
ImN1cnNvci12aXNpYmxlIj5mYWxzZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkg
bmFtZT0icGl4ZWxzLWJlbG93LWxpbmVzIj41PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9w
ZXJ0eSBuYW1lPSJsZWZ0LW1hcmdpbiI+NTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVy
dHkgbmFtZT0icmlnaHQtbWFyZ2luIj41PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICAgIDxwcm9wZXJ0
eSBuYW1lPSJidWZmZXIiPmNvbnNvbGVDb250ZW50PC9wcm9wZXJ0eT4KICAgICAgICAgICAgICA8L29i
amVjdD4KICAgICAgICAgICAgPC9jaGlsZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgPHBh
Y2tpbmc+CiAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJleHBhbmQiPlRydWU8L3Byb3BlcnR5Pgog
ICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iZmlsbCI+VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICA8
L3BhY2tpbmc+CiAgICAgICAgPC9jaGlsZD4KICAgICAgICA8Y2hpbGQgaW50ZXJuYWwtY2hpbGQ9ImFj
dGlvbl9hcmVhIj4KICAgICAgICAgIDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbkJveCIgaWQ9ImJ1dHRv
bl9ib3giPgogICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0ib3JpZW50YXRpb24iPkdUS19PUklFTlRB
VElPTl9IT1JJWk9OVEFMPC9wcm9wZXJ0eT4KICAgICAgICAgICAgPGNoaWxkPgogICAgICAgICAgICAg
IDxvYmplY3QgY2xhc3M9Ikd0a0J1dHRvbiIgaWQ9ImJ1dHRvbl9yZWZyZXNoIj4KICAgICAgICAgICAg
ICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCI+X1JlZnJlc2g8L3Byb3BlcnR5PgogICAgICAgICAgICAg
ICAgPHByb3BlcnR5IG5hbWU9InVzZS11bmRlcmxpbmUiPlRydWU8L3Byb3BlcnR5PgogICAgICAgICAg
ICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5kbGVyPSJvbl9yZWZyZXNoIiAvPgogICAgICAg
ICAgICAgIDwvb2JqZWN0PgogICAgICAgICAgICA8L2NoaWxkPgogICAgICAgICAgICA8Y2hpbGQ+CiAg
ICAgICAgICAgICAgPG9iamVjdCBjbGFzcz0iR3RrQnV0dG9uIiBpZD0iYnV0dG9uX2Nsb3NlIj4KICAg
ICAgICAgICAgICAgIDxwcm9wZXJ0eSBuYW1lPSJsYWJlbCIgdHJhbnNsYXRhYmxlPSJ5ZXMiPl9DbG9z
ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0idXNlLXVuZGVybGluZSI+
VHJ1ZTwvcHJvcGVydHk+CiAgICAgICAgICAgICAgICA8cHJvcGVydHkgbmFtZT0iY2FuLWRlZmF1bHQi
PnRydWU8L3Byb3BlcnR5PgogICAgICAgICAgICAgICAgPHNpZ25hbCBuYW1lPSJjbGlja2VkIiBoYW5k
bGVyPSJvbl9jbG9zZSIgLz4KICAgICAgICAgICAgICA8L29iamVjdD4KICAgICAgICAgICAgPC9jaGls
ZD4KICAgICAgICAgIDwvb2JqZWN0PgogICAgICAgIDwvY2hpbGQ+CiAgICAgIDwvb2JqZWN0PgogICAg
PC9jaGlsZD4KICAgIDxzdHlsZT4KICAgICAgPGNsYXNzIG5hbWU9ImNveWltIi8+CiAgICA8L3N0eWxl
PgogIDwvb2JqZWN0Pgo8L2ludGVyZmFjZT4K
`,
	},

	"/definitions/images/alert.svg": {
		local:   "definitions/images/alert.svg",
		size:    1582,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iMTUiIGhlaWdodD0iMTQiIHZpZXdCb3g9IjAgMCAxNSAxNCIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KICAgIDxwYXRoIGZpbGwtcnVsZT0iZXZl
bm9kZCIgY2xpcC1ydWxlPSJldmVub2RkIiBkPSJNOC41NzAyMiAxMS40OTY1VjkuOTA3ODdDOC41NzAy
MiA5LjgyOTgzIDguNTQzNzMgOS43NjQzNCA4LjQ5MDc4IDkuNzExMzlDOC40Mzc4MyA5LjY1ODQzIDgu
Mzc1MTIgOS42MzE5NiA4LjMwMjY2IDkuNjMxOTZINi42OTczNEM2LjYyNDg4IDkuNjMxOTYgNi41NjIx
NyA5LjY1ODQzIDYuNTA5MjEgOS43MTEzOUM2LjQ1NjI2IDkuNzY0MzQgNi40Mjk3OCA5LjgyOTgzIDYu
NDI5NzggOS45MDc4N1YxMS40OTY1QzYuNDI5NzggMTEuNTc0NSA2LjQ1NjI2IDExLjY0IDYuNTA5MjEg
MTEuNjkzQzYuNTYyMTcgMTEuNzQ1OSA2LjYyNDg4IDExLjc3MjQgNi42OTczNCAxMS43NzI0SDguMzAy
NjZDOC4zNzUxMiAxMS43NzI0IDguNDM3ODMgMTEuNzQ1OSA4LjQ5MDc4IDExLjY5M0M4LjU0MzczIDEx
LjY0IDguNTcwMjIgMTEuNTc0NSA4LjU3MDIyIDExLjQ5NjVaTTguNTUzNDkgOC4zNjk0M0w4LjcwMzk5
IDQuNTMxN0M4LjcwMzk5IDQuNDY0ODEgOC42NzYxMSA0LjQxMTg2IDguNjIwMzcgNC4zNzI4NEM4LjU0
NzkxIDQuMzExNTMgOC40ODEwNCA0LjI4MDg3IDguNDE5NzIgNC4yODA4N0g2LjU4MDI3QzYuNTE4OTYg
NC4yODA4NyA2LjQ1MjA4IDQuMzExNTMgNi4zNzk2MiA0LjM3Mjg0QzYuMzIzODggNC40MTE4NiA2LjI5
NiA0LjQ3MDM5IDYuMjk2IDQuNTQ4NDJMNi40MzgxNCA4LjM2OTQzQzYuNDM4MTQgOC40MjUxOCA2LjQ2
NjAyIDguNDcxMTYgNi41MjE3NiA4LjUwNzM5QzYuNTc3NSA4LjU0MzYyIDYuNjQ0MzkgOC41NjE3NCA2
LjcyMjQyIDguNTYxNzRIOC4yNjkyMkM4LjM0NzI2IDguNTYxNzQgOC40MTI3NCA4LjU0MzYyIDguNDY1
NyA4LjUwNzM5QzguNTE4NjUgOC40NzExNiA4LjU0NzkyIDguNDI1MTggOC41NTM0OSA4LjM2OTQzWk04
LjQzNjQ0IDAuNTYwMTkyTDE0Ljg1NzcgMTIuMzMyNkMxNS4wNTI4IDEyLjY4MzggMTUuMDQ3MyAxMy4w
MzQ5IDE0Ljg0MSAxMy4zODYxQzE0Ljc0NjMgMTMuNTQ3NyAxNC42MTY3IDEzLjY3NTkgMTQuNDUyMiAx
My43NzA3QzE0LjI4NzggMTMuODY1NCAxNC4xMTA4IDEzLjkxMjggMTMuOTIxMyAxMy45MTI4SDEuMDc4
NjlDMC44ODkxNjggMTMuOTEyOCAwLjcxMjE5MiAxMy44NjU0IDAuNTQ3NzU3IDEzLjc3MDdDMC4zODMz
MjIgMTMuNjc1OSAwLjI1MzczOCAxMy41NDc3IDAuMTU4OTc4IDEzLjM4NjFDLTAuMDQ3MjYyOCAxMy4w
MzQ5IC0wLjA1MjgzODQgMTIuNjgzOCAwLjE0MjI1NSAxMi4zMzI2TDYuNTYzNTUgMC41NjAxOTJDNi42
NTgzMSAwLjM4NzM5NiA2Ljc4OTMgMC4yNTA4MzMgNi45NTY1MiAwLjE1MDQ5OUM3LjEyMzc1IDAuMDUw
MTY1OSA3LjMwNDkgMCA3LjUgMEM3LjY5NTA5IDAgNy44NzYyNSAwLjA1MDE2NTkgOC4wNDM0NyAwLjE1
MDQ5OUM4LjIxMDY5IDAuMjUwODMzIDguMzQxNjkgMC4zODczOTYgOC40MzY0NCAwLjU2MDE5MloiIGZp
bGw9IiNGODlCMUMiLz4KPC9zdmc+Cg==
`,
	},

	"/definitions/images/failure.svg": {
		local:   "definitions/images/failure.svg",
		size:    1153,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0idXRmLTgiPz4KPCEtLSBHZW5lcmF0b3I6IEFkb2Jl
IElsbHVzdHJhdG9yIDIyLjAuMSwgU1ZHIEV4cG9ydCBQbHVnLUluIC4gU1ZHIFZlcnNpb246IDYuMDAg
QnVpbGQgMCkgIC0tPgo8c3ZnIHZlcnNpb249IjEuMSIgaWQ9IkxheWVyXzEiIHhtbG5zPSJodHRwOi8v
d3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkveGxp
bmsiIHg9IjBweCIgeT0iMHB4IgoJIHZpZXdCb3g9IjAgMCA1MiA1MyIgc3R5bGU9ImVuYWJsZS1iYWNr
Z3JvdW5kOm5ldyAwIDAgNTIgNTM7IiB4bWw6c3BhY2U9InByZXNlcnZlIj4KPHN0eWxlIHR5cGU9InRl
eHQvY3NzIj4KCS5zdDB7ZmlsbDpub25lO3N0cm9rZTojQ0MzNjM2O3N0cm9rZS1saW5lY2FwOnJvdW5k
O3N0cm9rZS1saW5lam9pbjpyb3VuZDtzdHJva2UtbWl0ZXJsaW1pdDoxMDt9Cgkuc3Qxe2ZpbGw6I0ZD
QUVBRTtzdHJva2U6I0NDMzYzNjtzdHJva2UtbGluZWNhcDpyb3VuZDtzdHJva2UtbGluZWpvaW46cm91
bmQ7c3Ryb2tlLW1pdGVybGltaXQ6MTA7fQo8L3N0eWxlPgo8Zz4KCTxjaXJjbGUgY2xhc3M9InN0MCIg
Y3g9IjI4LjYiIGN5PSIyNy41IiByPSIyMS4xIi8+Cgk8cGF0aCBjbGFzcz0ic3QwIiBkPSJNMTIuNSwy
My43YzAtMy43LDEuNy03LDQuNC05LjMiLz4KCTxwYXRoIGNsYXNzPSJzdDAiIGQ9Ik0xOC4zLDEzLjVj
MC4zLTAuMiwwLjYtMC40LDAuOS0wLjUiLz4KCTxwb2x5Z29uIGNsYXNzPSJzdDEiIHBvaW50cz0iMzcu
OCwyMi45IDMzLjksMTguOSAyOC42LDI0LjMgMjMuMiwxOC45IDE5LjMsMjIuOSAyNC42LDI4LjIgMTku
MywzMy42IDIzLjIsMzcuNSAyOC42LDMyLjIgCgkJMzMuOSwzNy41IDM3LjgsMzMuNiAzMi41LDI4LjIg
CSIvPgoJPHBhdGggY2xhc3M9InN0MCIgZD0iTTIxLjEsMTIuMWMxLjEtMC4zLDIuMi0wLjUsMy40LTAu
NSIvPgoJPHBhdGggY2xhc3M9InN0MCIgZD0iTTM1LjIsNDIuNmMtMS4xLDAuMy0yLjIsMC41LTMuNCww
LjUiLz4KCTxwYXRoIGNsYXNzPSJzdDAiIGQ9Ik00My45LDMxYzAsMy43LTEuNyw3LTQuNCw5LjMiLz4K
CTxwYXRoIGNsYXNzPSJzdDAiIGQ9Ik0zOC4xLDQxLjNjLTAuMywwLjItMC42LDAuNC0wLjksMC41Ii8+
CjwvZz4KPC9zdmc+Cg==
`,
	},

	"/definitions/images/field_error.svg": {
		local:   "definitions/images/field_error.svg",
		size:    1079,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iMTYiIGhlaWdodD0iMTYiIHZpZXdCb3g9IjAgMCAxNiAxNiIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2Rk
IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0xNiA4QzE2IDEwLjEyMTcgMTUuMTU3MSAxMi4xNTY2IDEz
LjY1NjkgMTMuNjU2OUMxMi4xNTY2IDE1LjE1NzEgMTAuMTIxNyAxNiA4IDE2QzUuODc4MjcgMTYgMy44
NDM0NCAxNS4xNTcxIDIuMzQzMTUgMTMuNjU2OUMwLjg0Mjg1NSAxMi4xNTY2IDAgMTAuMTIxNyAwIDhD
MCA1Ljg3ODI3IDAuODQyODU1IDMuODQzNDQgMi4zNDMxNSAyLjM0MzE1QzMuODQzNDQgMC44NDI4NTUg
NS44NzgyNyAwIDggMEMxMC4xMjE3IDAgMTIuMTU2NiAwLjg0Mjg1NSAxMy42NTY5IDIuMzQzMTVDMTUu
MTU3MSAzLjg0MzQ0IDE2IDUuODc4MjcgMTYgOFY4Wk05IDEyQzkgMTIuMjY1MiA4Ljg5NDY0IDEyLjUx
OTYgOC43MDcxMSAxMi43MDcxQzguNTE5NTcgMTIuODk0NiA4LjI2NTIyIDEzIDggMTNDNy43MzQ3OCAx
MyA3LjQ4MDQzIDEyLjg5NDYgNy4yOTI4OSAxMi43MDcxQzcuMTA1MzYgMTIuNTE5NiA3IDEyLjI2NTIg
NyAxMkM3IDExLjczNDggNy4xMDUzNiAxMS40ODA0IDcuMjkyODkgMTEuMjkyOUM3LjQ4MDQzIDExLjEw
NTQgNy43MzQ3OCAxMSA4IDExQzguMjY1MjIgMTEgOC41MTk1NyAxMS4xMDU0IDguNzA3MTEgMTEuMjky
OUM4Ljg5NDY0IDExLjQ4MDQgOSAxMS43MzQ4IDkgMTJWMTJaTTggM0M3LjczNDc4IDMgNy40ODA0MyAz
LjEwNTM2IDcuMjkyODkgMy4yOTI4OUM3LjEwNTM2IDMuNDgwNDMgNyAzLjczNDc4IDcgNFY4QzcgOC4y
NjUyMiA3LjEwNTM2IDguNTE5NTcgNy4yOTI4OSA4LjcwNzExQzcuNDgwNDMgOC44OTQ2NCA3LjczNDc4
IDkgOCA5QzguMjY1MjIgOSA4LjUxOTU3IDguODk0NjQgOC43MDcxMSA4LjcwNzExQzguODk0NjQgOC41
MTk1NyA5IDguMjY1MjIgOSA4VjRDOSAzLjczNDc4IDguODk0NjQgMy40ODA0MyA4LjcwNzExIDMuMjky
ODlDOC41MTk1NyAzLjEwNTM2IDguMjY1MjIgMyA4IDNaIiBmaWxsPSIjQkMxRDFEIi8+Cjwvc3ZnPgo=
`,
	},

	"/definitions/images/filetransfer_receive.svg": {
		local:   "definitions/images/filetransfer_receive.svg",
		size:    3030,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0idXRmLTgiPz4KPCEtLSBHZW5lcmF0b3I6IEFkb2Jl
IElsbHVzdHJhdG9yIDIyLjAuMSwgU1ZHIEV4cG9ydCBQbHVnLUluIC4gU1ZHIFZlcnNpb246IDYuMDAg
QnVpbGQgMCkgIC0tPgo8c3ZnIHZlcnNpb249IjEuMSIgaWQ9IkxheWVyXzEiIHhtbG5zPSJodHRwOi8v
d3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkveGxp
bmsiIHg9IjBweCIgeT0iMHB4IgoJIHZpZXdCb3g9IjAgMCA2MS41IDYzLjciIHN0eWxlPSJlbmFibGUt
YmFja2dyb3VuZDpuZXcgMCAwIDYxLjUgNjMuNzsiIHhtbDpzcGFjZT0icHJlc2VydmUiPgo8c3R5bGUg
dHlwZT0idGV4dC9jc3MiPgoJLnN0MHtmaWxsOiNGRkZCRjg7fQoJLnN0MXtmaWxsOiNGRkU1RDQ7fQoJ
LnN0MntmaWxsOiNGMEU5RkY7c3Ryb2tlOiM4OTgwQTA7c3Ryb2tlLXdpZHRoOjEuMjU7c3Ryb2tlLWxp
bmVjYXA6cm91bmQ7c3Ryb2tlLW1pdGVybGltaXQ6MTA7fQoJLnN0M3tmaWxsOiNGMEU5RkY7c3Ryb2tl
OiM4OTgwQTA7c3Ryb2tlLWxpbmVjYXA6cm91bmQ7c3Ryb2tlLWxpbmVqb2luOnJvdW5kO3N0cm9rZS1t
aXRlcmxpbWl0OjEwO30KCS5zdDR7ZmlsbDpub25lO3N0cm9rZTojODk4MEEwO3N0cm9rZS13aWR0aDox
LjI1O3N0cm9rZS1saW5lY2FwOnJvdW5kO3N0cm9rZS1taXRlcmxpbWl0OjEwO30KCS5zdDV7ZmlsbDpu
b25lO3N0cm9rZTojODk4MEEwO3N0cm9rZS13aWR0aDoxLjI1O3N0cm9rZS1saW5lY2FwOnJvdW5kO3N0
cm9rZS1saW5lam9pbjpyb3VuZDtzdHJva2UtbWl0ZXJsaW1pdDoxMDt9Cgkuc3Q2e2ZpbGw6bm9uZTtz
dHJva2U6Izg5ODBBMDtzdHJva2UtbGluZWNhcDpyb3VuZDtzdHJva2UtbGluZWpvaW46cm91bmQ7c3Ry
b2tlLW1pdGVybGltaXQ6MTA7fQoJLnN0N3tmaWxsOm5vbmU7c3Ryb2tlOiM4OTgwQTA7c3Ryb2tlLXdp
ZHRoOjAuNzU7c3Ryb2tlLWxpbmVjYXA6cm91bmQ7c3Ryb2tlLWxpbmVqb2luOnJvdW5kO3N0cm9rZS1t
aXRlcmxpbWl0OjEwO30KCS5zdDh7ZmlsbDojRjBFOUZGO3N0cm9rZTojODk4MEEwO3N0cm9rZS13aWR0
aDoxLjI1O3N0cm9rZS1saW5lY2FwOnJvdW5kO3N0cm9rZS1saW5lam9pbjpyb3VuZDtzdHJva2UtbWl0
ZXJsaW1pdDoxMDt9Cgkuc3Q5e2ZpbGw6I0YwRTlGRjt9Cjwvc3R5bGU+CjxjaXJjbGUgaWQ9IlhNTElE
XzhfIiBjbGFzcz0ic3QwIiBjeD0iMzIiIGN5PSIzMC45IiByPSIyNyIvPgo8cG9seWxpbmUgaWQ9IlJl
Y3RhbmdsZS00LUNvcHktMl8xXyIgY2xhc3M9InN0MSIgcG9pbnRzPSI0OCwxMi45IDQyLjcsMTIuOSA0
Mi43LDE4LjMgNDgsMTguMyAiLz4KPHBhdGggY2xhc3M9InN0MiIgZD0iTTI2LDE5aC00LjVjLTAuNyww
LTEuMywwLjUtMS4zLDEuMmwtMC4xLDEuM2MtMC4xLDAuNy0wLjYsMS4yLTEuMywxLjJoLTAuM2MtMC45
LDAtMS43LDAuNy0xLjgsMS42bC0wLjEsMS4xIgoJLz4KPHBhdGggY2xhc3M9InN0MiIgZD0iTTQ4LjMs
NDEuNWwyLjItMTcuM2MwLjEtMC45LTAuOS0xLjctMi4zLTEuN0gzMi40Yy0xLDAtMS44LTAuNi0xLjct
MS40bDAuMS0wLjdjMC4xLTAuNy0wLjYtMS40LTEuNy0xLjRoLTQuNSIKCS8+Cjxwb2x5bGluZSBjbGFz
cz0ic3QzIiBwb2ludHM9IjIxLjIsMjcuNCAyMS41LDI1LjcgNDYuOSwyNS43IDQ2LDMyICIvPgo8cGF0
aCBjbGFzcz0ic3Q0IiBkPSJNMTAuNiwyN2g3LjFjMC44LDAsMS41LDAuNSwxLjcsMS4zbDAuNCwxLjZj
MC4yLDAuNiwwLjcsMSwxLjMsMWgyMS4yYzAuOCwwLDEuNSwwLjUsMS43LDEuM2w0LjEsMTUuMkgxNS4z
CgljLTAuNiwwLTEuMS0wLjQtMS4zLTFMOS40LDI4LjciLz4KPGxpbmUgaWQ9IlhNTElEXzdfIiBjbGFz
cz0ic3Q1IiB4MT0iMy4xIiB5MT0iNDgiIHgyPSIxMC4yIiB5Mj0iNDgiLz4KPGxpbmUgaWQ9IlhNTElE
XzlfIiBjbGFzcz0ic3Q1IiB4MT0iNTIuMyIgeTE9IjQ4IiB4Mj0iNTQuMyIgeTI9IjQ4Ii8+CjxsaW5l
IGlkPSJYTUxJRF81XyIgY2xhc3M9InN0NiIgeDE9IjEzLjEiIHkxPSI0OCIgeDI9IjUwIiB5Mj0iNDgi
Lz4KPGxpbmUgaWQ9IlhNTElEXzRfIiBjbGFzcz0ic3Q3IiB4MT0iMTIuMiIgeTE9IjUxLjIiIHgyPSIx
OC4yIiB5Mj0iNTEuMiIvPgo8bGluZSBpZD0iWE1MSURfM18iIGNsYXNzPSJzdDciIHgxPSIyMS44IiB5
MT0iNTEuMiIgeDI9IjI3LjciIHkyPSI1MS4yIi8+CjxsaW5lIGlkPSJYTUxJRF8yXyIgY2xhc3M9InN0
NyIgeDE9IjMxLjEiIHkxPSI1MS4yIiB4Mj0iNDMuMiIgeTI9IjUxLjIiLz4KPGxpbmUgaWQ9IlhNTElE
XzFfIiBjbGFzcz0ic3Q3IiB4MT0iNDYuOSIgeTE9IjUxLjIiIHgyPSI1My44IiB5Mj0iNTEuMiIvPgo8
cmVjdCBpZD0iUmVjdGFuZ2xlLTQtQ29weS05XzFfIiB4PSI1Mi41IiB5PSIzNCIgY2xhc3M9InN0MSIg
d2lkdGg9IjIuNCIgaGVpZ2h0PSIzLjciLz4KPHBvbHlsaW5lIGlkPSJSZWN0YW5nbGUtNC1Db3B5LTE2
XzFfIiBjbGFzcz0ic3QxIiBwb2ludHM9IjE0LjMsMjIuMyAxNC4zLDE5LjggMTIuMSwxOS44IDEyLjEs
MjIuMyAiLz4KPHJlY3QgaWQ9IlJlY3RhbmdsZS00LUNvcHktMTlfMV8iIHg9IjE4LjUiIHk9IjE0LjQi
IGNsYXNzPSJzdDEiIHdpZHRoPSIyLjQiIGhlaWdodD0iMi40Ii8+CjxwYXRoIGNsYXNzPSJzdDgiIGQ9
Ik00Mi41LDIwLjFjLTAuOC0yLjUtMi4zLTQuNy00LjQtNi4zYy00LjktMy41LTExLjYtMi40LTE1Ljks
Mi40YzMuOC0xLjgsOC4yLTEuOCwxMS43LDAuNwoJYzEuNiwxLjEsMi44LDIuNiwzLjcsNC40bC0zLjks
MC45bDcsNS43bDQuOC04LjdMNDIuNSwyMC4xeiIvPgo8cGF0aCBjbGFzcz0ic3Q5IiBkPSJNMTIuNiwz
MC4xaDYuMmMwLjcsMCwxLjMsMC40LDEuNSwxbDAuMywxLjJjMC4yLDAuNCwwLjYsMC43LDEuMSwwLjdo
MTguNWMwLjcsMCwxLjMsMC40LDEuNSwxbDMuNiwxMS4zSDE2LjcKCWMtMC41LDAtMS0wLjMtMS4xLTAu
N2wtNC0xMy4yIi8+CjxsaW5lIGlkPSJYTUxJRF82XyIgY2xhc3M9InN0NSIgeDE9IjU2LjYiIHkxPSI0
OCIgeDI9IjU3LjgiIHkyPSI0OCIvPgo8L3N2Zz4K
`,
	},

	"/definitions/images/filetransfer_receive_send.svg": {
		local:   "definitions/images/filetransfer_receive_send.svg",
		size:    2786,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0idXRmLTgiPz4KPCEtLSBHZW5lcmF0b3I6IEFkb2Jl
IElsbHVzdHJhdG9yIDIyLjAuMSwgU1ZHIEV4cG9ydCBQbHVnLUluIC4gU1ZHIFZlcnNpb246IDYuMDAg
QnVpbGQgMCkgIC0tPgo8c3ZnIHZlcnNpb249IjEuMSIgaWQ9IkxheWVyXzEiIHhtbG5zPSJodHRwOi8v
d3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkveGxp
bmsiIHg9IjBweCIgeT0iMHB4IgoJIHZpZXdCb3g9IjAgMCA2MCA2Mi4yIiBzdHlsZT0iZW5hYmxlLWJh
Y2tncm91bmQ6bmV3IDAgMCA2MCA2Mi4yOyIgeG1sOnNwYWNlPSJwcmVzZXJ2ZSI+CjxzdHlsZSB0eXBl
PSJ0ZXh0L2NzcyI+Cgkuc3Qwe2ZpbGw6I0ZGRkJGODt9Cgkuc3Qxe2ZpbGw6I0ZGRTVENDt9Cgkuc3Qy
e2ZpbGw6I0YwRTlGRjtzdHJva2U6Izg5ODBBMDtzdHJva2Utd2lkdGg6MS4yNTtzdHJva2UtbGluZWNh
cDpyb3VuZDtzdHJva2UtbWl0ZXJsaW1pdDoxMDt9Cgkuc3Qze2ZpbGw6I0YwRTlGRjtzdHJva2U6Izg5
ODBBMDtzdHJva2Utd2lkdGg6MS4yNTtzdHJva2UtbGluZWNhcDpyb3VuZDtzdHJva2UtbGluZWpvaW46
cm91bmQ7c3Ryb2tlLW1pdGVybGltaXQ6MTA7fQoJLnN0NHtmaWxsOm5vbmU7c3Ryb2tlOiM4OTgwQTA7
c3Ryb2tlLWxpbmVjYXA6cm91bmQ7c3Ryb2tlLW1pdGVybGltaXQ6MTA7fQoJLnN0NXtmaWxsOm5vbmU7
c3Ryb2tlOiM4OTgwQTA7c3Ryb2tlLXdpZHRoOjEuMjU7c3Ryb2tlLWxpbmVjYXA6cm91bmQ7c3Ryb2tl
LWxpbmVqb2luOnJvdW5kO3N0cm9rZS1taXRlcmxpbWl0OjEwO30KCS5zdDZ7ZmlsbDpub25lO3N0cm9r
ZTojODk4MEEwO3N0cm9rZS13aWR0aDowLjc1O3N0cm9rZS1saW5lY2FwOnJvdW5kO3N0cm9rZS1saW5l
am9pbjpyb3VuZDtzdHJva2UtbWl0ZXJsaW1pdDoxMDt9Cjwvc3R5bGU+CjxjaXJjbGUgaWQ9IlhNTElE
XzExNzZfIiBjbGFzcz0ic3QwIiBjeD0iMzAuMyIgY3k9IjMwLjciIHI9IjI3Ii8+Cjxwb2x5bGluZSBp
ZD0iUmVjdGFuZ2xlLTQtQ29weS0yIiBjbGFzcz0ic3QxIiBwb2ludHM9IjQ2LjQsMTIuNyA0MSwxMi43
IDQxLDE4LjEgNDYuNCwxOC4xICIvPgo8cGF0aCBjbGFzcz0ic3QyIiBkPSJNMjYuNCwyMC40aC03LjNj
LTAuNSwwLTEsMC41LTEsMS4yTDE4LDIyLjljLTAuMSwwLjctMC41LDEuMi0xLDEuMmgtMC4yYy0wLjcs
MC0xLjMsMC43LTEuNCwxLjZsLTAuMSwxIi8+CjxwYXRoIGNsYXNzPSJzdDIiIGQ9Ik00Ny4xLDQzbDEu
NS0xOC4xYzAuMS0xLTAuNi0xLjktMS44LTEuOUgyOC41Ii8+CjxwYXRoIGNsYXNzPSJzdDMiIGQ9Ik0x
OS45LDMwLjZoMjAuOWMwLjgsMCwxLjUsMC41LDEuNywxLjNsNCwxNC45SDE0LjJjLTAuNiwwLTEuMS0w
LjQtMS4zLTFMOC4yLDI4LjRjLTAuMi0wLjgsMC40LTEuNywxLjMtMS43aDcKCWMwLjgsMCwxLjUsMC41
LDEuNywxLjNsMC40LDEuNkMxOC44LDMwLjIsMTkuMywzMC42LDE5LjksMzAuNnoiLz4KPHBvbHlsaW5l
IGNsYXNzPSJzdDQiIHBvaW50cz0iMzguNiwyNi4xIDQ2LjEsMjYuMSA0NS4xLDM1LjcgIi8+Cjxwb2x5
bGluZSBjbGFzcz0ic3Q0IiBwb2ludHM9IjE5LjksMjguMiAyMC4xLDI1LjQgMjUuNCwyNS40ICIvPgo8
cGF0aCBjbGFzcz0ic3QzIiBkPSJNMjUuOSwyNS43YzEuNiwxLjUsMy42LDIuNiw1LjksM2M1LjIsMC44
LDEwLjEtMi4zLDExLjctNy4yYy0yLjMsMi42LTUuOCw0LjEtOS42LDMuNQoJYy0xLjctMC4yLTMuMi0w
LjktNC42LTEuOWwyLjctMmwtNy43LTEuN2wtMC41LDcuOUwyNS45LDI1Ljd6Ii8+CjxwYXRoIGNsYXNz
PSJzdDMiIGQ9Ik00Mi41LDEzLjNsLTAuMS0wLjFjLTEuNS0xLjUtMy41LTIuNi01LjktM2MtNS4yLTAu
OC0xMC4xLDIuMy0xMS43LDcuMWMyLjMtMi42LDUuOS00LDkuNi0zLjQKCWMxLjcsMC4yLDMuMiwxLDQu
NiwxLjlsMC4xLDAuMUwzNi4zLDE4bDcuNywxLjhsMC42LTcuOUw0Mi41LDEzLjN6Ii8+CjxsaW5lIGlk
PSJYTUxJRF8xMTg5XyIgY2xhc3M9InN0NSIgeDE9IjIuNCIgeTE9IjQ2LjgiIHgyPSI5LjYiIHkyPSI0
Ni44Ii8+CjxsaW5lIGlkPSJYTUxJRF8xMTg4XyIgY2xhc3M9InN0NSIgeDE9IjUyLjEiIHkxPSI0Ni44
IiB4Mj0iNTciIHkyPSI0Ni44Ii8+CjxsaW5lIGlkPSJYTUxJRF8xMTg3XyIgY2xhc3M9InN0NSIgeDE9
IjEyLjQiIHkxPSI0Ni44IiB4Mj0iNDkuMyIgeTI9IjQ2LjgiLz4KPGxpbmUgaWQ9IlhNTElEXzExODNf
IiBjbGFzcz0ic3Q2IiB4MT0iMTAuNiIgeTE9IjQ5LjkiIHgyPSIxNi41IiB5Mj0iNDkuOSIvPgo8bGlu
ZSBpZD0iWE1MSURfMTE4Ml8iIGNsYXNzPSJzdDYiIHgxPSIyMC4xIiB5MT0iNDkuOSIgeDI9IjI2IiB5
Mj0iNDkuOSIvPgo8bGluZSBpZD0iWE1MSURfMTE4MV8iIGNsYXNzPSJzdDYiIHgxPSIyOS40IiB5MT0i
NDkuOSIgeDI9IjQxLjYiIHkyPSI0OS45Ii8+CjxsaW5lIGlkPSJYTUxJRF8xMTgwXyIgY2xhc3M9InN0
NiIgeDE9IjQ1LjIiIHkxPSI0OS45IiB4Mj0iNTIuMSIgeTI9IjQ5LjkiLz4KPHJlY3QgaWQ9IlJlY3Rh
bmdsZS00LUNvcHktOSIgeD0iNTAuOSIgeT0iMzMuNyIgY2xhc3M9InN0MSIgd2lkdGg9IjIuNCIgaGVp
Z2h0PSIzLjciLz4KPHBvbHlsaW5lIGlkPSJSZWN0YW5nbGUtNC1Db3B5LTE2IiBjbGFzcz0ic3QxIiBw
b2ludHM9IjEyLjcsMjIuMSAxMi43LDE5LjcgMTAuNCwxOS43IDEwLjQsMjIuMSAiLz4KPHJlY3QgaWQ9
IlJlY3RhbmdsZS00LUNvcHktMTkiIHg9IjE2LjgiIHk9IjE0LjIiIGNsYXNzPSJzdDEiIHdpZHRoPSIy
LjQiIGhlaWdodD0iMi40Ii8+Cjwvc3ZnPgo=
`,
	},

	"/definitions/images/filetransfer_send.svg": {
		local:   "definitions/images/filetransfer_send.svg",
		size:    2609,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0idXRmLTgiPz4KPCEtLSBHZW5lcmF0b3I6IEFkb2Jl
IElsbHVzdHJhdG9yIDIyLjAuMSwgU1ZHIEV4cG9ydCBQbHVnLUluIC4gU1ZHIFZlcnNpb246IDYuMDAg
QnVpbGQgMCkgIC0tPgo8c3ZnIHZlcnNpb249IjEuMSIgaWQ9IkxheWVyXzEiIHhtbG5zPSJodHRwOi8v
d3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkveGxp
bmsiIHg9IjBweCIgeT0iMHB4IgoJIHZpZXdCb3g9IjAgMCA2MS41IDYzLjciIHN0eWxlPSJlbmFibGUt
YmFja2dyb3VuZDpuZXcgMCAwIDYxLjUgNjMuNzsiIHhtbDpzcGFjZT0icHJlc2VydmUiPgo8c3R5bGUg
dHlwZT0idGV4dC9jc3MiPgoJLnN0MHtmaWxsOiNGRkU1RDQ7fQoJLnN0MXtmaWxsOiNGRkZCRjg7fQoJ
LnN0MntmaWxsOiNGMEU5RkY7c3Ryb2tlOiM4OTgwQTA7c3Ryb2tlLXdpZHRoOjEuMjU7c3Ryb2tlLWxp
bmVjYXA6cm91bmQ7c3Ryb2tlLW1pdGVybGltaXQ6MTA7fQoJLnN0M3tmaWxsOiNGMEU5RkY7c3Ryb2tl
OiM4OTgwQTA7c3Ryb2tlLWxpbmVjYXA6cm91bmQ7c3Ryb2tlLW1pdGVybGltaXQ6MTA7fQoJLnN0NHtm
aWxsOm5vbmU7c3Ryb2tlOiM4OTgwQTA7c3Ryb2tlLXdpZHRoOjEuMjU7c3Ryb2tlLWxpbmVjYXA6cm91
bmQ7c3Ryb2tlLWxpbmVqb2luOnJvdW5kO3N0cm9rZS1taXRlcmxpbWl0OjEwO30KCS5zdDV7ZmlsbDpu
b25lO3N0cm9rZTojODk4MEEwO3N0cm9rZS13aWR0aDowLjc1O3N0cm9rZS1saW5lY2FwOnJvdW5kO3N0
cm9rZS1saW5lam9pbjpyb3VuZDtzdHJva2UtbWl0ZXJsaW1pdDoxMDt9Cgkuc3Q2e2ZpbGw6I0YwRTlG
RjtzdHJva2U6Izg5ODBBMDtzdHJva2Utd2lkdGg6MS4yNTtzdHJva2UtbGluZWNhcDpyb3VuZDtzdHJv
a2UtbGluZWpvaW46cm91bmQ7c3Ryb2tlLW1pdGVybGltaXQ6MTA7fQo8L3N0eWxlPgo8cmVjdCBpZD0i
UmVjdGFuZ2xlLTQtQ29weS05IiB4PSI1Mi4xIiB5PSIzNS4zIiBjbGFzcz0ic3QwIiB3aWR0aD0iMi40
IiBoZWlnaHQ9IjMuNyIvPgo8Y2lyY2xlIGlkPSJYTUxJRF8xMTc2XyIgY2xhc3M9InN0MSIgY3g9IjMx
LjUiIGN5PSIzMi4zIiByPSIyNyIvPgo8cGF0aCBjbGFzcz0ic3QyIiBkPSJNMjQuOCwyMmgtNS40Yy0w
LjQsMC0wLjcsMC41LTAuNywxLjJsLTAuMSwxLjNjLTAuMSwwLjctMC4zLDEuMi0wLjcsMS4yaC0wLjFj
LTAuNSwwLTEsMC43LTEsMS42bC0wLjEsMSIvPgo8cGF0aCBjbGFzcz0ic3QyIiBkPSJNNDcuMiw0NC42
TDQ5LjksMjZjMC4xLTEtMS4xLTEuOS0zLjEtMS45SDMzLjciLz4KPHBvbHlsaW5lIGNsYXNzPSJzdDMi
IHBvaW50cz0iNDEsMjkuNSA0Ni4zLDI5LjUgNDUuNiwzNy4xICIvPgo8cG9seWxpbmUgY2xhc3M9InN0
MyIgcG9pbnRzPSIyMS4xLDI5LjggMjEuNCwyNyAyOS45LDI3ICIvPgo8bGluZSBpZD0iWE1MSURfMTE4
OV8iIGNsYXNzPSJzdDQiIHgxPSIzLjYiIHkxPSI0OC40IiB4Mj0iMTAuOCIgeTI9IjQ4LjQiLz4KPGxp
bmUgaWQ9IlhNTElEXzExODdfIiBjbGFzcz0ic3Q0IiB4MT0iMTMuNiIgeTE9IjQ4LjQiIHgyPSI1MC41
IiB5Mj0iNDguNCIvPgo8bGluZSBpZD0iWE1MSURfMTE4M18iIGNsYXNzPSJzdDUiIHgxPSIxMS44IiB5
MT0iNTEuNSIgeDI9IjE3LjciIHkyPSI1MS41Ii8+CjxsaW5lIGlkPSJYTUxJRF8xMTgyXyIgY2xhc3M9
InN0NSIgeDE9IjIxLjMiIHkxPSI1MS41IiB4Mj0iMjcuMiIgeTI9IjUxLjUiLz4KPGxpbmUgaWQ9IlhN
TElEXzExODFfIiBjbGFzcz0ic3Q1IiB4MT0iMzAuNiIgeTE9IjUxLjUiIHgyPSI0Mi44IiB5Mj0iNTEu
NSIvPgo8bGluZSBpZD0iWE1MSURfMTE4MF8iIGNsYXNzPSJzdDUiIHgxPSI0Ni40IiB5MT0iNTEuNSIg
eDI9IjUzLjMiIHkyPSI1MS41Ii8+CjxwYXRoIGNsYXNzPSJzdDYiIGQ9Ik0yMS4xLDMyLjJINDJjMC44
LDAsMS41LDAuNSwxLjcsMS4zbDQsMTQuOUgxNS40Yy0wLjYsMC0xLjEtMC40LTEuMy0xTDkuNCwzMGMt
MC4yLTAuOCwwLjQtMS43LDEuMy0xLjdoNwoJYzAuOCwwLDEuNSwwLjUsMS43LDEuM2wwLjQsMS42QzIw
LDMxLjgsMjAuNSwzMi4yLDIxLjEsMzIuMnoiLz4KPHBvbHlsaW5lIGlkPSJSZWN0YW5nbGUtNC1Db3B5
LTIiIGNsYXNzPSJzdDAiIHBvaW50cz0iNDcuNiwxNC4zIDQyLjIsMTQuMyA0Mi4yLDE5LjcgNDcuNiwx
OS43ICIvPgo8cG9seWxpbmUgaWQ9IlJlY3RhbmdsZS00LUNvcHktMTYiIGNsYXNzPSJzdDAiIHBvaW50
cz0iMTMuOSwyMy43IDEzLjksMjEuMyAxMS42LDIxLjMgMTEuNiwyMy43ICIvPgo8cmVjdCBpZD0iUmVj
dGFuZ2xlLTQtQ29weS0xOSIgeD0iMTgiIHk9IjE1LjgiIGNsYXNzPSJzdDAiIHdpZHRoPSIyLjQiIGhl
aWdodD0iMi40Ii8+CjxwYXRoIGNsYXNzPSJzdDYiIGQ9Ik0yMy45LDE3Yy0wLjEsMi42LDAuNSw1LjIs
MS45LDcuNGMzLjQsNC45LDEwLjEsNi4yLDE1LjcsMy4zYy00LjIsMC40LTguMy0xLjItMTAuNy00LjgK
CWMtMS4yLTEuNi0xLjctMy40LTEuOS01LjRsMy45LDAuNWwtNC41LTcuOGwtNy41LDYuNUwyMy45LDE3
eiIvPgo8bGluZSBpZD0iWE1MSURfMTE4OF8iIGNsYXNzPSJzdDQiIHgxPSI1NC41IiB5MT0iNDguNCIg
eDI9IjU4LjIiIHkyPSI0OC40Ii8+Cjwvc3ZnPgo=
`,
	},

	"/definitions/images/message_error.svg": {
		local:   "definitions/images/message_error.svg",
		size:    1079,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iMTYiIGhlaWdodD0iMTYiIHZpZXdCb3g9IjAgMCAxNiAxNiIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2Rk
IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0xNiA4QzE2IDEwLjEyMTcgMTUuMTU3MSAxMi4xNTY2IDEz
LjY1NjkgMTMuNjU2OUMxMi4xNTY2IDE1LjE1NzEgMTAuMTIxNyAxNiA4IDE2QzUuODc4MjcgMTYgMy44
NDM0NCAxNS4xNTcxIDIuMzQzMTUgMTMuNjU2OUMwLjg0Mjg1NSAxMi4xNTY2IDAgMTAuMTIxNyAwIDhD
MCA1Ljg3ODI3IDAuODQyODU1IDMuODQzNDQgMi4zNDMxNSAyLjM0MzE1QzMuODQzNDQgMC44NDI4NTUg
NS44NzgyNyAwIDggMEMxMC4xMjE3IDAgMTIuMTU2NiAwLjg0Mjg1NSAxMy42NTY5IDIuMzQzMTVDMTUu
MTU3MSAzLjg0MzQ0IDE2IDUuODc4MjcgMTYgOFY4Wk05IDEyQzkgMTIuMjY1MiA4Ljg5NDY0IDEyLjUx
OTYgOC43MDcxMSAxMi43MDcxQzguNTE5NTcgMTIuODk0NiA4LjI2NTIyIDEzIDggMTNDNy43MzQ3OCAx
MyA3LjQ4MDQzIDEyLjg5NDYgNy4yOTI4OSAxMi43MDcxQzcuMTA1MzYgMTIuNTE5NiA3IDEyLjI2NTIg
NyAxMkM3IDExLjczNDggNy4xMDUzNiAxMS40ODA0IDcuMjkyODkgMTEuMjkyOUM3LjQ4MDQzIDExLjEw
NTQgNy43MzQ3OCAxMSA4IDExQzguMjY1MjIgMTEgOC41MTk1NyAxMS4xMDU0IDguNzA3MTEgMTEuMjky
OUM4Ljg5NDY0IDExLjQ4MDQgOSAxMS43MzQ4IDkgMTJWMTJaTTggM0M3LjczNDc4IDMgNy40ODA0MyAz
LjEwNTM2IDcuMjkyODkgMy4yOTI4OUM3LjEwNTM2IDMuNDgwNDMgNyAzLjczNDc4IDcgNFY4QzcgOC4y
NjUyMiA3LjEwNTM2IDguNTE5NTcgNy4yOTI4OSA4LjcwNzExQzcuNDgwNDMgOC44OTQ2NCA3LjczNDc4
IDkgOCA5QzguMjY1MjIgOSA4LjUxOTU3IDguODk0NjQgOC43MDcxMSA4LjcwNzExQzguODk0NjQgOC41
MTk1NyA5IDguMjY1MjIgOSA4VjRDOSAzLjczNDc4IDguODk0NjQgMy40ODA0MyA4LjcwNzExIDMuMjky
ODlDOC41MTk1NyAzLjEwNTM2IDguMjY1MjIgMyA4IDNaIiBmaWxsPSIjRkVEQ0RDIi8+Cjwvc3ZnPgo=
`,
	},

	"/definitions/images/message_info.svg": {
		local:   "definitions/images/message_info.svg",
		size:    714,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iMTYiIGhlaWdodD0iMTYiIHZpZXdCb3g9IjAgMCAxNiAxNiIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2Rk
IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0xNC42OTQ3IDMuMjkyNzlDMTQuODgyMiAzLjQ4MDMxIDE0
Ljk4NzUgMy43MzQ2MiAxNC45ODc1IDMuOTk5NzlDMTQuOTg3NSA0LjI2NDk1IDE0Ljg4MjIgNC41MTky
NiAxNC42OTQ3IDQuNzA2NzlMNi42OTQ3MSAxMi43MDY4QzYuNTA3MTggMTIuODk0MyA2LjI1Mjg4IDEy
Ljk5OTYgNS45ODc3MSAxMi45OTk2QzUuNzIyNTUgMTIuOTk5NiA1LjQ2ODI0IDEyLjg5NDMgNS4yODA3
MSAxMi43MDY4TDEuMjgwNzEgOC43MDY3OUMxLjA5ODU1IDguNTE4MTggMC45OTc3NTkgOC4yNjU1OCAx
LjAwMDA0IDguMDAzMzlDMS4wMDIzMiA3Ljc0MTE5IDEuMTA3NDkgNy40OTAzOCAxLjI5Mjg5IDcuMzA0
OTdDMS40NzgzIDcuMTE5NTYgMS43MjkxMSA3LjAxNDM5IDEuOTkxMzEgNy4wMTIxMUMyLjI1MzUxIDcu
MDA5ODMgMi41MDYxMSA3LjExMDYzIDIuNjk0NzEgNy4yOTI3OUw1Ljk4NzcxIDEwLjU4NThMMTMuMjgw
NyAzLjI5Mjc5QzEzLjQ2ODIgMy4xMDUzMiAxMy43MjI1IDMgMTMuOTg3NyAzQzE0LjI1MjkgMyAxNC41
MDcyIDMuMTA1MzIgMTQuNjk0NyAzLjI5Mjc5WiIgZmlsbD0iI0E1RTNCQyIvPgo8L3N2Zz4K
`,
	},

	"/definitions/images/message_other.svg": {
		local:   "definitions/images/message_other.svg",
		size:    1223,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iMTYiIGhlaWdodD0iMTYiIHZpZXdCb3g9IjAgMCAxNiAxNiIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2Rk
IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0xNiA4QzE2IDEwLjEyMTcgMTUuMTU3MSAxMi4xNTY2IDEz
LjY1NjkgMTMuNjU2OUMxMi4xNTY2IDE1LjE1NzEgMTAuMTIxNyAxNiA4IDE2QzUuODc4MjcgMTYgMy44
NDM0NCAxNS4xNTcxIDIuMzQzMTUgMTMuNjU2OUMwLjg0Mjg1NSAxMi4xNTY2IDAgMTAuMTIxNyAwIDhD
MCA1Ljg3ODI3IDAuODQyODU1IDMuODQzNDQgMi4zNDMxNSAyLjM0MzE1QzMuODQzNDQgMC44NDI4NTUg
NS44NzgyNyAwIDggMEMxMC4xMjE3IDAgMTIuMTU2NiAwLjg0Mjg1NSAxMy42NTY5IDIuMzQzMTVDMTUu
MTU3MSAzLjg0MzQ0IDE2IDUuODc4MjcgMTYgOFpNOSA0QzkgNC4yNjUyMiA4Ljg5NDY0IDQuNTE5NTcg
OC43MDcxMSA0LjcwNzExQzguNTE5NTcgNC44OTQ2NCA4LjI2NTIyIDUgOCA1QzcuNzM0NzggNSA3LjQ4
MDQzIDQuODk0NjQgNy4yOTI4OSA0LjcwNzExQzcuMTA1MzYgNC41MTk1NyA3IDQuMjY1MjIgNyA0Qzcg
My43MzQ3OCA3LjEwNTM2IDMuNDgwNDMgNy4yOTI4OSAzLjI5Mjg5QzcuNDgwNDMgMy4xMDUzNiA3Ljcz
NDc4IDMgOCAzQzguMjY1MjIgMyA4LjUxOTU3IDMuMTA1MzYgOC43MDcxMSAzLjI5Mjg5QzguODk0NjQg
My40ODA0MyA5IDMuNzM0NzggOSA0Wk03IDdDNi43MzQ3OCA3IDYuNDgwNDMgNy4xMDUzNiA2LjI5Mjg5
IDcuMjkyODlDNi4xMDUzNiA3LjQ4MDQzIDYgNy43MzQ3OCA2IDhDNiA4LjI2NTIyIDYuMTA1MzYgOC41
MTk1NyA2LjI5Mjg5IDguNzA3MTFDNi40ODA0MyA4Ljg5NDY0IDYuNzM0NzggOSA3IDlWMTJDNyAxMi4y
NjUyIDcuMTA1MzYgMTIuNTE5NiA3LjI5Mjg5IDEyLjcwNzFDNy40ODA0MyAxMi44OTQ2IDcuNzM0Nzgg
MTMgOCAxM0g5QzkuMjY1MjIgMTMgOS41MTk1NyAxMi44OTQ2IDkuNzA3MTEgMTIuNzA3MUM5Ljg5NDY0
IDEyLjUxOTYgMTAgMTIuMjY1MiAxMCAxMkMxMCAxMS43MzQ4IDkuODk0NjQgMTEuNDgwNCA5LjcwNzEx
IDExLjI5MjlDOS41MTk1NyAxMS4xMDU0IDkuMjY1MjIgMTEgOSAxMVY4QzkgNy43MzQ3OCA4Ljg5NDY0
IDcuNDgwNDMgOC43MDcxMSA3LjI5Mjg5QzguNTE5NTcgNy4xMDUzNiA4LjI2NTIyIDcgOCA3SDdaIiBm
aWxsPSIjRDFFOEY0Ii8+Cjwvc3ZnPgo=
`,
	},

	"/definitions/images/message_question.svg": {
		local:   "definitions/images/message_question.svg",
		size:    1873,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iMTYiIGhlaWdodD0iMTYiIHZpZXdCb3g9IjAgMCAxNiAxNiIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2Rk
IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0xNiA4QzE2IDEwLjEyMTcgMTUuMTU3MSAxMi4xNTY2IDEz
LjY1NjkgMTMuNjU2OUMxMi4xNTY2IDE1LjE1NzEgMTAuMTIxNyAxNiA4IDE2QzUuODc4MjcgMTYgMy44
NDM0NCAxNS4xNTcxIDIuMzQzMTUgMTMuNjU2OUMwLjg0Mjg1NSAxMi4xNTY2IDAgMTAuMTIxNyAwIDhD
MCA1Ljg3ODI3IDAuODQyODU1IDMuODQzNDQgMi4zNDMxNSAyLjM0MzE1QzMuODQzNDQgMC44NDI4NTUg
NS44NzgyNyAwIDggMEMxMC4xMjE3IDAgMTIuMTU2NiAwLjg0Mjg1NSAxMy42NTY5IDIuMzQzMTVDMTUu
MTU3MSAzLjg0MzQ0IDE2IDUuODc4MjcgMTYgOFY4Wk04IDVDNy44MjQzIDQuOTk5ODMgNy42NTE2NSA1
LjA0NTk1IDcuNDk5NDUgNS4xMzM3M0M3LjM0NzI0IDUuMjIxNTEgNy4yMjA4NSA1LjM0Nzg0IDcuMTMz
IDUuNUM3LjA2OTU3IDUuNjE3ODggNi45ODMxMSA1LjcyMTgyIDYuODc4NzYgNS44MDU2NkM2Ljc3NDQx
IDUuODg5NSA2LjY1NDI5IDUuOTUxNTQgNi41MjU1MiA1Ljk4ODFDNi4zOTY3NSA2LjAyNDY2IDYuMjYx
OTQgNi4wMzQ5OSA2LjEyOTExIDYuMDE4NDlDNS45OTYyNyA2LjAwMTk4IDUuODY4MDkgNS45NTg5NyA1
Ljc1MjE4IDUuODkyMDFDNS42MzYyOCA1LjgyNTA1IDUuNTM0OTkgNS43MzU1IDUuNDU0MzMgNS42Mjg2
N0M1LjM3MzY3IDUuNTIxODQgNS4zMTUyOSA1LjM5OTkgNS4yODI2MyA1LjI3MDA4QzUuMjQ5OTcgNS4x
NDAyNyA1LjI0MzcxIDUuMDA1MjIgNS4yNjQyMSA0Ljg3Mjk0QzUuMjg0NzIgNC43NDA2NSA1LjMzMTU3
IDQuNjEzODQgNS40MDIgNC41QzUuNzMyMjIgMy45MjgxMSA2LjI0MTkxIDMuNDgxMTYgNi44NTIwMyAz
LjIyODQ2QzcuNDYyMTQgMi45NzU3NiA4LjEzODU5IDIuOTMxNDQgOC43NzY0NyAzLjEwMjM2QzkuNDE0
MzQgMy4yNzMyOCA5Ljk3OCAzLjY0OTg5IDEwLjM4IDQuMTczOEMxMC43ODIgNC42OTc3IDExIDUuMzM5
NjIgMTEgNkMxMS4wMDAyIDYuNjIwNjIgMTAuODA3OSA3LjIyNjAzIDEwLjQ0OTggNy43MzI4NUMxMC4w
OTE2IDguMjM5NjggOS41ODUwNyA4LjYyMjk5IDkgOC44M1Y5QzkgOS4yNjUyMiA4Ljg5NDY0IDkuNTE5
NTcgOC43MDcxMSA5LjcwNzExQzguNTE5NTcgOS44OTQ2NCA4LjI2NTIyIDEwIDggMTBDNy43MzQ3OCAx
MCA3LjQ4MDQzIDkuODk0NjQgNy4yOTI4OSA5LjcwNzExQzcuMTA1MzYgOS41MTk1NyA3IDkuMjY1MjIg
NyA5VjhDNyA3LjczNDc4IDcuMTA1MzYgNy40ODA0MyA3LjI5Mjg5IDcuMjkyODlDNy40ODA0MyA3LjEw
NTM2IDcuNzM0NzggNyA4IDdDOC4yNjUyMiA3IDguNTE5NTcgNi44OTQ2NCA4LjcwNzExIDYuNzA3MTFD
OC44OTQ2NCA2LjUxOTU3IDkgNi4yNjUyMiA5IDZDOSA1LjczNDc4IDguODk0NjQgNS40ODA0MyA4Ljcw
NzExIDUuMjkyODlDOC41MTk1NyA1LjEwNTM2IDguMjY1MjIgNSA4IDVaTTggMTNDOC4yNjUyMiAxMyA4
LjUxOTU3IDEyLjg5NDYgOC43MDcxMSAxMi43MDcxQzguODk0NjQgMTIuNTE5NiA5IDEyLjI2NTIgOSAx
MkM5IDExLjczNDggOC44OTQ2NCAxMS40ODA0IDguNzA3MTEgMTEuMjkyOUM4LjUxOTU3IDExLjEwNTQg
OC4yNjUyMiAxMSA4IDExQzcuNzM0NzggMTEgNy40ODA0MyAxMS4xMDU0IDcuMjkyODkgMTEuMjkyOUM3
LjEwNTM2IDExLjQ4MDQgNyAxMS43MzQ4IDcgMTJDNyAxMi4yNjUyIDcuMTA1MzYgMTIuNTE5NiA3LjI5
Mjg5IDEyLjcwNzFDNy40ODA0MyAxMi44OTQ2IDcuNzM0NzggMTMgOCAxM1YxM1oiIGZpbGw9IiNGOUQx
QjYiLz4KPC9zdmc+Cg==
`,
	},

	"/definitions/images/message_warning.svg": {
		local:   "definitions/images/message_warning.svg",
		size:    1155,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iMTYiIGhlaWdodD0iMTYiIHZpZXdCb3g9IjAgMCAxNiAxNiIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2Rk
IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik02LjE2MTExIDIuMDI1ODZDNi45NjgxOSAwLjY1ODA0NiA5
LjAzMjg1IDAuNjU4MDQ2IDkuODM4ODggMi4wMjU4NkwxNS43MjU4IDEyLjAwMjlDMTYuNTE3MSAxMy4z
NDQ1IDE1LjUwMTEgMTUgMTMuODg4IDE1SDIuMTEzMDNDMC40OTg4NjggMTUgLTAuNTE3MTA3IDEzLjM0
NDUgMC4yNzQxNTEgMTIuMDAyOUw2LjE2MTExIDIuMDI1ODZaTTkuMDU1IDExLjk4MzhDOS4wNTUgMTIu
MjUwNSA4Ljk0Mzg1IDEyLjUwNjMgOC43NDYgMTIuNjk0OUM4LjU0ODE1IDEyLjg4MzUgOC4yNzk4IDEy
Ljk4OTUgNy45OTk5OSAxMi45ODk1QzcuNzIwMTkgMTIuOTg5NSA3LjQ1MTg0IDEyLjg4MzUgNy4yNTM5
OSAxMi42OTQ5QzcuMDU2MTQgMTIuNTA2MyA2Ljk0NDk4IDEyLjI1MDUgNi45NDQ5OCAxMS45ODM4QzYu
OTQ0OTggMTEuNzE3IDcuMDU2MTQgMTEuNDYxMiA3LjI1Mzk5IDExLjI3MjZDNy40NTE4NCAxMS4wODQg
Ny43MjAxOSAxMC45NzggNy45OTk5OSAxMC45NzhDOC4yNzk4IDEwLjk3OCA4LjU0ODE1IDExLjA4NCA4
Ljc0NiAxMS4yNzI2QzguOTQzODUgMTEuNDYxMiA5LjA1NSAxMS43MTcgOS4wNTUgMTEuOTgzOFYxMS45
ODM4Wk03Ljk5OTk5IDMuOTM3NzlDNy43MjAxOSAzLjkzNzc5IDcuNDUxODQgNC4wNDM3NSA3LjI1Mzk5
IDQuMjMyMzZDNy4wNTYxNCA0LjQyMDk4IDYuOTQ0OTggNC42NzY3OSA2Ljk0NDk4IDQuOTQzNTNWNy45
NjA3OEM2Ljk0NDk4IDguMjI3NTIgNy4wNTYxNCA4LjQ4MzMzIDcuMjUzOTkgOC42NzE5NUM3LjQ1MTg0
IDguODYwNTYgNy43MjAxOSA4Ljk2NjUyIDcuOTk5OTkgOC45NjY1MkM4LjI3OTggOC45NjY1MiA4LjU0
ODE1IDguODYwNTYgOC43NDYgOC42NzE5NUM4Ljk0Mzg1IDguNDgzMzMgOS4wNTUgOC4yMjc1MiA5LjA1
NSA3Ljk2MDc4VjQuOTQzNTNDOS4wNTUgNC42NzY3OSA4Ljk0Mzg1IDQuNDIwOTggOC43NDYgNC4yMzIz
NkM4LjU0ODE1IDQuMDQzNzUgOC4yNzk4IDMuOTM3NzkgNy45OTk5OSAzLjkzNzc5WiIgZmlsbD0iI0ZG
RjVDMSIvPgo8L3N2Zz4K
`,
	},

	"/definitions/images/not_anonymous.svg": {
		local:   "definitions/images/not_anonymous.svg",
		size:    631,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iNjYiIGhlaWdodD0iNDkiIHZpZXdCb3g9IjAgMCA2NiA0OSIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZD0iTTEgMjQuMjcyN0MxIDI0
LjI3MjcgMTIuNjM2NCAxIDMzIDFDNTMuMzYzNiAxIDY1IDI0LjI3MjcgNjUgMjQuMjcyN0M2NSAyNC4y
NzI3IDUzLjM2MzYgNDcuNTQ1NSAzMyA0Ny41NDU1QzEyLjYzNjQgNDcuNTQ1NSAxIDI0LjI3MjcgMSAy
NC4yNzI3WiIgZmlsbD0iI0ZFRjNDNyIgc3Ryb2tlPSIjRDk3NzA2IiBzdHJva2Utd2lkdGg9IjIiIHN0
cm9rZS1saW5lY2FwPSJyb3VuZCIgc3Ryb2tlLWxpbmVqb2luPSJyb3VuZCIvPgo8cGF0aCBkPSJNMzMg
MzNDMzcuODE5OSAzMyA0MS43MjczIDI5LjA5MjYgNDEuNzI3MyAyNC4yNzI3QzQxLjcyNzMgMTkuNDUy
NyAzNy44MTk5IDE1LjU0NTQgMzMgMTUuNTQ1NEMyOC4xOCAxNS41NDU0IDI0LjI3MjcgMTkuNDUyNyAy
NC4yNzI3IDI0LjI3MjdDMjQuMjcyNyAyOS4wOTI2IDI4LjE4IDMzIDMzIDMzWiIgZmlsbD0id2hpdGUi
IHN0cm9rZT0iI0Q5NzcwNiIgc3Ryb2tlLXdpZHRoPSIyIiBzdHJva2UtbGluZWNhcD0icm91bmQiIHN0
cm9rZS1saW5lam9pbj0icm91bmQiLz4KPC9zdmc+Cg==
`,
	},

	"/definitions/images/not_encrypted.svg": {
		local:   "definitions/images/not_encrypted.svg",
		size:    778,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iNjYiIGhlaWdodD0iNjYiIHZpZXdCb3g9IjAgMCA2NiA2NiIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZD0iTTU1LjM3MDUgMzguODE3
OEM1NS45NTU1IDM2LjkzMzMgNTYuMjU5MyAzNC45NzI4IDU2LjI3MjMgMzIuOTk5NlYxMi42MzZMMzIu
OTk5NiAzLjkwODY5TDIzLjgwNjkgNy4zNDE0MiIgc3Ryb2tlPSIjRDk3NzA2IiBzdHJva2Utd2lkdGg9
IjIiIHN0cm9rZS1saW5lY2FwPSJyb3VuZCIgc3Ryb2tlLWxpbmVqb2luPSJyb3VuZCIvPgo8cGF0aCBk
PSJNMTEuODUwOSAxMS44NTExTDkuNzI3MjkgMTIuNjM2NVYzMy4wMDAyQzkuNzI3MjkgNTAuNDU0NyAz
MyA2Mi4wOTExIDMzIDYyLjA5MTFDMzkuMTU4NiA1OC44NDE0IDQ0LjY5MzcgNTQuNTI3NyA0OS4zNDkx
IDQ5LjM0OTMiIGZpbGw9IiNGRUYzQzciLz4KPHBhdGggZD0iTTExLjg1MDkgMTEuODUxMUw5LjcyNzI5
IDEyLjYzNjVWMzMuMDAwMkM5LjcyNzI5IDUwLjQ1NDcgMzMgNjIuMDkxMSAzMyA2Mi4wOTExQzM5LjE1
ODYgNTguODQxNCA0NC42OTM3IDU0LjUyNzcgNDkuMzQ5MSA0OS4zNDkzIiBzdHJva2U9IiNEOTc3MDYi
IHN0cm9rZS13aWR0aD0iMiIgc3Ryb2tlLWxpbmVjYXA9InJvdW5kIiBzdHJva2UtbGluZWpvaW49InJv
dW5kIi8+CjxwYXRoIGQ9Ik0xIDFMNjUgNjUiIHN0cm9rZT0iI0Q5NzcwNiIgc3Ryb2tlLXdpZHRoPSIy
IiBzdHJva2UtbGluZWNhcD0icm91bmQiIHN0cm9rZS1saW5lam9pbj0icm91bmQiLz4KPC9zdmc+Cg==
`,
	},

	"/definitions/images/padlock.svg": {
		local:   "definitions/images/padlock.svg",
		size:    2835,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiIHN0YW5kYWxvbmU9Im5vIj8+Cjxzdmcg
d2lkdGg9IjExcHgiIGhlaWdodD0iMTNweCIgdmlld0JveD0iMCAwIDExIDEzIiB2ZXJzaW9uPSIxLjEi
IHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cu
dzMub3JnLzE5OTkveGxpbmsiPgogICAgPCEtLSBHZW5lcmF0b3I6IFNrZXRjaCA0MS4yICgzNTM5Nykg
LSBodHRwOi8vd3d3LmJvaGVtaWFuY29kaW5nLmNvbS9za2V0Y2ggLS0+CiAgICA8dGl0bGU+R3JvdXA8
L3RpdGxlPgogICAgPGRlc2M+Q3JlYXRlZCB3aXRoIFNrZXRjaC48L2Rlc2M+CiAgICA8ZGVmcz48L2Rl
ZnM+CiAgICA8ZyBpZD0iU01QLWZsb3dzIiBzdHJva2U9Im5vbmUiIHN0cm9rZS13aWR0aD0iMSIgZmls
bD0ibm9uZSIgZmlsbC1ydWxlPSJldmVub2RkIj4KICAgICAgICA8ZyBpZD0iQm9iXzQiIHRyYW5zZm9y
bT0idHJhbnNsYXRlKC04NDMuMDAwMDAwLCAtMTY0LjAwMDAwMCkiPgogICAgICAgICAgICA8ZyBpZD0i
R3JvdXAiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDg0My4wMDAwMDAsIDE2NC4wMDAwMDApIj4KICAgICAg
ICAgICAgICAgIDxwYXRoIGQ9Ik0zLjA1NTU1NTU2LDUuOTA5MDkwOTEgTDcuOTQ0NDQ0NDQsNS45MDkw
OTA5MSBMNy45NDQ0NDQ0NCw0LjEzNjM2MzY0IEM3Ljk0NDQ0NDQ0LDMuNDgzODk4MjUgNy43MDU3MzE1
NSwyLjkyNjg0ODkgNy4yMjgyOTg2MSwyLjQ2NTE5ODg2IEM2Ljc1MDg2NTY3LDIuMDAzNTQ4ODMgNi4x
NzQ3NzE4OSwxLjc3MjcyNzI3IDUuNSwxLjc3MjcyNzI3IEM0LjgyNTIyODExLDEuNzcyNzI3MjcgNC4y
NDkxMzQzMywyLjAwMzU0ODgzIDMuNzcxNzAxMzksMi40NjUxOTg4NiBDMy4yOTQyNjg0NSwyLjkyNjg0
ODkgMy4wNTU1NTU1NiwzLjQ4Mzg5ODI1IDMuMDU1NTU1NTYsNC4xMzYzNjM2NCBMMy4wNTU1NTU1Niw1
LjkwOTA5MDkxIFogTTExLDYuNzk1NDU0NTUgTDExLDEyLjExMzYzNjQgQzExLDEyLjM1OTg0OTcgMTAu
OTEwODgwNSwxMi41NjkxMjc5IDEwLjczMjYzODksMTIuNzQxNDc3MyBDMTAuNTU0Mzk3MywxMi45MTM4
MjY2IDEwLjMzNzk2NDIsMTMgMTAuMDgzMzMzMywxMyBMMC45MTY2NjY2NjcsMTMgQzAuNjYyMDM1NzY0
LDEzIDAuNDQ1NjAyNzQzLDEyLjkxMzgyNjYgMC4yNjczNjExMTEsMTIuNzQxNDc3MyBDMC4wODkxMTk0
NzkyLDEyLjU2OTEyNzkgMCwxMi4zNTk4NDk3IDAsMTIuMTEzNjM2NCBMMCw2Ljc5NTQ1NDU1IEMwLDYu
NTQ5MjQxMTkgMC4wODkxMTk0NzkyLDYuMzM5OTYyOTggMC4yNjczNjExMTEsNi4xNjc2MTM2NCBDMC40
NDU2MDI3NDMsNS45OTUyNjQyOSAwLjY2MjAzNTc2NCw1LjkwOTA5MDkxIDAuOTE2NjY2NjY3LDUuOTA5
MDkwOTEgTDEuMjIyMjIyMjIsNS45MDkwOTA5MSBMMS4yMjIyMjIyMiw0LjEzNjM2MzY0IEMxLjIyMjIy
MjIyLDMuMDAzNzgyMjIgMS42NDIzNTY5MSwyLjAzMTI1NDA2IDIuNDgyNjM4ODksMS4yMTg3NSBDMy4z
MjI5MjA4NywwLjQwNjI0NTkzNyA0LjMyODY5Nzg1LDAgNS41LDAgQzYuNjcxMzAyMTUsMCA3LjY3NzA3
OTEzLDAuNDA2MjQ1OTM3IDguNTE3MzYxMTEsMS4yMTg3NSBDOS4zNTc2NDMwOSwyLjAzMTI1NDA2IDku
Nzc3Nzc3NzgsMy4wMDM3ODIyMiA5Ljc3Nzc3Nzc4LDQuMTM2MzYzNjQgTDkuNzc3Nzc3NzgsNS45MDkw
OTA5MSBMMTAuMDgzMzMzMyw1LjkwOTA5MDkxIEMxMC4zMzc5NjQyLDUuOTA5MDkwOTEgMTAuNTU0Mzk3
Myw1Ljk5NTI2NDI5IDEwLjczMjYzODksNi4xNjc2MTM2NCBDMTAuOTEwODgwNSw2LjMzOTk2Mjk4IDEx
LDYuNTQ5MjQxMTkgMTEsNi43OTU0NTQ1NSBMMTEsNi43OTU0NTQ1NSBaIiBpZD0i74CjLWNvcHktMiIg
ZmlsbD0iIzdFRDMyMSI+PC9wYXRoPgogICAgICAgICAgICAgICAgPHBhdGggZD0iTTguMzIzMjMxNzEs
OC4wMDQ5NDQ4NyBMNS4yMjA1NjI4MywxMS4wMDUwNDYyIEM1LjE1OTU3NjQxLDExLjA2NDAxNjUgNS4w
ODcxNTYxMSwxMS4wOTM1MDEzIDUuMDAzMjk5NzgsMTEuMDkzNTAxMyBDNC45MTk0NDM0NCwxMS4wOTM1
MDEzIDQuODQ3MDIzMTUsMTEuMDY0MDE2NSA0Ljc4NjAzNjcyLDExLjAwNTA0NjIgTDMuMTQ3MDM0NzQs
OS40MjAyMjYwOCBDMy4wODYwNDgzMSw5LjM2MTI1NTczIDMuMDU1NTU1NTYsOS4yOTEyMjk1IDMuMDU1
NTU1NTYsOS4yMTAxNDUyNyBDMy4wNTU1NTU1Niw5LjEyOTA2MTA1IDMuMDg2MDQ4MzEsOS4wNTkwMzQ4
MiAzLjE0NzAzNDc0LDkuMDAwMDY0NDcgTDMuNTY2MzE0MzEsOC41OTQ2NDUzNyBDMy42MjczMDA3NCw4
LjUzNTY3NTAzIDMuNjk5NzIxMDMsOC41MDYxOTAzIDMuNzgzNTc3MzcsOC41MDYxOTAzIEMzLjg2NzQz
MzcsOC41MDYxOTAzIDMuOTM5ODU0LDguNTM1Njc1MDMgNC4wMDA4NDA0Miw4LjU5NDY0NTM3IEw1LjAw
MzI5OTc4LDkuNTYzOTY1NTggTDcuNDY5NDI2MDIsNy4xNzkzNjQxNyBDNy41MzA0MTI0NSw3LjEyMDM5
MzgyIDcuNjAyODMyNzQsNy4wOTA5MDkwOSA3LjY4NjY4OTA4LDcuMDkwOTA5MDkgQzcuNzcwNTQ1NDEs
Ny4wOTA5MDkwOSA3Ljg0Mjk2NTcsNy4xMjAzOTM4MiA3LjkwMzk1MjEzLDcuMTc5MzY0MTcgTDguMzIz
MjMxNzEsNy41ODQ3ODMyNiBDOC4zODQyMTgxMyw3LjY0Mzc1MzYxIDguNDE0NzEwODksNy43MTM3Nzk4
NCA4LjQxNDcxMDg5LDcuNzk0ODY0MDcgQzguNDE0NzEwODksNy44NzU5NDgyOSA4LjM4NDIxODEzLDcu
OTQ1OTc0NTMgOC4zMjMyMzE3MSw4LjAwNDk0NDg3IEw4LjMyMzIzMTcxLDguMDA0OTQ0ODcgWiIgaWQ9
IlBhdGgiIGZpbGw9IiNGRkZGRkYiPjwvcGF0aD4KICAgICAgICAgICAgPC9nPgogICAgICAgIDwvZz4K
ICAgIDwvZz4KPC9zdmc+
`,
	},

	"/definitions/images/partially_anonymous.svg": {
		local:   "definitions/images/partially_anonymous.svg",
		size:    1761,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iNjYiIGhlaWdodD0iNjYiIHZpZXdCb3g9IjAgMCA2NiA2NiIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZD0iTTUwLjI4IDUwLjI3OTlD
NDUuMzA3MiA1NC4wNzA0IDM5LjI1MTkgNTYuMTcwNCAzMyA1Ni4yNzI2QzEyLjYzNjQgNTYuMjcyNiAx
IDMyLjk5OTkgMSAzMi45OTk5QzQuNjE4NTkgMjYuMjU2MyA5LjYzNzQ4IDIwLjM2NDUgMTUuNzIgMTUu
NzE5OUw1MC4yOCA1MC4yNzk5Wk0yNi44OTA5IDEwLjQyNTNDMjguODkzMyA5Ljk1NjU5IDMwLjk0MzUg
OS43MjIyOSAzMyA5LjcyNzEyQzUzLjM2MzYgOS43MjcxMiA2NSAzMi45OTk5IDY1IDMyLjk5OTlDNjMu
MjM0MSAzNi4zMDM0IDYxLjEyODIgMzkuNDEzNiA1OC43MTY0IDQyLjI3OTlMMjYuODkwOSAxMC40MjUz
Wk0zOS4xNjczIDM5LjE2NzFDMzguMzY4MyA0MC4wMjQ2IDM3LjQwNDggNDAuNzEyMyAzNi4zMzQzIDQx
LjE4OTNDMzUuMjYzNyA0MS42NjYzIDM0LjEwODEgNDEuOTIyOCAzMi45MzYzIDQxLjk0MzVDMzEuNzY0
NCA0MS45NjQxIDMwLjYwMDUgNDEuNzQ4NiAyOS41MTM4IDQxLjMwOTZDMjguNDI3MSA0MC44NzA3IDI3
LjQzOTkgNDAuMjE3NCAyNi42MTEyIDM5LjM4ODdDMjUuNzgyNSAzOC41NTk5IDI1LjEyOTEgMzcuNTcy
OCAyNC42OTAyIDM2LjQ4NjFDMjQuMjUxMyAzNS4zOTk0IDI0LjAzNTcgMzQuMjM1NCAyNC4wNTY0IDMz
LjA2MzZDMjQuMDc3MSAzMS44OTE4IDI0LjMzMzYgMzAuNzM2MSAyNC44MTA2IDI5LjY2NTZDMjUuMjg3
NSAyOC41OTUxIDI1Ljk3NTMgMjcuNjMxNiAyNi44MzI3IDI2LjgzMjYiIGZpbGw9IiNGRUYzQzciLz4K
PHBhdGggZD0iTTM5LjE2NzMgMzkuMTY3MUMzOC4zNjgzIDQwLjAyNDYgMzcuNDA0OCA0MC43MTIzIDM2
LjMzNDMgNDEuMTg5M0MzNS4yNjM3IDQxLjY2NjMgMzQuMTA4MSA0MS45MjI4IDMyLjkzNjMgNDEuOTQz
NUMzMS43NjQ0IDQxLjk2NDEgMzAuNjAwNSA0MS43NDg2IDI5LjUxMzggNDEuMzA5NkMyOC40MjcxIDQw
Ljg3MDcgMjcuNDM5OSA0MC4yMTc0IDI2LjYxMTIgMzkuMzg4N0MyNS43ODI1IDM4LjU1OTkgMjUuMTI5
MSAzNy41NzI4IDI0LjY5MDIgMzYuNDg2MUMyNC4yNTEzIDM1LjM5OTQgMjQuMDM1NyAzNC4yMzU0IDI0
LjA1NjQgMzMuMDYzNkMyNC4wNzcxIDMxLjg5MTggMjQuMzMzNiAzMC43MzYxIDI0LjgxMDYgMjkuNjY1
NkMyNS4yODc1IDI4LjU5NTEgMjUuOTc1MyAyNy42MzE2IDI2LjgzMjcgMjYuODMyNk01MC4yOCA1MC4y
Nzk5QzQ1LjMwNzIgNTQuMDcwNCAzOS4yNTE5IDU2LjE3MDQgMzMgNTYuMjcyNkMxMi42MzY0IDU2LjI3
MjYgMSAzMi45OTk5IDEgMzIuOTk5OUM0LjYxODU5IDI2LjI1NjMgOS42Mzc0OCAyMC4zNjQ1IDE1Ljcy
IDE1LjcxOTlMNTAuMjggNTAuMjc5OVpNMjYuODkwOSAxMC40MjUzQzI4Ljg5MzMgOS45NTY1OSAzMC45
NDM1IDkuNzIyMjkgMzMgOS43MjcxMkM1My4zNjM2IDkuNzI3MTIgNjUgMzIuOTk5OSA2NSAzMi45OTk5
QzYzLjIzNDEgMzYuMzAzNCA2MS4xMjgyIDM5LjQxMzYgNTguNzE2NCA0Mi4yNzk5TDI2Ljg5MDkgMTAu
NDI1M1oiIHN0cm9rZT0iI0Q5NzcwNiIgc3Ryb2tlLXdpZHRoPSIyIiBzdHJva2UtbGluZWNhcD0icm91
bmQiIHN0cm9rZS1saW5lam9pbj0icm91bmQiLz4KPHBhdGggZD0iTTEgMUw2NSA2NSIgc3Ryb2tlPSIj
RDk3NzA2IiBzdHJva2Utd2lkdGg9IjIiIHN0cm9rZS1saW5lY2FwPSJyb3VuZCIgc3Ryb2tlLWxpbmVq
b2luPSJyb3VuZCIvPgo8L3N2Zz4K
`,
	},

	"/definitions/images/publicly_logged.svg": {
		local:   "definitions/images/publicly_logged.svg",
		size:    917,
		modtime: 1489449600,
		compressed: `
PHN2ZyB3aWR0aD0iNTQiIGhlaWdodD0iNjYiIHZpZXdCb3g9IjAgMCA1NCA2NiIgZmlsbD0ibm9uZSIg
eG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZD0iTTMzIDFINy40QzUuNzAy
NjEgMSA0LjA3NDc1IDEuNjc0MjggMi44NzQ1MiAyLjg3NDUyQzEuNjc0MjggNC4wNzQ3NSAxIDUuNzAy
NjEgMSA3LjRWNTguNkMxIDYwLjI5NzQgMS42NzQyOCA2MS45MjUzIDIuODc0NTIgNjMuMTI1NUM0LjA3
NDc1IDY0LjMyNTcgNS43MDI2MSA2NSA3LjQgNjVINDUuOEM0Ny40OTc0IDY1IDQ5LjEyNTMgNjQuMzI1
NyA1MC4zMjU1IDYzLjEyNTVDNTEuNTI1NyA2MS45MjUzIDUyLjIgNjAuMjk3NCA1Mi4yIDU4LjZWMjAu
MkwzMyAxWiIgZmlsbD0iI0ZFRjNDNyIgc3Ryb2tlPSIjRDk3NzA2IiBzdHJva2Utd2lkdGg9IjIiIHN0
cm9rZS1saW5lY2FwPSJyb3VuZCIgc3Ryb2tlLWxpbmVqb2luPSJyb3VuZCIvPgo8cGF0aCBkPSJNMzMg
MVYyMC4ySDUyLjIiIHN0cm9rZT0iI0Q5NzcwNiIgc3Ryb2tlLXdpZHRoPSIyIiBzdHJva2UtbGluZWNh
cD0icm91bmQiIHN0cm9rZS1saW5lam9pbj0icm91bmQiLz4KPHBhdGggZD0iTTM5LjQgMzYuMjAwMkgx
My44IiBzdHJva2U9IiNEOTc3MDYiIHN0cm9rZS13aWR0aD0iMiIgc3Ryb2tlLWxpbmVjYXA9InJvdW5k
IiBzdHJva2UtbGluZWpvaW49InJvdW5kIi8+CjxwYXRoIGQ9Ik0zOS40IDQ5SDEzLjgiIHN0cm9rZT0i
I0Q5NzcwNiIgc3Ryb2tlLXdpZHRoPSIyIiBzdHJva2UtbGluZWNhcD0icm91bmQiIHN0cm9rZS1saW5l
am9pbj0icm91bmQiLz4KPHBhdGggZD0iTTIwLjIgMjMuMzk5OUgxN0gxMy44IiBzdHJva2U9IiNEOTc3
MDYiIHN0cm9rZS13aWR0aD0iMiIgc3Ryb2tlLWxpbmVjYXA9InJvdW5kIiBzdHJva2UtbGluZWpvaW49
InJvdW5kIi8+Cjwvc3ZnPgo=
`,
	},

	"/definitions/images/secure.svg": {
		local:   "definitions/images/secure.svg",
		size:    2255,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0idXRmLTgiPz4KPCEtLSBHZW5lcmF0b3I6IEFkb2Jl
IElsbHVzdHJhdG9yIDIyLjAuMSwgU1ZHIEV4cG9ydCBQbHVnLUluIC4gU1ZHIFZlcnNpb246IDYuMDAg
QnVpbGQgMCkgIC0tPgo8c3ZnIHZlcnNpb249IjEuMSIgaWQ9IkxheWVyXzEiIHhtbG5zPSJodHRwOi8v
d3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkveGxp
bmsiIHg9IjBweCIgeT0iMHB4IgoJIHZpZXdCb3g9IjAgMCA2MS4xIDYxIiBzdHlsZT0iZW5hYmxlLWJh
Y2tncm91bmQ6bmV3IDAgMCA2MS4xIDYxOyIgeG1sOnNwYWNlPSJwcmVzZXJ2ZSI+CjxzdHlsZSB0eXBl
PSJ0ZXh0L2NzcyI+Cgkuc3Qwe2ZpbGw6I0ZGRjFGMTt9Cgkuc3Qxe2ZpbGw6I0Y4QzhCQzt9Cgkuc3Qy
e2ZpbGw6I0QzRDNEMztzdHJva2U6IzNDMzczNztzdHJva2UtbGluZWNhcDpyb3VuZDtzdHJva2UtbGlu
ZWpvaW46cm91bmQ7c3Ryb2tlLW1pdGVybGltaXQ6MTA7fQoJLnN0M3tmaWxsOm5vbmU7c3Ryb2tlOiMz
QzM3Mzc7c3Ryb2tlLXdpZHRoOjEuNTtzdHJva2UtbGluZWpvaW46cm91bmQ7c3Ryb2tlLW1pdGVybGlt
aXQ6MTA7fQoJLnN0NHtmaWxsOiNGRkYzRjM7c3Ryb2tlOiMzQzM3Mzc7c3Ryb2tlLWxpbmVjYXA6cm91
bmQ7c3Ryb2tlLWxpbmVqb2luOnJvdW5kO3N0cm9rZS1taXRlcmxpbWl0OjEwO30KCS5zdDV7ZmlsbDpu
b25lO3N0cm9rZTojM0MzNzM3O3N0cm9rZS13aWR0aDoxLjU7c3Ryb2tlLWxpbmVjYXA6cm91bmQ7c3Ry
b2tlLWxpbmVqb2luOnJvdW5kO3N0cm9rZS1taXRlcmxpbWl0OjEwO30KCS5zdDZ7ZmlsbDpub25lO3N0
cm9rZTojM0MzNzM3O3N0cm9rZS13aWR0aDowLjc1O3N0cm9rZS1saW5lY2FwOnJvdW5kO3N0cm9rZS1s
aW5lam9pbjpyb3VuZDtzdHJva2UtbWl0ZXJsaW1pdDoxMDt9Cjwvc3R5bGU+CjxnPgoJPGNpcmNsZSBp
ZD0iWE1MSURfMTE3Nl8iIGNsYXNzPSJzdDAiIGN4PSIyOS44IiBjeT0iMzAuNSIgcj0iMjciLz4KCTxw
b2x5bGluZSBpZD0iUmVjdGFuZ2xlLTQtQ29weS0yIiBjbGFzcz0ic3QxIiBwb2ludHM9IjQzLjgsMTMu
OCA0MC44LDEzLjggNDAuOCwxNi44IDQzLjgsMTYuOCAJIi8+Cgk8cmVjdCBpZD0iUmVjdGFuZ2xlLTQt
Q29weS05IiB4PSI0NS4xIiB5PSIyNy4xIiBjbGFzcz0ic3QxIiB3aWR0aD0iMi43IiBoZWlnaHQ9IjQu
MiIvPgoJPHBvbHlsaW5lIGlkPSJSZWN0YW5nbGUtNC1Db3B5LTE2IiBjbGFzcz0ic3QxIiBwb2ludHM9
IjE0LjgsMjMuOCAxNC44LDIxLjQgMTIuNCwyMS40IDEyLjQsMjMuOCAJIi8+Cgk8cmVjdCBpZD0iUmVj
dGFuZ2xlLTQtQ29weS0xOSIgeD0iMTYuOSIgeT0iMTQuNiIgY2xhc3M9InN0MSIgd2lkdGg9IjIuNyIg
aGVpZ2h0PSIyLjciLz4KCTxyZWN0IHg9IjE2LjgiIHk9IjI0LjIiIGNsYXNzPSJzdDIiIHdpZHRoPSIy
NiIgaGVpZ2h0PSIxOS43Ii8+Cgk8cGF0aCBjbGFzcz0ic3QzIiBkPSJNMzYuOSwyMi44di01LjRjMC0y
LTIuNS0zLjYtNS4zLTMuNkgyOGMtMi45LDAtNS40LDEuNi01LjQsMy42djUuNCIvPgoJPHBvbHlnb24g
Y2xhc3M9InN0NCIgcG9pbnRzPSIzNC42LDMwLjcgMzIuNSwyOC43IDI5LjgsMzEuNCAyNywyOC43IDI1
LDMwLjcgMjcuNywzMy40IDI1LDM2LjEgMjcsMzguMyAyOS44LDM1LjYgMzIuNSwzOC4zIAoJCTM0LjYs
MzYuMSAzMS45LDMzLjQgCSIvPgoJPGxpbmUgaWQ9IlhNTElEXzExODlfIiBjbGFzcz0ic3Q1IiB4MT0i
My42IiB5MT0iNDQuNCIgeDI9IjEwLjgiIHkyPSI0NC40Ii8+Cgk8bGluZSBpZD0iWE1MSURfMTE4OF8i
IGNsYXNzPSJzdDUiIHgxPSI1MiIgeTE9IjQ0LjQiIHgyPSI1OC4yIiB5Mj0iNDQuNCIvPgoJPGxpbmUg
aWQ9IlhNTElEXzExODdfIiBjbGFzcz0ic3Q1IiB4MT0iMTMuNiIgeTE9IjQ0LjQiIHgyPSI1MC41IiB5
Mj0iNDQuNCIvPgoJPGxpbmUgaWQ9IlhNTElEXzExODNfIiBjbGFzcz0ic3Q2IiB4MT0iMTEuOCIgeTE9
IjQ3LjQiIHgyPSIxNy43IiB5Mj0iNDcuNCIvPgoJPGxpbmUgaWQ9IlhNTElEXzExODJfIiBjbGFzcz0i
c3Q2IiB4MT0iMjEuMyIgeTE9IjQ3LjQiIHgyPSIyNy4yIiB5Mj0iNDcuNCIvPgoJPGxpbmUgaWQ9IlhN
TElEXzExODFfIiBjbGFzcz0ic3Q2IiB4MT0iMzAuNiIgeTE9IjQ3LjQiIHgyPSI0Mi44IiB5Mj0iNDcu
NCIvPgoJPGxpbmUgaWQ9IlhNTElEXzExODBfIiBjbGFzcz0ic3Q2IiB4MT0iNDYuNCIgeTE9IjQ3LjQi
IHgyPSI1My4zIiB5Mj0iNDcuNCIvPgo8L2c+Cjwvc3ZnPgo=
`,
	},

	"/definitions/images/smp.svg": {
		local:   "definitions/images/smp.svg",
		size:    5415,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiPz4KPHN2ZyB3aWR0aD0iODRweCIgaGVp
Z2h0PSI3NXB4IiB2aWV3Qm94PSIwIDAgODQgNzUiIHZlcnNpb249IjEuMSIgeG1sbnM9Imh0dHA6Ly93
d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGlu
ayI+CiAgICA8IS0tIEdlbmVyYXRvcjogU2tldGNoIDQzLjEgKDM5MDEyKSAtIGh0dHA6Ly93d3cuYm9o
ZW1pYW5jb2RpbmcuY29tL3NrZXRjaCAtLT4KICAgIDx0aXRsZT5Hcm91cCA1IENvcHk8L3RpdGxlPgog
ICAgPGRlc2M+Q3JlYXRlZCB3aXRoIFNrZXRjaC48L2Rlc2M+CiAgICA8ZGVmcz4KICAgICAgICA8bGlu
ZWFyR3JhZGllbnQgeDE9Ii0xLjExMDIyMzAyZS0xNCUiIHkxPSI1MCUiIHgyPSIxMDAlIiB5Mj0iNTAl
IiBpZD0ibGluZWFyR3JhZGllbnQtMSI+CiAgICAgICAgICAgIDxzdG9wIHN0b3AtY29sb3I9IiNGQkRB
NjEiIG9mZnNldD0iMCUiPjwvc3RvcD4KICAgICAgICAgICAgPHN0b3Agc3RvcC1jb2xvcj0iI0Y3NkIx
QyIgb2Zmc2V0PSIxMDAlIj48L3N0b3A+CiAgICAgICAgPC9saW5lYXJHcmFkaWVudD4KICAgICAgICA8
bGluZWFyR3JhZGllbnQgeDE9IjEwMCUiIHkxPSI1MCUiIHgyPSIyLjMwOTg2OTI2JSIgeTI9IjUwJSIg
aWQ9ImxpbmVhckdyYWRpZW50LTIiPgogICAgICAgICAgICA8c3RvcCBzdG9wLWNvbG9yPSIjNUJFREFD
IiBvZmZzZXQ9IjAlIj48L3N0b3A+CiAgICAgICAgICAgIDxzdG9wIHN0b3AtY29sb3I9IiMyREFCNDki
IG9mZnNldD0iMTAwJSI+PC9zdG9wPgogICAgICAgIDwvbGluZWFyR3JhZGllbnQ+CiAgICA8L2RlZnM+
CiAgICA8ZyBpZD0iU01QLWZsb3dzIiBzdHJva2U9Im5vbmUiIHN0cm9rZS13aWR0aD0iMSIgZmlsbD0i
bm9uZSIgZmlsbC1ydWxlPSJldmVub2RkIj4KICAgICAgICA8ZyBpZD0iR3JvdXAtNS1Db3B5IiB0cmFu
c2Zvcm09InRyYW5zbGF0ZSgyLjAwMDAwMCwgMy4wMDAwMDApIj4KICAgICAgICAgICAgPGcgaWQ9Ikdy
b3VwLTMiPgogICAgICAgICAgICAgICAgPGcgaWQ9Ikdyb3VwIiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgw
LjM4NTk1MywgMC4wMDAwMDApIj4KICAgICAgICAgICAgICAgICAgICA8ZyBpZD0iR3JvdXAtMiIgdHJh
bnNmb3JtPSJ0cmFuc2xhdGUoMjUuMDAwMDAwLCAyNy4wMDAwMDApIiBmaWxsPSIjRjI1NjVCIj4KICAg
ICAgICAgICAgICAgICAgICAgICAgPHBhdGggZD0iTTIxLjgxODQ3ODEsMTEuODEzODUyNiBDMjEuNDY1
ODExNSwxMS44MTM4NTI2IDIxLjE3OTU4OTIsMTEuNDk0MzAyMyAyMS4xNzk1ODkyLDExLjEwMDU3MDYg
TDIxLjE3OTU4OTIsMC43MTMyODIwMzkgQzIxLjE3OTU4OTIsMC4zMTk1NTAzNTMgMjEuNDY1ODExNSww
IDIxLjgxODQ3ODEsMCBDMjIuMTcxMTQ0OCwwIDIyLjQ1NzM2NywwLjMxOTU1MDM1MyAyMi40NTczNjcs
MC43MTMyODIwMzkgTDIyLjQ1NzM2NywxMS4xMDA1NzA2IEMyMi40NTczNjcsMTEuNDk0MzAyMyAyMi4x
NzExNDQ4LDExLjgxMzg1MjYgMjEuODE4NDc4MSwxMS44MTM4NTI2IiBpZD0iRmlsbC0xMSI+PC9wYXRo
PgogICAgICAgICAgICAgICAgICAgICAgICA8cGF0aCBkPSJNMy4wNjczNzc3LDYuMDIxMjcwMjcgQzIu
NzI0NjM0NDEsNS42NTY5ODE1NyAyLjM0NDI0ODc5LDUuNDcyNzE5MjcgMS45MjYyMjA4NCw1LjQ3Mjcx
OTI3IEMxLjQ1MjcxOTk5LDUuNDcyNzE5MjcgMS4wNDY1NzkxLDUuNjU2OTgxNTcgMC43MDM4MzU4MDMs
Ni4wMjEyNzAyNyBDMC4zNjEwOTI1MSw2LjM4NzY3NjkyIDAuMTkyNjkyNjI3LDYuNzkyMjA2OCAwLjE5
MjY5MjYyNyw3LjI0MzMzMTc2IEMwLjE5MjY5MjYyNyw3Ljc0NzQwNTY1IDAuMzYxMDkyNTEsOC4xODM3
MDQ5IDAuNzAzODM1ODAzLDguNTQ3OTkzNTkgQzEuMDQ2NTc5MSw4LjkxMjI4MjI4IDEuNDUyNzE5OTks
OS4wOTQ0MjY2MyAxLjkyNjIyMDg0LDkuMDk0NDI2NjMgQzIuMzQ0MjQ4NzksOS4wOTQ0MjY2MyAyLjcy
NDYzNDQxLDguOTEyMjgyMjggMy4wNjczNzc3LDguNTQ3OTkzNTkgQzMuNDEwMTIxLDguMTgzNzA0OSAz
LjU3ODUyMDg4LDcuNzQ3NDA1NjUgMy41Nzg1MjA4OCw3LjI0MzMzMTc2IEMzLjU3ODUyMDg4LDYuNzky
MjA2OCAzLjQxMDEyMSw2LjM4NzY3NjkyIDMuMDY3Mzc3Nyw2LjAyMTI3MDI3IEwzLjA2NzM3NzcsNi4w
MjEyNzAyNyBaIiBpZD0iUGF0aCIgc3Ryb2tlPSIjRjI1NjVCIiBzdHJva2Utd2lkdGg9IjEuMyI+PC9w
YXRoPgogICAgICAgICAgICAgICAgICAgICAgICA8cGF0aCBkPSJNOS44MDMzNzMwNiw2LjAyMTI3MDI3
IEM5LjQ4ODM2NjIxLDUuNjU2OTgxNTcgOS4wOTQxMTIzNyw1LjQ3MjcxOTI3IDguNjIwNjExNTIsNS40
NzI3MTkyNyBDOC4xNDkwOTE4NCw1LjQ3MjcxOTI3IDcuNzU0ODM4LDUuNjU2OTgxNTcgNy40Mzk4MzEx
Niw2LjAyMTI3MDI3IEM3LjEyNDgyNDMxLDYuMzg3Njc2OTIgNi45NjYzMzAzMSw2Ljc5MjIwNjggNi45
NjYzMzAzMSw3LjI0MzMzMTc2IEM2Ljk2NjMzMDMxLDcuNzQ3NDA1NjUgNy4xMjQ4MjQzMSw4LjE4Mzcw
NDkgNy40Mzk4MzExNiw4LjU0Nzk5MzU5IEM3Ljc1NDgzOCw4LjkxMjI4MjI4IDguMTQ5MDkxODQsOS4w
OTQ0MjY2MyA4LjYyMDYxMTUyLDkuMDk0NDI2NjMgQzkuMDk0MTEyMzcsOS4wOTQ0MjY2MyA5LjQ4ODM2
NjIxLDguOTEyMjgyMjggOS44MDMzNzMwNiw4LjU0Nzk5MzU5IEMxMC4xMTgzNzk5LDguMTgzNzA0OSAx
MC4yNzY4NzM5LDcuNzQ3NDA1NjUgMTAuMjc2ODczOSw3LjI0MzMzMTc2IEMxMC4yNzY4NzM5LDYuNzky
MjA2OCAxMC4xMTgzNzk5LDYuMzg3Njc2OTIgOS44MDMzNzMwNiw2LjAyMTI3MDI3IEw5LjgwMzM3MzA2
LDYuMDIxMjcwMjcgWiIgaWQ9IlBhdGgiIHN0cm9rZT0iI0YyNTY1QiIgc3Ryb2tlLXdpZHRoPSIxLjMi
PjwvcGF0aD4KICAgICAgICAgICAgICAgICAgICAgICAgPHBhdGggZD0iTTE2LjUzOTM2ODQsNi4wMjEy
NzAyNyBDMTYuMTk2NjI1MSw1LjY1Njk4MTU3IDE1Ljc5MDQ4NDIsNS40NzI3MTkyNyAxNS4zMTg5NjQ1
LDUuNDcyNzE5MjcgQzE0Ljg5Njk3NDIsNS40NzI3MTkyNyAxNC41MTY1ODg2LDUuNjU2OTgxNTcgMTQu
MTc1ODI2NSw2LjAyMTI3MDI3IEMxMy44MzMwODMyLDYuMzg3Njc2OTIgMTMuNjYyNzAyMiw2Ljc5MjIw
NjggMTMuNjYyNzAyMiw3LjI0MzMzMTc2IEMxMy42NjI3MDIyLDcuNzQ3NDA1NjUgMTMuODMzMDgzMiw4
LjE4MzcwNDkgMTQuMTc1ODI2NSw4LjU0Nzk5MzU5IEMxNC41MTY1ODg2LDguOTEyMjgyMjggMTQuODk2
OTc0Miw5LjA5NDQyNjYzIDE1LjMxODk2NDUsOS4wOTQ0MjY2MyBDMTUuNzkwNDg0Miw5LjA5NDQyNjYz
IDE2LjE5NjYyNTEsOC45MTIyODIyOCAxNi41MzkzNjg0LDguNTQ3OTkzNTkgQzE2Ljg4MDEzMDUsOC4x
ODM3MDQ5IDE3LjA1MDUxMTYsNy43NDc0MDU2NSAxNy4wNTA1MTE2LDcuMjQzMzMxNzYgQzE3LjA1MDUx
MTYsNi43OTIyMDY4IDE2Ljg4MDEzMDUsNi4zODc2NzY5MiAxNi41MzkzNjg0LDYuMDIxMjcwMjcgTDE2
LjUzOTM2ODQsNi4wMjEyNzAyNyBaIiBpZD0iUGF0aCIgc3Ryb2tlPSIjRjI1NjVCIiBzdHJva2Utd2lk
dGg9IjEuMyI+PC9wYXRoPgogICAgICAgICAgICAgICAgICAgIDwvZz4KICAgICAgICAgICAgICAgICAg
ICA8cGF0aCBkPSJNMzUuMTQ5Mjg1LDY5LjMzMjM1NTEgQzE1LjczNzExNiw2OS4zMzIzNTUxIDAsNTMu
ODExNTM3MSAwLDM0LjY2NjE3NzUgQzAsMTUuNTIyNzQzOSAxNS43MzcxMTYsMCAzNS4xNDkyODUsMCBD
NTQuNTYxNDU0MSwwIDcwLjI5ODU3MDEsMTUuNTIyNzQzOSA3MC4yOTg1NzAxLDM0LjY2NjE3NzUgQzcw
LjI5ODU3MDEsNTMuODExNTM3MSA1NC41NjE0NTQxLDY5LjMzMjM1NTEgMzUuMTQ5Mjg1LDY5LjMzMjM1
NTEgWiIgaWQ9IlN0cm9rZS03LUNvcHktOCIgc3Ryb2tlPSJ1cmwoI2xpbmVhckdyYWRpZW50LTEpIiBz
dHJva2Utd2lkdGg9IjQuNTU5OTk5OTQiPjwvcGF0aD4KICAgICAgICAgICAgICAgIDwvZz4KICAgICAg
ICAgICAgICAgIDxwYXRoIGQ9Ik02Mi40NDQ3NDE3LDY1LjY1Nzc4ODQgQzYyLjA3NDk5NjIsNjYuMDI4
NzYyOCA2MS42MzU5Mjk5LDY2LjIxNDI0NzMgNjEuMTI3NTI5OCw2Ni4yMTQyNDczIEM2MC42MTkxMjk3
LDY2LjIxNDI0NzMgNjAuMTgwMDYzNSw2Ni4wMjg3NjI4IDU5LjgxMDMxNzksNjUuNjU3Nzg4NCBMNDku
ODczNDU2MSw1NS42ODc5IEM0OS41MDM3MTA2LDU1LjMxNjkyNTYgNDkuMzE4ODQwNiw1NC44NzY0IDQ5
LjMxODg0MDYsNTQuMzY2MzEwMiBDNDkuMzE4ODQwNiw1My44NTYyMjAzIDQ5LjUwMzcxMDYsNTMuNDE1
Njk0OCA0OS44NzM0NTYxLDUzLjA0NDcyMDMgTDUyLjQxNTQ0NCw1MC40OTQyODM4IEM1Mi43ODUxODk2
LDUwLjEyMzMwOTMgNTMuMjI0MjU1OCw0OS45Mzc4MjQ5IDUzLjczMjY1NTksNDkuOTM3ODI0OSBDNTQu
MjQxMDU2MSw0OS45Mzc4MjQ5IDU0LjY4MDEyMjMsNTAuMTIzMzA5MyA1NS4wNDk4Njc4LDUwLjQ5NDI4
MzggTDYxLjEyNzUyOTgsNTYuNTkyMTQ1NyBMNzYuMDc5MDQwNSw0MS41OTA5NDE2IEM3Ni40NDg3ODYs
NDEuMjE5OTY3MiA3Ni44ODc4NTIzLDQxLjAzNDQ4MjggNzcuMzk2MjUyNCw0MS4wMzQ0ODI4IEM3Ny45
MDQ2NTI1LDQxLjAzNDQ4MjggNzguMzQzNzE4OCw0MS4yMTk5NjcyIDc4LjcxMzQ2NDMsNDEuNTkwOTQx
NiBMODEuMjU1NDUyMiw0NC4xNDEzNzgyIEM4MS42MjUxOTc3LDQ0LjUxMjM1MjYgODEuODEwMDY3Nyw0
NC45NTI4NzgyIDgxLjgxMDA2NzcsNDUuNDYyOTY4IEM4MS44MTAwNjc3LDQ1Ljk3MzA1NzkgODEuNjI1
MTk3Nyw0Ni40MTM1ODM0IDgxLjI1NTQ1MjIsNDYuNzg0NTU3OSBMNjIuNDQ0NzQxNyw2NS42NTc3ODg0
IFoiIGlkPSJQYXRoLUNvcHktMiIgZmlsbD0idXJsKCNsaW5lYXJHcmFkaWVudC0yKSI+PC9wYXRoPgog
ICAgICAgICAgICA8L2c+CiAgICAgICAgICAgIDxnIGlkPSJHcm91cC00IiB0cmFuc2Zvcm09InRyYW5z
bGF0ZSgxNC4wMDAwMDAsIDE3LjAwMDAwMCkiIGZpbGw9IiNGMkE2NDEiPgogICAgICAgICAgICAgICAg
PGcgaWQ9Ikdyb3VwLTMiPgogICAgICAgICAgICAgICAgICAgIDxwYXRoIGQ9Ik00NCwzMy4wOTA5MDkx
IEwyNC43NSwzMy4wOTA5MDkxIEwyNC43NSwzNS44NDA5MDkxIEwzMywzNS44NDA5MDkxIEwzMywzOC41
OTA5MDkxIEwxMSwzOC41OTA5MDkxIEwxMSwzNS44NDA5MDkxIEwxOS4yNSwzNS44NDA5MDkxIEwxOS4y
NSwzMy4wOTA5MDkxIEwwLDMzLjA5MDkwOTEgTDAsMC4wOTA5MDkwOTA5IEw0NCwwLjA5MDkwOTA5MDkg
TDQ0LDMzLjA5MDkwOTEgWiBNMi43NSwzMC4zNDA5MDkxIEw0MS4yNSwzMC4zNDA5MDkxIEw0MS4yNSwy
Ljg0MDkwOTA5IEwyLjc1LDIuODQwOTA5MDkgTDIuNzUsMzAuMzQwOTA5MSBaIiBpZD0iWi1Db3B5LTIi
PjwvcGF0aD4KICAgICAgICAgICAgICAgIDwvZz4KICAgICAgICAgICAgPC9nPgogICAgICAgIDwvZz4K
ICAgIDwvZz4KPC9zdmc+
`,
	},

	"/definitions/images/smpsuccess.svg": {
		local:   "definitions/images/smpsuccess.svg",
		size:    15608,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiIHN0YW5kYWxvbmU9Im5vIj8+Cjxzdmcg
d2lkdGg9IjIzNHB4IiBoZWlnaHQ9IjU4cHgiIHZpZXdCb3g9IjAgMCAyMzQgNTgiIHZlcnNpb249IjEu
MSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3
dy53My5vcmcvMTk5OS94bGluayI+CiAgICA8IS0tIEdlbmVyYXRvcjogU2tldGNoIDQxLjIgKDM1Mzk3
KSAtIGh0dHA6Ly93d3cuYm9oZW1pYW5jb2RpbmcuY29tL3NrZXRjaCAtLT4KICAgIDx0aXRsZT5Hcm91
cCAxOTwvdGl0bGU+CiAgICA8ZGVzYz5DcmVhdGVkIHdpdGggU2tldGNoLjwvZGVzYz4KICAgIDxkZWZz
PgogICAgICAgIDxsaW5lYXJHcmFkaWVudCB4MT0iMTAwJSIgeTE9IjUwJSIgeDI9IjMuMDYxNjE3ZS0x
NSUiIHkyPSI1MCUiIGlkPSJsaW5lYXJHcmFkaWVudC0xIj4KICAgICAgICAgICAgPHN0b3Agc3RvcC1j
b2xvcj0iI0ZCREE2MSIgb2Zmc2V0PSIwJSI+PC9zdG9wPgogICAgICAgICAgICA8c3RvcCBzdG9wLWNv
bG9yPSIjRjc2QjFDIiBvZmZzZXQ9IjEwMCUiPjwvc3RvcD4KICAgICAgICA8L2xpbmVhckdyYWRpZW50
PgogICAgICAgIDxsaW5lYXJHcmFkaWVudCB4MT0iLTEuMTEwMjIzMDJlLTE0JSIgeTE9IjUwJSIgeDI9
IjEwMCUiIHkyPSI1MCUiIGlkPSJsaW5lYXJHcmFkaWVudC0yIj4KICAgICAgICAgICAgPHN0b3Agc3Rv
cC1jb2xvcj0iI0ZCREE2MSIgb2Zmc2V0PSIwJSI+PC9zdG9wPgogICAgICAgICAgICA8c3RvcCBzdG9w
LWNvbG9yPSIjRjc2QjFDIiBvZmZzZXQ9IjEwMCUiPjwvc3RvcD4KICAgICAgICA8L2xpbmVhckdyYWRp
ZW50PgogICAgPC9kZWZzPgogICAgPGcgaWQ9IlNNUC1mbG93cyIgc3Ryb2tlPSJub25lIiBzdHJva2Ut
d2lkdGg9IjEiIGZpbGw9Im5vbmUiIGZpbGwtcnVsZT0iZXZlbm9kZCI+CiAgICAgICAgPGcgaWQ9IkJv
Yl80IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgtNjE1LjAwMDAwMCwgLTI1Ni4wMDAwMDApIj4KICAgICAg
ICAgICAgPGcgaWQ9Ikdyb3VwLTI4LUNvcHkiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDU2MC4wMDAwMDAs
IDIwMC4wMDAwMDApIj4KICAgICAgICAgICAgICAgIDxnIGlkPSJHcm91cC0yNiIgdHJhbnNmb3JtPSJ0
cmFuc2xhdGUoNDIuMDAwMDAwLCA1Ni4wMDAwMDApIj4KICAgICAgICAgICAgICAgICAgICA8ZyBpZD0i
R3JvdXAtMTkiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDE0LjAwMDAwMCwgMC4wMDAwMDApIj4KICAgICAg
ICAgICAgICAgICAgICAgICAgPGcgaWQ9Ikdyb3VwIiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgxMDcuOTA2
OTc3LCAwLjAwMDAwMCkiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhdGggZD0iTTUuNjIw
MTU1MDQsMTEuMTAzODk2MSBMMTQuNjEyNDAzMSwxMS4xMDM4OTYxIEwxNC42MTI0MDMxLDcuNzcyNzI3
MjcgQzE0LjYxMjQwMzEsNi41NDY2NjU5NSAxNC4xNzMzMzI5LDUuNDk5OTAyODggMTMuMjk1MTc5Myw0
LjYzMjQwNjY2IEMxMi40MTcwMjU2LDMuNzY0OTEwNDQgMTEuMzU3NDAyOCwzLjMzMTE2ODgzIDEwLjEx
NjI3OTEsMy4zMzExNjg4MyBDOC44NzUxNTUyOSwzLjMzMTE2ODgzIDcuODE1NTMyNDksMy43NjQ5MTA0
NCA2LjkzNzM3ODg4LDQuNjMyNDA2NjYgQzYuMDU5MjI1MjYsNS40OTk5MDI4OCA1LjYyMDE1NTA0LDYu
NTQ2NjY1OTUgNS42MjAxNTUwNCw3Ljc3MjcyNzI3IEw1LjYyMDE1NTA0LDExLjEwMzg5NjEgWiBNMjAu
MjMyNTU4MSwxMi43Njk0ODA1IEwyMC4yMzI1NTgxLDIyLjc2Mjk4NyBDMjAuMjMyNTU4MSwyMy4yMjU2
NTE3IDIwLjA2ODYzODYsMjMuNjE4OTEwNyAxOS43NDA3OTQ2LDIzLjk0Mjc3NiBDMTkuNDEyOTUwNiwy
NC4yNjY2NDEyIDE5LjAxNDg2MDIsMjQuNDI4NTcxNCAxOC41NDY1MTE2LDI0LjQyODU3MTQgTDEuNjg2
MDQ2NTEsMjQuNDI4NTcxNCBDMS4yMTc2OTc5MiwyNC40Mjg1NzE0IDAuODE5NjA3NTgyLDI0LjI2NjY0
MTIgMC40OTE3NjM1NjYsMjMuOTQyNzc2IEMwLjE2MzkxOTU0OSwyMy42MTg5MTA3IDAsMjMuMjI1NjUx
NyAwLDIyLjc2Mjk4NyBMMCwxMi43Njk0ODA1IEMwLDEyLjMwNjgxNTkgMC4xNjM5MTk1NDksMTEuOTEz
NTU2OCAwLjQ5MTc2MzU2NiwxMS41ODk2OTE2IEMwLjgxOTYwNzU4MiwxMS4yNjU4MjYzIDEuMjE3Njk3
OTIsMTEuMTAzODk2MSAxLjY4NjA0NjUxLDExLjEwMzg5NjEgTDIuMjQ4MDYyMDIsMTEuMTAzODk2MSBM
Mi4yNDgwNjIwMiw3Ljc3MjcyNzI3IEMyLjI0ODA2MjAyLDUuNjQ0NDY5ODggMy4wMjA4MjU2MSwzLjgx
Njk3MTkyIDQuNTY2Mzc1OTcsMi4yOTAxNzg1NyBDNi4xMTE5MjYzMywwLjc2MzM4NTIyMyA3Ljk2MTg3
NTUzLDAgMTAuMTE2Mjc5MSwwIEMxMi4yNzA2ODI2LDAgMTQuMTIwNjMxOCwwLjc2MzM4NTIyMyAxNS42
NjYxODIyLDIuMjkwMTc4NTcgQzE3LjIxMTczMjUsMy44MTY5NzE5MiAxNy45ODQ0OTYxLDUuNjQ0NDY5
ODggMTcuOTg0NDk2MSw3Ljc3MjcyNzI3IEwxNy45ODQ0OTYxLDExLjEwMzg5NjEgTDE4LjU0NjUxMTYs
MTEuMTAzODk2MSBDMTkuMDE0ODYwMiwxMS4xMDM4OTYxIDE5LjQxMjk1MDYsMTEuMjY1ODI2MyAxOS43
NDA3OTQ2LDExLjU4OTY5MTYgQzIwLjA2ODYzODYsMTEuOTEzNTU2OCAyMC4yMzI1NTgxLDEyLjMwNjgx
NTkgMjAuMjMyNTU4MSwxMi43Njk0ODA1IEwyMC4yMzI1NTgxLDEyLjc2OTQ4MDUgWiIgaWQ9Iu+Aoy1j
b3B5LTIiIGZpbGw9IiM3RUQzMjEiPjwvcGF0aD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxw
YXRoIGQ9Ik0xNS4zMDkxMTU0LDE1LjA0MjI1OSBMOS42MDIzMDM3MywyMC42Nzk4MTIxIEM5LjQ5MDEy
OTk2LDIwLjc5MDYyNDUgOS4zNTY5MjU2MiwyMC44NDYwMjk4IDkuMjAyNjg2NjksMjAuODQ2MDI5OCBD
OS4wNDg0NDc3NywyMC44NDYwMjk4IDguOTE1MjQzNDIsMjAuNzkwNjI0NSA4LjgwMzA2OTY2LDIwLjY3
OTgxMjEgTDUuNzg4NDE0ODQsMTcuNzAxNzQzNSBDNS42NzYyNDEwOCwxNy41OTA5MzExIDUuNjIwMTU1
MDQsMTcuNDU5MzQzMyA1LjYyMDE1NTA0LDE3LjMwNjk3NjMgQzUuNjIwMTU1MDQsMTcuMTU0NjA5MiA1
LjY3NjI0MTA4LDE3LjAyMzAyMTUgNS43ODg0MTQ4NCwxNi45MTIyMDkxIEw2LjU1OTYwNTYxLDE2LjE1
MDM3NzYgQzYuNjcxNzc5MzcsMTYuMDM5NTY1MiA2LjgwNDk4MzcyLDE1Ljk4NDE1OTggNi45NTkyMjI2
NCwxNS45ODQxNTk4IEM3LjExMzQ2MTU3LDE1Ljk4NDE1OTggNy4yNDY2NjU5MSwxNi4wMzk1NjUyIDcu
MzU4ODM5NjgsMTYuMTUwMzc3NiBMOS4yMDI2ODY2OSwxNy45NzE4NDc0IEwxMy43Mzg2OTA2LDEzLjQ5
MDg5MzEgQzEzLjg1MDg2NDMsMTMuMzgwMDgwNyAxMy45ODQwNjg3LDEzLjMyNDY3NTMgMTQuMTM4MzA3
NiwxMy4zMjQ2NzUzIEMxNC4yOTI1NDY1LDEzLjMyNDY3NTMgMTQuNDI1NzUwOSwxMy4zODAwODA3IDE0
LjUzNzkyNDYsMTMuNDkwODkzMSBMMTUuMzA5MTE1NCwxNC4yNTI3MjQ2IEMxNS40MjEyODkyLDE0LjM2
MzUzNyAxNS40NzczNzUyLDE0LjQ5NTEyNDggMTUuNDc3Mzc1MiwxNC42NDc0OTE4IEMxNS40NzczNzUy
LDE0Ljc5OTg1ODkgMTUuNDIxMjg5MiwxNC45MzE0NDY2IDE1LjMwOTExNTQsMTUuMDQyMjU5IEwxNS4z
MDkxMTU0LDE1LjA0MjI1OSBaIiBpZD0iUGF0aCIgZmlsbD0iI0ZGRkZGRiI+PC9wYXRoPgogICAgICAg
ICAgICAgICAgICAgICAgICA8L2c+CiAgICAgICAgICAgICAgICAgICAgICAgIDxwYXRoIGQ9Ik0xNzUu
NDcyNzMsMjkuNDU5NTk2IEw1Ni44NDg2Mzk5LDI5LjQ1OTU5NiIgaWQ9IlN0cm9rZS0xMy1Db3B5IiBz
dHJva2U9InVybCgjbGluZWFyR3JhZGllbnQtMSkiIHN0cm9rZS13aWR0aD0iMS41NzQ2NCIgc3Ryb2tl
LWxpbmVjYXA9InJvdW5kIiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgxMTYuMTU2NTE3LCAyOS4yMTExMjkp
IHNjYWxlKC0xLCAxKSB0cmFuc2xhdGUoLTExNi4xNTY1MTcsIC0yOS4yMTExMjkpICI+PC9wYXRoPgog
ICAgICAgICAgICAgICAgICAgICAgICA8cmVjdCBpZD0iUmVjdGFuZ2xlLTMtQ29weS05IiBmaWxsPSIj
RTlFOUVBIiB4PSIxNi4xODYwNDY1IiB5PSIxNi4yODU3MTQzIiB3aWR0aD0iMS4zNDg4MzcyMSIgaGVp
Z2h0PSIyMS43MTQyODU3IiByeD0iMC42NzQ0MTg2MDUiPjwvcmVjdD4KICAgICAgICAgICAgICAgICAg
ICAgICAgPGcgaWQ9Ikdyb3VwLTE2IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgxNzUuMzQ4ODM3LCAxLjM1
NzE0MykiPgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhdGggZD0iTTI4LjUzOTM3NzksNTUu
NjQyODU3MSBDMTMuMDEzODYzOCw1NS42NDI4NTcxIDAuNDI3NTkzMDI5LDQzLjE4NjU4NTMgMC40Mjc1
OTMwMjksMjcuODIxNDI4NiBDMC40Mjc1OTMwMjksMTIuNDU3ODE3NSAxMy4wMTM4NjM4LDAgMjguNTM5
Mzc3OSwwIEM0NC4wNjQ4OTIsMCA1Ni42NTExNjI4LDEyLjQ1NzgxNzUgNTYuNjUxMTYyOCwyNy44MjE0
Mjg2IEM1Ni42NTExNjI4LDQzLjE4NjU4NTMgNDQuMDY0ODkyLDU1LjY0Mjg1NzEgMjguNTM5Mzc3OSw1
NS42NDI4NTcxIFoiIGlkPSJTdHJva2UtNyIgc3Ryb2tlPSJ1cmwoI2xpbmVhckdyYWRpZW50LTIpIiBz
dHJva2Utd2lkdGg9IjEuMTMzNzQwOCIgZmlsbD0iI0U4RThFNyI+PC9wYXRoPgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHBhdGggZD0iTTI3LjQxNDkwNjUsNTMuOTU2NzEgQzEyLjgyMDkyMzMsNTMu
OTU2NzEgMC45ODk4Mjg3MjcsNDIuMjU1MzYzNyAwLjk4OTgyODcyNywyNy44MjE0Mjg2IEMwLjk4OTgy
ODcyNywxMy4zODg5NDU0IDEyLjgyMDkyMzMsMS42ODYxNDcxOSAyNy40MTQ5MDY1LDEuNjg2MTQ3MTkg
QzQyLjAwODg4OTgsMS42ODYxNDcxOSA1My44Mzk5ODQzLDEzLjM4ODk0NTQgNTMuODM5OTg0MywyNy44
MjE0Mjg2IEM1My44Mzk5ODQzLDQyLjI1NTM2MzcgNDIuMDA4ODg5OCw1My45NTY3MSAyNy40MTQ5MDY1
LDUzLjk1NjcxIFoiIGlkPSJTdHJva2UtNy1Db3B5IiBmaWxsPSIjRkZGRkZGIj48L3BhdGg+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cmVjdCBpZD0iUmVjdGFuZ2xlLTMtQ29weS03IiBmaWxsPSIj
RTlFOUVBIiB0cmFuc2Zvcm09InRyYW5zbGF0ZSg0MC4zNDYzMjgsIDM3LjkzODMxMikgcm90YXRlKC05
MC4wMDAwMDApIHRyYW5zbGF0ZSgtNDAuMzQ2MzI4LCAtMzcuOTM4MzEyKSAiIHg9IjM5LjUwMjk3NCIg
eT0iMzEuMTkzNzIyOSIgd2lkdGg9IjEuNjg2NzA3MDkiIGhlaWdodD0iMTMuNDg5MTc3NSIgcng9IjAu
ODQzMzUzNTQ2Ij48L3JlY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8ZyBpZD0iR3JvdXAt
MTgiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDkuNjAxODkyLCAxNi4zMDkxMTMpIj4KICAgICAgICAgICAg
ICAgICAgICAgICAgICAgICAgICA8cmVjdCBpZD0iUmVjdGFuZ2xlLTMiIGZpbGw9IiNFOUU5RUEiIHg9
IjMzLjU1NTYxNCIgeT0iMC41NTIzNTg1NjEiIHdpZHRoPSIxLjY4NjcwNzA5IiBoZWlnaHQ9IjE5LjEw
OTY2ODEiIHJ4PSIwLjg0MzM1MzU0NiI+PC9yZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxyZWN0IGlkPSJSZWN0YW5nbGUtMy1Db3B5IiBmaWxsPSIjRTlFOUVBIiB0cmFuc2Zvcm09InRy
YW5zbGF0ZSgyOS42MTk5NjQsIDIwLjUwNTEwMCkgcm90YXRlKC05MC4wMDAwMDApIHRyYW5zbGF0ZSgt
MjkuNjE5OTY0LCAtMjAuNTA1MTAwKSAiIHg9IjI4Ljc3NjYxMDYiIHk9IjEwLjk1MDI2NjIiIHdpZHRo
PSIxLjY4NjcwNzA5IiBoZWlnaHQ9IjE5LjEwOTY2ODEiIHJ4PSIwLjg0MzM1MzU0NiI+PC9yZWN0Pgog
ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxyZWN0IGlkPSJSZWN0YW5nbGUtMy1Db3B5LTYi
IGZpbGw9IiNFOUU5RUEiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDI1LjEyMjA3OSwgMTguODE4OTUzKSBy
b3RhdGUoLTkwLjAwMDAwMCkgdHJhbnNsYXRlKC0yNS4xMjIwNzksIC0xOC44MTg5NTMpICIgeD0iMjQu
Mjc4NzI1IiB5PSI5LjI2NDExOTAyIiB3aWR0aD0iMS42ODY3MDcwOSIgaGVpZ2h0PSIxOS4xMDk2Njgx
IiByeD0iMC44NDMzNTM1NDYiPjwvcmVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cmVjdCBpZD0iUmVjdGFuZ2xlLTMtQ29weS04IiBmaWxsPSIjRTlFOUVBIiB0cmFuc2Zvcm09InRyYW5z
bGF0ZSgzMC4xODIyMDAsIDEuOTU3NDgxKSByb3RhdGUoLTkwLjAwMDAwMCkgdHJhbnNsYXRlKC0zMC4x
ODIyMDAsIC0xLjk1NzQ4MSkgIiB4PSIyOS4zMzg4NDYzIiB5PSItMi41Mzg5MTEyOCIgd2lkdGg9IjEu
Njg2NzA3MDkiIGhlaWdodD0iOC45OTI3ODQ5OSIgcng9IjAuODQzMzUzNTQ2Ij48L3JlY3Q+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICAgICAgPHBhdGggZD0iTTE5Ljg1NzA1OTEsMS41NDA5NTM2NyBD
MTkuNzA0NjE5OCwxLjU0MDk1MzY3IDE5LjYyOTk1NTYsMS42NDA3NDg3NSAxOS42Mjk5NTU2LDEuODQz
NDU3NTMgQzE5LjYyOTk1NTYsMS45OTYyNjg3NiAxOS43MDQ2MTk4LDIuMDcyNjc0MzggMTkuODU3MDU5
MSwyLjA3MjY3NDM4IEMyMC4wNTc3MTkxLDIuMDcyNjc0MzggMjAuMTYwMzgyNCwxLjk5NjI2ODc2IDIw
LjE2MDM4MjQsMS44NDM0NTc1MyBDMjAuMTYwMzgyNCwxLjY0MDc0ODc1IDIwLjA1NzcxOTEsMS41NDA5
NTM2NyAxOS44NTcwNTkxLDEuNTQwOTUzNjcgTDE5Ljg1NzA1OTEsMS41NDA5NTM2NyBaIE03LjA1MjE1
MjQsMTcuNTY1ODYyIEwzMi44ODkwNjk0LDE3LjU2NTg2MiBMMzIuODg5MDY5NCwzLjEzNDU1NjUgTDMw
LjU0MDI1ODgsMy4xMzQ1NTY1IEwyOS43ODI3Mjg1LDMuMTM0NTU2NSBMMjguNzIxODc0OSwzLjEzNDU1
NjUgTDI3Ljk2NDM0NDYsMy4xMzQ1NTY1IEwyNi42NzYzODc1LDMuMTM0NTU2NSBMMjUuOTE3MzAxNywz
LjEzNDU1NjUgTDcuMDUyMTUyNCwzLjEzNDU1NjUgTDcuMDUyMTUyNCwxNy41NjU4NjIgWiBNNi40NDU1
MDU5NCwxOC4xNzM5ODgzIEwzMy40OTQxNjA0LDE4LjE3Mzk4ODMgTDMzLjQ5NDE2MDQsMi41Mjc5ODk0
NyBMNi40NDU1MDU5NCwyLjUyNzk4OTQ3IEw2LjQ0NTUwNTk0LDE4LjE3Mzk4ODMgWiBNNS4xNTc1NDg4
NSwxOS4zODg2ODE3IEwzNC44NTk4OTI3LDE5LjM4ODY4MTcgTDM0Ljg1OTg5MjcsMS45OTYyNjg3NiBD
MzQuODU5ODkyNywxLjQzOTU5OTI4IDM0LjU4MTQ1NzUsMS4xNjA0ODQ4OSAzNC4wMjQ1ODcyLDEuMTYw
NDg0ODkgTDUuOTkxMjk4ODUsMS4xNjA0ODQ4OSBDNS40MzQ0Mjg1MiwxLjE2MDQ4NDg5IDUuMTU3NTQ4
ODUsMS40Mzk1OTkyOCA1LjE1NzU0ODg1LDEuOTk2MjY4NzYgTDUuMTU3NTQ4ODUsMTkuMzg4NjgxNyBa
IE0yLjk2MTE3NzU4LDIyLjQyNzc1NCBMMzYuNjAyMDU2OCwyMi40Mjc3NTQgTDM3LjQzNTgwNjgsMjEu
ODE5NjI3NyBMMi4wNTEyMDc4OSwyMS44MTk2Mjc3IEwyLjk2MTE3NzU4LDIyLjQyNzc1NCBaIE0zOC42
NDc1NDQzLDIwLjIyNDQ2NTYgQzM4LjY0NzU0NDMsMjAuMDIzMzE2MSAzOC41NDQ4ODEsMTkuOTIwNDAy
NCAzOC4zNDQyMjEsMTkuOTIwNDAyNCBMMS4yMTc0NTc4OSwxOS45MjA0MDI0IEMxLjAxNTI0MjQxLDE5
LjkyMDQwMjQgMC45MTQxMzQ2NjQsMjAuMDIzMzE2MSAwLjkxNDEzNDY2NCwyMC4yMjQ0NjU2IEwwLjkx
NDEzNDY2NCwyMC45ODU0MDMxIEMwLjkxNDEzNDY2NCwyMS4wMzUzMDA3IDAuOTkwMzU0MzQ3LDIxLjEz
NTA5NTcgMS4xNDI3OTM3MSwyMS4yODc5MDcgTDEuMTQyNzkzNzEsMjEuMjExNTAxNCBMMS4yMTc0NTc4
OSwyMS4yODc5MDcgTDM4LjM0NDIyMSwyMS4yODc5MDcgQzM4LjU0NDg4MSwyMS4yODc5MDcgMzguNjQ3
NTQ0MywyMS4xODgxMTE5IDM4LjY0NzU0NDMsMjAuOTg1NDAzMSBMMzguNjQ3NTQ0MywyMC4yMjQ0NjU2
IFogTTM5LjE3Nzk3MSwyMC4yMjQ0NjU2IEwzOS4xNzc5NzEsMjAuOTg1NDAzMSBDMzkuMTc3OTcxLDIx
LjQ0MDcxODIgMzguOTUwODY3NSwyMS43MTgyNzMzIDM4LjQ5NTEwNDksMjEuODE5NjI3NyBMMzYuODI5
MTYwNCwyMi45NTc5MTU0IEwzNi44MjkxNjA0LDIzLjAzNDMyMSBMMi42NTc4NTQzNSwyMy4wMzQzMjEg
TDIuNjU3ODU0MzUsMjIuOTU3OTE1NCBMMC45OTAzNTQzNDcsMjEuODE5NjI3NyBDMC41ODQzNjc4NzIs
MjEuNjY4Mzc1OCAwLjM4MzcwNzg5LDIxLjM4OTI2MTQgMC4zODM3MDc4OSwyMC45ODU0MDMxIEwwLjM4
MzcwNzg5LDIwLjIyNDQ2NTYgQzAuMzgzNzA3ODksMTkuNjY3Nzk2MSAwLjY2MDU4NzU1NSwxOS4zODg2
ODE3IDEuMjE3NDU3ODksMTkuMzg4NjgxNyBMNC41NTI0NTc5LDE5LjM4ODY4MTcgTDQuNTUyNDU3OSwx
Ljk5NjI2ODc2IEM0LjU1MjQ1NzksMS42NDA3NDg3NSA0LjY5MDg5NzczLDEuMzEzMjk2MTIgNC45Njc3
Nzc0LDEuMDA3NjczNjYgQzUuMjQ0NjU3MDYsMC43MDUxNjk3OTIgNS41ODUzMTIzOCwwLjU1MjM1ODU2
MSA1Ljk5MTI5ODg1LDAuNTUyMzU4NTYxIEwzNC4wMjQ1ODcyLDAuNTUyMzU4NTYxIEMzNC4zNzc2ODY1
LDAuNTUyMzU4NTYxIDM0LjY5NTAwOTMsMC42OTI2OTU0MDUgMzQuOTcxODg4OSwwLjk3MDI1MDQ5OCBD
MzUuMjUwMzI0MSwxLjI0OTM2NDg5IDM1LjM4ODc2MzksMS41OTA4NTEyMSAzNS4zODg3NjM5LDEuOTk2
MjY4NzYgTDM1LjM4ODc2MzksMTkuMzg4NjgxNyBMMzguMzQ0MjIxLDE5LjM4ODY4MTcgQzM4Ljg5OTUz
NTksMTkuMzg4NjgxNyAzOS4xNzc5NzEsMTkuNjY3Nzk2MSAzOS4xNzc5NzEsMjAuMjI0NDY1NiBMMzku
MTc3OTcxLDIwLjIyNDQ2NTYgWiIgaWQ9IkZpbGwtMyIgZmlsbD0iIzAwMDAwMCI+PC9wYXRoPgogICAg
ICAgICAgICAgICAgICAgICAgICAgICAgPC9nPgogICAgICAgICAgICAgICAgICAgICAgICA8L2c+CiAg
ICAgICAgICAgICAgICAgICAgICAgIDxnIGlkPSJHcm91cC0xNyIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUo
MC4wMDAwMDAsIDEuMzU3MTQzKSI+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGF0aCBkPSJN
MjguMTExNzg0OSw1NS42NDI4NTcxIEMxMi41ODYyNzA4LDU1LjY0Mjg1NzEgMCw0My4xODY1ODUzIDAs
MjcuODIxNDI4NiBDMCwxMi40NTc4MTc1IDEyLjU4NjI3MDgsMCAyOC4xMTE3ODQ5LDAgQzQzLjYzNzI5
OSwwIDU2LjIyMzU2OTgsMTIuNDU3ODE3NSA1Ni4yMjM1Njk4LDI3LjgyMTQyODYgQzU2LjIyMzU2OTgs
NDMuMTg2NTg1MyA0My42MzcyOTksNTUuNjQyODU3MSAyOC4xMTE3ODQ5LDU1LjY0Mjg1NzEgWiIgaWQ9
IlN0cm9rZS03LUNvcHktMiIgc3Ryb2tlPSJ1cmwoI2xpbmVhckdyYWRpZW50LTIpIiBzdHJva2Utd2lk
dGg9IjEuMTMzNzQwOCIgZmlsbD0iI0U4RThFNyIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoMjguMTExNzg1
LCAyNy44MjE0MjkpIHNjYWxlKC0xLCAxKSB0cmFuc2xhdGUoLTI4LjExMTc4NSwgLTI3LjgyMTQyOSkg
Ij48L3BhdGg+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cGF0aCBkPSJNMjkuMjM2MjU2Myw1
My45NTY3MSBDMTQuNjQyMjczLDUzLjk1NjcxIDIuODExMTc4NDksNDIuMjU1MzYzNyAyLjgxMTE3ODQ5
LDI3LjgyMTQyODYgQzIuODExMTc4NDksMTMuMzg4OTQ1NCAxNC42NDIyNzMsMS42ODYxNDcxOSAyOS4y
MzYyNTYzLDEuNjg2MTQ3MTkgQzQzLjgzMDIzOTUsMS42ODYxNDcxOSA1NS42NjEzMzQxLDEzLjM4ODk0
NTQgNTUuNjYxMzM0MSwyNy44MjE0Mjg2IEM1NS42NjEzMzQxLDQyLjI1NTM2MzcgNDMuODMwMjM5NSw1
My45NTY3MSAyOS4yMzYyNTYzLDUzLjk1NjcxIFoiIGlkPSJTdHJva2UtNy1Db3B5LTMiIGZpbGw9IiNG
RkZGRkYiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDI5LjIzNjI1NiwgMjcuODIxNDI5KSBzY2FsZSgtMSwg
MSkgdHJhbnNsYXRlKC0yOS4yMzYyNTYsIC0yNy44MjE0MjkpICI+PC9wYXRoPgogICAgICAgICAgICAg
ICAgICAgICAgICAgICAgPHJlY3QgaWQ9IlJlY3RhbmdsZS0zLUNvcHktMTAiIGZpbGw9IiNFOUU5RUEi
IHRyYW5zZm9ybT0idHJhbnNsYXRlKDE5LjY3ODI0OSwgMzUuNjkwMTE1KSByb3RhdGUoLTkwLjAwMDAw
MCkgdHJhbnNsYXRlKC0xOS42NzgyNDksIC0zNS42OTAxMTUpICIgeD0iMTguODM0ODk1OSIgeT0iMjYu
MTM1MjgxNCIgd2lkdGg9IjEuNjg2NzA3MDkiIGhlaWdodD0iMTkuMTA5NjY4MSIgcng9IjAuODQzMzUz
NTQ2Ij48L3JlY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cmVjdCBpZD0iUmVjdGFuZ2xl
LTMtQ29weS0xMSIgZmlsbD0iI0U5RTlFQSIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoMTcuOTkxNTQyLCAz
Ni44MTQyMTQpIHJvdGF0ZSgtOTAuMDAwMDAwKSB0cmFuc2xhdGUoLTE3Ljk5MTU0MiwgLTM2LjgxNDIx
NCkgIiB4PSIxNy4xNDgxODg4IiB5PSIzMC4wNjk2MjQ4IiB3aWR0aD0iMS42ODY3MDcwOSIgaGVpZ2h0
PSIxMy40ODkxNzc1IiByeD0iMC44NDMzNTM1NDYiPjwvcmVjdD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxyZWN0IGlkPSJSZWN0YW5nbGUtMy1Db3B5LTEyIiBmaWxsPSIjRTlFOUVBIiB0cmFuc2Zv
cm09InRyYW5zbGF0ZSgyMy42MTM4OTksIDM0LjAwMzk2OCkgcm90YXRlKC05MC4wMDAwMDApIHRyYW5z
bGF0ZSgtMjMuNjEzODk5LCAtMzQuMDAzOTY4KSAiIHg9IjIyLjc3MDU0NTgiIHk9IjI0LjQ0OTEzNDIi
IHdpZHRoPSIxLjY4NjcwNzA5IiBoZWlnaHQ9IjE5LjEwOTY2ODEiIHJ4PSIwLjg0MzM1MzU0NiI+PC9y
ZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHJlY3QgaWQ9IlJlY3RhbmdsZS0zLUNvcHkt
MTMiIGZpbGw9IiNFOUU5RUEiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDE5LjExNjAxNCwgMTcuMTQyNDk2
KSByb3RhdGUoLTkwLjAwMDAwMCkgdHJhbnNsYXRlKC0xOS4xMTYwMTQsIC0xNy4xNDI0OTYpICIgeD0i
MTguMjcyNjYwMiIgeT0iMTIuNjQ2MTAzOSIgd2lkdGg9IjEuNjg2NzA3MDkiIGhlaWdodD0iOC45OTI3
ODQ5OSIgcng9IjAuODQzMzUzNTQ2Ij48L3JlY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8
cGF0aCBkPSJNMjkuNTkzNTkzOCwxNi43MjU5Njg4IEMyOS40NDExNTQ0LDE2LjcyNTk2ODggMjkuMzY2
NDkwMywxNi44MjU3NjM5IDI5LjM2NjQ5MDMsMTcuMDI4NDcyNyBDMjkuMzY2NDkwMywxNy4xODEyODM5
IDI5LjQ0MTE1NDQsMTcuMjU3Njg5NiAyOS41OTM1OTM4LDE3LjI1NzY4OTYgQzI5Ljc5NDI1MzgsMTcu
MjU3Njg5NiAyOS44OTY5MTcsMTcuMTgxMjgzOSAyOS44OTY5MTcsMTcuMDI4NDcyNyBDMjkuODk2OTE3
LDE2LjgyNTc2MzkgMjkuNzk0MjUzOCwxNi43MjU5Njg4IDI5LjU5MzU5MzgsMTYuNzI1OTY4OCBMMjku
NTkzNTkzOCwxNi43MjU5Njg4IFogTTE2Ljc4ODY4NzEsMzIuNzUwODc3MiBMNDIuNjI1NjA0MSwzMi43
NTA4NzcyIEw0Mi42MjU2MDQxLDE4LjMxOTU3MTcgTDQwLjI3Njc5MzUsMTguMzE5NTcxNyBMMzkuNTE5
MjYzMSwxOC4zMTk1NzE3IEwzOC40NTg0MDk2LDE4LjMxOTU3MTcgTDM3LjcwMDg3OTMsMTguMzE5NTcx
NyBMMzYuNDEyOTIyMiwxOC4zMTk1NzE3IEwzNS42NTM4MzY0LDE4LjMxOTU3MTcgTDE2Ljc4ODY4NzEs
MTguMzE5NTcxNyBMMTYuNzg4Njg3MSwzMi43NTA4NzcyIFogTTE2LjE4MjA0MDYsMzMuMzU5MDAzNSBM
NDMuMjMwNjk1LDMzLjM1OTAwMzUgTDQzLjIzMDY5NSwxNy43MTMwMDQ2IEwxNi4xODIwNDA2LDE3Ljcx
MzAwNDYgTDE2LjE4MjA0MDYsMzMuMzU5MDAzNSBaIE0xNC44OTQwODM1LDM0LjU3MzY5NjkgTDQ0LjU5
NjQyNzMsMzQuNTczNjk2OSBMNDQuNTk2NDI3MywxNy4xODEyODM5IEM0NC41OTY0MjczLDE2LjYyNDYx
NDUgNDQuMzE3OTkyMiwxNi4zNDU1MDAxIDQzLjc2MTEyMTgsMTYuMzQ1NTAwMSBMMTUuNzI3ODMzNSwx
Ni4zNDU1MDAxIEMxNS4xNzA5NjMyLDE2LjM0NTUwMDEgMTQuODk0MDgzNSwxNi42MjQ2MTQ1IDE0Ljg5
NDA4MzUsMTcuMTgxMjgzOSBMMTQuODk0MDgzNSwzNC41NzM2OTY5IFogTTEyLjY5NzcxMjIsMzcuNjEy
NzY5MiBMNDYuMzM4NTkxNSwzNy42MTI3NjkyIEw0Ny4xNzIzNDE1LDM3LjAwNDY0MjkgTDExLjc4Nzc0
MjYsMzcuMDA0NjQyOSBMMTIuNjk3NzEyMiwzNy42MTI3NjkyIFogTTQ4LjM4NDA3ODksMzUuNDA5NDgw
NyBDNDguMzg0MDc4OSwzNS4yMDgzMzEzIDQ4LjI4MTQxNTcsMzUuMTA1NDE3NiA0OC4wODA3NTU3LDM1
LjEwNTQxNzYgTDEwLjk1Mzk5MjYsMzUuMTA1NDE3NiBDMTAuNzUxNzc3MSwzNS4xMDU0MTc2IDEwLjY1
MDY2OTMsMzUuMjA4MzMxMyAxMC42NTA2NjkzLDM1LjQwOTQ4MDcgTDEwLjY1MDY2OTMsMzYuMTcwNDE4
MyBDMTAuNjUwNjY5MywzNi4yMjAzMTU4IDEwLjcyNjg4OSwzNi4zMjAxMTA5IDEwLjg3OTMyODQsMzYu
NDcyOTIyMiBMMTAuODc5MzI4NCwzNi4zOTY1MTY1IEwxMC45NTM5OTI2LDM2LjQ3MjkyMjIgTDQ4LjA4
MDc1NTcsMzYuNDcyOTIyMiBDNDguMjgxNDE1NywzNi40NzI5MjIyIDQ4LjM4NDA3ODksMzYuMzczMTI3
MSA0OC4zODQwNzg5LDM2LjE3MDQxODMgTDQ4LjM4NDA3ODksMzUuNDA5NDgwNyBaIE00OC45MTQ1MDU3
LDM1LjQwOTQ4MDcgTDQ4LjkxNDUwNTcsMzYuMTcwNDE4MyBDNDguOTE0NTA1NywzNi42MjU3MzM0IDQ4
LjY4NzQwMjEsMzYuOTAzMjg4NSA0OC4yMzE2Mzk2LDM3LjAwNDY0MjkgTDQ2LjU2NTY5NTEsMzguMTQy
OTMwNiBMNDYuNTY1Njk1MSwzOC4yMTkzMzYyIEwxMi4zOTQzODksMzguMjE5MzM2MiBMMTIuMzk0Mzg5
LDM4LjE0MjkzMDYgTDEwLjcyNjg4OSwzNy4wMDQ2NDI5IEMxMC4zMjA5MDI1LDM2Ljg1MzM5MDkgMTAu
MTIwMjQyNiwzNi41NzQyNzY1IDEwLjEyMDI0MjYsMzYuMTcwNDE4MyBMMTAuMTIwMjQyNiwzNS40MDk0
ODA3IEMxMC4xMjAyNDI2LDM0Ljg1MjgxMTMgMTAuMzk3MTIyMiwzNC41NzM2OTY5IDEwLjk1Mzk5MjYs
MzQuNTczNjk2OSBMMTQuMjg4OTkyNiwzNC41NzM2OTY5IEwxNC4yODg5OTI2LDE3LjE4MTI4MzkgQzE0
LjI4ODk5MjYsMTYuODI1NzYzOSAxNC40Mjc0MzI0LDE2LjQ5ODMxMTMgMTQuNzA0MzEyMSwxNi4xOTI2
ODg4IEMxNC45ODExOTE3LDE1Ljg5MDE4NSAxNS4zMjE4NDcsMTUuNzM3MzczNyAxNS43Mjc4MzM1LDE1
LjczNzM3MzcgTDQzLjc2MTEyMTgsMTUuNzM3MzczNyBDNDQuMTE0MjIxMiwxNS43MzczNzM3IDQ0LjQz
MTU0MzksMTUuODc3NzEwNiA0NC43MDg0MjM2LDE2LjE1NTI2NTcgQzQ0Ljk4Njg1ODgsMTYuNDM0Mzgw
MSA0NS4xMjUyOTg2LDE2Ljc3NTg2NjQgNDUuMTI1Mjk4NiwxNy4xODEyODM5IEw0NS4xMjUyOTg2LDM0
LjU3MzY5NjkgTDQ4LjA4MDc1NTcsMzQuNTczNjk2OSBDNDguNjM2MDcwNSwzNC41NzM2OTY5IDQ4Ljkx
NDUwNTcsMzQuODUyODExMyA0OC45MTQ1MDU3LDM1LjQwOTQ4MDcgTDQ4LjkxNDUwNTcsMzUuNDA5NDgw
NyBaIiBpZD0iRmlsbC0zLUNvcHkiIGZpbGw9IiMwMDAwMDAiPjwvcGF0aD4KICAgICAgICAgICAgICAg
ICAgICAgICAgPC9nPgogICAgICAgICAgICAgICAgICAgICAgICA8cmVjdCBpZD0iUmVjdGFuZ2xlLTQi
IGZpbGw9IiNGRkVEQ0YiIHg9IjYwLjMxOTc5MDEiIHk9IjE0LjU1MTU4NzMiIHdpZHRoPSI4LjY4Nzg4
MDE4IiBoZWlnaHQ9IjguNzQ2MDMxNzUiPjwvcmVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgPHJl
Y3QgaWQ9IlJlY3RhbmdsZS00LUNvcHktMyIgZmlsbD0iI0ZGRDI4QSIgeD0iNjMuNTYzMjk3NCIgeT0i
MzUuMzIzNDEyNyIgd2lkdGg9IjYuMzE4NDU4MzIiIGhlaWdodD0iNi4zNjA3NTAzNiI+PC9yZWN0Pgog
ICAgICAgICAgICAgICAgICAgICAgICA8cmVjdCBpZD0iUmVjdGFuZ2xlLTQtQ29weSIgZmlsbD0iI0Y5
Q0M4NSIgeD0iNzYuNTM3MzI2OSIgeT0iMjEuMTExMTExMSIgd2lkdGg9IjQuNzM4ODQzNzQiIGhlaWdo
dD0iNC43NzA1NjI3NyI+PC9yZWN0PgogICAgICAgICAgICAgICAgICAgICAgICA8cmVjdCBpZD0iUmVj
dGFuZ2xlLTQtQ29weS00IiBmaWxsPSIjRjhDOEJDIiB4PSI5OS4yNDE4Nzg2IiB5PSIzMC45NTAzOTY4
IiB3aWR0aD0iMy4xNTkyMjkxNiIgaGVpZ2h0PSIzLjE4MDM3NTE4Ij48L3JlY3Q+CiAgICAgICAgICAg
ICAgICAgICAgICAgIDxyZWN0IGlkPSJSZWN0YW5nbGUtNC1Db3B5LTYiIGZpbGw9IiNGOEM4QkMiIHg9
IjkyLjc1NDg2MzgiIHk9IjI0LjM5MDg3MyIgd2lkdGg9IjIuMzY5NDIxODciIGhlaWdodD0iMi4zODUy
ODEzOSI+PC9yZWN0PgogICAgICAgICAgICAgICAgICAgICAgICA8cmVjdCBpZD0iUmVjdGFuZ2xlLTQt
Q29weS01IiBmaWxsPSIjRjhDOEJDIiB4PSI4My4wMjQzNDE3IiB5PSIzNC4yMzAxNTg3IiB3aWR0aD0i
My45NDkwMzY0NSIgaGVpZ2h0PSIzLjk3NTQ2ODk4Ij48L3JlY3Q+CiAgICAgICAgICAgICAgICAgICAg
ICAgIDxyZWN0IGlkPSJSZWN0YW5nbGUtNC1Db3B5LTE0IiBmaWxsPSIjRjhDOEJDIiB4PSI4Ni4yNjc4
NDkxIiB5PSIxNy44MzEzNDkyIiB3aWR0aD0iMi4zNjk0MjE4NyIgaGVpZ2h0PSIyLjM4NTI4MTM5Ij48
L3JlY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgIDxnIGlkPSJHcm91cC0xNSIgdHJhbnNmb3JtPSJ0
cmFuc2xhdGUoMTUwLjU5NzQxMiwgMjkuMzEwNTE2KSBzY2FsZSgtMSwgMSkgdHJhbnNsYXRlKC0xNTAu
NTk3NDEyLCAtMjkuMzEwNTE2KSB0cmFuc2xhdGUoMTI5LjUxNDYxNCwgMTUuNjQ0ODQxKSI+CiAgICAg
ICAgICAgICAgICAgICAgICAgICAgICA8cmVjdCBpZD0iUmVjdGFuZ2xlLTQtQ29weS0yIiBmaWxsPSIj
RkZFRENGIiB4PSIwIiB5PSIwIiB3aWR0aD0iOC42ODc4ODAxOCIgaGVpZ2h0PSI4Ljc0NjAzMTc1Ij48
L3JlY3Q+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICA8cmVjdCBpZD0iUmVjdGFuZ2xlLTQtQ29w
eS03IiBmaWxsPSIjRkZEMjhBIiB4PSI0LjMyNDY3NjUiIHk9IjIwLjc3MTgyNTQiIHdpZHRoPSI2LjMx
ODQ1ODMyIiBoZWlnaHQ9IjYuMzYwNzUwMzYiPjwvcmVjdD4KICAgICAgICAgICAgICAgICAgICAgICAg
ICAgIDxyZWN0IGlkPSJSZWN0YW5nbGUtNC1Db3B5LTkiIGZpbGw9IiNGOUNDODUiIHg9IjE2LjIxNzUz
NjkiIHk9IjUuNDY2MjY5ODQiIHdpZHRoPSI0LjczODg0Mzc0IiBoZWlnaHQ9IjQuNzcwNTYyNzciPjwv
cmVjdD4KICAgICAgICAgICAgICAgICAgICAgICAgICAgIDxyZWN0IGlkPSJSZWN0YW5nbGUtNC1Db3B5
LTEzIiBmaWxsPSIjRjhDOEJDIiB4PSIzOC45MjIwODg1IiB5PSIxNi4zOTg4MDk1IiB3aWR0aD0iMy4x
NTkyMjkxNiIgaGVpZ2h0PSIzLjE4MDM3NTE4Ij48L3JlY3Q+CiAgICAgICAgICAgICAgICAgICAgICAg
ICAgICA8cmVjdCBpZD0iUmVjdGFuZ2xlLTQtQ29weS0xNiIgZmlsbD0iI0Y4QzhCQyIgeD0iMzIuNDM1
MDczNyIgeT0iOC43NDYwMzE3NSIgd2lkdGg9IjIuMzY5NDIxODciIGhlaWdodD0iMi4zODUyODEzOSI+
PC9yZWN0PgogICAgICAgICAgICAgICAgICAgICAgICAgICAgPHJlY3QgaWQ9IlJlY3RhbmdsZS00LUNv
cHktMTgiIGZpbGw9IiNGOEM4QkMiIHg9IjIyLjcwNDU1MTYiIHk9IjE5LjY3ODU3MTQiIHdpZHRoPSIz
Ljk0OTAzNjQ1IiBoZWlnaHQ9IjMuOTc1NDY4OTgiPjwvcmVjdD4KICAgICAgICAgICAgICAgICAgICAg
ICAgICAgIDxyZWN0IGlkPSJSZWN0YW5nbGUtNC1Db3B5LTE5IiBmaWxsPSIjRjhDOEJDIiB4PSIyNS45
NDgwNTkiIHk9IjMuMjc5NzYxOSIgd2lkdGg9IjIuMzY5NDIxODciIGhlaWdodD0iMi4zODUyODEzOSI+
PC9yZWN0PgogICAgICAgICAgICAgICAgICAgICAgICA8L2c+CiAgICAgICAgICAgICAgICAgICAgPC9n
PgogICAgICAgICAgICAgICAgPC9nPgogICAgICAgICAgICA8L2c+CiAgICAgICAgPC9nPgogICAgPC9n
Pgo8L3N2Zz4=
`,
	},

	"/definitions/images/success.svg": {
		local:   "definitions/images/success.svg",
		size:    1103,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0idXRmLTgiPz4KPCEtLSBHZW5lcmF0b3I6IEFkb2Jl
IElsbHVzdHJhdG9yIDIyLjAuMSwgU1ZHIEV4cG9ydCBQbHVnLUluIC4gU1ZHIFZlcnNpb246IDYuMDAg
QnVpbGQgMCkgIC0tPgo8c3ZnIHZlcnNpb249IjEuMSIgaWQ9IkxheWVyXzEiIHhtbG5zPSJodHRwOi8v
d3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkveGxp
bmsiIHg9IjBweCIgeT0iMHB4IgoJIHZpZXdCb3g9IjAgMCA1MiA1MyIgc3R5bGU9ImVuYWJsZS1iYWNr
Z3JvdW5kOm5ldyAwIDAgNTIgNTM7IiB4bWw6c3BhY2U9InByZXNlcnZlIj4KPHN0eWxlIHR5cGU9InRl
eHQvY3NzIj4KCS5zdDB7ZmlsbDpub25lO3N0cm9rZTojODlBRjhGO3N0cm9rZS1taXRlcmxpbWl0OjEw
O30KCS5zdDF7ZmlsbDojRTFGOUUzO3N0cm9rZTojODlBRjhGO3N0cm9rZS1taXRlcmxpbWl0OjEwO30K
CS5zdDJ7ZmlsbDpub25lO3N0cm9rZTojODlBRjhGO3N0cm9rZS1saW5lY2FwOnJvdW5kO3N0cm9rZS1s
aW5lam9pbjpyb3VuZDtzdHJva2UtbWl0ZXJsaW1pdDoxMDt9Cjwvc3R5bGU+CjxnPgoJPGNpcmNsZSBj
bGFzcz0ic3QwIiBjeD0iMjguOSIgY3k9IjI3LjMiIHI9IjIxLjEiLz4KCTxwb2x5Z29uIGNsYXNzPSJz
dDEiIHBvaW50cz0iMTcuMSwyOC44IDI1LjIsMzYuOSA0MC4xLDIxLjkgMzYuOCwxOC42IDI1LjIsMzAu
MyAyMC4zLDI1LjQgCSIvPgoJPHBhdGggY2xhc3M9InN0MiIgZD0iTTIxLjQsMTEuOWMxLjEtMC4zLDIu
Mi0wLjUsMy40LTAuNSIvPgoJPHBhdGggY2xhc3M9InN0MiIgZD0iTTEyLjgsMjMuNWMwLTMuNywxLjct
Nyw0LjQtOS4zIi8+Cgk8cGF0aCBjbGFzcz0ic3QyIiBkPSJNMTguNiwxMy4zYzAuMy0wLjIsMC42LTAu
NCwwLjktMC41Ii8+Cgk8cGF0aCBjbGFzcz0ic3QyIiBkPSJNMzUuNSw0Mi40Yy0xLjEsMC4zLTIuMiww
LjUtMy40LDAuNSIvPgoJPHBhdGggY2xhc3M9InN0MiIgZD0iTTQ0LjIsMzAuOGMwLDMuNy0xLjcsNy00
LjQsOS4zIi8+Cgk8cGF0aCBjbGFzcz0ic3QyIiBkPSJNMzguNCw0MS4xYy0wLjMsMC4yLTAuNiwwLjQt
MC45LDAuNSIvPgo8L2c+Cjwvc3ZnPgo=
`,
	},

	"/definitions/images/waiting.svg": {
		local:   "definitions/images/waiting.svg",
		size:    2496,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0idXRmLTgiPz4KPCEtLSBHZW5lcmF0b3I6IEFkb2Jl
IElsbHVzdHJhdG9yIDIyLjAuMSwgU1ZHIEV4cG9ydCBQbHVnLUluIC4gU1ZHIFZlcnNpb246IDYuMDAg
QnVpbGQgMCkgIC0tPgo8c3ZnIHZlcnNpb249IjEuMSIgaWQ9IkxheWVyXzEiIHhtbG5zPSJodHRwOi8v
d3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkveGxp
bmsiIHg9IjBweCIgeT0iMHB4IgoJIHZpZXdCb3g9IjAgMCA1Ny43IDU3IiBzdHlsZT0iZW5hYmxlLWJh
Y2tncm91bmQ6bmV3IDAgMCA1Ny43IDU3OyIgeG1sOnNwYWNlPSJwcmVzZXJ2ZSI+CjxzdHlsZSB0eXBl
PSJ0ZXh0L2NzcyI+Cgkuc3Qwe2ZpbGw6I0ZGRjFGMTt9Cgkuc3Qxe2ZpbGw6I0Y4QzhCQzt9Cgkuc3Qy
e2ZpbGw6bm9uZTtzdHJva2U6IzY1NzI3RjtzdHJva2Utd2lkdGg6MS41O3N0cm9rZS1saW5lY2FwOnJv
dW5kO3N0cm9rZS1saW5lam9pbjpyb3VuZDtzdHJva2UtbWl0ZXJsaW1pdDoxMDt9Cgkuc3Qze2ZpbGw6
bm9uZTtzdHJva2U6IzY1NzI3RjtzdHJva2Utd2lkdGg6MC43NTtzdHJva2UtbGluZWNhcDpyb3VuZDtz
dHJva2UtbGluZWpvaW46cm91bmQ7c3Ryb2tlLW1pdGVybGltaXQ6MTA7fQoJLnN0NHtmaWxsOiM2NTcy
N0Y7fQo8L3N0eWxlPgo8Y2lyY2xlIGlkPSJYTUxJRF8xMTc2XyIgY2xhc3M9InN0MCIgY3g9IjI3LjYi
IGN5PSIyOC41IiByPSIyNyIvPgo8cG9seWxpbmUgaWQ9IlJlY3RhbmdsZS00LUNvcHktMiIgY2xhc3M9
InN0MSIgcG9pbnRzPSI0MS42LDExLjggMzguNiwxMS44IDM4LjYsMTQuOCA0MS42LDE0LjggIi8+Cjxy
ZWN0IGlkPSJSZWN0YW5nbGUtNC1Db3B5LTkiIHg9IjQyLjkiIHk9IjI1LjEiIGNsYXNzPSJzdDEiIHdp
ZHRoPSIyLjciIGhlaWdodD0iNC4yIi8+Cjxwb2x5bGluZSBpZD0iUmVjdGFuZ2xlLTQtQ29weS0xNiIg
Y2xhc3M9InN0MSIgcG9pbnRzPSIxMi42LDIxLjggMTIuNiwxOS40IDEwLjIsMTkuNCAxMC4yLDIxLjgg
Ii8+CjxyZWN0IGlkPSJSZWN0YW5nbGUtNC1Db3B5LTE5IiB4PSIxNC43IiB5PSIxMi43IiBjbGFzcz0i
c3QxIiB3aWR0aD0iMi43IiBoZWlnaHQ9IjIuNyIvPgo8bGluZSBpZD0iWE1MSURfMTE4OV8iIGNsYXNz
PSJzdDIiIHgxPSIxLjQiIHkxPSI0Mi40IiB4Mj0iOC42IiB5Mj0iNDIuNCIvPgo8bGluZSBpZD0iWE1M
SURfMTE4OF8iIGNsYXNzPSJzdDIiIHgxPSI1MS4xIiB5MT0iNDIuNCIgeDI9IjU2IiB5Mj0iNDIuNCIv
Pgo8bGluZSBpZD0iWE1MSURfMTE4N18iIGNsYXNzPSJzdDIiIHgxPSIxMS40IiB5MT0iNDIuNCIgeDI9
IjQ4LjMiIHkyPSI0Mi40Ii8+CjxsaW5lIGlkPSJYTUxJRF8xMTgzXyIgY2xhc3M9InN0MyIgeDE9Ijku
NiIgeTE9IjQ1LjQiIHgyPSIxNS41IiB5Mj0iNDUuNCIvPgo8bGluZSBpZD0iWE1MSURfMTE4Ml8iIGNs
YXNzPSJzdDMiIHgxPSIxOS4xIiB5MT0iNDUuNCIgeDI9IjI1IiB5Mj0iNDUuNCIvPgo8bGluZSBpZD0i
WE1MSURfMTE4MV8iIGNsYXNzPSJzdDMiIHgxPSIyOC40IiB5MT0iNDUuNCIgeDI9IjQwLjYiIHkyPSI0
NS40Ii8+CjxsaW5lIGlkPSJYTUxJRF8xMTgwXyIgY2xhc3M9InN0MyIgeDE9IjQ0LjIiIHkxPSI0NS40
IiB4Mj0iNTEuMSIgeTI9IjQ1LjQiLz4KPHBhdGggY2xhc3M9InN0NCIgZD0iTTMzLjYsMjYuOGMwLjUt
MC4yLDEuNi0xLjIsMS42LTMuMWwwLjEtMy45aDAuNWMwLjIsMCwwLjUtMC4xLDAuNS0wLjR2LTIuMWMw
LTAuMi0wLjEtMC41LTAuNC0wLjVsLTE3LjEtMC4xCgljLTAuMiwwLTAuNSwwLjEtMC41LDAuNHYyLjFj
MCwwLjIsMC4xLDAuNSwwLjQsMC41aDAuNWwtMC4xLDRjMCwxLjksMC41LDIuNCwxLjUsMi45bDQuNCwy
Ljd2MS4xYy0xLjIsMC42LTQsMi4yLTQuNCwyLjYKCWMtMC40LDAuMi0xLjYsMS4yLTEuNiwzdjMuM2gt
MC41Yy0wLjIsMC0wLjUsMC4xLTAuNSwwLjR2MmMwLDAuMiwwLjEsMC41LDAuNCwwLjVsMTcuMSwwLjJj
MC4yLDAsMC41LTAuMSwwLjUtMC40di0yLjEKCWMwLTAuMi0wLjEtMC41LTAuNC0wLjVoLTAuNXYtMy4z
YzAtMS45LTAuNS0yLjQtMS41LTIuOWwtNC40LTIuN3YtMS4xQzMwLjUsMjguNiwzMy4zLDI3LDMzLjYs
MjYuOHogTTI4LjEsMzEuNmw0LjcsMi45CgljMC40LDAuMiwwLjYsMC40LDAuNywwLjVjMC4xLDAuMiww
LjIsMC42LDAuMSwxLjF2My4zbC0xMi44LTAuMVYzNmMwLTEuMSwwLjYtMS40LDAuOS0xLjZjMC40LTAu
MiwyLjktMS42LDQuNy0yLjcKCWMwLjItMC4xLDAuNC0wLjQsMC41LTAuNmwwLjEtMmMwLTAuMi0wLjIt
MC42LTAuNC0wLjZsLTQuOS0zLjFsMCwwYy0wLjQtMC4yLTAuNi0wLjQtMC43LTAuNWMtMC4xLTAuMi0w
LjItMC42LTAuMS0xLjFsMC4xLTQKCWwxMi44LDAuMWwtMC4xLDMuOWMwLDEuMS0wLjYsMS40LTAuOSwx
LjZjLTAuNCwwLjItMi45LDEuNi00LjcsMi43Yy0wLjIsMC4xLTAuNCwwLjQtMC41LDAuNnYyQzI3LjYs
MzEuMiwyNy45LDMxLjMsMjguMSwzMS42eiIvPgo8L3N2Zz4K
`,
	},

	"/definitions/images/warning.svg": {
		local:   "definitions/images/warning.svg",
		size:    1203,
		modtime: 1489449600,
		compressed: `
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0idXRmLTgiPz4KPCEtLSBHZW5lcmF0b3I6IEFkb2Jl
IElsbHVzdHJhdG9yIDIyLjAuMSwgU1ZHIEV4cG9ydCBQbHVnLUluIC4gU1ZHIFZlcnNpb246IDYuMDAg
QnVpbGQgMCkgIC0tPgo8c3ZnIHZlcnNpb249IjEuMSIgaWQ9IkxheWVyXzEiIHhtbG5zPSJodHRwOi8v
d3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkveGxp
bmsiIHg9IjBweCIgeT0iMHB4IgoJIHZpZXdCb3g9IjAgMCA1NiA1NyIgc3R5bGU9ImVuYWJsZS1iYWNr
Z3JvdW5kOm5ldyAwIDAgNTYgNTc7IiB4bWw6c3BhY2U9InByZXNlcnZlIj4KPHN0eWxlIHR5cGU9InRl
eHQvY3NzIj4KCS5zdDB7ZmlsbDojQUE4MzgxO3N0cm9rZTojN0I1MTREO3N0cm9rZS13aWR0aDoxLjU7
c3Ryb2tlLWxpbmVjYXA6cm91bmQ7c3Ryb2tlLWxpbmVqb2luOnJvdW5kO3N0cm9rZS1taXRlcmxpbWl0
OjEwO30KCS5zdDF7ZmlsbDpub25lO3N0cm9rZTojN0I1MTREO3N0cm9rZS1saW5lY2FwOnJvdW5kO3N0
cm9rZS1saW5lam9pbjpyb3VuZDtzdHJva2UtbWl0ZXJsaW1pdDoxMDt9Cjwvc3R5bGU+CjxnPgoJPGNp
cmNsZSBjbGFzcz0ic3QwIiBjeD0iMjcuNCIgY3k9IjM4LjIiIHI9IjEuNCIvPgoJPHBhdGggY2xhc3M9
InN0MCIgZD0iTTI4LjYsMTkuNlYzMmMwLDAuNy0wLjYsMS40LTEuMywxLjRjLTAuNywwLTEuMy0wLjUt
MS4zLTEuM1YxOS42di0wLjFjMC0wLjcsMC42LTEuMywxLjMtMS4zCgkJUzI4LjYsMTguOSwyOC42LDE5
LjZ6Ii8+Cgk8Y2lyY2xlIGNsYXNzPSJzdDEiIGN4PSIyNy4zIiBjeT0iMjkuMyIgcj0iMjEuMSIvPgoJ
PHBhdGggY2xhc3M9InN0MSIgZD0iTTExLjIsMjUuNWMwLTMuNywxLjctNyw0LjQtOS4zIi8+Cgk8cGF0
aCBjbGFzcz0ic3QxIiBkPSJNMTcsMTUuM2MwLjMtMC4yLDAuNi0wLjQsMC45LTAuNSIvPgoJPHBhdGgg
Y2xhc3M9InN0MSIgZD0iTTE5LjgsMTMuOWMxLjEtMC4zLDIuMi0wLjUsMy40LTAuNSIvPgoJPHBhdGgg
Y2xhc3M9InN0MSIgZD0iTTMzLjksNDQuNGMtMS4xLDAuMy0yLjIsMC41LTMuNCwwLjUiLz4KCTxwYXRo
IGNsYXNzPSJzdDEiIGQ9Ik00Mi42LDMyLjhjMCwzLjctMS43LDctNC40LDkuMyIvPgoJPHBhdGggY2xh
c3M9InN0MSIgZD0iTTM2LjgsNDMuMWMtMC4zLDAuMi0wLjYsMC40LTAuOSwwLjUiLz4KPC9nPgo8L3N2
Zz4K
`,
	},

	"/": {
		isDir: true,
		local: "",
	},

	"/definitions": {
		isDir: true,
		local: "definitions",
	},

	"/definitions/images": {
		isDir: true,
		local: "definitions/images",
	},
}
