#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "chess.h"
#include "mprocs.h"
#include "moves.h"
#include "checks.h"
#include "misc.h"

#define CMSCORE               (50000000)

extern move movestack[MAX_PV][256],mvlst[1024];
extern tosquare temptsq[30];
extern char pvchar[1024];

 /*  Generate _all_ legal moves (incl. check test) */
 /*   and return them in a list format */
void mvs(board *sq,int side,int *num,int cpk,int cpq,move *lastmv,int resflg,int ply) {
  int a=-1,b,c,p,opt,flg,kx,ky,nmvs=0,n;
  move *mov;
  tosquare *tmv;

   /*  Generate King Moves Separately */
  kmvs(sq,side,num,ply);

  if (side==WHITE) {kx=sq->kpos[0];ky=sq->kpos[1];}
  else {kx=sq->kpos[2];ky=sq->kpos[3];}

  nmvs=*num;
 
  /*  Generate castling moves if possible. */
  /*  Quite slow if cpk and cpq are both still 1 */
  if (cpk && castleok(side,1,sq,cpk,cpq)==1) {
    movestack[ply][nmvs].fx=7;
    movestack[ply][nmvs].tx=5;
    if (side==WHITE) movestack[ply][nmvs].fy=movestack[ply][nmvs].ty=7;
    else movestack[ply][nmvs].fy=movestack[ply][nmvs].ty=0;
    movestack[ply][nmvs].score=0;
    movestack[ply][nmvs].castle=1;
    movestack[ply][nmvs].ep=movestack[ply][nmvs].promote=0;
    nmvs++;
  }
  if (cpq && castleok(side,2,sq,cpk,cpq)==1) {
    movestack[ply][nmvs].fx=0;
    movestack[ply][nmvs].tx=3;
    if (side==WHITE) movestack[ply][nmvs].fy=movestack[ply][nmvs].ty=7;
    else movestack[ply][nmvs].fy=movestack[ply][nmvs].ty=0;
    movestack[ply][nmvs].score=0;
    movestack[ply][nmvs].castle=2;
    movestack[ply][nmvs].ep=movestack[ply][nmvs].promote=0;
    nmvs++;
  }
   
  /*  Generate en-passant moves if possible */
  /*  Slow if one exists, but generally fairly fast */
  if (lastmv) a=en_passant_possible(sq,lastmv,side);
  else a=-1;
  if (a>-1) {
    assert(lastmv!=NULL);
    if (a==0 || a==2) {
      movestack[ply][nmvs].fx=lastmv->fx-1;
      movestack[ply][nmvs].tx=lastmv->fx;
      movestack[ply][nmvs].fy=2+side;
      if (side==WHITE) movestack[ply][nmvs].ty=2;
      if (side==BLACK) movestack[ply][nmvs].ty=5;
      movestack[ply][nmvs].score=movestack[ply][nmvs].castle=movestack[ply][nmvs].promote=0;
      movestack[ply][nmvs].ep=-1;
      nmvs++;
    }
    if (a==1 || a==2) {
      movestack[ply][nmvs].fx=lastmv->fx+1;
      movestack[ply][nmvs].tx=lastmv->fx;
      movestack[ply][nmvs].fy=2+side;
      if (side==WHITE) movestack[ply][nmvs].ty=2;
      if (side==BLACK) movestack[ply][nmvs].ty=5;
      movestack[ply][nmvs].score=movestack[ply][nmvs].castle=movestack[ply][nmvs].promote=0;
      movestack[ply][nmvs].ep=1;
      nmvs++;
    }
  }
   
  /*  Go through all squares on the board, and generate moves */
  /*  for each piece of the correct colour */
  for (a=0;a<8;a++) {
    for (b=0;b<8;b++) {
      p=sq->sq[a][b];
      if ((side==WHITE && p<1) || (side==BLACK && p>-1)) continue;
      if (abs(p)==6) continue;
      n=0;
      if (side==WHITE) {
    	switch(p) {
  	  case 1: n=mppl(b,a,sq,1);break;
	  case 2: n=mprl(b,a,sq,1);break;
	  case 3: n=mpnl(b,a,sq,1);break;
	  case 4: n=mpbl(b,a,sq,1,0);break;
	  case 5: n=mpql(b,a,sq,1);break;
	  default: break;
	}
      }
      else {
  	switch (p) {
	  case -1: n=mppl(b,a,sq,2);break;
	  case -2: n=mprl(b,a,sq,2);break;
	  case -3: n=mpnl(b,a,sq,2);break;
	  case -4: n=mpbl(b,a,sq,2,0);break;
	  case -5: n=mpql(b,a,sq,2);break;
	  default : break;
	}
      }
      /*  Go through each generated move, and prune out the illegal ones */
      for (c=0;c<n;c++) {
        tmv=&temptsq[c];
 	flg=0;
	opt=sq->sq[tmv->ty][tmv->tx]; /*  Value of target square */
	sq->sq[tmv->ty][tmv->tx]=p; /*  Do the move */
	sq->sq[a][b]=0; /*  Blank previous square */
        if (abs(opt)>1) (sq->npieces[2-side][abs(opt)-2])--; /*  Remove piece if has been captured */

         /*  If side was in check before move then test to see if still in check */
        if (resflg!=0) flg=chck(sq,side);
         /*  Also test to see if the move has revealed a check. */
	else flg=revchck(sq,side,b,a,tmv->tx,tmv->ty);

	/*  So if side isn't in check.... */
	if (flg==0) {
	  /*  ...then store the move */
          do {
	    mov=&movestack[ply][nmvs];
            mov->fy=a;mov->fx=b;
            mov->ty=tmv->ty;mov->tx=tmv->tx;
	    mov->score=mov->castle=mov->ep=0;
            mov->promote=0;
	    nmvs++;
            /*  If pawn promotion then one for each of QNBR */
	    if (abs(p)==1 && (tmv->ty==7 || tmv->ty==0)) {
              if (flg==0) mov->promote=5*p;
              if (flg==1) mov->promote=3*p;
              if (flg==2) mov->promote=4*p;
              if (flg==3) mov->promote=2*p;
	      flg++;
            }
	    else flg=5;
          } while (flg<3);
	}
	sq->sq[a][b]=p;
	sq->sq[tmv->ty][tmv->tx]=opt;
        if (abs(opt)>1) (sq->npieces[2-side][abs(opt)-2])++;
      }
    }
  }
  *num=nmvs;
}

 /*  Generate all legal king moves (except castling) */
void kmvs(board *sq,int side,int *num,int ply) {
  int c,d,kx,ky,opt,p,nmvs=0,kpiece;
  move *mov;
   
  if (side==WHITE) {kx=sq->kpos[0];ky=sq->kpos[1];kpiece=6;}
  else {kx=sq->kpos[2];ky=sq->kpos[3];kpiece=-6;}
  for (c=ky-1;c<ky+2;c++) {
    if (c<0 || c>7) continue;
    for (d=kx-1;d<kx+2;d++) {
      if (d<0 || d>7) continue;
      if (c==ky && d==kx) continue;
      p=sq->sq[c][d];
      if ((side==WHITE && p<1) || (side==BLACK && p>-1)) {
        opt=sq->sq[c][d];
        sq->sq[c][d]=kpiece;
        sq->sq[ky][kx]=0;
        if (side==WHITE) {sq->kpos[0]=d;sq->kpos[1]=c;}
        else {sq->kpos[2]=d;sq->kpos[3]=c;}
        if (abs(opt)>1) (sq->npieces[2-side][abs(opt)-2])--;
        if (chck(sq,side)==0) {
	  mov=&movestack[ply][nmvs];
	  mov->fx=kx;mov->fy=ky;
	  mov->tx=d;mov->ty=c;
	  mov->score=mov->castle=mov->ep=0;
          mov->promote=0;
          nmvs++;
	}
	if (side==WHITE) {sq->kpos[0]=kx;sq->kpos[1]=ky;}
	else {sq->kpos[2]=kx;sq->kpos[3]=ky;}
	sq->sq[ky][kx]=kpiece;
	sq->sq[c][d]=opt;
        if (abs(opt)>1) (sq->npieces[2-side][abs(opt)-2])++;
      }
    }
  }
  *num=nmvs;
}

 /*  Generate all capture moves */
 /*  Also generate pawn advances and direct checks. */
 /*  Don't bother with castling :) */
 /*  This function assumes that you're _not_ in check!! */
void mvs2(board *sq,int side,int *num,move *lastmv,int ply) {
  int a=-1,b,c,p,opt,flg,okx,oky,nmvs=0,n;
  move *mov;
  tosquare *tmv;

   /*  Generate king moves which capture enemy pieces separately */
  kmvs2(sq,side,num,ply);
 
  if (side==WHITE) {okx=sq->kpos[2];oky=sq->kpos[3];}
  else {okx=sq->kpos[0];oky=sq->kpos[1];}

  nmvs=*num;
  /*  Generate en-passant captures if possible */
  if (lastmv) a=en_passant_possible(sq,lastmv,side);
  if (a>-1) {
    if (a==0 || a==2) {
      movestack[ply][nmvs].fx=lastmv->fx-1;
      movestack[ply][nmvs].tx=lastmv->fx;
      movestack[ply][nmvs].fy=2+side;
      if (side==WHITE) movestack[ply][nmvs].ty=2;
      if (side==BLACK) movestack[ply][nmvs].ty=5;
      movestack[ply][nmvs].score=movestack[ply][nmvs].castle=movestack[ply][nmvs].promote=0;
      movestack[ply][nmvs].ep=-1;
      nmvs++;
    }
    if (a==1 || a==2) {
      movestack[ply][nmvs].fx=lastmv->fx+1;
      movestack[ply][nmvs].tx=lastmv->fx;
      movestack[ply][nmvs].fy=2+side;
      if (side==WHITE) movestack[ply][nmvs].ty=2;
      if (side==BLACK) movestack[ply][nmvs].ty=5;
      movestack[ply][nmvs].score=movestack[ply][nmvs].castle=movestack[ply][nmvs].promote=0;
      movestack[ply][nmvs].ep=1;
      nmvs++;
    }
  }
  for (a=0;a<8;a++) {
    for (b=0;b<8;b++) {
      p=sq->sq[a][b];
      if ((side==WHITE && p<1) || (side==BLACK && p>-1)) continue;
      if (abs(p)==6) continue;
      n=0;
      switch(abs(p)) {
	case 1: n=mppl_qui(b,a,sq,side);break;
	case 2: n=mprl_qui(b,a,sq,side,okx,oky);break;
	case 3: n=mpnl_qui(b,a,sq,side,okx,oky);break;
	case 4: n=mpbl_qui(b,a,sq,side,okx,oky,0);break;
	case 5: n=mpql_qui(b,a,sq,side,okx,oky);break;
	default: break;
      }
      for (c=0;c<n;c++) {
        tmv=&temptsq[c];
 	flg=0;
	 
	opt=sq->sq[tmv->ty][tmv->tx];
if (abs(opt)==6) {fprintf(stderr,"(%d,%d - %d,%d)",b,a,tmv->tx,tmv->ty);print_board(sq);}
	sq->sq[tmv->ty][tmv->tx]=p;
	sq->sq[a][b]=0;
        if (abs(opt)>1) (sq->npieces[2-side][abs(opt)-2])--;

	flg=revchck(sq,side,b,a,tmv->tx,tmv->ty);
	 
	sq->sq[a][b]=p;
	sq->sq[tmv->ty][tmv->tx]=opt;
        if (abs(opt)>1) (sq->npieces[2-side][abs(opt)-2])++;
	 
	if (flg==0) {
          do {
	    mov=&movestack[ply][nmvs];
            mov->fy=a;mov->fx=b;
            mov->ty=tmv->ty;mov->tx=tmv->tx;
	    mov->score=mov->castle=mov->ep=0;
            mov->promote=0;
	    nmvs++;
            if (abs(p)==1 && (tmv->ty==7 || tmv->ty==0)) {
              if (flg==0) mov->promote=5*p;  /*  If pawn promotion then one for each of queen & knight. */
              if (flg==1) mov->promote=3*p;
              if (flg==2) mov->promote=4*p;
              if (flg==3) mov->promote=2*p;
	      flg++;
            }
	    else flg=5;
          } while (flg<3);
	}
      }
    }
  }
  *num=nmvs;
}

 /*  Generate all capture moves */
void gencapt(board *sq,int side,int *num,move *lastmv,int resflg,int ply) {
  int a=-1,b,c,p,opt,flg,nmvs=0,n;
  move *mov;
  tosquare *tmv;

   /*  Generate king moves that capture enemy pieces separately. */
  kmvs2(sq,side,num,ply);
 
  resflg=chck(sq,side);
  nmvs=*num;
  /*  Generate en-passant moves if possible */
  if (lastmv) a=en_passant_possible(sq,lastmv,side);
  else a=-1;
  if (a>-1) {
    if ((a&1)==0) {
      movestack[ply][nmvs].fx=lastmv->fx-1;
      movestack[ply][nmvs].tx=lastmv->fx;
      movestack[ply][nmvs].fy=2+side;
      if (side==WHITE) movestack[ply][nmvs].ty=2;
      if (side==BLACK) movestack[ply][nmvs].ty=5;
      movestack[ply][nmvs].score=movestack[ply][nmvs].castle=movestack[ply][nmvs].promote=0;
      movestack[ply][nmvs].ep=-1;
      nmvs++;
    }
    if (a>0) {
      movestack[ply][nmvs].fx=lastmv->fx+1;
      movestack[ply][nmvs].tx=lastmv->fx;
      movestack[ply][nmvs].fy=2+side;
      if (side==WHITE) movestack[ply][nmvs].ty=2;
      if (side==BLACK) movestack[ply][nmvs].ty=5;
      movestack[ply][nmvs].score=movestack[ply][nmvs].castle=movestack[ply][nmvs].promote=0;
      movestack[ply][nmvs].ep=1;
      nmvs++;
    }
  }
  for (a=0;a<8;a++) {
    for (b=0;b<8;b++) {
      p=sq->sq[a][b];
      if ((side==WHITE && p<1) || (side==BLACK && p>-1)) continue;
      if (abs(p)==6) continue;
      n=0;
      switch(abs(p)) {
	case 1: n=mppl_capt(b,a,sq,side);break;
	case 2: n=mprl_capt(b,a,sq,side);break;
	case 3: n=mpnl_capt(b,a,sq,side);break;
	case 4: n=mpbl_capt(b,a,sq,side,0);break;
	case 5: n=mpql_capt(b,a,sq,side);break;
        default: break;
      }
      for (c=0;c<n;c++) {
        tmv=&temptsq[c];
 	flg=0;
	opt=sq->sq[tmv->ty][tmv->tx];
	sq->sq[tmv->ty][tmv->tx]=p;
	sq->sq[a][b]=0;
        if (abs(opt)>1) (sq->npieces[2-side][abs(opt)-2])--;
	 
	if (resflg!=0) flg=chck(sq,side);
	else flg=revchck(sq,side,b,a,tmv->tx,tmv->ty);

	sq->sq[a][b]=p;
	sq->sq[tmv->ty][tmv->tx]=opt;
        if (abs(opt)>1) (sq->npieces[2-side][abs(opt)-2])++;
	 
	if (flg==0) {
	  do {
	    mov=&movestack[ply][nmvs];
            mov->fy=a;mov->fx=b;
            mov->ty=tmv->ty;mov->tx=tmv->tx;
	    mov->score=mov->castle=mov->ep=mov->promote=0;
	    nmvs++;
            if (abs(p)==1 && (tmv->ty==7 || tmv->ty==0)) {
              if (flg==0) mov->promote=5*p;  /*  If pawn promotion then one for each of queen & knight. */
              if (flg==1) mov->promote=3*p;
              if (flg==2) mov->promote=4*p;
              if (flg==3) mov->promote=2*p;
	      flg++;
            }
	    else flg=5;
	  } while (flg<3);
	}
      }
    }
  }
  *num=nmvs;
}

 /*  Generate all king moves which capture opponent pieces */
void kmvs2(board *sq,int side,int *num,int ply) {
  int c,d,kx,ky,opt,p,nmvs=0,kpiece;
  move *mov;

  if (side==WHITE) {kx=sq->kpos[0];ky=sq->kpos[1];kpiece=6;}
  else {kx=sq->kpos[2];ky=sq->kpos[3];kpiece=-6;}
  for (c=ky-1;c<ky+2;c++) {
    if (c<0 || c>7) continue;
    for (d=kx-1;d<kx+2;d++) {
      if (d<0 || d>7) continue;
      if (c==ky && d==kx) continue;
      p=sq->sq[c][d];
      if ((side==WHITE && p<0) || (side==BLACK && p>0)) {
        opt=sq->sq[c][d];
        sq->sq[c][d]=kpiece;
        sq->sq[ky][kx]=0;
        if (abs(opt)>1) (sq->npieces[2-side][abs(opt)-2])--;
        if (side==WHITE) {sq->kpos[0]=d;sq->kpos[1]=c;}
        else {sq->kpos[2]=d;sq->kpos[3]=c;}
        if (chck(sq,side)==0) {
	  mov=&movestack[ply][nmvs];
	  mov->fx=kx;mov->fy=ky;
	  mov->tx=d;mov->ty=c;
	  mov->score=mov->castle=mov->ep=0;mov->promote=0;
          nmvs++;
	}
	if (side==WHITE) {sq->kpos[0]=kx;sq->kpos[1]=ky;}
	else {sq->kpos[2]=kx;sq->kpos[3]=ky;}
	sq->sq[ky][kx]=kpiece;
	sq->sq[c][d]=opt;
        if (abs(opt)>1) (sq->npieces[2-side][abs(opt)-2])++;
      }
    }
  }
  *num=nmvs;
}

 /*  Do move & return a new board */
board *do_move(const board *sq,const move *mv,int side) {
  int p=sq->sq[mv->fy][mv->fx],pt;
  board *bd;

  bd=malloc(sizeof(board));
  assert(bd!=NULL);
  *bd=*sq;

  if (mv->castle==0 && mv->ep==0) {
    pt=sq->sq[mv->ty][mv->tx];
    bd->sq[mv->ty][mv->tx]=bd->sq[mv->fy][mv->fx];
    bd->sq[mv->fy][mv->fx]=0;
    if (abs(pt)>1 && abs(pt)<6) (bd->npieces[2-side][abs(pt)-2])--;
    if (p==6) {bd->kpos[0]=mv->tx;bd->kpos[1]=mv->ty;}
    if (p==-6) {bd->kpos[2]=mv->tx;bd->kpos[3]=mv->ty;}
    if (abs(p)==1) {
      if (mv->promote!=0) (bd->npieces[side-1][abs(mv->promote)-2])++;
      if (mv->promote!=0) bd->sq[mv->ty][mv->tx]=mv->promote;
      bd->pawnfile[mv->fx][side-1]--;
      if (mv->promote==0) bd->pawnfile[mv->tx][side-1]++;
    }
    if (abs(pt)==1) {
      if (pt==1) bd->pawnfile[mv->tx][0]--;
      else bd->pawnfile[mv->tx][1]--;
    }
    return bd;
  }
  else if (mv->ep!=0) {
    bd->sq[mv->fy][mv->fx]=0;
    bd->sq[mv->fy][mv->fx-mv->ep]=0;
    (bd->pawnfile[mv->tx][2-side])--;
    (bd->pawnfile[mv->fx][side-1])--;
    (bd->pawnfile[mv->tx][side-1])++;
    if (mv->fy==3) bd->sq[2][mv->tx]=1;
    else if (mv->fy==4) bd->sq[5][mv->tx]=-1;
    return bd;
  }
  else if (mv->castle==1) {
    if (side==WHITE) {
      bd->sq[7][4]=0;
      bd->sq[7][6]=6;
      bd->sq[7][7]=0;
      bd->sq[7][5]=2;
      bd->kpos[0]=6;
    }
    else {
      bd->sq[0][4]=0;
      bd->sq[0][6]=-6;
      bd->sq[0][7]=0;
      bd->sq[0][5]=-2;
      bd->kpos[2]=6;
    }
    return bd;
  }
  else {
    if (side==WHITE) {
      bd->sq[7][4]=0;
      bd->sq[7][2]=6;
      bd->sq[7][0]=0;
      bd->sq[7][3]=2;
      bd->kpos[0]=2;
    }
    else {
      bd->sq[0][4]=0;
      bd->sq[0][2]=-6;
      bd->sq[0][0]=0;
      bd->sq[0][3]=-2;
      bd->kpos[2]=2;
    }
  }
  return bd;
}

 /*  Do move on existing board (requires undo later!) */
 /*  Hopefully this function isn't screwed up as it's quite difficult to */
 /*  remember everything. */
void do_move2(board *bd,move *mv,int side) {
  int p=0,pt=0;

  if (!mv) return;
  mv->capt=0;
  p=bd->sq[mv->fy][mv->fx];
  if (mv->castle==0 && mv->ep==0) {
    pt=bd->sq[mv->ty][mv->tx];
    bd->sq[mv->ty][mv->tx]=p; /*  Move the piece */
    bd->sq[mv->fy][mv->fx]=0;
    if (abs(pt)>1) (bd->npieces[2-side][abs(pt)-2])--; /*  Reduce count of that piece */
    if (p==6) {bd->kpos[0]=mv->tx;bd->kpos[1]=mv->ty;} /*  Remove castling possibilities if King move */
    else if (p==-6) {bd->kpos[2]=mv->tx;bd->kpos[3]=mv->ty;} /*  Ditto */
    if (abs(p)==1) { /*  If Pawn move */
      if (mv->promote!=0) { /*  If Pawn promotion */
	(bd->npieces[side-1][abs(mv->promote)-2])++; /*  Increase number of promotion piece */
        bd->sq[mv->ty][mv->tx]=mv->promote; /*  Turn pawn into promoted piece */
      }
      else bd->pawnfile[mv->tx][side-1]++; /*  Put pawn on target file */
      bd->pawnfile[mv->fx][side-1]--; /*  Remove pawn from source file */
    }
    if (pt==1) bd->pawnfile[mv->tx][0]--; /*  Remove pawn from target file */
    else if (pt==-1) bd->pawnfile[mv->tx][1]--;
    mv->capt=pt; /*  Set target square piece identifier */
    return;
  }
  else if (mv->ep!=0) { /*  Otherwise, if En-Passant Capture... */
    bd->sq[mv->fy][mv->fx]=0;
    bd->sq[mv->fy][mv->tx]=0;
    bd->sq[mv->ty][mv->tx]=p;
    (bd->pawnfile[mv->tx][2-side])--;
    (bd->pawnfile[mv->fx][side-1])--;
    (bd->pawnfile[mv->tx][side-1])++;
    mv->capt=-p;
    return;
  }
  else if (mv->castle==1) { /* If King side castle */
    if (side==WHITE) { /*  White */
      bd->sq[7][4]=0;
      bd->sq[7][6]=6;
      bd->sq[7][7]=0;
      bd->sq[7][5]=2;
      bd->kpos[0]=6;
    }
    else { /*  Black */
      bd->sq[0][4]=0;
      bd->sq[0][6]=-6;
      bd->sq[0][7]=0;
      bd->sq[0][5]=-2;
      bd->kpos[2]=6;
    }
    return;
  }
  else { /*  or queen's side castle */
    if (side==WHITE) { /*  White */
      bd->sq[7][4]=0;
      bd->sq[7][2]=6;
      bd->sq[7][0]=0;
      bd->sq[7][3]=2;
      bd->kpos[0]=2;
    }
    else { /*  Black */
      bd->sq[0][4]=0;
      bd->sq[0][2]=-6;
      bd->sq[0][0]=0;
      bd->sq[0][3]=-2;
      bd->kpos[2]=2;
    }
  }
  return; /*  return 0 for castle (target square empty always) */
}

 /*  Put a move back as it was!  pt is old target square identifier */
 /*  If this function is bugged then it might be quite subtle as it */
 /*  was pretty damn hard to write. */
void undo_move(board *bd,const move *mv,int side) {
  int p=0,pt=0;

  if (!mv) return;
  p=bd->sq[mv->ty][mv->tx];
  pt=mv->capt;
  if (mv->castle==0 && mv->ep==0) {
    bd->sq[mv->ty][mv->tx]=pt;
    bd->sq[mv->fy][mv->fx]=p;
    if (abs(pt)>1) (bd->npieces[2-side][abs(pt)-2])++;
    if (p==6) {bd->kpos[0]=mv->fx;bd->kpos[1]=mv->fy;}
    else if (p==-6) {bd->kpos[2]=mv->fx;bd->kpos[3]=mv->fy;}
    if (abs(p)==1 || mv->promote!=0) {
      if (mv->promote!=0) { /*  Pawn promotion */
	(bd->npieces[side-1][abs(mv->promote)-2])--; 
        if (mv->promote>0) bd->sq[mv->fy][mv->fx]=1;
        if (mv->promote<0) bd->sq[mv->fy][mv->fx]=-1;
      }
      if (mv->promote==0) bd->pawnfile[mv->tx][side-1]--;
      bd->pawnfile[mv->fx][side-1]++;
    }
    if (abs(pt)==1) { /*  Pawn is target */
      if (pt==1) bd->pawnfile[mv->tx][0]++;
      else bd->pawnfile[mv->tx][1]++;
    }
    return;
  }
  else if (mv->ep!=0) {
    bd->sq[mv->fy][mv->fx]=p;
    bd->sq[mv->ty][mv->tx]=0;
    bd->sq[mv->fy][mv->tx]=-p;
    (bd->pawnfile[mv->tx][2-side])++;
    (bd->pawnfile[mv->fx][side-1])++;
    (bd->pawnfile[mv->tx][side-1])--;
    return;
  }
  else if (mv->castle==1) {
    if (side==WHITE) {
      bd->sq[7][4]=6;
      bd->sq[7][6]=0;
      bd->sq[7][7]=2;
      bd->sq[7][5]=0;
      bd->kpos[0]=4;
    }
    else {
      bd->sq[0][4]=-6;
      bd->sq[0][6]=0;
      bd->sq[0][7]=-2;
      bd->sq[0][5]=0;
      bd->kpos[2]=4;
    }
    return;
  }
  else {
    if (side==WHITE) {
      bd->sq[7][4]=6;
      bd->sq[7][2]=0;
      bd->sq[7][0]=2;
      bd->sq[7][3]=0;
      bd->kpos[0]=4;
    }
    else {
      bd->sq[0][4]=-6;
      bd->sq[0][2]=0;
      bd->sq[0][0]=-2;
      bd->sq[0][3]=0;
      bd->kpos[2]=4;
    }
  }
  return;
}

 /*  Add move to the current move list. */
 /*  No need to be fast :) */
void add_move(int side,int mvno,int fx,int fy,int tx,int ty,int castle,int ep,int promote,int capt) {
  int nm;

  nm=(mvno*2)-1;
  if (side==WHITE) nm--; /*  nm = number of moves */
  /*  Put values of new element in */
  mvlst[nm].fx=fx;
  mvlst[nm].fy=fy;
  mvlst[nm].tx=tx;
  mvlst[nm].ty=ty;
  mvlst[nm].castle=castle;
  mvlst[nm].score=0;
  mvlst[nm].ep=ep;
  mvlst[nm].promote=promote;
  mvlst[nm].capt=capt;
}

 /*  Takes a list of moves, and places the best oe in first place */
void sort_from(int nm,int pos,int ply) {
  int n,bn=pos;
  long int bsc=movestack[ply][pos].score;
  move temp;
   
  if (pos>=nm) return;
  for (n=pos+1;n<nm;n++) {
    if (movestack[ply][n].score>bsc) {bn=n;bsc=movestack[ply][n].score;}
  }
  if (bn==pos) return;
  temp=movestack[ply][pos];
  movestack[ply][pos]=movestack[ply][bn];
  movestack[ply][bn]=temp;
}

 /*  Prints out a move in the standard CAPS, wide spaced format */
void printmove(move *mv) {
  if (mv==NULL) return;
  if (mv->fx==-1) {fprintf(stderr,".......");return;}
  if (mv->castle==0) {
    if (mv->promote==0) {
      fprintf(stderr,"%c%d ",(char)((mv->fx)+65),8-(mv->fy));
      if (mv->capt==0) fprintf(stderr,"- ");
      else fprintf(stderr,"x ");
    }
    else {
      fprintf(stderr,"P");
      if (mv->capt==0) fprintf(stderr,"-");
      else fprintf(stderr,"x");
    }
    fprintf(stderr,"%c%d",(char)((mv->tx)+65),8-(mv->ty));
    if (mv->promote!=0) fprintf(stderr,"(");
    if (abs(mv->promote)==2) fprintf(stderr,"R");
    if (abs(mv->promote)==3) fprintf(stderr,"N");
    if (abs(mv->promote)==4) fprintf(stderr,"B");
    if (abs(mv->promote)==5) fprintf(stderr,"Q");
    if (mv->promote!=0) fprintf(stderr,")");
  }
  else if (mv->castle==1) fprintf(stderr,"0-0    ");
  else if (mv->castle==2) fprintf(stderr,"0-0-0  ");
  else fprintf(stderr,"ERROR IN PRINTMOVE!");
}

/*  Prints move in a shortened lowercase format */
void printmove2(move *mv) {
  if (mv==NULL) return;
  if (mv->fx==-1) {fprintf(stderr,"...");return;}
  if (mv->castle==0) {
    if (mv->promote==0) fprintf(stderr,"%c%d%c%d",(char)((mv->fx)+97),8-(mv->fy),(char)((mv->tx)+97),8-(mv->ty));
    if (abs(mv->promote)==2) fprintf(stderr,"%c%d%c%dr",(char)((mv->fx)+97),8-(mv->fy),(char)((mv->tx)+97),8-(mv->ty));
    if (abs(mv->promote)==3) fprintf(stderr,"%c%d%c%dn",(char)((mv->fx)+97),8-(mv->fy),(char)((mv->tx)+97),8-(mv->ty));
    if (abs(mv->promote)==4) fprintf(stderr,"%c%d%c%db",(char)((mv->fx)+97),8-(mv->fy),(char)((mv->tx)+97),8-(mv->ty));
    if (abs(mv->promote)==5) fprintf(stderr,"%c%d%c%dq",(char)((mv->fx)+97),8-(mv->fy),(char)((mv->tx)+97),8-(mv->ty));
  }
  else if (mv->castle==1) {
    if (mv->fy==7) fprintf(stderr,"e1g1");
    else fprintf(stderr,"e8g8");
  }
  else if (mv->castle==2) {
    if (mv->fy==7) fprintf(stderr,"e1c1");
    else fprintf(stderr,"e8c8");
  }
  if (mv->score==CMSCORE-1) fprintf(stderr,"#");
  else if (mv->gchk) fprintf(stderr,"+");
}

 /*  Compare two moves, returning 1/0 if they are the same/different. */
int compare(move *m1,move *m2) {
  if (m1==NULL || m2==NULL) return 0;
  if (m1->castle==m2->castle && m1->castle!=0) return 1;
  if (m1->fx!=m2->fx) return 0;
  if (m1->fy!=m2->fy) return 0;
  if (m1->tx!=m2->tx) return 0;
  if (m1->ty!=m2->ty) return 0;
  if (m1->castle!=m2->castle) return 0;
  if (m1->ep!=m2->ep) return 0;
  if (m1->promote!=m2->promote) return 0;
  return 1;
}

 /*  Compare move 'mv' with movedata stored in hash format */
int compare2(move *mv,int moved,int mvars) {
  int nmove,nmvars;

  nmove=mv->fx+(mv->fy<<3)+(mv->tx<<6)+(mv->ty<<9);
  if (nmove!=moved) return 0;
  nmvars=mv->ep+1+(abs(mv->promote)<<2)+(mv->castle<<5);
  if (nmvars!=mvars) return 0;
  return 1;
}
 
										
										
